/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.accumulo.shell.commands;

import java.io.PrintWriter;
import java.io.Writer;
import java.lang.reflect.Field;
import java.util.EnumSet;

import org.apache.accumulo.core.client.Connector;
import org.apache.accumulo.core.client.IteratorSetting;
import org.apache.accumulo.core.client.admin.TableOperations;
import org.apache.accumulo.core.iterators.IteratorUtil.IteratorScope;
import org.apache.accumulo.core.iterators.SortedKeyValueIterator;
import org.apache.accumulo.shell.Shell;
import org.apache.accumulo.start.classloader.vfs.AccumuloVFSClassLoader;
import org.apache.commons.cli.CommandLine;
import org.easymock.EasyMock;
import org.junit.Before;
import org.junit.Test;

import jline.console.ConsoleReader;

public class SetIterCommandTest {
  private SetIterCommand cmd;

  @Before
  public void setup() {
    cmd = new SetIterCommand();

    // Initialize that internal state
    cmd.getOptions();
  }

  @Test
  public void addColumnAgeOffFilter() throws Exception {
    Connector conn = EasyMock.createMock(Connector.class);
    CommandLine cli = EasyMock.createMock(CommandLine.class);
    Shell shellState = EasyMock.createMock(Shell.class);
    ConsoleReader reader = EasyMock.createMock(ConsoleReader.class);
    Writer out = EasyMock.createMock(PrintWriter.class);
    TableOperations tableOperations = EasyMock.createMock(TableOperations.class);

    // Command line parsing
    EasyMock.expect(cli.hasOption("all")).andReturn(true);
    EasyMock.expect(cli.hasOption("all")).andReturn(true);
    EasyMock.expect(cli.hasOption("all")).andReturn(true);
    EasyMock.expect(cli.hasOption("t")).andReturn(true);
    EasyMock.expect(cli.hasOption("t")).andReturn(true);
    EasyMock.expect(cli.getOptionValue("t")).andReturn("foo");
    EasyMock.expect(cli.hasOption("ns")).andReturn(false);
    EasyMock.expect(cli.getOptionValue("p")).andReturn("21");
    EasyMock.expect(cli.getOptionValue("class"))
        .andReturn("org.apache.accumulo.core.iterators.user.ColumnAgeOffFilter");
    EasyMock.expect(cli.hasOption("ageoff")).andReturn(false);
    EasyMock.expect(cli.hasOption("agg")).andReturn(false);
    EasyMock.expect(cli.hasOption("regex")).andReturn(false);
    EasyMock.expect(cli.hasOption("reqvis")).andReturn(false);
    EasyMock.expect(cli.hasOption("vers")).andReturn(false);
    EasyMock.expect(cli.getOptionValue("n", null)).andReturn(null);

    // Loading the class
    EasyMock.expect(shellState.getClassLoader(cli, shellState))
        .andReturn(AccumuloVFSClassLoader.getClassLoader());

    // Set the output object
    Field field = reader.getClass().getSuperclass().getDeclaredField("out");
    field.setAccessible(true);
    field.set(reader, out);

    // Parsing iterator options
    reader.flush();
    EasyMock.expectLastCall().times(3);

    EasyMock.expect(shellState.getReader()).andReturn(reader);

    // Shell asking for negate option, we pass in an empty string to pickup the default value of
    // 'false'
    EasyMock.expect(reader.readLine(EasyMock.anyObject(String.class))).andReturn("");

    // Shell asking for the unnamed option for the column (a:a) and the TTL (1)
    EasyMock.expect(reader.readLine(EasyMock.anyObject(String.class))).andReturn("a:a 1");

    // Shell asking for another unnamed option; we pass in an empty string to signal that we are
    // done adding options
    EasyMock.expect(reader.readLine(EasyMock.anyObject(String.class))).andReturn("");
    EasyMock.expect(shellState.getConnector()).andReturn(conn);

    // Table exists
    EasyMock.expect(conn.tableOperations()).andReturn(tableOperations);
    EasyMock.expect(tableOperations.exists("foo")).andReturn(true);

    // Testing class load
    EasyMock.expect(shellState.getConnector()).andReturn(conn);
    EasyMock.expect(conn.tableOperations()).andReturn(tableOperations);
    EasyMock.expect(tableOperations.testClassLoad("foo",
        "org.apache.accumulo.core.iterators.user.ColumnAgeOffFilter",
        SortedKeyValueIterator.class.getName())).andReturn(true);

    // Attach iterator
    EasyMock.expect(shellState.getConnector()).andReturn(conn);
    EasyMock.expect(conn.tableOperations()).andReturn(tableOperations);
    tableOperations.attachIterator(EasyMock.eq("foo"), EasyMock.anyObject(IteratorSetting.class),
        EasyMock.eq(EnumSet.allOf(IteratorScope.class)));
    EasyMock.expectLastCall().once();

    EasyMock.replay(conn, cli, shellState, reader, tableOperations);

    cmd.execute(
        "setiter -all -p 21 -t foo"
            + " -class org.apache.accumulo.core.iterators.user.ColumnAgeOffFilter",
        cli, shellState);

    EasyMock.verify(conn, cli, shellState, reader, tableOperations);
  }
}
