/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.service;

import com.google.common.collect.ImmutableSet;
import java.net.InetSocketAddress;
import java.net.SocketAddress;
import java.util.Arrays;
import java.util.List;
import java.util.Set;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.cassandra.auth.AuthenticatedUser;
import org.apache.cassandra.auth.DataResource;
import org.apache.cassandra.auth.FunctionResource;
import org.apache.cassandra.auth.IResource;
import org.apache.cassandra.auth.Permission;
import org.apache.cassandra.auth.Resources;
import org.apache.cassandra.config.CFMetaData;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.config.Schema;
import org.apache.cassandra.config.SchemaConstants;
import org.apache.cassandra.cql3.QueryHandler;
import org.apache.cassandra.cql3.QueryProcessor;
import org.apache.cassandra.cql3.functions.Function;
import org.apache.cassandra.exceptions.AuthenticationException;
import org.apache.cassandra.exceptions.InvalidRequestException;
import org.apache.cassandra.exceptions.RequestExecutionException;
import org.apache.cassandra.exceptions.UnauthorizedException;
import org.apache.cassandra.schema.SchemaKeyspaceTables;
import org.apache.cassandra.service.ClientWarn;
import org.apache.cassandra.thrift.ThriftValidation;
import org.apache.cassandra.utils.CassandraVersion;
import org.apache.cassandra.utils.FBUtilities;
import org.apache.cassandra.utils.JVMStabilityInspector;
import org.apache.cassandra.utils.MD5Digest;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ClientState {
    private static final Logger logger = LoggerFactory.getLogger(ClientState.class);
    public static final CassandraVersion DEFAULT_CQL_VERSION = QueryProcessor.CQL_VERSION;
    public static final ImmutableSet<IResource> READABLE_SYSTEM_RESOURCES;
    public static final ImmutableSet<IResource> PROTECTED_AUTH_RESOURCES;
    public static final ImmutableSet<IResource> DROPPABLE_SYSTEM_AUTH_TABLES;
    private volatile AuthenticatedUser user;
    private volatile String keyspace;
    private volatile boolean issuedPreparedStatementsUseWarning;
    private volatile boolean noCompactMode;
    private static final QueryHandler cqlQueryHandler;
    public final boolean isInternal;
    private final InetSocketAddress remoteAddress;
    private static final AtomicLong lastTimestampMicros;

    private ClientState() {
        this.isInternal = true;
        this.remoteAddress = null;
    }

    protected ClientState(InetSocketAddress remoteAddress) {
        this.isInternal = false;
        this.remoteAddress = remoteAddress;
        if (!DatabaseDescriptor.getAuthenticator().requireAuthentication()) {
            this.user = AuthenticatedUser.ANONYMOUS_USER;
        }
    }

    public static ClientState forInternalCalls() {
        return new ClientState();
    }

    public static ClientState forExternalCalls(SocketAddress remoteAddress) {
        return new ClientState((InetSocketAddress)remoteAddress);
    }

    public long getTimestamp() {
        long current;
        long tstamp;
        long last;
        do {
            current = System.currentTimeMillis() * 1000L;
        } while (!lastTimestampMicros.compareAndSet(last, tstamp = (last = lastTimestampMicros.get()) >= current ? last + 1L : current));
        return tstamp;
    }

    public long getTimestampForPaxos(long minTimestampToUse) {
        long last;
        long tstamp;
        do {
            long current = Math.max(System.currentTimeMillis() * 1000L, minTimestampToUse);
            last = lastTimestampMicros.get();
            long l = tstamp = last >= current ? last + 1L : current;
        } while (tstamp != minTimestampToUse && !lastTimestampMicros.compareAndSet(last, tstamp));
        return tstamp;
    }

    public static QueryHandler getCQLQueryHandler() {
        return cqlQueryHandler;
    }

    public InetSocketAddress getRemoteAddress() {
        return this.remoteAddress;
    }

    public String getRawKeyspace() {
        return this.keyspace;
    }

    public String getKeyspace() throws InvalidRequestException {
        if (this.keyspace == null) {
            throw new InvalidRequestException("No keyspace has been specified. USE a keyspace, or explicitly specify keyspace.tablename");
        }
        return this.keyspace;
    }

    public void setKeyspace(String ks) throws InvalidRequestException {
        if (this.user != null && Schema.instance.getKSMetaData(ks) == null) {
            throw new InvalidRequestException("Keyspace '" + ks + "' does not exist");
        }
        this.keyspace = ks;
    }

    public void setNoCompactMode() {
        this.noCompactMode = true;
    }

    public boolean isNoCompactMode() {
        return this.noCompactMode;
    }

    public void login(AuthenticatedUser user) throws AuthenticationException {
        if (!user.isAnonymous() && !this.canLogin(user)) {
            throw new AuthenticationException(String.format("%s is not permitted to log in", user.getName()));
        }
        this.user = user;
    }

    private boolean canLogin(AuthenticatedUser user) {
        try {
            return DatabaseDescriptor.getRoleManager().canLogin(user.getPrimaryRole());
        }
        catch (RequestExecutionException e) {
            throw new AuthenticationException("Unable to perform authentication: " + e.getMessage(), e);
        }
    }

    public void hasAllKeyspacesAccess(Permission perm) throws UnauthorizedException {
        if (this.isInternal) {
            return;
        }
        this.validateLogin();
        this.ensureHasPermission(perm, DataResource.root());
    }

    public void hasKeyspaceAccess(String keyspace, Permission perm) throws UnauthorizedException, InvalidRequestException {
        this.hasAccess(keyspace, perm, DataResource.keyspace(keyspace));
    }

    public void hasColumnFamilyAccess(String keyspace, String columnFamily, Permission perm) throws UnauthorizedException, InvalidRequestException {
        ThriftValidation.validateColumnFamily(keyspace, columnFamily);
        this.hasAccess(keyspace, perm, DataResource.table(keyspace, columnFamily));
    }

    public void hasColumnFamilyAccess(CFMetaData cfm, Permission perm) throws UnauthorizedException, InvalidRequestException {
        this.hasAccess(cfm.ksName, perm, cfm.resource);
    }

    private void hasAccess(String keyspace, Permission perm, DataResource resource) throws UnauthorizedException, InvalidRequestException {
        ClientState.validateKeyspace(keyspace);
        if (this.isInternal) {
            return;
        }
        this.validateLogin();
        this.preventSystemKSSchemaModification(keyspace, resource, perm);
        if (perm == Permission.SELECT && READABLE_SYSTEM_RESOURCES.contains((Object)resource)) {
            return;
        }
        if (PROTECTED_AUTH_RESOURCES.contains((Object)resource) && (perm == Permission.CREATE || perm == Permission.ALTER || perm == Permission.DROP)) {
            throw new UnauthorizedException(String.format("%s schema is protected", resource));
        }
        this.ensureHasPermission(perm, resource);
    }

    public void ensureHasPermission(Permission perm, IResource resource) throws UnauthorizedException {
        String keyspace;
        DataResource dataResource;
        if (!DatabaseDescriptor.getAuthorizer().requireAuthorization()) {
            return;
        }
        if (resource instanceof FunctionResource && resource.hasParent() && ((FunctionResource)resource).getKeyspace().equals("system")) {
            return;
        }
        if (resource instanceof DataResource && !(dataResource = (DataResource)resource).isRootLevel() && SchemaConstants.isSystemKeyspace(keyspace = dataResource.getKeyspace())) {
            if (!this.user.isSystem() && !this.user.isSuper()) {
                this.ensurePermissionOnResourceChain(perm, Resources.chain(dataResource, IResource::hasParent));
            }
            return;
        }
        this.ensurePermissionOnResourceChain(perm, resource);
    }

    public void ensureHasPermission(Permission permission, Function function) {
        if (!DatabaseDescriptor.getAuthorizer().requireAuthorization()) {
            return;
        }
        if (function.isNative()) {
            return;
        }
        this.ensurePermissionOnResourceChain(permission, FunctionResource.function(function.name().keyspace, function.name().name, function.argTypes()));
    }

    private void ensurePermissionOnResourceChain(Permission perm, IResource resource) {
        this.ensurePermissionOnResourceChain(perm, Resources.chain(resource));
    }

    private void ensurePermissionOnResourceChain(Permission perm, List<? extends IResource> resources) {
        IResource resource = resources.get(0);
        for (IResource iResource : resources) {
            if (!this.authorize(iResource).contains((Object)perm)) continue;
            return;
        }
        throw new UnauthorizedException(String.format("User %s has no %s permission on %s or any of its parents", new Object[]{this.user.getName(), perm, resource}));
    }

    private void preventSystemKSSchemaModification(String keyspace, DataResource resource, Permission perm) throws UnauthorizedException {
        if (perm != Permission.ALTER && perm != Permission.DROP && perm != Permission.CREATE) {
            return;
        }
        if (SchemaConstants.isLocalSystemKeyspace(keyspace)) {
            throw new UnauthorizedException(keyspace + " keyspace is not user-modifiable.");
        }
        if (SchemaConstants.isReplicatedSystemKeyspace(keyspace)) {
            if (perm == Permission.ALTER && resource.isKeyspaceLevel()) {
                return;
            }
            if (perm == Permission.DROP && DROPPABLE_SYSTEM_AUTH_TABLES.contains((Object)resource)) {
                return;
            }
            throw new UnauthorizedException(String.format("Cannot %s %s", new Object[]{perm, resource}));
        }
    }

    public void validateLogin() throws UnauthorizedException {
        if (this.user == null) {
            throw new UnauthorizedException("You have not logged in");
        }
    }

    public void ensureNotAnonymous() throws UnauthorizedException {
        this.validateLogin();
        if (this.user.isAnonymous()) {
            throw new UnauthorizedException("You have to be logged in and not anonymous to perform this request");
        }
    }

    public void ensureIsSuper(String message) throws UnauthorizedException {
        if (DatabaseDescriptor.getAuthenticator().requireAuthentication() && (this.user == null || !this.user.isSuper())) {
            throw new UnauthorizedException(message);
        }
    }

    private static void validateKeyspace(String keyspace) throws InvalidRequestException {
        if (keyspace == null) {
            throw new InvalidRequestException("You have not set a keyspace for this session");
        }
    }

    public AuthenticatedUser getUser() {
        return this.user;
    }

    public static CassandraVersion[] getCQLSupportedVersion() {
        return new CassandraVersion[]{QueryProcessor.CQL_VERSION};
    }

    private Set<Permission> authorize(IResource resource) {
        return this.user.getPermissions(resource);
    }

    public void warnAboutUseWithPreparedStatements(MD5Digest statementId, String preparedKeyspace) {
        if (!this.issuedPreparedStatementsUseWarning) {
            ClientWarn.instance.warn(String.format("`USE <keyspace>` with prepared statements is considered to be an anti-pattern due to ambiguity in non-qualified table names. Please consider removing instances of `Session#setKeyspace(<keyspace>)`, `Session#execute(\"USE <keyspace>\")` and `cluster.newSession(<keyspace>)` from your code, and always use fully qualified table names (e.g. <keyspace>.<table>). Keyspace used: %s, statement keyspace: %s, statement id: %s", this.getRawKeyspace(), preparedKeyspace, statementId));
            this.issuedPreparedStatementsUseWarning = true;
        }
    }

    static {
        ImmutableSet.Builder readableBuilder = ImmutableSet.builder();
        for (String cf : Arrays.asList("local", "peers", "size_estimates")) {
            readableBuilder.add((Object)DataResource.table("system", cf));
        }
        SchemaKeyspaceTables.ALL.forEach(table -> readableBuilder.add((Object)DataResource.table("system_schema", table)));
        readableBuilder.add((Object)DataResource.table("system_traces", "events"));
        readableBuilder.add((Object)DataResource.table("system_traces", "sessions"));
        READABLE_SYSTEM_RESOURCES = readableBuilder.build();
        ImmutableSet.Builder protectedBuilder = ImmutableSet.builder();
        if (DatabaseDescriptor.isDaemonInitialized()) {
            protectedBuilder.addAll(DatabaseDescriptor.getAuthenticator().protectedResources());
            protectedBuilder.addAll(DatabaseDescriptor.getAuthorizer().protectedResources());
            protectedBuilder.addAll(DatabaseDescriptor.getRoleManager().protectedResources());
        }
        PROTECTED_AUTH_RESOURCES = protectedBuilder.build();
        ImmutableSet.Builder droppableBuilder = ImmutableSet.builder();
        droppableBuilder.add((Object)DataResource.table("system_auth", "credentials"));
        droppableBuilder.add((Object)DataResource.table("system_auth", "users"));
        droppableBuilder.add((Object)DataResource.table("system_auth", "permissions"));
        DROPPABLE_SYSTEM_AUTH_TABLES = droppableBuilder.build();
        QueryHandler handler = QueryProcessor.instance;
        String customHandlerClass = System.getProperty("cassandra.custom_query_handler_class");
        if (customHandlerClass != null) {
            try {
                handler = (QueryHandler)FBUtilities.construct(customHandlerClass, "QueryHandler");
                logger.info("Using {} as query handler for native protocol queries (as requested with -Dcassandra.custom_query_handler_class)", (Object)customHandlerClass);
            }
            catch (Exception e) {
                logger.error("Cannot use class {} as query handler", (Object)customHandlerClass, (Object)e);
                JVMStabilityInspector.killCurrentJVM(e, true);
            }
        }
        cqlQueryHandler = handler;
        lastTimestampMicros = new AtomicLong(0L);
    }
}

