/*
 * Decompiled with CFR 0.152.
 */
package org.apache.helix.task;

import com.google.common.collect.Lists;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.Map;
import java.util.Set;
import java.util.TimeZone;
import org.apache.helix.HelixDataAccessor;
import org.apache.helix.HelixManager;
import org.apache.helix.HelixProperty;
import org.apache.helix.PropertyKey;
import org.apache.helix.common.caches.TaskDataCache;
import org.apache.helix.controller.LogUtil;
import org.apache.helix.controller.dataproviders.WorkflowControllerDataProvider;
import org.apache.helix.controller.stages.BestPossibleStateOutput;
import org.apache.helix.controller.stages.CurrentStateOutput;
import org.apache.helix.model.ResourceAssignment;
import org.apache.helix.task.AbstractTaskDispatcher;
import org.apache.helix.task.JobConfig;
import org.apache.helix.task.JobDag;
import org.apache.helix.task.JobDispatcher;
import org.apache.helix.task.RuntimeJobDag;
import org.apache.helix.task.ScheduleConfig;
import org.apache.helix.task.TargetState;
import org.apache.helix.task.TaskConfig;
import org.apache.helix.task.TaskConstants;
import org.apache.helix.task.TaskDriver;
import org.apache.helix.task.TaskState;
import org.apache.helix.task.TaskUtil;
import org.apache.helix.task.Workflow;
import org.apache.helix.task.WorkflowConfig;
import org.apache.helix.task.WorkflowContext;
import org.apache.helix.zookeeper.datamodel.ZNRecord;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class WorkflowDispatcher
extends AbstractTaskDispatcher {
    private static final Logger LOG = LoggerFactory.getLogger(WorkflowDispatcher.class);
    private WorkflowControllerDataProvider _clusterDataCache;
    private JobDispatcher _jobDispatcher;

    public void updateCache(WorkflowControllerDataProvider cache) {
        this._clusterDataCache = cache;
        if (this._jobDispatcher == null) {
            this._jobDispatcher = new JobDispatcher();
        }
        this._jobDispatcher.init(this._manager);
        this._jobDispatcher.updateCache(cache);
        this._jobDispatcher.setClusterStatusMonitor(this._clusterStatusMonitor);
    }

    public void updateWorkflowStatus(String workflow, WorkflowConfig workflowCfg, WorkflowContext workflowCtx, CurrentStateOutput currentStateOutput, BestPossibleStateOutput bestPossibleOutput) {
        if (workflowCfg == null) {
            LOG.warn("Workflow configuration is NULL for {}", (Object)workflow);
            return;
        }
        TargetState targetState = workflowCfg.getTargetState();
        if (targetState == TargetState.DELETE) {
            LOG.debug("Workflow is marked as deleted {} cleaning up the workflow context.", (Object)workflow);
            this.updateInflightJobs(workflow, workflowCtx, currentStateOutput, bestPossibleOutput);
            this.cleanupWorkflow(workflow);
            return;
        }
        if (!workflowCfg.isJobQueue() && !TaskConstants.FINAL_STATES.contains((Object)workflowCtx.getWorkflowState())) {
            this.scheduleRebalanceForTimeout(workflow, workflowCtx.getStartTime(), workflowCfg.getTimeout());
            if (!TaskState.TIMED_OUT.equals((Object)workflowCtx.getWorkflowState()) && this.isTimeout(workflowCtx.getStartTime(), workflowCfg.getTimeout())) {
                workflowCtx.setWorkflowState(TaskState.TIMED_OUT);
                this._clusterDataCache.updateWorkflowContext(workflow, workflowCtx);
            }
        }
        long currentTime = System.currentTimeMillis();
        if (workflowCtx.getFinishTime() == -1L && this.isWorkflowFinished(workflowCtx, workflowCfg, this._clusterDataCache.getJobConfigMap(), this._clusterDataCache)) {
            workflowCtx.setFinishTime(currentTime);
            this.updateWorkflowMonitor(workflowCtx, workflowCfg);
            this._clusterDataCache.updateWorkflowContext(workflow, workflowCtx);
        }
        if (workflowCtx.getFinishTime() != -1L) {
            LOG.debug("Workflow {} is finished.", (Object)workflow);
            this.updateInflightJobs(workflow, workflowCtx, currentStateOutput, bestPossibleOutput);
            long expiryTime = workflowCfg.getExpiry();
            if (workflowCtx.getFinishTime() + expiryTime <= currentTime) {
                LOG.info("Workflow {} passed expiry time, cleaning up the workflow context.", (Object)workflow);
                this.cleanupWorkflow(workflow);
            } else {
                long cleanupTime = workflowCtx.getFinishTime() + expiryTime;
                _rebalanceScheduler.scheduleRebalance(this._manager, workflow, cleanupTime);
            }
            return;
        }
        if (!workflowCfg.isTerminable() || workflowCfg.isJobQueue()) {
            HashSet<String> jobWithFinalStates = new HashSet<String>(workflowCtx.getJobStates().keySet());
            jobWithFinalStates.removeAll(workflowCfg.getJobDag().getAllNodes());
            if (jobWithFinalStates.size() > 0) {
                workflowCtx.setLastJobPurgeTime(System.currentTimeMillis());
                workflowCtx.removeJobStates(jobWithFinalStates);
                workflowCtx.removeJobStartTime(jobWithFinalStates);
            }
        }
        this.updateInflightJobs(workflow, workflowCtx, currentStateOutput, bestPossibleOutput);
        if (!TaskConstants.FINAL_STATES.contains((Object)workflowCtx.getWorkflowState()) && TargetState.STOP.equals((Object)targetState)) {
            if (this.isWorkflowStopped(workflowCtx, workflowCfg) && workflowCtx.getWorkflowState() != TaskState.STOPPED) {
                LOG.debug("Workflow {} is marked as stopped. Workflow state is {}", (Object)workflow, (Object)workflowCtx.getWorkflowState());
                workflowCtx.setWorkflowState(TaskState.STOPPED);
                this._clusterDataCache.updateWorkflowContext(workflow, workflowCtx);
            }
            return;
        }
        if (targetState.equals((Object)TargetState.START) && workflowCtx.getWorkflowState() == TaskState.STOPPED) {
            workflowCtx.setWorkflowState(TaskState.IN_PROGRESS);
        }
        this._clusterDataCache.updateWorkflowContext(workflow, workflowCtx);
    }

    private void updateInflightJobs(String workflow, WorkflowContext workflowCtx, CurrentStateOutput currentStateOutput, BestPossibleStateOutput bestPossibleOutput) {
        RuntimeJobDag runtimeJobDag = this._clusterDataCache.getTaskDataCache().getRuntimeJobDag(workflow);
        if (runtimeJobDag != null) {
            for (String inflightJob : runtimeJobDag.getInflightJobList()) {
                if (System.currentTimeMillis() < workflowCtx.getJobStartTime(inflightJob)) continue;
                this.processJob(inflightJob, currentStateOutput, bestPossibleOutput, workflowCtx);
            }
        } else {
            LOG.warn("Failed to find runtime job DAG for workflow {}, existing runtime jobs may not be processed correctly for it", (Object)workflow);
        }
    }

    public void assignWorkflow(String workflow, WorkflowConfig workflowCfg, WorkflowContext workflowCtx, CurrentStateOutput currentStateOutput, BestPossibleStateOutput bestPossibleOutput) {
        if (workflowCfg == null) {
            return;
        }
        if (!this.isWorkflowReadyForSchedule(workflowCfg)) {
            LOG.info("Workflow {} is not ready to schedule, schedule future rebalance at {}", (Object)workflow, (Object)workflowCfg.getStartTime().getTime());
            _rebalanceScheduler.scheduleRebalance(this._manager, workflow, workflowCfg.getStartTime().getTime());
            return;
        }
        boolean isReady = this.scheduleWorkflowIfReady(workflow, workflowCfg, workflowCtx, this._clusterDataCache.getTaskDataCache());
        if (isReady) {
            this.scheduleJobs(workflow, workflowCfg, workflowCtx, this._clusterDataCache.getJobConfigMap(), this._clusterDataCache, currentStateOutput, bestPossibleOutput);
        } else {
            LOG.debug("Workflow {} is not ready to be scheduled.", (Object)workflow);
        }
        this._clusterDataCache.updateWorkflowContext(workflow, workflowCtx);
    }

    public WorkflowContext getOrInitializeWorkflowContext(String workflowName, TaskDataCache cache) {
        WorkflowContext workflowCtx = cache.getWorkflowContext(workflowName);
        if (workflowCtx == null) {
            if (cache.getWorkflowConfig(workflowName) != null) {
                workflowCtx = new WorkflowContext(new ZNRecord("WorkflowContext"));
                workflowCtx.setStartTime(System.currentTimeMillis());
                workflowCtx.setName(workflowName);
                LOG.debug("Workflow context is created for " + workflowName);
            } else {
                LOG.error("Workflow context is not created for {}. Workflow config is missing!", (Object)workflowName);
            }
        }
        return workflowCtx;
    }

    private void scheduleJobs(String workflow, WorkflowConfig workflowCfg, WorkflowContext workflowCtx, Map<String, JobConfig> jobConfigMap, WorkflowControllerDataProvider clusterDataCache, CurrentStateOutput currentStateOutput, BestPossibleStateOutput bestPossibleOutput) {
        long currentScheduledTime;
        ScheduleConfig scheduleConfig = workflowCfg.getScheduleConfig();
        if (scheduleConfig != null && scheduleConfig.isRecurring()) {
            LOG.debug("Jobs from recurring workflow {} are not schedule-able", (Object)workflow);
            return;
        }
        int inCompleteAllJobCount = TaskUtil.getInCompleteJobCount(workflowCfg, workflowCtx);
        int scheduledJobs = 0;
        long timeToSchedule = Long.MAX_VALUE;
        JobDag jobDag = clusterDataCache.getTaskDataCache().getRuntimeJobDag(workflow);
        if (jobDag == null) {
            jobDag = workflowCfg.getJobDag();
        }
        String nextJob = jobDag.getNextJob();
        while (nextJob != null) {
            String job = nextJob;
            TaskState jobState = workflowCtx.getJobState(job);
            if (jobState != null && !jobState.equals((Object)TaskState.NOT_STARTED)) {
                LOG.debug("Job {} is already started or completed.", (Object)job);
                this.processJob(job, currentStateOutput, bestPossibleOutput, workflowCtx);
                nextJob = jobDag.getNextJob();
                continue;
            }
            if (workflowCfg.isJobQueue() && scheduledJobs >= workflowCfg.getParallelJobs()) {
                LOG.debug("Workflow {} already have enough job in progress, scheduledJobs(s)={}, stop scheduling more jobs", (Object)workflow, (Object)scheduledJobs);
                break;
            }
            if (this.isJobReadyToSchedule(job, workflowCfg, workflowCtx, inCompleteAllJobCount, jobConfigMap, clusterDataCache, clusterDataCache.getAssignableInstanceManager())) {
                JobConfig jobConfig = jobConfigMap.get(job);
                long calculatedStartTime = this.computeStartTimeForJob(workflowCtx, job, jobConfig);
                if (System.currentTimeMillis() >= workflowCtx.getJobStartTime(job)) {
                    workflowCtx.setJobState(job, TaskState.NOT_STARTED);
                    this.processJob(job, currentStateOutput, bestPossibleOutput, workflowCtx);
                    ++scheduledJobs;
                } else {
                    timeToSchedule = Math.min(timeToSchedule, calculatedStartTime);
                }
            }
            nextJob = jobDag.getNextJob();
        }
        long l = currentScheduledTime = _rebalanceScheduler.getRebalanceTime(workflow) == -1L ? Long.MAX_VALUE : _rebalanceScheduler.getRebalanceTime(workflow);
        if (timeToSchedule < currentScheduledTime) {
            _rebalanceScheduler.scheduleRebalance(this._manager, workflow, timeToSchedule);
        }
    }

    private void processJob(String job, CurrentStateOutput currentStateOutput, BestPossibleStateOutput bestPossibleOutput, WorkflowContext workflowCtx) {
        this._clusterDataCache.getTaskDataCache().dispatchJob(job);
        try {
            ResourceAssignment resourceAssignment = this._jobDispatcher.processJobStatusUpdateAndAssignment(job, currentStateOutput, workflowCtx);
            this.updateBestPossibleStateOutput(job, resourceAssignment, bestPossibleOutput);
        }
        catch (Exception e) {
            LogUtil.logWarn(LOG, this._clusterDataCache.getClusterEventId(), String.format("Failed to compute job assignment for job %s", job), e);
        }
    }

    public void processJobForDrop(String resourceName, CurrentStateOutput currentStateOutput, BestPossibleStateOutput bestPossibleStateOutput) {
        JobConfig jobConfig = this._clusterDataCache.getJobConfig(resourceName);
        if (jobConfig == null || this._clusterDataCache.getWorkflowConfig(jobConfig.getWorkflow()) == null || this._clusterDataCache.getWorkflowContext(jobConfig.getWorkflow()) == null) {
            ResourceAssignment emptyAssignment = this.buildEmptyAssignment(resourceName, currentStateOutput);
            this.updateBestPossibleStateOutput(resourceName, emptyAssignment, bestPossibleStateOutput);
        }
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    private boolean scheduleWorkflowIfReady(String workflow, WorkflowConfig workflowCfg, WorkflowContext workflowCtx, TaskDataCache cache) {
        if (workflowCfg == null || workflowCfg.getScheduleConfig() == null) {
            return true;
        }
        ScheduleConfig scheduleConfig = workflowCfg.getScheduleConfig();
        Date startTime = scheduleConfig.getStartTime();
        long currentTime = new Date().getTime();
        long delayFromStart = startTime.getTime() - currentTime;
        if (delayFromStart <= 0L) {
            if (scheduleConfig.isRecurring()) {
                WorkflowContext lastWorkflowCtx;
                if (!workflowCfg.getTargetState().equals((Object)TargetState.START)) {
                    LOG.debug("Skip scheduling since the workflow {} has not been started", (Object)workflow);
                    return false;
                }
                String lastScheduled = workflowCtx.getLastScheduledSingleWorkflow();
                if (lastScheduled != null && (lastWorkflowCtx = cache.getWorkflowContext(lastScheduled)) != null && lastWorkflowCtx.getFinishTime() == -1L) {
                    LOG.info("Skip scheduling workflow {} since last schedule {} has not completed yet.", (Object)workflow, (Object)lastScheduled);
                    return false;
                }
                long period = scheduleConfig.getRecurrenceUnit().toMillis(scheduleConfig.getRecurrenceInterval());
                long offsetMultiplier = -delayFromStart / period;
                long timeToSchedule = period * offsetMultiplier + startTime.getTime();
                SimpleDateFormat df = new SimpleDateFormat("yyyyMMdd'T'HHmmss");
                df.setTimeZone(TimeZone.getTimeZone("UTC"));
                String newWorkflowName = workflow + "_" + df.format(new Date(timeToSchedule));
                LOG.debug("Ready to start workflow {}", (Object)newWorkflowName);
                if (!newWorkflowName.equals(lastScheduled)) {
                    Workflow clonedWf = WorkflowDispatcher.cloneWorkflow(this._manager, workflow, newWorkflowName, new Date(timeToSchedule));
                    TaskDriver driver = new TaskDriver(this._manager);
                    if (clonedWf != null) {
                        try {
                            driver.start(clonedWf);
                        }
                        catch (Exception e) {
                            LOG.error("Failed to schedule cloned workflow {}. ", (Object)newWorkflowName, (Object)e);
                            this._clusterStatusMonitor.updateWorkflowCounters(clonedWf.getWorkflowConfig(), TaskState.FAILED);
                        }
                    }
                    workflowCtx.setLastScheduledSingleWorkflow(newWorkflowName);
                }
                _rebalanceScheduler.scheduleRebalance(this._manager, workflow, timeToSchedule + period);
                return false;
            }
            long scheduledTime = _rebalanceScheduler.getRebalanceTime(workflow);
            if (scheduledTime <= 0L || currentTime <= scheduledTime) return true;
            _rebalanceScheduler.removeScheduledRebalance(workflow);
            return true;
        }
        _rebalanceScheduler.scheduleRebalance(this._manager, workflow, startTime.getTime());
        return false;
    }

    public static Workflow cloneWorkflow(HelixManager manager, String origWorkflowName, String newWorkflowName, Date newStartTime) {
        PropertyKey.Builder keyBuilder;
        HelixDataAccessor accessor = manager.getHelixDataAccessor();
        Map resourceConfigMap = accessor.getChildValuesMap((keyBuilder = accessor.keyBuilder()).resourceConfigs(), true);
        if (!resourceConfigMap.containsKey(origWorkflowName)) {
            LOG.error("No such workflow named {}", (Object)origWorkflowName);
            return null;
        }
        if (resourceConfigMap.containsKey(newWorkflowName)) {
            LOG.error("Workflow with name {} already exists!", (Object)newWorkflowName);
            return null;
        }
        Map<String, String> workflowConfigsMap = ((HelixProperty)resourceConfigMap.get(origWorkflowName)).getRecord().getSimpleFields();
        WorkflowConfig.Builder workflowConfigBlder = WorkflowConfig.Builder.fromMap(workflowConfigsMap);
        if (newStartTime != null) {
            ScheduleConfig scheduleConfig = ScheduleConfig.oneTimeDelayedStart(newStartTime);
            workflowConfigBlder.setScheduleConfig(scheduleConfig);
        }
        workflowConfigBlder.setTerminable(true);
        WorkflowConfig workflowConfig = workflowConfigBlder.build();
        JobDag jobDag = workflowConfig.getJobDag();
        Map<String, Set<String>> parentsToChildren = jobDag.getParentsToChildren();
        Workflow.Builder workflowBuilder = new Workflow.Builder(newWorkflowName);
        workflowBuilder.setWorkflowConfig(workflowConfig);
        Set<String> namespacedJobs = jobDag.getAllNodes();
        for (String namespacedJob : namespacedJobs) {
            if (!resourceConfigMap.containsKey(namespacedJob)) continue;
            String job = TaskUtil.getDenamespacedJobName(origWorkflowName, namespacedJob);
            HelixProperty jobConfig = (HelixProperty)resourceConfigMap.get(namespacedJob);
            Map<String, String> jobSimpleFields = jobConfig.getRecord().getSimpleFields();
            JobConfig.Builder jobCfgBuilder = JobConfig.Builder.fromMap(jobSimpleFields);
            jobCfgBuilder.setWorkflow(newWorkflowName);
            Map<String, Map<String, String>> rawTaskConfigMap = jobConfig.getRecord().getMapFields();
            LinkedList taskConfigs = Lists.newLinkedList();
            for (Map<String, String> rawTaskConfig : rawTaskConfigMap.values()) {
                TaskConfig taskConfig = TaskConfig.Builder.from(rawTaskConfig);
                taskConfigs.add(taskConfig);
            }
            jobCfgBuilder.addTaskConfigs(taskConfigs);
            workflowBuilder.addJob(job, jobCfgBuilder);
            Set<String> children = parentsToChildren.get(namespacedJob);
            if (children == null) continue;
            for (String namespacedChild : children) {
                String child = TaskUtil.getDenamespacedJobName(origWorkflowName, namespacedChild);
                workflowBuilder.addParentChildDependency(job, child);
            }
        }
        return workflowBuilder.build();
    }

    private void cleanupWorkflow(String workflow) {
        LOG.info("Cleaning up workflow: " + workflow);
        WorkflowConfig workflowcfg = this._clusterDataCache.getWorkflowConfig(workflow);
        if (workflowcfg.isTerminable() || workflowcfg.getTargetState() == TargetState.DELETE) {
            Set<String> jobs = workflowcfg.getJobDag().getAllNodes();
            _rebalanceScheduler.removeScheduledRebalance(workflow);
            for (String job : jobs) {
                _rebalanceScheduler.removeScheduledRebalance(job);
            }
            if (!TaskUtil.removeWorkflow(this._manager.getHelixDataAccessor(), this._manager.getHelixPropertyStore(), workflow, jobs)) {
                LOG.warn("Failed to clean up workflow {}", (Object)workflow);
            } else {
                this.removeContexts(workflow, jobs, this._clusterDataCache.getTaskDataCache());
            }
        } else {
            LOG.info("Did not clean up workflow {} because neither the workflow is non-terminable nor is set to DELETE.", (Object)workflow);
        }
    }

    private void removeContexts(String workflow, Set<String> jobs, TaskDataCache cache) {
        if (jobs != null) {
            for (String job : jobs) {
                cache.removeContext(job);
            }
        }
        cache.removeContext(workflow);
    }

    private long computeStartTimeForJob(WorkflowContext workflowCtx, String job, JobConfig jobConfig) {
        long calculatedStartTime = workflowCtx.getJobStartTime(job);
        if (calculatedStartTime < 0L) {
            calculatedStartTime = System.currentTimeMillis();
            if (jobConfig.getExecutionDelay() >= 0L) {
                calculatedStartTime += jobConfig.getExecutionDelay();
            }
            calculatedStartTime = Math.max(calculatedStartTime, jobConfig.getExecutionStart());
            workflowCtx.setJobStartTime(job, calculatedStartTime);
        }
        return calculatedStartTime;
    }
}

