/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.jackrabbit.ocm.manager.query;

import java.util.ArrayList;
import java.util.Collection;

import javax.jcr.Node;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import junit.framework.Test;
import junit.framework.TestSuite;

import org.apache.jackrabbit.ocm.AnnotationRepositoryTestBase;
import org.apache.jackrabbit.ocm.exception.JcrMappingException;
import org.apache.jackrabbit.ocm.manager.ObjectContentManager;
import org.apache.jackrabbit.ocm.manager.impl.ObjectIterator;
import org.apache.jackrabbit.ocm.query.Filter;
import org.apache.jackrabbit.ocm.query.Query;
import org.apache.jackrabbit.ocm.query.QueryManager;
import org.apache.jackrabbit.ocm.testmodel.Page;
import org.apache.jackrabbit.ocm.testmodel.Paragraph;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


/**
 * Test Object Iterator
 *
 * @author <a href="mailto:christophe.lombart@sword-technologies.com">Christophe Lombart</a>
 */
public class AnnotationIteratorQueryTest extends AnnotationRepositoryTestBase
{
    private final static Logger log = LoggerFactory.getLogger(AnnotationIteratorQueryTest.class);


    public static Test suite()
    {
        // All methods starting with "test" will be executed in the test suite.
        return new TestSuite(AnnotationIteratorQueryTest.class);
    }

    /**
     * @see junit.framework.TestCase#setUp()
     */
    protected void setUp() throws Exception
    {
        super.setUp();
		this.importData();

    }

    /**
     * Test equalTo
     *
     */
    public void testsetScope()
    {

    	try
    	{
    		
    		  // No scope    		
    	      QueryManager queryManager = getObjectContentManager().getQueryManager();
    	      Filter filter = queryManager.createFilter(Page.class);        	
    	      Query query = queryManager.createQuery(filter);
              // we need this to be sure about objects order in the iterator test
              query.addOrderByAscending("title");
    	      ObjectContentManager ocm = this.getObjectContentManager();
    	
    	      long  start = System.currentTimeMillis();
              ObjectIterator iterator = (ObjectIterator) ocm.getObjectIterator(query);
              log.info("getObjectIterator takes : " + (System.currentTimeMillis() - start));

    	      start = System.currentTimeMillis();
    	      Collection result = ocm.getObjects(query);
    	      log.info("getObject takes : " + (System.currentTimeMillis() - start));
    	

    	      start = System.currentTimeMillis();
    	      iterator = (ObjectIterator) ocm.getObjectIterator(query);
              log.info("getObjectIterator takes : " + (System.currentTimeMillis() - start));
              assertEquals("wrong results count", 4, iterator.getSize());
              assertEquals("wrong initial position", 0, iterator.getPosition());
              assertTrue("no objects returned", iterator.hasNext());
              Page page = (Page) iterator.next();
              assertEquals("wrong first result", "Page 1 Title", page.getTitle());
              iterator.skip(1);
              assertEquals("wrong position after skip", 2, iterator.getPosition());
              page = (Page) iterator.next();
              assertEquals("wrong third result", "Page 3 Title", page.getTitle());
        }
        catch (Exception e)
        {
            e.printStackTrace();
            fail("Exception occurs during the unit test : " + e);
        }

    }


    private void importData() throws JcrMappingException
    {

    	try
		{
    		ObjectContentManager ocm = getObjectContentManager();
    		
        	if (ocm.objectExists("/test"))
            {
                ocm.remove("/test");
            }

			Session session = ocm.getSession();
			Node root = session.getRootNode();
			root.addNode("test");
			root.addNode("test/node1");
			root.addNode("test/node2");

            session.save();
			
			Page page = new Page();
			page.setPath("/test/node1/page1");
			page.setTitle("Page 1 Title");
			
			ArrayList paragraphs = new ArrayList();
			
			paragraphs.add(new Paragraph("Para 1"));
			paragraphs.add(new Paragraph("Para 2"));
			paragraphs.add(new Paragraph("Para 3"));
			paragraphs.add(new Paragraph("Another Para "));
			page.setParagraphs(paragraphs);
			
			ocm.insert(page);
						
			
			page = new Page();
			page.setPath("/test/node1/page2");
			page.setTitle("Page 2 Title");
			
			paragraphs = new ArrayList();
			
			paragraphs.add(new Paragraph("Para 1"));
			paragraphs.add(new Paragraph("Para 2"));
			paragraphs.add(new Paragraph("Para 5"));
			paragraphs.add(new Paragraph("Another Para"));
			page.setParagraphs(paragraphs);
			
			ocm.insert(page);
			
			page = new Page();
			page.setPath("/test/node2/page1");
			page.setTitle("Page 3 Title");
			
			paragraphs = new ArrayList();
			
			paragraphs.add(new Paragraph("Para 1"));
			paragraphs.add(new Paragraph("Para 4"));
			paragraphs.add(new Paragraph("Para 5"));
			paragraphs.add(new Paragraph("Another Para"));
			page.setParagraphs(paragraphs);
			
			ocm.insert(page);

			page = new Page();
			page.setPath("/test/node2/page2");
			page.setTitle("Page 4 Title");
			
			paragraphs = new ArrayList();
			
			paragraphs.add(new Paragraph("Para 6"));
			paragraphs.add(new Paragraph("Para 7"));
			paragraphs.add(new Paragraph("Para 8"));
			paragraphs.add(new Paragraph("Another Para"));
			page.setParagraphs(paragraphs);
			
			ocm.insert(page);
			
			ocm.save();
			

			
		}
		catch (RepositoryException e)
		{
			
			e.printStackTrace();
		}


    }
}