/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jena.geosparql.implementation.parsers.gml;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.lang.invoke.MethodHandles;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import org.apache.jena.datatypes.DatatypeFormatException;
import org.apache.jena.geosparql.implementation.DimensionInfo;
import org.apache.jena.geosparql.implementation.SRSInfo;
import org.apache.jena.geosparql.implementation.SRSInfoException;
import org.apache.jena.geosparql.implementation.UnitsOfMeasure;
import org.apache.jena.geosparql.implementation.jts.CoordinateSequenceDimensions;
import org.apache.jena.geosparql.implementation.jts.CustomCoordinateSequence;
import org.apache.jena.geosparql.implementation.jts.CustomGeometryFactory;
import org.apache.jena.geosparql.implementation.parsers.ParserReader;
import org.jdom2.Document;
import org.jdom2.Element;
import org.jdom2.JDOMException;
import org.jdom2.Namespace;
import org.jdom2.input.SAXBuilder;
import org.locationtech.jts.geom.Coordinate;
import org.locationtech.jts.geom.CoordinateXY;
import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.geom.GeometryFactory;
import org.locationtech.jts.geom.LineSegment;
import org.locationtech.jts.geom.LineString;
import org.locationtech.jts.geom.LinearRing;
import org.locationtech.jts.geom.Point;
import org.locationtech.jts.geom.Polygon;
import org.locationtech.jts.operation.union.CascadedPolygonUnion;
import org.locationtech.jts.util.GeometricShapeFactory;
import org.opengis.referencing.crs.CoordinateReferenceSystem;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class GMLReader
implements ParserReader {
    private static final Logger LOGGER = LoggerFactory.getLogger(MethodHandles.lookup().lookupClass());
    private static final GeometryFactory GEOMETRY_FACTORY = CustomGeometryFactory.theInstance();
    private final Geometry geometry;
    private final String srsURI;
    private final CoordinateSequenceDimensions dims;
    private final DimensionInfo dimensionInfo;
    private static final Namespace GML_NAMESPACE = Namespace.getNamespace("gml", "http://www.opengis.net/gml/3.2");
    private static Boolean isSRSNameWarningIssued = false;
    private static final String EMPTY_GML_TEXT = "<gml:Point xmlns:gml='http://www.opengis.net/gml/3.2' srsName=\"http://www.opengis.net/def/crs/OGC/1.3/CRS84\" />";

    protected GMLReader(Element gmlElement) throws DatatypeFormatException, SRSInfoException {
        this.srsURI = GMLReader.getSrsURI(gmlElement);
        SRSInfo srsInfo = new SRSInfo(this.srsURI);
        CoordinateReferenceSystem crs = srsInfo.getCrs();
        int srsDimension = crs.getCoordinateSystem().getDimension();
        this.dims = CoordinateSequenceDimensions.find(srsDimension);
        String geometryType = gmlElement.getName();
        this.geometry = GMLReader.buildGeometry(geometryType, gmlElement, this.dims, srsInfo);
        this.dimensionInfo = new DimensionInfo(this.dims, this.geometry.getDimension());
    }

    protected GMLReader(Geometry geometry, int srsDimension, String srsURI) {
        this.srsURI = srsURI;
        this.geometry = geometry;
        this.dims = CoordinateSequenceDimensions.find(srsDimension);
        this.dimensionInfo = new DimensionInfo(this.dims, geometry.getDimension());
    }

    protected GMLReader(Geometry geometry, int srsDimension) {
        this(geometry, srsDimension, "http://www.opengis.net/def/crs/OGC/1.3/CRS84");
    }

    @Override
    public Geometry getGeometry() {
        return this.geometry;
    }

    @Override
    public String getSrsURI() {
        return this.srsURI;
    }

    @Override
    public CoordinateSequenceDimensions getDimensions() {
        return this.dims;
    }

    @Override
    public DimensionInfo getDimensionInfo() {
        return this.dimensionInfo;
    }

    private static String getSrsURI(Element gmlElement) {
        String srsNameURI = gmlElement.getAttributeValue("srsName");
        if (srsNameURI == null) {
            srsNameURI = "http://www.opengis.net/def/crs/OGC/1.3/CRS84";
            if (!isSRSNameWarningIssued.booleanValue()) {
                LOGGER.warn("GML Literal with no srsName. Defaulting to CRS84 {} used as WKT default SRS. This warning will be issued once.", (Object)srsNameURI);
                isSRSNameWarningIssued = true;
            }
        }
        return srsNameURI;
    }

    private static Geometry buildGeometry(String shape, Element gmlElement, CoordinateSequenceDimensions dims, SRSInfo srsInfo) throws DatatypeFormatException {
        Geometry geo;
        try {
            switch (shape) {
                case "Point": {
                    geo = GMLReader.buildPoint(gmlElement, dims);
                    break;
                }
                case "LineString": {
                    geo = GMLReader.buildLineString(gmlElement, dims);
                    break;
                }
                case "Curve": {
                    geo = GMLReader.buildCurve(gmlElement, dims, srsInfo);
                    break;
                }
                case "Polygon": {
                    geo = GMLReader.buildPolygon(gmlElement, dims);
                    break;
                }
                case "Surface": {
                    geo = GMLReader.buildSurface(gmlElement, dims, srsInfo);
                    break;
                }
                case "MultiPoint": {
                    geo = GMLReader.buildMultiPoint(gmlElement, dims);
                    break;
                }
                case "MultiCurve": {
                    geo = GMLReader.buildMultiCurve(gmlElement, dims, srsInfo);
                    break;
                }
                case "MultiSurface": {
                    geo = GMLReader.buildMultiSurface(gmlElement, dims, srsInfo);
                    break;
                }
                case "MultiGeometry": {
                    geo = GMLReader.buildMultiGeometry(gmlElement, dims, srsInfo);
                    break;
                }
                default: {
                    throw new DatatypeFormatException("Geometry shape not supported ([10-100r3], page 22): " + shape);
                }
            }
        }
        catch (ArrayIndexOutOfBoundsException ex) {
            throw new DatatypeFormatException("Build GML Geometry Exception - Shape: " + shape + ", Element: " + String.valueOf(gmlElement) + ". " + ex.getMessage());
        }
        return geo;
    }

    private static CustomCoordinateSequence extractPos(Element gmlElement, CoordinateSequenceDimensions dims) {
        String coordinates = gmlElement.getChildTextNormalize("pos", GML_NAMESPACE);
        if (coordinates == null) {
            coordinates = "";
        }
        return new CustomCoordinateSequence(dims, coordinates);
    }

    private static String extractPosList(Element gmlElement, int srsDimension) {
        String posList = gmlElement.getChildTextNormalize("posList", GML_NAMESPACE);
        if (posList == null) {
            return "";
        }
        StringBuilder sb = new StringBuilder("");
        String[] coordinates = posList.trim().split(" ");
        int mod = coordinates.length % srsDimension;
        if (mod != 0) {
            throw new DatatypeFormatException("GML Pos List does not divide into srs dimension: " + coordinates.length + " divide " + srsDimension + " remainder " + mod + ".");
        }
        int finalCoordinate = coordinates.length - 1;
        for (int i = 0; i < coordinates.length; ++i) {
            if (i != 0 & i % srsDimension == 0) {
                sb.append(",");
            }
            String coordinate = coordinates[i];
            sb.append(coordinate);
            if (i == finalCoordinate) continue;
            sb.append(" ");
        }
        return sb.toString();
    }

    private static List<Coordinate> buildCoordinateList(Element gmlElement, int srsDimension) {
        String posList = gmlElement.getChildTextNormalize("posList", GML_NAMESPACE);
        String[] coordinates = posList.trim().split(" ");
        int mod = coordinates.length % srsDimension;
        if (mod != 0) {
            throw new DatatypeFormatException("GML Pos List does not divide into srs dimension: " + coordinates.length + " divide " + srsDimension + " remainder " + mod + ".");
        }
        ArrayList<Coordinate> coordinateList = new ArrayList<Coordinate>();
        for (int i = 0; i < coordinates.length; i += srsDimension) {
            coordinateList.add(switch (srsDimension) {
                case 2 -> new CoordinateXY(Double.parseDouble(coordinates[i]), Double.parseDouble(coordinates[i + 1]));
                case 3 -> new Coordinate(Double.parseDouble(coordinates[i]), Double.parseDouble(coordinates[i + 1]), Double.parseDouble(coordinates[i + 2]));
                default -> throw new DatatypeFormatException("SRS dimension " + srsDimension + " is not supported.");
            });
        }
        return coordinateList;
    }

    private static Point buildPoint(Element gmlElement, CoordinateSequenceDimensions dims) {
        CustomCoordinateSequence coordinateSequence = GMLReader.extractPos(gmlElement, dims);
        return GEOMETRY_FACTORY.createPoint(coordinateSequence);
    }

    private static LineString buildLineString(Element gmlElement, CoordinateSequenceDimensions dims) {
        int srsDimension = CoordinateSequenceDimensions.convertToInt(dims);
        String posList = GMLReader.extractPosList(gmlElement, srsDimension);
        CustomCoordinateSequence coordinateSequence = new CustomCoordinateSequence(dims, posList);
        return GEOMETRY_FACTORY.createLineString(coordinateSequence);
    }

    private static LineString buildCurve(Element gmlElement, CoordinateSequenceDimensions dims, SRSInfo srsInfo) {
        Element segmentsElement = gmlElement.getChild("segments", GML_NAMESPACE);
        if (segmentsElement == null) {
            return GEOMETRY_FACTORY.createLineString();
        }
        List<Element> segments = segmentsElement.getChildren("LineStringSegment", GML_NAMESPACE);
        if (!segments.isEmpty()) {
            return GMLReader.buildLineStringSegments(segments, dims);
        }
        segments = segmentsElement.getChildren("Arc", GML_NAMESPACE);
        if (!segments.isEmpty()) {
            return GMLReader.buildArc(segments, dims);
        }
        segments = segmentsElement.getChildren("Circle", GML_NAMESPACE);
        if (!segments.isEmpty()) {
            return GMLReader.buildCircle(segments, dims);
        }
        segments = segmentsElement.getChildren("CircleByCenterPoint", GML_NAMESPACE);
        if (!segments.isEmpty()) {
            return GMLReader.buildCircleByCentrePoint(segments, dims, srsInfo);
        }
        throw new DatatypeFormatException("GML Curve segments not supported or empty: " + String.valueOf(gmlElement));
    }

    private static LineString buildLineStringSegments(List<Element> segments, CoordinateSequenceDimensions dims) {
        int srsDimension = CoordinateSequenceDimensions.convertToInt(dims);
        List<Coordinate> lineStringCoords = new ArrayList<Coordinate>();
        for (Element segment : segments) {
            List<Coordinate> segmentCoords = GMLReader.buildCoordinateList(segment, srsDimension);
            if (lineStringCoords.isEmpty()) {
                lineStringCoords = segmentCoords;
                continue;
            }
            if (segmentCoords.isEmpty()) continue;
            Coordinate lastCoord = lineStringCoords.get(lineStringCoords.size() - 1);
            Coordinate firstCoord = segmentCoords.get(0);
            if (!firstCoord.equals2D(lastCoord)) {
                throw new DatatypeFormatException("GML LineString segments do not have matching last and first coordinates: " + String.valueOf(lineStringCoords) + " - " + String.valueOf(segmentCoords));
            }
            segmentCoords.remove(0);
            lineStringCoords.addAll(segmentCoords);
        }
        CustomCoordinateSequence coordinateSequence = new CustomCoordinateSequence(dims, lineStringCoords);
        return GEOMETRY_FACTORY.createLineString(coordinateSequence);
    }

    private static final LineString buildArc(List<Element> segments, CoordinateSequenceDimensions dims) {
        int srsDimension = CoordinateSequenceDimensions.convertToInt(dims);
        Element posListElement = segments.get(0);
        List<Coordinate> coordinates = GMLReader.buildCoordinateList(posListElement, srsDimension);
        if (coordinates.size() != 3) {
            throw new DatatypeFormatException("GML Arc posList does not contain 3 coordinates: " + String.valueOf(posListElement));
        }
        Coordinate centre = GMLReader.findCentre(coordinates);
        Coordinate coord0 = coordinates.get(0);
        double radius = Math.hypot(centre.x - coord0.x, centre.y - coord0.y);
        GeometricShapeFactory geometricShapeFactory = new GeometricShapeFactory(GEOMETRY_FACTORY);
        geometricShapeFactory.setCentre(centre);
        geometricShapeFactory.setWidth(radius * 2.0);
        double startAng = GMLReader.findAngle(centre, coord0);
        double angExtent = GMLReader.findAngle(centre, coordinates.get(2));
        return geometricShapeFactory.createArc(startAng, angExtent);
    }

    protected static Coordinate findCentre(List<Coordinate> coordinates) {
        if (coordinates.size() < 3) {
            throw new DatatypeFormatException("GML posList does not contain 3 coordinates: " + String.valueOf(coordinates));
        }
        double x1 = coordinates.get((int)0).x;
        double y1 = coordinates.get((int)0).y;
        double x2 = coordinates.get((int)1).x;
        double y2 = coordinates.get((int)1).y;
        double x3 = coordinates.get((int)2).x;
        double y3 = coordinates.get((int)2).y;
        double A = x1 * (y2 - y3) - y1 * (x2 - x3) + x2 * y3 - x3 * y2;
        double B = (x1 * x1 + y1 * y1) * (y3 - y2) + (x2 * x2 + y2 * y2) * (y1 - y3) + (x3 * x3 + y3 * y3) * (y2 - y1);
        double C2 = (x1 * x1 + y1 * y1) * (x2 - x3) + (x2 * x2 + y2 * y2) * (x3 - x1) + (x3 * x3 + y3 * y3) * (x1 - x2);
        double x = -(B / (2.0 * A));
        double y = -(C2 / (2.0 * A));
        Coordinate centre = new Coordinate(x, y);
        return centre;
    }

    protected static double findAngle(Coordinate coord0, Coordinate coord1) {
        LineSegment line = new LineSegment(coord0, coord1);
        double angle = line.angle();
        return angle;
    }

    private static LineString buildCircle(List<Element> segments, CoordinateSequenceDimensions dims) {
        int srsDimension = CoordinateSequenceDimensions.convertToInt(dims);
        Element posListElement = segments.get(0);
        List<Coordinate> coordinates = GMLReader.buildCoordinateList(posListElement, srsDimension);
        if (coordinates.size() != 3) {
            throw new DatatypeFormatException("GML Circle posList does not contain 3 coordinates: " + String.valueOf(posListElement));
        }
        Coordinate centre = GMLReader.findCentre(coordinates);
        Coordinate coord = coordinates.get(0);
        double radius = Math.hypot(centre.x - coord.x, centre.y - coord.y);
        GeometricShapeFactory geometricShapeFactory = new GeometricShapeFactory(GEOMETRY_FACTORY);
        geometricShapeFactory.setCentre(centre);
        geometricShapeFactory.setWidth(radius * 2.0);
        Polygon circlePolygon = geometricShapeFactory.createCircle();
        return circlePolygon.getExteriorRing();
    }

    private static LineString buildCircleByCentrePoint(List<Element> segments, CoordinateSequenceDimensions dims, SRSInfo srsInfo) {
        Element circleElement = segments.get(0);
        Point point = GMLReader.buildPoint(circleElement, dims);
        Coordinate centre = point.getCoordinate();
        Element radiusElement = circleElement.getChild("radius", GML_NAMESPACE);
        double radius = Double.parseDouble(radiusElement.getTextNormalize());
        String uomURI = radiusElement.getAttributeValue("uom");
        if (uomURI != null) {
            UnitsOfMeasure uom = new UnitsOfMeasure(uomURI);
            radius = UnitsOfMeasure.conversion(radius, uom, srsInfo.getUnitsOfMeasure());
        }
        GeometricShapeFactory geometricShapeFactory = new GeometricShapeFactory(GEOMETRY_FACTORY);
        geometricShapeFactory.setCentre(centre);
        geometricShapeFactory.setSize(radius * 2.0);
        Polygon circlePolygon = geometricShapeFactory.createCircle();
        return circlePolygon.getExteriorRing();
    }

    private static Polygon buildPolygon(Element gmlElement, CoordinateSequenceDimensions dims) {
        Polygon polygon;
        LinearRing exteriorLinearRing;
        Element exteriorElement = gmlElement.getChild("exterior", GML_NAMESPACE);
        if (exteriorElement != null) {
            Element exteriorLinearRingElement = exteriorElement.getChild("LinearRing", GML_NAMESPACE);
            exteriorLinearRing = GMLReader.buildLinearRing(exteriorLinearRingElement, dims);
        } else {
            exteriorLinearRing = GEOMETRY_FACTORY.createLinearRing();
        }
        List<Element> interiorElements = gmlElement.getChildren("interior", GML_NAMESPACE);
        ArrayList<LinearRing> interiorList = new ArrayList<LinearRing>();
        for (Element interiorElement : interiorElements) {
            Element interiorLinearRingElement = interiorElement.getChild("LinearRing", GML_NAMESPACE);
            LinearRing linearRing = GMLReader.buildLinearRing(interiorLinearRingElement, dims);
            interiorList.add(linearRing);
        }
        if (interiorList.isEmpty()) {
            polygon = GEOMETRY_FACTORY.createPolygon(exteriorLinearRing);
        } else {
            LinearRing[] interiorLinearRings = interiorList.toArray(new LinearRing[interiorList.size()]);
            polygon = GEOMETRY_FACTORY.createPolygon(exteriorLinearRing, interiorLinearRings);
        }
        return polygon;
    }

    private static LinearRing buildLinearRing(Element gmlElement, CoordinateSequenceDimensions dims) {
        int srsDimension = CoordinateSequenceDimensions.convertToInt(dims);
        String posList = GMLReader.extractPosList(gmlElement, srsDimension);
        CustomCoordinateSequence sequence = new CustomCoordinateSequence(dims, posList);
        LinearRing linearRing = GEOMETRY_FACTORY.createLinearRing(sequence);
        return linearRing;
    }

    private static Polygon buildSurface(Element gmlElement, CoordinateSequenceDimensions dims, SRSInfo srsInfo) {
        Element patchesElement = gmlElement.getChild("patches", GML_NAMESPACE);
        if (patchesElement == null) {
            throw new DatatypeFormatException("GML Surface does not contain patches child: " + String.valueOf(gmlElement));
        }
        List<Element> polygonPatches = patchesElement.getChildren("PolygonPatch", GML_NAMESPACE);
        if (polygonPatches.isEmpty()) {
            throw new DatatypeFormatException("GML Surface does not contain PolygonPatches. Only gml:PolygonPatches are supported in Simple Features profile ([10-100r3], page 22):" + String.valueOf(gmlElement));
        }
        ArrayList<Polygon> polys = new ArrayList<Polygon>();
        for (Element patch : polygonPatches) {
            Polygon polygon;
            LinearRing exteriorLinearRing;
            Element exteriorElement = patch.getChild("exterior", GML_NAMESPACE);
            if (exteriorElement != null) {
                LinearRing exteriorGeom = GMLReader.buildSurfacePatch(exteriorElement, dims, srsInfo);
                exteriorLinearRing = GEOMETRY_FACTORY.createLinearRing(((Geometry)exteriorGeom).getCoordinates());
            } else {
                exteriorLinearRing = GEOMETRY_FACTORY.createLinearRing();
            }
            List<Element> interiorElements = gmlElement.getChildren("interior", GML_NAMESPACE);
            ArrayList<LinearRing> interiorList = new ArrayList<LinearRing>();
            for (Element interiorElement : interiorElements) {
                LinearRing interiorGeom = GMLReader.buildSurfacePatch(interiorElement, dims, srsInfo);
                LinearRing interiorLinearRing = GEOMETRY_FACTORY.createLinearRing(((Geometry)interiorGeom).getCoordinates());
                interiorList.add(interiorLinearRing);
            }
            if (interiorList.isEmpty()) {
                polygon = GEOMETRY_FACTORY.createPolygon(exteriorLinearRing);
            } else {
                LinearRing[] interiorLinearRings = interiorList.toArray(new LinearRing[interiorList.size()]);
                polygon = GEOMETRY_FACTORY.createPolygon(exteriorLinearRing, interiorLinearRings);
            }
            polys.add(polygon);
        }
        Geometry unionGeom = CascadedPolygonUnion.union(polys);
        if (!(unionGeom instanceof Polygon)) {
            throw new AssertionError((Object)"CascadePolygonUnion has not produced a Polygon geometry in GML Reader.");
        }
        Polygon unionPolygon = (Polygon)unionGeom;
        return unionPolygon;
    }

    private static LinearRing buildSurfacePatch(Element gmlElement, CoordinateSequenceDimensions dims, SRSInfo srsInfo) {
        Element linearRingElement = gmlElement.getChild("LinearRing", GML_NAMESPACE);
        LinearRing linearRing = null;
        if (linearRingElement != null) {
            linearRing = GMLReader.buildLinearRing(linearRingElement, dims);
        } else {
            Element curveElement;
            Element curveMemberElement;
            Element ringElement = gmlElement.getChild("Ring", GML_NAMESPACE);
            if (ringElement == null && (curveMemberElement = gmlElement.getChild("curveMember", GML_NAMESPACE)) != null && (curveElement = curveMemberElement.getChild("Curve", GML_NAMESPACE)) != null) {
                LineString lineString = GMLReader.buildCurve(curveElement, dims, srsInfo);
                linearRing = GEOMETRY_FACTORY.createLinearRing(lineString.getCoordinateSequence());
            }
        }
        if (linearRing == null) {
            throw new DatatypeFormatException("GML Surface does not contain correct LinearRing or Ring elements ([10-100r3], page 22):" + String.valueOf(gmlElement));
        }
        return linearRing;
    }

    private static Geometry buildMultiPoint(Element gmlElement, CoordinateSequenceDimensions dims) {
        List<Element> memberElements = GMLReader.getMembers(gmlElement, "pointMember");
        ArrayList<Point> points = new ArrayList<Point>(memberElements.size());
        for (Element member : memberElements) {
            Point point = GMLReader.buildPoint(member, dims);
            points.add(point);
        }
        Point[] pointArray = points.toArray(new Point[points.size()]);
        return GEOMETRY_FACTORY.createMultiPoint(pointArray);
    }

    private static Geometry buildMultiCurve(Element gmlElement, CoordinateSequenceDimensions dims, SRSInfo srsInfo) {
        List<Element> memberElements = GMLReader.getMembers(gmlElement, "curveMember");
        ArrayList<LineString> lineStrings = new ArrayList<LineString>(memberElements.size());
        for (Element member : memberElements) {
            String shape;
            lineStrings.add(switch (shape = member.getName()) {
                case "LineString" -> GMLReader.buildLineString(member, dims);
                case "Curve" -> GMLReader.buildCurve(member, dims, srsInfo);
                default -> throw new DatatypeFormatException("GML MultiCurve does not contain LineString or Curve elements ([10-100r3], page 22):" + String.valueOf(gmlElement));
            });
        }
        LineString[] lineStringArray = lineStrings.toArray(new LineString[lineStrings.size()]);
        return GEOMETRY_FACTORY.createMultiLineString(lineStringArray);
    }

    private static Geometry buildMultiSurface(Element gmlElement, CoordinateSequenceDimensions dims, SRSInfo srsInfo) {
        List<Element> memberElements = GMLReader.getMembers(gmlElement, "surfaceMember");
        ArrayList<Polygon> polygons = new ArrayList<Polygon>(memberElements.size());
        for (Element member : memberElements) {
            String shape;
            polygons.add(switch (shape = member.getName()) {
                case "Polygon" -> GMLReader.buildPolygon(member, dims);
                case "Surface" -> GMLReader.buildSurface(member, dims, srsInfo);
                default -> throw new DatatypeFormatException("GML MultiSurface does not contain Polygon or Surface elements ([10-100r3], page 22):" + String.valueOf(gmlElement));
            });
        }
        Polygon[] polygonArray = polygons.toArray(new Polygon[polygons.size()]);
        return GEOMETRY_FACTORY.createMultiPolygon(polygonArray);
    }

    private static Geometry buildMultiGeometry(Element gmlElement, CoordinateSequenceDimensions dims, SRSInfo srsInfo) {
        List<Element> memberElements = GMLReader.getMembers(gmlElement, "geometryMember");
        ArrayList<Geometry> geometries = new ArrayList<Geometry>(memberElements.size());
        for (Element member : memberElements) {
            String shape = member.getName();
            Geometry geom = GMLReader.buildGeometry(shape, member, dims, srsInfo);
            geometries.add(geom);
        }
        Geometry[] geometryArray = geometries.toArray(new Geometry[geometries.size()]);
        return GEOMETRY_FACTORY.createGeometryCollection(geometryArray);
    }

    private static List<Element> getMembers(Element gmlElement, String memberLabel) {
        List<Element> memberElements;
        String membersLabel = memberLabel + "s";
        Element membersElement = gmlElement.getChild(membersLabel, GML_NAMESPACE);
        if (membersElement != null) {
            memberElements = membersElement.getChildren();
        } else {
            List<Element> memberElementList = gmlElement.getChildren(memberLabel, GML_NAMESPACE);
            memberElements = new ArrayList<Element>();
            for (Element memberElement : memberElementList) {
                List<Element> childElements = memberElement.getChildren();
                memberElements.addAll(childElements);
            }
        }
        return memberElements;
    }

    public static GMLReader extract(String gmlText) throws JDOMException, IOException {
        if (gmlText.isEmpty()) {
            gmlText = EMPTY_GML_TEXT;
        }
        SAXBuilder jdomBuilder = GMLReader.newSAXBuilder();
        ByteArrayInputStream stream = new ByteArrayInputStream(gmlText.getBytes(StandardCharsets.UTF_8));
        Document xmlDoc = jdomBuilder.build(stream);
        Element gmlElement = xmlDoc.getRootElement();
        return new GMLReader(gmlElement);
    }

    private static SAXBuilder newSAXBuilder() {
        SAXBuilder builder = new SAXBuilder();
        builder.setFeature("http://apache.org/xml/features/disallow-doctype-decl", true);
        builder.setFeature("http://xml.org/sax/features/external-general-entities", false);
        builder.setFeature("http://xml.org/sax/features/external-parameter-entities", false);
        builder.setExpandEntities(false);
        return builder;
    }

    public int hashCode() {
        int hash = 7;
        hash = 89 * hash + Objects.hashCode(this.geometry);
        hash = 89 * hash + Objects.hashCode(this.srsURI);
        hash = 89 * hash + Objects.hashCode((Object)this.dims);
        hash = 89 * hash + Objects.hashCode(this.dimensionInfo);
        return hash;
    }

    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (this.getClass() != obj.getClass()) {
            return false;
        }
        GMLReader other = (GMLReader)obj;
        if (!Objects.equals(this.srsURI, other.srsURI)) {
            return false;
        }
        if (!Objects.equals(this.geometry, other.geometry)) {
            return false;
        }
        if (this.dims != other.dims) {
            return false;
        }
        return Objects.equals(this.dimensionInfo, other.dimensionInfo);
    }

    public String toString() {
        return "GMLReader{geometry=" + String.valueOf(this.geometry) + ", srsURI=" + this.srsURI + ", dims=" + String.valueOf((Object)this.dims) + ", dimensionInfo=" + String.valueOf(this.dimensionInfo) + "}";
    }
}

