/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.myfaces.orchestra.conversation;

import org.apache.myfaces.orchestra.conversation.basic.LogConversationMessager;
import org.apache.myfaces.orchestra.conversation.model.UserData;
import org.apache.myfaces.orchestra.frameworkAdapter.FrameworkAdapter;
import org.apache.myfaces.orchestra.frameworkAdapter.local.LocalFrameworkAdapter;
import org.springframework.test.AbstractDependencyInjectionSpringContextTests;

/**
 * Test various aspects of the conversation handling
 */
public class TestConversationPersistence extends AbstractDependencyInjectionSpringContextTests
{
    protected String[] getConfigLocations()
    {
        return new String[]
            {
                "classpath:testApplicationContext.xml"
            };
    }

    protected void onSetUp() throws Exception
    {
        super.onSetUp();

        LocalFrameworkAdapter frameworkAdapter = new LocalFrameworkAdapter();
        frameworkAdapter.setApplicationContext(applicationContext);
        frameworkAdapter.setConversationMessager(new LogConversationMessager());
        FrameworkAdapter.setCurrentInstance(frameworkAdapter);
    }

    public void testPersistence()
    {
        final String BEAN_NAME = "persistentBackingBean";

        PersistentBackingBean conv = (PersistentBackingBean) applicationContext.getBean(BEAN_NAME);

        /* create and reread a user */
        UserData user = conv.createUser();
        UserData user2 = conv.readUser(user.getId());

        assertSame("has to be the same user",user , user2);

        conv.invalidateConversation();

        /* restart conversatin */
        UserData user3 = conv.readUser(user.getId());

        assertNotSame("should not be the same user", user ,user3);
    }

    public void testRestartConversation()
    {
        final String BEAN_NAME = "persistentBackingBean";

        PersistentBackingBean bean = (PersistentBackingBean) applicationContext.getBean(BEAN_NAME);

        UserData user = bean.createUser();

        bean.invalidateAndRestartConversation();

        /* here we access the new conversation */
        UserData restartedUser = bean.getRestartedUser();

        assertNotNull("should have got a user", restartedUser);
        assertNotSame("should not be the same user", user, restartedUser);
        assertEquals("has to be the same user id", user.getId(), restartedUser.getId());

        /* end all conversations*/
        ConversationManager.getInstance().clearCurrentConversationContext();

        bean.updateUser(user.getId(), "test2");

        /* invalidate conversation */
        bean.invalidateConversation();

        UserData user4 = bean.readUser(user.getId());

        assertNotNull(user4);
        assertEquals(user.getId(), user4.getId());
        assertEquals("test", user4.getUsername());
    }
}
