"use strict";
/* Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to you under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
const chai_1 = require("chai");
const mocha_1 = require("mocha");
const rxjs_1 = require("rxjs");
const umd_1 = require("mona-dish/dist/js/umd");
const tobago_with_header_1 = require("./markups/tobago-with-header");
const tobago_without_header_1 = require("./markups/tobago-without-header");
const jsdom = require("jsdom");
const { JSDOM } = jsdom;
var trim = umd_1.Lang.trim;
global.window = {};
let dom = null;
(0, mocha_1.describe)('DOMQuery tests', function () {
    beforeEach(function () {
        dom = new JSDOM(`
            <!DOCTYPE html>
        <html lang="en">
        <head>
            <meta charset="UTF-8">
            <title>Title</title>
            </head>
            <body>
                <div id="id_1"></div>
                <div id="id_2"  booga="blarg" class="blarg2"></div>
                <div id="id_3" class="blarg1 blarg2"></div>
                <div id="id_4"></div>
            </body>
            </html>
    
    `, {
            contentType: "text/html",
            runScripts: "dangerously",
            resources: "usable",
            url: `file://${__dirname}/index.html`
        });
        let window = dom.window;
        global.dom = dom;
        global.window = window;
        global.body = window.document.body;
        global.document = window.document;
        global.navigator = {
            language: "en-En"
        };
    });
    this.afterEach(function () {
    });
    (0, mocha_1.it)('basic init', function () {
        let probe1 = new umd_1.DomQuery(window.document.body);
        let probe2 = umd_1.DomQuery.querySelectorAll("div");
        let probe3 = new umd_1.DomQuery(probe1, probe2);
        let probe4 = new umd_1.DomQuery(window.document.body, probe3);
        (0, chai_1.expect)(probe1.length).to.be.eq(1);
        (0, chai_1.expect)(probe2.length == 4).to.be.true;
        (0, chai_1.expect)(probe3.length == 5).to.be.true;
        //still under discussion (we might index to avoid doubles)
        (0, chai_1.expect)(probe4.length == 6).to.be.true;
    });
    (0, mocha_1.it)('proper iterator api and rxjs mapping', function () {
        let probe1 = new umd_1.DomQuery(window.document.body);
        let probe2 = umd_1.DomQuery.querySelectorAll("div");
        let o1 = (0, rxjs_1.from)(umd_1.Stream.ofDataSource(probe1));
        let o2 = (0, rxjs_1.from)(umd_1.Stream.ofDataSource(probe2));
        let cnt1 = 0;
        let isDQuery = false;
        let cnt2 = 0;
        o1.subscribe((item) => {
            cnt1++;
        });
        o2.subscribe((item) => {
            cnt2++;
            isDQuery = (item.length == 1) && (item instanceof umd_1.DomQuery);
        });
        (0, chai_1.expect)(probe1.length).to.be.eq(1);
        (0, chai_1.expect)(probe2.length == 4).to.be.true;
        (0, chai_1.expect)(isDQuery).to.be.true;
    });
    (0, mocha_1.it)('proper iterator api and rxjs mapping with observable', function () {
        let probe1 = new umd_1.DomQuery(window.document.body);
        let probe2 = umd_1.DomQuery.querySelectorAll("div");
        let o1 = (0, rxjs_1.from)(umd_1.Stream.ofDataSource(probe1));
        let o2 = (0, rxjs_1.from)(umd_1.Stream.ofDataSource(probe2));
        let cnt1 = 0;
        let isDQuery = false;
        let cnt2 = 0;
        o1.subscribe((item) => {
            cnt1++;
        });
        o2.subscribe((item) => {
            cnt2++;
            isDQuery = (item.length == 1) && (item instanceof umd_1.DomQuery);
        });
        (0, chai_1.expect)(probe1.length).to.be.eq(1);
        (0, chai_1.expect)(probe2.length == 4).to.be.true;
        (0, chai_1.expect)(isDQuery).to.be.true;
    });
    (0, mocha_1.it)('domquery ops test filter', function () {
        let probe2 = umd_1.DomQuery.querySelectorAll("div");
        probe2 = probe2.filter((item) => item.id.match((id) => id != "id_1"));
        (0, chai_1.expect)(probe2.length == 3);
    });
    (0, mocha_1.it)('global eval test', function () {
        let probe2 = umd_1.DomQuery.querySelectorAll("div");
        probe2 = probe2.filter((item) => item.id.match((id) => id != "id_1"));
        (0, chai_1.expect)(probe2.length == 3);
    });
    (0, mocha_1.it)('must detach', function () {
        let probe2 = umd_1.DomQuery.querySelectorAll("div#id_1");
        probe2.detach();
        (0, chai_1.expect)(umd_1.DomQuery.querySelectorAll("div#id_1").isPresent()).to.be.false;
        probe2.appendTo(umd_1.DomQuery.querySelectorAll("body"));
        (0, chai_1.expect)(umd_1.DomQuery.querySelectorAll("div#id_1").isPresent()).to.be.true;
    });
    (0, mocha_1.it)('domquery ops test2 each', () => {
        let probe2 = umd_1.DomQuery.querySelectorAll("div#id_1");
        umd_1.DomQuery.globalEval("document.getElementById('id_1').innerHTML = 'hello'");
        (0, chai_1.expect)(probe2.html().value).to.eq("hello");
        (0, chai_1.expect)(umd_1.DomQuery.byId(document.head).innerHTML.indexOf("document.getElementById('id_1').innerHTML = 'hello'")).to.eq(-1);
        umd_1.DomQuery.globalEval("document.getElementById('id_1').innerHTML = 'hello2'", "nonci");
        (0, chai_1.expect)(probe2.html().value).to.eq("hello2");
    });
    (0, mocha_1.it)('domquery ops test2 with sticky eval code', () => {
        let probe2 = umd_1.DomQuery.querySelectorAll("div#id_1");
        umd_1.DomQuery.globalEvalSticky("document.getElementById('id_1').innerHTML = 'hello'");
        (0, chai_1.expect)(probe2.html().value).to.eq("hello");
        (0, chai_1.expect)(umd_1.DomQuery.byId(document.head).innerHTML.indexOf("document.getElementById('id_1').innerHTML = 'hello'")).not.to.eq(-1);
        umd_1.DomQuery.globalEvalSticky("document.getElementById('id_1').innerHTML = 'hello2'", "nonci");
        (0, chai_1.expect)(probe2.html().value).to.eq("hello2");
        (0, chai_1.expect)(umd_1.DomQuery.byId(document.head).innerHTML.indexOf("document.getElementById('id_1').innerHTML = 'hello2'")).not.to.eq(-1);
    });
    (0, mocha_1.it)('domquery ops test2 eachNode', function () {
        let probe2 = umd_1.DomQuery.querySelectorAll("div");
        let noIter = 0;
        probe2 = probe2.each((item, cnt) => {
            (0, chai_1.expect)(item instanceof umd_1.DomQuery).to.be.true;
            (0, chai_1.expect)(noIter == cnt).to.be.true;
            noIter++;
        });
        (0, chai_1.expect)(noIter == 4).to.be.true;
    });
    (0, mocha_1.it)('domquery ops test2 byId', function () {
        let probe2 = umd_1.DomQuery.byId("id_1");
        (0, chai_1.expect)(probe2.length == 1).to.be.true;
        probe2 = umd_1.DomQuery.byTagName("div");
        (0, chai_1.expect)(probe2.length == 4).to.be.true;
    });
    (0, mocha_1.it)('outerhtml and eval tests', function () {
        let probe1 = new umd_1.DomQuery(window.document.body);
        probe1.querySelectorAll("#id_1").outerHTML(`
            <div id='barg'>
            
            </div>
            <script type="text/javascript">
                document.getElementById('blarg').innerHTML = 'hello world';
            </script>
            `, true, true);
        (0, chai_1.expect)(window.document.body.innerHTML.indexOf("hello world") != -1).to.be.true;
        (0, chai_1.expect)(window.document.head.innerHTML.indexOf("hello world") == -1).to.be.true;
        (0, chai_1.expect)(window.document.body.innerHTML.indexOf("id_1") == -1).to.be.true;
        (0, chai_1.expect)(window.document.body.innerHTML.indexOf("blarg") != -1).to.be.true;
    });
    (0, mocha_1.it)('attr test and eval tests', function () {
        let probe1 = new umd_1.DomQuery(document);
        probe1.querySelectorAll("div#id_2").attr("style").value = "border=1;";
        let blarg = probe1.querySelectorAll("div#id_2").attr("booga").value;
        let style = probe1.querySelectorAll("div#id_2").attr("style").value;
        let nonexistent = probe1.querySelectorAll("div#id_2").attr("buhaha").value;
        (0, chai_1.expect)(blarg).to.be.eq("blarg");
        (0, chai_1.expect)(style).to.be.eq("border=1;");
        (0, chai_1.expect)(nonexistent).to.be.eq(null);
    });
    (0, mocha_1.it)('style must work ', function () {
        let probe1 = new umd_1.DomQuery(document);
        let probe = probe1.querySelectorAll("div#id_2");
        probe.style("border").value = "10px solid red";
        probe.style("color").value = "blue";
        let styleNodeLevel = probe.getAsElem(0).value.style['color'];
        (0, chai_1.expect)(probe.style("border").value).to.eq("10px solid red");
        (0, chai_1.expect)(probe.style("color").value).to.eq("blue");
        (0, chai_1.expect)(styleNodeLevel).to.eq('blue');
    });
    (0, mocha_1.it)('must perform addClass and hasClass correctly', function () {
        let probe1 = new umd_1.DomQuery(document);
        let element = probe1.querySelectorAll("div#id_2");
        element.addClass("booga").addClass("Booga2");
        let classdef = element.attr("class").value;
        (0, chai_1.expect)(classdef).to.eq("blarg2 booga Booga2");
        element.removeClass("booga2");
        (0, chai_1.expect)(element.hasClass("booga2")).to.be.false;
        (0, chai_1.expect)(element.hasClass("booga")).to.be.true;
    });
    (0, mocha_1.it)('must perform addClass and hasClass correctly 2', function () {
        let probe1 = new umd_1.DomQuery(document);
        let element = probe1.querySelectorAll(".blarg2");
        element.addClass("booga").addClass("Booga2");
        let classdef = element.attr("class").value;
        (0, chai_1.expect)(classdef).to.eq("blarg2 booga Booga2");
        element.removeClass("booga2");
        (0, chai_1.expect)(element.hasClass("booga2")).to.be.false;
        (0, chai_1.expect)(element.hasClass("booga")).to.be.true;
        (0, chai_1.expect)(element.hasClass("blarg2")).to.be.true;
    });
    (0, mocha_1.it)('must perform addClass and hasClass correctly 2', function () {
        let probe1 = new umd_1.DomQuery(document);
        let element = probe1.querySelectorAll(".blarg2");
        element.addClass("booga").addClass("Booga2");
        (0, chai_1.expect)(probe1.querySelectorAll(".Booga2").length).eq(2);
    });
    (0, mocha_1.it)('must perform insert before and insert after correctly', function () {
        let probe1 = new umd_1.DomQuery(document).querySelectorAll("#id_2");
        let insert = umd_1.DomQuery.fromMarkup("<div id='insertedBefore'></div><div id='insertedBefore2'></div>");
        let insert2 = umd_1.DomQuery.fromMarkup("<div id='insertedAfter'></div><div id='insertedAfter2'></div>");
        probe1.insertBefore(insert);
        probe1.insertAfter(insert2);
        (0, chai_1.expect)(umd_1.DomQuery.querySelectorAll("#insertedBefore").isPresent()).to.be.true;
        (0, chai_1.expect)(umd_1.DomQuery.querySelectorAll("#insertedBefore2").isPresent()).to.be.true;
        (0, chai_1.expect)(umd_1.DomQuery.querySelectorAll("#id_2").isPresent()).to.be.true;
        (0, chai_1.expect)(umd_1.DomQuery.querySelectorAll("#insertedAfter").isPresent()).to.be.true;
        (0, chai_1.expect)(umd_1.DomQuery.querySelectorAll("#insertedAfter2").isPresent()).to.be.true;
    });
    (0, mocha_1.it)('do not create new <html> tag on <header', function () {
        const fromMarkupWithHeader = umd_1.DomQuery.fromMarkup(tobago_with_header_1.tobagoSheetWithHeader);
        const fromMarkupWithoutHeader = umd_1.DomQuery.fromMarkup(tobago_without_header_1.tobagoSheetWithoutHeader);
        (0, chai_1.expect)(fromMarkupWithHeader.tagName.value === "HTML").to.be.false;
        (0, chai_1.expect)(fromMarkupWithoutHeader.tagName.value === "HTML").to.be.false;
    });
    (0, mocha_1.it)('do not falsely assume standard tag', function () {
        const fromMarkup1 = umd_1.DomQuery.fromMarkup(`
        <head-mine>booga</head-mine>
        `);
        const fromMarkup2 = umd_1.DomQuery.fromMarkup(`
        <body_mine>booga</body_mine>
        
        `);
        (0, chai_1.expect)(fromMarkup1.tagName.value === "HTML").to.be.false;
        (0, chai_1.expect)(fromMarkup1.tagName.value === "HTML").to.be.false;
        (0, chai_1.expect)(fromMarkup1.tagName.value === "HEAD").to.be.false;
        (0, chai_1.expect)(fromMarkup2.tagName.value === "BODY").to.be.false;
    });
    (0, mocha_1.it)('it must stream', function () {
        let probe1 = new umd_1.DomQuery(document).querySelectorAll("div");
        let coll = umd_1.Stream.ofDomQuery(probe1).collect(new umd_1.ArrayCollector());
        (0, chai_1.expect)(coll.length == 4).to.be.true;
        coll = umd_1.LazyStream.ofDomQuery(probe1).collect(new umd_1.ArrayCollector());
        (0, chai_1.expect)(coll.length == 4).to.be.true;
    });
    (0, mocha_1.it)('it must stream - DQ API (dynamically added)', function () {
        let probe1 = new umd_1.DomQuery(document).querySelectorAll("div");
        let coll = probe1.asArray;
        (0, chai_1.expect)(coll.length == 4).to.be.true;
    });
    (0, mocha_1.it)('it must stream to a domquery', function () {
        let probe1 = new umd_1.DomQuery(document).querySelectorAll("div");
        let coll = umd_1.Stream.ofDataSource(probe1).collect(new umd_1.DomQueryCollector());
        (0, chai_1.expect)(coll.length == 4).to.be.true;
        probe1.reset();
        coll = umd_1.LazyStream.ofStreamDataSource(probe1).collect(new umd_1.DomQueryCollector());
        (0, chai_1.expect)(coll.length == 4).to.be.true;
    });
    (0, mocha_1.it)('it must have parents', function () {
        let probe1 = new umd_1.DomQuery(document).querySelectorAll("div");
        let coll = umd_1.Stream.ofDataSource(probe1.parentsWhileMatch("body")).collect(new umd_1.ArrayCollector());
        (0, chai_1.expect)(coll.length == 1).to.be.true;
    });
    (0, mocha_1.it)("must have a working insertBefore and insertAfter", function () {
        let probe1 = new umd_1.DomQuery(document).byId("id_2");
        probe1.insertBefore(umd_1.DomQuery.fromMarkup(` <div id="id_x_0"></div><div id="id_x_1"></div>`));
        probe1.insertAfter(umd_1.DomQuery.fromMarkup(` <div id="id_x_0_1"></div><div id="id_x_1_1"></div>`));
        (0, chai_1.expect)(umd_1.DomQuery.querySelectorAll("div").length).to.eq(8);
        umd_1.DomQuery.querySelectorAll("body").innerHTML = trim(umd_1.DomQuery.querySelectorAll("body").innerHTML.replace(/>\s*</gi, "><"));
        (0, chai_1.expect)(umd_1.DomQuery.querySelectorAll("body").childNodes.length).to.eq(8);
        let innerHtml = umd_1.DomQuery.querySelectorAll("body").innerHTML;
        (0, chai_1.expect)(innerHtml.indexOf("id_x_0") < innerHtml.indexOf("id_x_1")).to.be.true;
        (0, chai_1.expect)(innerHtml.indexOf("id_x_0") < innerHtml.indexOf("id_2")).to.be.true;
        (0, chai_1.expect)(innerHtml.indexOf("id_x_0") > 0).to.be.true;
        (0, chai_1.expect)(innerHtml.indexOf("id_x_0_1") > innerHtml.indexOf("id_2")).to.be.true;
        (0, chai_1.expect)(innerHtml.indexOf("id_x_1_1") > innerHtml.indexOf("id_x_0_1")).to.be.true;
    });
    (0, mocha_1.it)("must have a working replace", function () {
        let probe1 = new umd_1.DomQuery(document).byId("id_1");
        probe1.replace(umd_1.DomQuery.fromMarkup(` <div id="id_x_0"></div><div id="id_x_1"></div>`));
        (0, chai_1.expect)(umd_1.DomQuery.querySelectorAll("div").length).to.eq(5);
        let innerHtml = umd_1.DomQuery.querySelectorAll("body").innerHTML;
        (0, chai_1.expect)(innerHtml.indexOf("id_x_0") > 0).to.be.true;
        (0, chai_1.expect)(innerHtml.indexOf("id_x_0") < innerHtml.indexOf("id_2")).to.be.true;
        (0, chai_1.expect)(innerHtml.indexOf("id_x_0") < innerHtml.indexOf("id_3")).to.be.true;
        (0, chai_1.expect)(innerHtml.indexOf("id_x_0") < innerHtml.indexOf("id_x_1")).to.be.true;
        (0, chai_1.expect)(innerHtml.indexOf("id_x_1") > 0).to.be.true;
        (0, chai_1.expect)(innerHtml.indexOf("id_x_1") < innerHtml.indexOf("id_2")).to.be.true;
        (0, chai_1.expect)(innerHtml.indexOf("id_x_1") < innerHtml.indexOf("id_3")).to.be.true;
        (0, chai_1.expect)(innerHtml.indexOf("id_1") == -1).to.be.true;
    });
    (0, mocha_1.it)("must have a working replace - 2", function () {
        let probe1 = new umd_1.DomQuery(document).byId("id_2");
        probe1.replace(umd_1.DomQuery.fromMarkup(` <div id="id_x_0"></div><div id="id_x_1"></div>`));
        (0, chai_1.expect)(umd_1.DomQuery.querySelectorAll("div").length).to.eq(5);
        let innerHtml = umd_1.DomQuery.querySelectorAll("body").innerHTML;
        (0, chai_1.expect)(innerHtml.indexOf("id_x_0") > innerHtml.indexOf("id_1")).to.be.true;
        (0, chai_1.expect)(innerHtml.indexOf("id_x_0") > 0).to.be.true;
        (0, chai_1.expect)(innerHtml.indexOf("id_x_0") > innerHtml.indexOf("id_0")).to.be.true;
        (0, chai_1.expect)(innerHtml.indexOf("id_x_0") < innerHtml.indexOf("id_3")).to.be.true;
        (0, chai_1.expect)(innerHtml.indexOf("id_x_1") > 0).to.be.true;
        (0, chai_1.expect)(innerHtml.indexOf("id_x_1") > innerHtml.indexOf("id_0")).to.be.true;
        (0, chai_1.expect)(innerHtml.indexOf("id_x_1") < innerHtml.indexOf("id_3")).to.be.true;
        (0, chai_1.expect)(innerHtml.indexOf("id_2") == -1).to.be.true;
    });
    (0, mocha_1.it)("must have a working replace - 3", function () {
        let probe1 = new umd_1.DomQuery(document).byId("id_4");
        probe1.replace(umd_1.DomQuery.fromMarkup(` <div id="id_x_0"></div><div id="id_x_1"></div>`));
        (0, chai_1.expect)(umd_1.DomQuery.querySelectorAll("div").length).to.eq(5);
        let innerHtml = umd_1.DomQuery.querySelectorAll("body").innerHTML;
        (0, chai_1.expect)(innerHtml.indexOf("id_x_0") > 0).to.be.true;
        (0, chai_1.expect)(innerHtml.indexOf("id_x_0") > innerHtml.indexOf("id_1")).to.be.true;
        (0, chai_1.expect)(innerHtml.indexOf("id_x_0") > innerHtml.indexOf("id_2")).to.be.true;
        (0, chai_1.expect)(innerHtml.indexOf("id_x_0") > innerHtml.indexOf("id_3")).to.be.true;
        (0, chai_1.expect)(innerHtml.indexOf("id_x_0") < innerHtml.indexOf("id_x_1")).to.be.true;
        (0, chai_1.expect)(innerHtml.indexOf("id_x_1") > 0).to.be.true;
        (0, chai_1.expect)(innerHtml.indexOf("id_x_1") > innerHtml.indexOf("id_1")).to.be.true;
        (0, chai_1.expect)(innerHtml.indexOf("id_x_1") > innerHtml.indexOf("id_2")).to.be.true;
        (0, chai_1.expect)(innerHtml.indexOf("id_x_1") > innerHtml.indexOf("id_3")).to.be.true;
        (0, chai_1.expect)(innerHtml.indexOf("id_4") == -1).to.be.true;
    });
    (0, mocha_1.it)("must have a working input handling", function () {
        umd_1.DomQuery.querySelectorAll("body").innerHTML = `<form id="blarg">
    <div id="embed1">
        <input type="text" id="id_1" name="id_1" value="id_1_val"></input>
        <input type="text" id="id_2" name="id_2" value="id_2_val" disabled="disabled"> </input>
        <textarea type="text" id="id_3" name="id_3">textareaVal</textarea>

        <fieldset>
            <input type="radio" id="mc" name="cc_1" value="Mastercard" checked="checked"></input>
            <label for="mc"> Mastercard</label>
            <input type="radio" id="vi" name="cc_1" value="Visa"></input>
            <label for="vi"> Visa</label>
            <input type="radio" id="ae" name="cc_1" value="AmericanExpress"></input>
            <label for="ae"> American Express</label>
        </fieldset>
        <select id="val_5" name="val_5" name="top5" size="5">
            <option>barg</option>
            <option>jjj</option>
            <option selected>akaka</option>
            <option>blon</option>
            <option>slashs</option>
        </select>
    </div>
</form>
       `;
        let length = umd_1.DomQuery.querySelectorAll("form").elements.length;
        (0, chai_1.expect)(length == 8).to.be.true;
        let length1 = umd_1.DomQuery.querySelectorAll("body").elements.length;
        (0, chai_1.expect)(length1 == 8).to.be.true;
        let length2 = umd_1.DomQuery.byId("embed1").elements.length;
        (0, chai_1.expect)(length2 == 8).to.be.true;
        let count = umd_1.Stream.ofDataSource(umd_1.DomQuery.byId("embed1").elements)
            .map(item => item.disabled ? 1 : 0)
            .reduce((val1, val2) => val1 + val2, 0);
        (0, chai_1.expect)(count.value).to.eq(1);
        umd_1.Stream.ofDataSource(umd_1.DomQuery.byId("embed1").elements)
            .filter(item => item.disabled)
            .each(item => item.disabled = false);
        count = umd_1.Stream.ofDataSource(umd_1.DomQuery.byId("embed1").elements)
            .map(item => item.disabled ? 1 : 0)
            .reduce((val1, val2) => val1 + val2, 0);
        (0, chai_1.expect)(count.value).to.eq(0);
        count = umd_1.Stream.ofDataSource(umd_1.DomQuery.byId("embed1").elements)
            .map(item => item.attr("checked").isPresent() ? 1 : 0)
            .reduce((val1, val2) => val1 + val2, 0);
        (0, chai_1.expect)(count.value).to.eq(1);
        (0, chai_1.expect)(umd_1.DomQuery.byId("id_1").inputValue.value == "id_1_val").to.be.true;
        umd_1.DomQuery.byId("id_1").inputValue.value = "booga";
        (0, chai_1.expect)(umd_1.DomQuery.byId("id_1").inputValue.value == "booga").to.be.true;
        (0, chai_1.expect)(umd_1.DomQuery.byId("id_3").inputValue.value).to.eq("textareaVal");
        umd_1.DomQuery.byId("id_3").inputValue.value = "hello world";
        (0, chai_1.expect)(umd_1.DomQuery.byId("id_3").inputValue.value).to.eq("hello world");
        let cfg = new umd_1.Config(umd_1.DomQuery.querySelectorAll("form").elements.encodeFormElement());
        (0, chai_1.expect)(cfg.getIf("id_1").value[0]).to.eq("booga");
        (0, chai_1.expect)(cfg.getIf("id_2").value[0]).to.eq("id_2_val");
        (0, chai_1.expect)(cfg.getIf("id_3").value[0]).to.eq("hello world");
        (0, chai_1.expect)(cfg.getIf("cc_1").value[0]).to.eq("Mastercard");
        (0, chai_1.expect)(cfg.getIf("val_5").value[0]).to.eq("akaka");
    });
    (0, mocha_1.it)("must have a proper loadScriptEval execution", function (done) {
        umd_1.DomQuery.byTagName("body").loadScriptEval("./fixtures/test.js");
        setTimeout(() => {
            (0, chai_1.expect)(umd_1.DomQuery.byId("id_1").innerHTML == "hello world").to.be.true;
            done();
        }, 100);
    });
    (0, mocha_1.it)("must have first etc working", function () {
        (0, chai_1.expect)(umd_1.DomQuery.querySelectorAll("div").first().id.value).to.eq("id_1");
    });
    (0, mocha_1.it)("runscript runcss", function (done) {
        umd_1.DomQuery.byTagName("body").innerHTML = `
            <div id="first"></div>
            <div id="second"></div>
            <div id="third"></div>
            <div id="fourth"></div>
            
            <script type="text/javascript">
                document.getElementById("first").innerHTML = "hello world";
            </script>
            <script type="text/javascript">
            //<![CDATA[
                document.getElementById("second").innerHTML = "hello world";
            //]]>    
            </script>
            <script type="text/javascript">
            <!--
                document.getElementById("third").innerHTML = "hello world";
            //-->   
            </script>
              <script type="text/javascript">
            //<!--
                document.getElementById("fourth").innerHTML = "hello world";
            //-->   
            </script>
        
            <style type="text/css">
                #first {
                    border: 1px solid black;
                }
            </style>
            
            <link rel="stylesheet" href="./fixtures/blank.css"></link>
        `;
        let content = umd_1.DomQuery.byTagName("body").runScripts().runCss();
        (0, chai_1.expect)(content.byId("first").innerHTML).to.eq("hello world");
        (0, chai_1.expect)(content.byId("second").innerHTML).to.eq("hello world");
        (0, chai_1.expect)(content.byId("third").innerHTML).to.eq("hello world");
        (0, chai_1.expect)(content.byId("fourth").innerHTML).to.eq("hello world");
        (0, chai_1.expect)(umd_1.DomQuery.byTagName("body")
            .querySelectorAll("link[rel='stylesheet'][href='./fixtures/blank.css']").length).to.eq(1);
        done();
    });
    //TODO defer does not work in jsdom
    (0, mocha_1.it)("must have a proper loadScriptEval deferred", function (done) {
        umd_1.DomQuery.byId(document.body).loadScriptEval("./fixtures/test2.js", 200);
        setTimeout(() => {
            (0, chai_1.expect)(umd_1.DomQuery.byId("id_1").innerHTML == "hello world").to.be.false;
        }, 100);
        setTimeout(() => {
            (0, chai_1.expect)(umd_1.DomQuery.byId("id_1").innerHTML == "hello world").to.be.true;
            done();
        }, 1500);
    });
    (0, mocha_1.it)("it must handle events properly", function () {
        let clicked = 0;
        let listener = (evt) => {
            clicked++;
        };
        let eventReceiver = umd_1.DomQuery.byId("id_1");
        eventReceiver.addEventListener("click", listener);
        eventReceiver.click();
        (0, chai_1.expect)(clicked).to.eq(1);
        eventReceiver.removeEventListener("click", listener);
        eventReceiver.click();
        (0, chai_1.expect)(clicked).to.eq(1);
    });
    (0, mocha_1.it)("it must handle innerText properly", function (done) {
        //jsdom bug
        Object.defineProperty(Object.prototype, 'innerText', {
            get() {
                return this.textContent;
            },
        });
        let probe = umd_1.DomQuery.byId("id_1");
        probe.innerHTML = "<div>hello</div><div>world</div>";
        (0, chai_1.expect)(probe.innerText()).to.eq("helloworld");
        done();
    });
    (0, mocha_1.it)("it must handle textContent properly", function () {
        let probe = umd_1.DomQuery.byId("id_1");
        probe.innerHTML = "<div>hello</div><div>world</div>";
        (0, chai_1.expect)(probe.textContent()).to.eq("helloworld");
    });
    (0, mocha_1.it)("it must handle iterations properly", function () {
        let probe = umd_1.DomQuery.byTagName("div");
        let resArr = umd_1.LazyStream.ofStreamDataSource(probe).collect(new umd_1.ArrayCollector());
        (0, chai_1.expect)(resArr.length).to.eq(4);
        probe.reset();
        while (probe.hasNext()) {
            let el = probe.next();
            (0, chai_1.expect)(el.tagName.value.toLowerCase()).to.eq("div");
        }
        (0, chai_1.expect)(probe.next()).to.eq(null);
        let probe2 = umd_1.DomQuery.byTagName("div").limits(2);
        resArr = umd_1.LazyStream.ofStreamDataSource(probe2).collect(new umd_1.ArrayCollector());
        (0, chai_1.expect)(resArr.length).to.eq(2);
    });
    (0, mocha_1.it)("it must handle subnodes properly", function () {
        let probe = umd_1.DomQuery.byTagName("div");
        (0, chai_1.expect)(probe.subNodes(1, 3).length).to.eq(2);
        probe = umd_1.DomQuery.byTagName("body").childNodes.subNodes(0, 2);
        (0, chai_1.expect)(probe.length).to.eq(2);
        probe = umd_1.DomQuery.byTagName("div").subNodes(2);
        (0, chai_1.expect)(probe.length).to.eq(2);
    });
    (0, mocha_1.it)("it must ensure shadow dom creation works properly", function () {
        let probe = umd_1.DomQuery.byTagName("div");
        try {
            //probably not testable atm, mocha does not have shadow dom support
            //we might be able to shim it in one way or the other
            let element = probe.attachShadow();
            (0, chai_1.expect)(element.length > 0).to.eq(true);
        }
        catch (e) {
            //not supported we still need to get an error here
            (0, chai_1.expect)(e.message.indexOf("not supported") != -1).to.be.true;
        }
    });
    (0, mocha_1.it)("parent must break shadow barriers", function () {
        let probe = umd_1.DomQuery.fromMarkup("<div id='shadowItem'>hello</div>'");
        try {
            //probably not testable atm, mocha does not have shadow dom support
            //we might be able to shim it in one way or the other
            let element = umd_1.DomQuery.byId("id_1").attachShadow();
            element.append(probe);
            (0, chai_1.expect)(probe.firstParent("#id_1").length > 0).to.eq(true);
        }
        catch (e) {
            //not supported we still need to get an error here
            (0, chai_1.expect)(e.message.indexOf("not supported") != -1).to.be.true;
        }
    });
    (0, mocha_1.it)('it must have a working wait for dom with mut observer and must detect condition after change', function () {
        return __awaiter(this, void 0, void 0, function* () {
            let probe = umd_1.DomQuery.byId('id_1');
            probe.innerHTML = 'true';
            let ret = yield probe.waitUntilDom((element) => element.innerHTML.indexOf('true') != -1);
            (0, chai_1.expect)(ret.isPresent());
            probe = umd_1.DomQuery.byId('bosushsdhs');
            ret = yield probe.waitUntilDom((element) => element.isAbsent());
            (0, chai_1.expect)(ret.isAbsent());
        });
    });
    (0, mocha_1.it)('it must have a working wait for dom with mut observer', function () {
        return __awaiter(this, void 0, void 0, function* () {
            let probe = umd_1.DomQuery.byId('id_1');
            setTimeout(() => probe.innerHTML = 'true', 300);
            let ret = yield probe.waitUntilDom((element) => element.innerHTML.indexOf('true') != -1);
            delete window.MutationObserver;
            delete global.MutationObserver;
            probe.innerHTML = "";
            setTimeout(() => probe.innerHTML = 'true', 300);
            let ret2 = yield probe.waitUntilDom((element) => element.innerHTML.indexOf('true') != -1);
            (0, chai_1.expect)(ret.isPresent() && ret2.isPresent());
        });
    });
    (0, mocha_1.it)('it must have a timeout', function () {
        return __awaiter(this, void 0, void 0, function* () {
            let probe = umd_1.DomQuery.byId('booga');
            try {
                setTimeout(() => probe.innerHTML = 'true', 300);
                yield probe.waitUntilDom((element) => element.innerHTML.indexOf('true') != -1);
                chai_1.expect.fail("must have a timeout");
            }
            catch (ex) {
                (0, chai_1.expect)(!!ex);
            }
            try {
                delete window.MutationObserver;
                delete global.MutationObserver;
                probe.innerHTML = "";
                setTimeout(() => probe.innerHTML = 'true', 300);
                yield probe.waitUntilDom((element) => element.innerHTML.indexOf('true') != -1);
                chai_1.expect.fail("must have a timeout");
            }
            catch (ex2) {
                (0, chai_1.expect)(!!ex2);
            }
        });
    });
    (0, mocha_1.it)('must handle null inputs correctly', function () {
        const dq = new umd_1.DomQuery(null);
        (0, chai_1.expect)(dq.isAbsent()).to.eq(true);
    });
    (0, mocha_1.it)('concat must work as expected resulting', function () {
        let probe = umd_1.DomQuery.querySelectorAll("div");
        let probe2 = umd_1.DomQuery.querySelectorAll("body");
        let result = probe.concat(probe2);
        (0, chai_1.expect)(result.length).to.eq(probe.length + probe2.length);
        //lets now check for filter double
        probe2 = umd_1.DomQuery.querySelectorAll('div');
        result = probe.concat(probe2);
        (0, chai_1.expect)(result.length).to.eq(probe.length);
    });
    (0, mocha_1.it)('must handle match correctly', function () {
        let probe = umd_1.DomQuery.querySelectorAll("div").first();
        let probe2 = umd_1.DomQuery.querySelectorAll("body").first();
        (0, chai_1.expect)(probe.matchesSelector("div")).to.eq(true);
        (0, chai_1.expect)(probe2.matchesSelector("body")).to.eq(true);
        (0, chai_1.expect)(probe2.matchesSelector("div")).to.eq(false);
    });
    (0, mocha_1.it)('must by recycleable', function () {
        let probe = umd_1.DomQuery.querySelectorAll("div");
        let probe2 = umd_1.DomQuery.querySelectorAll("body");
        let res1 = probe.filter(item => item.matchesSelector("div"));
        (0, chai_1.expect)(res1.length).to.eq(4);
        let res2 = probe.filter(item => item.matchesSelector("div"));
        (0, chai_1.expect)(res2.length).to.eq(4);
    });
    (0, mocha_1.it)('delete must work', function () {
        let probe = umd_1.DomQuery.querySelectorAll("body");
        let probe2 = umd_1.DomQuery.fromMarkup("<div id='deleteprobe1'>snafu</div>");
        probe2.appendTo(probe);
        (0, chai_1.expect)(probe.querySelectorAll("#deleteprobe1").isPresent()).to.eq(true);
        probe2.delete();
        (0, chai_1.expect)(probe.querySelectorAll("#deleteprobe1").isAbsent()).to.eq(true);
    });
    (0, mocha_1.it)('must work with rxjs and domquery', function () {
        let probe = umd_1.DomQuery.querySelectorAll("div");
        let probe2 = umd_1.DomQuery.querySelectorAll("div");
        let probeCnt = 0;
        let probe2Cnt = 0;
        (0, rxjs_1.from)(probe).subscribe(el => probeCnt++);
        (0, rxjs_1.from)(umd_1.Stream.ofDataSource(probe2)).subscribe(el => probe2Cnt++);
        (0, chai_1.expect)(probeCnt).to.be.above(0);
        (0, chai_1.expect)(probeCnt).to.eq(probe2Cnt);
    });
    (0, mocha_1.it)('must handle closest properly', function () {
        let probe = umd_1.DomQuery.byId("id_1");
        probe.innerHTML = "<div id='inner_elem'>hello world<div id='inner_elem2'></div></div>";
        let probe2 = umd_1.DomQuery.byId("inner_elem");
        (0, chai_1.expect)(probe2.closest("div#id_1").id.value).to.eq("id_1");
        (0, chai_1.expect)(probe2.parent().closest("div").id.value).to.eq("id_1");
        probe2 = umd_1.DomQuery.byId("inner_elem2");
        (0, chai_1.expect)(probe2.closest("div").id.value).to.eq("inner_elem2");
        (0, chai_1.expect)(probe2.closest("div#id_1").id.value).to.eq("id_1");
        (0, chai_1.expect)(probe2.parent().parent().closest("div").id.value).to.eq("id_1");
    });
});
//# sourceMappingURL=DomQueryTest.spec.js.map