/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.security.util.crypto;

import java.math.BigInteger;
import java.util.concurrent.TimeUnit;
import org.apache.nifi.security.util.crypto.AbstractSecureHasher;
import org.apache.nifi.security.util.crypto.scrypt.Scrypt;
import org.bouncycastle.util.encoders.Hex;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ScryptSecureHasher
extends AbstractSecureHasher {
    private static final Logger logger = LoggerFactory.getLogger(ScryptSecureHasher.class);
    private static final int DEFAULT_N = Double.valueOf(Math.pow(2.0, 14.0)).intValue();
    private static final int DEFAULT_R = 8;
    private static final int DEFAULT_P = 1;
    private static final int DEFAULT_DK_LENGTH = 32;
    private static final int DEFAULT_SALT_LENGTH = Scrypt.getDefaultSaltLength();
    private static final int MIN_P = 1;
    private static final int MIN_DK_LENGTH = 1;
    private static final int MIN_N = 1;
    private static final int MIN_R = 1;
    private static final int MAX_R = Double.valueOf(Math.pow(2.0, 31.0)).intValue() - 1;
    private static final int MIN_SALT_LENGTH = 8;
    private static final int MAX_SALT_LENGTH = Double.valueOf(Math.pow(2.0, 31.0)).intValue() - 1;
    private final int n;
    private final int r;
    private final int p;
    private final int dkLength;

    public ScryptSecureHasher() {
        this(DEFAULT_N, 8, 1, 32, 0);
    }

    public ScryptSecureHasher(int dkLength) {
        this(DEFAULT_N, 8, 1, dkLength, 0);
    }

    public ScryptSecureHasher(int n, int r, int p, int dkLength) {
        this(n, r, p, dkLength, 0);
    }

    public ScryptSecureHasher(int n, int r, int p, int dkLength, int saltLength) {
        this.validateParameters(n, r, p, dkLength, saltLength);
        this.n = n;
        this.r = r;
        this.p = p;
        this.dkLength = dkLength;
        this.saltLength = saltLength;
    }

    private void validateParameters(Integer n, Integer r, int p, Integer dkLength, Integer saltLength) {
        if (!ScryptSecureHasher.isRValid(r)) {
            logger.error("The provided block size r {} ( * 128 bytes) is outside the boundary of 1 to 2^31 - 1.", (Object)r);
            throw new IllegalArgumentException("Invalid r is not within the memory boundary.");
        }
        if (!ScryptSecureHasher.isNValid(n, r)) {
            logger.error("The iteration count N {} is outside the boundary of powers of 2 from 1 to 2^(128 * r / 8).", (Object)n);
            throw new IllegalArgumentException("Invalid N exceeds the iterations boundary.");
        }
        if (!ScryptSecureHasher.isPValid(p, r)) {
            logger.error("The provided parallelization factor {} is outside the boundary of 1 to ((2^32 - 1) * 32) / (128 * r).", (Object)p);
            throw new IllegalArgumentException("Invalid p exceeds the parallelism boundary.");
        }
        if (!ScryptSecureHasher.isDKLengthValid(dkLength)) {
            logger.error("The provided hash length {} is outside the boundary of 1 to (2^32 - 1) * 32.", (Object)dkLength);
            throw new IllegalArgumentException("Invalid hash length is not within the dkLength boundary.");
        }
        this.initializeSalt(saltLength);
    }

    @Override
    byte[] hash(byte[] input) {
        byte[] rawSalt = this.getSalt();
        return this.hash(input, rawSalt);
    }

    @Override
    byte[] hash(byte[] input, byte[] rawSalt) {
        logger.debug("Creating {} byte Scrypt hash with salt [{}]", (Object)this.dkLength, (Object)Hex.toHexString((byte[])rawSalt));
        if (!this.isSaltLengthValid(rawSalt.length)) {
            throw new IllegalArgumentException("The salt length (" + rawSalt.length + " bytes) is invalid");
        }
        long startNanos = System.nanoTime();
        byte[] hash = Scrypt.scrypt(input, rawSalt, this.n, this.r, this.p, this.dkLength * 8);
        long generateNanos = System.nanoTime();
        long totalDurationMillis = TimeUnit.NANOSECONDS.toMillis(generateNanos - startNanos);
        logger.debug("Generated Scrypt hash in {} ms", (Object)totalDurationMillis);
        return hash;
    }

    protected static boolean isNValid(Integer n, int r) {
        if (n < DEFAULT_N) {
            logger.warn("The provided iteration count N {} is below the recommended minimum {}.", (Object)n, (Object)DEFAULT_N);
        }
        return n >= 1 && n <= Double.valueOf(Math.pow(2.0, (double)(128 * r) / 8.0)).intValue();
    }

    protected static boolean isRValid(int r) {
        if (r < 8) {
            logger.warn("The provided r size {} * 128 B is below the recommended minimum {}.", (Object)r, (Object)8);
        }
        return r >= 1 && r <= MAX_R;
    }

    protected static boolean isPValid(int p, int r) {
        if (p < 1) {
            logger.warn("The provided parallelization factor {} is below the recommended minimum {}.", (Object)p, (Object)1);
        }
        long dividend = Double.valueOf((Math.pow(2.0, 32.0) - 1.0) * 32.0).longValue();
        int divisor = 128 * r;
        BigInteger MAX_P = new BigInteger(String.valueOf(dividend)).divide(new BigInteger(String.valueOf(divisor)));
        logger.debug("Calculated maximum p value as (2^32 - 1) * 32 [{}] / (128 * r) [{}] = {}", new Object[]{dividend, divisor, MAX_P.intValue()});
        return p >= 1 && p <= MAX_P.intValue();
    }

    protected static boolean isDKLengthValid(Integer dkLength) {
        return dkLength >= 1 && dkLength <= UPPER_BOUNDARY;
    }

    @Override
    public int getDefaultSaltLength() {
        return DEFAULT_SALT_LENGTH;
    }

    @Override
    public int getMinSaltLength() {
        return 8;
    }

    @Override
    public int getMaxSaltLength() {
        return MAX_SALT_LENGTH;
    }

    @Override
    String getAlgorithmName() {
        return "Scrypt";
    }

    @Override
    boolean acceptsEmptyInput() {
        return false;
    }
}

