/*
 * Decompiled with CFR 0.152.
 */
package tachyon.util.network;

import com.google.common.base.Preconditions;
import com.google.common.base.Throwables;
import java.io.IOException;
import java.lang.reflect.Field;
import java.net.Inet4Address;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.NetworkInterface;
import java.net.ServerSocket;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Enumeration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import tachyon.Constants;
import tachyon.TachyonURI;
import tachyon.conf.TachyonConf;
import tachyon.org.apache.thrift.transport.TServerSocket;
import tachyon.thrift.NetAddress;
import tachyon.util.OSUtils;

public final class NetworkAddressUtils {
    public static final String WILDCARD_ADDRESS = "0.0.0.0";
    public static final boolean WINDOWS = OSUtils.isWindows();
    private static final Logger LOG = LoggerFactory.getLogger((String)Constants.LOGGER_TYPE);
    private static String sLocalHost;
    private static String sLocalIP;

    private NetworkAddressUtils() {
    }

    public static void assertValidPort(int port, TachyonConf tachyonConf) {
        Preconditions.checkNotNull((Object)tachyonConf);
        Preconditions.checkArgument((port < 65536 ? 1 : 0) != 0, (Object)"Port must be less than 65536");
        if (!tachyonConf.getBoolean("tachyon.test.mode")) {
            Preconditions.checkArgument((port > 0 ? 1 : 0) != 0, (Object)"Port is only allowed to be zero in test mode.");
        }
    }

    public static void assertValidPort(InetSocketAddress address, TachyonConf tachyonConf) {
        NetworkAddressUtils.assertValidPort(address.getPort(), tachyonConf);
    }

    public static InetSocketAddress getConnectAddress(ServiceType service, TachyonConf conf) {
        return new InetSocketAddress(NetworkAddressUtils.getConnectHost(service, conf), NetworkAddressUtils.getPort(service, conf));
    }

    public static String getConnectHost(ServiceType service, TachyonConf conf) {
        String bindHost;
        String connectHost;
        if (conf.containsKey(service.mHostNameKey) && !(connectHost = conf.get(service.mHostNameKey)).isEmpty() && !connectHost.equals(WILDCARD_ADDRESS)) {
            return connectHost;
        }
        if (conf.containsKey(service.mBindHostKey) && !(bindHost = conf.get(service.mBindHostKey)).isEmpty() && !bindHost.equals(WILDCARD_ADDRESS)) {
            return bindHost;
        }
        return NetworkAddressUtils.getLocalHostName(conf);
    }

    public static int getPort(ServiceType service, TachyonConf conf) {
        return conf.getInt(service.mPortKey);
    }

    public static InetSocketAddress getBindAddress(ServiceType service, TachyonConf conf) {
        int port = NetworkAddressUtils.getPort(service, conf);
        NetworkAddressUtils.assertValidPort(port, conf);
        String host = conf.containsKey(service.mBindHostKey) && !conf.get(service.mBindHostKey).isEmpty() ? conf.get(service.mBindHostKey) : NetworkAddressUtils.getLocalHostName(conf);
        return new InetSocketAddress(host, port);
    }

    public static String getLocalHostName(TachyonConf conf) {
        if (sLocalHost != null) {
            return sLocalHost;
        }
        int hostResolutionTimeout = conf.getInt("tachyon.network.host.resolution.timeout.ms");
        return NetworkAddressUtils.getLocalHostName(hostResolutionTimeout);
    }

    public static String getLocalHostName(int timeout) {
        if (sLocalHost != null) {
            return sLocalHost;
        }
        try {
            sLocalHost = InetAddress.getByName(NetworkAddressUtils.getLocalIpAddress(timeout)).getCanonicalHostName();
            return sLocalHost;
        }
        catch (UnknownHostException e) {
            LOG.error(e.getMessage(), (Throwable)e);
            throw Throwables.propagate((Throwable)e);
        }
    }

    public static String getLocalIpAddress(TachyonConf conf) {
        if (sLocalIP != null) {
            return sLocalIP;
        }
        int hostResolutionTimeout = conf.getInt("tachyon.network.host.resolution.timeout.ms");
        return NetworkAddressUtils.getLocalIpAddress(hostResolutionTimeout);
    }

    public static String getLocalIpAddress(int timeout) {
        if (sLocalIP != null) {
            return sLocalIP;
        }
        try {
            InetAddress address = InetAddress.getLocalHost();
            LOG.debug("address: {} isLoopbackAddress: {}, with host {} {}", new Object[]{address, address.isLoopbackAddress(), address.getHostAddress(), address.getHostName()});
            if (!NetworkAddressUtils.isValidAddress(address, timeout)) {
                Enumeration<NetworkInterface> networkInterfaces = NetworkInterface.getNetworkInterfaces();
                if (!WINDOWS) {
                    ArrayList<NetworkInterface> netIFs = Collections.list(networkInterfaces);
                    Collections.reverse(netIFs);
                    networkInterfaces = Collections.enumeration(netIFs);
                }
                while (networkInterfaces.hasMoreElements()) {
                    NetworkInterface ni = networkInterfaces.nextElement();
                    Enumeration<InetAddress> addresses = ni.getInetAddresses();
                    while (addresses.hasMoreElements()) {
                        address = addresses.nextElement();
                        if (!NetworkAddressUtils.isValidAddress(address, timeout)) continue;
                        sLocalIP = address.getHostAddress();
                        return sLocalIP;
                    }
                }
                LOG.warn("Your hostname, " + InetAddress.getLocalHost().getHostName() + " resolves to" + " a loopback/non-reachable address: " + address.getHostAddress() + ", but we couldn't find any external IP address!");
            }
            sLocalIP = address.getHostAddress();
            return sLocalIP;
        }
        catch (IOException e) {
            LOG.error(e.getMessage(), (Throwable)e);
            throw Throwables.propagate((Throwable)e);
        }
    }

    private static boolean isValidAddress(InetAddress address, int timeout) throws IOException {
        return !address.isAnyLocalAddress() && !address.isLinkLocalAddress() && !address.isLoopbackAddress() && address.isReachable(timeout) && address instanceof Inet4Address;
    }

    public static TachyonURI replaceHostName(TachyonURI path) throws UnknownHostException {
        if (path == null) {
            return null;
        }
        if (path.hasAuthority() && path.getPort() != -1) {
            String authority = NetworkAddressUtils.resolveHostName(path.getHost());
            if (path.getPort() != -1) {
                authority = authority + ":" + path.getPort();
            }
            return new TachyonURI(path.getScheme(), authority, path.getPath());
        }
        return path;
    }

    public static String resolveHostName(String hostname) throws UnknownHostException {
        if (hostname == null || hostname.isEmpty()) {
            return null;
        }
        return InetAddress.getByName(hostname).getCanonicalHostName();
    }

    public static String getFqdnHost(InetSocketAddress addr) {
        return addr.getAddress().getCanonicalHostName();
    }

    public static String getFqdnHost(NetAddress addr) throws UnknownHostException {
        return NetworkAddressUtils.resolveHostName(addr.getHost());
    }

    public static int getThriftPort(TServerSocket thriftSocket) {
        return NetworkAddressUtils.getThriftSocket(thriftSocket).getLocalPort();
    }

    public static ServerSocket getThriftSocket(TServerSocket thriftSocket) {
        try {
            Field field = TServerSocket.class.getDeclaredField("serverSocket_");
            field.setAccessible(true);
            return (ServerSocket)field.get(thriftSocket);
        }
        catch (NoSuchFieldException e) {
            throw Throwables.propagate((Throwable)e);
        }
        catch (IllegalAccessException e) {
            throw Throwables.propagate((Throwable)e);
        }
    }

    public static InetSocketAddress parseInetSocketAddress(String address) throws IOException {
        if (address == null) {
            return null;
        }
        String[] strArr = address.split(":");
        if (strArr.length != 2) {
            throw new IOException("Invalid InetSocketAddress " + address);
        }
        return new InetSocketAddress(strArr[0], Integer.parseInt(strArr[1]));
    }

    public static enum ServiceType {
        MASTER_RPC("Tachyon Master RPC service", "tachyon.master.hostname", "tachyon.master.bind.host", "tachyon.master.port", 19998),
        MASTER_WEB("Tachyon Master Web service", "tachyon.master.web.hostname", "tachyon.master.web.bind.host", "tachyon.master.web.port", 19999),
        WORKER_RPC("Tachyon Worker RPC service", "tachyon.worker.hostname", "tachyon.worker.bind.host", "tachyon.worker.port", 29998),
        WORKER_DATA("Tachyon Worker data service", "tachyon.worker.data.hostname", "tachyon.worker.data.bind.host", "tachyon.worker.data.port", 29999),
        WORKER_WEB("Tachyon Worker Web service", "tachyon.worker.web.hostname", "tachyon.worker.web.bind.host", "tachyon.worker.web.port", 30000);

        private final String mServiceName;
        private final String mHostNameKey;
        private final String mBindHostKey;
        private final String mPortKey;
        private final int mDefaultPort;

        private ServiceType(String serviceName, String hostNameKey, String bindHostKey, String portKey, int defaultPort) {
            this.mServiceName = serviceName;
            this.mHostNameKey = hostNameKey;
            this.mBindHostKey = bindHostKey;
            this.mPortKey = portKey;
            this.mDefaultPort = defaultPort;
        }

        public String getServiceName() {
            return this.mServiceName;
        }

        public String getHostNameKey() {
            return this.mHostNameKey;
        }

        public String getBindHostKey() {
            return this.mBindHostKey;
        }

        public String getPortKey() {
            return this.mPortKey;
        }

        public int getDefaultPort() {
            return this.mDefaultPort;
        }
    }
}

