/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.gems.webwidget;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.portlet.ActionRequest;
import javax.portlet.ActionResponse;
import javax.portlet.PortletConfig;
import javax.portlet.PortletException;
import javax.portlet.PortletPreferences;
import javax.portlet.RenderRequest;
import javax.portlet.RenderResponse;
import javax.portlet.ResourceRequest;
import javax.portlet.ResourceResponse;
import javax.portlet.WindowState;

import org.apache.commons.collections.keyvalue.DefaultMapEntry;
import org.apache.commons.lang.StringUtils;
import org.apache.portals.bridges.velocity.GenericVelocityPortlet;

/**
 * WebWidgetIFramePortlet
 *
 * @version $Id: WebWidgetIFramePortlet.java 833030 2009-11-05 11:09:39Z woonsan $
 */
public class WebWidgetIFramePortlet extends GenericVelocityPortlet
{
    
    private static final String DEFAULT_WEB_WIDGET_SOURCE = 
        "<script src=\"http://www.gmodules.com/ig/ifr?url=http://www.google.com/ig/modules/wikipedia.xml&amp;up_mylang=en&amp;synd=open&amp;w=320&amp;h=46&amp;title=Wikipedia&amp;lang=en&amp;country=ALL&amp;border=%23ffffff%7C3px%2C1px+solid+%23999999&amp;output=js\"></script>";
    
    private Map<String, String> attributes = new HashMap<String, String>();
    
    private Map<String, String> maxAttributes = new HashMap<String, String>();
    
    @Override
    public void init(PortletConfig config) throws PortletException
    {
        super.init(config);
        
        attributes.put("TITLE", "");
        attributes.put("ALIGN", "BOTTOM");
        attributes.put("CLASS", "");
        attributes.put("FRAMEBORDER", "0");
        attributes.put("ID", "");
        attributes.put("MARGINHEIGHT", "0");
        attributes.put("MARGINWIDTH", "0");
        attributes.put("NAME", "");

        attributes.put("HEIGHT", "");
        attributes.put("WIDTH", "100%");
        attributes.put("SCROLLING", "NO");
        attributes.put("STYLE", "");

        attributes.put("WIDGETSOURCE", DEFAULT_WEB_WIDGET_SOURCE);
        
        maxAttributes.put("HEIGHT", "800");
        maxAttributes.put("WIDTH", "100%");
        maxAttributes.put("SCROLLING", "AUTO");
        maxAttributes.put("STYLE", "");
    }

    private String getAttributePreference(PortletPreferences prefs, String attribute)
    {
        return this.getMappedAttributePreference(prefs, attribute, attributes);
    }

    private String getMaxAttributePreference(PortletPreferences prefs, String attribute)
    {
        return this.getMappedAttributePreference(prefs, "MAX-" + attribute, maxAttributes);
    }

    private String getMappedAttributePreference(PortletPreferences prefs, String attribute, Map<String, String> map)
    {
        return prefs.getValue(attribute, map.get(attribute));
    }

    private void appendAttribute(PortletPreferences prefs, StringBuilder content, String attribute, Map<String, String> map)
    {
        String value;

        if (map == maxAttributes)
            value = getMaxAttributePreference(prefs, attribute);
        else
            value = getAttributePreference(prefs, attribute);

        if (value == null || value.length() == 0) { return; }
        content.append(" ").append(attribute).append("=\"").append(value).append("\"");
    }

    private void appendAttribute(PortletPreferences prefs, StringBuilder content, String attribute)
    {
        appendAttribute(prefs, content, attribute, attributes);
    }

    private void appendMaxAttribute(PortletPreferences prefs, StringBuilder content, String attribute)
    {
        appendAttribute(prefs, content, attribute, maxAttributes);
    }
    
    @Override
    public void doView(RenderRequest request, RenderResponse response) throws PortletException, IOException
    {
        String viewPage = (String) request.getAttribute(PARAM_VIEW_PAGE);
        
        if (viewPage != null)
        {
            super.doView(request, response);
        }
        else
        {
            doIFrame(request, response);
        }
    }
    
    @Override
    public void setupPreferencesEdit(RenderRequest request, RenderResponse response)
    {
        PortletPreferences prefs = request.getPreferences();
        
        String editablePrefNames = getAttributePreference(prefs, "EDITABLEPREFS");
        
        if (StringUtils.isBlank(editablePrefNames))
        {
            super.setupPreferencesEdit(request, response);
        }
        else
        {
            List<DefaultMapEntry> prefEntryList = new ArrayList<DefaultMapEntry>();
            Map<String, String> prefsMap = new HashMap<String, String>();
            String [] prefNames = StringUtils.split(editablePrefNames, ",");
            String [] emptyValues = {};
            
            for (String prefName : prefNames)
            {
                prefName = prefName.trim();
                String [] prefValues = prefs.getValues(prefName, emptyValues);
                prefsMap.put(prefName, prefValues.length == 0 ? "" : prefValues[0]);
                prefEntryList.add(new DefaultMapEntry(prefName, prefValues));
            }
            
            getContext(request, response).put("prefs", prefEntryList.iterator());
            getContext(request, response).put("prefsMap", prefsMap);
        }
    }
    
    @Override
    public void doEdit(RenderRequest request, RenderResponse response) throws PortletException, IOException
    {
        response.setContentType("text/html");
        doPreferencesEdit(request, response);
    }
    
    @Override
    public void doHelp(RenderRequest request, RenderResponse response) throws PortletException, IOException
    {
        super.doHelp(request, response);
    }
    
    @Override
    public void doCustom(RenderRequest request, RenderResponse response) throws PortletException, IOException
    {
        super.doCustom(request, response);
    }
    
    /**
     * Render IFRAME content
     */
    protected void doIFrame(RenderRequest request, RenderResponse response) throws IOException
    {
        PortletPreferences prefs = request.getPreferences();
        // generate HTML IFRAME content
        StringBuilder content = new StringBuilder(256);
        
        // fix JS2-349
        content.append("<TABLE CLASS='iframePortletTableContainer' WIDTH='100%'><TBODY CLASS='iframePortletTbodyContainer'><TR><TD>");

        content.append("<IFRAME");
        
        // IFrame source can be a portlet resource url or a simple direct url.
        // In the former case, the serveResource() method will write the widget source
        // which are normally starting with '<' for "<script ...", "<object ...", "<iframe ..." or "<embed ...".
        // (the resource url is invoked from the generated IFrame.)
        // In the latter case, the direct url will serve the widget directly.
        String source = response.createResourceURL().toString();
        String widgetSource = getAttributePreference(request.getPreferences(), "WIDGETSOURCE");
        if (!StringUtils.isBlank(widgetSource))
        {
            String trimmedWidgetSource = widgetSource.trim();
            // if the widget source does not start with '<', then it is regarded as a direct source url of this iframe.
            if (!trimmedWidgetSource.startsWith("<"))
            {
                source = trimmedWidgetSource;
            }
        }
        
        content.append(" ").append("SRC").append("=\"").append(source).append("\"");

        appendAttribute(prefs, content, "ALIGN");
        appendAttribute(prefs, content, "CLASS");
        appendAttribute(prefs, content, "FRAMEBORDER");
        appendAttribute(prefs, content, "ID");
        appendAttribute(prefs, content, "MARGINHEIGHT");
        appendAttribute(prefs, content, "MARGINWIDTH");
        appendAttribute(prefs, content, "NAME");
        
        if (request.getWindowState().equals(WindowState.MAXIMIZED))
        {
            appendMaxAttribute(prefs, content, "HEIGHT");
            appendMaxAttribute(prefs, content, "WIDTH");
            appendMaxAttribute(prefs, content, "SCROLLING");
            appendMaxAttribute(prefs, content, "STYLE");
        }
        else
        {
            appendAttribute(prefs, content, "HEIGHT");
            appendAttribute(prefs, content, "WIDTH");
            appendAttribute(prefs, content, "SCROLLING");
            appendAttribute(prefs, content, "STYLE");
        }
        
        content.append(">");
        content.append("<P STYLE=\"textAlign:center\"><A HREF=\"").append(source).append("\">").append(source).append(
                "</A></P>");
        content.append("</IFRAME>");

        // end fix JS2-349
        content.append("</TD></TR></TBODY></TABLE>");
        
        // set required content type and write HTML IFRAME content
        response.setContentType("text/html");
        response.getWriter().print(content.toString());
    }
    
    /**
     * Save the prefs
     */
    @Override
    public void processAction(ActionRequest request, ActionResponse actionResponse) throws PortletException,
            IOException
    {
        processPreferencesAction(request, actionResponse);
    }
    
    @Override
    protected String getTitle(RenderRequest request) 
    {
        String title = getAttributePreference(request.getPreferences(), "TITLE");
        
        if (!StringUtils.isEmpty(title))
        {
            return title;
        }
        else
        {
            return super.getTitle(request);
        }
    }
    
    @Override
    public void serveResource(ResourceRequest request, ResourceResponse response) throws PortletException, IOException 
    {
        String widgetSource = getAttributePreference(request.getPreferences(), "WIDGETSOURCE");
        
        if (!StringUtils.isEmpty(widgetSource))
        {
            response.setContentType("text/html");
            response.getWriter().print(widgetSource);
        }
    }
    
}
