/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.gems.util;

import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

/**
 * JSON utility
 * 
 * @author <a href="mailto:smilek@apache.org">Steve Milek</a>
 * @version $Id: JSONUtils.java 763984 2009-04-10 17:35:20Z woonsan $
 */
public class JSONUtils
{
    public static StringBuffer makeJSONObject(Map objectMap, boolean whenEmptyReturnNewObject)
    {
        return makeJSONObject(null, new Map[] { objectMap }, whenEmptyReturnNewObject);
    }

    public static StringBuffer makeJSONObject(Map[] objectMaps, boolean whenEmptyReturnNewObject)
    {
        return makeJSONObject(null, objectMaps, whenEmptyReturnNewObject);
    }

    public static StringBuffer makeJSONObject(StringBuffer jsonBuffer, Map objectMap, boolean whenEmptyReturnNewObject)
    {
        return makeJSONObject(jsonBuffer, new Map[] { objectMap }, whenEmptyReturnNewObject);
    }

    public static StringBuffer makeJSONObject(StringBuffer jsonBuffer, Map[] objectMaps, boolean whenEmptyReturnNewObject)
    {
        if (jsonBuffer == null)
            jsonBuffer = new StringBuffer();
        int added = 0;
        int objMapsLen = (objectMaps == null ? 0 : objectMaps.length);
        if (objMapsLen > 0)
        {
            for (int i = 0; i < objMapsLen; i++)
            {
                Map objectMap = objectMaps[i];
                if (objectMap != null && objectMap.size() > 0)
                {
                    if (added == 0)
                        jsonBuffer.append("{");
                    Map.Entry objEntry;
                    Object objKey, objVal;
                    Iterator objMapIter = objectMap.entrySet().iterator();
                    while (objMapIter.hasNext())
                    {
                        objEntry = (Map.Entry) objMapIter.next();
                        objKey = objEntry.getKey();
                        if (objKey != null)
                        {
                            if (added > 0)
                                jsonBuffer.append(", ");
                            jsonBuffer.append("\"").append(objKey.toString()).append("\":");
                            objVal = objEntry.getValue();
                            if (objVal == null)
                                objVal = "";
                            jsonBuffer.append("\"").append(objVal.toString()).append("\"");
                            added++;
                        }
                    }
                }
            }
        }
        if (added > 0)
        {
            jsonBuffer.append("}");
        }
        else if (whenEmptyReturnNewObject)
        {
            jsonBuffer.append("{}");
        }
        else
        {
            return null;
        }
        return jsonBuffer;
    }

    public static String makeJSONStringArray(Collection stringList)
    {
        return makeJSONStringArray(stringList, null);
    }

    public static String makeJSONStringArray(Collection stringList, List compiledUniqueValues)
    {
        if (stringList != null && stringList.size() > 0)
        {
            StringBuffer stringListContent = new StringBuffer();
            Iterator stringListIter = stringList.iterator();
            while (stringListIter.hasNext())
            {
                String value = (String) stringListIter.next();
                if (value != null && value.length() > 0)
                {
                    if (stringListContent.length() > 0)
                    {
                        stringListContent.append(", ");
                    }
                    else
                    {
                        stringListContent.append("[ ");
                    }
                    stringListContent.append("\"").append(value).append("\"");
                    if (compiledUniqueValues != null)
                    {
                        if (!compiledUniqueValues.contains(value))
                        {
                            compiledUniqueValues.add(value);
                        }
                    }
                }
            }
            if (stringListContent.length() > 0)
            {
                stringListContent.append(" ]");
                return stringListContent.toString();
            }
        }
        return null;
    }

    public static String makeJSONInteger(Object source, boolean quote)
    {
        String sourceStr = ((source == null) ? (String) null : source.toString());
        if (sourceStr != null)
        {
            try
            {
                Integer.parseInt(sourceStr);
                if (quote)
                {
                    sourceStr = "\"" + sourceStr + "\"";
                }
            }
            catch (NumberFormatException nex)
            {
                sourceStr = null;
            }
        }
        return sourceStr;
    }

    public static String makeJSONBoolean(Object source)
    {
        String boolStr = ((source == null) ? (String) null : source.toString());
        if (boolStr != null && (!boolStr.equals("false")) && (!boolStr.equals("true")))
        {
            boolStr = null;
        }
        return boolStr;
    }
}
