/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.applications.webcontent.rewriter;

import java.util.Iterator;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import org.apache.portals.applications.webcontent.rewriter.rules.Attribute;
import org.apache.portals.applications.webcontent.rewriter.rules.Rule;
import org.apache.portals.applications.webcontent.rewriter.rules.Ruleset;
import org.apache.portals.applications.webcontent.rewriter.rules.Tag;


/**
 * RuleBasedRewriter
 *
 * @author <a href="mailto:taylor@apache.org">David Sean Taylor</a>
 * @version $Id: RulesetRewriterImpl.java 891414 2009-12-16 20:19:02Z rwatler $
 */
public class RulesetRewriterImpl extends BasicRewriter implements RulesetRewriter
{
    protected final static Logger log = LoggerFactory.getLogger(RulesetRewriterImpl.class);
    
    private Ruleset ruleset = null;
    private boolean removeComments = false;

    public boolean shouldStripTag(String tagid)
    {        
        if (null == ruleset)
        {
            return false;
        }
        
        Tag tag = ruleset.getTag(tagid.toUpperCase());
        if (null == tag)
        {
            return false;
        }
        return tag.getStrip();        
    }
            
    public boolean shouldRemoveTag(String tagid)
    {        
        if (null == ruleset)
        {
            return false;
        }
        
        Tag tag = ruleset.getTag(tagid.toUpperCase());
        if (null == tag)
        {
            return false;
        }
        return tag.getRemove();
    }

    public void setRuleset(Ruleset ruleset)
    {
        this.ruleset = ruleset;
    }
    
    public Ruleset getRuleset()
    {
        return this.ruleset;
    }

    public boolean shouldRemoveComments()
    {
        if (null == ruleset)
        {
            return false;
        }
        
        return ruleset.getRemoveComments();                
    }

    public void enterConvertTagEvent(String tagid, MutableAttributes attributes)
    {
        if (null == ruleset)
        {
            return;
        }
        
         Tag tag = ruleset.getTag(tagid.toUpperCase());
        if (null == tag)
        {
             return;
        }

        Iterator attribRules = tag.getAttributes().iterator();
        while (attribRules.hasNext())
        {
            Attribute attribute = (Attribute)attribRules.next();
            String name = attribute.getId();
            String value = attributes.getValue(name);
 
            if (value != null) // && name.equalsIgnoreCase(attribute.getId()))
            {
                Rule rule = attribute.getRule();
                if (null == rule)
                {
                    continue;
                }
                
                if (!rule.shouldRewrite(value))
                {
                    continue;
                }                                        
                
                String rewritten = rewriteUrl(value, tag.getId(), name, attributes);
                if (null != rewritten) // return null indicates "don't rewrite" 
                {
                    if (rule.getSuffix() != null)
                    {
                        rewritten = rewritten.concat(rule.getSuffix());
                    }
                    
                    attributes.addAttribute(name, rewritten);
                                        
                    if (rule.getPopup())
                    {
                        attributes.addAttribute("TARGET", "_BLANK");                        
                    }
                }
            }            
        }
    }
    
    /**
     * rewriteURL
     * 
     * @param url
     * @param tag
     * @param attribute
     * @param otherAttributes
     * @return the modified url which is a portlet action
     * 
     * Rewrites all urls HREFS with a portlet action
     */
    public String rewriteUrl(String url, String tag, String attribute, MutableAttributes otherAttributes)
    {
        return getBaseRelativeUrl(url);
    }

    public String enterConvertTextEvent(String tagid, String text)
    {
        return rewriteText(tagid, text);
    }
}
