/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.applications.webcontent2.rewriter.impl;

import java.io.BufferedWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.Reader;
import java.io.Writer;

import org.apache.commons.io.IOUtils;
import org.apache.commons.io.LineIterator;
import org.apache.portals.applications.webcontent2.rewriter.ContentRewriter;
import org.apache.portals.applications.webcontent2.rewriter.ContentRewritingContext;
import org.apache.portals.applications.webcontent2.rewriter.ContentRewritingException;
import org.apache.portals.applications.webcontent2.rewriter.Sink;
import org.apache.portals.applications.webcontent2.rewriter.Source;

/**
 * Abstract text line based content rewriter base implementation.
 */
public abstract class AbstractTextLineContentRewriter implements ContentRewriter
{

    /**
     * {@inheritDoc}
     * <p>
     * This method basically gets a {@link java.io.Reader} from the <code>source</code>
     * and reads each text line from the reader. And it invokes {@link #rewriteLine(String, ContentRewritingContext)} method
     * to translate the line and write the translated line to the writer retrieved from the <code>sink</code>.
     * </p>
     * <p>
     * If both <code>source</code> and <code>sink<code> are all text-based, and if it is possible to translate line by line,
     * then you can simply create a derived class from this class with implementing {@link #rewriteLine(String, ContentRewritingContext)} method.
     * </p>
     */
    public void rewrite(Source source, Sink sink, ContentRewritingContext context) throws ContentRewritingException, IOException
    {
        Reader reader = null;
        Writer writer = null;
        BufferedWriter bw = null;
        PrintWriter out = null;

        try
        {
            reader = source.getReader();
            writer = sink.getWriter();
            bw = new BufferedWriter(writer);
            out = new PrintWriter(bw);
            String line = null;

            for (LineIterator lineIt = IOUtils.lineIterator(reader); lineIt.hasNext();)
            {
                line = rewriteLine(lineIt.nextLine(), context);
                out.println(line);
            }

            out.flush();
        }
        finally
        {
            IOUtils.closeQuietly(reader);
            IOUtils.closeQuietly(out);
            IOUtils.closeQuietly(bw);
            IOUtils.closeQuietly(writer);
        }
    }

    /**
     * Translates the given <code>line</code> and returns the translated string.
     * <p>
     * For example, suppose the <code>source</code> contains the following lines:
     * <pre>
     * ${message.greeting}
     * Copyright ${copyright.holder} ${copyright.years}
     * </pre>
     * </p>
     * <p>
     * Then, one implementation of this method might want to return "Hello, World!" for the first text line,
     * and "Copyright Apache Software Foundation 2014" for instance to let the <code>sink</code> have the following:
     * <pre>
     * Hello, World!
     * Copyright Apache Software Foundation 2014
     * </pre>
     * </p>
     * <p>
     * Even better, for example, if the given <code>context</code> contains all those variables (i.e, ${copyright.holder} and ${copyright.years}),
     * then the implementation of this method can resolve those variables dynamically and translates them based on the variables given
     * by the runtime.
     * </p>
     * 
     * @param line a text line from the <code>source</code>
     * @param context the content rewriting context given by the runtime
     * @return a translated line
     * @throws ContentRewritingException
     * @throws IOException
     */
    protected abstract String rewriteLine(String line, ContentRewritingContext context) throws ContentRewritingException, IOException;

}
