/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.applications.webcontent2.rewriter;

import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Reader;
import java.io.Writer;

import org.apache.commons.io.output.TeeOutputStream;
import org.apache.commons.lang.StringUtils;
import org.apache.portals.applications.webcontent2.rewriter.htmlcleaner.DefaultSerializerFactory;
import org.apache.portals.applications.webcontent2.rewriter.htmlcleaner.HtmlCleanerContentRewriter;
import org.apache.portals.applications.webcontent2.rewriter.impl.SimpleContentRewritingContext;
import org.htmlcleaner.HtmlNode;
import org.htmlcleaner.SimpleHtmlSerializer;
import org.htmlcleaner.TagNode;
import org.htmlcleaner.TagNodeVisitor;
import org.junit.Before;
import org.junit.Test;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HtmlCleanerContentRewriterTest
{

    private static Logger log = LoggerFactory.getLogger(HtmlCleanerContentRewriterTest.class);

    private static String [] HTMLCLEANER_TRNSFORM_INFO = 
                {
                    "cfoutput",
                    "c:block=div,false",
                    "font=span,true",
                    "font.size",
                    "font.face",
                    "font.style=${style};font-family=${face};font-size=${size};"
                };

    private Source source;
    private Sink sink;
    private ByteArrayOutputStream sinkTeeOut;
    private HtmlCleanerContentRewriter contentRewriter;

    @Before
    public void before() 
    {
        source = new Source() 
        {
            public InputStream getInputStream() throws IOException
            {
                return HtmlCleanerContentRewriterTest.class.getResourceAsStream("guidelines.html");
            }
            public Reader getReader() throws IOException
            {
                return new InputStreamReader(getInputStream());
            }
        };

        sinkTeeOut = new ByteArrayOutputStream();
        sink = new Sink() 
        {
            public OutputStream getOutputStream() throws IOException
            {
                return new TeeOutputStream(System.out, sinkTeeOut);
            }
            public Writer getWriter() throws IOException
            {
                return new OutputStreamWriter(getOutputStream());
            }
        };

        DefaultSerializerFactory serializerFactory = new DefaultSerializerFactory();
        serializerFactory.setSerializerClass(SimpleHtmlSerializer.class);

        contentRewriter = new HtmlCleanerContentRewriter();
        contentRewriter.setSerializerFactory(serializerFactory);
        contentRewriter.setXpathExpression("body");
        contentRewriter.setInnerHtmlOnly(true);
        contentRewriter.setCleanerTransformationStringArray(HTMLCLEANER_TRNSFORM_INFO);

        final String siteUrl = "http://www.example.com/";
        contentRewriter.addTagNodeVisitor(new TagNodeVisitor() 
        {
            public boolean visit(TagNode tagNode, HtmlNode htmlNode) 
            {
                if (htmlNode instanceof TagNode) 
                {
                    TagNode tag = (TagNode) htmlNode;
                    String tagName = tag.getName();

                    if ("a".equals(tagName) || "link".equals(tagName)) 
                    {
                        String href = tag.getAttributeByName("href");

                        if (href != null) 
                        {
                            tag.addAttribute("href", siteUrl + StringUtils.removeStart(href, "./"));
                        }
                    }

                    if ("a".equals(tagName) && "Project_Guidelines".equals(tag.getAttributeByName("name"))) 
                    {
                        tag.removeFromTree();
                    }
                }

                // tells visitor to continue traversing the DOM tree
                return true;
            }
        });
    }

    @Test
    public void testRewriter() throws Exception {
        ContentRewritingContext rewritingContext = new SimpleContentRewritingContext();
        contentRewriter.rewrite(source, sink, rewritingContext);
        String output = sinkTeeOut.toString();
        log.debug("OUTPUT: {}", output);

        assertFalse(output.contains("name=\"Project_Guidelines\""));
        assertTrue(output.contains("<a href=\"http://www.example.com/roles.html\">Roles and Responsibilities</a>"));
        assertTrue(output.contains("general@portals <a href=\"http://www.example.com/mail-lists.html#general\">"));

        assertFalse(output.contains("<cfoutput>"));
        assertTrue(output.contains("Yin and yang describe the polar effects of phenomena."));
        assertFalse(output.contains("<c:block"));
        assertFalse(output.contains("<font"));
        assertTrue(output.contains("<span id=\"f21\" style=\"color: red;font-family=Arial;font-size=12;\">"));
    }

}
