/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.applications.webcontent2.proxy.builder;

import java.util.ArrayList;
import java.util.List;

import org.apache.commons.chain.Chain;
import org.apache.commons.chain.Command;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.portals.applications.webcontent2.proxy.HttpClientContextBuilder;
import org.apache.portals.applications.webcontent2.proxy.ProxyMappingRegistry;
import org.apache.portals.applications.webcontent2.proxy.impl.ProxyProcessingChain;

/**
 * Abstract {@link ProxyProcessingChain} builder class providing basic functionality
 * to add or remove internal {@link Command}s.
 * <p>
 * An inherited implementation can simply implement {@link #initializeCommands(ProxyMappingRegistry, HttpClientBuilder, HttpClientContextBuilder)}
 * in order to add or remove {@link Command}s before {@link #build(ProxyMappingRegistry, HttpClientBuilder, HttpClientContextBuilder)} gets called.
 * </p>
 */
public abstract class AbstractProxyProcessingChainBuilder implements ProxyProcessingChainBuilder
{

    /**
     * The root reverse proxy processing {@link Chain}.
     */
    private ProxyProcessingChain chain;

    /**
     * The internal reverse proxy processing {@link Command}s.
     */
    private List<Command> commands;

    /**
     * Zero-argument default constructor.
     */
    public AbstractProxyProcessingChainBuilder()
    {
    }

    /**
     * {@inheritDoc}
     */
    public ProxyProcessingChain build(ProxyMappingRegistry proxyMappingRegistry, HttpClientBuilder httpClientBuilder,
                                      HttpClientContextBuilder httpClientContextBuilder)
    {
        if (chain == null)
        {
            chain = new ProxyProcessingChain();
        }

        initializeCommands(proxyMappingRegistry, httpClientBuilder, httpClientContextBuilder);

        if (commands != null)
        {
            for (Command command : commands)
            {
                chain.addCommand(command);
            }
        }

        return chain;
    }

    /**
     * Adds the given <code>command</code> to the internal {@link Command} list
     * which will be added to the root {@link Chain} on {@link #build(ProxyMappingRegistry, HttpClientBuilder, HttpClientContextBuilder)} call.
     * @param command
     */
    public void addCommand(Command command)
    {
        if (commands == null)
        {
            commands = new ArrayList<Command>();
        }

        commands.add(command);
    }

    /**
     * Removes the given <code>command</code> instance from the internal {@link Command} list.
     * @param command
     */
    public void removeCommand(Command command)
    {
        if (commands != null)
        {
            commands.remove(command);
        }
    }

    /**
     * Removes all the commands from the internal {@link Command} list.
     */
    public void removeAllCommands()
    {
        if (commands != null)
        {
            commands.clear();
        }
    }

    /**
     * Initializes all the internal commands to be used by the root {@link Chain} on
     * {@link #build(ProxyMappingRegistry, HttpClientBuilder, HttpClientContextBuilder)} call.
     * <p>
     * An implementation may invoke {@link #addCommand(Command)}, {@link #removeCommand(Command)}
     * and {@link #removeAllCommands()} during the initialization.
     * </p>
     * 
     * @param proxyMappingRegistry
     * @param httpClientBuilder
     * @param httpClientContextBuilder
     */
    protected abstract void initializeCommands(ProxyMappingRegistry proxyMappingRegistry, HttpClientBuilder httpClientBuilder,
                                               HttpClientContextBuilder httpClientContextBuilder);
}
