/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.applications.webcontent2.proxy.impl;

import org.apache.commons.lang.StringUtils;
import org.apache.portals.applications.webcontent2.proxy.URICleaner;

/**
 * Default simple {@link URICleaner} implementation.
 * <p>
 * This simply finds any strings given by {@link #setSearchList(String[])}
 * from a URI string and replace each string in the URI string
 * by the corresponding strings given by {@link #setReplacementList(String[])}.
 * </p>
 * <p>
 * By default, it replaces a space (" ") by "%20".
 * </p>
 */
public class DefaultURICleaner implements URICleaner
{
    private String[] searchList = { " " };
    private String[] replacementList = { "%20" };

    /**
     * Zero-argument default constructor.
     */
    public DefaultURICleaner()
    {
    }

    /**
     * Sets the search string array to find from URI strings.
     * @param searchList
     */
    public void setSearchList(String[] searchList)
    {
        this.searchList = null;

        if (searchList != null) {
            this.searchList = new String[searchList.length];
            System.arraycopy(searchList, 0, this.searchList, 0, searchList.length);
        }
    }

    /**
     * Sets the replace string array to replace URI strings with.
     * @param replacementList
     */
    public void setReplacementList(String[] replacementList)
    {
        this.replacementList = null;

        if (replacementList != null) {
            this.replacementList = new String[replacementList.length];
            System.arraycopy(replacementList, 0, this.replacementList, 0, replacementList.length);
        }
    }

    /**
     * {@inheritDoc}
     */
    public String clean(String uri)
    {
        return StringUtils.replaceEach(uri, searchList, replacementList);
    }
}
