/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.applications.webcontent2.proxy.impl;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertSame;

import org.apache.commons.chain.Command;
import org.apache.portals.applications.webcontent2.proxy.command.CleanupCommand;
import org.apache.portals.applications.webcontent2.proxy.command.ExecuteHttpClientCommand;
import org.apache.portals.applications.webcontent2.proxy.command.InitializationCommand;
import org.apache.portals.applications.webcontent2.proxy.command.SerializeHttpEntityContentCommand;
import org.junit.Test;

public class ProxyProcessingChainIndexingAdditionRemovalTest
{

    @Test
    public void testCommandIndexing() throws Exception {
        Command [] commands =  new Command [] {
            new InitializationCommand(), 
            new ExecuteHttpClientCommand(), 
            new CleanupCommand()
        };

        ProxyProcessingChain chain = new ProxyProcessingChain();

        for (Command command : commands) {
            chain.addCommand(command);
        }

        assertEquals(3, chain.getCommandCount());

        for (int i = 0; i < chain.getCommandCount(); i++) {
            assertSame(commands[i], chain.getCommand(i));
            assertSame(i, chain.getCommandIndex(commands[i]));
        }

        assertEquals(0, chain.getCommandIndex(Command.class));
        assertEquals(0, chain.getCommandIndex(InitializationCommand.class));
        assertEquals(1, chain.getCommandIndex(ExecuteHttpClientCommand.class));
        assertEquals(2, chain.getCommandIndex(CleanupCommand.class));
        assertEquals(-1, chain.getCommandIndex(SerializeHttpEntityContentCommand.class));
    }

    @Test
    public void testCommandAdditionRemoval() throws Exception {
        ProxyProcessingChain chain = new ProxyProcessingChain();

        assertEquals(0, chain.getCommandCount());

        Command serializeHttpEntityContentCommand = new SerializeHttpEntityContentCommand();
        chain.addCommand(serializeHttpEntityContentCommand);
        assertEquals(1, chain.getCommandCount());
        assertSame(serializeHttpEntityContentCommand, chain.getCommand(0));
        assertSame(0, chain.getCommandIndex(serializeHttpEntityContentCommand));

        Command initializationCommand = new InitializationCommand();
        chain.addCommand(0, initializationCommand);
        assertEquals(2, chain.getCommandCount());
        assertSame(initializationCommand, chain.getCommand(0));
        assertSame(0, chain.getCommandIndex(initializationCommand));

        Command executeHttpClientCommand = new ExecuteHttpClientCommand();
        chain.addCommand(chain.getCommandIndex(serializeHttpEntityContentCommand), executeHttpClientCommand);
        assertEquals(3, chain.getCommandCount());
        assertSame(executeHttpClientCommand, chain.getCommand(1));
        assertSame(1, chain.getCommandIndex(executeHttpClientCommand));

        Command cleanupCommand = new CleanupCommand();
        chain.addCommand(chain.getCommandCount(), cleanupCommand);
        assertEquals(4, chain.getCommandCount());
        assertSame(cleanupCommand, chain.getCommand(3));
        assertSame(3, chain.getCommandIndex(cleanupCommand));

        assertSame(initializationCommand, chain.getCommand(0));
        assertSame(0, chain.getCommandIndex(initializationCommand));
        assertSame(executeHttpClientCommand, chain.getCommand(1));
        assertSame(1, chain.getCommandIndex(executeHttpClientCommand));
        assertSame(serializeHttpEntityContentCommand, chain.getCommand(2));
        assertSame(2, chain.getCommandIndex(serializeHttpEntityContentCommand));
        assertSame(cleanupCommand, chain.getCommand(3));
        assertSame(3, chain.getCommandIndex(cleanupCommand));
    }
}
