/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.applications.webcontent2.proxy.impl;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

import java.net.URI;

import org.junit.Test;

public class RegexProxyMappingTest
{

    @Test
    public void testRegexMappings() throws Exception
    {
        RegexProxyMapping mapping = new RegexProxyMapping();

        mapping.setLocalPattern("^/apache/(\\w+)/(.*)$");
        mapping.setRemoteReplace("http://$1.apache.org/$2");
        mapping.setRemotePattern("^http://(\\w+)\\.apache\\.org/(.*)$");
        mapping.setLocalReplace("/apache/$1/$2");

        assertTrue(mapping.matchesLocal("/apache/portals/"));
        assertTrue(mapping.matchesLocal("/apache/portals/mission.html"));

        assertTrue(mapping.matchesRemote(URI.create("http://portals.apache.org/")));
        assertTrue(mapping.matchesRemote(URI.create("http://portals.apache.org/mission.html")));

        assertEquals("http://portals.apache.org/", mapping.resolveRemoteFromLocal("/apache/portals/"));
        assertEquals("http://portals.apache.org/mission.html", mapping.resolveRemoteFromLocal("/apache/portals/mission.html"));

        assertEquals("/apache/portals/", mapping.resolveLocalFromRemote(URI.create("http://portals.apache.org/")));
        assertEquals("/apache/portals/mission.html", mapping.resolveLocalFromRemote(URI.create("http://portals.apache.org/mission.html")));
    }

    @Test
    public void testRegexMappingsWithoutScheme() throws Exception
    {
        RegexProxyMapping mapping = new RegexProxyMapping();

        mapping.setLocalPattern("^/apache/(\\w+)/(.*)$");
        mapping.setRemoteReplace("//$1.apache.org/$2");
        mapping.setRemotePattern("^https?://(\\w+)\\.apache\\.org/(.*)$");
        mapping.setLocalReplace("/apache/$1/$2");

        assertTrue(mapping.matchesLocal("/apache/portals/"));
        assertTrue(mapping.matchesLocal("/apache/portals/mission.html"));

        assertTrue(mapping.matchesRemote(URI.create("http://portals.apache.org/")));
        assertTrue(mapping.matchesRemote(URI.create("http://portals.apache.org/mission.html")));

        assertEquals("//portals.apache.org/", mapping.resolveRemoteFromLocal("/apache/portals/"));
        assertEquals("//portals.apache.org/mission.html", mapping.resolveRemoteFromLocal("/apache/portals/mission.html"));

        assertEquals("/apache/portals/", mapping.resolveLocalFromRemote(URI.create("http://portals.apache.org/")));
        assertEquals("/apache/portals/mission.html", mapping.resolveLocalFromRemote(URI.create("http://portals.apache.org/mission.html")));
    }
}
