/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.bridges.script;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;

import javax.script.ScriptEngine;
import javax.script.ScriptEngineManager;
import javax.script.ScriptException;

/**
 * ScriptEngineUtils
 * 
 * @author <a href="mailto:woonsan@apache.org">Woonsan Ko</a>
 * @version $Id: ScriptEngineUtils.java 936949 2010-04-22 16:44:40Z woonsan $
 */
public class ScriptEngineUtils
{
    private ScriptEngineUtils()
    {
        
    }
    
    /**
     * Creates script engine based on scriptEngineName and scriptSource.
     * <P>
     * When scriptEngineName is not null, it creates a script engine by the scriptEngineName.
     * When scriptEngineName is not null, it creates a script engine by the mimeType of the scriptSource if available.
     * Otherwise, it creates a script engine by the script extension.
     * </P>
     * 
     * @param scriptEngineName
     * @param scriptSource
     * @return
     * @throws ScriptException
     */
    public static ScriptEngine createScriptEngine(final String scriptEngineName, final ScriptSource scriptSource) throws ScriptException
    {
        ScriptEngine scriptEngine = null;
        ScriptEngineManager manager = new ScriptEngineManager();
        
        if (scriptEngineName != null)
        {
            scriptEngine = manager.getEngineByName(scriptEngineName);
        }
        else
        {
            if (scriptSource.getMimeType() != null)
            {
                scriptEngine = manager.getEngineByMimeType(scriptSource.getMimeType());
            }
            
            if (scriptEngine == null)
            {
                if (scriptSource.getExtension() != null)
                {
                    scriptEngine = manager.getEngineByExtension(scriptSource.getExtension());
                }
            }
        }
        
        if (scriptEngine == null)
        {
            throw new ScriptException("Cannot create script engine. { scriptEngineName: " + scriptEngineName + ", scriptSource: " + scriptSource);
        }
        
        return scriptEngine;
    }
    
    /**
     * Evaluates the scriptSource through the scriptEngine and returns the evaluated object.
     * <P>
     * If the evalKey is null, it returns the last evaluated object.
     * Otherwise, it retrieves the object by the evalKey from the scriptEngine.
     * </P>
     * 
     * @param scriptEngine
     * @param scriptSource
     * @param evalKey
     * @param fallback
     * @return
     * @throws ScriptException
     * @throws IOException
     */
    public static Object evaluateScript(final ScriptEngine scriptEngine, final ScriptSource scriptSource, final String evalKey, final boolean fallback) throws ScriptException, IOException
    {
        Object evaledObject = null;
        
        InputStream input = null;
        InputStreamReader isr = null;
        BufferedReader br = null;
        
        try
        {
            input = scriptSource.getInputStream();
            
            if (scriptSource.getCharacterEncoding() != null)
            {
                isr = new InputStreamReader(input, scriptSource.getCharacterEncoding());
            }
            else
            {
                isr = new InputStreamReader(input);
            }
            
            br = new BufferedReader(isr);
            
            Object ret = scriptEngine.eval(br);
            
            if (evalKey != null)
            {
                evaledObject = scriptEngine.get(evalKey);
                
                if (evaledObject == null && fallback)
                {
                    evaledObject = ret;
                }
            }
            
            if (evaledObject == null)
            {
                evaledObject = ret;
            }
            
            if (evaledObject == null)
            {
                throw new ScriptException("The evaluated object is null. (" + evalKey + ")");
            }
        }
        finally
        {
            if (br != null)
            {
                try
                {
                    br.close();
                }
                catch (Exception ignore) { }
            }
            if (isr != null)
            {
                try
                {
                    isr.close();
                }
                catch (Exception ignore) { }
            }
            if (input != null)
            {
                try
                {
                    input.close();
                }
                catch (Exception ignore) { }
            }
        }
        
        return evaledObject;
    }
    
    /**
     * Checks if the script source has been modified after the specified time.
     * 
     * @param scriptSource
     * @param after
     * @return
     */
    public static boolean isScriptModified(final ScriptSource scriptSource, final long after)
    {
        return (scriptSource.lastModified() > after);
    }
}
