/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.bridges.script;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;

/**
 * SimpleScriptSource
 * <P>
 * Simple ScriptSource implementation to load scripts from a file system, classpath resource. 
 * </P>
 * 
 * @author <a href="mailto:woonsan@apache.org">Woonsan Ko</a>
 * @version $Id: SimpleScriptSource.java 937248 2010-04-23 11:06:13Z woonsan $
 */
public class SimpleScriptSource implements ScriptSource
{
    
    /** The file name part of the script source. */
    private String name;
    
    /** The file extension of the script source without dot prefix. */
    private String extension;
    
    /** The mime type of this script. */
    private String mimeType;
    
    /** The character encoding of the content of this script. */
    private String characterEncoding;
    
    /** Phyical file of this script source if acessible. */
    private File scriptFile;
    
    /** The script content bytes if the script source file is not accessible. */
    private byte [] scriptBytes;
    
    public SimpleScriptSource(String name, String extension, String characterEncoding, File scriptFile, byte [] scriptContent) throws IOException
    {
        this.name = name;
        this.extension = extension;
        this.characterEncoding = characterEncoding;
        this.scriptFile = scriptFile;
        this.scriptBytes = scriptContent;
    }
    
    public String getName()
    {
        return name;
    }
    
    public String getExtension()
    {
        return extension;
    }
    
    public String getCharacterEncoding()
    {
        return characterEncoding;
    }
    
    public void setMimeType(String mimeType)
    {
        this.mimeType = mimeType;
    }
    
    public String getMimeType()
    {
        return mimeType;
    }
    
    public long lastModified()
    {
        if (scriptFile != null)
        {
            return scriptFile.lastModified();
        }
        else
        {
            return 0L;
        }
    }
    
    public InputStream getInputStream() throws IOException
    {
        if (scriptFile != null)
        {
            return new FileInputStream(scriptFile);
        }
        else
        {
            return new ByteArrayInputStream(scriptBytes);
        }
    }
}
