/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.bridges.script;

import java.io.BufferedInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.net.URLDecoder;

/**
 * ScriptSourceFactory
 * <P>
 * Simple factory class to create ScriptSource instances by using SimpleScriptSource class.
 * </P>
 * 
 * @author <a href="mailto:woonsan@apache.org">Woonsan Ko</a>
 * @version $Id: SimpleScriptSourceFactory.java 937248 2010-04-23 11:06:13Z woonsan $
 */
public class SimpleScriptSourceFactory implements ScriptSourceFactory
{
    
    public ScriptSource createScriptSource(final String uri) throws IOException
    {
        return createScriptSource(uri, null);
    }
    
    public ScriptSource createScriptSource(final String uri, final String uriEncoding) throws IOException
    {
        return createScriptSource(uri, uriEncoding, null);
    }
    
    public ScriptSource createScriptSource(final String uri, final String uriEncoding, final String characterEncoding) throws IOException
    {
        String name = null;
        String extension = null;
        File scriptFile = null;
        byte [] scriptContent = null;
        
        if (uri.startsWith("file:"))
        {
            String decodedUri = uri;
            
            if (uriEncoding != null)
            {
                decodedUri = URLDecoder.decode(uri, uriEncoding);
            }
            
            scriptFile = new File(decodedUri.substring(5));
        }
        else if (uri.startsWith("classpath:"))
        {
            URL resourceURL = Thread.currentThread().getContextClassLoader().getResource(uri.substring(10));
            String resourceURLString = resourceURL.toString();

            if (resourceURLString.startsWith("file:"))
            {
                String decodedUri = resourceURLString;
                
                if (uriEncoding != null)
                {
                    decodedUri = URLDecoder.decode(resourceURLString, uriEncoding);
                }
                
                scriptFile = new File(decodedUri.substring(5));
            }
            else
            {
                InputStream is = null;
                BufferedInputStream bis = null;
                ByteArrayOutputStream baos = null;
                
                try
                {
                    is = resourceURL.openStream();
                    bis = new BufferedInputStream(is);
                    baos = new ByteArrayOutputStream(4096);
                    
                    byte [] bytes = new byte[4096];
                    
                    int readLen = bis.read(bytes, 0, 4096);
                    while (readLen != -1)
                    {
                        baos.write(bytes, 0, readLen);
                        readLen = bis.read(bytes, 0, 4096);
                    }
                    
                    scriptContent = baos.toByteArray();
                }
                finally
                {
                    if (baos != null)
                    {
                        try
                        {
                            baos.close();
                        }
                        catch (Exception ignore) { }
                    }
                    if (bis != null)
                    {
                        try
                        {
                            bis.close();
                        }
                        catch (Exception ignore) { }
                    }
                    if (is != null)
                    {
                        try
                        {
                            is.close();
                        }
                        catch (Exception ignore) { }
                    }
                }
            }
        }
        else
        {
            String relPath = uri;
            scriptFile = new File(relPath);
        }
        
        if (scriptFile != null)
        {
            name = scriptFile.getName();
        }
        else
        {
            name = uri;
            int offset = name.lastIndexOf('/');
            if (offset != -1)
            {
                name = name.substring(offset + 1);
            }
        }
        
        int offset = name.lastIndexOf('.');
        
        if (offset != -1)
        {
            extension = name.substring(offset + 1);
        }
        
        return new SimpleScriptSource(name, extension, characterEncoding, scriptFile, scriptContent);
    }
}
