/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.bridges.script;

import static org.easymock.EasyMock.createNiceMock;
import static org.easymock.EasyMock.expect;
import static org.easymock.EasyMock.replay;

import java.io.IOException;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.ResourceBundle;

import javax.portlet.Event;
import javax.portlet.EventPortlet;
import javax.portlet.EventRequest;
import javax.portlet.EventResponse;
import javax.portlet.Portlet;
import javax.portlet.PortletContext;
import javax.portlet.PortletException;
import javax.portlet.ResourceRequest;
import javax.portlet.ResourceResponse;
import javax.portlet.ResourceServingPortlet;
import javax.xml.namespace.QName;

import junit.framework.TestCase;

import org.springframework.core.io.DefaultResourceLoader;
import org.springframework.mock.web.portlet.MockActionRequest;
import org.springframework.mock.web.portlet.MockActionResponse;
import org.springframework.mock.web.portlet.MockPortletConfig;
import org.springframework.mock.web.portlet.MockPortletContext;
import org.springframework.mock.web.portlet.MockPortletPreferences;
import org.springframework.mock.web.portlet.MockRenderRequest;
import org.springframework.mock.web.portlet.MockRenderResponse;

/**
 * AbstractSimpleScriptPortletTestCase
 * 
 * @author <a href="mailto:woonsan@apache.org">Woonsan Ko</a>
 * @version $Id: AbstractSimpleScriptPortletTestCase.java 937251 2010-04-23 11:12:46Z woonsan $
 */
public abstract class AbstractSimpleScriptPortletTestCase extends TestCase
{
    protected PortletContext portletContext;
    protected MockPortletConfig portletConfig;
    protected ScriptPortlet scriptPortlet;
    protected Portlet scriptPortletInstance;
    protected long scriptSourceLastEvaluated;
    
    abstract protected Map<String, String> getPortletInitParameters();
    
    abstract protected ResourceBundle getPortletResourceBundle();
    
    abstract protected String getScriptMimeType();
    
    public void setUp() throws Exception
    {
        portletContext = new MockPortletContext(new DefaultResourceLoader());
        portletConfig = new MockPortletConfig(portletContext);
        
        for (Map.Entry<String, String> entry : getPortletInitParameters().entrySet())
        {
            portletConfig.addInitParameter(entry.getKey(), entry.getValue());
        }
        
        portletConfig.setResourceBundle(Locale.ENGLISH, getPortletResourceBundle());
        
        scriptPortlet = new ScriptPortlet();
        
        try
        {
            scriptPortlet.setScriptSourceFactory(new LastModifiedMutableScriptSourceFactory());
            scriptPortlet.init(portletConfig);
            scriptPortletInstance = scriptPortlet.getScriptPortletInstance();
            scriptSourceLastEvaluated = scriptPortlet.getScriptSourceLastEvaluated();
            assertTrue(scriptPortlet.getScriptSourceLastEvaluated() > 0L);
        }
        catch (PortletException e)
        {
            e.printStackTrace();
            fail("Failed to initialize portlet: " + e);
        }
    }
    
    @Override
    public void tearDown()
    {
        if (scriptPortlet != null)
        {
            scriptPortlet.destroy();
        }
    }
    
    public void testInit() throws Exception
    {
        String scriptEngine = getPortletInitParameters().get(ScriptPortlet.ENGINE);
        String evalPortletKey = getPortletInitParameters().get(ScriptPortlet.EVAL_KEY);
        String scriptSource = getPortletInitParameters().get(ScriptPortlet.SOURCE);
        
        /*
         * 1. initialize script engine by script engine name
         */
        
        PortletContext portletContext = new MockPortletContext(new DefaultResourceLoader());
        MockPortletConfig portletConfig = new MockPortletConfig(portletContext);
        portletConfig.addInitParameter(ScriptPortlet.ENGINE, scriptEngine);
        if (evalPortletKey != null)
        {
            portletConfig.addInitParameter(ScriptPortlet.EVAL_KEY, evalPortletKey);
        }
        portletConfig.addInitParameter(ScriptPortlet.SOURCE, scriptSource);
        
        ScriptPortlet helloScriptPortlet = new ScriptPortlet();
        
        try
        {
            helloScriptPortlet.init(portletConfig);
            assertTrue(helloScriptPortlet.getScriptSourceLastEvaluated() > 0L);
        }
        catch (PortletException e)
        {
            e.printStackTrace();
            fail("Failed to initialize portlet: " + e);
        }
        
        /*
         * 2. initialize script engine by script file extension
         */
        
        portletContext = new MockPortletContext(new DefaultResourceLoader());
        portletConfig = new MockPortletConfig(portletContext);
        if (evalPortletKey != null)
        {
            portletConfig.addInitParameter(ScriptPortlet.EVAL_KEY, evalPortletKey);
        }
        portletConfig.addInitParameter(ScriptPortlet.SOURCE, scriptSource);
        
        helloScriptPortlet = new ScriptPortlet();
        
        try
        {
            helloScriptPortlet.init(portletConfig);
            assertTrue(helloScriptPortlet.getScriptSourceLastEvaluated() > 0L);
        }
        catch (PortletException e)
        {
            e.printStackTrace();
            fail("Failed to initialize portlet: " + e);
        }
        
        /*
         * 3. initialize script engine by script mime type
         */
        
        portletContext = new MockPortletContext(new DefaultResourceLoader())
        {
            @Override
            public String getMimeType(String filePath)
            {
                return getScriptMimeType();
            }
        };
        portletConfig = new MockPortletConfig(portletContext);
        if (evalPortletKey != null)
        {
            portletConfig.addInitParameter(ScriptPortlet.EVAL_KEY, evalPortletKey);
        }
        portletConfig.addInitParameter(ScriptPortlet.SOURCE, scriptSource);
        
        helloScriptPortlet = new ScriptPortlet();
        
        try
        {
            helloScriptPortlet.init(portletConfig);
            assertTrue(helloScriptPortlet.getScriptSourceLastEvaluated() > 0L);
        }
        catch (PortletException e)
        {
            e.printStackTrace();
            fail("Failed to initialize portlet: " + e);
        }
    }
    
    public void testRender() throws Exception
    {
        MockRenderRequest request = new MockRenderRequest(portletContext);
        MockPortletPreferences preferences = new MockPortletPreferences();
        request.setPreferences(preferences);
        request.addLocale(Locale.ENGLISH);
        
        MockRenderResponse response = new MockRenderResponse();
        
        scriptPortlet.render(request, response);
        assertFalse("The script is refreshed " +
                   "(scriptSourceLastEvaluated: " + scriptPortlet.getScriptSourceLastEvaluated() + ", " +
                   "initial-scriptSourceLastEvaluated: " + scriptSourceLastEvaluated + ")",
                   scriptPortlet.getScriptSourceLastEvaluated() > scriptSourceLastEvaluated);
        
        String content = response.getContentAsString();
        assertNotNull(content);
        //System.out.println("### content: " + content);
        String greeting = "<H1>" + portletConfig.getResourceBundle(Locale.ENGLISH).getString("javax.portlet.title") + "</H1>";
        assertEquals(greeting, content.trim());
        
        // to check not refreshed...
        Thread.sleep(10);
        scriptPortlet.render(request, response);
        assertSame(scriptPortletInstance, scriptPortlet.getScriptPortletInstance());
        assertTrue("The script is refreshed " +
                   "(scriptSourceLastEvaluated: " + scriptPortlet.getScriptSourceLastEvaluated() + ", " +
                   "initial-scriptSourceLastEvaluated: " + scriptSourceLastEvaluated + ")",
                   scriptPortlet.getScriptSourceLastEvaluated() == scriptSourceLastEvaluated);
        
        // to force to refresh...
        Thread.sleep(10);
        scriptPortlet.setRefreshDelay(0L);
        ((LastModifiedMutableScriptSource) scriptPortlet.getScriptSource()).setLastModified(System.currentTimeMillis());
        scriptPortlet.render(request, response);
        assertNotSame(scriptPortletInstance, scriptPortlet.getScriptPortletInstance());
        assertTrue("The script is not refreshed " +
                   "(scriptSourceLastEvaluated: " + scriptPortlet.getScriptSourceLastEvaluated() + ", " +
                   "initial-scriptSourceLastEvaluated: " + scriptSourceLastEvaluated + ")",
                   scriptPortlet.getScriptSourceLastEvaluated() > scriptSourceLastEvaluated);
        
        // to check not refreshed when refresh delay is a big number...
        Thread.sleep(10);
        scriptPortletInstance = scriptPortlet.getScriptPortletInstance();
        scriptSourceLastEvaluated = scriptPortlet.getScriptSourceLastEvaluated();
        scriptPortlet.setRefreshDelay(600000L);
        ((LastModifiedMutableScriptSource) scriptPortlet.getScriptSource()).setLastModified(System.currentTimeMillis());
        scriptPortlet.render(request, response);
        assertSame(scriptPortletInstance, scriptPortlet.getScriptPortletInstance());
        assertTrue("The script is refreshed " +
                   "(scriptSourceLastEvaluated: " + scriptPortlet.getScriptSourceLastEvaluated() + ", " +
                   "initial-scriptSourceLastEvaluated: " + scriptSourceLastEvaluated + ")",
                   scriptPortlet.getScriptSourceLastEvaluated() == scriptSourceLastEvaluated);
        
        // to check refreshed after refresh delay...
        Thread.sleep(20);
        scriptPortletInstance = scriptPortlet.getScriptPortletInstance();
        scriptSourceLastEvaluated = scriptPortlet.getScriptSourceLastEvaluated();
        scriptPortlet.setRefreshDelay(10L);
        ((LastModifiedMutableScriptSource) scriptPortlet.getScriptSource()).setLastModified(System.currentTimeMillis());
        scriptPortlet.render(request, response);
        assertNotSame(scriptPortletInstance, scriptPortlet.getScriptPortletInstance());
        assertTrue("The script is not refreshed " +
                   "(scriptSourceLastEvaluated: " + scriptPortlet.getScriptSourceLastEvaluated() + ", " +
                   "initial-scriptSourceLastEvaluated: " + scriptSourceLastEvaluated + ")",
                   scriptPortlet.getScriptSourceLastEvaluated() > scriptSourceLastEvaluated);
    }
    
    public void testProcessAction() throws Exception
    {
        MockActionRequest request = new MockActionRequest(portletContext);
        MockPortletPreferences preferences = new MockPortletPreferences();
        
        if (scriptPortlet.getValidatorInstance() != null)
        {
            preferences.setPreferencesValidator(scriptPortlet.getValidatorInstance());
        }
        
        request.setPreferences(preferences);
        request.addLocale(Locale.ENGLISH);
        
        MockActionResponse response = new MockActionResponse();
        
        try
        {
            scriptPortlet.processAction(request, response);
            fail("Validator exception should occur here!!!");
        }
        catch (Exception e)
        {
            // expected ...
        }
        
        request.addParameter("message", "hello");
        
        scriptPortlet.processAction(request, response);
    }
    
    public void testProcessEvent() throws Exception
    {
        assertTrue(scriptPortlet instanceof EventPortlet);
        
        Map<String, String> values = new HashMap<String, String>();
        EventRequest request = createNiceMock(EventRequest.class);
        EventResponse response = createNiceMock(EventResponse.class);
        
        Event event = createNiceMock(Event.class);
        QName qname = new QName("unknown");
        
        expect(event.getQName()).andReturn(qname);
        expect(request.getEvent()).andReturn(event);
        expect(request.getAttribute("values")).andReturn(values);
        
        replay(event);
        replay(request);
        replay(response);
        
        scriptPortlet.processEvent(request, response);
        assertEquals("hello", values.get("message"));
    }
    
    public void testServeResource() throws Exception
    {
        assertTrue(scriptPortlet instanceof ResourceServingPortlet);
        
        Map<String, String> values = new HashMap<String, String>();
        ResourceRequest request = createNiceMock(ResourceRequest.class);
        ResourceResponse response = createNiceMock(ResourceResponse.class);
        
        expect(request.getAttribute("values")).andReturn(values);
        
        replay(request);
        replay(response);
        
        scriptPortlet.serveResource(request, response);
        assertEquals("hello", values.get("message"));
    }
    
    private static class LastModifiedMutableScriptSource extends DelegatingScriptSourceAdaptor
    {
        private long lastModified = -1L;
        
        public LastModifiedMutableScriptSource(final ScriptSource delegatee) throws IOException
        {
            super(delegatee);
        }
        
        @Override
        public long lastModified()
        {
            if (-1L == lastModified)
            {
                return super.lastModified();
            }
            else
            {
                return lastModified;
            }
        }
        
        private void setLastModified(long lastModified)
        {
            this.lastModified = lastModified;
        }
    }
    
    private static class LastModifiedMutableScriptSourceFactory extends SimpleScriptSourceFactory
    {
        @Override
        public ScriptSource createScriptSource(final String uri, final String uriEncoding, final String characterEncoding) throws IOException
        {
            ScriptSource scriptSource = super.createScriptSource(uri, uriEncoding, characterEncoding);
            return new LastModifiedMutableScriptSource(scriptSource);
        }
    }
}
