package org.apache.jetspeed.portletcontainer;

// jetspeed
import org.apache.jetspeed.portlet.*;
import org.apache.jetspeed.portletcontainer.util.*;
import org.apache.jetspeed.portletcontainer.filter.*;
import org.apache.jetspeed.portletcontainer.information.*;
import org.apache.jetspeed.portletcontainer.invoker.*;
import org.apache.jetspeed.portletcontainer.om.portletinstanceregistry.PortletInstanceEntry;
import org.apache.jetspeed.portletcontainer.om.portletregistry.ConcretePortletEntry;

// turbine
import org.apache.turbine.om.security.User;
import org.apache.turbine.util.Log;
import org.apache.turbine.services.resources.TurbineResources;

// java
import javax.servlet.*;
import javax.servlet.http.*;
import java.util.*;
import java.io.IOException;
import java.lang.reflect.*;

import com.ibm.wps.portletcontainer.information.*;

public class PortletRequestImpl extends com.ibm.servlet.engine.webapp.HttpServletRequestProxy implements PortletRequest
{
    private PortletInstanceEntry        portletInstance = null;
    private HttpServletRequest          servletRequest = null;
    private PortletConfigImpl           portletConfig = null;
    private PortalInformationProvider   provider = null;
    private Map                         requestParameters = null;

    private PortletWindowImpl           window = null;
    private Portlet.Mode                mode = null;
    private Portlet.Mode                previousMode = null;
    private ClientImpl                  client = null;
    private PortletData                 portletData = null;
    private PortletSessionImpl          portletSession = null;
    private PortletSettingsImpl         portletSettings = null;

    private PortletRequestFilter        requestFilter = null;

    /**
     * Defines whether this request participates in an event loop.
     */
    private boolean                     eventBasedRequest = false;

    public PortletRequestImpl( PortletInstanceEntry portletInstance,
                               HttpServletRequest servletRequest,
                               PortletConfigImpl portletConfig,
                               PortalInformationProvider provider,
                               boolean eventBasedRequest )
    {
        this.portletInstance = portletInstance;
        this.servletRequest = servletRequest;
        this.portletConfig = portletConfig;
        this.provider = provider;
        this.eventBasedRequest = eventBasedRequest;

        this.mode = provider.getPortletMode(portletInstance);

        this.previousMode = provider.getPreviousPortletMode(portletInstance);

        initFilter();
    }

    public HttpServletRequest getProxiedHttpServletRequest()
    {
        return servletRequest;
    }

    public void setProxiedRequest(HttpServletRequest request)
    {
        this.servletRequest = request;
        requestFilter.setServletRequest(servletRequest);
        if (portletSession!=null)
            portletSession.setHttpSession(request.getSession());
    }

    public Portlet.Mode getMode ()
    {
        return mode;
    }

    public Portlet.Mode getPreviousMode ()
    {
        return previousMode;
    }

    public void setModeModifier(Portlet.ModeModifier modeModifier)
        throws AccessDeniedException
    {
        if (!eventBasedRequest)
            throw new AccessDeniedException();

        provider.changePortletMode(portletInstance,
                                   modeModifier);
    }

    public javax.servlet.ServletInputStream getInputStream() 
        throws java.io.IOException
    {
        if (!eventBasedRequest)
            throw new IllegalStateException();

        return servletRequest.getInputStream();
    }

    public String getMethod ()
    {
        return servletRequest.getMethod();
    }

    public Locale getLocale ()
    {
        return provider.getLocale( );
    }

    public Enumeration getLocales()
    {
        Vector locales = new Vector();
        locales.add(getLocale());
        return locales.elements();
    }

    public Client getClient ()
    {
        if (this.client==null)
        {
            this.client = new ClientImpl(provider);
        }
        return this.client;
    }

    public String getParameter (String name)
    {
        String encodedName = PortletURIImpl.encode(name);
        if (requestParameters==null) requestParameters = provider.getParameterMap();
        String[] values = (String[])requestParameters.get(
            PortletNamespaceMapper.encode(portletInstance.getPiid(),
                                          encodedName));
        if (values == null) { 
            // if the parameter was not found in the portlet request check if it is in the servlet request
            values = (String[])requestParameters.get(encodedName);
        }
        if ((values!=null) && (values.length>0)) {
            String decodedValue = PortletURIImpl.decode(values[0]);
            return decodedValue; //values[0];
        }
        return null;
    }

    public Map getParameterMap ()
    {
// Contrary to attributes, parameters that are not in a portlet's namespace will not
// be included in the returnvalue
        
        if (requestParameters==null) 
            requestParameters = provider.getParameterMap();

        Hashtable portlet_parameters = new Hashtable();

        Iterator parameters = requestParameters.keySet().iterator();

        while (parameters.hasNext())
        {
            String element = (String)parameters.next();

            String decodedElement = PortletURIImpl.decode(element);
            
            if (PortletNamespaceMapper.isInNamespace(portletInstance.getPiid(), 
                                                     decodedElement))
            {
                String name = PortletNamespaceMapper.decode(portletInstance.getPiid(), 
                                                            decodedElement);

                if (!portlet_parameters.containsKey(name))
                {
                    String[] values = (String[])requestParameters.get(element);

                    if ((values!=null) && (values.length>0))
                    {
                        String decodedValue = PortletURIImpl.decode(values[0]);
                        portlet_parameters.put(
                            name,
                            decodedValue);
                    }
                }
            }
        }
        return portlet_parameters;
    }

    public Enumeration getParameterNames ()
    {
// Contrary to attributes, parameters that are not in a portlet's namespace will not
// be included in the returnvalue
        
        if (requestParameters==null) 
            requestParameters = provider.getParameterMap();
        
        Hashtable portlet_parameters = new Hashtable();
        
        Iterator parameters = requestParameters.keySet().iterator();
        
        while (parameters.hasNext())
        {
            String element = (String)parameters.next();

            String decodedElement = PortletURIImpl.decode(element);
            
            if (PortletNamespaceMapper.isInNamespace(portletInstance.getPiid(), 
                                                     decodedElement))
            {
                String name = PortletNamespaceMapper.decode(portletInstance.getPiid(), 
                                                            decodedElement);

                portlet_parameters.put(name, name);
            }
        }
        return portlet_parameters.elements();
    }

    public String [] getParameterValues (String name)
    {
        String encodedName = PortletURIImpl.encode(name);

        if (requestParameters==null) 
            requestParameters = provider.getParameterMap();

        String[] values = (String[])requestParameters.get(PortletNamespaceMapper.encode(
            portletInstance.getPiid(),encodedName));

        if (values == null)
        { // if the parameter was not found in the portlet request check if it is in the servlet request
            values = (String[])requestParameters.get(encodedName);
        }
        if (values == null)
        {
            return new String[0];
        }
        
        String[] decodedValues = new String[values.length];
        
        for (int i = 0; i < decodedValues.length; i++)
        {
            decodedValues[i] = PortletURIImpl.decode(values[i]);
        }

        return decodedValues;
    }

    public void setAttribute (String name, Object value)
    {
        servletRequest.setAttribute(PortletNamespaceMapper.encode(portletInstance.getPiid(),
                                                                  name), 
                                    value);
    }

    public void removeAttribute (String name)
    {
        servletRequest.removeAttribute(PortletNamespaceMapper.encode(portletInstance.getPiid(),
                                                                     name));
        // when not found in the portlet request we do NOT delete it in the servlet request
    }

    public Object getAttribute (String name)
    {
        Object attribute = servletRequest.getAttribute(
            PortletNamespaceMapper.encode(portletInstance.getPiid(),
                                          name));
        if (attribute == null)
        { // if the attribute was not found in the portlet request check if it is in the servlet request
            attribute = servletRequest.getAttribute(name);
        }

        return attribute;
    }

    public Enumeration getAttributeNames ()
    {
        Vector portlet_attributes = new Vector();
        
        Enumeration attributes = servletRequest.getAttributeNames();
        
        while (attributes.hasMoreElements())
        {
            String element = (String)attributes.nextElement();

            // check if this cookie is in the namespace of this portlet
            if (PortletNamespaceMapper.isInNamespace(portletInstance.getPiid(), 
                                                     element))
            {
                portlet_attributes.add(PortletNamespaceMapper.decode(portletInstance.getPiid(), 
                                                                     element));
            }
        }

        return portlet_attributes.elements();
    }

    public boolean isSecure ()
    {
        return servletRequest.isSecure();
    }

    public PortletWindow getWindow ()
    {
        if (this.window==null)
        {
            this.window = new PortletWindowImpl( portletInstance, 
                                                 this, 
                                                 portletConfig, 
                                                 provider, 
                                                 eventBasedRequest );
        }
        return this.window;
    }

    public PortletData getData ()
    {
        if (this.portletData==null)
        {
            this.portletData = new PortletDataImpl( portletInstance,
                                                    this,
                                                    (PortletContextImpl)portletConfig.getContext(),
                                                    provider );
        }
        return this.portletData;
    }

    public PortletSettings getPortletSettings()
    {
        if (this.portletSettings==null)
        {
            ConcretePortletEntry concretePortlet = portletInstance.getConcretePortlet();

            this.portletSettings = new PortletSettingsImpl( concretePortlet,
                                                            provider );
        }
        return this.portletSettings;
    }

    public PortletSession getPortletSession ()
    {
        return getPortletSession(true);
    }

    public PortletSession getPortletSession (boolean create)
    {
        HttpSession session = servletRequest.getSession(create);

        if (session!=null)
        {
            this.portletSession = PortletSessionManager.getPortletSession(
                portletInstance.getApplicationInstance().getAiid().toString(),
                session );

            if (this.portletSession==null)
            {
                if ((PortletSessionManager.isUserLoggedIn(session)) || 
                    (create))
                {
                    this.portletSession = PortletSessionManager.createPortletSession(
                        portletInstance.getApplicationInstance().getAiid().toString(),
                        session );
                }
            }
        }
        return this.portletSession;
    }

    public javax.servlet.http.Cookie[] getCookies()
    {
        javax.servlet.http.Cookie[] cookies = servletRequest.getCookies();
        if (cookies!=null)
        {
            Vector newCookies = new Vector();
            for (int i=0; i<cookies.length; i++)
            {
                // check if this cookie is in the namespace of this portlet
                if (PortletNamespaceMapper.isInNamespace(portletInstance.getPiid(),
                                                         cookies[i].getName()))
                {
                    newCookies.add(CookieConverter.decode(portletInstance.getPiid(), cookies[i]));
                }
                // or if this portlet is in no portlet's namespace
                else if (!PortletNamespaceMapper.isInNamespace(cookies[i].getName()))
                {
                    newCookies.add(cookies[i]);
                }
            }
            cookies = new javax.servlet.http.Cookie[newCookies.size()];
            newCookies.toArray(cookies);
        }
        return cookies;
    }

    public org.apache.jetspeed.portlet.User getUser ()
    {
        if (getSession(false)!=null)
        {
            return getPortletSession().getUser();
        }
        else return null;
    }

    public long getDateHeader(String name)
    {
        return servletRequest.getDateHeader(name);
    }

    public String getHeader(String name)
    {
        return servletRequest.getHeader(name);
    }

    public Enumeration getHeaders(String name)
    {
        return servletRequest.getHeaders(name);
    }

    public Enumeration getHeaderNames()
    {
        return servletRequest.getHeaderNames();
    }

    public int getIntHeader(String name)
    {
        return servletRequest.getIntHeader(name);
    }

    public void invalidateCache()
    {
        // tbd
    }

    // intermediate methods ... between servlet and portlet

    public HttpSession getSession(boolean create)
    {
        return getPortletSession(create);
    }

    public HttpSession getSession()
    {
        return getPortletSession();
    }

    // servlet methods, which cannot be changed by the filter, because they are deprecated

    public boolean isRequestedSessionIdFromUrl()
    {
        // deprectated
        return isRequestedSessionIdFromURL();
    }
    
    public String getRealPath(String path)
    {
        // deprectated
        return portletConfig.getContext().getRealPath(path);
    }

    // servlet methods, which can be changed by the filter

    public String getAuthType()
    {
        return requestFilter.getAuthType();
    }

    public String getPathTranslated()
    {
        return requestFilter.getPathTranslated();
    }

    public boolean isUserInRole(String role)
    {
        return requestFilter.isUserInRole(role);
    }

    public String getPathInfo()
    {
        return requestFilter.getPathInfo();
    }

    public String getServletPath()
    {
        return requestFilter.getServletPath();
    }

    public String getContextPath()
    {
        return requestFilter.getContextPath();
    }

    public String getQueryString()
    {
        return requestFilter.getQueryString();
    }

    public String getRemoteUser()
    {
        return requestFilter.getRemoteUser();
    }

    public boolean isRequestedSessionIdFromCookie()
    {
        return requestFilter.isRequestedSessionIdFromCookie();
    }

    public java.security.Principal getUserPrincipal()
    {
        return requestFilter.getUserPrincipal();
    }

    public String getRequestedSessionId()
    {
        // TODO: we should perhaps return a unique SessionID with PIID or AIID attached
        return requestFilter.getRequestedSessionId();
    }
    
    public String getRequestURI()
    {
        return requestFilter.getRequestURI();
    }

    public boolean isRequestedSessionIdValid()
    {
        // TODO
        return requestFilter.isRequestedSessionIdValid();
    }

    public boolean isRequestedSessionIdFromURL()
    {
        // TODO
        return requestFilter.isRequestedSessionIdFromURL();
    }

    public RequestDispatcher getRequestDispatcher(String path)
    {
        return requestFilter.getRequestDispatcher(path);
    }

    public String getCharacterEncoding()
    {
        return requestFilter.getCharacterEncoding();
    }

    public int getContentLength()
    {
        return requestFilter.getContentLength();
    }

    public String getContentType()
    {
        return requestFilter.getContentType();
    }

    public String getProtocol()
    {
        return requestFilter.getProtocol();
    }

    public String getScheme()
    {
        return requestFilter.getScheme();
    }

    public java.io.BufferedReader getReader() throws java.io.IOException
    {
        return requestFilter.getReader();
    }

    public String getRemoteAddr()
    {
        return requestFilter.getRemoteAddr();
    }

    public String getServerName()
    {
        return requestFilter.getServerName();
    }

    public int getServerPort()
    {
        return requestFilter.getServerPort();
    }

    public String getRemoteHost()
    {
        return requestFilter.getRemoteHost();
    }

    // additional methods

    public boolean isEventBasedRequest()
    {
        return eventBasedRequest;
    }

    public void setParameterMap(Map parameters)
    {
        requestParameters = parameters;
    }

    public PortalInformationProvider getProvider()
    {
        return provider;
    }

    public void setProvider(PortalInformationProvider provider)
    {
        // check if we have already applied a WebModuleInformationProvider
        // if so, we don't need another wrapper
        if (!(this.provider instanceof WebModuleInformationProvider))
            this.provider = provider;
    }

    public PortletInstanceEntry getPortletInstanceEntry()
    {
        return portletInstance;
    }

    public PortletConfig getPortletConfig()
    {
        return portletConfig;
    }

    private void initFilter()
    {
        String filterClassName = TurbineResources.getString("portletcontainer.filter.request");
        if (filterClassName==null) {
            requestFilter = new PortletRequestFilter(servletRequest,
                                                     this);
        }
        else {
            Class[] parameterClasses = {HttpServletRequest.class, PortletRequest.class};
            Constructor constructor = null;
            try {
                constructor = Class.forName(filterClassName).getConstructor(parameterClasses);
            }
            catch (NoSuchMethodException e) {
                Log.error("Invalid PortletRequestFilter defined in resources key portletcontainer.filter.request.",e);
                throw new IllegalArgumentException("Invalid PortletRequestFilter defined in resources key portletcontainer.filter.request.");
            }
            catch (ClassNotFoundException e) {
                Log.error("Invalid PortletRequestFilter defined in resources key portletcontainer.filter.request.",e);
                throw new IllegalArgumentException("Invalid PortletRequestFilter defined in resources key portletcontainer.filter.request.");
            }
            Object[] parameters = {servletRequest, this };
            try {
                Object filter = constructor.newInstance(parameters);
                if (!(filter instanceof PortletRequestFilter)) {
                    Log.error("PortletRequestFilter defined in resources key portletcontainer.filter.request does not derive from PortletRequestFilter.");
                    throw new IllegalArgumentException("PortletRequestFilter defined in resources key portletcontainer.filter.request does not derive from PortletRequestFilter.");
                }
                requestFilter = (PortletRequestFilter)filter;
            }
            catch (InstantiationException e) {
                Log.error("PortletRequestFilter defined in resources key portletcontainer.filter.request could not be instantiated.",e);
                throw new IllegalArgumentException("PortletRequestFilter defined in resources key portletcontainer.filter.request could not be instantiated.");
            }
            catch (IllegalAccessException e) {
                Log.error("PortletRequestFilter defined in resources key portletcontainer.filter.request could not be instantiated.",e);
                throw new IllegalArgumentException("PortletRequestFilter defined in resources key portletcontainer.filter.request could not be instantiated.");
            }
            catch (InvocationTargetException e) {
                Log.error("PortletRequestFilter defined in resources key portletcontainer.filter.request could not be instantiated.",e);
                throw new IllegalArgumentException("PortletRequestFilter defined in resources key portletcontainer.filter.request could not be instantiated.");
            }
        }
    }

}
