package org.apache.jetspeed.portletcontainer.event;

// jetspeed
import org.apache.jetspeed.portlet.*;
import org.apache.jetspeed.portlet.event.*;
import org.apache.jetspeed.portletcontainer.*;
import org.apache.jetspeed.portletcontainer.event.*;
import org.apache.jetspeed.portletcontainer.information.*;
import org.apache.jetspeed.portletcontainer.invoker.*;
import org.apache.jetspeed.portletcontainer.util.*;

// turbine
import org.apache.turbine.util.Log;
import org.apache.turbine.services.resources.TurbineResources;

// java
import javax.servlet.*;
import javax.servlet.http.*;
import java.io.InputStream;
import java.io.IOException;
import java.util.LinkedList;
import java.util.Enumeration;

public class EventQueueManager
{
	// helper var to detect and escape from message loops.
	// if there a more than 1000 messages queued within one event loop, we asssume that 
	// this is a 'ping-pong' meassage and break the loop.  
    private static int maxProcessedEvents = TurbineResources.getInt("portletcontainer.MaxProcessedEvents",1000);

    /**
     * Adds an event to the queue
     * 
     * @param request the servlet request
     * @param event   the event to be added
     */
    public static void addEvent(ServletRequest request, EventImpl event)
    {
        LinkedList queue = getEventQueue(request);
        queue.addFirst(event);
    }

    /**
     * 
     * @param request the servlet request
     * @param evtEnv  the event environment
     */
    public static void processEventLoop(HttpServletRequest request, EventEnvironment evtEnv)
    {
        if (Log.isDebugEnabled)
            Log.debug( "EventQueueManager.processEventLoop: starting event loop...");

        setEventCount(request,0);
        LinkedList queue = getEventQueue(request);
        // process event
        while (!queue.isEmpty()) {
            EventImpl event = getEvent(queue, request);
            try {
                event.prepare(evtEnv);
            }
            catch (Throwable t) { 
                // catch every exception, because the portal should go on.
                Log.error("EventQueueManager: Error during execution of event.",t);
            }
            int count = getEventCount(request);
            if (count>maxProcessedEvents) {
                Log.error("EventQueueManager: Event queue processing was interrupted. Reason: Too much events! Maybe a portlet created an endless loop.");
                break;
            }
            else setEventCount(request,++count);
        }

        if (Log.isDebugEnabled)
            Log.debug( "EventQueueManager.processEventLoop: finished event loop.");
    }

    /**
     * Returns the event inserted first and removes it from the queue.
     * If no event waits in the queue anymore, <CODE>NULL</CODE> is returned.
     * 
     * @param queue   the event queue
     * @param request the servlet request
     * @return the event inserted first
     */
    private static EventImpl getEvent(LinkedList queue, ServletRequest request)
    {
        try {
            return (EventImpl)queue.removeLast();
        }
        catch (java.util.NoSuchElementException ex)
        {
            Log.error("EventQueueManager.getEvent(): "+ex.toString());
            return null;
        }
    }
    
    /**
     * Returns the event queue allocated for this request.
     * The queue will be created, if not already existing.
     * 
     * @param request the servlet request
     * @return the event queue
     */
    private static LinkedList getEventQueue(ServletRequest request)
    {
        LinkedList queue = (LinkedList)request.getAttribute("PC_EventQueue");
        if (queue==null) {
            queue = new LinkedList();
            request.setAttribute("PC_EventQueue", queue);
        }
        return queue;
    }

    /**
     * Returns the numbers of events processed so far.
     * 
     * The counter is a means to detect event loops.
     * Currently we will stop the queue, if 1000 (customizable)
     * events are processed within one process interval.
     * 
     * @param request the servlet request
     * @return the numbers of events processed 
     */
    private static int getEventCount(ServletRequest request)
    {
        return ((Integer)request.getAttribute("PC_EventCount")).intValue();
    }

    /**
     * Sets the number of events processed in the request queue.
     * 
     * The counter is a means to detect event loops.
     * Currently we will stop the queue, if 1000 (customizable)
     * events are processed within one process interval.
     * 
     * @param request the servlet request
     * @param the number of events processed from the queue
     */
    private static void setEventCount(ServletRequest request, int count)
    {
        request.setAttribute("PC_EventCount",new Integer(count));
    }

}
