/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2001 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" and
 *     "Apache Jetspeed" must not be used to endorse or promote products
 *    derived from this software without prior written permission. For
 *    written permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache" or
 *    "Apache Jetspeed", nor may "Apache" appear in their name, without
 *    prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */
 
package org.apache.jetspeed.portletcontainer.tags;

import java.util.Enumeration;
import java.io.IOException;
import javax.servlet.jsp.*;
import javax.servlet.jsp.tagext.BodyTagSupport;
import org.apache.jetspeed.portlet.*;
import org.apache.jetspeed.portletcontainer.Constants;
import org.apache.regexp.*;

/**
 ** Supporting class for the <CODE>settingsLoop</CODE> tag.
 ** Loops over all attributes in <CODE>PortletSettings</CODE> of the current portlet instance.
 ** Tag parameter:
 ** <UL>
 ** <LI><CODE>pattern</CODE> (optional, default: all attributes are looped) - a regular expression that defines which attributes should be looped
 ** </UL>
 ** @see   SettingsAttributeTag
 **
 ** @author   <a href="mailto:h.schwarz@de.ibm.com">Hagen Schwarz</a>
 **/
public class SettingsLoopTag extends BodyTagSupport {   

    private Enumeration iAttributes = null;
    private String iAttribute = null;   //String for pertinent settings attribute

    /**
     * Processes the <CODE>settingsLoop</CODE> start tag.
     * @return the result of <CODE>doAfterBody()</CODE>
     */
    public int doStartTag ()
    {
        PortletRequest portletRequest = (PortletRequest)pageContext.getRequest().getAttribute(Constants.PORTLET_REQUEST);
        PortletSettings settings = portletRequest.getPortletSettings();
        
        if (settings != null) iAttributes = settings.getAttributeNames();

        return (doAfterBody ());
    }

    /**
     * Steps to the next attribute in <CODE>settingsLoop</CODE>. Sets the variable <CODE>portletSettings</CODE> with this attribute.
     * @return <CODE>SKIP_BODY</CODE> - if no more attributes or attribute does not match <CODE>pattern</CODE>
     *         <BR><CODE>EVAL_BODY_TAG</CODE> - else
     */
    public int doAfterBody ()
    {
        int result = SKIP_BODY;

        try
        {
            if (bodyContent != null)
            {
                bodyContent.writeOut (bodyContent.getEnclosingWriter ());
                bodyContent.clearBuffer();
            }

            iAttribute = null;
            while (iAttributes != null && iAttributes.hasMoreElements() && iAttribute == null)
            {
                iAttribute = (String) iAttributes.nextElement();
                if ( iPattern == null ||   //No pattern specified
                     iPattern.match(iAttribute) )   //Pattern matches - only evaluated if iPattern != null
                {
                    pageContext.setAttribute ("portletSettings", iAttribute);
                    result = EVAL_BODY_TAG;   //Attribute found, while-loop is ending
                }
                else   
                    iAttribute = null;   //Attribute found did not match pattern, is reset
            }
        }
        catch (IOException exc)
        {
            PortletConfig portletConfig = (PortletConfig)pageContext.getRequest().getAttribute(Constants.PORTLET_CONFIG);
            PortletContext portletContext = portletConfig.getContext();

            PortletLog portletLog = portletContext.getLog();
            portletLog.error ("SettingsLoopTag: An I/O error occurred.", exc);
        }

        return (result);
    }

    /**
     * Processes the <CODE>settingsLoop</CODE> end tag.
     * @return <CODE>EVAL_PAGE</CODE>
     */
    public int doEndTag ()
    {
        try
        {
            if (bodyContent != null)
                bodyContent.writeOut (bodyContent.getEnclosingWriter ());
        }
        catch (IOException exc)
        {
            PortletConfig portletConfig = (PortletConfig)pageContext.getRequest().getAttribute(Constants.PORTLET_CONFIG);
            PortletContext portletContext = portletConfig.getContext();

            PortletLog portletLog = portletContext.getLog();
            portletLog.error ( "SettingsLoopTag: An I/O error occurred: ", exc);
        }

        resetCustomAttributes ();

        return (EVAL_PAGE);
    }

    private void resetCustomAttributes ()
    {
    
        iAttribute = null;
        iAttributes = null;
        iPattern = null;
    }

    /**
     * Removes the variable <CODE>portletData</CODE>.
     */
    public void release ()
    {
        pageContext.removeAttribute ("portletSettings");
    }

    private RE iPattern = null;
    /**
    * The setter for the <CODE>pattern</CODE> parameter.
    * @param aPattern java.lang.String
    */
    public void setPattern(String aPattern)
    {
        try
        {
            iPattern = new RE(aPattern);
            //Case sensitive if RE.setMatchFlags not used
        }
        catch (RESyntaxException exc)
        {
            PortletConfig portletConfig = (PortletConfig)pageContext.getRequest().getAttribute(Constants.PORTLET_CONFIG);
            PortletContext portletContext = portletConfig.getContext();

            PortletLog portletLog = portletContext.getLog();
            portletLog.error ( "SettingsLoopTag: Pattern syntax exception: ", exc);

            iPattern = null;   //Treated as if no pattern specified 
        }
    }

}