/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2001 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" and
 *     "Apache Jetspeed" must not be used to endorse or promote products
 *    derived from this software without prior written permission. For
 *    written permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache" or
 *    "Apache Jetspeed", nor may "Apache" appear in their name, without
 *    prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

package org.apache.jetspeed.services.portletregistry;

import org.apache.jetspeed.portletcontainer.om.applicationregistry.*;
import org.apache.jetspeed.portletcontainer.om.portletregistry.*;
import org.apache.turbine.services.Service;
import java.util.*;
import java.io.IOException;

import com.ibm.wps.util.ObjectID;

/**
 * <P>
 * The <CODE>PortletRegistryService</CODE> interface represents all
 * portlets and portlet applications available in the portal. It is accessed
 * by the portlet container to get information about the portlets and
 * portlet applications.
 * </P>
 * <P>
 * The interfaces defined in this package represent an abstract object
 * model (OM) that is applicable for different implementations.
 * The abstract OM defines only how the data is stored and accessed
 * in the memory. Each implementation can store the data in different ways.
 * </P>
 * <P>
 * This abstraction layer helps to generalize the portlet container from
 * special implementations like data storage and moreover it is not bound
 * to a special Application Server.
 * </P>
 * 
 * @author <a href="shesmer@raleigh.ibm.com">Stephan Hesmer</a>
 */
public interface PortletRegistryService extends Service
{
 
    /** The name of this service */
    public String SERVICE_NAME = "PortletRegistry";

    /**
     * Returns all portlet applications definied in the registry
     * as an enumeration of <CODE>ApplicationEntry</CODE> elements.
     * 
     * @return an enumeration of <CODE>ApplicationEntry</CODE> elements
     * @throws IOException
     */
    public Enumeration getApplications() throws IOException;

    /**
     * Checks whether the application with the given identifier is a concrete application or not.
     * 
     * @param aid the application identifier
     * @return true if the application identifier refers to a concrete application
     */
    public boolean isConcreteApplication(ObjectID aid) throws IOException;

    /**
     * Returns the portlet application with the given name.<BR>
     * If the given identifier refers to a concrete portlet application this function
     * returns null.
     * 
     * @param aid    the identifier of the portlet application to be returned
     * @return the portlet application
     * @exception IOException
     */
    public ApplicationEntry getApplication(ObjectID aid) throws IOException;
   
    /**
     * Returns all portlet applications definied in the registry
     * as an enumeration of <CODE>ConcreteApplicationEntry</CODE> elements.
     * 
     * @return an enumeration of <CODE>ConcreteApplicationEntry</CODE> elements
     * @throws IOException
     */
    public Enumeration getConcreteApplications() throws IOException;

    /**
     * Returns the concrete portlet application with the given name.<BR>
     * If the given identifier refers not to a concrete portlet application this function
     * returns null.
     * 
     * @param caid    the identifier of the concrete portlet application to be returned
     * @return the concrete portlet application
     * @exception IOException
     */
    public ConcreteApplicationEntry getConcreteApplication(ObjectID caid) throws IOException;

    /**
     * Returns all portlets definied in the registry
     * as an enumeration of <CODE>PortletEntry</CODE> elements.
     * 
     * @return an enumeration of <CODE>PortletEntry</CODE> elements
     * @throws IOException
     */
    public Enumeration getPortlets() throws IOException;

    /**
     * Returns all portlets of the given application definied in the registry
     * as an enumeration of <CODE>PortletEntry</CODE> elements.
     * 
     * @return an enumeration of <CODE>PortletEntry</CODE> elements
     * @throws IOException
     */
    public Enumeration getPortlets(ApplicationEntry application) throws IOException;

    /**
     * Checks whether the portlet with the given identifier is a concrete portlet or not.
     * 
     * @param pid the portlet identifier
     * @return true if the portlet identifier refers to a concrete portlet
     */
    public boolean isConcretePortlet(ObjectID pid) throws IOException;

    /**
     * Returns the portlet with the given portlet identifier.<BR>
     * If the given identifier refers to a concrete portlet this function
     * returns null.
     * 
     * @param pid    the portlet identifier
     * @return the portlet
     * @exception IOException
     */
    public PortletEntry getPortlet(ObjectID pid) throws IOException;

    /**
     * Returns all portlets definied in the registry
     * as an enumeration of <CODE>ConcretePortletEntry</CODE> elements.
     * 
     * @return an enumeration of <CODE>ConcretePortletEntry</CODE> elements
     * @throws IOException
     */
    public Enumeration getConcretePortlets() throws IOException;

    /**
     * Returns all portlets of the given concrete application definied in the registry
     * as an enumeration of <CODE>ConcretePortletEntry</CODE> elements.
     * 
     * @return an enumeration of <CODE>ConcretePortletEntry</CODE> elements
     * @throws IOException
     */
    public Enumeration getConcretePortlets(ConcreteApplicationEntry application) throws IOException;

    /**
     * Returns the concrete portlet with the given portlet identifier.<BR>
     * If the given identifier refers not to a concrete portlet this function
     * returns null.
     * 
     * @param cpid    the concrete portlet identifier
     * @return the concrete portlet
     * @exception IOException
     */
    public ConcretePortletEntry getConcretePortlet(ObjectID cpid) throws IOException;

    /**
     * Checks if application settings are changed and sends the corresponding
     * events to all portlets of the given application except to the given
     * portlet.
     * 
     * @param concretePortlet
     *               the concrete portlet that changed the application settings
     * @param orgSettings
     *               Map containing the settings of the original application ([String]/[String])
     */
    public void applicationSettingsChanged(ConcretePortletEntry concretePortlet,
                                           Map orgSettings);

}
