/*
* Licensed to the Apache Software Foundation (ASF) under one or more
* contributor license agreements.  See the NOTICE file distributed with
* this work for additional information regarding copyright ownership.
* The ASF licenses this file to You under the Apache License, Version 2.0
* (the "License"); you may not use this file except in compliance with
* the License.  You may obtain a copy of the License at
*
*     http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/
package org.apache.jetspeed.aggregator;

import java.io.PrintWriter;
import java.util.List;

import javax.portlet.PortletMode;
import javax.portlet.WindowState;

import org.apache.jetspeed.cache.ContentCacheKey;
import org.apache.jetspeed.portlet.HeadElement;
import org.apache.jetspeed.util.KeyValue;

/**
 * <p>
 * PortletContent
 * </p>
 * @author <a href="mailto:weaver@apache.org">Scott T. Weaver</a>
 * @author <a href="mailto:taylor@apache.org">David S. Taylor</a>  
 * @version $Id: PortletContent.java 889695 2009-12-11 16:53:45Z woonsan $
 *
 */
public interface PortletContent
{
    /**
     * Retrieve the actual content of a portlet as a string
     * 
     * @return
     */
    String getContent();
    
    /** 
     * Has the renderer completed rendering the content?
     * 
     * @return
     */
    boolean isComplete();
    
    /**
     * Notify that this content is completed.
     *
     */
    void complete();
    
    /**
     * Get a writer to the content to stream content into this object
     * @return
     */
    PrintWriter getWriter();
    
    /**
     * Get the expiration setting for this content if it is cached.
     * @return
     */
    int getExpiration();
    
    /**
     * Sets the expiration setting for this content.
     * 
     * @param expiration
     */
    void setExpiration(int expiration);
    
    /**
     * Get the cache key used to cache this content
     * @since 2.1.2 
     * @return
     */
    ContentCacheKey getCacheKey();
    
    /**
     * Get the title of the portlet, used during caching
     * 
     * @return
     */
    String getTitle();
    
    /**
     * Set the title of this portlet, used during caching
     * @param title
     */
    void setTitle(String title);
    
    /**
     * Gets the content type of this portlet content.
     * 
     * @return
     */
    String getContentType();
    
    /**
     * Sets the content type of this portlet content.
     * 
     * @param contentType
     */
    void setContentType(String contentType);
    
    /**
     * Gets the portlet mode of this portlet content.
     * 
     * @return
     */
    PortletMode getPortletMode();
    
    /**
     * Gets the window state of this portlet content.
     * 
     * @return
     */
    WindowState getWindowState();
    
    /**
     * Adds an header element to be contributed to the page.
     * If keyHint argument is provided and if a header element 
     * with the provided key hint already exists, then 
     * the element will be overwritten to the item.
     * If the element is null and keyHint is not null, 
     * the element with the keyHint is removed from the response.
     * If the element is not null and keyHint is null,
     * then the keyHint will be automatically generated by the element content. 
     * Meanwhile, the element should implement java.io.Serializable.
     * Otherwise it will throw a java.io.NotSerializableException.
     * 
     * @param element
     * @param keyHint
     */
    void addHeadElement(HeadElement element, String keyHint);
    
    /**
     * Retrieves header element key value pairs to be contributed to the page.
     * Because the insertion order might be important for web development, it should be list instead of map.
     * 
     * @return
     */
    List<KeyValue<String, HeadElement>> getHeadElements(); 
    
    void reset();
    
    void resetBuffer();
    
    /**
     * Release the buffers used by the portlet content cache. Note the actual release may not occur until garbage collection. 
     *
     */
    void release();                    
}