/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.crypto.key;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URL;
import java.security.Key;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.UnrecoverableKeyException;
import java.security.cert.CertificateException;
import java.util.ArrayList;
import java.util.Date;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import javax.crypto.KeyGenerator;
import javax.crypto.spec.SecretKeySpec;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.crypto.key.KeyProvider;
import org.apache.hadoop.crypto.key.KeyProviderFactory;
import org.apache.hadoop.crypto.key.RangerAWSKMSProvider;
import org.apache.hadoop.crypto.key.RangerAzureKeyVaultKeyGenerator;
import org.apache.hadoop.crypto.key.RangerGoogleCloudHSMProvider;
import org.apache.hadoop.crypto.key.RangerHSM;
import org.apache.hadoop.crypto.key.RangerKMSDB;
import org.apache.hadoop.crypto.key.RangerKMSMKI;
import org.apache.hadoop.crypto.key.RangerKeyStore;
import org.apache.hadoop.crypto.key.RangerMasterKey;
import org.apache.hadoop.crypto.key.RangerSafenetKeySecure;
import org.apache.hadoop.crypto.key.RangerTencentKMSProvider;
import org.apache.hadoop.fs.Path;
import org.apache.ranger.credentialapi.CredentialReader;
import org.apache.ranger.kms.dao.DaoManager;
import org.apache.ranger.plugin.util.AutoClosableLock;
import org.apache.ranger.plugin.util.JsonUtilsV2;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
public class RangerKeyStoreProvider
extends KeyProvider {
    private static final Logger logger = LoggerFactory.getLogger(RangerKeyStoreProvider.class);
    public static final String SCHEME_NAME = "dbks";
    public static final String KMS_CONFIG_DIR = "kms.config.dir";
    public static final String DBKS_SITE_XML = "dbks-site.xml";
    public static final String ENCRYPTION_KEY = "ranger.db.encrypt.key.password";
    private static final String KEY_METADATA = "KeyMetadata";
    private static final String CREDENTIAL_PATH = "ranger.ks.jpa.jdbc.credential.provider.path";
    private static final String MK_CREDENTIAL_ALIAS = "ranger.ks.masterkey.credential.alias";
    private static final String DB_CREDENTIAL_ALIAS = "ranger.ks.jpa.jdbc.credential.alias";
    private static final String DB_PASSWORD = "ranger.ks.jpa.jdbc.password";
    private static final String HSM_ENABLED = "ranger.ks.hsm.enabled";
    private static final String HSM_PARTITION_PASSWORD_ALIAS = "ranger.ks.hsm.partition.password.alias";
    private static final String HSM_PARTITION_PASSWORD = "ranger.ks.hsm.partition.password";
    private static final String KEYSECURE_ENABLED = "ranger.kms.keysecure.enabled";
    private static final String KEYSECURE_USERNAME = "ranger.kms.keysecure.login.username";
    private static final String KEYSECURE_PASSWORD_ALIAS = "ranger.kms.keysecure.login.password.alias";
    private static final String KEYSECURE_PASSWORD = "ranger.kms.keysecure.login.password";
    private static final String KEYSECURE_LOGIN = "ranger.kms.keysecure.login";
    private static final String AZURE_KEYVAULT_ENABLED = "ranger.kms.azurekeyvault.enabled";
    private static final String AZURE_CLIENT_SECRET_ALIAS = "ranger.kms.azure.client.secret.alias";
    private static final String AZURE_CLIENT_SECRET = "ranger.kms.azure.client.secret";
    private static final String AWS_KMS_ENABLED = "ranger.kms.awskms.enabled";
    private static final String AWS_CLIENT_SECRETKEY_ALIAS = "ranger.kms.aws.client.secretkey.alias";
    private static final String AWS_CLIENT_SECRETKEY = "ranger.kms.aws.client.secretkey";
    private static final String TENCENT_KMS_ENABLED = "ranger.kms.tencentkms.enabled";
    private static final String TENCENT_CLIENT_SECRET = "ranger.kms.tencent.client.secret";
    private static final String TENCENT_CLIENT_SECRET_ALIAS = "ranger.kms.tencent.client.secret.alias";
    private static final String IS_GCP_ENABLED = "ranger.kms.gcp.enabled";
    private final RangerKeyStore dbStore;
    private final char[] masterKey;
    private final Map<String, KeyProvider.Metadata> cache = new HashMap<String, KeyProvider.Metadata>();
    private final ReadWriteLock lock = new ReentrantReadWriteLock(true);
    private final boolean keyVaultEnabled;
    private boolean changed = false;

    public RangerKeyStoreProvider(Configuration conf) throws Throwable {
        super(conf);
        if (logger.isDebugEnabled()) {
            logger.debug("==> RangerKeyStoreProvider(conf)");
        }
        conf = RangerKeyStoreProvider.getDBKSConf();
        RangerKeyStoreProvider.getFromJceks(conf, CREDENTIAL_PATH, MK_CREDENTIAL_ALIAS, ENCRYPTION_KEY);
        RangerKeyStoreProvider.getFromJceks(conf, CREDENTIAL_PATH, DB_CREDENTIAL_ALIAS, DB_PASSWORD);
        RangerKeyStoreProvider.getFromJceks(conf, CREDENTIAL_PATH, HSM_PARTITION_PASSWORD_ALIAS, HSM_PARTITION_PASSWORD);
        String password = conf.get(ENCRYPTION_KEY);
        if (password == null || password.trim().equals("") || password.trim().equals("_") || password.trim().equals("crypted")) {
            throw new IOException("The Ranger MasterKey Password is empty or not a valid Password");
        }
        boolean isHSMEnabled = conf.getBoolean(HSM_ENABLED, false);
        boolean isKeySecureEnabled = conf.getBoolean(KEYSECURE_ENABLED, false);
        boolean isAzureKeyVaultEnabled = conf.getBoolean(AZURE_KEYVAULT_ENABLED, false);
        boolean isAWSKMSEnabled = conf.getBoolean(AWS_KMS_ENABLED, false);
        boolean isGCPEnabled = conf.getBoolean(IS_GCP_ENABLED, false);
        boolean isTencentKMSEnabled = conf.getBoolean(TENCENT_KMS_ENABLED, false);
        this.keyVaultEnabled = isAzureKeyVaultEnabled || isGCPEnabled || isTencentKMSEnabled;
        RangerKMSDB rangerKMSDB = new RangerKMSDB(conf);
        DaoManager daoManager = rangerKMSDB.getDaoManager();
        if (isHSMEnabled) {
            logger.info("Ranger KMS HSM is enabled for storing master key.");
            String partitionPasswd = conf.get(HSM_PARTITION_PASSWORD);
            if (partitionPasswd == null || partitionPasswd.trim().equals("") || partitionPasswd.trim().equals("_") || partitionPasswd.trim().equals("crypted")) {
                throw new IOException("Partition Password doesn't exists");
            }
            RangerHSM masterKeyProvider = new RangerHSM(conf);
            this.dbStore = new RangerKeyStore(daoManager);
            this.masterKey = this.generateAndGetMasterKey(masterKeyProvider, password);
        } else if (isKeySecureEnabled) {
            logger.info("KeySecure is enabled for storing the master key.");
            RangerKeyStoreProvider.getFromJceks(conf, CREDENTIAL_PATH, KEYSECURE_PASSWORD_ALIAS, KEYSECURE_PASSWORD);
            String keySecureLoginCred = conf.get(KEYSECURE_USERNAME).trim() + ":" + conf.get(KEYSECURE_PASSWORD);
            conf.set(KEYSECURE_LOGIN, keySecureLoginCred);
            RangerSafenetKeySecure masterKeyProvider = new RangerSafenetKeySecure(conf);
            this.dbStore = new RangerKeyStore(daoManager);
            this.masterKey = this.generateAndGetMasterKey(masterKeyProvider, password);
        } else if (isAzureKeyVaultEnabled) {
            logger.info("Azure Key Vault is enabled for storing the master key.");
            RangerKeyStoreProvider.getFromJceks(conf, CREDENTIAL_PATH, AZURE_CLIENT_SECRET_ALIAS, AZURE_CLIENT_SECRET);
            try {
                RangerAzureKeyVaultKeyGenerator masterKeyProvider = new RangerAzureKeyVaultKeyGenerator(conf);
                masterKeyProvider.onInitialization();
                masterKeyProvider.generateMasterKey(password);
                this.dbStore = new RangerKeyStore(daoManager, true, masterKeyProvider);
                this.masterKey = null;
            }
            catch (Exception ex) {
                throw new Exception("Error while generating master key and master key secret in Azure Key Vault. Error : " + ex);
            }
        } else if (isAWSKMSEnabled) {
            logger.info("AWS KMS is enabled for storing the master key.");
            RangerKeyStoreProvider.getFromJceks(conf, CREDENTIAL_PATH, AWS_CLIENT_SECRETKEY_ALIAS, AWS_CLIENT_SECRETKEY);
            try {
                RangerAWSKMSProvider masterKeyProvider = new RangerAWSKMSProvider(conf);
                masterKeyProvider.onInitialization();
                masterKeyProvider.generateMasterKey(password);
                this.dbStore = new RangerKeyStore(daoManager, true, masterKeyProvider);
                this.masterKey = null;
            }
            catch (Exception ex) {
                throw new Exception("Error while generating master key and master key secret in AWS KMS. Error : " + ex);
            }
        } else if (isTencentKMSEnabled) {
            logger.info("Ranger KMS Tencent KMS is enabled for storing master key.");
            RangerKeyStoreProvider.getFromJceks(conf, CREDENTIAL_PATH, TENCENT_CLIENT_SECRET_ALIAS, TENCENT_CLIENT_SECRET);
            try {
                RangerTencentKMSProvider masterKeyProvider = new RangerTencentKMSProvider(conf);
                masterKeyProvider.onInitialization();
                masterKeyProvider.generateMasterKey(password);
                this.dbStore = new RangerKeyStore(daoManager, true, masterKeyProvider);
                this.masterKey = null;
            }
            catch (Exception ex) {
                throw new Exception("Error while generating master key and master key secret in Tencent KMS. Error : " + ex);
            }
        } else if (isGCPEnabled) {
            logger.info("Google Cloud HSM is enabled for storing the master key.");
            RangerGoogleCloudHSMProvider masterKeyProvider = new RangerGoogleCloudHSMProvider(conf);
            masterKeyProvider.onInitialization();
            masterKeyProvider.generateMasterKey(password);
            this.dbStore = new RangerKeyStore(daoManager, true, masterKeyProvider);
            this.masterKey = null;
        } else {
            logger.info("Ranger KMS Database is enabled for storing master key.");
            RangerMasterKey masterKeyProvider = new RangerMasterKey(daoManager);
            this.dbStore = new RangerKeyStore(daoManager);
            this.masterKey = this.generateAndGetMasterKey(masterKeyProvider, password);
        }
        this.reloadKeys();
    }

    public static Configuration getDBKSConf() {
        Configuration newConfig = RangerKeyStoreProvider.getConfiguration(true, DBKS_SITE_XML);
        RangerKeyStoreProvider.getFromJceks(newConfig, CREDENTIAL_PATH, MK_CREDENTIAL_ALIAS, ENCRYPTION_KEY);
        RangerKeyStoreProvider.getFromJceks(newConfig, CREDENTIAL_PATH, DB_CREDENTIAL_ALIAS, DB_PASSWORD);
        return newConfig;
    }

    public KeyProvider.KeyVersion createKey(String name, byte[] material, KeyProvider.Options options) throws IOException {
        KeyProvider.KeyVersion ret;
        if (logger.isDebugEnabled()) {
            logger.debug("==> createKey({})", (Object)name);
        }
        try (AutoClosableLock.AutoClosableWriteLock ignored = new AutoClosableLock.AutoClosableWriteLock(this.lock);){
            this.reloadKeys();
            if (this.dbStore.engineContainsAlias(name) || this.cache.containsKey(name)) {
                throw new IOException("Key " + name + " already exists");
            }
            KeyProvider.Metadata meta = new KeyProvider.Metadata(options.getCipher(), options.getBitLength(), options.getDescription(), options.getAttributes(), new Date(), 1);
            if (options.getBitLength() != 8 * material.length) {
                throw new IOException("Wrong key length. Required " + options.getBitLength() + ", but got " + 8 * material.length);
            }
            String versionName = RangerKeyStoreProvider.buildVersionName((String)name, (int)0);
            ret = this.innerSetKeyVersion(name, versionName, material, meta);
        }
        if (logger.isDebugEnabled()) {
            logger.debug("<== createKey({})", (Object)name);
        }
        return ret;
    }

    public void deleteKey(String name) throws IOException {
        if (logger.isDebugEnabled()) {
            logger.debug("==> deleteKey({})", (Object)name);
        }
        try (AutoClosableLock.AutoClosableWriteLock ignored = new AutoClosableLock.AutoClosableWriteLock(this.lock);){
            this.reloadKeys();
            KeyProvider.Metadata meta = this.getMetadata(name);
            if (meta == null) {
                throw new IOException("Key " + name + " does not exist");
            }
            for (int v = 0; v < meta.getVersions(); ++v) {
                String versionName = RangerKeyStoreProvider.buildVersionName((String)name, (int)v);
                try {
                    if (!this.dbStore.engineContainsAlias(versionName)) continue;
                    this.dbStore.engineDeleteEntry(versionName);
                    continue;
                }
                catch (KeyStoreException e) {
                    throw new IOException("Problem removing " + versionName, e);
                }
            }
            try {
                if (this.dbStore.engineContainsAlias(name)) {
                    this.dbStore.engineDeleteEntry(name);
                }
            }
            catch (KeyStoreException e) {
                throw new IOException("Problem removing " + name + " from " + (Object)((Object)this), e);
            }
            this.cache.remove(name);
            this.changed = true;
        }
        if (logger.isDebugEnabled()) {
            logger.debug("<== deleteKey({})", (Object)name);
        }
    }

    public void flush() throws IOException {
        if (logger.isDebugEnabled()) {
            logger.debug("==> flush()");
        }
        if (this.changed) {
            try (AutoClosableLock.AutoClosableWriteLock ignored = new AutoClosableLock.AutoClosableWriteLock(this.lock);){
                try {
                    this.dbStore.engineStore(null, this.masterKey);
                    this.reloadKeys();
                }
                catch (NoSuchAlgorithmException e) {
                    throw new IOException("No such algorithm storing key", e);
                }
                catch (CertificateException e) {
                    throw new IOException("Certificate exception storing key", e);
                }
                this.changed = false;
            }
            catch (IOException ioe) {
                this.reloadKeys();
                throw ioe;
            }
        }
        if (logger.isDebugEnabled()) {
            logger.debug("<== flush()");
        }
    }

    public KeyProvider.KeyVersion getKeyVersion(String versionName) throws IOException {
        KeyProvider.KeyVersion ret;
        block29: {
            if (logger.isDebugEnabled()) {
                logger.debug("==> getKeyVersion({})", (Object)versionName);
            }
            ret = null;
            try (AutoClosableLock.AutoClosableReadLock ignored = new AutoClosableLock.AutoClosableReadLock(this.lock);){
                if (this.keyVaultEnabled) {
                    try {
                        byte[] decryptKeyByte;
                        boolean versionNameExists = this.dbStore.engineContainsAlias(versionName);
                        if (!versionNameExists) {
                            this.dbStore.engineLoad(null, this.masterKey);
                            versionNameExists = this.dbStore.engineContainsAlias(versionName);
                        }
                        if (!versionNameExists) break block29;
                        try {
                            decryptKeyByte = this.dbStore.engineGetDecryptedZoneKeyByte(versionName);
                        }
                        catch (Exception e) {
                            throw new RuntimeException("Error while getting decrypted key." + e);
                        }
                        if (decryptKeyByte != null && decryptKeyByte.length > 0) {
                            ret = new KeyProvider.KeyVersion(RangerKeyStoreProvider.getBaseName((String)versionName), versionName, decryptKeyByte);
                        }
                        break block29;
                    }
                    catch (NoSuchAlgorithmException e) {
                        throw new IOException("Can't get algorithm for key " + e.getMessage());
                    }
                    catch (CertificateException e) {
                        throw new IOException("Certificate exception storing key", e);
                    }
                }
                SecretKeySpec key = null;
                try {
                    boolean versionNameExists = this.dbStore.engineContainsAlias(versionName);
                    if (!versionNameExists) {
                        this.dbStore.engineLoad(null, this.masterKey);
                        versionNameExists = this.dbStore.engineContainsAlias(versionName);
                    }
                    if (versionNameExists) {
                        key = (SecretKeySpec)this.dbStore.engineGetKey(versionName, this.masterKey);
                    }
                }
                catch (NoSuchAlgorithmException e) {
                    throw new IOException("Can't get algorithm for key " + key, e);
                }
                catch (UnrecoverableKeyException e) {
                    throw new IOException("Can't recover key " + key, e);
                }
                catch (CertificateException e) {
                    throw new IOException("Certificate exception storing key", e);
                }
                if (key != null) {
                    ret = new KeyProvider.KeyVersion(RangerKeyStoreProvider.getBaseName((String)versionName), versionName, key.getEncoded());
                }
            }
        }
        if (logger.isDebugEnabled()) {
            logger.debug("<== getKeyVersion({})", (Object)versionName);
        }
        return ret;
    }

    public List<KeyProvider.KeyVersion> getKeyVersions(String name) throws IOException {
        if (logger.isDebugEnabled()) {
            logger.debug("==> getKeyVersions({})", (Object)name);
        }
        ArrayList<KeyProvider.KeyVersion> ret = new ArrayList<KeyProvider.KeyVersion>();
        try (AutoClosableLock.AutoClosableReadLock ignored = new AutoClosableLock.AutoClosableReadLock(this.lock);){
            KeyProvider.Metadata km = this.getMetadata(name);
            if (km != null) {
                int latestVersion = km.getVersions();
                for (int i = 0; i < latestVersion; ++i) {
                    String versionName = RangerKeyStoreProvider.buildVersionName((String)name, (int)i);
                    KeyProvider.KeyVersion v = this.getKeyVersion(versionName);
                    if (v == null) continue;
                    ret.add(v);
                }
            }
        }
        if (logger.isDebugEnabled()) {
            logger.debug("<== getKeyVersions({}): count={}", (Object)name, (Object)ret.size());
        }
        return ret;
    }

    public List<String> getKeys() throws IOException {
        if (logger.isDebugEnabled()) {
            logger.debug("==> getKeys()");
        }
        ArrayList<String> ret = new ArrayList<String>();
        this.reloadKeys();
        Enumeration<String> e = this.dbStore.engineAliases();
        while (e.hasMoreElements()) {
            String alias = e.nextElement();
            if (alias.contains("@")) continue;
            ret.add(alias);
        }
        if (logger.isDebugEnabled()) {
            logger.debug("<== getKeys(): count={}", (Object)ret.size());
        }
        return ret;
    }

    public KeyProvider.Metadata getMetadata(String name) throws IOException {
        KeyProvider.Metadata ret;
        Throwable throwable;
        if (logger.isDebugEnabled()) {
            logger.debug("==> getMetadata({})", (Object)name);
        }
        boolean addToCache = false;
        try {
            throwable = null;
            try (AutoClosableLock.AutoClosableReadLock ignored = new AutoClosableLock.AutoClosableReadLock(this.lock);){
                ret = this.cache.get(name);
                if (ret == null) {
                    if (!this.dbStore.engineContainsAlias(name)) {
                        this.dbStore.engineLoad(null, this.masterKey);
                    }
                    if (this.dbStore.engineContainsAlias(name)) {
                        if (this.keyVaultEnabled) {
                            ret = this.dbStore.engineGetKeyMetadata(name);
                            addToCache = ret != null;
                        } else {
                            Key key = this.dbStore.engineGetKey(name, this.masterKey);
                            if (key != null) {
                                ret = ((KeyMetadata)key).metadata;
                                addToCache = ret != null;
                            }
                        }
                    }
                }
            }
            catch (Throwable throwable2) {
                throwable = throwable2;
                throw throwable2;
            }
        }
        catch (NoSuchAlgorithmException e) {
            throw new IOException("Can't get algorithm for " + name, e);
        }
        catch (UnrecoverableKeyException e) {
            throw new IOException("Can't recover key for " + name, e);
        }
        catch (Exception e) {
            throw new IOException("Please try again ", e);
        }
        if (ret != null && addToCache) {
            throwable = null;
            try (AutoClosableLock.AutoClosableTryWriteLock writeLock = new AutoClosableLock.AutoClosableTryWriteLock(this.lock);){
                if (writeLock.isLocked()) {
                    this.cache.put(name, ret);
                } else {
                    logger.debug("{} not added to cache - writeLock couldn't be obtained", (Object)name);
                }
            }
            catch (Throwable throwable3) {
                throwable = throwable3;
                throw throwable3;
            }
        }
        if (logger.isDebugEnabled()) {
            logger.debug("<== getMetadata({}): ret={}", (Object)name, (Object)ret);
        }
        return ret;
    }

    public KeyProvider.KeyVersion rollNewVersion(String name, byte[] material) throws IOException {
        if (logger.isDebugEnabled()) {
            logger.debug("==> rollNewVersion({})", (Object)name);
        }
        KeyProvider.KeyVersion ret = null;
        try (AutoClosableLock.AutoClosableWriteLock ignored = new AutoClosableLock.AutoClosableWriteLock(this.lock);){
            this.reloadKeys();
            KeyProvider.Metadata meta = this.getMetadata(name);
            if (meta == null) {
                throw new IOException("Key " + name + " not found");
            }
            if (meta.getBitLength() != 8 * material.length) {
                throw new IOException("Wrong key length. Required " + meta.getBitLength() + ", but got " + 8 * material.length);
            }
            int nextVersion = meta.addVersion();
            String versionName = RangerKeyStoreProvider.buildVersionName((String)name, (int)nextVersion);
            ret = this.innerSetKeyVersion(name, versionName, material, meta);
        }
        if (logger.isDebugEnabled()) {
            logger.debug("<== rollNewVersion({}): ret={}", (Object)name, (Object)ret);
        }
        return ret;
    }

    private static Configuration getConfiguration(boolean loadHadoopDefaults, String ... resources) {
        if (logger.isDebugEnabled()) {
            logger.debug("==> getConfiguration()");
        }
        Configuration conf = new Configuration(loadHadoopDefaults);
        String confDir = System.getProperty(KMS_CONFIG_DIR);
        if (confDir != null) {
            try {
                Path confPath = new Path(confDir);
                if (!confPath.isUriPathAbsolute()) {
                    throw new RuntimeException("System property 'kms.config.dir' must be an absolute path: " + confDir);
                }
                for (String resource : resources) {
                    conf.addResource(new URL("file://" + new Path(confDir, resource).toUri()));
                }
            }
            catch (MalformedURLException ex) {
                logger.error("getConfiguration() error", (Throwable)ex);
                throw new RuntimeException(ex);
            }
        } else {
            for (String resource : resources) {
                conf.addResource(resource);
            }
        }
        if (logger.isDebugEnabled()) {
            logger.debug("<== getConfiguration()");
        }
        return conf;
    }

    private static void getFromJceks(Configuration conf, String path, String alias, String key) {
        if (logger.isDebugEnabled()) {
            logger.debug("==> getFromJceks()");
        }
        if (conf != null) {
            String pathValue = conf.get(path);
            String aliasValue = conf.get(alias);
            if (pathValue != null && aliasValue != null) {
                String storeType = conf.get("ranger.keystore.file.type", KeyStore.getDefaultType());
                String xaDBPassword = CredentialReader.getDecryptedString((String)pathValue.trim(), (String)aliasValue.trim(), (String)storeType);
                if (xaDBPassword != null && !xaDBPassword.trim().isEmpty() && !xaDBPassword.trim().equalsIgnoreCase("none")) {
                    conf.set(key, xaDBPassword);
                } else {
                    logger.info("Credential keystore password not applied for KMS; clear text password shall be applicable");
                }
            }
        }
        if (logger.isDebugEnabled()) {
            logger.debug("<== getFromJceks()");
        }
    }

    private char[] generateAndGetMasterKey(RangerKMSMKI masterKeyProvider, String password) {
        char[] ret;
        if (logger.isDebugEnabled()) {
            logger.debug("==> generateAndGetMasterKey()");
        }
        try {
            masterKeyProvider.generateMasterKey(password);
        }
        catch (Throwable cause) {
            throw new RuntimeException("Error while generating Ranger Master key, Error - ", cause);
        }
        try {
            ret = masterKeyProvider.getMasterKey(password).toCharArray();
        }
        catch (Throwable cause) {
            throw new RuntimeException("Error while getting Ranger Master key, Error - ", cause);
        }
        if (logger.isDebugEnabled()) {
            logger.debug("<== generateAndGetMasterKey()");
        }
        return ret;
    }

    private void loadKeys(char[] masterKey) throws NoSuchAlgorithmException, CertificateException, IOException {
        if (logger.isDebugEnabled()) {
            logger.debug("==> loadKeys()");
        }
        this.dbStore.engineLoad(null, masterKey);
        if (logger.isDebugEnabled()) {
            logger.debug("<== loadKeys()");
        }
    }

    private KeyProvider.KeyVersion innerSetKeyVersion(String name, String versionName, byte[] material, KeyProvider.Metadata meta) throws IOException {
        if (logger.isDebugEnabled()) {
            logger.debug("==> innerSetKeyVersion(name={}, versionName={})", (Object)name, (Object)versionName);
        }
        this.saveKey(name, meta);
        try {
            String cipher = meta.getCipher();
            int bitLength = meta.getBitLength();
            String description = meta.getDescription();
            int version = meta.getVersions();
            String attribute = JsonUtilsV2.mapToJson((Map)meta.getAttributes());
            if (this.keyVaultEnabled) {
                this.dbStore.addSecureKeyByteEntry(versionName, new SecretKeySpec(material, cipher), cipher, bitLength, description, version, attribute);
            } else {
                this.dbStore.addKeyEntry(versionName, new SecretKeySpec(material, cipher), this.masterKey, cipher, bitLength, description, version, attribute);
            }
        }
        catch (Exception e) {
            throw new IOException("Can't store key " + versionName, e);
        }
        this.changed = true;
        KeyProvider.KeyVersion ret = new KeyProvider.KeyVersion(name, versionName, material);
        if (logger.isDebugEnabled()) {
            logger.debug("<== innerSetKeyVersion(name={}, versionName={}): ret={}", new Object[]{name, versionName, ret});
        }
        return ret;
    }

    private void saveKey(String name, KeyProvider.Metadata metadata) throws IOException {
        try {
            String attributes = JsonUtilsV2.mapToJson((Map)metadata.getAttributes());
            if (this.keyVaultEnabled) {
                Key ezkey = new KeyMetadata(metadata);
                if (ezkey.getEncoded().length == 0) {
                    KeyGenerator keyGenerator = KeyGenerator.getInstance(metadata.getAlgorithm());
                    keyGenerator.init(metadata.getBitLength());
                    byte[] key = keyGenerator.generateKey().getEncoded();
                    ezkey = new SecretKeySpec(key, metadata.getCipher());
                }
                this.dbStore.addSecureKeyByteEntry(name, ezkey, metadata.getCipher(), metadata.getBitLength(), metadata.getDescription(), metadata.getVersions(), attributes);
            } else {
                this.dbStore.addKeyEntry(name, new KeyMetadata(metadata), this.masterKey, metadata.getAlgorithm(), metadata.getBitLength(), metadata.getDescription(), metadata.getVersions(), attributes);
            }
            this.cache.put(name, metadata);
        }
        catch (Exception e) {
            throw new IOException("Can't set metadata key " + name, e);
        }
    }

    private void reloadKeys() throws IOException {
        if (logger.isDebugEnabled()) {
            logger.debug("==> reloadKeys()");
        }
        try (AutoClosableLock.AutoClosableWriteLock ignored = new AutoClosableLock.AutoClosableWriteLock(this.lock);){
            this.cache.clear();
            this.loadKeys(this.masterKey);
        }
        catch (NoSuchAlgorithmException | CertificateException e) {
            throw new IOException("Can't load Keys");
        }
        if (logger.isDebugEnabled()) {
            logger.debug("<== reloadKeys()");
        }
    }

    public static class KeyMetadata
    implements Key,
    Serializable {
        private static final long serialVersionUID = 8405872419967874451L;
        KeyProvider.Metadata metadata;

        protected KeyMetadata(KeyProvider.Metadata meta) {
            this.metadata = meta;
        }

        @Override
        public String getAlgorithm() {
            return this.metadata.getCipher();
        }

        @Override
        public String getFormat() {
            return RangerKeyStoreProvider.KEY_METADATA;
        }

        @Override
        public byte[] getEncoded() {
            return new byte[0];
        }

        private void writeObject(ObjectOutputStream out) throws IOException {
            byte[] serialized = this.metadata.serialize();
            out.writeInt(serialized.length);
            out.write(serialized);
        }

        private void readObject(ObjectInputStream in) throws IOException, ClassNotFoundException {
            byte[] buf = new byte[in.readInt()];
            in.readFully(buf);
            this.metadata = new KeyProvider.Metadata(buf);
        }
    }

    public static class Factory
    extends KeyProviderFactory {
        public KeyProvider createProvider(URI providerName, Configuration conf) {
            if (logger.isDebugEnabled()) {
                logger.debug("==> createProvider({})", (Object)providerName);
            }
            RangerKeyStoreProvider ret = null;
            try {
                if (RangerKeyStoreProvider.SCHEME_NAME.equals(providerName.getScheme())) {
                    ret = new RangerKeyStoreProvider(conf);
                } else {
                    logger.warn(providerName.getScheme() + ": unrecognized schema");
                }
            }
            catch (Throwable e) {
                logger.error("createProvider() error", e);
            }
            if (logger.isDebugEnabled()) {
                logger.debug("<== createProvider({})", (Object)providerName);
            }
            return ret;
        }
    }
}

