/*
 * ExecTemplate.java
 *
 * Brazil project web application toolkit,
 * export version: 2.1 
 * Copyright (c) 2001-2004 Sun Microsystems, Inc.
 *
 * Sun Public License Notice
 *
 * The contents of this file are subject to the Sun Public License Version 
 * 1.0 (the "License"). You may not use this file except in compliance with 
 * the License. A copy of the License is included as the file "license.terms",
 * and also available at http://www.sun.com/
 * 
 * The Original Code is from:
 *    Brazil project web application toolkit release 2.1.
 * The Initial Developer of the Original Code is: suhler.
 * Portions created by suhler are Copyright (C) Sun Microsystems, Inc.
 * All Rights Reserved.
 * 
 * Contributor(s): suhler.
 *
 * Version:  2.3
 * Created by suhler on 01/05/21
 * Last modified by suhler on 04/11/30 15:19:43
 */

package sunlabs.brazil.sunlabs;

import java.io.ByteArrayOutputStream;
import java.io.OutputStream;
import sunlabs.brazil.template.RewriteContext;
import sunlabs.brazil.template.Template;
import sunlabs.brazil.util.http.HttpInputStream;

/**
 * template to exec a program, and return its arguments into request
 * properties .
 * <p>This template processes the <code>&lt;exec ...&gt;</code> tag.
 * The following attributes are supported. ${...} substitutions are
 * preformed before the command is run.
 * <dl class=attrib>
 * <dt>command
 * <dd>
 * The command to run.  The environment (and path) are inherited
 * from the server.  This is a required parameter.
 * <dt>usesh
 * <dd>There is a bug in "exec" that prevents passing arguments to a
 * command with embedded whitespace.  If this flag is present, then
 * the command "/bin/sh -c [command]" is run.  This only works on systems
 * where "/bin/sh" may be executed.
 * <dt>prepend
 * <dd>The name prepended to the properties produced by this tag
 * <dt>stdin
 * <dd>The standard input to send to the command (if any)
 * <dt>encoding
 * <dd>The character set encoding to use when converting the stdout
 * and stderr properties.  If no encoding attribute is present, the
 * encoding property is used instead.  Defaults to the default encoding.
 * </dl>
 * The following request properties are set as a side effect:
 * <dl class=props>
 * <dt>stdout
 * <dd>The standard output produced by the program, converted to a String
 * using the default encoding.
 * <dt>stderr
 * <dd>The standard error output produced by the program, converted to
 * a String using the default encoding.
 * <dt>code
 * <dd>The exit code for the program.
 * <dt>error
 * <dd>The error message, if something went wrong.
 * </dl>
 * <p>
 * Currently, there is no way to set the environment or current
 * directory for the program.
 *
 * @author      Stephen Uhler
 * @version		2.3
 */

public class
ExecTemplate extends Template {

    public void tag_exec(RewriteContext hr) {
        debug(hr);
	boolean usesh = hr.isTrue("usesh");
	String command = hr.get("command");
	if (command == null) {
	    debug(hr, "Missing command attribute");
	    return;
	}
	String prepend = hr.get("prepend");	// property names prefix
	String encoding = hr.get("encoding", null);
	if (prepend == null) {
	    prepend = hr.prefix + ".";
	} else if (!prepend.endsWith(".")) {
	    prepend = prepend + ".";
	}

	try {
	    Process process;
	    if (usesh) {
		String sh[] = {"/bin/sh", "-c", ""};
		sh[2] = command;
		process = Runtime.getRuntime().exec(sh);
	    } else {
		process = Runtime.getRuntime().exec(command);
	    }
	    HttpInputStream in = new HttpInputStream(process.getInputStream());
	    HttpInputStream err = new HttpInputStream(process.getErrorStream());

	    // If we have stdin data in a property, send it to the process

	    String stdin = hr.get("stdin");
	    if (stdin != null) {
		OutputStream out = process.getOutputStream();
		out.write(stdin.getBytes());
		out.close();
	    }

	    // Read the streams into the proper properties

	    ByteArrayOutputStream buff = new ByteArrayOutputStream();
	    in.copyTo(buff);
	    in.close();
	    String enc = null;
	    if (encoding != null) {
	       enc = buff.toString(encoding);
	    } else {
	       enc = buff.toString();
	    }
	    hr.request.props.put(prepend + "stdout", enc);
	    buff.reset();

	    err.copyTo(buff);
	    err.close();
	    if (encoding != null) {
	       enc = buff.toString(encoding);
	    } else {
	       enc = buff.toString();
	    }
	    hr.request.props.put(prepend + "stderr", enc);

	    process.waitFor();

	    hr.request.props.put(prepend + "code", "" + process.exitValue());
	} catch (Exception e) {
	    hr.request.props.put(prepend + "error", e.getMessage());
	}

    }
}
