package sunlabs.brazil.rexxshell;
//Leo Chao, August 8, 2002
import sunlabs.brazil.server.Request;
import sunlabs.brazil.server.Server;
import sunlabs.brazil.template.RewriteContext;
import sunlabs.brazil.template.Template;
import COM.ibm.netrexx.process.NetRexxA;
import java.io.FileOutputStream;
import java.io.PrintWriter;
import java.io.PrintStream;
import java.io.ByteArrayOutputStream;
import java.lang.reflect.Method;
import sunlabs.brazil.util.Format;

/**
 * ReXXShell is a template that enables users to use embedded netrexx code within
 * their HTML pages.  Simply use the &lg;rexxshell&gt; tag and place all the netrexx
 * code within.  The current implementation is, to put it simply, bad.  There must
 * be less hackish ways to do this, but this implementations keeps the netrexx
 * code completely intact and as a seperate package.  All stdout output generated
 * from the screen is redirected to the HTML page, stderr output is not by default.
 * <p>
 * hr.request, hr.server and hr.prefix are visible to the rexxshell script as 
 * <i>req</i>, <i>serv</i>, and <i>pre</i> respectively.
 * <p>
 * Configuration File Parameters:<br>
 * <dl class=props>
 * <dt>stdout</dt>
 * <dd>One of <i>page</i> or <i>console</i>.  Defaults to <i>page</i>.</dd>
 * <dt>stderr</dt>
 * <dd>One of <i>page</i> or <i>console</i>.  Defaults to <i>console</i>.</dd>
 * </dl>
 */
public class ReXXShell extends Template
{
    private NetRexxA interpreter = null;
    private Server cSer;
    private boolean cPipeOut = true;
    private boolean cPipeErr = false;
    public static Request cRequest;
    public static Server cServer;
    public static String cPrefix;

    /**
     * Initializes the template, does NOT start off an interpreter.
     * @param	hr
     *		The request and associated HTML document that will be
     *		processed.
     * @return true always 
     */
    public boolean init(RewriteContext hr)
    {
	String pout, perr;
	if (interpreter == null) {
	    interpreter = new NetRexxA();
	}

	pout = hr.request.props.getProperty("query.stdout");
	if (pout == null) {
	    pout = hr.request.props.getProperty(hr.prefix+"stdout","page");
	}

	if (pout.equals("console")) {
	    cPipeOut = false;
	}

	perr = hr.request.props.getProperty("query.stderr");
	if (perr == null) {
	    perr = hr.request.props.getProperty(hr.prefix+"stderr","console");
	}

	if (perr.equals("page")) {
	    cPipeErr = true;
	}

	cSer = hr.server;
	return true;
    }

    /**
     * Implements the rexxshell tag, reads the body of the tag, puts that into a
     * wrapper NetRexx script.  Then parses and interprets the whole thing.
     *
     * @param	hr
     *		The request and associated HTML document that will be
     *		processed.
     */
    public void tag_rexxshell(RewriteContext hr)
    {
	boolean eval = hr.isTrue("eval");
	hr.accumulate(false);
	hr.nextToken();

	String script = hr.getBody();

	if (eval) {
	    script =  Format.subst(hr.request.props, script);
	}

	if (interpreter != null) {
	    cRequest=hr.request;
	    cServer=hr.server;
	    cPrefix=hr.prefix;
	    //Setup a wrapper script for proper execution and allows
	    //us to put the namespace in the script's scope
	    script = "import sunlabs.brazil.template.RewriteContext\n"+
		     "import sunlabs.brazil.properties.PropertiesList\n"+
		     "import sunlabs.brazil.rexxshell.ReXXShell\n"+
		     "import sunlabs.brazil.server.Request\n"+
		     "import sunlabs.brazil.server.Server\n\n"+
		     "class RxScript extends ReXXShell\n" +
		     "properties public\n"+
		     "  req=Request ReXXShell.cRequest\n"+
		     "  serv=Server ReXXShell.cServer\n"+
		     "  pre=String ReXXShell.cPrefix"+
		     "\n"+
		     "method RxScript public\n"+
		     "  this.script()\n\n"+
		     "method _script_run public static\n"+
		     "  runner=RxScript()\n"+  //Causes a warning, need a detour
		     "method script public\n"+		     
		     script;
	    String returnCode = RexxEval(script, hr);
	    if (!returnCode.equals("OK")) {
		hr.server.log(Server.LOG_ERROR, "rexxshell interpreter", "Error: " + returnCode);
	    }
	}
	hr.nextToken();
	hr.accumulate(true);
    }

    private String RexxEval(String script, RewriteContext hr)
    {
	String[] files = {"RxScript.nrx"};
	String[] args  = {"verbose0"};
	PrintWriter pw;
	FileOutputStream fos;	

	try {
	    fos = new FileOutputStream(files[0]);
	    pw = new PrintWriter(fos);
	} catch (Exception e) {
	    cSer.log(Server.LOG_ERROR, this, "File " + files[0] + " cannot be created");
	    return "File Creation Error";
	}
      
	cSer.log(Server.LOG_DIAGNOSTIC, this, "ReXX Script:\n"+script+"\n--end script--");
	try {
	    pw.write(script);

	    pw.flush();
	    pw.close();
	    fos.close();
	} catch (Exception e) {
	    cSer.log(Server.LOG_ERROR, this, "Error writing to " + files[0]);
	    return "File Access Error";
	}

	if (interpreter.parse(files, args)) {
	    Class rxclass = interpreter.getClassObject(null, "RxScript");
	    
	    try {
		Method rxmain = rxclass.getMethod("_script_run", null);
		PrintStream out = System.out;
		PrintStream err = System.err;
		ByteArrayOutputStream baos = new ByteArrayOutputStream();
		PrintStream pageOut = new PrintStream(baos);
		
		if (cPipeOut)
		    System.setOut(pageOut);
		if (cPipeErr)
		    System.setErr(pageOut);

		rxmain.invoke(null, null);
		pageOut.flush();
		System.setOut(out);
		System.setErr(err);

		hr.append(baos.toString());
		pageOut.close();
	    } catch (Exception e) {
		cSer.log(Server.LOG_ERROR, this, "Cannot interpret script");
		return "Exception was:\n"+e;
	    }

	} else {
	    cSer.log(Server.LOG_ERROR, "ReXX Script:", "Parse Error");
	    return "Parse Error";
	}

	return "OK";
    }
}


