# ABSTRACT: Driver for the Czech tagset of the CoNLL 2009 Shared Task.
# This tagset differs slightly from the Czech tagset of CoNLL 2006 and 2007.
# Copyright © 2009, 2014 Dan Zeman <zeman@ufal.mff.cuni.cz>

package Lingua::Interset::Tagset::CS::Conll2009;
use strict;
use warnings;
our $VERSION = '2.050';

use utf8;
use open ':utf8';
use namespace::autoclean;
use Moose;
# Note that we could derive CS::Conll2009 from CS::Conll instead of directly from CS::Pdt but the double conversions would slow down everything.
extends 'Lingua::Interset::Tagset::CS::Pdt';



#------------------------------------------------------------------------------
# Creates atomic drivers for 12 surface features (11 inherited from Pdt).
#------------------------------------------------------------------------------
sub _create_atoms
{
    my $self = shift;
    my $atoms = $self->SUPER::_create_atoms();
    # SEMANTIC CLASS ####################
    # Sem contains the character after ";" in lemma suffix. It encodes types of proper nouns (named entities):
    $atoms->{nametype} = $self->create_simple_atom
    (
        'intfeature' => 'nametype',
        'simple_decode_map' =>
        {
            'G' => 'geo',
            'Y' => 'giv',
            'S' => 'sur',
            'E' => 'nat',
            'R' => 'pro',
            'K' => 'com',
            'm' => 'oth',
            'o' => 'col',
            'H' => 'che',
            'U' => 'med',
            'L' => 'sci',
            'j' => 'jus',
            'g' => 'tec',
            'c' => 'cel',
            'y' => 'hob',
            'b' => 'fin',
            'u' => 'cul',
            'w' => 'spo',
            'p' => 'gov',
            'z' => 'env'
        }
    );
    return $atoms;
}



#------------------------------------------------------------------------------
# Decodes a physical tag (string) and returns the corresponding feature
# structure.
#------------------------------------------------------------------------------
sub decode
{
    my $self = shift;
    my $tag = shift;
    my %conll_features;
    my $pdt = _conll_to_pdt($tag, \%conll_features);
    my $fs = $self->SUPER::decode($pdt);
    if($conll_features{Sem})
    {
        my $atoms = $self->atoms();
        $atoms->{nametype}->decode_and_merge_hard($conll_features{Sem}, $fs);
        $fs->set_nountype('prop');
    }
    # Here we could set $fs->set_tagset('cs::conll2009') but we will not so that all
    # the descendants of cs::pdt can share the same feature structures.
    # (The cs::conll feature structures are not identical to cs::pdt but they do
    # not add their own 'other' values and that matters.)
    return $fs;
}



#------------------------------------------------------------------------------
# Takes feature structure and returns the corresponding physical tag (string).
#------------------------------------------------------------------------------
sub encode
{
    my $self = shift;
    my $fs = shift; # Lingua::Interset::FeatureStructure
    my $pdt = $self->SUPER::encode($fs);
    my $conll = _pdt_to_conll($pdt);
    if($fs->is_proper_noun() || $fs->nametype() ne '')
    {
        my $atoms = $self->atoms();
        my $sem = $atoms->{nametype}->encode($fs);
        $sem = 'm' if($sem eq '');
        if($conll =~ s/_$//)
        {
            $conll .= "Sem=$sem";
        }
        else
        {
            $conll .= "|Sem=$sem";
        }
    }
    return $conll;
}



#------------------------------------------------------------------------------
# Converts a PDT tag to CoNLL 2009 format (two columns, POS and FEAT, separated
# by tab). This is similar but not identical to conversion between PDT and
# CoNLL 2006. We could derive cs::conll2009 from cs::conll instead of cs::pdt
# but it would slow down everything significantly (two translations instead of
# one).
#------------------------------------------------------------------------------
sub _pdt_to_conll
{
    my $tag = shift;
    my @chars = split(//, $tag);
    my $pos = shift(@chars);
    my @features_conll;
    foreach my $name (qw(SubPOS Gen Num Cas PGe PNu Per Ten Gra Neg Voi Rs1 Rs2 Var))
    {
        my $char = shift(@chars);
        if($char ne '-')
        {
            push(@features_conll, "$name=$char");
        }
    }
    my $features = (@features_conll) ? join('|', @features_conll) : '_';
    $tag = "$pos\t$features";
    # For some reason, CoNLL 2009 data do not set number and person of the word "by" while older data did so.
    # In fact, PDT 2.0 morphology returns "Vc-------------" for this word, and only CoNLL 2006 data had "Num=X|Per=3".
    # CoNLL 2007 data did not have person for "by" but otherwise they can be decoded using the old cs::conll driver.
    if($tag eq "V\tSubPOS=c|Num=X|Per=3")
    {
        $tag = "V\tSubPOS=c";
    }
    return $tag;
}



#------------------------------------------------------------------------------
# Converts a CoNLL 2009 tag (except the Sem feature) to PDT.
# This is similar but not identical to conversion between PDT and CoNLL 2006.
# We could derive cs::conll2009 from cs::conll instead of cs::pdt but it would
# slow down everything significantly (two translations instead of one).
#------------------------------------------------------------------------------
sub _conll_to_pdt
{
    my $tag = shift;
    my $features_conll = shift; # output parameter: hash reference; the caller will want to access the Sem feature!
    my %storage;
    if(ref($features_conll) ne 'HASH')
    {
        $features_conll = \%storage;
    }
    # For some reason, CoNLL 2009 data do not set number and person of the word "by" while older data did so.
    # In fact, PDT 2.0 morphology returns "Vc-------------" for this word, and only CoNLL 2006 data had "Num=X|Per=3".
    # CoNLL 2007 data did not have person for "by" but otherwise they can be decoded using the old cs::conll driver.
    if($tag eq "V\tSubPOS=c")
    {
        $tag = "V\tSubPOS=c|Num=X|Per=3";
    }
    # two components: part of speech, features
    # example: N\tSubPOS=N|Gen=M|Num=S|Cas=1|Neg=A
    my ($pos, $features) = split(/\s+/, $tag);
    # The CoNLL tagset is derived from the PDT tagset.
    # Part of speech (coarse-grained) is the first character of the PDT tag.
    # Features are the rest: SubPOS Gen Num Cas PGe PNu Per Ten Gra Neg Voi Rs1 Rs2 Var
    # The Sem feature comes from PDT lemma, not tag.
    my $pdttag;
    if($features eq '_')
    {
        $pdttag = $pos.'--------------';
    }
    else
    {
        $pdttag = $pos;
        my @features_conll = split(/\|/, $features);
        foreach my $f (@features_conll)
        {
            if($f =~ m/^(\w+)=(.)$/)
            {
                $features_conll->{$1} = $2;
            }
        }
        foreach my $name (qw(SubPOS Gen Num Cas PGe PNu Per Ten Gra Neg Voi Rs1 Rs2 Var))
        {
            if(defined($features_conll->{$name}) && $features_conll->{$name} ne '')
            {
                $pdttag .= $features_conll->{$name};
            }
            else
            {
                $pdttag .= '-';
            }
        }
    }
    return $pdttag;
}



#------------------------------------------------------------------------------
# Returns reference to list of known tags.
#------------------------------------------------------------------------------
sub list
{
    my $self = shift;
    my $list = <<end_of_list
A	SubPOS=.
A	SubPOS=2|Neg=A
A	SubPOS=2|Neg=A|Sem=G
A	SubPOS=2|Neg=A|Sem=K
A	SubPOS=2|Neg=A|Sem=S
A	SubPOS=2|Neg=A|Sem=Y
A	SubPOS=2|Neg=A|Sem=m
A	SubPOS=2|Neg=A|Var=1
A	SubPOS=2|Neg=N
A	SubPOS=A|Gen=F|Num=D|Cas=7|Gra=1|Neg=A
A	SubPOS=A|Gen=F|Num=D|Cas=7|Gra=1|Neg=A|Var=6
A	SubPOS=A|Gen=F|Num=D|Cas=7|Gra=1|Neg=N
A	SubPOS=A|Gen=F|Num=D|Cas=7|Gra=1|Neg=N|Var=6
A	SubPOS=A|Gen=F|Num=D|Cas=7|Gra=2|Neg=A
A	SubPOS=A|Gen=F|Num=D|Cas=7|Gra=2|Neg=A|Var=6
A	SubPOS=A|Gen=F|Num=D|Cas=7|Gra=2|Neg=N
A	SubPOS=A|Gen=F|Num=D|Cas=7|Gra=2|Neg=N|Var=6
A	SubPOS=A|Gen=F|Num=D|Cas=7|Gra=3|Neg=A
A	SubPOS=A|Gen=F|Num=D|Cas=7|Gra=3|Neg=A|Var=6
A	SubPOS=A|Gen=F|Num=D|Cas=7|Gra=3|Neg=N
A	SubPOS=A|Gen=F|Num=D|Cas=7|Gra=3|Neg=N|Var=6
A	SubPOS=A|Gen=F|Num=P|Cas=1|Gra=1|Neg=A
A	SubPOS=A|Gen=F|Num=P|Cas=1|Gra=1|Neg=A|Sem=G
A	SubPOS=A|Gen=F|Num=P|Cas=1|Gra=1|Neg=A|Sem=K
A	SubPOS=A|Gen=F|Num=P|Cas=1|Gra=1|Neg=A|Sem=R
A	SubPOS=A|Gen=F|Num=P|Cas=1|Gra=1|Neg=A|Sem=U
A	SubPOS=A|Gen=F|Num=P|Cas=1|Gra=1|Neg=A|Sem=m
A	SubPOS=A|Gen=F|Num=P|Cas=1|Gra=1|Neg=A|Sem=o
A	SubPOS=A|Gen=F|Num=P|Cas=1|Gra=1|Neg=A|Var=6
A	SubPOS=A|Gen=F|Num=P|Cas=1|Gra=1|Neg=N
A	SubPOS=A|Gen=F|Num=P|Cas=1|Gra=1|Neg=N|Var=6
A	SubPOS=A|Gen=F|Num=P|Cas=1|Gra=2|Neg=A
A	SubPOS=A|Gen=F|Num=P|Cas=1|Gra=2|Neg=A|Var=6
A	SubPOS=A|Gen=F|Num=P|Cas=1|Gra=2|Neg=N
A	SubPOS=A|Gen=F|Num=P|Cas=1|Gra=2|Neg=N|Var=6
A	SubPOS=A|Gen=F|Num=P|Cas=1|Gra=3|Neg=A
A	SubPOS=A|Gen=F|Num=P|Cas=1|Gra=3|Neg=A|Var=6
A	SubPOS=A|Gen=F|Num=P|Cas=1|Gra=3|Neg=N
A	SubPOS=A|Gen=F|Num=P|Cas=1|Gra=3|Neg=N|Var=6
A	SubPOS=A|Gen=F|Num=P|Cas=2|Gra=1|Neg=A
A	SubPOS=A|Gen=F|Num=P|Cas=2|Gra=1|Neg=A|Sem=G
A	SubPOS=A|Gen=F|Num=P|Cas=2|Gra=1|Neg=A|Sem=R
A	SubPOS=A|Gen=F|Num=P|Cas=2|Gra=1|Neg=A|Sem=S
A	SubPOS=A|Gen=F|Num=P|Cas=2|Gra=1|Neg=A|Sem=m
A	SubPOS=A|Gen=F|Num=P|Cas=2|Gra=1|Neg=A|Var=6
A	SubPOS=A|Gen=F|Num=P|Cas=2|Gra=1|Neg=N
A	SubPOS=A|Gen=F|Num=P|Cas=2|Gra=1|Neg=N|Var=6
A	SubPOS=A|Gen=F|Num=P|Cas=2|Gra=2|Neg=A
A	SubPOS=A|Gen=F|Num=P|Cas=2|Gra=2|Neg=A|Var=6
A	SubPOS=A|Gen=F|Num=P|Cas=2|Gra=2|Neg=N
A	SubPOS=A|Gen=F|Num=P|Cas=2|Gra=2|Neg=N|Var=6
A	SubPOS=A|Gen=F|Num=P|Cas=2|Gra=3|Neg=A
A	SubPOS=A|Gen=F|Num=P|Cas=2|Gra=3|Neg=A|Var=6
A	SubPOS=A|Gen=F|Num=P|Cas=2|Gra=3|Neg=N
A	SubPOS=A|Gen=F|Num=P|Cas=2|Gra=3|Neg=N|Var=6
A	SubPOS=A|Gen=F|Num=P|Cas=3|Gra=1|Neg=A
A	SubPOS=A|Gen=F|Num=P|Cas=3|Gra=1|Neg=A|Sem=G
A	SubPOS=A|Gen=F|Num=P|Cas=3|Gra=1|Neg=A|Sem=m
A	SubPOS=A|Gen=F|Num=P|Cas=3|Gra=1|Neg=A|Var=6
A	SubPOS=A|Gen=F|Num=P|Cas=3|Gra=1|Neg=N
A	SubPOS=A|Gen=F|Num=P|Cas=3|Gra=1|Neg=N|Var=6
A	SubPOS=A|Gen=F|Num=P|Cas=3|Gra=2|Neg=A
A	SubPOS=A|Gen=F|Num=P|Cas=3|Gra=2|Neg=A|Var=6
A	SubPOS=A|Gen=F|Num=P|Cas=3|Gra=2|Neg=N
A	SubPOS=A|Gen=F|Num=P|Cas=3|Gra=2|Neg=N|Var=6
A	SubPOS=A|Gen=F|Num=P|Cas=3|Gra=3|Neg=A
A	SubPOS=A|Gen=F|Num=P|Cas=3|Gra=3|Neg=A|Var=6
A	SubPOS=A|Gen=F|Num=P|Cas=3|Gra=3|Neg=N
A	SubPOS=A|Gen=F|Num=P|Cas=3|Gra=3|Neg=N|Var=6
A	SubPOS=A|Gen=F|Num=P|Cas=4|Gra=1|Neg=A
A	SubPOS=A|Gen=F|Num=P|Cas=4|Gra=1|Neg=A|Sem=G
A	SubPOS=A|Gen=F|Num=P|Cas=4|Gra=1|Neg=A|Sem=m
A	SubPOS=A|Gen=F|Num=P|Cas=4|Gra=1|Neg=A|Var=6
A	SubPOS=A|Gen=F|Num=P|Cas=4|Gra=1|Neg=N
A	SubPOS=A|Gen=F|Num=P|Cas=4|Gra=1|Neg=N|Var=6
A	SubPOS=A|Gen=F|Num=P|Cas=4|Gra=2|Neg=A
A	SubPOS=A|Gen=F|Num=P|Cas=4|Gra=2|Neg=A|Var=6
A	SubPOS=A|Gen=F|Num=P|Cas=4|Gra=2|Neg=N
A	SubPOS=A|Gen=F|Num=P|Cas=4|Gra=2|Neg=N|Var=6
A	SubPOS=A|Gen=F|Num=P|Cas=4|Gra=3|Neg=A
A	SubPOS=A|Gen=F|Num=P|Cas=4|Gra=3|Neg=A|Var=6
A	SubPOS=A|Gen=F|Num=P|Cas=4|Gra=3|Neg=N
A	SubPOS=A|Gen=F|Num=P|Cas=4|Gra=3|Neg=N|Var=6
A	SubPOS=A|Gen=F|Num=P|Cas=5|Gra=1|Neg=A
A	SubPOS=A|Gen=F|Num=P|Cas=5|Gra=1|Neg=A|Var=6
A	SubPOS=A|Gen=F|Num=P|Cas=5|Gra=1|Neg=N
A	SubPOS=A|Gen=F|Num=P|Cas=5|Gra=1|Neg=N|Var=6
A	SubPOS=A|Gen=F|Num=P|Cas=5|Gra=2|Neg=A
A	SubPOS=A|Gen=F|Num=P|Cas=5|Gra=2|Neg=A|Var=6
A	SubPOS=A|Gen=F|Num=P|Cas=5|Gra=2|Neg=N
A	SubPOS=A|Gen=F|Num=P|Cas=5|Gra=2|Neg=N|Var=6
A	SubPOS=A|Gen=F|Num=P|Cas=5|Gra=3|Neg=A
A	SubPOS=A|Gen=F|Num=P|Cas=5|Gra=3|Neg=A|Var=6
A	SubPOS=A|Gen=F|Num=P|Cas=5|Gra=3|Neg=N
A	SubPOS=A|Gen=F|Num=P|Cas=5|Gra=3|Neg=N|Var=6
A	SubPOS=A|Gen=F|Num=P|Cas=6|Gra=1|Neg=A
A	SubPOS=A|Gen=F|Num=P|Cas=6|Gra=1|Neg=A|Sem=G
A	SubPOS=A|Gen=F|Num=P|Cas=6|Gra=1|Neg=A|Sem=S
A	SubPOS=A|Gen=F|Num=P|Cas=6|Gra=1|Neg=A|Sem=m
A	SubPOS=A|Gen=F|Num=P|Cas=6|Gra=1|Neg=A|Sem=o
A	SubPOS=A|Gen=F|Num=P|Cas=6|Gra=1|Neg=A|Var=6
A	SubPOS=A|Gen=F|Num=P|Cas=6|Gra=1|Neg=N
A	SubPOS=A|Gen=F|Num=P|Cas=6|Gra=1|Neg=N|Var=6
A	SubPOS=A|Gen=F|Num=P|Cas=6|Gra=2|Neg=A
A	SubPOS=A|Gen=F|Num=P|Cas=6|Gra=2|Neg=A|Var=6
A	SubPOS=A|Gen=F|Num=P|Cas=6|Gra=2|Neg=N
A	SubPOS=A|Gen=F|Num=P|Cas=6|Gra=2|Neg=N|Var=6
A	SubPOS=A|Gen=F|Num=P|Cas=6|Gra=3|Neg=A
A	SubPOS=A|Gen=F|Num=P|Cas=6|Gra=3|Neg=A|Var=6
A	SubPOS=A|Gen=F|Num=P|Cas=6|Gra=3|Neg=N
A	SubPOS=A|Gen=F|Num=P|Cas=6|Gra=3|Neg=N|Var=6
A	SubPOS=A|Gen=F|Num=P|Cas=7|Gra=1|Neg=A
A	SubPOS=A|Gen=F|Num=P|Cas=7|Gra=1|Neg=A|Var=6
A	SubPOS=A|Gen=F|Num=P|Cas=7|Gra=1|Neg=A|Var=7
A	SubPOS=A|Gen=F|Num=P|Cas=7|Gra=1|Neg=N
A	SubPOS=A|Gen=F|Num=P|Cas=7|Gra=1|Neg=N|Var=6
A	SubPOS=A|Gen=F|Num=P|Cas=7|Gra=1|Neg=N|Var=7
A	SubPOS=A|Gen=F|Num=P|Cas=7|Gra=2|Neg=A
A	SubPOS=A|Gen=F|Num=P|Cas=7|Gra=2|Neg=A|Var=6
A	SubPOS=A|Gen=F|Num=P|Cas=7|Gra=2|Neg=A|Var=7
A	SubPOS=A|Gen=F|Num=P|Cas=7|Gra=2|Neg=N
A	SubPOS=A|Gen=F|Num=P|Cas=7|Gra=2|Neg=N|Var=6
A	SubPOS=A|Gen=F|Num=P|Cas=7|Gra=2|Neg=N|Var=7
A	SubPOS=A|Gen=F|Num=P|Cas=7|Gra=3|Neg=A
A	SubPOS=A|Gen=F|Num=P|Cas=7|Gra=3|Neg=A|Var=6
A	SubPOS=A|Gen=F|Num=P|Cas=7|Gra=3|Neg=A|Var=7
A	SubPOS=A|Gen=F|Num=P|Cas=7|Gra=3|Neg=N
A	SubPOS=A|Gen=F|Num=P|Cas=7|Gra=3|Neg=N|Var=6
A	SubPOS=A|Gen=F|Num=P|Cas=7|Gra=3|Neg=N|Var=7
A	SubPOS=A|Gen=F|Num=S|Cas=1|Gra=1|Neg=A
A	SubPOS=A|Gen=F|Num=S|Cas=1|Gra=1|Neg=A|Sem=G
A	SubPOS=A|Gen=F|Num=S|Cas=1|Gra=1|Neg=A|Sem=K
A	SubPOS=A|Gen=F|Num=S|Cas=1|Gra=1|Neg=A|Sem=L
A	SubPOS=A|Gen=F|Num=S|Cas=1|Gra=1|Neg=A|Sem=R
A	SubPOS=A|Gen=F|Num=S|Cas=1|Gra=1|Neg=A|Sem=S
A	SubPOS=A|Gen=F|Num=S|Cas=1|Gra=1|Neg=A|Sem=m
A	SubPOS=A|Gen=F|Num=S|Cas=1|Gra=1|Neg=A|Sem=o
A	SubPOS=A|Gen=F|Num=S|Cas=1|Gra=1|Neg=A|Var=6
A	SubPOS=A|Gen=F|Num=S|Cas=1|Gra=1|Neg=N
A	SubPOS=A|Gen=F|Num=S|Cas=1|Gra=1|Neg=N|Var=6
A	SubPOS=A|Gen=F|Num=S|Cas=1|Gra=2|Neg=A
A	SubPOS=A|Gen=F|Num=S|Cas=1|Gra=2|Neg=A|Var=6
A	SubPOS=A|Gen=F|Num=S|Cas=1|Gra=2|Neg=N
A	SubPOS=A|Gen=F|Num=S|Cas=1|Gra=2|Neg=N|Var=6
A	SubPOS=A|Gen=F|Num=S|Cas=1|Gra=3|Neg=A
A	SubPOS=A|Gen=F|Num=S|Cas=1|Gra=3|Neg=A|Var=6
A	SubPOS=A|Gen=F|Num=S|Cas=1|Gra=3|Neg=N
A	SubPOS=A|Gen=F|Num=S|Cas=1|Gra=3|Neg=N|Var=6
A	SubPOS=A|Gen=F|Num=S|Cas=2|Gra=1|Neg=A
A	SubPOS=A|Gen=F|Num=S|Cas=2|Gra=1|Neg=A|Sem=G
A	SubPOS=A|Gen=F|Num=S|Cas=2|Gra=1|Neg=A|Sem=K
A	SubPOS=A|Gen=F|Num=S|Cas=2|Gra=1|Neg=A|Sem=R
A	SubPOS=A|Gen=F|Num=S|Cas=2|Gra=1|Neg=A|Sem=m
A	SubPOS=A|Gen=F|Num=S|Cas=2|Gra=1|Neg=A|Sem=o
A	SubPOS=A|Gen=F|Num=S|Cas=2|Gra=1|Neg=A|Var=6
A	SubPOS=A|Gen=F|Num=S|Cas=2|Gra=1|Neg=N
A	SubPOS=A|Gen=F|Num=S|Cas=2|Gra=1|Neg=N|Var=6
A	SubPOS=A|Gen=F|Num=S|Cas=2|Gra=2|Neg=A
A	SubPOS=A|Gen=F|Num=S|Cas=2|Gra=2|Neg=A|Var=6
A	SubPOS=A|Gen=F|Num=S|Cas=2|Gra=2|Neg=N
A	SubPOS=A|Gen=F|Num=S|Cas=2|Gra=2|Neg=N|Var=6
A	SubPOS=A|Gen=F|Num=S|Cas=2|Gra=3|Neg=A
A	SubPOS=A|Gen=F|Num=S|Cas=2|Gra=3|Neg=A|Var=6
A	SubPOS=A|Gen=F|Num=S|Cas=2|Gra=3|Neg=N
A	SubPOS=A|Gen=F|Num=S|Cas=2|Gra=3|Neg=N|Var=6
A	SubPOS=A|Gen=F|Num=S|Cas=3|Gra=1|Neg=A
A	SubPOS=A|Gen=F|Num=S|Cas=3|Gra=1|Neg=A|Sem=G
A	SubPOS=A|Gen=F|Num=S|Cas=3|Gra=1|Neg=A|Sem=m
A	SubPOS=A|Gen=F|Num=S|Cas=3|Gra=1|Neg=A|Var=6
A	SubPOS=A|Gen=F|Num=S|Cas=3|Gra=1|Neg=N
A	SubPOS=A|Gen=F|Num=S|Cas=3|Gra=1|Neg=N|Var=6
A	SubPOS=A|Gen=F|Num=S|Cas=3|Gra=2|Neg=A
A	SubPOS=A|Gen=F|Num=S|Cas=3|Gra=2|Neg=A|Var=6
A	SubPOS=A|Gen=F|Num=S|Cas=3|Gra=2|Neg=N
A	SubPOS=A|Gen=F|Num=S|Cas=3|Gra=2|Neg=N|Var=6
A	SubPOS=A|Gen=F|Num=S|Cas=3|Gra=3|Neg=A
A	SubPOS=A|Gen=F|Num=S|Cas=3|Gra=3|Neg=A|Var=6
A	SubPOS=A|Gen=F|Num=S|Cas=3|Gra=3|Neg=N
A	SubPOS=A|Gen=F|Num=S|Cas=3|Gra=3|Neg=N|Var=6
A	SubPOS=A|Gen=F|Num=S|Cas=4|Gra=1|Neg=A
A	SubPOS=A|Gen=F|Num=S|Cas=4|Gra=1|Neg=A|Sem=G
A	SubPOS=A|Gen=F|Num=S|Cas=4|Gra=1|Neg=A|Sem=K
A	SubPOS=A|Gen=F|Num=S|Cas=4|Gra=1|Neg=A|Sem=S
A	SubPOS=A|Gen=F|Num=S|Cas=4|Gra=1|Neg=A|Sem=m
A	SubPOS=A|Gen=F|Num=S|Cas=4|Gra=1|Neg=A|Sem=o
A	SubPOS=A|Gen=F|Num=S|Cas=4|Gra=1|Neg=A|Var=6
A	SubPOS=A|Gen=F|Num=S|Cas=4|Gra=1|Neg=N
A	SubPOS=A|Gen=F|Num=S|Cas=4|Gra=1|Neg=N|Var=6
A	SubPOS=A|Gen=F|Num=S|Cas=4|Gra=2|Neg=A
A	SubPOS=A|Gen=F|Num=S|Cas=4|Gra=2|Neg=A|Var=6
A	SubPOS=A|Gen=F|Num=S|Cas=4|Gra=2|Neg=N
A	SubPOS=A|Gen=F|Num=S|Cas=4|Gra=2|Neg=N|Var=6
A	SubPOS=A|Gen=F|Num=S|Cas=4|Gra=3|Neg=A
A	SubPOS=A|Gen=F|Num=S|Cas=4|Gra=3|Neg=A|Var=6
A	SubPOS=A|Gen=F|Num=S|Cas=4|Gra=3|Neg=N
A	SubPOS=A|Gen=F|Num=S|Cas=4|Gra=3|Neg=N|Var=6
A	SubPOS=A|Gen=F|Num=S|Cas=5|Gra=1|Neg=A
A	SubPOS=A|Gen=F|Num=S|Cas=5|Gra=1|Neg=A|Sem=R
A	SubPOS=A|Gen=F|Num=S|Cas=5|Gra=1|Neg=A|Sem=m
A	SubPOS=A|Gen=F|Num=S|Cas=5|Gra=1|Neg=A|Var=6
A	SubPOS=A|Gen=F|Num=S|Cas=5|Gra=1|Neg=N
A	SubPOS=A|Gen=F|Num=S|Cas=5|Gra=1|Neg=N|Var=6
A	SubPOS=A|Gen=F|Num=S|Cas=5|Gra=2|Neg=A
A	SubPOS=A|Gen=F|Num=S|Cas=5|Gra=2|Neg=A|Var=6
A	SubPOS=A|Gen=F|Num=S|Cas=5|Gra=2|Neg=N
A	SubPOS=A|Gen=F|Num=S|Cas=5|Gra=2|Neg=N|Var=6
A	SubPOS=A|Gen=F|Num=S|Cas=5|Gra=3|Neg=A
A	SubPOS=A|Gen=F|Num=S|Cas=5|Gra=3|Neg=A|Var=6
A	SubPOS=A|Gen=F|Num=S|Cas=5|Gra=3|Neg=N
A	SubPOS=A|Gen=F|Num=S|Cas=5|Gra=3|Neg=N|Var=6
A	SubPOS=A|Gen=F|Num=S|Cas=6|Gra=1|Neg=A
A	SubPOS=A|Gen=F|Num=S|Cas=6|Gra=1|Neg=A|Sem=G
A	SubPOS=A|Gen=F|Num=S|Cas=6|Gra=1|Neg=A|Sem=K
A	SubPOS=A|Gen=F|Num=S|Cas=6|Gra=1|Neg=A|Sem=m
A	SubPOS=A|Gen=F|Num=S|Cas=6|Gra=1|Neg=A|Sem=o
A	SubPOS=A|Gen=F|Num=S|Cas=6|Gra=1|Neg=A|Var=6
A	SubPOS=A|Gen=F|Num=S|Cas=6|Gra=1|Neg=N
A	SubPOS=A|Gen=F|Num=S|Cas=6|Gra=1|Neg=N|Var=6
A	SubPOS=A|Gen=F|Num=S|Cas=6|Gra=2|Neg=A
A	SubPOS=A|Gen=F|Num=S|Cas=6|Gra=2|Neg=A|Var=6
A	SubPOS=A|Gen=F|Num=S|Cas=6|Gra=2|Neg=N
A	SubPOS=A|Gen=F|Num=S|Cas=6|Gra=2|Neg=N|Var=6
A	SubPOS=A|Gen=F|Num=S|Cas=6|Gra=3|Neg=A
A	SubPOS=A|Gen=F|Num=S|Cas=6|Gra=3|Neg=A|Var=6
A	SubPOS=A|Gen=F|Num=S|Cas=6|Gra=3|Neg=N
A	SubPOS=A|Gen=F|Num=S|Cas=6|Gra=3|Neg=N|Var=6
A	SubPOS=A|Gen=F|Num=S|Cas=7|Gra=1|Neg=A
A	SubPOS=A|Gen=F|Num=S|Cas=7|Gra=1|Neg=A|Var=6
A	SubPOS=A|Gen=F|Num=S|Cas=7|Gra=1|Neg=N
A	SubPOS=A|Gen=F|Num=S|Cas=7|Gra=1|Neg=N|Var=6
A	SubPOS=A|Gen=F|Num=S|Cas=7|Gra=2|Neg=A
A	SubPOS=A|Gen=F|Num=S|Cas=7|Gra=2|Neg=A|Var=6
A	SubPOS=A|Gen=F|Num=S|Cas=7|Gra=2|Neg=N
A	SubPOS=A|Gen=F|Num=S|Cas=7|Gra=2|Neg=N|Var=6
A	SubPOS=A|Gen=F|Num=S|Cas=7|Gra=3|Neg=A
A	SubPOS=A|Gen=F|Num=S|Cas=7|Gra=3|Neg=A|Var=6
A	SubPOS=A|Gen=F|Num=S|Cas=7|Gra=3|Neg=N
A	SubPOS=A|Gen=F|Num=S|Cas=7|Gra=3|Neg=N|Var=6
A	SubPOS=A|Gen=F|Num=S|Cas=X|Gra=1|Neg=A
A	SubPOS=A|Gen=F|Num=S|Cas=X|Gra=1|Neg=A|Sem=G
A	SubPOS=A|Gen=F|Num=S|Cas=X|Gra=1|Neg=A|Sem=K
A	SubPOS=A|Gen=F|Num=S|Cas=X|Gra=1|Neg=A|Sem=m
A	SubPOS=A|Gen=F|Num=X|Cas=X|Gra=1|Neg=A
A	SubPOS=A|Gen=F|Num=X|Cas=X|Gra=1|Neg=A|Sem=G
A	SubPOS=A|Gen=F|Num=X|Cas=X|Gra=1|Neg=A|Sem=m
A	SubPOS=A|Gen=F|Num=X|Cas=X|Gra=1|Neg=A|Var=8
A	SubPOS=A|Gen=F|Num=X|Cas=X|Gra=1|Neg=N|Var=8
A	SubPOS=A|Gen=I|Num=P|Cas=1|Gra=1|Neg=A
A	SubPOS=A|Gen=I|Num=P|Cas=1|Gra=1|Neg=A|Sem=G
A	SubPOS=A|Gen=I|Num=P|Cas=1|Gra=1|Neg=A|Sem=R
A	SubPOS=A|Gen=I|Num=P|Cas=1|Gra=1|Neg=A|Sem=m
A	SubPOS=A|Gen=I|Num=P|Cas=1|Gra=1|Neg=A|Var=6
A	SubPOS=A|Gen=I|Num=P|Cas=1|Gra=1|Neg=N
A	SubPOS=A|Gen=I|Num=P|Cas=1|Gra=1|Neg=N|Var=6
A	SubPOS=A|Gen=I|Num=P|Cas=1|Gra=2|Neg=A
A	SubPOS=A|Gen=I|Num=P|Cas=1|Gra=2|Neg=A|Var=6
A	SubPOS=A|Gen=I|Num=P|Cas=1|Gra=2|Neg=N
A	SubPOS=A|Gen=I|Num=P|Cas=1|Gra=2|Neg=N|Var=6
A	SubPOS=A|Gen=I|Num=P|Cas=1|Gra=3|Neg=A
A	SubPOS=A|Gen=I|Num=P|Cas=1|Gra=3|Neg=A|Var=6
A	SubPOS=A|Gen=I|Num=P|Cas=1|Gra=3|Neg=N
A	SubPOS=A|Gen=I|Num=P|Cas=1|Gra=3|Neg=N|Var=6
A	SubPOS=A|Gen=I|Num=P|Cas=2|Gra=1|Neg=A
A	SubPOS=A|Gen=I|Num=P|Cas=2|Gra=1|Neg=A|Sem=G
A	SubPOS=A|Gen=I|Num=P|Cas=2|Gra=1|Neg=A|Sem=S
A	SubPOS=A|Gen=I|Num=P|Cas=2|Gra=1|Neg=A|Sem=m
A	SubPOS=A|Gen=I|Num=P|Cas=2|Gra=1|Neg=A|Var=6
A	SubPOS=A|Gen=I|Num=P|Cas=2|Gra=1|Neg=N
A	SubPOS=A|Gen=I|Num=P|Cas=2|Gra=1|Neg=N|Var=6
A	SubPOS=A|Gen=I|Num=P|Cas=2|Gra=2|Neg=A
A	SubPOS=A|Gen=I|Num=P|Cas=2|Gra=2|Neg=A|Var=6
A	SubPOS=A|Gen=I|Num=P|Cas=2|Gra=2|Neg=N
A	SubPOS=A|Gen=I|Num=P|Cas=2|Gra=2|Neg=N|Var=6
A	SubPOS=A|Gen=I|Num=P|Cas=2|Gra=3|Neg=A
A	SubPOS=A|Gen=I|Num=P|Cas=2|Gra=3|Neg=A|Var=6
A	SubPOS=A|Gen=I|Num=P|Cas=2|Gra=3|Neg=N
A	SubPOS=A|Gen=I|Num=P|Cas=2|Gra=3|Neg=N|Var=6
A	SubPOS=A|Gen=I|Num=P|Cas=3|Gra=1|Neg=A
A	SubPOS=A|Gen=I|Num=P|Cas=3|Gra=1|Neg=A|Sem=G
A	SubPOS=A|Gen=I|Num=P|Cas=3|Gra=1|Neg=A|Sem=m
A	SubPOS=A|Gen=I|Num=P|Cas=3|Gra=1|Neg=A|Var=6
A	SubPOS=A|Gen=I|Num=P|Cas=3|Gra=1|Neg=N
A	SubPOS=A|Gen=I|Num=P|Cas=3|Gra=1|Neg=N|Var=6
A	SubPOS=A|Gen=I|Num=P|Cas=3|Gra=2|Neg=A
A	SubPOS=A|Gen=I|Num=P|Cas=3|Gra=2|Neg=A|Var=6
A	SubPOS=A|Gen=I|Num=P|Cas=3|Gra=2|Neg=N
A	SubPOS=A|Gen=I|Num=P|Cas=3|Gra=2|Neg=N|Var=6
A	SubPOS=A|Gen=I|Num=P|Cas=3|Gra=3|Neg=A
A	SubPOS=A|Gen=I|Num=P|Cas=3|Gra=3|Neg=A|Var=6
A	SubPOS=A|Gen=I|Num=P|Cas=3|Gra=3|Neg=N
A	SubPOS=A|Gen=I|Num=P|Cas=3|Gra=3|Neg=N|Var=6
A	SubPOS=A|Gen=I|Num=P|Cas=4|Gra=1|Neg=A
A	SubPOS=A|Gen=I|Num=P|Cas=4|Gra=1|Neg=A|Var=6
A	SubPOS=A|Gen=I|Num=P|Cas=4|Gra=1|Neg=N
A	SubPOS=A|Gen=I|Num=P|Cas=4|Gra=1|Neg=N|Var=6
A	SubPOS=A|Gen=I|Num=P|Cas=4|Gra=2|Neg=A
A	SubPOS=A|Gen=I|Num=P|Cas=4|Gra=2|Neg=A|Var=6
A	SubPOS=A|Gen=I|Num=P|Cas=4|Gra=2|Neg=N
A	SubPOS=A|Gen=I|Num=P|Cas=4|Gra=2|Neg=N|Var=6
A	SubPOS=A|Gen=I|Num=P|Cas=4|Gra=3|Neg=A
A	SubPOS=A|Gen=I|Num=P|Cas=4|Gra=3|Neg=A|Var=6
A	SubPOS=A|Gen=I|Num=P|Cas=4|Gra=3|Neg=N
A	SubPOS=A|Gen=I|Num=P|Cas=4|Gra=3|Neg=N|Var=6
A	SubPOS=A|Gen=I|Num=P|Cas=5|Gra=1|Neg=A
A	SubPOS=A|Gen=I|Num=P|Cas=5|Gra=1|Neg=A|Var=6
A	SubPOS=A|Gen=I|Num=P|Cas=5|Gra=1|Neg=N
A	SubPOS=A|Gen=I|Num=P|Cas=5|Gra=1|Neg=N|Var=6
A	SubPOS=A|Gen=I|Num=P|Cas=5|Gra=2|Neg=A
A	SubPOS=A|Gen=I|Num=P|Cas=5|Gra=2|Neg=A|Var=6
A	SubPOS=A|Gen=I|Num=P|Cas=5|Gra=2|Neg=N
A	SubPOS=A|Gen=I|Num=P|Cas=5|Gra=2|Neg=N|Var=6
A	SubPOS=A|Gen=I|Num=P|Cas=5|Gra=3|Neg=A
A	SubPOS=A|Gen=I|Num=P|Cas=5|Gra=3|Neg=A|Var=6
A	SubPOS=A|Gen=I|Num=P|Cas=5|Gra=3|Neg=N
A	SubPOS=A|Gen=I|Num=P|Cas=5|Gra=3|Neg=N|Var=6
A	SubPOS=A|Gen=I|Num=P|Cas=6|Gra=1|Neg=A
A	SubPOS=A|Gen=I|Num=P|Cas=6|Gra=1|Neg=A|Sem=R
A	SubPOS=A|Gen=I|Num=P|Cas=6|Gra=1|Neg=A|Sem=m
A	SubPOS=A|Gen=I|Num=P|Cas=6|Gra=1|Neg=A|Var=6
A	SubPOS=A|Gen=I|Num=P|Cas=6|Gra=1|Neg=N
A	SubPOS=A|Gen=I|Num=P|Cas=6|Gra=1|Neg=N|Var=6
A	SubPOS=A|Gen=I|Num=P|Cas=6|Gra=2|Neg=A
A	SubPOS=A|Gen=I|Num=P|Cas=6|Gra=2|Neg=A|Var=6
A	SubPOS=A|Gen=I|Num=P|Cas=6|Gra=2|Neg=N
A	SubPOS=A|Gen=I|Num=P|Cas=6|Gra=2|Neg=N|Var=6
A	SubPOS=A|Gen=I|Num=P|Cas=6|Gra=3|Neg=A
A	SubPOS=A|Gen=I|Num=P|Cas=6|Gra=3|Neg=A|Var=6
A	SubPOS=A|Gen=I|Num=P|Cas=6|Gra=3|Neg=N
A	SubPOS=A|Gen=I|Num=P|Cas=6|Gra=3|Neg=N|Var=6
A	SubPOS=A|Gen=I|Num=P|Cas=7|Gra=1|Neg=A
A	SubPOS=A|Gen=I|Num=P|Cas=7|Gra=1|Neg=A|Sem=G
A	SubPOS=A|Gen=I|Num=P|Cas=7|Gra=1|Neg=A|Sem=m
A	SubPOS=A|Gen=I|Num=P|Cas=7|Gra=1|Neg=A|Sem=o
A	SubPOS=A|Gen=I|Num=P|Cas=7|Gra=1|Neg=A|Var=6
A	SubPOS=A|Gen=I|Num=P|Cas=7|Gra=1|Neg=A|Var=7
A	SubPOS=A|Gen=I|Num=P|Cas=7|Gra=1|Neg=N
A	SubPOS=A|Gen=I|Num=P|Cas=7|Gra=1|Neg=N|Var=6
A	SubPOS=A|Gen=I|Num=P|Cas=7|Gra=1|Neg=N|Var=7
A	SubPOS=A|Gen=I|Num=P|Cas=7|Gra=2|Neg=A
A	SubPOS=A|Gen=I|Num=P|Cas=7|Gra=2|Neg=A|Var=6
A	SubPOS=A|Gen=I|Num=P|Cas=7|Gra=2|Neg=A|Var=7
A	SubPOS=A|Gen=I|Num=P|Cas=7|Gra=2|Neg=N
A	SubPOS=A|Gen=I|Num=P|Cas=7|Gra=2|Neg=N|Var=6
A	SubPOS=A|Gen=I|Num=P|Cas=7|Gra=2|Neg=N|Var=7
A	SubPOS=A|Gen=I|Num=P|Cas=7|Gra=3|Neg=A
A	SubPOS=A|Gen=I|Num=P|Cas=7|Gra=3|Neg=A|Var=6
A	SubPOS=A|Gen=I|Num=P|Cas=7|Gra=3|Neg=A|Var=7
A	SubPOS=A|Gen=I|Num=P|Cas=7|Gra=3|Neg=N
A	SubPOS=A|Gen=I|Num=P|Cas=7|Gra=3|Neg=N|Var=6
A	SubPOS=A|Gen=I|Num=P|Cas=7|Gra=3|Neg=N|Var=7
A	SubPOS=A|Gen=I|Num=P|Cas=X|Gra=1|Neg=A
A	SubPOS=A|Gen=I|Num=P|Cas=X|Gra=1|Neg=A|Sem=G
A	SubPOS=A|Gen=I|Num=P|Cas=X|Gra=1|Neg=A|Sem=m
A	SubPOS=A|Gen=I|Num=P|Cas=X|Gra=1|Neg=N
A	SubPOS=A|Gen=I|Num=P|Cas=X|Gra=2|Neg=A
A	SubPOS=A|Gen=I|Num=P|Cas=X|Gra=2|Neg=N
A	SubPOS=A|Gen=I|Num=P|Cas=X|Gra=3|Neg=A
A	SubPOS=A|Gen=I|Num=P|Cas=X|Gra=3|Neg=N
A	SubPOS=A|Gen=I|Num=S|Cas=1|Gra=1|Neg=A
A	SubPOS=A|Gen=I|Num=S|Cas=1|Gra=1|Neg=A|Sem=G
A	SubPOS=A|Gen=I|Num=S|Cas=1|Gra=1|Neg=A|Sem=H
A	SubPOS=A|Gen=I|Num=S|Cas=1|Gra=1|Neg=A|Sem=K
A	SubPOS=A|Gen=I|Num=S|Cas=1|Gra=1|Neg=A|Sem=S
A	SubPOS=A|Gen=I|Num=S|Cas=1|Gra=1|Neg=A|Sem=m
A	SubPOS=A|Gen=I|Num=S|Cas=1|Gra=1|Neg=A|Sem=o
A	SubPOS=A|Gen=I|Num=S|Cas=1|Gra=1|Neg=A|Var=6
A	SubPOS=A|Gen=I|Num=S|Cas=1|Gra=1|Neg=N
A	SubPOS=A|Gen=I|Num=S|Cas=1|Gra=1|Neg=N|Var=6
A	SubPOS=A|Gen=I|Num=S|Cas=1|Gra=2|Neg=A
A	SubPOS=A|Gen=I|Num=S|Cas=1|Gra=2|Neg=A|Var=6
A	SubPOS=A|Gen=I|Num=S|Cas=1|Gra=2|Neg=N
A	SubPOS=A|Gen=I|Num=S|Cas=1|Gra=2|Neg=N|Var=6
A	SubPOS=A|Gen=I|Num=S|Cas=1|Gra=3|Neg=A
A	SubPOS=A|Gen=I|Num=S|Cas=1|Gra=3|Neg=A|Var=6
A	SubPOS=A|Gen=I|Num=S|Cas=1|Gra=3|Neg=N
A	SubPOS=A|Gen=I|Num=S|Cas=1|Gra=3|Neg=N|Var=6
A	SubPOS=A|Gen=I|Num=S|Cas=2|Gra=1|Neg=A
A	SubPOS=A|Gen=I|Num=S|Cas=2|Gra=1|Neg=A|Sem=G
A	SubPOS=A|Gen=I|Num=S|Cas=2|Gra=1|Neg=A|Sem=H
A	SubPOS=A|Gen=I|Num=S|Cas=2|Gra=1|Neg=A|Sem=S
A	SubPOS=A|Gen=I|Num=S|Cas=2|Gra=1|Neg=A|Sem=m
A	SubPOS=A|Gen=I|Num=S|Cas=2|Gra=1|Neg=A|Sem=o
A	SubPOS=A|Gen=I|Num=S|Cas=2|Gra=1|Neg=A|Var=6
A	SubPOS=A|Gen=I|Num=S|Cas=2|Gra=1|Neg=N
A	SubPOS=A|Gen=I|Num=S|Cas=2|Gra=1|Neg=N|Var=6
A	SubPOS=A|Gen=I|Num=S|Cas=2|Gra=2|Neg=A
A	SubPOS=A|Gen=I|Num=S|Cas=2|Gra=2|Neg=A|Var=6
A	SubPOS=A|Gen=I|Num=S|Cas=2|Gra=2|Neg=N
A	SubPOS=A|Gen=I|Num=S|Cas=2|Gra=2|Neg=N|Var=6
A	SubPOS=A|Gen=I|Num=S|Cas=2|Gra=3|Neg=A
A	SubPOS=A|Gen=I|Num=S|Cas=2|Gra=3|Neg=A|Var=6
A	SubPOS=A|Gen=I|Num=S|Cas=2|Gra=3|Neg=N
A	SubPOS=A|Gen=I|Num=S|Cas=2|Gra=3|Neg=N|Var=6
A	SubPOS=A|Gen=I|Num=S|Cas=3|Gra=1|Neg=A
A	SubPOS=A|Gen=I|Num=S|Cas=3|Gra=1|Neg=A|Var=6
A	SubPOS=A|Gen=I|Num=S|Cas=3|Gra=1|Neg=N
A	SubPOS=A|Gen=I|Num=S|Cas=3|Gra=1|Neg=N|Var=6
A	SubPOS=A|Gen=I|Num=S|Cas=3|Gra=2|Neg=A
A	SubPOS=A|Gen=I|Num=S|Cas=3|Gra=2|Neg=A|Var=6
A	SubPOS=A|Gen=I|Num=S|Cas=3|Gra=2|Neg=N
A	SubPOS=A|Gen=I|Num=S|Cas=3|Gra=2|Neg=N|Var=6
A	SubPOS=A|Gen=I|Num=S|Cas=3|Gra=3|Neg=A
A	SubPOS=A|Gen=I|Num=S|Cas=3|Gra=3|Neg=A|Var=6
A	SubPOS=A|Gen=I|Num=S|Cas=3|Gra=3|Neg=N
A	SubPOS=A|Gen=I|Num=S|Cas=3|Gra=3|Neg=N|Var=6
A	SubPOS=A|Gen=I|Num=S|Cas=4|Gra=1|Neg=A
A	SubPOS=A|Gen=I|Num=S|Cas=4|Gra=1|Neg=A|Sem=G
A	SubPOS=A|Gen=I|Num=S|Cas=4|Gra=1|Neg=A|Sem=S
A	SubPOS=A|Gen=I|Num=S|Cas=4|Gra=1|Neg=A|Sem=m
A	SubPOS=A|Gen=I|Num=S|Cas=4|Gra=1|Neg=A|Var=6
A	SubPOS=A|Gen=I|Num=S|Cas=4|Gra=1|Neg=N
A	SubPOS=A|Gen=I|Num=S|Cas=4|Gra=1|Neg=N|Var=6
A	SubPOS=A|Gen=I|Num=S|Cas=4|Gra=2|Neg=A
A	SubPOS=A|Gen=I|Num=S|Cas=4|Gra=2|Neg=A|Var=6
A	SubPOS=A|Gen=I|Num=S|Cas=4|Gra=2|Neg=N
A	SubPOS=A|Gen=I|Num=S|Cas=4|Gra=2|Neg=N|Var=6
A	SubPOS=A|Gen=I|Num=S|Cas=4|Gra=3|Neg=A
A	SubPOS=A|Gen=I|Num=S|Cas=4|Gra=3|Neg=A|Var=6
A	SubPOS=A|Gen=I|Num=S|Cas=4|Gra=3|Neg=N
A	SubPOS=A|Gen=I|Num=S|Cas=4|Gra=3|Neg=N|Var=6
A	SubPOS=A|Gen=I|Num=S|Cas=5|Gra=1|Neg=A
A	SubPOS=A|Gen=I|Num=S|Cas=5|Gra=1|Neg=A|Var=6
A	SubPOS=A|Gen=I|Num=S|Cas=5|Gra=1|Neg=N
A	SubPOS=A|Gen=I|Num=S|Cas=5|Gra=1|Neg=N|Var=6
A	SubPOS=A|Gen=I|Num=S|Cas=5|Gra=2|Neg=A
A	SubPOS=A|Gen=I|Num=S|Cas=5|Gra=2|Neg=A|Var=6
A	SubPOS=A|Gen=I|Num=S|Cas=5|Gra=2|Neg=N
A	SubPOS=A|Gen=I|Num=S|Cas=5|Gra=2|Neg=N|Var=6
A	SubPOS=A|Gen=I|Num=S|Cas=5|Gra=3|Neg=A
A	SubPOS=A|Gen=I|Num=S|Cas=5|Gra=3|Neg=A|Var=6
A	SubPOS=A|Gen=I|Num=S|Cas=5|Gra=3|Neg=N
A	SubPOS=A|Gen=I|Num=S|Cas=5|Gra=3|Neg=N|Var=6
A	SubPOS=A|Gen=I|Num=S|Cas=6|Gra=1|Neg=A
A	SubPOS=A|Gen=I|Num=S|Cas=6|Gra=1|Neg=A|Sem=G
A	SubPOS=A|Gen=I|Num=S|Cas=6|Gra=1|Neg=A|Sem=S
A	SubPOS=A|Gen=I|Num=S|Cas=6|Gra=1|Neg=A|Sem=m
A	SubPOS=A|Gen=I|Num=S|Cas=6|Gra=1|Neg=A|Sem=o
A	SubPOS=A|Gen=I|Num=S|Cas=6|Gra=1|Neg=A|Var=6
A	SubPOS=A|Gen=I|Num=S|Cas=6|Gra=1|Neg=A|Var=7
A	SubPOS=A|Gen=I|Num=S|Cas=6|Gra=1|Neg=N
A	SubPOS=A|Gen=I|Num=S|Cas=6|Gra=1|Neg=N|Var=6
A	SubPOS=A|Gen=I|Num=S|Cas=6|Gra=1|Neg=N|Var=7
A	SubPOS=A|Gen=I|Num=S|Cas=6|Gra=2|Neg=A
A	SubPOS=A|Gen=I|Num=S|Cas=6|Gra=2|Neg=A|Var=6
A	SubPOS=A|Gen=I|Num=S|Cas=6|Gra=2|Neg=N
A	SubPOS=A|Gen=I|Num=S|Cas=6|Gra=2|Neg=N|Var=6
A	SubPOS=A|Gen=I|Num=S|Cas=6|Gra=3|Neg=A
A	SubPOS=A|Gen=I|Num=S|Cas=6|Gra=3|Neg=A|Var=6
A	SubPOS=A|Gen=I|Num=S|Cas=6|Gra=3|Neg=N
A	SubPOS=A|Gen=I|Num=S|Cas=6|Gra=3|Neg=N|Var=6
A	SubPOS=A|Gen=I|Num=S|Cas=7|Gra=1|Neg=A
A	SubPOS=A|Gen=I|Num=S|Cas=7|Gra=1|Neg=A|Sem=H
A	SubPOS=A|Gen=I|Num=S|Cas=7|Gra=1|Neg=A|Sem=m
A	SubPOS=A|Gen=I|Num=S|Cas=7|Gra=1|Neg=A|Sem=o
A	SubPOS=A|Gen=I|Num=S|Cas=7|Gra=1|Neg=A|Var=2
A	SubPOS=A|Gen=I|Num=S|Cas=7|Gra=1|Neg=A|Var=6
A	SubPOS=A|Gen=I|Num=S|Cas=7|Gra=1|Neg=N
A	SubPOS=A|Gen=I|Num=S|Cas=7|Gra=1|Neg=N|Var=2
A	SubPOS=A|Gen=I|Num=S|Cas=7|Gra=1|Neg=N|Var=6
A	SubPOS=A|Gen=I|Num=S|Cas=7|Gra=2|Neg=A
A	SubPOS=A|Gen=I|Num=S|Cas=7|Gra=2|Neg=A|Var=6
A	SubPOS=A|Gen=I|Num=S|Cas=7|Gra=2|Neg=N
A	SubPOS=A|Gen=I|Num=S|Cas=7|Gra=2|Neg=N|Var=6
A	SubPOS=A|Gen=I|Num=S|Cas=7|Gra=3|Neg=A
A	SubPOS=A|Gen=I|Num=S|Cas=7|Gra=3|Neg=A|Var=6
A	SubPOS=A|Gen=I|Num=S|Cas=7|Gra=3|Neg=N
A	SubPOS=A|Gen=I|Num=S|Cas=7|Gra=3|Neg=N|Var=6
A	SubPOS=A|Gen=I|Num=S|Cas=X|Gra=1|Neg=A
A	SubPOS=A|Gen=I|Num=S|Cas=X|Gra=1|Neg=A|Sem=K
A	SubPOS=A|Gen=I|Num=S|Cas=X|Gra=1|Neg=A|Sem=R
A	SubPOS=A|Gen=I|Num=S|Cas=X|Gra=1|Neg=A|Sem=m
A	SubPOS=A|Gen=I|Num=X|Cas=X|Gra=1|Neg=A
A	SubPOS=A|Gen=I|Num=X|Cas=X|Gra=1|Neg=A|Sem=m
A	SubPOS=A|Gen=M|Num=P|Cas=1|Gra=1|Neg=A
A	SubPOS=A|Gen=M|Num=P|Cas=1|Gra=1|Neg=A|Sem=G
A	SubPOS=A|Gen=M|Num=P|Cas=1|Gra=1|Neg=A|Sem=K
A	SubPOS=A|Gen=M|Num=P|Cas=1|Gra=1|Neg=A|Sem=m
A	SubPOS=A|Gen=M|Num=P|Cas=1|Gra=1|Neg=A|Var=6
A	SubPOS=A|Gen=M|Num=P|Cas=1|Gra=1|Neg=N
A	SubPOS=A|Gen=M|Num=P|Cas=1|Gra=1|Neg=N|Var=6
A	SubPOS=A|Gen=M|Num=P|Cas=1|Gra=2|Neg=A
A	SubPOS=A|Gen=M|Num=P|Cas=1|Gra=2|Neg=A|Var=6
A	SubPOS=A|Gen=M|Num=P|Cas=1|Gra=2|Neg=N
A	SubPOS=A|Gen=M|Num=P|Cas=1|Gra=2|Neg=N|Var=6
A	SubPOS=A|Gen=M|Num=P|Cas=1|Gra=3|Neg=A
A	SubPOS=A|Gen=M|Num=P|Cas=1|Gra=3|Neg=A|Var=6
A	SubPOS=A|Gen=M|Num=P|Cas=1|Gra=3|Neg=N
A	SubPOS=A|Gen=M|Num=P|Cas=1|Gra=3|Neg=N|Var=6
A	SubPOS=A|Gen=M|Num=P|Cas=2|Gra=1|Neg=A
A	SubPOS=A|Gen=M|Num=P|Cas=2|Gra=1|Neg=A|Sem=G
A	SubPOS=A|Gen=M|Num=P|Cas=2|Gra=1|Neg=A|Sem=S
A	SubPOS=A|Gen=M|Num=P|Cas=2|Gra=1|Neg=A|Sem=m
A	SubPOS=A|Gen=M|Num=P|Cas=2|Gra=1|Neg=A|Var=6
A	SubPOS=A|Gen=M|Num=P|Cas=2|Gra=1|Neg=N
A	SubPOS=A|Gen=M|Num=P|Cas=2|Gra=1|Neg=N|Var=6
A	SubPOS=A|Gen=M|Num=P|Cas=2|Gra=2|Neg=A
A	SubPOS=A|Gen=M|Num=P|Cas=2|Gra=2|Neg=A|Var=6
A	SubPOS=A|Gen=M|Num=P|Cas=2|Gra=2|Neg=N
A	SubPOS=A|Gen=M|Num=P|Cas=2|Gra=2|Neg=N|Var=6
A	SubPOS=A|Gen=M|Num=P|Cas=2|Gra=3|Neg=A
A	SubPOS=A|Gen=M|Num=P|Cas=2|Gra=3|Neg=A|Var=6
A	SubPOS=A|Gen=M|Num=P|Cas=2|Gra=3|Neg=N
A	SubPOS=A|Gen=M|Num=P|Cas=2|Gra=3|Neg=N|Var=6
A	SubPOS=A|Gen=M|Num=P|Cas=3|Gra=1|Neg=A
A	SubPOS=A|Gen=M|Num=P|Cas=3|Gra=1|Neg=A|Var=6
A	SubPOS=A|Gen=M|Num=P|Cas=3|Gra=1|Neg=N
A	SubPOS=A|Gen=M|Num=P|Cas=3|Gra=1|Neg=N|Var=6
A	SubPOS=A|Gen=M|Num=P|Cas=3|Gra=2|Neg=A
A	SubPOS=A|Gen=M|Num=P|Cas=3|Gra=2|Neg=A|Var=6
A	SubPOS=A|Gen=M|Num=P|Cas=3|Gra=2|Neg=N
A	SubPOS=A|Gen=M|Num=P|Cas=3|Gra=2|Neg=N|Var=6
A	SubPOS=A|Gen=M|Num=P|Cas=3|Gra=3|Neg=A
A	SubPOS=A|Gen=M|Num=P|Cas=3|Gra=3|Neg=A|Var=6
A	SubPOS=A|Gen=M|Num=P|Cas=3|Gra=3|Neg=N
A	SubPOS=A|Gen=M|Num=P|Cas=3|Gra=3|Neg=N|Var=6
A	SubPOS=A|Gen=M|Num=P|Cas=4|Gra=1|Neg=A
A	SubPOS=A|Gen=M|Num=P|Cas=4|Gra=1|Neg=A|Var=6
A	SubPOS=A|Gen=M|Num=P|Cas=4|Gra=1|Neg=N
A	SubPOS=A|Gen=M|Num=P|Cas=4|Gra=1|Neg=N|Var=6
A	SubPOS=A|Gen=M|Num=P|Cas=4|Gra=2|Neg=A
A	SubPOS=A|Gen=M|Num=P|Cas=4|Gra=2|Neg=A|Var=6
A	SubPOS=A|Gen=M|Num=P|Cas=4|Gra=2|Neg=N
A	SubPOS=A|Gen=M|Num=P|Cas=4|Gra=2|Neg=N|Var=6
A	SubPOS=A|Gen=M|Num=P|Cas=4|Gra=3|Neg=A
A	SubPOS=A|Gen=M|Num=P|Cas=4|Gra=3|Neg=A|Var=6
A	SubPOS=A|Gen=M|Num=P|Cas=4|Gra=3|Neg=N
A	SubPOS=A|Gen=M|Num=P|Cas=4|Gra=3|Neg=N|Var=6
A	SubPOS=A|Gen=M|Num=P|Cas=5|Gra=1|Neg=A
A	SubPOS=A|Gen=M|Num=P|Cas=5|Gra=1|Neg=A|Var=6
A	SubPOS=A|Gen=M|Num=P|Cas=5|Gra=1|Neg=N
A	SubPOS=A|Gen=M|Num=P|Cas=5|Gra=1|Neg=N|Var=6
A	SubPOS=A|Gen=M|Num=P|Cas=5|Gra=2|Neg=A
A	SubPOS=A|Gen=M|Num=P|Cas=5|Gra=2|Neg=A|Var=6
A	SubPOS=A|Gen=M|Num=P|Cas=5|Gra=2|Neg=N
A	SubPOS=A|Gen=M|Num=P|Cas=5|Gra=2|Neg=N|Var=6
A	SubPOS=A|Gen=M|Num=P|Cas=5|Gra=3|Neg=A
A	SubPOS=A|Gen=M|Num=P|Cas=5|Gra=3|Neg=A|Var=6
A	SubPOS=A|Gen=M|Num=P|Cas=5|Gra=3|Neg=N
A	SubPOS=A|Gen=M|Num=P|Cas=5|Gra=3|Neg=N|Var=6
A	SubPOS=A|Gen=M|Num=P|Cas=6|Gra=1|Neg=A
A	SubPOS=A|Gen=M|Num=P|Cas=6|Gra=1|Neg=A|Var=6
A	SubPOS=A|Gen=M|Num=P|Cas=6|Gra=1|Neg=N
A	SubPOS=A|Gen=M|Num=P|Cas=6|Gra=1|Neg=N|Var=6
A	SubPOS=A|Gen=M|Num=P|Cas=6|Gra=2|Neg=A
A	SubPOS=A|Gen=M|Num=P|Cas=6|Gra=2|Neg=A|Var=6
A	SubPOS=A|Gen=M|Num=P|Cas=6|Gra=2|Neg=N
A	SubPOS=A|Gen=M|Num=P|Cas=6|Gra=2|Neg=N|Var=6
A	SubPOS=A|Gen=M|Num=P|Cas=6|Gra=3|Neg=A
A	SubPOS=A|Gen=M|Num=P|Cas=6|Gra=3|Neg=A|Var=6
A	SubPOS=A|Gen=M|Num=P|Cas=6|Gra=3|Neg=N
A	SubPOS=A|Gen=M|Num=P|Cas=6|Gra=3|Neg=N|Var=6
A	SubPOS=A|Gen=M|Num=P|Cas=7|Gra=1|Neg=A
A	SubPOS=A|Gen=M|Num=P|Cas=7|Gra=1|Neg=A|Var=6
A	SubPOS=A|Gen=M|Num=P|Cas=7|Gra=1|Neg=A|Var=7
A	SubPOS=A|Gen=M|Num=P|Cas=7|Gra=1|Neg=N
A	SubPOS=A|Gen=M|Num=P|Cas=7|Gra=1|Neg=N|Var=6
A	SubPOS=A|Gen=M|Num=P|Cas=7|Gra=1|Neg=N|Var=7
A	SubPOS=A|Gen=M|Num=P|Cas=7|Gra=2|Neg=A
A	SubPOS=A|Gen=M|Num=P|Cas=7|Gra=2|Neg=A|Var=6
A	SubPOS=A|Gen=M|Num=P|Cas=7|Gra=2|Neg=A|Var=7
A	SubPOS=A|Gen=M|Num=P|Cas=7|Gra=2|Neg=N
A	SubPOS=A|Gen=M|Num=P|Cas=7|Gra=2|Neg=N|Var=6
A	SubPOS=A|Gen=M|Num=P|Cas=7|Gra=2|Neg=N|Var=7
A	SubPOS=A|Gen=M|Num=P|Cas=7|Gra=3|Neg=A
A	SubPOS=A|Gen=M|Num=P|Cas=7|Gra=3|Neg=A|Var=6
A	SubPOS=A|Gen=M|Num=P|Cas=7|Gra=3|Neg=A|Var=7
A	SubPOS=A|Gen=M|Num=P|Cas=7|Gra=3|Neg=N
A	SubPOS=A|Gen=M|Num=P|Cas=7|Gra=3|Neg=N|Var=6
A	SubPOS=A|Gen=M|Num=P|Cas=7|Gra=3|Neg=N|Var=7
A	SubPOS=A|Gen=M|Num=S|Cas=1|Gra=1|Neg=A
A	SubPOS=A|Gen=M|Num=S|Cas=1|Gra=1|Neg=A|Sem=G
A	SubPOS=A|Gen=M|Num=S|Cas=1|Gra=1|Neg=A|Sem=S
A	SubPOS=A|Gen=M|Num=S|Cas=1|Gra=1|Neg=A|Sem=U
A	SubPOS=A|Gen=M|Num=S|Cas=1|Gra=1|Neg=A|Sem=m
A	SubPOS=A|Gen=M|Num=S|Cas=1|Gra=1|Neg=A|Var=6
A	SubPOS=A|Gen=M|Num=S|Cas=1|Gra=1|Neg=N
A	SubPOS=A|Gen=M|Num=S|Cas=1|Gra=1|Neg=N|Var=6
A	SubPOS=A|Gen=M|Num=S|Cas=1|Gra=2|Neg=A
A	SubPOS=A|Gen=M|Num=S|Cas=1|Gra=2|Neg=A|Var=6
A	SubPOS=A|Gen=M|Num=S|Cas=1|Gra=2|Neg=N
A	SubPOS=A|Gen=M|Num=S|Cas=1|Gra=2|Neg=N|Var=6
A	SubPOS=A|Gen=M|Num=S|Cas=1|Gra=3|Neg=A
A	SubPOS=A|Gen=M|Num=S|Cas=1|Gra=3|Neg=A|Var=6
A	SubPOS=A|Gen=M|Num=S|Cas=1|Gra=3|Neg=N
A	SubPOS=A|Gen=M|Num=S|Cas=1|Gra=3|Neg=N|Var=6
A	SubPOS=A|Gen=M|Num=S|Cas=2|Gra=1|Neg=A
A	SubPOS=A|Gen=M|Num=S|Cas=2|Gra=1|Neg=A|Sem=S
A	SubPOS=A|Gen=M|Num=S|Cas=2|Gra=1|Neg=A|Sem=m
A	SubPOS=A|Gen=M|Num=S|Cas=2|Gra=1|Neg=A|Var=6
A	SubPOS=A|Gen=M|Num=S|Cas=2|Gra=1|Neg=N
A	SubPOS=A|Gen=M|Num=S|Cas=2|Gra=1|Neg=N|Var=6
A	SubPOS=A|Gen=M|Num=S|Cas=2|Gra=2|Neg=A
A	SubPOS=A|Gen=M|Num=S|Cas=2|Gra=2|Neg=A|Var=6
A	SubPOS=A|Gen=M|Num=S|Cas=2|Gra=2|Neg=N
A	SubPOS=A|Gen=M|Num=S|Cas=2|Gra=2|Neg=N|Var=6
A	SubPOS=A|Gen=M|Num=S|Cas=2|Gra=3|Neg=A
A	SubPOS=A|Gen=M|Num=S|Cas=2|Gra=3|Neg=A|Var=6
A	SubPOS=A|Gen=M|Num=S|Cas=2|Gra=3|Neg=N
A	SubPOS=A|Gen=M|Num=S|Cas=2|Gra=3|Neg=N|Var=6
A	SubPOS=A|Gen=M|Num=S|Cas=3|Gra=1|Neg=A
A	SubPOS=A|Gen=M|Num=S|Cas=3|Gra=1|Neg=A|Sem=S
A	SubPOS=A|Gen=M|Num=S|Cas=3|Gra=1|Neg=A|Sem=m
A	SubPOS=A|Gen=M|Num=S|Cas=3|Gra=1|Neg=A|Var=6
A	SubPOS=A|Gen=M|Num=S|Cas=3|Gra=1|Neg=N
A	SubPOS=A|Gen=M|Num=S|Cas=3|Gra=1|Neg=N|Var=6
A	SubPOS=A|Gen=M|Num=S|Cas=3|Gra=2|Neg=A
A	SubPOS=A|Gen=M|Num=S|Cas=3|Gra=2|Neg=A|Var=6
A	SubPOS=A|Gen=M|Num=S|Cas=3|Gra=2|Neg=N
A	SubPOS=A|Gen=M|Num=S|Cas=3|Gra=2|Neg=N|Var=6
A	SubPOS=A|Gen=M|Num=S|Cas=3|Gra=3|Neg=A
A	SubPOS=A|Gen=M|Num=S|Cas=3|Gra=3|Neg=A|Var=6
A	SubPOS=A|Gen=M|Num=S|Cas=3|Gra=3|Neg=N
A	SubPOS=A|Gen=M|Num=S|Cas=3|Gra=3|Neg=N|Var=6
A	SubPOS=A|Gen=M|Num=S|Cas=4|Gra=1|Neg=A
A	SubPOS=A|Gen=M|Num=S|Cas=4|Gra=1|Neg=A|Sem=S
A	SubPOS=A|Gen=M|Num=S|Cas=4|Gra=1|Neg=A|Sem=m
A	SubPOS=A|Gen=M|Num=S|Cas=4|Gra=1|Neg=A|Var=6
A	SubPOS=A|Gen=M|Num=S|Cas=4|Gra=1|Neg=N
A	SubPOS=A|Gen=M|Num=S|Cas=4|Gra=1|Neg=N|Var=6
A	SubPOS=A|Gen=M|Num=S|Cas=4|Gra=2|Neg=A
A	SubPOS=A|Gen=M|Num=S|Cas=4|Gra=2|Neg=A|Var=6
A	SubPOS=A|Gen=M|Num=S|Cas=4|Gra=2|Neg=N
A	SubPOS=A|Gen=M|Num=S|Cas=4|Gra=2|Neg=N|Var=6
A	SubPOS=A|Gen=M|Num=S|Cas=4|Gra=3|Neg=A
A	SubPOS=A|Gen=M|Num=S|Cas=4|Gra=3|Neg=A|Var=6
A	SubPOS=A|Gen=M|Num=S|Cas=4|Gra=3|Neg=N
A	SubPOS=A|Gen=M|Num=S|Cas=4|Gra=3|Neg=N|Var=6
A	SubPOS=A|Gen=M|Num=S|Cas=5|Gra=1|Neg=A
A	SubPOS=A|Gen=M|Num=S|Cas=5|Gra=1|Neg=A|Var=6
A	SubPOS=A|Gen=M|Num=S|Cas=5|Gra=1|Neg=N
A	SubPOS=A|Gen=M|Num=S|Cas=5|Gra=1|Neg=N|Var=6
A	SubPOS=A|Gen=M|Num=S|Cas=5|Gra=2|Neg=A
A	SubPOS=A|Gen=M|Num=S|Cas=5|Gra=2|Neg=A|Var=6
A	SubPOS=A|Gen=M|Num=S|Cas=5|Gra=2|Neg=N
A	SubPOS=A|Gen=M|Num=S|Cas=5|Gra=2|Neg=N|Var=6
A	SubPOS=A|Gen=M|Num=S|Cas=5|Gra=3|Neg=A
A	SubPOS=A|Gen=M|Num=S|Cas=5|Gra=3|Neg=A|Var=6
A	SubPOS=A|Gen=M|Num=S|Cas=5|Gra=3|Neg=N
A	SubPOS=A|Gen=M|Num=S|Cas=5|Gra=3|Neg=N|Var=6
A	SubPOS=A|Gen=M|Num=S|Cas=6|Gra=1|Neg=A
A	SubPOS=A|Gen=M|Num=S|Cas=6|Gra=1|Neg=A|Sem=S
A	SubPOS=A|Gen=M|Num=S|Cas=6|Gra=1|Neg=A|Sem=m
A	SubPOS=A|Gen=M|Num=S|Cas=6|Gra=1|Neg=A|Var=6
A	SubPOS=A|Gen=M|Num=S|Cas=6|Gra=1|Neg=A|Var=7
A	SubPOS=A|Gen=M|Num=S|Cas=6|Gra=1|Neg=N
A	SubPOS=A|Gen=M|Num=S|Cas=6|Gra=1|Neg=N|Var=6
A	SubPOS=A|Gen=M|Num=S|Cas=6|Gra=1|Neg=N|Var=7
A	SubPOS=A|Gen=M|Num=S|Cas=6|Gra=2|Neg=A
A	SubPOS=A|Gen=M|Num=S|Cas=6|Gra=2|Neg=A|Var=6
A	SubPOS=A|Gen=M|Num=S|Cas=6|Gra=2|Neg=N
A	SubPOS=A|Gen=M|Num=S|Cas=6|Gra=2|Neg=N|Var=6
A	SubPOS=A|Gen=M|Num=S|Cas=6|Gra=3|Neg=A
A	SubPOS=A|Gen=M|Num=S|Cas=6|Gra=3|Neg=A|Var=6
A	SubPOS=A|Gen=M|Num=S|Cas=6|Gra=3|Neg=N
A	SubPOS=A|Gen=M|Num=S|Cas=6|Gra=3|Neg=N|Var=6
A	SubPOS=A|Gen=M|Num=S|Cas=7|Gra=1|Neg=A
A	SubPOS=A|Gen=M|Num=S|Cas=7|Gra=1|Neg=A|Sem=S
A	SubPOS=A|Gen=M|Num=S|Cas=7|Gra=1|Neg=A|Sem=m
A	SubPOS=A|Gen=M|Num=S|Cas=7|Gra=1|Neg=A|Var=6
A	SubPOS=A|Gen=M|Num=S|Cas=7|Gra=1|Neg=N
A	SubPOS=A|Gen=M|Num=S|Cas=7|Gra=1|Neg=N|Var=6
A	SubPOS=A|Gen=M|Num=S|Cas=7|Gra=2|Neg=A
A	SubPOS=A|Gen=M|Num=S|Cas=7|Gra=2|Neg=A|Var=6
A	SubPOS=A|Gen=M|Num=S|Cas=7|Gra=2|Neg=N
A	SubPOS=A|Gen=M|Num=S|Cas=7|Gra=2|Neg=N|Var=6
A	SubPOS=A|Gen=M|Num=S|Cas=7|Gra=3|Neg=A
A	SubPOS=A|Gen=M|Num=S|Cas=7|Gra=3|Neg=A|Var=6
A	SubPOS=A|Gen=M|Num=S|Cas=7|Gra=3|Neg=N
A	SubPOS=A|Gen=M|Num=S|Cas=7|Gra=3|Neg=N|Var=6
A	SubPOS=A|Gen=M|Num=S|Cas=X|Gra=1|Neg=A
A	SubPOS=A|Gen=N|Num=D|Cas=7|Gra=1|Neg=A
A	SubPOS=A|Gen=N|Num=D|Cas=7|Gra=1|Neg=A|Var=6
A	SubPOS=A|Gen=N|Num=D|Cas=7|Gra=1|Neg=N
A	SubPOS=A|Gen=N|Num=D|Cas=7|Gra=1|Neg=N|Var=6
A	SubPOS=A|Gen=N|Num=D|Cas=7|Gra=2|Neg=A
A	SubPOS=A|Gen=N|Num=D|Cas=7|Gra=2|Neg=A|Var=6
A	SubPOS=A|Gen=N|Num=D|Cas=7|Gra=2|Neg=N
A	SubPOS=A|Gen=N|Num=D|Cas=7|Gra=2|Neg=N|Var=6
A	SubPOS=A|Gen=N|Num=D|Cas=7|Gra=3|Neg=A
A	SubPOS=A|Gen=N|Num=D|Cas=7|Gra=3|Neg=A|Var=6
A	SubPOS=A|Gen=N|Num=D|Cas=7|Gra=3|Neg=N
A	SubPOS=A|Gen=N|Num=D|Cas=7|Gra=3|Neg=N|Var=6
A	SubPOS=A|Gen=N|Num=P|Cas=1|Gra=1|Neg=A
A	SubPOS=A|Gen=N|Num=P|Cas=1|Gra=1|Neg=A|Var=6
A	SubPOS=A|Gen=N|Num=P|Cas=1|Gra=1|Neg=N
A	SubPOS=A|Gen=N|Num=P|Cas=1|Gra=1|Neg=N|Var=6
A	SubPOS=A|Gen=N|Num=P|Cas=1|Gra=2|Neg=A
A	SubPOS=A|Gen=N|Num=P|Cas=1|Gra=2|Neg=A|Var=6
A	SubPOS=A|Gen=N|Num=P|Cas=1|Gra=2|Neg=N
A	SubPOS=A|Gen=N|Num=P|Cas=1|Gra=2|Neg=N|Var=6
A	SubPOS=A|Gen=N|Num=P|Cas=1|Gra=3|Neg=A
A	SubPOS=A|Gen=N|Num=P|Cas=1|Gra=3|Neg=A|Var=6
A	SubPOS=A|Gen=N|Num=P|Cas=1|Gra=3|Neg=N
A	SubPOS=A|Gen=N|Num=P|Cas=1|Gra=3|Neg=N|Var=6
A	SubPOS=A|Gen=N|Num=P|Cas=2|Gra=1|Neg=A
A	SubPOS=A|Gen=N|Num=P|Cas=2|Gra=1|Neg=A|Var=6
A	SubPOS=A|Gen=N|Num=P|Cas=2|Gra=1|Neg=N
A	SubPOS=A|Gen=N|Num=P|Cas=2|Gra=1|Neg=N|Var=6
A	SubPOS=A|Gen=N|Num=P|Cas=2|Gra=2|Neg=A
A	SubPOS=A|Gen=N|Num=P|Cas=2|Gra=2|Neg=A|Var=6
A	SubPOS=A|Gen=N|Num=P|Cas=2|Gra=2|Neg=N
A	SubPOS=A|Gen=N|Num=P|Cas=2|Gra=2|Neg=N|Var=6
A	SubPOS=A|Gen=N|Num=P|Cas=2|Gra=3|Neg=A
A	SubPOS=A|Gen=N|Num=P|Cas=2|Gra=3|Neg=A|Var=6
A	SubPOS=A|Gen=N|Num=P|Cas=2|Gra=3|Neg=N
A	SubPOS=A|Gen=N|Num=P|Cas=2|Gra=3|Neg=N|Var=6
A	SubPOS=A|Gen=N|Num=P|Cas=3|Gra=1|Neg=A
A	SubPOS=A|Gen=N|Num=P|Cas=3|Gra=1|Neg=A|Sem=m
A	SubPOS=A|Gen=N|Num=P|Cas=3|Gra=1|Neg=A|Sem=o
A	SubPOS=A|Gen=N|Num=P|Cas=3|Gra=1|Neg=A|Var=6
A	SubPOS=A|Gen=N|Num=P|Cas=3|Gra=1|Neg=N
A	SubPOS=A|Gen=N|Num=P|Cas=3|Gra=1|Neg=N|Var=6
A	SubPOS=A|Gen=N|Num=P|Cas=3|Gra=2|Neg=A
A	SubPOS=A|Gen=N|Num=P|Cas=3|Gra=2|Neg=A|Var=6
A	SubPOS=A|Gen=N|Num=P|Cas=3|Gra=2|Neg=N
A	SubPOS=A|Gen=N|Num=P|Cas=3|Gra=2|Neg=N|Var=6
A	SubPOS=A|Gen=N|Num=P|Cas=3|Gra=3|Neg=A
A	SubPOS=A|Gen=N|Num=P|Cas=3|Gra=3|Neg=A|Var=6
A	SubPOS=A|Gen=N|Num=P|Cas=3|Gra=3|Neg=N
A	SubPOS=A|Gen=N|Num=P|Cas=3|Gra=3|Neg=N|Var=6
A	SubPOS=A|Gen=N|Num=P|Cas=4|Gra=1|Neg=A
A	SubPOS=A|Gen=N|Num=P|Cas=4|Gra=1|Neg=A|Var=6
A	SubPOS=A|Gen=N|Num=P|Cas=4|Gra=1|Neg=N
A	SubPOS=A|Gen=N|Num=P|Cas=4|Gra=1|Neg=N|Var=6
A	SubPOS=A|Gen=N|Num=P|Cas=4|Gra=2|Neg=A
A	SubPOS=A|Gen=N|Num=P|Cas=4|Gra=2|Neg=A|Var=6
A	SubPOS=A|Gen=N|Num=P|Cas=4|Gra=2|Neg=N
A	SubPOS=A|Gen=N|Num=P|Cas=4|Gra=2|Neg=N|Var=6
A	SubPOS=A|Gen=N|Num=P|Cas=4|Gra=3|Neg=A
A	SubPOS=A|Gen=N|Num=P|Cas=4|Gra=3|Neg=A|Var=6
A	SubPOS=A|Gen=N|Num=P|Cas=4|Gra=3|Neg=N
A	SubPOS=A|Gen=N|Num=P|Cas=4|Gra=3|Neg=N|Var=6
A	SubPOS=A|Gen=N|Num=P|Cas=5|Gra=1|Neg=A
A	SubPOS=A|Gen=N|Num=P|Cas=5|Gra=1|Neg=A|Var=6
A	SubPOS=A|Gen=N|Num=P|Cas=5|Gra=1|Neg=N
A	SubPOS=A|Gen=N|Num=P|Cas=5|Gra=1|Neg=N|Var=6
A	SubPOS=A|Gen=N|Num=P|Cas=5|Gra=2|Neg=A
A	SubPOS=A|Gen=N|Num=P|Cas=5|Gra=2|Neg=A|Var=6
A	SubPOS=A|Gen=N|Num=P|Cas=5|Gra=2|Neg=N
A	SubPOS=A|Gen=N|Num=P|Cas=5|Gra=2|Neg=N|Var=6
A	SubPOS=A|Gen=N|Num=P|Cas=5|Gra=3|Neg=A
A	SubPOS=A|Gen=N|Num=P|Cas=5|Gra=3|Neg=A|Var=6
A	SubPOS=A|Gen=N|Num=P|Cas=5|Gra=3|Neg=N
A	SubPOS=A|Gen=N|Num=P|Cas=5|Gra=3|Neg=N|Var=6
A	SubPOS=A|Gen=N|Num=P|Cas=6|Gra=1|Neg=A
A	SubPOS=A|Gen=N|Num=P|Cas=6|Gra=1|Neg=A|Var=6
A	SubPOS=A|Gen=N|Num=P|Cas=6|Gra=1|Neg=N
A	SubPOS=A|Gen=N|Num=P|Cas=6|Gra=1|Neg=N|Var=6
A	SubPOS=A|Gen=N|Num=P|Cas=6|Gra=2|Neg=A
A	SubPOS=A|Gen=N|Num=P|Cas=6|Gra=2|Neg=A|Var=6
A	SubPOS=A|Gen=N|Num=P|Cas=6|Gra=2|Neg=N
A	SubPOS=A|Gen=N|Num=P|Cas=6|Gra=2|Neg=N|Var=6
A	SubPOS=A|Gen=N|Num=P|Cas=6|Gra=3|Neg=A
A	SubPOS=A|Gen=N|Num=P|Cas=6|Gra=3|Neg=A|Var=6
A	SubPOS=A|Gen=N|Num=P|Cas=6|Gra=3|Neg=N
A	SubPOS=A|Gen=N|Num=P|Cas=6|Gra=3|Neg=N|Var=6
A	SubPOS=A|Gen=N|Num=P|Cas=7|Gra=1|Neg=A
A	SubPOS=A|Gen=N|Num=P|Cas=7|Gra=1|Neg=A|Var=6
A	SubPOS=A|Gen=N|Num=P|Cas=7|Gra=1|Neg=A|Var=7
A	SubPOS=A|Gen=N|Num=P|Cas=7|Gra=1|Neg=N
A	SubPOS=A|Gen=N|Num=P|Cas=7|Gra=1|Neg=N|Var=6
A	SubPOS=A|Gen=N|Num=P|Cas=7|Gra=1|Neg=N|Var=7
A	SubPOS=A|Gen=N|Num=P|Cas=7|Gra=2|Neg=A
A	SubPOS=A|Gen=N|Num=P|Cas=7|Gra=2|Neg=A|Var=6
A	SubPOS=A|Gen=N|Num=P|Cas=7|Gra=2|Neg=A|Var=7
A	SubPOS=A|Gen=N|Num=P|Cas=7|Gra=2|Neg=N
A	SubPOS=A|Gen=N|Num=P|Cas=7|Gra=2|Neg=N|Var=6
A	SubPOS=A|Gen=N|Num=P|Cas=7|Gra=2|Neg=N|Var=7
A	SubPOS=A|Gen=N|Num=P|Cas=7|Gra=3|Neg=A
A	SubPOS=A|Gen=N|Num=P|Cas=7|Gra=3|Neg=A|Var=6
A	SubPOS=A|Gen=N|Num=P|Cas=7|Gra=3|Neg=A|Var=7
A	SubPOS=A|Gen=N|Num=P|Cas=7|Gra=3|Neg=N
A	SubPOS=A|Gen=N|Num=P|Cas=7|Gra=3|Neg=N|Var=6
A	SubPOS=A|Gen=N|Num=P|Cas=7|Gra=3|Neg=N|Var=7
A	SubPOS=A|Gen=N|Num=S|Cas=1|Gra=1|Neg=A
A	SubPOS=A|Gen=N|Num=S|Cas=1|Gra=1|Neg=A|Sem=G
A	SubPOS=A|Gen=N|Num=S|Cas=1|Gra=1|Neg=A|Sem=K
A	SubPOS=A|Gen=N|Num=S|Cas=1|Gra=1|Neg=A|Sem=R
A	SubPOS=A|Gen=N|Num=S|Cas=1|Gra=1|Neg=A|Sem=S
A	SubPOS=A|Gen=N|Num=S|Cas=1|Gra=1|Neg=A|Sem=m
A	SubPOS=A|Gen=N|Num=S|Cas=1|Gra=1|Neg=A|Var=6
A	SubPOS=A|Gen=N|Num=S|Cas=1|Gra=1|Neg=N
A	SubPOS=A|Gen=N|Num=S|Cas=1|Gra=1|Neg=N|Var=6
A	SubPOS=A|Gen=N|Num=S|Cas=1|Gra=2|Neg=A
A	SubPOS=A|Gen=N|Num=S|Cas=1|Gra=2|Neg=A|Var=6
A	SubPOS=A|Gen=N|Num=S|Cas=1|Gra=2|Neg=N
A	SubPOS=A|Gen=N|Num=S|Cas=1|Gra=2|Neg=N|Var=6
A	SubPOS=A|Gen=N|Num=S|Cas=1|Gra=3|Neg=A
A	SubPOS=A|Gen=N|Num=S|Cas=1|Gra=3|Neg=A|Var=6
A	SubPOS=A|Gen=N|Num=S|Cas=1|Gra=3|Neg=N
A	SubPOS=A|Gen=N|Num=S|Cas=1|Gra=3|Neg=N|Var=6
A	SubPOS=A|Gen=N|Num=S|Cas=2|Gra=1|Neg=A
A	SubPOS=A|Gen=N|Num=S|Cas=2|Gra=1|Neg=A|Sem=G
A	SubPOS=A|Gen=N|Num=S|Cas=2|Gra=1|Neg=A|Sem=S
A	SubPOS=A|Gen=N|Num=S|Cas=2|Gra=1|Neg=A|Sem=m
A	SubPOS=A|Gen=N|Num=S|Cas=2|Gra=1|Neg=A|Sem=o
A	SubPOS=A|Gen=N|Num=S|Cas=2|Gra=1|Neg=A|Var=6
A	SubPOS=A|Gen=N|Num=S|Cas=2|Gra=1|Neg=N
A	SubPOS=A|Gen=N|Num=S|Cas=2|Gra=1|Neg=N|Var=6
A	SubPOS=A|Gen=N|Num=S|Cas=2|Gra=2|Neg=A
A	SubPOS=A|Gen=N|Num=S|Cas=2|Gra=2|Neg=A|Var=6
A	SubPOS=A|Gen=N|Num=S|Cas=2|Gra=2|Neg=N
A	SubPOS=A|Gen=N|Num=S|Cas=2|Gra=2|Neg=N|Var=6
A	SubPOS=A|Gen=N|Num=S|Cas=2|Gra=3|Neg=A
A	SubPOS=A|Gen=N|Num=S|Cas=2|Gra=3|Neg=A|Var=6
A	SubPOS=A|Gen=N|Num=S|Cas=2|Gra=3|Neg=N
A	SubPOS=A|Gen=N|Num=S|Cas=2|Gra=3|Neg=N|Var=6
A	SubPOS=A|Gen=N|Num=S|Cas=3|Gra=1|Neg=A
A	SubPOS=A|Gen=N|Num=S|Cas=3|Gra=1|Neg=A|Var=6
A	SubPOS=A|Gen=N|Num=S|Cas=3|Gra=1|Neg=N
A	SubPOS=A|Gen=N|Num=S|Cas=3|Gra=1|Neg=N|Var=6
A	SubPOS=A|Gen=N|Num=S|Cas=3|Gra=2|Neg=A
A	SubPOS=A|Gen=N|Num=S|Cas=3|Gra=2|Neg=A|Var=6
A	SubPOS=A|Gen=N|Num=S|Cas=3|Gra=2|Neg=N
A	SubPOS=A|Gen=N|Num=S|Cas=3|Gra=2|Neg=N|Var=6
A	SubPOS=A|Gen=N|Num=S|Cas=3|Gra=3|Neg=A
A	SubPOS=A|Gen=N|Num=S|Cas=3|Gra=3|Neg=A|Var=6
A	SubPOS=A|Gen=N|Num=S|Cas=3|Gra=3|Neg=N
A	SubPOS=A|Gen=N|Num=S|Cas=3|Gra=3|Neg=N|Var=6
A	SubPOS=A|Gen=N|Num=S|Cas=4|Gra=1|Neg=A
A	SubPOS=A|Gen=N|Num=S|Cas=4|Gra=1|Neg=A|Sem=G
A	SubPOS=A|Gen=N|Num=S|Cas=4|Gra=1|Neg=A|Sem=m
A	SubPOS=A|Gen=N|Num=S|Cas=4|Gra=1|Neg=A|Var=6
A	SubPOS=A|Gen=N|Num=S|Cas=4|Gra=1|Neg=N
A	SubPOS=A|Gen=N|Num=S|Cas=4|Gra=1|Neg=N|Var=6
A	SubPOS=A|Gen=N|Num=S|Cas=4|Gra=2|Neg=A
A	SubPOS=A|Gen=N|Num=S|Cas=4|Gra=2|Neg=A|Var=6
A	SubPOS=A|Gen=N|Num=S|Cas=4|Gra=2|Neg=N
A	SubPOS=A|Gen=N|Num=S|Cas=4|Gra=2|Neg=N|Var=6
A	SubPOS=A|Gen=N|Num=S|Cas=4|Gra=3|Neg=A
A	SubPOS=A|Gen=N|Num=S|Cas=4|Gra=3|Neg=A|Var=6
A	SubPOS=A|Gen=N|Num=S|Cas=4|Gra=3|Neg=N
A	SubPOS=A|Gen=N|Num=S|Cas=4|Gra=3|Neg=N|Var=6
A	SubPOS=A|Gen=N|Num=S|Cas=5|Gra=1|Neg=A
A	SubPOS=A|Gen=N|Num=S|Cas=5|Gra=1|Neg=A|Var=6
A	SubPOS=A|Gen=N|Num=S|Cas=5|Gra=1|Neg=N
A	SubPOS=A|Gen=N|Num=S|Cas=5|Gra=1|Neg=N|Var=6
A	SubPOS=A|Gen=N|Num=S|Cas=5|Gra=2|Neg=A
A	SubPOS=A|Gen=N|Num=S|Cas=5|Gra=2|Neg=A|Var=6
A	SubPOS=A|Gen=N|Num=S|Cas=5|Gra=2|Neg=N
A	SubPOS=A|Gen=N|Num=S|Cas=5|Gra=2|Neg=N|Var=6
A	SubPOS=A|Gen=N|Num=S|Cas=5|Gra=3|Neg=A
A	SubPOS=A|Gen=N|Num=S|Cas=5|Gra=3|Neg=A|Var=6
A	SubPOS=A|Gen=N|Num=S|Cas=5|Gra=3|Neg=N
A	SubPOS=A|Gen=N|Num=S|Cas=5|Gra=3|Neg=N|Var=6
A	SubPOS=A|Gen=N|Num=S|Cas=6|Gra=1|Neg=A
A	SubPOS=A|Gen=N|Num=S|Cas=6|Gra=1|Neg=A|Sem=G
A	SubPOS=A|Gen=N|Num=S|Cas=6|Gra=1|Neg=A|Sem=S
A	SubPOS=A|Gen=N|Num=S|Cas=6|Gra=1|Neg=A|Sem=m
A	SubPOS=A|Gen=N|Num=S|Cas=6|Gra=1|Neg=A|Var=6
A	SubPOS=A|Gen=N|Num=S|Cas=6|Gra=1|Neg=A|Var=7
A	SubPOS=A|Gen=N|Num=S|Cas=6|Gra=1|Neg=N
A	SubPOS=A|Gen=N|Num=S|Cas=6|Gra=1|Neg=N|Var=6
A	SubPOS=A|Gen=N|Num=S|Cas=6|Gra=1|Neg=N|Var=7
A	SubPOS=A|Gen=N|Num=S|Cas=6|Gra=2|Neg=A
A	SubPOS=A|Gen=N|Num=S|Cas=6|Gra=2|Neg=A|Var=6
A	SubPOS=A|Gen=N|Num=S|Cas=6|Gra=2|Neg=N
A	SubPOS=A|Gen=N|Num=S|Cas=6|Gra=2|Neg=N|Var=6
A	SubPOS=A|Gen=N|Num=S|Cas=6|Gra=3|Neg=A
A	SubPOS=A|Gen=N|Num=S|Cas=6|Gra=3|Neg=A|Var=6
A	SubPOS=A|Gen=N|Num=S|Cas=6|Gra=3|Neg=N
A	SubPOS=A|Gen=N|Num=S|Cas=6|Gra=3|Neg=N|Var=6
A	SubPOS=A|Gen=N|Num=S|Cas=7|Gra=1|Neg=A
A	SubPOS=A|Gen=N|Num=S|Cas=7|Gra=1|Neg=A|Sem=G
A	SubPOS=A|Gen=N|Num=S|Cas=7|Gra=1|Neg=A|Sem=S
A	SubPOS=A|Gen=N|Num=S|Cas=7|Gra=1|Neg=A|Sem=m
A	SubPOS=A|Gen=N|Num=S|Cas=7|Gra=1|Neg=A|Var=6
A	SubPOS=A|Gen=N|Num=S|Cas=7|Gra=1|Neg=A|Var=8
A	SubPOS=A|Gen=N|Num=S|Cas=7|Gra=1|Neg=N
A	SubPOS=A|Gen=N|Num=S|Cas=7|Gra=1|Neg=N|Var=6
A	SubPOS=A|Gen=N|Num=S|Cas=7|Gra=2|Neg=A
A	SubPOS=A|Gen=N|Num=S|Cas=7|Gra=2|Neg=A|Var=6
A	SubPOS=A|Gen=N|Num=S|Cas=7|Gra=2|Neg=N
A	SubPOS=A|Gen=N|Num=S|Cas=7|Gra=2|Neg=N|Var=6
A	SubPOS=A|Gen=N|Num=S|Cas=7|Gra=3|Neg=A
A	SubPOS=A|Gen=N|Num=S|Cas=7|Gra=3|Neg=A|Var=6
A	SubPOS=A|Gen=N|Num=S|Cas=7|Gra=3|Neg=N
A	SubPOS=A|Gen=N|Num=S|Cas=7|Gra=3|Neg=N|Var=6
A	SubPOS=A|Gen=N|Num=S|Cas=X|Gra=1|Neg=A
A	SubPOS=A|Gen=N|Num=S|Cas=X|Gra=1|Neg=A|Sem=G
A	SubPOS=A|Gen=N|Num=S|Cas=X|Gra=1|Neg=A|Sem=m
A	SubPOS=A|Gen=X|Num=P|Cas=X|Gra=1|Neg=A
A	SubPOS=A|Gen=X|Num=P|Cas=X|Gra=1|Neg=N
A	SubPOS=A|Gen=X|Num=X|Cas=X|Gra=1|Neg=A
A	SubPOS=A|Gen=X|Num=X|Cas=X|Gra=1|Neg=A|Sem=E
A	SubPOS=A|Gen=X|Num=X|Cas=X|Gra=1|Neg=A|Sem=G
A	SubPOS=A|Gen=X|Num=X|Cas=X|Gra=1|Neg=A|Sem=K
A	SubPOS=A|Gen=X|Num=X|Cas=X|Gra=1|Neg=A|Sem=R
A	SubPOS=A|Gen=X|Num=X|Cas=X|Gra=1|Neg=A|Sem=S
A	SubPOS=A|Gen=X|Num=X|Cas=X|Gra=1|Neg=A|Sem=Y
A	SubPOS=A|Gen=X|Num=X|Cas=X|Gra=1|Neg=A|Sem=m
A	SubPOS=A|Gen=X|Num=X|Cas=X|Gra=1|Neg=A|Sem=w
A	SubPOS=A|Gen=X|Num=X|Cas=X|Gra=1|Neg=A|Var=1
A	SubPOS=A|Gen=X|Num=X|Cas=X|Gra=1|Neg=A|Var=8
A	SubPOS=A|Gen=X|Num=X|Cas=X|Gra=1|Neg=A|Var=8|Sem=G
A	SubPOS=A|Gen=X|Num=X|Cas=X|Gra=1|Neg=A|Var=8|Sem=K
A	SubPOS=A|Gen=X|Num=X|Cas=X|Gra=1|Neg=A|Var=8|Sem=R
A	SubPOS=A|Gen=X|Num=X|Cas=X|Gra=1|Neg=A|Var=8|Sem=m
A	SubPOS=A|Gen=X|Num=X|Cas=X|Gra=1|Neg=A|Var=9
A	SubPOS=A|Gen=X|Num=X|Cas=X|Gra=1|Neg=N
A	SubPOS=A|Gen=X|Num=X|Cas=X|Gra=1|Neg=N|Var=8
A	SubPOS=A|Gen=X|Num=X|Cas=X|Gra=1|Neg=N|Var=9
A	SubPOS=A|Gen=X|Num=X|Cas=X|Gra=2|Neg=A|Var=8
A	SubPOS=A|Gen=X|Num=X|Cas=X|Gra=3|Neg=A|Var=8
A	SubPOS=C|Gen=F|Num=S|Cas=4|Neg=A
A	SubPOS=C|Gen=F|Num=S|Cas=4|Neg=N
A	SubPOS=C|Gen=M|Num=P|Neg=A
A	SubPOS=C|Gen=M|Num=P|Neg=N
A	SubPOS=C|Gen=N|Num=S|Neg=A
A	SubPOS=C|Gen=N|Num=S|Neg=N
A	SubPOS=C|Gen=Q|Num=W|Neg=A
A	SubPOS=C|Gen=Q|Num=W|Neg=N
A	SubPOS=C|Gen=T|Num=P|Neg=A
A	SubPOS=C|Gen=T|Num=P|Neg=N
A	SubPOS=C|Gen=Y|Num=S|Neg=A
A	SubPOS=C|Gen=Y|Num=S|Neg=N
A	SubPOS=G|Gen=F|Num=D|Cas=7|Neg=A
A	SubPOS=G|Gen=F|Num=D|Cas=7|Neg=A|Var=6
A	SubPOS=G|Gen=F|Num=D|Cas=7|Neg=N
A	SubPOS=G|Gen=F|Num=D|Cas=7|Neg=N|Var=6
A	SubPOS=G|Gen=F|Num=P|Cas=1|Neg=A
A	SubPOS=G|Gen=F|Num=P|Cas=1|Neg=A|Var=6
A	SubPOS=G|Gen=F|Num=P|Cas=1|Neg=N
A	SubPOS=G|Gen=F|Num=P|Cas=1|Neg=N|Var=6
A	SubPOS=G|Gen=F|Num=P|Cas=2|Neg=A
A	SubPOS=G|Gen=F|Num=P|Cas=2|Neg=A|Var=6
A	SubPOS=G|Gen=F|Num=P|Cas=2|Neg=N
A	SubPOS=G|Gen=F|Num=P|Cas=2|Neg=N|Var=6
A	SubPOS=G|Gen=F|Num=P|Cas=3|Neg=A
A	SubPOS=G|Gen=F|Num=P|Cas=3|Neg=A|Var=6
A	SubPOS=G|Gen=F|Num=P|Cas=3|Neg=N
A	SubPOS=G|Gen=F|Num=P|Cas=3|Neg=N|Var=6
A	SubPOS=G|Gen=F|Num=P|Cas=4|Neg=A
A	SubPOS=G|Gen=F|Num=P|Cas=4|Neg=A|Var=6
A	SubPOS=G|Gen=F|Num=P|Cas=4|Neg=N
A	SubPOS=G|Gen=F|Num=P|Cas=4|Neg=N|Var=6
A	SubPOS=G|Gen=F|Num=P|Cas=5|Neg=A
A	SubPOS=G|Gen=F|Num=P|Cas=5|Neg=A|Var=6
A	SubPOS=G|Gen=F|Num=P|Cas=5|Neg=N
A	SubPOS=G|Gen=F|Num=P|Cas=5|Neg=N|Var=6
A	SubPOS=G|Gen=F|Num=P|Cas=6|Neg=A
A	SubPOS=G|Gen=F|Num=P|Cas=6|Neg=A|Var=6
A	SubPOS=G|Gen=F|Num=P|Cas=6|Neg=N
A	SubPOS=G|Gen=F|Num=P|Cas=6|Neg=N|Var=6
A	SubPOS=G|Gen=F|Num=P|Cas=7|Neg=A
A	SubPOS=G|Gen=F|Num=P|Cas=7|Neg=A|Var=6
A	SubPOS=G|Gen=F|Num=P|Cas=7|Neg=N
A	SubPOS=G|Gen=F|Num=P|Cas=7|Neg=N|Var=6
A	SubPOS=G|Gen=F|Num=S|Cas=1|Neg=A
A	SubPOS=G|Gen=F|Num=S|Cas=1|Neg=A|Var=6
A	SubPOS=G|Gen=F|Num=S|Cas=1|Neg=N
A	SubPOS=G|Gen=F|Num=S|Cas=1|Neg=N|Var=6
A	SubPOS=G|Gen=F|Num=S|Cas=2|Neg=A
A	SubPOS=G|Gen=F|Num=S|Cas=2|Neg=A|Var=6
A	SubPOS=G|Gen=F|Num=S|Cas=2|Neg=N
A	SubPOS=G|Gen=F|Num=S|Cas=2|Neg=N|Var=6
A	SubPOS=G|Gen=F|Num=S|Cas=3|Neg=A
A	SubPOS=G|Gen=F|Num=S|Cas=3|Neg=A|Var=6
A	SubPOS=G|Gen=F|Num=S|Cas=3|Neg=N
A	SubPOS=G|Gen=F|Num=S|Cas=3|Neg=N|Var=6
A	SubPOS=G|Gen=F|Num=S|Cas=4|Neg=A
A	SubPOS=G|Gen=F|Num=S|Cas=4|Neg=A|Var=6
A	SubPOS=G|Gen=F|Num=S|Cas=4|Neg=N
A	SubPOS=G|Gen=F|Num=S|Cas=4|Neg=N|Var=6
A	SubPOS=G|Gen=F|Num=S|Cas=5|Neg=A
A	SubPOS=G|Gen=F|Num=S|Cas=5|Neg=A|Var=6
A	SubPOS=G|Gen=F|Num=S|Cas=5|Neg=N
A	SubPOS=G|Gen=F|Num=S|Cas=5|Neg=N|Var=6
A	SubPOS=G|Gen=F|Num=S|Cas=6|Neg=A
A	SubPOS=G|Gen=F|Num=S|Cas=6|Neg=A|Var=6
A	SubPOS=G|Gen=F|Num=S|Cas=6|Neg=N
A	SubPOS=G|Gen=F|Num=S|Cas=6|Neg=N|Var=6
A	SubPOS=G|Gen=F|Num=S|Cas=7|Neg=A
A	SubPOS=G|Gen=F|Num=S|Cas=7|Neg=A|Var=6
A	SubPOS=G|Gen=F|Num=S|Cas=7|Neg=N
A	SubPOS=G|Gen=F|Num=S|Cas=7|Neg=N|Var=6
A	SubPOS=G|Gen=I|Num=P|Cas=1|Neg=A
A	SubPOS=G|Gen=I|Num=P|Cas=1|Neg=A|Var=6
A	SubPOS=G|Gen=I|Num=P|Cas=1|Neg=N
A	SubPOS=G|Gen=I|Num=P|Cas=1|Neg=N|Var=6
A	SubPOS=G|Gen=I|Num=P|Cas=2|Neg=A
A	SubPOS=G|Gen=I|Num=P|Cas=2|Neg=A|Var=6
A	SubPOS=G|Gen=I|Num=P|Cas=2|Neg=N
A	SubPOS=G|Gen=I|Num=P|Cas=2|Neg=N|Var=6
A	SubPOS=G|Gen=I|Num=P|Cas=3|Neg=A
A	SubPOS=G|Gen=I|Num=P|Cas=3|Neg=A|Var=6
A	SubPOS=G|Gen=I|Num=P|Cas=3|Neg=N
A	SubPOS=G|Gen=I|Num=P|Cas=3|Neg=N|Var=6
A	SubPOS=G|Gen=I|Num=P|Cas=4|Neg=A
A	SubPOS=G|Gen=I|Num=P|Cas=4|Neg=A|Var=6
A	SubPOS=G|Gen=I|Num=P|Cas=4|Neg=N
A	SubPOS=G|Gen=I|Num=P|Cas=4|Neg=N|Var=6
A	SubPOS=G|Gen=I|Num=P|Cas=5|Neg=A
A	SubPOS=G|Gen=I|Num=P|Cas=5|Neg=A|Var=6
A	SubPOS=G|Gen=I|Num=P|Cas=5|Neg=N
A	SubPOS=G|Gen=I|Num=P|Cas=5|Neg=N|Var=6
A	SubPOS=G|Gen=I|Num=P|Cas=6|Neg=A
A	SubPOS=G|Gen=I|Num=P|Cas=6|Neg=A|Var=6
A	SubPOS=G|Gen=I|Num=P|Cas=6|Neg=N
A	SubPOS=G|Gen=I|Num=P|Cas=6|Neg=N|Var=6
A	SubPOS=G|Gen=I|Num=P|Cas=7|Neg=A
A	SubPOS=G|Gen=I|Num=P|Cas=7|Neg=A|Var=6
A	SubPOS=G|Gen=I|Num=P|Cas=7|Neg=N
A	SubPOS=G|Gen=I|Num=P|Cas=7|Neg=N|Var=6
A	SubPOS=G|Gen=I|Num=S|Cas=1|Neg=A
A	SubPOS=G|Gen=I|Num=S|Cas=1|Neg=A|Var=6
A	SubPOS=G|Gen=I|Num=S|Cas=1|Neg=N
A	SubPOS=G|Gen=I|Num=S|Cas=1|Neg=N|Var=6
A	SubPOS=G|Gen=I|Num=S|Cas=2|Neg=A
A	SubPOS=G|Gen=I|Num=S|Cas=2|Neg=A|Var=6
A	SubPOS=G|Gen=I|Num=S|Cas=2|Neg=N
A	SubPOS=G|Gen=I|Num=S|Cas=2|Neg=N|Var=6
A	SubPOS=G|Gen=I|Num=S|Cas=3|Neg=A
A	SubPOS=G|Gen=I|Num=S|Cas=3|Neg=A|Var=6
A	SubPOS=G|Gen=I|Num=S|Cas=3|Neg=N
A	SubPOS=G|Gen=I|Num=S|Cas=3|Neg=N|Var=6
A	SubPOS=G|Gen=I|Num=S|Cas=4|Neg=A
A	SubPOS=G|Gen=I|Num=S|Cas=4|Neg=A|Var=6
A	SubPOS=G|Gen=I|Num=S|Cas=4|Neg=N
A	SubPOS=G|Gen=I|Num=S|Cas=4|Neg=N|Var=6
A	SubPOS=G|Gen=I|Num=S|Cas=5|Neg=A
A	SubPOS=G|Gen=I|Num=S|Cas=5|Neg=A|Var=6
A	SubPOS=G|Gen=I|Num=S|Cas=5|Neg=N
A	SubPOS=G|Gen=I|Num=S|Cas=5|Neg=N|Var=6
A	SubPOS=G|Gen=I|Num=S|Cas=6|Neg=A
A	SubPOS=G|Gen=I|Num=S|Cas=6|Neg=A|Var=6
A	SubPOS=G|Gen=I|Num=S|Cas=6|Neg=A|Var=7
A	SubPOS=G|Gen=I|Num=S|Cas=6|Neg=N
A	SubPOS=G|Gen=I|Num=S|Cas=6|Neg=N|Var=6
A	SubPOS=G|Gen=I|Num=S|Cas=6|Neg=N|Var=7
A	SubPOS=G|Gen=I|Num=S|Cas=7|Neg=A
A	SubPOS=G|Gen=I|Num=S|Cas=7|Neg=A|Var=6
A	SubPOS=G|Gen=I|Num=S|Cas=7|Neg=N
A	SubPOS=G|Gen=I|Num=S|Cas=7|Neg=N|Var=6
A	SubPOS=G|Gen=M|Num=P|Cas=1|Neg=A
A	SubPOS=G|Gen=M|Num=P|Cas=1|Neg=A|Var=6
A	SubPOS=G|Gen=M|Num=P|Cas=1|Neg=N
A	SubPOS=G|Gen=M|Num=P|Cas=1|Neg=N|Var=6
A	SubPOS=G|Gen=M|Num=P|Cas=2|Neg=A
A	SubPOS=G|Gen=M|Num=P|Cas=2|Neg=A|Var=6
A	SubPOS=G|Gen=M|Num=P|Cas=2|Neg=N
A	SubPOS=G|Gen=M|Num=P|Cas=2|Neg=N|Var=6
A	SubPOS=G|Gen=M|Num=P|Cas=3|Neg=A
A	SubPOS=G|Gen=M|Num=P|Cas=3|Neg=A|Var=6
A	SubPOS=G|Gen=M|Num=P|Cas=3|Neg=N
A	SubPOS=G|Gen=M|Num=P|Cas=3|Neg=N|Var=6
A	SubPOS=G|Gen=M|Num=P|Cas=4|Neg=A
A	SubPOS=G|Gen=M|Num=P|Cas=4|Neg=A|Var=6
A	SubPOS=G|Gen=M|Num=P|Cas=4|Neg=N
A	SubPOS=G|Gen=M|Num=P|Cas=4|Neg=N|Var=6
A	SubPOS=G|Gen=M|Num=P|Cas=5|Neg=A
A	SubPOS=G|Gen=M|Num=P|Cas=5|Neg=A|Var=6
A	SubPOS=G|Gen=M|Num=P|Cas=5|Neg=N
A	SubPOS=G|Gen=M|Num=P|Cas=5|Neg=N|Var=6
A	SubPOS=G|Gen=M|Num=P|Cas=6|Neg=A
A	SubPOS=G|Gen=M|Num=P|Cas=6|Neg=A|Var=6
A	SubPOS=G|Gen=M|Num=P|Cas=6|Neg=N
A	SubPOS=G|Gen=M|Num=P|Cas=6|Neg=N|Var=6
A	SubPOS=G|Gen=M|Num=P|Cas=7|Neg=A
A	SubPOS=G|Gen=M|Num=P|Cas=7|Neg=A|Var=6
A	SubPOS=G|Gen=M|Num=P|Cas=7|Neg=N
A	SubPOS=G|Gen=M|Num=P|Cas=7|Neg=N|Var=6
A	SubPOS=G|Gen=M|Num=S|Cas=1|Neg=A
A	SubPOS=G|Gen=M|Num=S|Cas=1|Neg=A|Var=6
A	SubPOS=G|Gen=M|Num=S|Cas=1|Neg=N
A	SubPOS=G|Gen=M|Num=S|Cas=1|Neg=N|Var=6
A	SubPOS=G|Gen=M|Num=S|Cas=2|Neg=A
A	SubPOS=G|Gen=M|Num=S|Cas=2|Neg=A|Var=6
A	SubPOS=G|Gen=M|Num=S|Cas=2|Neg=N
A	SubPOS=G|Gen=M|Num=S|Cas=2|Neg=N|Var=6
A	SubPOS=G|Gen=M|Num=S|Cas=3|Neg=A
A	SubPOS=G|Gen=M|Num=S|Cas=3|Neg=A|Var=6
A	SubPOS=G|Gen=M|Num=S|Cas=3|Neg=N
A	SubPOS=G|Gen=M|Num=S|Cas=3|Neg=N|Var=6
A	SubPOS=G|Gen=M|Num=S|Cas=4|Neg=A
A	SubPOS=G|Gen=M|Num=S|Cas=4|Neg=A|Var=6
A	SubPOS=G|Gen=M|Num=S|Cas=4|Neg=N
A	SubPOS=G|Gen=M|Num=S|Cas=4|Neg=N|Var=6
A	SubPOS=G|Gen=M|Num=S|Cas=5|Neg=A
A	SubPOS=G|Gen=M|Num=S|Cas=5|Neg=A|Var=6
A	SubPOS=G|Gen=M|Num=S|Cas=5|Neg=N
A	SubPOS=G|Gen=M|Num=S|Cas=5|Neg=N|Var=6
A	SubPOS=G|Gen=M|Num=S|Cas=6|Neg=A
A	SubPOS=G|Gen=M|Num=S|Cas=6|Neg=A|Var=6
A	SubPOS=G|Gen=M|Num=S|Cas=6|Neg=A|Var=7
A	SubPOS=G|Gen=M|Num=S|Cas=6|Neg=N
A	SubPOS=G|Gen=M|Num=S|Cas=6|Neg=N|Var=6
A	SubPOS=G|Gen=M|Num=S|Cas=6|Neg=N|Var=7
A	SubPOS=G|Gen=M|Num=S|Cas=7|Neg=A
A	SubPOS=G|Gen=M|Num=S|Cas=7|Neg=A|Var=6
A	SubPOS=G|Gen=M|Num=S|Cas=7|Neg=N
A	SubPOS=G|Gen=M|Num=S|Cas=7|Neg=N|Var=6
A	SubPOS=G|Gen=N|Num=D|Cas=7|Neg=A
A	SubPOS=G|Gen=N|Num=D|Cas=7|Neg=A|Var=6
A	SubPOS=G|Gen=N|Num=D|Cas=7|Neg=N
A	SubPOS=G|Gen=N|Num=D|Cas=7|Neg=N|Var=6
A	SubPOS=G|Gen=N|Num=P|Cas=1|Neg=A
A	SubPOS=G|Gen=N|Num=P|Cas=1|Neg=A|Var=6
A	SubPOS=G|Gen=N|Num=P|Cas=1|Neg=N
A	SubPOS=G|Gen=N|Num=P|Cas=1|Neg=N|Var=6
A	SubPOS=G|Gen=N|Num=P|Cas=2|Neg=A
A	SubPOS=G|Gen=N|Num=P|Cas=2|Neg=A|Var=6
A	SubPOS=G|Gen=N|Num=P|Cas=2|Neg=N
A	SubPOS=G|Gen=N|Num=P|Cas=2|Neg=N|Var=6
A	SubPOS=G|Gen=N|Num=P|Cas=3|Neg=A
A	SubPOS=G|Gen=N|Num=P|Cas=3|Neg=A|Var=6
A	SubPOS=G|Gen=N|Num=P|Cas=3|Neg=N
A	SubPOS=G|Gen=N|Num=P|Cas=3|Neg=N|Var=6
A	SubPOS=G|Gen=N|Num=P|Cas=4|Neg=A
A	SubPOS=G|Gen=N|Num=P|Cas=4|Neg=A|Var=6
A	SubPOS=G|Gen=N|Num=P|Cas=4|Neg=N
A	SubPOS=G|Gen=N|Num=P|Cas=4|Neg=N|Var=6
A	SubPOS=G|Gen=N|Num=P|Cas=5|Neg=A
A	SubPOS=G|Gen=N|Num=P|Cas=5|Neg=A|Var=6
A	SubPOS=G|Gen=N|Num=P|Cas=5|Neg=N
A	SubPOS=G|Gen=N|Num=P|Cas=5|Neg=N|Var=6
A	SubPOS=G|Gen=N|Num=P|Cas=6|Neg=A
A	SubPOS=G|Gen=N|Num=P|Cas=6|Neg=A|Var=6
A	SubPOS=G|Gen=N|Num=P|Cas=6|Neg=N
A	SubPOS=G|Gen=N|Num=P|Cas=6|Neg=N|Var=6
A	SubPOS=G|Gen=N|Num=P|Cas=7|Neg=A
A	SubPOS=G|Gen=N|Num=P|Cas=7|Neg=A|Var=6
A	SubPOS=G|Gen=N|Num=P|Cas=7|Neg=N
A	SubPOS=G|Gen=N|Num=P|Cas=7|Neg=N|Var=6
A	SubPOS=G|Gen=N|Num=S|Cas=1|Neg=A
A	SubPOS=G|Gen=N|Num=S|Cas=1|Neg=A|Var=6
A	SubPOS=G|Gen=N|Num=S|Cas=1|Neg=N
A	SubPOS=G|Gen=N|Num=S|Cas=1|Neg=N|Var=6
A	SubPOS=G|Gen=N|Num=S|Cas=2|Neg=A
A	SubPOS=G|Gen=N|Num=S|Cas=2|Neg=A|Var=6
A	SubPOS=G|Gen=N|Num=S|Cas=2|Neg=N
A	SubPOS=G|Gen=N|Num=S|Cas=2|Neg=N|Var=6
A	SubPOS=G|Gen=N|Num=S|Cas=3|Neg=A
A	SubPOS=G|Gen=N|Num=S|Cas=3|Neg=A|Var=6
A	SubPOS=G|Gen=N|Num=S|Cas=3|Neg=N
A	SubPOS=G|Gen=N|Num=S|Cas=3|Neg=N|Var=6
A	SubPOS=G|Gen=N|Num=S|Cas=4|Neg=A
A	SubPOS=G|Gen=N|Num=S|Cas=4|Neg=A|Var=6
A	SubPOS=G|Gen=N|Num=S|Cas=4|Neg=N
A	SubPOS=G|Gen=N|Num=S|Cas=4|Neg=N|Var=6
A	SubPOS=G|Gen=N|Num=S|Cas=5|Neg=A
A	SubPOS=G|Gen=N|Num=S|Cas=5|Neg=A|Var=6
A	SubPOS=G|Gen=N|Num=S|Cas=5|Neg=N
A	SubPOS=G|Gen=N|Num=S|Cas=5|Neg=N|Var=6
A	SubPOS=G|Gen=N|Num=S|Cas=6|Neg=A
A	SubPOS=G|Gen=N|Num=S|Cas=6|Neg=A|Var=6
A	SubPOS=G|Gen=N|Num=S|Cas=6|Neg=A|Var=7
A	SubPOS=G|Gen=N|Num=S|Cas=6|Neg=N
A	SubPOS=G|Gen=N|Num=S|Cas=6|Neg=N|Var=6
A	SubPOS=G|Gen=N|Num=S|Cas=6|Neg=N|Var=7
A	SubPOS=G|Gen=N|Num=S|Cas=7|Neg=A
A	SubPOS=G|Gen=N|Num=S|Cas=7|Neg=A|Var=6
A	SubPOS=G|Gen=N|Num=S|Cas=7|Neg=N
A	SubPOS=G|Gen=N|Num=S|Cas=7|Neg=N|Var=6
A	SubPOS=M|Gen=F|Num=D|Cas=7|Neg=A
A	SubPOS=M|Gen=F|Num=D|Cas=7|Neg=A|Var=6
A	SubPOS=M|Gen=F|Num=D|Cas=7|Neg=N
A	SubPOS=M|Gen=F|Num=D|Cas=7|Neg=N|Var=6
A	SubPOS=M|Gen=F|Num=P|Cas=1|Neg=A
A	SubPOS=M|Gen=F|Num=P|Cas=1|Neg=A|Var=6
A	SubPOS=M|Gen=F|Num=P|Cas=1|Neg=N
A	SubPOS=M|Gen=F|Num=P|Cas=1|Neg=N|Var=6
A	SubPOS=M|Gen=F|Num=P|Cas=2|Neg=A
A	SubPOS=M|Gen=F|Num=P|Cas=2|Neg=A|Var=6
A	SubPOS=M|Gen=F|Num=P|Cas=2|Neg=N
A	SubPOS=M|Gen=F|Num=P|Cas=2|Neg=N|Var=6
A	SubPOS=M|Gen=F|Num=P|Cas=3|Neg=A
A	SubPOS=M|Gen=F|Num=P|Cas=3|Neg=A|Var=6
A	SubPOS=M|Gen=F|Num=P|Cas=3|Neg=N
A	SubPOS=M|Gen=F|Num=P|Cas=3|Neg=N|Var=6
A	SubPOS=M|Gen=F|Num=P|Cas=4|Neg=A
A	SubPOS=M|Gen=F|Num=P|Cas=4|Neg=A|Var=6
A	SubPOS=M|Gen=F|Num=P|Cas=4|Neg=N
A	SubPOS=M|Gen=F|Num=P|Cas=4|Neg=N|Var=6
A	SubPOS=M|Gen=F|Num=P|Cas=5|Neg=A
A	SubPOS=M|Gen=F|Num=P|Cas=5|Neg=A|Var=6
A	SubPOS=M|Gen=F|Num=P|Cas=5|Neg=N
A	SubPOS=M|Gen=F|Num=P|Cas=5|Neg=N|Var=6
A	SubPOS=M|Gen=F|Num=P|Cas=6|Neg=A
A	SubPOS=M|Gen=F|Num=P|Cas=6|Neg=A|Var=6
A	SubPOS=M|Gen=F|Num=P|Cas=6|Neg=N
A	SubPOS=M|Gen=F|Num=P|Cas=6|Neg=N|Var=6
A	SubPOS=M|Gen=F|Num=P|Cas=7|Neg=A
A	SubPOS=M|Gen=F|Num=P|Cas=7|Neg=A|Var=6
A	SubPOS=M|Gen=F|Num=P|Cas=7|Neg=N
A	SubPOS=M|Gen=F|Num=P|Cas=7|Neg=N|Var=6
A	SubPOS=M|Gen=F|Num=S|Cas=1|Neg=A
A	SubPOS=M|Gen=F|Num=S|Cas=1|Neg=A|Var=6
A	SubPOS=M|Gen=F|Num=S|Cas=1|Neg=N
A	SubPOS=M|Gen=F|Num=S|Cas=1|Neg=N|Var=6
A	SubPOS=M|Gen=F|Num=S|Cas=2|Neg=A
A	SubPOS=M|Gen=F|Num=S|Cas=2|Neg=A|Var=6
A	SubPOS=M|Gen=F|Num=S|Cas=2|Neg=N
A	SubPOS=M|Gen=F|Num=S|Cas=2|Neg=N|Var=6
A	SubPOS=M|Gen=F|Num=S|Cas=3|Neg=A
A	SubPOS=M|Gen=F|Num=S|Cas=3|Neg=A|Var=6
A	SubPOS=M|Gen=F|Num=S|Cas=3|Neg=N
A	SubPOS=M|Gen=F|Num=S|Cas=3|Neg=N|Var=6
A	SubPOS=M|Gen=F|Num=S|Cas=4|Neg=A
A	SubPOS=M|Gen=F|Num=S|Cas=4|Neg=A|Var=6
A	SubPOS=M|Gen=F|Num=S|Cas=4|Neg=N
A	SubPOS=M|Gen=F|Num=S|Cas=4|Neg=N|Var=6
A	SubPOS=M|Gen=F|Num=S|Cas=5|Neg=A
A	SubPOS=M|Gen=F|Num=S|Cas=5|Neg=A|Var=6
A	SubPOS=M|Gen=F|Num=S|Cas=5|Neg=N
A	SubPOS=M|Gen=F|Num=S|Cas=5|Neg=N|Var=6
A	SubPOS=M|Gen=F|Num=S|Cas=6|Neg=A
A	SubPOS=M|Gen=F|Num=S|Cas=6|Neg=A|Var=6
A	SubPOS=M|Gen=F|Num=S|Cas=6|Neg=N
A	SubPOS=M|Gen=F|Num=S|Cas=6|Neg=N|Var=6
A	SubPOS=M|Gen=F|Num=S|Cas=7|Neg=A
A	SubPOS=M|Gen=F|Num=S|Cas=7|Neg=A|Var=6
A	SubPOS=M|Gen=F|Num=S|Cas=7|Neg=N
A	SubPOS=M|Gen=F|Num=S|Cas=7|Neg=N|Var=6
A	SubPOS=M|Gen=I|Num=P|Cas=1|Neg=A
A	SubPOS=M|Gen=I|Num=P|Cas=1|Neg=A|Var=6
A	SubPOS=M|Gen=I|Num=P|Cas=1|Neg=N
A	SubPOS=M|Gen=I|Num=P|Cas=1|Neg=N|Var=6
A	SubPOS=M|Gen=I|Num=P|Cas=2|Neg=A
A	SubPOS=M|Gen=I|Num=P|Cas=2|Neg=A|Var=6
A	SubPOS=M|Gen=I|Num=P|Cas=2|Neg=N
A	SubPOS=M|Gen=I|Num=P|Cas=2|Neg=N|Var=6
A	SubPOS=M|Gen=I|Num=P|Cas=3|Neg=A
A	SubPOS=M|Gen=I|Num=P|Cas=3|Neg=A|Var=6
A	SubPOS=M|Gen=I|Num=P|Cas=3|Neg=N
A	SubPOS=M|Gen=I|Num=P|Cas=3|Neg=N|Var=6
A	SubPOS=M|Gen=I|Num=P|Cas=4|Neg=A
A	SubPOS=M|Gen=I|Num=P|Cas=4|Neg=A|Var=6
A	SubPOS=M|Gen=I|Num=P|Cas=4|Neg=N
A	SubPOS=M|Gen=I|Num=P|Cas=4|Neg=N|Var=6
A	SubPOS=M|Gen=I|Num=P|Cas=5|Neg=A
A	SubPOS=M|Gen=I|Num=P|Cas=5|Neg=A|Var=6
A	SubPOS=M|Gen=I|Num=P|Cas=5|Neg=N
A	SubPOS=M|Gen=I|Num=P|Cas=5|Neg=N|Var=6
A	SubPOS=M|Gen=I|Num=P|Cas=6|Neg=A
A	SubPOS=M|Gen=I|Num=P|Cas=6|Neg=A|Var=6
A	SubPOS=M|Gen=I|Num=P|Cas=6|Neg=N
A	SubPOS=M|Gen=I|Num=P|Cas=6|Neg=N|Var=6
A	SubPOS=M|Gen=I|Num=P|Cas=7|Neg=A
A	SubPOS=M|Gen=I|Num=P|Cas=7|Neg=A|Var=6
A	SubPOS=M|Gen=I|Num=P|Cas=7|Neg=N
A	SubPOS=M|Gen=I|Num=P|Cas=7|Neg=N|Var=6
A	SubPOS=M|Gen=I|Num=S|Cas=1|Neg=A
A	SubPOS=M|Gen=I|Num=S|Cas=1|Neg=A|Var=6
A	SubPOS=M|Gen=I|Num=S|Cas=1|Neg=N
A	SubPOS=M|Gen=I|Num=S|Cas=1|Neg=N|Var=6
A	SubPOS=M|Gen=I|Num=S|Cas=2|Neg=A
A	SubPOS=M|Gen=I|Num=S|Cas=2|Neg=A|Var=6
A	SubPOS=M|Gen=I|Num=S|Cas=2|Neg=N
A	SubPOS=M|Gen=I|Num=S|Cas=2|Neg=N|Var=6
A	SubPOS=M|Gen=I|Num=S|Cas=3|Neg=A
A	SubPOS=M|Gen=I|Num=S|Cas=3|Neg=A|Var=6
A	SubPOS=M|Gen=I|Num=S|Cas=3|Neg=N
A	SubPOS=M|Gen=I|Num=S|Cas=3|Neg=N|Var=6
A	SubPOS=M|Gen=I|Num=S|Cas=4|Neg=A
A	SubPOS=M|Gen=I|Num=S|Cas=4|Neg=A|Var=6
A	SubPOS=M|Gen=I|Num=S|Cas=4|Neg=N
A	SubPOS=M|Gen=I|Num=S|Cas=4|Neg=N|Var=6
A	SubPOS=M|Gen=I|Num=S|Cas=5|Neg=A
A	SubPOS=M|Gen=I|Num=S|Cas=5|Neg=A|Var=6
A	SubPOS=M|Gen=I|Num=S|Cas=5|Neg=N
A	SubPOS=M|Gen=I|Num=S|Cas=5|Neg=N|Var=6
A	SubPOS=M|Gen=I|Num=S|Cas=6|Neg=A
A	SubPOS=M|Gen=I|Num=S|Cas=6|Neg=A|Var=6
A	SubPOS=M|Gen=I|Num=S|Cas=6|Neg=A|Var=7
A	SubPOS=M|Gen=I|Num=S|Cas=6|Neg=N
A	SubPOS=M|Gen=I|Num=S|Cas=6|Neg=N|Var=6
A	SubPOS=M|Gen=I|Num=S|Cas=6|Neg=N|Var=7
A	SubPOS=M|Gen=I|Num=S|Cas=7|Neg=A
A	SubPOS=M|Gen=I|Num=S|Cas=7|Neg=A|Var=6
A	SubPOS=M|Gen=I|Num=S|Cas=7|Neg=N
A	SubPOS=M|Gen=I|Num=S|Cas=7|Neg=N|Var=6
A	SubPOS=M|Gen=M|Num=P|Cas=1|Neg=A
A	SubPOS=M|Gen=M|Num=P|Cas=1|Neg=A|Var=6
A	SubPOS=M|Gen=M|Num=P|Cas=1|Neg=N
A	SubPOS=M|Gen=M|Num=P|Cas=1|Neg=N|Var=6
A	SubPOS=M|Gen=M|Num=P|Cas=2|Neg=A
A	SubPOS=M|Gen=M|Num=P|Cas=2|Neg=A|Var=6
A	SubPOS=M|Gen=M|Num=P|Cas=2|Neg=N
A	SubPOS=M|Gen=M|Num=P|Cas=2|Neg=N|Var=6
A	SubPOS=M|Gen=M|Num=P|Cas=3|Neg=A
A	SubPOS=M|Gen=M|Num=P|Cas=3|Neg=A|Var=6
A	SubPOS=M|Gen=M|Num=P|Cas=3|Neg=N
A	SubPOS=M|Gen=M|Num=P|Cas=3|Neg=N|Var=6
A	SubPOS=M|Gen=M|Num=P|Cas=4|Neg=A
A	SubPOS=M|Gen=M|Num=P|Cas=4|Neg=A|Var=6
A	SubPOS=M|Gen=M|Num=P|Cas=4|Neg=N
A	SubPOS=M|Gen=M|Num=P|Cas=4|Neg=N|Var=6
A	SubPOS=M|Gen=M|Num=P|Cas=5|Neg=A
A	SubPOS=M|Gen=M|Num=P|Cas=5|Neg=A|Var=6
A	SubPOS=M|Gen=M|Num=P|Cas=5|Neg=N
A	SubPOS=M|Gen=M|Num=P|Cas=5|Neg=N|Var=6
A	SubPOS=M|Gen=M|Num=P|Cas=6|Neg=A
A	SubPOS=M|Gen=M|Num=P|Cas=6|Neg=A|Var=6
A	SubPOS=M|Gen=M|Num=P|Cas=6|Neg=N
A	SubPOS=M|Gen=M|Num=P|Cas=6|Neg=N|Var=6
A	SubPOS=M|Gen=M|Num=P|Cas=7|Neg=A
A	SubPOS=M|Gen=M|Num=P|Cas=7|Neg=A|Var=6
A	SubPOS=M|Gen=M|Num=P|Cas=7|Neg=N
A	SubPOS=M|Gen=M|Num=P|Cas=7|Neg=N|Var=6
A	SubPOS=M|Gen=M|Num=S|Cas=1|Neg=A
A	SubPOS=M|Gen=M|Num=S|Cas=1|Neg=A|Var=6
A	SubPOS=M|Gen=M|Num=S|Cas=1|Neg=N
A	SubPOS=M|Gen=M|Num=S|Cas=1|Neg=N|Var=6
A	SubPOS=M|Gen=M|Num=S|Cas=2|Neg=A
A	SubPOS=M|Gen=M|Num=S|Cas=2|Neg=A|Var=6
A	SubPOS=M|Gen=M|Num=S|Cas=2|Neg=N
A	SubPOS=M|Gen=M|Num=S|Cas=2|Neg=N|Var=6
A	SubPOS=M|Gen=M|Num=S|Cas=3|Neg=A
A	SubPOS=M|Gen=M|Num=S|Cas=3|Neg=A|Var=6
A	SubPOS=M|Gen=M|Num=S|Cas=3|Neg=N
A	SubPOS=M|Gen=M|Num=S|Cas=3|Neg=N|Var=6
A	SubPOS=M|Gen=M|Num=S|Cas=4|Neg=A
A	SubPOS=M|Gen=M|Num=S|Cas=4|Neg=A|Var=6
A	SubPOS=M|Gen=M|Num=S|Cas=4|Neg=N
A	SubPOS=M|Gen=M|Num=S|Cas=4|Neg=N|Var=6
A	SubPOS=M|Gen=M|Num=S|Cas=5|Neg=A
A	SubPOS=M|Gen=M|Num=S|Cas=5|Neg=A|Var=6
A	SubPOS=M|Gen=M|Num=S|Cas=5|Neg=N
A	SubPOS=M|Gen=M|Num=S|Cas=5|Neg=N|Var=6
A	SubPOS=M|Gen=M|Num=S|Cas=6|Neg=A
A	SubPOS=M|Gen=M|Num=S|Cas=6|Neg=A|Var=6
A	SubPOS=M|Gen=M|Num=S|Cas=6|Neg=A|Var=7
A	SubPOS=M|Gen=M|Num=S|Cas=6|Neg=N
A	SubPOS=M|Gen=M|Num=S|Cas=6|Neg=N|Var=6
A	SubPOS=M|Gen=M|Num=S|Cas=6|Neg=N|Var=7
A	SubPOS=M|Gen=M|Num=S|Cas=7|Neg=A
A	SubPOS=M|Gen=M|Num=S|Cas=7|Neg=A|Var=6
A	SubPOS=M|Gen=M|Num=S|Cas=7|Neg=N
A	SubPOS=M|Gen=M|Num=S|Cas=7|Neg=N|Var=6
A	SubPOS=M|Gen=N|Num=D|Cas=7|Neg=A
A	SubPOS=M|Gen=N|Num=D|Cas=7|Neg=A|Var=6
A	SubPOS=M|Gen=N|Num=D|Cas=7|Neg=N
A	SubPOS=M|Gen=N|Num=D|Cas=7|Neg=N|Var=6
A	SubPOS=M|Gen=N|Num=P|Cas=1|Neg=A
A	SubPOS=M|Gen=N|Num=P|Cas=1|Neg=A|Var=6
A	SubPOS=M|Gen=N|Num=P|Cas=1|Neg=N
A	SubPOS=M|Gen=N|Num=P|Cas=1|Neg=N|Var=6
A	SubPOS=M|Gen=N|Num=P|Cas=2|Neg=A
A	SubPOS=M|Gen=N|Num=P|Cas=2|Neg=A|Var=6
A	SubPOS=M|Gen=N|Num=P|Cas=2|Neg=N
A	SubPOS=M|Gen=N|Num=P|Cas=2|Neg=N|Var=6
A	SubPOS=M|Gen=N|Num=P|Cas=3|Neg=A
A	SubPOS=M|Gen=N|Num=P|Cas=3|Neg=A|Var=6
A	SubPOS=M|Gen=N|Num=P|Cas=3|Neg=N
A	SubPOS=M|Gen=N|Num=P|Cas=3|Neg=N|Var=6
A	SubPOS=M|Gen=N|Num=P|Cas=4|Neg=A
A	SubPOS=M|Gen=N|Num=P|Cas=4|Neg=A|Var=6
A	SubPOS=M|Gen=N|Num=P|Cas=4|Neg=N
A	SubPOS=M|Gen=N|Num=P|Cas=4|Neg=N|Var=6
A	SubPOS=M|Gen=N|Num=P|Cas=5|Neg=A
A	SubPOS=M|Gen=N|Num=P|Cas=5|Neg=A|Var=6
A	SubPOS=M|Gen=N|Num=P|Cas=5|Neg=N
A	SubPOS=M|Gen=N|Num=P|Cas=5|Neg=N|Var=6
A	SubPOS=M|Gen=N|Num=P|Cas=6|Neg=A
A	SubPOS=M|Gen=N|Num=P|Cas=6|Neg=A|Var=6
A	SubPOS=M|Gen=N|Num=P|Cas=6|Neg=N
A	SubPOS=M|Gen=N|Num=P|Cas=6|Neg=N|Var=6
A	SubPOS=M|Gen=N|Num=P|Cas=7|Neg=A
A	SubPOS=M|Gen=N|Num=P|Cas=7|Neg=A|Var=6
A	SubPOS=M|Gen=N|Num=P|Cas=7|Neg=N
A	SubPOS=M|Gen=N|Num=P|Cas=7|Neg=N|Var=6
A	SubPOS=M|Gen=N|Num=S|Cas=1|Neg=A
A	SubPOS=M|Gen=N|Num=S|Cas=1|Neg=A|Var=6
A	SubPOS=M|Gen=N|Num=S|Cas=1|Neg=N
A	SubPOS=M|Gen=N|Num=S|Cas=1|Neg=N|Var=6
A	SubPOS=M|Gen=N|Num=S|Cas=2|Neg=A
A	SubPOS=M|Gen=N|Num=S|Cas=2|Neg=A|Var=6
A	SubPOS=M|Gen=N|Num=S|Cas=2|Neg=N
A	SubPOS=M|Gen=N|Num=S|Cas=2|Neg=N|Var=6
A	SubPOS=M|Gen=N|Num=S|Cas=3|Neg=A
A	SubPOS=M|Gen=N|Num=S|Cas=3|Neg=A|Var=6
A	SubPOS=M|Gen=N|Num=S|Cas=3|Neg=N
A	SubPOS=M|Gen=N|Num=S|Cas=3|Neg=N|Var=6
A	SubPOS=M|Gen=N|Num=S|Cas=4|Neg=A
A	SubPOS=M|Gen=N|Num=S|Cas=4|Neg=A|Var=6
A	SubPOS=M|Gen=N|Num=S|Cas=4|Neg=N
A	SubPOS=M|Gen=N|Num=S|Cas=4|Neg=N|Var=6
A	SubPOS=M|Gen=N|Num=S|Cas=5|Neg=A
A	SubPOS=M|Gen=N|Num=S|Cas=5|Neg=A|Var=6
A	SubPOS=M|Gen=N|Num=S|Cas=5|Neg=N
A	SubPOS=M|Gen=N|Num=S|Cas=5|Neg=N|Var=6
A	SubPOS=M|Gen=N|Num=S|Cas=6|Neg=A
A	SubPOS=M|Gen=N|Num=S|Cas=6|Neg=A|Var=6
A	SubPOS=M|Gen=N|Num=S|Cas=6|Neg=A|Var=7
A	SubPOS=M|Gen=N|Num=S|Cas=6|Neg=N
A	SubPOS=M|Gen=N|Num=S|Cas=6|Neg=N|Var=6
A	SubPOS=M|Gen=N|Num=S|Cas=6|Neg=N|Var=7
A	SubPOS=M|Gen=N|Num=S|Cas=7|Neg=A
A	SubPOS=M|Gen=N|Num=S|Cas=7|Neg=A|Var=6
A	SubPOS=M|Gen=N|Num=S|Cas=7|Neg=N
A	SubPOS=M|Gen=N|Num=S|Cas=7|Neg=N|Var=6
A	SubPOS=O|Gen=F|Num=P
A	SubPOS=O|Gen=F|Num=P|Var=1
A	SubPOS=O|Gen=F|Num=P|Var=6
A	SubPOS=O|Gen=F|Num=S
A	SubPOS=O|Gen=F|Num=S|Var=1
A	SubPOS=O|Gen=I|Num=P
A	SubPOS=O|Gen=I|Num=P|Var=1
A	SubPOS=O|Gen=I|Num=P|Var=6
A	SubPOS=O|Gen=M|Num=P
A	SubPOS=O|Gen=M|Num=P|Var=1
A	SubPOS=O|Gen=M|Num=P|Var=6
A	SubPOS=O|Gen=N|Num=P
A	SubPOS=O|Gen=N|Num=P|Var=1
A	SubPOS=O|Gen=N|Num=P|Var=6
A	SubPOS=O|Gen=N|Num=S
A	SubPOS=O|Gen=N|Num=S|Var=1
A	SubPOS=O|Gen=N|Num=S|Var=6
A	SubPOS=O|Gen=Y|Num=S
A	SubPOS=O|Gen=Y|Num=S|Var=6
A	SubPOS=U|Gen=F|Num=D|Cas=7|PGe=F
A	SubPOS=U|Gen=F|Num=D|Cas=7|PGe=F|Var=6
A	SubPOS=U|Gen=F|Num=D|Cas=7|PGe=M
A	SubPOS=U|Gen=F|Num=D|Cas=7|PGe=M|Var=6
A	SubPOS=U|Gen=F|Num=P|Cas=1|PGe=F
A	SubPOS=U|Gen=F|Num=P|Cas=1|PGe=F|Sem=Y
A	SubPOS=U|Gen=F|Num=P|Cas=1|PGe=F|Sem=m
A	SubPOS=U|Gen=F|Num=P|Cas=1|PGe=F|Var=6
A	SubPOS=U|Gen=F|Num=P|Cas=1|PGe=M
A	SubPOS=U|Gen=F|Num=P|Cas=1|PGe=M|Sem=S
A	SubPOS=U|Gen=F|Num=P|Cas=1|PGe=M|Sem=Y
A	SubPOS=U|Gen=F|Num=P|Cas=1|PGe=M|Sem=m
A	SubPOS=U|Gen=F|Num=P|Cas=1|PGe=M|Var=5
A	SubPOS=U|Gen=F|Num=P|Cas=1|PGe=M|Var=6
A	SubPOS=U|Gen=F|Num=P|Cas=2|PGe=F
A	SubPOS=U|Gen=F|Num=P|Cas=2|PGe=F|Sem=Y
A	SubPOS=U|Gen=F|Num=P|Cas=2|PGe=F|Sem=m
A	SubPOS=U|Gen=F|Num=P|Cas=2|PGe=F|Var=6
A	SubPOS=U|Gen=F|Num=P|Cas=2|PGe=M
A	SubPOS=U|Gen=F|Num=P|Cas=2|PGe=M|Sem=S
A	SubPOS=U|Gen=F|Num=P|Cas=2|PGe=M|Sem=Y
A	SubPOS=U|Gen=F|Num=P|Cas=2|PGe=M|Sem=m
A	SubPOS=U|Gen=F|Num=P|Cas=2|PGe=M|Var=6
A	SubPOS=U|Gen=F|Num=P|Cas=3|PGe=F
A	SubPOS=U|Gen=F|Num=P|Cas=3|PGe=F|Var=6
A	SubPOS=U|Gen=F|Num=P|Cas=3|PGe=M
A	SubPOS=U|Gen=F|Num=P|Cas=3|PGe=M|Sem=S
A	SubPOS=U|Gen=F|Num=P|Cas=3|PGe=M|Sem=Y
A	SubPOS=U|Gen=F|Num=P|Cas=3|PGe=M|Sem=m
A	SubPOS=U|Gen=F|Num=P|Cas=3|PGe=M|Var=6
A	SubPOS=U|Gen=F|Num=P|Cas=4|PGe=F
A	SubPOS=U|Gen=F|Num=P|Cas=4|PGe=F|Sem=Y
A	SubPOS=U|Gen=F|Num=P|Cas=4|PGe=F|Sem=m
A	SubPOS=U|Gen=F|Num=P|Cas=4|PGe=F|Var=6
A	SubPOS=U|Gen=F|Num=P|Cas=4|PGe=M
A	SubPOS=U|Gen=F|Num=P|Cas=4|PGe=M|Sem=E
A	SubPOS=U|Gen=F|Num=P|Cas=4|PGe=M|Sem=S
A	SubPOS=U|Gen=F|Num=P|Cas=4|PGe=M|Sem=m
A	SubPOS=U|Gen=F|Num=P|Cas=4|PGe=M|Var=6
A	SubPOS=U|Gen=F|Num=P|Cas=5|PGe=F
A	SubPOS=U|Gen=F|Num=P|Cas=5|PGe=F|Var=6
A	SubPOS=U|Gen=F|Num=P|Cas=5|PGe=M
A	SubPOS=U|Gen=F|Num=P|Cas=5|PGe=M|Var=6
A	SubPOS=U|Gen=F|Num=P|Cas=6|PGe=F
A	SubPOS=U|Gen=F|Num=P|Cas=6|PGe=F|Var=6
A	SubPOS=U|Gen=F|Num=P|Cas=6|PGe=M
A	SubPOS=U|Gen=F|Num=P|Cas=6|PGe=M|Sem=S
A	SubPOS=U|Gen=F|Num=P|Cas=6|PGe=M|Sem=Y
A	SubPOS=U|Gen=F|Num=P|Cas=6|PGe=M|Sem=m
A	SubPOS=U|Gen=F|Num=P|Cas=6|PGe=M|Var=6
A	SubPOS=U|Gen=F|Num=P|Cas=7|PGe=F
A	SubPOS=U|Gen=F|Num=P|Cas=7|PGe=F|Var=6
A	SubPOS=U|Gen=F|Num=P|Cas=7|PGe=F|Var=7
A	SubPOS=U|Gen=F|Num=P|Cas=7|PGe=M
A	SubPOS=U|Gen=F|Num=P|Cas=7|PGe=M|Sem=S
A	SubPOS=U|Gen=F|Num=P|Cas=7|PGe=M|Sem=m
A	SubPOS=U|Gen=F|Num=P|Cas=7|PGe=M|Var=6
A	SubPOS=U|Gen=F|Num=P|Cas=7|PGe=M|Var=7
A	SubPOS=U|Gen=F|Num=S|Cas=1|PGe=F
A	SubPOS=U|Gen=F|Num=S|Cas=1|PGe=F|Sem=Y
A	SubPOS=U|Gen=F|Num=S|Cas=1|PGe=F|Sem=m
A	SubPOS=U|Gen=F|Num=S|Cas=1|PGe=F|Var=6
A	SubPOS=U|Gen=F|Num=S|Cas=1|PGe=M
A	SubPOS=U|Gen=F|Num=S|Cas=1|PGe=M|Sem=G
A	SubPOS=U|Gen=F|Num=S|Cas=1|PGe=M|Sem=S
A	SubPOS=U|Gen=F|Num=S|Cas=1|PGe=M|Sem=Y
A	SubPOS=U|Gen=F|Num=S|Cas=1|PGe=M|Sem=m
A	SubPOS=U|Gen=F|Num=S|Cas=1|PGe=M|Var=6
A	SubPOS=U|Gen=F|Num=S|Cas=2|PGe=F
A	SubPOS=U|Gen=F|Num=S|Cas=2|PGe=F|Sem=Y
A	SubPOS=U|Gen=F|Num=S|Cas=2|PGe=F|Sem=m
A	SubPOS=U|Gen=F|Num=S|Cas=2|PGe=F|Var=6
A	SubPOS=U|Gen=F|Num=S|Cas=2|PGe=M
A	SubPOS=U|Gen=F|Num=S|Cas=2|PGe=M|Sem=S
A	SubPOS=U|Gen=F|Num=S|Cas=2|PGe=M|Sem=Y
A	SubPOS=U|Gen=F|Num=S|Cas=2|PGe=M|Sem=m
A	SubPOS=U|Gen=F|Num=S|Cas=2|PGe=M|Var=6
A	SubPOS=U|Gen=F|Num=S|Cas=3|PGe=F
A	SubPOS=U|Gen=F|Num=S|Cas=3|PGe=F|Var=6
A	SubPOS=U|Gen=F|Num=S|Cas=3|PGe=M
A	SubPOS=U|Gen=F|Num=S|Cas=3|PGe=M|Sem=S
A	SubPOS=U|Gen=F|Num=S|Cas=3|PGe=M|Sem=Y
A	SubPOS=U|Gen=F|Num=S|Cas=3|PGe=M|Sem=m
A	SubPOS=U|Gen=F|Num=S|Cas=3|PGe=M|Var=6
A	SubPOS=U|Gen=F|Num=S|Cas=4|PGe=F
A	SubPOS=U|Gen=F|Num=S|Cas=4|PGe=F|Sem=S
A	SubPOS=U|Gen=F|Num=S|Cas=4|PGe=F|Sem=m
A	SubPOS=U|Gen=F|Num=S|Cas=4|PGe=F|Var=6
A	SubPOS=U|Gen=F|Num=S|Cas=4|PGe=M
A	SubPOS=U|Gen=F|Num=S|Cas=4|PGe=M|Sem=S
A	SubPOS=U|Gen=F|Num=S|Cas=4|PGe=M|Sem=Y
A	SubPOS=U|Gen=F|Num=S|Cas=4|PGe=M|Sem=m
A	SubPOS=U|Gen=F|Num=S|Cas=4|PGe=M|Var=6
A	SubPOS=U|Gen=F|Num=S|Cas=5|PGe=F
A	SubPOS=U|Gen=F|Num=S|Cas=5|PGe=F|Var=6
A	SubPOS=U|Gen=F|Num=S|Cas=5|PGe=M
A	SubPOS=U|Gen=F|Num=S|Cas=5|PGe=M|Var=6
A	SubPOS=U|Gen=F|Num=S|Cas=6|PGe=F
A	SubPOS=U|Gen=F|Num=S|Cas=6|PGe=F|Var=6
A	SubPOS=U|Gen=F|Num=S|Cas=6|PGe=M
A	SubPOS=U|Gen=F|Num=S|Cas=6|PGe=M|Sem=S
A	SubPOS=U|Gen=F|Num=S|Cas=6|PGe=M|Sem=Y
A	SubPOS=U|Gen=F|Num=S|Cas=6|PGe=M|Sem=m
A	SubPOS=U|Gen=F|Num=S|Cas=6|PGe=M|Var=6
A	SubPOS=U|Gen=F|Num=S|Cas=7|PGe=F
A	SubPOS=U|Gen=F|Num=S|Cas=7|PGe=F|Var=6
A	SubPOS=U|Gen=F|Num=S|Cas=7|PGe=M
A	SubPOS=U|Gen=F|Num=S|Cas=7|PGe=M|Sem=S
A	SubPOS=U|Gen=F|Num=S|Cas=7|PGe=M|Sem=Y
A	SubPOS=U|Gen=F|Num=S|Cas=7|PGe=M|Sem=m
A	SubPOS=U|Gen=F|Num=S|Cas=7|PGe=M|Var=6
A	SubPOS=U|Gen=I|Num=P|Cas=1|PGe=F
A	SubPOS=U|Gen=I|Num=P|Cas=1|PGe=F|Sem=Y
A	SubPOS=U|Gen=I|Num=P|Cas=1|PGe=F|Sem=m
A	SubPOS=U|Gen=I|Num=P|Cas=1|PGe=F|Var=6
A	SubPOS=U|Gen=I|Num=P|Cas=1|PGe=M
A	SubPOS=U|Gen=I|Num=P|Cas=1|PGe=M|Sem=E
A	SubPOS=U|Gen=I|Num=P|Cas=1|PGe=M|Sem=G
A	SubPOS=U|Gen=I|Num=P|Cas=1|PGe=M|Sem=S
A	SubPOS=U|Gen=I|Num=P|Cas=1|PGe=M|Sem=Y
A	SubPOS=U|Gen=I|Num=P|Cas=1|PGe=M|Sem=m
A	SubPOS=U|Gen=I|Num=P|Cas=1|PGe=M|Var=6
A	SubPOS=U|Gen=I|Num=P|Cas=2|PGe=F
A	SubPOS=U|Gen=I|Num=P|Cas=2|PGe=F|Var=6
A	SubPOS=U|Gen=I|Num=P|Cas=2|PGe=M
A	SubPOS=U|Gen=I|Num=P|Cas=2|PGe=M|Sem=S
A	SubPOS=U|Gen=I|Num=P|Cas=2|PGe=M|Sem=Y
A	SubPOS=U|Gen=I|Num=P|Cas=2|PGe=M|Sem=m
A	SubPOS=U|Gen=I|Num=P|Cas=2|PGe=M|Var=6
A	SubPOS=U|Gen=I|Num=P|Cas=3|PGe=F
A	SubPOS=U|Gen=I|Num=P|Cas=3|PGe=F|Var=6
A	SubPOS=U|Gen=I|Num=P|Cas=3|PGe=M
A	SubPOS=U|Gen=I|Num=P|Cas=3|PGe=M|Sem=S
A	SubPOS=U|Gen=I|Num=P|Cas=3|PGe=M|Sem=Y
A	SubPOS=U|Gen=I|Num=P|Cas=3|PGe=M|Sem=m
A	SubPOS=U|Gen=I|Num=P|Cas=3|PGe=M|Var=6
A	SubPOS=U|Gen=I|Num=P|Cas=4|PGe=F
A	SubPOS=U|Gen=I|Num=P|Cas=4|PGe=F|Var=6
A	SubPOS=U|Gen=I|Num=P|Cas=4|PGe=M
A	SubPOS=U|Gen=I|Num=P|Cas=4|PGe=M|Sem=S
A	SubPOS=U|Gen=I|Num=P|Cas=4|PGe=M|Sem=Y
A	SubPOS=U|Gen=I|Num=P|Cas=4|PGe=M|Sem=m
A	SubPOS=U|Gen=I|Num=P|Cas=4|PGe=M|Var=6
A	SubPOS=U|Gen=I|Num=P|Cas=5|PGe=F
A	SubPOS=U|Gen=I|Num=P|Cas=5|PGe=F|Var=6
A	SubPOS=U|Gen=I|Num=P|Cas=5|PGe=M
A	SubPOS=U|Gen=I|Num=P|Cas=5|PGe=M|Var=6
A	SubPOS=U|Gen=I|Num=P|Cas=6|PGe=F
A	SubPOS=U|Gen=I|Num=P|Cas=6|PGe=F|Var=6
A	SubPOS=U|Gen=I|Num=P|Cas=6|PGe=M
A	SubPOS=U|Gen=I|Num=P|Cas=6|PGe=M|Sem=G
A	SubPOS=U|Gen=I|Num=P|Cas=6|PGe=M|Sem=S
A	SubPOS=U|Gen=I|Num=P|Cas=6|PGe=M|Sem=Y
A	SubPOS=U|Gen=I|Num=P|Cas=6|PGe=M|Sem=m
A	SubPOS=U|Gen=I|Num=P|Cas=6|PGe=M|Var=6
A	SubPOS=U|Gen=I|Num=P|Cas=7|PGe=F
A	SubPOS=U|Gen=I|Num=P|Cas=7|PGe=F|Var=6
A	SubPOS=U|Gen=I|Num=P|Cas=7|PGe=F|Var=7
A	SubPOS=U|Gen=I|Num=P|Cas=7|PGe=M
A	SubPOS=U|Gen=I|Num=P|Cas=7|PGe=M|Sem=S
A	SubPOS=U|Gen=I|Num=P|Cas=7|PGe=M|Sem=m
A	SubPOS=U|Gen=I|Num=P|Cas=7|PGe=M|Var=6
A	SubPOS=U|Gen=I|Num=P|Cas=7|PGe=M|Var=7
A	SubPOS=U|Gen=I|Num=S|Cas=1|PGe=F
A	SubPOS=U|Gen=I|Num=S|Cas=1|PGe=F|Sem=Y
A	SubPOS=U|Gen=I|Num=S|Cas=1|PGe=F|Sem=m
A	SubPOS=U|Gen=I|Num=S|Cas=1|PGe=F|Var=6
A	SubPOS=U|Gen=I|Num=S|Cas=1|PGe=M
A	SubPOS=U|Gen=I|Num=S|Cas=1|PGe=M|Sem=E
A	SubPOS=U|Gen=I|Num=S|Cas=1|PGe=M|Sem=G
A	SubPOS=U|Gen=I|Num=S|Cas=1|PGe=M|Sem=S
A	SubPOS=U|Gen=I|Num=S|Cas=1|PGe=M|Sem=Y
A	SubPOS=U|Gen=I|Num=S|Cas=1|PGe=M|Sem=m
A	SubPOS=U|Gen=I|Num=S|Cas=1|PGe=M|Var=6
A	SubPOS=U|Gen=I|Num=S|Cas=2|PGe=F
A	SubPOS=U|Gen=I|Num=S|Cas=2|PGe=F|Sem=Y
A	SubPOS=U|Gen=I|Num=S|Cas=2|PGe=F|Sem=m
A	SubPOS=U|Gen=I|Num=S|Cas=2|PGe=F|Var=6
A	SubPOS=U|Gen=I|Num=S|Cas=2|PGe=M
A	SubPOS=U|Gen=I|Num=S|Cas=2|PGe=M|Sem=G
A	SubPOS=U|Gen=I|Num=S|Cas=2|PGe=M|Sem=S
A	SubPOS=U|Gen=I|Num=S|Cas=2|PGe=M|Sem=Y
A	SubPOS=U|Gen=I|Num=S|Cas=2|PGe=M|Sem=m
A	SubPOS=U|Gen=I|Num=S|Cas=2|PGe=M|Var=6
A	SubPOS=U|Gen=I|Num=S|Cas=3|PGe=F
A	SubPOS=U|Gen=I|Num=S|Cas=3|PGe=F|Var=6
A	SubPOS=U|Gen=I|Num=S|Cas=3|PGe=M
A	SubPOS=U|Gen=I|Num=S|Cas=3|PGe=M|Sem=S
A	SubPOS=U|Gen=I|Num=S|Cas=3|PGe=M|Sem=m
A	SubPOS=U|Gen=I|Num=S|Cas=3|PGe=M|Var=6
A	SubPOS=U|Gen=I|Num=S|Cas=4|PGe=F
A	SubPOS=U|Gen=I|Num=S|Cas=4|PGe=F|Var=6
A	SubPOS=U|Gen=I|Num=S|Cas=4|PGe=M
A	SubPOS=U|Gen=I|Num=S|Cas=4|PGe=M|Sem=E
A	SubPOS=U|Gen=I|Num=S|Cas=4|PGe=M|Sem=G
A	SubPOS=U|Gen=I|Num=S|Cas=4|PGe=M|Sem=S
A	SubPOS=U|Gen=I|Num=S|Cas=4|PGe=M|Sem=Y
A	SubPOS=U|Gen=I|Num=S|Cas=4|PGe=M|Sem=m
A	SubPOS=U|Gen=I|Num=S|Cas=4|PGe=M|Var=6
A	SubPOS=U|Gen=I|Num=S|Cas=5|PGe=F
A	SubPOS=U|Gen=I|Num=S|Cas=5|PGe=F|Var=6
A	SubPOS=U|Gen=I|Num=S|Cas=5|PGe=M
A	SubPOS=U|Gen=I|Num=S|Cas=5|PGe=M|Var=6
A	SubPOS=U|Gen=I|Num=S|Cas=6|PGe=F
A	SubPOS=U|Gen=I|Num=S|Cas=6|PGe=F|Var=1
A	SubPOS=U|Gen=I|Num=S|Cas=6|PGe=F|Var=6
A	SubPOS=U|Gen=I|Num=S|Cas=6|PGe=M
A	SubPOS=U|Gen=I|Num=S|Cas=6|PGe=M|Sem=G
A	SubPOS=U|Gen=I|Num=S|Cas=6|PGe=M|Sem=S
A	SubPOS=U|Gen=I|Num=S|Cas=6|PGe=M|Sem=Y
A	SubPOS=U|Gen=I|Num=S|Cas=6|PGe=M|Sem=m
A	SubPOS=U|Gen=I|Num=S|Cas=6|PGe=M|Var=1
A	SubPOS=U|Gen=I|Num=S|Cas=6|PGe=M|Var=6
A	SubPOS=U|Gen=I|Num=S|Cas=7|PGe=F
A	SubPOS=U|Gen=I|Num=S|Cas=7|PGe=F|Var=6
A	SubPOS=U|Gen=I|Num=S|Cas=7|PGe=M
A	SubPOS=U|Gen=I|Num=S|Cas=7|PGe=M|Sem=S
A	SubPOS=U|Gen=I|Num=S|Cas=7|PGe=M|Sem=Y
A	SubPOS=U|Gen=I|Num=S|Cas=7|PGe=M|Sem=m
A	SubPOS=U|Gen=I|Num=S|Cas=7|PGe=M|Var=6
A	SubPOS=U|Gen=M|Num=P|Cas=1|PGe=F
A	SubPOS=U|Gen=M|Num=P|Cas=1|PGe=F|Sem=Y
A	SubPOS=U|Gen=M|Num=P|Cas=1|PGe=F|Sem=m
A	SubPOS=U|Gen=M|Num=P|Cas=1|PGe=F|Var=6
A	SubPOS=U|Gen=M|Num=P|Cas=1|PGe=M
A	SubPOS=U|Gen=M|Num=P|Cas=1|PGe=M|Sem=S
A	SubPOS=U|Gen=M|Num=P|Cas=1|PGe=M|Sem=m
A	SubPOS=U|Gen=M|Num=P|Cas=1|PGe=M|Var=6
A	SubPOS=U|Gen=M|Num=P|Cas=2|PGe=F
A	SubPOS=U|Gen=M|Num=P|Cas=2|PGe=F|Var=6
A	SubPOS=U|Gen=M|Num=P|Cas=2|PGe=M
A	SubPOS=U|Gen=M|Num=P|Cas=2|PGe=M|Sem=G
A	SubPOS=U|Gen=M|Num=P|Cas=2|PGe=M|Sem=S
A	SubPOS=U|Gen=M|Num=P|Cas=2|PGe=M|Sem=Y
A	SubPOS=U|Gen=M|Num=P|Cas=2|PGe=M|Sem=m
A	SubPOS=U|Gen=M|Num=P|Cas=2|PGe=M|Var=6
A	SubPOS=U|Gen=M|Num=P|Cas=3|PGe=F
A	SubPOS=U|Gen=M|Num=P|Cas=3|PGe=F|Var=6
A	SubPOS=U|Gen=M|Num=P|Cas=3|PGe=M
A	SubPOS=U|Gen=M|Num=P|Cas=3|PGe=M|Sem=S
A	SubPOS=U|Gen=M|Num=P|Cas=3|PGe=M|Sem=m
A	SubPOS=U|Gen=M|Num=P|Cas=3|PGe=M|Var=6
A	SubPOS=U|Gen=M|Num=P|Cas=4|PGe=F
A	SubPOS=U|Gen=M|Num=P|Cas=4|PGe=F|Var=6
A	SubPOS=U|Gen=M|Num=P|Cas=4|PGe=M
A	SubPOS=U|Gen=M|Num=P|Cas=4|PGe=M|Sem=S
A	SubPOS=U|Gen=M|Num=P|Cas=4|PGe=M|Sem=m
A	SubPOS=U|Gen=M|Num=P|Cas=4|PGe=M|Var=6
A	SubPOS=U|Gen=M|Num=P|Cas=5|PGe=F
A	SubPOS=U|Gen=M|Num=P|Cas=5|PGe=F|Var=6
A	SubPOS=U|Gen=M|Num=P|Cas=5|PGe=M
A	SubPOS=U|Gen=M|Num=P|Cas=5|PGe=M|Var=6
A	SubPOS=U|Gen=M|Num=P|Cas=6|PGe=F
A	SubPOS=U|Gen=M|Num=P|Cas=6|PGe=F|Var=6
A	SubPOS=U|Gen=M|Num=P|Cas=6|PGe=M
A	SubPOS=U|Gen=M|Num=P|Cas=6|PGe=M|Sem=S
A	SubPOS=U|Gen=M|Num=P|Cas=6|PGe=M|Sem=m
A	SubPOS=U|Gen=M|Num=P|Cas=6|PGe=M|Var=6
A	SubPOS=U|Gen=M|Num=P|Cas=7|PGe=F
A	SubPOS=U|Gen=M|Num=P|Cas=7|PGe=F|Var=6
A	SubPOS=U|Gen=M|Num=P|Cas=7|PGe=F|Var=7
A	SubPOS=U|Gen=M|Num=P|Cas=7|PGe=M
A	SubPOS=U|Gen=M|Num=P|Cas=7|PGe=M|Sem=S
A	SubPOS=U|Gen=M|Num=P|Cas=7|PGe=M|Sem=m
A	SubPOS=U|Gen=M|Num=P|Cas=7|PGe=M|Var=6
A	SubPOS=U|Gen=M|Num=P|Cas=7|PGe=M|Var=7
A	SubPOS=U|Gen=M|Num=S|Cas=1|PGe=F
A	SubPOS=U|Gen=M|Num=S|Cas=1|PGe=F|Var=6
A	SubPOS=U|Gen=M|Num=S|Cas=1|PGe=M
A	SubPOS=U|Gen=M|Num=S|Cas=1|PGe=M|Sem=S
A	SubPOS=U|Gen=M|Num=S|Cas=1|PGe=M|Sem=Y
A	SubPOS=U|Gen=M|Num=S|Cas=1|PGe=M|Sem=m
A	SubPOS=U|Gen=M|Num=S|Cas=1|PGe=M|Var=6
A	SubPOS=U|Gen=M|Num=S|Cas=2|PGe=F
A	SubPOS=U|Gen=M|Num=S|Cas=2|PGe=F|Var=6
A	SubPOS=U|Gen=M|Num=S|Cas=2|PGe=M
A	SubPOS=U|Gen=M|Num=S|Cas=2|PGe=M|Sem=G
A	SubPOS=U|Gen=M|Num=S|Cas=2|PGe=M|Sem=S
A	SubPOS=U|Gen=M|Num=S|Cas=2|PGe=M|Sem=Y
A	SubPOS=U|Gen=M|Num=S|Cas=2|PGe=M|Sem=m
A	SubPOS=U|Gen=M|Num=S|Cas=2|PGe=M|Var=6
A	SubPOS=U|Gen=M|Num=S|Cas=3|PGe=F
A	SubPOS=U|Gen=M|Num=S|Cas=3|PGe=F|Var=6
A	SubPOS=U|Gen=M|Num=S|Cas=3|PGe=M
A	SubPOS=U|Gen=M|Num=S|Cas=3|PGe=M|Sem=S
A	SubPOS=U|Gen=M|Num=S|Cas=3|PGe=M|Sem=m
A	SubPOS=U|Gen=M|Num=S|Cas=3|PGe=M|Var=6
A	SubPOS=U|Gen=M|Num=S|Cas=4|PGe=F
A	SubPOS=U|Gen=M|Num=S|Cas=4|PGe=F|Sem=Y
A	SubPOS=U|Gen=M|Num=S|Cas=4|PGe=F|Sem=m
A	SubPOS=U|Gen=M|Num=S|Cas=4|PGe=F|Var=6
A	SubPOS=U|Gen=M|Num=S|Cas=4|PGe=M
A	SubPOS=U|Gen=M|Num=S|Cas=4|PGe=M|Sem=S
A	SubPOS=U|Gen=M|Num=S|Cas=4|PGe=M|Sem=Y
A	SubPOS=U|Gen=M|Num=S|Cas=4|PGe=M|Sem=m
A	SubPOS=U|Gen=M|Num=S|Cas=4|PGe=M|Var=6
A	SubPOS=U|Gen=M|Num=S|Cas=5|PGe=F
A	SubPOS=U|Gen=M|Num=S|Cas=5|PGe=F|Var=6
A	SubPOS=U|Gen=M|Num=S|Cas=5|PGe=M
A	SubPOS=U|Gen=M|Num=S|Cas=5|PGe=M|Var=6
A	SubPOS=U|Gen=M|Num=S|Cas=6|PGe=F
A	SubPOS=U|Gen=M|Num=S|Cas=6|PGe=F|Var=1
A	SubPOS=U|Gen=M|Num=S|Cas=6|PGe=F|Var=6
A	SubPOS=U|Gen=M|Num=S|Cas=6|PGe=M
A	SubPOS=U|Gen=M|Num=S|Cas=6|PGe=M|Sem=S
A	SubPOS=U|Gen=M|Num=S|Cas=6|PGe=M|Sem=m
A	SubPOS=U|Gen=M|Num=S|Cas=6|PGe=M|Var=1
A	SubPOS=U|Gen=M|Num=S|Cas=6|PGe=M|Var=6
A	SubPOS=U|Gen=M|Num=S|Cas=7|PGe=F
A	SubPOS=U|Gen=M|Num=S|Cas=7|PGe=F|Var=6
A	SubPOS=U|Gen=M|Num=S|Cas=7|PGe=M
A	SubPOS=U|Gen=M|Num=S|Cas=7|PGe=M|Sem=K
A	SubPOS=U|Gen=M|Num=S|Cas=7|PGe=M|Sem=S
A	SubPOS=U|Gen=M|Num=S|Cas=7|PGe=M|Sem=m
A	SubPOS=U|Gen=M|Num=S|Cas=7|PGe=M|Var=6
A	SubPOS=U|Gen=N|Num=D|Cas=7|PGe=F
A	SubPOS=U|Gen=N|Num=D|Cas=7|PGe=F|Var=6
A	SubPOS=U|Gen=N|Num=D|Cas=7|PGe=M
A	SubPOS=U|Gen=N|Num=D|Cas=7|PGe=M|Var=6
A	SubPOS=U|Gen=N|Num=P|Cas=1|PGe=F
A	SubPOS=U|Gen=N|Num=P|Cas=1|PGe=F|Var=6
A	SubPOS=U|Gen=N|Num=P|Cas=1|PGe=M
A	SubPOS=U|Gen=N|Num=P|Cas=1|PGe=M|Sem=S
A	SubPOS=U|Gen=N|Num=P|Cas=1|PGe=M|Sem=m
A	SubPOS=U|Gen=N|Num=P|Cas=1|PGe=M|Var=6
A	SubPOS=U|Gen=N|Num=P|Cas=2|PGe=F
A	SubPOS=U|Gen=N|Num=P|Cas=2|PGe=F|Var=6
A	SubPOS=U|Gen=N|Num=P|Cas=2|PGe=M
A	SubPOS=U|Gen=N|Num=P|Cas=2|PGe=M|Sem=S
A	SubPOS=U|Gen=N|Num=P|Cas=2|PGe=M|Sem=m
A	SubPOS=U|Gen=N|Num=P|Cas=2|PGe=M|Var=6
A	SubPOS=U|Gen=N|Num=P|Cas=3|PGe=F
A	SubPOS=U|Gen=N|Num=P|Cas=3|PGe=F|Var=6
A	SubPOS=U|Gen=N|Num=P|Cas=3|PGe=M
A	SubPOS=U|Gen=N|Num=P|Cas=3|PGe=M|Sem=S
A	SubPOS=U|Gen=N|Num=P|Cas=3|PGe=M|Sem=m
A	SubPOS=U|Gen=N|Num=P|Cas=3|PGe=M|Var=6
A	SubPOS=U|Gen=N|Num=P|Cas=4|PGe=F
A	SubPOS=U|Gen=N|Num=P|Cas=4|PGe=F|Var=6
A	SubPOS=U|Gen=N|Num=P|Cas=4|PGe=M
A	SubPOS=U|Gen=N|Num=P|Cas=4|PGe=M|Sem=S
A	SubPOS=U|Gen=N|Num=P|Cas=4|PGe=M|Sem=Y
A	SubPOS=U|Gen=N|Num=P|Cas=4|PGe=M|Sem=m
A	SubPOS=U|Gen=N|Num=P|Cas=4|PGe=M|Var=6
A	SubPOS=U|Gen=N|Num=P|Cas=5|PGe=F
A	SubPOS=U|Gen=N|Num=P|Cas=5|PGe=F|Var=6
A	SubPOS=U|Gen=N|Num=P|Cas=5|PGe=M
A	SubPOS=U|Gen=N|Num=P|Cas=5|PGe=M|Var=6
A	SubPOS=U|Gen=N|Num=P|Cas=6|PGe=F
A	SubPOS=U|Gen=N|Num=P|Cas=6|PGe=F|Var=6
A	SubPOS=U|Gen=N|Num=P|Cas=6|PGe=M
A	SubPOS=U|Gen=N|Num=P|Cas=6|PGe=M|Sem=S
A	SubPOS=U|Gen=N|Num=P|Cas=6|PGe=M|Sem=m
A	SubPOS=U|Gen=N|Num=P|Cas=6|PGe=M|Var=6
A	SubPOS=U|Gen=N|Num=P|Cas=7|PGe=F
A	SubPOS=U|Gen=N|Num=P|Cas=7|PGe=F|Var=6
A	SubPOS=U|Gen=N|Num=P|Cas=7|PGe=F|Var=7
A	SubPOS=U|Gen=N|Num=P|Cas=7|PGe=M
A	SubPOS=U|Gen=N|Num=P|Cas=7|PGe=M|Sem=S
A	SubPOS=U|Gen=N|Num=P|Cas=7|PGe=M|Sem=m
A	SubPOS=U|Gen=N|Num=P|Cas=7|PGe=M|Var=6
A	SubPOS=U|Gen=N|Num=P|Cas=7|PGe=M|Var=7
A	SubPOS=U|Gen=N|Num=S|Cas=1|PGe=F
A	SubPOS=U|Gen=N|Num=S|Cas=1|PGe=F|Var=6
A	SubPOS=U|Gen=N|Num=S|Cas=1|PGe=M
A	SubPOS=U|Gen=N|Num=S|Cas=1|PGe=M|Sem=E
A	SubPOS=U|Gen=N|Num=S|Cas=1|PGe=M|Sem=G
A	SubPOS=U|Gen=N|Num=S|Cas=1|PGe=M|Sem=S
A	SubPOS=U|Gen=N|Num=S|Cas=1|PGe=M|Sem=Y
A	SubPOS=U|Gen=N|Num=S|Cas=1|PGe=M|Sem=m
A	SubPOS=U|Gen=N|Num=S|Cas=1|PGe=M|Var=6
A	SubPOS=U|Gen=N|Num=S|Cas=2|PGe=F
A	SubPOS=U|Gen=N|Num=S|Cas=2|PGe=F|Sem=Y
A	SubPOS=U|Gen=N|Num=S|Cas=2|PGe=F|Sem=m
A	SubPOS=U|Gen=N|Num=S|Cas=2|PGe=F|Var=6
A	SubPOS=U|Gen=N|Num=S|Cas=2|PGe=M
A	SubPOS=U|Gen=N|Num=S|Cas=2|PGe=M|Sem=S
A	SubPOS=U|Gen=N|Num=S|Cas=2|PGe=M|Sem=Y
A	SubPOS=U|Gen=N|Num=S|Cas=2|PGe=M|Sem=m
A	SubPOS=U|Gen=N|Num=S|Cas=2|PGe=M|Var=6
A	SubPOS=U|Gen=N|Num=S|Cas=3|PGe=F
A	SubPOS=U|Gen=N|Num=S|Cas=3|PGe=F|Var=6
A	SubPOS=U|Gen=N|Num=S|Cas=3|PGe=M
A	SubPOS=U|Gen=N|Num=S|Cas=3|PGe=M|Sem=S
A	SubPOS=U|Gen=N|Num=S|Cas=3|PGe=M|Sem=m
A	SubPOS=U|Gen=N|Num=S|Cas=3|PGe=M|Var=6
A	SubPOS=U|Gen=N|Num=S|Cas=4|PGe=F
A	SubPOS=U|Gen=N|Num=S|Cas=4|PGe=F|Sem=Y
A	SubPOS=U|Gen=N|Num=S|Cas=4|PGe=F|Sem=m
A	SubPOS=U|Gen=N|Num=S|Cas=4|PGe=F|Var=6
A	SubPOS=U|Gen=N|Num=S|Cas=4|PGe=M
A	SubPOS=U|Gen=N|Num=S|Cas=4|PGe=M|Sem=E
A	SubPOS=U|Gen=N|Num=S|Cas=4|PGe=M|Sem=S
A	SubPOS=U|Gen=N|Num=S|Cas=4|PGe=M|Sem=Y
A	SubPOS=U|Gen=N|Num=S|Cas=4|PGe=M|Sem=m
A	SubPOS=U|Gen=N|Num=S|Cas=4|PGe=M|Var=6
A	SubPOS=U|Gen=N|Num=S|Cas=5|PGe=F
A	SubPOS=U|Gen=N|Num=S|Cas=5|PGe=F|Var=6
A	SubPOS=U|Gen=N|Num=S|Cas=5|PGe=M
A	SubPOS=U|Gen=N|Num=S|Cas=5|PGe=M|Var=6
A	SubPOS=U|Gen=N|Num=S|Cas=6|PGe=F
A	SubPOS=U|Gen=N|Num=S|Cas=6|PGe=F|Var=1
A	SubPOS=U|Gen=N|Num=S|Cas=6|PGe=F|Var=6
A	SubPOS=U|Gen=N|Num=S|Cas=6|PGe=M
A	SubPOS=U|Gen=N|Num=S|Cas=6|PGe=M|Sem=G
A	SubPOS=U|Gen=N|Num=S|Cas=6|PGe=M|Sem=S
A	SubPOS=U|Gen=N|Num=S|Cas=6|PGe=M|Sem=Y
A	SubPOS=U|Gen=N|Num=S|Cas=6|PGe=M|Sem=m
A	SubPOS=U|Gen=N|Num=S|Cas=6|PGe=M|Var=1
A	SubPOS=U|Gen=N|Num=S|Cas=6|PGe=M|Var=6
A	SubPOS=U|Gen=N|Num=S|Cas=7|PGe=F
A	SubPOS=U|Gen=N|Num=S|Cas=7|PGe=F|Var=6
A	SubPOS=U|Gen=N|Num=S|Cas=7|PGe=M
A	SubPOS=U|Gen=N|Num=S|Cas=7|PGe=M|Sem=G
A	SubPOS=U|Gen=N|Num=S|Cas=7|PGe=M|Sem=S
A	SubPOS=U|Gen=N|Num=S|Cas=7|PGe=M|Sem=Y
A	SubPOS=U|Gen=N|Num=S|Cas=7|PGe=M|Sem=m
A	SubPOS=U|Gen=N|Num=S|Cas=7|PGe=M|Var=6
A	SubPOS=U|Gen=X|Num=X|Cas=X|PGe=F|Var=8
A	SubPOS=U|Gen=X|Num=X|Cas=X|PGe=M
A	SubPOS=U|Gen=X|Num=X|Cas=X|PGe=M|Sem=S
A	SubPOS=U|Gen=X|Num=X|Cas=X|PGe=M|Sem=m
A	SubPOS=U|Gen=X|Num=X|Cas=X|PGe=M|Var=6
A	SubPOS=U|Gen=X|Num=X|Cas=X|PGe=M|Var=6|Sem=S
A	SubPOS=U|Gen=X|Num=X|Cas=X|PGe=M|Var=6|Sem=m
A	SubPOS=U|Gen=X|Num=X|Cas=X|PGe=M|Var=8
C	SubPOS=3
C	SubPOS==
C	SubPOS=?|Cas=1
C	SubPOS=?|Cas=2
C	SubPOS=?|Cas=3
C	SubPOS=?|Cas=4
C	SubPOS=?|Cas=6
C	SubPOS=?|Cas=7
C	SubPOS=a|Cas=1
C	SubPOS=a|Cas=2
C	SubPOS=a|Cas=2|Var=1
C	SubPOS=a|Cas=3
C	SubPOS=a|Cas=4
C	SubPOS=a|Cas=5
C	SubPOS=a|Cas=6
C	SubPOS=a|Cas=7
C	SubPOS=a|Cas=X
C	SubPOS=a|Cas=X|Var=8
C	SubPOS=d|Gen=F|Num=D|Cas=7
C	SubPOS=d|Gen=F|Num=D|Cas=7|Var=6
C	SubPOS=d|Gen=F|Num=P|Cas=1
C	SubPOS=d|Gen=F|Num=P|Cas=1|Var=6
C	SubPOS=d|Gen=F|Num=P|Cas=2
C	SubPOS=d|Gen=F|Num=P|Cas=2|Var=6
C	SubPOS=d|Gen=F|Num=P|Cas=3
C	SubPOS=d|Gen=F|Num=P|Cas=3|Var=6
C	SubPOS=d|Gen=F|Num=P|Cas=4
C	SubPOS=d|Gen=F|Num=P|Cas=4|Var=6
C	SubPOS=d|Gen=F|Num=P|Cas=5
C	SubPOS=d|Gen=F|Num=P|Cas=5|Var=6
C	SubPOS=d|Gen=F|Num=P|Cas=6
C	SubPOS=d|Gen=F|Num=P|Cas=6|Var=6
C	SubPOS=d|Gen=F|Num=P|Cas=7
C	SubPOS=d|Gen=F|Num=P|Cas=7|Var=6
C	SubPOS=d|Gen=F|Num=P|Cas=7|Var=7
C	SubPOS=d|Gen=F|Num=S|Cas=1
C	SubPOS=d|Gen=F|Num=S|Cas=2
C	SubPOS=d|Gen=F|Num=S|Cas=2|Var=6
C	SubPOS=d|Gen=F|Num=S|Cas=3
C	SubPOS=d|Gen=F|Num=S|Cas=3|Var=6
C	SubPOS=d|Gen=F|Num=S|Cas=4
C	SubPOS=d|Gen=F|Num=S|Cas=4|Var=2
C	SubPOS=d|Gen=F|Num=S|Cas=5
C	SubPOS=d|Gen=F|Num=S|Cas=6
C	SubPOS=d|Gen=F|Num=S|Cas=6|Var=6
C	SubPOS=d|Gen=F|Num=S|Cas=7
C	SubPOS=d|Gen=I|Num=P|Cas=1
C	SubPOS=d|Gen=I|Num=P|Cas=1|Var=6
C	SubPOS=d|Gen=I|Num=P|Cas=2
C	SubPOS=d|Gen=I|Num=P|Cas=2|Var=6
C	SubPOS=d|Gen=I|Num=P|Cas=3
C	SubPOS=d|Gen=I|Num=P|Cas=3|Var=6
C	SubPOS=d|Gen=I|Num=P|Cas=4
C	SubPOS=d|Gen=I|Num=P|Cas=4|Var=6
C	SubPOS=d|Gen=I|Num=P|Cas=5
C	SubPOS=d|Gen=I|Num=P|Cas=5|Var=6
C	SubPOS=d|Gen=I|Num=P|Cas=6
C	SubPOS=d|Gen=I|Num=P|Cas=6|Var=6
C	SubPOS=d|Gen=I|Num=P|Cas=7
C	SubPOS=d|Gen=I|Num=P|Cas=7|Var=6
C	SubPOS=d|Gen=I|Num=P|Cas=7|Var=7
C	SubPOS=d|Gen=I|Num=S|Cas=1
C	SubPOS=d|Gen=I|Num=S|Cas=1|Var=6
C	SubPOS=d|Gen=I|Num=S|Cas=2
C	SubPOS=d|Gen=I|Num=S|Cas=2|Var=6
C	SubPOS=d|Gen=I|Num=S|Cas=3
C	SubPOS=d|Gen=I|Num=S|Cas=3|Var=6
C	SubPOS=d|Gen=I|Num=S|Cas=4
C	SubPOS=d|Gen=I|Num=S|Cas=4|Var=6
C	SubPOS=d|Gen=I|Num=S|Cas=5
C	SubPOS=d|Gen=I|Num=S|Cas=5|Var=6
C	SubPOS=d|Gen=I|Num=S|Cas=6
C	SubPOS=d|Gen=I|Num=S|Cas=6|Var=6
C	SubPOS=d|Gen=I|Num=S|Cas=7
C	SubPOS=d|Gen=I|Num=S|Cas=7|Var=6
C	SubPOS=d|Gen=M|Num=P|Cas=1
C	SubPOS=d|Gen=M|Num=P|Cas=1|Var=6
C	SubPOS=d|Gen=M|Num=P|Cas=2
C	SubPOS=d|Gen=M|Num=P|Cas=2|Var=6
C	SubPOS=d|Gen=M|Num=P|Cas=3
C	SubPOS=d|Gen=M|Num=P|Cas=3|Var=6
C	SubPOS=d|Gen=M|Num=P|Cas=4
C	SubPOS=d|Gen=M|Num=P|Cas=4|Var=6
C	SubPOS=d|Gen=M|Num=P|Cas=5
C	SubPOS=d|Gen=M|Num=P|Cas=5|Var=6
C	SubPOS=d|Gen=M|Num=P|Cas=6
C	SubPOS=d|Gen=M|Num=P|Cas=6|Var=6
C	SubPOS=d|Gen=M|Num=P|Cas=7
C	SubPOS=d|Gen=M|Num=P|Cas=7|Var=6
C	SubPOS=d|Gen=M|Num=P|Cas=7|Var=7
C	SubPOS=d|Gen=M|Num=S|Cas=1
C	SubPOS=d|Gen=M|Num=S|Cas=1|Var=6
C	SubPOS=d|Gen=M|Num=S|Cas=2
C	SubPOS=d|Gen=M|Num=S|Cas=2|Var=6
C	SubPOS=d|Gen=M|Num=S|Cas=3
C	SubPOS=d|Gen=M|Num=S|Cas=3|Var=6
C	SubPOS=d|Gen=M|Num=S|Cas=4
C	SubPOS=d|Gen=M|Num=S|Cas=4|Var=6
C	SubPOS=d|Gen=M|Num=S|Cas=5
C	SubPOS=d|Gen=M|Num=S|Cas=5|Var=6
C	SubPOS=d|Gen=M|Num=S|Cas=6
C	SubPOS=d|Gen=M|Num=S|Cas=6|Var=6
C	SubPOS=d|Gen=M|Num=S|Cas=7
C	SubPOS=d|Gen=M|Num=S|Cas=7|Var=6
C	SubPOS=d|Gen=N|Num=D|Cas=7
C	SubPOS=d|Gen=N|Num=P|Cas=1
C	SubPOS=d|Gen=N|Num=P|Cas=1|Var=6
C	SubPOS=d|Gen=N|Num=P|Cas=2
C	SubPOS=d|Gen=N|Num=P|Cas=2|Var=6
C	SubPOS=d|Gen=N|Num=P|Cas=3
C	SubPOS=d|Gen=N|Num=P|Cas=3|Var=6
C	SubPOS=d|Gen=N|Num=P|Cas=4
C	SubPOS=d|Gen=N|Num=P|Cas=4|Var=6
C	SubPOS=d|Gen=N|Num=P|Cas=5
C	SubPOS=d|Gen=N|Num=P|Cas=5|Var=6
C	SubPOS=d|Gen=N|Num=P|Cas=6
C	SubPOS=d|Gen=N|Num=P|Cas=6|Var=6
C	SubPOS=d|Gen=N|Num=P|Cas=7
C	SubPOS=d|Gen=N|Num=P|Cas=7|Var=6
C	SubPOS=d|Gen=N|Num=P|Cas=7|Var=7
C	SubPOS=d|Gen=N|Num=S|Cas=1
C	SubPOS=d|Gen=N|Num=S|Cas=1|Var=1
C	SubPOS=d|Gen=N|Num=S|Cas=1|Var=6
C	SubPOS=d|Gen=N|Num=S|Cas=2
C	SubPOS=d|Gen=N|Num=S|Cas=2|Var=6
C	SubPOS=d|Gen=N|Num=S|Cas=3
C	SubPOS=d|Gen=N|Num=S|Cas=3|Var=6
C	SubPOS=d|Gen=N|Num=S|Cas=4
C	SubPOS=d|Gen=N|Num=S|Cas=4|Var=1
C	SubPOS=d|Gen=N|Num=S|Cas=4|Var=6
C	SubPOS=d|Gen=N|Num=S|Cas=5
C	SubPOS=d|Gen=N|Num=S|Cas=5|Var=6
C	SubPOS=d|Gen=N|Num=S|Cas=6
C	SubPOS=d|Gen=N|Num=S|Cas=6|Var=6
C	SubPOS=d|Gen=N|Num=S|Cas=7
C	SubPOS=d|Gen=N|Num=S|Cas=7|Var=6
C	SubPOS=d|Gen=X|Num=P|Cas=1
C	SubPOS=d|Gen=X|Num=P|Cas=1|Var=1
C	SubPOS=d|Gen=X|Num=P|Cas=2
C	SubPOS=d|Gen=X|Num=P|Cas=3
C	SubPOS=d|Gen=X|Num=P|Cas=4
C	SubPOS=d|Gen=X|Num=P|Cas=4|Var=1
C	SubPOS=d|Gen=X|Num=P|Cas=5
C	SubPOS=d|Gen=X|Num=P|Cas=5|Var=1
C	SubPOS=d|Gen=X|Num=P|Cas=6
C	SubPOS=d|Gen=X|Num=P|Cas=7
C	SubPOS=d|Gen=X|Num=S|Cas=1
C	SubPOS=d|Gen=X|Num=S|Cas=5
C	SubPOS=d|Gen=Y|Num=S|Cas=2
C	SubPOS=d|Gen=Y|Num=S|Cas=3
C	SubPOS=d|Gen=Y|Num=S|Cas=6
C	SubPOS=d|Gen=Y|Num=S|Cas=7
C	SubPOS=h|Gen=F|Num=D|Cas=7
C	SubPOS=h|Gen=F|Num=P|Cas=1
C	SubPOS=h|Gen=F|Num=P|Cas=4
C	SubPOS=h|Gen=F|Num=P|Cas=5
C	SubPOS=h|Gen=I|Num=P|Cas=1
C	SubPOS=h|Gen=I|Num=P|Cas=5
C	SubPOS=h|Gen=M|Num=P|Cas=1
C	SubPOS=h|Gen=M|Num=P|Cas=5
C	SubPOS=h|Gen=N|Num=P|Cas=1
C	SubPOS=h|Gen=N|Num=P|Cas=4
C	SubPOS=h|Gen=N|Num=P|Cas=5
C	SubPOS=h|Gen=X|Num=P|Cas=2
C	SubPOS=h|Gen=X|Num=P|Cas=3
C	SubPOS=h|Gen=X|Num=P|Cas=6
C	SubPOS=h|Gen=X|Num=P|Cas=7
C	SubPOS=h|Gen=Y|Num=P|Cas=4
C	SubPOS=j|Num=S|Cas=1
C	SubPOS=j|Num=S|Cas=2
C	SubPOS=j|Num=S|Cas=2|Var=1
C	SubPOS=j|Num=S|Cas=3
C	SubPOS=j|Num=S|Cas=3|Var=1
C	SubPOS=j|Num=S|Cas=4
C	SubPOS=j|Num=S|Cas=5
C	SubPOS=j|Num=S|Cas=6
C	SubPOS=j|Num=S|Cas=6|Var=1
C	SubPOS=j|Num=S|Cas=7
C	SubPOS=j|Num=S|Cas=7|Var=1
C	SubPOS=k|Num=P|Cas=1
C	SubPOS=k|Num=P|Cas=2
C	SubPOS=k|Num=P|Cas=3
C	SubPOS=k|Num=P|Cas=4
C	SubPOS=k|Num=P|Cas=5
C	SubPOS=k|Num=P|Cas=6
C	SubPOS=k|Num=P|Cas=7
C	SubPOS=l|Gen=F|Num=D|Cas=7
C	SubPOS=l|Gen=F|Num=D|Cas=7|Var=6
C	SubPOS=l|Gen=F|Num=D|Cas=7|Var=9
C	SubPOS=l|Gen=F|Num=S|Cas=1
C	SubPOS=l|Gen=F|Num=S|Cas=2
C	SubPOS=l|Gen=F|Num=S|Cas=3
C	SubPOS=l|Gen=F|Num=S|Cas=4
C	SubPOS=l|Gen=F|Num=S|Cas=5
C	SubPOS=l|Gen=F|Num=S|Cas=6
C	SubPOS=l|Gen=F|Num=S|Cas=7
C	SubPOS=l|Gen=H|Num=P|Cas=1
C	SubPOS=l|Gen=H|Num=P|Cas=4
C	SubPOS=l|Gen=H|Num=P|Cas=5
C	SubPOS=l|Gen=I|Num=S|Cas=4
C	SubPOS=l|Gen=M|Num=S|Cas=4
C	SubPOS=l|Gen=N|Num=S|Cas=1
C	SubPOS=l|Gen=N|Num=S|Cas=4
C	SubPOS=l|Gen=N|Num=S|Cas=5
C	SubPOS=l|Gen=X|Num=P|Cas=1
C	SubPOS=l|Gen=X|Num=P|Cas=1|Var=6
C	SubPOS=l|Gen=X|Num=P|Cas=2
C	SubPOS=l|Gen=X|Num=P|Cas=2|Var=1
C	SubPOS=l|Gen=X|Num=P|Cas=2|Var=6
C	SubPOS=l|Gen=X|Num=P|Cas=3
C	SubPOS=l|Gen=X|Num=P|Cas=3|Var=1
C	SubPOS=l|Gen=X|Num=P|Cas=3|Var=2
C	SubPOS=l|Gen=X|Num=P|Cas=3|Var=6
C	SubPOS=l|Gen=X|Num=P|Cas=3|Var=9
C	SubPOS=l|Gen=X|Num=P|Cas=4
C	SubPOS=l|Gen=X|Num=P|Cas=4|Var=6
C	SubPOS=l|Gen=X|Num=P|Cas=5
C	SubPOS=l|Gen=X|Num=P|Cas=5|Var=6
C	SubPOS=l|Gen=X|Num=P|Cas=6
C	SubPOS=l|Gen=X|Num=P|Cas=6|Var=1
C	SubPOS=l|Gen=X|Num=P|Cas=6|Var=2
C	SubPOS=l|Gen=X|Num=P|Cas=6|Var=6
C	SubPOS=l|Gen=X|Num=P|Cas=7
C	SubPOS=l|Gen=X|Num=P|Cas=7|Var=1
C	SubPOS=l|Gen=X|Num=P|Cas=7|Var=2
C	SubPOS=l|Gen=X|Num=P|Cas=7|Var=6
C	SubPOS=l|Gen=X|Num=P|Cas=7|Var=9
C	SubPOS=l|Gen=X|Num=P|Cas=X
C	SubPOS=l|Gen=X|Num=P|Cas=X|Sem=K
C	SubPOS=l|Gen=X|Num=P|Cas=X|Sem=m
C	SubPOS=l|Gen=X|Num=S|Cas=2
C	SubPOS=l|Gen=X|Num=S|Cas=3
C	SubPOS=l|Gen=X|Num=S|Cas=6
C	SubPOS=l|Gen=X|Num=S|Cas=7
C	SubPOS=l|Gen=X|Num=S|Cas=X
C	SubPOS=l|Gen=Y|Num=P|Cas=1
C	SubPOS=l|Gen=Y|Num=P|Cas=4
C	SubPOS=l|Gen=Y|Num=P|Cas=5
C	SubPOS=l|Gen=Y|Num=S|Cas=1
C	SubPOS=l|Gen=Y|Num=S|Cas=5
C	SubPOS=l|Gen=Z|Num=S|Cas=2
C	SubPOS=l|Gen=Z|Num=S|Cas=3
C	SubPOS=l|Gen=Z|Num=S|Cas=6
C	SubPOS=l|Gen=Z|Num=S|Cas=7
C	SubPOS=n|Num=P|Cas=2
C	SubPOS=n|Num=P|Cas=2|Var=1
C	SubPOS=n|Num=P|Cas=3
C	SubPOS=n|Num=P|Cas=3|Var=1
C	SubPOS=n|Num=P|Cas=6
C	SubPOS=n|Num=P|Cas=6|Var=1
C	SubPOS=n|Num=P|Cas=7
C	SubPOS=n|Num=P|Cas=7|Var=1
C	SubPOS=n|Num=S|Cas=1
C	SubPOS=n|Num=S|Cas=1|Sem=K
C	SubPOS=n|Num=S|Cas=1|Sem=m
C	SubPOS=n|Num=S|Cas=1|Var=1
C	SubPOS=n|Num=S|Cas=4
C	SubPOS=n|Num=S|Cas=4|Var=1
C	SubPOS=n|Num=S|Cas=5
C	SubPOS=n|Num=S|Cas=5|Var=1
C	SubPOS=n|Num=S|Cas=X
C	SubPOS=n|Num=X|Cas=X
C	SubPOS=o|Var=1
C	SubPOS=o
C	SubPOS=r|Gen=F|Num=D|Cas=7
C	SubPOS=r|Gen=F|Num=D|Cas=7|Var=6
C	SubPOS=r|Gen=F|Num=P|Cas=1
C	SubPOS=r|Gen=F|Num=P|Cas=1|Var=6
C	SubPOS=r|Gen=F|Num=P|Cas=2
C	SubPOS=r|Gen=F|Num=P|Cas=2|Var=6
C	SubPOS=r|Gen=F|Num=P|Cas=3
C	SubPOS=r|Gen=F|Num=P|Cas=3|Var=6
C	SubPOS=r|Gen=F|Num=P|Cas=4
C	SubPOS=r|Gen=F|Num=P|Cas=4|Var=6
C	SubPOS=r|Gen=F|Num=P|Cas=5
C	SubPOS=r|Gen=F|Num=P|Cas=5|Var=6
C	SubPOS=r|Gen=F|Num=P|Cas=6
C	SubPOS=r|Gen=F|Num=P|Cas=6|Var=6
C	SubPOS=r|Gen=F|Num=P|Cas=7
C	SubPOS=r|Gen=F|Num=P|Cas=7|Var=6
C	SubPOS=r|Gen=F|Num=P|Cas=7|Var=7
C	SubPOS=r|Gen=F|Num=S|Cas=1
C	SubPOS=r|Gen=F|Num=S|Cas=2
C	SubPOS=r|Gen=F|Num=S|Cas=2|Var=6
C	SubPOS=r|Gen=F|Num=S|Cas=3
C	SubPOS=r|Gen=F|Num=S|Cas=3|Var=6
C	SubPOS=r|Gen=F|Num=S|Cas=4
C	SubPOS=r|Gen=F|Num=S|Cas=5
C	SubPOS=r|Gen=F|Num=S|Cas=6
C	SubPOS=r|Gen=F|Num=S|Cas=6|Var=6
C	SubPOS=r|Gen=F|Num=S|Cas=7
C	SubPOS=r|Gen=I|Num=P|Cas=1
C	SubPOS=r|Gen=I|Num=P|Cas=1|Var=6
C	SubPOS=r|Gen=I|Num=P|Cas=2
C	SubPOS=r|Gen=I|Num=P|Cas=2|Var=6
C	SubPOS=r|Gen=I|Num=P|Cas=3
C	SubPOS=r|Gen=I|Num=P|Cas=3|Var=6
C	SubPOS=r|Gen=I|Num=P|Cas=4
C	SubPOS=r|Gen=I|Num=P|Cas=4|Var=6
C	SubPOS=r|Gen=I|Num=P|Cas=5
C	SubPOS=r|Gen=I|Num=P|Cas=5|Var=6
C	SubPOS=r|Gen=I|Num=P|Cas=6
C	SubPOS=r|Gen=I|Num=P|Cas=6|Var=6
C	SubPOS=r|Gen=I|Num=P|Cas=7
C	SubPOS=r|Gen=I|Num=P|Cas=7|Var=6
C	SubPOS=r|Gen=I|Num=P|Cas=7|Var=7
C	SubPOS=r|Gen=I|Num=S|Cas=1
C	SubPOS=r|Gen=I|Num=S|Cas=1|Var=6
C	SubPOS=r|Gen=I|Num=S|Cas=2
C	SubPOS=r|Gen=I|Num=S|Cas=2|Var=6
C	SubPOS=r|Gen=I|Num=S|Cas=3
C	SubPOS=r|Gen=I|Num=S|Cas=3|Var=6
C	SubPOS=r|Gen=I|Num=S|Cas=4
C	SubPOS=r|Gen=I|Num=S|Cas=4|Var=6
C	SubPOS=r|Gen=I|Num=S|Cas=5
C	SubPOS=r|Gen=I|Num=S|Cas=5|Var=6
C	SubPOS=r|Gen=I|Num=S|Cas=6
C	SubPOS=r|Gen=I|Num=S|Cas=6|Var=6
C	SubPOS=r|Gen=I|Num=S|Cas=6|Var=7
C	SubPOS=r|Gen=I|Num=S|Cas=7
C	SubPOS=r|Gen=I|Num=S|Cas=7|Var=6
C	SubPOS=r|Gen=M|Num=P|Cas=1
C	SubPOS=r|Gen=M|Num=P|Cas=1|Var=6
C	SubPOS=r|Gen=M|Num=P|Cas=2
C	SubPOS=r|Gen=M|Num=P|Cas=2|Var=6
C	SubPOS=r|Gen=M|Num=P|Cas=3
C	SubPOS=r|Gen=M|Num=P|Cas=3|Var=6
C	SubPOS=r|Gen=M|Num=P|Cas=4
C	SubPOS=r|Gen=M|Num=P|Cas=4|Var=6
C	SubPOS=r|Gen=M|Num=P|Cas=5
C	SubPOS=r|Gen=M|Num=P|Cas=5|Var=6
C	SubPOS=r|Gen=M|Num=P|Cas=6
C	SubPOS=r|Gen=M|Num=P|Cas=6|Var=6
C	SubPOS=r|Gen=M|Num=P|Cas=7
C	SubPOS=r|Gen=M|Num=P|Cas=7|Var=6
C	SubPOS=r|Gen=M|Num=P|Cas=7|Var=7
C	SubPOS=r|Gen=M|Num=S|Cas=1
C	SubPOS=r|Gen=M|Num=S|Cas=1|Var=6
C	SubPOS=r|Gen=M|Num=S|Cas=2
C	SubPOS=r|Gen=M|Num=S|Cas=2|Var=6
C	SubPOS=r|Gen=M|Num=S|Cas=3
C	SubPOS=r|Gen=M|Num=S|Cas=3|Var=6
C	SubPOS=r|Gen=M|Num=S|Cas=4
C	SubPOS=r|Gen=M|Num=S|Cas=4|Var=6
C	SubPOS=r|Gen=M|Num=S|Cas=5
C	SubPOS=r|Gen=M|Num=S|Cas=5|Var=6
C	SubPOS=r|Gen=M|Num=S|Cas=6
C	SubPOS=r|Gen=M|Num=S|Cas=6|Var=6
C	SubPOS=r|Gen=M|Num=S|Cas=6|Var=7
C	SubPOS=r|Gen=M|Num=S|Cas=7
C	SubPOS=r|Gen=M|Num=S|Cas=7|Var=6
C	SubPOS=r|Gen=N|Num=D|Cas=7
C	SubPOS=r|Gen=N|Num=P|Cas=1
C	SubPOS=r|Gen=N|Num=P|Cas=1|Var=6
C	SubPOS=r|Gen=N|Num=P|Cas=2
C	SubPOS=r|Gen=N|Num=P|Cas=2|Var=6
C	SubPOS=r|Gen=N|Num=P|Cas=3
C	SubPOS=r|Gen=N|Num=P|Cas=3|Var=6
C	SubPOS=r|Gen=N|Num=P|Cas=4
C	SubPOS=r|Gen=N|Num=P|Cas=4|Var=6
C	SubPOS=r|Gen=N|Num=P|Cas=5
C	SubPOS=r|Gen=N|Num=P|Cas=5|Var=6
C	SubPOS=r|Gen=N|Num=P|Cas=6
C	SubPOS=r|Gen=N|Num=P|Cas=6|Var=6
C	SubPOS=r|Gen=N|Num=P|Cas=7
C	SubPOS=r|Gen=N|Num=P|Cas=7|Var=6
C	SubPOS=r|Gen=N|Num=P|Cas=7|Var=7
C	SubPOS=r|Gen=N|Num=S|Cas=1
C	SubPOS=r|Gen=N|Num=S|Cas=1|Var=6
C	SubPOS=r|Gen=N|Num=S|Cas=2
C	SubPOS=r|Gen=N|Num=S|Cas=2|Var=6
C	SubPOS=r|Gen=N|Num=S|Cas=3
C	SubPOS=r|Gen=N|Num=S|Cas=3|Var=6
C	SubPOS=r|Gen=N|Num=S|Cas=4
C	SubPOS=r|Gen=N|Num=S|Cas=4|Var=6
C	SubPOS=r|Gen=N|Num=S|Cas=5
C	SubPOS=r|Gen=N|Num=S|Cas=5|Var=6
C	SubPOS=r|Gen=N|Num=S|Cas=6
C	SubPOS=r|Gen=N|Num=S|Cas=6|Var=6
C	SubPOS=r|Gen=N|Num=S|Cas=6|Var=7
C	SubPOS=r|Gen=N|Num=S|Cas=7
C	SubPOS=r|Gen=N|Num=S|Cas=7|Var=6
C	SubPOS=r|Gen=X|Num=X|Cas=X
C	SubPOS=r|Gen=X|Num=X|Cas=X|Sem=K
C	SubPOS=r|Gen=X|Num=X|Cas=X|Sem=m
C	SubPOS=u
C	SubPOS=v|Var=1
C	SubPOS=v|Var=6
C	SubPOS=v
C	SubPOS=w|Gen=F|Num=D|Cas=7
C	SubPOS=w|Gen=F|Num=P|Cas=1
C	SubPOS=w|Gen=F|Num=P|Cas=4
C	SubPOS=w|Gen=F|Num=P|Cas=5
C	SubPOS=w|Gen=F|Num=S|Cas=1
C	SubPOS=w|Gen=F|Num=S|Cas=2
C	SubPOS=w|Gen=F|Num=S|Cas=3
C	SubPOS=w|Gen=F|Num=S|Cas=4
C	SubPOS=w|Gen=F|Num=S|Cas=5
C	SubPOS=w|Gen=F|Num=S|Cas=6
C	SubPOS=w|Gen=F|Num=S|Cas=7
C	SubPOS=w|Gen=I|Num=P|Cas=1
C	SubPOS=w|Gen=I|Num=P|Cas=5
C	SubPOS=w|Gen=I|Num=S|Cas=4
C	SubPOS=w|Gen=M|Num=P|Cas=1
C	SubPOS=w|Gen=M|Num=P|Cas=5
C	SubPOS=w|Gen=M|Num=S|Cas=4
C	SubPOS=w|Gen=N|Num=P|Cas=1
C	SubPOS=w|Gen=N|Num=P|Cas=4
C	SubPOS=w|Gen=N|Num=P|Cas=5
C	SubPOS=w|Gen=N|Num=S|Cas=1
C	SubPOS=w|Gen=N|Num=S|Cas=4
C	SubPOS=w|Gen=N|Num=S|Cas=5
C	SubPOS=w|Gen=X|Num=P|Cas=2
C	SubPOS=w|Gen=X|Num=P|Cas=3
C	SubPOS=w|Gen=X|Num=P|Cas=6
C	SubPOS=w|Gen=X|Num=P|Cas=7
C	SubPOS=w|Gen=Y|Num=P|Cas=4
C	SubPOS=w|Gen=Y|Num=S|Cas=1
C	SubPOS=w|Gen=Y|Num=S|Cas=5
C	SubPOS=w|Gen=Z|Num=S|Cas=2
C	SubPOS=w|Gen=Z|Num=S|Cas=3
C	SubPOS=w|Gen=Z|Num=S|Cas=6
C	SubPOS=w|Gen=Z|Num=S|Cas=7
C	SubPOS=y|Gen=F|Num=P|Cas=1
C	SubPOS=y|Gen=F|Num=P|Cas=2
C	SubPOS=y|Gen=F|Num=P|Cas=3
C	SubPOS=y|Gen=F|Num=P|Cas=4
C	SubPOS=y|Gen=F|Num=P|Cas=5
C	SubPOS=y|Gen=F|Num=P|Cas=6
C	SubPOS=y|Gen=F|Num=P|Cas=7
C	SubPOS=y|Gen=F|Num=P|Cas=7|Var=6
C	SubPOS=y|Gen=F|Num=S|Cas=1
C	SubPOS=y|Gen=F|Num=S|Cas=2
C	SubPOS=y|Gen=F|Num=S|Cas=3
C	SubPOS=y|Gen=F|Num=S|Cas=4
C	SubPOS=y|Gen=F|Num=S|Cas=5
C	SubPOS=y|Gen=F|Num=S|Cas=6
C	SubPOS=y|Gen=F|Num=S|Cas=7
C	SubPOS=z|Gen=F|Num=D|Cas=7
C	SubPOS=z|Gen=F|Num=P|Cas=1
C	SubPOS=z|Gen=F|Num=P|Cas=4
C	SubPOS=z|Gen=F|Num=S|Cas=1
C	SubPOS=z|Gen=F|Num=S|Cas=2
C	SubPOS=z|Gen=F|Num=S|Cas=3
C	SubPOS=z|Gen=F|Num=S|Cas=4
C	SubPOS=z|Gen=F|Num=S|Cas=6
C	SubPOS=z|Gen=F|Num=S|Cas=7
C	SubPOS=z|Gen=I|Num=P|Cas=1
C	SubPOS=z|Gen=I|Num=S|Cas=4
C	SubPOS=z|Gen=M|Num=P|Cas=1
C	SubPOS=z|Gen=M|Num=S|Cas=4
C	SubPOS=z|Gen=N|Num=P|Cas=1
C	SubPOS=z|Gen=N|Num=P|Cas=4
C	SubPOS=z|Gen=N|Num=S|Cas=1
C	SubPOS=z|Gen=N|Num=S|Cas=4
C	SubPOS=z|Gen=X|Num=P|Cas=2
C	SubPOS=z|Gen=X|Num=P|Cas=3
C	SubPOS=z|Gen=X|Num=P|Cas=6
C	SubPOS=z|Gen=X|Num=P|Cas=7
C	SubPOS=z|Gen=Y|Num=P|Cas=4
C	SubPOS=z|Gen=Y|Num=S|Cas=1
C	SubPOS=z|Gen=Z|Num=S|Cas=2
C	SubPOS=z|Gen=Z|Num=S|Cas=3
C	SubPOS=z|Gen=Z|Num=S|Cas=6
C	SubPOS=z|Gen=Z|Num=S|Cas=7
C	SubPOS=}|Var=1
C	SubPOS=}|Var=2
C	SubPOS=}
D	SubPOS=!
D	SubPOS=b|Neg=A
D	SubPOS=b|Neg=N
D	SubPOS=b|Sem=R
D	SubPOS=b|Sem=m
D	SubPOS=b|Var=1
D	SubPOS=b|Var=2
D	SubPOS=b|Var=4
D	SubPOS=b|Var=6
D	SubPOS=b|Var=7
D	SubPOS=b|Var=8
D	SubPOS=b|Var=9
D	SubPOS=b
D	SubPOS=g|Gra=1|Neg=A
D	SubPOS=g|Gra=1|Neg=A|Sem=S
D	SubPOS=g|Gra=1|Neg=A|Sem=m
D	SubPOS=g|Gra=1|Neg=A|Var=1
D	SubPOS=g|Gra=1|Neg=A|Var=3
D	SubPOS=g|Gra=1|Neg=A|Var=8
D	SubPOS=g|Gra=1|Neg=N
D	SubPOS=g|Gra=1|Neg=N|Var=3
D	SubPOS=g|Gra=1|Neg=N|Var=8
D	SubPOS=g|Gra=2|Neg=A
D	SubPOS=g|Gra=2|Neg=A|Sem=K
D	SubPOS=g|Gra=2|Neg=A|Sem=m
D	SubPOS=g|Gra=2|Neg=A|Var=1
D	SubPOS=g|Gra=2|Neg=A|Var=2
D	SubPOS=g|Gra=2|Neg=A|Var=3
D	SubPOS=g|Gra=2|Neg=A|Var=6
D	SubPOS=g|Gra=2|Neg=N
D	SubPOS=g|Gra=2|Neg=N|Var=1
D	SubPOS=g|Gra=2|Neg=N|Var=2
D	SubPOS=g|Gra=2|Neg=N|Var=3
D	SubPOS=g|Gra=2|Neg=N|Var=6
D	SubPOS=g|Gra=3|Neg=A
D	SubPOS=g|Gra=3|Neg=A|Var=1
D	SubPOS=g|Gra=3|Neg=A|Var=2
D	SubPOS=g|Gra=3|Neg=A|Var=3
D	SubPOS=g|Gra=3|Neg=A|Var=6
D	SubPOS=g|Gra=3|Neg=N
D	SubPOS=g|Gra=3|Neg=N|Var=1
D	SubPOS=g|Gra=3|Neg=N|Var=2
D	SubPOS=g|Gra=3|Neg=N|Var=3
D	SubPOS=g|Gra=3|Neg=N|Var=6
I	SubPOS=I|Sem=K
I	SubPOS=I|Sem=m
I	SubPOS=I
J	SubPOS=*
J	SubPOS=,|Num=P|Per=1
J	SubPOS=,|Num=P|Per=2
J	SubPOS=,|Num=S|Per=1
J	SubPOS=,|Num=S|Per=2
J	SubPOS=,|Num=X|Per=3
J	SubPOS=,|Var=1
J	SubPOS=,|Var=8
J	SubPOS=,
J	SubPOS=^|Sem=K
J	SubPOS=^|Var=1
J	SubPOS=^|Var=2
J	SubPOS=^|Var=8
J	SubPOS=^
N	SubPOS=;
N	SubPOS=N|Gen=F|Num=D|Cas=7|Neg=A
N	SubPOS=N|Gen=F|Num=D|Cas=7|Neg=A|Sem=m
N	SubPOS=N|Gen=F|Num=D|Cas=7|Neg=N
N	SubPOS=N|Gen=F|Num=P|Cas=1|Neg=A
N	SubPOS=N|Gen=F|Num=P|Cas=1|Neg=A|Sem=E
N	SubPOS=N|Gen=F|Num=P|Cas=1|Neg=A|Sem=G
N	SubPOS=N|Gen=F|Num=P|Cas=1|Neg=A|Sem=K
N	SubPOS=N|Gen=F|Num=P|Cas=1|Neg=A|Sem=R
N	SubPOS=N|Gen=F|Num=P|Cas=1|Neg=A|Sem=Y
N	SubPOS=N|Gen=F|Num=P|Cas=1|Neg=A|Sem=m
N	SubPOS=N|Gen=F|Num=P|Cas=1|Neg=A|Var=1
N	SubPOS=N|Gen=F|Num=P|Cas=1|Neg=A|Var=1|Sem=G
N	SubPOS=N|Gen=F|Num=P|Cas=1|Neg=A|Var=1|Sem=m
N	SubPOS=N|Gen=F|Num=P|Cas=1|Neg=A|Var=4
N	SubPOS=N|Gen=F|Num=P|Cas=1|Neg=A|Var=6
N	SubPOS=N|Gen=F|Num=P|Cas=1|Neg=A|Var=8
N	SubPOS=N|Gen=F|Num=P|Cas=1|Neg=N
N	SubPOS=N|Gen=F|Num=P|Cas=1|Neg=N|Var=1
N	SubPOS=N|Gen=F|Num=P|Cas=1|Neg=N|Var=4
N	SubPOS=N|Gen=F|Num=P|Cas=1|Neg=N|Var=6
N	SubPOS=N|Gen=F|Num=P|Cas=1|Neg=N|Var=8
N	SubPOS=N|Gen=F|Num=P|Cas=2|Neg=A
N	SubPOS=N|Gen=F|Num=P|Cas=2|Neg=A|Sem=E
N	SubPOS=N|Gen=F|Num=P|Cas=2|Neg=A|Sem=G
N	SubPOS=N|Gen=F|Num=P|Cas=2|Neg=A|Sem=K
N	SubPOS=N|Gen=F|Num=P|Cas=2|Neg=A|Sem=R
N	SubPOS=N|Gen=F|Num=P|Cas=2|Neg=A|Sem=S
N	SubPOS=N|Gen=F|Num=P|Cas=2|Neg=A|Sem=Y
N	SubPOS=N|Gen=F|Num=P|Cas=2|Neg=A|Sem=m
N	SubPOS=N|Gen=F|Num=P|Cas=2|Neg=A|Var=1
N	SubPOS=N|Gen=F|Num=P|Cas=2|Neg=A|Var=6
N	SubPOS=N|Gen=F|Num=P|Cas=2|Neg=A|Var=8
N	SubPOS=N|Gen=F|Num=P|Cas=2|Neg=N
N	SubPOS=N|Gen=F|Num=P|Cas=2|Neg=N|Var=1
N	SubPOS=N|Gen=F|Num=P|Cas=2|Neg=N|Var=6
N	SubPOS=N|Gen=F|Num=P|Cas=2|Neg=N|Var=8
N	SubPOS=N|Gen=F|Num=P|Cas=3|Neg=A
N	SubPOS=N|Gen=F|Num=P|Cas=3|Neg=A|Sem=E
N	SubPOS=N|Gen=F|Num=P|Cas=3|Neg=A|Sem=G
N	SubPOS=N|Gen=F|Num=P|Cas=3|Neg=A|Sem=m
N	SubPOS=N|Gen=F|Num=P|Cas=3|Neg=A|Var=1
N	SubPOS=N|Gen=F|Num=P|Cas=3|Neg=A|Var=6
N	SubPOS=N|Gen=F|Num=P|Cas=3|Neg=A|Var=8
N	SubPOS=N|Gen=F|Num=P|Cas=3|Neg=N
N	SubPOS=N|Gen=F|Num=P|Cas=3|Neg=N|Var=1
N	SubPOS=N|Gen=F|Num=P|Cas=3|Neg=N|Var=6
N	SubPOS=N|Gen=F|Num=P|Cas=3|Neg=N|Var=8
N	SubPOS=N|Gen=F|Num=P|Cas=4|Neg=A
N	SubPOS=N|Gen=F|Num=P|Cas=4|Neg=A|Sem=E
N	SubPOS=N|Gen=F|Num=P|Cas=4|Neg=A|Sem=G
N	SubPOS=N|Gen=F|Num=P|Cas=4|Neg=A|Sem=K
N	SubPOS=N|Gen=F|Num=P|Cas=4|Neg=A|Sem=R
N	SubPOS=N|Gen=F|Num=P|Cas=4|Neg=A|Sem=m
N	SubPOS=N|Gen=F|Num=P|Cas=4|Neg=A|Var=1
N	SubPOS=N|Gen=F|Num=P|Cas=4|Neg=A|Var=4
N	SubPOS=N|Gen=F|Num=P|Cas=4|Neg=A|Var=6
N	SubPOS=N|Gen=F|Num=P|Cas=4|Neg=A|Var=8
N	SubPOS=N|Gen=F|Num=P|Cas=4|Neg=N
N	SubPOS=N|Gen=F|Num=P|Cas=4|Neg=N|Var=1
N	SubPOS=N|Gen=F|Num=P|Cas=4|Neg=N|Var=4
N	SubPOS=N|Gen=F|Num=P|Cas=4|Neg=N|Var=6
N	SubPOS=N|Gen=F|Num=P|Cas=4|Neg=N|Var=8
N	SubPOS=N|Gen=F|Num=P|Cas=5|Neg=A
N	SubPOS=N|Gen=F|Num=P|Cas=5|Neg=A|Sem=m
N	SubPOS=N|Gen=F|Num=P|Cas=5|Neg=A|Var=1
N	SubPOS=N|Gen=F|Num=P|Cas=5|Neg=A|Var=4
N	SubPOS=N|Gen=F|Num=P|Cas=5|Neg=A|Var=6
N	SubPOS=N|Gen=F|Num=P|Cas=5|Neg=A|Var=8
N	SubPOS=N|Gen=F|Num=P|Cas=5|Neg=N
N	SubPOS=N|Gen=F|Num=P|Cas=5|Neg=N|Var=1
N	SubPOS=N|Gen=F|Num=P|Cas=5|Neg=N|Var=4
N	SubPOS=N|Gen=F|Num=P|Cas=5|Neg=N|Var=6
N	SubPOS=N|Gen=F|Num=P|Cas=5|Neg=N|Var=8
N	SubPOS=N|Gen=F|Num=P|Cas=6|Neg=A
N	SubPOS=N|Gen=F|Num=P|Cas=6|Neg=A|Sem=G
N	SubPOS=N|Gen=F|Num=P|Cas=6|Neg=A|Sem=K
N	SubPOS=N|Gen=F|Num=P|Cas=6|Neg=A|Sem=R
N	SubPOS=N|Gen=F|Num=P|Cas=6|Neg=A|Sem=m
N	SubPOS=N|Gen=F|Num=P|Cas=6|Neg=A|Var=1
N	SubPOS=N|Gen=F|Num=P|Cas=6|Neg=A|Var=6
N	SubPOS=N|Gen=F|Num=P|Cas=6|Neg=A|Var=7
N	SubPOS=N|Gen=F|Num=P|Cas=6|Neg=A|Var=8
N	SubPOS=N|Gen=F|Num=P|Cas=6|Neg=N
N	SubPOS=N|Gen=F|Num=P|Cas=6|Neg=N|Var=1
N	SubPOS=N|Gen=F|Num=P|Cas=6|Neg=N|Var=6
N	SubPOS=N|Gen=F|Num=P|Cas=6|Neg=N|Var=7
N	SubPOS=N|Gen=F|Num=P|Cas=6|Neg=N|Var=8
N	SubPOS=N|Gen=F|Num=P|Cas=7|Neg=A
N	SubPOS=N|Gen=F|Num=P|Cas=7|Neg=A|Sem=G
N	SubPOS=N|Gen=F|Num=P|Cas=7|Neg=A|Sem=m
N	SubPOS=N|Gen=F|Num=P|Cas=7|Neg=A|Var=1
N	SubPOS=N|Gen=F|Num=P|Cas=7|Neg=A|Var=2
N	SubPOS=N|Gen=F|Num=P|Cas=7|Neg=A|Var=6
N	SubPOS=N|Gen=F|Num=P|Cas=7|Neg=A|Var=7
N	SubPOS=N|Gen=F|Num=P|Cas=7|Neg=A|Var=8
N	SubPOS=N|Gen=F|Num=P|Cas=7|Neg=N
N	SubPOS=N|Gen=F|Num=P|Cas=7|Neg=N|Var=1
N	SubPOS=N|Gen=F|Num=P|Cas=7|Neg=N|Var=2
N	SubPOS=N|Gen=F|Num=P|Cas=7|Neg=N|Var=6
N	SubPOS=N|Gen=F|Num=P|Cas=7|Neg=N|Var=7
N	SubPOS=N|Gen=F|Num=P|Cas=7|Neg=N|Var=8
N	SubPOS=N|Gen=F|Num=P|Cas=X|Neg=A
N	SubPOS=N|Gen=F|Num=P|Cas=X|Neg=A|Sem=G
N	SubPOS=N|Gen=F|Num=P|Cas=X|Neg=A|Sem=K
N	SubPOS=N|Gen=F|Num=P|Cas=X|Neg=A|Sem=R
N	SubPOS=N|Gen=F|Num=P|Cas=X|Neg=A|Sem=m
N	SubPOS=N|Gen=F|Num=P|Cas=X|Neg=A|Var=8
N	SubPOS=N|Gen=F|Num=P|Cas=X|Neg=A|Var=8|Sem=K
N	SubPOS=N|Gen=F|Num=P|Cas=X|Neg=A|Var=8|Sem=R
N	SubPOS=N|Gen=F|Num=P|Cas=X|Neg=A|Var=8|Sem=m
N	SubPOS=N|Gen=F|Num=P|Cas=X|Neg=A|Var=8|Sem=w
N	SubPOS=N|Gen=F|Num=P|Cas=X|Neg=N
N	SubPOS=N|Gen=F|Num=P|Cas=X|Neg=N|Var=8
N	SubPOS=N|Gen=F|Num=S|Cas=1|Neg=A
N	SubPOS=N|Gen=F|Num=S|Cas=1|Neg=A|Sem=E
N	SubPOS=N|Gen=F|Num=S|Cas=1|Neg=A|Sem=G
N	SubPOS=N|Gen=F|Num=S|Cas=1|Neg=A|Sem=K
N	SubPOS=N|Gen=F|Num=S|Cas=1|Neg=A|Sem=L
N	SubPOS=N|Gen=F|Num=S|Cas=1|Neg=A|Sem=R
N	SubPOS=N|Gen=F|Num=S|Cas=1|Neg=A|Sem=S
N	SubPOS=N|Gen=F|Num=S|Cas=1|Neg=A|Sem=U
N	SubPOS=N|Gen=F|Num=S|Cas=1|Neg=A|Sem=Y
N	SubPOS=N|Gen=F|Num=S|Cas=1|Neg=A|Sem=j
N	SubPOS=N|Gen=F|Num=S|Cas=1|Neg=A|Sem=m
N	SubPOS=N|Gen=F|Num=S|Cas=1|Neg=A|Var=1
N	SubPOS=N|Gen=F|Num=S|Cas=1|Neg=A|Var=1|Sem=G
N	SubPOS=N|Gen=F|Num=S|Cas=1|Neg=A|Var=1|Sem=K
N	SubPOS=N|Gen=F|Num=S|Cas=1|Neg=A|Var=1|Sem=R
N	SubPOS=N|Gen=F|Num=S|Cas=1|Neg=A|Var=1|Sem=Y
N	SubPOS=N|Gen=F|Num=S|Cas=1|Neg=A|Var=1|Sem=m
N	SubPOS=N|Gen=F|Num=S|Cas=1|Neg=A|Var=8
N	SubPOS=N|Gen=F|Num=S|Cas=1|Neg=N
N	SubPOS=N|Gen=F|Num=S|Cas=1|Neg=N|Var=1
N	SubPOS=N|Gen=F|Num=S|Cas=1|Neg=N|Var=8
N	SubPOS=N|Gen=F|Num=S|Cas=2|Neg=A
N	SubPOS=N|Gen=F|Num=S|Cas=2|Neg=A|Sem=E
N	SubPOS=N|Gen=F|Num=S|Cas=2|Neg=A|Sem=G
N	SubPOS=N|Gen=F|Num=S|Cas=2|Neg=A|Sem=K
N	SubPOS=N|Gen=F|Num=S|Cas=2|Neg=A|Sem=L
N	SubPOS=N|Gen=F|Num=S|Cas=2|Neg=A|Sem=R
N	SubPOS=N|Gen=F|Num=S|Cas=2|Neg=A|Sem=S
N	SubPOS=N|Gen=F|Num=S|Cas=2|Neg=A|Sem=U
N	SubPOS=N|Gen=F|Num=S|Cas=2|Neg=A|Sem=Y
N	SubPOS=N|Gen=F|Num=S|Cas=2|Neg=A|Sem=j
N	SubPOS=N|Gen=F|Num=S|Cas=2|Neg=A|Sem=m
N	SubPOS=N|Gen=F|Num=S|Cas=2|Neg=A|Var=1
N	SubPOS=N|Gen=F|Num=S|Cas=2|Neg=A|Var=1|Sem=G
N	SubPOS=N|Gen=F|Num=S|Cas=2|Neg=A|Var=1|Sem=m
N	SubPOS=N|Gen=F|Num=S|Cas=2|Neg=A|Var=2
N	SubPOS=N|Gen=F|Num=S|Cas=2|Neg=A|Var=6
N	SubPOS=N|Gen=F|Num=S|Cas=2|Neg=A|Var=8
N	SubPOS=N|Gen=F|Num=S|Cas=2|Neg=A|Var=8|Sem=G
N	SubPOS=N|Gen=F|Num=S|Cas=2|Neg=A|Var=8|Sem=m
N	SubPOS=N|Gen=F|Num=S|Cas=2|Neg=N
N	SubPOS=N|Gen=F|Num=S|Cas=2|Neg=N|Var=1
N	SubPOS=N|Gen=F|Num=S|Cas=2|Neg=N|Var=2
N	SubPOS=N|Gen=F|Num=S|Cas=2|Neg=N|Var=6
N	SubPOS=N|Gen=F|Num=S|Cas=2|Neg=N|Var=8
N	SubPOS=N|Gen=F|Num=S|Cas=3|Neg=A
N	SubPOS=N|Gen=F|Num=S|Cas=3|Neg=A|Sem=E
N	SubPOS=N|Gen=F|Num=S|Cas=3|Neg=A|Sem=G
N	SubPOS=N|Gen=F|Num=S|Cas=3|Neg=A|Sem=K
N	SubPOS=N|Gen=F|Num=S|Cas=3|Neg=A|Sem=R
N	SubPOS=N|Gen=F|Num=S|Cas=3|Neg=A|Sem=S
N	SubPOS=N|Gen=F|Num=S|Cas=3|Neg=A|Sem=Y
N	SubPOS=N|Gen=F|Num=S|Cas=3|Neg=A|Sem=m
N	SubPOS=N|Gen=F|Num=S|Cas=3|Neg=A|Var=1
N	SubPOS=N|Gen=F|Num=S|Cas=3|Neg=A|Var=1|Sem=G
N	SubPOS=N|Gen=F|Num=S|Cas=3|Neg=A|Var=1|Sem=m
N	SubPOS=N|Gen=F|Num=S|Cas=3|Neg=A|Var=2
N	SubPOS=N|Gen=F|Num=S|Cas=3|Neg=A|Var=6
N	SubPOS=N|Gen=F|Num=S|Cas=3|Neg=A|Var=8
N	SubPOS=N|Gen=F|Num=S|Cas=3|Neg=N
N	SubPOS=N|Gen=F|Num=S|Cas=3|Neg=N|Var=1
N	SubPOS=N|Gen=F|Num=S|Cas=3|Neg=N|Var=2
N	SubPOS=N|Gen=F|Num=S|Cas=3|Neg=N|Var=6
N	SubPOS=N|Gen=F|Num=S|Cas=3|Neg=N|Var=8
N	SubPOS=N|Gen=F|Num=S|Cas=4|Neg=A
N	SubPOS=N|Gen=F|Num=S|Cas=4|Neg=A|Sem=E
N	SubPOS=N|Gen=F|Num=S|Cas=4|Neg=A|Sem=G
N	SubPOS=N|Gen=F|Num=S|Cas=4|Neg=A|Sem=K
N	SubPOS=N|Gen=F|Num=S|Cas=4|Neg=A|Sem=R
N	SubPOS=N|Gen=F|Num=S|Cas=4|Neg=A|Sem=S
N	SubPOS=N|Gen=F|Num=S|Cas=4|Neg=A|Sem=Y
N	SubPOS=N|Gen=F|Num=S|Cas=4|Neg=A|Sem=j
N	SubPOS=N|Gen=F|Num=S|Cas=4|Neg=A|Sem=m
N	SubPOS=N|Gen=F|Num=S|Cas=4|Neg=A|Var=1
N	SubPOS=N|Gen=F|Num=S|Cas=4|Neg=A|Var=8
N	SubPOS=N|Gen=F|Num=S|Cas=4|Neg=N
N	SubPOS=N|Gen=F|Num=S|Cas=4|Neg=N|Var=1
N	SubPOS=N|Gen=F|Num=S|Cas=4|Neg=N|Var=8
N	SubPOS=N|Gen=F|Num=S|Cas=5|Neg=A
N	SubPOS=N|Gen=F|Num=S|Cas=5|Neg=A|Sem=G
N	SubPOS=N|Gen=F|Num=S|Cas=5|Neg=A|Sem=R
N	SubPOS=N|Gen=F|Num=S|Cas=5|Neg=A|Sem=Y
N	SubPOS=N|Gen=F|Num=S|Cas=5|Neg=A|Sem=m
N	SubPOS=N|Gen=F|Num=S|Cas=5|Neg=A|Var=1
N	SubPOS=N|Gen=F|Num=S|Cas=5|Neg=A|Var=8
N	SubPOS=N|Gen=F|Num=S|Cas=5|Neg=N
N	SubPOS=N|Gen=F|Num=S|Cas=5|Neg=N|Var=1
N	SubPOS=N|Gen=F|Num=S|Cas=5|Neg=N|Var=8
N	SubPOS=N|Gen=F|Num=S|Cas=6|Neg=A
N	SubPOS=N|Gen=F|Num=S|Cas=6|Neg=A|Sem=G
N	SubPOS=N|Gen=F|Num=S|Cas=6|Neg=A|Sem=K
N	SubPOS=N|Gen=F|Num=S|Cas=6|Neg=A|Sem=R
N	SubPOS=N|Gen=F|Num=S|Cas=6|Neg=A|Sem=S
N	SubPOS=N|Gen=F|Num=S|Cas=6|Neg=A|Sem=Y
N	SubPOS=N|Gen=F|Num=S|Cas=6|Neg=A|Sem=j
N	SubPOS=N|Gen=F|Num=S|Cas=6|Neg=A|Sem=m
N	SubPOS=N|Gen=F|Num=S|Cas=6|Neg=A|Var=1
N	SubPOS=N|Gen=F|Num=S|Cas=6|Neg=A|Var=1|Sem=G
N	SubPOS=N|Gen=F|Num=S|Cas=6|Neg=A|Var=1|Sem=K
N	SubPOS=N|Gen=F|Num=S|Cas=6|Neg=A|Var=1|Sem=m
N	SubPOS=N|Gen=F|Num=S|Cas=6|Neg=A|Var=6
N	SubPOS=N|Gen=F|Num=S|Cas=6|Neg=A|Var=8
N	SubPOS=N|Gen=F|Num=S|Cas=6|Neg=N
N	SubPOS=N|Gen=F|Num=S|Cas=6|Neg=N|Var=1
N	SubPOS=N|Gen=F|Num=S|Cas=6|Neg=N|Var=6
N	SubPOS=N|Gen=F|Num=S|Cas=6|Neg=N|Var=8
N	SubPOS=N|Gen=F|Num=S|Cas=7|Neg=A
N	SubPOS=N|Gen=F|Num=S|Cas=7|Neg=A|Sem=E
N	SubPOS=N|Gen=F|Num=S|Cas=7|Neg=A|Sem=G
N	SubPOS=N|Gen=F|Num=S|Cas=7|Neg=A|Sem=K
N	SubPOS=N|Gen=F|Num=S|Cas=7|Neg=A|Sem=L
N	SubPOS=N|Gen=F|Num=S|Cas=7|Neg=A|Sem=R
N	SubPOS=N|Gen=F|Num=S|Cas=7|Neg=A|Sem=S
N	SubPOS=N|Gen=F|Num=S|Cas=7|Neg=A|Sem=Y
N	SubPOS=N|Gen=F|Num=S|Cas=7|Neg=A|Sem=m
N	SubPOS=N|Gen=F|Num=S|Cas=7|Neg=A|Var=1
N	SubPOS=N|Gen=F|Num=S|Cas=7|Neg=A|Var=1|Sem=G
N	SubPOS=N|Gen=F|Num=S|Cas=7|Neg=A|Var=1|Sem=Y
N	SubPOS=N|Gen=F|Num=S|Cas=7|Neg=A|Var=1|Sem=m
N	SubPOS=N|Gen=F|Num=S|Cas=7|Neg=A|Var=3
N	SubPOS=N|Gen=F|Num=S|Cas=7|Neg=A|Var=6
N	SubPOS=N|Gen=F|Num=S|Cas=7|Neg=A|Var=8
N	SubPOS=N|Gen=F|Num=S|Cas=7|Neg=N
N	SubPOS=N|Gen=F|Num=S|Cas=7|Neg=N|Var=1
N	SubPOS=N|Gen=F|Num=S|Cas=7|Neg=N|Var=3
N	SubPOS=N|Gen=F|Num=S|Cas=7|Neg=N|Var=6
N	SubPOS=N|Gen=F|Num=S|Cas=7|Neg=N|Var=8
N	SubPOS=N|Gen=F|Num=S|Cas=X|Neg=A
N	SubPOS=N|Gen=F|Num=S|Cas=X|Neg=A|Sem=G
N	SubPOS=N|Gen=F|Num=S|Cas=X|Neg=A|Sem=K
N	SubPOS=N|Gen=F|Num=S|Cas=X|Neg=A|Sem=R
N	SubPOS=N|Gen=F|Num=S|Cas=X|Neg=A|Sem=S
N	SubPOS=N|Gen=F|Num=S|Cas=X|Neg=A|Sem=Y
N	SubPOS=N|Gen=F|Num=S|Cas=X|Neg=A|Sem=m
N	SubPOS=N|Gen=F|Num=S|Cas=X|Neg=A|Var=8
N	SubPOS=N|Gen=F|Num=S|Cas=X|Neg=A|Var=8|Sem=R
N	SubPOS=N|Gen=F|Num=S|Cas=X|Neg=A|Var=8|Sem=m
N	SubPOS=N|Gen=F|Num=S|Cas=X|Neg=N
N	SubPOS=N|Gen=F|Num=S|Cas=X|Neg=N|Var=8
N	SubPOS=N|Gen=F|Num=X|Cas=X|Neg=A
N	SubPOS=N|Gen=F|Num=X|Cas=X|Neg=A|Sem=G
N	SubPOS=N|Gen=F|Num=X|Cas=X|Neg=A|Sem=K
N	SubPOS=N|Gen=F|Num=X|Cas=X|Neg=A|Sem=R
N	SubPOS=N|Gen=F|Num=X|Cas=X|Neg=A|Sem=S
N	SubPOS=N|Gen=F|Num=X|Cas=X|Neg=A|Sem=Y
N	SubPOS=N|Gen=F|Num=X|Cas=X|Neg=A|Sem=m
N	SubPOS=N|Gen=F|Num=X|Cas=X|Neg=A|Var=1
N	SubPOS=N|Gen=F|Num=X|Cas=X|Neg=A|Var=2
N	SubPOS=N|Gen=F|Num=X|Cas=X|Neg=A|Var=8
N	SubPOS=N|Gen=F|Num=X|Cas=X|Neg=A|Var=8|Sem=G
N	SubPOS=N|Gen=F|Num=X|Cas=X|Neg=A|Var=8|Sem=K
N	SubPOS=N|Gen=F|Num=X|Cas=X|Neg=A|Var=8|Sem=R
N	SubPOS=N|Gen=F|Num=X|Cas=X|Neg=A|Var=8|Sem=S
N	SubPOS=N|Gen=F|Num=X|Cas=X|Neg=A|Var=8|Sem=Y
N	SubPOS=N|Gen=F|Num=X|Cas=X|Neg=A|Var=8|Sem=b
N	SubPOS=N|Gen=F|Num=X|Cas=X|Neg=A|Var=8|Sem=j
N	SubPOS=N|Gen=F|Num=X|Cas=X|Neg=A|Var=8|Sem=m
N	SubPOS=N|Gen=F|Num=X|Cas=X|Neg=A|Var=8|Sem=p
N	SubPOS=N|Gen=F|Num=X|Cas=X|Neg=A|Var=8|Sem=w
N	SubPOS=N|Gen=F|Num=X|Cas=X|Neg=A|Var=9
N	SubPOS=N|Gen=F|Num=X|Cas=X|Neg=A|Var=9|Sem=m
N	SubPOS=N|Gen=F|Num=X|Cas=X|Neg=N
N	SubPOS=N|Gen=F|Num=X|Cas=X|Neg=N|Var=1
N	SubPOS=N|Gen=F|Num=X|Cas=X|Neg=N|Var=2
N	SubPOS=N|Gen=F|Num=X|Cas=X|Neg=N|Var=8
N	SubPOS=N|Gen=F|Num=X|Cas=X|Neg=N|Var=9
N	SubPOS=N|Gen=I|Num=P|Cas=1|Neg=A
N	SubPOS=N|Gen=I|Num=P|Cas=1|Neg=A|Sem=G
N	SubPOS=N|Gen=I|Num=P|Cas=1|Neg=A|Sem=K
N	SubPOS=N|Gen=I|Num=P|Cas=1|Neg=A|Sem=R
N	SubPOS=N|Gen=I|Num=P|Cas=1|Neg=A|Sem=b
N	SubPOS=N|Gen=I|Num=P|Cas=1|Neg=A|Sem=m
N	SubPOS=N|Gen=I|Num=P|Cas=1|Neg=A|Var=1
N	SubPOS=N|Gen=I|Num=P|Cas=1|Neg=A|Var=2
N	SubPOS=N|Gen=I|Num=P|Cas=1|Neg=A|Var=3
N	SubPOS=N|Gen=I|Num=P|Cas=1|Neg=A|Var=6
N	SubPOS=N|Gen=I|Num=P|Cas=1|Neg=A|Var=8
N	SubPOS=N|Gen=I|Num=P|Cas=1|Neg=A|Var=9
N	SubPOS=N|Gen=I|Num=P|Cas=1|Neg=N
N	SubPOS=N|Gen=I|Num=P|Cas=1|Neg=N|Var=1
N	SubPOS=N|Gen=I|Num=P|Cas=1|Neg=N|Var=2
N	SubPOS=N|Gen=I|Num=P|Cas=1|Neg=N|Var=3
N	SubPOS=N|Gen=I|Num=P|Cas=1|Neg=N|Var=6
N	SubPOS=N|Gen=I|Num=P|Cas=1|Neg=N|Var=8
N	SubPOS=N|Gen=I|Num=P|Cas=1|Neg=N|Var=9
N	SubPOS=N|Gen=I|Num=P|Cas=2|Neg=A
N	SubPOS=N|Gen=I|Num=P|Cas=2|Neg=A|Sem=G
N	SubPOS=N|Gen=I|Num=P|Cas=2|Neg=A|Sem=H
N	SubPOS=N|Gen=I|Num=P|Cas=2|Neg=A|Sem=K
N	SubPOS=N|Gen=I|Num=P|Cas=2|Neg=A|Sem=R
N	SubPOS=N|Gen=I|Num=P|Cas=2|Neg=A|Sem=b
N	SubPOS=N|Gen=I|Num=P|Cas=2|Neg=A|Sem=m
N	SubPOS=N|Gen=I|Num=P|Cas=2|Neg=A|Var=1
N	SubPOS=N|Gen=I|Num=P|Cas=2|Neg=A|Var=1|Sem=G
N	SubPOS=N|Gen=I|Num=P|Cas=2|Neg=A|Var=1|Sem=m
N	SubPOS=N|Gen=I|Num=P|Cas=2|Neg=A|Var=2
N	SubPOS=N|Gen=I|Num=P|Cas=2|Neg=A|Var=3
N	SubPOS=N|Gen=I|Num=P|Cas=2|Neg=A|Var=6
N	SubPOS=N|Gen=I|Num=P|Cas=2|Neg=A|Var=8
N	SubPOS=N|Gen=I|Num=P|Cas=2|Neg=A|Var=9
N	SubPOS=N|Gen=I|Num=P|Cas=2|Neg=N
N	SubPOS=N|Gen=I|Num=P|Cas=2|Neg=N|Var=1
N	SubPOS=N|Gen=I|Num=P|Cas=2|Neg=N|Var=2
N	SubPOS=N|Gen=I|Num=P|Cas=2|Neg=N|Var=3
N	SubPOS=N|Gen=I|Num=P|Cas=2|Neg=N|Var=6
N	SubPOS=N|Gen=I|Num=P|Cas=2|Neg=N|Var=8
N	SubPOS=N|Gen=I|Num=P|Cas=2|Neg=N|Var=9
N	SubPOS=N|Gen=I|Num=P|Cas=3|Neg=A
N	SubPOS=N|Gen=I|Num=P|Cas=3|Neg=A|Sem=G
N	SubPOS=N|Gen=I|Num=P|Cas=3|Neg=A|Sem=R
N	SubPOS=N|Gen=I|Num=P|Cas=3|Neg=A|Sem=m
N	SubPOS=N|Gen=I|Num=P|Cas=3|Neg=A|Var=1
N	SubPOS=N|Gen=I|Num=P|Cas=3|Neg=A|Var=2
N	SubPOS=N|Gen=I|Num=P|Cas=3|Neg=A|Var=6
N	SubPOS=N|Gen=I|Num=P|Cas=3|Neg=A|Var=7
N	SubPOS=N|Gen=I|Num=P|Cas=3|Neg=A|Var=8
N	SubPOS=N|Gen=I|Num=P|Cas=3|Neg=A|Var=9
N	SubPOS=N|Gen=I|Num=P|Cas=3|Neg=N
N	SubPOS=N|Gen=I|Num=P|Cas=3|Neg=N|Var=1
N	SubPOS=N|Gen=I|Num=P|Cas=3|Neg=N|Var=2
N	SubPOS=N|Gen=I|Num=P|Cas=3|Neg=N|Var=6
N	SubPOS=N|Gen=I|Num=P|Cas=3|Neg=N|Var=7
N	SubPOS=N|Gen=I|Num=P|Cas=3|Neg=N|Var=8
N	SubPOS=N|Gen=I|Num=P|Cas=3|Neg=N|Var=9
N	SubPOS=N|Gen=I|Num=P|Cas=4|Neg=A
N	SubPOS=N|Gen=I|Num=P|Cas=4|Neg=A|Sem=G
N	SubPOS=N|Gen=I|Num=P|Cas=4|Neg=A|Sem=L
N	SubPOS=N|Gen=I|Num=P|Cas=4|Neg=A|Sem=R
N	SubPOS=N|Gen=I|Num=P|Cas=4|Neg=A|Sem=b
N	SubPOS=N|Gen=I|Num=P|Cas=4|Neg=A|Sem=m
N	SubPOS=N|Gen=I|Num=P|Cas=4|Neg=A|Var=1
N	SubPOS=N|Gen=I|Num=P|Cas=4|Neg=A|Var=2
N	SubPOS=N|Gen=I|Num=P|Cas=4|Neg=A|Var=3
N	SubPOS=N|Gen=I|Num=P|Cas=4|Neg=A|Var=6
N	SubPOS=N|Gen=I|Num=P|Cas=4|Neg=A|Var=8
N	SubPOS=N|Gen=I|Num=P|Cas=4|Neg=A|Var=9
N	SubPOS=N|Gen=I|Num=P|Cas=4|Neg=N
N	SubPOS=N|Gen=I|Num=P|Cas=4|Neg=N|Var=1
N	SubPOS=N|Gen=I|Num=P|Cas=4|Neg=N|Var=2
N	SubPOS=N|Gen=I|Num=P|Cas=4|Neg=N|Var=3
N	SubPOS=N|Gen=I|Num=P|Cas=4|Neg=N|Var=6
N	SubPOS=N|Gen=I|Num=P|Cas=4|Neg=N|Var=8
N	SubPOS=N|Gen=I|Num=P|Cas=4|Neg=N|Var=9
N	SubPOS=N|Gen=I|Num=P|Cas=5|Neg=A
N	SubPOS=N|Gen=I|Num=P|Cas=5|Neg=A|Var=1
N	SubPOS=N|Gen=I|Num=P|Cas=5|Neg=A|Var=2
N	SubPOS=N|Gen=I|Num=P|Cas=5|Neg=A|Var=3
N	SubPOS=N|Gen=I|Num=P|Cas=5|Neg=A|Var=6
N	SubPOS=N|Gen=I|Num=P|Cas=5|Neg=A|Var=8
N	SubPOS=N|Gen=I|Num=P|Cas=5|Neg=A|Var=9
N	SubPOS=N|Gen=I|Num=P|Cas=5|Neg=N
N	SubPOS=N|Gen=I|Num=P|Cas=5|Neg=N|Var=1
N	SubPOS=N|Gen=I|Num=P|Cas=5|Neg=N|Var=2
N	SubPOS=N|Gen=I|Num=P|Cas=5|Neg=N|Var=3
N	SubPOS=N|Gen=I|Num=P|Cas=5|Neg=N|Var=6
N	SubPOS=N|Gen=I|Num=P|Cas=5|Neg=N|Var=8
N	SubPOS=N|Gen=I|Num=P|Cas=5|Neg=N|Var=9
N	SubPOS=N|Gen=I|Num=P|Cas=6|Neg=A
N	SubPOS=N|Gen=I|Num=P|Cas=6|Neg=A|Sem=G
N	SubPOS=N|Gen=I|Num=P|Cas=6|Neg=A|Sem=K
N	SubPOS=N|Gen=I|Num=P|Cas=6|Neg=A|Sem=R
N	SubPOS=N|Gen=I|Num=P|Cas=6|Neg=A|Sem=b
N	SubPOS=N|Gen=I|Num=P|Cas=6|Neg=A|Sem=m
N	SubPOS=N|Gen=I|Num=P|Cas=6|Neg=A|Var=1
N	SubPOS=N|Gen=I|Num=P|Cas=6|Neg=A|Var=1|Sem=G
N	SubPOS=N|Gen=I|Num=P|Cas=6|Neg=A|Var=1|Sem=m
N	SubPOS=N|Gen=I|Num=P|Cas=6|Neg=A|Var=2
N	SubPOS=N|Gen=I|Num=P|Cas=6|Neg=A|Var=3
N	SubPOS=N|Gen=I|Num=P|Cas=6|Neg=A|Var=6
N	SubPOS=N|Gen=I|Num=P|Cas=6|Neg=A|Var=6|Sem=G
N	SubPOS=N|Gen=I|Num=P|Cas=6|Neg=A|Var=6|Sem=m
N	SubPOS=N|Gen=I|Num=P|Cas=6|Neg=A|Var=8
N	SubPOS=N|Gen=I|Num=P|Cas=6|Neg=A|Var=9
N	SubPOS=N|Gen=I|Num=P|Cas=6|Neg=N
N	SubPOS=N|Gen=I|Num=P|Cas=6|Neg=N|Var=1
N	SubPOS=N|Gen=I|Num=P|Cas=6|Neg=N|Var=2
N	SubPOS=N|Gen=I|Num=P|Cas=6|Neg=N|Var=3
N	SubPOS=N|Gen=I|Num=P|Cas=6|Neg=N|Var=6
N	SubPOS=N|Gen=I|Num=P|Cas=6|Neg=N|Var=8
N	SubPOS=N|Gen=I|Num=P|Cas=6|Neg=N|Var=9
N	SubPOS=N|Gen=I|Num=P|Cas=7|Neg=A
N	SubPOS=N|Gen=I|Num=P|Cas=7|Neg=A|Sem=G
N	SubPOS=N|Gen=I|Num=P|Cas=7|Neg=A|Sem=R
N	SubPOS=N|Gen=I|Num=P|Cas=7|Neg=A|Sem=b
N	SubPOS=N|Gen=I|Num=P|Cas=7|Neg=A|Sem=m
N	SubPOS=N|Gen=I|Num=P|Cas=7|Neg=A|Var=1
N	SubPOS=N|Gen=I|Num=P|Cas=7|Neg=A|Var=2
N	SubPOS=N|Gen=I|Num=P|Cas=7|Neg=A|Var=6
N	SubPOS=N|Gen=I|Num=P|Cas=7|Neg=A|Var=7
N	SubPOS=N|Gen=I|Num=P|Cas=7|Neg=A|Var=8
N	SubPOS=N|Gen=I|Num=P|Cas=7|Neg=A|Var=9
N	SubPOS=N|Gen=I|Num=P|Cas=7|Neg=N
N	SubPOS=N|Gen=I|Num=P|Cas=7|Neg=N|Var=1
N	SubPOS=N|Gen=I|Num=P|Cas=7|Neg=N|Var=2
N	SubPOS=N|Gen=I|Num=P|Cas=7|Neg=N|Var=6
N	SubPOS=N|Gen=I|Num=P|Cas=7|Neg=N|Var=7
N	SubPOS=N|Gen=I|Num=P|Cas=7|Neg=N|Var=8
N	SubPOS=N|Gen=I|Num=P|Cas=7|Neg=N|Var=9
N	SubPOS=N|Gen=I|Num=P|Cas=X|Neg=A
N	SubPOS=N|Gen=I|Num=P|Cas=X|Neg=A|Sem=G
N	SubPOS=N|Gen=I|Num=P|Cas=X|Neg=A|Sem=K
N	SubPOS=N|Gen=I|Num=P|Cas=X|Neg=A|Sem=R
N	SubPOS=N|Gen=I|Num=P|Cas=X|Neg=A|Sem=m
N	SubPOS=N|Gen=I|Num=P|Cas=X|Neg=A|Var=1
N	SubPOS=N|Gen=I|Num=P|Cas=X|Neg=A|Var=8
N	SubPOS=N|Gen=I|Num=P|Cas=X|Neg=A|Var=8|Sem=G
N	SubPOS=N|Gen=I|Num=P|Cas=X|Neg=A|Var=8|Sem=K
N	SubPOS=N|Gen=I|Num=P|Cas=X|Neg=A|Var=8|Sem=R
N	SubPOS=N|Gen=I|Num=P|Cas=X|Neg=A|Var=8|Sem=m
N	SubPOS=N|Gen=I|Num=P|Cas=X|Neg=N
N	SubPOS=N|Gen=I|Num=P|Cas=X|Neg=N|Var=1
N	SubPOS=N|Gen=I|Num=P|Cas=X|Neg=N|Var=8
N	SubPOS=N|Gen=I|Num=S|Cas=1|Neg=A
N	SubPOS=N|Gen=I|Num=S|Cas=1|Neg=A|Sem=E
N	SubPOS=N|Gen=I|Num=S|Cas=1|Neg=A|Sem=G
N	SubPOS=N|Gen=I|Num=S|Cas=1|Neg=A|Sem=H
N	SubPOS=N|Gen=I|Num=S|Cas=1|Neg=A|Sem=K
N	SubPOS=N|Gen=I|Num=S|Cas=1|Neg=A|Sem=L
N	SubPOS=N|Gen=I|Num=S|Cas=1|Neg=A|Sem=R
N	SubPOS=N|Gen=I|Num=S|Cas=1|Neg=A|Sem=S
N	SubPOS=N|Gen=I|Num=S|Cas=1|Neg=A|Sem=U
N	SubPOS=N|Gen=I|Num=S|Cas=1|Neg=A|Sem=Y
N	SubPOS=N|Gen=I|Num=S|Cas=1|Neg=A|Sem=b
N	SubPOS=N|Gen=I|Num=S|Cas=1|Neg=A|Sem=m
N	SubPOS=N|Gen=I|Num=S|Cas=1|Neg=A|Var=1
N	SubPOS=N|Gen=I|Num=S|Cas=1|Neg=A|Var=1|Sem=G
N	SubPOS=N|Gen=I|Num=S|Cas=1|Neg=A|Var=1|Sem=m
N	SubPOS=N|Gen=I|Num=S|Cas=1|Neg=A|Var=2
N	SubPOS=N|Gen=I|Num=S|Cas=1|Neg=A|Var=6
N	SubPOS=N|Gen=I|Num=S|Cas=1|Neg=A|Var=8
N	SubPOS=N|Gen=I|Num=S|Cas=1|Neg=N
N	SubPOS=N|Gen=I|Num=S|Cas=1|Neg=N|Var=1
N	SubPOS=N|Gen=I|Num=S|Cas=1|Neg=N|Var=2
N	SubPOS=N|Gen=I|Num=S|Cas=1|Neg=N|Var=6
N	SubPOS=N|Gen=I|Num=S|Cas=1|Neg=N|Var=8
N	SubPOS=N|Gen=I|Num=S|Cas=2|Neg=A
N	SubPOS=N|Gen=I|Num=S|Cas=2|Neg=A|Sem=G
N	SubPOS=N|Gen=I|Num=S|Cas=2|Neg=A|Sem=H
N	SubPOS=N|Gen=I|Num=S|Cas=2|Neg=A|Sem=K
N	SubPOS=N|Gen=I|Num=S|Cas=2|Neg=A|Sem=L
N	SubPOS=N|Gen=I|Num=S|Cas=2|Neg=A|Sem=R
N	SubPOS=N|Gen=I|Num=S|Cas=2|Neg=A|Sem=S
N	SubPOS=N|Gen=I|Num=S|Cas=2|Neg=A|Sem=Y
N	SubPOS=N|Gen=I|Num=S|Cas=2|Neg=A|Sem=b
N	SubPOS=N|Gen=I|Num=S|Cas=2|Neg=A|Sem=m
N	SubPOS=N|Gen=I|Num=S|Cas=2|Neg=A|Sem=u
N	SubPOS=N|Gen=I|Num=S|Cas=2|Neg=A|Sem=w
N	SubPOS=N|Gen=I|Num=S|Cas=2|Neg=A|Var=1
N	SubPOS=N|Gen=I|Num=S|Cas=2|Neg=A|Var=1|Sem=G
N	SubPOS=N|Gen=I|Num=S|Cas=2|Neg=A|Var=1|Sem=K
N	SubPOS=N|Gen=I|Num=S|Cas=2|Neg=A|Var=1|Sem=R
N	SubPOS=N|Gen=I|Num=S|Cas=2|Neg=A|Var=1|Sem=m
N	SubPOS=N|Gen=I|Num=S|Cas=2|Neg=A|Var=2
N	SubPOS=N|Gen=I|Num=S|Cas=2|Neg=A|Var=2|Sem=G
N	SubPOS=N|Gen=I|Num=S|Cas=2|Neg=A|Var=2|Sem=m
N	SubPOS=N|Gen=I|Num=S|Cas=2|Neg=A|Var=6
N	SubPOS=N|Gen=I|Num=S|Cas=2|Neg=A|Var=8
N	SubPOS=N|Gen=I|Num=S|Cas=2|Neg=A|Var=9
N	SubPOS=N|Gen=I|Num=S|Cas=2|Neg=N
N	SubPOS=N|Gen=I|Num=S|Cas=2|Neg=N|Var=1
N	SubPOS=N|Gen=I|Num=S|Cas=2|Neg=N|Var=2
N	SubPOS=N|Gen=I|Num=S|Cas=2|Neg=N|Var=6
N	SubPOS=N|Gen=I|Num=S|Cas=2|Neg=N|Var=8
N	SubPOS=N|Gen=I|Num=S|Cas=2|Neg=N|Var=9
N	SubPOS=N|Gen=I|Num=S|Cas=3|Neg=A
N	SubPOS=N|Gen=I|Num=S|Cas=3|Neg=A|Sem=E
N	SubPOS=N|Gen=I|Num=S|Cas=3|Neg=A|Sem=G
N	SubPOS=N|Gen=I|Num=S|Cas=3|Neg=A|Sem=K
N	SubPOS=N|Gen=I|Num=S|Cas=3|Neg=A|Sem=R
N	SubPOS=N|Gen=I|Num=S|Cas=3|Neg=A|Sem=Y
N	SubPOS=N|Gen=I|Num=S|Cas=3|Neg=A|Sem=b
N	SubPOS=N|Gen=I|Num=S|Cas=3|Neg=A|Sem=m
N	SubPOS=N|Gen=I|Num=S|Cas=3|Neg=A|Var=1
N	SubPOS=N|Gen=I|Num=S|Cas=3|Neg=A|Var=2
N	SubPOS=N|Gen=I|Num=S|Cas=3|Neg=A|Var=6
N	SubPOS=N|Gen=I|Num=S|Cas=3|Neg=A|Var=8
N	SubPOS=N|Gen=I|Num=S|Cas=3|Neg=A|Var=9
N	SubPOS=N|Gen=I|Num=S|Cas=3|Neg=N
N	SubPOS=N|Gen=I|Num=S|Cas=3|Neg=N|Var=1
N	SubPOS=N|Gen=I|Num=S|Cas=3|Neg=N|Var=2
N	SubPOS=N|Gen=I|Num=S|Cas=3|Neg=N|Var=6
N	SubPOS=N|Gen=I|Num=S|Cas=3|Neg=N|Var=8
N	SubPOS=N|Gen=I|Num=S|Cas=3|Neg=N|Var=9
N	SubPOS=N|Gen=I|Num=S|Cas=4|Neg=A
N	SubPOS=N|Gen=I|Num=S|Cas=4|Neg=A|Sem=G
N	SubPOS=N|Gen=I|Num=S|Cas=4|Neg=A|Sem=K
N	SubPOS=N|Gen=I|Num=S|Cas=4|Neg=A|Sem=L
N	SubPOS=N|Gen=I|Num=S|Cas=4|Neg=A|Sem=R
N	SubPOS=N|Gen=I|Num=S|Cas=4|Neg=A|Sem=U
N	SubPOS=N|Gen=I|Num=S|Cas=4|Neg=A|Sem=Y
N	SubPOS=N|Gen=I|Num=S|Cas=4|Neg=A|Sem=b
N	SubPOS=N|Gen=I|Num=S|Cas=4|Neg=A|Sem=m
N	SubPOS=N|Gen=I|Num=S|Cas=4|Neg=A|Sem=w
N	SubPOS=N|Gen=I|Num=S|Cas=4|Neg=A|Var=1
N	SubPOS=N|Gen=I|Num=S|Cas=4|Neg=A|Var=1|Sem=R
N	SubPOS=N|Gen=I|Num=S|Cas=4|Neg=A|Var=1|Sem=m
N	SubPOS=N|Gen=I|Num=S|Cas=4|Neg=A|Var=2
N	SubPOS=N|Gen=I|Num=S|Cas=4|Neg=A|Var=6
N	SubPOS=N|Gen=I|Num=S|Cas=4|Neg=A|Var=6|Sem=R
N	SubPOS=N|Gen=I|Num=S|Cas=4|Neg=A|Var=6|Sem=m
N	SubPOS=N|Gen=I|Num=S|Cas=4|Neg=A|Var=8
N	SubPOS=N|Gen=I|Num=S|Cas=4|Neg=N
N	SubPOS=N|Gen=I|Num=S|Cas=4|Neg=N|Var=1
N	SubPOS=N|Gen=I|Num=S|Cas=4|Neg=N|Var=2
N	SubPOS=N|Gen=I|Num=S|Cas=4|Neg=N|Var=6
N	SubPOS=N|Gen=I|Num=S|Cas=4|Neg=N|Var=8
N	SubPOS=N|Gen=I|Num=S|Cas=5|Neg=A
N	SubPOS=N|Gen=I|Num=S|Cas=5|Neg=A|Var=1
N	SubPOS=N|Gen=I|Num=S|Cas=5|Neg=A|Var=2
N	SubPOS=N|Gen=I|Num=S|Cas=5|Neg=A|Var=6
N	SubPOS=N|Gen=I|Num=S|Cas=5|Neg=A|Var=8
N	SubPOS=N|Gen=I|Num=S|Cas=5|Neg=A|Var=9
N	SubPOS=N|Gen=I|Num=S|Cas=5|Neg=N
N	SubPOS=N|Gen=I|Num=S|Cas=5|Neg=N|Var=1
N	SubPOS=N|Gen=I|Num=S|Cas=5|Neg=N|Var=2
N	SubPOS=N|Gen=I|Num=S|Cas=5|Neg=N|Var=6
N	SubPOS=N|Gen=I|Num=S|Cas=5|Neg=N|Var=8
N	SubPOS=N|Gen=I|Num=S|Cas=5|Neg=N|Var=9
N	SubPOS=N|Gen=I|Num=S|Cas=6|Neg=A
N	SubPOS=N|Gen=I|Num=S|Cas=6|Neg=A|Sem=G
N	SubPOS=N|Gen=I|Num=S|Cas=6|Neg=A|Sem=K
N	SubPOS=N|Gen=I|Num=S|Cas=6|Neg=A|Sem=R
N	SubPOS=N|Gen=I|Num=S|Cas=6|Neg=A|Sem=S
N	SubPOS=N|Gen=I|Num=S|Cas=6|Neg=A|Sem=Y
N	SubPOS=N|Gen=I|Num=S|Cas=6|Neg=A|Sem=m
N	SubPOS=N|Gen=I|Num=S|Cas=6|Neg=A|Sem=w
N	SubPOS=N|Gen=I|Num=S|Cas=6|Neg=A|Var=1
N	SubPOS=N|Gen=I|Num=S|Cas=6|Neg=A|Var=1|Sem=E
N	SubPOS=N|Gen=I|Num=S|Cas=6|Neg=A|Var=1|Sem=G
N	SubPOS=N|Gen=I|Num=S|Cas=6|Neg=A|Var=1|Sem=K
N	SubPOS=N|Gen=I|Num=S|Cas=6|Neg=A|Var=1|Sem=R
N	SubPOS=N|Gen=I|Num=S|Cas=6|Neg=A|Var=1|Sem=S
N	SubPOS=N|Gen=I|Num=S|Cas=6|Neg=A|Var=1|Sem=m
N	SubPOS=N|Gen=I|Num=S|Cas=6|Neg=A|Var=2
N	SubPOS=N|Gen=I|Num=S|Cas=6|Neg=A|Var=2|Sem=G
N	SubPOS=N|Gen=I|Num=S|Cas=6|Neg=A|Var=2|Sem=m
N	SubPOS=N|Gen=I|Num=S|Cas=6|Neg=A|Var=6
N	SubPOS=N|Gen=I|Num=S|Cas=6|Neg=A|Var=7
N	SubPOS=N|Gen=I|Num=S|Cas=6|Neg=A|Var=8
N	SubPOS=N|Gen=I|Num=S|Cas=6|Neg=A|Var=9
N	SubPOS=N|Gen=I|Num=S|Cas=6|Neg=N
N	SubPOS=N|Gen=I|Num=S|Cas=6|Neg=N|Var=1
N	SubPOS=N|Gen=I|Num=S|Cas=6|Neg=N|Var=2
N	SubPOS=N|Gen=I|Num=S|Cas=6|Neg=N|Var=6
N	SubPOS=N|Gen=I|Num=S|Cas=6|Neg=N|Var=7
N	SubPOS=N|Gen=I|Num=S|Cas=6|Neg=N|Var=8
N	SubPOS=N|Gen=I|Num=S|Cas=6|Neg=N|Var=9
N	SubPOS=N|Gen=I|Num=S|Cas=7|Neg=A
N	SubPOS=N|Gen=I|Num=S|Cas=7|Neg=A|Sem=G
N	SubPOS=N|Gen=I|Num=S|Cas=7|Neg=A|Sem=K
N	SubPOS=N|Gen=I|Num=S|Cas=7|Neg=A|Sem=R
N	SubPOS=N|Gen=I|Num=S|Cas=7|Neg=A|Sem=Y
N	SubPOS=N|Gen=I|Num=S|Cas=7|Neg=A|Sem=b
N	SubPOS=N|Gen=I|Num=S|Cas=7|Neg=A|Sem=m
N	SubPOS=N|Gen=I|Num=S|Cas=7|Neg=A|Var=1
N	SubPOS=N|Gen=I|Num=S|Cas=7|Neg=A|Var=2
N	SubPOS=N|Gen=I|Num=S|Cas=7|Neg=A|Var=6
N	SubPOS=N|Gen=I|Num=S|Cas=7|Neg=A|Var=8
N	SubPOS=N|Gen=I|Num=S|Cas=7|Neg=A|Var=9
N	SubPOS=N|Gen=I|Num=S|Cas=7|Neg=N
N	SubPOS=N|Gen=I|Num=S|Cas=7|Neg=N|Var=1
N	SubPOS=N|Gen=I|Num=S|Cas=7|Neg=N|Var=2
N	SubPOS=N|Gen=I|Num=S|Cas=7|Neg=N|Var=6
N	SubPOS=N|Gen=I|Num=S|Cas=7|Neg=N|Var=8
N	SubPOS=N|Gen=I|Num=S|Cas=7|Neg=N|Var=9
N	SubPOS=N|Gen=I|Num=S|Cas=X|Neg=A
N	SubPOS=N|Gen=I|Num=S|Cas=X|Neg=A|Sem=G
N	SubPOS=N|Gen=I|Num=S|Cas=X|Neg=A|Sem=K
N	SubPOS=N|Gen=I|Num=S|Cas=X|Neg=A|Sem=R
N	SubPOS=N|Gen=I|Num=S|Cas=X|Neg=A|Sem=m
N	SubPOS=N|Gen=I|Num=S|Cas=X|Neg=A|Var=8
N	SubPOS=N|Gen=I|Num=S|Cas=X|Neg=A|Var=8|Sem=K
N	SubPOS=N|Gen=I|Num=S|Cas=X|Neg=A|Var=8|Sem=m
N	SubPOS=N|Gen=I|Num=S|Cas=X|Neg=N
N	SubPOS=N|Gen=I|Num=S|Cas=X|Neg=N|Var=8
N	SubPOS=N|Gen=I|Num=X|Cas=X|Neg=A
N	SubPOS=N|Gen=I|Num=X|Cas=X|Neg=A|Sem=E
N	SubPOS=N|Gen=I|Num=X|Cas=X|Neg=A|Sem=G
N	SubPOS=N|Gen=I|Num=X|Cas=X|Neg=A|Sem=H
N	SubPOS=N|Gen=I|Num=X|Cas=X|Neg=A|Sem=K
N	SubPOS=N|Gen=I|Num=X|Cas=X|Neg=A|Sem=R
N	SubPOS=N|Gen=I|Num=X|Cas=X|Neg=A|Sem=S
N	SubPOS=N|Gen=I|Num=X|Cas=X|Neg=A|Sem=Y
N	SubPOS=N|Gen=I|Num=X|Cas=X|Neg=A|Sem=m
N	SubPOS=N|Gen=I|Num=X|Cas=X|Neg=A|Var=1
N	SubPOS=N|Gen=I|Num=X|Cas=X|Neg=A|Var=8
N	SubPOS=N|Gen=I|Num=X|Cas=X|Neg=A|Var=8|Sem=G
N	SubPOS=N|Gen=I|Num=X|Cas=X|Neg=A|Var=8|Sem=H
N	SubPOS=N|Gen=I|Num=X|Cas=X|Neg=A|Var=8|Sem=K
N	SubPOS=N|Gen=I|Num=X|Cas=X|Neg=A|Var=8|Sem=L
N	SubPOS=N|Gen=I|Num=X|Cas=X|Neg=A|Var=8|Sem=R
N	SubPOS=N|Gen=I|Num=X|Cas=X|Neg=A|Var=8|Sem=U
N	SubPOS=N|Gen=I|Num=X|Cas=X|Neg=A|Var=8|Sem=b
N	SubPOS=N|Gen=I|Num=X|Cas=X|Neg=A|Var=8|Sem=c
N	SubPOS=N|Gen=I|Num=X|Cas=X|Neg=A|Var=8|Sem=g
N	SubPOS=N|Gen=I|Num=X|Cas=X|Neg=A|Var=8|Sem=m
N	SubPOS=N|Gen=I|Num=X|Cas=X|Neg=A|Var=8|Sem=p
N	SubPOS=N|Gen=I|Num=X|Cas=X|Neg=N
N	SubPOS=N|Gen=I|Num=X|Cas=X|Neg=N|Var=1
N	SubPOS=N|Gen=I|Num=X|Cas=X|Neg=N|Var=8
N	SubPOS=N|Gen=M|Num=P|Cas=1|Neg=A
N	SubPOS=N|Gen=M|Num=P|Cas=1|Neg=A|Sem=E
N	SubPOS=N|Gen=M|Num=P|Cas=1|Neg=A|Sem=K
N	SubPOS=N|Gen=M|Num=P|Cas=1|Neg=A|Sem=S
N	SubPOS=N|Gen=M|Num=P|Cas=1|Neg=A|Sem=Y
N	SubPOS=N|Gen=M|Num=P|Cas=1|Neg=A|Sem=m
N	SubPOS=N|Gen=M|Num=P|Cas=1|Neg=A|Var=1
N	SubPOS=N|Gen=M|Num=P|Cas=1|Neg=A|Var=1|Sem=E
N	SubPOS=N|Gen=M|Num=P|Cas=1|Neg=A|Var=1|Sem=S
N	SubPOS=N|Gen=M|Num=P|Cas=1|Neg=A|Var=1|Sem=m
N	SubPOS=N|Gen=M|Num=P|Cas=1|Neg=A|Var=2
N	SubPOS=N|Gen=M|Num=P|Cas=1|Neg=A|Var=6
N	SubPOS=N|Gen=M|Num=P|Cas=1|Neg=A|Var=6|Sem=S
N	SubPOS=N|Gen=M|Num=P|Cas=1|Neg=A|Var=6|Sem=Y
N	SubPOS=N|Gen=M|Num=P|Cas=1|Neg=A|Var=6|Sem=m
N	SubPOS=N|Gen=M|Num=P|Cas=1|Neg=A|Var=7
N	SubPOS=N|Gen=M|Num=P|Cas=1|Neg=A|Var=8
N	SubPOS=N|Gen=M|Num=P|Cas=1|Neg=N
N	SubPOS=N|Gen=M|Num=P|Cas=1|Neg=N|Var=1
N	SubPOS=N|Gen=M|Num=P|Cas=1|Neg=N|Var=2
N	SubPOS=N|Gen=M|Num=P|Cas=1|Neg=N|Var=6
N	SubPOS=N|Gen=M|Num=P|Cas=1|Neg=N|Var=7
N	SubPOS=N|Gen=M|Num=P|Cas=1|Neg=N|Var=8
N	SubPOS=N|Gen=M|Num=P|Cas=2|Neg=A
N	SubPOS=N|Gen=M|Num=P|Cas=2|Neg=A|Sem=E
N	SubPOS=N|Gen=M|Num=P|Cas=2|Neg=A|Sem=G
N	SubPOS=N|Gen=M|Num=P|Cas=2|Neg=A|Sem=K
N	SubPOS=N|Gen=M|Num=P|Cas=2|Neg=A|Sem=S
N	SubPOS=N|Gen=M|Num=P|Cas=2|Neg=A|Sem=Y
N	SubPOS=N|Gen=M|Num=P|Cas=2|Neg=A|Sem=m
N	SubPOS=N|Gen=M|Num=P|Cas=2|Neg=A|Var=1
N	SubPOS=N|Gen=M|Num=P|Cas=2|Neg=A|Var=2
N	SubPOS=N|Gen=M|Num=P|Cas=2|Neg=A|Var=6
N	SubPOS=N|Gen=M|Num=P|Cas=2|Neg=A|Var=8
N	SubPOS=N|Gen=M|Num=P|Cas=2|Neg=N
N	SubPOS=N|Gen=M|Num=P|Cas=2|Neg=N|Var=1
N	SubPOS=N|Gen=M|Num=P|Cas=2|Neg=N|Var=2
N	SubPOS=N|Gen=M|Num=P|Cas=2|Neg=N|Var=6
N	SubPOS=N|Gen=M|Num=P|Cas=2|Neg=N|Var=8
N	SubPOS=N|Gen=M|Num=P|Cas=3|Neg=A
N	SubPOS=N|Gen=M|Num=P|Cas=3|Neg=A|Sem=E
N	SubPOS=N|Gen=M|Num=P|Cas=3|Neg=A|Sem=S
N	SubPOS=N|Gen=M|Num=P|Cas=3|Neg=A|Sem=m
N	SubPOS=N|Gen=M|Num=P|Cas=3|Neg=A|Var=1
N	SubPOS=N|Gen=M|Num=P|Cas=3|Neg=A|Var=2
N	SubPOS=N|Gen=M|Num=P|Cas=3|Neg=A|Var=6
N	SubPOS=N|Gen=M|Num=P|Cas=3|Neg=A|Var=7
N	SubPOS=N|Gen=M|Num=P|Cas=3|Neg=A|Var=8
N	SubPOS=N|Gen=M|Num=P|Cas=3|Neg=N
N	SubPOS=N|Gen=M|Num=P|Cas=3|Neg=N|Var=1
N	SubPOS=N|Gen=M|Num=P|Cas=3|Neg=N|Var=2
N	SubPOS=N|Gen=M|Num=P|Cas=3|Neg=N|Var=6
N	SubPOS=N|Gen=M|Num=P|Cas=3|Neg=N|Var=7
N	SubPOS=N|Gen=M|Num=P|Cas=3|Neg=N|Var=8
N	SubPOS=N|Gen=M|Num=P|Cas=4|Neg=A
N	SubPOS=N|Gen=M|Num=P|Cas=4|Neg=A|Sem=E
N	SubPOS=N|Gen=M|Num=P|Cas=4|Neg=A|Sem=S
N	SubPOS=N|Gen=M|Num=P|Cas=4|Neg=A|Sem=Y
N	SubPOS=N|Gen=M|Num=P|Cas=4|Neg=A|Sem=m
N	SubPOS=N|Gen=M|Num=P|Cas=4|Neg=A|Var=1
N	SubPOS=N|Gen=M|Num=P|Cas=4|Neg=A|Var=2
N	SubPOS=N|Gen=M|Num=P|Cas=4|Neg=A|Var=6
N	SubPOS=N|Gen=M|Num=P|Cas=4|Neg=A|Var=7
N	SubPOS=N|Gen=M|Num=P|Cas=4|Neg=A|Var=8
N	SubPOS=N|Gen=M|Num=P|Cas=4|Neg=N
N	SubPOS=N|Gen=M|Num=P|Cas=4|Neg=N|Var=1
N	SubPOS=N|Gen=M|Num=P|Cas=4|Neg=N|Var=2
N	SubPOS=N|Gen=M|Num=P|Cas=4|Neg=N|Var=6
N	SubPOS=N|Gen=M|Num=P|Cas=4|Neg=N|Var=7
N	SubPOS=N|Gen=M|Num=P|Cas=4|Neg=N|Var=8
N	SubPOS=N|Gen=M|Num=P|Cas=5|Neg=A
N	SubPOS=N|Gen=M|Num=P|Cas=5|Neg=A|Sem=E
N	SubPOS=N|Gen=M|Num=P|Cas=5|Neg=A|Sem=m
N	SubPOS=N|Gen=M|Num=P|Cas=5|Neg=A|Var=1
N	SubPOS=N|Gen=M|Num=P|Cas=5|Neg=A|Var=2
N	SubPOS=N|Gen=M|Num=P|Cas=5|Neg=A|Var=5
N	SubPOS=N|Gen=M|Num=P|Cas=5|Neg=A|Var=6
N	SubPOS=N|Gen=M|Num=P|Cas=5|Neg=A|Var=7
N	SubPOS=N|Gen=M|Num=P|Cas=5|Neg=A|Var=8
N	SubPOS=N|Gen=M|Num=P|Cas=5|Neg=N
N	SubPOS=N|Gen=M|Num=P|Cas=5|Neg=N|Var=1
N	SubPOS=N|Gen=M|Num=P|Cas=5|Neg=N|Var=2
N	SubPOS=N|Gen=M|Num=P|Cas=5|Neg=N|Var=5
N	SubPOS=N|Gen=M|Num=P|Cas=5|Neg=N|Var=6
N	SubPOS=N|Gen=M|Num=P|Cas=5|Neg=N|Var=7
N	SubPOS=N|Gen=M|Num=P|Cas=5|Neg=N|Var=8
N	SubPOS=N|Gen=M|Num=P|Cas=6|Neg=A
N	SubPOS=N|Gen=M|Num=P|Cas=6|Neg=A|Sem=E
N	SubPOS=N|Gen=M|Num=P|Cas=6|Neg=A|Sem=S
N	SubPOS=N|Gen=M|Num=P|Cas=6|Neg=A|Sem=Y
N	SubPOS=N|Gen=M|Num=P|Cas=6|Neg=A|Sem=m
N	SubPOS=N|Gen=M|Num=P|Cas=6|Neg=A|Var=1
N	SubPOS=N|Gen=M|Num=P|Cas=6|Neg=A|Var=2
N	SubPOS=N|Gen=M|Num=P|Cas=6|Neg=A|Var=6
N	SubPOS=N|Gen=M|Num=P|Cas=6|Neg=A|Var=7
N	SubPOS=N|Gen=M|Num=P|Cas=6|Neg=A|Var=8
N	SubPOS=N|Gen=M|Num=P|Cas=6|Neg=N
N	SubPOS=N|Gen=M|Num=P|Cas=6|Neg=N|Var=1
N	SubPOS=N|Gen=M|Num=P|Cas=6|Neg=N|Var=2
N	SubPOS=N|Gen=M|Num=P|Cas=6|Neg=N|Var=6
N	SubPOS=N|Gen=M|Num=P|Cas=6|Neg=N|Var=7
N	SubPOS=N|Gen=M|Num=P|Cas=6|Neg=N|Var=8
N	SubPOS=N|Gen=M|Num=P|Cas=7|Neg=A
N	SubPOS=N|Gen=M|Num=P|Cas=7|Neg=A|Sem=E
N	SubPOS=N|Gen=M|Num=P|Cas=7|Neg=A|Sem=S
N	SubPOS=N|Gen=M|Num=P|Cas=7|Neg=A|Sem=Y
N	SubPOS=N|Gen=M|Num=P|Cas=7|Neg=A|Sem=m
N	SubPOS=N|Gen=M|Num=P|Cas=7|Neg=A|Var=1
N	SubPOS=N|Gen=M|Num=P|Cas=7|Neg=A|Var=2
N	SubPOS=N|Gen=M|Num=P|Cas=7|Neg=A|Var=6
N	SubPOS=N|Gen=M|Num=P|Cas=7|Neg=A|Var=7
N	SubPOS=N|Gen=M|Num=P|Cas=7|Neg=A|Var=8
N	SubPOS=N|Gen=M|Num=P|Cas=7|Neg=N
N	SubPOS=N|Gen=M|Num=P|Cas=7|Neg=N|Var=1
N	SubPOS=N|Gen=M|Num=P|Cas=7|Neg=N|Var=2
N	SubPOS=N|Gen=M|Num=P|Cas=7|Neg=N|Var=6
N	SubPOS=N|Gen=M|Num=P|Cas=7|Neg=N|Var=7
N	SubPOS=N|Gen=M|Num=P|Cas=7|Neg=N|Var=8
N	SubPOS=N|Gen=M|Num=P|Cas=X|Neg=A
N	SubPOS=N|Gen=M|Num=P|Cas=X|Neg=A|Sem=K
N	SubPOS=N|Gen=M|Num=P|Cas=X|Neg=A|Sem=R
N	SubPOS=N|Gen=M|Num=P|Cas=X|Neg=A|Sem=Y
N	SubPOS=N|Gen=M|Num=P|Cas=X|Neg=A|Sem=m
N	SubPOS=N|Gen=M|Num=P|Cas=X|Neg=A|Var=8
N	SubPOS=N|Gen=M|Num=P|Cas=X|Neg=A|Var=8|Sem=K
N	SubPOS=N|Gen=M|Num=P|Cas=X|Neg=A|Var=8|Sem=m
N	SubPOS=N|Gen=M|Num=P|Cas=X|Neg=N
N	SubPOS=N|Gen=M|Num=P|Cas=X|Neg=N|Var=8
N	SubPOS=N|Gen=M|Num=S|Cas=1|Neg=A
N	SubPOS=N|Gen=M|Num=S|Cas=1|Neg=A|Sem=E
N	SubPOS=N|Gen=M|Num=S|Cas=1|Neg=A|Sem=G
N	SubPOS=N|Gen=M|Num=S|Cas=1|Neg=A|Sem=K
N	SubPOS=N|Gen=M|Num=S|Cas=1|Neg=A|Sem=R
N	SubPOS=N|Gen=M|Num=S|Cas=1|Neg=A|Sem=S
N	SubPOS=N|Gen=M|Num=S|Cas=1|Neg=A|Sem=Y
N	SubPOS=N|Gen=M|Num=S|Cas=1|Neg=A|Sem=m
N	SubPOS=N|Gen=M|Num=S|Cas=1|Neg=A|Sem=y
N	SubPOS=N|Gen=M|Num=S|Cas=1|Neg=A|Var=1
N	SubPOS=N|Gen=M|Num=S|Cas=1|Neg=A|Var=1|Sem=S
N	SubPOS=N|Gen=M|Num=S|Cas=1|Neg=A|Var=1|Sem=Y
N	SubPOS=N|Gen=M|Num=S|Cas=1|Neg=A|Var=1|Sem=m
N	SubPOS=N|Gen=M|Num=S|Cas=1|Neg=A|Var=2
N	SubPOS=N|Gen=M|Num=S|Cas=1|Neg=A|Var=6
N	SubPOS=N|Gen=M|Num=S|Cas=1|Neg=A|Var=8
N	SubPOS=N|Gen=M|Num=S|Cas=1|Neg=N
N	SubPOS=N|Gen=M|Num=S|Cas=1|Neg=N|Var=1
N	SubPOS=N|Gen=M|Num=S|Cas=1|Neg=N|Var=2
N	SubPOS=N|Gen=M|Num=S|Cas=1|Neg=N|Var=6
N	SubPOS=N|Gen=M|Num=S|Cas=1|Neg=N|Var=8
N	SubPOS=N|Gen=M|Num=S|Cas=2|Neg=A
N	SubPOS=N|Gen=M|Num=S|Cas=2|Neg=A|Sem=E
N	SubPOS=N|Gen=M|Num=S|Cas=2|Neg=A|Sem=G
N	SubPOS=N|Gen=M|Num=S|Cas=2|Neg=A|Sem=K
N	SubPOS=N|Gen=M|Num=S|Cas=2|Neg=A|Sem=S
N	SubPOS=N|Gen=M|Num=S|Cas=2|Neg=A|Sem=Y
N	SubPOS=N|Gen=M|Num=S|Cas=2|Neg=A|Sem=m
N	SubPOS=N|Gen=M|Num=S|Cas=2|Neg=A|Var=1
N	SubPOS=N|Gen=M|Num=S|Cas=2|Neg=A|Var=1|Sem=S
N	SubPOS=N|Gen=M|Num=S|Cas=2|Neg=A|Var=1|Sem=Y
N	SubPOS=N|Gen=M|Num=S|Cas=2|Neg=A|Var=1|Sem=m
N	SubPOS=N|Gen=M|Num=S|Cas=2|Neg=A|Var=2
N	SubPOS=N|Gen=M|Num=S|Cas=2|Neg=A|Var=2|Sem=S
N	SubPOS=N|Gen=M|Num=S|Cas=2|Neg=A|Var=2|Sem=Y
N	SubPOS=N|Gen=M|Num=S|Cas=2|Neg=A|Var=2|Sem=m
N	SubPOS=N|Gen=M|Num=S|Cas=2|Neg=A|Var=3
N	SubPOS=N|Gen=M|Num=S|Cas=2|Neg=A|Var=6
N	SubPOS=N|Gen=M|Num=S|Cas=2|Neg=A|Var=7
N	SubPOS=N|Gen=M|Num=S|Cas=2|Neg=A|Var=8
N	SubPOS=N|Gen=M|Num=S|Cas=2|Neg=N
N	SubPOS=N|Gen=M|Num=S|Cas=2|Neg=N|Var=1
N	SubPOS=N|Gen=M|Num=S|Cas=2|Neg=N|Var=2
N	SubPOS=N|Gen=M|Num=S|Cas=2|Neg=N|Var=3
N	SubPOS=N|Gen=M|Num=S|Cas=2|Neg=N|Var=6
N	SubPOS=N|Gen=M|Num=S|Cas=2|Neg=N|Var=7
N	SubPOS=N|Gen=M|Num=S|Cas=2|Neg=N|Var=8
N	SubPOS=N|Gen=M|Num=S|Cas=3|Neg=A
N	SubPOS=N|Gen=M|Num=S|Cas=3|Neg=A|Sem=E
N	SubPOS=N|Gen=M|Num=S|Cas=3|Neg=A|Sem=G
N	SubPOS=N|Gen=M|Num=S|Cas=3|Neg=A|Sem=S
N	SubPOS=N|Gen=M|Num=S|Cas=3|Neg=A|Sem=Y
N	SubPOS=N|Gen=M|Num=S|Cas=3|Neg=A|Sem=m
N	SubPOS=N|Gen=M|Num=S|Cas=3|Neg=A|Var=1
N	SubPOS=N|Gen=M|Num=S|Cas=3|Neg=A|Var=1|Sem=E
N	SubPOS=N|Gen=M|Num=S|Cas=3|Neg=A|Var=1|Sem=S
N	SubPOS=N|Gen=M|Num=S|Cas=3|Neg=A|Var=1|Sem=Y
N	SubPOS=N|Gen=M|Num=S|Cas=3|Neg=A|Var=1|Sem=m
N	SubPOS=N|Gen=M|Num=S|Cas=3|Neg=A|Var=2
N	SubPOS=N|Gen=M|Num=S|Cas=3|Neg=A|Var=3
N	SubPOS=N|Gen=M|Num=S|Cas=3|Neg=A|Var=6
N	SubPOS=N|Gen=M|Num=S|Cas=3|Neg=A|Var=7
N	SubPOS=N|Gen=M|Num=S|Cas=3|Neg=A|Var=8
N	SubPOS=N|Gen=M|Num=S|Cas=3|Neg=N
N	SubPOS=N|Gen=M|Num=S|Cas=3|Neg=N|Var=1
N	SubPOS=N|Gen=M|Num=S|Cas=3|Neg=N|Var=2
N	SubPOS=N|Gen=M|Num=S|Cas=3|Neg=N|Var=3
N	SubPOS=N|Gen=M|Num=S|Cas=3|Neg=N|Var=6
N	SubPOS=N|Gen=M|Num=S|Cas=3|Neg=N|Var=7
N	SubPOS=N|Gen=M|Num=S|Cas=3|Neg=N|Var=8
N	SubPOS=N|Gen=M|Num=S|Cas=4|Neg=A
N	SubPOS=N|Gen=M|Num=S|Cas=4|Neg=A|Sem=E
N	SubPOS=N|Gen=M|Num=S|Cas=4|Neg=A|Sem=G
N	SubPOS=N|Gen=M|Num=S|Cas=4|Neg=A|Sem=K
N	SubPOS=N|Gen=M|Num=S|Cas=4|Neg=A|Sem=R
N	SubPOS=N|Gen=M|Num=S|Cas=4|Neg=A|Sem=S
N	SubPOS=N|Gen=M|Num=S|Cas=4|Neg=A|Sem=Y
N	SubPOS=N|Gen=M|Num=S|Cas=4|Neg=A|Sem=m
N	SubPOS=N|Gen=M|Num=S|Cas=4|Neg=A|Var=1
N	SubPOS=N|Gen=M|Num=S|Cas=4|Neg=A|Var=1|Sem=Y
N	SubPOS=N|Gen=M|Num=S|Cas=4|Neg=A|Var=1|Sem=m
N	SubPOS=N|Gen=M|Num=S|Cas=4|Neg=A|Var=2
N	SubPOS=N|Gen=M|Num=S|Cas=4|Neg=A|Var=6
N	SubPOS=N|Gen=M|Num=S|Cas=4|Neg=A|Var=7
N	SubPOS=N|Gen=M|Num=S|Cas=4|Neg=A|Var=8
N	SubPOS=N|Gen=M|Num=S|Cas=4|Neg=N
N	SubPOS=N|Gen=M|Num=S|Cas=4|Neg=N|Var=1
N	SubPOS=N|Gen=M|Num=S|Cas=4|Neg=N|Var=2
N	SubPOS=N|Gen=M|Num=S|Cas=4|Neg=N|Var=6
N	SubPOS=N|Gen=M|Num=S|Cas=4|Neg=N|Var=7
N	SubPOS=N|Gen=M|Num=S|Cas=4|Neg=N|Var=8
N	SubPOS=N|Gen=M|Num=S|Cas=5|Neg=A
N	SubPOS=N|Gen=M|Num=S|Cas=5|Neg=A|Sem=S
N	SubPOS=N|Gen=M|Num=S|Cas=5|Neg=A|Sem=Y
N	SubPOS=N|Gen=M|Num=S|Cas=5|Neg=A|Sem=m
N	SubPOS=N|Gen=M|Num=S|Cas=5|Neg=A|Var=1
N	SubPOS=N|Gen=M|Num=S|Cas=5|Neg=A|Var=2
N	SubPOS=N|Gen=M|Num=S|Cas=5|Neg=A|Var=4
N	SubPOS=N|Gen=M|Num=S|Cas=5|Neg=A|Var=5
N	SubPOS=N|Gen=M|Num=S|Cas=5|Neg=A|Var=6
N	SubPOS=N|Gen=M|Num=S|Cas=5|Neg=A|Var=7
N	SubPOS=N|Gen=M|Num=S|Cas=5|Neg=A|Var=8
N	SubPOS=N|Gen=M|Num=S|Cas=5|Neg=N
N	SubPOS=N|Gen=M|Num=S|Cas=5|Neg=N|Var=1
N	SubPOS=N|Gen=M|Num=S|Cas=5|Neg=N|Var=2
N	SubPOS=N|Gen=M|Num=S|Cas=5|Neg=N|Var=4
N	SubPOS=N|Gen=M|Num=S|Cas=5|Neg=N|Var=5
N	SubPOS=N|Gen=M|Num=S|Cas=5|Neg=N|Var=6
N	SubPOS=N|Gen=M|Num=S|Cas=5|Neg=N|Var=7
N	SubPOS=N|Gen=M|Num=S|Cas=5|Neg=N|Var=8
N	SubPOS=N|Gen=M|Num=S|Cas=6|Neg=A
N	SubPOS=N|Gen=M|Num=S|Cas=6|Neg=A|Sem=E
N	SubPOS=N|Gen=M|Num=S|Cas=6|Neg=A|Sem=S
N	SubPOS=N|Gen=M|Num=S|Cas=6|Neg=A|Sem=Y
N	SubPOS=N|Gen=M|Num=S|Cas=6|Neg=A|Sem=m
N	SubPOS=N|Gen=M|Num=S|Cas=6|Neg=A|Var=1
N	SubPOS=N|Gen=M|Num=S|Cas=6|Neg=A|Var=1|Sem=G
N	SubPOS=N|Gen=M|Num=S|Cas=6|Neg=A|Var=1|Sem=S
N	SubPOS=N|Gen=M|Num=S|Cas=6|Neg=A|Var=1|Sem=Y
N	SubPOS=N|Gen=M|Num=S|Cas=6|Neg=A|Var=1|Sem=m
N	SubPOS=N|Gen=M|Num=S|Cas=6|Neg=A|Var=2
N	SubPOS=N|Gen=M|Num=S|Cas=6|Neg=A|Var=3
N	SubPOS=N|Gen=M|Num=S|Cas=6|Neg=A|Var=6
N	SubPOS=N|Gen=M|Num=S|Cas=6|Neg=A|Var=7
N	SubPOS=N|Gen=M|Num=S|Cas=6|Neg=A|Var=8
N	SubPOS=N|Gen=M|Num=S|Cas=6|Neg=N
N	SubPOS=N|Gen=M|Num=S|Cas=6|Neg=N|Var=1
N	SubPOS=N|Gen=M|Num=S|Cas=6|Neg=N|Var=2
N	SubPOS=N|Gen=M|Num=S|Cas=6|Neg=N|Var=3
N	SubPOS=N|Gen=M|Num=S|Cas=6|Neg=N|Var=6
N	SubPOS=N|Gen=M|Num=S|Cas=6|Neg=N|Var=7
N	SubPOS=N|Gen=M|Num=S|Cas=6|Neg=N|Var=8
N	SubPOS=N|Gen=M|Num=S|Cas=7|Neg=A
N	SubPOS=N|Gen=M|Num=S|Cas=7|Neg=A|Sem=E
N	SubPOS=N|Gen=M|Num=S|Cas=7|Neg=A|Sem=G
N	SubPOS=N|Gen=M|Num=S|Cas=7|Neg=A|Sem=K
N	SubPOS=N|Gen=M|Num=S|Cas=7|Neg=A|Sem=R
N	SubPOS=N|Gen=M|Num=S|Cas=7|Neg=A|Sem=S
N	SubPOS=N|Gen=M|Num=S|Cas=7|Neg=A|Sem=Y
N	SubPOS=N|Gen=M|Num=S|Cas=7|Neg=A|Sem=m
N	SubPOS=N|Gen=M|Num=S|Cas=7|Neg=A|Var=1
N	SubPOS=N|Gen=M|Num=S|Cas=7|Neg=A|Var=2
N	SubPOS=N|Gen=M|Num=S|Cas=7|Neg=A|Var=6
N	SubPOS=N|Gen=M|Num=S|Cas=7|Neg=A|Var=8
N	SubPOS=N|Gen=M|Num=S|Cas=7|Neg=N
N	SubPOS=N|Gen=M|Num=S|Cas=7|Neg=N|Var=1
N	SubPOS=N|Gen=M|Num=S|Cas=7|Neg=N|Var=2
N	SubPOS=N|Gen=M|Num=S|Cas=7|Neg=N|Var=6
N	SubPOS=N|Gen=M|Num=S|Cas=7|Neg=N|Var=8
N	SubPOS=N|Gen=M|Num=S|Cas=X|Neg=A
N	SubPOS=N|Gen=M|Num=S|Cas=X|Neg=A|Sem=G
N	SubPOS=N|Gen=M|Num=S|Cas=X|Neg=A|Sem=K
N	SubPOS=N|Gen=M|Num=S|Cas=X|Neg=A|Sem=S
N	SubPOS=N|Gen=M|Num=S|Cas=X|Neg=A|Sem=Y
N	SubPOS=N|Gen=M|Num=S|Cas=X|Neg=A|Sem=m
N	SubPOS=N|Gen=M|Num=S|Cas=X|Neg=A|Var=1
N	SubPOS=N|Gen=M|Num=S|Cas=X|Neg=A|Var=8
N	SubPOS=N|Gen=M|Num=S|Cas=X|Neg=N
N	SubPOS=N|Gen=M|Num=S|Cas=X|Neg=N|Var=1
N	SubPOS=N|Gen=M|Num=S|Cas=X|Neg=N|Var=8
N	SubPOS=N|Gen=M|Num=X|Cas=X|Neg=A
N	SubPOS=N|Gen=M|Num=X|Cas=X|Neg=A|Sem=E
N	SubPOS=N|Gen=M|Num=X|Cas=X|Neg=A|Sem=G
N	SubPOS=N|Gen=M|Num=X|Cas=X|Neg=A|Sem=K
N	SubPOS=N|Gen=M|Num=X|Cas=X|Neg=A|Sem=R
N	SubPOS=N|Gen=M|Num=X|Cas=X|Neg=A|Sem=S
N	SubPOS=N|Gen=M|Num=X|Cas=X|Neg=A|Sem=Y
N	SubPOS=N|Gen=M|Num=X|Cas=X|Neg=A|Sem=m
N	SubPOS=N|Gen=M|Num=X|Cas=X|Neg=A|Var=1
N	SubPOS=N|Gen=M|Num=X|Cas=X|Neg=A|Var=8
N	SubPOS=N|Gen=M|Num=X|Cas=X|Neg=A|Var=8|Sem=G
N	SubPOS=N|Gen=M|Num=X|Cas=X|Neg=A|Var=8|Sem=K
N	SubPOS=N|Gen=M|Num=X|Cas=X|Neg=A|Var=8|Sem=S
N	SubPOS=N|Gen=M|Num=X|Cas=X|Neg=A|Var=8|Sem=Y
N	SubPOS=N|Gen=M|Num=X|Cas=X|Neg=A|Var=8|Sem=m
N	SubPOS=N|Gen=M|Num=X|Cas=X|Neg=N
N	SubPOS=N|Gen=M|Num=X|Cas=X|Neg=N|Var=1
N	SubPOS=N|Gen=M|Num=X|Cas=X|Neg=N|Var=8
N	SubPOS=N|Gen=N|Num=P|Cas=1|Neg=A
N	SubPOS=N|Gen=N|Num=P|Cas=1|Neg=A|Sem=G
N	SubPOS=N|Gen=N|Num=P|Cas=1|Neg=A|Sem=K
N	SubPOS=N|Gen=N|Num=P|Cas=1|Neg=A|Sem=R
N	SubPOS=N|Gen=N|Num=P|Cas=1|Neg=A|Sem=U
N	SubPOS=N|Gen=N|Num=P|Cas=1|Neg=A|Sem=m
N	SubPOS=N|Gen=N|Num=P|Cas=1|Neg=A|Var=1
N	SubPOS=N|Gen=N|Num=P|Cas=1|Neg=A|Var=2
N	SubPOS=N|Gen=N|Num=P|Cas=1|Neg=A|Var=3
N	SubPOS=N|Gen=N|Num=P|Cas=1|Neg=A|Var=6
N	SubPOS=N|Gen=N|Num=P|Cas=1|Neg=A|Var=8
N	SubPOS=N|Gen=N|Num=P|Cas=1|Neg=N
N	SubPOS=N|Gen=N|Num=P|Cas=1|Neg=N|Var=1
N	SubPOS=N|Gen=N|Num=P|Cas=1|Neg=N|Var=2
N	SubPOS=N|Gen=N|Num=P|Cas=1|Neg=N|Var=3
N	SubPOS=N|Gen=N|Num=P|Cas=1|Neg=N|Var=6
N	SubPOS=N|Gen=N|Num=P|Cas=1|Neg=N|Var=8
N	SubPOS=N|Gen=N|Num=P|Cas=2|Neg=A
N	SubPOS=N|Gen=N|Num=P|Cas=2|Neg=A|Sem=G
N	SubPOS=N|Gen=N|Num=P|Cas=2|Neg=A|Sem=K
N	SubPOS=N|Gen=N|Num=P|Cas=2|Neg=A|Sem=R
N	SubPOS=N|Gen=N|Num=P|Cas=2|Neg=A|Sem=U
N	SubPOS=N|Gen=N|Num=P|Cas=2|Neg=A|Sem=m
N	SubPOS=N|Gen=N|Num=P|Cas=2|Neg=A|Var=1
N	SubPOS=N|Gen=N|Num=P|Cas=2|Neg=A|Var=2
N	SubPOS=N|Gen=N|Num=P|Cas=2|Neg=A|Var=3
N	SubPOS=N|Gen=N|Num=P|Cas=2|Neg=A|Var=6
N	SubPOS=N|Gen=N|Num=P|Cas=2|Neg=A|Var=8
N	SubPOS=N|Gen=N|Num=P|Cas=2|Neg=N
N	SubPOS=N|Gen=N|Num=P|Cas=2|Neg=N|Var=1
N	SubPOS=N|Gen=N|Num=P|Cas=2|Neg=N|Var=2
N	SubPOS=N|Gen=N|Num=P|Cas=2|Neg=N|Var=3
N	SubPOS=N|Gen=N|Num=P|Cas=2|Neg=N|Var=6
N	SubPOS=N|Gen=N|Num=P|Cas=2|Neg=N|Var=8
N	SubPOS=N|Gen=N|Num=P|Cas=3|Neg=A
N	SubPOS=N|Gen=N|Num=P|Cas=3|Neg=A|Sem=m
N	SubPOS=N|Gen=N|Num=P|Cas=3|Neg=A|Var=1
N	SubPOS=N|Gen=N|Num=P|Cas=3|Neg=A|Var=2
N	SubPOS=N|Gen=N|Num=P|Cas=3|Neg=A|Var=3
N	SubPOS=N|Gen=N|Num=P|Cas=3|Neg=A|Var=6
N	SubPOS=N|Gen=N|Num=P|Cas=3|Neg=A|Var=7
N	SubPOS=N|Gen=N|Num=P|Cas=3|Neg=A|Var=8
N	SubPOS=N|Gen=N|Num=P|Cas=3|Neg=N
N	SubPOS=N|Gen=N|Num=P|Cas=3|Neg=N|Var=1
N	SubPOS=N|Gen=N|Num=P|Cas=3|Neg=N|Var=2
N	SubPOS=N|Gen=N|Num=P|Cas=3|Neg=N|Var=3
N	SubPOS=N|Gen=N|Num=P|Cas=3|Neg=N|Var=6
N	SubPOS=N|Gen=N|Num=P|Cas=3|Neg=N|Var=7
N	SubPOS=N|Gen=N|Num=P|Cas=3|Neg=N|Var=8
N	SubPOS=N|Gen=N|Num=P|Cas=4|Neg=A
N	SubPOS=N|Gen=N|Num=P|Cas=4|Neg=A|Sem=m
N	SubPOS=N|Gen=N|Num=P|Cas=4|Neg=A|Var=1
N	SubPOS=N|Gen=N|Num=P|Cas=4|Neg=A|Var=2
N	SubPOS=N|Gen=N|Num=P|Cas=4|Neg=A|Var=3
N	SubPOS=N|Gen=N|Num=P|Cas=4|Neg=A|Var=6
N	SubPOS=N|Gen=N|Num=P|Cas=4|Neg=A|Var=8
N	SubPOS=N|Gen=N|Num=P|Cas=4|Neg=N
N	SubPOS=N|Gen=N|Num=P|Cas=4|Neg=N|Var=1
N	SubPOS=N|Gen=N|Num=P|Cas=4|Neg=N|Var=2
N	SubPOS=N|Gen=N|Num=P|Cas=4|Neg=N|Var=3
N	SubPOS=N|Gen=N|Num=P|Cas=4|Neg=N|Var=6
N	SubPOS=N|Gen=N|Num=P|Cas=4|Neg=N|Var=8
N	SubPOS=N|Gen=N|Num=P|Cas=5|Neg=A
N	SubPOS=N|Gen=N|Num=P|Cas=5|Neg=A|Sem=m
N	SubPOS=N|Gen=N|Num=P|Cas=5|Neg=A|Var=1
N	SubPOS=N|Gen=N|Num=P|Cas=5|Neg=A|Var=2
N	SubPOS=N|Gen=N|Num=P|Cas=5|Neg=A|Var=3
N	SubPOS=N|Gen=N|Num=P|Cas=5|Neg=A|Var=6
N	SubPOS=N|Gen=N|Num=P|Cas=5|Neg=A|Var=8
N	SubPOS=N|Gen=N|Num=P|Cas=5|Neg=N
N	SubPOS=N|Gen=N|Num=P|Cas=5|Neg=N|Var=1
N	SubPOS=N|Gen=N|Num=P|Cas=5|Neg=N|Var=2
N	SubPOS=N|Gen=N|Num=P|Cas=5|Neg=N|Var=3
N	SubPOS=N|Gen=N|Num=P|Cas=5|Neg=N|Var=6
N	SubPOS=N|Gen=N|Num=P|Cas=5|Neg=N|Var=8
N	SubPOS=N|Gen=N|Num=P|Cas=6|Neg=A
N	SubPOS=N|Gen=N|Num=P|Cas=6|Neg=A|Sem=m
N	SubPOS=N|Gen=N|Num=P|Cas=6|Neg=A|Var=1
N	SubPOS=N|Gen=N|Num=P|Cas=6|Neg=A|Var=2
N	SubPOS=N|Gen=N|Num=P|Cas=6|Neg=A|Var=3
N	SubPOS=N|Gen=N|Num=P|Cas=6|Neg=A|Var=6
N	SubPOS=N|Gen=N|Num=P|Cas=6|Neg=A|Var=8
N	SubPOS=N|Gen=N|Num=P|Cas=6|Neg=N
N	SubPOS=N|Gen=N|Num=P|Cas=6|Neg=N|Var=1
N	SubPOS=N|Gen=N|Num=P|Cas=6|Neg=N|Var=2
N	SubPOS=N|Gen=N|Num=P|Cas=6|Neg=N|Var=3
N	SubPOS=N|Gen=N|Num=P|Cas=6|Neg=N|Var=6
N	SubPOS=N|Gen=N|Num=P|Cas=6|Neg=N|Var=8
N	SubPOS=N|Gen=N|Num=P|Cas=7|Neg=A
N	SubPOS=N|Gen=N|Num=P|Cas=7|Neg=A|Sem=G
N	SubPOS=N|Gen=N|Num=P|Cas=7|Neg=A|Sem=m
N	SubPOS=N|Gen=N|Num=P|Cas=7|Neg=A|Var=1
N	SubPOS=N|Gen=N|Num=P|Cas=7|Neg=A|Var=2
N	SubPOS=N|Gen=N|Num=P|Cas=7|Neg=A|Var=3
N	SubPOS=N|Gen=N|Num=P|Cas=7|Neg=A|Var=6
N	SubPOS=N|Gen=N|Num=P|Cas=7|Neg=A|Var=7
N	SubPOS=N|Gen=N|Num=P|Cas=7|Neg=A|Var=8
N	SubPOS=N|Gen=N|Num=P|Cas=7|Neg=N
N	SubPOS=N|Gen=N|Num=P|Cas=7|Neg=N|Var=1
N	SubPOS=N|Gen=N|Num=P|Cas=7|Neg=N|Var=2
N	SubPOS=N|Gen=N|Num=P|Cas=7|Neg=N|Var=3
N	SubPOS=N|Gen=N|Num=P|Cas=7|Neg=N|Var=6
N	SubPOS=N|Gen=N|Num=P|Cas=7|Neg=N|Var=7
N	SubPOS=N|Gen=N|Num=P|Cas=7|Neg=N|Var=8
N	SubPOS=N|Gen=N|Num=P|Cas=X|Neg=A
N	SubPOS=N|Gen=N|Num=P|Cas=X|Neg=A|Sem=K
N	SubPOS=N|Gen=N|Num=P|Cas=X|Neg=A|Sem=m
N	SubPOS=N|Gen=N|Num=P|Cas=X|Neg=A|Var=1
N	SubPOS=N|Gen=N|Num=P|Cas=X|Neg=A|Var=8
N	SubPOS=N|Gen=N|Num=P|Cas=X|Neg=A|Var=8|Sem=K
N	SubPOS=N|Gen=N|Num=P|Cas=X|Neg=A|Var=8|Sem=m
N	SubPOS=N|Gen=N|Num=P|Cas=X|Neg=N
N	SubPOS=N|Gen=N|Num=P|Cas=X|Neg=N|Var=1
N	SubPOS=N|Gen=N|Num=P|Cas=X|Neg=N|Var=8
N	SubPOS=N|Gen=N|Num=S|Cas=1|Neg=A
N	SubPOS=N|Gen=N|Num=S|Cas=1|Neg=A|Sem=G
N	SubPOS=N|Gen=N|Num=S|Cas=1|Neg=A|Sem=K
N	SubPOS=N|Gen=N|Num=S|Cas=1|Neg=A|Sem=R
N	SubPOS=N|Gen=N|Num=S|Cas=1|Neg=A|Sem=S
N	SubPOS=N|Gen=N|Num=S|Cas=1|Neg=A|Sem=U
N	SubPOS=N|Gen=N|Num=S|Cas=1|Neg=A|Sem=Y
N	SubPOS=N|Gen=N|Num=S|Cas=1|Neg=A|Sem=m
N	SubPOS=N|Gen=N|Num=S|Cas=1|Neg=A|Var=1
N	SubPOS=N|Gen=N|Num=S|Cas=1|Neg=A|Var=2
N	SubPOS=N|Gen=N|Num=S|Cas=1|Neg=A|Var=6
N	SubPOS=N|Gen=N|Num=S|Cas=1|Neg=A|Var=8
N	SubPOS=N|Gen=N|Num=S|Cas=1|Neg=N
N	SubPOS=N|Gen=N|Num=S|Cas=1|Neg=N|Var=1
N	SubPOS=N|Gen=N|Num=S|Cas=1|Neg=N|Var=2
N	SubPOS=N|Gen=N|Num=S|Cas=1|Neg=N|Var=6
N	SubPOS=N|Gen=N|Num=S|Cas=1|Neg=N|Var=8
N	SubPOS=N|Gen=N|Num=S|Cas=2|Neg=A
N	SubPOS=N|Gen=N|Num=S|Cas=2|Neg=A|Sem=G
N	SubPOS=N|Gen=N|Num=S|Cas=2|Neg=A|Sem=H
N	SubPOS=N|Gen=N|Num=S|Cas=2|Neg=A|Sem=K
N	SubPOS=N|Gen=N|Num=S|Cas=2|Neg=A|Sem=R
N	SubPOS=N|Gen=N|Num=S|Cas=2|Neg=A|Sem=Y
N	SubPOS=N|Gen=N|Num=S|Cas=2|Neg=A|Sem=m
N	SubPOS=N|Gen=N|Num=S|Cas=2|Neg=A|Var=1
N	SubPOS=N|Gen=N|Num=S|Cas=2|Neg=A|Var=2
N	SubPOS=N|Gen=N|Num=S|Cas=2|Neg=A|Var=3
N	SubPOS=N|Gen=N|Num=S|Cas=2|Neg=A|Var=6
N	SubPOS=N|Gen=N|Num=S|Cas=2|Neg=A|Var=8
N	SubPOS=N|Gen=N|Num=S|Cas=2|Neg=N
N	SubPOS=N|Gen=N|Num=S|Cas=2|Neg=N|Var=1
N	SubPOS=N|Gen=N|Num=S|Cas=2|Neg=N|Var=2
N	SubPOS=N|Gen=N|Num=S|Cas=2|Neg=N|Var=3
N	SubPOS=N|Gen=N|Num=S|Cas=2|Neg=N|Var=6
N	SubPOS=N|Gen=N|Num=S|Cas=2|Neg=N|Var=8
N	SubPOS=N|Gen=N|Num=S|Cas=3|Neg=A
N	SubPOS=N|Gen=N|Num=S|Cas=3|Neg=A|Sem=G
N	SubPOS=N|Gen=N|Num=S|Cas=3|Neg=A|Sem=K
N	SubPOS=N|Gen=N|Num=S|Cas=3|Neg=A|Sem=Y
N	SubPOS=N|Gen=N|Num=S|Cas=3|Neg=A|Sem=m
N	SubPOS=N|Gen=N|Num=S|Cas=3|Neg=A|Var=1
N	SubPOS=N|Gen=N|Num=S|Cas=3|Neg=A|Var=2
N	SubPOS=N|Gen=N|Num=S|Cas=3|Neg=A|Var=3
N	SubPOS=N|Gen=N|Num=S|Cas=3|Neg=A|Var=6
N	SubPOS=N|Gen=N|Num=S|Cas=3|Neg=A|Var=8
N	SubPOS=N|Gen=N|Num=S|Cas=3|Neg=N
N	SubPOS=N|Gen=N|Num=S|Cas=3|Neg=N|Var=1
N	SubPOS=N|Gen=N|Num=S|Cas=3|Neg=N|Var=2
N	SubPOS=N|Gen=N|Num=S|Cas=3|Neg=N|Var=3
N	SubPOS=N|Gen=N|Num=S|Cas=3|Neg=N|Var=6
N	SubPOS=N|Gen=N|Num=S|Cas=3|Neg=N|Var=8
N	SubPOS=N|Gen=N|Num=S|Cas=4|Neg=A
N	SubPOS=N|Gen=N|Num=S|Cas=4|Neg=A|Sem=G
N	SubPOS=N|Gen=N|Num=S|Cas=4|Neg=A|Sem=K
N	SubPOS=N|Gen=N|Num=S|Cas=4|Neg=A|Sem=R
N	SubPOS=N|Gen=N|Num=S|Cas=4|Neg=A|Sem=m
N	SubPOS=N|Gen=N|Num=S|Cas=4|Neg=A|Sem=o
N	SubPOS=N|Gen=N|Num=S|Cas=4|Neg=A|Var=1
N	SubPOS=N|Gen=N|Num=S|Cas=4|Neg=A|Var=2
N	SubPOS=N|Gen=N|Num=S|Cas=4|Neg=A|Var=6
N	SubPOS=N|Gen=N|Num=S|Cas=4|Neg=A|Var=8
N	SubPOS=N|Gen=N|Num=S|Cas=4|Neg=N
N	SubPOS=N|Gen=N|Num=S|Cas=4|Neg=N|Var=1
N	SubPOS=N|Gen=N|Num=S|Cas=4|Neg=N|Var=2
N	SubPOS=N|Gen=N|Num=S|Cas=4|Neg=N|Var=6
N	SubPOS=N|Gen=N|Num=S|Cas=4|Neg=N|Var=8
N	SubPOS=N|Gen=N|Num=S|Cas=5|Neg=A
N	SubPOS=N|Gen=N|Num=S|Cas=5|Neg=A|Sem=G
N	SubPOS=N|Gen=N|Num=S|Cas=5|Neg=A|Sem=m
N	SubPOS=N|Gen=N|Num=S|Cas=5|Neg=A|Var=1
N	SubPOS=N|Gen=N|Num=S|Cas=5|Neg=A|Var=2
N	SubPOS=N|Gen=N|Num=S|Cas=5|Neg=A|Var=6
N	SubPOS=N|Gen=N|Num=S|Cas=5|Neg=A|Var=8
N	SubPOS=N|Gen=N|Num=S|Cas=5|Neg=N
N	SubPOS=N|Gen=N|Num=S|Cas=5|Neg=N|Var=1
N	SubPOS=N|Gen=N|Num=S|Cas=5|Neg=N|Var=2
N	SubPOS=N|Gen=N|Num=S|Cas=5|Neg=N|Var=6
N	SubPOS=N|Gen=N|Num=S|Cas=5|Neg=N|Var=8
N	SubPOS=N|Gen=N|Num=S|Cas=6|Neg=A
N	SubPOS=N|Gen=N|Num=S|Cas=6|Neg=A|Sem=G
N	SubPOS=N|Gen=N|Num=S|Cas=6|Neg=A|Sem=K
N	SubPOS=N|Gen=N|Num=S|Cas=6|Neg=A|Sem=R
N	SubPOS=N|Gen=N|Num=S|Cas=6|Neg=A|Sem=S
N	SubPOS=N|Gen=N|Num=S|Cas=6|Neg=A|Sem=U
N	SubPOS=N|Gen=N|Num=S|Cas=6|Neg=A|Sem=Y
N	SubPOS=N|Gen=N|Num=S|Cas=6|Neg=A|Sem=m
N	SubPOS=N|Gen=N|Num=S|Cas=6|Neg=A|Var=1
N	SubPOS=N|Gen=N|Num=S|Cas=6|Neg=A|Var=1|Sem=G
N	SubPOS=N|Gen=N|Num=S|Cas=6|Neg=A|Var=1|Sem=m
N	SubPOS=N|Gen=N|Num=S|Cas=6|Neg=A|Var=2
N	SubPOS=N|Gen=N|Num=S|Cas=6|Neg=A|Var=6
N	SubPOS=N|Gen=N|Num=S|Cas=6|Neg=A|Var=7
N	SubPOS=N|Gen=N|Num=S|Cas=6|Neg=A|Var=8
N	SubPOS=N|Gen=N|Num=S|Cas=6|Neg=N
N	SubPOS=N|Gen=N|Num=S|Cas=6|Neg=N|Var=1
N	SubPOS=N|Gen=N|Num=S|Cas=6|Neg=N|Var=2
N	SubPOS=N|Gen=N|Num=S|Cas=6|Neg=N|Var=6
N	SubPOS=N|Gen=N|Num=S|Cas=6|Neg=N|Var=7
N	SubPOS=N|Gen=N|Num=S|Cas=6|Neg=N|Var=8
N	SubPOS=N|Gen=N|Num=S|Cas=7|Neg=A
N	SubPOS=N|Gen=N|Num=S|Cas=7|Neg=A|Sem=G
N	SubPOS=N|Gen=N|Num=S|Cas=7|Neg=A|Sem=K
N	SubPOS=N|Gen=N|Num=S|Cas=7|Neg=A|Sem=S
N	SubPOS=N|Gen=N|Num=S|Cas=7|Neg=A|Sem=m
N	SubPOS=N|Gen=N|Num=S|Cas=7|Neg=A|Var=1
N	SubPOS=N|Gen=N|Num=S|Cas=7|Neg=A|Var=2
N	SubPOS=N|Gen=N|Num=S|Cas=7|Neg=A|Var=6
N	SubPOS=N|Gen=N|Num=S|Cas=7|Neg=A|Var=8
N	SubPOS=N|Gen=N|Num=S|Cas=7|Neg=A|Var=8|Sem=G
N	SubPOS=N|Gen=N|Num=S|Cas=7|Neg=A|Var=8|Sem=m
N	SubPOS=N|Gen=N|Num=S|Cas=7|Neg=N
N	SubPOS=N|Gen=N|Num=S|Cas=7|Neg=N|Var=1
N	SubPOS=N|Gen=N|Num=S|Cas=7|Neg=N|Var=2
N	SubPOS=N|Gen=N|Num=S|Cas=7|Neg=N|Var=6
N	SubPOS=N|Gen=N|Num=S|Cas=7|Neg=N|Var=8
N	SubPOS=N|Gen=N|Num=S|Cas=X|Neg=A
N	SubPOS=N|Gen=N|Num=S|Cas=X|Neg=A|Sem=E
N	SubPOS=N|Gen=N|Num=S|Cas=X|Neg=A|Sem=G
N	SubPOS=N|Gen=N|Num=S|Cas=X|Neg=A|Sem=K
N	SubPOS=N|Gen=N|Num=S|Cas=X|Neg=A|Sem=m
N	SubPOS=N|Gen=N|Num=S|Cas=X|Neg=A|Var=8
N	SubPOS=N|Gen=N|Num=S|Cas=X|Neg=N
N	SubPOS=N|Gen=N|Num=S|Cas=X|Neg=N|Var=8
N	SubPOS=N|Gen=N|Num=X|Cas=X|Neg=A
N	SubPOS=N|Gen=N|Num=X|Cas=X|Neg=A|Sem=G
N	SubPOS=N|Gen=N|Num=X|Cas=X|Neg=A|Sem=K
N	SubPOS=N|Gen=N|Num=X|Cas=X|Neg=A|Sem=R
N	SubPOS=N|Gen=N|Num=X|Cas=X|Neg=A|Sem=S
N	SubPOS=N|Gen=N|Num=X|Cas=X|Neg=A|Sem=Y
N	SubPOS=N|Gen=N|Num=X|Cas=X|Neg=A|Sem=m
N	SubPOS=N|Gen=N|Num=X|Cas=X|Neg=A|Sem=w
N	SubPOS=N|Gen=N|Num=X|Cas=X|Neg=A|Var=1
N	SubPOS=N|Gen=N|Num=X|Cas=X|Neg=A|Var=2
N	SubPOS=N|Gen=N|Num=X|Cas=X|Neg=A|Var=2|Sem=G
N	SubPOS=N|Gen=N|Num=X|Cas=X|Neg=A|Var=2|Sem=m
N	SubPOS=N|Gen=N|Num=X|Cas=X|Neg=A|Var=8
N	SubPOS=N|Gen=N|Num=X|Cas=X|Neg=A|Var=8|Sem=G
N	SubPOS=N|Gen=N|Num=X|Cas=X|Neg=A|Var=8|Sem=K
N	SubPOS=N|Gen=N|Num=X|Cas=X|Neg=A|Var=8|Sem=R
N	SubPOS=N|Gen=N|Num=X|Cas=X|Neg=A|Var=8|Sem=g
N	SubPOS=N|Gen=N|Num=X|Cas=X|Neg=A|Var=8|Sem=m
N	SubPOS=N|Gen=N|Num=X|Cas=X|Neg=A|Var=8|Sem=w
N	SubPOS=N|Gen=N|Num=X|Cas=X|Neg=A|Var=8|Sem=y
N	SubPOS=N|Gen=N|Num=X|Cas=X|Neg=A|Var=9
N	SubPOS=N|Gen=N|Num=X|Cas=X|Neg=N
N	SubPOS=N|Gen=N|Num=X|Cas=X|Neg=N|Var=1
N	SubPOS=N|Gen=N|Num=X|Cas=X|Neg=N|Var=2
N	SubPOS=N|Gen=N|Num=X|Cas=X|Neg=N|Var=8
N	SubPOS=N|Gen=N|Num=X|Cas=X|Neg=N|Var=9
N	SubPOS=N|Gen=X|Num=P|Cas=X|Neg=A
N	SubPOS=N|Gen=X|Num=P|Cas=X|Neg=A|Sem=K
N	SubPOS=N|Gen=X|Num=P|Cas=X|Neg=A|Sem=m
N	SubPOS=N|Gen=X|Num=P|Cas=X|Neg=N
N	SubPOS=N|Gen=X|Num=S|Cas=X|Neg=A
N	SubPOS=N|Gen=X|Num=S|Cas=X|Neg=A|Sem=K
N	SubPOS=N|Gen=X|Num=S|Cas=X|Neg=A|Sem=R
N	SubPOS=N|Gen=X|Num=S|Cas=X|Neg=A|Sem=m
N	SubPOS=N|Gen=X|Num=S|Cas=X|Neg=N
N	SubPOS=N|Gen=X|Num=X|Cas=X|Neg=A
N	SubPOS=N|Gen=X|Num=X|Cas=X|Neg=A|Sem=G
N	SubPOS=N|Gen=X|Num=X|Cas=X|Neg=A|Sem=K
N	SubPOS=N|Gen=X|Num=X|Cas=X|Neg=A|Sem=R
N	SubPOS=N|Gen=X|Num=X|Cas=X|Neg=A|Sem=S
N	SubPOS=N|Gen=X|Num=X|Cas=X|Neg=A|Sem=Y
N	SubPOS=N|Gen=X|Num=X|Cas=X|Neg=A|Sem=m
N	SubPOS=N|Gen=X|Num=X|Cas=X|Neg=A|Var=8
N	SubPOS=N|Gen=X|Num=X|Cas=X|Neg=A|Var=8|Sem=G
N	SubPOS=N|Gen=X|Num=X|Cas=X|Neg=A|Var=8|Sem=K
N	SubPOS=N|Gen=X|Num=X|Cas=X|Neg=A|Var=8|Sem=R
N	SubPOS=N|Gen=X|Num=X|Cas=X|Neg=A|Var=8|Sem=S
N	SubPOS=N|Gen=X|Num=X|Cas=X|Neg=A|Var=8|Sem=Y
N	SubPOS=N|Gen=X|Num=X|Cas=X|Neg=A|Var=8|Sem=m
N	SubPOS=N|Gen=X|Num=X|Cas=X|Neg=N
N	SubPOS=N|Gen=X|Num=X|Cas=X|Neg=N|Var=8
P	SubPOS=0
P	SubPOS=1|Gen=F|Num=D|Cas=7|PGe=F|PNu=S|Per=3
P	SubPOS=1|Gen=F|Num=D|Cas=7|PGe=F|PNu=S|Per=3|Var=2
P	SubPOS=1|Gen=F|Num=S|Cas=X|PGe=F|PNu=S|Per=3
P	SubPOS=1|Gen=F|Num=S|Cas=X|PGe=F|PNu=S|Per=3|Var=2
P	SubPOS=1|Gen=I|Num=S|Cas=4|PGe=F|PNu=S|Per=3
P	SubPOS=1|Gen=I|Num=S|Cas=4|PGe=F|PNu=S|Per=3|Var=2
P	SubPOS=1|Gen=M|Num=S|Cas=4|PGe=F|PNu=S|Per=3
P	SubPOS=1|Gen=M|Num=S|Cas=4|PGe=F|PNu=S|Per=3|Var=2
P	SubPOS=1|Gen=N|Num=S|Cas=4|PGe=F|PNu=S|Per=3
P	SubPOS=1|Gen=N|Num=S|Cas=4|PGe=F|PNu=S|Per=3|Var=2
P	SubPOS=1|Gen=X|Num=P|Cas=1|PGe=F|PNu=S|Per=3
P	SubPOS=1|Gen=X|Num=P|Cas=1|PGe=F|PNu=S|Per=3|Var=2
P	SubPOS=1|Gen=X|Num=P|Cas=2|PGe=F|PNu=S|Per=3
P	SubPOS=1|Gen=X|Num=P|Cas=2|PGe=F|PNu=S|Per=3|Var=2
P	SubPOS=1|Gen=X|Num=P|Cas=3|PGe=F|PNu=S|Per=3
P	SubPOS=1|Gen=X|Num=P|Cas=3|PGe=F|PNu=S|Per=3|Var=2
P	SubPOS=1|Gen=X|Num=P|Cas=4|PGe=F|PNu=S|Per=3
P	SubPOS=1|Gen=X|Num=P|Cas=4|PGe=F|PNu=S|Per=3|Var=2
P	SubPOS=1|Gen=X|Num=P|Cas=6|PGe=F|PNu=S|Per=3
P	SubPOS=1|Gen=X|Num=P|Cas=6|PGe=F|PNu=S|Per=3|Var=2
P	SubPOS=1|Gen=X|Num=P|Cas=7|PGe=F|PNu=S|Per=3
P	SubPOS=1|Gen=X|Num=P|Cas=7|PGe=F|PNu=S|Per=3|Var=2
P	SubPOS=1|Gen=X|Num=X|Cas=X|PGe=X|PNu=P|Per=3
P	SubPOS=1|Gen=X|Num=X|Cas=X|PGe=X|PNu=P|Per=3|Var=2
P	SubPOS=1|Gen=X|Num=X|Cas=X|PGe=Z|PNu=S|Per=3
P	SubPOS=1|Gen=X|Num=X|Cas=X|PGe=Z|PNu=S|Per=3|Var=2
P	SubPOS=1|Gen=Z|Num=S|Cas=1|PGe=F|PNu=S|Per=3
P	SubPOS=1|Gen=Z|Num=S|Cas=1|PGe=F|PNu=S|Per=3|Var=2
P	SubPOS=1|Gen=Z|Num=S|Cas=2|PGe=F|PNu=S|Per=3
P	SubPOS=1|Gen=Z|Num=S|Cas=2|PGe=F|PNu=S|Per=3|Var=2
P	SubPOS=1|Gen=Z|Num=S|Cas=3|PGe=F|PNu=S|Per=3
P	SubPOS=1|Gen=Z|Num=S|Cas=3|PGe=F|PNu=S|Per=3|Var=2
P	SubPOS=1|Gen=Z|Num=S|Cas=6|PGe=F|PNu=S|Per=3
P	SubPOS=1|Gen=Z|Num=S|Cas=6|PGe=F|PNu=S|Per=3|Var=2
P	SubPOS=1|Gen=Z|Num=S|Cas=7|PGe=F|PNu=S|Per=3
P	SubPOS=1|Gen=Z|Num=S|Cas=7|PGe=F|PNu=S|Per=3|Var=2
P	SubPOS=4|Gen=F|Num=D|Cas=7
P	SubPOS=4|Gen=F|Num=D|Cas=7|Var=6
P	SubPOS=4|Gen=F|Num=P|Cas=1
P	SubPOS=4|Gen=F|Num=P|Cas=1|Var=6
P	SubPOS=4|Gen=F|Num=P|Cas=4
P	SubPOS=4|Gen=F|Num=P|Cas=4|Var=6
P	SubPOS=4|Gen=F|Num=S|Cas=1
P	SubPOS=4|Gen=F|Num=S|Cas=1|Var=3
P	SubPOS=4|Gen=F|Num=S|Cas=1|Var=6
P	SubPOS=4|Gen=F|Num=S|Cas=2
P	SubPOS=4|Gen=F|Num=S|Cas=2|Var=6
P	SubPOS=4|Gen=F|Num=S|Cas=3
P	SubPOS=4|Gen=F|Num=S|Cas=3|Var=6
P	SubPOS=4|Gen=F|Num=S|Cas=4
P	SubPOS=4|Gen=F|Num=S|Cas=4|Var=6
P	SubPOS=4|Gen=F|Num=S|Cas=6
P	SubPOS=4|Gen=F|Num=S|Cas=6|Var=6
P	SubPOS=4|Gen=F|Num=S|Cas=7
P	SubPOS=4|Gen=F|Num=S|Cas=7|Var=6
P	SubPOS=4|Gen=I|Num=P|Cas=1
P	SubPOS=4|Gen=I|Num=P|Cas=1|Var=6
P	SubPOS=4|Gen=I|Num=S|Cas=4
P	SubPOS=4|Gen=I|Num=S|Cas=4|Var=6
P	SubPOS=4|Gen=M|Num=P|Cas=1
P	SubPOS=4|Gen=M|Num=P|Cas=1|Var=6
P	SubPOS=4|Gen=M|Num=S|Cas=4
P	SubPOS=4|Gen=M|Num=S|Cas=4|Var=6
P	SubPOS=4|Gen=N|Num=P|Cas=1
P	SubPOS=4|Gen=N|Num=P|Cas=1|Var=6
P	SubPOS=4|Gen=N|Num=P|Cas=1|Var=7
P	SubPOS=4|Gen=N|Num=P|Cas=4
P	SubPOS=4|Gen=N|Num=P|Cas=4|Var=6
P	SubPOS=4|Gen=N|Num=S|Cas=1
P	SubPOS=4|Gen=N|Num=S|Cas=1|Var=6
P	SubPOS=4|Gen=N|Num=S|Cas=4
P	SubPOS=4|Gen=N|Num=S|Cas=4|Var=6
P	SubPOS=4|Gen=X|Num=D|Cas=7
P	SubPOS=4|Gen=X|Num=P|Cas=2
P	SubPOS=4|Gen=X|Num=P|Cas=2|Var=6
P	SubPOS=4|Gen=X|Num=P|Cas=3
P	SubPOS=4|Gen=X|Num=P|Cas=3|Var=6
P	SubPOS=4|Gen=X|Num=P|Cas=6
P	SubPOS=4|Gen=X|Num=P|Cas=6|Var=6
P	SubPOS=4|Gen=X|Num=P|Cas=7
P	SubPOS=4|Gen=X|Num=P|Cas=7|Var=6
P	SubPOS=4|Gen=X|Num=P|Cas=7|Var=7
P	SubPOS=4|Gen=X|Num=X|Cas=X
P	SubPOS=4|Gen=Y|Num=P|Cas=4
P	SubPOS=4|Gen=Y|Num=P|Cas=4|Var=6
P	SubPOS=4|Gen=Y|Num=S|Cas=1
P	SubPOS=4|Gen=Y|Num=S|Cas=1|Var=3
P	SubPOS=4|Gen=Y|Num=S|Cas=1|Var=6
P	SubPOS=4|Gen=Z|Num=S|Cas=2
P	SubPOS=4|Gen=Z|Num=S|Cas=2|Var=6
P	SubPOS=4|Gen=Z|Num=S|Cas=3
P	SubPOS=4|Gen=Z|Num=S|Cas=3|Var=6
P	SubPOS=4|Gen=Z|Num=S|Cas=6
P	SubPOS=4|Gen=Z|Num=S|Cas=6|Var=6
P	SubPOS=4|Gen=Z|Num=S|Cas=7
P	SubPOS=4|Gen=Z|Num=S|Cas=7|Var=6
P	SubPOS=5|Gen=F|Num=S|Cas=2|Per=3
P	SubPOS=5|Gen=F|Num=S|Cas=3|Per=3
P	SubPOS=5|Gen=F|Num=S|Cas=4|Per=3
P	SubPOS=5|Gen=F|Num=S|Cas=4|Per=3|Var=6
P	SubPOS=5|Gen=F|Num=S|Cas=6|Per=3
P	SubPOS=5|Gen=F|Num=S|Cas=7|Per=3
P	SubPOS=5|Gen=N|Num=S|Cas=4|Per=3
P	SubPOS=5|Gen=X|Num=P|Cas=2|Per=3
P	SubPOS=5|Gen=X|Num=P|Cas=3|Per=3
P	SubPOS=5|Gen=X|Num=P|Cas=4|Per=3
P	SubPOS=5|Gen=X|Num=P|Cas=6|Per=3
P	SubPOS=5|Gen=X|Num=P|Cas=7|Per=3
P	SubPOS=5|Gen=Z|Num=S|Cas=2|Per=3
P	SubPOS=5|Gen=Z|Num=S|Cas=2|Per=3|Var=1
P	SubPOS=5|Gen=Z|Num=S|Cas=3|Per=3
P	SubPOS=5|Gen=Z|Num=S|Cas=4|Per=3
P	SubPOS=5|Gen=Z|Num=S|Cas=4|Per=3|Var=1
P	SubPOS=5|Gen=Z|Num=S|Cas=6|Per=3
P	SubPOS=5|Gen=Z|Num=S|Cas=7|Per=3
P	SubPOS=6|Num=X|Cas=2
P	SubPOS=6|Num=X|Cas=3
P	SubPOS=6|Num=X|Cas=4
P	SubPOS=6|Num=X|Cas=6
P	SubPOS=6|Num=X|Cas=7
P	SubPOS=7|Num=S|Cas=3
P	SubPOS=7|Num=S|Cas=3|Per=2
P	SubPOS=7|Num=S|Cas=4
P	SubPOS=7|Num=S|Cas=4|Per=2
P	SubPOS=7|Num=X|Cas=3
P	SubPOS=7|Num=X|Cas=4
P	SubPOS=8|Gen=F|Num=D|Cas=7
P	SubPOS=8|Gen=F|Num=D|Cas=7|Var=6
P	SubPOS=8|Gen=F|Num=P|Cas=1|Var=1
P	SubPOS=8|Gen=F|Num=P|Cas=4|Var=1
P	SubPOS=8|Gen=F|Num=S|Cas=1|Var=1
P	SubPOS=8|Gen=F|Num=S|Cas=2
P	SubPOS=8|Gen=F|Num=S|Cas=2|Var=1
P	SubPOS=8|Gen=F|Num=S|Cas=2|Var=6
P	SubPOS=8|Gen=F|Num=S|Cas=3
P	SubPOS=8|Gen=F|Num=S|Cas=3|Var=1
P	SubPOS=8|Gen=F|Num=S|Cas=3|Var=6
P	SubPOS=8|Gen=F|Num=S|Cas=4
P	SubPOS=8|Gen=F|Num=S|Cas=4|Var=1
P	SubPOS=8|Gen=F|Num=S|Cas=4|Var=6
P	SubPOS=8|Gen=F|Num=S|Cas=5|Var=1
P	SubPOS=8|Gen=F|Num=S|Cas=6
P	SubPOS=8|Gen=F|Num=S|Cas=6|Var=1
P	SubPOS=8|Gen=F|Num=S|Cas=6|Var=6
P	SubPOS=8|Gen=F|Num=S|Cas=7
P	SubPOS=8|Gen=F|Num=S|Cas=7|Var=1
P	SubPOS=8|Gen=H|Num=P|Cas=1
P	SubPOS=8|Gen=H|Num=P|Cas=5
P	SubPOS=8|Gen=H|Num=P|Cas=5|Var=7
P	SubPOS=8|Gen=H|Num=S|Cas=1
P	SubPOS=8|Gen=H|Num=S|Cas=5
P	SubPOS=8|Gen=I|Num=P|Cas=1
P	SubPOS=8|Gen=I|Num=P|Cas=1|Var=1
P	SubPOS=8|Gen=I|Num=P|Cas=1|Var=7
P	SubPOS=8|Gen=I|Num=P|Cas=5
P	SubPOS=8|Gen=I|Num=P|Cas=5|Var=1
P	SubPOS=8|Gen=I|Num=P|Cas=5|Var=7
P	SubPOS=8|Gen=I|Num=S|Cas=4
P	SubPOS=8|Gen=I|Num=S|Cas=4|Var=6
P	SubPOS=8|Gen=M|Num=P|Cas=1
P	SubPOS=8|Gen=M|Num=P|Cas=1|Var=1
P	SubPOS=8|Gen=M|Num=P|Cas=5
P	SubPOS=8|Gen=M|Num=P|Cas=5|Var=1
P	SubPOS=8|Gen=M|Num=S|Cas=4
P	SubPOS=8|Gen=M|Num=S|Cas=4|Var=6
P	SubPOS=8|Gen=N|Num=P|Cas=1|Var=1
P	SubPOS=8|Gen=N|Num=P|Cas=4|Var=1
P	SubPOS=8|Gen=N|Num=P|Cas=5|Var=1
P	SubPOS=8|Gen=N|Num=S|Cas=1|Var=1
P	SubPOS=8|Gen=N|Num=S|Cas=4
P	SubPOS=8|Gen=N|Num=S|Cas=4|Var=1
P	SubPOS=8|Gen=N|Num=S|Cas=5|Var=1
P	SubPOS=8|Gen=X|Num=P|Cas=2
P	SubPOS=8|Gen=X|Num=P|Cas=2|Var=6
P	SubPOS=8|Gen=X|Num=P|Cas=3
P	SubPOS=8|Gen=X|Num=P|Cas=3|Var=6
P	SubPOS=8|Gen=X|Num=P|Cas=4
P	SubPOS=8|Gen=X|Num=P|Cas=4|Var=7
P	SubPOS=8|Gen=X|Num=P|Cas=6
P	SubPOS=8|Gen=X|Num=P|Cas=6|Var=6
P	SubPOS=8|Gen=X|Num=P|Cas=7
P	SubPOS=8|Gen=Y|Num=P|Cas=4|Var=1
P	SubPOS=8|Gen=Y|Num=S|Cas=1
P	SubPOS=8|Gen=Y|Num=S|Cas=1|Var=6
P	SubPOS=8|Gen=Y|Num=S|Cas=5
P	SubPOS=8|Gen=Y|Num=S|Cas=5|Var=6
P	SubPOS=8|Gen=Z|Num=S|Cas=2
P	SubPOS=8|Gen=Z|Num=S|Cas=2|Var=6
P	SubPOS=8|Gen=Z|Num=S|Cas=3
P	SubPOS=8|Gen=Z|Num=S|Cas=3|Var=6
P	SubPOS=8|Gen=Z|Num=S|Cas=6
P	SubPOS=8|Gen=Z|Num=S|Cas=6|Var=6
P	SubPOS=8|Gen=Z|Num=S|Cas=6|Var=7
P	SubPOS=8|Gen=Z|Num=S|Cas=7
P	SubPOS=8|Gen=Z|Num=S|Cas=7|Var=6
P	SubPOS=9|Gen=F|Num=S|Cas=2
P	SubPOS=9|Gen=F|Num=S|Cas=2|Var=2
P	SubPOS=9|Gen=F|Num=S|Cas=3
P	SubPOS=9|Gen=F|Num=S|Cas=3|Var=2
P	SubPOS=9|Gen=F|Num=S|Cas=4
P	SubPOS=9|Gen=F|Num=S|Cas=4|Var=2
P	SubPOS=9|Gen=F|Num=S|Cas=6
P	SubPOS=9|Gen=F|Num=S|Cas=6|Var=2
P	SubPOS=9|Gen=F|Num=S|Cas=7
P	SubPOS=9|Gen=F|Num=S|Cas=7|Var=2
P	SubPOS=9|Gen=N|Num=S|Cas=4
P	SubPOS=9|Gen=N|Num=S|Cas=4|Var=2
P	SubPOS=9|Gen=X|Num=P|Cas=2
P	SubPOS=9|Gen=X|Num=P|Cas=2|Var=2
P	SubPOS=9|Gen=X|Num=P|Cas=3
P	SubPOS=9|Gen=X|Num=P|Cas=3|Var=2
P	SubPOS=9|Gen=X|Num=P|Cas=4
P	SubPOS=9|Gen=X|Num=P|Cas=4|Var=2
P	SubPOS=9|Gen=X|Num=P|Cas=6
P	SubPOS=9|Gen=X|Num=P|Cas=6|Var=2
P	SubPOS=9|Gen=X|Num=P|Cas=7
P	SubPOS=9|Gen=X|Num=P|Cas=7|Var=2
P	SubPOS=9|Gen=Z|Num=S|Cas=2
P	SubPOS=9|Gen=Z|Num=S|Cas=2|Var=1
P	SubPOS=9|Gen=Z|Num=S|Cas=2|Var=2
P	SubPOS=9|Gen=Z|Num=S|Cas=2|Var=3
P	SubPOS=9|Gen=Z|Num=S|Cas=3
P	SubPOS=9|Gen=Z|Num=S|Cas=3|Var=2
P	SubPOS=9|Gen=Z|Num=S|Cas=4
P	SubPOS=9|Gen=Z|Num=S|Cas=4|Var=1
P	SubPOS=9|Gen=Z|Num=S|Cas=4|Var=2
P	SubPOS=9|Gen=Z|Num=S|Cas=4|Var=3
P	SubPOS=9|Gen=Z|Num=S|Cas=6
P	SubPOS=9|Gen=Z|Num=S|Cas=6|Var=2
P	SubPOS=9|Gen=Z|Num=S|Cas=7
P	SubPOS=9|Gen=Z|Num=S|Cas=7|Var=2
P	SubPOS=D|Gen=F|Num=D|Cas=7
P	SubPOS=D|Gen=F|Num=D|Cas=7|Var=2
P	SubPOS=D|Gen=F|Num=D|Cas=7|Var=5
P	SubPOS=D|Gen=F|Num=D|Cas=7|Var=6
P	SubPOS=D|Gen=F|Num=P|Cas=1
P	SubPOS=D|Gen=F|Num=P|Cas=1|Var=5
P	SubPOS=D|Gen=F|Num=P|Cas=1|Var=6
P	SubPOS=D|Gen=F|Num=P|Cas=2
P	SubPOS=D|Gen=F|Num=P|Cas=3
P	SubPOS=D|Gen=F|Num=P|Cas=4
P	SubPOS=D|Gen=F|Num=P|Cas=4|Var=5
P	SubPOS=D|Gen=F|Num=P|Cas=4|Var=6
P	SubPOS=D|Gen=F|Num=P|Cas=6
P	SubPOS=D|Gen=F|Num=P|Cas=7
P	SubPOS=D|Gen=F|Num=S|Cas=1
P	SubPOS=D|Gen=F|Num=S|Cas=1|Var=5
P	SubPOS=D|Gen=F|Num=S|Cas=1|Var=6
P	SubPOS=D|Gen=F|Num=S|Cas=1|Var=8
P	SubPOS=D|Gen=F|Num=S|Cas=2
P	SubPOS=D|Gen=F|Num=S|Cas=2|Var=5
P	SubPOS=D|Gen=F|Num=S|Cas=2|Var=6
P	SubPOS=D|Gen=F|Num=S|Cas=2|Var=8
P	SubPOS=D|Gen=F|Num=S|Cas=3
P	SubPOS=D|Gen=F|Num=S|Cas=3|Var=5
P	SubPOS=D|Gen=F|Num=S|Cas=3|Var=6
P	SubPOS=D|Gen=F|Num=S|Cas=3|Var=8
P	SubPOS=D|Gen=F|Num=S|Cas=4
P	SubPOS=D|Gen=F|Num=S|Cas=4|Var=5
P	SubPOS=D|Gen=F|Num=S|Cas=4|Var=6
P	SubPOS=D|Gen=F|Num=S|Cas=4|Var=8
P	SubPOS=D|Gen=F|Num=S|Cas=6
P	SubPOS=D|Gen=F|Num=S|Cas=6|Var=5
P	SubPOS=D|Gen=F|Num=S|Cas=6|Var=6
P	SubPOS=D|Gen=F|Num=S|Cas=6|Var=8
P	SubPOS=D|Gen=F|Num=S|Cas=7
P	SubPOS=D|Gen=F|Num=S|Cas=7|Var=5
P	SubPOS=D|Gen=F|Num=S|Cas=7|Var=6
P	SubPOS=D|Gen=F|Num=S|Cas=7|Var=8
P	SubPOS=D|Gen=I|Num=P|Cas=1
P	SubPOS=D|Gen=I|Num=P|Cas=1|Var=5
P	SubPOS=D|Gen=I|Num=P|Cas=1|Var=6
P	SubPOS=D|Gen=I|Num=P|Cas=2
P	SubPOS=D|Gen=I|Num=P|Cas=3
P	SubPOS=D|Gen=I|Num=P|Cas=4
P	SubPOS=D|Gen=I|Num=P|Cas=4|Var=5
P	SubPOS=D|Gen=I|Num=P|Cas=4|Var=6
P	SubPOS=D|Gen=I|Num=P|Cas=6
P	SubPOS=D|Gen=I|Num=P|Cas=7
P	SubPOS=D|Gen=I|Num=S|Cas=1|Var=8
P	SubPOS=D|Gen=I|Num=S|Cas=2
P	SubPOS=D|Gen=I|Num=S|Cas=2|Var=8
P	SubPOS=D|Gen=I|Num=S|Cas=3
P	SubPOS=D|Gen=I|Num=S|Cas=3|Var=8
P	SubPOS=D|Gen=I|Num=S|Cas=4
P	SubPOS=D|Gen=I|Num=S|Cas=4|Var=5
P	SubPOS=D|Gen=I|Num=S|Cas=4|Var=6
P	SubPOS=D|Gen=I|Num=S|Cas=4|Var=8
P	SubPOS=D|Gen=I|Num=S|Cas=6
P	SubPOS=D|Gen=I|Num=S|Cas=6|Var=8
P	SubPOS=D|Gen=I|Num=S|Cas=7
P	SubPOS=D|Gen=I|Num=S|Cas=7|Var=8
P	SubPOS=D|Gen=M|Num=P|Cas=1
P	SubPOS=D|Gen=M|Num=P|Cas=1|Var=1
P	SubPOS=D|Gen=M|Num=P|Cas=1|Var=5
P	SubPOS=D|Gen=M|Num=P|Cas=1|Var=6
P	SubPOS=D|Gen=M|Num=P|Cas=2
P	SubPOS=D|Gen=M|Num=P|Cas=3
P	SubPOS=D|Gen=M|Num=P|Cas=4
P	SubPOS=D|Gen=M|Num=P|Cas=4|Var=5
P	SubPOS=D|Gen=M|Num=P|Cas=4|Var=6
P	SubPOS=D|Gen=M|Num=P|Cas=6
P	SubPOS=D|Gen=M|Num=P|Cas=7
P	SubPOS=D|Gen=M|Num=S|Cas=1
P	SubPOS=D|Gen=M|Num=S|Cas=1|Var=8
P	SubPOS=D|Gen=M|Num=S|Cas=2
P	SubPOS=D|Gen=M|Num=S|Cas=2|Var=8
P	SubPOS=D|Gen=M|Num=S|Cas=3
P	SubPOS=D|Gen=M|Num=S|Cas=3|Var=8
P	SubPOS=D|Gen=M|Num=S|Cas=4
P	SubPOS=D|Gen=M|Num=S|Cas=4|Var=5
P	SubPOS=D|Gen=M|Num=S|Cas=4|Var=6
P	SubPOS=D|Gen=M|Num=S|Cas=4|Var=8
P	SubPOS=D|Gen=M|Num=S|Cas=6|Var=8
P	SubPOS=D|Gen=M|Num=S|Cas=7
P	SubPOS=D|Gen=M|Num=S|Cas=7|Var=8
P	SubPOS=D|Gen=N|Num=P|Cas=1
P	SubPOS=D|Gen=N|Num=P|Cas=1|Var=5
P	SubPOS=D|Gen=N|Num=P|Cas=1|Var=6
P	SubPOS=D|Gen=N|Num=P|Cas=2
P	SubPOS=D|Gen=N|Num=P|Cas=3
P	SubPOS=D|Gen=N|Num=P|Cas=4
P	SubPOS=D|Gen=N|Num=P|Cas=4|Var=5
P	SubPOS=D|Gen=N|Num=P|Cas=4|Var=6
P	SubPOS=D|Gen=N|Num=P|Cas=6
P	SubPOS=D|Gen=N|Num=P|Cas=7
P	SubPOS=D|Gen=N|Num=S|Cas=1
P	SubPOS=D|Gen=N|Num=S|Cas=1|Sem=m
P	SubPOS=D|Gen=N|Num=S|Cas=1|Var=2
P	SubPOS=D|Gen=N|Num=S|Cas=1|Var=5
P	SubPOS=D|Gen=N|Num=S|Cas=1|Var=6
P	SubPOS=D|Gen=N|Num=S|Cas=1|Var=8
P	SubPOS=D|Gen=N|Num=S|Cas=2
P	SubPOS=D|Gen=N|Num=S|Cas=2|Var=8
P	SubPOS=D|Gen=N|Num=S|Cas=3
P	SubPOS=D|Gen=N|Num=S|Cas=3|Var=8
P	SubPOS=D|Gen=N|Num=S|Cas=4
P	SubPOS=D|Gen=N|Num=S|Cas=4|Var=2
P	SubPOS=D|Gen=N|Num=S|Cas=4|Var=5
P	SubPOS=D|Gen=N|Num=S|Cas=4|Var=6
P	SubPOS=D|Gen=N|Num=S|Cas=4|Var=8
P	SubPOS=D|Gen=N|Num=S|Cas=6
P	SubPOS=D|Gen=N|Num=S|Cas=6|Var=8
P	SubPOS=D|Gen=N|Num=S|Cas=7
P	SubPOS=D|Gen=N|Num=S|Cas=7|Var=8
P	SubPOS=D|Gen=X|Num=P|Cas=2
P	SubPOS=D|Gen=X|Num=P|Cas=2|Var=1
P	SubPOS=D|Gen=X|Num=P|Cas=2|Var=2
P	SubPOS=D|Gen=X|Num=P|Cas=2|Var=5
P	SubPOS=D|Gen=X|Num=P|Cas=2|Var=6
P	SubPOS=D|Gen=X|Num=P|Cas=3
P	SubPOS=D|Gen=X|Num=P|Cas=3|Var=1
P	SubPOS=D|Gen=X|Num=P|Cas=3|Var=2
P	SubPOS=D|Gen=X|Num=P|Cas=3|Var=5
P	SubPOS=D|Gen=X|Num=P|Cas=3|Var=6
P	SubPOS=D|Gen=X|Num=P|Cas=6
P	SubPOS=D|Gen=X|Num=P|Cas=6|Var=1
P	SubPOS=D|Gen=X|Num=P|Cas=6|Var=2
P	SubPOS=D|Gen=X|Num=P|Cas=6|Var=5
P	SubPOS=D|Gen=X|Num=P|Cas=6|Var=6
P	SubPOS=D|Gen=X|Num=P|Cas=7
P	SubPOS=D|Gen=X|Num=P|Cas=7|Var=2
P	SubPOS=D|Gen=X|Num=P|Cas=7|Var=5
P	SubPOS=D|Gen=X|Num=P|Cas=7|Var=6
P	SubPOS=D|Gen=X|Num=P|Cas=X
P	SubPOS=D|Gen=X|Num=P|Cas=X|Var=8
P	SubPOS=D|Gen=X|Num=S|Cas=1
P	SubPOS=D|Gen=X|Num=S|Cas=2
P	SubPOS=D|Gen=X|Num=S|Cas=3
P	SubPOS=D|Gen=X|Num=S|Cas=4
P	SubPOS=D|Gen=X|Num=S|Cas=6
P	SubPOS=D|Gen=X|Num=S|Cas=7
P	SubPOS=D|Gen=X|Num=S|Cas=X
P	SubPOS=D|Gen=X|Num=S|Cas=X|Var=8
P	SubPOS=D|Gen=X|Num=X|Cas=X
P	SubPOS=D|Gen=X|Num=X|Cas=X|Var=8
P	SubPOS=D|Gen=Y|Num=S|Cas=1
P	SubPOS=D|Gen=Y|Num=S|Cas=1|Var=5
P	SubPOS=D|Gen=Y|Num=S|Cas=1|Var=6
P	SubPOS=D|Gen=Y|Num=S|Cas=4
P	SubPOS=D|Gen=Z|Num=S|Cas=2
P	SubPOS=D|Gen=Z|Num=S|Cas=2|Var=5
P	SubPOS=D|Gen=Z|Num=S|Cas=2|Var=6
P	SubPOS=D|Gen=Z|Num=S|Cas=3
P	SubPOS=D|Gen=Z|Num=S|Cas=3|Var=5
P	SubPOS=D|Gen=Z|Num=S|Cas=3|Var=6
P	SubPOS=D|Gen=Z|Num=S|Cas=6
P	SubPOS=D|Gen=Z|Num=S|Cas=6|Var=1
P	SubPOS=D|Gen=Z|Num=S|Cas=6|Var=2
P	SubPOS=D|Gen=Z|Num=S|Cas=6|Var=5
P	SubPOS=D|Gen=Z|Num=S|Cas=6|Var=6
P	SubPOS=D|Gen=Z|Num=S|Cas=7
P	SubPOS=D|Gen=Z|Num=S|Cas=7|Var=5
P	SubPOS=D|Gen=Z|Num=S|Cas=7|Var=6
P	SubPOS=E|Cas=1
P	SubPOS=E|Cas=2
P	SubPOS=E|Cas=3
P	SubPOS=E|Cas=4
P	SubPOS=E|Cas=6
P	SubPOS=E|Cas=7
P	SubPOS=H|Gen=Z|Num=S|Cas=2|Per=3
P	SubPOS=H|Gen=Z|Num=S|Cas=3|Per=3
P	SubPOS=H|Gen=Z|Num=S|Cas=4|Per=3
P	SubPOS=H|Num=S|Cas=2|Per=1
P	SubPOS=H|Num=S|Cas=2|Per=2
P	SubPOS=H|Num=S|Cas=3|Per=1
P	SubPOS=H|Num=S|Cas=3|Per=2
P	SubPOS=H|Num=S|Cas=4|Per=1
P	SubPOS=H|Num=S|Cas=4|Per=2
P	SubPOS=J|Gen=F|Num=D|Cas=7
P	SubPOS=J|Gen=F|Num=P|Cas=1
P	SubPOS=J|Gen=F|Num=P|Cas=4
P	SubPOS=J|Gen=F|Num=S|Cas=1
P	SubPOS=J|Gen=F|Num=S|Cas=1|Var=2
P	SubPOS=J|Gen=F|Num=S|Cas=2
P	SubPOS=J|Gen=F|Num=S|Cas=2|Var=2
P	SubPOS=J|Gen=F|Num=S|Cas=3
P	SubPOS=J|Gen=F|Num=S|Cas=3|Var=2
P	SubPOS=J|Gen=F|Num=S|Cas=4
P	SubPOS=J|Gen=F|Num=S|Cas=4|Var=2
P	SubPOS=J|Gen=F|Num=S|Cas=6
P	SubPOS=J|Gen=F|Num=S|Cas=7
P	SubPOS=J|Gen=F|Num=S|Cas=7|Var=2
P	SubPOS=J|Gen=I|Num=P|Cas=1
P	SubPOS=J|Gen=I|Num=S|Cas=4
P	SubPOS=J|Gen=I|Num=S|Cas=4|Var=2
P	SubPOS=J|Gen=M|Num=P|Cas=1
P	SubPOS=J|Gen=M|Num=S|Cas=4
P	SubPOS=J|Gen=M|Num=S|Cas=4|Var=2
P	SubPOS=J|Gen=N|Num=P|Cas=1
P	SubPOS=J|Gen=N|Num=P|Cas=4
P	SubPOS=J|Gen=N|Num=S|Cas=1
P	SubPOS=J|Gen=N|Num=S|Cas=1|Var=2
P	SubPOS=J|Gen=N|Num=S|Cas=4
P	SubPOS=J|Gen=N|Num=S|Cas=4|Var=2
P	SubPOS=J|Gen=X|Num=P|Cas=1
P	SubPOS=J|Gen=X|Num=P|Cas=1|Var=2
P	SubPOS=J|Gen=X|Num=P|Cas=2
P	SubPOS=J|Gen=X|Num=P|Cas=2|Var=2
P	SubPOS=J|Gen=X|Num=P|Cas=3
P	SubPOS=J|Gen=X|Num=P|Cas=3|Var=2
P	SubPOS=J|Gen=X|Num=P|Cas=4
P	SubPOS=J|Gen=X|Num=P|Cas=4|Var=2
P	SubPOS=J|Gen=X|Num=P|Cas=6
P	SubPOS=J|Gen=X|Num=P|Cas=7
P	SubPOS=J|Gen=X|Num=P|Cas=7|Var=2
P	SubPOS=J|Gen=Y|Num=P|Cas=4
P	SubPOS=J|Gen=Y|Num=S|Cas=1
P	SubPOS=J|Gen=Y|Num=S|Cas=1|Var=2
P	SubPOS=J|Gen=Z|Num=S|Cas=2
P	SubPOS=J|Gen=Z|Num=S|Cas=2|Var=1
P	SubPOS=J|Gen=Z|Num=S|Cas=2|Var=2
P	SubPOS=J|Gen=Z|Num=S|Cas=2|Var=3
P	SubPOS=J|Gen=Z|Num=S|Cas=3
P	SubPOS=J|Gen=Z|Num=S|Cas=3|Var=2
P	SubPOS=J|Gen=Z|Num=S|Cas=4|Var=1
P	SubPOS=J|Gen=Z|Num=S|Cas=4|Var=3
P	SubPOS=J|Gen=Z|Num=S|Cas=6
P	SubPOS=J|Gen=Z|Num=S|Cas=7
P	SubPOS=J|Gen=Z|Num=S|Cas=7|Var=2
P	SubPOS=K|Gen=M|Cas=1
P	SubPOS=K|Gen=M|Cas=1|Per=2
P	SubPOS=K|Gen=M|Cas=1|Var=1
P	SubPOS=K|Gen=M|Cas=1|Var=2
P	SubPOS=K|Gen=M|Cas=2
P	SubPOS=K|Gen=M|Cas=2|Per=2
P	SubPOS=K|Gen=M|Cas=2|Var=2
P	SubPOS=K|Gen=M|Cas=3
P	SubPOS=K|Gen=M|Cas=3|Per=2
P	SubPOS=K|Gen=M|Cas=3|Var=2
P	SubPOS=K|Gen=M|Cas=4
P	SubPOS=K|Gen=M|Cas=4|Per=2
P	SubPOS=K|Gen=M|Cas=4|Var=2
P	SubPOS=K|Gen=M|Cas=6
P	SubPOS=K|Gen=M|Cas=6|Per=2
P	SubPOS=K|Gen=M|Cas=6|Var=2
P	SubPOS=K|Gen=M|Cas=7
P	SubPOS=K|Gen=M|Cas=7|Per=2
P	SubPOS=K|Gen=M|Cas=7|Var=2
P	SubPOS=L|Gen=F|Num=D|Cas=7
P	SubPOS=L|Gen=F|Num=P|Cas=1
P	SubPOS=L|Gen=F|Num=P|Cas=1|Var=6
P	SubPOS=L|Gen=F|Num=P|Cas=4
P	SubPOS=L|Gen=F|Num=P|Cas=5
P	SubPOS=L|Gen=F|Num=S|Cas=1
P	SubPOS=L|Gen=F|Num=S|Cas=2
P	SubPOS=L|Gen=F|Num=S|Cas=3
P	SubPOS=L|Gen=F|Num=S|Cas=4
P	SubPOS=L|Gen=F|Num=S|Cas=5
P	SubPOS=L|Gen=F|Num=S|Cas=5|Var=1
P	SubPOS=L|Gen=F|Num=S|Cas=6
P	SubPOS=L|Gen=F|Num=S|Cas=7
P	SubPOS=L|Gen=I|Num=P|Cas=1
P	SubPOS=L|Gen=I|Num=P|Cas=1|Var=6
P	SubPOS=L|Gen=I|Num=P|Cas=5
P	SubPOS=L|Gen=I|Num=S|Cas=4
P	SubPOS=L|Gen=I|Num=S|Cas=4|Var=1
P	SubPOS=L|Gen=M|Num=P|Cas=1
P	SubPOS=L|Gen=M|Num=P|Cas=1|Var=6
P	SubPOS=L|Gen=M|Num=P|Cas=5
P	SubPOS=L|Gen=M|Num=S|Cas=4
P	SubPOS=L|Gen=N|Num=P|Cas=1
P	SubPOS=L|Gen=N|Num=P|Cas=1|Var=6
P	SubPOS=L|Gen=N|Num=P|Cas=4
P	SubPOS=L|Gen=N|Num=P|Cas=5
P	SubPOS=L|Gen=N|Num=S|Cas=1
P	SubPOS=L|Gen=N|Num=S|Cas=1|Var=1
P	SubPOS=L|Gen=N|Num=S|Cas=1|Var=6
P	SubPOS=L|Gen=N|Num=S|Cas=4
P	SubPOS=L|Gen=N|Num=S|Cas=4|Sem=m
P	SubPOS=L|Gen=N|Num=S|Cas=4|Var=1
P	SubPOS=L|Gen=N|Num=S|Cas=4|Var=6
P	SubPOS=L|Gen=N|Num=S|Cas=5
P	SubPOS=L|Gen=N|Num=S|Cas=5|Var=1
P	SubPOS=L|Gen=X|Num=P|Cas=2
P	SubPOS=L|Gen=X|Num=P|Cas=3
P	SubPOS=L|Gen=X|Num=P|Cas=6
P	SubPOS=L|Gen=X|Num=P|Cas=7
P	SubPOS=L|Gen=X|Num=P|Cas=7|Var=6
P	SubPOS=L|Gen=X|Num=X|Cas=X
P	SubPOS=L|Gen=X|Num=X|Cas=X|Sem=K
P	SubPOS=L|Gen=X|Num=X|Cas=X|Sem=m
P	SubPOS=L|Gen=X|Num=X|Cas=X|Var=8
P	SubPOS=L|Gen=Y|Num=P|Cas=4
P	SubPOS=L|Gen=Y|Num=S|Cas=1
P	SubPOS=L|Gen=Y|Num=S|Cas=1|Var=1
P	SubPOS=L|Gen=Y|Num=S|Cas=4
P	SubPOS=L|Gen=Y|Num=S|Cas=5
P	SubPOS=L|Gen=Y|Num=S|Cas=5|Var=1
P	SubPOS=L|Gen=Z|Num=S|Cas=2
P	SubPOS=L|Gen=Z|Num=S|Cas=3
P	SubPOS=L|Gen=Z|Num=S|Cas=6
P	SubPOS=L|Gen=Z|Num=S|Cas=7
P	SubPOS=P|Gen=F|Num=P|Cas=1|Per=3
P	SubPOS=P|Gen=F|Num=P|Cas=2|Per=3
P	SubPOS=P|Gen=F|Num=P|Cas=3|Per=3
P	SubPOS=P|Gen=F|Num=P|Cas=4|Per=3
P	SubPOS=P|Gen=F|Num=P|Cas=6|Per=3
P	SubPOS=P|Gen=F|Num=P|Cas=7|Per=3
P	SubPOS=P|Gen=F|Num=P|Cas=X|Per=3
P	SubPOS=P|Gen=F|Num=S|Cas=1|Per=3
P	SubPOS=P|Gen=F|Num=S|Cas=2|Per=3
P	SubPOS=P|Gen=F|Num=S|Cas=3|Per=3
P	SubPOS=P|Gen=F|Num=S|Cas=4|Per=3
P	SubPOS=P|Gen=F|Num=S|Cas=6|Per=3
P	SubPOS=P|Gen=F|Num=S|Cas=7|Per=3
P	SubPOS=P|Gen=F|Num=S|Cas=X|Per=3
P	SubPOS=P|Gen=I|Num=P|Cas=1|Per=3
P	SubPOS=P|Gen=I|Num=P|Cas=2|Per=3
P	SubPOS=P|Gen=I|Num=P|Cas=3|Per=3
P	SubPOS=P|Gen=I|Num=P|Cas=4|Per=3
P	SubPOS=P|Gen=I|Num=P|Cas=6|Per=3
P	SubPOS=P|Gen=I|Num=P|Cas=7|Per=3
P	SubPOS=P|Gen=I|Num=S|Cas=2|Per=3
P	SubPOS=P|Gen=I|Num=S|Cas=3|Per=3
P	SubPOS=P|Gen=I|Num=S|Cas=4|Per=3
P	SubPOS=P|Gen=I|Num=S|Cas=6|Per=3
P	SubPOS=P|Gen=I|Num=S|Cas=7|Per=3
P	SubPOS=P|Gen=M|Num=P|Cas=1|Per=3
P	SubPOS=P|Gen=M|Num=P|Cas=2|Per=3
P	SubPOS=P|Gen=M|Num=P|Cas=3|Per=3
P	SubPOS=P|Gen=M|Num=P|Cas=4|Per=3
P	SubPOS=P|Gen=M|Num=P|Cas=6|Per=3
P	SubPOS=P|Gen=M|Num=P|Cas=7|Per=3
P	SubPOS=P|Gen=M|Num=P|Cas=X|Per=3
P	SubPOS=P|Gen=M|Num=S|Cas=1|Per=3
P	SubPOS=P|Gen=M|Num=S|Cas=1|Per=3|Sem=R
P	SubPOS=P|Gen=M|Num=S|Cas=1|Per=3|Sem=m
P	SubPOS=P|Gen=M|Num=S|Cas=2|Per=3
P	SubPOS=P|Gen=M|Num=S|Cas=3|Per=3
P	SubPOS=P|Gen=M|Num=S|Cas=4|Per=3
P	SubPOS=P|Gen=M|Num=S|Cas=6|Per=3
P	SubPOS=P|Gen=M|Num=S|Cas=7|Per=3
P	SubPOS=P|Gen=M|Num=S|Cas=X|Per=3
P	SubPOS=P|Gen=N|Num=P|Cas=1|Per=3
P	SubPOS=P|Gen=N|Num=P|Cas=2|Per=3
P	SubPOS=P|Gen=N|Num=P|Cas=3|Per=3
P	SubPOS=P|Gen=N|Num=P|Cas=4|Per=3
P	SubPOS=P|Gen=N|Num=P|Cas=7|Per=3
P	SubPOS=P|Gen=N|Num=P|Cas=X|Per=3
P	SubPOS=P|Gen=N|Num=S|Cas=1|Per=3
P	SubPOS=P|Gen=N|Num=S|Cas=2|Per=3
P	SubPOS=P|Gen=N|Num=S|Cas=4|Per=3
P	SubPOS=P|Gen=N|Num=S|Cas=6|Per=3
P	SubPOS=P|Gen=N|Num=S|Cas=7|Per=3
P	SubPOS=P|Gen=N|Num=S|Cas=X|Per=3
P	SubPOS=P|Gen=N|Num=S|Cas=X|Per=3|Sem=R
P	SubPOS=P|Gen=N|Num=S|Cas=X|Per=3|Sem=m
P	SubPOS=P|Gen=X|Num=P|Cas=1|Per=3
P	SubPOS=P|Gen=X|Num=P|Cas=2|Per=3
P	SubPOS=P|Gen=X|Num=P|Cas=3|Per=3
P	SubPOS=P|Gen=X|Num=P|Cas=4|Per=3
P	SubPOS=P|Gen=X|Num=P|Cas=7|Per=3
P	SubPOS=P|Gen=X|Num=P|Cas=X|Per=3
P	SubPOS=P|Gen=X|Num=S|Cas=3|Per=3
P	SubPOS=P|Gen=Y|Num=S|Cas=1|Per=3
P	SubPOS=P|Gen=Y|Num=S|Cas=2|Per=3
P	SubPOS=P|Gen=Y|Num=S|Cas=4|Per=3
P	SubPOS=P|Gen=Z|Num=S|Cas=2|Per=3|Var=1
P	SubPOS=P|Gen=Z|Num=S|Cas=3|Per=3
P	SubPOS=P|Gen=Z|Num=S|Cas=4|Per=3|Var=2
P	SubPOS=P|Gen=Z|Num=S|Cas=7|Per=3
P	SubPOS=P|Num=P|Cas=1|Per=1
P	SubPOS=P|Num=P|Cas=1|Per=2
P	SubPOS=P|Num=P|Cas=2|Per=1
P	SubPOS=P|Num=P|Cas=2|Per=2
P	SubPOS=P|Num=P|Cas=3|Per=1
P	SubPOS=P|Num=P|Cas=3|Per=2
P	SubPOS=P|Num=P|Cas=4|Per=1
P	SubPOS=P|Num=P|Cas=4|Per=2
P	SubPOS=P|Num=P|Cas=5|Per=1
P	SubPOS=P|Num=P|Cas=5|Per=2
P	SubPOS=P|Num=P|Cas=6|Per=1
P	SubPOS=P|Num=P|Cas=6|Per=2
P	SubPOS=P|Num=P|Cas=7|Per=1
P	SubPOS=P|Num=P|Cas=7|Per=1|Var=6
P	SubPOS=P|Num=P|Cas=7|Per=2
P	SubPOS=P|Num=P|Cas=X|Per=1
P	SubPOS=P|Num=P|Cas=X|Per=2
P	SubPOS=P|Num=S|Cas=1|Per=1
P	SubPOS=P|Num=S|Cas=1|Per=2
P	SubPOS=P|Num=S|Cas=1|Per=2|Ten=P|Neg=A|Voi=A
P	SubPOS=P|Num=S|Cas=2|Per=1
P	SubPOS=P|Num=S|Cas=2|Per=2
P	SubPOS=P|Num=S|Cas=3|Per=1
P	SubPOS=P|Num=S|Cas=3|Per=2
P	SubPOS=P|Num=S|Cas=4|Per=1
P	SubPOS=P|Num=S|Cas=4|Per=2
P	SubPOS=P|Num=S|Cas=4|Per=2|Sem=m
P	SubPOS=P|Num=S|Cas=5|Per=1
P	SubPOS=P|Num=S|Cas=5|Per=2
P	SubPOS=P|Num=S|Cas=5|Per=2|Sem=R
P	SubPOS=P|Num=S|Cas=5|Per=2|Sem=m
P	SubPOS=P|Num=S|Cas=6|Per=1
P	SubPOS=P|Num=S|Cas=6|Per=2
P	SubPOS=P|Num=S|Cas=7|Per=1
P	SubPOS=P|Num=S|Cas=7|Per=2
P	SubPOS=P|Num=S|Cas=X|Per=1
P	SubPOS=P|Num=S|Cas=X|Per=2
P	SubPOS=P|Num=X|Cas=X|Per=2
P	SubPOS=Q|Cas=1
P	SubPOS=Q|Cas=1|Var=9
P	SubPOS=Q|Cas=2
P	SubPOS=Q|Cas=2|Var=9
P	SubPOS=Q|Cas=3
P	SubPOS=Q|Cas=3|Var=9
P	SubPOS=Q|Cas=4
P	SubPOS=Q|Cas=4|Var=9
P	SubPOS=Q|Cas=6
P	SubPOS=Q|Cas=6|Var=9
P	SubPOS=Q|Cas=7
P	SubPOS=Q|Cas=7|Var=9
P	SubPOS=Q|Cas=X
P	SubPOS=Q|Cas=X|Var=9
P	SubPOS=S|Gen=F|Num=D|Cas=7|PGe=F|PNu=S|Per=3
P	SubPOS=S|Gen=F|Num=D|Cas=7|PNu=P|Per=1
P	SubPOS=S|Gen=F|Num=D|Cas=7|PNu=P|Per=2
P	SubPOS=S|Gen=F|Num=D|Cas=7|PNu=S|Per=1
P	SubPOS=S|Gen=F|Num=D|Cas=7|PNu=S|Per=1|Var=6
P	SubPOS=S|Gen=F|Num=D|Cas=7|PNu=S|Per=2
P	SubPOS=S|Gen=F|Num=D|Cas=7|PNu=S|Per=2|Var=6
P	SubPOS=S|Gen=F|Num=P|Cas=1|PNu=S|Per=1|Var=1
P	SubPOS=S|Gen=F|Num=P|Cas=1|PNu=S|Per=2|Var=1
P	SubPOS=S|Gen=F|Num=P|Cas=4|PNu=S|Per=1|Var=1
P	SubPOS=S|Gen=F|Num=P|Cas=4|PNu=S|Per=2|Var=1
P	SubPOS=S|Gen=F|Num=P|Cas=X|PNu=P|Per=1
P	SubPOS=S|Gen=F|Num=P|Cas=X|PNu=S|Per=1
P	SubPOS=S|Gen=F|Num=P|Cas=X|PNu=X|Per=1
P	SubPOS=S|Gen=F|Num=S|Cas=1|PNu=S|Per=1|Var=1
P	SubPOS=S|Gen=F|Num=S|Cas=1|PNu=S|Per=2|Var=1
P	SubPOS=S|Gen=F|Num=S|Cas=2|PNu=P|Per=1
P	SubPOS=S|Gen=F|Num=S|Cas=2|PNu=P|Per=2
P	SubPOS=S|Gen=F|Num=S|Cas=2|PNu=S|Per=1
P	SubPOS=S|Gen=F|Num=S|Cas=2|PNu=S|Per=1|Var=1
P	SubPOS=S|Gen=F|Num=S|Cas=2|PNu=S|Per=1|Var=6
P	SubPOS=S|Gen=F|Num=S|Cas=2|PNu=S|Per=2
P	SubPOS=S|Gen=F|Num=S|Cas=2|PNu=S|Per=2|Var=1
P	SubPOS=S|Gen=F|Num=S|Cas=2|PNu=S|Per=2|Var=6
P	SubPOS=S|Gen=F|Num=S|Cas=3|PNu=P|Per=1
P	SubPOS=S|Gen=F|Num=S|Cas=3|PNu=P|Per=2
P	SubPOS=S|Gen=F|Num=S|Cas=3|PNu=S|Per=1
P	SubPOS=S|Gen=F|Num=S|Cas=3|PNu=S|Per=1|Var=1
P	SubPOS=S|Gen=F|Num=S|Cas=3|PNu=S|Per=1|Var=6
P	SubPOS=S|Gen=F|Num=S|Cas=3|PNu=S|Per=2
P	SubPOS=S|Gen=F|Num=S|Cas=3|PNu=S|Per=2|Var=1
P	SubPOS=S|Gen=F|Num=S|Cas=3|PNu=S|Per=2|Var=6
P	SubPOS=S|Gen=F|Num=S|Cas=4|PNu=P|Per=1
P	SubPOS=S|Gen=F|Num=S|Cas=4|PNu=P|Per=1|Var=6
P	SubPOS=S|Gen=F|Num=S|Cas=4|PNu=P|Per=2
P	SubPOS=S|Gen=F|Num=S|Cas=4|PNu=P|Per=2|Var=6
P	SubPOS=S|Gen=F|Num=S|Cas=4|PNu=S|Per=1
P	SubPOS=S|Gen=F|Num=S|Cas=4|PNu=S|Per=1|Var=1
P	SubPOS=S|Gen=F|Num=S|Cas=4|PNu=S|Per=1|Var=6
P	SubPOS=S|Gen=F|Num=S|Cas=4|PNu=S|Per=2
P	SubPOS=S|Gen=F|Num=S|Cas=4|PNu=S|Per=2|Var=1
P	SubPOS=S|Gen=F|Num=S|Cas=4|PNu=S|Per=2|Var=6
P	SubPOS=S|Gen=F|Num=S|Cas=5|PNu=S|Per=1|Var=1
P	SubPOS=S|Gen=F|Num=S|Cas=5|PNu=S|Per=2|Var=1
P	SubPOS=S|Gen=F|Num=S|Cas=6|PNu=P|Per=1
P	SubPOS=S|Gen=F|Num=S|Cas=6|PNu=P|Per=2
P	SubPOS=S|Gen=F|Num=S|Cas=6|PNu=S|Per=1
P	SubPOS=S|Gen=F|Num=S|Cas=6|PNu=S|Per=1|Var=1
P	SubPOS=S|Gen=F|Num=S|Cas=6|PNu=S|Per=1|Var=6
P	SubPOS=S|Gen=F|Num=S|Cas=6|PNu=S|Per=2
P	SubPOS=S|Gen=F|Num=S|Cas=6|PNu=S|Per=2|Var=1
P	SubPOS=S|Gen=F|Num=S|Cas=6|PNu=S|Per=2|Var=6
P	SubPOS=S|Gen=F|Num=S|Cas=7|PNu=P|Per=1
P	SubPOS=S|Gen=F|Num=S|Cas=7|PNu=P|Per=2
P	SubPOS=S|Gen=F|Num=S|Cas=7|PNu=S|Per=1
P	SubPOS=S|Gen=F|Num=S|Cas=7|PNu=S|Per=1|Var=1
P	SubPOS=S|Gen=F|Num=S|Cas=7|PNu=S|Per=2
P	SubPOS=S|Gen=F|Num=S|Cas=7|PNu=S|Per=2|Var=1
P	SubPOS=S|Gen=F|Num=S|Cas=X|PGe=F|PNu=S|Per=3
P	SubPOS=S|Gen=F|Num=S|Cas=X|PNu=P|Per=1
P	SubPOS=S|Gen=F|Num=S|Cas=X|PNu=P|Per=1|Sem=m
P	SubPOS=S|Gen=F|Num=S|Cas=X|PNu=S|Per=1
P	SubPOS=S|Gen=F|Num=S|Cas=X|PNu=X|Per=1
P	SubPOS=S|Gen=H|Num=P|Cas=1|PNu=P|Per=1
P	SubPOS=S|Gen=H|Num=P|Cas=1|PNu=P|Per=2
P	SubPOS=S|Gen=H|Num=P|Cas=1|PNu=S|Per=1
P	SubPOS=S|Gen=H|Num=P|Cas=1|PNu=S|Per=2
P	SubPOS=S|Gen=H|Num=P|Cas=5|PNu=S|Per=1
P	SubPOS=S|Gen=H|Num=P|Cas=5|PNu=S|Per=1|Var=7
P	SubPOS=S|Gen=H|Num=P|Cas=5|PNu=S|Per=2
P	SubPOS=S|Gen=H|Num=P|Cas=5|PNu=S|Per=2|Var=7
P	SubPOS=S|Gen=H|Num=S|Cas=1|PNu=P|Per=1
P	SubPOS=S|Gen=H|Num=S|Cas=1|PNu=P|Per=2
P	SubPOS=S|Gen=H|Num=S|Cas=1|PNu=S|Per=1
P	SubPOS=S|Gen=H|Num=S|Cas=1|PNu=S|Per=2
P	SubPOS=S|Gen=H|Num=S|Cas=5|PNu=P|Per=1
P	SubPOS=S|Gen=H|Num=S|Cas=5|PNu=P|Per=2
P	SubPOS=S|Gen=H|Num=S|Cas=5|PNu=S|Per=1
P	SubPOS=S|Gen=H|Num=S|Cas=5|PNu=S|Per=2
P	SubPOS=S|Gen=I|Num=P|Cas=1|PNu=P|Per=1
P	SubPOS=S|Gen=I|Num=P|Cas=1|PNu=P|Per=2
P	SubPOS=S|Gen=I|Num=P|Cas=1|PNu=S|Per=1
P	SubPOS=S|Gen=I|Num=P|Cas=1|PNu=S|Per=1|Var=1
P	SubPOS=S|Gen=I|Num=P|Cas=1|PNu=S|Per=1|Var=7
P	SubPOS=S|Gen=I|Num=P|Cas=1|PNu=S|Per=2
P	SubPOS=S|Gen=I|Num=P|Cas=1|PNu=S|Per=2|Var=1
P	SubPOS=S|Gen=I|Num=P|Cas=1|PNu=S|Per=2|Var=7
P	SubPOS=S|Gen=I|Num=P|Cas=5|PNu=S|Per=1
P	SubPOS=S|Gen=I|Num=P|Cas=5|PNu=S|Per=1|Var=1
P	SubPOS=S|Gen=I|Num=P|Cas=5|PNu=S|Per=1|Var=7
P	SubPOS=S|Gen=I|Num=P|Cas=5|PNu=S|Per=2
P	SubPOS=S|Gen=I|Num=P|Cas=5|PNu=S|Per=2|Var=1
P	SubPOS=S|Gen=I|Num=P|Cas=5|PNu=S|Per=2|Var=7
P	SubPOS=S|Gen=I|Num=P|Cas=X|PNu=P|Per=1
P	SubPOS=S|Gen=I|Num=P|Cas=X|PNu=S|Per=1
P	SubPOS=S|Gen=I|Num=P|Cas=X|PNu=X|Per=1
P	SubPOS=S|Gen=I|Num=S|Cas=4|PGe=F|PNu=S|Per=3
P	SubPOS=S|Gen=I|Num=S|Cas=4|PNu=P|Per=1
P	SubPOS=S|Gen=I|Num=S|Cas=4|PNu=P|Per=2
P	SubPOS=S|Gen=I|Num=S|Cas=4|PNu=S|Per=1
P	SubPOS=S|Gen=I|Num=S|Cas=4|PNu=S|Per=1|Var=6
P	SubPOS=S|Gen=I|Num=S|Cas=4|PNu=S|Per=2
P	SubPOS=S|Gen=I|Num=S|Cas=4|PNu=S|Per=2|Var=6
P	SubPOS=S|Gen=I|Num=S|Cas=X|PNu=P|Per=1
P	SubPOS=S|Gen=I|Num=S|Cas=X|PNu=S|Per=1
P	SubPOS=S|Gen=I|Num=S|Cas=X|PNu=X|Per=1
P	SubPOS=S|Gen=M|Num=P|Cas=1|PNu=P|Per=1
P	SubPOS=S|Gen=M|Num=P|Cas=1|PNu=P|Per=2
P	SubPOS=S|Gen=M|Num=P|Cas=1|PNu=S|Per=1
P	SubPOS=S|Gen=M|Num=P|Cas=1|PNu=S|Per=1|Var=1
P	SubPOS=S|Gen=M|Num=P|Cas=1|PNu=S|Per=1|Var=7
P	SubPOS=S|Gen=M|Num=P|Cas=1|PNu=S|Per=2
P	SubPOS=S|Gen=M|Num=P|Cas=1|PNu=S|Per=2|Var=1
P	SubPOS=S|Gen=M|Num=P|Cas=5|PNu=P|Per=1
P	SubPOS=S|Gen=M|Num=P|Cas=5|PNu=P|Per=2
P	SubPOS=S|Gen=M|Num=P|Cas=5|PNu=S|Per=1
P	SubPOS=S|Gen=M|Num=P|Cas=5|PNu=S|Per=1|Var=1
P	SubPOS=S|Gen=M|Num=P|Cas=5|PNu=S|Per=1|Var=7
P	SubPOS=S|Gen=M|Num=P|Cas=5|PNu=S|Per=2
P	SubPOS=S|Gen=M|Num=P|Cas=5|PNu=S|Per=2|Var=1
P	SubPOS=S|Gen=M|Num=P|Cas=X|PNu=P|Per=1
P	SubPOS=S|Gen=M|Num=P|Cas=X|PNu=S|Per=1
P	SubPOS=S|Gen=M|Num=P|Cas=X|PNu=X|Per=1
P	SubPOS=S|Gen=M|Num=S|Cas=4|PGe=F|PNu=S|Per=3
P	SubPOS=S|Gen=M|Num=S|Cas=4|PNu=P|Per=1
P	SubPOS=S|Gen=M|Num=S|Cas=4|PNu=P|Per=2
P	SubPOS=S|Gen=M|Num=S|Cas=4|PNu=S|Per=1
P	SubPOS=S|Gen=M|Num=S|Cas=4|PNu=S|Per=1|Var=6
P	SubPOS=S|Gen=M|Num=S|Cas=4|PNu=S|Per=2
P	SubPOS=S|Gen=M|Num=S|Cas=4|PNu=S|Per=2|Var=6
P	SubPOS=S|Gen=M|Num=S|Cas=X|PNu=P|Per=1
P	SubPOS=S|Gen=M|Num=S|Cas=X|PNu=S|Per=1
P	SubPOS=S|Gen=M|Num=S|Cas=X|PNu=X|Per=1
P	SubPOS=S|Gen=N|Num=P|Cas=1|PNu=S|Per=1|Var=1
P	SubPOS=S|Gen=N|Num=P|Cas=1|PNu=S|Per=2|Var=1
P	SubPOS=S|Gen=N|Num=P|Cas=4|PNu=S|Per=1|Var=1
P	SubPOS=S|Gen=N|Num=P|Cas=4|PNu=S|Per=2|Var=1
P	SubPOS=S|Gen=N|Num=P|Cas=5|PNu=S|Per=1|Var=1
P	SubPOS=S|Gen=N|Num=P|Cas=5|PNu=S|Per=2|Var=1
P	SubPOS=S|Gen=N|Num=P|Cas=X|PNu=P|Per=1
P	SubPOS=S|Gen=N|Num=P|Cas=X|PNu=S|Per=1
P	SubPOS=S|Gen=N|Num=P|Cas=X|PNu=X|Per=1
P	SubPOS=S|Gen=N|Num=S|Cas=1|PNu=S|Per=1|Var=1
P	SubPOS=S|Gen=N|Num=S|Cas=1|PNu=S|Per=2|Var=1
P	SubPOS=S|Gen=N|Num=S|Cas=4|PGe=F|PNu=S|Per=3
P	SubPOS=S|Gen=N|Num=S|Cas=4|PNu=P|Per=1
P	SubPOS=S|Gen=N|Num=S|Cas=4|PNu=P|Per=2
P	SubPOS=S|Gen=N|Num=S|Cas=4|PNu=S|Per=1
P	SubPOS=S|Gen=N|Num=S|Cas=4|PNu=S|Per=1|Var=1
P	SubPOS=S|Gen=N|Num=S|Cas=4|PNu=S|Per=2
P	SubPOS=S|Gen=N|Num=S|Cas=4|PNu=S|Per=2|Var=1
P	SubPOS=S|Gen=N|Num=S|Cas=5|PNu=S|Per=1|Var=1
P	SubPOS=S|Gen=N|Num=S|Cas=5|PNu=S|Per=2|Var=1
P	SubPOS=S|Gen=N|Num=S|Cas=X|PNu=P|Per=1
P	SubPOS=S|Gen=N|Num=S|Cas=X|PNu=S|Per=1
P	SubPOS=S|Gen=N|Num=S|Cas=X|PNu=X|Per=1
P	SubPOS=S|Gen=X|Num=P|Cas=1|PGe=F|PNu=S|Per=3
P	SubPOS=S|Gen=X|Num=P|Cas=2|PGe=F|PNu=S|Per=3
P	SubPOS=S|Gen=X|Num=P|Cas=2|PNu=P|Per=1
P	SubPOS=S|Gen=X|Num=P|Cas=2|PNu=P|Per=2
P	SubPOS=S|Gen=X|Num=P|Cas=2|PNu=S|Per=1
P	SubPOS=S|Gen=X|Num=P|Cas=2|PNu=S|Per=1|Var=6
P	SubPOS=S|Gen=X|Num=P|Cas=2|PNu=S|Per=2
P	SubPOS=S|Gen=X|Num=P|Cas=2|PNu=S|Per=2|Var=6
P	SubPOS=S|Gen=X|Num=P|Cas=3|PGe=F|PNu=S|Per=3
P	SubPOS=S|Gen=X|Num=P|Cas=3|PNu=P|Per=1
P	SubPOS=S|Gen=X|Num=P|Cas=3|PNu=P|Per=2
P	SubPOS=S|Gen=X|Num=P|Cas=3|PNu=S|Per=1
P	SubPOS=S|Gen=X|Num=P|Cas=3|PNu=S|Per=1|Var=6
P	SubPOS=S|Gen=X|Num=P|Cas=3|PNu=S|Per=2
P	SubPOS=S|Gen=X|Num=P|Cas=3|PNu=S|Per=2|Var=6
P	SubPOS=S|Gen=X|Num=P|Cas=4|PGe=F|PNu=S|Per=3
P	SubPOS=S|Gen=X|Num=P|Cas=4|PNu=P|Per=1
P	SubPOS=S|Gen=X|Num=P|Cas=4|PNu=P|Per=2
P	SubPOS=S|Gen=X|Num=P|Cas=4|PNu=S|Per=1
P	SubPOS=S|Gen=X|Num=P|Cas=4|PNu=S|Per=1|Var=7
P	SubPOS=S|Gen=X|Num=P|Cas=4|PNu=S|Per=2
P	SubPOS=S|Gen=X|Num=P|Cas=4|PNu=S|Per=2|Var=7
P	SubPOS=S|Gen=X|Num=P|Cas=5|PGe=F|PNu=S|Per=3
P	SubPOS=S|Gen=X|Num=P|Cas=6|PGe=F|PNu=S|Per=3
P	SubPOS=S|Gen=X|Num=P|Cas=6|PNu=P|Per=1
P	SubPOS=S|Gen=X|Num=P|Cas=6|PNu=P|Per=2
P	SubPOS=S|Gen=X|Num=P|Cas=6|PNu=S|Per=1
P	SubPOS=S|Gen=X|Num=P|Cas=6|PNu=S|Per=1|Var=6
P	SubPOS=S|Gen=X|Num=P|Cas=6|PNu=S|Per=2
P	SubPOS=S|Gen=X|Num=P|Cas=6|PNu=S|Per=2|Var=6
P	SubPOS=S|Gen=X|Num=P|Cas=7|PGe=F|PNu=S|Per=3
P	SubPOS=S|Gen=X|Num=P|Cas=7|PGe=F|PNu=S|Per=3|Var=6
P	SubPOS=S|Gen=X|Num=P|Cas=7|PNu=P|Per=1
P	SubPOS=S|Gen=X|Num=P|Cas=7|PNu=P|Per=1|Var=6
P	SubPOS=S|Gen=X|Num=P|Cas=7|PNu=P|Per=2
P	SubPOS=S|Gen=X|Num=P|Cas=7|PNu=P|Per=2|Var=6
P	SubPOS=S|Gen=X|Num=P|Cas=7|PNu=S|Per=1
P	SubPOS=S|Gen=X|Num=P|Cas=7|PNu=S|Per=2
P	SubPOS=S|Gen=X|Num=X|Cas=X|PGe=N|PNu=S|Per=3
P	SubPOS=S|Gen=X|Num=X|Cas=X|PGe=X|PNu=P|Per=3
P	SubPOS=S|Gen=X|Num=X|Cas=X|PGe=Y|PNu=S|Per=3
P	SubPOS=S|Gen=X|Num=X|Cas=X|PGe=Z|PNu=S|Per=3
P	SubPOS=S|Gen=X|Num=X|Cas=X|PNu=P|Per=1
P	SubPOS=S|Gen=X|Num=X|Cas=X|PNu=S|Per=1
P	SubPOS=S|Gen=X|Num=X|Cas=X|PNu=X|Per=2
P	SubPOS=S|Gen=X|Num=X|Cas=X|PNu=X|Per=2|Sem=R
P	SubPOS=S|Gen=X|Num=X|Cas=X|PNu=X|Per=2|Sem=m
P	SubPOS=S|Gen=Y|Num=P|Cas=4|PNu=S|Per=1|Var=1
P	SubPOS=S|Gen=Y|Num=P|Cas=4|PNu=S|Per=1|Var=7
P	SubPOS=S|Gen=Y|Num=P|Cas=4|PNu=S|Per=2|Var=1
P	SubPOS=S|Gen=Y|Num=S|Cas=1|PNu=P|Per=1
P	SubPOS=S|Gen=Y|Num=S|Cas=1|PNu=P|Per=2
P	SubPOS=S|Gen=Y|Num=S|Cas=1|PNu=S|Per=1
P	SubPOS=S|Gen=Y|Num=S|Cas=1|PNu=S|Per=1|Var=6
P	SubPOS=S|Gen=Y|Num=S|Cas=1|PNu=S|Per=2
P	SubPOS=S|Gen=Y|Num=S|Cas=1|PNu=S|Per=2|Var=6
P	SubPOS=S|Gen=Y|Num=S|Cas=5|PNu=P|Per=1
P	SubPOS=S|Gen=Y|Num=S|Cas=5|PNu=P|Per=2
P	SubPOS=S|Gen=Y|Num=S|Cas=5|PNu=S|Per=1
P	SubPOS=S|Gen=Y|Num=S|Cas=5|PNu=S|Per=1|Var=6
P	SubPOS=S|Gen=Y|Num=S|Cas=5|PNu=S|Per=2
P	SubPOS=S|Gen=Y|Num=S|Cas=5|PNu=S|Per=2|Var=6
P	SubPOS=S|Gen=Z|Num=S|Cas=1|PGe=F|PNu=S|Per=3
P	SubPOS=S|Gen=Z|Num=S|Cas=2|PGe=F|PNu=S|Per=3
P	SubPOS=S|Gen=Z|Num=S|Cas=2|PNu=P|Per=1
P	SubPOS=S|Gen=Z|Num=S|Cas=2|PNu=P|Per=2
P	SubPOS=S|Gen=Z|Num=S|Cas=2|PNu=S|Per=1
P	SubPOS=S|Gen=Z|Num=S|Cas=2|PNu=S|Per=1|Var=6
P	SubPOS=S|Gen=Z|Num=S|Cas=2|PNu=S|Per=2
P	SubPOS=S|Gen=Z|Num=S|Cas=2|PNu=S|Per=2|Var=6
P	SubPOS=S|Gen=Z|Num=S|Cas=3|PGe=F|PNu=S|Per=3
P	SubPOS=S|Gen=Z|Num=S|Cas=3|PNu=P|Per=1
P	SubPOS=S|Gen=Z|Num=S|Cas=3|PNu=P|Per=2
P	SubPOS=S|Gen=Z|Num=S|Cas=3|PNu=S|Per=1
P	SubPOS=S|Gen=Z|Num=S|Cas=3|PNu=S|Per=1|Var=6
P	SubPOS=S|Gen=Z|Num=S|Cas=3|PNu=S|Per=2
P	SubPOS=S|Gen=Z|Num=S|Cas=3|PNu=S|Per=2|Var=6
P	SubPOS=S|Gen=Z|Num=S|Cas=5|PGe=F|PNu=S|Per=3
P	SubPOS=S|Gen=Z|Num=S|Cas=6|PGe=F|PNu=S|Per=3
P	SubPOS=S|Gen=Z|Num=S|Cas=6|PNu=P|Per=1
P	SubPOS=S|Gen=Z|Num=S|Cas=6|PNu=P|Per=2
P	SubPOS=S|Gen=Z|Num=S|Cas=6|PNu=S|Per=1
P	SubPOS=S|Gen=Z|Num=S|Cas=6|PNu=S|Per=1|Var=6
P	SubPOS=S|Gen=Z|Num=S|Cas=6|PNu=S|Per=1|Var=7
P	SubPOS=S|Gen=Z|Num=S|Cas=6|PNu=S|Per=2
P	SubPOS=S|Gen=Z|Num=S|Cas=6|PNu=S|Per=2|Var=6
P	SubPOS=S|Gen=Z|Num=S|Cas=6|PNu=S|Per=2|Var=7
P	SubPOS=S|Gen=Z|Num=S|Cas=7|PGe=F|PNu=S|Per=3
P	SubPOS=S|Gen=Z|Num=S|Cas=7|PNu=P|Per=1
P	SubPOS=S|Gen=Z|Num=S|Cas=7|PNu=P|Per=1|Var=6
P	SubPOS=S|Gen=Z|Num=S|Cas=7|PNu=P|Per=1|Var=8
P	SubPOS=S|Gen=Z|Num=S|Cas=7|PNu=P|Per=2
P	SubPOS=S|Gen=Z|Num=S|Cas=7|PNu=P|Per=2|Var=6
P	SubPOS=S|Gen=Z|Num=S|Cas=7|PNu=S|Per=1
P	SubPOS=S|Gen=Z|Num=S|Cas=7|PNu=S|Per=1|Var=6
P	SubPOS=S|Gen=Z|Num=S|Cas=7|PNu=S|Per=2
P	SubPOS=S|Gen=Z|Num=S|Cas=7|PNu=S|Per=2|Var=6
P	SubPOS=W|Cas=1
P	SubPOS=W|Cas=2
P	SubPOS=W|Cas=2|Var=2
P	SubPOS=W|Cas=3
P	SubPOS=W|Cas=3|Var=2
P	SubPOS=W|Cas=4
P	SubPOS=W|Cas=6
P	SubPOS=W|Cas=6|Var=2
P	SubPOS=W|Cas=7
P	SubPOS=W|Cas=7|Var=2
P	SubPOS=W|Cas=X
P	SubPOS=W|Gen=F|Num=D|Cas=7
P	SubPOS=W|Gen=F|Num=D|Cas=7|Var=6
P	SubPOS=W|Gen=F|Num=P|Cas=1
P	SubPOS=W|Gen=F|Num=P|Cas=1|Var=6
P	SubPOS=W|Gen=F|Num=P|Cas=4
P	SubPOS=W|Gen=F|Num=P|Cas=4|Var=6
P	SubPOS=W|Gen=F|Num=P|Cas=5
P	SubPOS=W|Gen=F|Num=P|Cas=5|Var=6
P	SubPOS=W|Gen=F|Num=S|Cas=1
P	SubPOS=W|Gen=F|Num=S|Cas=2
P	SubPOS=W|Gen=F|Num=S|Cas=2|Var=6
P	SubPOS=W|Gen=F|Num=S|Cas=3
P	SubPOS=W|Gen=F|Num=S|Cas=3|Var=6
P	SubPOS=W|Gen=F|Num=S|Cas=4
P	SubPOS=W|Gen=F|Num=S|Cas=5
P	SubPOS=W|Gen=F|Num=S|Cas=6
P	SubPOS=W|Gen=F|Num=S|Cas=6|Var=6
P	SubPOS=W|Gen=F|Num=S|Cas=7
P	SubPOS=W|Gen=I|Num=P|Cas=1
P	SubPOS=W|Gen=I|Num=P|Cas=1|Var=6
P	SubPOS=W|Gen=I|Num=P|Cas=5
P	SubPOS=W|Gen=I|Num=P|Cas=5|Var=6
P	SubPOS=W|Gen=I|Num=S|Cas=4
P	SubPOS=W|Gen=I|Num=S|Cas=4|Var=6
P	SubPOS=W|Gen=M|Cas=1
P	SubPOS=W|Gen=M|Cas=2
P	SubPOS=W|Gen=M|Cas=3
P	SubPOS=W|Gen=M|Cas=4
P	SubPOS=W|Gen=M|Cas=6
P	SubPOS=W|Gen=M|Cas=7
P	SubPOS=W|Gen=M|Num=P|Cas=1
P	SubPOS=W|Gen=M|Num=P|Cas=1|Var=6
P	SubPOS=W|Gen=M|Num=P|Cas=5
P	SubPOS=W|Gen=M|Num=P|Cas=5|Var=6
P	SubPOS=W|Gen=M|Num=S|Cas=4
P	SubPOS=W|Gen=M|Num=S|Cas=4|Var=6
P	SubPOS=W|Gen=N|Num=P|Cas=1
P	SubPOS=W|Gen=N|Num=P|Cas=1|Var=6
P	SubPOS=W|Gen=N|Num=P|Cas=1|Var=7
P	SubPOS=W|Gen=N|Num=P|Cas=4
P	SubPOS=W|Gen=N|Num=P|Cas=4|Var=6
P	SubPOS=W|Gen=N|Num=P|Cas=5
P	SubPOS=W|Gen=N|Num=P|Cas=5|Var=6
P	SubPOS=W|Gen=N|Num=S|Cas=1
P	SubPOS=W|Gen=N|Num=S|Cas=1|Var=6
P	SubPOS=W|Gen=N|Num=S|Cas=4
P	SubPOS=W|Gen=N|Num=S|Cas=4|Var=6
P	SubPOS=W|Gen=N|Num=S|Cas=5
P	SubPOS=W|Gen=N|Num=S|Cas=5|Var=6
P	SubPOS=W|Gen=X|Num=P|Cas=2
P	SubPOS=W|Gen=X|Num=P|Cas=2|Var=6
P	SubPOS=W|Gen=X|Num=P|Cas=3
P	SubPOS=W|Gen=X|Num=P|Cas=3|Var=6
P	SubPOS=W|Gen=X|Num=P|Cas=6
P	SubPOS=W|Gen=X|Num=P|Cas=6|Var=6
P	SubPOS=W|Gen=X|Num=P|Cas=7
P	SubPOS=W|Gen=X|Num=P|Cas=7|Var=6
P	SubPOS=W|Gen=X|Num=P|Cas=7|Var=7
P	SubPOS=W|Gen=Y|Num=P|Cas=4
P	SubPOS=W|Gen=Y|Num=P|Cas=4|Var=6
P	SubPOS=W|Gen=Y|Num=S|Cas=1
P	SubPOS=W|Gen=Y|Num=S|Cas=1|Var=6
P	SubPOS=W|Gen=Y|Num=S|Cas=5
P	SubPOS=W|Gen=Y|Num=S|Cas=5|Var=6
P	SubPOS=W|Gen=Z|Num=S|Cas=2
P	SubPOS=W|Gen=Z|Num=S|Cas=2|Var=6
P	SubPOS=W|Gen=Z|Num=S|Cas=3
P	SubPOS=W|Gen=Z|Num=S|Cas=3|Var=6
P	SubPOS=W|Gen=Z|Num=S|Cas=6
P	SubPOS=W|Gen=Z|Num=S|Cas=6|Var=6
P	SubPOS=W|Gen=Z|Num=S|Cas=6|Var=7
P	SubPOS=W|Gen=Z|Num=S|Cas=7
P	SubPOS=W|Gen=Z|Num=S|Cas=7|Var=6
P	SubPOS=Y
P	SubPOS=Z|Cas=1
P	SubPOS=Z|Cas=1|Var=1
P	SubPOS=Z|Cas=1|Var=2
P	SubPOS=Z|Cas=1|Var=4
P	SubPOS=Z|Cas=2
P	SubPOS=Z|Cas=2|Var=1
P	SubPOS=Z|Cas=2|Var=2
P	SubPOS=Z|Cas=3
P	SubPOS=Z|Cas=3|Var=1
P	SubPOS=Z|Cas=3|Var=2
P	SubPOS=Z|Cas=4
P	SubPOS=Z|Cas=4|Var=1
P	SubPOS=Z|Cas=4|Var=2
P	SubPOS=Z|Cas=4|Var=4
P	SubPOS=Z|Cas=5
P	SubPOS=Z|Cas=5|Var=1
P	SubPOS=Z|Cas=5|Var=2
P	SubPOS=Z|Cas=6
P	SubPOS=Z|Cas=6|Var=1
P	SubPOS=Z|Cas=6|Var=2
P	SubPOS=Z|Cas=7
P	SubPOS=Z|Cas=7|Var=1
P	SubPOS=Z|Cas=7|Var=2
P	SubPOS=Z|Gen=F|Num=D|Cas=7
P	SubPOS=Z|Gen=F|Num=D|Cas=7|Var=1
P	SubPOS=Z|Gen=F|Num=D|Cas=7|Var=6
P	SubPOS=Z|Gen=F|Num=P|Cas=1
P	SubPOS=Z|Gen=F|Num=P|Cas=1|Var=1
P	SubPOS=Z|Gen=F|Num=P|Cas=1|Var=6
P	SubPOS=Z|Gen=F|Num=P|Cas=4
P	SubPOS=Z|Gen=F|Num=P|Cas=4|Var=1
P	SubPOS=Z|Gen=F|Num=P|Cas=4|Var=6
P	SubPOS=Z|Gen=F|Num=P|Cas=5
P	SubPOS=Z|Gen=F|Num=P|Cas=5|Var=1
P	SubPOS=Z|Gen=F|Num=P|Cas=5|Var=6
P	SubPOS=Z|Gen=F|Num=S|Cas=1
P	SubPOS=Z|Gen=F|Num=S|Cas=1|Var=1
P	SubPOS=Z|Gen=F|Num=S|Cas=1|Var=6
P	SubPOS=Z|Gen=F|Num=S|Cas=2
P	SubPOS=Z|Gen=F|Num=S|Cas=2|Var=1
P	SubPOS=Z|Gen=F|Num=S|Cas=2|Var=6
P	SubPOS=Z|Gen=F|Num=S|Cas=3
P	SubPOS=Z|Gen=F|Num=S|Cas=3|Var=1
P	SubPOS=Z|Gen=F|Num=S|Cas=3|Var=6
P	SubPOS=Z|Gen=F|Num=S|Cas=4
P	SubPOS=Z|Gen=F|Num=S|Cas=4|Var=1
P	SubPOS=Z|Gen=F|Num=S|Cas=4|Var=6
P	SubPOS=Z|Gen=F|Num=S|Cas=5
P	SubPOS=Z|Gen=F|Num=S|Cas=5|Var=1
P	SubPOS=Z|Gen=F|Num=S|Cas=5|Var=6
P	SubPOS=Z|Gen=F|Num=S|Cas=6
P	SubPOS=Z|Gen=F|Num=S|Cas=6|Var=1
P	SubPOS=Z|Gen=F|Num=S|Cas=6|Var=6
P	SubPOS=Z|Gen=F|Num=S|Cas=7
P	SubPOS=Z|Gen=F|Num=S|Cas=7|Var=1
P	SubPOS=Z|Gen=F|Num=S|Cas=7|Var=6
P	SubPOS=Z|Gen=I|Num=P|Cas=1
P	SubPOS=Z|Gen=I|Num=P|Cas=1|Var=1
P	SubPOS=Z|Gen=I|Num=P|Cas=1|Var=6
P	SubPOS=Z|Gen=I|Num=P|Cas=5
P	SubPOS=Z|Gen=I|Num=P|Cas=5|Var=1
P	SubPOS=Z|Gen=I|Num=P|Cas=5|Var=6
P	SubPOS=Z|Gen=I|Num=S|Cas=4
P	SubPOS=Z|Gen=I|Num=S|Cas=4|Var=1
P	SubPOS=Z|Gen=I|Num=S|Cas=4|Var=6
P	SubPOS=Z|Gen=I|Num=S|Cas=6|Var=7
P	SubPOS=Z|Gen=M|Cas=1
P	SubPOS=Z|Gen=M|Cas=1|Var=1
P	SubPOS=Z|Gen=M|Cas=2
P	SubPOS=Z|Gen=M|Cas=2|Var=1
P	SubPOS=Z|Gen=M|Cas=3
P	SubPOS=Z|Gen=M|Cas=3|Var=1
P	SubPOS=Z|Gen=M|Cas=4
P	SubPOS=Z|Gen=M|Cas=4|Var=1
P	SubPOS=Z|Gen=M|Cas=5
P	SubPOS=Z|Gen=M|Cas=5|Var=1
P	SubPOS=Z|Gen=M|Cas=6
P	SubPOS=Z|Gen=M|Cas=6|Var=1
P	SubPOS=Z|Gen=M|Cas=7
P	SubPOS=Z|Gen=M|Cas=7|Var=1
P	SubPOS=Z|Gen=M|Num=P|Cas=1
P	SubPOS=Z|Gen=M|Num=P|Cas=1|Var=1
P	SubPOS=Z|Gen=M|Num=P|Cas=1|Var=6
P	SubPOS=Z|Gen=M|Num=P|Cas=5
P	SubPOS=Z|Gen=M|Num=P|Cas=5|Var=1
P	SubPOS=Z|Gen=M|Num=P|Cas=5|Var=6
P	SubPOS=Z|Gen=M|Num=S|Cas=4
P	SubPOS=Z|Gen=M|Num=S|Cas=4|Var=1
P	SubPOS=Z|Gen=M|Num=S|Cas=4|Var=6
P	SubPOS=Z|Gen=M|Num=S|Cas=6|Var=7
P	SubPOS=Z|Gen=N|Num=P|Cas=1
P	SubPOS=Z|Gen=N|Num=P|Cas=1|Var=1
P	SubPOS=Z|Gen=N|Num=P|Cas=1|Var=6
P	SubPOS=Z|Gen=N|Num=P|Cas=4
P	SubPOS=Z|Gen=N|Num=P|Cas=4|Var=1
P	SubPOS=Z|Gen=N|Num=P|Cas=4|Var=6
P	SubPOS=Z|Gen=N|Num=P|Cas=5
P	SubPOS=Z|Gen=N|Num=P|Cas=5|Var=1
P	SubPOS=Z|Gen=N|Num=P|Cas=5|Var=6
P	SubPOS=Z|Gen=N|Num=S|Cas=1
P	SubPOS=Z|Gen=N|Num=S|Cas=1|Var=1
P	SubPOS=Z|Gen=N|Num=S|Cas=1|Var=6
P	SubPOS=Z|Gen=N|Num=S|Cas=4
P	SubPOS=Z|Gen=N|Num=S|Cas=4|Var=1
P	SubPOS=Z|Gen=N|Num=S|Cas=4|Var=6
P	SubPOS=Z|Gen=N|Num=S|Cas=5
P	SubPOS=Z|Gen=N|Num=S|Cas=5|Var=1
P	SubPOS=Z|Gen=N|Num=S|Cas=5|Var=6
P	SubPOS=Z|Gen=N|Num=S|Cas=6|Var=7
P	SubPOS=Z|Gen=X|Num=P|Cas=2
P	SubPOS=Z|Gen=X|Num=P|Cas=2|Var=1
P	SubPOS=Z|Gen=X|Num=P|Cas=2|Var=6
P	SubPOS=Z|Gen=X|Num=P|Cas=3
P	SubPOS=Z|Gen=X|Num=P|Cas=3|Var=1
P	SubPOS=Z|Gen=X|Num=P|Cas=3|Var=6
P	SubPOS=Z|Gen=X|Num=P|Cas=6
P	SubPOS=Z|Gen=X|Num=P|Cas=6|Var=1
P	SubPOS=Z|Gen=X|Num=P|Cas=6|Var=6
P	SubPOS=Z|Gen=X|Num=P|Cas=7
P	SubPOS=Z|Gen=X|Num=P|Cas=7|Var=1
P	SubPOS=Z|Gen=X|Num=P|Cas=7|Var=6
P	SubPOS=Z|Gen=X|Num=P|Cas=7|Var=7
P	SubPOS=Z|Gen=X|Num=X|Cas=X
P	SubPOS=Z|Gen=Y|Num=P|Cas=4
P	SubPOS=Z|Gen=Y|Num=P|Cas=4|Var=1
P	SubPOS=Z|Gen=Y|Num=P|Cas=4|Var=6
P	SubPOS=Z|Gen=Y|Num=S|Cas=1
P	SubPOS=Z|Gen=Y|Num=S|Cas=1|Var=1
P	SubPOS=Z|Gen=Y|Num=S|Cas=1|Var=6
P	SubPOS=Z|Gen=Y|Num=S|Cas=2
P	SubPOS=Z|Gen=Y|Num=S|Cas=3
P	SubPOS=Z|Gen=Y|Num=S|Cas=5
P	SubPOS=Z|Gen=Y|Num=S|Cas=5|Var=1
P	SubPOS=Z|Gen=Y|Num=S|Cas=5|Var=6
P	SubPOS=Z|Gen=Z|Num=S|Cas=2
P	SubPOS=Z|Gen=Z|Num=S|Cas=2|Var=1
P	SubPOS=Z|Gen=Z|Num=S|Cas=2|Var=6
P	SubPOS=Z|Gen=Z|Num=S|Cas=3
P	SubPOS=Z|Gen=Z|Num=S|Cas=3|Var=1
P	SubPOS=Z|Gen=Z|Num=S|Cas=3|Var=6
P	SubPOS=Z|Gen=Z|Num=S|Cas=6
P	SubPOS=Z|Gen=Z|Num=S|Cas=6|Var=1
P	SubPOS=Z|Gen=Z|Num=S|Cas=6|Var=6
P	SubPOS=Z|Gen=Z|Num=S|Cas=6|Var=7
P	SubPOS=Z|Gen=Z|Num=S|Cas=7
P	SubPOS=Z|Gen=Z|Num=S|Cas=7|Var=1
P	SubPOS=Z|Gen=Z|Num=S|Cas=7|Var=6
R	SubPOS=F
R	SubPOS=R|Cas=1
R	SubPOS=R|Cas=1|Var=8
R	SubPOS=R|Cas=2
R	SubPOS=R|Cas=2|Var=1
R	SubPOS=R|Cas=2|Var=3
R	SubPOS=R|Cas=2|Var=8
R	SubPOS=R|Cas=3
R	SubPOS=R|Cas=3|Var=8
R	SubPOS=R|Cas=4
R	SubPOS=R|Cas=4|Var=8
R	SubPOS=R|Cas=6
R	SubPOS=R|Cas=7
R	SubPOS=R|Cas=7|Var=8
R	SubPOS=R|Cas=X
R	SubPOS=R|Cas=X|Sem=G
R	SubPOS=R|Cas=X|Sem=K
R	SubPOS=R|Cas=X|Sem=R
R	SubPOS=R|Cas=X|Sem=S
R	SubPOS=R|Cas=X|Sem=Y
R	SubPOS=R|Cas=X|Sem=m
R	SubPOS=R|Cas=X|Var=8
R	SubPOS=V|Cas=1
R	SubPOS=V|Cas=2
R	SubPOS=V|Cas=2|Var=1
R	SubPOS=V|Cas=3
R	SubPOS=V|Cas=3|Var=1
R	SubPOS=V|Cas=4
R	SubPOS=V|Cas=4|Var=1
R	SubPOS=V|Cas=6
R	SubPOS=V|Cas=7
T	SubPOS=T|Sem=K
T	SubPOS=T|Sem=S
T	SubPOS=T|Sem=m
T	SubPOS=T|Var=1
T	SubPOS=T|Var=8
T	SubPOS=T
V	SubPOS=B|Num=P|Per=1|Ten=F|Neg=A|Voi=A
V	SubPOS=B|Num=P|Per=1|Ten=F|Neg=A|Voi=A|Var=6
V	SubPOS=B|Num=P|Per=1|Ten=F|Neg=A|Voi=A|Var=7
V	SubPOS=B|Num=P|Per=1|Ten=F|Neg=A|Voi=A|Var=8
V	SubPOS=B|Num=P|Per=1|Ten=F|Neg=N|Voi=A
V	SubPOS=B|Num=P|Per=1|Ten=F|Neg=N|Voi=A|Var=6
V	SubPOS=B|Num=P|Per=1|Ten=F|Neg=N|Voi=A|Var=7
V	SubPOS=B|Num=P|Per=1|Ten=F|Neg=N|Voi=A|Var=8
V	SubPOS=B|Num=P|Per=1|Ten=P|Neg=A|Voi=A
V	SubPOS=B|Num=P|Per=1|Ten=P|Neg=A|Voi=A|Var=1
V	SubPOS=B|Num=P|Per=1|Ten=P|Neg=A|Voi=A|Var=2
V	SubPOS=B|Num=P|Per=1|Ten=P|Neg=A|Voi=A|Var=3
V	SubPOS=B|Num=P|Per=1|Ten=P|Neg=A|Voi=A|Var=6
V	SubPOS=B|Num=P|Per=1|Ten=P|Neg=A|Voi=A|Var=7
V	SubPOS=B|Num=P|Per=1|Ten=P|Neg=N|Voi=A
V	SubPOS=B|Num=P|Per=1|Ten=P|Neg=N|Voi=A|Var=1
V	SubPOS=B|Num=P|Per=1|Ten=P|Neg=N|Voi=A|Var=2
V	SubPOS=B|Num=P|Per=1|Ten=P|Neg=N|Voi=A|Var=3
V	SubPOS=B|Num=P|Per=1|Ten=P|Neg=N|Voi=A|Var=6
V	SubPOS=B|Num=P|Per=1|Ten=P|Neg=N|Voi=A|Var=7
V	SubPOS=B|Num=P|Per=2|Ten=F|Neg=A|Voi=A
V	SubPOS=B|Num=P|Per=2|Ten=F|Neg=A|Voi=A|Var=7
V	SubPOS=B|Num=P|Per=2|Ten=F|Neg=N|Voi=A
V	SubPOS=B|Num=P|Per=2|Ten=F|Neg=N|Voi=A|Var=7
V	SubPOS=B|Num=P|Per=2|Ten=P|Neg=A|Voi=A
V	SubPOS=B|Num=P|Per=2|Ten=P|Neg=A|Voi=A|Var=1
V	SubPOS=B|Num=P|Per=2|Ten=P|Neg=A|Voi=A|Var=2
V	SubPOS=B|Num=P|Per=2|Ten=P|Neg=A|Voi=A|Var=3
V	SubPOS=B|Num=P|Per=2|Ten=P|Neg=A|Voi=A|Var=6
V	SubPOS=B|Num=P|Per=2|Ten=P|Neg=N|Voi=A
V	SubPOS=B|Num=P|Per=2|Ten=P|Neg=N|Voi=A|Var=1
V	SubPOS=B|Num=P|Per=2|Ten=P|Neg=N|Voi=A|Var=2
V	SubPOS=B|Num=P|Per=2|Ten=P|Neg=N|Voi=A|Var=3
V	SubPOS=B|Num=P|Per=2|Ten=P|Neg=N|Voi=A|Var=6
V	SubPOS=B|Num=P|Per=3|Ten=F|Neg=A|Voi=A
V	SubPOS=B|Num=P|Per=3|Ten=F|Neg=A|Voi=A|Var=1
V	SubPOS=B|Num=P|Per=3|Ten=F|Neg=A|Voi=A|Var=7
V	SubPOS=B|Num=P|Per=3|Ten=F|Neg=N|Voi=A
V	SubPOS=B|Num=P|Per=3|Ten=F|Neg=N|Voi=A|Var=1
V	SubPOS=B|Num=P|Per=3|Ten=F|Neg=N|Voi=A|Var=7
V	SubPOS=B|Num=P|Per=3|Ten=P|Neg=A|Voi=A
V	SubPOS=B|Num=P|Per=3|Ten=P|Neg=A|Voi=A|Sem=m
V	SubPOS=B|Num=P|Per=3|Ten=P|Neg=A|Voi=A|Var=1
V	SubPOS=B|Num=P|Per=3|Ten=P|Neg=A|Voi=A|Var=2
V	SubPOS=B|Num=P|Per=3|Ten=P|Neg=A|Voi=A|Var=3
V	SubPOS=B|Num=P|Per=3|Ten=P|Neg=A|Voi=A|Var=4
V	SubPOS=B|Num=P|Per=3|Ten=P|Neg=A|Voi=A|Var=5
V	SubPOS=B|Num=P|Per=3|Ten=P|Neg=A|Voi=A|Var=6
V	SubPOS=B|Num=P|Per=3|Ten=P|Neg=A|Voi=A|Var=7
V	SubPOS=B|Num=P|Per=3|Ten=P|Neg=N|Voi=A
V	SubPOS=B|Num=P|Per=3|Ten=P|Neg=N|Voi=A|Var=1
V	SubPOS=B|Num=P|Per=3|Ten=P|Neg=N|Voi=A|Var=2
V	SubPOS=B|Num=P|Per=3|Ten=P|Neg=N|Voi=A|Var=3
V	SubPOS=B|Num=P|Per=3|Ten=P|Neg=N|Voi=A|Var=4
V	SubPOS=B|Num=P|Per=3|Ten=P|Neg=N|Voi=A|Var=5
V	SubPOS=B|Num=P|Per=3|Ten=P|Neg=N|Voi=A|Var=6
V	SubPOS=B|Num=P|Per=3|Ten=P|Neg=N|Voi=A|Var=7
V	SubPOS=B|Num=S|Per=1|Ten=F|Neg=A|Voi=A
V	SubPOS=B|Num=S|Per=1|Ten=F|Neg=A|Voi=A|Var=1
V	SubPOS=B|Num=S|Per=1|Ten=F|Neg=A|Voi=A|Var=7
V	SubPOS=B|Num=S|Per=1|Ten=F|Neg=N|Voi=A
V	SubPOS=B|Num=S|Per=1|Ten=F|Neg=N|Voi=A|Var=1
V	SubPOS=B|Num=S|Per=1|Ten=F|Neg=N|Voi=A|Var=7
V	SubPOS=B|Num=S|Per=1|Ten=P|Neg=A|Voi=A
V	SubPOS=B|Num=S|Per=1|Ten=P|Neg=A|Voi=A|Sem=m
V	SubPOS=B|Num=S|Per=1|Ten=P|Neg=A|Voi=A|Var=1
V	SubPOS=B|Num=S|Per=1|Ten=P|Neg=A|Voi=A|Var=2
V	SubPOS=B|Num=S|Per=1|Ten=P|Neg=A|Voi=A|Var=3
V	SubPOS=B|Num=S|Per=1|Ten=P|Neg=A|Voi=A|Var=4
V	SubPOS=B|Num=S|Per=1|Ten=P|Neg=A|Voi=A|Var=6
V	SubPOS=B|Num=S|Per=1|Ten=P|Neg=A|Voi=A|Var=7
V	SubPOS=B|Num=S|Per=1|Ten=P|Neg=N|Voi=A
V	SubPOS=B|Num=S|Per=1|Ten=P|Neg=N|Voi=A|Var=1
V	SubPOS=B|Num=S|Per=1|Ten=P|Neg=N|Voi=A|Var=2
V	SubPOS=B|Num=S|Per=1|Ten=P|Neg=N|Voi=A|Var=3
V	SubPOS=B|Num=S|Per=1|Ten=P|Neg=N|Voi=A|Var=4
V	SubPOS=B|Num=S|Per=1|Ten=P|Neg=N|Voi=A|Var=6
V	SubPOS=B|Num=S|Per=2|Ten=F|Neg=A|Voi=A
V	SubPOS=B|Num=S|Per=2|Ten=F|Neg=A|Voi=A|Var=7
V	SubPOS=B|Num=S|Per=2|Ten=F|Neg=N|Voi=A
V	SubPOS=B|Num=S|Per=2|Ten=F|Neg=N|Voi=A|Var=7
V	SubPOS=B|Num=S|Per=2|Ten=P|Neg=A|Voi=A
V	SubPOS=B|Num=S|Per=2|Ten=P|Neg=A|Voi=A|Var=1
V	SubPOS=B|Num=S|Per=2|Ten=P|Neg=A|Voi=A|Var=2
V	SubPOS=B|Num=S|Per=2|Ten=P|Neg=A|Voi=A|Var=3
V	SubPOS=B|Num=S|Per=2|Ten=P|Neg=A|Voi=A|Var=6
V	SubPOS=B|Num=S|Per=2|Ten=P|Neg=A|Voi=A|Var=7
V	SubPOS=B|Num=S|Per=2|Ten=P|Neg=N|Voi=A
V	SubPOS=B|Num=S|Per=2|Ten=P|Neg=N|Voi=A|Var=1
V	SubPOS=B|Num=S|Per=2|Ten=P|Neg=N|Voi=A|Var=2
V	SubPOS=B|Num=S|Per=2|Ten=P|Neg=N|Voi=A|Var=3
V	SubPOS=B|Num=S|Per=2|Ten=P|Neg=N|Voi=A|Var=6
V	SubPOS=B|Num=S|Per=3|Ten=F|Neg=A|Voi=A
V	SubPOS=B|Num=S|Per=3|Ten=F|Neg=A|Voi=A|Var=7
V	SubPOS=B|Num=S|Per=3|Ten=F|Neg=N|Voi=A
V	SubPOS=B|Num=S|Per=3|Ten=F|Neg=N|Voi=A|Var=7
V	SubPOS=B|Num=S|Per=3|Ten=P|Neg=A|Voi=A
V	SubPOS=B|Num=S|Per=3|Ten=P|Neg=A|Voi=A|Sem=m
V	SubPOS=B|Num=S|Per=3|Ten=P|Neg=A|Voi=A|Var=1
V	SubPOS=B|Num=S|Per=3|Ten=P|Neg=A|Voi=A|Var=2
V	SubPOS=B|Num=S|Per=3|Ten=P|Neg=A|Voi=A|Var=3
V	SubPOS=B|Num=S|Per=3|Ten=P|Neg=A|Voi=A|Var=4
V	SubPOS=B|Num=S|Per=3|Ten=P|Neg=A|Voi=A|Var=5
V	SubPOS=B|Num=S|Per=3|Ten=P|Neg=A|Voi=A|Var=6
V	SubPOS=B|Num=S|Per=3|Ten=P|Neg=A|Voi=A|Var=7
V	SubPOS=B|Num=S|Per=3|Ten=P|Neg=A|Voi=A|Var=8
V	SubPOS=B|Num=S|Per=3|Ten=P|Neg=N|Voi=A
V	SubPOS=B|Num=S|Per=3|Ten=P|Neg=N|Voi=A|Var=1
V	SubPOS=B|Num=S|Per=3|Ten=P|Neg=N|Voi=A|Var=2
V	SubPOS=B|Num=S|Per=3|Ten=P|Neg=N|Voi=A|Var=3
V	SubPOS=B|Num=S|Per=3|Ten=P|Neg=N|Voi=A|Var=4
V	SubPOS=B|Num=S|Per=3|Ten=P|Neg=N|Voi=A|Var=5
V	SubPOS=B|Num=S|Per=3|Ten=P|Neg=N|Voi=A|Var=6
V	SubPOS=B|Num=S|Per=3|Ten=P|Neg=N|Voi=A|Var=7
V	SubPOS=B|Num=X|Per=X|Ten=F|Neg=A|Voi=A
V	SubPOS=B|Num=X|Per=X|Ten=F|Neg=N|Voi=A
V	SubPOS=B|Num=X|Per=X|Ten=P|Neg=A|Voi=A
V	SubPOS=B|Num=X|Per=X|Ten=P|Neg=A|Voi=A|Sem=m
V	SubPOS=B|Num=X|Per=X|Ten=P|Neg=N|Voi=A
V	SubPOS=c|Num=P|Per=1
V	SubPOS=c|Num=P|Per=1|Var=6
V	SubPOS=c|Num=P|Per=2
V	SubPOS=c|Num=S|Per=1
V	SubPOS=c|Num=S|Per=1|Var=6
V	SubPOS=c|Num=S|Per=2
V	SubPOS=c
V	SubPOS=e|Gen=H|Num=S|Neg=A
V	SubPOS=e|Gen=H|Num=S|Neg=A|Var=2
V	SubPOS=e|Gen=H|Num=S|Neg=N
V	SubPOS=e|Gen=H|Num=S|Neg=N|Var=2
V	SubPOS=e|Gen=X|Num=P|Neg=A
V	SubPOS=e|Gen=X|Num=P|Neg=A|Var=2
V	SubPOS=e|Gen=X|Num=P|Neg=N
V	SubPOS=e|Gen=X|Num=P|Neg=N|Var=2
V	SubPOS=e|Gen=Y|Num=S|Neg=A
V	SubPOS=e|Gen=Y|Num=S|Neg=N
V	SubPOS=f|Neg=A
V	SubPOS=f|Neg=A|Sem=K
V	SubPOS=f|Neg=A|Sem=R
V	SubPOS=f|Neg=A|Sem=m
V	SubPOS=f|Neg=A|Var=1
V	SubPOS=f|Neg=A|Var=2
V	SubPOS=f|Neg=A|Var=3
V	SubPOS=f|Neg=A|Var=4
V	SubPOS=f|Neg=A|Var=6
V	SubPOS=f|Neg=A|Var=8
V	SubPOS=f|Neg=N
V	SubPOS=f|Neg=N|Var=1
V	SubPOS=f|Neg=N|Var=2
V	SubPOS=f|Neg=N|Var=3
V	SubPOS=f|Neg=N|Var=4
V	SubPOS=f|Neg=N|Var=6
V	SubPOS=i|Num=P|Per=1|Neg=A
V	SubPOS=i|Num=P|Per=1|Neg=A|Var=1
V	SubPOS=i|Num=P|Per=1|Neg=A|Var=2
V	SubPOS=i|Num=P|Per=1|Neg=A|Var=3
V	SubPOS=i|Num=P|Per=1|Neg=A|Var=6
V	SubPOS=i|Num=P|Per=1|Neg=N
V	SubPOS=i|Num=P|Per=1|Neg=N|Var=1
V	SubPOS=i|Num=P|Per=1|Neg=N|Var=2
V	SubPOS=i|Num=P|Per=1|Neg=N|Var=3
V	SubPOS=i|Num=P|Per=1|Neg=N|Var=6
V	SubPOS=i|Num=P|Per=2|Neg=A
V	SubPOS=i|Num=P|Per=2|Neg=A|Var=1
V	SubPOS=i|Num=P|Per=2|Neg=A|Var=2
V	SubPOS=i|Num=P|Per=2|Neg=A|Var=3
V	SubPOS=i|Num=P|Per=2|Neg=A|Var=6
V	SubPOS=i|Num=P|Per=2|Neg=A|Var=7
V	SubPOS=i|Num=P|Per=2|Neg=N
V	SubPOS=i|Num=P|Per=2|Neg=N|Var=1
V	SubPOS=i|Num=P|Per=2|Neg=N|Var=2
V	SubPOS=i|Num=P|Per=2|Neg=N|Var=3
V	SubPOS=i|Num=P|Per=2|Neg=N|Var=6
V	SubPOS=i|Num=P|Per=2|Neg=N|Var=7
V	SubPOS=i|Num=P|Per=3|Neg=A|Var=1
V	SubPOS=i|Num=P|Per=3|Neg=A|Var=2
V	SubPOS=i|Num=P|Per=3|Neg=A|Var=3
V	SubPOS=i|Num=P|Per=3|Neg=A|Var=4
V	SubPOS=i|Num=P|Per=3|Neg=A|Var=9
V	SubPOS=i|Num=P|Per=3|Neg=N|Var=1
V	SubPOS=i|Num=P|Per=3|Neg=N|Var=2
V	SubPOS=i|Num=P|Per=3|Neg=N|Var=3
V	SubPOS=i|Num=P|Per=3|Neg=N|Var=4
V	SubPOS=i|Num=P|Per=3|Neg=N|Var=9
V	SubPOS=i|Num=S|Per=2|Neg=A
V	SubPOS=i|Num=S|Per=2|Neg=A|Var=1
V	SubPOS=i|Num=S|Per=2|Neg=A|Var=2
V	SubPOS=i|Num=S|Per=2|Neg=A|Var=6
V	SubPOS=i|Num=S|Per=2|Neg=A|Var=7
V	SubPOS=i|Num=S|Per=2|Neg=A|Var=8
V	SubPOS=i|Num=S|Per=2|Neg=N
V	SubPOS=i|Num=S|Per=2|Neg=N|Var=1
V	SubPOS=i|Num=S|Per=2|Neg=N|Var=2
V	SubPOS=i|Num=S|Per=2|Neg=N|Var=6
V	SubPOS=i|Num=S|Per=2|Neg=N|Var=7
V	SubPOS=i|Num=S|Per=3|Neg=A
V	SubPOS=i|Num=S|Per=3|Neg=A|Var=2
V	SubPOS=i|Num=S|Per=3|Neg=A|Var=3
V	SubPOS=i|Num=S|Per=3|Neg=A|Var=4
V	SubPOS=i|Num=S|Per=3|Neg=A|Var=9
V	SubPOS=i|Num=S|Per=3|Neg=N
V	SubPOS=i|Num=S|Per=3|Neg=N|Var=2
V	SubPOS=i|Num=S|Per=3|Neg=N|Var=3
V	SubPOS=i|Num=S|Per=3|Neg=N|Var=4
V	SubPOS=i|Num=S|Per=3|Neg=N|Var=9
V	SubPOS=i|Num=X|Per=2|Neg=A
V	SubPOS=i|Num=X|Per=2|Neg=N
V	SubPOS=m|Gen=H|Num=S|Neg=A
V	SubPOS=m|Gen=H|Num=S|Neg=A|Var=4
V	SubPOS=m|Gen=H|Num=S|Neg=N
V	SubPOS=m|Gen=H|Num=S|Neg=N|Var=4
V	SubPOS=m|Gen=X|Num=P|Neg=A
V	SubPOS=m|Gen=X|Num=P|Neg=A|Var=4
V	SubPOS=m|Gen=X|Num=P|Neg=N
V	SubPOS=m|Gen=X|Num=P|Neg=N|Var=4
V	SubPOS=m|Gen=Y|Num=S|Neg=A
V	SubPOS=m|Gen=Y|Num=S|Neg=A|Var=4
V	SubPOS=m|Gen=Y|Num=S|Neg=N
V	SubPOS=m|Gen=Y|Num=S|Neg=N|Var=4
V	SubPOS=p|Gen=F|Num=S|Per=2|Ten=R|Neg=A|Voi=A
V	SubPOS=p|Gen=F|Num=S|Per=2|Ten=R|Neg=A|Voi=A|Var=1
V	SubPOS=p|Gen=F|Num=S|Per=2|Ten=R|Neg=A|Voi=A|Var=6
V	SubPOS=p|Gen=F|Num=S|Per=2|Ten=R|Neg=N|Voi=A
V	SubPOS=p|Gen=F|Num=S|Per=2|Ten=R|Neg=N|Voi=A|Var=1
V	SubPOS=p|Gen=F|Num=S|Per=2|Ten=R|Neg=N|Voi=A|Var=6
V	SubPOS=p|Gen=M|Num=P|Per=X|Ten=R|Neg=A|Voi=A
V	SubPOS=p|Gen=M|Num=P|Per=X|Ten=R|Neg=A|Voi=A|Var=1
V	SubPOS=p|Gen=M|Num=P|Per=X|Ten=R|Neg=A|Voi=A|Var=3
V	SubPOS=p|Gen=M|Num=P|Per=X|Ten=R|Neg=A|Voi=A|Var=6
V	SubPOS=p|Gen=M|Num=P|Per=X|Ten=R|Neg=N|Voi=A
V	SubPOS=p|Gen=M|Num=P|Per=X|Ten=R|Neg=N|Voi=A|Var=1
V	SubPOS=p|Gen=M|Num=P|Per=X|Ten=R|Neg=N|Voi=A|Var=3
V	SubPOS=p|Gen=M|Num=P|Per=X|Ten=R|Neg=N|Voi=A|Var=6
V	SubPOS=p|Gen=N|Num=S|Per=2|Ten=R|Neg=A|Voi=A
V	SubPOS=p|Gen=N|Num=S|Per=2|Ten=R|Neg=A|Voi=A|Var=1
V	SubPOS=p|Gen=N|Num=S|Per=2|Ten=R|Neg=A|Voi=A|Var=6
V	SubPOS=p|Gen=N|Num=S|Per=2|Ten=R|Neg=N|Voi=A
V	SubPOS=p|Gen=N|Num=S|Per=2|Ten=R|Neg=N|Voi=A|Var=1
V	SubPOS=p|Gen=N|Num=S|Per=2|Ten=R|Neg=N|Voi=A|Var=6
V	SubPOS=p|Gen=N|Num=S|Per=X|Ten=R|Neg=A|Voi=A
V	SubPOS=p|Gen=N|Num=S|Per=X|Ten=R|Neg=A|Voi=A|Var=1
V	SubPOS=p|Gen=N|Num=S|Per=X|Ten=R|Neg=A|Voi=A|Var=3
V	SubPOS=p|Gen=N|Num=S|Per=X|Ten=R|Neg=A|Voi=A|Var=6
V	SubPOS=p|Gen=N|Num=S|Per=X|Ten=R|Neg=N|Voi=A
V	SubPOS=p|Gen=N|Num=S|Per=X|Ten=R|Neg=N|Voi=A|Var=1
V	SubPOS=p|Gen=N|Num=S|Per=X|Ten=R|Neg=N|Voi=A|Var=3
V	SubPOS=p|Gen=N|Num=S|Per=X|Ten=R|Neg=N|Voi=A|Var=6
V	SubPOS=p|Gen=Q|Num=W|Per=X|Ten=R|Neg=A|Voi=A
V	SubPOS=p|Gen=Q|Num=W|Per=X|Ten=R|Neg=A|Voi=A|Var=1
V	SubPOS=p|Gen=Q|Num=W|Per=X|Ten=R|Neg=A|Voi=A|Var=3
V	SubPOS=p|Gen=Q|Num=W|Per=X|Ten=R|Neg=A|Voi=A|Var=6
V	SubPOS=p|Gen=Q|Num=W|Per=X|Ten=R|Neg=N|Voi=A
V	SubPOS=p|Gen=Q|Num=W|Per=X|Ten=R|Neg=N|Voi=A|Var=1
V	SubPOS=p|Gen=Q|Num=W|Per=X|Ten=R|Neg=N|Voi=A|Var=3
V	SubPOS=p|Gen=Q|Num=W|Per=X|Ten=R|Neg=N|Voi=A|Var=6
V	SubPOS=p|Gen=T|Num=P|Per=X|Ten=R|Neg=A|Voi=A
V	SubPOS=p|Gen=T|Num=P|Per=X|Ten=R|Neg=A|Voi=A|Var=1
V	SubPOS=p|Gen=T|Num=P|Per=X|Ten=R|Neg=A|Voi=A|Var=3
V	SubPOS=p|Gen=T|Num=P|Per=X|Ten=R|Neg=A|Voi=A|Var=6
V	SubPOS=p|Gen=T|Num=P|Per=X|Ten=R|Neg=N|Voi=A
V	SubPOS=p|Gen=T|Num=P|Per=X|Ten=R|Neg=N|Voi=A|Var=1
V	SubPOS=p|Gen=T|Num=P|Per=X|Ten=R|Neg=N|Voi=A|Var=3
V	SubPOS=p|Gen=T|Num=P|Per=X|Ten=R|Neg=N|Voi=A|Var=6
V	SubPOS=p|Gen=X|Num=P|Per=X|Ten=R|Neg=A|Voi=A
V	SubPOS=p|Gen=X|Num=P|Per=X|Ten=R|Neg=N|Voi=A
V	SubPOS=p|Gen=X|Num=S|Per=X|Ten=R|Neg=A|Voi=A
V	SubPOS=p|Gen=X|Num=S|Per=X|Ten=R|Neg=N|Voi=A
V	SubPOS=p|Gen=X|Num=X|Per=X|Ten=R|Neg=A|Voi=A
V	SubPOS=p|Gen=X|Num=X|Per=X|Ten=R|Neg=A|Voi=A|Sem=m
V	SubPOS=p|Gen=X|Num=X|Per=X|Ten=R|Neg=N|Voi=A
V	SubPOS=p|Gen=Y|Num=S|Per=2|Ten=R|Neg=A|Voi=A
V	SubPOS=p|Gen=Y|Num=S|Per=2|Ten=R|Neg=A|Voi=A|Var=1
V	SubPOS=p|Gen=Y|Num=S|Per=2|Ten=R|Neg=A|Voi=A|Var=6
V	SubPOS=p|Gen=Y|Num=S|Per=2|Ten=R|Neg=N|Voi=A
V	SubPOS=p|Gen=Y|Num=S|Per=2|Ten=R|Neg=N|Voi=A|Var=1
V	SubPOS=p|Gen=Y|Num=S|Per=2|Ten=R|Neg=N|Voi=A|Var=6
V	SubPOS=p|Gen=Y|Num=S|Per=X|Ten=R|Neg=A|Voi=A
V	SubPOS=p|Gen=Y|Num=S|Per=X|Ten=R|Neg=A|Voi=A|Var=1
V	SubPOS=p|Gen=Y|Num=S|Per=X|Ten=R|Neg=A|Voi=A|Var=6
V	SubPOS=p|Gen=Y|Num=S|Per=X|Ten=R|Neg=A|Voi=A|Var=8
V	SubPOS=p|Gen=Y|Num=S|Per=X|Ten=R|Neg=A|Voi=A|Var=9
V	SubPOS=p|Gen=Y|Num=S|Per=X|Ten=R|Neg=N|Voi=A
V	SubPOS=p|Gen=Y|Num=S|Per=X|Ten=R|Neg=N|Voi=A|Var=1
V	SubPOS=p|Gen=Y|Num=S|Per=X|Ten=R|Neg=N|Voi=A|Var=6
V	SubPOS=p|Gen=Y|Num=S|Per=X|Ten=R|Neg=N|Voi=A|Var=8
V	SubPOS=q|Gen=M|Num=P|Per=X|Ten=R|Neg=A|Voi=A|Var=2
V	SubPOS=q|Gen=M|Num=P|Per=X|Ten=R|Neg=A|Voi=A|Var=3
V	SubPOS=q|Gen=M|Num=P|Per=X|Ten=R|Neg=A|Voi=A|Var=7
V	SubPOS=q|Gen=M|Num=P|Per=X|Ten=R|Neg=N|Voi=A|Var=2
V	SubPOS=q|Gen=M|Num=P|Per=X|Ten=R|Neg=N|Voi=A|Var=3
V	SubPOS=q|Gen=M|Num=P|Per=X|Ten=R|Neg=N|Voi=A|Var=7
V	SubPOS=q|Gen=N|Num=S|Per=X|Ten=R|Neg=A|Voi=A|Var=2
V	SubPOS=q|Gen=N|Num=S|Per=X|Ten=R|Neg=A|Voi=A|Var=3
V	SubPOS=q|Gen=N|Num=S|Per=X|Ten=R|Neg=A|Voi=A|Var=7
V	SubPOS=q|Gen=N|Num=S|Per=X|Ten=R|Neg=N|Voi=A|Var=2
V	SubPOS=q|Gen=N|Num=S|Per=X|Ten=R|Neg=N|Voi=A|Var=3
V	SubPOS=q|Gen=N|Num=S|Per=X|Ten=R|Neg=N|Voi=A|Var=7
V	SubPOS=q|Gen=Q|Num=W|Per=X|Ten=R|Neg=A|Voi=A|Var=2
V	SubPOS=q|Gen=Q|Num=W|Per=X|Ten=R|Neg=A|Voi=A|Var=3
V	SubPOS=q|Gen=Q|Num=W|Per=X|Ten=R|Neg=A|Voi=A|Var=7
V	SubPOS=q|Gen=Q|Num=W|Per=X|Ten=R|Neg=N|Voi=A|Var=2
V	SubPOS=q|Gen=Q|Num=W|Per=X|Ten=R|Neg=N|Voi=A|Var=3
V	SubPOS=q|Gen=Q|Num=W|Per=X|Ten=R|Neg=N|Voi=A|Var=7
V	SubPOS=q|Gen=T|Num=P|Per=X|Ten=R|Neg=A|Voi=A|Var=2
V	SubPOS=q|Gen=T|Num=P|Per=X|Ten=R|Neg=A|Voi=A|Var=3
V	SubPOS=q|Gen=T|Num=P|Per=X|Ten=R|Neg=A|Voi=A|Var=7
V	SubPOS=q|Gen=T|Num=P|Per=X|Ten=R|Neg=N|Voi=A|Var=2
V	SubPOS=q|Gen=T|Num=P|Per=X|Ten=R|Neg=N|Voi=A|Var=3
V	SubPOS=q|Gen=T|Num=P|Per=X|Ten=R|Neg=N|Voi=A|Var=7
V	SubPOS=q|Gen=Y|Num=S|Per=X|Ten=R|Neg=A|Voi=A|Var=2
V	SubPOS=q|Gen=Y|Num=S|Per=X|Ten=R|Neg=A|Voi=A|Var=3
V	SubPOS=q|Gen=Y|Num=S|Per=X|Ten=R|Neg=A|Voi=A|Var=7
V	SubPOS=q|Gen=Y|Num=S|Per=X|Ten=R|Neg=N|Voi=A|Var=2
V	SubPOS=q|Gen=Y|Num=S|Per=X|Ten=R|Neg=N|Voi=A|Var=3
V	SubPOS=q|Gen=Y|Num=S|Per=X|Ten=R|Neg=N|Voi=A|Var=7
V	SubPOS=s|Gen=F|Num=S|Cas=4|Per=X|Ten=X|Neg=A|Voi=P
V	SubPOS=s|Gen=F|Num=S|Cas=4|Per=X|Ten=X|Neg=A|Voi=P|Var=2
V	SubPOS=s|Gen=F|Num=S|Cas=4|Per=X|Ten=X|Neg=N|Voi=P
V	SubPOS=s|Gen=F|Num=S|Cas=4|Per=X|Ten=X|Neg=N|Voi=P|Var=2
V	SubPOS=s|Gen=F|Num=S|Per=2|Ten=H|Neg=A|Voi=P
V	SubPOS=s|Gen=F|Num=S|Per=2|Ten=H|Neg=A|Voi=P|Var=2
V	SubPOS=s|Gen=F|Num=S|Per=2|Ten=H|Neg=N|Voi=P
V	SubPOS=s|Gen=F|Num=S|Per=2|Ten=H|Neg=N|Voi=P|Var=2
V	SubPOS=s|Gen=M|Num=P|Per=X|Ten=X|Neg=A|Voi=P
V	SubPOS=s|Gen=M|Num=P|Per=X|Ten=X|Neg=A|Voi=P|Var=2
V	SubPOS=s|Gen=M|Num=P|Per=X|Ten=X|Neg=N|Voi=P
V	SubPOS=s|Gen=M|Num=P|Per=X|Ten=X|Neg=N|Voi=P|Var=2
V	SubPOS=s|Gen=N|Num=S|Per=2|Ten=H|Neg=A|Voi=P
V	SubPOS=s|Gen=N|Num=S|Per=2|Ten=H|Neg=A|Voi=P|Var=2
V	SubPOS=s|Gen=N|Num=S|Per=2|Ten=H|Neg=N|Voi=P
V	SubPOS=s|Gen=N|Num=S|Per=2|Ten=H|Neg=N|Voi=P|Var=2
V	SubPOS=s|Gen=N|Num=S|Per=X|Ten=X|Neg=A|Voi=P
V	SubPOS=s|Gen=N|Num=S|Per=X|Ten=X|Neg=A|Voi=P|Var=2
V	SubPOS=s|Gen=N|Num=S|Per=X|Ten=X|Neg=N|Voi=P
V	SubPOS=s|Gen=N|Num=S|Per=X|Ten=X|Neg=N|Voi=P|Var=2
V	SubPOS=s|Gen=Q|Num=W|Per=X|Ten=X|Neg=A|Voi=P
V	SubPOS=s|Gen=Q|Num=W|Per=X|Ten=X|Neg=A|Voi=P|Var=2
V	SubPOS=s|Gen=Q|Num=W|Per=X|Ten=X|Neg=N|Voi=P
V	SubPOS=s|Gen=Q|Num=W|Per=X|Ten=X|Neg=N|Voi=P|Var=2
V	SubPOS=s|Gen=T|Num=P|Per=X|Ten=X|Neg=A|Voi=P
V	SubPOS=s|Gen=T|Num=P|Per=X|Ten=X|Neg=A|Voi=P|Var=2
V	SubPOS=s|Gen=T|Num=P|Per=X|Ten=X|Neg=N|Voi=P
V	SubPOS=s|Gen=T|Num=P|Per=X|Ten=X|Neg=N|Voi=P|Var=2
V	SubPOS=s|Gen=Y|Num=S|Per=2|Ten=H|Neg=A|Voi=P
V	SubPOS=s|Gen=Y|Num=S|Per=X|Ten=X|Neg=A|Voi=P
V	SubPOS=s|Gen=Y|Num=S|Per=X|Ten=X|Neg=A|Voi=P|Var=2
V	SubPOS=s|Gen=Y|Num=S|Per=X|Ten=X|Neg=N|Voi=P
V	SubPOS=s|Gen=Y|Num=S|Per=X|Ten=X|Neg=N|Voi=P|Var=2
V	SubPOS=t|Num=P|Per=1|Ten=F|Neg=A|Voi=A|Var=2
V	SubPOS=t|Num=P|Per=1|Ten=F|Neg=A|Voi=A|Var=3
V	SubPOS=t|Num=P|Per=1|Ten=F|Neg=N|Voi=A|Var=2
V	SubPOS=t|Num=P|Per=1|Ten=F|Neg=N|Voi=A|Var=3
V	SubPOS=t|Num=P|Per=1|Ten=P|Neg=A|Voi=A|Var=2
V	SubPOS=t|Num=P|Per=1|Ten=P|Neg=A|Voi=A|Var=3
V	SubPOS=t|Num=P|Per=1|Ten=P|Neg=N|Voi=A|Var=2
V	SubPOS=t|Num=P|Per=1|Ten=P|Neg=N|Voi=A|Var=3
V	SubPOS=t|Num=P|Per=2|Ten=F|Neg=A|Voi=A|Var=2
V	SubPOS=t|Num=P|Per=2|Ten=F|Neg=N|Voi=A|Var=2
V	SubPOS=t|Num=P|Per=2|Ten=P|Neg=A|Voi=A|Var=2
V	SubPOS=t|Num=P|Per=2|Ten=P|Neg=N|Voi=A|Var=2
V	SubPOS=t|Num=P|Per=3|Ten=F|Neg=A|Voi=A|Var=2
V	SubPOS=t|Num=P|Per=3|Ten=F|Neg=A|Voi=A|Var=3
V	SubPOS=t|Num=P|Per=3|Ten=F|Neg=N|Voi=A|Var=2
V	SubPOS=t|Num=P|Per=3|Ten=F|Neg=N|Voi=A|Var=3
V	SubPOS=t|Num=P|Per=3|Ten=P|Neg=A|Voi=A|Var=2
V	SubPOS=t|Num=P|Per=3|Ten=P|Neg=A|Voi=A|Var=3
V	SubPOS=t|Num=P|Per=3|Ten=P|Neg=A|Voi=A|Var=9
V	SubPOS=t|Num=P|Per=3|Ten=P|Neg=N|Voi=A|Var=2
V	SubPOS=t|Num=P|Per=3|Ten=P|Neg=N|Voi=A|Var=3
V	SubPOS=t|Num=P|Per=3|Ten=P|Neg=N|Voi=A|Var=9
V	SubPOS=t|Num=S|Per=1|Ten=F|Neg=A|Voi=A|Var=2
V	SubPOS=t|Num=S|Per=1|Ten=F|Neg=A|Voi=A|Var=3
V	SubPOS=t|Num=S|Per=1|Ten=F|Neg=N|Voi=A|Var=2
V	SubPOS=t|Num=S|Per=1|Ten=F|Neg=N|Voi=A|Var=3
V	SubPOS=t|Num=S|Per=1|Ten=P|Neg=A|Voi=A|Var=2
V	SubPOS=t|Num=S|Per=1|Ten=P|Neg=A|Voi=A|Var=3
V	SubPOS=t|Num=S|Per=1|Ten=P|Neg=N|Voi=A|Var=2
V	SubPOS=t|Num=S|Per=1|Ten=P|Neg=N|Voi=A|Var=3
V	SubPOS=t|Num=S|Per=2|Ten=F|Neg=A|Voi=A|Var=2
V	SubPOS=t|Num=S|Per=2|Ten=F|Neg=N|Voi=A|Var=2
V	SubPOS=t|Num=S|Per=2|Ten=P|Neg=A|Voi=A|Var=2
V	SubPOS=t|Num=S|Per=2|Ten=P|Neg=N|Voi=A|Var=2
V	SubPOS=t|Num=S|Per=3|Ten=F|Neg=A|Voi=A|Var=2
V	SubPOS=t|Num=S|Per=3|Ten=F|Neg=N|Voi=A|Var=2
V	SubPOS=t|Num=S|Per=3|Ten=P|Neg=A|Voi=A|Var=2
V	SubPOS=t|Num=S|Per=3|Ten=P|Neg=N|Voi=A|Var=2
V	SubPOS=~
X	SubPOS=@|Sem=G
X	SubPOS=@|Sem=m
X	SubPOS=@|Var=0
X	SubPOS=@|Var=1
X	SubPOS=@
X	SubPOS=X|Var=8
X	SubPOS=X
X	SubPOS=x|Sem=K
X	SubPOS=x|Sem=m
X	SubPOS=x
Z	SubPOS=#
Z	SubPOS=:
end_of_list
    ;
    # Protect from editors that replace tabs by spaces.
    $list =~ s/ \s+/\t/sg;
    my @list = split(/\r?\n/, $list);
    return \@list;
}



1;

__END__

=pod

=encoding UTF-8

=head1 NAME

Lingua::Interset::Tagset::CS::Conll2009 - Driver for the Czech tagset of the CoNLL 2009 Shared Task.

=head1 VERSION

version 2.050

=head1 SYNOPSIS

  use Lingua::Interset::Tagset::CS::Conll2009;
  my $driver = Lingua::Interset::Tagset::CS::Conll2009->new();
  my $fs = $driver->decode("N\tSubPOS=N|Gen=M|Num=S|Cas=1|Neg=A");

or

  use Lingua::Interset qw(decode);
  my $fs = decode('cs::conll2009', "N\tSubPOS=N|Gen=M|Num=S|Cas=1|Neg=A");

=head1 DESCRIPTION

Interset driver for the Czech tagset of the CoNLL 2009 Shared Task.
CoNLL 2009 tagsets in Interset are traditionally two values separated by tabs.
The values come from the CoNLL 2009 columns POS and FEAT. For Czech,
these values are derived from the tagset of the Prague Dependency Treebank; however,
there is an additional surface feature C<Sem>, which is derived from PDT lemmas.
The CoNLL 2009 tagset differs slightly from CoNLL 2006 and 2007:
the (fine-grained) C<POS> column of 2006 and 2007 has been moved to the C<FEAT>
column as a new feature called C<SubPOS>.
This driver is a translation layer above the C<cs::conll> driver.

=head1 SEE ALSO

L<Lingua::Interset>,
L<Lingua::Interset::Tagset>,
L<Lingua::Interset::Tagset::CS::Pdt>,
L<Lingua::Interset::Tagset::CS::Conll>,
L<Lingua::Interset::FeatureStructure>

=head1 AUTHOR

Dan Zeman <zeman@ufal.mff.cuni.cz>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2014 by Univerzita Karlova v Praze (Charles University in Prague).

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
