package DBIx::DBSchema::DBD;

use strict;
use vars qw($VERSION);

$VERSION = '0.04';

=head1 NAME

DBIx::DBSchema::DBD - DBIx::DBSchema Driver Writer's Guide and Base Class

=head1 SYNOPSIS

  perldoc DBIx::DBSchema::DBD

  package DBIx::DBSchema::DBD::FooBase
  use DBIx::DBSchema::DBD;
  @ISA = qw(DBIx::DBSchema::DBD);

=head1 DESCRIPTION

Drivers should be named DBIx::DBSchema::DBD::DatabaseName, where DatabaseName
is the same as the DBD:: driver for this database.  Drivers should implement the
following class methods:

=over 4

=item columns CLASS DBI_DBH TABLE

Given an active DBI database handle, return a listref of listrefs (see
L<perllol>), each containing six elements: column name, column type,
nullability, column length, column default, and a field reserved for
driver-specific use.

=item column CLASS DBI_DBH TABLE COLUMN

Same as B<columns> above, except return the listref for a single column.  You
can inherit from DBIx::DBSchema::DBD to provide this function.

=cut

sub column {
  my($proto, $dbh, $table, $column) = @_;
  #@a = grep { $_->[0] eq $column } @{ $proto->columns( $dbh, $table ) };
  #$a[0];
  @{ [
    grep { $_->[0] eq $column } @{ $proto->columns( $dbh, $table ) }
  ] }[0]; #force list context on grep, return scalar of first element
}

=item primary_key CLASS DBI_DBH TABLE

Given an active DBI database handle, return the primary key for the specified
table.

=item unique CLASS DBI_DBH TABLE

Deprecated method - see the B<indices> method for new drivers.

Given an active DBI database handle, return a hashref of unique indices.  The
keys of the hashref are index names, and the values are arrayrefs which point
a list of column names for each.  See L<perldsc/"HASHES OF LISTS"> and
L<DBIx::DBSchema::Index>.

=item index CLASS DBI_DBH TABLE

Deprecated method - see the B<indices> method for new drivers.

Given an active DBI database handle, return a hashref of (non-unique) indices.
The keys of the hashref are index names, and the values are arrayrefs which
point a list of column names for each.  See L<perldsc/"HASHES OF LISTS"> and
L<DBIx::DBSchema::Index>.

=item indices CLASS DBI_DBH TABLE

Given an active DBI database handle, return a hashref of all indices, both
unique and non-unique.  The keys of the hashref are index names, and the values
are again hashrefs with the following keys:

=over 8

=item name - Index name (redundant)

=item using - Optional index method

=item unique - Boolean indicating whether or not this is a unique index

=item columns - List reference of column names (or expressions)

=back

(See L<FS::DBIx::DBSchema::Index>)

New drivers are advised to implement this method, and existing drivers are
advised to (eventually) provide this method instead of B<index> and B<unique>.

For backwards-compatibility with current drivers, the base DBIx::DBSchema::DBD
class provides an B<indices> method which uses the old B<index> and B<unique>
methods to provide this data.

=cut

sub indices {
  #my($proto, $dbh, $table) = @_;
  my($proto, @param) = @_;

  my $unique_hr = $proto->unique( @param );
  my $index_hr  = $proto->index(  @param );

  scalar(
    {
  
      (
        map {
              $_ => { 'name'    => $_,
                      'unique'  => 1,
                      'columns' => $unique_hr->{$_},
                    },
            }
            keys %$unique_hr
      ),
  
      (
        map {
              $_ => { 'name'    => $_,
                      'unique'  => 0,
                      'columns' => $index_hr->{$_},
                    },
            }
            keys %$index_hr
      ),
  
    }
  );
}

=item default_db_catalog

Returns the default database catalog for the DBI table_info command.
Inheriting from DBIx::DBSchema::DBD will provide the default empty string.

=cut

sub default_db_catalog { ''; }

=item default_db_catalog

Returns the default database schema for the DBI table_info command.
Inheriting from DBIx::DBSchema::DBD will provide the default empty string.

=cut

sub default_db_schema { ''; }

=back

=head1 TYPE MAPPING

You can define a %typemap array for your driver to map "standard" data    
types to database-specific types.  For example, the MySQL TIMESTAMP field
has non-standard auto-updating semantics; the MySQL DATETIME type is 
what other databases and the ODBC standard call TIMESTAMP, so one of the   
entries in the MySQL %typemap is:

  'TIMESTAMP' => 'DATETIME',

Another example is the Pg %typemap which maps the standard types BLOB and
LONG VARBINARY to the Pg-specific BYTEA:

  'BLOB' => 'BYTEA',
  'LONG VARBINARY' => 'BYTEA',

Make sure you use all uppercase-keys.

=head1 AUTHOR

Ivan Kohler <ivan-dbix-dbschema@420.am>

=head1 COPYRIGHT

Copyright (c) 2000-2005 Ivan Kohler
All rights reserved.
This program is free software; you can redistribute it and/or modify it under
the same terms as Perl itself.

=head1 BUGS

=head1 SEE ALSO

L<DBIx::DBSchema>, L<DBIx::DBSchema::DBD::mysql>, L<DBIx::DBSchema::DBD::Pg>,
L<DBIx::DBSchema::Index>, L<DBI>, L<DBI::DBD>, L<perllol>,
L<perldsc/"HASHES OF LISTS">

=cut 

1;

