package Net::Domain::Parts;

use strict;
use warnings;

use Carp qw(croak);
use Data::Dumper;
use JSON;

use Exporter;
our @ISA = qw(Exporter);
our @EXPORT = qw(
    domain_parts
);

our @EXPORT_OK = qw(
    tld_list
    tld_struct
);

our %EXPORT_TAGS;
$EXPORT_TAGS{all} = [@EXPORT, @EXPORT_OK];

our $VERSION = '0.04';

my $json;
{
    local $/;
    $json = <DATA>;
}

my $tlds = decode_json($json);

sub domain_parts {
    my ($domain_name) = @_;

    if (! defined $domain_name) {
        croak(__PACKAGE__ . "::domain_parts() must be sent in a domain name.");
    }

    if ($domain_name !~ /.+\./) {
        croak(__PACKAGE__ . "::domain_parts() domain param invalid: $domain_name");
    }

    my ($sub_domain, $domain, $tld) = _sort_domain_parts($domain_name);

    return($sub_domain, $domain, $tld);
}
sub tld_list {
    my %tld_list;

    for my $key (keys %$tlds) {
        next if $key =~ /version/;
        $tld_list{$_} = 1 for keys %{ $tlds->{$key} };
    }

    return \%tld_list;
}
sub tld_struct {
    return $tlds;
}
sub version {
    return $tlds->{version};
}

sub _sort_domain_parts {
    my ($dns_entry) = @_;

    my $dot_count = () = $dns_entry =~ /\./g;

    my $tld_match;

    if ($dot_count > 2) {
        # Could possibly be a three-level TLD

        my $tld_entries = $tlds->{third_level_domain};

        for (sort { length($b) <=> length($a) } keys %{ $tld_entries }) {
            if ($dns_entry =~ /$_$/) {
                $tld_match = $_;
                last;
            }
        }
    }
    if (! $tld_match && $dot_count > 1) {
        # Could possibly match a second-level TLD (eg. co.uk)

        my $tld_entries = $tlds->{second_level_domain};

        for (sort { length($b) <=> length($a) } keys %{ $tld_entries }) {
            if ($dns_entry =~ /$_$/) {
                $tld_match = $_;
                last;
            }
        }
    }
    if (! $tld_match) {
        # If no match yet, check the domain against top-level TLDs

        my $tld_entries = $tlds->{top_level_domain};

        for (sort { length($b) <=> length($a) } keys %{ $tld_entries }) {
            if ($dns_entry =~ /$_$/) {
                $tld_match = $_;
                last;
            }
        }
    }

    return(undef, undef, undef) if ! $tld_match;

    my ($sub_domains, $domain);

    $dns_entry =~ s/\.$tld_match//;

    if ($dns_entry =~ /(.*)\.(.*)/) {
        $sub_domains = $1;
        $domain = "$2.$tld_match";
    }
    else {
        $domain = "$dns_entry.$tld_match";
    }

    return($sub_domains, $domain, $tld_match)
}

sub __placeholder {}

1;

=pod

=head1 NAME

Net::Domain::Parts - Extract sub-domain, domain and TLD parts of a domain name.

=for html
<a href="https://github.com/stevieb9/net-domain-parts/actions"><img src="https://github.com/stevieb9/net-domain-parts/workflows/CI/badge.svg"/></a>
<a href='https://coveralls.io/github/stevieb9/net-domain-parts?branch=main'><img src='https://coveralls.io/repos/stevieb9/net-domain-parts/badge.svg?branch=main&service=github' alt='Coverage Status' /></a>


=head1 SYNOPSIS

    use Net::Domain::Parts;

    my $domain_name = 'www.perlmonks.org';

    my ($subdomain, $domain, $tld) = domain_parts($domain_name);

    printf(
        "Domain %s, Subdomain %s, TLD: %s\n",
        $domain,
        $subdomain,
        $tld
    );

    # Domain: perlmonks.org, Subdomain: www, TLD: org

=head1 DESCRIPTION

This module takes a fully qualified domain name, and breaks it down into its
core components... the TLD, the domain name itself, and any subdomains.

=head1 FUNCTIONS

=head2 domain_parts($domain_name)

Exported by default. Breaks up an FQDN into it's core components.

I<Parameters>:

    $domain_name

I<Mandatory, String>: A fully qualified, valid domain name string.

I<Returns>: A list of C<subdomain> (undef if not present), C<domain> (with TLD
attached) and the C<tld>. If the domain isn't valid (ie. doesn't exist), three
C<undef>s will be returned.

=head2 version

Never exported. Call it as a fully qualified package function. Returns the
version string of the TLD data being used.

    my $version = Net::Domain::Parts::version();
    print "$version\n";

    # 2025-01-21_09-07-06_UTC

=head2 tld_struct

Not exported by default. Returns the internal store of TLD data.

I<Returns>: Hash reference.

    {
        version => 'version_string',
        third_level_domain => {
            'witd.gov.pl'       => 1,
            'fudai.iwate.jp'    => 1,
        },
        second_level_domain => {
            'co.uk'     => 1,
            'prato.it'  => 1,
        },
        top_level_domain => {
            'com'   => 1,
            'ca'    => 1,
            'org'   => 1,
        },
    }

=head2 tld_list

Not exported by default. Returns the entire list of TLDs, each TLD as a hash
key.

I<Returns>: Hash reference.

    {
        'org'                       => 1,
        'com'                       => 1,
        'co.uk'                     => 1,
        'org.im'                    => 1,
        'kashiwazaki.niigata.jp'    => 1,
    }

=head1 AUTHOR

Steve Bertrand, C<< <steveb at cpan.org> >>

=head1 LICENSE AND COPYRIGHT

Copyright 2025 Steve Bertrand.

This program is free software; you can redistribute it and/or modify it
under the terms of the the Artistic License (2.0). You may obtain a
copy of the full license at:

L<http://www.perlfoundation.org/artistic_license_2_0>

=cut

__DATA__
{
   "third_level_domain" : {
      "tachiarai.fukuoka.jp" : 1,
      "so.gov.pl" : 1,
      "yonaguni.okinawa.jp" : 1,
      "moroyama.saitama.jp" : 1,
      "rs.gov.br" : 1,
      "matsumae.hokkaido.jp" : 1,
      "sdn.gov.pl" : 1,
      "mitsue.nara.jp" : 1,
      "nantan.kyoto.jp" : 1,
      "hirokawa.fukuoka.jp" : 1,
      "ikeda.fukui.jp" : 1,
      "kasaoka.okayama.jp" : 1,
      "shimamaki.hokkaido.jp" : 1,
      "tsuga.tochigi.jp" : 1,
      "yoshinogari.saga.jp" : 1,
      "funabashi.chiba.jp" : 1,
      "nishi.fukuoka.jp" : 1,
      "yokoshibahikari.chiba.jp" : 1,
      "kiyama.saga.jp" : 1,
      "kuchinotsu.nagasaki.jp" : 1,
      "kokonoe.oita.jp" : 1,
      "namegata.ibaraki.jp" : 1,
      "masuda.shimane.jp" : 1,
      "choyo.kumamoto.jp" : 1,
      "nes.buskerud.no" : 1,
      "koori.fukushima.jp" : 1,
      "katsushika.tokyo.jp" : 1,
      "nisshin.aichi.jp" : 1,
      "mima.tokushima.jp" : 1,
      "seranishi.hiroshima.jp" : 1,
      "k12.wy.us" : 1,
      "cc.wa.us" : 1,
      "kashiba.nara.jp" : 1,
      "tokashiki.okinawa.jp" : 1,
      "cc.ut.us" : 1,
      "fujimino.saitama.jp" : 1,
      "oiso.kanagawa.jp" : 1,
      "masaki.ehime.jp" : 1,
      "lib.wi.us" : 1,
      "suzu.ishikawa.jp" : 1,
      "gosen.niigata.jp" : 1,
      "osaki.miyagi.jp" : 1,
      "esashi.hokkaido.jp" : 1,
      "naganohara.gunma.jp" : 1,
      "kamisunagawa.hokkaido.jp" : 1,
      "witd.gov.pl" : 1,
      "geisei.kochi.jp" : 1,
      "nikko.tochigi.jp" : 1,
      "tomari.hokkaido.jp" : 1,
      "se.gov.br" : 1,
      "gs.nt.no" : 1,
      "ug.gov.pl" : 1,
      "nakatane.kagoshima.jp" : 1,
      "matsubushi.saitama.jp" : 1,
      "kariwa.niigata.jp" : 1,
      "isshiki.aichi.jp" : 1,
      "gs.st.no" : 1,
      "sosa.chiba.jp" : 1,
      "k12.tn.us" : 1,
      "komono.mie.jp" : 1,
      "k12.pr.us" : 1,
      "shimamoto.osaka.jp" : 1,
      "konan.aichi.jp" : 1,
      "hiranai.aomori.jp" : 1,
      "bø.telemark.no" : 1,
      "uki.kumamoto.jp" : 1,
      "kanonji.kagawa.jp" : 1,
      "minobu.yamanashi.jp" : 1,
      "takikawa.hokkaido.jp" : 1,
      "miho.ibaraki.jp" : 1,
      "omachi.saga.jp" : 1,
      "joso.ibaraki.jp" : 1,
      "lib.vt.us" : 1,
      "akaiwa.okayama.jp" : 1,
      "kaminokawa.tochigi.jp" : 1,
      "tamatsukuri.ibaraki.jp" : 1,
      "sukumo.kochi.jp" : 1,
      "shingu.fukuoka.jp" : 1,
      "higashi.fukuoka.jp" : 1,
      "tomioka.gunma.jp" : 1,
      "mihama.wakayama.jp" : 1,
      "tsurugashima.saitama.jp" : 1,
      "chosei.chiba.jp" : 1,
      "higashikagura.hokkaido.jp" : 1,
      "gotemba.shizuoka.jp" : 1,
      "fujimi.saitama.jp" : 1,
      "k12.oh.us" : 1,
      "toyota.aichi.jp" : 1,
      "naie.hokkaido.jp" : 1,
      "hiraizumi.iwate.jp" : 1,
      "shibetsu.hokkaido.jp" : 1,
      "tsubata.ishikawa.jp" : 1,
      "kakegawa.shizuoka.jp" : 1,
      "ogawa.ibaraki.jp" : 1,
      "futtsu.chiba.jp" : 1,
      "lib.sc.us" : 1,
      "higashi.fukushima.jp" : 1,
      "yoshino.nara.jp" : 1,
      "minamitane.kagoshima.jp" : 1,
      "ap.gov.br" : 1,
      "kurobe.toyama.jp" : 1,
      "iwanuma.miyagi.jp" : 1,
      "ono.fukushima.jp" : 1,
      "fukudomi.saga.jp" : 1,
      "nakamichi.yamanashi.jp" : 1,
      "yasuoka.nagano.jp" : 1,
      "chikujo.fukuoka.jp" : 1,
      "iwakura.aichi.jp" : 1,
      "mihama.chiba.jp" : 1,
      "meguro.tokyo.jp" : 1,
      "yamaga.kumamoto.jp" : 1,
      "takatsuki.osaka.jp" : 1,
      "hikari.yamaguchi.jp" : 1,
      "kitadaito.okinawa.jp" : 1,
      "sakai.osaka.jp" : 1,
      "mishima.fukushima.jp" : 1,
      "torahime.shiga.jp" : 1,
      "komagane.nagano.jp" : 1,
      "koriyama.fukushima.jp" : 1,
      "fukui.fukui.jp" : 1,
      "hidaka.wakayama.jp" : 1,
      "ujiie.tochigi.jp" : 1,
      "otoineppu.hokkaido.jp" : 1,
      "tomi.nagano.jp" : 1,
      "lib.ar.us" : 1,
      "fuji.shizuoka.jp" : 1,
      "anpachi.gifu.jp" : 1,
      "kuji.iwate.jp" : 1,
      "sc.gov.br" : 1,
      "fuchu.tokyo.jp" : 1,
      "gamagori.aichi.jp" : 1,
      "ichinohe.iwate.jp" : 1,
      "akkeshi.hokkaido.jp" : 1,
      "shizuoka.shizuoka.jp" : 1,
      "kujukuri.chiba.jp" : 1,
      "atami.shizuoka.jp" : 1,
      "takagi.nagano.jp" : 1,
      "mus.mi.us" : 1,
      "hirosaki.aomori.jp" : 1,
      "sa.edu.au" : 1,
      "shimosuwa.nagano.jp" : 1,
      "erimo.hokkaido.jp" : 1,
      "minoh.osaka.jp" : 1,
      "rokunohe.aomori.jp" : 1,
      "higashisumiyoshi.osaka.jp" : 1,
      "zushi.kanagawa.jp" : 1,
      "cc.fl.us" : 1,
      "tomobe.ibaraki.jp" : 1,
      "kitami.hokkaido.jp" : 1,
      "murakami.niigata.jp" : 1,
      "misato.saitama.jp" : 1,
      "ono.hyogo.jp" : 1,
      "kimobetsu.hokkaido.jp" : 1,
      "cc.nm.us" : 1,
      "hinohara.tokyo.jp" : 1,
      "sennan.osaka.jp" : 1,
      "shizukuishi.iwate.jp" : 1,
      "iwata.shizuoka.jp" : 1,
      "city.kawasaki.jp" : 1,
      "tendo.yamagata.jp" : 1,
      "matsukawa.nagano.jp" : 1,
      "minamioguni.kumamoto.jp" : 1,
      "matsuda.kanagawa.jp" : 1,
      "k12.vi.us" : 1,
      "kameyama.mie.jp" : 1,
      "iwaizumi.iwate.jp" : 1,
      "tosu.saga.jp" : 1,
      "nagato.yamaguchi.jp" : 1,
      "oum.gov.pl" : 1,
      "kawara.fukuoka.jp" : 1,
      "maebashi.gunma.jp" : 1,
      "shingo.aomori.jp" : 1,
      "itako.ibaraki.jp" : 1,
      "ojiya.niigata.jp" : 1,
      "k12.ma.us" : 1,
      "kyuragi.saga.jp" : 1,
      "oji.nara.jp" : 1,
      "lib.dc.us" : 1,
      "miyake.nara.jp" : 1,
      "shari.hokkaido.jp" : 1,
      "shinjo.okayama.jp" : 1,
      "dst.mi.us" : 1,
      "obama.nagasaki.jp" : 1,
      "hakusan.ishikawa.jp" : 1,
      "city.sapporo.jp" : 1,
      "kwpsp.gov.pl" : 1,
      "k12.il.us" : 1,
      "tokai.aichi.jp" : 1,
      "kuju.oita.jp" : 1,
      "shirakawa.fukushima.jp" : 1,
      "aisai.aichi.jp" : 1,
      "minamiise.mie.jp" : 1,
      "mishima.shizuoka.jp" : 1,
      "higashikawa.hokkaido.jp" : 1,
      "higashiura.aichi.jp" : 1,
      "gs.of.no" : 1,
      "noda.iwate.jp" : 1,
      "shisui.chiba.jp" : 1,
      "shimodate.ibaraki.jp" : 1,
      "saitama.saitama.jp" : 1,
      "isahaya.nagasaki.jp" : 1,
      "lib.nc.us" : 1,
      "sowa.ibaraki.jp" : 1,
      "otaki.chiba.jp" : 1,
      "city.kitakyushu.jp" : 1,
      "washtenaw.mi.us" : 1,
      "takasu.hokkaido.jp" : 1,
      "ichinomiya.aichi.jp" : 1,
      "lib.nh.us" : 1,
      "ooshika.nagano.jp" : 1,
      "amagasaki.hyogo.jp" : 1,
      "tahara.aichi.jp" : 1,
      "shimoji.okinawa.jp" : 1,
      "yotsukaido.chiba.jp" : 1,
      "kamakura.kanagawa.jp" : 1,
      "tokushima.tokushima.jp" : 1,
      "kanazawa.ishikawa.jp" : 1,
      "nasushiobara.tochigi.jp" : 1,
      "lib.ia.us" : 1,
      "wsa.gov.pl" : 1,
      "ushiku.ibaraki.jp" : 1,
      "kadena.okinawa.jp" : 1,
      "nose.osaka.jp" : 1,
      "mup.gov.pl" : 1,
      "matsushima.miyagi.jp" : 1,
      "miyako.fukuoka.jp" : 1,
      "saga.saga.jp" : 1,
      "bibai.hokkaido.jp" : 1,
      "heroy.nordland.no" : 1,
      "anan.nagano.jp" : 1,
      "tateyama.chiba.jp" : 1,
      "nishiazai.shiga.jp" : 1,
      "toyotsu.fukuoka.jp" : 1,
      "cc.ma.us" : 1,
      "ina.saitama.jp" : 1,
      "tajiri.osaka.jp" : 1,
      "kanan.osaka.jp" : 1,
      "kasamatsu.gifu.jp" : 1,
      "funahashi.toyama.jp" : 1,
      "niki.hokkaido.jp" : 1,
      "rzgw.gov.pl" : 1,
      "ichikawa.hyogo.jp" : 1,
      "tokamachi.niigata.jp" : 1,
      "tagawa.fukuoka.jp" : 1,
      "kai.yamanashi.jp" : 1,
      "tonami.toyama.jp" : 1,
      "k12.nv.us" : 1,
      "ichikawa.chiba.jp" : 1,
      "imizu.toyama.jp" : 1,
      "tozawa.yamagata.jp" : 1,
      "manno.kagawa.jp" : 1,
      "akabira.hokkaido.jp" : 1,
      "lib.nm.us" : 1,
      "seika.kyoto.jp" : 1,
      "shitara.aichi.jp" : 1,
      "yashio.saitama.jp" : 1,
      "koshu.yamanashi.jp" : 1,
      "lib.ny.us" : 1,
      "ouda.nara.jp" : 1,
      "shinjo.yamagata.jp" : 1,
      "pr.gov.br" : 1,
      "shima.mie.jp" : 1,
      "yakumo.hokkaido.jp" : 1,
      "heguri.nara.jp" : 1,
      "taiwa.miyagi.jp" : 1,
      "biratori.hokkaido.jp" : 1,
      "ebetsu.hokkaido.jp" : 1,
      "mw.gov.pl" : 1,
      "isehara.kanagawa.jp" : 1,
      "takahama.aichi.jp" : 1,
      "kaga.ishikawa.jp" : 1,
      "k12.id.us" : 1,
      "cc.pr.us" : 1,
      "urausu.hokkaido.jp" : 1,
      "yawara.ibaraki.jp" : 1,
      "sango.nara.jp" : 1,
      "ise.mie.jp" : 1,
      "toda.saitama.jp" : 1,
      "suwa.nagano.jp" : 1,
      "yamatsuri.fukushima.jp" : 1,
      "matsuzaki.shizuoka.jp" : 1,
      "taishi.hyogo.jp" : 1,
      "cc.in.us" : 1,
      "oto.fukuoka.jp" : 1,
      "miyashiro.saitama.jp" : 1,
      "kutchan.hokkaido.jp" : 1,
      "uozu.toyama.jp" : 1,
      "ritto.shiga.jp" : 1,
      "nichinan.tottori.jp" : 1,
      "hamura.tokyo.jp" : 1,
      "toyono.osaka.jp" : 1,
      "ukiha.fukuoka.jp" : 1,
      "k12.az.us" : 1,
      "edogawa.tokyo.jp" : 1,
      "tosashimizu.kochi.jp" : 1,
      "shinshinotsu.hokkaido.jp" : 1,
      "nishiokoppe.hokkaido.jp" : 1,
      "cc.nc.us" : 1,
      "tarama.okinawa.jp" : 1,
      "kushiro.hokkaido.jp" : 1,
      "noboribetsu.hokkaido.jp" : 1,
      "koya.wakayama.jp" : 1,
      "cc.sc.us" : 1,
      "k12.ak.us" : 1,
      "tosa.kochi.jp" : 1,
      "qld.edu.au" : 1,
      "cc.id.us" : 1,
      "chijiwa.nagasaki.jp" : 1,
      "sumoto.kumamoto.jp" : 1,
      "towada.aomori.jp" : 1,
      "komatsu.ishikawa.jp" : 1,
      "ookuwa.nagano.jp" : 1,
      "sagae.yamagata.jp" : 1,
      "fukusaki.hyogo.jp" : 1,
      "minamidaito.okinawa.jp" : 1,
      "k12.ca.us" : 1,
      "itayanagi.aomori.jp" : 1,
      "nayoro.hokkaido.jp" : 1,
      "kamiizumi.saitama.jp" : 1,
      "kanuma.tochigi.jp" : 1,
      "uchinomi.kagawa.jp" : 1,
      "am.gov.br" : 1,
      "kawanishi.hyogo.jp" : 1,
      "aoki.nagano.jp" : 1,
      "shimane.shimane.jp" : 1,
      "nanmoku.gunma.jp" : 1,
      "yamagata.gifu.jp" : 1,
      "futaba.fukushima.jp" : 1,
      "city.kobe.jp" : 1,
      "wuoz.gov.pl" : 1,
      "higashishirakawa.gifu.jp" : 1,
      "izumiotsu.osaka.jp" : 1,
      "hitachi.ibaraki.jp" : 1,
      "yawatahama.ehime.jp" : 1,
      "kitahata.saga.jp" : 1,
      "mukawa.hokkaido.jp" : 1,
      "kawanehon.shizuoka.jp" : 1,
      "cc.vi.us" : 1,
      "ayagawa.kagawa.jp" : 1,
      "haibara.shizuoka.jp" : 1,
      "lib.md.us" : 1,
      "koga.fukuoka.jp" : 1,
      "iwaki.fukushima.jp" : 1,
      "uchihara.ibaraki.jp" : 1,
      "sande.more-og-romsdal.no" : 1,
      "lib.al.us" : 1,
      "kanzaki.saga.jp" : 1,
      "otari.nagano.jp" : 1,
      "ginoza.okinawa.jp" : 1,
      "oshu.iwate.jp" : 1,
      "toyone.aichi.jp" : 1,
      "tawaramoto.nara.jp" : 1,
      "minami.kyoto.jp" : 1,
      "mihara.kochi.jp" : 1,
      "bungotakada.oita.jp" : 1,
      "inabe.mie.jp" : 1,
      "motegi.tochigi.jp" : 1,
      "marugame.kagawa.jp" : 1,
      "amami.kagoshima.jp" : 1,
      "ureshino.mie.jp" : 1,
      "gs.sf.no" : 1,
      "ako.hyogo.jp" : 1,
      "kuromatsunai.hokkaido.jp" : 1,
      "sado.niigata.jp" : 1,
      "kotoura.tottori.jp" : 1,
      "oia.gov.pl" : 1,
      "kozagawa.wakayama.jp" : 1,
      "wiw.gov.pl" : 1,
      "upow.gov.pl" : 1,
      "chitose.hokkaido.jp" : 1,
      "lib.pa.us" : 1,
      "gamo.shiga.jp" : 1,
      "genkai.saga.jp" : 1,
      "ishikawa.okinawa.jp" : 1,
      "minamisanriku.miyagi.jp" : 1,
      "omachi.nagano.jp" : 1,
      "nemuro.hokkaido.jp" : 1,
      "ms.gov.br" : 1,
      "mashiki.kumamoto.jp" : 1,
      "toyonaka.osaka.jp" : 1,
      "nosegawa.nara.jp" : 1,
      "hirono.fukushima.jp" : 1,
      "shinonsen.hyogo.jp" : 1,
      "tokorozawa.saitama.jp" : 1,
      "meiwa.mie.jp" : 1,
      "tenei.fukushima.jp" : 1,
      "uda.nara.jp" : 1,
      "honjyo.akita.jp" : 1,
      "shiriuchi.hokkaido.jp" : 1,
      "mitane.akita.jp" : 1,
      "musashimurayama.tokyo.jp" : 1,
      "higashiagatsuma.gunma.jp" : 1,
      "yamato.kanagawa.jp" : 1,
      "hachioji.tokyo.jp" : 1,
      "yaita.tochigi.jp" : 1,
      "kudamatsu.yamaguchi.jp" : 1,
      "ugim.gov.pl" : 1,
      "kitaakita.akita.jp" : 1,
      "nago.okinawa.jp" : 1,
      "hidaka.hokkaido.jp" : 1,
      "gs.tm.no" : 1,
      "arakawa.tokyo.jp" : 1,
      "utashinai.hokkaido.jp" : 1,
      "takasago.hyogo.jp" : 1,
      "kamiamakusa.kumamoto.jp" : 1,
      "nishi.osaka.jp" : 1,
      "nobeoka.miyazaki.jp" : 1,
      "hamatama.saga.jp" : 1,
      "tenri.nara.jp" : 1,
      "maibara.shiga.jp" : 1,
      "moriyama.shiga.jp" : 1,
      "anjo.aichi.jp" : 1,
      "tenkawa.nara.jp" : 1,
      "cc.ct.us" : 1,
      "kitayama.wakayama.jp" : 1,
      "k12.me.us" : 1,
      "oyama.tochigi.jp" : 1,
      "kisarazu.chiba.jp" : 1,
      "satsumasendai.kagoshima.jp" : 1,
      "kin.okinawa.jp" : 1,
      "shimizu.shizuoka.jp" : 1,
      "kofu.yamanashi.jp" : 1,
      "yaizu.shizuoka.jp" : 1,
      "samukawa.kanagawa.jp" : 1,
      "fukushima.fukushima.jp" : 1,
      "um.gov.pl" : 1,
      "yokote.akita.jp" : 1,
      "ogasawara.tokyo.jp" : 1,
      "miharu.fukushima.jp" : 1,
      "azumino.nagano.jp" : 1,
      "gs.mr.no" : 1,
      "mamurogawa.yamagata.jp" : 1,
      "ueno.gunma.jp" : 1,
      "pa.gov.pl" : 1,
      "kashiwa.chiba.jp" : 1,
      "higashimurayama.tokyo.jp" : 1,
      "kishiwada.osaka.jp" : 1,
      "ginowan.okinawa.jp" : 1,
      "asahi.toyama.jp" : 1,
      "sa.gov.au" : 1,
      "abira.hokkaido.jp" : 1,
      "kiyosu.aichi.jp" : 1,
      "iwamizawa.hokkaido.jp" : 1,
      "urasoe.okinawa.jp" : 1,
      "honjo.saitama.jp" : 1,
      "nahari.kochi.jp" : 1,
      "sakae.chiba.jp" : 1,
      "cc.nj.us" : 1,
      "naka.hiroshima.jp" : 1,
      "cc.tn.us" : 1,
      "chikuhoku.nagano.jp" : 1,
      "kamifurano.hokkaido.jp" : 1,
      "yanaizu.fukushima.jp" : 1,
      "taira.toyama.jp" : 1,
      "nishiizu.shizuoka.jp" : 1,
      "yasugi.shimane.jp" : 1,
      "tateshina.nagano.jp" : 1,
      "tsukigata.hokkaido.jp" : 1,
      "amakusa.kumamoto.jp" : 1,
      "gifu.gifu.jp" : 1,
      "k12.mt.us" : 1,
      "katagami.akita.jp" : 1,
      "kamikitayama.nara.jp" : 1,
      "hakui.ishikawa.jp" : 1,
      "tama.tokyo.jp" : 1,
      "k12.tx.us" : 1,
      "susaki.kochi.jp" : 1,
      "asaka.saitama.jp" : 1,
      "lib.as.us" : 1,
      "hinode.tokyo.jp" : 1,
      "chuo.yamanashi.jp" : 1,
      "nanyo.yamagata.jp" : 1,
      "iwatsuki.saitama.jp" : 1,
      "nishihara.okinawa.jp" : 1,
      "kawanishi.nara.jp" : 1,
      "chikugo.fukuoka.jp" : 1,
      "omi.nagano.jp" : 1,
      "k12.ks.us" : 1,
      "koshigaya.saitama.jp" : 1,
      "fujisato.akita.jp" : 1,
      "dazaifu.fukuoka.jp" : 1,
      "shimoichi.nara.jp" : 1,
      "minato.tokyo.jp" : 1,
      "arita.saga.jp" : 1,
      "nanae.hokkaido.jp" : 1,
      "asahi.yamagata.jp" : 1,
      "kumenan.okayama.jp" : 1,
      "izunokuni.shizuoka.jp" : 1,
      "minamiechizen.fukui.jp" : 1,
      "inatsuki.fukuoka.jp" : 1,
      "shimogo.fukushima.jp" : 1,
      "nagaoka.niigata.jp" : 1,
      "kisosaki.mie.jp" : 1,
      "yamazoe.nara.jp" : 1,
      "inawashiro.fukushima.jp" : 1,
      "kyotamba.kyoto.jp" : 1,
      "iida.nagano.jp" : 1,
      "lib.ne.us" : 1,
      "moriguchi.osaka.jp" : 1,
      "rishirifuji.hokkaido.jp" : 1,
      "pippu.hokkaido.jp" : 1,
      "kawasaki.miyagi.jp" : 1,
      "shimokitayama.nara.jp" : 1,
      "yabu.hyogo.jp" : 1,
      "nakayama.yamagata.jp" : 1,
      "tanabe.kyoto.jp" : 1,
      "rr.gov.br" : 1,
      "seiyo.ehime.jp" : 1,
      "akashi.hyogo.jp" : 1,
      "beppu.oita.jp" : 1,
      "hazu.aichi.jp" : 1,
      "cc.al.us" : 1,
      "isumi.chiba.jp" : 1,
      "owariasahi.aichi.jp" : 1,
      "yatsushiro.kumamoto.jp" : 1,
      "lib.or.us" : 1,
      "akishima.tokyo.jp" : 1,
      "ebino.miyazaki.jp" : 1,
      "motobu.okinawa.jp" : 1,
      "toyo.kochi.jp" : 1,
      "ogawa.nagano.jp" : 1,
      "shirakawa.gifu.jp" : 1,
      "ide.kyoto.jp" : 1,
      "matsudo.chiba.jp" : 1,
      "cc.ks.us" : 1,
      "kmpsp.gov.pl" : 1,
      "omaezaki.shizuoka.jp" : 1,
      "mihara.hiroshima.jp" : 1,
      "shiraoi.hokkaido.jp" : 1,
      "oirase.aomori.jp" : 1,
      "sekigahara.gifu.jp" : 1,
      "cc.mo.us" : 1,
      "jinsekikogen.hiroshima.jp" : 1,
      "kiryu.gunma.jp" : 1,
      "yokosuka.kanagawa.jp" : 1,
      "yakage.okayama.jp" : 1,
      "tsukuba.ibaraki.jp" : 1,
      "yusuhara.kochi.jp" : 1,
      "higashimatsuyama.saitama.jp" : 1,
      "arai.shizuoka.jp" : 1,
      "rebun.hokkaido.jp" : 1,
      "soeda.fukuoka.jp" : 1,
      "tomisato.chiba.jp" : 1,
      "yonezawa.yamagata.jp" : 1,
      "kiwa.mie.jp" : 1,
      "cc.wy.us" : 1,
      "ina.nagano.jp" : 1,
      "nagiso.nagano.jp" : 1,
      "sagamihara.kanagawa.jp" : 1,
      "ishigaki.okinawa.jp" : 1,
      "miyakonojo.miyazaki.jp" : 1,
      "soni.nara.jp" : 1,
      "tamba.hyogo.jp" : 1,
      "ohda.shimane.jp" : 1,
      "tamayu.shimane.jp" : 1,
      "yamanobe.yamagata.jp" : 1,
      "kimino.wakayama.jp" : 1,
      "showa.fukushima.jp" : 1,
      "k12.fl.us" : 1,
      "miyazaki.miyazaki.jp" : 1,
      "muroran.hokkaido.jp" : 1,
      "tsukui.kanagawa.jp" : 1,
      "fukumitsu.toyama.jp" : 1,
      "chofu.tokyo.jp" : 1,
      "nabari.mie.jp" : 1,
      "kuki.saitama.jp" : 1,
      "kawachinagano.osaka.jp" : 1,
      "pup.gov.pl" : 1,
      "kisofukushima.nagano.jp" : 1,
      "niikappu.hokkaido.jp" : 1,
      "kamigori.hyogo.jp" : 1,
      "cc.vt.us" : 1,
      "kiho.mie.jp" : 1,
      "k12.wa.us" : 1,
      "tatsuno.nagano.jp" : 1,
      "nonoichi.ishikawa.jp" : 1,
      "fukuroi.shizuoka.jp" : 1,
      "kahoku.yamagata.jp" : 1,
      "shibata.miyagi.jp" : 1,
      "chikusei.ibaraki.jp" : 1,
      "ochi.kochi.jp" : 1,
      "zamami.okinawa.jp" : 1,
      "kita.osaka.jp" : 1,
      "cc.la.us" : 1,
      "wassamu.hokkaido.jp" : 1,
      "rn.gov.br" : 1,
      "kamikoani.akita.jp" : 1,
      "shikabe.hokkaido.jp" : 1,
      "aridagawa.wakayama.jp" : 1,
      "iide.yamagata.jp" : 1,
      "murata.miyagi.jp" : 1,
      "ginan.gifu.jp" : 1,
      "okoppe.hokkaido.jp" : 1,
      "zentsuji.kagawa.jp" : 1,
      "hokuryu.hokkaido.jp" : 1,
      "inami.wakayama.jp" : 1,
      "nagahama.shiga.jp" : 1,
      "namie.fukushima.jp" : 1,
      "nishinoomote.kagoshima.jp" : 1,
      "mihama.fukui.jp" : 1,
      "noda.chiba.jp" : 1,
      "uzs.gov.pl" : 1,
      "higashi.okinawa.jp" : 1,
      "k12.co.us" : 1,
      "tainai.niigata.jp" : 1,
      "nanbu.yamanashi.jp" : 1,
      "kanoya.kagoshima.jp" : 1,
      "nakai.kanagawa.jp" : 1,
      "kadoma.osaka.jp" : 1,
      "tadotsu.kagawa.jp" : 1,
      "saijo.ehime.jp" : 1,
      "wazuka.kyoto.jp" : 1,
      "kuroishi.aomori.jp" : 1,
      "ichiba.tokushima.jp" : 1,
      "ikawa.akita.jp" : 1,
      "choshi.chiba.jp" : 1,
      "eniwa.hokkaido.jp" : 1,
      "shirosato.ibaraki.jp" : 1,
      "kaneyama.fukushima.jp" : 1,
      "daigo.ibaraki.jp" : 1,
      "city.yokohama.jp" : 1,
      "soka.saitama.jp" : 1,
      "minamiminowa.nagano.jp" : 1,
      "toyota.yamaguchi.jp" : 1,
      "mitaka.tokyo.jp" : 1,
      "yukuhashi.fukuoka.jp" : 1,
      "kazo.saitama.jp" : 1,
      "nsw.edu.au" : 1,
      "shiranuka.hokkaido.jp" : 1,
      "myoko.niigata.jp" : 1,
      "saroma.hokkaido.jp" : 1,
      "oishida.yamagata.jp" : 1,
      "wanouchi.gifu.jp" : 1,
      "kani.gifu.jp" : 1,
      "yachiyo.chiba.jp" : 1,
      "shinto.gunma.jp" : 1,
      "kitamoto.saitama.jp" : 1,
      "gobo.wakayama.jp" : 1,
      "sa.gov.pl" : 1,
      "omiya.saitama.jp" : 1,
      "yabuki.fukushima.jp" : 1,
      "kakogawa.hyogo.jp" : 1,
      "fuchu.toyama.jp" : 1,
      "fujikawaguchiko.yamanashi.jp" : 1,
      "kamioka.akita.jp" : 1,
      "gyokuto.kumamoto.jp" : 1,
      "otaki.saitama.jp" : 1,
      "k12.nj.us" : 1,
      "kariya.aichi.jp" : 1,
      "kawamata.fukushima.jp" : 1,
      "oyabe.toyama.jp" : 1,
      "ibaraki.ibaraki.jp" : 1,
      "hachinohe.aomori.jp" : 1,
      "kira.aichi.jp" : 1,
      "tomakomai.hokkaido.jp" : 1,
      "oschr.gov.pl" : 1,
      "kashiwazaki.niigata.jp" : 1,
      "higashikagawa.kagawa.jp" : 1,
      "otsu.shiga.jp" : 1,
      "chuo.chiba.jp" : 1,
      "odate.akita.jp" : 1,
      "yoka.hyogo.jp" : 1,
      "toba.mie.jp" : 1,
      "ishikawa.fukushima.jp" : 1,
      "kusu.oita.jp" : 1,
      "kawai.nara.jp" : 1,
      "fujishiro.ibaraki.jp" : 1,
      "watarai.mie.jp" : 1,
      "tsu.mie.jp" : 1,
      "oarai.ibaraki.jp" : 1,
      "inagawa.hyogo.jp" : 1,
      "ishikari.hokkaido.jp" : 1,
      "yamanashi.yamanashi.jp" : 1,
      "toyooka.hyogo.jp" : 1,
      "higashiizu.shizuoka.jp" : 1,
      "lib.va.us" : 1,
      "shiojiri.nagano.jp" : 1,
      "uenohara.yamanashi.jp" : 1,
      "tabayama.yamanashi.jp" : 1,
      "hirata.fukushima.jp" : 1,
      "kainan.tokushima.jp" : 1,
      "moriya.ibaraki.jp" : 1,
      "oirm.gov.pl" : 1,
      "lib.mi.us" : 1,
      "gotsu.shimane.jp" : 1,
      "yamada.fukuoka.jp" : 1,
      "uppo.gov.pl" : 1,
      "tomigusuku.okinawa.jp" : 1,
      "soja.okayama.jp" : 1,
      "ohira.tochigi.jp" : 1,
      "unnan.shimane.jp" : 1,
      "chuo.tokyo.jp" : 1,
      "lib.ky.us" : 1,
      "nagawa.nagano.jp" : 1,
      "tamano.okayama.jp" : 1,
      "kakinoki.shimane.jp" : 1,
      "minamifurano.hokkaido.jp" : 1,
      "k12.gu.us" : 1,
      "shiroi.chiba.jp" : 1,
      "asuke.aichi.jp" : 1,
      "wsse.gov.pl" : 1,
      "yokkaichi.mie.jp" : 1,
      "obuse.nagano.jp" : 1,
      "pe.gov.br" : 1,
      "kitakata.fukushima.jp" : 1,
      "achi.nagano.jp" : 1,
      "hashimoto.wakayama.jp" : 1,
      "taketomi.okinawa.jp" : 1,
      "kamikawa.saitama.jp" : 1,
      "nasu.tochigi.jp" : 1,
      "miasa.nagano.jp" : 1,
      "tamakawa.fukushima.jp" : 1,
      "gs.va.no" : 1,
      "togane.chiba.jp" : 1,
      "yakumo.shimane.jp" : 1,
      "suginami.tokyo.jp" : 1,
      "tome.miyagi.jp" : 1,
      "mitoyo.kagawa.jp" : 1,
      "k12.in.us" : 1,
      "po.gov.pl" : 1,
      "keisen.fukuoka.jp" : 1,
      "shiroishi.saga.jp" : 1,
      "niimi.okayama.jp" : 1,
      "nakagyo.kyoto.jp" : 1,
      "katsuragi.wakayama.jp" : 1,
      "tsurugi.ishikawa.jp" : 1,
      "mikasa.hokkaido.jp" : 1,
      "yusui.kagoshima.jp" : 1,
      "yono.saitama.jp" : 1,
      "kurotaki.nara.jp" : 1,
      "gs.ah.no" : 1,
      "yanagawa.fukuoka.jp" : 1,
      "aibetsu.hokkaido.jp" : 1,
      "fujiyoshida.yamanashi.jp" : 1,
      "kaita.hiroshima.jp" : 1,
      "nt.edu.au" : 1,
      "oshima.tokyo.jp" : 1,
      "daiwa.hiroshima.jp" : 1,
      "kanegasaki.iwate.jp" : 1,
      "cc.ny.us" : 1,
      "annaka.gunma.jp" : 1,
      "anamizu.ishikawa.jp" : 1,
      "nakasatsunai.hokkaido.jp" : 1,
      "asahikawa.hokkaido.jp" : 1,
      "aguni.okinawa.jp" : 1,
      "kawakami.nagano.jp" : 1,
      "tokuyama.yamaguchi.jp" : 1,
      "tarui.gifu.jp" : 1,
      "umig.gov.pl" : 1,
      "kppsp.gov.pl" : 1,
      "morioka.iwate.jp" : 1,
      "kawazu.shizuoka.jp" : 1,
      "takaharu.miyazaki.jp" : 1,
      "uchinada.ishikawa.jp" : 1,
      "rikubetsu.hokkaido.jp" : 1,
      "asaminami.hiroshima.jp" : 1,
      "k12.la.us" : 1,
      "takamori.kumamoto.jp" : 1,
      "warabi.saitama.jp" : 1,
      "oketo.hokkaido.jp" : 1,
      "k12.ga.us" : 1,
      "toki.gifu.jp" : 1,
      "obu.aichi.jp" : 1,
      "tokigawa.saitama.jp" : 1,
      "inuyama.aichi.jp" : 1,
      "honbetsu.hokkaido.jp" : 1,
      "shinyoshitomi.fukuoka.jp" : 1,
      "ami.ibaraki.jp" : 1,
      "nakanoto.ishikawa.jp" : 1,
      "ikeda.gifu.jp" : 1,
      "rifu.miyagi.jp" : 1,
      "iwanai.hokkaido.jp" : 1,
      "toyama.toyama.jp" : 1,
      "seto.aichi.jp" : 1,
      "higashiyama.kyoto.jp" : 1,
      "tsuruga.fukui.jp" : 1,
      "kahoku.ishikawa.jp" : 1,
      "yoita.niigata.jp" : 1,
      "seirou.niigata.jp" : 1,
      "iizuka.fukuoka.jp" : 1,
      "aogaki.hyogo.jp" : 1,
      "nes.akershus.no" : 1,
      "kashiwara.osaka.jp" : 1,
      "kotohira.kagawa.jp" : 1,
      "arao.kumamoto.jp" : 1,
      "ozora.hokkaido.jp" : 1,
      "es.gov.br" : 1,
      "yoro.gifu.jp" : 1,
      "saikai.nagasaki.jp" : 1,
      "doshi.yamanashi.jp" : 1,
      "urawa.saitama.jp" : 1,
      "seiro.niigata.jp" : 1,
      "awaji.hyogo.jp" : 1,
      "sue.fukuoka.jp" : 1,
      "uw.gov.pl" : 1,
      "shimonoseki.yamaguchi.jp" : 1,
      "shinshiro.aichi.jp" : 1,
      "wake.okayama.jp" : 1,
      "ouchi.saga.jp" : 1,
      "matsubara.osaka.jp" : 1,
      "k12.mo.us" : 1,
      "rj.gov.br" : 1,
      "izumizaki.fukushima.jp" : 1,
      "komatsushima.tokushima.jp" : 1,
      "chikuma.nagano.jp" : 1,
      "lib.mn.us" : 1,
      "mimata.miyazaki.jp" : 1,
      "yorii.saitama.jp" : 1,
      "shintoku.hokkaido.jp" : 1,
      "ube.yamaguchi.jp" : 1,
      "takayama.gifu.jp" : 1,
      "gose.nara.jp" : 1,
      "saiki.oita.jp" : 1,
      "takahagi.ibaraki.jp" : 1,
      "higashimatsushima.miyagi.jp" : 1,
      "otsuchi.iwate.jp" : 1,
      "hongo.hiroshima.jp" : 1,
      "itoman.okinawa.jp" : 1,
      "tsubetsu.hokkaido.jp" : 1,
      "tadaoka.osaka.jp" : 1,
      "onojo.fukuoka.jp" : 1,
      "kyotango.kyoto.jp" : 1,
      "yamato.fukushima.jp" : 1,
      "tanabe.wakayama.jp" : 1,
      "yamanouchi.nagano.jp" : 1,
      "mt.gov.br" : 1,
      "shinjo.nara.jp" : 1,
      "aomori.aomori.jp" : 1,
      "vic.gov.au" : 1,
      "furubira.hokkaido.jp" : 1,
      "sannan.hyogo.jp" : 1,
      "suifu.ibaraki.jp" : 1,
      "kuzumaki.iwate.jp" : 1,
      "mizumaki.fukuoka.jp" : 1,
      "kita.kyoto.jp" : 1,
      "yoshikawa.saitama.jp" : 1,
      "iiyama.nagano.jp" : 1,
      "chichibu.saitama.jp" : 1,
      "numazu.shizuoka.jp" : 1,
      "taki.mie.jp" : 1,
      "aya.miyazaki.jp" : 1,
      "koshimizu.hokkaido.jp" : 1,
      "naruto.tokushima.jp" : 1,
      "konan.shiga.jp" : 1,
      "nakagawa.nagano.jp" : 1,
      "matsusaka.mie.jp" : 1,
      "våler.hedmark.no" : 1,
      "hashima.gifu.jp" : 1,
      "tas.edu.au" : 1,
      "kochi.kochi.jp" : 1,
      "ayase.kanagawa.jp" : 1,
      "nishinoshima.shimane.jp" : 1,
      "psp.gov.pl" : 1,
      "naka.ibaraki.jp" : 1,
      "ashiya.fukuoka.jp" : 1,
      "gojome.akita.jp" : 1,
      "unzen.nagasaki.jp" : 1,
      "shimabara.nagasaki.jp" : 1,
      "izumo.shimane.jp" : 1,
      "sakai.fukui.jp" : 1,
      "shirahama.wakayama.jp" : 1,
      "chiyoda.gunma.jp" : 1,
      "sakaiminato.tottori.jp" : 1,
      "daito.osaka.jp" : 1,
      "al.gov.br" : 1,
      "kitanakagusuku.okinawa.jp" : 1,
      "takamatsu.kagawa.jp" : 1,
      "iwama.ibaraki.jp" : 1,
      "nishiaizu.fukushima.jp" : 1,
      "biei.hokkaido.jp" : 1,
      "mikawa.yamagata.jp" : 1,
      "taka.hyogo.jp" : 1,
      "kasuga.fukuoka.jp" : 1,
      "ama.aichi.jp" : 1,
      "tsuchiura.ibaraki.jp" : 1,
      "okuizumo.shimane.jp" : 1,
      "hyuga.miyazaki.jp" : 1,
      "fujikawa.shizuoka.jp" : 1,
      "obanazawa.yamagata.jp" : 1,
      "oguchi.aichi.jp" : 1,
      "suita.osaka.jp" : 1,
      "watari.miyagi.jp" : 1,
      "minamiuonuma.niigata.jp" : 1,
      "sumoto.hyogo.jp" : 1,
      "nagatoro.saitama.jp" : 1,
      "cc.dc.us" : 1,
      "nakatombetsu.hokkaido.jp" : 1,
      "shiiba.miyazaki.jp" : 1,
      "onna.okinawa.jp" : 1,
      "tsushima.nagasaki.jp" : 1,
      "cc.gu.us" : 1,
      "kamitsue.oita.jp" : 1,
      "ainan.ehime.jp" : 1,
      "furano.hokkaido.jp" : 1,
      "goshiki.hyogo.jp" : 1,
      "otsuki.yamanashi.jp" : 1,
      "kasuya.fukuoka.jp" : 1,
      "ikeda.osaka.jp" : 1,
      "cc.ga.us" : 1,
      "oharu.aichi.jp" : 1,
      "hanamaki.iwate.jp" : 1,
      "bando.ibaraki.jp" : 1,
      "eaton.mi.us" : 1,
      "okaya.nagano.jp" : 1,
      "gushikami.okinawa.jp" : 1,
      "k12.ct.us" : 1,
      "natori.miyagi.jp" : 1,
      "winb.gov.pl" : 1,
      "k12.ut.us" : 1,
      "lib.ms.us" : 1,
      "matsumoto.nagano.jp" : 1,
      "aso.kumamoto.jp" : 1,
      "k12.ok.us" : 1,
      "odawara.kanagawa.jp" : 1,
      "yazu.tottori.jp" : 1,
      "akagi.shimane.jp" : 1,
      "agematsu.nagano.jp" : 1,
      "bizen.okayama.jp" : 1,
      "tottori.tottori.jp" : 1,
      "iyo.ehime.jp" : 1,
      "hiraya.nagano.jp" : 1,
      "izumi.osaka.jp" : 1,
      "hachijo.tokyo.jp" : 1,
      "kawagoe.mie.jp" : 1,
      "tohnosho.chiba.jp" : 1,
      "abiko.chiba.jp" : 1,
      "asahi.chiba.jp" : 1,
      "sakawa.kochi.jp" : 1,
      "nogata.fukuoka.jp" : 1,
      "nishihara.kumamoto.jp" : 1,
      "hanawa.fukushima.jp" : 1,
      "kami.kochi.jp" : 1,
      "shimonita.gunma.jp" : 1,
      "nomi.ishikawa.jp" : 1,
      "kagamiishi.fukushima.jp" : 1,
      "kikonai.hokkaido.jp" : 1,
      "gs.nl.no" : 1,
      "okinawa.okinawa.jp" : 1,
      "sakura.tochigi.jp" : 1,
      "k12.ar.us" : 1,
      "ichinoseki.iwate.jp" : 1,
      "ozu.kumamoto.jp" : 1,
      "pinb.gov.pl" : 1,
      "rikuzentakata.iwate.jp" : 1,
      "obama.fukui.jp" : 1,
      "hasuda.saitama.jp" : 1,
      "tamamura.gunma.jp" : 1,
      "misaki.osaka.jp" : 1,
      "ito.shizuoka.jp" : 1,
      "sukagawa.fukushima.jp" : 1,
      "ikeda.nagano.jp" : 1,
      "aizubange.fukushima.jp" : 1,
      "otake.hiroshima.jp" : 1,
      "hichiso.gifu.jp" : 1,
      "shijonawate.osaka.jp" : 1,
      "ninomiya.kanagawa.jp" : 1,
      "echizen.fukui.jp" : 1,
      "takaoka.toyama.jp" : 1,
      "omitama.ibaraki.jp" : 1,
      "takatsuki.shiga.jp" : 1,
      "nara.nara.jp" : 1,
      "sabae.fukui.jp" : 1,
      "yokaichiba.chiba.jp" : 1,
      "shimokawa.hokkaido.jp" : 1,
      "mochizuki.nagano.jp" : 1,
      "ogaki.gifu.jp" : 1,
      "meiwa.gunma.jp" : 1,
      "minami.fukuoka.jp" : 1,
      "sakuho.nagano.jp" : 1,
      "cog.mi.us" : 1,
      "karatsu.saga.jp" : 1,
      "utsunomiya.tochigi.jp" : 1,
      "tsuwano.shimane.jp" : 1,
      "yamagata.nagano.jp" : 1,
      "oyamazaki.kyoto.jp" : 1,
      "gonohe.aomori.jp" : 1,
      "kawaguchi.saitama.jp" : 1,
      "chtr.k12.ma.us" : 1,
      "samegawa.fukushima.jp" : 1,
      "nagasaki.nagasaki.jp" : 1,
      "sakae.nagano.jp" : 1,
      "onomichi.hiroshima.jp" : 1,
      "ueda.nagano.jp" : 1,
      "nakano.nagano.jp" : 1,
      "muroto.kochi.jp" : 1,
      "lib.vi.us" : 1,
      "gs.hm.no" : 1,
      "k12.dc.us" : 1,
      "utazu.kagawa.jp" : 1,
      "os.hedmark.no" : 1,
      "toyosato.shiga.jp" : 1,
      "shinagawa.tokyo.jp" : 1,
      "takahata.yamagata.jp" : 1,
      "gs.fm.no" : 1,
      "lib.ma.us" : 1,
      "asahi.ibaraki.jp" : 1,
      "morotsuka.miyazaki.jp" : 1,
      "sarufutsu.hokkaido.jp" : 1,
      "usui.fukuoka.jp" : 1,
      "cc.va.us" : 1,
      "setagaya.tokyo.jp" : 1,
      "fujioka.gunma.jp" : 1,
      "kamoenai.hokkaido.jp" : 1,
      "lib.il.us" : 1,
      "kizu.kyoto.jp" : 1,
      "cc.oh.us" : 1,
      "yamamoto.miyagi.jp" : 1,
      "joetsu.niigata.jp" : 1,
      "toyako.hokkaido.jp" : 1,
      "sanjo.niigata.jp" : 1,
      "ic.gov.pl" : 1,
      "kakuda.miyagi.jp" : 1,
      "esan.hokkaido.jp" : 1,
      "shiwa.iwate.jp" : 1,
      "uchiko.ehime.jp" : 1,
      "tara.saga.jp" : 1,
      "shirako.chiba.jp" : 1,
      "miyako.iwate.jp" : 1,
      "wa.edu.au" : 1,
      "kamikawa.hokkaido.jp" : 1,
      "k12.nc.us" : 1,
      "kurate.fukuoka.jp" : 1,
      "showa.gunma.jp" : 1,
      "hanyu.saitama.jp" : 1,
      "sera.hiroshima.jp" : 1,
      "k12.nh.us" : 1,
      "nishiawakura.okayama.jp" : 1,
      "fukuyama.hiroshima.jp" : 1,
      "kuwana.mie.jp" : 1,
      "fuefuki.yamanashi.jp" : 1,
      "kamisato.saitama.jp" : 1,
      "minato.osaka.jp" : 1,
      "date.fukushima.jp" : 1,
      "kamisu.ibaraki.jp" : 1,
      "kanmaki.nara.jp" : 1,
      "fujinomiya.shizuoka.jp" : 1,
      "k12.ia.us" : 1,
      "itano.tokushima.jp" : 1,
      "uryu.hokkaido.jp" : 1,
      "kaminoyama.yamagata.jp" : 1,
      "ac.gov.br" : 1,
      "takasaki.gunma.jp" : 1,
      "yugawa.fukushima.jp" : 1,
      "kaizuka.osaka.jp" : 1,
      "moseushi.hokkaido.jp" : 1,
      "nishikawa.yamagata.jp" : 1,
      "karuizawa.nagano.jp" : 1,
      "nakagawa.hokkaido.jp" : 1,
      "tono.iwate.jp" : 1,
      "semine.miyagi.jp" : 1,
      "himeshima.oita.jp" : 1,
      "oi.kanagawa.jp" : 1,
      "kijo.miyazaki.jp" : 1,
      "kaneyama.yamagata.jp" : 1,
      "hasami.nagasaki.jp" : 1,
      "kusatsu.gunma.jp" : 1,
      "shingu.wakayama.jp" : 1,
      "izumi.kagoshima.jp" : 1,
      "ibigawa.gifu.jp" : 1,
      "qld.gov.au" : 1,
      "tabuse.yamaguchi.jp" : 1,
      "takahama.fukui.jp" : 1,
      "us.gov.pl" : 1,
      "tsumagoi.gunma.jp" : 1,
      "asahi.mie.jp" : 1,
      "toyokawa.aichi.jp" : 1,
      "kunigami.okinawa.jp" : 1,
      "gs.vf.no" : 1,
      "karumai.iwate.jp" : 1,
      "makurazaki.kagoshima.jp" : 1,
      "saito.miyazaki.jp" : 1,
      "sugito.saitama.jp" : 1,
      "kasugai.aichi.jp" : 1,
      "hamada.shimane.jp" : 1,
      "sth.ac.at" : 1,
      "misasa.tottori.jp" : 1,
      "karasuyama.tochigi.jp" : 1,
      "osakasayama.osaka.jp" : 1,
      "sp.gov.br" : 1,
      "shiso.hyogo.jp" : 1,
      "shonai.yamagata.jp" : 1,
      "kihoku.ehime.jp" : 1,
      "cc.ok.us" : 1,
      "våler.østfold.no" : 1,
      "yuasa.wakayama.jp" : 1,
      "oguni.kumamoto.jp" : 1,
      "ikoma.nara.jp" : 1,
      "nishinomiya.hyogo.jp" : 1,
      "higashiizumo.shimane.jp" : 1,
      "toyohashi.aichi.jp" : 1,
      "okayama.okayama.jp" : 1,
      "nakagawa.tokushima.jp" : 1,
      "kasama.ibaraki.jp" : 1,
      "kitaura.miyazaki.jp" : 1,
      "yamanakako.yamanashi.jp" : 1,
      "kakamigahara.gifu.jp" : 1,
      "konsulat.gov.pl" : 1,
      "gs.oslo.no" : 1,
      "mutsu.aomori.jp" : 1,
      "himeji.hyogo.jp" : 1,
      "misawa.aomori.jp" : 1,
      "lib.wy.us" : 1,
      "buzen.fukuoka.jp" : 1,
      "higashiomi.shiga.jp" : 1,
      "izumozaki.niigata.jp" : 1,
      "musashino.tokyo.jp" : 1,
      "noheji.aomori.jp" : 1,
      "shingu.hyogo.jp" : 1,
      "kagoshima.kagoshima.jp" : 1,
      "neyagawa.osaka.jp" : 1,
      "k12.wi.us" : 1,
      "isen.kagoshima.jp" : 1,
      "takashima.shiga.jp" : 1,
      "tochigi.tochigi.jp" : 1,
      "kiyose.tokyo.jp" : 1,
      "uonuma.niigata.jp" : 1,
      "fukaya.saitama.jp" : 1,
      "hatsukaichi.hiroshima.jp" : 1,
      "arida.wakayama.jp" : 1,
      "cc.ak.us" : 1,
      "tanagura.fukushima.jp" : 1,
      "lib.tn.us" : 1,
      "omihachiman.shiga.jp" : 1,
      "motosu.gifu.jp" : 1,
      "kouyama.kagoshima.jp" : 1,
      "gujo.gifu.jp" : 1,
      "marumori.miyagi.jp" : 1,
      "morimachi.shizuoka.jp" : 1,
      "asago.hyogo.jp" : 1,
      "oga.akita.jp" : 1,
      "kashihara.nara.jp" : 1,
      "hakodate.hokkaido.jp" : 1,
      "lib.pr.us" : 1,
      "is.gov.pl" : 1,
      "asakuchi.okayama.jp" : 1,
      "futsu.nagasaki.jp" : 1,
      "kikuchi.kumamoto.jp" : 1,
      "hiji.oita.jp" : 1,
      "kitashiobara.fukushima.jp" : 1,
      "yasu.shiga.jp" : 1,
      "sakata.yamagata.jp" : 1,
      "okinoshima.shimane.jp" : 1,
      "higashiosaka.osaka.jp" : 1,
      "naoshima.kagawa.jp" : 1,
      "k12.vt.us" : 1,
      "miyoshi.tokushima.jp" : 1,
      "nishimera.miyazaki.jp" : 1,
      "mihama.aichi.jp" : 1,
      "kainan.wakayama.jp" : 1,
      "machida.tokyo.jp" : 1,
      "lib.oh.us" : 1,
      "ba.gov.br" : 1,
      "uwajima.ehime.jp" : 1,
      "hiratsuka.kanagawa.jp" : 1,
      "aizumisato.fukushima.jp" : 1,
      "fukuchi.fukuoka.jp" : 1,
      "hino.tokyo.jp" : 1,
      "city.sendai.jp" : 1,
      "taiji.wakayama.jp" : 1,
      "iwakuni.yamaguchi.jp" : 1,
      "mitou.yamaguchi.jp" : 1,
      "tobishima.aichi.jp" : 1,
      "kamitonda.wakayama.jp" : 1,
      "fujieda.shizuoka.jp" : 1,
      "unazuki.toyama.jp" : 1,
      "kyotanabe.kyoto.jp" : 1,
      "shunan.yamaguchi.jp" : 1,
      "ina.ibaraki.jp" : 1,
      "shintomi.miyazaki.jp" : 1,
      "kiyosato.hokkaido.jp" : 1,
      "aki.kochi.jp" : 1,
      "kozaki.chiba.jp" : 1,
      "oumu.hokkaido.jp" : 1,
      "k12.sc.us" : 1,
      "minamimaki.nagano.jp" : 1,
      "kure.hiroshima.jp" : 1,
      "nankoku.kochi.jp" : 1,
      "cc.tx.us" : 1,
      "zama.kanagawa.jp" : 1,
      "takahashi.okayama.jp" : 1,
      "ma.gov.br" : 1,
      "shonai.fukuoka.jp" : 1,
      "otsuki.kochi.jp" : 1,
      "shiraoka.saitama.jp" : 1,
      "ome.tokyo.jp" : 1,
      "abashiri.hokkaido.jp" : 1,
      "habikino.osaka.jp" : 1,
      "okawa.fukuoka.jp" : 1,
      "sakuragawa.ibaraki.jp" : 1,
      "kawai.iwate.jp" : 1,
      "setouchi.okayama.jp" : 1,
      "ujitawara.kyoto.jp" : 1,
      "tajimi.gifu.jp" : 1,
      "matsuno.ehime.jp" : 1,
      "eiheiji.fukui.jp" : 1,
      "psse.gov.pl" : 1,
      "kawakita.ishikawa.jp" : 1,
      "kumatori.osaka.jp" : 1,
      "mihama.mie.jp" : 1,
      "tatsuno.hyogo.jp" : 1,
      "yuu.yamaguchi.jp" : 1,
      "tanohata.iwate.jp" : 1,
      "assabu.hokkaido.jp" : 1,
      "nishiwaki.hyogo.jp" : 1,
      "cc.ri.us" : 1,
      "abu.yamaguchi.jp" : 1,
      "minami-alps.yamanashi.jp" : 1,
      "fujisawa.iwate.jp" : 1,
      "takinoue.hokkaido.jp" : 1,
      "haboro.hokkaido.jp" : 1,
      "itakura.gunma.jp" : 1,
      "higashitsuno.kochi.jp" : 1,
      "wkz.gov.pl" : 1,
      "sande.vestfold.no" : 1,
      "nishigo.fukushima.jp" : 1,
      "iheya.okinawa.jp" : 1,
      "yoshida.shizuoka.jp" : 1,
      "handa.aichi.jp" : 1,
      "cc.wi.us" : 1,
      "hokuto.yamanashi.jp" : 1,
      "ohkura.yamagata.jp" : 1,
      "yamagata.yamagata.jp" : 1,
      "seki.gifu.jp" : 1,
      "wakuya.miyagi.jp" : 1,
      "ann-arbor.mi.us" : 1,
      "shinichi.hiroshima.jp" : 1,
      "hiroo.hokkaido.jp" : 1,
      "narashino.chiba.jp" : 1,
      "k12.pa.us" : 1,
      "cc.nh.us" : 1,
      "cc.az.us" : 1,
      "nanbu.tottori.jp" : 1,
      "kumakogen.ehime.jp" : 1,
      "ikeda.hokkaido.jp" : 1,
      "wa.gov.au" : 1,
      "numata.gunma.jp" : 1,
      "ayabe.kyoto.jp" : 1,
      "yoshioka.gunma.jp" : 1,
      "miyoshi.aichi.jp" : 1,
      "higashikurume.tokyo.jp" : 1,
      "shibuya.tokyo.jp" : 1,
      "yuza.yamagata.jp" : 1,
      "saka.hiroshima.jp" : 1,
      "ranzan.saitama.jp" : 1,
      "yamatotakada.nara.jp" : 1,
      "pvt.k12.ma.us" : 1,
      "ogawara.miyagi.jp" : 1,
      "misato.wakayama.jp" : 1,
      "ryugasaki.ibaraki.jp" : 1,
      "yame.fukuoka.jp" : 1,
      "kanie.aichi.jp" : 1,
      "kumiyama.kyoto.jp" : 1,
      "higashichichibu.saitama.jp" : 1,
      "tohma.hokkaido.jp" : 1,
      "minamata.kumamoto.jp" : 1,
      "iitate.fukushima.jp" : 1,
      "nogi.tochigi.jp" : 1,
      "otaki.nagano.jp" : 1,
      "hirakata.osaka.jp" : 1,
      "ohtawara.tochigi.jp" : 1,
      "hayashima.okayama.jp" : 1,
      "kinko.kagoshima.jp" : 1,
      "kurashiki.okayama.jp" : 1,
      "griw.gov.pl" : 1,
      "satosho.okayama.jp" : 1,
      "yonabaru.okinawa.jp" : 1,
      "katori.chiba.jp" : 1,
      "ine.kyoto.jp" : 1,
      "sayo.hyogo.jp" : 1,
      "tsuruta.aomori.jp" : 1,
      "nagaokakyo.kyoto.jp" : 1,
      "udono.mie.jp" : 1,
      "yatomi.aichi.jp" : 1,
      "yamatokoriyama.nara.jp" : 1,
      "obihiro.hokkaido.jp" : 1,
      "takino.hyogo.jp" : 1,
      "tamaki.mie.jp" : 1,
      "tsushima.aichi.jp" : 1,
      "lib.me.us" : 1,
      "koganei.tokyo.jp" : 1,
      "nagara.chiba.jp" : 1,
      "kasumigaura.ibaraki.jp" : 1,
      "narusawa.yamanashi.jp" : 1,
      "sasebo.nagasaki.jp" : 1,
      "takarazuka.hyogo.jp" : 1,
      "miyoshi.hiroshima.jp" : 1,
      "ando.nara.jp" : 1,
      "agano.niigata.jp" : 1,
      "adachi.tokyo.jp" : 1,
      "kamimine.saga.jp" : 1,
      "gs.ol.no" : 1,
      "kushima.miyazaki.jp" : 1,
      "ichinomiya.chiba.jp" : 1,
      "fukagawa.hokkaido.jp" : 1,
      "yugawara.kanagawa.jp" : 1,
      "cc.co.us" : 1,
      "shishikui.tokushima.jp" : 1,
      "lib.ri.us" : 1,
      "aizumi.tokushima.jp" : 1,
      "fujikawa.yamanashi.jp" : 1,
      "noto.ishikawa.jp" : 1,
      "takata.fukuoka.jp" : 1,
      "cc.de.us" : 1,
      "tsukiyono.gunma.jp" : 1,
      "rishiri.hokkaido.jp" : 1,
      "fujiidera.osaka.jp" : 1,
      "sanuki.kagawa.jp" : 1,
      "yatsuka.shimane.jp" : 1,
      "sannohe.aomori.jp" : 1,
      "inazawa.aichi.jp" : 1,
      "ena.gifu.jp" : 1,
      "iijima.nagano.jp" : 1,
      "akiruno.tokyo.jp" : 1,
      "mifune.kumamoto.jp" : 1,
      "teshikaga.hokkaido.jp" : 1,
      "takaishi.osaka.jp" : 1,
      "mugi.tokushima.jp" : 1,
      "kushimoto.wakayama.jp" : 1,
      "shikokuchuo.ehime.jp" : 1,
      "takanezawa.tochigi.jp" : 1,
      "act.edu.au" : 1,
      "tachikawa.tokyo.jp" : 1,
      "otaru.hokkaido.jp" : 1,
      "kunohe.iwate.jp" : 1,
      "nanjo.okinawa.jp" : 1,
      "kosuge.yamanashi.jp" : 1,
      "owani.aomori.jp" : 1,
      "gs.rl.no" : 1,
      "chihayaakasaka.osaka.jp" : 1,
      "tsuiki.fukuoka.jp" : 1,
      "isesaki.gunma.jp" : 1,
      "kosaka.akita.jp" : 1,
      "minakami.gunma.jp" : 1,
      "nagano.nagano.jp" : 1,
      "bunkyo.tokyo.jp" : 1,
      "koto.shiga.jp" : 1,
      "ogata.akita.jp" : 1,
      "nishio.aichi.jp" : 1,
      "tateyama.toyama.jp" : 1,
      "kameoka.kyoto.jp" : 1,
      "tsunan.niigata.jp" : 1,
      "maniwa.okayama.jp" : 1,
      "date.hokkaido.jp" : 1,
      "toho.fukuoka.jp" : 1,
      "ino.kochi.jp" : 1,
      "pb.gov.br" : 1,
      "hitachinaka.ibaraki.jp" : 1,
      "hokuto.hokkaido.jp" : 1,
      "okegawa.saitama.jp" : 1,
      "bo.nordland.no" : 1,
      "nerima.tokyo.jp" : 1,
      "takanabe.miyazaki.jp" : 1,
      "isa.kagoshima.jp" : 1,
      "k12.nm.us" : 1,
      "shinanomachi.nagano.jp" : 1,
      "kitakami.iwate.jp" : 1,
      "lib.nv.us" : 1,
      "toshima.tokyo.jp" : 1,
      "otama.fukushima.jp" : 1,
      "sayama.saitama.jp" : 1,
      "wzmiuw.gov.pl" : 1,
      "yashiro.hyogo.jp" : 1,
      "mibu.tochigi.jp" : 1,
      "k12.ny.us" : 1,
      "shimizu.hokkaido.jp" : 1,
      "katsuura.chiba.jp" : 1,
      "kouhoku.saga.jp" : 1,
      "notogawa.shiga.jp" : 1,
      "ltd.co.im" : 1,
      "chigasaki.kanagawa.jp" : 1,
      "funagata.yamagata.jp" : 1,
      "toyotomi.hokkaido.jp" : 1,
      "sr.gov.pl" : 1,
      "hirogawa.wakayama.jp" : 1,
      "cc.il.us" : 1,
      "niiza.saitama.jp" : 1,
      "pi.gov.br" : 1,
      "shikatsu.aichi.jp" : 1,
      "daisen.akita.jp" : 1,
      "lib.id.us" : 1,
      "hirado.nagasaki.jp" : 1,
      "horokanai.hokkaido.jp" : 1,
      "naha.okinawa.jp" : 1,
      "akita.akita.jp" : 1,
      "nanporo.hokkaido.jp" : 1,
      "oizumi.gunma.jp" : 1,
      "minamiboso.chiba.jp" : 1,
      "hida.gifu.jp" : 1,
      "shimoda.shizuoka.jp" : 1,
      "shimofusa.chiba.jp" : 1,
      "shinjuku.tokyo.jp" : 1,
      "usa.oita.jp" : 1,
      "kurume.fukuoka.jp" : 1,
      "otoyo.kochi.jp" : 1,
      "kawaba.gunma.jp" : 1,
      "showa.yamanashi.jp" : 1,
      "yaotsu.gifu.jp" : 1,
      "oshima.yamaguchi.jp" : 1,
      "misato.akita.jp" : 1,
      "takayama.gunma.jp" : 1,
      "miyawaka.fukuoka.jp" : 1,
      "hayakawa.yamanashi.jp" : 1,
      "bifuka.hokkaido.jp" : 1,
      "kunitomi.miyazaki.jp" : 1,
      "kyonan.chiba.jp" : 1,
      "otobe.hokkaido.jp" : 1,
      "kawanabe.kagoshima.jp" : 1,
      "wakayama.wakayama.jp" : 1,
      "oyodo.nara.jp" : 1,
      "shibukawa.gunma.jp" : 1,
      "lib.az.us" : 1,
      "muika.niigata.jp" : 1,
      "wios.gov.pl" : 1,
      "toyoake.aichi.jp" : 1,
      "niyodogawa.kochi.jp" : 1,
      "ryokami.saitama.jp" : 1,
      "cc.mi.us" : 1,
      "tsuno.miyazaki.jp" : 1,
      "ohi.fukui.jp" : 1,
      "takko.aomori.jp" : 1,
      "catholic.edu.au" : 1,
      "kota.aichi.jp" : 1,
      "ogawa.saitama.jp" : 1,
      "mutsuzawa.chiba.jp" : 1,
      "nango.fukushima.jp" : 1,
      "sakai.ibaraki.jp" : 1,
      "lib.ak.us" : 1,
      "fuso.aichi.jp" : 1,
      "lib.ca.us" : 1,
      "kannami.shizuoka.jp" : 1,
      "nakadomari.aomori.jp" : 1,
      "takatori.nara.jp" : 1,
      "nagi.okayama.jp" : 1,
      "iwafune.tochigi.jp" : 1,
      "kitahiroshima.hokkaido.jp" : 1,
      "koga.ibaraki.jp" : 1,
      "taiki.mie.jp" : 1,
      "nirasaki.yamanashi.jp" : 1,
      "kikugawa.shizuoka.jp" : 1,
      "namikata.ehime.jp" : 1,
      "itami.hyogo.jp" : 1,
      "inzai.chiba.jp" : 1,
      "mg.gov.br" : 1,
      "fujisawa.kanagawa.jp" : 1,
      "shiki.saitama.jp" : 1,
      "ashikaga.tochigi.jp" : 1,
      "goto.nagasaki.jp" : 1,
      "kitagawa.miyazaki.jp" : 1,
      "kunneppu.hokkaido.jp" : 1,
      "toya.hokkaido.jp" : 1,
      "tec.mi.us" : 1,
      "hidaka.saitama.jp" : 1,
      "seihi.nagasaki.jp" : 1,
      "ogose.saitama.jp" : 1,
      "plc.co.im" : 1,
      "yachiyo.ibaraki.jp" : 1,
      "koza.wakayama.jp" : 1,
      "minokamo.gifu.jp" : 1,
      "ashibetsu.hokkaido.jp" : 1,
      "numata.hokkaido.jp" : 1,
      "bihoro.hokkaido.jp" : 1,
      "hikimi.shimane.jp" : 1,
      "yokawa.hyogo.jp" : 1,
      "k12.al.us" : 1,
      "k12.md.us" : 1,
      "sekikawa.niigata.jp" : 1,
      "hashikami.aomori.jp" : 1,
      "minami.tokushima.jp" : 1,
      "hisayama.fukuoka.jp" : 1,
      "hakuba.nagano.jp" : 1,
      "kuriyama.hokkaido.jp" : 1,
      "okutama.tokyo.jp" : 1,
      "cc.ky.us" : 1,
      "hadano.kanagawa.jp" : 1,
      "minamiyamashiro.kyoto.jp" : 1,
      "shikaoi.hokkaido.jp" : 1,
      "togitsu.nagasaki.jp" : 1,
      "tagami.niigata.jp" : 1,
      "kamo.niigata.jp" : 1,
      "kunimi.fukushima.jp" : 1,
      "sakegawa.yamagata.jp" : 1,
      "ashoro.hokkaido.jp" : 1,
      "ogano.saitama.jp" : 1,
      "hidaka.kochi.jp" : 1,
      "yuki.ibaraki.jp" : 1,
      "cc.md.us" : 1,
      "sano.tochigi.jp" : 1,
      "lib.co.us" : 1,
      "tonosho.kagawa.jp" : 1,
      "sakurai.nara.jp" : 1,
      "sakado.saitama.jp" : 1,
      "nishikatsura.yamanashi.jp" : 1,
      "izu.shizuoka.jp" : 1,
      "matsuura.nagasaki.jp" : 1,
      "nakijin.okinawa.jp" : 1,
      "lib.nd.us" : 1,
      "cc.as.us" : 1,
      "nagai.yamagata.jp" : 1,
      "nichinan.miyazaki.jp" : 1,
      "honjo.akita.jp" : 1,
      "mitake.gifu.jp" : 1,
      "kinokawa.wakayama.jp" : 1,
      "misato.miyagi.jp" : 1,
      "shoo.okayama.jp" : 1,
      "tochio.niigata.jp" : 1,
      "bungoono.oita.jp" : 1,
      "omigawa.chiba.jp" : 1,
      "oshino.yamanashi.jp" : 1,
      "yao.osaka.jp" : 1,
      "asahi.nagano.jp" : 1,
      "obira.hokkaido.jp" : 1,
      "wiih.gov.pl" : 1,
      "midori.chiba.jp" : 1,
      "kitaaiki.nagano.jp" : 1,
      "bø.nordland.no" : 1,
      "aisho.shiga.jp" : 1,
      "hannan.osaka.jp" : 1,
      "niigata.niigata.jp" : 1,
      "kamogawa.chiba.jp" : 1,
      "toyoura.hokkaido.jp" : 1,
      "nishiarita.saga.jp" : 1,
      "takayama.nagano.jp" : 1,
      "minamiashigara.kanagawa.jp" : 1,
      "taketa.oita.jp" : 1,
      "kumano.mie.jp" : 1,
      "johana.toyama.jp" : 1,
      "ora.gunma.jp" : 1,
      "kiso.nagano.jp" : 1,
      "ota.gunma.jp" : 1,
      "kayabe.hokkaido.jp" : 1,
      "ce.gov.br" : 1,
      "uji.kyoto.jp" : 1,
      "ota.tokyo.jp" : 1,
      "kumamoto.kumamoto.jp" : 1,
      "yawata.kyoto.jp" : 1,
      "wakasa.tottori.jp" : 1,
      "kanna.gunma.jp" : 1,
      "shiogama.miyagi.jp" : 1,
      "sakahogi.gifu.jp" : 1,
      "oseto.nagasaki.jp" : 1,
      "mombetsu.hokkaido.jp" : 1,
      "kawatana.nagasaki.jp" : 1,
      "valer.ostfold.no" : 1,
      "arakawa.saitama.jp" : 1,
      "taito.tokyo.jp" : 1,
      "oamishirasato.chiba.jp" : 1,
      "bato.tochigi.jp" : 1,
      "kamaishi.iwate.jp" : 1,
      "inami.toyama.jp" : 1,
      "miyama.fukuoka.jp" : 1,
      "yurihonjo.akita.jp" : 1,
      "kawagoe.saitama.jp" : 1,
      "kitagawa.kochi.jp" : 1,
      "soma.fukushima.jp" : 1,
      "yaese.okinawa.jp" : 1,
      "tagajo.miyagi.jp" : 1,
      "togo.aichi.jp" : 1,
      "nakagawa.fukuoka.jp" : 1,
      "oguni.yamagata.jp" : 1,
      "sanagochi.tokushima.jp" : 1,
      "onjuku.chiba.jp" : 1,
      "suzaka.nagano.jp" : 1,
      "happou.akita.jp" : 1,
      "matsumoto.kagoshima.jp" : 1,
      "cc.me.us" : 1,
      "onagawa.miyagi.jp" : 1,
      "cc.or.us" : 1,
      "yamada.iwate.jp" : 1,
      "lib.nj.us" : 1,
      "ikaruga.nara.jp" : 1,
      "kamagaya.chiba.jp" : 1,
      "muko.kyoto.jp" : 1,
      "shichikashuku.miyagi.jp" : 1,
      "yahiko.niigata.jp" : 1,
      "tado.mie.jp" : 1,
      "hitachiota.ibaraki.jp" : 1,
      "imakane.hokkaido.jp" : 1,
      "gs.svalbard.no" : 1,
      "oe.yamagata.jp" : 1,
      "go.gov.br" : 1,
      "tsuru.yamanashi.jp" : 1,
      "mizuho.tokyo.jp" : 1,
      "bo.telemark.no" : 1,
      "gs.tr.no" : 1,
      "iizuna.nagano.jp" : 1,
      "miyada.nagano.jp" : 1,
      "yamashina.kyoto.jp" : 1,
      "semboku.akita.jp" : 1,
      "city.nagoya.jp" : 1,
      "yonago.tottori.jp" : 1,
      "katano.osaka.jp" : 1,
      "settsu.osaka.jp" : 1,
      "abeno.osaka.jp" : 1,
      "kawahara.tottori.jp" : 1,
      "kamiichi.toyama.jp" : 1,
      "mino.gifu.jp" : 1,
      "cc.ar.us" : 1,
      "kembuchi.hokkaido.jp" : 1,
      "taku.saga.jp" : 1,
      "kadogawa.miyazaki.jp" : 1,
      "cc.nv.us" : 1,
      "kunisaki.oita.jp" : 1,
      "to.gov.br" : 1,
      "ichikawamisato.yamanashi.jp" : 1,
      "taiki.hokkaido.jp" : 1,
      "cc.mn.us" : 1,
      "rankoshi.hokkaido.jp" : 1,
      "furukawa.miyagi.jp" : 1,
      "honai.ehime.jp" : 1,
      "kui.hiroshima.jp" : 1,
      "chita.aichi.jp" : 1,
      "mashike.hokkaido.jp" : 1,
      "yasuda.kochi.jp" : 1,
      "k12.va.us" : 1,
      "umi.fukuoka.jp" : 1,
      "hatogaya.saitama.jp" : 1,
      "hamatonbetsu.hokkaido.jp" : 1,
      "pa.gov.br" : 1,
      "kamishihoro.hokkaido.jp" : 1,
      "zao.miyagi.jp" : 1,
      "kumejima.okinawa.jp" : 1,
      "piw.gov.pl" : 1,
      "chiyoda.tokyo.jp" : 1,
      "chikuzen.fukuoka.jp" : 1,
      "hikawa.shimane.jp" : 1,
      "wada.nagano.jp" : 1,
      "ono.fukui.jp" : 1,
      "k12.mi.us" : 1,
      "kimitsu.chiba.jp" : 1,
      "wakasa.fukui.jp" : 1,
      "kamo.kyoto.jp" : 1,
      "kasai.hyogo.jp" : 1,
      "hanno.saitama.jp" : 1,
      "hachirogata.akita.jp" : 1,
      "komae.tokyo.jp" : 1,
      "lib.tx.us" : 1,
      "pr.gov.pl" : 1,
      "chizu.tottori.jp" : 1,
      "himi.toyama.jp" : 1,
      "paroch.k12.ma.us" : 1,
      "lib.mt.us" : 1,
      "kwp.gov.pl" : 1,
      "shichinohe.aomori.jp" : 1,
      "moka.tochigi.jp" : 1,
      "ryuoh.shiga.jp" : 1,
      "sanda.hyogo.jp" : 1,
      "taishin.fukushima.jp" : 1,
      "koge.tottori.jp" : 1,
      "kosai.shizuoka.jp" : 1,
      "ogi.saga.jp" : 1,
      "takamori.nagano.jp" : 1,
      "higashihiroshima.hiroshima.jp" : 1,
      "tas.gov.au" : 1,
      "hatoyama.saitama.jp" : 1,
      "nyuzen.toyama.jp" : 1,
      "urakawa.hokkaido.jp" : 1,
      "k12.as.us" : 1,
      "higashiyoshino.nara.jp" : 1,
      "higashinaruse.akita.jp" : 1,
      "kuroiso.tochigi.jp" : 1,
      "aga.niigata.jp" : 1,
      "sayama.osaka.jp" : 1,
      "gs.bu.no" : 1,
      "lib.ks.us" : 1,
      "kunitachi.tokyo.jp" : 1,
      "misugi.mie.jp" : 1,
      "miyota.nagano.jp" : 1,
      "shirataka.yamagata.jp" : 1,
      "hizen.saga.jp" : 1,
      "kawaue.gifu.jp" : 1,
      "ogimi.okinawa.jp" : 1,
      "satte.saitama.jp" : 1,
      "yamakita.kanagawa.jp" : 1,
      "tsuruoka.yamagata.jp" : 1,
      "herøy.møre-og-romsdal.no" : 1,
      "k12.ne.us" : 1,
      "toon.ehime.jp" : 1,
      "nakanojo.gunma.jp" : 1,
      "vic.edu.au" : 1,
      "imabari.ehime.jp" : 1,
      "anan.tokushima.jp" : 1,
      "takehara.hiroshima.jp" : 1,
      "lib.sd.us" : 1,
      "tarumizu.kagoshima.jp" : 1,
      "shimotsuke.tochigi.jp" : 1,
      "ibara.okayama.jp" : 1,
      "tsukumi.oita.jp" : 1,
      "kitakata.miyazaki.jp" : 1,
      "matsuyama.ehime.jp" : 1,
      "omuta.fukuoka.jp" : 1,
      "yamada.toyama.jp" : 1,
      "chuo.osaka.jp" : 1,
      "saigawa.fukuoka.jp" : 1,
      "takazaki.miyazaki.jp" : 1,
      "soo.kagoshima.jp" : 1,
      "akune.kagoshima.jp" : 1,
      "yoshimi.saitama.jp" : 1,
      "kashima.saga.jp" : 1,
      "nikaho.akita.jp" : 1,
      "chino.nagano.jp" : 1,
      "hikone.shiga.jp" : 1,
      "kyowa.hokkaido.jp" : 1,
      "kiyokawa.kanagawa.jp" : 1,
      "gs.jan-mayen.no" : 1,
      "godo.gifu.jp" : 1,
      "k12.or.us" : 1,
      "onga.fukuoka.jp" : 1,
      "nachikatsuura.wakayama.jp" : 1,
      "kodaira.tokyo.jp" : 1,
      "yachimata.chiba.jp" : 1,
      "kudoyama.wakayama.jp" : 1,
      "chikuho.fukuoka.jp" : 1,
      "aikawa.kanagawa.jp" : 1,
      "yomitan.okinawa.jp" : 1,
      "sakyo.kyoto.jp" : 1,
      "tsuyama.okayama.jp" : 1,
      "fujimi.nagano.jp" : 1,
      "ama.shimane.jp" : 1,
      "kumagaya.saitama.jp" : 1,
      "cc.ms.us" : 1,
      "uruma.okinawa.jp" : 1,
      "gen.mi.us" : 1,
      "miyama.mie.jp" : 1,
      "ishinomaki.miyagi.jp" : 1,
      "kouzushima.tokyo.jp" : 1,
      "cc.mt.us" : 1,
      "nozawaonsen.nagano.jp" : 1,
      "tobe.ehime.jp" : 1,
      "tsuno.kochi.jp" : 1,
      "wif.gov.pl" : 1,
      "ofunato.iwate.jp" : 1,
      "sobetsu.hokkaido.jp" : 1,
      "kibichuo.okayama.jp" : 1,
      "matsue.shimane.jp" : 1,
      "furudono.fukushima.jp" : 1,
      "uto.kumamoto.jp" : 1,
      "ikata.ehime.jp" : 1,
      "ogori.fukuoka.jp" : 1,
      "tokai.ibaraki.jp" : 1,
      "shakotan.hokkaido.jp" : 1,
      "osakikamijima.hiroshima.jp" : 1,
      "niihama.ehime.jp" : 1,
      "katsuragi.nara.jp" : 1,
      "lib.fl.us" : 1,
      "omotego.fukushima.jp" : 1,
      "koto.tokyo.jp" : 1,
      "hakone.kanagawa.jp" : 1,
      "sasaguri.fukuoka.jp" : 1,
      "usuki.oita.jp" : 1,
      "nakatsugawa.gifu.jp" : 1,
      "misato.shimane.jp" : 1,
      "katashina.gunma.jp" : 1,
      "hioki.kagoshima.jp" : 1,
      "kami.miyagi.jp" : 1,
      "lib.wa.us" : 1,
      "yamagata.ibaraki.jp" : 1,
      "yoshida.saitama.jp" : 1,
      "urayasu.chiba.jp" : 1,
      "minamiaiki.nagano.jp" : 1,
      "sakura.chiba.jp" : 1,
      "nagareyama.chiba.jp" : 1,
      "moriyoshi.akita.jp" : 1,
      "saku.nagano.jp" : 1,
      "mizusawa.iwate.jp" : 1,
      "oke.gov.pl" : 1,
      "nakaniikawa.toyama.jp" : 1,
      "maizuru.kyoto.jp" : 1,
      "shiroishi.miyagi.jp" : 1,
      "shimada.shizuoka.jp" : 1,
      "nagasu.kumamoto.jp" : 1,
      "cc.wv.us" : 1,
      "togakushi.nagano.jp" : 1,
      "kawanishi.yamagata.jp" : 1,
      "kamikawa.hyogo.jp" : 1,
      "hanamigawa.chiba.jp" : 1,
      "midori.gunma.jp" : 1,
      "minowa.nagano.jp" : 1,
      "iruma.saitama.jp" : 1,
      "fudai.iwate.jp" : 1,
      "katsuyama.fukui.jp" : 1,
      "kawajima.saitama.jp" : 1,
      "aizuwakamatsu.fukushima.jp" : 1,
      "mito.ibaraki.jp" : 1,
      "horonobe.hokkaido.jp" : 1,
      "tobetsu.hokkaido.jp" : 1,
      "tako.chiba.jp" : 1,
      "hara.nagano.jp" : 1,
      "atsuma.hokkaido.jp" : 1,
      "hirara.okinawa.jp" : 1,
      "okagaki.fukuoka.jp" : 1,
      "kasahara.gifu.jp" : 1,
      "haebaru.okinawa.jp" : 1,
      "kumano.hiroshima.jp" : 1,
      "herøy.nordland.no" : 1,
      "yura.wakayama.jp" : 1,
      "inagi.tokyo.jp" : 1,
      "komoro.nagano.jp" : 1,
      "tondabayashi.osaka.jp" : 1,
      "joyo.kyoto.jp" : 1,
      "kobayashi.miyazaki.jp" : 1,
      "heroy.more-og-romsdal.no" : 1,
      "omura.nagasaki.jp" : 1,
      "kitagata.saga.jp" : 1,
      "shikama.miyagi.jp" : 1,
      "okuma.fukushima.jp" : 1,
      "valer.hedmark.no" : 1,
      "munakata.fukuoka.jp" : 1,
      "kitagata.gifu.jp" : 1,
      "sumida.tokyo.jp" : 1,
      "hakata.fukuoka.jp" : 1,
      "toei.aichi.jp" : 1,
      "okazaki.aichi.jp" : 1,
      "sko.gov.pl" : 1,
      "sasayama.hyogo.jp" : 1,
      "ariake.saga.jp" : 1,
      "mitsuke.niigata.jp" : 1,
      "itoigawa.niigata.jp" : 1,
      "yahaba.iwate.jp" : 1,
      "ashiya.hyogo.jp" : 1,
      "yasaka.nagano.jp" : 1,
      "lib.hi.us" : 1,
      "omi.niigata.jp" : 1,
      "iwade.wakayama.jp" : 1,
      "miyoshi.saitama.jp" : 1,
      "koka.shiga.jp" : 1,
      "komaki.aichi.jp" : 1,
      "kanra.gunma.jp" : 1,
      "ichikai.tochigi.jp" : 1,
      "nanao.ishikawa.jp" : 1,
      "wajima.ishikawa.jp" : 1,
      "shika.ishikawa.jp" : 1,
      "shimotsuma.ibaraki.jp" : 1,
      "higashiyamato.tokyo.jp" : 1,
      "okawa.kochi.jp" : 1,
      "izena.okinawa.jp" : 1,
      "tomika.gifu.jp" : 1,
      "wskr.gov.pl" : 1,
      "miyazu.kyoto.jp" : 1,
      "fukushima.hokkaido.jp" : 1,
      "asakawa.fukushima.jp" : 1,
      "kusatsu.shiga.jp" : 1,
      "kasuga.hyogo.jp" : 1,
      "joboji.iwate.jp" : 1,
      "haga.tochigi.jp" : 1,
      "matsushige.tokushima.jp" : 1,
      "toride.ibaraki.jp" : 1,
      "hekinan.aichi.jp" : 1,
      "lib.ct.us" : 1,
      "oow.gov.pl" : 1,
      "misaki.okayama.jp" : 1,
      "hitachiomiya.ibaraki.jp" : 1,
      "ohira.miyagi.jp" : 1,
      "kokubunji.tokyo.jp" : 1,
      "cc.pa.us" : 1,
      "lib.ok.us" : 1,
      "taishi.osaka.jp" : 1,
      "k12.ms.us" : 1,
      "lib.ut.us" : 1,
      "gs.aa.no" : 1,
      "tatebayashi.gunma.jp" : 1,
      "chikushino.fukuoka.jp" : 1,
      "chippubetsu.hokkaido.jp" : 1,
      "hita.oita.jp" : 1,
      "chonan.chiba.jp" : 1,
      "kashima.ibaraki.jp" : 1,
      "motoyama.kochi.jp" : 1,
      "umaji.kochi.jp" : 1,
      "hirono.iwate.jp" : 1,
      "kounosu.saitama.jp" : 1,
      "nakagusuku.okinawa.jp" : 1,
      "shibata.niigata.jp" : 1,
      "mobara.chiba.jp" : 1,
      "yamato.kumamoto.jp" : 1,
      "kaisei.kanagawa.jp" : 1,
      "kawaminami.miyazaki.jp" : 1,
      "miura.kanagawa.jp" : 1,
      "ichihara.chiba.jp" : 1,
      "fussa.tokyo.jp" : 1,
      "hino.tottori.jp" : 1,
      "bandai.fukushima.jp" : 1,
      "gokase.miyazaki.jp" : 1,
      "tokoname.aichi.jp" : 1,
      "k12.ky.us" : 1,
      "ro.gov.br" : 1,
      "minamiizu.shizuoka.jp" : 1,
      "namegawa.saitama.jp" : 1,
      "tsugaru.aomori.jp" : 1,
      "namerikawa.toyama.jp" : 1,
      "kyowa.akita.jp" : 1,
      "lib.gu.us" : 1,
      "ozu.ehime.jp" : 1,
      "shinkamigoto.nagasaki.jp" : 1,
      "aioi.hyogo.jp" : 1,
      "kasukabe.saitama.jp" : 1,
      "df.gov.br" : 1,
      "starostwo.gov.pl" : 1,
      "tomiya.miyagi.jp" : 1,
      "toga.toyama.jp" : 1,
      "wakkanai.hokkaido.jp" : 1,
      "oki.fukuoka.jp" : 1,
      "atsugi.kanagawa.jp" : 1,
      "nishikata.tochigi.jp" : 1,
      "yufu.oita.jp" : 1,
      "higashiyodogawa.osaka.jp" : 1,
      "lib.in.us" : 1,
      "cc.sd.us" : 1,
      "kawakami.nara.jp" : 1,
      "kosei.shiga.jp" : 1,
      "nanto.toyama.jp" : 1,
      "mizunami.gifu.jp" : 1,
      "kazuno.akita.jp" : 1,
      "nakamura.kochi.jp" : 1,
      "kaho.fukuoka.jp" : 1,
      "cc.nd.us" : 1,
      "kagami.kochi.jp" : 1,
      "miki.hyogo.jp" : 1,
      "susono.shizuoka.jp" : 1,
      "fuchu.hiroshima.jp" : 1,
      "togura.nagano.jp" : 1,
      "yuzawa.niigata.jp" : 1,
      "imari.saga.jp" : 1,
      "cc.ia.us" : 1,
      "aogashima.tokyo.jp" : 1,
      "os.hordaland.no" : 1,
      "hasama.oita.jp" : 1,
      "embetsu.hokkaido.jp" : 1,
      "oita.oita.jp" : 1,
      "cc.hi.us" : 1,
      "nakama.fukuoka.jp" : 1,
      "wajiki.tokushima.jp" : 1,
      "etajima.hiroshima.jp" : 1,
      "lib.la.us" : 1,
      "sodegaura.chiba.jp" : 1,
      "itabashi.tokyo.jp" : 1,
      "ebina.kanagawa.jp" : 1,
      "fukuchiyama.kyoto.jp" : 1,
      "shibecha.hokkaido.jp" : 1,
      "shioya.tochigi.jp" : 1,
      "minano.saitama.jp" : 1,
      "gs.hl.no" : 1,
      "kita.tokyo.jp" : 1,
      "lib.ga.us" : 1,
      "minamiawaji.hyogo.jp" : 1,
      "iwate.iwate.jp" : 1,
      "hagi.yamaguchi.jp" : 1,
      "nishitosa.kochi.jp" : 1,
      "nakano.tokyo.jp" : 1,
      "murayama.yamagata.jp" : 1,
      "makinohara.shizuoka.jp" : 1,
      "kamijima.ehime.jp" : 1,
      "ibaraki.osaka.jp" : 1,
      "suzuka.mie.jp" : 1,
      "izumisano.osaka.jp" : 1,
      "schools.nsw.edu.au" : 1,
      "kagamino.okayama.jp" : 1,
      "hofu.yamaguchi.jp" : 1,
      "tsubame.niigata.jp" : 1,
      "hamamatsu.shizuoka.jp" : 1,
      "yoichi.hokkaido.jp" : 1,
      "otofuke.hokkaido.jp" : 1,
      "tone.ibaraki.jp" : 1,
      "koryo.nara.jp" : 1,
      "yokoze.saitama.jp" : 1,
      "higashine.yamagata.jp" : 1,
      "sumita.iwate.jp" : 1,
      "cc.ca.us" : 1,
      "narita.chiba.jp" : 1,
      "chuo.fukuoka.jp" : 1,
      "zpisdn.gov.pl" : 1,
      "harima.hyogo.jp" : 1,
      "chiryu.aichi.jp" : 1,
      "kurogi.fukuoka.jp" : 1,
      "noshiro.akita.jp" : 1,
      "ninohe.iwate.jp" : 1,
      "cc.ne.us" : 1,
      "k12.mn.us" : 1,
      "sakaki.nagano.jp" : 1,
      "sunagawa.hokkaido.jp" : 1,
      "ikusaka.nagano.jp" : 1,
      "sande.møre-og-romsdal.no" : 1,
      "lib.mo.us" : 1,
      "gov.nc.tr" : 1,
      "iki.nagasaki.jp" : 1,
      "ap.gov.pl" : 1,
      "zp.gov.pl" : 1,
      "inashiki.ibaraki.jp" : 1,
      "mashiko.tochigi.jp" : 1,
      "shobara.hiroshima.jp" : 1,
      "tonaki.okinawa.jp" : 1
   },
   "second_level_domain" : {
      "fauske.no" : 1,
      "ac.th" : 1,
      "hl.no" : 1,
      "com.ws" : 1,
      "gouv.fr" : 1,
      "rendalen.no" : 1,
      "association.aero" : 1,
      "edu.pr" : 1,
      "bl.it" : 1,
      "at.it" : 1,
      "mil.cn" : 1,
      "hokksund.no" : 1,
      "trentino-alto-adige.it" : 1,
      "andriabarlettatrani.it" : 1,
      "net.kw" : 1,
      "gov.pk" : 1,
      "hvaler.no" : 1,
      "net.vu" : 1,
      "bremanger.no" : 1,
      "org.je" : 1,
      "agdenes.no" : 1,
      "pulawy.pl" : 1,
      "im.it" : 1,
      "web.nf" : 1,
      "net.ag" : 1,
      "gov.sy" : 1,
      "org.ng" : 1,
      "piemonte.it" : 1,
      "målselv.no" : 1,
      "net.lr" : 1,
      "abc.br" : 1,
      "ac.rw" : 1,
      "kiwi.nz" : 1,
      "dovre.no" : 1,
      "law.za" : 1,
      "kg.kr" : 1,
      "gov.tm" : 1,
      "florence.it" : 1,
      "org.ma" : 1,
      "rec.nf" : 1,
      "prd.fr" : 1,
      "arq.br" : 1,
      "mil.zm" : 1,
      "com.bt" : 1,
      "aerobatic.aero" : 1,
      "org.mk" : 1,
      "sv.it" : 1,
      "educator.aero" : 1,
      "time.no" : 1,
      "ud.it" : 1,
      "go.id" : 1,
      "com.al" : 1,
      "fm.no" : 1,
      "net.mv" : 1,
      "com.vc" : 1,
      "campania.it" : 1,
      "org.il" : 1,
      "lt.it" : 1,
      "org.rs" : 1,
      "luhansk.ua" : 1,
      "vagsoy.no" : 1,
      "tokke.no" : 1,
      "lødingen.no" : 1,
      "co.tm" : 1,
      "fhs.no" : 1,
      "haiphong.vn" : 1,
      "szex.hu" : 1,
      "abr.it" : 1,
      "co.ae" : 1,
      "flog.br" : 1,
      "seoul.kr" : 1,
      "pb.ao" : 1,
      "鳥取.jp" : 1,
      "or.at" : 1,
      "meråker.no" : 1,
      "dønna.no" : 1,
      "stavanger.no" : 1,
      "an.it" : 1,
      "haugesund.no" : 1,
      "si.it" : 1,
      "valleeaoste.it" : 1,
      "lorenskog.no" : 1,
      "konyvelo.hu" : 1,
      "trentino-a-adige.it" : 1,
      "k.bg" : 1,
      "notteroy.no" : 1,
      "kobe.jp" : 1,
      "in.ua" : 1,
      "va.it" : 1,
      "school.nz" : 1,
      "gildeskål.no" : 1,
      "firm.ht" : 1,
      "eidsvoll.no" : 1,
      "com.st" : 1,
      "club.aero" : 1,
      "org.mu" : 1,
      "org.gh" : 1,
      "loabát.no" : 1,
      "lapy.pl" : 1,
      "gov.by" : 1,
      "rawa-maz.pl" : 1,
      "orkdal.no" : 1,
      "aejrie.no" : 1,
      "edu.lc" : 1,
      "edu.lb" : 1,
      "gov.bt" : 1,
      "shop.ht" : 1,
      "te.ua" : 1,
      "miasta.pl" : 1,
      "fin.ec" : 1,
      "com.tm" : 1,
      "biz.bb" : 1,
      "ha.cn" : 1,
      "net.ph" : 1,
      "tjøme.no" : 1,
      "sorum.no" : 1,
      "priv.me" : 1,
      "org.sg" : 1,
      "foggia.it" : 1,
      "lesja.no" : 1,
      "com.sy" : 1,
      "edu.bs" : 1,
      "com.pa" : 1,
      "carraramassa.it" : 1,
      "blog.br" : 1,
      "nom.mg" : 1,
      "råholt.no" : 1,
      "pa.it" : 1,
      "miyagi.jp" : 1,
      "com.pk" : 1,
      "org.es" : 1,
      "org.sn" : 1,
      "jl.cn" : 1,
      "rieti.it" : 1,
      "econo.bj" : 1,
      "biz.ls" : 1,
      "revista.bo" : 1,
      "wloclawek.pl" : 1,
      "go.cr" : 1,
      "amot.no" : 1,
      "int.la" : 1,
      "haiduong.vn" : 1,
      "levanger.no" : 1,
      "adv.br" : 1,
      "tokushima.jp" : 1,
      "网絡.hk" : 1,
      "name.ng" : 1,
      "osteroy.no" : 1,
      "gov.ws" : 1,
      "int.lk" : 1,
      "ascolipiceno.it" : 1,
      "frosinone.it" : 1,
      "nom.io" : 1,
      "nordkapp.no" : 1,
      "bio.br" : 1,
      "edu.eg" : 1,
      "vn.ua" : 1,
      "des.br" : 1,
      "ol.no" : 1,
      "natal.br" : 1,
      "yamaguchi.jp" : 1,
      "cs.it" : 1,
      "tm.no" : 1,
      "com.by" : 1,
      "ciencia.bo" : 1,
      "nj.us" : 1,
      "frei.no" : 1,
      "pro.ec" : 1,
      "tananger.no" : 1,
      "co.at" : 1,
      "vikna.no" : 1,
      "store.st" : 1,
      "tw.cn" : 1,
      "co.cm" : 1,
      "co.rs" : 1,
      "gen.tr" : 1,
      "fvg.it" : 1,
      "埼玉.jp" : 1,
      "gjøvik.no" : 1,
      "hamar.no" : 1,
      "galsa.no" : 1,
      "stokke.no" : 1,
      "gob.hn" : 1,
      "rep.kp" : 1,
      "seljord.no" : 1,
      "utazas.hu" : 1,
      "kawasaki.jp" : 1,
      "recreation.aero" : 1,
      "岐阜.jp" : 1,
      "ac.mw" : 1,
      "gouv.sn" : 1,
      "vindafjord.no" : 1,
      "ce.it" : 1,
      "jor.br" : 1,
      "edu.mz" : 1,
      "lodi.it" : 1,
      "tmp.br" : 1,
      "ln.cn" : 1,
      "suwalki.pl" : 1,
      "høylandet.no" : 1,
      "dyroy.no" : 1,
      "edu.gp" : 1,
      "sveio.no" : 1,
      "sapporo.jp" : 1,
      "aosta-valley.it" : 1,
      "sld.pa" : 1,
      "edu.ss" : 1,
      "gov.vc" : 1,
      "mátta-várjjat.no" : 1,
      "gov.al" : 1,
      "gniezno.pl" : 1,
      "org.bn" : 1,
      "com.to" : 1,
      "静岡.jp" : 1,
      "pg.in" : 1,
      "gujarat.in" : 1,
      "nagasaki.jp" : 1,
      "gov.ar" : 1,
      "other.nf" : 1,
      "perugia.it" : 1,
      "組織.hk" : 1,
      "pa.us" : 1,
      "bahccavuotna.no" : 1,
      "leikanger.no" : 1,
      "aarborte.no" : 1,
      "baidar.no" : 1,
      "cuiaba.br" : 1,
      "matera.it" : 1,
      "de.us" : 1,
      "trentino-stirol.it" : 1,
      "fi.cr" : 1,
      "incheon.kr" : 1,
      "ancona.it" : 1,
      "net.bz" : 1,
      "sør-fron.no" : 1,
      "edu.ua" : 1,
      "co.tj" : 1,
      "h.se" : 1,
      "grong.no" : 1,
      "mil.ba" : 1,
      "aircraft.aero" : 1,
      "com.tw" : 1,
      "zp.ua" : 1,
      "froland.no" : 1,
      "eun.eg" : 1,
      "org.om" : 1,
      "navigation.aero" : 1,
      "komforb.se" : 1,
      "etne.no" : 1,
      "ivgu.no" : 1,
      "musica.ar" : 1,
      "�司.hk" : 1,
      "鹿�島.jp" : 1,
      "net.ve" : 1,
      "so.it" : 1,
      "oystre-slidre.no" : 1,
      "suldal.no" : 1,
      "edu.vg" : 1,
      "org.me" : 1,
      "mil.py" : 1,
      "cagliari.it" : 1,
      "gouv.km" : 1,
      "edu.gy" : 1,
      "edu.au" : 1,
      "psc.br" : 1,
      "net.ms" : 1,
      "fl.us" : 1,
      "教育.hk" : 1,
      "ac.eg" : 1,
      "kitakyushu.jp" : 1,
      "rio.br" : 1,
      "info.tt" : 1,
      "aid.pl" : 1,
      "tv.tr" : 1,
      "austrheim.no" : 1,
      "ia.us" : 1,
      "jeonbuk.kr" : 1,
      "skedsmokorset.no" : 1,
      "saltdal.no" : 1,
      "ac.tz" : 1,
      "pescara.it" : 1,
      "com.ro" : 1,
      "tv.in" : 1,
      "tas.au" : 1,
      "biz.et" : 1,
      "org.bi" : 1,
      "mil.tz" : 1,
      "edu.vn" : 1,
      "co.bb" : 1,
      "uz.ua" : 1,
      "s.bg" : 1,
      "tourism.bj" : 1,
      "swidnica.pl" : 1,
      "marker.no" : 1,
      "gov.ec" : 1,
      "udi.br" : 1,
      "midsund.no" : 1,
      "ne.ke" : 1,
      "熊本.jp" : 1,
      "nagano.jp" : 1,
      "padua.it" : 1,
      "jørpeland.no" : 1,
      "net.ht" : 1,
      "nghean.vn" : 1,
      "odo.br" : 1,
      "stjordal.no" : 1,
      "control.aero" : 1,
      "vennesla.no" : 1,
      "org.ly" : 1,
      "greta.fr" : 1,
      "org.cu" : 1,
      "nt.au" : 1,
      "enf.br" : 1,
      "高知.jp" : 1,
      "ne.kr" : 1,
      "net.dz" : 1,
      "info.nf" : 1,
      "tgory.pl" : 1,
      "hadsel.no" : 1,
      "røyken.no" : 1,
      "pc.pl" : 1,
      "org.ni" : 1,
      "ac.za" : 1,
      "edu.qa" : 1,
      "mil.zw" : 1,
      "snaase.no" : 1,
      "hotel.hu" : 1,
      "bib.br" : 1,
      "gov.tw" : 1,
      "savona.it" : 1,
      "jaworzno.pl" : 1,
      "id.cv" : 1,
      "rennebu.no" : 1,
      "ltd.gi" : 1,
      "net.ai" : 1,
      "cherkassy.ua" : 1,
      "nøtterøy.no" : 1,
      "trentinosued-tirol.it" : 1,
      "net.cv" : 1,
      "am.br" : 1,
      "kropyvnytskyi.ua" : 1,
      "auto.pl" : 1,
      "cam.it" : 1,
      "co.ss" : 1,
      "tm.hu" : 1,
      "høyanger.no" : 1,
      "skierva.no" : 1,
      "ehime.jp" : 1,
      "com.ar" : 1,
      "bacgiang.vn" : 1,
      "oristano.it" : 1,
      "gov.to" : 1,
      "org.ir" : 1,
      "mil.io" : 1,
      "skånland.no" : 1,
      "i.se" : 1,
      "kh.ua" : 1,
      "网络.cn" : 1,
      "edu.gt" : 1,
      "info.ke" : 1,
      "org.uz" : 1,
      "media.hu" : 1,
      "mb.it" : 1,
      "com.ec" : 1,
      "va.us" : 1,
      "vallee-d-aoste.it" : 1,
      "brønnøy.no" : 1,
      "org.az" : 1,
      "gifu.jp" : 1,
      "p.se" : 1,
      "ac.bw" : 1,
      "com.pe" : 1,
      "avellino.it" : 1,
      "sandnessjøen.no" : 1,
      "o.se" : 1,
      "sardegna.it" : 1,
      "个人.hk" : 1,
      "mil.mg" : 1,
      "oygarden.no" : 1,
      "dielddanuorri.no" : 1,
      "edu.sv" : 1,
      "on.ca" : 1,
      "idv.tw" : 1,
      "campobasso.it" : 1,
      "aéroport.ci" : 1,
      "tonsberg.no" : 1,
      "yalta.ua" : 1,
      "klabu.no" : 1,
      "biz.tt" : 1,
      "gouv.ci" : 1,
      "caa.aero" : 1,
      "aukra.no" : 1,
      "net.nz" : 1,
      "gov.rw" : 1,
      "edu.pl" : 1,
      "agents.aero" : 1,
      "ac.ci" : 1,
      "est.pr" : 1,
      "kongsberg.no" : 1,
      "friuli-ve-giulia.it" : 1,
      "åfjord.no" : 1,
      "ed.jp" : 1,
      "sunndal.no" : 1,
      "e.bg" : 1,
      "palermo.it" : 1,
      "cremona.it" : 1,
      "org.kn" : 1,
      "edu.hn" : 1,
      "kvafjord.no" : 1,
      "va.no" : 1,
      "ha.no" : 1,
      "riopreto.br" : 1,
      "consulado.st" : 1,
      "net.gg" : 1,
      "kharkov.ua" : 1,
      "ac.pr" : 1,
      "ba.it" : 1,
      "daknong.vn" : 1,
      "dr.in" : 1,
      "sund.no" : 1,
      "ac.in" : 1,
      "tv.tz" : 1,
      "babia-gora.pl" : 1,
      "lindås.no" : 1,
      "pri.ee" : 1,
      "2000.hu" : 1,
      "com.pt" : 1,
      "org.ao" : 1,
      "sd.us" : 1,
      "mil.ar" : 1,
      "student.aero" : 1,
      "dr.tr" : 1,
      "com.io" : 1,
      "aip.ee" : 1,
      "vda.it" : 1,
      "net.ml" : 1,
      "utsira.no" : 1,
      "gialai.vn" : 1,
      "caxias.br" : 1,
      "isla.pr" : 1,
      "com.ci" : 1,
      "tv.eg" : 1,
      "meraker.no" : 1,
      "fukuoka.jp" : 1,
      "edu.br" : 1,
      "bájddar.no" : 1,
      "nc.tr" : 1,
      "ås.no" : 1,
      "fm.it" : 1,
      "piacenza.it" : 1,
      "go.th" : 1,
      "com.ky" : 1,
      "org.kg" : 1,
      "org.iq" : 1,
      "edu.tj" : 1,
      "net.gn" : 1,
      "med.ly" : 1,
      "gov.ba" : 1,
      "kutno.pl" : 1,
      "tranby.no" : 1,
      "kagawa.jp" : 1,
      "rec.br" : 1,
      "naklo.pl" : 1,
      "net.do" : 1,
      "urbinopesaro.it" : 1,
      "9guacu.br" : 1,
      "trentinoalto-adige.it" : 1,
      "le.it" : 1,
      "bulsan-sudtirol.it" : 1,
      "eidskog.no" : 1,
      "alta.no" : 1,
      "camau.vn" : 1,
      "trentinosüd-tirol.it" : 1,
      "nat.cu" : 1,
      "net.za" : 1,
      "nom.tm" : 1,
      "sch.sa" : 1,
      "com.gl" : 1,
      "nt.no" : 1,
      "sp.it" : 1,
      "gov.py" : 1,
      "phd.jo" : 1,
      "poltava.ua" : 1,
      "org.ac" : 1,
      "ky.us" : 1,
      "sklep.pl" : 1,
      "e164.arpa" : 1,
      "biz.mv" : 1,
      "香川.jp" : 1,
      "skaun.no" : 1,
      "vevelstad.no" : 1,
      "consulting.aero" : 1,
      "edu.sd" : 1,
      "mil.ec" : 1,
      "edu.jo" : 1,
      "kr.ua" : 1,
      "laakesvuemie.no" : 1,
      "gov.sa" : 1,
      "com.na" : 1,
      "name.qa" : 1,
      "economia.bo" : 1,
      "vibo-valentia.it" : 1,
      "ac.id" : 1,
      "volyn.ua" : 1,
      "edu.ge" : 1,
      "asn.lv" : 1,
      "profesional.bo" : 1,
      "cesena-forlì.it" : 1,
      "ta.it" : 1,
      "mil.pe" : 1,
      "baclieu.vn" : 1,
      "milano.it" : 1,
      "adv.mz" : 1,
      "nord-fron.no" : 1,
      "org.my" : 1,
      "pesarourbino.it" : 1,
      "grane.no" : 1,
      "walbrzych.pl" : 1,
      "com.mg" : 1,
      "nom.pa" : 1,
      "web.tj" : 1,
      "cieszyn.pl" : 1,
      "airport.aero" : 1,
      "edu.ps" : 1,
      "com.ba" : 1,
      "mil.tw" : 1,
      "u.bg" : 1,
      "alto-adige.it" : 1,
      "ac.cr" : 1,
      "gov.zw" : 1,
      "agri.jo" : 1,
      "vallee-aoste.it" : 1,
      "quangninh.vn" : 1,
      "store.nf" : 1,
      "org.mt" : 1,
      "varggat.no" : 1,
      "gyeongnam.kr" : 1,
      "angiang.vn" : 1,
      "commune.am" : 1,
      "net.bb" : 1,
      "co.ls" : 1,
      "thainguyen.vn" : 1,
      "hl.cn" : 1,
      "barlettatraniandria.it" : 1,
      "edu.pf" : 1,
      "med.om" : 1,
      "jeju.kr" : 1,
      "málatvuopmi.no" : 1,
      "net.so" : 1,
      "home.arpa" : 1,
      "co.nz" : 1,
      "gov.io" : 1,
      "org.tr" : 1,
      "mil.to" : 1,
      "www.ro" : 1,
      "research.aero" : 1,
      "kristiansund.no" : 1,
      "co.ni" : 1,
      "production.aero" : 1,
      "gov.pt" : 1,
      "ac.zw" : 1,
      "asso.nc" : 1,
      "donetsk.ua" : 1,
      "info.et" : 1,
      "net.ls" : 1,
      "io.in" : 1,
      "pruszkow.pl" : 1,
      "int.az" : 1,
      "monzabrianza.it" : 1,
      "org.bj" : 1,
      "net.bh" : 1,
      "cn.it" : 1,
      "bel.tr" : 1,
      "hagiang.vn" : 1,
      "net.fm" : 1,
      "orskog.no" : 1,
      "ong.br" : 1,
      "bialystok.pl" : 1,
      "napoli.it" : 1,
      "firenze.it" : 1,
      "gov.mg" : 1,
      "tel.tr" : 1,
      "edu.nr" : 1,
      "holtalen.no" : 1,
      "snillfjord.no" : 1,
      "arendal.no" : 1,
      "vlog.br" : 1,
      "net.bo" : 1,
      "vefsn.no" : 1,
      "slupsk.pl" : 1,
      "inderoy.no" : 1,
      "x.se" : 1,
      "patria.bo" : 1,
      "info.ro" : 1,
      "org.ye" : 1,
      "com.sa" : 1,
      "school.ge" : 1,
      "hm.no" : 1,
      "gov.na" : 1,
      "inf.cu" : 1,
      "wielun.pl" : 1,
      "ninhbinh.vn" : 1,
      "net.sb" : 1,
      "nb.ca" : 1,
      "tt.im" : 1,
      "net.sc" : 1,
      "edu.co" : 1,
      "org.pn" : 1,
      "int.ni" : 1,
      "k12.vi" : 1,
      "r.se" : 1,
      "net.sh" : 1,
      "trapani.it" : 1,
      "ct.it" : 1,
      "sos.pl" : 1,
      "net.bw" : 1,
      "ab.ca" : 1,
      "com.py" : 1,
      "mil.rw" : 1,
      "gov.mn" : 1,
      "padova.it" : 1,
      "halsa.no" : 1,
      "pila.pl" : 1,
      "nis.za" : 1,
      "沖縄.jp" : 1,
      "balsan.it" : 1,
      "kartuzy.pl" : 1,
      "lutsk.ua" : 1,
      "edu.cw" : 1,
      "gob.sv" : 1,
      "t.se" : 1,
      "zhytomyr.ua" : 1,
      "iz.hr" : 1,
      "nom.pe" : 1,
      "gov.gd" : 1,
      "monza-e-della-brianza.it" : 1,
      "ne.jp" : 1,
      "limanowa.pl" : 1,
      "箇人.hk" : 1,
      "sch.zm" : 1,
      "øyer.no" : 1,
      "sørfold.no" : 1,
      "gunma.jp" : 1,
      "slask.pl" : 1,
      "gov.gr" : 1,
      "gov.cn" : 1,
      "tsk.tr" : 1,
      "piedmont.it" : 1,
      "vix.br" : 1,
      "edu.mw" : 1,
      "vibovalentia.it" : 1,
      "y.bg" : 1,
      "web.gu" : 1,
      "oyer.no" : 1,
      "trd.br" : 1,
      "skanland.no" : 1,
      "zaporizhzhia.ua" : 1,
      "bahcavuotna.no" : 1,
      "茨城.jp" : 1,
      "com.af" : 1,
      "cantho.vn" : 1,
      "tv.bo" : 1,
      "assn.lk" : 1,
      "gov.zm" : 1,
      "glogow.pl" : 1,
      "averoy.no" : 1,
      "nl.ca" : 1,
      "melhus.no" : 1,
      "co.cl" : 1,
      "mil.tm" : 1,
      "组织.hk" : 1,
      "kirovograd.ua" : 1,
      "gov.kp" : 1,
      "gaular.no" : 1,
      "turystyka.pl" : 1,
      "gloppen.no" : 1,
      "net.hk" : 1,
      "g.se" : 1,
      "sevastopol.ua" : 1,
      "ålgård.no" : 1,
      "edu.mo" : 1,
      "dongthap.vn" : 1,
      "mil.sy" : 1,
      "aomori.jp" : 1,
      "leksvik.no" : 1,
      "logistics.aero" : 1,
      "jorpeland.no" : 1,
      "m.se" : 1,
      "net.bm" : 1,
      "rm.it" : 1,
      "granvin.no" : 1,
      "ct.us" : 1,
      "org.fj" : 1,
      "sd.cn" : 1,
      "langevåg.no" : 1,
      "7.bg" : 1,
      "edu.uy" : 1,
      "radio.br" : 1,
      "福島.jp" : 1,
      "coop.br" : 1,
      "chiba.jp" : 1,
      "royken.no" : 1,
      "go.tz" : 1,
      "edu.gu" : 1,
      "gob.gt" : 1,
      "eid.no" : 1,
      "cahcesuolo.no" : 1,
      "gmina.pl" : 1,
      "mi.th" : 1,
      "ålesund.no" : 1,
      "lubin.pl" : 1,
      "il.us" : 1,
      "hb.cn" : 1,
      "inf.mk" : 1,
      "forde.no" : 1,
      "sondre-land.no" : 1,
      "千葉.jp" : 1,
      "mil.by" : 1,
      "hotel.lk" : 1,
      "edu.in" : 1,
      "casino.hu" : 1,
      "navuotna.no" : 1,
      "cesena-forli.it" : 1,
      "pmn.it" : 1,
      "net.et" : 1,
      "friulive-giulia.it" : 1,
      "como.it" : 1,
      "info.gu" : 1,
      "modena.it" : 1,
      "sor-varanger.no" : 1,
      "basilicata.it" : 1,
      "b.se" : 1,
      "rodoy.no" : 1,
      "gangaviika.no" : 1,
      "bjarkoy.no" : 1,
      "lodingen.no" : 1,
      "asso.ht" : 1,
      "szkola.pl" : 1,
      "sálát.no" : 1,
      "homebuilt.aero" : 1,
      "edu.sl" : 1,
      "steigen.no" : 1,
      "as.us" : 1,
      "aosta.it" : 1,
      "org.ki" : 1,
      "f.se" : 1,
      "com.kp" : 1,
      "gov.as" : 1,
      "com.im" : 1,
      "gok.pk" : 1,
      "my.id" : 1,
      "com.zm" : 1,
      "ra.it" : 1,
      "org.am" : 1,
      "leasing.aero" : 1,
      "ne.us" : 1,
      "gov.af" : 1,
      "aurskog-høland.no" : 1,
      "org.cy" : 1,
      "vb.it" : 1,
      "gaivuotna.no" : 1,
      "trentino-südtirol.it" : 1,
      "ma.us" : 1,
      "naustdal.no" : 1,
      "tec.br" : 1,
      "beardu.no" : 1,
      "agro.bo" : 1,
      "engineer.aero" : 1,
      "gouv.ml" : 1,
      "arte.bo" : 1,
      "com.cn" : 1,
      "curitiba.br" : 1,
      "lu.it" : 1,
      "games.hu" : 1,
      "com.gr" : 1,
      "sa.cr" : 1,
      "lowicz.pl" : 1,
      "nom.ro" : 1,
      "lombardia.it" : 1,
      "siena.it" : 1,
      "aju.br" : 1,
      "sandoy.no" : 1,
      "karlsoy.no" : 1,
      "net.lv" : 1,
      "pe.ca" : 1,
      "championship.aero" : 1,
      "fukui.jp" : 1,
      "turek.pl" : 1,
      "org.lk" : 1,
      "go.ci" : 1,
      "ed.ao" : 1,
      "gratangen.no" : 1,
      "5g.in" : 1,
      "tm.mc" : 1,
      "co.az" : 1,
      "net.dm" : 1,
      "org.la" : 1,
      "mil.st" : 1,
      "coop.rw" : 1,
      "ac.mu" : 1,
      "unjárga.no" : 1,
      "co.io" : 1,
      "vadso.no" : 1,
      "sc.ug" : 1,
      "nu.it" : 1,
      "ac.ma" : 1,
      "barum.no" : 1,
      "priv.no" : 1,
      "広島.jp" : 1,
      "lindas.no" : 1,
      "net.kz" : 1,
      "mil.al" : 1,
      "exchange.aero" : 1,
      "froya.no" : 1,
      "trentin-sud-tirol.it" : 1,
      "mil.vc" : 1,
      "nedre-eiker.no" : 1,
      "ga.us" : 1,
      "luster.no" : 1,
      "dienbien.vn" : 1,
      "net.tt" : 1,
      "macerata.it" : 1,
      "not.br" : 1,
      "gran.no" : 1,
      "net.mx" : 1,
      "bolzano.it" : 1,
      "løten.no" : 1,
      "laspezia.it" : 1,
      "gulen.no" : 1,
      "rl.no" : 1,
      "friulivegiulia.it" : 1,
      "org.bh" : 1,
      "net.bj" : 1,
      "gov.au" : 1,
      "co.ci" : 1,
      "com.pl" : 1,
      "res.aero" : 1,
      "gov.gy" : 1,
      "stat.no" : 1,
      "大阪.jp" : 1,
      "lierne.no" : 1,
      "org.fm" : 1,
      "vanylven.no" : 1,
      "hobøl.no" : 1,
      "østre-toten.no" : 1,
      "com.sv" : 1,
      "net.tr" : 1,
      "drammen.no" : 1,
      "ai.in" : 1,
      "me.ke" : 1,
      "ote.bj" : 1,
      "randaberg.no" : 1,
      "riik.ee" : 1,
      "finnøy.no" : 1,
      "org.ls" : 1,
      "or.tz" : 1,
      "gáivuotna.no" : 1,
      "co.bw" : 1,
      "edu.pe" : 1,
      "rindal.no" : 1,
      "barueri.br" : 1,
      "sk.ca" : 1,
      "forlicesena.it" : 1,
      "org.so" : 1,
      "pu.it" : 1,
      "ca.us" : 1,
      "kragero.no" : 1,
      "jdf.br" : 1,
      "bolt.hu" : 1,
      "edu.ec" : 1,
      "hemne.no" : 1,
      "org.bb" : 1,
      "kommunalforbund.se" : 1,
      "government.aero" : 1,
      "mil.jo" : 1,
      "net.mt" : 1,
      "vaapste.no" : 1,
      "fm.br" : 1,
      "øygarden.no" : 1,
      "birkenes.no" : 1,
      "fj.cn" : 1,
      "co.mg" : 1,
      "gov.vn" : 1,
      "co.ve" : 1,
      "muosát.no" : 1,
      "ruovat.no" : 1,
      "monzaedellabrianza.it" : 1,
      "råde.no" : 1,
      "tempio-olbia.it" : 1,
      "trentinsudtirol.it" : 1,
      "author.aero" : 1,
      "com.gt" : 1,
      "edu.ar" : 1,
      "giehtavuoatna.no" : 1,
      "gov.ua" : 1,
      "mil.hn" : 1,
      "pomorze.pl" : 1,
      "org.sh" : 1,
      "青森.jp" : 1,
      "torsken.no" : 1,
      "aquila.it" : 1,
      "gov.uk" : 1,
      "trani-andria-barletta.it" : 1,
      "me.uk" : 1,
      "nome.cv" : 1,
      "marketplace.aero" : 1,
      "info.bj" : 1,
      "org.bw" : 1,
      "ns.ca" : 1,
      "sch.uk" : 1,
      "滋賀.jp" : 1,
      "tromsø.no" : 1,
      "uri.arpa" : 1,
      "asso.fr" : 1,
      "net.ye" : 1,
      "tecnologia.bo" : 1,
      "eco.bj" : 1,
      "news.hu" : 1,
      "net.pn" : 1,
      "co.za" : 1,
      "int.cv" : 1,
      "guovdageaidnu.no" : 1,
      "emiliaromagna.it" : 1,
      "mil.tj" : 1,
      "org.sb" : 1,
      "lugansk.ua" : 1,
      "com.qa" : 1,
      "prd.mg" : 1,
      "org.sc" : 1,
      "friuliveneziagiulia.it" : 1,
      "3.bg" : 1,
      "mil.br" : 1,
      "org.bo" : 1,
      "aaa.pro" : 1,
      "av.tr" : 1,
      "tos.it" : 1,
      "dgca.aero" : 1,
      "valdaosta.it" : 1,
      "or.ci" : 1,
      "net.iq" : 1,
      "net.kg" : 1,
      "com.vn" : 1,
      "stange.no" : 1,
      "mil.co" : 1,
      "lombardy.it" : 1,
      "bamble.no" : 1,
      "tana.no" : 1,
      "academia.bo" : 1,
      "med.ht" : 1,
      "org.gn" : 1,
      "potenza.it" : 1,
      "per.la" : 1,
      "čáhcesuolo.no" : 1,
      "co.je" : 1,
      "lazio.it" : 1,
      "org.ml" : 1,
      "co.tz" : 1,
      "certification.aero" : 1,
      "sebastopol.ua" : 1,
      "cesenaforli.it" : 1,
      "d.se" : 1,
      "rn.it" : 1,
      "arezzo.it" : 1,
      "cal.it" : 1,
      "cooperativa.bo" : 1,
      "name.et" : 1,
      "kaszuby.pl" : 1,
      "mn.us" : 1,
      "it.ao" : 1,
      "lebork.pl" : 1,
      "rep.br" : 1,
      "wodzislaw.pl" : 1,
      "net.kn" : 1,
      "tj.cn" : 1,
      "gjerdrum.no" : 1,
      "gov.pl" : 1,
      "com.au" : 1,
      "bievát.no" : 1,
      "com.gy" : 1,
      "hareid.no" : 1,
      "org.gg" : 1,
      "brindisi.it" : 1,
      "al.us" : 1,
      "genova.it" : 1,
      "mo-i-rana.no" : 1,
      "toyama.jp" : 1,
      "carrara-massa.it" : 1,
      "fin.tn" : 1,
      "åmot.no" : 1,
      "net.my" : 1,
      "bj.cn" : 1,
      "massacarrara.it" : 1,
      "psi.br" : 1,
      "gov.qa" : 1,
      "edu.tw" : 1,
      "aostavalley.it" : 1,
      "gliding.aero" : 1,
      "dyrøy.no" : 1,
      "ráhkkerávju.no" : 1,
      "tm.se" : 1,
      "gorlice.pl" : 1,
      "ms.kr" : 1,
      "hornindal.no" : 1,
      "perso.ht" : 1,
      "monza.it" : 1,
      "grajewo.pl" : 1,
      "rømskog.no" : 1,
      "0.bg" : 1,
      "namsskogan.no" : 1,
      "sauda.no" : 1,
      "sr.it" : 1,
      "org.za" : 1,
      "ferrara.it" : 1,
      "sch.qa" : 1,
      "net.ac" : 1,
      "rennesøy.no" : 1,
      "tn.us" : 1,
      "thaibinh.vn" : 1,
      "søndre-land.no" : 1,
      "com.ua" : 1,
      "bu.no" : 1,
      "ac.tj" : 1,
      "osoyro.no" : 1,
      "c.se" : 1,
      "int.ve" : 1,
      "rec.ro" : 1,
      "tjeldsund.no" : 1,
      "pro.pr" : 1,
      "org.do" : 1,
      "fortal.br" : 1,
      "edu.to" : 1,
      "univ.sn" : 1,
      "nom.fr" : 1,
      "mt.us" : 1,
      "jondal.no" : 1,
      "org.gi" : 1,
      "nom.br" : 1,
      "mn.it" : 1,
      "com.ss" : 1,
      "ve.it" : 1,
      "mil.uy" : 1,
      "com.gp" : 1,
      "aa.no" : 1,
      "charter.aero" : 1,
      "mjondalen.no" : 1,
      "org.lv" : 1,
      "alt.za" : 1,
      "com.vi" : 1,
      "ggf.br" : 1,
      "北海道.jp" : 1,
      "opole.pl" : 1,
      "biz.id" : 1,
      "trøgstad.no" : 1,
      "trentinoa-adige.it" : 1,
      "sch.ae" : 1,
      "sørreisa.no" : 1,
      "sc.ke" : 1,
      "web.pk" : 1,
      "erotika.hu" : 1,
      "新潟.jp" : 1,
      "co.mw" : 1,
      "rome.it" : 1,
      "binhduong.vn" : 1,
      "health.nz" : 1,
      "net.am" : 1,
      "plc.uk" : 1,
      "ac.at" : 1,
      "aichi.jp" : 1,
      "vs.it" : 1,
      "floro.no" : 1,
      "ac.rs" : 1,
      "skiervá.no" : 1,
      "kiengiang.vn" : 1,
      "mobi.tz" : 1,
      "insurance.aero" : 1,
      "asso.km" : 1,
      "cn.ua" : 1,
      "software.aero" : 1,
      "net.cy" : 1,
      "sogndal.no" : 1,
      "ponpes.id" : 1,
      "gov.ae" : 1,
      "pvt.ge" : 1,
      "wiki.br" : 1,
      "snåase.no" : 1,
      "oita.jp" : 1,
      "biz.ni" : 1,
      "hurum.no" : 1,
      "gen.nz" : 1,
      "sorfold.no" : 1,
      "goiania.br" : 1,
      "com.eg" : 1,
      "bacninh.vn" : 1,
      "alstahaug.no" : 1,
      "badaddja.no" : 1,
      "net.ki" : 1,
      "pl.ua" : 1,
      "coop.py" : 1,
      "bn.it" : 1,
      "wi.us" : 1,
      "gov.tn" : 1,
      "mil.in" : 1,
      "org.mx" : 1,
      "chungnam.kr" : 1,
      "vaksdal.no" : 1,
      "business.in" : 1,
      "gov.pr" : 1,
      "perso.sn" : 1,
      "网络.hk" : 1,
      "bytom.pl" : 1,
      "org.tt" : 1,
      "stranda.no" : 1,
      "tn.it" : 1,
      "敎育.hk" : 1,
      "carboniaiglesias.it" : 1,
      "matta-varjjat.no" : 1,
      "nesna.no" : 1,
      "elk.pl" : 1,
      "snasa.no" : 1,
      "the.br" : 1,
      "salud.bo" : 1,
      "aeroclub.aero" : 1,
      "mt.it" : 1,
      "museum.om" : 1,
      "balsfjord.no" : 1,
      "org.kz" : 1,
      "iveland.no" : 1,
      "com.bs" : 1,
      "edu.pa" : 1,
      "evenášši.no" : 1,
      "szczecin.pl" : 1,
      "wv.us" : 1,
      "porsanger.no" : 1,
      "lukow.pl" : 1,
      "re.kr" : 1,
      "edu.pk" : 1,
      "lanbib.se" : 1,
      "biz.az" : 1,
      "skoczow.pl" : 1,
      "øksnes.no" : 1,
      "lig.it" : 1,
      "xj.cn" : 1,
      "rana.no" : 1,
      "pomorskie.pl" : 1,
      "edu.sy" : 1,
      "berlevåg.no" : 1,
      "krym.ua" : 1,
      "rødøy.no" : 1,
      "stargard.pl" : 1,
      "bt.it" : 1,
      "al.it" : 1,
      "skydiving.aero" : 1,
      "palmas.br" : 1,
      "net.lk" : 1,
      "com.lc" : 1,
      "dnipropetrovsk.ua" : 1,
      "com.lb" : 1,
      "or.th" : 1,
      "assur.bj" : 1,
      "edu.tm" : 1,
      "dn.ua" : 1,
      "hatinh.vn" : 1,
      "org.dm" : 1,
      "net.la" : 1,
      "fylkesbibl.no" : 1,
      "stryn.no" : 1,
      "sf.no" : 1,
      "boleslawiec.pl" : 1,
      "modalen.no" : 1,
      "com.tn" : 1,
      "org.hk" : 1,
      "is.it" : 1,
      "ørland.no" : 1,
      "africa.bj" : 1,
      "ca.it" : 1,
      "群馬.jp" : 1,
      "sch.ss" : 1,
      "pro.vn" : 1,
      "asnes.no" : 1,
      "macapa.br" : 1,
      "sci.eg" : 1,
      "hanoi.vn" : 1,
      "haugiang.vn" : 1,
      "ringebu.no" : 1,
      "crew.aero" : 1,
      "pisz.pl" : 1,
      "ebiz.tw" : 1,
      "gov.eg" : 1,
      "int.mv" : 1,
      "sex.pl" : 1,
      "info.pl" : 1,
      "bykle.no" : 1,
      "waw.pl" : 1,
      "gjerstad.no" : 1,
      "baria-vungtau.vn" : 1,
      "asso.ci" : 1,
      "parti.se" : 1,
      "fedje.no" : 1,
      "z.bg" : 1,
      "sola.no" : 1,
      "meland.no" : 1,
      "paragliding.aero" : 1,
      "edu.st" : 1,
      "olsztyn.pl" : 1,
      "tozsde.hu" : 1,
      "us.in" : 1,
      "svelvik.no" : 1,
      "gov.mz" : 1,
      "loisirs.bj" : 1,
      "6.bg" : 1,
      "ac.ae" : 1,
      "nu.ca" : 1,
      "vagan.no" : 1,
      "leilao.br" : 1,
      "solund.no" : 1,
      "zaporizhzhe.ua" : 1,
      "forlì-cesena.it" : 1,
      "przeworsk.pl" : 1,
      "ip6.arpa" : 1,
      "stathelle.no" : 1,
      "com.nf" : 1,
      "ing.pa" : 1,
      "báhcavuotna.no" : 1,
      "edu.vc" : 1,
      "med.ee" : 1,
      "åmli.no" : 1,
      "edu.al" : 1,
      "ind.kw" : 1,
      "bindal.no" : 1,
      "w.bg" : 1,
      "gov.ss" : 1,
      "hámmárfeasta.no" : 1,
      "reggiocalabria.it" : 1,
      "org.et" : 1,
      "�司.cn" : 1,
      "kyiv.ua" : 1,
      "mielec.pl" : 1,
      "gov.lb" : 1,
      "tm.dz" : 1,
      "gov.lc" : 1,
      "pe.it" : 1,
      "vao.it" : 1,
      "ac.lk" : 1,
      "valléedaoste.it" : 1,
      "kagoshima.jp" : 1,
      "edu.bt" : 1,
      "tranibarlettaandria.it" : 1,
      "org.hu" : 1,
      "fhv.se" : 1,
      "store.ro" : 1,
      "co.rw" : 1,
      "prato.it" : 1,
      "trieste.it" : 1,
      "tokyo.jp" : 1,
      "thanhhoa.vn" : 1,
      "hasvik.no" : 1,
      "iglesias-carbonia.it" : 1,
      "maintenance.aero" : 1,
      "health.vn" : 1,
      "co.uz" : 1,
      "fm.jo" : 1,
      "catania.it" : 1,
      "huissier-justice.fr" : 1,
      "trentinosüdtirol.it" : 1,
      "gov.bs" : 1,
      "søgne.no" : 1,
      "org.bm" : 1,
      "co.th" : 1,
      "name.hr" : 1,
      "nom.co" : 1,
      "sumy.ua" : 1,
      "edu.ws" : 1,
      "山梨.jp" : 1,
      "kr.it" : 1,
      "skodje.no" : 1,
      "sc.kr" : 1,
      "fhsk.se" : 1,
      "troandin.no" : 1,
      "info.ki" : 1,
      "gov.bf" : 1,
      "net.fj" : 1,
      "com.pr" : 1,
      "he.cn" : 1,
      "kep.tr" : 1,
      "pp.az" : 1,
      "trentinosuedtirol.it" : 1,
      "toscana.it" : 1,
      "valleedaoste.it" : 1,
      "ven.it" : 1,
      "teramo.it" : 1,
      "org.ee" : 1,
      "biz.tr" : 1,
      "trentin-sudtirol.it" : 1,
      "valle-d-aosta.it" : 1,
      "rzeszow.pl" : 1,
      "agr.br" : 1,
      "gov.gu" : 1,
      "info.la" : 1,
      "midtre-gauldal.no" : 1,
      "gov.cm" : 1,
      "a.se" : 1,
      "org.ph" : 1,
      "gov.in" : 1,
      "co.mu" : 1,
      "nm.us" : 1,
      "com.fr" : 1,
      "lier.no" : 1,
      "k12.ec" : 1,
      "unjarga.no" : 1,
      "co.ma" : 1,
      "net.sg" : 1,
      "n.se" : 1,
      "rybnik.pl" : 1,
      "freight.aero" : 1,
      "nic.za" : 1,
      "mil.ae" : 1,
      "krødsherad.no" : 1,
      "hole.no" : 1,
      "kristiansand.no" : 1,
      "ibaraki.jp" : 1,
      "bjerkreim.no" : 1,
      "oppegård.no" : 1,
      "quangbinh.vn" : 1,
      "osen.no" : 1,
      "edu.gd" : 1,
      "kvam.no" : 1,
      "karasjok.no" : 1,
      "grp.lk" : 1,
      "kråanghke.no" : 1,
      "czest.pl" : 1,
      "molde.no" : 1,
      "net.bn" : 1,
      "uzhhorod.ua" : 1,
      "accident-prevention.aero" : 1,
      "prof.pr" : 1,
      "yk.ca" : 1,
      "nt.ca" : 1,
      "gov.mw" : 1,
      "fot.br" : 1,
      "l.se" : 1,
      "in.us" : 1,
      "karmoy.no" : 1,
      "edu.gr" : 1,
      "edu.cn" : 1,
      "edu.zm" : 1,
      "pistoia.it" : 1,
      "tksat.bo" : 1,
      "art.br" : 1,
      "masoy.no" : 1,
      "gv.at" : 1,
      "pe.kr" : 1,
      "cb.it" : 1,
      "agric.za" : 1,
      "com.sl" : 1,
      "bologna.it" : 1,
      "klepp.no" : 1,
      "bozen-sudtirol.it" : 1,
      "gov.mo" : 1,
      "info.in" : 1,
      "cri.nz" : 1,
      "nordreisa.no" : 1,
      "ac.gn" : 1,
      "bulsan.it" : 1,
      "kherson.ua" : 1,
      "gálsá.no" : 1,
      "edu.kp" : 1,
      "佐賀.jp" : 1,
      "trentino-suedtirol.it" : 1,
      "tromsa.no" : 1,
      "ráisa.no" : 1,
      "hs.kr" : 1,
      "noticias.bo" : 1,
      "lezajsk.pl" : 1,
      "net.ma" : 1,
      "mil.kr" : 1,
      "rar.ve" : 1,
      "la.us" : 1,
      "vågå.no" : 1,
      "net.mk" : 1,
      "mil.eg" : 1,
      "journalist.aero" : 1,
      "flå.no" : 1,
      "net.ng" : 1,
      "org.lr" : 1,
      "powiat.pl" : 1,
      "malopolska.pl" : 1,
      "jølster.no" : 1,
      "com.in" : 1,
      "wegrow.pl" : 1,
      "lecce.it" : 1,
      "net.je" : 1,
      "org.vu" : 1,
      "nowaruda.pl" : 1,
      "tm.km" : 1,
      "org.ug" : 1,
      "jus.br" : 1,
      "podlasie.pl" : 1,
      "org.kw" : 1,
      "com.cm" : 1,
      "sc.cn" : 1,
      "or.mu" : 1,
      "vt.us" : 1,
      "com.uy" : 1,
      "batsfjord.no" : 1,
      "báidár.no" : 1,
      "gob.ar" : 1,
      "sigdal.no" : 1,
      "malatvuopmi.no" : 1,
      "org.ag" : 1,
      "gildeskal.no" : 1,
      "com.gu" : 1,
      "hobol.no" : 1,
      "wolomin.pl" : 1,
      "ac.cy" : 1,
      "grimstad.no" : 1,
      "rel.ht" : 1,
      "verran.no" : 1,
      "kv.ua" : 1,
      "vegarshei.no" : 1,
      "gjesdal.no" : 1,
      "presse.km" : 1,
      "vinhlong.vn" : 1,
      "mil.mz" : 1,
      "mus.br" : 1,
      "alaheadju.no" : 1,
      "balsan-sudtirol.it" : 1,
      "gob.pe" : 1,
      "info.bb" : 1,
      "cim.br" : 1,
      "tysvar.no" : 1,
      "sx.cn" : 1,
      "gov.sl" : 1,
      "sauherad.no" : 1,
      "namdinh.vn" : 1,
      "sassari.it" : 1,
      "com.mo" : 1,
      "ms.us" : 1,
      "biz.my" : 1,
      "edu.af" : 1,
      "forsand.no" : 1,
      "pro.br" : 1,
      "gob.ec" : 1,
      "info.zm" : 1,
      "skanit.no" : 1,
      "net.mu" : 1,
      "nom.pl" : 1,
      "realestate.pl" : 1,
      "go.tj" : 1,
      "drobak.no" : 1,
      "sykkylven.no" : 1,
      "tranoy.no" : 1,
      "sondrio.it" : 1,
      "tychy.pl" : 1,
      "laquila.it" : 1,
      "andria-trani-barletta.it" : 1,
      "com.mw" : 1,
      "coop.in" : 1,
      "sor-aurdal.no" : 1,
      "net.il" : 1,
      "dongnai.vn" : 1,
      "me.us" : 1,
      "vestvågøy.no" : 1,
      "medecin.km" : 1,
      "org.mv" : 1,
      "zgora.pl" : 1,
      "omasvuotna.no" : 1,
      "re.it" : 1,
      "mar.it" : 1,
      "svalbard.no" : 1,
      "molise.it" : 1,
      "bronnoysund.no" : 1,
      "al.no" : 1,
      "senasa.ar" : 1,
      "info.nr" : 1,
      "tochigi.jp" : 1,
      "binhdinh.vn" : 1,
      "kharkiv.ua" : 1,
      "int.bo" : 1,
      "or.id" : 1,
      "com.cw" : 1,
      "org.km" : 1,
      "net.ir" : 1,
      "vt.it" : 1,
      "bieszczady.pl" : 1,
      "city.hu" : 1,
      "geek.nz" : 1,
      "taxi.br" : 1,
      "caserta.it" : 1,
      "info.ve" : 1,
      "org.cv" : 1,
      "net.id" : 1,
      "fuossko.no" : 1,
      "ulsan.kr" : 1,
      "org.ai" : 1,
      "te.it" : 1,
      "gov.sx" : 1,
      "sarpsborg.no" : 1,
      "sch.jo" : 1,
      "edu.py" : 1,
      "co.gg" : 1,
      "co.am" : 1,
      "kvalsund.no" : 1,
      "venice.it" : 1,
      "vardo.no" : 1,
      "biz.ki" : 1,
      "sør-varanger.no" : 1,
      "edu.sa" : 1,
      "gov.jo" : 1,
      "org.dz" : 1,
      "ge.it" : 1,
      "com.co" : 1,
      "māori.nz" : 1,
      "net.ni" : 1,
      "gov.sd" : 1,
      "net.ly" : 1,
      "nl.no" : 1,
      "com.nr" : 1,
      "rost.no" : 1,
      "org.ht" : 1,
      "natural.bo" : 1,
      "biz.cy" : 1,
      "lavangen.no" : 1,
      "個人.hk" : 1,
      "reggio-calabria.it" : 1,
      "gov.ge" : 1,
      "dni.us" : 1,
      "ts.it" : 1,
      "net.cu" : 1,
      "rissa.no" : 1,
      "ltd.cy" : 1,
      "bs.it" : 1,
      "surnadal.no" : 1,
      "journal.aero" : 1,
      "kafjord.no" : 1,
      "org.nz" : 1,
      "ketrzyn.pl" : 1,
      "pordenone.it" : 1,
      "venezia.it" : 1,
      "mazury.pl" : 1,
      "ac.nz" : 1,
      "ullensaker.no" : 1,
      "parliament.nz" : 1,
      "ac.ni" : 1,
      "na.it" : 1,
      "co.zw" : 1,
      "hoabinh.vn" : 1,
      "media.pl" : 1,
      "nesoddtangen.no" : 1,
      "me.it" : 1,
      "com.pf" : 1,
      "ac.ls" : 1,
      "gov.br" : 1,
      "konin.pl" : 1,
      "leg.br" : 1,
      "gu.us" : 1,
      "leka.no" : 1,
      "ltd.lk" : 1,
      "szczytno.pl" : 1,
      "name.tt" : 1,
      "návuotna.no" : 1,
      "vgs.no" : 1,
      "dlugoleka.pl" : 1,
      "nm.cn" : 1,
      "principe.st" : 1,
      "novara.it" : 1,
      "gov.tj" : 1,
      "net.uz" : 1,
      "com.ps" : 1,
      "trader.aero" : 1,
      "edu.ba" : 1,
      "co.cr" : 1,
      "chernigov.ua" : 1,
      "info.ec" : 1,
      "lahppi.no" : 1,
      "net.az" : 1,
      "ms.it" : 1,
      "lyngen.no" : 1,
      "cl.it" : 1,
      "eigersund.no" : 1,
      "co.tt" : 1,
      "honefoss.no" : 1,
      "ishikawa.jp" : 1,
      "co.id" : 1,
      "wy.us" : 1,
      "com.ge" : 1,
      "fet.no" : 1,
      "roan.no" : 1,
      "org.ve" : 1,
      "gov.nr" : 1,
      "mragowo.pl" : 1,
      "edu.mg" : 1,
      "årdal.no" : 1,
      "barletta-trani-andria.it" : 1,
      "fe.it" : 1,
      "com.sd" : 1,
      "ringsaker.no" : 1,
      "pro.in" : 1,
      "pn.it" : 1,
      "isa.us" : 1,
      "mosjoen.no" : 1,
      "pisa.it" : 1,
      "gov.co" : 1,
      "stalowa-wola.pl" : 1,
      "cng.br" : 1,
      "net.om" : 1,
      "binhphuoc.vn" : 1,
      "publ.pt" : 1,
      "asso.dz" : 1,
      "habmer.no" : 1,
      "press.aero" : 1,
      "haram.no" : 1,
      "com.jo" : 1,
      "flora.no" : 1,
      "co.vi" : 1,
      "rovno.ua" : 1,
      "hn.cn" : 1,
      "edu.gl" : 1,
      "donna.no" : 1,
      "asso.mc" : 1,
      "mil.pl" : 1,
      "org.bz" : 1,
      "mb.ca" : 1,
      "gob.pa" : 1,
      "namdalseid.no" : 1,
      "udine.it" : 1,
      "láhppi.no" : 1,
      "bodo.no" : 1,
      "info.hu" : 1,
      "nore-og-uvdal.no" : 1,
      "edu.mn" : 1,
      "gob.pk" : 1,
      "sømna.no" : 1,
      "sampa.br" : 1,
      "ne.ug" : 1,
      "game.tw" : 1,
      "act.au" : 1,
      "govt.nz" : 1,
      "ot.it" : 1,
      "pt.it" : 1,
      "adm.br" : 1,
      "res.in" : 1,
      "edu.ky" : 1,
      "phuyen.vn" : 1,
      "vinnytsia.ua" : 1,
      "gov.ps" : 1,
      "hjartdal.no" : 1,
      "asso.ml" : 1,
      "com.tj" : 1,
      "sh.cn" : 1,
      "mil.qa" : 1,
      "info.sd" : 1,
      "edu.ci" : 1,
      "conf.au" : 1,
      "jogasz.hu" : 1,
      "algard.no" : 1,
      "campidano-medio.it" : 1,
      "lt.ua" : 1,
      "balat.no" : 1,
      "com.br" : 1,
      "rakkestad.no" : 1,
      "biz.fj" : 1,
      "co.in" : 1,
      "sandnes.no" : 1,
      "gouv.ht" : 1,
      "org.ms" : 1,
      "skjervoy.no" : 1,
      "store.bb" : 1,
      "js.cn" : 1,
      "phutho.vn" : 1,
      "vinnica.ua" : 1,
      "edu.io" : 1,
      "mil.gt" : 1,
      "奈良.jp" : 1,
      "kvitsøy.no" : 1,
      "sport.eg" : 1,
      "edu.pt" : 1,
      "net.me" : 1,
      "sc.us" : 1,
      "caltanissetta.it" : 1,
      "or.cr" : 1,
      "gs.cn" : 1,
      "com.hn" : 1,
      "tm.fr" : 1,
      "bardu.no" : 1,
      "deporte.bo" : 1,
      "tv.bb" : 1,
      "org.sz" : 1,
      "tv.sd" : 1,
      "bearalvahki.no" : 1,
      "org.eg" : 1,
      "tønsberg.no" : 1,
      "ar.it" : 1,
      "f.bg" : 1,
      "gov.hk" : 1,
      "mo.cn" : 1,
      "info.fj" : 1,
      "net.im" : 1,
      "mobi.ng" : 1,
      "wlocl.pl" : 1,
      "大分.jp" : 1,
      "store.ve" : 1,
      "kazimierz-dolny.pl" : 1,
      "net.zm" : 1,
      "bievat.no" : 1,
      "isernia.it" : 1,
      "art.ml" : 1,
      "vercelli.it" : 1,
      "gov.ie" : 1,
      "risor.no" : 1,
      "asso.gp" : 1,
      "flakstad.no" : 1,
      "net.cn" : 1,
      "stjørdal.no" : 1,
      "org.mz" : 1,
      "restaurant.bj" : 1,
      "net.gr" : 1,
      "ovre-eiker.no" : 1,
      "nic.tj" : 1,
      "salvador.br" : 1,
      "org.gp" : 1,
      "hápmir.no" : 1,
      "taa.it" : 1,
      "enebakk.no" : 1,
      "lenvik.no" : 1,
      "org.ss" : 1,
      "com.gi" : 1,
      "cargo.aero" : 1,
      "soc.dz" : 1,
      "org.vi" : 1,
      "trentinos-tirol.it" : 1,
      "fh.se" : 1,
      "com.lv" : 1,
      "slattum.no" : 1,
      "edu.bn" : 1,
      "video.hu" : 1,
      "mi.us" : 1,
      "ks.ua" : 1,
      "org.lc" : 1,
      "zlg.br" : 1,
      "org.lb" : 1,
      "balsan-suedtirol.it" : 1,
      "ntr.br" : 1,
      "hammerfest.no" : 1,
      "etnedal.no" : 1,
      "ri.it" : 1,
      "intl.tn" : 1,
      "com.dm" : 1,
      "gov.et" : 1,
      "iglesiascarbonia.it" : 1,
      "niigata.jp" : 1,
      "sic.it" : 1,
      "museum.no" : 1,
      "imb.br" : 1,
      "lib.ee" : 1,
      "contagem.br" : 1,
      "flight.aero" : 1,
      "divttasvuotna.no" : 1,
      "edu.sg" : 1,
      "bielawa.pl" : 1,
      "az.us" : 1,
      "org.bs" : 1,
      "nom.ve" : 1,
      "mil.mv" : 1,
      "øvre-eiker.no" : 1,
      "com.kz" : 1,
      "musica.bo" : 1,
      "vallée-d-aoste.it" : 1,
      "alessandria.it" : 1,
      "okayama.jp" : 1,
      "trentinosud-tirol.it" : 1,
      "sicily.it" : 1,
      "trondheim.no" : 1,
      "edu.es" : 1,
      "cs.in" : 1,
      "edu.sn" : 1,
      "in.rs" : 1,
      "info.ht" : 1,
      "nordre-land.no" : 1,
      "stjordalshalsen.no" : 1,
      "aq.it" : 1,
      "com.tt" : 1,
      "fla.no" : 1,
      "lillehammer.no" : 1,
      "com.mx" : 1,
      "idv.hk" : 1,
      "nsw.au" : 1,
      "gov.bm" : 1,
      "k12.tr" : 1,
      "name.vn" : 1,
      "ac.zm" : 1,
      "art.do" : 1,
      "florø.no" : 1,
      "alesund.no" : 1,
      "ballooning.aero" : 1,
      "verona.it" : 1,
      "gov.lv" : 1,
      "sortland.no" : 1,
      "go.it" : 1,
      "co.me" : 1,
      "forum.hu" : 1,
      "architectes.bj" : 1,
      "t.bg" : 1,
      "edu.rs" : 1,
      "網絡.cn" : 1,
      "info.eg" : 1,
      "friuli-vegiulia.it" : 1,
      "gov.gi" : 1,
      "co.im" : 1,
      "båtsfjord.no" : 1,
      "saotome.st" : 1,
      "pd.it" : 1,
      "id.ir" : 1,
      "y.se" : 1,
      "5.bg" : 1,
      "bozen.it" : 1,
      "cci.fr" : 1,
      "j.bg" : 1,
      "int.mw" : 1,
      "prochowice.pl" : 1,
      "gjovik.no" : 1,
      "moss.no" : 1,
      "net.af" : 1,
      "org.ae" : 1,
      "tarnobrzeg.pl" : 1,
      "宮崎.jp" : 1,
      "nhs.uk" : 1,
      "sor-odal.no" : 1,
      "edu.gh" : 1,
      "olbia-tempio.it" : 1,
      "to.it" : 1,
      "com.hk" : 1,
      "rel.pl" : 1,
      "2.bg" : 1,
      "riobranco.br" : 1,
      "mil.ph" : 1,
      "g.bg" : 1,
      "org.tn" : 1,
      "presse.ml" : 1,
      "ind.in" : 1,
      "belem.br" : 1,
      "lindesnes.no" : 1,
      "tolga.no" : 1,
      "odesa.ua" : 1,
      "qh.cn" : 1,
      "recht.pro" : 1,
      "narviika.no" : 1,
      "m.bg" : 1,
      "avocats.bj" : 1,
      "leirfjord.no" : 1,
      "com.bm" : 1,
      "org.pr" : 1,
      "chernovtsy.ua" : 1,
      "bo.it" : 1,
      "idrett.no" : 1,
      "gov.tt" : 1,
      "suedtirol.it" : 1,
      "log.br" : 1,
      "int.in" : 1,
      "coz.br" : 1,
      "神奈川.jp" : 1,
      "福岡.jp" : 1,
      "coop.ar" : 1,
      "ac.pk" : 1,
      "ok.us" : 1,
      "bjarkøy.no" : 1,
      "åsnes.no" : 1,
      "kongsvinger.no" : 1,
      "gov.kz" : 1,
      "nom.km" : 1,
      "tur.ar" : 1,
      "edu.ng" : 1,
      "g12.br" : 1,
      "gov.mr" : 1,
      "mantova.it" : 1,
      "oppdal.no" : 1,
      "airline.aero" : 1,
      "maringa.br" : 1,
      "veterinaire.km" : 1,
      "muosat.no" : 1,
      "coop.ht" : 1,
      "gov.dm" : 1,
      "ulvik.no" : 1,
      "langson.vn" : 1,
      "com.et" : 1,
      "ed.cr" : 1,
      "mo.it" : 1,
      "horten.no" : 1,
      "davvenjárga.no" : 1,
      "gov.cl" : 1,
      "eidfjord.no" : 1,
      "nesseby.no" : 1,
      "edu.mk" : 1,
      "valledaosta.it" : 1,
      "b.bg" : 1,
      "egersund.no" : 1,
      "up.in" : 1,
      "net.ba" : 1,
      "morena.br" : 1,
      "vinhphuc.vn" : 1,
      "u.se" : 1,
      "parma.it" : 1,
      "gov.gn" : 1,
      "ambulance.aero" : 1,
      "uk.in" : 1,
      "sjc.br" : 1,
      "røyrvik.no" : 1,
      "bet.br" : 1,
      "edu.az" : 1,
      "lebesby.no" : 1,
      "oslo.no" : 1,
      "sanok.pl" : 1,
      "com.bb" : 1,
      "mod.gi" : 1,
      "ac.fj" : 1,
      "meldal.no" : 1,
      "anani.br" : 1,
      "klæbu.no" : 1,
      "lucania.it" : 1,
      "hemnes.no" : 1,
      "ac.kr" : 1,
      "gov.cd" : 1,
      "gáŋgaviika.no" : 1,
      "mil.ve" : 1,
      "ato.br" : 1,
      "web.ni" : 1,
      "gov.ml" : 1,
      "com.so" : 1,
      "med.sd" : 1,
      "divtasvuodna.no" : 1,
      "bærum.no" : 1,
      "campidanomedio.it" : 1,
      "fi.it" : 1,
      "web.id" : 1,
      "org.sv" : 1,
      "hábmer.no" : 1,
      "åkrehamn.no" : 1,
      "gov.cx" : 1,
      "bhz.br" : 1,
      "org.pl" : 1,
      "hotel.tz" : 1,
      "biz.pk" : 1,
      "ch.it" : 1,
      "com.bh" : 1,
      "info.az" : 1,
      "snåsa.no" : 1,
      "repbody.aero" : 1,
      "balsan-südtirol.it" : 1,
      "hammarfeasta.no" : 1,
      "com.fm" : 1,
      "steinkjer.no" : 1,
      "liguria.it" : 1,
      "hol.no" : 1,
      "nærøy.no" : 1,
      "nom.ag" : 1,
      "nsn.us" : 1,
      "vestre-toten.no" : 1,
      "ngo.za" : 1,
      "edu.ly" : 1,
      "hk.cn" : 1,
      "com.bo" : 1,
      "edu.cu" : 1,
      "warszawa.pl" : 1,
      "id.au" : 1,
      "air-surveillance.aero" : 1,
      "audnedaln.no" : 1,
      "roros.no" : 1,
      "io.vn" : 1,
      "ro.it" : 1,
      "álaheadju.no" : 1,
      "biella.it" : 1,
      "med.br" : 1,
      "alvdal.no" : 1,
      "net.sa" : 1,
      "id.us" : 1,
      "x.bg" : 1,
      "torino.it" : 1,
      "mo.us" : 1,
      "bedzin.pl" : 1,
      "info.mv" : 1,
      "edu.ni" : 1,
      "forli-cesena.it" : 1,
      "org.qa" : 1,
      "com.sb" : 1,
      "hoylandet.no" : 1,
      "com.sc" : 1,
      "livorno.it" : 1,
      "trento.it" : 1,
      "vega.no" : 1,
      "bentre.vn" : 1,
      "friuli-veneziagiulia.it" : 1,
      "com.sh" : 1,
      "hanam.vn" : 1,
      "embaixada.st" : 1,
      "fjell.no" : 1,
      "r.bg" : 1,
      "skedsmo.no" : 1,
      "conference.aero" : 1,
      "tranøy.no" : 1,
      "hanggliding.aero" : 1,
      "inderøy.no" : 1,
      "bømlo.no" : 1,
      "nuoro.it" : 1,
      "net.py" : 1,
      "pub.sa" : 1,
      "olecko.pl" : 1,
      "gov.za" : 1,
      "langevag.no" : 1,
      "londrina.br" : 1,
      "tydal.no" : 1,
      "id.ly" : 1,
      "i.ph" : 1,
      "石川.jp" : 1,
      "name.mk" : 1,
      "grosseto.it" : 1,
      "bari.it" : 1,
      "sex.hu" : 1,
      "kumamoto.jp" : 1,
      "pro.tt" : 1,
      "nic.in" : 1,
      "gov.do" : 1,
      "org.gt" : 1,
      "int.tj" : 1,
      "e.se" : 1,
      "arts.ve" : 1,
      "edu.me" : 1,
      "thuathienhue.vn" : 1,
      "voagat.no" : 1,
      "gov.bh" : 1,
      "aseral.no" : 1,
      "網络.hk" : 1,
      "tm.za" : 1,
      "org.gy" : 1,
      "wakayama.jp" : 1,
      "org.au" : 1,
      "gov.ls" : 1,
      "sld.do" : 1,
      "bergen.no" : 1,
      "bolivia.bo" : 1,
      "firm.nf" : 1,
      "net.pt" : 1,
      "saobernardo.br" : 1,
      "mil.km" : 1,
      "kvænangen.no" : 1,
      "drøbak.no" : 1,
      "net.io" : 1,
      "ar.us" : 1,
      "cv.ua" : 1,
      "com.ml" : 1,
      "kraanghke.no" : 1,
      "club.tw" : 1,
      "trentino-sued-tirol.it" : 1,
      "dc.us" : 1,
      "pg.it" : 1,
      "kvæfjord.no" : 1,
      "gov.so" : 1,
      "bc.ca" : 1,
      "og.it" : 1,
      "aca.pro" : 1,
      "net.ci" : 1,
      "council.aero" : 1,
      "bolzano-altoadige.it" : 1,
      "aurskog-holand.no" : 1,
      "zj.cn" : 1,
      "taxi.aero" : 1,
      "services.aero" : 1,
      "vik.no" : 1,
      "genoa.it" : 1,
      "trentino-sud-tirol.it" : 1,
      "trentino-altoadige.it" : 1,
      "ac.uk" : 1,
      "edu.bi" : 1,
      "net.ky" : 1,
      "co.st" : 1,
      "me.ss" : 1,
      "gwangju.kr" : 1,
      "gov.bb" : 1,
      "aparecida.br" : 1,
      "org.vn" : 1,
      "cesenaforlì.it" : 1,
      "ah.no" : 1,
      "inf.br" : 1,
      "com.gn" : 1,
      "go.jp" : 1,
      "hiroshima.jp" : 1,
      "empresa.bo" : 1,
      "nikolaev.ua" : 1,
      "tm.ro" : 1,
      "spjelkavik.no" : 1,
      "radoy.no" : 1,
      "lg.ua" : 1,
      "tc.br" : 1,
      "binhthuan.vn" : 1,
      "com.do" : 1,
      "kochi.jp" : 1,
      "trentinostirol.it" : 1,
      "tingvoll.no" : 1,
      "sor-fron.no" : 1,
      "coop.mv" : 1,
      "ac.ml" : 1,
      "bbs.tr" : 1,
      "tv.it" : 1,
      "lom.it" : 1,
      "trentin-süd-tirol.it" : 1,
      "karasjohka.no" : 1,
      "jan-mayen.no" : 1,
      "mil.nz" : 1,
      "moareke.no" : 1,
      "org.uk" : 1,
      "lucca.it" : 1,
      "bi.it" : 1,
      "gov.bw" : 1,
      "friuli-venezia-giulia.it" : 1,
      "net.gl" : 1,
      "mol.it" : 1,
      "klodzko.pl" : 1,
      "org.ua" : 1,
      "fst.br" : 1,
      "pavia.it" : 1,
      "budejju.no" : 1,
      "karmøy.no" : 1,
      "gen.in" : 1,
      "gov.sh" : 1,
      "gausdal.no" : 1,
      "dell-ogliastra.it" : 1,
      "ri.us" : 1,
      "sicilia.it" : 1,
      "gov.sc" : 1,
      "gov.sb" : 1,
      "lerdal.no" : 1,
      "net.na" : 1,
      "gc.ca" : 1,
      "salangen.no" : 1,
      "edu.om" : 1,
      "mi.it" : 1,
      "ne.tz" : 1,
      "ac.vn" : 1,
      "kvanangen.no" : 1,
      "ind.br" : 1,
      "perso.tn" : 1,
      "ac.ke" : 1,
      "khmelnitskiy.ua" : 1,
      "org.nr" : 1,
      "mil.no" : 1,
      "com.ht" : 1,
      "gov.ve" : 1,
      "øystre-slidre.no" : 1,
      "biz.zm" : 1,
      "info.tn" : 1,
      "bálát.no" : 1,
      "warmia.pl" : 1,
      "com.dz" : 1,
      "jessheim.no" : 1,
      "edu.ye" : 1,
      "agrigento.it" : 1,
      "museum.mv" : 1,
      "nx.cn" : 1,
      "cn.in" : 1,
      "konskowola.pl" : 1,
      "vi.us" : 1,
      "name.my" : 1,
      "pvh.br" : 1,
      "hi.us" : 1,
      "lunner.no" : 1,
      "org.co" : 1,
      "guam.gu" : 1,
      "vestnes.no" : 1,
      "in.ni" : 1,
      "mie.jp" : 1,
      "edu.pn" : 1,
      "jab.br" : 1,
      "naturbruksgymn.se" : 1,
      "co.bi" : 1,
      "ostroleka.pl" : 1,
      "jeonnam.kr" : 1,
      "com.ai" : 1,
      "web.tr" : 1,
      "trentin-sued-tirol.it" : 1,
      "co.bz" : 1,
      "gov.bz" : 1,
      "com.cv" : 1,
      "lyngdal.no" : 1,
      "pol.ht" : 1,
      "friuli-vgiulia.it" : 1,
      "sa.au" : 1,
      "evenassi.no" : 1,
      "daklak.vn" : 1,
      "money.bj" : 1,
      "agro.pl" : 1,
      "terni.it" : 1,
      "i.bg" : 1,
      "ایران.ir" : 1,
      "net.ar" : 1,
      "stjørdalshalsen.no" : 1,
      "com.km" : 1,
      "org.cw" : 1,
      "mutual.ar" : 1,
      "pol.dz" : 1,
      "org.ps" : 1,
      "media.aero" : 1,
      "net.ec" : 1,
      "danang.vn" : 1,
      "nh.us" : 1,
      "ac.cn" : 1,
      "edu.mt" : 1,
      "fitjar.no" : 1,
      "or.ug" : 1,
      "crimea.ua" : 1,
      "hokkaido.jp" : 1,
      "nittedal.no" : 1,
      "mykolaiv.ua" : 1,
      "trysil.no" : 1,
      "o.bg" : 1,
      "evje-og-hornnes.no" : 1,
      "net.pe" : 1,
      "magazine.aero" : 1,
      "parachuting.aero" : 1,
      "org.pf" : 1,
      "p.bg" : 1,
      "mincom.tn" : 1,
      "nysa.pl" : 1,
      "政府.hk" : 1,
      "film.hu" : 1,
      "mil.do" : 1,
      "po.it" : 1,
      "edu.tr" : 1,
      "ind.gt" : 1,
      "aland.fi" : 1,
      "cq.cn" : 1,
      "askoy.no" : 1,
      "nissedal.no" : 1,
      "trentino-s-tirol.it" : 1,
      "malvik.no" : 1,
      "ribeirao.br" : 1,
      "emp.br" : 1,
      "fuoisku.no" : 1,
      "daejeon.kr" : 1,
      "akita.jp" : 1,
      "gov.ms" : 1,
      "skiptvet.no" : 1,
      "andoy.no" : 1,
      "mil.za" : 1,
      "lakas.hu" : 1,
      "edu.bj" : 1,
      "saogonca.br" : 1,
      "messina.it" : 1,
      "ardal.no" : 1,
      "sport.hu" : 1,
      "vardø.no" : 1,
      "bialowieza.pl" : 1,
      "ballangen.no" : 1,
      "net.to" : 1,
      "far.br" : 1,
      "gov.km" : 1,
      "holtålen.no" : 1,
      "medicina.bo" : 1,
      "trentinoaltoadige.it" : 1,
      "swinoujscie.pl" : 1,
      "lecco.it" : 1,
      "gd.cn" : 1,
      "gob.es" : 1,
      "larvik.no" : 1,
      "rg.it" : 1,
      "aure.no" : 1,
      "plurinacional.bo" : 1,
      "aero.mv" : 1,
      "v.bg" : 1,
      "com.bz" : 1,
      "愛知.jp" : 1,
      "h.bg" : 1,
      "vestby.no" : 1,
      "hof.no" : 1,
      "edu.ac" : 1,
      "k12.il" : 1,
      "ecn.br" : 1,
      "og.ao" : 1,
      "net.tw" : 1,
      "or.bi" : 1,
      "podhale.pl" : 1,
      "friuli-v-giulia.it" : 1,
      "org.sd" : 1,
      "org.jo" : 1,
      "co.sz" : 1,
      "lviv.ua" : 1,
      "rivne.ua" : 1,
      "gov.dz" : 1,
      "eng.jo" : 1,
      "post.in" : 1,
      "volda.no" : 1,
      "hyllestad.no" : 1,
      "org.ge" : 1,
      "miyazaki.jp" : 1,
      "olbiatempio.it" : 1,
      "zhitomir.ua" : 1,
      "edu.my" : 1,
      "bydgoszcz.pl" : 1,
      "info.ni" : 1,
      "com.ve" : 1,
      "mil.sh" : 1,
      "show.aero" : 1,
      "khmelnytskyi.ua" : 1,
      "edu.kn" : 1,
      "org.hn" : 1,
      "trentinsuedtirol.it" : 1,
      "pro.mv" : 1,
      "co.ug" : 1,
      "maori.nz" : 1,
      "joinville.br" : 1,
      "microlight.aero" : 1,
      "sendai.jp" : 1,
      "net.rw" : 1,
      "int.vn" : 1,
      "meløy.no" : 1,
      "kragerø.no" : 1,
      "group.aero" : 1,
      "com.ms" : 1,
      "gorizia.it" : 1,
      "firm.ro" : 1,
      "polkowice.pl" : 1,
      "tysvær.no" : 1,
      "gog.pk" : 1,
      "edu.ao" : 1,
      "tayninh.vn" : 1,
      "siljan.no" : 1,
      "ac.ir" : 1,
      "岡山.jp" : 1,
      "skjak.no" : 1,
      "cr.it" : 1,
      "hå.no" : 1,
      "秋田.jp" : 1,
      "tourism.tn" : 1,
      "org.br" : 1,
      "mil.bo" : 1,
      "travel.in" : 1,
      "kvinnherad.no" : 1,
      "wiki.bo" : 1,
      "hønefoss.no" : 1,
      "narvik.no" : 1,
      "eng.br" : 1,
      "urn.arpa" : 1,
      "masfjorden.no" : 1,
      "nombre.bo" : 1,
      "hagebostad.no" : 1,
      "info.tr" : 1,
      "s.se" : 1,
      "askvoll.no" : 1,
      "edu.iq" : 1,
      "edu.kg" : 1,
      "thanhphohochiminh.vn" : 1,
      "lærdal.no" : 1,
      "abo.pa" : 1,
      "org.tj" : 1,
      "voss.no" : 1,
      "sec.ps" : 1,
      "gv.ao" : 1,
      "三重.jp" : 1,
      "bådåddjå.no" : 1,
      "net.th" : 1,
      "ákŋoluokta.no" : 1,
      "mazowsze.pl" : 1,
      "lc.it" : 1,
      "naroy.no" : 1,
      "bozen-suedtirol.it" : 1,
      "legnica.pl" : 1,
      "lom.no" : 1,
      "srv.br" : 1,
      "lv.ua" : 1,
      "edu.lk" : 1,
      "hemsedal.no" : 1,
      "bg.it" : 1,
      "saitama.jp" : 1,
      "delhi.in" : 1,
      "pro.ht" : 1,
      "folkebibl.no" : 1,
      "kárášjohka.no" : 1,
      "�庫.jp" : 1,
      "edu.la" : 1,
      "nom.za" : 1,
      "net.tm" : 1,
      "holmestrand.no" : 1,
      "press.ma" : 1,
      "b.br" : 1,
      "com.ph" : 1,
      "gov.lr" : 1,
      "osterøy.no" : 1,
      "zachpomor.pl" : 1,
      "pv.it" : 1,
      "works.aero" : 1,
      "co.mz" : 1,
      "cz.it" : 1,
      "latina.it" : 1,
      "net.sy" : 1,
      "plo.ps" : 1,
      "net.pa" : 1,
      "ah.cn" : 1,
      "viterbo.it" : 1,
      "nt.ro" : 1,
      "go.kr" : 1,
      "長野.jp" : 1,
      "net.pk" : 1,
      "etc.br" : 1,
      "kontum.vn" : 1,
      "gov.kw" : 1,
      "transporte.bo" : 1,
      "loten.no" : 1,
      "nc.us" : 1,
      "ternopil.ua" : 1,
      "suli.hu" : 1,
      "internet.in" : 1,
      "brand.se" : 1,
      "red.sv" : 1,
      "airtraffic.aero" : 1,
      "nom.nc" : 1,
      "dellogliastra.it" : 1,
      "com.ee" : 1,
      "jur.pro" : 1,
      "zakarpattia.ua" : 1,
      "komvux.se" : 1,
      "web.lk" : 1,
      "fredrikstad.no" : 1,
      "ascoli-piceno.it" : 1,
      "urbino-pesaro.it" : 1,
      "org.sl" : 1,
      "fuel.aero" : 1,
      "edu.ki" : 1,
      "hi.cn" : 1,
      "atm.pl" : 1,
      "drangedal.no" : 1,
      "deatnu.no" : 1,
      "opoczno.pl" : 1,
      "santoandre.br" : 1,
      "name.tj" : 1,
      "sorreisa.no" : 1,
      "halden.no" : 1,
      "treviso.it" : 1,
      "marnardal.no" : 1,
      "gangwon.kr" : 1,
      "gol.no" : 1,
      "vet.br" : 1,
      "nat.tn" : 1,
      "co.il" : 1,
      "ai.jo" : 1,
      "adult.ht" : 1,
      "emergency.aero" : 1,
      "jelenia-gora.pl" : 1,
      "pi.it" : 1,
      "gov.mv" : 1,
      "boavista.br" : 1,
      "net.ws" : 1,
      "ngo.ph" : 1,
      "gov.ee" : 1,
      "gyeongbuk.kr" : 1,
      "ed.ci" : 1,
      "sa.it" : 1,
      "ac.jp" : 1,
      "hamaroy.no" : 1,
      "sc.ls" : 1,
      "industria.bo" : 1,
      "info.tz" : 1,
      "federation.aero" : 1,
      "org.uy" : 1,
      "aremark.no" : 1,
      "scientist.aero" : 1,
      "finnoy.no" : 1,
      "com.kw" : 1,
      "com.ug" : 1,
      "com.vu" : 1,
      "emilia-romagna.it" : 1,
      "qld.au" : 1,
      "org.gu" : 1,
      "dp.ua" : 1,
      "com.ag" : 1,
      "mil.lv" : 1,
      "frøya.no" : 1,
      "nesset.no" : 1,
      "odessa.ua" : 1,
      "acct.pro" : 1,
      "tp.it" : 1,
      "id.vn" : 1,
      "com.lr" : 1,
      "art.dz" : 1,
      "ind.tn" : 1,
      "ivano-frankivsk.ua" : 1,
      "porsáŋgu.no" : 1,
      "org.in" : 1,
      "gov.ph" : 1,
      "md.us" : 1,
      "school.za" : 1,
      "gob.cu" : 1,
      "hitra.no" : 1,
      "rovigo.it" : 1,
      "ostre-toten.no" : 1,
      "vi.it" : 1,
      "sørum.no" : 1,
      "kåfjord.no" : 1,
      "trentinsüdtirol.it" : 1,
      "net.bt" : 1,
      "art.ht" : 1,
      "ravenna.it" : 1,
      "info.pr" : 1,
      "gob.ni" : 1,
      "net.al" : 1,
      "if.ua" : 1,
      "jgora.pl" : 1,
      "mil.kz" : 1,
      "accident-investigation.aero" : 1,
      "fnd.br" : 1,
      "law.pro" : 1,
      "com.mv" : 1,
      "ogliastra.it" : 1,
      "net.vc" : 1,
      "seg.br" : 1,
      "lurøy.no" : 1,
      "pesaro-urbino.it" : 1,
      "eidsberg.no" : 1,
      "go.ke" : 1,
      "beiarn.no" : 1,
      "info.ls" : 1,
      "chungbuk.kr" : 1,
      "skien.no" : 1,
      "ck.ua" : 1,
      "strand.no" : 1,
      "jolster.no" : 1,
      "sula.no" : 1,
      "gov.it" : 1,
      "mil.tt" : 1,
      "akrehamn.no" : 1,
      "vv.it" : 1,
      "com.re" : 1,
      "ايران.ir" : 1,
      "cri.br" : 1,
      "air-traffic-control.aero" : 1,
      "org.mw" : 1,
      "lavagis.no" : 1,
      "京都.jp" : 1,
      "marche.it" : 1,
      "busan.kr" : 1,
      "mil.cl" : 1,
      "k.se" : 1,
      "chieti.it" : 1,
      "hyogo.jp" : 1,
      "bygland.no" : 1,
      "net.st" : 1,
      "floripa.br" : 1,
      "sonla.vn" : 1,
      "gov.pw" : 1,
      "fg.it" : 1,
      "徳島.jp" : 1,
      "org.mo" : 1,
      "milan.it" : 1,
      "sirdal.no" : 1,
      "wroclaw.pl" : 1,
      "agro.bj" : 1,
      "yokohama.jp" : 1,
      "friulivgiulia.it" : 1,
      "tempioolbia.it" : 1,
      "ørsta.no" : 1,
      "島根.jp" : 1,
      "info.bo" : 1,
      "net.ge" : 1,
      "modelling.aero" : 1,
      "栃木.jp" : 1,
      "laz.it" : 1,
      "co.ke" : 1,
      "vaga.no" : 1,
      "gov.ly" : 1,
      "sch.ir" : 1,
      "net.sd" : 1,
      "org.tw" : 1,
      "sch.id" : 1,
      "gr.jp" : 1,
      "idf.il" : 1,
      "com.om" : 1,
      "farsund.no" : 1,
      "net.jo" : 1,
      "massa-carrara.it" : 1,
      "emb.kw" : 1,
      "info.ml" : 1,
      "art.sn" : 1,
      "edu.za" : 1,
      "gz.cn" : 1,
      "træna.no" : 1,
      "qsl.br" : 1,
      "krodsherad.no" : 1,
      "resto.bj" : 1,
      "prd.km" : 1,
      "trentinoaadige.it" : 1,
      "salerno.it" : 1,
      "varoy.no" : 1,
      "groundhandling.aero" : 1,
      "sandøy.no" : 1,
      "edu.do" : 1,
      "id.lv" : 1,
      "andasuolo.no" : 1,
      "sch.ly" : 1,
      "org.to" : 1,
      "mil.tr" : 1,
      "gov.ir" : 1,
      "workinggroup.aero" : 1,
      "1.bg" : 1,
      "eco.br" : 1,
      "press.se" : 1,
      "gov.az" : 1,
      "9.bg" : 1,
      "trentino-süd-tirol.it" : 1,
      "rimini.it" : 1,
      "co.uk" : 1,
      "hattfjelldal.no" : 1,
      "måsøy.no" : 1,
      "co.ao" : 1,
      "nord-odal.no" : 1,
      "com.bi" : 1,
      "altoadige.it" : 1,
      "mil.ye" : 1,
      "røst.no" : 1,
      "evenes.no" : 1,
      "ac.pa" : 1,
      "kvitsoy.no" : 1,
      "sn.cn" : 1,
      "gsm.pl" : 1,
      "ac.be" : 1,
      "bd.se" : 1,
      "davvenjarga.no" : 1,
      "union.aero" : 1,
      "net.tj" : 1,
      "reggio-emilia.it" : 1,
      "edu.gn" : 1,
      "co.hu" : 1,
      "int.pt" : 1,
      "edu.ml" : 1,
      "samnanger.no" : 1,
      "ppg.br" : 1,
      "med.sa" : 1,
      "bozen-südtirol.it" : 1,
      "mat.br" : 1,
      "arts.ro" : 1,
      "veneto.it" : 1,
      "org.ro" : 1,
      "umb.it" : 1,
      "net.br" : 1,
      "岩手.jp" : 1,
      "varese.it" : 1,
      "sokndal.no" : 1,
      "web.za" : 1,
      "or.kr" : 1,
      "biz.pr" : 1,
      "crotone.it" : 1,
      "police.uk" : 1,
      "ecologia.bo" : 1,
      "beskidy.pl" : 1,
      "组織.hk" : 1,
      "vic.au" : 1,
      "int.ci" : 1,
      "rotorcraft.aero" : 1,
      "frana.no" : 1,
      "la-spezia.it" : 1,
      "conf.lv" : 1,
      "網絡.hk" : 1,
      "web.do" : 1,
      "swiebodzin.pl" : 1,
      "net.hn" : 1,
      "backan.vn" : 1,
      "its.me" : 1,
      "org.rw" : 1,
      "shizuoka.jp" : 1,
      "mk.ua" : 1,
      "pilot.aero" : 1,
      "recife.br" : 1,
      "songdalen.no" : 1,
      "sosnowiec.pl" : 1,
      "røros.no" : 1,
      "lardal.no" : 1,
      "me.tz" : 1,
      "fermo.it" : 1,
      "net.cw" : 1,
      "or.ke" : 1,
      "org.ar" : 1,
      "dep.no" : 1,
      "bokn.no" : 1,
      "rade.no" : 1,
      "åseral.no" : 1,
      "bz.it" : 1,
      "poa.br" : 1,
      "hoyanger.no" : 1,
      "belluno.it" : 1,
      "nååmesjevuemie.no" : 1,
      "biz.ss" : 1,
      "abruzzo.it" : 1,
      "mil.iq" : 1,
      "mil.kg" : 1,
      "averøy.no" : 1,
      "kirkenes.no" : 1,
      "niteroi.br" : 1,
      "santamaria.br" : 1,
      "co.lc" : 1,
      "sorocaba.br" : 1,
      "gov.om" : 1,
      "lillesand.no" : 1,
      "net.co" : 1,
      "com.ni" : 1,
      "8.bg" : 1,
      "edu.sb" : 1,
      "tuscany.it" : 1,
      "priv.hu" : 1,
      "edu.sc" : 1,
      "slg.br" : 1,
      "fræna.no" : 1,
      "med.pro" : 1,
      "com.ly" : 1,
      "andebu.no" : 1,
      "net.nr" : 1,
      "consultant.aero" : 1,
      "me.eg" : 1,
      "pr.ml" : 1,
      "politica.bo" : 1,
      "vågan.no" : 1,
      "com.cu" : 1,
      "edu.bo" : 1,
      "edu.bh" : 1,
      "ac.se" : 1,
      "mil.ac" : 1,
      "sm.ua" : 1,
      "express.aero" : 1,
      "fosnes.no" : 1,
      "edu.fm" : 1,
      "bulsan-suedtirol.it" : 1,
      "coop.mw" : 1,
      "gov.me" : 1,
      "fyresdal.no" : 1,
      "test.tj" : 1,
      "av.it" : 1,
      "bmd.br" : 1,
      "soc.lk" : 1,
      "imperia.it" : 1,
      "edu.ls" : 1,
      "loabat.no" : 1,
      "cnt.br" : 1,
      "org.pe" : 1,
      "nome.pt" : 1,
      "co.kr" : 1,
      "trana.no" : 1,
      "enna.it" : 1,
      "edu.so" : 1,
      "rygge.no" : 1,
      "andria-barletta-trani.it" : 1,
      "ninhthuan.vn" : 1,
      "co.ag" : 1,
      "emr.it" : 1,
      "mil.my" : 1,
      "app.br" : 1,
      "org.ec" : 1,
      "amli.no" : 1,
      "com.uz" : 1,
      "selbu.no" : 1,
      "net.ps" : 1,
      "co.dm" : 1,
      "edu.bb" : 1,
      "name.tr" : 1,
      "web.bo" : 1,
      "li.it" : 1,
      "vossevangen.no" : 1,
      "com.az" : 1,
      "bib.ve" : 1,
      "gov.lt" : 1,
      "orland.no" : 1,
      "ekloges.cy" : 1,
      "sel.no" : 1,
      "áltá.no" : 1,
      "edu.et" : 1,
      "smøla.no" : 1,
      "mil.cy" : 1,
      "go.ug" : 1,
      "com.mk" : 1,
      "org.bt" : 1,
      "pro.om" : 1,
      "caobang.vn" : 1,
      "gov.sg" : 1,
      "com.ng" : 1,
      "tr.it" : 1,
      "aerodrome.aero" : 1,
      "fam.pk" : 1,
      "grondar.za" : 1,
      "ski.no" : 1,
      "moskenes.no" : 1,
      "net.in" : 1,
      "manaus.br" : 1,
      "宮城.jp" : 1,
      "kvinesdal.no" : 1,
      "biz.vn" : 1,
      "東京.jp" : 1,
      "gr.it" : 1,
      "elverum.no" : 1,
      "山形.jp" : 1,
      "puglia.it" : 1,
      "soctrang.vn" : 1,
      "net.cm" : 1,
      "aoste.it" : 1,
      "val-d-aosta.it" : 1,
      "ca.in" : 1,
      "net.uy" : 1,
      "net.gu" : 1,
      "khanhhoa.vn" : 1,
      "edu.bm" : 1,
      "org.ws" : 1,
      "eu.int" : 1,
      "me.so" : 1,
      "kautokeino.no" : 1,
      "leangaviika.no" : 1,
      "starachowice.pl" : 1,
      "osøyro.no" : 1,
      "mosvik.no" : 1,
      "edu.hk" : 1,
      "com.gh" : 1,
      "avoues.fr" : 1,
      "tjome.no" : 1,
      "mail.pl" : 1,
      "in.th" : 1,
      "tynset.no" : 1,
      "quangnam.vn" : 1,
      "cuneo.it" : 1,
      "valle-aosta.it" : 1,
      "taranto.it" : 1,
      "net.mo" : 1,
      "nv.us" : 1,
      "zagan.pl" : 1,
      "sandnessjoen.no" : 1,
      "targi.pl" : 1,
      "feira.br" : 1,
      "com.mu" : 1,
      "várggát.no" : 1,
      "travel.pl" : 1,
      "org.st" : 1,
      "jevnaker.no" : 1,
      "tvedestrand.no" : 1,
      "roma.it" : 1,
      "vang.no" : 1,
      "oppegard.no" : 1,
      "bearalváhki.no" : 1,
      "skjervøy.no" : 1,
      "富山.jp" : 1,
      "travinh.vn" : 1,
      "yamanashi.jp" : 1,
      "br.it" : 1,
      "campinagrande.br" : 1,
      "safety.aero" : 1,
      "notaires.km" : 1,
      "net.mw" : 1,
      "stord.no" : 1,
      "mobi.gp" : 1,
      "ltd.uk" : 1,
      "org.vc" : 1,
      "gov.bn" : 1,
      "plc.ly" : 1,
      "org.al" : 1,
      "ac.me" : 1,
      "giske.no" : 1,
      "naples.it" : 1,
      "inst.ml" : 1,
      "rollag.no" : 1,
      "shop.pl" : 1,
      "balestrand.no" : 1,
      "ac.im" : 1,
      "pc.it" : 1,
      "cr.ua" : 1,
      "edu.mx" : 1,
      "nord-aurdal.no" : 1,
      "trentin-suedtirol.it" : 1,
      "zarow.pl" : 1,
      "edu.tt" : 1,
      "sc.tz" : 1,
      "vestre-slidre.no" : 1,
      "oksnes.no" : 1,
      "radøy.no" : 1,
      "passenger-association.aero" : 1,
      "trentinosudtirol.it" : 1,
      "co.zm" : 1,
      "quangtri.vn" : 1,
      "tottori.jp" : 1,
      "山口.jp" : 1,
      "trogstad.no" : 1,
      "katowice.pl" : 1,
      "rahkkeravju.no" : 1,
      "edu.kz" : 1,
      "malbork.pl" : 1,
      "org.pa" : 1,
      "sch.ng" : 1,
      "a.bg" : 1,
      "fr.it" : 1,
      "biz.pl" : 1,
      "com.es" : 1,
      "flesberg.no" : 1,
      "com.sn" : 1,
      "org.pk" : 1,
      "stavern.no" : 1,
      "no.it" : 1,
      "berg.no" : 1,
      "mielno.pl" : 1,
      "raisa.no" : 1,
      "n.bg" : 1,
      "org.sy" : 1,
      "kolobrzeg.pl" : 1,
      "ilawa.pl" : 1,
      "com.sg" : 1,
      "gob.ve" : 1,
      "civilaviation.aero" : 1,
      "bajddar.no" : 1,
      "ao.it" : 1,
      "gov.ng" : 1,
      "gov.mk" : 1,
      "engerdal.no" : 1,
      "vc.it" : 1,
      "name.pr" : 1,
      "tromso.no" : 1,
      "bulsan-südtirol.it" : 1,
      "kyoto.jp" : 1,
      "pug.it" : 1,
      "gov.ma" : 1,
      "nf.ca" : 1,
      "org.tm" : 1,
      "edu.dm" : 1,
      "brumunddal.no" : 1,
      "gop.pk" : 1,
      "oh.us" : 1,
      "edu.gi" : 1,
      "sogne.no" : 1,
      "austevoll.no" : 1,
      "i.ng" : 1,
      "olawa.pl" : 1,
      "stordal.no" : 1,
      "xz.cn" : 1,
      "vestvagoy.no" : 1,
      "zgorzelec.pl" : 1,
      "com.bn" : 1,
      "porsgrunn.no" : 1,
      "stor-elvdal.no" : 1,
      "edu.lv" : 1,
      "hungyen.vn" : 1,
      "gov.rs" : 1,
      "bjugn.no" : 1,
      "gov.il" : 1,
      "yenbai.vn" : 1,
      "tiengiang.vn" : 1,
      "kiev.ua" : 1,
      "vegårshei.no" : 1,
      "l.bg" : 1,
      "radom.pl" : 1,
      "mediocampidano.it" : 1,
      "catanzaro.it" : 1,
      "bergamo.it" : 1,
      "km.ua" : 1,
      "mil.fj" : 1,
      "愛媛.jp" : 1,
      "ens.tn" : 1,
      "südtirol.it" : 1,
      "lo.it" : 1,
      "com.hr" : 1,
      "gov.mu" : 1,
      "ragusa.it" : 1,
      "democracia.bo" : 1,
      "vald-aosta.it" : 1,
      "net.sl" : 1,
      "nannestad.no" : 1,
      "mp.br" : 1,
      "per.jo" : 1,
      "pro.az" : 1,
      "osaka.jp" : 1,
      "gov.gh" : 1,
      "herad.no" : 1,
      "gov.ki" : 1,
      "tv.im" : 1,
      "engine.aero" : 1,
      "net.tn" : 1,
      "hapmir.no" : 1,
      "movimiento.bo" : 1,
      "val-daosta.it" : 1,
      "q.bg" : 1,
      "iwi.nz" : 1,
      "in-addr.arpa" : 1,
      "equipment.aero" : 1,
      "tv.jo" : 1,
      "trainer.aero" : 1,
      "trentino-sudtirol.it" : 1,
      "siellak.no" : 1,
      "teo.br" : 1,
      "olkusz.pl" : 1,
      "royrvik.no" : 1,
      "karpacz.pl" : 1,
      "net.ae" : 1,
      "slz.br" : 1,
      "of.by" : 1,
      "org.af" : 1,
      "gov.cy" : 1,
      "moåreke.no" : 1,
      "es.kr" : 1,
      "name.fj" : 1,
      "monzaebrianza.it" : 1,
      "afjord.no" : 1,
      "z.se" : 1,
      "yn.cn" : 1,
      "sejny.pl" : 1,
      "asker.no" : 1,
      "norddal.no" : 1,
      "from.hr" : 1,
      "net.nf" : 1,
      "press.cy" : 1,
      "med.pa" : 1,
      "skánit.no" : 1,
      "kommune.no" : 1,
      "aurland.no" : 1,
      "daegu.kr" : 1,
      "edu.mv" : 1,
      "w.se" : 1,
      "benevento.it" : 1,
      "geo.br" : 1,
      "tinn.no" : 1,
      "pz.it" : 1,
      "組织.hk" : 1,
      "tm.pl" : 1,
      "storfjord.no" : 1,
      "gov.la" : 1,
      "shimane.jp" : 1,
      "trading.aero" : 1,
      "andøy.no" : 1,
      "pharmaciens.km" : 1,
      "fie.ee" : 1,
      "gov.lk" : 1,
      "rc.it" : 1,
      "aknoluokta.no" : 1,
      "salat.no" : 1,
      "biz.tj" : 1,
      "edu.lr" : 1,
      "trentinsüd-tirol.it" : 1,
      "gob.bo" : 1,
      "coop.km" : 1,
      "ci.it" : 1,
      "askim.no" : 1,
      "lørenskog.no" : 1,
      "mandal.no" : 1,
      "of.no" : 1,
      "askøy.no" : 1,
      "rælingen.no" : 1,
      "edu.vu" : 1,
      "kalisz.pl" : 1,
      "ad.jp" : 1,
      "edu.kw" : 1,
      "bodø.no" : 1,
      "ngo.lk" : 1,
      "ralingen.no" : 1,
      "or.us" : 1,
      "rennesoy.no" : 1,
      "trentin-südtirol.it" : 1,
      "valle-daosta.it" : 1,
      "jampa.br" : 1,
      "notodden.no" : 1,
      "ag.it" : 1,
      "valleaosta.it" : 1,
      "okinawa.jp" : 1,
      "pr.us" : 1,
      "tx.us" : 1,
      "ks.us" : 1,
      "mjøndalen.no" : 1,
      "nom.ni" : 1,
      "værøy.no" : 1,
      "info.vn" : 1,
      "lomza.pl" : 1,
      "firm.in" : 1,
      "smola.no" : 1,
      "sandefjord.no" : 1,
      "sch.lk" : 1,
      "co.jp" : 1,
      "calabria.it" : 1,
      "com.fj" : 1,
      "net.pr" : 1,
      "fukushima.jp" : 1,
      "name.eg" : 1,
      "net.ss" : 1,
      "orsta.no" : 1,
      "ac.il" : 1,
      "tur.br" : 1,
      "net.gp" : 1,
      "loppa.no" : 1,
      "lamdong.vn" : 1,
      "nagoya.jp" : 1,
      "catering.aero" : 1,
      "st.no" : 1,
      "umbria.it" : 1,
      "priv.pl" : 1,
      "design.aero" : 1,
      "net.vi" : 1,
      "vadsø.no" : 1,
      "verbania.it" : 1,
      "saga.jp" : 1,
      "turin.it" : 1,
      "vf.no" : 1,
      "raholt.no" : 1,
      "ustka.pl" : 1,
      "org.se" : 1,
      "org.gr" : 1,
      "edu.it" : 1,
      "net.mz" : 1,
      "erotica.hu" : 1,
      "org.cn" : 1,
      "com.am" : 1,
      "yamagata.jp" : 1,
      "bronnoy.no" : 1,
      "friulivenezia-giulia.it" : 1,
      "romskog.no" : 1,
      "org.zm" : 1,
      "tm.cy" : 1,
      "verdal.no" : 1,
      "ostrowwlkp.pl" : 1,
      "org.im" : 1,
      "biz.nr" : 1,
      "eti.br" : 1,
      "pueblo.bo" : 1,
      "brønnøysund.no" : 1,
      "長崎.jp" : 1,
      "com.cy" : 1,
      "nieruchomosci.pl" : 1,
      "gkp.pk" : 1,
      "mil.ng" : 1,
      "lund.no" : 1,
      "ullensvang.no" : 1,
      "alt.na" : 1,
      "orkanger.no" : 1,
      "tysfjord.no" : 1,
      "net.eg" : 1,
      "chernihiv.ua" : 1,
      "ap.it" : 1,
      "org.kp" : 1,
      "com.ki" : 1,
      "ut.us" : 1,
      "reklam.hu" : 1,
      "cherkasy.ua" : 1,
      "gov.fj" : 1,
      "davvesiida.no" : 1,
      "bas.it" : 1,
      "tec.ve" : 1,
      "grue.no" : 1,
      "romsa.no" : 1,
      "nd.us" : 1,
      "edu.ee" : 1,
      "ss.it" : 1,
      "gob.do" : 1,
      "gov.tl" : 1,
      "det.br" : 1,
      "net.bs" : 1,
      "trentino-aadige.it" : 1,
      "bomlo.no" : 1,
      "somna.no" : 1,
      "campinas.br" : 1,
      "sør-aurdal.no" : 1,
      "edu.ph" : 1,
      "wa.us" : 1,
      "báhccavuotna.no" : 1,
      "mil.gh" : 1,
      "福井.jp" : 1,
      "friuliv-giulia.it" : 1,
      "meloy.no" : 1,
      "flekkefjord.no" : 1,
      "ac.mz" : 1,
      "co.us" : 1,
      "kobierzyce.pl" : 1,
      "laocai.vn" : 1,
      "arna.no" : 1,
      "ingatlan.hu" : 1,
      "osasco.br" : 1,
      "com.lk" : 1,
      "6g.in" : 1,
      "net.lc" : 1,
      "def.br" : 1,
      "net.lb" : 1,
      "asn.au" : 1,
      "com.la" : 1,
      "wa.au" : 1,
      "or.jp" : 1,
      "biz.in" : 1,
      "com.iq" : 1,
      "com.kg" : 1,
      "fjaler.no" : 1,
      "net.vn" : 1,
      "pp.se" : 1,
      "porsangu.no" : 1,
      "org.ky" : 1,
      "firm.ve" : 1,
      "brescia.it" : 1,
      "gx.cn" : 1,
      "vr.it" : 1,
      "com.aw" : 1,
      "mil.ni" : 1,
      "od.ua" : 1,
      "eng.pro" : 1,
      "org.zw" : 1,
      "web.ve" : 1,
      "snoasa.no" : 1,
      "bsb.br" : 1,
      "org.ci" : 1,
      "quangngai.vn" : 1,
      "ak.us" : 1,
      "trentino.it" : 1,
      "gjemnes.no" : 1,
      "forlìcesena.it" : 1,
      "gamvik.no" : 1,
      "co.gl" : 1,
      "hjelmeland.no" : 1,
      "ringerike.no" : 1,
      "lg.jp" : 1,
      "førde.no" : 1,
      "arts.nf" : 1,
      "skjåk.no" : 1,
      "uzhgorod.ua" : 1,
      "am.in" : 1,
      "leirvik.no" : 1,
      "nara.jp" : 1,
      "jx.cn" : 1,
      "rec.ve" : 1,
      "iwate.jp" : 1,
      "bihar.in" : 1,
      "sør-odal.no" : 1,
      "d.bg" : 1,
      "iris.arpa" : 1,
      "edu.ms" : 1,
      "carbonia-iglesias.it" : 1,
      "publ.cv" : 1,
      "www.ck" : 1,
      "co.om" : 1,
      "fc.it" : 1,
      "trentinsued-tirol.it" : 1,
      "org.io" : 1,
      "gov.tr" : 1,
      "sálat.no" : 1,
      "co.ir" : 1,
      "org.pt" : 1,
      "reggioemilia.it" : 1,
      "mil.id" : 1,
      "e12.ve" : 1,
      "andriatranibarletta.it" : 1,
      "leaŋgaviika.no" : 1,
      "er.in" : 1,
      "co.pn" : 1,
      "sardinia.it" : 1,
      "kopervik.no" : 1,
      "mr.no" : 1,
      "net.au" : 1,
      "trentinsud-tirol.it" : 1,
      "entertainment.aero" : 1,
      "ac.ug" : 1,
      "longan.vn" : 1,
      "net.gy" : 1,
      "risør.no" : 1,
      "name.az" : 1,
      "kanagawa.jp" : 1,
      "cpa.pro" : 1,
      "trani-barletta-andria.it" : 1,
      "co.it" : 1,
      "nom.es" : 1,
      "shiga.jp" : 1,
      "ørskog.no" : 1,
      "ibestad.no" : 1,
      "elblag.pl" : 1,
      "kviteseid.no" : 1,
      "hurdal.no" : 1,
      "fusa.no" : 1,
      "edu.ve" : 1,
      "med.ec" : 1,
      "valléeaoste.it" : 1,
      "sar.it" : 1,
      "overhalla.no" : 1,
      "berlevag.no" : 1,
      "medio-campidano.it" : 1,
      "org.mg" : 1,
      "asti.it" : 1,
      "gyeonggi.kr" : 1,
      "com.my" : 1,
      "name.mv" : 1,
      "modum.no" : 1,
      "laichau.vn" : 1,
      "broker.aero" : 1,
      "agrar.hu" : 1,
      "gov.pn" : 1,
      "mosjøen.no" : 1,
      "namsos.no" : 1,
      "vallée-aoste.it" : 1,
      "gob.cl" : 1,
      "qc.ca" : 1,
      "gov.ye" : 1,
      "dev.br" : 1,
      "ac.sz" : 1,
      "shop.hu" : 1,
      "org.na" : 1,
      "frosta.no" : 1,
      "mil.az" : 1,
      "valled-aosta.it" : 1,
      "valle.no" : 1,
      "org.gl" : 1,
      "gob.mx" : 1,
      "net.uk" : 1,
      "fetsund.no" : 1,
      "mobi.ke" : 1,
      "blog.bo" : 1,
      "tourism.pl" : 1,
      "com.ac" : 1,
      "net.ua" : 1,
      "edu.bz" : 1,
      "avocat.pro" : 1,
      "chernivtsi.ua" : 1,
      "bet.ar" : 1,
      "traniandriabarletta.it" : 1,
      "biz.mw" : 1,
      "malselv.no" : 1,
      "maceio.br" : 1,
      "c.bg" : 1,
      "pro.fj" : 1,
      "org.mn" : 1,
      "asso.re" : 1,
      "flatanger.no" : 1,
      "tr.no" : 1,
      "selje.no" : 1,
      "co.gy" : 1,
      "indigena.bo" : 1,
      "off.ai" : 1,
      "com.bj" : 1,
      "net.pl" : 1,
      "gos.pk" : 1,
      "gov.kn" : 1,
      "oz.au" : 1,
      "bar.pro" : 1,
      "gov.ao" : 1,
      "和歌山.jp" : 1,
      "co.na" : 1,
      "com.tr" : 1,
      "desa.id" : 1,
      "4.bg" : 1,
      "ny.us" : 1,
      "zt.ua" : 1,
      "vicenza.it" : 1,
      "monza-brianza.it" : 1,
      "tuyenquang.vn" : 1,
      "frogn.no" : 1,
      "krokstadelva.no" : 1,
      "co.bj" : 1,
      "pro.cy" : 1,
      "cosenza.it" : 1,
      "augustow.pl" : 1,
      "foz.br" : 1,
      "siracusa.it" : 1,
      "rv.ua" : 1,
      "muni.il" : 1,
      "ostrowiec.pl" : 1,
      "tra.kp" : 1,
      "org.ba" : 1,
      "ass.km" : 1,
      "pol.tr" : 1,
      "com.mt" : 1,
      "czeladz.pl" : 1,
      "gov.kg" : 1,
      "gov.iq" : 1,
      "en.it" : 1,
      "dnepropetrovsk.ua" : 1,
      "vågsøy.no" : 1,
      "ål.no" : 1,
      "rauma.no" : 1,
      "pr.it" : 1,
      "per.nf" : 1,
      "univ.bj" : 1,
      "or.it" : 1,
      "edu.km" : 1,
      "net.gt" : 1,
      "gru.br" : 1,
      "edu.cv" : 1,
      "nesodden.no" : 1,
      "ostroda.pl" : 1,
      "gov.ac" : 1,
      "odda.no" : 1,
      "lel.br" : 1,
      "folldal.no" : 1,
      "hægebostad.no" : 1,
      "org.py" : 1,
      "tv.br" : 1,
      "org.sa" : 1,
      "tysnes.no" : 1,
      "com.ye" : 1,
      "naamesjevuemie.no" : 1,
      "harstad.no" : 1,
      "edu.dz" : 1,
      "spydeberg.no" : 1,
      "mc.it" : 1,
      "ai.vn" : 1,
      "net.qa" : 1,
      "sb.ua" : 1,
      "bryne.no" : 1,
      "luroy.no" : 1,
      "gov.my" : 1,
      "edu.ht" : 1,
      "esp.br" : 1,
      "int.ar" : 1,
      "kepno.pl" : 1,
      "gub.uy" : 1,
      "me.in" : 1
   },
   "version" : "2025-01-21_09-07-06_UTC",
   "top_level_domain" : {
      "football" : 1,
      "fidelity" : 1,
      "tatar" : 1,
      "work" : 1,
      "politie" : 1,
      "motorcycles" : 1,
      "anquan" : 1,
      "blackfriday" : 1,
      "cloud" : 1,
      "google" : 1,
      "maif" : 1,
      "td" : 1,
      "sarl" : 1,
      "esq" : 1,
      "ga" : 1,
      "man" : 1,
      "nikon" : 1,
      "ly" : 1,
      "mattel" : 1,
      "group" : 1,
      "credit" : 1,
      "website" : 1,
      "otsuka" : 1,
      "km" : 1,
      "sc" : 1,
      "fedex" : 1,
      "ng" : 1,
      "aaa" : 1,
      "allfinanz" : 1,
      "xbox" : 1,
      "leclerc" : 1,
      "aig" : 1,
      "staples" : 1,
      "ci" : 1,
      "pars" : 1,
      "dds" : 1,
      "llc" : 1,
      "case" : 1,
      "navy" : 1,
      "jobs" : 1,
      "care" : 1,
      "auto" : 1,
      "ericsson" : 1,
      "goo" : 1,
      "bzh" : 1,
      "java" : 1,
      "int" : 1,
      "stcgroup" : 1,
      "sk" : 1,
      "ag" : 1,
      "info" : 1,
      "pizza" : 1,
      "dentist" : 1,
      "shiksha" : 1,
      "ls" : 1,
      "delta" : 1,
      "buy" : 1,
      "education" : 1,
      "safe" : 1,
      "ott" : 1,
      "book" : 1,
      "markets" : 1,
      "lotte" : 1,
      "host" : 1,
      "lacaixa" : 1,
      "luxury" : 1,
      "kred" : 1,
      "audible" : 1,
      "kitchen" : 1,
      "nokia" : 1,
      "fail" : 1,
      "yahoo" : 1,
      "bauhaus" : 1,
      "ismaili" : 1,
      "software" : 1,
      "gd" : 1,
      "lotto" : 1,
      "army" : 1,
      "silk" : 1,
      "bing" : 1,
      "tvs" : 1,
      "flowers" : 1,
      "moi" : 1,
      "sap" : 1,
      "marshalls" : 1,
      "hotels" : 1,
      "unicom" : 1,
      "ba" : 1,
      "io" : 1,
      "vegas" : 1,
      "discount" : 1,
      "racing" : 1,
      "cab" : 1,
      "express" : 1,
      "jll" : 1,
      "ae" : 1,
      "george" : 1,
      "taobao" : 1,
      "fun" : 1,
      "cbn" : 1,
      "gdn" : 1,
      "ventures" : 1,
      "ws" : 1,
      "bank" : 1,
      "md" : 1,
      "you" : 1,
      "baseball" : 1,
      "cern" : 1,
      "ally" : 1,
      "gea" : 1,
      "boo" : 1,
      "ups" : 1,
      "hangout" : 1,
      "ril" : 1,
      "casa" : 1,
      "alipay" : 1,
      "bike" : 1,
      "construction" : 1,
      "sanofi" : 1,
      "globo" : 1,
      "democrat" : 1,
      "academy" : 1,
      "kim" : 1,
      "pccw" : 1,
      "careers" : 1,
      "clinic" : 1,
      "fm" : 1,
      "samsclub" : 1,
      "mini" : 1,
      "page" : 1,
      "surgery" : 1,
      "as" : 1,
      "dclk" : 1,
      "sydney" : 1,
      "next" : 1,
      "ftr" : 1,
      "kh" : 1,
      "engineering" : 1,
      "vip" : 1,
      "cv" : 1,
      "hdfc" : 1,
      "school" : 1,
      "living" : 1,
      "yokohama" : 1,
      "ceo" : 1,
      "bd" : 1,
      "kosher" : 1,
      "monster" : 1,
      "komatsu" : 1,
      "surf" : 1,
      "car" : 1,
      "flir" : 1,
      "akdn" : 1,
      "movie" : 1,
      "ne" : 1,
      "rest" : 1,
      "deals" : 1,
      "sx" : 1,
      "ma" : 1,
      "fast" : 1,
      "coach" : 1,
      "protection" : 1,
      "homesense" : 1,
      "toyota" : 1,
      "il" : 1,
      "plus" : 1,
      "bet" : 1,
      "ss" : 1,
      "gu" : 1,
      "download" : 1,
      "th" : 1,
      "woodside" : 1,
      "select" : 1,
      "travel" : 1,
      "citadel" : 1,
      "versicherung" : 1,
      "training" : 1,
      "plumbing" : 1,
      "datsun" : 1,
      "airforce" : 1,
      "seek" : 1,
      "lgbt" : 1,
      "ax" : 1,
      "rentals" : 1,
      "istanbul" : 1,
      "bm" : 1,
      "pohl" : 1,
      "team" : 1,
      "rs" : 1,
      "here" : 1,
      "cam" : 1,
      "kerryhotels" : 1,
      "landrover" : 1,
      "phone" : 1,
      "gf" : 1,
      "city" : 1,
      "media" : 1,
      "prod" : 1,
      "oracle" : 1,
      "airtel" : 1,
      "honda" : 1,
      "miami" : 1,
      "cheap" : 1,
      "mtr" : 1,
      "foo" : 1,
      "mormon" : 1,
      "vision" : 1,
      "pramerica" : 1,
      "mint" : 1,
      "rogers" : 1,
      "pru" : 1,
      "americanfamily" : 1,
      "lexus" : 1,
      "country" : 1,
      "uz" : 1,
      "se" : 1,
      "sy" : 1,
      "family" : 1,
      "tech" : 1,
      "blockbuster" : 1,
      "tienda" : 1,
      "song" : 1,
      "mm" : 1,
      "cr" : 1,
      "diy" : 1,
      "tf" : 1,
      "lc" : 1,
      "pw" : 1,
      "fire" : 1,
      "re" : 1,
      "athleta" : 1,
      "nrw" : 1,
      "viking" : 1,
      "gh" : 1,
      "healthcare" : 1,
      "lk" : 1,
      "goldpoint" : 1,
      "services" : 1,
      "coop" : 1,
      "temasek" : 1,
      "barclays" : 1,
      "management" : 1,
      "lat" : 1,
      "yt" : 1,
      "smile" : 1,
      "ren" : 1,
      "aws" : 1,
      "company" : 1,
      "sbi" : 1,
      "institute" : 1,
      "fan" : 1,
      "delivery" : 1,
      "np" : 1,
      "accountant" : 1,
      "cooking" : 1,
      "shop" : 1,
      "bbc" : 1,
      "schaeffler" : 1,
      "mh" : 1,
      "jm" : 1,
      "nc" : 1,
      "solar" : 1,
      "sandvikcoromant" : 1,
      "business" : 1,
      "soccer" : 1,
      "sg" : 1,
      "hyundai" : 1,
      "tkmaxx" : 1,
      "rio" : 1,
      "dating" : 1,
      "bf" : 1,
      "accountants" : 1,
      "dtv" : 1,
      "vermögensberater" : 1,
      "gm" : 1,
      "xyz" : 1,
      "nexus" : 1,
      "med" : 1,
      "courses" : 1,
      "hyatt" : 1,
      "marriott" : 1,
      "gay" : 1,
      "mckinsey" : 1,
      "able" : 1,
      "post" : 1,
      "safety" : 1,
      "hair" : 1,
      "target" : 1,
      "calvinklein" : 1,
      "toray" : 1,
      "guide" : 1,
      "makeup" : 1,
      "chrome" : 1,
      "tm" : 1,
      "gifts" : 1,
      "taxi" : 1,
      "latrobe" : 1,
      "statefarm" : 1,
      "pl" : 1,
      "aramco" : 1,
      "watch" : 1,
      "trade" : 1,
      "mu" : 1,
      "dunlop" : 1,
      "hamburg" : 1,
      "bh" : 1,
      "diamonds" : 1,
      "cleaning" : 1,
      "qpon" : 1,
      "verisign" : 1,
      "sky" : 1,
      "dot" : 1,
      "vanguard" : 1,
      "play" : 1,
      "cn" : 1,
      "pid" : 1,
      "srl" : 1,
      "msd" : 1,
      "actor" : 1,
      "ac" : 1,
      "wow" : 1,
      "dz" : 1,
      "sener" : 1,
      "boats" : 1,
      "properties" : 1,
      "lidl" : 1,
      "energy" : 1,
      "email" : 1,
      "cba" : 1,
      "cisco" : 1,
      "genting" : 1,
      "partners" : 1,
      "forsale" : 1,
      "bs" : 1,
      "jaguar" : 1,
      "co" : 1,
      "gg" : 1,
      "nike" : 1,
      "wedding" : 1,
      "kp" : 1,
      "immo" : 1,
      "giving" : 1,
      "me" : 1,
      "global" : 1,
      "hn" : 1,
      "na" : 1,
      "fox" : 1,
      "now" : 1,
      "vn" : 1,
      "jio" : 1,
      "ad" : 1,
      "wtc" : 1,
      "commbank" : 1,
      "xihuan" : 1,
      "africa" : 1,
      "sm" : 1,
      "store" : 1,
      "lefrak" : 1,
      "my" : 1,
      "video" : 1,
      "nextdirect" : 1,
      "osaka" : 1,
      "run" : 1,
      "dnp" : 1,
      "by" : 1,
      "ggee" : 1,
      "furniture" : 1,
      "fairwinds" : 1,
      "gift" : 1,
      "vote" : 1,
      "club" : 1,
      "barclaycard" : 1,
      "exposed" : 1,
      "cuisinella" : 1,
      "be" : 1,
      "spa" : 1,
      "lease" : 1,
      "statebank" : 1,
      "tokyo" : 1,
      "ms" : 1,
      "nagoya" : 1,
      "garden" : 1,
      "cars" : 1,
      "graphics" : 1,
      "coupons" : 1,
      "bond" : 1,
      "intuit" : 1,
      "pink" : 1,
      "guru" : 1,
      "health" : 1,
      "playstation" : 1,
      "frontier" : 1,
      "firestone" : 1,
      "pictures" : 1,
      "room" : 1,
      "reit" : 1,
      "chat" : 1,
      "tg" : 1,
      "okinawa" : 1,
      "fishing" : 1,
      "hr" : 1,
      "barcelona" : 1,
      "press" : 1,
      "pin" : 1,
      "su" : 1,
      "helsinki" : 1,
      "gs" : 1,
      "jmp" : 1,
      "bg" : 1,
      "thd" : 1,
      "eurovision" : 1,
      "ht" : 1,
      "edeka" : 1,
      "hot" : 1,
      "clubmed" : 1,
      "bio" : 1,
      "mobile" : 1,
      "ifm" : 1,
      "london" : 1,
      "compare" : 1,
      "rexroth" : 1,
      "voto" : 1,
      "kyoto" : 1,
      "cz" : 1,
      "basketball" : 1,
      "capital" : 1,
      "holiday" : 1,
      "koeln" : 1,
      "teva" : 1,
      "build" : 1,
      "ru" : 1,
      "dance" : 1,
      "beer" : 1,
      "singles" : 1,
      "events" : 1,
      "origins" : 1,
      "madrid" : 1,
      "abb" : 1,
      "do" : 1,
      "bayern" : 1,
      "codes" : 1,
      "bestbuy" : 1,
      "tunes" : 1,
      "tjx" : 1,
      "amsterdam" : 1,
      "style" : 1,
      "pfizer" : 1,
      "amex" : 1,
      "schmidt" : 1,
      "ge" : 1,
      "fk" : 1,
      "men" : 1,
      "marketing" : 1,
      "lasalle" : 1,
      "equipment" : 1,
      "la" : 1,
      "gy" : 1,
      "agakhan" : 1,
      "gallo" : 1,
      "taipei" : 1,
      "gallery" : 1,
      "alsace" : 1,
      "je" : 1,
      "foundation" : 1,
      "like" : 1,
      "town" : 1,
      "fitness" : 1,
      "abbott" : 1,
      "bingo" : 1,
      "coffee" : 1,
      "sh" : 1,
      "dhl" : 1,
      "kpn" : 1,
      "church" : 1,
      "mg" : 1,
      "red" : 1,
      "pharmacy" : 1,
      "tatamotors" : 1,
      "app" : 1,
      "archi" : 1,
      "report" : 1,
      "dish" : 1,
      "bosch" : 1,
      "grainger" : 1,
      "estate" : 1,
      "nf" : 1,
      "sbs" : 1,
      "gap" : 1,
      "kpmg" : 1,
      "asda" : 1,
      "aquarelle" : 1,
      "direct" : 1,
      "frl" : 1,
      "samsung" : 1,
      "yodobashi" : 1,
      "fans" : 1,
      "boutique" : 1,
      "pictet" : 1,
      "pt" : 1,
      "florist" : 1,
      "save" : 1,
      "vin" : 1,
      "imamat" : 1,
      "eus" : 1,
      "creditcard" : 1,
      "pr" : 1,
      "one" : 1,
      "cw" : 1,
      "office" : 1,
      "wf" : 1,
      "seven" : 1,
      "cal" : 1,
      "sling" : 1,
      "mma" : 1,
      "nu" : 1,
      "nowruz" : 1,
      "wme" : 1,
      "iq" : 1,
      "youtube" : 1,
      "immobilien" : 1,
      "dj" : 1,
      "enterprises" : 1,
      "wed" : 1,
      "axa" : 1,
      "cool" : 1,
      "mc" : 1,
      "solutions" : 1,
      "abc" : 1,
      "physio" : 1,
      "mk" : 1,
      "nico" : 1,
      "au" : 1,
      "ninja" : 1,
      "mp" : 1,
      "chanel" : 1,
      "tirol" : 1,
      "af" : 1,
      "aol" : 1,
      "ke" : 1,
      "eco" : 1,
      "banamex" : 1,
      "in" : 1,
      "scot" : 1,
      "epson" : 1,
      "bharti" : 1,
      "lplfinancial" : 1,
      "digital" : 1,
      "spot" : 1,
      "deal" : 1,
      "ky" : 1,
      "museum" : 1,
      "study" : 1,
      "author" : 1,
      "works" : 1,
      "zuerich" : 1,
      "vivo" : 1,
      "whoswho" : 1,
      "itv" : 1,
      "mls" : 1,
      "alibaba" : 1,
      "merckmsd" : 1,
      "forex" : 1,
      "mx" : 1,
      "smart" : 1,
      "film" : 1,
      "sa" : 1,
      "walmart" : 1,
      "cl" : 1,
      "jp" : 1,
      "cruise" : 1,
      "gmbh" : 1,
      "net" : 1,
      "repair" : 1,
      "college" : 1,
      "stc" : 1,
      "prof" : 1,
      "house" : 1,
      "pioneer" : 1,
      "property" : 1,
      "hermes" : 1,
      "forum" : 1,
      "er" : 1,
      "hiv" : 1,
      "kg" : 1,
      "autos" : 1,
      "pn" : 1,
      "bom" : 1,
      "gp" : 1,
      "earth" : 1,
      "claims" : 1,
      "lawyer" : 1,
      "et" : 1,
      "rocks" : 1,
      "tc" : 1,
      "ir" : 1,
      "best" : 1,
      "mom" : 1,
      "box" : 1,
      "it" : 1,
      "lu" : 1,
      "reviews" : 1,
      "pub" : 1,
      "swiss" : 1,
      "organic" : 1,
      "bcn" : 1,
      "watches" : 1,
      "tk" : 1,
      "suzuki" : 1,
      "limo" : 1,
      "discover" : 1,
      "hotmail" : 1,
      "mil" : 1,
      "mlb" : 1,
      "abogado" : 1,
      "limited" : 1,
      "map" : 1,
      "am" : 1,
      "viajes" : 1,
      "irish" : 1,
      "zm" : 1,
      "abbvie" : 1,
      "dental" : 1,
      "sd" : 1,
      "tires" : 1,
      "viva" : 1,
      "vi" : 1,
      "citic" : 1,
      "lol" : 1,
      "aarp" : 1,
      "kuokgroup" : 1,
      "tj" : 1,
      "shangrila" : 1,
      "nhk" : 1,
      "pay" : 1,
      "zappos" : 1,
      "mit" : 1,
      "guge" : 1,
      "lt" : 1,
      "lr" : 1,
      "ford" : 1,
      "citi" : 1,
      "gl" : 1,
      "shoes" : 1,
      "gmo" : 1,
      "cc" : 1,
      "dupont" : 1,
      "coupon" : 1,
      "doctor" : 1,
      "ink" : 1,
      "talk" : 1,
      "data" : 1,
      "loans" : 1,
      "golf" : 1,
      "weibo" : 1,
      "si" : 1,
      "camera" : 1,
      "bw" : 1,
      "storage" : 1,
      "frogans" : 1,
      "computer" : 1,
      "firmdale" : 1,
      "erni" : 1,
      "homes" : 1,
      "studio" : 1,
      "bbt" : 1,
      "ferrari" : 1,
      "ck" : 1,
      "christmas" : 1,
      "scholarships" : 1,
      "casino" : 1,
      "va" : 1,
      "lundbeck" : 1,
      "supplies" : 1,
      "how" : 1,
      "cafe" : 1,
      "mw" : 1,
      "contractors" : 1,
      "buzz" : 1,
      "aco" : 1,
      "name" : 1,
      "merck" : 1,
      "meme" : 1,
      "lancaster" : 1,
      "tours" : 1,
      "meet" : 1,
      "jewelry" : 1,
      "ibm" : 1,
      "tennis" : 1,
      "pet" : 1,
      "photos" : 1,
      "americanexpress" : 1,
      "weber" : 1,
      "joy" : 1,
      "eu" : 1,
      "godaddy" : 1,
      "quebec" : 1,
      "cymru" : 1,
      "orange" : 1,
      "berlin" : 1,
      "kia" : 1,
      "pm" : 1,
      "open" : 1,
      "goodyear" : 1,
      "sb" : 1,
      "tl" : 1,
      "kaufen" : 1,
      "green" : 1,
      "fage" : 1,
      "ua" : 1,
      "aq" : 1,
      "om" : 1,
      "sina" : 1,
      "ar" : 1,
      "onion" : 1,
      "xin" : 1,
      "gle" : 1,
      "barefoot" : 1,
      "travelersinsurance" : 1,
      "sandvik" : 1,
      "infiniti" : 1,
      "kz" : 1,
      "read" : 1,
      "at" : 1,
      "anz" : 1,
      "insurance" : 1,
      "secure" : 1,
      "reisen" : 1,
      "amfam" : 1,
      "jcb" : 1,
      "haus" : 1,
      "dk" : 1,
      "tui" : 1,
      "softbank" : 1,
      "tmall" : 1,
      "norton" : 1,
      "bid" : 1,
      "gw" : 1,
      "com" : 1,
      "consulting" : 1,
      "date" : 1,
      "reise" : 1,
      "chase" : 1,
      "omega" : 1,
      "fresenius" : 1,
      "bmw" : 1,
      "imdb" : 1,
      "im" : 1,
      "gives" : 1,
      "dad" : 1,
      "fish" : 1,
      "ph" : 1,
      "hkt" : 1,
      "poker" : 1,
      "jeep" : 1,
      "gal" : 1,
      "sony" : 1,
      "deloitte" : 1,
      "fly" : 1,
      "apple" : 1,
      "sew" : 1,
      "ist" : 1,
      "tw" : 1,
      "cat" : 1,
      "gent" : 1,
      "lifestyle" : 1,
      "ikano" : 1,
      "visa" : 1,
      "cfa" : 1,
      "holdings" : 1,
      "sv" : 1,
      "nr" : 1,
      "cards" : 1,
      "vana" : 1,
      "tushu" : 1,
      "sas" : 1,
      "fo" : 1,
      "krd" : 1,
      "channel" : 1,
      "ubs" : 1,
      "kiwi" : 1,
      "icu" : 1,
      "bj" : 1,
      "ml" : 1,
      "cipriani" : 1,
      "dubai" : 1,
      "schwarz" : 1,
      "cyou" : 1,
      "help" : 1,
      "cx" : 1,
      "pf" : 1,
      "kw" : 1,
      "bradesco" : 1,
      "baidu" : 1,
      "netbank" : 1,
      "bo" : 1,
      "mobi" : 1,
      "fj" : 1,
      "sakura" : 1,
      "financial" : 1,
      "durban" : 1,
      "kfh" : 1,
      "productions" : 1,
      "cfd" : 1,
      "comsec" : 1,
      "kddi" : 1,
      "today" : 1,
      "mango" : 1,
      "maison" : 1,
      "ping" : 1,
      "faith" : 1,
      "latino" : 1,
      "neustar" : 1,
      "total" : 1,
      "travelers" : 1,
      "nab" : 1,
      "mov" : 1,
      "audio" : 1,
      "lb" : 1,
      "wang" : 1,
      "pnc" : 1,
      "reliance" : 1,
      "praxi" : 1,
      "villas" : 1,
      "tz" : 1,
      "ollo" : 1,
      "microsoft" : 1,
      "shell" : 1,
      "vodka" : 1,
      "aeg" : 1,
      "win" : 1,
      "cy" : 1,
      "ads" : 1,
      "beats" : 1,
      "melbourne" : 1,
      "st" : 1,
      "hu" : 1,
      "jnj" : 1,
      "uol" : 1,
      "vu" : 1,
      "science" : 1,
      "weir" : 1,
      "kerrylogistics" : 1,
      "sr" : 1,
      "theatre" : 1,
      "asia" : 1,
      "saxo" : 1,
      "contact" : 1,
      "netflix" : 1,
      "emerck" : 1,
      "li" : 1,
      "photo" : 1,
      "id" : 1,
      "engineer" : 1,
      "search" : 1,
      "zara" : 1,
      "arab" : 1,
      "mo" : 1,
      "qa" : 1,
      "nowtv" : 1,
      "link" : 1,
      "expert" : 1,
      "fido" : 1,
      "mba" : 1,
      "apartments" : 1,
      "new" : 1,
      "realtor" : 1,
      "juniper" : 1,
      "dvag" : 1,
      "review" : 1,
      "ngo" : 1,
      "juegos" : 1,
      "swatch" : 1,
      "fit" : 1,
      "bargains" : 1,
      "trading" : 1,
      "flickr" : 1,
      "ricoh" : 1,
      "ferrero" : 1,
      "wiki" : 1,
      "recipes" : 1,
      "ni" : 1,
      "weatherchannel" : 1,
      "wine" : 1,
      "ye" : 1,
      "joburg" : 1,
      "skype" : 1,
      "mitsubishi" : 1,
      "olayangroup" : 1,
      "inc" : 1,
      "cg" : 1,
      "lv" : 1,
      "hospital" : 1,
      "itau" : 1,
      "gucci" : 1,
      "final" : 1,
      "dev" : 1,
      "bz" : 1,
      "hitachi" : 1,
      "canon" : 1,
      "land" : 1,
      "dell" : 1,
      "broker" : 1,
      "porn" : 1,
      "jo" : 1,
      "ubank" : 1,
      "richardli" : 1,
      "beauty" : 1,
      "prime" : 1,
      "nissay" : 1,
      "sfr" : 1,
      "vet" : 1,
      "seat" : 1,
      "futbol" : 1,
      "sexy" : 1,
      "lds" : 1,
      "llp" : 1,
      "luxe" : 1,
      "ski" : 1,
      "promo" : 1,
      "mz" : 1,
      "quest" : 1,
      "trust" : 1,
      "lpl" : 1,
      "investments" : 1,
      "pa" : 1,
      "gov" : 1,
      "dealer" : 1,
      "webcam" : 1,
      "baby" : 1,
      "radio" : 1,
      "center" : 1,
      "wales" : 1,
      "lego" : 1,
      "associates" : 1,
      "exchange" : 1,
      "brussels" : 1,
      "cash" : 1,
      "tiaa" : 1,
      "de" : 1,
      "cruises" : 1,
      "moe" : 1,
      "attorney" : 1,
      "live" : 1,
      "hm" : 1,
      "games" : 1,
      "panasonic" : 1,
      "desi" : 1,
      "republican" : 1,
      "sn" : 1,
      "ai" : 1,
      "flights" : 1,
      "to" : 1,
      "salon" : 1,
      "ooo" : 1,
      "bloomberg" : 1,
      "auction" : 1,
      "ki" : 1,
      "nec" : 1,
      "yachts" : 1,
      "tel" : 1,
      "design" : 1,
      "builders" : 1,
      "hbo" : 1,
      "music" : 1,
      "systems" : 1,
      "moto" : 1,
      "tn" : 1,
      "blue" : 1,
      "so" : 1,
      "brother" : 1,
      "nyc" : 1,
      "market" : 1,
      "hsbc" : 1,
      "pics" : 1,
      "sncf" : 1,
      "crs" : 1,
      "gq" : 1,
      "nl" : 1,
      "corsica" : 1,
      "ice" : 1,
      "vig" : 1,
      "eat" : 1,
      "locus" : 1,
      "olayan" : 1,
      "drive" : 1,
      "amica" : 1,
      "ipiranga" : 1,
      "tci" : 1,
      "world" : 1,
      "cm" : 1,
      "grocery" : 1,
      "hughes" : 1,
      "theater" : 1,
      "boston" : 1,
      "hiphop" : 1,
      "mr" : 1,
      "fujitsu" : 1,
      "moscow" : 1,
      "lilly" : 1,
      "mt" : 1,
      "finance" : 1,
      "schule" : 1,
      "ro" : 1,
      "vg" : 1,
      "al" : 1,
      "lamborghini" : 1,
      "bt" : 1,
      "caravan" : 1,
      "amazon" : 1,
      "br" : 1,
      "toshiba" : 1,
      "legal" : 1,
      "paris" : 1,
      "auspost" : 1,
      "fashion" : 1,
      "lighting" : 1,
      "money" : 1,
      "fyi" : 1,
      "sport" : 1,
      "clinique" : 1,
      "bofa" : 1,
      "mortgage" : 1,
      "crown" : 1,
      "cpa" : 1,
      "bms" : 1,
      "bot" : 1,
      "tips" : 1,
      "bible" : 1,
      "jprs" : 1,
      "tattoo" : 1,
      "ong" : 1,
      "university" : 1,
      "realestate" : 1,
      "game" : 1,
      "bnpparibas" : 1,
      "realty" : 1,
      "gn" : 1,
      "lanxess" : 1,
      "ug" : 1,
      "icbc" : 1,
      "bridgestone" : 1,
      "xxx" : 1,
      "cologne" : 1,
      "tjmaxx" : 1,
      "pk" : 1,
      "booking" : 1,
      "security" : 1,
      "xerox" : 1,
      "horse" : 1,
      "cu" : 1,
      "hosting" : 1,
      "gop" : 1,
      "technology" : 1,
      "nra" : 1,
      "blog" : 1,
      "charity" : 1,
      "career" : 1,
      "ve" : 1,
      "yamaxun" : 1,
      "community" : 1,
      "dm" : 1,
      "virgin" : 1,
      "mn" : 1,
      "vlaanderen" : 1,
      "feedback" : 1,
      "saarland" : 1,
      "sz" : 1,
      "moda" : 1,
      "toys" : 1,
      "shopping" : 1,
      "williamhill" : 1,
      "aetna" : 1,
      "yun" : 1,
      "tax" : 1,
      "tt" : 1,
      "gold" : 1,
      "degree" : 1,
      "life" : 1,
      "ing" : 1,
      "pro" : 1,
      "tube" : 1,
      "uy" : 1,
      "supply" : 1,
      "tr" : 1,
      "lifeinsurance" : 1,
      "cf" : 1,
      "ryukyu" : 1,
      "cricket" : 1,
      "docs" : 1,
      "mq" : 1,
      "menu" : 1,
      "online" : 1,
      "ec" : 1,
      "food" : 1,
      "org" : 1,
      "got" : 1,
      "android" : 1,
      "stockholm" : 1,
      "gt" : 1,
      "mtn" : 1,
      "capetown" : 1,
      "lincoln" : 1,
      "gr" : 1,
      "place" : 1,
      "fi" : 1,
      "yandex" : 1,
      "gratis" : 1,
      "bn" : 1,
      "day" : 1,
      "zip" : 1,
      "abudhabi" : 1,
      "walter" : 1,
      "condos" : 1,
      "jot" : 1,
      "us" : 1,
      "bcg" : 1,
      "top" : 1,
      "allstate" : 1,
      "tickets" : 1,
      "ch" : 1,
      "gallup" : 1,
      "support" : 1,
      "kindle" : 1,
      "shouji" : 1,
      "diet" : 1,
      "zw" : 1,
      "aw" : 1,
      "trv" : 1,
      "tv" : 1,
      "rw" : 1,
      "sucks" : 1,
      "show" : 1,
      "pe" : 1,
      "py" : 1,
      "circle" : 1,
      "boehringer" : 1,
      "arte" : 1,
      "vacations" : 1,
      "catholic" : 1,
      "farmers" : 1,
      "nz" : 1,
      "rip" : 1,
      "fund" : 1,
      "kids" : 1,
      "onl" : 1,
      "wanggou" : 1,
      "vermögensberatung" : 1,
      "stream" : 1,
      "ovh" : 1,
      "tools" : 1,
      "winners" : 1,
      "phd" : 1,
      "catering" : 1,
      "homegoods" : 1,
      "space" : 1,
      "dvr" : 1,
      "insure" : 1,
      "adult" : 1,
      "scb" : 1,
      "bi" : 1,
      "philips" : 1,
      "party" : 1,
      "directory" : 1,
      "restaurant" : 1,
      "shia" : 1,
      "star" : 1,
      "prudential" : 1,
      "weather" : 1,
      "ps" : 1,
      "hockey" : 1,
      "broadway" : 1,
      "edu" : 1,
      "stada" : 1,
      "yoga" : 1,
      "photography" : 1,
      "ruhr" : 1,
      "industries" : 1,
      "tdk" : 1,
      "gripe" : 1,
      "law" : 1,
      "skin" : 1,
      "voting" : 1,
      "dog" : 1,
      "memorial" : 1,
      "jetzt" : 1,
      "click" : 1,
      "alstom" : 1,
      "gmail" : 1,
      "gbiz" : 1,
      "rent" : 1,
      "accenture" : 1,
      "bb" : 1,
      "volvo" : 1,
      "bbva" : 1,
      "tab" : 1,
      "jpmorgan" : 1,
      "eg" : 1,
      "kr" : 1,
      "gmx" : 1,
      "glass" : 1,
      "windows" : 1,
      "az" : 1,
      "sharp" : 1,
      "social" : 1,
      "sex" : 1,
      "mv" : 1,
      "ltd" : 1,
      "rehab" : 1,
      "ieee" : 1,
      "pwc" : 1,
      "sohu" : 1,
      "analytics" : 1,
      "is" : 1,
      "monash" : 1,
      "agency" : 1,
      "ca" : 1,
      "rwe" : 1,
      "goog" : 1,
      "sl" : 1,
      "biz" : 1,
      "hdfcbank" : 1,
      "network" : 1,
      "international" : 1,
      "rsvp" : 1,
      "ee" : 1,
      "bostik" : 1,
      "no" : 1,
      "loan" : 1,
      "nfl" : 1,
      "guitars" : 1,
      "fr" : 1,
      "audi" : 1,
      "gi" : 1,
      "nba" : 1,
      "redstone" : 1,
      "uno" : 1,
      "wien" : 1,
      "nissan" : 1,
      "domains" : 1,
      "site" : 1,
      "observer" : 1,
      "ntt" : 1,
      "extraspace" : 1,
      "cbre" : 1,
      "afl" : 1,
      "ao" : 1,
      "obi" : 1,
      "soy" : 1,
      "call" : 1,
      "parts" : 1,
      "lamer" : 1,
      "ltda" : 1,
      "ie" : 1,
      "arpa" : 1,
      "art" : 1,
      "voyage" : 1,
      "capitalone" : 1,
      "uk" : 1,
      "creditunion" : 1,
      "redumbrella" : 1,
      "progressive" : 1,
      "kn" : 1,
      "aero" : 1,
      "zero" : 1,
      "wtf" : 1,
      "love" : 1,
      "clothing" : 1,
      "pg" : 1,
      "rugby" : 1,
      "camp" : 1,
      "hk" : 1,
      "es" : 1,
      "band" : 1,
      "azure" : 1,
      "sj" : 1,
      "black" : 1,
      "wolterskluwer" : 1,
      "kerryproperties" : 1,
      "airbus" : 1,
      "homedepot" : 1,
      "zone" : 1,
      "cd" : 1,
      "rodeo" : 1,
      "lipsy" : 1,
      "hisamitsu" : 1,
      "farm" : 1,
      "bentley" : 1,
      "sale" : 1,
      "bv" : 1,
      "rich" : 1,
      "free" : 1,
      "chintai" : 1,
      "bar" : 1,
      "news" : 1,
      "locker" : 1,
      "gb" : 1,
      "vc" : 1
   }
}
