// 
//  Copyright (C) 1995,2007,2010  Smithsonian Astrophysical Observatory
//
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License along
//  with this program; if not, write to the Free Software Foundation, Inc.,
//  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
//

/*H****************************************************************************
* FILE NAME: %M%
 
* DEVELOPEMENT: UI
 
* DESCRIPTION:
 
        This file contains the class definition for the Host Level 
	Parameter interface. This class incapsulates the IRAF Compatible
	Parameter interface library into a class.
 
* NOTES:
 
        This class is based on the IRAF Compatible Parameter interface.
	This class does not support the structure option.
	This class is tuned to work directly with the ASCDS Parameter
	Dialog Box.

        This class utilizes the SAORD parameter interface library to
        perform reading of redirected values as well as updates to 
        existing parameter values. 

* REVISION HISTORY:
 
        Ref. No.        Date
        --------        ----
	%I%		%G%
 
*H****************************************************************************/

#include "Parameter.hh"
#include "parameter.h"
#include <unistd.h>
#include <string.h>
#include <stdlib.h>

#include <sstream>
#include <iostream>
#include <fstream>

using namespace std;


char *_baseName (char *path);


char *
_baseName (char *path)
{
  char *ptr;
  unsigned int i;

  // Perform some simple tests first
  if (path == NULL || strlen (path) == 0)
    return NULL;

  if (!strcmp (path, "/"))
    return path;

  // But what if the user passed in "/////////////"
  for (i = 0; path[i] == '/'; i++);
  if (i == strlen (path))
    return &path[strlen (path) - 1];

  // remove trailing slashes since string has more than just /'s
  for (i = strlen (path) - 1; path[strlen (path) - 1] == '/'; i--)
    path[strlen (path) - 1] = 0;

  if ((ptr = (char *) strrchr (path, '/')) == NULL)
    ptr = path;
  else
    ptr++;
  return ptr;
}


const int MAX_PARAM_LEN = 512;



//-----------------------------------------------------------------------
//              Initialize static class members
//-----------------------------------------------------------------------

// Names of each item w/in a parameter file entry
const char *
  Parameter::ITEMNAMES[] = {
  "Name",
  "CurrValue",
  "Default",
  "Prompt",
  "Type",
  "MinValue",
  "MaxValue",
  "Mode"
};

// How many items per parameter?
const int
  Parameter::ITEMNAME_COUNT = 8;

// Which field (eventually might want fields?) is editable
// (for indexing into above zero-based char**)
const int
  Parameter::EDITABLE_ITEM = 1;

// The default widths for displaying the values of each parameter item
const short
Parameter::ITEMNAME_WIDTHS[] = { 10, 32, 20, 20, 4, 10, 10, 4 };

// The maxiumn such widths
const int
Parameter::ITEMNAME_MAXWIDTHS[] = { 64, 250, 250, 150, 4, 150, 150, 4 };


//-----------------------------------------------------------------------


// Constructor
Parameter::Parameter ()
{
  read_only = true;
}

Parameter::Parameter (string pfname, int argc, char **argv, int arg1)
{
  start_arg = arg1;
  filename = pfname;
  _fileRead = "\0";
  param_path = NULL;
  read_only = true;

  Load (argc, argv);
}

Parameter::Parameter (int argc, char **argv)
{
  param_path = NULL;
  filename = argv[0];
  filename.append (".par");
  read_only = true;
  for (int i = 0; i < argc; i++)
    {
      if (strcmp (argv[i], "-pfile") == 0)
	{
	  filename = argv[++i];
	  break;
	}
    }
  _fileRead = "\0";
  start_arg = 1;
  Load (argc, argv);
}

// Destructor
Parameter::~Parameter ()
{
  if (param_path)
    {
      delete param_path;
      param_path = NULL;
    }
}

// Opens and reads the parameter contents into memory. An error code 
// of 0 is returned if an error occurs.
int
Parameter::Load (int argc, char **argv)
{
  int error = 0;
  paramfile parfile = NULL;

  // open the parameter file with read/write access 
  if ((parfile =
       paramopen ((char *) filename.c_str (), NULL, 0, "rwH")) == NULL)
    {
      // if the open failed, the user may have explicitly attempted to
      // open a file which only has read access- try opening the file
      // read-only
      parfile = paramopen ((char *) filename.c_str (), NULL, 0, "rH");
    }
  else
    {
      read_only = false;
    }

  if (parfile)
    {
      char *temp_path = paramgetpath (parfile);

      if (temp_path)
	{
	  // read the par file as ascii text
	  fstream infile (temp_path, ios::in);

	  if (infile)
	    {
	      // disable auto-exit for errors occurring when in H mode
	      set_paramerror (0);

	      // Read the paremeter file
	      Read (infile, &parfile, temp_path);
	      _fileRead = temp_path;

	      // Set the values that were entered on the commmand line
	      SetValues (argc, argv);

	      if (param_path)
		{
		  delete param_path;
		}
	      param_path = new string (temp_path);

	      error = 1;
	    }
	  free (temp_path);
	}

      // close the param file
      paramclose (parfile);
    }

  return error;
}

// Returns a string containing param1=value1 param2=value2 ...
string Parameter::GetValues (int evaluate)
{
  string
    result;
  result.append (GetBooleanValues (evaluate));
  result.append (GetIntegerValues (evaluate));
  result.append (GetRealValues (evaluate));
  result.append (GetStringValues (evaluate));
  return result;
}

// Returns the values of the boolean parameters
string Parameter::GetBooleanValues (int evaluate)
{
  string
    result,
    redir;
  list < ParamBoolean >::iterator it;

  for (it = bool_list.begin (); it != bool_list.end (); it++)
    {
      redir = (*it).GetRedirectCurrent ();
      if (!evaluate && redir.length ())
	{
	  int
	    quote = (redir.find ('"') == string::npos);

	  result.append ((*it).GetName ());
	  result.append ("=");
	  if (quote)
	    result.append ("\"");
	  result.append (redir);
	  if (quote)
	    result.append ("\"");
	}
      else
	result.append ((*it).GetNewValue ());
      result.append (" ");
    }
  return result;
}

string Parameter::GetIntegerValues (int evaluate)
{
  string
    result,
    redir;
  list < ParamInteger >::iterator it;
  for (it = int_list.begin (); it != int_list.end (); it++)
    {
      redir = (*it).GetRedirectCurrent ();
      if (!evaluate && redir.length ())
	{
	  int
	    quote = (redir.find ('"') == string::npos);

	  result.append ((*it).GetName ());
	  result.append ("=");
	  if (quote)
	    result.append ("\"");
	  result.append (redir);
	  if (quote)
	    result.append ("\"");
	}
      else
	result.append ((*it).GetNewValue ());
      result.append (" ");
    }
  return result;
}

string Parameter::GetRealValues (int evaluate)
{
  string
    result,
    redir;
  list < ParamReal >::iterator it;
  for (it = real_list.begin (); it != real_list.end (); it++)
    {
      redir = (*it).GetRedirectCurrent ();
      if (!evaluate && redir.length ())
	{
	  int
	    quote = (redir.find ('"') == string::npos);

	  result.append ((*it).GetName ());
	  result.append ("=");
	  if (quote)
	    result.append ("\"");
	  result.append (redir);
	  if (quote)
	    result.append ("\"");
	}
      else
	result.append ((*it).GetNewValue ());
      result.append (" ");
    }
  return result;
}

string Parameter::GetStringValues (int evaluate)
{
  string
    result,
    redir;
  list < ParamString >::iterator it;
  for (it = str_list.begin (); it != str_list.end (); it++)
    {
      redir = (*it).GetRedirectCurrent ();
      if (!evaluate && redir.length ())
	{
	  int
	    quote = (redir.find ('"') == string::npos);

	  result.append ((*it).GetName ());
	  result.append ("=");
	  if (quote)
	    result.append ("\"");
	  result.append (redir);
	  if (quote)
	    result.append ("\"");
	}
      else
	result.append ((*it).GetNewValue ());
      result.append (" ");
    }
  return result;
}

// Returns the changes made to the parameter file
string Parameter::GetChanges ()
{
  string
    result;
  result.append (GetBooleanChanges ());
  result.append (GetIntegerChanges ());
  result.append (GetRealChanges ());
  result.append (GetStringChanges ());
  return result;
}

string Parameter::GetParamPath ()
{
  string
    result = "\0";
  if (param_path)
    result = param_path->c_str ();
  return result;
}


string Parameter::EvalIndir (char *param, char *val)
{
  string
    result = "\0";
  paramfile
    pf;

  if (param && val && val[0] == ')')
    {
      char *retval = NULL;

      if (param_path)
	pf = paramopen ((char *) param_path->c_str (), NULL, 0, "rH");
      else if (filename.length () > 0)
	pf = paramopen ((char *) filename.c_str (), NULL, 0, "rH");
      else
	pf = NULL;

      if (pf)
	{
	  retval = evaluateIndir (pf, param, val);
	  paramclose (pf);
	}
      if (retval)
	result = retval;
    }
  return result;
}

// Returns the changes made to the boolean parameters
string Parameter::GetBooleanChanges ()
{
  string
    result;
  list < ParamBoolean >::iterator it;

  for (it = bool_list.begin (); it != bool_list.end (); it++)
    {
      if ((*it).GetValue () != (*it).GetCurrent ())
	{
	  result.append ((*it).GetNewValue ());
	  result.append (" ");
	}
    }
  return result;
}

// Returns the changes made to integer the parameters
string Parameter::GetIntegerChanges ()
{
  string
    result;
  list < ParamInteger >::iterator it;
  for (it = int_list.begin (); it != int_list.end (); it++)
    {
      if ((*it).GetValue () != (*it).GetCurrent ())
	{
	  result.append ((*it).GetNewValue ());
	  result.append (" ");
	}
    }
  return result;
}

// Returns the changes made to real the parameters
string Parameter::GetRealChanges ()
{
  string
    result;
  list < ParamReal >::iterator it;
  for (it = real_list.begin (); it != real_list.end (); it++)
    {
      if ((*it).GetValue () != (*it).GetCurrent ())
	{
	  result.append ((*it).GetNewValue ());
	  result.append (" ");
	}
    }
  return result;
}

// Returns the changes made to string the parameters
string Parameter::GetStringChanges ()
{
  string
    result;
  list < ParamString >::iterator it;
  for (it = str_list.begin (); it != str_list.end (); it++)
    {
      if ((*it).GetValue () != (*it).GetCurrent ())
	{
	  result.append ((*it).GetNewValue ());
	  result.append (" ");
	}
    }
  return result;
}

// Writes out the parameter file. An error code of 0 is returned if an 
// error occurs.
int
Parameter::Write (string fname, int updateHidden)
{
  int ii;
  char toolstring[256];
  char *toolname = NULL;
  int status = 0;
  paramfile parfile = NULL;

  strcpy (toolstring, fname.c_str ());
  if ((toolname = _baseName (toolstring)) != NULL)
    {
      string mode = "rwH";
      if (updateHidden)
	mode.append ("L");

      // open the parameter file 
      if ((parfile =
	   paramopen (toolname, NULL, 0, (char *) mode.c_str ())) != 0)
	{
	  char *temp_path = paramgetpath (parfile);
	  if (temp_path)
	    {
	      if (param_path)
		{
		  delete param_path;
		}
	      param_path = new string (temp_path);
	      //cout << "write to " <<  temp_path << endl; 
	      free (temp_path);
	    }

	  // iterate through all parameters in the file 
	  for (ii = 0; ii < GetLength (); ii++)
	    {
	      bool found = false;

	      // check if current parameter is in the integer list 
	      list < ParamInteger >::iterator iit;
	      for (iit = int_list.begin (); iit != int_list.end (); iit++)
		{
		  if ((found = ((*iit).GetPosition () == ii)))
		    {
		      // has the value changed? 
		      if (((*iit).RedirectChange ()) ||
			  ((*iit).Changed ()) ||
			  ((*iit).GetCurrent () != (*iit).GetValue ()))
			{
			  if (paccess (parfile,
				       (char *) ((string) (*iit).GetName ()).
				       c_str ()))
			    {
			      char *temp = (*iit).GetRedirectCurrent ();

			      if (*temp == ')')
				{
				  pputstr (parfile,
					   (char *) ((string) (*iit).
						     GetName ()).c_str (),
					   temp);
				}
			      else if ((*iit).GetCurrent () == INDEF_INT)
				{
				  pputstr (parfile, (char *) ((string)
							      (*iit).
							      GetName ()).
					   c_str (), (char *) "INDEF");
				}

			      else
				{
				  pputi (parfile,
					 (char *) ((string) (*iit).
						   GetName ()).c_str (),
					 (*iit).GetCurrent ());
				}
			    }
			}
		    }
		}		// end for (iit) 

	      // check if current parameter is in the boolean list 
	      list < ParamBoolean >::iterator bit;
	      for (bit = bool_list.begin (); bit != bool_list.end (); bit++)
		{
		  if ((found = ((*bit).GetPosition () == ii)))
		    {
		      // has the value changed? 
		      if (((*bit).RedirectChange ()) ||
			  ((*bit).Changed ()) ||
			  ((*bit).GetCurrent () != (*bit).GetValue ()))
			{
			  if (paccess (parfile,
				       (char *) ((string) (*bit).GetName ()).
				       c_str ()))
			    {
			      char *temp = (*bit).GetRedirectCurrent ();

			      if (*temp == ')')
				{
				  pputstr (parfile,
					   (char *) ((string) (*bit).
						     GetName ()).c_str (),
					   temp);
				}
			      else
				{
				  pputb (parfile,
					 (char *) ((string) (*bit).
						   GetName ()).c_str (),
					 (*bit).GetCurrent ());
				}
			    }
			}
		    }
		}		// end for (bit) 

	      // check if current parameter is in the real list 
	      list < ParamReal >::iterator rit;
	      for (rit = real_list.begin (); rit != real_list.end (); rit++)
		{
		  if ((found = ((*rit).GetPosition () == ii)))
		    {
		      // has the value changed? 
		      if (((*rit).RedirectChange ()) ||
			  ((*rit).Changed ()) ||
			  ((*rit).GetCurrent () != (*rit).GetValue ()))
			{
			  if (paccess (parfile,
				       (char *) ((string) (*rit).GetName ()).
				       c_str ()))
			    {
			      char *temp = (*rit).GetRedirectCurrent ();

			      if (*temp == ')')
				{
				  pputstr (parfile,
					   (char *) ((string) (*rit).
						     GetName ()).c_str (),
					   temp);
				}
			      else if ((*rit).GetCurrent () == INDEF_REAL)
				{
				  pputstr (parfile, (char *) ((string)
							      (*rit).
							      GetName ()).
					   c_str (), (char *) "INDEF");
				}
			      else
				{
				  pputd (parfile,
					 (char *) ((string) (*rit).
						   GetName ()).c_str (),
					 (*rit).GetCurrent ());
				}
			    }
			}
		    }
		}		// end for (rit)

	      // check if current parameter is in the string list 
	      list < ParamString >::iterator sit;
	      for (sit = str_list.begin (); sit != str_list.end (); sit++)
		{
		  if ((found = ((*sit).GetPosition () == ii)))
		    {
		      // has the value changed? 
		      if ((strcmp (((string) (*sit).GetCurrent ()).c_str (),
				   ((string) (*sit).GetValue ()).c_str ()) !=
			   0) || ((*sit).RedirectChange ())
			  || ((*sit).Changed ()))
			{
			  if (paccess (parfile,
				       (char *) ((string) (*sit).GetName ()).
				       c_str ()))
			    {
			      int len;
			      char *temp_value;
			      char *temp = (*sit).GetRedirectCurrent ();

			      if (*temp == ')')
				{
				  pputstr (parfile,
					   (char *) ((string) (*sit).
						     GetName ()).c_str (),
					   temp);
				}
			      else
				{
				  temp_value = (char *) malloc (((len =
								  ((string)
								   (*sit).
								   GetCurrent
								   ()).
								  length ()) +
								 1) *
								sizeof
								(char));
				  strcpy (temp_value,
					  ((string) (*sit).GetCurrent ()).
					  c_str ());
				  if ((temp_value[0] == '"')
				      && (temp_value[len - 1] == '"'))
				    {
				      char *adj_value;
				      adj_value =
					(char *) malloc ((len) *
							 sizeof (char));
				      strcpy (adj_value, &temp_value[1]);
				      adj_value[len - 2] = '\0';
				      pputstr (parfile,
					       (char *) ((string) (*sit).
							 GetName ()).c_str (),
					       adj_value);
				      free (adj_value);
				    }
				  else
				    {
				      pputstr (parfile,
					       (char *) ((string) (*sit).
							 GetName ()).c_str (),
					       temp_value);
				    }
				  free (temp_value);
				}
			    }
			}
		    }
		}		// end for (sit)
	    }			// end for (ii)  
	  paramclose (parfile);
	  status = 1;
	}
      else
	{
	  // parameter file open failed- error message ? 
	}
    }
  return (status);
}

// Save As - write the input parameter file to a new file. 
// An error code of 0 is returned if an error occurs.
int
Parameter::SaveAs (char *fname, int overwrite, int updateHidden)
{
  int ii;
  int status = 0;
  paramfile parfile = NULL;
  string mode = "rwH";

  if (!fname)
    return status;
  if ((access (fname, F_OK) == 0) && !overwrite)
    return status;
  if (!_fileRead.length ())
    return status;
  if (updateHidden)
    mode.append ("L");

  // create copy of file 
  if (_fileRead != fname)
    {
      ifstream in ((char *) _fileRead.c_str ());
      ofstream out (fname);
      out << in.rdbuf ();
      in.close ();
      out.close ();
    }
  else
    {
      string tmpn;
      char *tn = getenv ((char *) "ASCDS_WORK_PATH");
      if (tn)
	tmpn = tn;
      else
	{
	  tn = getenv ((char *) "ASCDS_TMP");
	  if (tn)
	    tmpn = tn;
	  else
	    tn = (char *) "/tmp";
	}

      char *tmpnam = tempnam (NULL, NULL);
      tmpn.append ((char *) "/.PEG.");
      tmpn.append (_baseName (tmpnam));
      free (tmpnam);

      ifstream in ((char *) _fileRead.c_str ());
      ofstream tmp ((char *) tmpn.c_str ());
      tmp << in.rdbuf ();
      in.close ();
      tmp.close ();
      ifstream intmp ((char *) tmpn.c_str ());
      ofstream out (fname);
      out << intmp.rdbuf ();
      intmp.close ();
      out.close ();
      string cmd = "/bin/rm -f ";
      cmd.append (tmpn);
      cmd.append (" &");
      system (cmd.c_str ());
    }

  // open the copy parameter file 
  if ((parfile = paramopen (fname, NULL, 0, (char *) mode.c_str ())) != 0)
    {
      // iterate through all parameters in the file 
      for (ii = 0; ii < GetLength (); ii++)
	{
	  bool found = false;

	  // check if current parameter is in the integer list 
	  list < ParamInteger >::iterator iit;
	  for (iit = int_list.begin (); iit != int_list.end (); iit++)
	    {
	      if ((found = ((*iit).GetPosition () == ii)))
		{
		  // has the value changed? 
		  if (((*iit).RedirectChange ()) ||
		      ((*iit).GetCurrent () != (*iit).GetValue ()))
		    {
		      if (paccess (parfile,
				   (char *) ((string) (*iit).GetName ()).
				   c_str ()))
			{
			  char *temp = (*iit).GetRedirectCurrent ();

			  if (*temp == ')')
			    {
			      pputstr (parfile,
				       (char *) ((string) (*iit).GetName ()).
				       c_str (), temp);
			    }
			  else if ((*iit).GetCurrent () == INDEF_INT)
			    {
			      pputstr (parfile, (char *) ((string)
							  (*iit).GetName ()).
				       c_str (), (char *) "INDEF");
			    }

			  else
			    {
			      pputi (parfile,
				     (char *) ((string) (*iit).GetName ()).
				     c_str (), (*iit).GetCurrent ());
			    }
			}
		    }
		}
	    }			// end for (iit) 

	  // check if current parameter is in the boolean list 
	  list < ParamBoolean >::iterator bit;
	  for (bit = bool_list.begin (); bit != bool_list.end (); bit++)
	    {
	      if ((found = ((*bit).GetPosition () == ii)))
		{
		  // has the value changed? 
		  if (((*bit).RedirectChange ()) ||
		      ((*bit).GetCurrent () != (*bit).GetValue ()))
		    {
		      if (paccess (parfile,
				   (char *) ((string) (*bit).GetName ()).
				   c_str ()))
			{
			  char *temp = (*bit).GetRedirectCurrent ();

			  if (*temp == ')')
			    {
			      pputstr (parfile,
				       (char *) ((string) (*bit).GetName ()).
				       c_str (), temp);
			    }
			  else
			    {
			      pputb (parfile,
				     (char *) ((string) (*bit).GetName ()).
				     c_str (), (*bit).GetCurrent ());
			    }
			}
		    }
		}
	    }			// end for (bit) 

	  // check if current parameter is in the real list 
	  list < ParamReal >::iterator rit;
	  for (rit = real_list.begin (); rit != real_list.end (); rit++)
	    {
	      if ((found = ((*rit).GetPosition () == ii)))
		{
		  // has the value changed? 
		  if (((*rit).RedirectChange ()) ||
		      ((*rit).GetCurrent () != (*rit).GetValue ()))
		    {
		      if (paccess (parfile,
				   (char *) ((string) (*rit).GetName ()).
				   c_str ()))
			{
			  char *temp = (*rit).GetRedirectCurrent ();

			  if (*temp == ')')
			    {
			      pputstr (parfile,
				       (char *) ((string) (*rit).GetName ()).
				       c_str (), temp);
			    }
			  else if ((*rit).GetCurrent () == INDEF_REAL)
			    {
			      pputstr (parfile, (char *) ((string)
							  (*rit).GetName ()).
				       c_str (), (char *) "INDEF");
			    }
			  else
			    {
			      pputd (parfile,
				     (char *) ((string) (*rit).GetName ()).
				     c_str (), (*rit).GetCurrent ());
			    }
			}
		    }
		}
	    }			// end for (rit)

	  // check if current parameter is in the string list 
	  list < ParamString >::iterator sit;
	  for (sit = str_list.begin (); sit != str_list.end (); sit++)
	    {
	      if ((found = ((*sit).GetPosition () == ii)))
		{
		  // has the value changed? 
		  if ((strcmp (((string) (*sit).GetCurrent ()).c_str (),
			       ((string) (*sit).GetValue ()).c_str ()) != 0)
		      || ((*sit).RedirectChange ()))
		    {
		      if (paccess (parfile,
				   (char *) ((string) (*sit).GetName ()).
				   c_str ()))
			{
			  int len;
			  char *temp_value;
			  char *temp = (*sit).GetRedirectCurrent ();

			  if (*temp == ')')
			    {
			      pputstr (parfile,
				       (char *) ((string) (*sit).GetName ()).
				       c_str (), temp);
			    }
			  else
			    {
			      temp_value = (char *) malloc (((len =
							      ((string)
							       (*sit).
							       GetCurrent ()).
							      length ()) +
							     1) *
							    sizeof (char));
			      strcpy (temp_value,
				      ((string) (*sit).GetCurrent ()).
				      c_str ());
			      if ((temp_value[0] == '"')
				  && (temp_value[len - 1] == '"'))
				{
				  char *adj_value;
				  adj_value =
				    (char *) malloc ((len) * sizeof (char));
				  strcpy (adj_value, &temp_value[1]);
				  adj_value[len - 2] = '\0';
				  pputstr (parfile,
					   (char *) ((string) (*sit).
						     GetName ()).c_str (),
					   adj_value);
				  free (adj_value);
				}
			      else
				{
				  pputstr (parfile,
					   (char *) ((string) (*sit).
						     GetName ()).c_str (),
					   temp_value);
				}
			      free (temp_value);
			    }
			}
		    }
		}
	    }			// end for (sit)
	}			// end for (ii)  
      paramclose (parfile);
      status = 1;
    }
  return (status);
}


// Sets the values from the command line
void
Parameter::SetValues (int argc, char **argv)
{
  for (int i = start_arg; i < argc; i++)
    {
      // Parse the parameter
      string param = argv[i];
      int pos, llen, check = 0;

      if ((llen = param.length ()) > 1)
	{
	  string name, value;
	  if ((pos = param.find ('=')) > 0)
	    {
	      check = 1;
	      name = param.substr (0, pos);
	      value = param.substr (pos + 1);
	    }
	  else if ((pos = param.find ('+')) == (llen - 1))
	    {
	      check = 2;
	      name = param.substr (0, pos);
	      value = "yes";
	    }
	  else if ((pos = param.find ('-')) == (llen - 1))
	    {
	      check = 3;
	      name = param.substr (0, pos);
	      value = "no";
	    }

	  if (check)
	    {
	      int found = 0;

	      // Check the boolean list
	      list < ParamBoolean >::iterator bit;
	      for (bit = bool_list.begin (); bit != bool_list.end (); bit++)
		{
		  if ((*bit).GetName () == name)
		    {
		      (*bit).SetValue (value);
		      found++;
		    }
		}

	      if (check == 1)
		{
		  if (!found)
		    {
		      // Check the integer list
		      list < ParamInteger >::iterator iit;
		      for (iit = int_list.begin (); iit != int_list.end ();
			   iit++)
			{
			  if ((*iit).GetName () == name)
			    {
			      (*iit).SetValue (value);
			      found++;
			    }
			}
		    }

		  if (!found)
		    {
		      // Check the real list
		      list < ParamReal >::iterator rit;
		      for (rit = real_list.begin (); rit != real_list.end ();
			   rit++)
			{
			  if ((*rit).GetName () == name)
			    {
			      (*rit).SetValue (value);
			      found++;
			    }
			}
		    }

		  if (!found)
		    {
		      // Check the string list
		      list < ParamString >::iterator sit;
		      for (sit = str_list.begin (); sit != str_list.end ();
			   sit++)
			{
			  if ((*sit).GetName () == name)
			    {
			      (*sit).SetValue (value);
			      found++;
			    }
			}
		    }
		}		// end if(check == 1) 
	    }
	}
    }
}

// Returns the number parameters in the list
int
Parameter::GetLength ()
{
  int count = 0;
  count += bool_list.size ();
  count += int_list.size ();
  count += real_list.size ();
  count += str_list.size ();
  return count;
}



bool Parameter::GetParameterByName (string name, ParamItem & item)
{
  return GetParameterByName (name.c_str (), item);
}

// ----------------------------------------
// Return parameter for given name - note: This routine does not work for 
// parameters which contain string values. GetParameterByName(const char*,
// ParamItem** should be used instead of this routine.
// ----------------------------------------
bool Parameter::GetParameterByName (const char *name, ParamItem & item)
{
  string
    str;

  list < ParamBoolean >::iterator bit;
  for (bit = bool_list.begin (); bit != bool_list.end (); bit++)
    {
      str = (*bit).GetName ();
      if (strcmp (str.c_str (), name) == 0)
	{
	  ParamBoolean *
	    bitem = (ParamBoolean *) & item;
	  *bitem = (*bit);
	  return true;
	}
    }

  list < ParamInteger >::iterator iit;
  for (iit = int_list.begin (); iit != int_list.end (); iit++)
    {
      str = (*iit).GetName ();
      if (strcmp (str.c_str (), name) == 0)
	{
	  ParamInteger *
	    iitem = (ParamInteger *) & item;
	  *iitem = (*iit);
	  return true;
	}
    }

  list < ParamReal >::iterator rit;
  for (rit = real_list.begin (); rit != real_list.end (); rit++)
    {
      str = (*rit).GetName ();
      if (strcmp (str.c_str (), name) == 0)
	{
	  ParamReal *
	    ritem = (ParamReal *) & item;
	  *ritem = (*rit);
	  return true;
	}
    }

  list < ParamString >::iterator sit;
  for (sit = str_list.begin (); sit != str_list.end (); sit++)
    {
      str = (*sit).GetName ();
      if (strcmp (str.c_str (), name) == 0)
	{
	  ParamString *
	    sitem = (ParamString *) & item;
	  *sitem = (*sit);
	  return true;
	}
    }

  return false;
}

bool Parameter::GetParameterByName (string name, ParamItem ** item_p)
{
  return GetParameterByName (name.c_str (), item_p);
}

// ----------------------------------------
// Return parameter for given name - note: This routine instantiates
// a new ParamItem object which should be freed by the calling routine.
// ----------------------------------------
bool Parameter::GetParameterByName (const char *name, ParamItem ** item_p)
{
  string
    str;

  list < ParamBoolean >::iterator bit;
  for (bit = bool_list.begin (); bit != bool_list.end (); bit++)
    {
      str = (*bit).GetName ();
      if (strcmp (str.c_str (), name) == 0)
	{
	  *item_p = new ParamBoolean ((*bit));
	  return true;
	}
    }

  list < ParamInteger >::iterator iit;
  for (iit = int_list.begin (); iit != int_list.end (); iit++)
    {
      str = (*iit).GetName ();
      if (strcmp (str.c_str (), name) == 0)
	{
	  *item_p = new ParamInteger ((*iit));
	  return true;
	}
    }

  list < ParamReal >::iterator rit;
  for (rit = real_list.begin (); rit != real_list.end (); rit++)
    {
      str = (*rit).GetName ();
      if (strcmp (str.c_str (), name) == 0)
	{
	  *item_p = new ParamReal ((*rit));
	  return true;
	}
    }

  list < ParamString >::iterator sit;
  for (sit = str_list.begin (); sit != str_list.end (); sit++)
    {
      str = (*sit).GetName ();
      if (strcmp (str.c_str (), name) == 0)
	{
	  *item_p = new ParamString ((*sit));
	  return true;
	}
    }
  return false;
}

// Sets a new value in the parameter at position. A 0 is returned
// if an error was encountered.
int
Parameter::SetValueAtPosition (int pos, string value)
{

  // determine if the position falls in the boolean list
  list < ParamBoolean >::iterator bit;
  for (bit = bool_list.begin (); bit != bool_list.end (); bit++)
    {
      if ((*bit).GetPosition () == pos)
	return (*bit).SetValue (value);
    }

  // detemine if the position falls in the integer list
  list < ParamInteger >::iterator iit;
  for (iit = int_list.begin (); iit != int_list.end (); iit++)
    {
      if ((*iit).GetPosition () == pos)
	return (*iit).SetValue (value);
    }

  // detemine if the position falls in the real list
  list < ParamReal >::iterator rit;
  for (rit = real_list.begin (); rit != real_list.end (); rit++)
    {
      if ((*rit).GetPosition () == pos)
	return (*rit).SetValue (value);
    }

  // detemine if the position falls in the string list
  list < ParamString >::iterator sit;
  for (sit = str_list.begin (); sit != str_list.end (); sit++)
    {
      if ((*sit).GetPosition () == pos)
	return (*sit).SetValue (value);
    }

  return 0;
}

// Read the parameter file from the input stream
void
Parameter::Read (istream & iss, paramfile * parfile)
{
  int ii = 0;

  while (!iss.eof ())
    {

      // Read a line from the line
      char temp[MAX_PARAM_LEN];

      iss.getline (temp, MAX_PARAM_LEN);

      // Determine if '#' is the first character in the line.
      string test = temp;
//WLM      test.remove_leading_white_space();
      if (test.find ('#') != 0)
	{
	  // The base parameter information
	  istringstream issitem (temp);
	  ParamItem item;
	  issitem >> item;
	  item.SetPosition (ii);

	  item.SetParam (parfile);


	  // Determine what type of parameter was just read
	  switch (item.GetType ())
	    {
	    case ParamItem::BOOLEAN_TYPE:
	      {
		ParamBoolean value (item);
		issitem >> value;
		if (value.IsValid ())
		  {
		    bool_list.push_back (value);
		    ii++;
		  }
		break;
	      }
	    case ParamItem::INTEGER_TYPE:
	      {
		ParamInteger value (item);
		issitem >> value;
		if (value.IsValid ())
		  {
		    int_list.push_back (value);
		    ii++;
		  }
		break;
	      }
	    case ParamItem::REAL_TYPE:
	      {
		ParamReal value (item);
		issitem >> value;
		if (value.IsValid ())
		  {
		    real_list.push_back (value);
		    ii++;
		  }
		break;
	      }
	    case ParamItem::STRING_TYPE:
	    case ParamItem::FILENAME_TYPE:
	    case ParamItem::PSET_TYPE:
	      {
		ParamString value (item);
		issitem >> value;
		if (value.IsValid ())
		  {
		    str_list.push_back (value);
		    ii++;
		  }
		break;
	      }
	    case ParamItem::NO_PARAM_TYPE:	// fallthrough intended
	    default:
	      // do nothing
	      break;

	    }
	}
    }
}

// Read the parameter file from the input stream
void
Parameter::Read (istream & iss, paramfile * parfile, char *filenam)
{
  int ii = 0;

  while (!iss.eof ())
    {
      char temp[MAX_PARAM_LEN];

      // Read a line from the line
      iss.getline (temp, MAX_PARAM_LEN);

      // Determine if '#' is the first character in the line.
      string test = temp;
//WLM      test.remove_leading_white_space();
      if (test.find ('#') != 0)
	{
	  // The base parameter information
	  istringstream issitem (temp);
	  ParamItem item;
	  issitem >> item;
	  item.SetPosition (ii);

	  item.SetParam (parfile);
	  item.SetFilename (filenam);


	  // Determine what type of parameter was just read
	  switch (item.GetType ())
	    {
	    case ParamItem::BOOLEAN_TYPE:
	      {
		ParamBoolean value (item);
		issitem >> value;
		if (value.IsValid ())
		  {
		    bool_list.push_back (value);
		    ii++;
		  }
		break;
	      }
	    case ParamItem::INTEGER_TYPE:
	      {
		ParamInteger value (item);
		issitem >> value;
		if (value.IsValid ())
		  {
		    int_list.push_back (value);
		    ii++;
		  }
		break;
	      }
	    case ParamItem::REAL_TYPE:
	      {
		ParamReal value (item);
		issitem >> value;
		if (value.IsValid ())
		  {
		    real_list.push_back (value);
		    ii++;
		  }
		break;
	      }
	    case ParamItem::STRING_TYPE:
	    case ParamItem::FILENAME_TYPE:
	    case ParamItem::PSET_TYPE:
	      {
		ParamString value (item);
		issitem >> value;
		if (value.IsValid ())
		  {
		    str_list.push_back (value);
		    ii++;
		  }
		break;
	      }
	    case ParamItem::NO_PARAM_TYPE:	// fallthrough intended
	    default:
	      // do nothing
	      break;
	    }
	}
    }
}


// Read the parameter file from the input stream
void
Parameter::Read (istream & iss)
{
  int ii = 0;

  while (!iss.eof ())
    {
      char temp[MAX_PARAM_LEN];

      // Read a line from the line
      iss.getline (temp, MAX_PARAM_LEN);

      // Determine if '#' is the first character in the line.
      string test = temp;
//WLM      test.remove_leading_white_space();
      if (test.find ('#') != 0)
	{
	  // The base parameter information
	  istringstream issitem (temp);
	  ParamItem item;
	  issitem >> item;
	  item.SetPosition (ii);

	  // Determine what type of parameter was just read
	  switch (item.GetType ())
	    {
	    case ParamItem::BOOLEAN_TYPE:
	      {
		ParamBoolean value (item);
		issitem >> value;
		if (value.IsValid ())
		  {
		    bool_list.push_back (value);
		    ii++;
		  }
		break;
	      }
	    case ParamItem::INTEGER_TYPE:
	      {
		ParamInteger value (item);
		issitem >> value;
		if (value.IsValid ())
		  {
		    int_list.push_back (value);
		    ii++;
		  }
		break;
	      }
	    case ParamItem::REAL_TYPE:
	      {
		ParamReal value (item);
		issitem >> value;
		if (value.IsValid ())
		  {
		    real_list.push_back (value);
		    ii++;
		  }
		break;
	      }
	    case ParamItem::STRING_TYPE:
	    case ParamItem::FILENAME_TYPE:
	    case ParamItem::PSET_TYPE:
	      {
		ParamString value (item);
		issitem >> value;
		if (value.IsValid ())
		  {
		    str_list.push_back (value);
		    ii++;
		  }
		break;
	      }
	    case ParamItem::NO_PARAM_TYPE:	// fallthrough intended
	    default:
	      // do nothing
	      break;
	    }
	}
    }
}

// Print the parameter file to the output stream
void
Parameter::Print (ostream & oss)
{
  int ii;

  for (ii = 0; ii < GetLength (); ii++)
    {
      list < ParamBoolean >::iterator bit;
      for (bit = bool_list.begin (); bit != bool_list.end (); bit++)
	if ((*bit).GetPosition () == ii)
	  {
	    oss << (*bit) << endl;
	    continue;
	  }

      list < ParamInteger >::iterator iit;
      for (iit = int_list.begin (); iit != int_list.end (); iit++)
	if ((*iit).GetPosition () == ii)
	  {
	    oss << (*iit) << endl;
	    continue;
	  }

      list < ParamReal >::iterator rit;
      for (rit = real_list.begin (); rit != real_list.end (); rit++)
	if ((*rit).GetPosition () == ii)
	  {
	    oss << (*rit) << endl;
	    continue;
	  }

      list < ParamString >::iterator sit;
      for (sit = str_list.begin (); sit != str_list.end (); sit++)
	if ((*sit).GetPosition () == ii)
	  {
	    oss << (*sit) << endl;
	    continue;
	  }
    }
}

// Reads a parameter and saves it to the linked list.
int
Parameter::ReadParam (string param)
{
  return 0;
}

// Get characters separated by a comma for the given position
char **
Parameter::GetCharsByPosition (int pos)
{
  list < ParamBoolean >::iterator bit;
  for (bit = bool_list.begin (); bit != bool_list.end (); bit++)
    if ((*bit).GetPosition () == pos)
      {
	return ((*bit).GetChars ());
      }

  list < ParamInteger >::iterator iit;
  for (iit = int_list.begin (); iit != int_list.end (); iit++)
    if ((*iit).GetPosition () == pos)
      {
	return ((*iit).GetChars ());
      }

  list < ParamReal >::iterator rit;
  for (rit = real_list.begin (); rit != real_list.end (); rit++)
    if ((*rit).GetPosition () == pos)
      {
	return ((*rit).GetChars ());
      }

  list < ParamString >::iterator sit;
  for (sit = str_list.begin (); sit != str_list.end (); sit++)
    if ((*sit).GetPosition () == pos)
      {
	return ((*sit).GetChars ());
      }

  return NULL;

}

// get widths of parameter item for a given position
short *
Parameter::GetWidthsByPosition (int pos)
{
  list < ParamBoolean >::iterator bit;
  for (bit = bool_list.begin (); bit != bool_list.end (); bit++)
    if ((*bit).GetPosition () == pos)
      {
	return ((*bit).GetWidths ());
      }

  list < ParamInteger >::iterator iit;
  for (iit = int_list.begin (); iit != int_list.end (); iit++)
    if ((*iit).GetPosition () == pos)
      {
	return ((*iit).GetWidths ());
      }

  list < ParamReal >::iterator rit;
  for (rit = real_list.begin (); rit != real_list.end (); rit++)
    if ((*rit).GetPosition () == pos)
      {
	return ((*rit).GetWidths ());
      }

  list < ParamString >::iterator sit;
  for (sit = str_list.begin (); sit != str_list.end (); sit++)
    if ((*sit).GetPosition () == pos)
      {
	return ((*sit).GetWidths ());
      }

  return NULL;
}


// ----------------------------------------------
// Returns the Parameter for a given position
// NOTE:  this only returns the ParamItem  which
//        does NOT include the value
// ----------------------------------------------
void
Parameter::GetPosition (int pos, ParamItem & item)
{

  list < ParamBoolean >::iterator bit;
  for (bit = bool_list.begin (); bit != bool_list.end (); bit++)
    if ((*bit).GetPosition () == pos)
      {
	item = (*bit);
	break;
      }

  list < ParamInteger >::iterator iit;
  for (iit = int_list.begin (); iit != int_list.end (); iit++)
    if ((*iit).GetPosition () == pos)
      {
	item = (*iit);
	break;
      }

  list < ParamReal >::iterator rit;
  for (rit = real_list.begin (); rit != real_list.end (); rit++)
    if ((*rit).GetPosition () == pos)
      {
	item = (*rit);
	break;
      }

  list < ParamString >::iterator sit;
  for (sit = str_list.begin (); sit != str_list.end (); sit++)
    if ((*sit).GetPosition () == pos)
      {
	item = (*sit);
	break;
      }

  return;
}


// cancel all changes made
void
Parameter::CancelChanges ()
{

  list < ParamBoolean >::iterator bit;
  for (bit = bool_list.begin (); bit != bool_list.end (); bit++)
    if ((*bit).Changed ())
      {
	(*bit).SetValue ((*bit).GetValue ());
      }

  list < ParamInteger >::iterator iit;
  for (iit = int_list.begin (); iit != int_list.end (); iit++)
    if ((*iit).Changed ())
      {
	(*iit).SetValue ((*iit).GetValue ());
      }

  list < ParamReal >::iterator rit;
  for (rit = real_list.begin (); rit != real_list.end (); rit++)
    if ((*rit).Changed ())
      {
	(*rit).SetValue ((*rit).GetValue ());
      }

  list < ParamString >::iterator sit;
  for (sit = str_list.begin (); sit != str_list.end (); sit++)
    if ((*sit).Changed ())
      {
	string tmp = (*sit).GetValue ();
	(*sit).SetValue (tmp);
      }

}



int
Parameter::CheckValidity (char *pname, char *pval, char *eval)
{
  int retval = 0;

  if (pname && pval && filename.length ())
    {
      paramfile parfile = NULL;

      // open the parameter file with read access
      if ((parfile =
	   paramopen ((char *) filename.c_str (), NULL, 0, "rH")) != NULL)
	{
	  if (paccess (parfile, pname))
	    {
	      retval = paramvalid (parfile, pname, pval, eval);
	    }
	}
      paramclose (parfile);
    }

  return retval;
}



ostream & operator<< (ostream & oss, Parameter & obj)
{
  obj.Print (oss);
  return oss;
}

istream & operator>> (istream & iss, Parameter & obj)
{
  obj.Read (iss);
  return iss;
}
