!!****f* ABINIT/contstr23
!! NAME
!! contstr23
!!
!! FUNCTION
!! Carries out specialized metric tensor operations needed for contraction
!! of the 2nd strain derivative of the l=0,1,2,3 nonlocal Kleinman-Bylander
!! pseudopotential operation.  Derivatives are wrt a pair of cartesian
!! strain components.
!! Full advantage is taken of the full permutational symmetry of these
!! tensors.
!!
!! COPYRIGHT
!! Copyright (C) 1998-2020 ABINIT group (DRH)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors, see ~abinit/doc/developers/contributors.txt.
!!
!! INPUTS
!!  istr1=1,...6 specifies cartesian strain component 11,22,33,32,31,21
!!  istr2=seccond strain component
!!  rank=angular momentum
!!  gm(3,3)=metric tensor (array is symmetric but stored as 3x3)
!!  gprimd(3,3)=reciprocal space dimensional primitive translations
!!  aa(2,*)=unique elements of complex right-hand tensor
!!  bb(2,*)=unique elements of complex left-hand tensor
!!
!! OUTPUT
!!  e2nl=contraction for nonlocal 2nd-order strain derivative energy
!!
!! NOTES
!! All tensors are stored in a compressed storage mode defined below;
!! input and output conform to this scheme.
!! When tensor elements occur repeatedly due to symmetry, the
!! WEIGHT IS INCLUDED in the output tensor element to simplify later
!! contractions with other tensors of the same rank and form, i.e. the
!! next contraction is then simply a dot product over the unique elements.
!!
!! PARENTS
!!      m_nonlop_pl
!!
!! CHILDREN
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

#include "abi_common.h"


module m_contstr23
contains
!!**



subroutine contstr23(istr1,istr2,rank,gm,gprimd,e2nl,aa,bb)


 use defs_basis
 use m_abicore
 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: istr1,istr2,rank
 real(dp),intent(out) :: e2nl
!arrays
 real(dp),intent(in) :: aa(2,((rank+1)*(rank+2))/2),bb(2,((rank+3)*(rank+4))/2)
 real(dp),intent(in) :: gm(3,3),gprimd(3,3)

!Local variables-------------------------------
!scalars
 integer,parameter :: mrank=3
 integer :: ii,jj,ka,kb,kd,kg
!arrays
 integer,save :: idx(12)=(/1,1,2,2,3,3,3,2,3,1,2,1/)
 real(dp) :: d2gm(3,3),dgm01(3,3),dgm10(3,3),tmp(2)
 real(dp),allocatable :: cm(:,:)

! *************************************************************************
 ABI_ALLOCATE(cm,(((mrank+1)*(mrank+2))/2,((mrank+3)*(mrank+4))/2))

 ka=idx(2*istr1-1);kb=idx(2*istr1);kg=idx(2*istr2-1);kd=idx(2*istr2)

 do ii = 1,3
   dgm01(:,ii)=-(gprimd(ka,:)*gprimd(kb,ii)+gprimd(kb,:)*gprimd(ka,ii))
   dgm10(:,ii)=-(gprimd(kg,:)*gprimd(kd,ii)+gprimd(kd,:)*gprimd(kg,ii))
 end do

 d2gm(:,:)=0.d0
 do ii = 1,3
   if(ka==kg) d2gm(:,ii)=d2gm(:,ii)&
&   +gprimd(kb,:)*gprimd(kd,ii)+gprimd(kd,:)*gprimd(kb,ii)
   if(ka==kd) d2gm(:,ii)=d2gm(:,ii)&
&   +gprimd(kb,:)*gprimd(kg,ii)+gprimd(kg,:)*gprimd(kb,ii)
   if(kb==kg) d2gm(:,ii)=d2gm(:,ii)&
&   +gprimd(ka,:)*gprimd(kd,ii)+gprimd(kd,:)*gprimd(ka,ii)
   if(kb==kd) d2gm(:,ii)=d2gm(:,ii)&
&   +gprimd(ka,:)*gprimd(kg,ii)+gprimd(kg,:)*gprimd(ka,ii)
 end do
 d2gm(:,:)=0.5d0*d2gm(:,:)

!
!The code below was written by a Mathematica program and formatted by
!a combination of editing scripts.  It is not intended to be read
!by human beings, and certainly not to be modified by one.  Conceivably
!it could be shortened somewhat by identifying common subexpressions.
!
 if(rank==0)then
   cm(1,1)=d2gm(1,1)
   cm(1,2)=d2gm(2,2)
   cm(1,3)=d2gm(3,3)
   cm(1,4)=2*d2gm(2,3)
   cm(1,5)=2*d2gm(1,3)
   cm(1,6)=2*d2gm(1,2)
 elseif(rank==1)then
   cm(1,1)=2*dgm01(1,1)*dgm10(1,1)+gm(1,1)*d2gm(1,1)
   cm(2,1)=dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2)+gm(1,2)&
&   *d2gm(1,1)
   cm(3,1)=dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3)+gm(1,3)&
&   *d2gm(1,1)
   cm(1,2)=dgm01(2,2)*dgm10(1,1)+4*dgm01(1,2)*dgm10(1,2)+dgm01(1,1)&
&   *dgm10(2,2)+2*gm(1,2)*d2gm(1,2)+gm(1,1)*d2gm(2,2)
   cm(2,2)=3*dgm01(2,2)*dgm10(1,2)+3*dgm01(1,2)*dgm10(2,2)+2*gm(2,2)&
&   *d2gm(1,2)+gm(1,2)*d2gm(2,2)
   cm(3,2)=2*dgm01(2,3)*dgm10(1,2)+dgm01(2,2)*dgm10(1,3)+dgm01(1,3)&
&   *dgm10(2,2)+2*dgm01(1,2)*dgm10(2,3)+2*gm(2,3)*d2gm(1,2)+gm(1,3)&
&   *d2gm(2,2)
   cm(1,3)=dgm01(3,3)*dgm10(1,1)+4*dgm01(1,3)*dgm10(1,3)+dgm01(1,1)&
&   *dgm10(3,3)+2*gm(1,3)*d2gm(1,3)+gm(1,1)*d2gm(3,3)
   cm(2,3)=dgm01(3,3)*dgm10(1,2)+2*dgm01(2,3)*dgm10(1,3)+2*dgm01(1,3)&
&   *dgm10(2,3)+dgm01(1,2)*dgm10(3,3)+2*gm(2,3)*d2gm(1,3)+gm(1,2)&
&   *d2gm(3,3)
   cm(3,3)=3*dgm01(3,3)*dgm10(1,3)+3*dgm01(1,3)*dgm10(3,3)+2*gm(3,3)&
&   *d2gm(1,3)+gm(1,3)*d2gm(3,3)
   cm(1,4)=2*(dgm01(2,3)*dgm10(1,1)+2*dgm01(1,3)*dgm10(1,2)+2*dgm01(1,2)&
&   *dgm10(1,3)+dgm01(1,1)*dgm10(2,3)+gm(1,3)*d2gm(1,2)+gm(1,2)*d2gm(1,3)&
&   +gm(1,1)*d2gm(2,3))
   cm(2,4)=2*(2*dgm01(2,3)*dgm10(1,2)+dgm01(2,2)*dgm10(1,3)+dgm01(1,3)&
&   *dgm10(2,2)+2*dgm01(1,2)*dgm10(2,3)+gm(2,3)*d2gm(1,2)+gm(2,2)&
&   *d2gm(1,3)+gm(1,2)*d2gm(2,3))
   cm(3,4)=2*(dgm01(3,3)*dgm10(1,2)+2*dgm01(2,3)*dgm10(1,3)+2*dgm01(1,3)&
&   *dgm10(2,3)+dgm01(1,2)*dgm10(3,3)+gm(3,3)*d2gm(1,2)+gm(2,3)*d2gm(1,3)&
&   +gm(1,3)*d2gm(2,3))
   cm(1,5)=3*dgm01(1,3)*dgm10(1,1)+3*dgm01(1,1)*dgm10(1,3)+gm(1,3)&
&   *d2gm(1,1)+2*gm(1,1)*d2gm(1,3)
   cm(2,5)=dgm01(2,3)*dgm10(1,1)+2*dgm01(1,3)*dgm10(1,2)+2*dgm01(1,2)&
&   *dgm10(1,3)+dgm01(1,1)*dgm10(2,3)+gm(2,3)*d2gm(1,1)+2*gm(1,2)&
&   *d2gm(1,3)
   cm(3,5)=dgm01(3,3)*dgm10(1,1)+4*dgm01(1,3)*dgm10(1,3)+dgm01(1,1)&
&   *dgm10(3,3)+gm(3,3)*d2gm(1,1)+2*gm(1,3)*d2gm(1,3)
   cm(1,6)=3*dgm01(1,2)*dgm10(1,1)+3*dgm01(1,1)*dgm10(1,2)+gm(1,2)&
&   *d2gm(1,1)+2*gm(1,1)*d2gm(1,2)
   cm(2,6)=dgm01(2,2)*dgm10(1,1)+4*dgm01(1,2)*dgm10(1,2)+dgm01(1,1)&
&   *dgm10(2,2)+gm(2,2)*d2gm(1,1)+2*gm(1,2)*d2gm(1,2)
   cm(3,6)=dgm01(2,3)*dgm10(1,1)+2*dgm01(1,3)*dgm10(1,2)+2*dgm01(1,2)&
&   *dgm10(1,3)+dgm01(1,1)*dgm10(2,3)+gm(2,3)*d2gm(1,1)+2*gm(1,3)&
&   *d2gm(1,2)
   cm(1,7)=dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2)+gm(1,2)&
&   *d2gm(2,2)
   cm(2,7)=2*dgm01(2,2)*dgm10(2,2)+gm(2,2)*d2gm(2,2)
   cm(3,7)=dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3)+gm(2,3)&
&   *d2gm(2,2)
   cm(1,8)=dgm01(3,3)*dgm10(1,2)+2*dgm01(2,3)*dgm10(1,3)+2*dgm01(1,3)&
&   *dgm10(2,3)+dgm01(1,2)*dgm10(3,3)+2*gm(1,3)*d2gm(2,3)+gm(1,2)&
&   *d2gm(3,3)
   cm(2,8)=dgm01(3,3)*dgm10(2,2)+4*dgm01(2,3)*dgm10(2,3)+dgm01(2,2)&
&   *dgm10(3,3)+2*gm(2,3)*d2gm(2,3)+gm(2,2)*d2gm(3,3)
   cm(3,8)=3*dgm01(3,3)*dgm10(2,3)+3*dgm01(2,3)*dgm10(3,3)+2*gm(3,3)&
&   *d2gm(2,3)+gm(2,3)*d2gm(3,3)
   cm(1,9)=2*dgm01(2,3)*dgm10(1,2)+dgm01(2,2)*dgm10(1,3)+dgm01(1,3)&
&   *dgm10(2,2)+2*dgm01(1,2)*dgm10(2,3)+gm(1,3)*d2gm(2,2)+2*gm(1,2)&
&   *d2gm(2,3)
   cm(2,9)=3*dgm01(2,3)*dgm10(2,2)+3*dgm01(2,2)*dgm10(2,3)+gm(2,3)&
&   *d2gm(2,2)+2*gm(2,2)*d2gm(2,3)
   cm(3,9)=dgm01(3,3)*dgm10(2,2)+4*dgm01(2,3)*dgm10(2,3)+dgm01(2,2)&
&   *dgm10(3,3)+gm(3,3)*d2gm(2,2)+2*gm(2,3)*d2gm(2,3)
   cm(1,10)=dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3)+gm(1,3)&
&   *d2gm(3,3)
   cm(2,10)=dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)+gm(2,3)&
&   *d2gm(3,3)
   cm(3,10)=2*dgm01(3,3)*dgm10(3,3)+gm(3,3)*d2gm(3,3)
 elseif(rank==2)then
   cm(1,1)=gm(1,1)*(4*dgm01(1,1)*dgm10(1,1)+gm(1,1)*d2gm(1,1))
   cm(2,1)=3*gm(1,2)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))&
&   -0.5d0*gm(1,1)*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2))&
&   +1.5d0*gm(1,2)**2*d2gm(1,1)+gm(2,2)*(-dgm01(1,1)*dgm10(1,1)-0.5d0*gm(1,1)&
&   *d2gm(1,1))
   cm(3,1)=3*gm(1,3)*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))&
&   -0.5d0*gm(1,1)*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))&
&   +1.5d0*gm(1,3)**2*d2gm(1,1)+gm(3,3)*(-dgm01(1,1)*dgm10(1,1)-0.5d0*gm(1,1)&
&   *d2gm(1,1))
   cm(4,1)=((-2*gm(2,3)*dgm01(1,1)+6*gm(1,3)*dgm01(1,2)+6*gm(1,2)&
&   *dgm01(1,3)-2*gm(1,1)*dgm01(2,3))*dgm10(1,1)+dgm01(1,1)*(-2*gm(2,3)&
&   *dgm10(1,1)+6*gm(1,3)*dgm10(1,2)+6*gm(1,2)*dgm10(1,3)-2*gm(1,1)&
&   *dgm10(2,3))+(6*gm(1,2)*gm(1,3)-2*gm(1,1)*gm(2,3))*d2gm(1,1))&
&   /2.d0
   cm(5,1)=2*gm(1,1)*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))&
&   +gm(1,3)*(4*dgm01(1,1)*dgm10(1,1)+2*gm(1,1)*d2gm(1,1))
   cm(6,1)=2*gm(1,1)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))&
&   +gm(1,2)*(4*dgm01(1,1)*dgm10(1,1)+2*gm(1,1)*d2gm(1,1))
   cm(1,2)=1.5d0*gm(1,2)**2*d2gm(1,1)+gm(2,2)*(-dgm01(1,1)*dgm10(1,1)&
&   -0.5d0*gm(1,1)*d2gm(1,1))+gm(1,2)*(7*dgm01(1,2)*dgm10(1,1)+7*dgm01(1,1)&
&   *dgm10(1,2)+4*gm(1,1)*d2gm(1,2))+gm(1,1)*(1.5d0*dgm01(2,2)*dgm10(1,1)&
&   +8*dgm01(1,2)*dgm10(1,2)+1.5d0*dgm01(1,1)*dgm10(2,2)+gm(1,1)&
&   *d2gm(2,2))
   cm(2,2)=-gm(1,1)*dgm01(2,2)*dgm10(2,2)+7*gm(1,2)*(dgm01(2,2)&
&   *dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+gm(2,2)**2*d2gm(1,1)+1.5d0*gm(1,2)&
&   **2*d2gm(2,2)+gm(2,2)*(1.5d0*dgm01(2,2)*dgm10(1,1)+8*dgm01(1,2)&
&   *dgm10(1,2)+1.5d0*dgm01(1,1)*dgm10(2,2)+4*gm(1,2)*d2gm(1,2)-0.5d0*gm(1,1)&
&   *d2gm(2,2))
   cm(3,2)=((-2*gm(3,3)*dgm01(2,2)+12*gm(2,3)*dgm01(2,3)-2*gm(2,2)&
&   *dgm01(3,3))*dgm10(1,1)+4*(-2*gm(3,3)*dgm01(1,2)+6*gm(2,3)*dgm01(1,3)&
&   +6*gm(1,3)*dgm01(2,3)-2*gm(1,2)*dgm01(3,3))*dgm10(1,2)+(-2*gm(3,3)&
&   *dgm01(1,1)+12*gm(1,3)*dgm01(1,3)-2*gm(1,1)*dgm01(3,3))*dgm10(2,2)&
&   +dgm01(2,2)*(-2*gm(3,3)*dgm10(1,1)+12*gm(1,3)*dgm10(1,3)-2*gm(1,1)&
&   *dgm10(3,3))+4*dgm01(1,2)*(-2*gm(3,3)*dgm10(1,2)+6*gm(2,3)*dgm10(1,3)&
&   +6*gm(1,3)*dgm10(2,3)-2*gm(1,2)*dgm10(3,3))+dgm01(1,1)*(-2*gm(3,3)&
&   *dgm10(2,2)+12*gm(2,3)*dgm10(2,3)-2*gm(2,2)*dgm10(3,3))+(6*gm(2,3)&
&   **2-2*gm(2,2)*gm(3,3))*d2gm(1,1)+4*(6*gm(1,3)*gm(2,3)-2*gm(1,2)&
&   *gm(3,3))*d2gm(1,2)+(6*gm(1,3)**2-2*gm(1,1)*gm(3,3))*d2gm(2,2))&
&   /4.d0
   cm(4,2)=9*gm(1,3)*dgm01(2,2)*dgm10(1,2)+2*gm(1,2)*dgm01(2,3)&
&   *dgm10(1,2)+3*gm(1,2)*dgm01(2,2)*dgm10(1,3)+9*gm(1,3)*dgm01(1,2)&
&   *dgm10(2,2)+3*gm(1,2)*dgm01(1,3)*dgm10(2,2)-gm(1,1)*dgm01(2,3)&
&   *dgm10(2,2)+2*gm(1,2)*dgm01(1,2)*dgm10(2,3)-gm(1,1)*dgm01(2,2)&
&   *dgm10(2,3)+gm(2,2)*(2*dgm01(2,3)*dgm10(1,1)+6*dgm01(1,3)*dgm10(1,2)&
&   +6*dgm01(1,2)*dgm10(1,3)+2*dgm01(1,1)*dgm10(2,3)+6*gm(1,3)*d2gm(1,2))&
&   +3*gm(1,2)*gm(1,3)*d2gm(2,2)+gm(2,3)*(1*dgm01(2,2)*dgm10(1,1)&
&   +4*dgm01(1,2)*dgm10(1,2)+dgm01(1,1)*dgm10(2,2)+2*gm(2,2)*d2gm(1,1)&
&   +2*gm(1,2)*d2gm(1,2)-gm(1,1)*d2gm(2,2))
   cm(5,2)=gm(1,3)*dgm01(2,2)*dgm10(1,1)+3*gm(1,2)*dgm01(2,3)*dgm10(1,1)&
&   +4*gm(1,3)*dgm01(1,2)*dgm10(1,2)+2*gm(1,2)*dgm01(1,3)*dgm10(1,2)&
&   +6*gm(1,1)*dgm01(2,3)*dgm10(1,2)+2*gm(1,2)*dgm01(1,2)*dgm10(1,3)&
&   +2*gm(1,1)*dgm01(2,2)*dgm10(1,3)+gm(1,3)*dgm01(1,1)*dgm10(2,2)&
&   +2*gm(1,1)*dgm01(1,3)*dgm10(2,2)+3*gm(1,2)*dgm01(1,1)*dgm10(2,3)&
&   +6*gm(1,1)*dgm01(1,2)*dgm10(2,3)-gm(2,2)*(dgm01(1,3)*dgm10(1,1)&
&   +dgm01(1,1)*dgm10(1,3)+gm(1,3)*d2gm(1,1))+2*gm(1,2)*gm(1,3)*d2gm(1,2)&
&   +gm(2,3)*(9*dgm01(1,2)*dgm10(1,1)+9*dgm01(1,1)*dgm10(1,2)+3*gm(1,2)&
&   *d2gm(1,1)+6*gm(1,1)*d2gm(1,2))+2*gm(1,1)*gm(1,3)*d2gm(2,2)
   cm(6,2)=8*gm(1,1)*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))&
&   +2*gm(1,2)**2*d2gm(1,2)+gm(2,2)*(8*dgm01(1,2)*dgm10(1,1)+8*dgm01(1,1)&
&   *dgm10(1,2)+2*gm(1,2)*d2gm(1,1)+6*gm(1,1)*d2gm(1,2))+gm(1,2)&
&   *(4*dgm01(2,2)*dgm10(1,1)+8*dgm01(1,2)*dgm10(1,2)+4*dgm01(1,1)&
&   *dgm10(2,2)+2*gm(1,1)*d2gm(2,2))
   cm(1,3)=1.5d0*gm(1,3)**2*d2gm(1,1)+gm(3,3)*(-dgm01(1,1)*dgm10(1,1)&
&   -0.5d0*gm(1,1)*d2gm(1,1))+gm(1,3)*(7*dgm01(1,3)*dgm10(1,1)+7*dgm01(1,1)&
&   *dgm10(1,3)+4*gm(1,1)*d2gm(1,3))+gm(1,1)*(1.5d0*dgm01(3,3)*dgm10(1,1)&
&   +8*dgm01(1,3)*dgm10(1,3)+1.5d0*dgm01(1,1)*dgm10(3,3)+gm(1,1)&
&   *d2gm(3,3))
   cm(2,3)=((-2*gm(3,3)*dgm01(2,2)+12*gm(2,3)*dgm01(2,3)-2*gm(2,2)&
&   *dgm01(3,3))*dgm10(1,1)+4*(6*gm(2,3)*dgm01(1,2)-2*gm(2,2)*dgm01(1,3)&
&   -2*gm(1,3)*dgm01(2,2)+6*gm(1,2)*dgm01(2,3))*dgm10(1,3)+dgm01(3,3)&
&   *(-2*gm(2,2)*dgm10(1,1)+12*gm(1,2)*dgm10(1,2)-2*gm(1,1)*dgm10(2,2))&
&   +4*dgm01(1,3)*(6*gm(2,3)*dgm10(1,2)-2*gm(2,2)*dgm10(1,3)-2*gm(1,3)&
&   *dgm10(2,2)+6*gm(1,2)*dgm10(2,3))+(-2*gm(2,2)*dgm01(1,1)+12*gm(1,2)&
&   *dgm01(1,2)-2*gm(1,1)*dgm01(2,2))*dgm10(3,3)+dgm01(1,1)*(-2*gm(3,3)&
&   *dgm10(2,2)+12*gm(2,3)*dgm10(2,3)-2*gm(2,2)*dgm10(3,3))+(6*gm(2,3)&
&   **2-2*gm(2,2)*gm(3,3))*d2gm(1,1)+4*(-2*gm(1,3)*gm(2,2)+6*gm(1,2)&
&   *gm(2,3))*d2gm(1,3)+(6*gm(1,2)**2-2*gm(1,1)*gm(2,2))*d2gm(3,3))&
&   /4.d0
   cm(3,3)=-gm(1,1)*dgm01(3,3)*dgm10(3,3)+7*gm(1,3)*(dgm01(3,3)&
&   *dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+gm(3,3)**2*d2gm(1,1)+1.5d0*gm(1,3)&
&   **2*d2gm(3,3)+gm(3,3)*(1.5d0*dgm01(3,3)*dgm10(1,1)+8*dgm01(1,3)&
&   *dgm10(1,3)+1.5d0*dgm01(1,1)*dgm10(3,3)+4*gm(1,3)*d2gm(1,3)-0.5d0*gm(1,1)&
&   *d2gm(3,3))
   cm(4,3)=3*gm(1,3)*dgm01(3,3)*dgm10(1,2)+2*gm(1,3)*dgm01(2,3)&
&   *dgm10(1,3)+9*gm(1,2)*dgm01(3,3)*dgm10(1,3)+2*gm(1,3)*dgm01(1,3)&
&   *dgm10(2,3)-gm(1,1)*dgm01(3,3)*dgm10(2,3)+3*gm(1,3)*dgm01(1,2)&
&   *dgm10(3,3)+9*gm(1,2)*dgm01(1,3)*dgm10(3,3)-gm(1,1)*dgm01(2,3)&
&   *dgm10(3,3)+gm(3,3)*(2*dgm01(2,3)*dgm10(1,1)+6*dgm01(1,3)*dgm10(1,2)&
&   +6*dgm01(1,2)*dgm10(1,3)+2*dgm01(1,1)*dgm10(2,3)+2*gm(2,3)*d2gm(1,1)&
&   +6*gm(1,2)*d2gm(1,3))+3*gm(1,2)*gm(1,3)*d2gm(3,3)+gm(2,3)*(1*dgm01(3,3)&
&   *dgm10(1,1)+4*dgm01(1,3)*dgm10(1,3)+dgm01(1,1)*dgm10(3,3)+2*gm(1,3)&
&   *d2gm(1,3)-gm(1,1)*d2gm(3,3))
   cm(5,3)=8*gm(1,1)*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))&
&   +2*gm(1,3)**2*d2gm(1,3)+gm(3,3)*(8*dgm01(1,3)*dgm10(1,1)+8*dgm01(1,1)&
&   *dgm10(1,3)+2*gm(1,3)*d2gm(1,1)+6*gm(1,1)*d2gm(1,3))+gm(1,3)&
&   *(4*dgm01(3,3)*dgm10(1,1)+8*dgm01(1,3)*dgm10(1,3)+4*dgm01(1,1)&
&   *dgm10(3,3)+2*gm(1,1)*d2gm(3,3))
   cm(6,3)=3*gm(1,3)*dgm01(2,3)*dgm10(1,1)+gm(1,2)*dgm01(3,3)*dgm10(1,1)&
&   +2*gm(1,3)*dgm01(1,3)*dgm10(1,2)+2*gm(1,1)*dgm01(3,3)*dgm10(1,2)&
&   +2*gm(1,3)*dgm01(1,2)*dgm10(1,3)+4*gm(1,2)*dgm01(1,3)*dgm10(1,3)&
&   +6*gm(1,1)*dgm01(2,3)*dgm10(1,3)+3*gm(1,3)*dgm01(1,1)*dgm10(2,3)&
&   +6*gm(1,1)*dgm01(1,3)*dgm10(2,3)+gm(1,2)*dgm01(1,1)*dgm10(3,3)&
&   +2*gm(1,1)*dgm01(1,2)*dgm10(3,3)-gm(3,3)*(dgm01(1,2)*dgm10(1,1)&
&   +dgm01(1,1)*dgm10(1,2)+gm(1,2)*d2gm(1,1))+2*gm(1,2)*gm(1,3)*d2gm(1,3)&
&   +gm(2,3)*(9*dgm01(1,3)*dgm10(1,1)+9*dgm01(1,1)*dgm10(1,3)+3*gm(1,3)&
&   *d2gm(1,1)+6*gm(1,1)*d2gm(1,3))+2*gm(1,1)*gm(1,2)*d2gm(3,3)
   cm(1,4)=7*gm(1,2)*dgm01(1,3)*dgm10(1,1)+3*gm(1,1)*dgm01(2,3)&
&   *dgm10(1,1)+8*gm(1,1)*dgm01(1,3)*dgm10(1,2)+7*gm(1,2)*dgm01(1,1)&
&   *dgm10(1,3)+8*gm(1,1)*dgm01(1,2)*dgm10(1,3)+3*gm(1,1)*dgm01(1,1)&
&   *dgm10(2,3)+gm(2,3)*(-2*dgm01(1,1)*dgm10(1,1)-gm(1,1)*d2gm(1,1))&
&   +gm(1,3)*(7*dgm01(1,2)*dgm10(1,1)+7*dgm01(1,1)*dgm10(1,2)+3*gm(1,2)&
&   *d2gm(1,1)+4*gm(1,1)*d2gm(1,2))+4*gm(1,1)*gm(1,2)*d2gm(1,3)+2*gm(1,1)&
&   **2*d2gm(2,3)
   cm(2,4)=-2*gm(1,3)*dgm01(2,2)*dgm10(1,2)+12*gm(1,2)*dgm01(2,3)&
&   *dgm10(1,2)+4*gm(1,2)*dgm01(2,2)*dgm10(1,3)-2*gm(1,3)*dgm01(1,2)&
&   *dgm10(2,2)+4*gm(1,2)*dgm01(1,3)*dgm10(2,2)-gm(1,1)*dgm01(2,3)&
&   *dgm10(2,2)+12*gm(1,2)*dgm01(1,2)*dgm10(2,3)-gm(1,1)*dgm01(2,2)&
&   *dgm10(2,3)+gm(2,3)*(2*dgm01(2,2)*dgm10(1,1)+12*dgm01(1,2)*dgm10(1,2)&
&   +2*dgm01(1,1)*dgm10(2,2)+2*gm(2,2)*d2gm(1,1)+6*gm(1,2)*d2gm(1,2))&
&   +3*gm(1,2)**2*d2gm(2,3)+gm(2,2)*(1*dgm01(2,3)*dgm10(1,1)+2*dgm01(1,3)&
&   *dgm10(1,2)+2*dgm01(1,2)*dgm10(1,3)+dgm01(1,1)*dgm10(2,3)-2*gm(1,3)&
&   *d2gm(1,2)+4*gm(1,2)*d2gm(1,3)-gm(1,1)*d2gm(2,3))
   cm(3,4)=4*gm(1,3)*dgm01(3,3)*dgm10(1,2)+12*gm(1,3)*dgm01(2,3)&
&   *dgm10(1,3)-2*gm(1,2)*dgm01(3,3)*dgm10(1,3)+12*gm(1,3)*dgm01(1,3)&
&   *dgm10(2,3)-gm(1,1)*dgm01(3,3)*dgm10(2,3)+4*gm(1,3)*dgm01(1,2)&
&   *dgm10(3,3)-2*gm(1,2)*dgm01(1,3)*dgm10(3,3)-gm(1,1)*dgm01(2,3)&
&   *dgm10(3,3)+gm(2,3)*(2*dgm01(3,3)*dgm10(1,1)+12*dgm01(1,3)*dgm10(1,3)&
&   +2*dgm01(1,1)*dgm10(3,3)+6*gm(1,3)*d2gm(1,3))+3*gm(1,3)**2*d2gm(2,3)&
&   +gm(3,3)*(1*dgm01(2,3)*dgm10(1,1)+2*dgm01(1,3)*dgm10(1,2)+2*dgm01(1,2)&
&   *dgm10(1,3)+dgm01(1,1)*dgm10(2,3)+2*gm(2,3)*d2gm(1,1)+4*gm(1,3)&
&   *d2gm(1,2)-2*gm(1,2)*d2gm(1,3)-gm(1,1)*d2gm(2,3))
   cm(4,4)=3*gm(2,2)*dgm01(3,3)*dgm10(1,1)+8*gm(1,3)*dgm01(2,3)&
&   *dgm10(1,2)+6*gm(1,2)*dgm01(3,3)*dgm10(1,2)+12*gm(2,2)*dgm01(1,3)&
&   *dgm10(1,3)+6*gm(1,3)*dgm01(2,2)*dgm10(1,3)+8*gm(1,2)*dgm01(2,3)&
&   *dgm10(1,3)+6*gm(1,3)*dgm01(1,3)*dgm10(2,2)+8*gm(1,3)*dgm01(1,2)&
&   *dgm10(2,3)+8*gm(1,2)*dgm01(1,3)*dgm10(2,3)-4*gm(1,1)*dgm01(2,3)&
&   *dgm10(2,3)+3*gm(2,2)*dgm01(1,1)*dgm10(3,3)+6*gm(1,2)*dgm01(1,2)&
&   *dgm10(3,3)+gm(2,3)**2*d2gm(1,1)+gm(3,3)*(3*dgm01(2,2)*dgm10(1,1)&
&   +12*dgm01(1,2)*dgm10(1,2)+3*dgm01(1,1)*dgm10(2,2)+3*gm(2,2)*d2gm(1,1)&
&   +6*gm(1,2)*d2gm(1,2))+6*gm(1,3)*gm(2,2)*d2gm(1,3)+6*gm(1,2)*gm(1,3)&
&   *d2gm(2,3)+gm(2,3)*(0*dgm01(2,3)*dgm10(1,1)+4*dgm01(1,3)*dgm10(1,2)&
&   +4*dgm01(1,2)*dgm10(1,3)+0*dgm01(1,1)*dgm10(2,3)+2*gm(1,3)*d2gm(1,2)&
&   +2*gm(1,2)*d2gm(1,3)-2*gm(1,1)*d2gm(2,3))
   cm(5,4)=5*gm(1,3)*dgm01(2,3)*dgm10(1,1)+3*gm(1,2)*dgm01(3,3)&
&   *dgm10(1,1)+6*gm(1,3)*dgm01(1,3)*dgm10(1,2)+6*gm(1,1)*dgm01(3,3)&
&   *dgm10(1,2)+6*gm(1,3)*dgm01(1,2)*dgm10(1,3)+4*gm(1,2)*dgm01(1,3)&
&   *dgm10(1,3)+10*gm(1,1)*dgm01(2,3)*dgm10(1,3)+5*gm(1,3)*dgm01(1,1)&
&   *dgm10(2,3)+10*gm(1,1)*dgm01(1,3)*dgm10(2,3)+3*gm(1,2)*dgm01(1,1)&
&   *dgm10(3,3)+6*gm(1,1)*dgm01(1,2)*dgm10(3,3)+2*gm(1,3)**2*d2gm(1,2)&
&   +gm(3,3)*(9*dgm01(1,2)*dgm10(1,1)+9*dgm01(1,1)*dgm10(1,2)+3*gm(1,2)&
&   *d2gm(1,1)+6*gm(1,1)*d2gm(1,2))+2*gm(1,2)*gm(1,3)*d2gm(1,3)+gm(2,3)&
&   *(7*dgm01(1,3)*dgm10(1,1)+7*dgm01(1,1)*dgm10(1,3)+gm(1,3)*d2gm(1,1)&
&   +6*gm(1,1)*d2gm(1,3))+4*gm(1,1)*gm(1,3)*d2gm(2,3)
   cm(6,4)=3*gm(1,3)*dgm01(2,2)*dgm10(1,1)+5*gm(1,2)*dgm01(2,3)&
&   *dgm10(1,1)+4*gm(1,3)*dgm01(1,2)*dgm10(1,2)+6*gm(1,2)*dgm01(1,3)&
&   *dgm10(1,2)+10*gm(1,1)*dgm01(2,3)*dgm10(1,2)+6*gm(1,2)*dgm01(1,2)&
&   *dgm10(1,3)+6*gm(1,1)*dgm01(2,2)*dgm10(1,3)+3*gm(1,3)*dgm01(1,1)&
&   *dgm10(2,2)+6*gm(1,1)*dgm01(1,3)*dgm10(2,2)+5*gm(1,2)*dgm01(1,1)&
&   *dgm10(2,3)+10*gm(1,1)*dgm01(1,2)*dgm10(2,3)+2*gm(1,2)*gm(1,3)&
&   *d2gm(1,2)+gm(2,3)*(7*dgm01(1,2)*dgm10(1,1)+7*dgm01(1,1)*dgm10(1,2)&
&   +gm(1,2)*d2gm(1,1)+6*gm(1,1)*d2gm(1,2))+2*gm(1,2)**2*d2gm(1,3)&
&   +gm(2,2)*(9*dgm01(1,3)*dgm10(1,1)+9*dgm01(1,1)*dgm10(1,3)+3*gm(1,3)&
&   *d2gm(1,1)+6*gm(1,1)*d2gm(1,3))+4*gm(1,1)*gm(1,2)*d2gm(2,3)
   cm(1,5)=gm(1,3)*(4*dgm01(1,1)*dgm10(1,1)+2*gm(1,1)*d2gm(1,1))&
&   +gm(1,1)*(6*dgm01(1,3)*dgm10(1,1)+6*dgm01(1,1)*dgm10(1,3)+2*gm(1,1)&
&   *d2gm(1,3))
   cm(2,5)=((6*gm(2,3)*dgm01(1,2)-2*gm(2,2)*dgm01(1,3)-2*gm(1,3)&
&   *dgm01(2,2)+6*gm(1,2)*dgm01(2,3))*dgm10(1,1)+(-2*gm(2,2)*dgm01(1,1)&
&   +12*gm(1,2)*dgm01(1,2)-2*gm(1,1)*dgm01(2,2))*dgm10(1,3)+dgm01(1,3)&
&   *(-2*gm(2,2)*dgm10(1,1)+12*gm(1,2)*dgm10(1,2)-2*gm(1,1)*dgm10(2,2))&
&   +dgm01(1,1)*(6*gm(2,3)*dgm10(1,2)-2*gm(2,2)*dgm10(1,3)-2*gm(1,3)&
&   *dgm10(2,2)+6*gm(1,2)*dgm10(2,3))+(-2*gm(1,3)*gm(2,2)+6*gm(1,2)&
&   *gm(2,3))*d2gm(1,1)+(6*gm(1,2)**2-2*gm(1,1)*gm(2,2))*d2gm(1,3))&
&   /2.d0
   cm(3,5)=gm(1,3)*(2*dgm01(3,3)*dgm10(1,1)+12*dgm01(1,3)*dgm10(1,3)&
&   +2*dgm01(1,1)*dgm10(3,3))-gm(1,1)*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)&
&   *dgm10(3,3))+3*gm(1,3)**2*d2gm(1,3)+gm(3,3)*(1*dgm01(1,3)*dgm10(1,1)&
&   +dgm01(1,1)*dgm10(1,3)+2*gm(1,3)*d2gm(1,1)-gm(1,1)*d2gm(1,3))
   cm(4,5)=(3*gm(3,3)*dgm01(1,2)+gm(2,3)*dgm01(1,3)+gm(1,3)*dgm01(2,3)&
&   +3*gm(1,2)*dgm01(3,3))*dgm10(1,1)+(-2*gm(2,3)*dgm01(1,1)+6*gm(1,3)&
&   *dgm01(1,2)+6*gm(1,2)*dgm01(1,3)-2*gm(1,1)*dgm01(2,3))*dgm10(1,3)&
&   +dgm01(1,3)*(-2*gm(2,3)*dgm10(1,1)+6*gm(1,3)*dgm10(1,2)+6*gm(1,2)&
&   *dgm10(1,3)-2*gm(1,1)*dgm10(2,3))+dgm01(1,1)*(3*gm(3,3)*dgm10(1,2)&
&   +gm(2,3)*dgm10(1,3)+gm(1,3)*dgm10(2,3)+3*gm(1,2)*dgm10(3,3))&
&   +(1*gm(1,3)*gm(2,3)+3*gm(1,2)*gm(3,3))*d2gm(1,1)+(6*gm(1,2)*gm(1,3)&
&   -2*gm(1,1)*gm(2,3))*d2gm(1,3)
   cm(5,5)=gm(1,1)*(3*dgm01(3,3)*dgm10(1,1)+8*dgm01(1,3)*dgm10(1,3)&
&   +3*dgm01(1,1)*dgm10(3,3))+gm(1,3)**2*d2gm(1,1)+gm(3,3)*(6*dgm01(1,1)&
&   *dgm10(1,1)+3*gm(1,1)*d2gm(1,1))+gm(1,3)*(6*dgm01(1,3)*dgm10(1,1)&
&   +6*dgm01(1,1)*dgm10(1,3)+4*gm(1,1)*d2gm(1,3))
   cm(6,5)=5*gm(1,2)*dgm01(1,3)*dgm10(1,1)+3*gm(1,1)*dgm01(2,3)&
&   *dgm10(1,1)+4*gm(1,1)*dgm01(1,3)*dgm10(1,2)+5*gm(1,2)*dgm01(1,1)&
&   *dgm10(1,3)+4*gm(1,1)*dgm01(1,2)*dgm10(1,3)+3*gm(1,1)*dgm01(1,1)&
&   *dgm10(2,3)+gm(2,3)*(6*dgm01(1,1)*dgm10(1,1)+3*gm(1,1)*d2gm(1,1))&
&   +gm(1,3)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2)+gm(1,2)&
&   *d2gm(1,1))+4*gm(1,1)*gm(1,2)*d2gm(1,3)
   cm(1,6)=gm(1,2)*(4*dgm01(1,1)*dgm10(1,1)+2*gm(1,1)*d2gm(1,1))&
&   +gm(1,1)*(6*dgm01(1,2)*dgm10(1,1)+6*dgm01(1,1)*dgm10(1,2)+2*gm(1,1)&
&   *d2gm(1,2))
   cm(2,6)=gm(1,2)*(2*dgm01(2,2)*dgm10(1,1)+12*dgm01(1,2)*dgm10(1,2)&
&   +2*dgm01(1,1)*dgm10(2,2))-gm(1,1)*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)&
&   *dgm10(2,2))+3*gm(1,2)**2*d2gm(1,2)+gm(2,2)*(1*dgm01(1,2)*dgm10(1,1)&
&   +dgm01(1,1)*dgm10(1,2)+2*gm(1,2)*d2gm(1,1)-gm(1,1)*d2gm(1,2))
   cm(3,6)=((-2*gm(3,3)*dgm01(1,2)+6*gm(2,3)*dgm01(1,3)+6*gm(1,3)&
&   *dgm01(2,3)-2*gm(1,2)*dgm01(3,3))*dgm10(1,1)+(-2*gm(3,3)*dgm01(1,1)&
&   +12*gm(1,3)*dgm01(1,3)-2*gm(1,1)*dgm01(3,3))*dgm10(1,2)+dgm01(1,2)&
&   *(-2*gm(3,3)*dgm10(1,1)+12*gm(1,3)*dgm10(1,3)-2*gm(1,1)*dgm10(3,3))&
&   +dgm01(1,1)*(-2*gm(3,3)*dgm10(1,2)+6*gm(2,3)*dgm10(1,3)+6*gm(1,3)&
&   *dgm10(2,3)-2*gm(1,2)*dgm10(3,3))+(6*gm(1,3)*gm(2,3)-2*gm(1,2)&
&   *gm(3,3))*d2gm(1,1)+(6*gm(1,3)**2-2*gm(1,1)*gm(3,3))*d2gm(1,2))&
&   /2.d0
   cm(4,6)=(1*gm(2,3)*dgm01(1,2)+3*gm(2,2)*dgm01(1,3)+3*gm(1,3)&
&   *dgm01(2,2)+gm(1,2)*dgm01(2,3))*dgm10(1,1)+(-2*gm(2,3)*dgm01(1,1)&
&   +6*gm(1,3)*dgm01(1,2)+6*gm(1,2)*dgm01(1,3)-2*gm(1,1)*dgm01(2,3))&
&   *dgm10(1,2)+dgm01(1,2)*(-2*gm(2,3)*dgm10(1,1)+6*gm(1,3)*dgm10(1,2)&
&   +6*gm(1,2)*dgm10(1,3)-2*gm(1,1)*dgm10(2,3))+dgm01(1,1)*(1*gm(2,3)&
&   *dgm10(1,2)+3*gm(2,2)*dgm10(1,3)+3*gm(1,3)*dgm10(2,2)+gm(1,2)&
&   *dgm10(2,3))+(3*gm(1,3)*gm(2,2)+gm(1,2)*gm(2,3))*d2gm(1,1)+(6*gm(1,2)&
&   *gm(1,3)-2*gm(1,1)*gm(2,3))*d2gm(1,2)
   cm(5,6)=gm(1,2)*dgm01(1,3)*dgm10(1,1)+3*gm(1,1)*dgm01(2,3)*dgm10(1,1)&
&   +4*gm(1,1)*dgm01(1,3)*dgm10(1,2)+gm(1,2)*dgm01(1,1)*dgm10(1,3)&
&   +4*gm(1,1)*dgm01(1,2)*dgm10(1,3)+3*gm(1,1)*dgm01(1,1)*dgm10(2,3)&
&   +gm(2,3)*(6*dgm01(1,1)*dgm10(1,1)+3*gm(1,1)*d2gm(1,1))+gm(1,3)&
&   *(5*dgm01(1,2)*dgm10(1,1)+5*dgm01(1,1)*dgm10(1,2)+gm(1,2)*d2gm(1,1)&
&   +4*gm(1,1)*d2gm(1,2))
   cm(6,6)=gm(1,1)*(3*dgm01(2,2)*dgm10(1,1)+8*dgm01(1,2)*dgm10(1,2)&
&   +3*dgm01(1,1)*dgm10(2,2))+gm(1,2)**2*d2gm(1,1)+gm(2,2)*(6*dgm01(1,1)&
&   *dgm10(1,1)+3*gm(1,1)*d2gm(1,1))+gm(1,2)*(6*dgm01(1,2)*dgm10(1,1)&
&   +6*dgm01(1,1)*dgm10(1,2)+4*gm(1,1)*d2gm(1,2))
   cm(1,7)=gm(1,1)*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))&
&   +3*gm(1,2)**2*d2gm(1,2)-gm(2,2)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)&
&   *dgm10(1,2)+gm(1,1)*d2gm(1,2))+gm(1,2)*(2*dgm01(2,2)*dgm10(1,1)&
&   +12*dgm01(1,2)*dgm10(1,2)+2*dgm01(1,1)*dgm10(2,2)+2*gm(1,1)*d2gm(2,2))
   cm(2,7)=4*gm(1,2)*dgm01(2,2)*dgm10(2,2)+2*gm(2,2)**2*d2gm(1,2)&
&   +gm(2,2)*(6*dgm01(2,2)*dgm10(1,2)+6*dgm01(1,2)*dgm10(2,2)+2*gm(1,2)&
&   *d2gm(2,2))
   cm(3,7)=((-2*gm(3,3)*dgm01(2,2)+12*gm(2,3)*dgm01(2,3)-2*gm(2,2)&
&   *dgm01(3,3))*dgm10(1,2)+(-2*gm(3,3)*dgm01(1,2)+6*gm(2,3)*dgm01(1,3)&
&   +6*gm(1,3)*dgm01(2,3)-2*gm(1,2)*dgm01(3,3))*dgm10(2,2)+dgm01(2,2)&
&   *(-2*gm(3,3)*dgm10(1,2)+6*gm(2,3)*dgm10(1,3)+6*gm(1,3)*dgm10(2,3)&
&   -2*gm(1,2)*dgm10(3,3))+dgm01(1,2)*(-2*gm(3,3)*dgm10(2,2)+12*gm(2,3)&
&   *dgm10(2,3)-2*gm(2,2)*dgm10(3,3))+(6*gm(2,3)**2-2*gm(2,2)*gm(3,3))&
&   *d2gm(1,2)+(6*gm(1,3)*gm(2,3)-2*gm(1,2)*gm(3,3))*d2gm(2,2))/2.d0
   cm(4,7)=6*gm(1,3)*dgm01(2,2)*dgm10(2,2)+gm(1,2)*dgm01(2,3)*dgm10(2,2)&
&   +gm(1,2)*dgm01(2,2)*dgm10(2,3)+gm(2,3)*(5*dgm01(2,2)*dgm10(1,2)&
&   +5*dgm01(1,2)*dgm10(2,2)+4*gm(2,2)*d2gm(1,2)+gm(1,2)*d2gm(2,2))&
&   +gm(2,2)*(4*dgm01(2,3)*dgm10(1,2)+3*dgm01(2,2)*dgm10(1,3)+3*dgm01(1,3)&
&   *dgm10(2,2)+4*dgm01(1,2)*dgm10(2,3)+3*gm(1,3)*d2gm(2,2))
   cm(5,7)=-gm(1,3)*dgm01(2,2)*dgm10(1,2)+6*gm(1,2)*dgm01(2,3)&
&   *dgm10(1,2)+gm(1,2)*dgm01(2,2)*dgm10(1,3)-gm(1,3)*dgm01(1,2)&
&   *dgm10(2,2)+gm(1,2)*dgm01(1,3)*dgm10(2,2)+3*gm(1,1)*dgm01(2,3)&
&   *dgm10(2,2)+6*gm(1,2)*dgm01(1,2)*dgm10(2,3)+3*gm(1,1)*dgm01(2,2)&
&   *dgm10(2,3)-2*gm(2,2)*(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)*dgm10(1,3)&
&   +gm(1,3)*d2gm(1,2))+gm(1,2)*gm(1,3)*d2gm(2,2)+gm(2,3)*(3*dgm01(2,2)&
&   *dgm10(1,1)+12*dgm01(1,2)*dgm10(1,2)+3*dgm01(1,1)*dgm10(2,2)&
&   +6*gm(1,2)*d2gm(1,2)+3*gm(1,1)*d2gm(2,2))
   cm(6,7)=6*gm(1,1)*dgm01(2,2)*dgm10(2,2)+6*gm(1,2)*(dgm01(2,2)&
&   *dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+gm(1,2)**2*d2gm(2,2)+gm(2,2)&
&   *(3*dgm01(2,2)*dgm10(1,1)+8*dgm01(1,2)*dgm10(1,2)+3*dgm01(1,1)&
&   *dgm10(2,2)+4*gm(1,2)*d2gm(1,2)+3*gm(1,1)*d2gm(2,2))
   cm(1,8)=4*gm(1,3)*dgm01(2,3)*dgm10(1,1)+2*gm(1,2)*dgm01(3,3)&
&   *dgm10(1,1)+12*gm(1,3)*dgm01(1,3)*dgm10(1,2)+gm(1,1)*dgm01(3,3)&
&   *dgm10(1,2)+12*gm(1,3)*dgm01(1,2)*dgm10(1,3)+12*gm(1,2)*dgm01(1,3)&
&   *dgm10(1,3)+2*gm(1,1)*dgm01(2,3)*dgm10(1,3)+4*gm(1,3)*dgm01(1,1)&
&   *dgm10(2,3)+2*gm(1,1)*dgm01(1,3)*dgm10(2,3)+2*gm(1,2)*dgm01(1,1)&
&   *dgm10(3,3)+gm(1,1)*dgm01(1,2)*dgm10(3,3)+3*gm(1,3)**2*d2gm(1,2)&
&   -gm(3,3)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2)+gm(1,1)&
&   *d2gm(1,2))+6*gm(1,2)*gm(1,3)*d2gm(1,3)-2*gm(2,3)*(dgm01(1,3)&
&   *dgm10(1,1)+dgm01(1,1)*dgm10(1,3)+gm(1,1)*d2gm(1,3))+4*gm(1,1)&
&   *gm(1,3)*d2gm(2,3)+2*gm(1,1)*gm(1,2)*d2gm(3,3)
   cm(2,8)=gm(2,2)*dgm01(3,3)*dgm10(1,2)+2*gm(2,2)*dgm01(2,3)*dgm10(1,3)&
&   -2*gm(1,3)*dgm01(2,3)*dgm10(2,2)+2*gm(1,2)*dgm01(3,3)*dgm10(2,2)&
&   +2*gm(2,2)*dgm01(1,3)*dgm10(2,3)-2*gm(1,3)*dgm01(2,2)*dgm10(2,3)&
&   +12*gm(1,2)*dgm01(2,3)*dgm10(2,3)+gm(2,2)*dgm01(1,2)*dgm10(3,3)&
&   +2*gm(1,2)*dgm01(2,2)*dgm10(3,3)+3*gm(2,3)**2*d2gm(1,2)-gm(3,3)&
&   *(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2)+gm(2,2)*d2gm(1,2))&
&   -2*gm(1,3)*gm(2,2)*d2gm(2,3)+gm(2,3)*(12*dgm01(2,3)*dgm10(1,2)&
&   +4*dgm01(2,2)*dgm10(1,3)+4*dgm01(1,3)*dgm10(2,2)+12*dgm01(1,2)&
&   *dgm10(2,3)+4*gm(2,2)*d2gm(1,3)+6*gm(1,2)*d2gm(2,3))+2*gm(1,2)&
&   *gm(2,2)*d2gm(3,3)
   cm(3,8)=7*gm(1,3)*dgm01(3,3)*dgm10(2,3)+7*gm(1,3)*dgm01(2,3)&
&   *dgm10(3,3)-2*gm(1,2)*dgm01(3,3)*dgm10(3,3)+2*gm(3,3)**2*d2gm(1,2)&
&   +gm(3,3)*(3*dgm01(3,3)*dgm10(1,2)+8*dgm01(2,3)*dgm10(1,3)+8*dgm01(1,3)&
&   *dgm10(2,3)+3*dgm01(1,2)*dgm10(3,3)+4*gm(2,3)*d2gm(1,3)+4*gm(1,3)&
&   *d2gm(2,3)-gm(1,2)*d2gm(3,3))+gm(2,3)*(7*dgm01(3,3)*dgm10(1,3)&
&   +7*dgm01(1,3)*dgm10(3,3)+3*gm(1,3)*d2gm(3,3))
   cm(4,8)=9*gm(2,2)*dgm01(3,3)*dgm10(1,3)+3*gm(1,3)*dgm01(3,3)&
&   *dgm10(2,2)+4*gm(1,3)*dgm01(2,3)*dgm10(2,3)+7*gm(1,2)*dgm01(3,3)&
&   *dgm10(2,3)+9*gm(2,2)*dgm01(1,3)*dgm10(3,3)+3*gm(1,3)*dgm01(2,2)&
&   *dgm10(3,3)+7*gm(1,2)*dgm01(2,3)*dgm10(3,3)+2*gm(2,3)**2*d2gm(1,3)&
&   +gm(3,3)*(10*dgm01(2,3)*dgm10(1,2)+6*dgm01(2,2)*dgm10(1,3)+6*dgm01(1,3)&
&   *dgm10(2,2)+10*dgm01(1,2)*dgm10(2,3)+4*gm(2,3)*d2gm(1,2)+6*gm(2,2)&
&   *d2gm(1,3)+6*gm(1,2)*d2gm(2,3))+3*gm(1,3)*gm(2,2)*d2gm(3,3)+gm(2,3)&
&   *(5*dgm01(3,3)*dgm10(1,2)+6*dgm01(2,3)*dgm10(1,3)+6*dgm01(1,3)&
&   *dgm10(2,3)+5*dgm01(1,2)*dgm10(3,3)+2*gm(1,3)*d2gm(2,3)+gm(1,2)&
&   *d2gm(3,3))
   cm(5,8)=5*gm(1,3)*dgm01(3,3)*dgm10(1,2)+6*gm(1,3)*dgm01(2,3)&
&   *dgm10(1,3)+7*gm(1,2)*dgm01(3,3)*dgm10(1,3)+6*gm(1,3)*dgm01(1,3)&
&   *dgm10(2,3)+9*gm(1,1)*dgm01(3,3)*dgm10(2,3)+5*gm(1,3)*dgm01(1,2)&
&   *dgm10(3,3)+7*gm(1,2)*dgm01(1,3)*dgm10(3,3)+9*gm(1,1)*dgm01(2,3)&
&   *dgm10(3,3)+2*gm(1,3)**2*d2gm(2,3)+gm(3,3)*(6*dgm01(2,3)*dgm10(1,1)&
&   +10*dgm01(1,3)*dgm10(1,2)+10*dgm01(1,2)*dgm10(1,3)+6*dgm01(1,1)&
&   *dgm10(2,3)+4*gm(1,3)*d2gm(1,2)+6*gm(1,2)*d2gm(1,3)+6*gm(1,1)&
&   *d2gm(2,3))+gm(1,2)*gm(1,3)*d2gm(3,3)+gm(2,3)*(3*dgm01(3,3)*dgm10(1,1)&
&   +4*dgm01(1,3)*dgm10(1,3)+3*dgm01(1,1)*dgm10(3,3)+2*gm(1,3)*d2gm(1,3)&
&   +3*gm(1,1)*d2gm(3,3))
   cm(6,8)=-4*gm(3,3)*dgm01(1,2)*dgm10(1,2)+8*gm(1,3)*dgm01(2,3)&
&   *dgm10(1,2)+0*gm(1,2)*dgm01(3,3)*dgm10(1,2)+6*gm(1,3)*dgm01(2,2)&
&   *dgm10(1,3)+4*gm(1,2)*dgm01(2,3)*dgm10(1,3)+6*gm(1,3)*dgm01(1,3)&
&   *dgm10(2,2)+3*gm(1,1)*dgm01(3,3)*dgm10(2,2)+8*gm(1,3)*dgm01(1,2)&
&   *dgm10(2,3)+4*gm(1,2)*dgm01(1,3)*dgm10(2,3)+12*gm(1,1)*dgm01(2,3)&
&   *dgm10(2,3)+0*gm(1,2)*dgm01(1,2)*dgm10(3,3)+3*gm(1,1)*dgm01(2,2)&
&   *dgm10(3,3)-2*gm(1,2)*gm(3,3)*d2gm(1,2)+2*gm(1,2)*gm(1,3)*d2gm(2,3)&
&   +gm(2,3)*(6*dgm01(2,3)*dgm10(1,1)+8*dgm01(1,3)*dgm10(1,2)+8*dgm01(1,2)&
&   *dgm10(1,3)+6*dgm01(1,1)*dgm10(2,3)+6*gm(1,3)*d2gm(1,2)+2*gm(1,2)&
&   *d2gm(1,3)+6*gm(1,1)*d2gm(2,3))+gm(1,2)**2*d2gm(3,3)+gm(2,2)&
&   *(3*dgm01(3,3)*dgm10(1,1)+12*dgm01(1,3)*dgm10(1,3)+3*dgm01(1,1)&
&   *dgm10(3,3)+6*gm(1,3)*d2gm(1,3)+3*gm(1,1)*d2gm(3,3))
   cm(1,9)=2*gm(1,3)*dgm01(2,2)*dgm10(1,1)+4*gm(1,2)*dgm01(2,3)&
&   *dgm10(1,1)+12*gm(1,3)*dgm01(1,2)*dgm10(1,2)+12*gm(1,2)*dgm01(1,3)&
&   *dgm10(1,2)+2*gm(1,1)*dgm01(2,3)*dgm10(1,2)+12*gm(1,2)*dgm01(1,2)&
&   *dgm10(1,3)+gm(1,1)*dgm01(2,2)*dgm10(1,3)+2*gm(1,3)*dgm01(1,1)&
&   *dgm10(2,2)+gm(1,1)*dgm01(1,3)*dgm10(2,2)+4*gm(1,2)*dgm01(1,1)&
&   *dgm10(2,3)+2*gm(1,1)*dgm01(1,2)*dgm10(2,3)+6*gm(1,2)*gm(1,3)&
&   *d2gm(1,2)-2*gm(2,3)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2)&
&   +gm(1,1)*d2gm(1,2))+3*gm(1,2)**2*d2gm(1,3)-gm(2,2)*(dgm01(1,3)&
&   *dgm10(1,1)+dgm01(1,1)*dgm10(1,3)+gm(1,1)*d2gm(1,3))+2*gm(1,1)&
&   *gm(1,3)*d2gm(2,2)+4*gm(1,1)*gm(1,2)*d2gm(2,3)
   cm(2,9)=-2*gm(1,3)*dgm01(2,2)*dgm10(2,2)+7*gm(1,2)*dgm01(2,3)&
&   *dgm10(2,2)+7*gm(1,2)*dgm01(2,2)*dgm10(2,3)+2*gm(2,2)**2*d2gm(1,3)&
&   +gm(2,3)*(7*dgm01(2,2)*dgm10(1,2)+7*dgm01(1,2)*dgm10(2,2)+4*gm(2,2)&
&   *d2gm(1,2)+3*gm(1,2)*d2gm(2,2))+gm(2,2)*(8*dgm01(2,3)*dgm10(1,2)&
&   +3*dgm01(2,2)*dgm10(1,3)+3*dgm01(1,3)*dgm10(2,2)+8*dgm01(1,2)&
&   *dgm10(2,3)-gm(1,3)*d2gm(2,2)+4*gm(1,2)*d2gm(2,3))
   cm(3,9)=-gm(2,2)*dgm01(3,3)*dgm10(1,3)+2*gm(1,3)*dgm01(3,3)&
&   *dgm10(2,2)+12*gm(1,3)*dgm01(2,3)*dgm10(2,3)-2*gm(1,2)*dgm01(3,3)&
&   *dgm10(2,3)-gm(2,2)*dgm01(1,3)*dgm10(3,3)+2*gm(1,3)*dgm01(2,2)&
&   *dgm10(3,3)-2*gm(1,2)*dgm01(2,3)*dgm10(3,3)+3*gm(2,3)**2*d2gm(1,3)&
&   +gm(3,3)*(2*dgm01(2,3)*dgm10(1,2)+dgm01(2,2)*dgm10(1,3)+dgm01(1,3)&
&   *dgm10(2,2)+2*dgm01(1,2)*dgm10(2,3)+4*gm(2,3)*d2gm(1,2)-gm(2,2)&
&   *d2gm(1,3)+2*gm(1,3)*d2gm(2,2)-2*gm(1,2)*d2gm(2,3))+gm(2,3)*(4*dgm01(3,3)&
&   *dgm10(1,2)+12*dgm01(2,3)*dgm10(1,3)+12*dgm01(1,3)*dgm10(2,3)&
&   +4*dgm01(1,2)*dgm10(3,3)+6*gm(1,3)*d2gm(2,3))
   cm(4,9)=6*gm(2,2)*dgm01(3,3)*dgm10(1,2)+10*gm(2,2)*dgm01(2,3)&
&   *dgm10(1,3)+7*gm(1,3)*dgm01(2,3)*dgm10(2,2)+3*gm(1,2)*dgm01(3,3)&
&   *dgm10(2,2)+10*gm(2,2)*dgm01(1,3)*dgm10(2,3)+7*gm(1,3)*dgm01(2,2)&
&   *dgm10(2,3)+4*gm(1,2)*dgm01(2,3)*dgm10(2,3)+6*gm(2,2)*dgm01(1,2)&
&   *dgm10(3,3)+3*gm(1,2)*dgm01(2,2)*dgm10(3,3)+2*gm(2,3)**2*d2gm(1,2)&
&   +gm(3,3)*(9*dgm01(2,2)*dgm10(1,2)+9*dgm01(1,2)*dgm10(2,2)+6*gm(2,2)&
&   *d2gm(1,2)+3*gm(1,2)*d2gm(2,2))+6*gm(1,3)*gm(2,2)*d2gm(2,3)+gm(2,3)&
&   *(6*dgm01(2,3)*dgm10(1,2)+5*dgm01(2,2)*dgm10(1,3)+5*dgm01(1,3)&
&   *dgm10(2,2)+6*dgm01(1,2)*dgm10(2,3)+4*gm(2,2)*d2gm(1,3)+gm(1,3)&
&   *d2gm(2,2)+2*gm(1,2)*d2gm(2,3))
   cm(5,9)=4*gm(1,3)*dgm01(2,3)*dgm10(1,2)+6*gm(1,2)*dgm01(3,3)&
&   *dgm10(1,2)-4*gm(2,2)*dgm01(1,3)*dgm10(1,3)+0*gm(1,3)*dgm01(2,2)&
&   *dgm10(1,3)+8*gm(1,2)*dgm01(2,3)*dgm10(1,3)+0*gm(1,3)*dgm01(1,3)&
&   *dgm10(2,2)+3*gm(1,1)*dgm01(3,3)*dgm10(2,2)+4*gm(1,3)*dgm01(1,2)&
&   *dgm10(2,3)+8*gm(1,2)*dgm01(1,3)*dgm10(2,3)+12*gm(1,1)*dgm01(2,3)&
&   *dgm10(2,3)+6*gm(1,2)*dgm01(1,2)*dgm10(3,3)+3*gm(1,1)*dgm01(2,2)&
&   *dgm10(3,3)-2*gm(1,3)*gm(2,2)*d2gm(1,3)+gm(1,3)**2*d2gm(2,2)&
&   +gm(3,3)*(3*dgm01(2,2)*dgm10(1,1)+12*dgm01(1,2)*dgm10(1,2)+3*dgm01(1,1)&
&   *dgm10(2,2)+6*gm(1,2)*d2gm(1,2)+3*gm(1,1)*d2gm(2,2))+2*gm(1,2)&
&   *gm(1,3)*d2gm(2,3)+gm(2,3)*(6*dgm01(2,3)*dgm10(1,1)+8*dgm01(1,3)&
&   *dgm10(1,2)+8*dgm01(1,2)*dgm10(1,3)+6*dgm01(1,1)*dgm10(2,3)+2*gm(1,3)&
&   *d2gm(1,2)+6*gm(1,2)*d2gm(1,3)+6*gm(1,1)*d2gm(2,3))
   cm(6,9)=7*gm(1,3)*dgm01(2,2)*dgm10(1,2)+6*gm(1,2)*dgm01(2,3)&
&   *dgm10(1,2)+5*gm(1,2)*dgm01(2,2)*dgm10(1,3)+7*gm(1,3)*dgm01(1,2)&
&   *dgm10(2,2)+5*gm(1,2)*dgm01(1,3)*dgm10(2,2)+9*gm(1,1)*dgm01(2,3)&
&   *dgm10(2,2)+6*gm(1,2)*dgm01(1,2)*dgm10(2,3)+9*gm(1,1)*dgm01(2,2)&
&   *dgm10(2,3)+gm(1,2)*gm(1,3)*d2gm(2,2)+gm(2,3)*(3*dgm01(2,2)*dgm10(1,1)&
&   +4*dgm01(1,2)*dgm10(1,2)+3*dgm01(1,1)*dgm10(2,2)+2*gm(1,2)*d2gm(1,2)&
&   +3*gm(1,1)*d2gm(2,2))+2*gm(1,2)**2*d2gm(2,3)+gm(2,2)*(6*dgm01(2,3)&
&   *dgm10(1,1)+10*dgm01(1,3)*dgm10(1,2)+10*dgm01(1,2)*dgm10(1,3)&
&   +6*dgm01(1,1)*dgm10(2,3)+6*gm(1,3)*d2gm(1,2)+4*gm(1,2)*d2gm(1,3)&
&   +6*gm(1,1)*d2gm(2,3))
   cm(1,10)=gm(1,1)*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))&
&   +3*gm(1,3)**2*d2gm(1,3)-gm(3,3)*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)&
&   *dgm10(1,3)+gm(1,1)*d2gm(1,3))+gm(1,3)*(2*dgm01(3,3)*dgm10(1,1)&
&   +12*dgm01(1,3)*dgm10(1,3)+2*dgm01(1,1)*dgm10(3,3)+2*gm(1,1)*d2gm(3,3))
   cm(2,10)=((-2*gm(3,3)*dgm01(2,2)+12*gm(2,3)*dgm01(2,3)-2*gm(2,2)&
&   *dgm01(3,3))*dgm10(1,3)+dgm01(3,3)*(6*gm(2,3)*dgm10(1,2)-2*gm(2,2)&
&   *dgm10(1,3)-2*gm(1,3)*dgm10(2,2)+6*gm(1,2)*dgm10(2,3))+(6*gm(2,3)&
&   *dgm01(1,2)-2*gm(2,2)*dgm01(1,3)-2*gm(1,3)*dgm01(2,2)+6*gm(1,2)&
&   *dgm01(2,3))*dgm10(3,3)+dgm01(1,3)*(-2*gm(3,3)*dgm10(2,2)+12*gm(2,3)&
&   *dgm10(2,3)-2*gm(2,2)*dgm10(3,3))+(6*gm(2,3)**2-2*gm(2,2)*gm(3,3))&
&   *d2gm(1,3)+(-2*gm(1,3)*gm(2,2)+6*gm(1,2)*gm(2,3))*d2gm(3,3))&
&   /2.d0
   cm(3,10)=4*gm(1,3)*dgm01(3,3)*dgm10(3,3)+2*gm(3,3)**2*d2gm(1,3)&
&   +gm(3,3)*(6*dgm01(3,3)*dgm10(1,3)+6*dgm01(1,3)*dgm10(3,3)+2*gm(1,3)&
&   *d2gm(3,3))
   cm(4,10)=gm(1,3)*dgm01(3,3)*dgm10(2,3)+gm(1,3)*dgm01(2,3)*dgm10(3,3)&
&   +6*gm(1,2)*dgm01(3,3)*dgm10(3,3)+gm(3,3)*(3*dgm01(3,3)*dgm10(1,2)&
&   +4*dgm01(2,3)*dgm10(1,3)+4*dgm01(1,3)*dgm10(2,3)+3*dgm01(1,2)&
&   *dgm10(3,3)+4*gm(2,3)*d2gm(1,3)+3*gm(1,2)*d2gm(3,3))+gm(2,3)&
&   *(5*dgm01(3,3)*dgm10(1,3)+5*dgm01(1,3)*dgm10(3,3)+gm(1,3)*d2gm(3,3))
   cm(5,10)=6*gm(1,1)*dgm01(3,3)*dgm10(3,3)+6*gm(1,3)*(dgm01(3,3)&
&   *dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+gm(1,3)**2*d2gm(3,3)+gm(3,3)&
&   *(3*dgm01(3,3)*dgm10(1,1)+8*dgm01(1,3)*dgm10(1,3)+3*dgm01(1,1)&
&   *dgm10(3,3)+4*gm(1,3)*d2gm(1,3)+3*gm(1,1)*d2gm(3,3))
   cm(6,10)=gm(1,3)*dgm01(3,3)*dgm10(1,2)+6*gm(1,3)*dgm01(2,3)&
&   *dgm10(1,3)-gm(1,2)*dgm01(3,3)*dgm10(1,3)+6*gm(1,3)*dgm01(1,3)&
&   *dgm10(2,3)+3*gm(1,1)*dgm01(3,3)*dgm10(2,3)+gm(1,3)*dgm01(1,2)&
&   *dgm10(3,3)-gm(1,2)*dgm01(1,3)*dgm10(3,3)+3*gm(1,1)*dgm01(2,3)&
&   *dgm10(3,3)-2*gm(3,3)*(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)*dgm10(1,3)&
&   +gm(1,2)*d2gm(1,3))+gm(1,2)*gm(1,3)*d2gm(3,3)+gm(2,3)*(3*dgm01(3,3)&
&   *dgm10(1,1)+12*dgm01(1,3)*dgm10(1,3)+3*dgm01(1,1)*dgm10(3,3)&
&   +6*gm(1,3)*d2gm(1,3)+3*gm(1,1)*d2gm(3,3))
   cm(1,11)=-gm(1,1)*dgm01(2,2)*dgm10(2,2)+3*gm(1,2)*(dgm01(2,2)&
&   *dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+1.5d0*gm(1,2)**2*d2gm(2,2)&
&   -0.5d0*gm(2,2)*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2)+gm(1,1)&
&   *d2gm(2,2))
   cm(2,11)=gm(2,2)*(4*dgm01(2,2)*dgm10(2,2)+gm(2,2)*d2gm(2,2))
   cm(3,11)=3*gm(2,3)*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))&
&   -0.5d0*gm(2,2)*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3))&
&   +1.5d0*gm(2,3)**2*d2gm(2,2)+gm(3,3)*(-dgm01(2,2)*dgm10(2,2)-0.5d0*gm(2,2)&
&   *d2gm(2,2))
   cm(4,11)=2*gm(2,2)*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))&
&   +gm(2,3)*(4*dgm01(2,2)*dgm10(2,2)+2*gm(2,2)*d2gm(2,2))
   cm(5,11)=((6*gm(2,3)*dgm01(1,2)-2*gm(2,2)*dgm01(1,3)-2*gm(1,3)&
&   *dgm01(2,2)+6*gm(1,2)*dgm01(2,3))*dgm10(2,2)+dgm01(2,2)*(6*gm(2,3)&
&   *dgm10(1,2)-2*gm(2,2)*dgm10(1,3)-2*gm(1,3)*dgm10(2,2)+6*gm(1,2)&
&   *dgm10(2,3))+(-2*gm(1,3)*gm(2,2)+6*gm(1,2)*gm(2,3))*d2gm(2,2))&
&   /2.d0
   cm(6,11)=4*gm(1,2)*dgm01(2,2)*dgm10(2,2)+2*gm(2,2)*(dgm01(2,2)&
&   *dgm10(1,2)+dgm01(1,2)*dgm10(2,2)+gm(1,2)*d2gm(2,2))
   cm(1,12)=((-2*gm(3,3)*dgm01(1,1)+12*gm(1,3)*dgm01(1,3)-2*gm(1,1)&
&   *dgm01(3,3))*dgm10(2,2)+dgm01(3,3)*(-2*gm(2,2)*dgm10(1,1)+12*gm(1,2)&
&   *dgm10(1,2)-2*gm(1,1)*dgm10(2,2))+4*(-2*gm(2,3)*dgm01(1,1)+6*gm(1,3)&
&   *dgm01(1,2)+6*gm(1,2)*dgm01(1,3)-2*gm(1,1)*dgm01(2,3))*dgm10(2,3)&
&   +4*dgm01(2,3)*(-2*gm(2,3)*dgm10(1,1)+6*gm(1,3)*dgm10(1,2)+6*gm(1,2)&
&   *dgm10(1,3)-2*gm(1,1)*dgm10(2,3))+(-2*gm(2,2)*dgm01(1,1)+12*gm(1,2)&
&   *dgm01(1,2)-2*gm(1,1)*dgm01(2,2))*dgm10(3,3)+dgm01(2,2)*(-2*gm(3,3)&
&   *dgm10(1,1)+12*gm(1,3)*dgm10(1,3)-2*gm(1,1)*dgm10(3,3))+(6*gm(1,3)&
&   **2-2*gm(1,1)*gm(3,3))*d2gm(2,2)+4*(6*gm(1,2)*gm(1,3)-2*gm(1,1)&
&   *gm(2,3))*d2gm(2,3)+(6*gm(1,2)**2-2*gm(1,1)*gm(2,2))*d2gm(3,3))&
&   /4.d0
   cm(2,12)=1.5d0*gm(2,3)**2*d2gm(2,2)+gm(3,3)*(-dgm01(2,2)*dgm10(2,2)&
&   -0.5d0*gm(2,2)*d2gm(2,2))+gm(2,3)*(7*dgm01(2,3)*dgm10(2,2)+7*dgm01(2,2)&
&   *dgm10(2,3)+4*gm(2,2)*d2gm(2,3))+gm(2,2)*(1.5d0*dgm01(3,3)*dgm10(2,2)&
&   +8*dgm01(2,3)*dgm10(2,3)+1.5d0*dgm01(2,2)*dgm10(3,3)+gm(2,2)&
&   *d2gm(3,3))
   cm(3,12)=-gm(2,2)*dgm01(3,3)*dgm10(3,3)+7*gm(2,3)*(dgm01(3,3)&
&   *dgm10(2,3)+dgm01(2,3)*dgm10(3,3))+gm(3,3)**2*d2gm(2,2)+1.5d0*gm(2,3)&
&   **2*d2gm(3,3)+gm(3,3)*(1.5d0*dgm01(3,3)*dgm10(2,2)+8*dgm01(2,3)&
&   *dgm10(2,3)+1.5d0*dgm01(2,2)*dgm10(3,3)+4*gm(2,3)*d2gm(2,3)-0.5d0*gm(2,2)&
&   *d2gm(3,3))
   cm(4,12)=8*gm(2,2)*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3))&
&   +2*gm(2,3)**2*d2gm(2,3)+gm(3,3)*(8*dgm01(2,3)*dgm10(2,2)+8*dgm01(2,2)&
&   *dgm10(2,3)+2*gm(2,3)*d2gm(2,2)+6*gm(2,2)*d2gm(2,3))+gm(2,3)&
&   *(4*dgm01(3,3)*dgm10(2,2)+8*dgm01(2,3)*dgm10(2,3)+4*dgm01(2,2)&
&   *dgm10(3,3)+2*gm(2,2)*d2gm(3,3))
   cm(5,12)=-gm(2,2)*dgm01(3,3)*dgm10(1,3)+gm(1,3)*dgm01(3,3)*dgm10(2,2)&
&   +4*gm(1,3)*dgm01(2,3)*dgm10(2,3)+9*gm(1,2)*dgm01(3,3)*dgm10(2,3)&
&   -gm(2,2)*dgm01(1,3)*dgm10(3,3)+gm(1,3)*dgm01(2,2)*dgm10(3,3)&
&   +9*gm(1,2)*dgm01(2,3)*dgm10(3,3)+gm(3,3)*(6*dgm01(2,3)*dgm10(1,2)&
&   +2*dgm01(2,2)*dgm10(1,3)+2*dgm01(1,3)*dgm10(2,2)+6*dgm01(1,2)&
&   *dgm10(2,3)+2*gm(1,3)*d2gm(2,2)+6*gm(1,2)*d2gm(2,3))-gm(1,3)&
&   *gm(2,2)*d2gm(3,3)+gm(2,3)*(3*dgm01(3,3)*dgm10(1,2)+2*dgm01(2,3)&
&   *dgm10(1,3)+2*dgm01(1,3)*dgm10(2,3)+3*dgm01(1,2)*dgm10(3,3)+2*gm(1,3)&
&   *d2gm(2,3)+3*gm(1,2)*d2gm(3,3))
   cm(6,12)=2*gm(2,2)*dgm01(3,3)*dgm10(1,2)+6*gm(2,2)*dgm01(2,3)&
&   *dgm10(1,3)+9*gm(1,3)*dgm01(2,3)*dgm10(2,2)+gm(1,2)*dgm01(3,3)&
&   *dgm10(2,2)+6*gm(2,2)*dgm01(1,3)*dgm10(2,3)+9*gm(1,3)*dgm01(2,2)&
&   *dgm10(2,3)+4*gm(1,2)*dgm01(2,3)*dgm10(2,3)+2*gm(2,2)*dgm01(1,2)&
&   *dgm10(3,3)+gm(1,2)*dgm01(2,2)*dgm10(3,3)-gm(3,3)*(dgm01(2,2)&
&   *dgm10(1,2)+dgm01(1,2)*dgm10(2,2)+gm(1,2)*d2gm(2,2))+6*gm(1,3)&
&   *gm(2,2)*d2gm(2,3)+gm(2,3)*(2*dgm01(2,3)*dgm10(1,2)+3*dgm01(2,2)&
&   *dgm10(1,3)+3*dgm01(1,3)*dgm10(2,2)+2*dgm01(1,2)*dgm10(2,3)+3*gm(1,3)&
&   *d2gm(2,2)+2*gm(1,2)*d2gm(2,3))+2*gm(1,2)*gm(2,2)*d2gm(3,3)
   cm(1,13)=((-2*gm(2,3)*dgm01(1,1)+6*gm(1,3)*dgm01(1,2)+6*gm(1,2)&
&   *dgm01(1,3)-2*gm(1,1)*dgm01(2,3))*dgm10(2,2)+dgm01(2,3)*(-2*gm(2,2)&
&   *dgm10(1,1)+12*gm(1,2)*dgm10(1,2)-2*gm(1,1)*dgm10(2,2))+(-2*gm(2,2)&
&   *dgm01(1,1)+12*gm(1,2)*dgm01(1,2)-2*gm(1,1)*dgm01(2,2))*dgm10(2,3)&
&   +dgm01(2,2)*(-2*gm(2,3)*dgm10(1,1)+6*gm(1,3)*dgm10(1,2)+6*gm(1,2)&
&   *dgm10(1,3)-2*gm(1,1)*dgm10(2,3))+(6*gm(1,2)*gm(1,3)-2*gm(1,1)&
&   *gm(2,3))*d2gm(2,2)+(6*gm(1,2)**2-2*gm(1,1)*gm(2,2))*d2gm(2,3))&
&   /2.d0
   cm(2,13)=gm(2,3)*(4*dgm01(2,2)*dgm10(2,2)+2*gm(2,2)*d2gm(2,2))&
&   +gm(2,2)*(6*dgm01(2,3)*dgm10(2,2)+6*dgm01(2,2)*dgm10(2,3)+2*gm(2,2)&
&   *d2gm(2,3))
   cm(3,13)=gm(2,3)*(2*dgm01(3,3)*dgm10(2,2)+12*dgm01(2,3)*dgm10(2,3)&
&   +2*dgm01(2,2)*dgm10(3,3))-gm(2,2)*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)&
&   *dgm10(3,3))+3*gm(2,3)**2*d2gm(2,3)+gm(3,3)*(1*dgm01(2,3)*dgm10(2,2)&
&   +dgm01(2,2)*dgm10(2,3)+2*gm(2,3)*d2gm(2,2)-gm(2,2)*d2gm(2,3))
   cm(4,13)=gm(2,2)*(3*dgm01(3,3)*dgm10(2,2)+8*dgm01(2,3)*dgm10(2,3)&
&   +3*dgm01(2,2)*dgm10(3,3))+gm(2,3)**2*d2gm(2,2)+gm(3,3)*(6*dgm01(2,2)&
&   *dgm10(2,2)+3*gm(2,2)*d2gm(2,2))+gm(2,3)*(6*dgm01(2,3)*dgm10(2,2)&
&   +6*dgm01(2,2)*dgm10(2,3)+4*gm(2,2)*d2gm(2,3))
   cm(5,13)=-2*gm(2,2)*dgm01(2,3)*dgm10(1,3)-gm(1,3)*dgm01(2,3)&
&   *dgm10(2,2)+3*gm(1,2)*dgm01(3,3)*dgm10(2,2)-2*gm(2,2)*dgm01(1,3)&
&   *dgm10(2,3)-gm(1,3)*dgm01(2,2)*dgm10(2,3)+12*gm(1,2)*dgm01(2,3)&
&   *dgm10(2,3)+3*gm(1,2)*dgm01(2,2)*dgm10(3,3)+3*gm(3,3)*(dgm01(2,2)&
&   *dgm10(1,2)+dgm01(1,2)*dgm10(2,2)+gm(1,2)*d2gm(2,2))-2*gm(1,3)&
&   *gm(2,2)*d2gm(2,3)+gm(2,3)*(6*dgm01(2,3)*dgm10(1,2)+dgm01(2,2)&
&   *dgm10(1,3)+dgm01(1,3)*dgm10(2,2)+6*dgm01(1,2)*dgm10(2,3)+gm(1,3)&
&   *d2gm(2,2)+6*gm(1,2)*d2gm(2,3))
   cm(6,13)=6*gm(1,3)*dgm01(2,2)*dgm10(2,2)+5*gm(1,2)*dgm01(2,3)&
&   *dgm10(2,2)+5*gm(1,2)*dgm01(2,2)*dgm10(2,3)+gm(2,3)*(dgm01(2,2)&
&   *dgm10(1,2)+dgm01(1,2)*dgm10(2,2)+gm(1,2)*d2gm(2,2))+gm(2,2)&
&   *(4*dgm01(2,3)*dgm10(1,2)+3*dgm01(2,2)*dgm10(1,3)+3*dgm01(1,3)&
&   *dgm10(2,2)+4*dgm01(1,2)*dgm10(2,3)+3*gm(1,3)*d2gm(2,2)+4*gm(1,2)&
&   *d2gm(2,3))
   cm(1,14)=((-2*gm(3,3)*dgm01(1,1)+12*gm(1,3)*dgm01(1,3)-2*gm(1,1)&
&   *dgm01(3,3))*dgm10(2,3)+dgm01(3,3)*(-2*gm(2,3)*dgm10(1,1)+6*gm(1,3)&
&   *dgm10(1,2)+6*gm(1,2)*dgm10(1,3)-2*gm(1,1)*dgm10(2,3))+(-2*gm(2,3)&
&   *dgm01(1,1)+6*gm(1,3)*dgm01(1,2)+6*gm(1,2)*dgm01(1,3)-2*gm(1,1)&
&   *dgm01(2,3))*dgm10(3,3)+dgm01(2,3)*(-2*gm(3,3)*dgm10(1,1)+12*gm(1,3)&
&   *dgm10(1,3)-2*gm(1,1)*dgm10(3,3))+(6*gm(1,3)**2-2*gm(1,1)*gm(3,3))&
&   *d2gm(2,3)+(6*gm(1,2)*gm(1,3)-2*gm(1,1)*gm(2,3))*d2gm(3,3))/2.d0
   cm(2,14)=gm(2,2)*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3))&
&   +3*gm(2,3)**2*d2gm(2,3)-gm(3,3)*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)&
&   *dgm10(2,3)+gm(2,2)*d2gm(2,3))+gm(2,3)*(2*dgm01(3,3)*dgm10(2,2)&
&   +12*dgm01(2,3)*dgm10(2,3)+2*dgm01(2,2)*dgm10(3,3)+2*gm(2,2)*d2gm(3,3))
   cm(3,14)=4*gm(2,3)*dgm01(3,3)*dgm10(3,3)+2*gm(3,3)**2*d2gm(2,3)&
&   +gm(3,3)*(6*dgm01(3,3)*dgm10(2,3)+6*dgm01(2,3)*dgm10(3,3)+2*gm(2,3)&
&   *d2gm(3,3))
   cm(4,14)=6*gm(2,2)*dgm01(3,3)*dgm10(3,3)+6*gm(2,3)*(dgm01(3,3)&
&   *dgm10(2,3)+dgm01(2,3)*dgm10(3,3))+gm(2,3)**2*d2gm(3,3)+gm(3,3)&
&   *(3*dgm01(3,3)*dgm10(2,2)+8*dgm01(2,3)*dgm10(2,3)+3*dgm01(2,2)&
&   *dgm10(3,3)+4*gm(2,3)*d2gm(2,3)+3*gm(2,2)*d2gm(3,3))
   cm(5,14)=5*gm(1,3)*dgm01(3,3)*dgm10(2,3)+5*gm(1,3)*dgm01(2,3)&
&   *dgm10(3,3)+6*gm(1,2)*dgm01(3,3)*dgm10(3,3)+gm(3,3)*(3*dgm01(3,3)&
&   *dgm10(1,2)+4*dgm01(2,3)*dgm10(1,3)+4*dgm01(1,3)*dgm10(2,3)+3*dgm01(1,2)&
&   *dgm10(3,3)+4*gm(1,3)*d2gm(2,3)+3*gm(1,2)*d2gm(3,3))+gm(2,3)&
&   *(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3)+gm(1,3)*d2gm(3,3))
   cm(6,14)=3*gm(2,2)*dgm01(3,3)*dgm10(1,3)+3*gm(1,3)*dgm01(3,3)&
&   *dgm10(2,2)+12*gm(1,3)*dgm01(2,3)*dgm10(2,3)-gm(1,2)*dgm01(3,3)&
&   *dgm10(2,3)+3*gm(2,2)*dgm01(1,3)*dgm10(3,3)+3*gm(1,3)*dgm01(2,2)&
&   *dgm10(3,3)-gm(1,2)*dgm01(2,3)*dgm10(3,3)-2*gm(3,3)*(dgm01(2,3)&
&   *dgm10(1,2)+dgm01(1,2)*dgm10(2,3)+gm(1,2)*d2gm(2,3))+3*gm(1,3)&
&   *gm(2,2)*d2gm(3,3)+gm(2,3)*(1*dgm01(3,3)*dgm10(1,2)+6*dgm01(2,3)&
&   *dgm10(1,3)+6*dgm01(1,3)*dgm10(2,3)+dgm01(1,2)*dgm10(3,3)+6*gm(1,3)&
&   *d2gm(2,3)+gm(1,2)*d2gm(3,3))
   cm(1,15)=-gm(1,1)*dgm01(3,3)*dgm10(3,3)+3*gm(1,3)*(dgm01(3,3)&
&   *dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+1.5d0*gm(1,3)**2*d2gm(3,3)&
&   -0.5d0*gm(3,3)*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3)+gm(1,1)&
&   *d2gm(3,3))
   cm(2,15)=-gm(2,2)*dgm01(3,3)*dgm10(3,3)+3*gm(2,3)*(dgm01(3,3)&
&   *dgm10(2,3)+dgm01(2,3)*dgm10(3,3))+1.5d0*gm(2,3)**2*d2gm(3,3)&
&   -0.5d0*gm(3,3)*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3)+gm(2,2)&
&   *d2gm(3,3))
   cm(3,15)=gm(3,3)*(4*dgm01(3,3)*dgm10(3,3)+gm(3,3)*d2gm(3,3))
   cm(4,15)=4*gm(2,3)*dgm01(3,3)*dgm10(3,3)+2*gm(3,3)*(dgm01(3,3)&
&   *dgm10(2,3)+dgm01(2,3)*dgm10(3,3)+gm(2,3)*d2gm(3,3))
   cm(5,15)=4*gm(1,3)*dgm01(3,3)*dgm10(3,3)+2*gm(3,3)*(dgm01(3,3)&
&   *dgm10(1,3)+dgm01(1,3)*dgm10(3,3)+gm(1,3)*d2gm(3,3))
   cm(6,15)=((-2*gm(3,3)*dgm01(1,2)+6*gm(2,3)*dgm01(1,3)+6*gm(1,3)&
&   *dgm01(2,3)-2*gm(1,2)*dgm01(3,3))*dgm10(3,3)+dgm01(3,3)*(-2*gm(3,3)&
&   *dgm10(1,2)+6*gm(2,3)*dgm10(1,3)+6*gm(1,3)*dgm10(2,3)-2*gm(1,2)&
&   *dgm10(3,3))+(6*gm(1,3)*gm(2,3)-2*gm(1,2)*gm(3,3))*d2gm(3,3))&
&   /2.d0
 elseif(rank==3)then
   cm(1,1)=gm(1,1)**2*(6*dgm01(1,1)*dgm10(1,1)+gm(1,1)*d2gm(1,1))
   cm(2,1)=9*gm(1,1)*gm(1,2)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)&
&   *dgm10(1,2))+gm(1,2)**2*(9*dgm01(1,1)*dgm10(1,1)+4.5d0*gm(1,1)&
&   *d2gm(1,1))+gm(1,1)*(-1.5d0*gm(1,1)*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)&
&   *dgm10(2,2))+gm(2,2)*(-6*dgm01(1,1)*dgm10(1,1)-1.5d0*gm(1,1)&
&   *d2gm(1,1)))
   cm(3,1)=9*gm(1,1)*gm(1,3)*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)&
&   *dgm10(1,3))+gm(1,3)**2*(9*dgm01(1,1)*dgm10(1,1)+4.5d0*gm(1,1)&
&   *d2gm(1,1))+gm(1,1)*(-1.5d0*gm(1,1)*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)&
&   *dgm10(3,3))+gm(3,3)*(-6*dgm01(1,1)*dgm10(1,1)-1.5d0*gm(1,1)&
&   *d2gm(1,1)))
   cm(4,1)=gm(1,1)*(9*gm(1,3)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)&
&   *dgm10(1,2))-3*gm(1,1)*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3))&
&   +gm(2,3)*(-12*dgm01(1,1)*dgm10(1,1)-3*gm(1,1)*d2gm(1,1)))+gm(1,2)&
&   *(9*gm(1,1)*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+gm(1,3)&
&   *(18*dgm01(1,1)*dgm10(1,1)+9*gm(1,1)*d2gm(1,1)))
   cm(5,1)=gm(1,1)*(3*gm(1,1)*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)&
&   *dgm10(1,3))+gm(1,3)*(12*dgm01(1,1)*dgm10(1,1)+3*gm(1,1)*d2gm(1,1)))
   cm(6,1)=gm(1,1)*(3*gm(1,1)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)&
&   *dgm10(1,2))+gm(1,2)*(12*dgm01(1,1)*dgm10(1,1)+3*gm(1,1)*d2gm(1,1)))
   cm(7,1)=(270*gm(1,2)**2*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))&
&   -54*gm(1,1)*gm(2,2)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))&
&   +90*gm(1,2)**3*d2gm(1,1)+gm(1,2)*(-54*gm(1,1)*(dgm01(2,2)*dgm10(1,1)&
&   +dgm01(1,1)*dgm10(2,2))+gm(2,2)*(-108*dgm01(1,1)*dgm10(1,1)-54*gm(1,1)&
&   *d2gm(1,1))))/36.d0
   cm(8,1)=((90*gm(1,3)**2*dgm01(1,2)+gm(1,1)*(-18*gm(3,3)*dgm01(1,2)&
&   -36*gm(2,3)*dgm01(1,3))+gm(1,3)*(-36*gm(2,3)*dgm01(1,1)+180*gm(1,2)&
&   *dgm01(1,3)-36*gm(1,1)*dgm01(2,3))-18*gm(1,2)*(gm(3,3)*dgm01(1,1)&
&   +gm(1,1)*dgm01(3,3)))*dgm10(1,1)+dgm01(1,1)*(90*gm(1,3)**2*dgm10(1,2)&
&   +gm(1,1)*(-18*gm(3,3)*dgm10(1,2)-36*gm(2,3)*dgm10(1,3))+gm(1,3)&
&   *(-36*gm(2,3)*dgm10(1,1)+180*gm(1,2)*dgm10(1,3)-36*gm(1,1)*dgm10(2,3))&
&   -18*gm(1,2)*(gm(3,3)*dgm10(1,1)+gm(1,1)*dgm10(3,3)))+(-36*gm(1,1)&
&   *gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)**2-18*gm(1,1)*gm(3,3)))&
&   *d2gm(1,1))/12.d0
   cm(9,1)=((90*gm(1,2)**2*dgm01(1,3)+gm(1,1)*(-36*gm(2,3)*dgm01(1,2)&
&   -18*gm(2,2)*dgm01(1,3))+gm(1,3)*(-18*gm(2,2)*dgm01(1,1)+180*gm(1,2)&
&   *dgm01(1,2)-18*gm(1,1)*dgm01(2,2))-36*gm(1,2)*(gm(2,3)*dgm01(1,1)&
&   +gm(1,1)*dgm01(2,3)))*dgm10(1,1)+dgm01(1,1)*(90*gm(1,2)**2*dgm10(1,3)&
&   +gm(1,1)*(-36*gm(2,3)*dgm10(1,2)-18*gm(2,2)*dgm10(1,3))+gm(1,3)&
&   *(-18*gm(2,2)*dgm10(1,1)+180*gm(1,2)*dgm10(1,2)-18*gm(1,1)*dgm10(2,2))&
&   -36*gm(1,2)*(gm(2,3)*dgm10(1,1)+gm(1,1)*dgm10(2,3)))+(90*gm(1,2)&
&   **2*gm(1,3)-18*gm(1,1)*gm(1,3)*gm(2,2)-36*gm(1,1)*gm(1,2)*gm(2,3))&
&   *d2gm(1,1))/12.d0
   cm(10,1)=(270*gm(1,3)**2*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))&
&   -54*gm(1,1)*gm(3,3)*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))&
&   +90*gm(1,3)**3*d2gm(1,1)+gm(1,3)*(-54*gm(1,1)*(dgm01(3,3)*dgm10(1,1)&
&   +dgm01(1,1)*dgm10(3,3))+gm(3,3)*(-108*dgm01(1,1)*dgm10(1,1)-54*gm(1,1)&
&   *d2gm(1,1))))/36.d0
   cm(1,2)=gm(1,2)**2*(9*dgm01(1,1)*dgm10(1,1)+4.5d0*gm(1,1)*d2gm(1,1))&
&   +gm(1,1)*gm(1,2)*(21*dgm01(1,2)*dgm10(1,1)+21*dgm01(1,1)*dgm10(1,2)&
&   +6*gm(1,1)*d2gm(1,2))+gm(1,1)*(gm(2,2)*(-6*dgm01(1,1)*dgm10(1,1)&
&   -1.5d0*gm(1,1)*d2gm(1,1))+gm(1,1)*(1.5d0*dgm01(2,2)*dgm10(1,1)&
&   +12*dgm01(1,2)*dgm10(1,2)+1.5d0*dgm01(1,1)*dgm10(2,2)+gm(1,1)&
&   *d2gm(2,2)))
   cm(2,2)=-3*gm(1,1)**2*dgm01(2,2)*dgm10(2,2)+24*gm(1,1)*gm(1,2)&
&   *(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+gm(2,2)**2*(12*dgm01(1,1)&
&   *dgm10(1,1)+6*gm(1,1)*d2gm(1,1))+3*gm(1,2)**3*d2gm(1,2)+gm(1,2)&
&   **2*(7.5d0*dgm01(2,2)*dgm10(1,1)+18*dgm01(1,2)*dgm10(1,2)+7.5d0*dgm01(1,1)&
&   *dgm10(2,2)+4.5d0*gm(1,1)*d2gm(2,2))+gm(2,2)*(3*gm(1,2)**2*d2gm(1,1)&
&   +gm(1,2)*(21*dgm01(1,2)*dgm10(1,1)+21*dgm01(1,1)*dgm10(1,2)+15*gm(1,1)&
&   *d2gm(1,2))+gm(1,1)*(9*dgm01(2,2)*dgm10(1,1)+30*dgm01(1,2)*dgm10(1,2)&
&   +9*dgm01(1,1)*dgm10(2,2)-1.5d0*gm(1,1)*d2gm(2,2)))
   cm(3,2)=-15*gm(1,2)*gm(3,3)*dgm01(1,2)*dgm10(1,1)+1.5d0*gm(1,3)&
&   **2*dgm01(2,2)*dgm10(1,1)-4.5d0*gm(1,1)*gm(3,3)*dgm01(2,2)*dgm10(1,1)&
&   +9*gm(1,2)*gm(1,3)*dgm01(2,3)*dgm10(1,1)-3*gm(1,2)**2*dgm01(3,3)&
&   *dgm10(1,1)-15*gm(1,2)*gm(3,3)*dgm01(1,1)*dgm10(1,2)+6*gm(1,3)&
&   **2*dgm01(1,2)*dgm10(1,2)-18*gm(1,1)*gm(3,3)*dgm01(1,2)*dgm10(1,2)&
&   +6*gm(1,2)*gm(1,3)*dgm01(1,3)*dgm10(1,2)+24*gm(1,1)*gm(1,3)*dgm01(2,3)&
&   *dgm10(1,2)-9*gm(1,1)*gm(1,2)*dgm01(3,3)*dgm10(1,2)+6*gm(1,2)&
&   *gm(1,3)*dgm01(1,2)*dgm10(1,3)+9*gm(1,1)*gm(1,3)*dgm01(2,2)*dgm10(1,3)&
&   +1.5d0*gm(1,3)**2*dgm01(1,1)*dgm10(2,2)-4.5d0*gm(1,1)*gm(3,3)&
&   *dgm01(1,1)*dgm10(2,2)+9*gm(1,1)*gm(1,3)*dgm01(1,3)*dgm10(2,2)&
&   -1.5d0*gm(1,1)**2*dgm01(3,3)*dgm10(2,2)+9*gm(1,2)*gm(1,3)*dgm01(1,1)&
&   *dgm10(2,3)+24*gm(1,1)*gm(1,3)*dgm01(1,2)*dgm10(2,3)-3*gm(1,2)&
&   **2*dgm01(1,1)*dgm10(3,3)-9*gm(1,1)*gm(1,2)*dgm01(1,2)*dgm10(3,3)&
&   -1.5d0*gm(1,1)**2*dgm01(2,2)*dgm10(3,3)-3*gm(1,2)**2*gm(3,3)&
&   *d2gm(1,1)+gm(2,3)**2*(15*dgm01(1,1)*dgm10(1,1)+7.5d0*gm(1,1)&
&   *d2gm(1,1))+gm(2,2)*(-6*gm(1,3)*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)&
&   *dgm10(1,3))-1.5d0*gm(1,1)*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)&
&   *dgm10(3,3))-3*gm(1,3)**2*d2gm(1,1)+gm(3,3)*(-3*dgm01(1,1)*dgm10(1,1)&
&   -1.5d0*gm(1,1)*d2gm(1,1)))+3*gm(1,2)*gm(1,3)**2*d2gm(1,2)-9*gm(1,1)&
&   *gm(1,2)*gm(3,3)*d2gm(1,2)+gm(2,3)*(9*gm(1,2)*(dgm01(1,3)*dgm10(1,1)&
&   +dgm01(1,1)*dgm10(1,3))+gm(1,1)*(15*dgm01(2,3)*dgm10(1,1)+24*dgm01(1,3)&
&   *dgm10(1,2)+24*dgm01(1,2)*dgm10(1,3)+15*dgm01(1,1)*dgm10(2,3))&
&   +gm(1,3)*(33*dgm01(1,2)*dgm10(1,1)+33*dgm01(1,1)*dgm10(1,2)+9*gm(1,2)&
&   *d2gm(1,1)+24*gm(1,1)*d2gm(1,2)))+4.5d0*gm(1,1)*gm(1,3)**2*d2gm(2,2)&
&   -1.5d0*gm(1,1)**2*gm(3,3)*d2gm(2,2)
   cm(4,2)=gm(1,2)**2*(3*dgm01(2,3)*dgm10(1,1)+6*dgm01(1,3)*dgm10(1,2)&
&   +6*dgm01(1,2)*dgm10(1,3)+3*dgm01(1,1)*dgm10(2,3)+3*gm(2,3)*d2gm(1,1)&
&   +6*gm(1,3)*d2gm(1,2))+gm(2,2)*(3*gm(1,2)*dgm01(1,3)*dgm10(1,1)&
&   +12*gm(1,1)*dgm01(2,3)*dgm10(1,1)+24*gm(1,1)*dgm01(1,3)*dgm10(1,2)&
&   +3*gm(1,2)*dgm01(1,1)*dgm10(1,3)+24*gm(1,1)*dgm01(1,2)*dgm10(1,3)&
&   +12*gm(1,1)*dgm01(1,1)*dgm10(2,3)+gm(2,3)*(24*dgm01(1,1)*dgm10(1,1)&
&   +12*gm(1,1)*d2gm(1,1))+gm(1,3)*(27*dgm01(1,2)*dgm10(1,1)+27*dgm01(1,1)&
&   *dgm10(1,2)+3*gm(1,2)*d2gm(1,1)+24*gm(1,1)*d2gm(1,2)))+gm(1,1)&
&   *(33*gm(1,3)*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))-3*gm(1,1)&
&   *(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+gm(2,3)*(6*dgm01(2,2)&
&   *dgm10(1,1)+12*dgm01(1,2)*dgm10(1,2)+6*dgm01(1,1)*dgm10(2,2)&
&   -3*gm(1,1)*d2gm(2,2)))+gm(1,2)*(gm(1,1)*(6*dgm01(2,3)*dgm10(1,2)&
&   +9*dgm01(2,2)*dgm10(1,3)+9*dgm01(1,3)*dgm10(2,2)+6*dgm01(1,2)&
&   *dgm10(2,3))+gm(2,3)*(12*dgm01(1,2)*dgm10(1,1)+12*dgm01(1,1)&
&   *dgm10(1,2)+6*gm(1,1)*d2gm(1,2))+gm(1,3)*(12*dgm01(2,2)*dgm10(1,1)&
&   +24*dgm01(1,2)*dgm10(1,2)+12*dgm01(1,1)*dgm10(2,2)+9*gm(1,1)&
&   *d2gm(2,2)))
   cm(5,2)=1.5d0*gm(1,2)**2*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))&
&   +gm(1,2)*(gm(1,1)*(12*dgm01(2,3)*dgm10(1,1)+6*dgm01(1,3)*dgm10(1,2)&
&   +6*dgm01(1,2)*dgm10(1,3)+12*dgm01(1,1)*dgm10(2,3))+gm(2,3)*(24*dgm01(1,1)&
&   *dgm10(1,1)+12*gm(1,1)*d2gm(1,1)))+gm(1,1)*(-4.5d0*gm(2,2)*(dgm01(1,3)&
&   *dgm10(1,1)+dgm01(1,1)*dgm10(1,3))+gm(1,1)*(12*dgm01(2,3)*dgm10(1,2)&
&   +3*dgm01(2,2)*dgm10(1,3)+3*dgm01(1,3)*dgm10(2,2)+12*dgm01(1,2)&
&   *dgm10(2,3))+gm(2,3)*(36*dgm01(1,2)*dgm10(1,1)+36*dgm01(1,1)&
&   *dgm10(1,2)+12*gm(1,1)*d2gm(1,2)))+gm(1,3)*(1.5d0*gm(1,2)**2*d2gm(1,1)&
&   +gm(2,2)*(-9*dgm01(1,1)*dgm10(1,1)-4.5d0*gm(1,1)*d2gm(1,1))+gm(1,2)&
&   *(9*dgm01(1,2)*dgm10(1,1)+9*dgm01(1,1)*dgm10(1,2)+6*gm(1,1)*d2gm(1,2))&
&   +gm(1,1)*(1.5d0*dgm01(2,2)*dgm10(1,1)+12*dgm01(1,2)*dgm10(1,2)&
&   +1.5d0*dgm01(1,1)*dgm10(2,2)+3*gm(1,1)*d2gm(2,2)))
   cm(6,2)=1.5d0*gm(1,2)**3*d2gm(1,1)+gm(1,2)**2*(10.5d0*dgm01(1,2)&
&   *dgm10(1,1)+10.5d0*dgm01(1,1)*dgm10(1,2)+6*gm(1,1)*d2gm(1,2))&
&   +gm(1,1)*(15*gm(1,1)*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))&
&   +gm(2,2)*(31.5d0*dgm01(1,2)*dgm10(1,1)+31.5d0*dgm01(1,1)*dgm10(1,2)&
&   +12*gm(1,1)*d2gm(1,2)))+gm(1,2)*(gm(2,2)*(15*dgm01(1,1)*dgm10(1,1)&
&   +7.5d0*gm(1,1)*d2gm(1,1))+gm(1,1)*(13.5d0*dgm01(2,2)*dgm10(1,1)&
&   +24*dgm01(1,2)*dgm10(1,2)+13.5d0*dgm01(1,1)*dgm10(2,2)+3*gm(1,1)&
&   *d2gm(2,2)))
   cm(7,2)=gm(2,2)**2*(0*dgm01(1,2)*dgm10(1,1)+0*dgm01(1,1)*dgm10(1,2)&
&   +3*gm(1,2)*d2gm(1,1)-3*gm(1,1)*d2gm(1,2))+gm(1,2)*(-3*gm(1,1)&
&   *dgm01(2,2)*dgm10(2,2)+16.5d0*gm(1,2)*(dgm01(2,2)*dgm10(1,2)&
&   +dgm01(1,2)*dgm10(2,2))+2.5d0*gm(1,2)**2*d2gm(2,2))+gm(2,2)*(-7.5d0*gm(1,1)&
&   *(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+9*gm(1,2)**2*d2gm(1,2)&
&   +gm(1,2)*(4.5d0*dgm01(2,2)*dgm10(1,1)+36*dgm01(1,2)*dgm10(1,2)&
&   +4.5d0*dgm01(1,1)*dgm10(2,2)-1.5d0*gm(1,1)*d2gm(2,2)))
   cm(8,2)=-6*gm(1,2)*gm(3,3)*dgm01(2,2)*dgm10(1,1)-24*gm(1,2)&
&   *gm(3,3)*dgm01(1,2)*dgm10(1,2)+22.5d0*gm(1,3)**2*dgm01(2,2)*dgm10(1,2)&
&   -4.5d0*gm(1,1)*gm(3,3)*dgm01(2,2)*dgm10(1,2)+18*gm(1,2)*gm(1,3)&
&   *dgm01(2,3)*dgm10(1,2)-6*gm(1,2)**2*dgm01(3,3)*dgm10(1,2)+15*gm(1,2)&
&   *gm(1,3)*dgm01(2,2)*dgm10(1,3)-6*gm(1,2)*gm(3,3)*dgm01(1,1)*dgm10(2,2)&
&   +22.5d0*gm(1,3)**2*dgm01(1,2)*dgm10(2,2)-4.5d0*gm(1,1)*gm(3,3)&
&   *dgm01(1,2)*dgm10(2,2)+15*gm(1,2)*gm(1,3)*dgm01(1,3)*dgm10(2,2)&
&   -3*gm(1,1)*gm(1,3)*dgm01(2,3)*dgm10(2,2)-1.5d0*gm(1,1)*gm(1,2)&
&   *dgm01(3,3)*dgm10(2,2)+18*gm(1,2)*gm(1,3)*dgm01(1,2)*dgm10(2,3)&
&   -3*gm(1,1)*gm(1,3)*dgm01(2,2)*dgm10(2,3)-6*gm(1,2)**2*dgm01(1,2)&
&   *dgm10(3,3)-1.5d0*gm(1,1)*gm(1,2)*dgm01(2,2)*dgm10(3,3)-6*gm(1,2)&
&   **2*gm(3,3)*d2gm(1,2)+gm(2,3)**2*(-4.5d0*dgm01(1,2)*dgm10(1,1)&
&   -4.5d0*dgm01(1,1)*dgm10(1,2)+1.5d0*gm(1,2)*d2gm(1,1)-6*gm(1,1)&
&   *d2gm(1,2))+gm(2,2)*(-4.5d0*gm(1,2)*dgm01(3,3)*dgm10(1,1)-3*gm(1,1)&
&   *dgm01(3,3)*dgm10(1,2)+gm(1,3)*(12*dgm01(2,3)*dgm10(1,1)+30*dgm01(1,3)&
&   *dgm10(1,2)+30*dgm01(1,2)*dgm10(1,3)+12*dgm01(1,1)*dgm10(2,3))&
&   -4.5d0*gm(1,2)*dgm01(1,1)*dgm10(3,3)-3*gm(1,1)*dgm01(1,2)*dgm10(3,3)&
&   +15*gm(1,3)**2*d2gm(1,2)+gm(3,3)*(-7.5d0*dgm01(1,2)*dgm10(1,1)&
&   -7.5d0*dgm01(1,1)*dgm10(1,2)-4.5d0*gm(1,2)*d2gm(1,1)-3*gm(1,1)&
&   *d2gm(1,2)))+7.5d0*gm(1,2)*gm(1,3)**2*d2gm(2,2)-1.5d0*gm(1,1)&
&   *gm(1,2)*gm(3,3)*d2gm(2,2)+gm(2,3)*(3*gm(1,2)*dgm01(2,3)*dgm10(1,1)&
&   +18*gm(1,2)*dgm01(1,3)*dgm10(1,2)-12*gm(1,1)*dgm01(2,3)*dgm10(1,2)&
&   +18*gm(1,2)*dgm01(1,2)*dgm10(1,3)-3*gm(1,1)*dgm01(2,2)*dgm10(1,3)&
&   -3*gm(1,1)*dgm01(1,3)*dgm10(2,2)+3*gm(1,2)*dgm01(1,1)*dgm10(2,3)&
&   -12*gm(1,1)*dgm01(1,2)*dgm10(2,3)+12*gm(2,2)*(dgm01(1,3)*dgm10(1,1)&
&   +dgm01(1,1)*dgm10(1,3)+gm(1,3)*d2gm(1,1))+gm(1,3)*(9*dgm01(2,2)&
&   *dgm10(1,1)+36*dgm01(1,2)*dgm10(1,2)+9*dgm01(1,1)*dgm10(2,2)&
&   +18*gm(1,2)*d2gm(1,2)-3*gm(1,1)*d2gm(2,2)))
   cm(9,2)=gm(1,1)*(-3*gm(1,3)*dgm01(2,2)*dgm10(2,2)-12*gm(2,3)&
&   *(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2)))+6*gm(2,2)**2*(dgm01(1,3)&
&   *dgm10(1,1)+dgm01(1,1)*dgm10(1,3)+gm(1,3)*d2gm(1,1))+gm(1,2)&
&   **2*(3*dgm01(2,3)*dgm10(1,2)+7.5d0*dgm01(2,2)*dgm10(1,3)+7.5d0*dgm01(1,3)&
&   *dgm10(2,2)+3*dgm01(1,2)*dgm10(2,3)+3*gm(2,3)*d2gm(1,2)+7.5d0*gm(1,3)&
&   *d2gm(2,2))+gm(1,2)*(39*gm(1,3)*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)&
&   *dgm10(2,2))-3*gm(1,1)*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))&
&   +gm(2,3)*(0*dgm01(2,2)*dgm10(1,1)+12*dgm01(1,2)*dgm10(1,2)+0*dgm01(1,1)&
&   *dgm10(2,2)-3*gm(1,1)*d2gm(2,2)))+gm(2,2)*(3*gm(1,2)*dgm01(2,3)&
&   *dgm10(1,1)+24*gm(1,2)*dgm01(1,3)*dgm10(1,2)-9*gm(1,1)*dgm01(2,3)&
&   *dgm10(1,2)+24*gm(1,2)*dgm01(1,2)*dgm10(1,3)-1.5d0*gm(1,1)*dgm01(2,2)&
&   *dgm10(1,3)-1.5d0*gm(1,1)*dgm01(1,3)*dgm10(2,2)+3*gm(1,2)*dgm01(1,1)&
&   *dgm10(2,3)-9*gm(1,1)*dgm01(1,2)*dgm10(2,3)+gm(2,3)*(-6*dgm01(1,2)&
&   *dgm10(1,1)-6*dgm01(1,1)*dgm10(1,2)+3*gm(1,2)*d2gm(1,1)-9*gm(1,1)&
&   *d2gm(1,2))+gm(1,3)*(10.5d0*dgm01(2,2)*dgm10(1,1)+48*dgm01(1,2)&
&   *dgm10(1,2)+10.5d0*dgm01(1,1)*dgm10(2,2)+24*gm(1,2)*d2gm(1,2)&
&   -1.5d0*gm(1,1)*d2gm(2,2)))
   cm(10,2)=(3*(90*gm(2,3)**2*dgm01(1,3)+gm(3,3)*(-18*gm(1,3)*dgm01(2,2)&
&   -36*gm(1,2)*dgm01(2,3))+gm(2,3)*(-36*gm(3,3)*dgm01(1,2)+180*gm(1,3)&
&   *dgm01(2,3)-36*gm(1,2)*dgm01(3,3))-18*gm(2,2)*(gm(3,3)*dgm01(1,3)&
&   +gm(1,3)*dgm01(3,3)))*dgm10(1,1)+6*(90*gm(1,3)**2*dgm01(2,3)&
&   +gm(3,3)*(-36*gm(1,2)*dgm01(1,3)-18*gm(1,1)*dgm01(2,3))+gm(2,3)&
&   *(-18*gm(3,3)*dgm01(1,1)+180*gm(1,3)*dgm01(1,3)-18*gm(1,1)*dgm01(3,3))&
&   -36*gm(1,3)*(gm(3,3)*dgm01(1,2)+gm(1,2)*dgm01(3,3)))*dgm10(1,2)&
&   +(270*gm(1,3)**2*dgm01(1,3)-54*gm(1,1)*gm(3,3)*dgm01(1,3)-54*gm(1,3)&
&   *(gm(3,3)*dgm01(1,1)+gm(1,1)*dgm01(3,3)))*dgm10(2,2)+dgm01(2,2)&
&   *(270*gm(1,3)**2*dgm10(1,3)-54*gm(1,1)*gm(3,3)*dgm10(1,3)-54*gm(1,3)&
&   *(gm(3,3)*dgm10(1,1)+gm(1,1)*dgm10(3,3)))+6*dgm01(1,2)*(90*gm(1,3)&
&   **2*dgm10(2,3)+gm(3,3)*(-36*gm(1,2)*dgm10(1,3)-18*gm(1,1)*dgm10(2,3))&
&   +gm(2,3)*(-18*gm(3,3)*dgm10(1,1)+180*gm(1,3)*dgm10(1,3)-18*gm(1,1)&
&   *dgm10(3,3))-36*gm(1,3)*(gm(3,3)*dgm10(1,2)+gm(1,2)*dgm10(3,3)))&
&   +3*dgm01(1,1)*(90*gm(2,3)**2*dgm10(1,3)+gm(3,3)*(-18*gm(1,3)&
&   *dgm10(2,2)-36*gm(1,2)*dgm10(2,3))+gm(2,3)*(-36*gm(3,3)*dgm10(1,2)&
&   +180*gm(1,3)*dgm10(2,3)-36*gm(1,2)*dgm10(3,3))-18*gm(2,2)*(gm(3,3)&
&   *dgm10(1,3)+gm(1,3)*dgm10(3,3)))+3*(-36*gm(1,2)*gm(2,3)*gm(3,3)&
&   +gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*d2gm(1,1)+6*(90*gm(1,3)&
&   **2*gm(2,3)-18*(2*gm(1,2)*gm(1,3)+gm(1,1)*gm(2,3))*gm(3,3))*d2gm(1,2)&
&   +(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))*d2gm(2,2))/36.d0
   cm(1,3)=gm(1,3)**2*(9*dgm01(1,1)*dgm10(1,1)+4.5d0*gm(1,1)*d2gm(1,1))&
&   +gm(1,1)*gm(1,3)*(21*dgm01(1,3)*dgm10(1,1)+21*dgm01(1,1)*dgm10(1,3)&
&   +6*gm(1,1)*d2gm(1,3))+gm(1,1)*(gm(3,3)*(-6*dgm01(1,1)*dgm10(1,1)&
&   -1.5d0*gm(1,1)*d2gm(1,1))+gm(1,1)*(1.5d0*dgm01(3,3)*dgm10(1,1)&
&   +12*dgm01(1,3)*dgm10(1,3)+1.5d0*dgm01(1,1)*dgm10(3,3)+gm(1,1)&
&   *d2gm(3,3)))
   cm(2,3)=-6*gm(1,2)*gm(3,3)*dgm01(1,2)*dgm10(1,1)-3*gm(1,3)**2*dgm01(2,2)&
&   *dgm10(1,1)-1.5d0*gm(1,1)*gm(3,3)*dgm01(2,2)*dgm10(1,1)+9*gm(1,2)&
&   *gm(1,3)*dgm01(2,3)*dgm10(1,1)+1.5d0*gm(1,2)**2*dgm01(3,3)*dgm10(1,1)&
&   -6*gm(1,2)*gm(3,3)*dgm01(1,1)*dgm10(1,2)+6*gm(1,2)*gm(1,3)*dgm01(1,3)&
&   *dgm10(1,2)+9*gm(1,1)*gm(1,2)*dgm01(3,3)*dgm10(1,2)+6*gm(1,2)&
&   *gm(1,3)*dgm01(1,2)*dgm10(1,3)+6*gm(1,2)**2*dgm01(1,3)*dgm10(1,3)&
&   -9*gm(1,1)*gm(1,3)*dgm01(2,2)*dgm10(1,3)+24*gm(1,1)*gm(1,2)*dgm01(2,3)&
&   *dgm10(1,3)-3*gm(1,3)**2*dgm01(1,1)*dgm10(2,2)-1.5d0*gm(1,1)&
&   *gm(3,3)*dgm01(1,1)*dgm10(2,2)-9*gm(1,1)*gm(1,3)*dgm01(1,3)*dgm10(2,2)&
&   -1.5d0*gm(1,1)**2*dgm01(3,3)*dgm10(2,2)+9*gm(1,2)*gm(1,3)*dgm01(1,1)&
&   *dgm10(2,3)+24*gm(1,1)*gm(1,2)*dgm01(1,3)*dgm10(2,3)+1.5d0*gm(1,2)&
&   **2*dgm01(1,1)*dgm10(3,3)+9*gm(1,1)*gm(1,2)*dgm01(1,2)*dgm10(3,3)&
&   -1.5d0*gm(1,1)**2*dgm01(2,2)*dgm10(3,3)-3*gm(1,2)**2*gm(3,3)&
&   *d2gm(1,1)+gm(2,3)**2*(15*dgm01(1,1)*dgm10(1,1)+7.5d0*gm(1,1)&
&   *d2gm(1,1))+3*gm(1,2)**2*gm(1,3)*d2gm(1,3)+gm(2,3)*(gm(1,1)*(15*dgm01(2,3)&
&   *dgm10(1,1)+24*dgm01(1,3)*dgm10(1,2)+24*dgm01(1,2)*dgm10(1,3)&
&   +15*dgm01(1,1)*dgm10(2,3))+9*gm(1,3)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)&
&   *dgm10(1,2)+gm(1,2)*d2gm(1,1))+gm(1,2)*(33*dgm01(1,3)*dgm10(1,1)&
&   +33*dgm01(1,1)*dgm10(1,3)+24*gm(1,1)*d2gm(1,3)))+4.5d0*gm(1,1)&
&   *gm(1,2)**2*d2gm(3,3)+gm(2,2)*(-3*gm(1,3)**2*d2gm(1,1)+gm(3,3)&
&   *(-3*dgm01(1,1)*dgm10(1,1)-1.5d0*gm(1,1)*d2gm(1,1))+gm(1,3)*(-15*dgm01(1,3)&
&   *dgm10(1,1)-15*dgm01(1,1)*dgm10(1,3)-9*gm(1,1)*d2gm(1,3))+gm(1,1)&
&   *(-4.5d0*dgm01(3,3)*dgm10(1,1)-18*dgm01(1,3)*dgm10(1,3)-4.5d0*dgm01(1,1)&
&   *dgm10(3,3)-1.5d0*gm(1,1)*d2gm(3,3)))
   cm(3,3)=-3*gm(1,1)**2*dgm01(3,3)*dgm10(3,3)+24*gm(1,1)*gm(1,3)&
&   *(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+gm(3,3)**2*(12*dgm01(1,1)&
&   *dgm10(1,1)+6*gm(1,1)*d2gm(1,1))+3*gm(1,3)**3*d2gm(1,3)+gm(1,3)&
&   **2*(7.5d0*dgm01(3,3)*dgm10(1,1)+18*dgm01(1,3)*dgm10(1,3)+7.5d0*dgm01(1,1)&
&   *dgm10(3,3)+4.5d0*gm(1,1)*d2gm(3,3))+gm(3,3)*(3*gm(1,3)**2*d2gm(1,1)&
&   +gm(1,3)*(21*dgm01(1,3)*dgm10(1,1)+21*dgm01(1,1)*dgm10(1,3)+15*gm(1,1)&
&   *d2gm(1,3))+gm(1,1)*(9*dgm01(3,3)*dgm10(1,1)+30*dgm01(1,3)*dgm10(1,3)&
&   +9*dgm01(1,1)*dgm10(3,3)-1.5d0*gm(1,1)*d2gm(3,3)))
   cm(4,3)=27*gm(1,2)*gm(3,3)*dgm01(1,3)*dgm10(1,1)+12*gm(1,1)&
&   *gm(3,3)*dgm01(2,3)*dgm10(1,1)+24*gm(1,1)*gm(3,3)*dgm01(1,3)&
&   *dgm10(1,2)+27*gm(1,2)*gm(3,3)*dgm01(1,1)*dgm10(1,3)+24*gm(1,1)&
&   *gm(3,3)*dgm01(1,2)*dgm10(1,3)+33*gm(1,1)*gm(1,2)*dgm01(3,3)&
&   *dgm10(1,3)+12*gm(1,1)*gm(3,3)*dgm01(1,1)*dgm10(2,3)-3*gm(1,1)&
&   **2*dgm01(3,3)*dgm10(2,3)+33*gm(1,1)*gm(1,2)*dgm01(1,3)*dgm10(3,3)&
&   -3*gm(1,1)**2*dgm01(2,3)*dgm10(3,3)+24*gm(1,1)*gm(1,2)*gm(3,3)&
&   *d2gm(1,3)+gm(1,3)**2*(3*dgm01(2,3)*dgm10(1,1)+6*dgm01(1,3)*dgm10(1,2)&
&   +6*dgm01(1,2)*dgm10(1,3)+3*dgm01(1,1)*dgm10(2,3)+6*gm(1,2)*d2gm(1,3))&
&   +gm(2,3)*(3*gm(1,3)**2*d2gm(1,1)+gm(3,3)*(24*dgm01(1,1)*dgm10(1,1)&
&   +12*gm(1,1)*d2gm(1,1))+gm(1,3)*(12*dgm01(1,3)*dgm10(1,1)+12*dgm01(1,1)&
&   *dgm10(1,3)+6*gm(1,1)*d2gm(1,3))+gm(1,1)*(6*dgm01(3,3)*dgm10(1,1)&
&   +12*dgm01(1,3)*dgm10(1,3)+6*dgm01(1,1)*dgm10(3,3)-3*gm(1,1)*d2gm(3,3)))&
&   +gm(1,3)*(gm(1,1)*(9*dgm01(3,3)*dgm10(1,2)+6*dgm01(2,3)*dgm10(1,3)&
&   +6*dgm01(1,3)*dgm10(2,3)+9*dgm01(1,2)*dgm10(3,3))+3*gm(3,3)*(dgm01(1,2)&
&   *dgm10(1,1)+dgm01(1,1)*dgm10(1,2)+gm(1,2)*d2gm(1,1))+gm(1,2)&
&   *(12*dgm01(3,3)*dgm10(1,1)+24*dgm01(1,3)*dgm10(1,3)+12*dgm01(1,1)&
&   *dgm10(3,3)+9*gm(1,1)*d2gm(3,3)))
   cm(5,3)=1.5d0*gm(1,3)**3*d2gm(1,1)+gm(1,3)**2*(10.5d0*dgm01(1,3)&
&   *dgm10(1,1)+10.5d0*dgm01(1,1)*dgm10(1,3)+6*gm(1,1)*d2gm(1,3))&
&   +gm(1,1)*(15*gm(1,1)*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))&
&   +gm(3,3)*(31.5d0*dgm01(1,3)*dgm10(1,1)+31.5d0*dgm01(1,1)*dgm10(1,3)&
&   +12*gm(1,1)*d2gm(1,3)))+gm(1,3)*(gm(3,3)*(15*dgm01(1,1)*dgm10(1,1)&
&   +7.5d0*gm(1,1)*d2gm(1,1))+gm(1,1)*(13.5d0*dgm01(3,3)*dgm10(1,1)&
&   +24*dgm01(1,3)*dgm10(1,3)+13.5d0*dgm01(1,1)*dgm10(3,3)+3*gm(1,1)&
&   *d2gm(3,3)))
   cm(6,3)=1.5d0*gm(1,3)**2*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2)&
&   +gm(1,2)*d2gm(1,1))+gm(1,3)*(gm(1,1)*(12*dgm01(2,3)*dgm10(1,1)&
&   +6*dgm01(1,3)*dgm10(1,2)+6*dgm01(1,2)*dgm10(1,3)+12*dgm01(1,1)&
&   *dgm10(2,3))+gm(2,3)*(24*dgm01(1,1)*dgm10(1,1)+12*gm(1,1)*d2gm(1,1))&
&   +gm(1,2)*(9*dgm01(1,3)*dgm10(1,1)+9*dgm01(1,1)*dgm10(1,3)+6*gm(1,1)&
&   *d2gm(1,3)))+gm(1,1)*(-4.5d0*gm(3,3)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)&
&   *dgm10(1,2))+gm(1,1)*(3*dgm01(3,3)*dgm10(1,2)+12*dgm01(2,3)*dgm10(1,3)&
&   +12*dgm01(1,3)*dgm10(2,3)+3*dgm01(1,2)*dgm10(3,3))+gm(2,3)*(36*dgm01(1,3)&
&   *dgm10(1,1)+36*dgm01(1,1)*dgm10(1,3)+12*gm(1,1)*d2gm(1,3)))+gm(1,2)&
&   *(gm(3,3)*(-9*dgm01(1,1)*dgm10(1,1)-4.5d0*gm(1,1)*d2gm(1,1))&
&   +gm(1,1)*(1.5d0*dgm01(3,3)*dgm10(1,1)+12*dgm01(1,3)*dgm10(1,3)&
&   +1.5d0*dgm01(1,1)*dgm10(3,3)+3*gm(1,1)*d2gm(3,3)))
   cm(7,3)=(3*(90*gm(2,3)**2*dgm01(1,2)-18*gm(1,2)*gm(3,3)*dgm01(2,2)&
&   +gm(2,3)*(-36*gm(2,2)*dgm01(1,3)-36*gm(1,3)*dgm01(2,2)+180*gm(1,2)&
&   *dgm01(2,3))+gm(2,2)*(-18*gm(3,3)*dgm01(1,2)-36*gm(1,3)*dgm01(2,3)&
&   -18*gm(1,2)*dgm01(3,3)))*dgm10(1,1)+6*(-18*gm(1,1)*gm(2,3)*dgm01(2,2)&
&   +gm(1,2)*(180*gm(2,3)*dgm01(1,2)-36*gm(1,3)*dgm01(2,2))+90*gm(1,2)&
&   **2*dgm01(2,3)+gm(2,2)*(-18*gm(2,3)*dgm01(1,1)-36*gm(1,3)*dgm01(1,2)&
&   -36*gm(1,2)*dgm01(1,3)-18*gm(1,1)*dgm01(2,3)))*dgm10(1,3)+dgm01(3,3)&
&   *(270*gm(1,2)**2*dgm10(1,2)-54*gm(1,1)*gm(2,2)*dgm10(1,2)-54*gm(1,2)&
&   *(gm(2,2)*dgm10(1,1)+gm(1,1)*dgm10(2,2)))+6*dgm01(1,3)*(-18*gm(1,1)&
&   *gm(2,3)*dgm10(2,2)+gm(1,2)*(180*gm(2,3)*dgm10(1,2)-36*gm(1,3)&
&   *dgm10(2,2))+90*gm(1,2)**2*dgm10(2,3)+gm(2,2)*(-18*gm(2,3)*dgm10(1,1)&
&   -36*gm(1,3)*dgm10(1,2)-36*gm(1,2)*dgm10(1,3)-18*gm(1,1)*dgm10(2,3)))&
&   +(270*gm(1,2)**2*dgm01(1,2)-54*gm(1,1)*gm(2,2)*dgm01(1,2)-54*gm(1,2)&
&   *(gm(2,2)*dgm01(1,1)+gm(1,1)*dgm01(2,2)))*dgm10(3,3)+3*dgm01(1,1)&
&   *(90*gm(2,3)**2*dgm10(1,2)-18*gm(1,2)*gm(3,3)*dgm10(2,2)+gm(2,3)&
&   *(-36*gm(2,2)*dgm10(1,3)-36*gm(1,3)*dgm10(2,2)+180*gm(1,2)*dgm10(2,3))&
&   +gm(2,2)*(-18*gm(3,3)*dgm10(1,2)-36*gm(1,3)*dgm10(2,3)-18*gm(1,2)&
&   *dgm10(3,3)))+3*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*d2gm(1,1)+6*(90*gm(1,2)**2*gm(2,3)-18*gm(2,2)&
&   *(2*gm(1,2)*gm(1,3)+gm(1,1)*gm(2,3)))*d2gm(1,3)+(90*gm(1,2)**3-54*gm(1,1)&
&   *gm(1,2)*gm(2,2))*d2gm(3,3))/36.d0
   cm(8,3)=gm(1,1)*(-3*gm(1,2)*dgm01(3,3)*dgm10(3,3)-12*gm(2,3)&
&   *(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3)))+6*gm(3,3)**2*(dgm01(1,2)&
&   *dgm10(1,1)+dgm01(1,1)*dgm10(1,2)+gm(1,2)*d2gm(1,1))+gm(1,3)&
&   **2*(7.5d0*dgm01(3,3)*dgm10(1,2)+3*dgm01(2,3)*dgm10(1,3)+3*dgm01(1,3)&
&   *dgm10(2,3)+7.5d0*dgm01(1,2)*dgm10(3,3)+3*gm(2,3)*d2gm(1,3)+7.5d0*gm(1,2)&
&   *d2gm(3,3))+gm(3,3)*(10.5d0*gm(1,2)*dgm01(3,3)*dgm10(1,1)-1.5d0*gm(1,1)&
&   *dgm01(3,3)*dgm10(1,2)+48*gm(1,2)*dgm01(1,3)*dgm10(1,3)-9*gm(1,1)&
&   *dgm01(2,3)*dgm10(1,3)-9*gm(1,1)*dgm01(1,3)*dgm10(2,3)+10.5d0*gm(1,2)&
&   *dgm01(1,1)*dgm10(3,3)-1.5d0*gm(1,1)*dgm01(1,2)*dgm10(3,3)+gm(2,3)&
&   *(-6*dgm01(1,3)*dgm10(1,1)-6*dgm01(1,1)*dgm10(1,3)+3*gm(1,3)&
&   *d2gm(1,1)-9*gm(1,1)*d2gm(1,3))+gm(1,3)*(3*dgm01(2,3)*dgm10(1,1)&
&   +24*dgm01(1,3)*dgm10(1,2)+24*dgm01(1,2)*dgm10(1,3)+3*dgm01(1,1)&
&   *dgm10(2,3)+24*gm(1,2)*d2gm(1,3))-1.5d0*gm(1,1)*gm(1,2)*d2gm(3,3))&
&   +gm(1,3)*(39*gm(1,2)*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))&
&   -3*gm(1,1)*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3))+gm(2,3)&
&   *(0*dgm01(3,3)*dgm10(1,1)+12*dgm01(1,3)*dgm10(1,3)+0*dgm01(1,1)&
&   *dgm10(3,3)-3*gm(1,1)*d2gm(3,3)))
   cm(9,3)=-4.5d0*gm(1,3)*gm(3,3)*dgm01(2,2)*dgm10(1,1)+12*gm(1,2)&
&   *gm(3,3)*dgm01(2,3)*dgm10(1,1)+30*gm(1,2)*gm(3,3)*dgm01(1,3)&
&   *dgm10(1,2)+15*gm(1,2)*gm(1,3)*dgm01(3,3)*dgm10(1,2)+30*gm(1,2)&
&   *gm(3,3)*dgm01(1,2)*dgm10(1,3)-6*gm(1,3)**2*dgm01(2,2)*dgm10(1,3)&
&   -3*gm(1,1)*gm(3,3)*dgm01(2,2)*dgm10(1,3)+18*gm(1,2)*gm(1,3)*dgm01(2,3)&
&   *dgm10(1,3)+22.5d0*gm(1,2)**2*dgm01(3,3)*dgm10(1,3)-4.5d0*gm(1,3)&
&   *gm(3,3)*dgm01(1,1)*dgm10(2,2)-6*gm(1,3)**2*dgm01(1,3)*dgm10(2,2)&
&   -3*gm(1,1)*gm(3,3)*dgm01(1,3)*dgm10(2,2)-1.5d0*gm(1,1)*gm(1,3)&
&   *dgm01(3,3)*dgm10(2,2)+12*gm(1,2)*gm(3,3)*dgm01(1,1)*dgm10(2,3)&
&   +18*gm(1,2)*gm(1,3)*dgm01(1,3)*dgm10(2,3)-3*gm(1,1)*gm(1,2)*dgm01(3,3)&
&   *dgm10(2,3)+15*gm(1,2)*gm(1,3)*dgm01(1,2)*dgm10(3,3)+22.5d0*gm(1,2)&
&   **2*dgm01(1,3)*dgm10(3,3)-1.5d0*gm(1,1)*gm(1,3)*dgm01(2,2)*dgm10(3,3)&
&   -3*gm(1,1)*gm(1,2)*dgm01(2,3)*dgm10(3,3)+15*gm(1,2)**2*gm(3,3)&
&   *d2gm(1,3)+gm(2,3)**2*(-4.5d0*dgm01(1,3)*dgm10(1,1)-4.5d0*dgm01(1,1)&
&   *dgm10(1,3)+1.5d0*gm(1,3)*d2gm(1,1)-6*gm(1,1)*d2gm(1,3))+7.5d0*gm(1,2)&
&   **2*gm(1,3)*d2gm(3,3)+gm(2,3)*(9*gm(1,2)*dgm01(3,3)*dgm10(1,1)&
&   -3*gm(1,1)*dgm01(3,3)*dgm10(1,2)+36*gm(1,2)*dgm01(1,3)*dgm10(1,3)&
&   -12*gm(1,1)*dgm01(2,3)*dgm10(1,3)-12*gm(1,1)*dgm01(1,3)*dgm10(2,3)&
&   +9*gm(1,2)*dgm01(1,1)*dgm10(3,3)-3*gm(1,1)*dgm01(1,2)*dgm10(3,3)&
&   +12*gm(3,3)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2)+gm(1,2)&
&   *d2gm(1,1))+gm(1,3)*(3*dgm01(2,3)*dgm10(1,1)+18*dgm01(1,3)*dgm10(1,2)&
&   +18*dgm01(1,2)*dgm10(1,3)+3*dgm01(1,1)*dgm10(2,3)+18*gm(1,2)&
&   *d2gm(1,3))-3*gm(1,1)*gm(1,2)*d2gm(3,3))+gm(2,2)*(-4.5d0*gm(1,1)&
&   *(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))-6*gm(1,3)**2*d2gm(1,3)&
&   +gm(3,3)*(-7.5d0*dgm01(1,3)*dgm10(1,1)-7.5d0*dgm01(1,1)*dgm10(1,3)&
&   -4.5d0*gm(1,3)*d2gm(1,1)-3*gm(1,1)*d2gm(1,3))+gm(1,3)*(-6*dgm01(3,3)&
&   *dgm10(1,1)-24*dgm01(1,3)*dgm10(1,3)-6*dgm01(1,1)*dgm10(3,3)&
&   -1.5d0*gm(1,1)*d2gm(3,3)))
   cm(10,3)=gm(3,3)**2*(0*dgm01(1,3)*dgm10(1,1)+0*dgm01(1,1)*dgm10(1,3)&
&   +3*gm(1,3)*d2gm(1,1)-3*gm(1,1)*d2gm(1,3))+gm(1,3)*(-3*gm(1,1)&
&   *dgm01(3,3)*dgm10(3,3)+16.5d0*gm(1,3)*(dgm01(3,3)*dgm10(1,3)&
&   +dgm01(1,3)*dgm10(3,3))+2.5d0*gm(1,3)**2*d2gm(3,3))+gm(3,3)*(-7.5d0*gm(1,1)&
&   *(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+9*gm(1,3)**2*d2gm(1,3)&
&   +gm(1,3)*(4.5d0*dgm01(3,3)*dgm10(1,1)+36*dgm01(1,3)*dgm10(1,3)&
&   +4.5d0*dgm01(1,1)*dgm10(3,3)-1.5d0*gm(1,1)*d2gm(3,3)))
   cm(1,4)=gm(1,2)*(gm(1,3)*(18*dgm01(1,1)*dgm10(1,1)+9*gm(1,1)&
&   *d2gm(1,1))+gm(1,1)*(21*dgm01(1,3)*dgm10(1,1)+21*dgm01(1,1)*dgm10(1,3)&
&   +6*gm(1,1)*d2gm(1,3)))+gm(1,1)*(gm(2,3)*(-12*dgm01(1,1)*dgm10(1,1)&
&   -3*gm(1,1)*d2gm(1,1))+gm(1,3)*(21*dgm01(1,2)*dgm10(1,1)+21*dgm01(1,1)&
&   *dgm10(1,2)+6*gm(1,1)*d2gm(1,2))+gm(1,1)*(3*dgm01(2,3)*dgm10(1,1)&
&   +12*dgm01(1,3)*dgm10(1,2)+12*dgm01(1,2)*dgm10(1,3)+3*dgm01(1,1)&
&   *dgm10(2,3)+2*gm(1,1)*d2gm(2,3)))
   cm(2,4)=gm(1,1)*(gm(2,3)*(12*dgm01(2,2)*dgm10(1,1)+48*dgm01(1,2)&
&   *dgm10(1,2)+12*dgm01(1,1)*dgm10(2,2))-9*gm(1,3)*(dgm01(2,2)*dgm10(1,2)&
&   +dgm01(1,2)*dgm10(2,2))-3*gm(1,1)*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)&
&   *dgm10(2,3)))+gm(1,2)*(gm(1,3)*(3*dgm01(2,2)*dgm10(1,1)+12*dgm01(1,2)&
&   *dgm10(1,2)+3*dgm01(1,1)*dgm10(2,2))+gm(1,1)*(42*dgm01(2,3)*dgm10(1,2)&
&   +15*dgm01(2,2)*dgm10(1,3)+15*dgm01(1,3)*dgm10(2,2)+42*dgm01(1,2)&
&   *dgm10(2,3))+gm(2,3)*(30*dgm01(1,2)*dgm10(1,1)+30*dgm01(1,1)&
&   *dgm10(1,2)+24*gm(1,1)*d2gm(1,2)))+3*gm(1,2)**3*d2gm(1,3)+gm(1,2)&
&   **2*(12*dgm01(2,3)*dgm10(1,1)+12*dgm01(1,3)*dgm10(1,2)+12*dgm01(1,2)&
&   *dgm10(1,3)+12*dgm01(1,1)*dgm10(2,3)+3*gm(2,3)*d2gm(1,1)+3*gm(1,3)&
&   *d2gm(1,2)+9*gm(1,1)*d2gm(2,3))+gm(2,2)*(18*gm(1,2)*dgm01(1,3)&
&   *dgm10(1,1)+6*gm(1,1)*dgm01(2,3)*dgm10(1,1)+6*gm(1,1)*dgm01(1,3)&
&   *dgm10(1,2)+18*gm(1,2)*dgm01(1,1)*dgm10(1,3)+6*gm(1,1)*dgm01(1,2)&
&   *dgm10(1,3)+6*gm(1,1)*dgm01(1,1)*dgm10(2,3)+gm(2,3)*(24*dgm01(1,1)&
&   *dgm10(1,1)+12*gm(1,1)*d2gm(1,1))+gm(1,3)*(-6*dgm01(1,2)*dgm10(1,1)&
&   -6*dgm01(1,1)*dgm10(1,2)+3*gm(1,2)*d2gm(1,1)-9*gm(1,1)*d2gm(1,2))&
&   +15*gm(1,1)*gm(1,2)*d2gm(1,3)-3*gm(1,1)**2*d2gm(2,3))
   cm(3,4)=-6*gm(1,2)*gm(3,3)*dgm01(1,3)*dgm10(1,1)+6*gm(1,1)*gm(3,3)&
&   *dgm01(2,3)*dgm10(1,1)+6*gm(1,1)*gm(3,3)*dgm01(1,3)*dgm10(1,2)&
&   -6*gm(1,2)*gm(3,3)*dgm01(1,1)*dgm10(1,3)+6*gm(1,1)*gm(3,3)*dgm01(1,2)&
&   *dgm10(1,3)-9*gm(1,1)*gm(1,2)*dgm01(3,3)*dgm10(1,3)+6*gm(1,1)&
&   *gm(3,3)*dgm01(1,1)*dgm10(2,3)-3*gm(1,1)**2*dgm01(3,3)*dgm10(2,3)&
&   -9*gm(1,1)*gm(1,2)*dgm01(1,3)*dgm10(3,3)-3*gm(1,1)**2*dgm01(2,3)&
&   *dgm10(3,3)+3*gm(1,3)**3*d2gm(1,2)+gm(1,3)*(gm(1,2)*(3*dgm01(3,3)&
&   *dgm10(1,1)+12*dgm01(1,3)*dgm10(1,3)+3*dgm01(1,1)*dgm10(3,3))&
&   +gm(1,1)*(15*dgm01(3,3)*dgm10(1,2)+42*dgm01(2,3)*dgm10(1,3)+42*dgm01(1,3)&
&   *dgm10(2,3)+15*dgm01(1,2)*dgm10(3,3))+gm(3,3)*(18*dgm01(1,2)&
&   *dgm10(1,1)+18*dgm01(1,1)*dgm10(1,2)+3*gm(1,2)*d2gm(1,1)+15*gm(1,1)&
&   *d2gm(1,2)))-9*gm(1,1)*gm(1,2)*gm(3,3)*d2gm(1,3)+gm(2,3)*(gm(1,1)&
&   *(12*dgm01(3,3)*dgm10(1,1)+48*dgm01(1,3)*dgm10(1,3)+12*dgm01(1,1)&
&   *dgm10(3,3))+3*gm(1,3)**2*d2gm(1,1)+gm(3,3)*(24*dgm01(1,1)*dgm10(1,1)&
&   +12*gm(1,1)*d2gm(1,1))+gm(1,3)*(30*dgm01(1,3)*dgm10(1,1)+30*dgm01(1,1)&
&   *dgm10(1,3)+24*gm(1,1)*d2gm(1,3)))-3*gm(1,1)**2*gm(3,3)*d2gm(2,3)&
&   +gm(1,3)**2*(12*dgm01(2,3)*dgm10(1,1)+12*dgm01(1,3)*dgm10(1,2)&
&   +12*dgm01(1,2)*dgm10(1,3)+12*dgm01(1,1)*dgm10(2,3)+3*gm(1,2)&
&   *d2gm(1,3)+9*gm(1,1)*d2gm(2,3))
   cm(4,4)=42*gm(1,2)*gm(3,3)*dgm01(1,2)*dgm10(1,1)+9*gm(1,3)**2*dgm01(2,2)&
&   *dgm10(1,1)+15*gm(1,1)*gm(3,3)*dgm01(2,2)*dgm10(1,1)+12*gm(1,2)&
&   *gm(1,3)*dgm01(2,3)*dgm10(1,1)+9*gm(1,2)**2*dgm01(3,3)*dgm10(1,1)&
&   +42*gm(1,2)*gm(3,3)*dgm01(1,1)*dgm10(1,2)+12*gm(1,3)**2*dgm01(1,2)&
&   *dgm10(1,2)+48*gm(1,1)*gm(3,3)*dgm01(1,2)*dgm10(1,2)+24*gm(1,2)&
&   *gm(1,3)*dgm01(1,3)*dgm10(1,2)+24*gm(1,1)*gm(1,3)*dgm01(2,3)&
&   *dgm10(1,2)+24*gm(1,1)*gm(1,2)*dgm01(3,3)*dgm10(1,2)+24*gm(1,2)&
&   *gm(1,3)*dgm01(1,2)*dgm10(1,3)+12*gm(1,2)**2*dgm01(1,3)*dgm10(1,3)&
&   +24*gm(1,1)*gm(1,3)*dgm01(2,2)*dgm10(1,3)+24*gm(1,1)*gm(1,2)&
&   *dgm01(2,3)*dgm10(1,3)+9*gm(1,3)**2*dgm01(1,1)*dgm10(2,2)+15*gm(1,1)&
&   *gm(3,3)*dgm01(1,1)*dgm10(2,2)+24*gm(1,1)*gm(1,3)*dgm01(1,3)&
&   *dgm10(2,2)+12*gm(1,2)*gm(1,3)*dgm01(1,1)*dgm10(2,3)+24*gm(1,1)&
&   *gm(1,3)*dgm01(1,2)*dgm10(2,3)+24*gm(1,1)*gm(1,2)*dgm01(1,3)&
&   *dgm10(2,3)-12*gm(1,1)**2*dgm01(2,3)*dgm10(2,3)+9*gm(1,2)**2*dgm01(1,1)&
&   *dgm10(3,3)+24*gm(1,1)*gm(1,2)*dgm01(1,2)*dgm10(3,3)+9*gm(1,2)&
&   **2*gm(3,3)*d2gm(1,1)+gm(2,3)**2*(18*dgm01(1,1)*dgm10(1,1)+9*gm(1,1)&
&   *d2gm(1,1))+6*gm(1,2)*gm(1,3)**2*d2gm(1,2)+24*gm(1,1)*gm(1,2)&
&   *gm(3,3)*d2gm(1,2)+6*gm(1,2)**2*gm(1,3)*d2gm(1,3)+gm(2,2)*(gm(1,1)&
&   *(15*dgm01(3,3)*dgm10(1,1)+48*dgm01(1,3)*dgm10(1,3)+15*dgm01(1,1)&
&   *dgm10(3,3))+9*gm(1,3)**2*d2gm(1,1)+gm(3,3)*(30*dgm01(1,1)*dgm10(1,1)&
&   +15*gm(1,1)*d2gm(1,1))+gm(1,3)*(42*dgm01(1,3)*dgm10(1,1)+42*dgm01(1,1)&
&   *dgm10(1,3)+24*gm(1,1)*d2gm(1,3)))+18*gm(1,1)*gm(1,2)*gm(1,3)&
&   *d2gm(2,3)+gm(2,3)*(gm(1,3)*(0*dgm01(1,2)*dgm10(1,1)+0*dgm01(1,1)&
&   *dgm10(1,2)-6*gm(1,2)*d2gm(1,1)+6*gm(1,1)*d2gm(1,2))+gm(1,2)&
&   *(0*dgm01(1,3)*dgm10(1,1)+0*dgm01(1,1)*dgm10(1,3)+6*gm(1,1)*d2gm(1,3))&
&   +gm(1,1)*(6*dgm01(2,3)*dgm10(1,1)+12*dgm01(1,3)*dgm10(1,2)+12*dgm01(1,2)&
&   *dgm10(1,3)+6*dgm01(1,1)*dgm10(2,3)-6*gm(1,1)*d2gm(2,3)))
   cm(5,4)=gm(1,2)*(12*gm(1,1)*(dgm01(3,3)*dgm10(1,1)+dgm01(1,3)&
&   *dgm10(1,3)+dgm01(1,1)*dgm10(3,3))+gm(3,3)*(24*dgm01(1,1)*dgm10(1,1)&
&   +12*gm(1,1)*d2gm(1,1)))+gm(1,3)**2*(9*dgm01(1,2)*dgm10(1,1)+9*dgm01(1,1)&
&   *dgm10(1,2)+3*gm(1,2)*d2gm(1,1)+6*gm(1,1)*d2gm(1,2))+gm(1,1)&
&   *(gm(1,1)*(12*dgm01(3,3)*dgm10(1,2)+18*dgm01(2,3)*dgm10(1,3)&
&   +18*dgm01(1,3)*dgm10(2,3)+12*dgm01(1,2)*dgm10(3,3))+gm(3,3)*(36*dgm01(1,2)&
&   *dgm10(1,1)+36*dgm01(1,1)*dgm10(1,2)+12*gm(1,1)*d2gm(1,2))+gm(2,3)&
&   *(27*dgm01(1,3)*dgm10(1,1)+27*dgm01(1,1)*dgm10(1,3)+12*gm(1,1)&
&   *d2gm(1,3)))+gm(1,3)*(gm(2,3)*(6*dgm01(1,1)*dgm10(1,1)+3*gm(1,1)&
&   *d2gm(1,1))+gm(1,2)*(12*dgm01(1,3)*dgm10(1,1)+12*dgm01(1,1)*dgm10(1,3)&
&   +6*gm(1,1)*d2gm(1,3))+gm(1,1)*(15*dgm01(2,3)*dgm10(1,1)+18*dgm01(1,3)&
&   *dgm10(1,2)+18*dgm01(1,2)*dgm10(1,3)+15*dgm01(1,1)*dgm10(2,3)&
&   +6*gm(1,1)*d2gm(2,3)))
   cm(6,4)=gm(1,3)*(12*gm(1,1)*(dgm01(2,2)*dgm10(1,1)+dgm01(1,2)&
&   *dgm10(1,2)+dgm01(1,1)*dgm10(2,2))+3*gm(1,2)**2*d2gm(1,1)+gm(2,2)&
&   *(24*dgm01(1,1)*dgm10(1,1)+12*gm(1,1)*d2gm(1,1))+gm(1,2)*(12*dgm01(1,2)&
&   *dgm10(1,1)+12*dgm01(1,1)*dgm10(1,2)+6*gm(1,1)*d2gm(1,2)))+gm(1,2)&
&   **2*(9*dgm01(1,3)*dgm10(1,1)+9*dgm01(1,1)*dgm10(1,3)+6*gm(1,1)&
&   *d2gm(1,3))+gm(1,1)*(gm(1,1)*(18*dgm01(2,3)*dgm10(1,2)+12*dgm01(2,2)&
&   *dgm10(1,3)+12*dgm01(1,3)*dgm10(2,2)+18*dgm01(1,2)*dgm10(2,3))&
&   +gm(2,3)*(27*dgm01(1,2)*dgm10(1,1)+27*dgm01(1,1)*dgm10(1,2)+12*gm(1,1)&
&   *d2gm(1,2))+gm(2,2)*(36*dgm01(1,3)*dgm10(1,1)+36*dgm01(1,1)*dgm10(1,3)&
&   +12*gm(1,1)*d2gm(1,3)))+gm(1,2)*(gm(2,3)*(6*dgm01(1,1)*dgm10(1,1)&
&   +3*gm(1,1)*d2gm(1,1))+gm(1,1)*(15*dgm01(2,3)*dgm10(1,1)+18*dgm01(1,3)&
&   *dgm10(1,2)+18*dgm01(1,2)*dgm10(1,3)+15*dgm01(1,1)*dgm10(2,3)&
&   +6*gm(1,1)*d2gm(2,3)))
   cm(7,4)=-3*gm(1,1)*gm(2,3)*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)&
&   *dgm10(2,2))+gm(1,2)*(gm(2,3)*(9*dgm01(2,2)*dgm10(1,1)+60*dgm01(1,2)&
&   *dgm10(1,2)+9*dgm01(1,1)*dgm10(2,2))-6*gm(1,3)*(dgm01(2,2)*dgm10(1,2)&
&   +dgm01(1,2)*dgm10(2,2))-3*gm(1,1)*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)&
&   *dgm10(2,3)))+gm(1,2)**2*(30*dgm01(2,3)*dgm10(1,2)+9*dgm01(2,2)&
&   *dgm10(1,3)+9*dgm01(1,3)*dgm10(2,2)+30*dgm01(1,2)*dgm10(2,3)&
&   +15*gm(2,3)*d2gm(1,2))+gm(2,2)**2*(-6*dgm01(1,3)*dgm10(1,1)-6*dgm01(1,1)&
&   *dgm10(1,3)-3*(gm(1,3)*d2gm(1,1)+gm(1,1)*d2gm(1,3)))+5*gm(1,2)&
&   **3*d2gm(2,3)+gm(2,2)*(6*gm(1,2)*dgm01(2,3)*dgm10(1,1)+12*gm(1,2)&
&   *dgm01(1,3)*dgm10(1,2)-6*gm(1,1)*dgm01(2,3)*dgm10(1,2)+12*gm(1,2)&
&   *dgm01(1,2)*dgm10(1,3)-6*gm(1,1)*dgm01(2,2)*dgm10(1,3)-6*gm(1,1)&
&   *dgm01(1,3)*dgm10(2,2)+6*gm(1,2)*dgm01(1,1)*dgm10(2,3)-6*gm(1,1)&
&   *dgm01(1,2)*dgm10(2,3)+gm(2,3)*(6*dgm01(1,2)*dgm10(1,1)+6*dgm01(1,1)&
&   *dgm10(1,2)+9*gm(1,2)*d2gm(1,1)-3*gm(1,1)*d2gm(1,2))+gm(1,3)&
&   *(-6*dgm01(2,2)*dgm10(1,1)-12*dgm01(1,2)*dgm10(1,2)-6*dgm01(1,1)&
&   *dgm10(2,2)-6*gm(1,2)*d2gm(1,2))+9*gm(1,2)**2*d2gm(1,3)-3*gm(1,1)&
&   *gm(1,2)*d2gm(2,3))
   cm(8,4)=12*gm(1,3)*gm(3,3)*dgm01(2,2)*dgm10(1,1)+0*gm(1,2)*gm(3,3)&
&   *dgm01(2,3)*dgm10(1,1)+48*gm(1,3)*gm(3,3)*dgm01(1,2)*dgm10(1,2)&
&   +12*gm(1,2)*gm(3,3)*dgm01(1,3)*dgm10(1,2)+18*gm(1,3)**2*dgm01(2,3)&
&   *dgm10(1,2)-12*gm(1,1)*gm(3,3)*dgm01(2,3)*dgm10(1,2)+24*gm(1,2)&
&   *gm(1,3)*dgm01(3,3)*dgm10(1,2)+12*gm(1,2)*gm(3,3)*dgm01(1,2)&
&   *dgm10(1,3)+15*gm(1,3)**2*dgm01(2,2)*dgm10(1,3)-3*gm(1,1)*gm(3,3)&
&   *dgm01(2,2)*dgm10(1,3)+48*gm(1,2)*gm(1,3)*dgm01(2,3)*dgm10(1,3)&
&   -6*gm(1,2)**2*dgm01(3,3)*dgm10(1,3)+12*gm(1,3)*gm(3,3)*dgm01(1,1)&
&   *dgm10(2,2)+15*gm(1,3)**2*dgm01(1,3)*dgm10(2,2)-3*gm(1,1)*gm(3,3)&
&   *dgm01(1,3)*dgm10(2,2)+0*gm(1,2)*gm(3,3)*dgm01(1,1)*dgm10(2,3)&
&   +18*gm(1,3)**2*dgm01(1,2)*dgm10(2,3)-12*gm(1,1)*gm(3,3)*dgm01(1,2)&
&   *dgm10(2,3)+48*gm(1,2)*gm(1,3)*dgm01(1,3)*dgm10(2,3)-12*gm(1,1)&
&   *gm(1,3)*dgm01(2,3)*dgm10(2,3)-3*gm(1,1)*gm(1,2)*dgm01(3,3)*dgm10(2,3)&
&   +24*gm(1,2)*gm(1,3)*dgm01(1,2)*dgm10(3,3)-6*gm(1,2)**2*dgm01(1,3)&
&   *dgm10(3,3)-3*gm(1,1)*gm(1,2)*dgm01(2,3)*dgm10(3,3)+24*gm(1,2)&
&   *gm(1,3)*gm(3,3)*d2gm(1,2)-6*gm(1,2)**2*gm(3,3)*d2gm(1,3)+gm(2,3)&
&   **2*(-3*dgm01(1,3)*dgm10(1,1)-3*dgm01(1,1)*dgm10(1,3)+3*gm(1,3)&
&   *d2gm(1,1)-6*gm(1,1)*d2gm(1,3))+gm(2,2)*(gm(1,3)*(12*dgm01(3,3)&
&   *dgm10(1,1)+60*dgm01(1,3)*dgm10(1,3)+12*dgm01(1,1)*dgm10(3,3))&
&   -3*gm(1,1)*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+15*gm(1,3)&
&   **2*d2gm(1,3)+gm(3,3)*(9*dgm01(1,3)*dgm10(1,1)+9*dgm01(1,1)*dgm10(1,3)&
&   +12*gm(1,3)*d2gm(1,1)-3*gm(1,1)*d2gm(1,3)))+15*gm(1,2)*gm(1,3)&
&   **2*d2gm(2,3)-3*gm(1,1)*gm(1,2)*gm(3,3)*d2gm(2,3)+gm(2,3)*(3*gm(1,2)&
&   *dgm01(3,3)*dgm10(1,1)-9*gm(1,1)*dgm01(3,3)*dgm10(1,2)+36*gm(1,2)&
&   *dgm01(1,3)*dgm10(1,3)-18*gm(1,1)*dgm01(2,3)*dgm10(1,3)-18*gm(1,1)&
&   *dgm01(1,3)*dgm10(2,3)+3*gm(1,2)*dgm01(1,1)*dgm10(3,3)-9*gm(1,1)&
&   *dgm01(1,2)*dgm10(3,3)+3*gm(1,3)**2*d2gm(1,2)+gm(3,3)*(-6*dgm01(1,2)&
&   *dgm10(1,1)-6*dgm01(1,1)*dgm10(1,2)+3*gm(1,2)*d2gm(1,1)-9*gm(1,1)&
&   *d2gm(1,2))+gm(1,3)*(0*dgm01(2,3)*dgm10(1,1)+24*dgm01(1,3)*dgm10(1,2)&
&   +24*dgm01(1,2)*dgm10(1,3)+0*dgm01(1,1)*dgm10(2,3)+18*gm(1,2)&
&   *d2gm(1,3)-6*gm(1,1)*d2gm(2,3)))
   cm(9,4)=12*gm(1,2)*gm(3,3)*dgm01(2,2)*dgm10(1,1)+60*gm(1,2)&
&   *gm(3,3)*dgm01(1,2)*dgm10(1,2)-6*gm(1,3)**2*dgm01(2,2)*dgm10(1,2)&
&   -3*gm(1,1)*gm(3,3)*dgm01(2,2)*dgm10(1,2)+48*gm(1,2)*gm(1,3)*dgm01(2,3)&
&   *dgm10(1,2)+15*gm(1,2)**2*dgm01(3,3)*dgm10(1,2)+24*gm(1,2)*gm(1,3)&
&   *dgm01(2,2)*dgm10(1,3)+18*gm(1,2)**2*dgm01(2,3)*dgm10(1,3)+12*gm(1,2)&
&   *gm(3,3)*dgm01(1,1)*dgm10(2,2)-6*gm(1,3)**2*dgm01(1,2)*dgm10(2,2)&
&   -3*gm(1,1)*gm(3,3)*dgm01(1,2)*dgm10(2,2)+24*gm(1,2)*gm(1,3)*dgm01(1,3)&
&   *dgm10(2,2)-3*gm(1,1)*gm(1,3)*dgm01(2,3)*dgm10(2,2)+48*gm(1,2)&
&   *gm(1,3)*dgm01(1,2)*dgm10(2,3)+18*gm(1,2)**2*dgm01(1,3)*dgm10(2,3)&
&   -3*gm(1,1)*gm(1,3)*dgm01(2,2)*dgm10(2,3)-12*gm(1,1)*gm(1,2)*dgm01(2,3)&
&   *dgm10(2,3)+15*gm(1,2)**2*dgm01(1,2)*dgm10(3,3)+15*gm(1,2)**2*gm(3,3)&
&   *d2gm(1,2)+gm(2,3)**2*(-3*dgm01(1,2)*dgm10(1,1)-3*dgm01(1,1)&
&   *dgm10(1,2)+3*gm(1,2)*d2gm(1,1)-6*gm(1,1)*d2gm(1,2))+15*gm(1,2)&
&   **2*gm(1,3)*d2gm(2,3)+gm(2,3)*(0*gm(1,2)*dgm01(2,3)*dgm10(1,1)&
&   +24*gm(1,2)*dgm01(1,3)*dgm10(1,2)-18*gm(1,1)*dgm01(2,3)*dgm10(1,2)&
&   +24*gm(1,2)*dgm01(1,2)*dgm10(1,3)-9*gm(1,1)*dgm01(2,2)*dgm10(1,3)&
&   -9*gm(1,1)*dgm01(1,3)*dgm10(2,2)+0*gm(1,2)*dgm01(1,1)*dgm10(2,3)&
&   -18*gm(1,1)*dgm01(1,2)*dgm10(2,3)+gm(1,3)*(3*dgm01(2,2)*dgm10(1,1)&
&   +36*dgm01(1,2)*dgm10(1,2)+3*dgm01(1,1)*dgm10(2,2)+18*gm(1,2)&
&   *d2gm(1,2))+3*gm(1,2)**2*d2gm(1,3)+gm(2,2)*(-6*dgm01(1,3)*dgm10(1,1)&
&   -6*dgm01(1,1)*dgm10(1,3)+3*gm(1,3)*d2gm(1,1)-9*gm(1,1)*d2gm(1,3))&
&   -6*gm(1,1)*gm(1,2)*d2gm(2,3))+gm(2,2)*(12*gm(1,2)*dgm01(3,3)&
&   *dgm10(1,1)-3*gm(1,1)*dgm01(3,3)*dgm10(1,2)+48*gm(1,2)*dgm01(1,3)&
&   *dgm10(1,3)-12*gm(1,1)*dgm01(2,3)*dgm10(1,3)-12*gm(1,1)*dgm01(1,3)&
&   *dgm10(2,3)+12*gm(1,2)*dgm01(1,1)*dgm10(3,3)-3*gm(1,1)*dgm01(1,2)&
&   *dgm10(3,3)-6*gm(1,3)**2*d2gm(1,2)+gm(3,3)*(9*dgm01(1,2)*dgm10(1,1)&
&   +9*dgm01(1,1)*dgm10(1,2)+12*gm(1,2)*d2gm(1,1)-3*gm(1,1)*d2gm(1,2))&
&   +gm(1,3)*(0*dgm01(2,3)*dgm10(1,1)+12*dgm01(1,3)*dgm10(1,2)+12*dgm01(1,2)&
&   *dgm10(1,3)+0*dgm01(1,1)*dgm10(2,3)+24*gm(1,2)*d2gm(1,3)-3*gm(1,1)&
&   *d2gm(2,3)))
   cm(10,4)=-3*gm(1,1)*gm(2,3)*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)&
&   *dgm10(3,3))+gm(1,3)*(gm(2,3)*(9*dgm01(3,3)*dgm10(1,1)+60*dgm01(1,3)&
&   *dgm10(1,3)+9*dgm01(1,1)*dgm10(3,3))-6*gm(1,2)*(dgm01(3,3)*dgm10(1,3)&
&   +dgm01(1,3)*dgm10(3,3))-3*gm(1,1)*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)&
&   *dgm10(3,3)))+gm(3,3)**2*(-6*dgm01(1,2)*dgm10(1,1)-6*dgm01(1,1)&
&   *dgm10(1,2)-3*(gm(1,2)*d2gm(1,1)+gm(1,1)*d2gm(1,2)))+gm(1,3)&
&   **2*(9*dgm01(3,3)*dgm10(1,2)+30*dgm01(2,3)*dgm10(1,3)+30*dgm01(1,3)&
&   *dgm10(2,3)+9*dgm01(1,2)*dgm10(3,3)+15*gm(2,3)*d2gm(1,3))+5*gm(1,3)&
&   **3*d2gm(2,3)+gm(3,3)*(-6*gm(1,2)*dgm01(3,3)*dgm10(1,1)-6*gm(1,1)&
&   *dgm01(3,3)*dgm10(1,2)-12*gm(1,2)*dgm01(1,3)*dgm10(1,3)-6*gm(1,1)&
&   *dgm01(2,3)*dgm10(1,3)-6*gm(1,1)*dgm01(1,3)*dgm10(2,3)-6*gm(1,2)&
&   *dgm01(1,1)*dgm10(3,3)-6*gm(1,1)*dgm01(1,2)*dgm10(3,3)+9*gm(1,3)&
&   **2*d2gm(1,2)+gm(2,3)*(6*dgm01(1,3)*dgm10(1,1)+6*dgm01(1,1)*dgm10(1,3)&
&   +9*gm(1,3)*d2gm(1,1)-3*gm(1,1)*d2gm(1,3))+gm(1,3)*(6*dgm01(2,3)&
&   *dgm10(1,1)+12*dgm01(1,3)*dgm10(1,2)+12*dgm01(1,2)*dgm10(1,3)&
&   +6*dgm01(1,1)*dgm10(2,3)-6*gm(1,2)*d2gm(1,3)-3*gm(1,1)*d2gm(2,3)))
   cm(1,5)=gm(1,1)*(gm(1,3)*(12*dgm01(1,1)*dgm10(1,1)+3*gm(1,1)&
&   *d2gm(1,1))+gm(1,1)*(9*dgm01(1,3)*dgm10(1,1)+9*dgm01(1,1)*dgm10(1,3)&
&   +2*gm(1,1)*d2gm(1,3)))
   cm(2,5)=gm(1,3)*(3*gm(1,2)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)&
&   *dgm10(1,2))-4.5d0*gm(1,1)*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)&
&   *dgm10(2,2))+1.5d0*gm(1,2)**2*d2gm(1,1)+gm(2,2)*(-9*dgm01(1,1)&
&   *dgm10(1,1)-4.5d0*gm(1,1)*d2gm(1,1)))+gm(1,2)*(gm(1,1)*(12*dgm01(2,3)&
&   *dgm10(1,1)+18*dgm01(1,3)*dgm10(1,2)+18*dgm01(1,2)*dgm10(1,3)&
&   +12*dgm01(1,1)*dgm10(2,3))+gm(2,3)*(24*dgm01(1,1)*dgm10(1,1)&
&   +12*gm(1,1)*d2gm(1,1)))+gm(1,2)**2*(10.5d0*dgm01(1,3)*dgm10(1,1)&
&   +10.5d0*dgm01(1,1)*dgm10(1,3)+9*gm(1,1)*d2gm(1,3))+gm(1,1)*(12*gm(2,3)&
&   *(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2))-3*gm(1,1)*(dgm01(2,2)&
&   *dgm10(1,3)+dgm01(1,3)*dgm10(2,2))+gm(2,2)*(-10.5d0*dgm01(1,3)&
&   *dgm10(1,1)-10.5d0*dgm01(1,1)*dgm10(1,3)-3*gm(1,1)*d2gm(1,3)))
   cm(3,5)=1.5d0*gm(1,3)**3*d2gm(1,1)+gm(1,3)*(gm(1,1)*(7.5d0*dgm01(3,3)&
&   *dgm10(1,1)+36*dgm01(1,3)*dgm10(1,3)+7.5d0*dgm01(1,1)*dgm10(3,3))&
&   +gm(3,3)*(15*dgm01(1,1)*dgm10(1,1)+7.5d0*gm(1,1)*d2gm(1,1)))&
&   +gm(1,3)**2*(13.5d0*dgm01(1,3)*dgm10(1,1)+13.5d0*dgm01(1,1)*dgm10(1,3)&
&   +9*gm(1,1)*d2gm(1,3))+gm(1,1)*(-3*gm(1,1)*(dgm01(3,3)*dgm10(1,3)&
&   +dgm01(1,3)*dgm10(3,3))+gm(3,3)*(1.5d0*dgm01(1,3)*dgm10(1,1)&
&   +1.5d0*dgm01(1,1)*dgm10(1,3)-3*gm(1,1)*d2gm(1,3)))
   cm(4,5)=3*gm(1,3)**2*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2)&
&   +gm(1,2)*d2gm(1,1))+gm(1,2)*(gm(1,1)*(12*dgm01(3,3)*dgm10(1,1)&
&   +36*dgm01(1,3)*dgm10(1,3)+12*dgm01(1,1)*dgm10(3,3))+gm(3,3)*(24*dgm01(1,1)&
&   *dgm10(1,1)+12*gm(1,1)*d2gm(1,1)))+gm(1,1)*(12*gm(3,3)*(dgm01(1,2)&
&   *dgm10(1,1)+dgm01(1,1)*dgm10(1,2))-6*gm(1,1)*(dgm01(2,3)*dgm10(1,3)&
&   +dgm01(1,3)*dgm10(2,3))+gm(2,3)*(-9*dgm01(1,3)*dgm10(1,1)-9*dgm01(1,1)&
&   *dgm10(1,3)-6*gm(1,1)*d2gm(1,3)))+gm(1,3)*(gm(1,1)*(3*dgm01(2,3)&
&   *dgm10(1,1)+18*dgm01(1,3)*dgm10(1,2)+18*dgm01(1,2)*dgm10(1,3)&
&   +3*dgm01(1,1)*dgm10(2,3))+gm(2,3)*(6*dgm01(1,1)*dgm10(1,1)+3*gm(1,1)&
&   *d2gm(1,1))+gm(1,2)*(24*dgm01(1,3)*dgm10(1,1)+24*dgm01(1,1)*dgm10(1,3)&
&   +18*gm(1,1)*d2gm(1,3)))
   cm(5,5)=gm(1,3)**2*(6*dgm01(1,1)*dgm10(1,1)+3*gm(1,1)*d2gm(1,1))&
&   +gm(1,1)*(gm(1,1)*(6*dgm01(3,3)*dgm10(1,1)+12*dgm01(1,3)*dgm10(1,3)&
&   +6*dgm01(1,1)*dgm10(3,3))+gm(3,3)*(24*dgm01(1,1)*dgm10(1,1)+6*gm(1,1)&
&   *d2gm(1,1)))+gm(1,1)*gm(1,3)*(18*dgm01(1,3)*dgm10(1,1)+18*dgm01(1,1)&
&   *dgm10(1,3)+6*gm(1,1)*d2gm(1,3))
   cm(6,5)=gm(1,1)*(3*gm(1,3)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)&
&   *dgm10(1,2))+6*gm(1,1)*(dgm01(2,3)*dgm10(1,1)+dgm01(1,3)*dgm10(1,2)&
&   +dgm01(1,2)*dgm10(1,3)+dgm01(1,1)*dgm10(2,3))+gm(2,3)*(24*dgm01(1,1)&
&   *dgm10(1,1)+6*gm(1,1)*d2gm(1,1)))+gm(1,2)*(gm(1,3)*(6*dgm01(1,1)&
&   *dgm10(1,1)+3*gm(1,1)*d2gm(1,1))+gm(1,1)*(15*dgm01(1,3)*dgm10(1,1)&
&   +15*dgm01(1,1)*dgm10(1,3)+6*gm(1,1)*d2gm(1,3)))
   cm(7,5)=(3*(-18*gm(1,1)*gm(2,3)*dgm01(2,2)+gm(1,2)*(180*gm(2,3)&
&   *dgm01(1,2)-36*gm(1,3)*dgm01(2,2))+90*gm(1,2)**2*dgm01(2,3)+gm(2,2)&
&   *(-18*gm(2,3)*dgm01(1,1)-36*gm(1,3)*dgm01(1,2)-36*gm(1,2)*dgm01(1,3)&
&   -18*gm(1,1)*dgm01(2,3)))*dgm10(1,1)+2*(270*gm(1,2)**2*dgm01(1,2)&
&   -54*gm(1,1)*gm(2,2)*dgm01(1,2)-54*gm(1,2)*(gm(2,2)*dgm01(1,1)&
&   +gm(1,1)*dgm01(2,2)))*dgm10(1,3)+2*dgm01(1,3)*(270*gm(1,2)**2*dgm10(1,2)&
&   -54*gm(1,1)*gm(2,2)*dgm10(1,2)-54*gm(1,2)*(gm(2,2)*dgm10(1,1)&
&   +gm(1,1)*dgm10(2,2)))+3*dgm01(1,1)*(-18*gm(1,1)*gm(2,3)*dgm10(2,2)&
&   +gm(1,2)*(180*gm(2,3)*dgm10(1,2)-36*gm(1,3)*dgm10(2,2))+90*gm(1,2)&
&   **2*dgm10(2,3)+gm(2,2)*(-18*gm(2,3)*dgm10(1,1)-36*gm(1,3)*dgm10(1,2)&
&   -36*gm(1,2)*dgm10(1,3)-18*gm(1,1)*dgm10(2,3)))+3*(90*gm(1,2)&
&   **2*gm(2,3)-18*gm(2,2)*(2*gm(1,2)*gm(1,3)+gm(1,1)*gm(2,3)))*d2gm(1,1)&
&   +2*(90*gm(1,2)**3-54*gm(1,1)*gm(1,2)*gm(2,2))*d2gm(1,3))/36.d0
   cm(8,5)=9*gm(1,2)*gm(3,3)*dgm01(1,3)*dgm10(1,1)-4.5d0*gm(1,1)&
&   *gm(3,3)*dgm01(2,3)*dgm10(1,1)-3*gm(1,1)*gm(3,3)*dgm01(1,3)*dgm10(1,2)&
&   +9*gm(1,2)*gm(3,3)*dgm01(1,1)*dgm10(1,3)-3*gm(1,1)*gm(3,3)*dgm01(1,2)&
&   *dgm10(1,3)-3*gm(1,1)*gm(1,2)*dgm01(3,3)*dgm10(1,3)-4.5d0*gm(1,1)&
&   *gm(3,3)*dgm01(1,1)*dgm10(2,3)-3*gm(1,1)*gm(1,2)*dgm01(1,3)*dgm10(3,3)&
&   +gm(1,3)*(-6*gm(1,1)*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3))&
&   +gm(1,2)*(12*dgm01(3,3)*dgm10(1,1)+60*dgm01(1,3)*dgm10(1,3)+12*dgm01(1,1)&
&   *dgm10(3,3))+12*gm(3,3)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2)&
&   +gm(1,2)*d2gm(1,1)))-3*gm(1,1)*gm(1,2)*gm(3,3)*d2gm(1,3)+gm(1,3)&
&   **2*(1.5d0*dgm01(2,3)*dgm10(1,1)+15*dgm01(1,3)*dgm10(1,2)+15*dgm01(1,2)&
&   *dgm10(1,3)+1.5d0*dgm01(1,1)*dgm10(2,3)+15*gm(1,2)*d2gm(1,3))&
&   +gm(2,3)*(gm(1,1)*(-4.5d0*dgm01(3,3)*dgm10(1,1)-12*dgm01(1,3)&
&   *dgm10(1,3)-4.5d0*dgm01(1,1)*dgm10(3,3))+1.5d0*gm(1,3)**2*d2gm(1,1)&
&   +gm(3,3)*(-9*dgm01(1,1)*dgm10(1,1)-4.5d0*gm(1,1)*d2gm(1,1))+gm(1,3)&
&   *(-3*dgm01(1,3)*dgm10(1,1)-3*dgm01(1,1)*dgm10(1,3)-6*gm(1,1)&
&   *d2gm(1,3)))
   cm(9,5)=15*gm(1,2)*gm(3,3)*dgm01(1,2)*dgm10(1,1)-3*gm(1,3)**2*dgm01(2,2)&
&   *dgm10(1,1)-1.5d0*gm(1,1)*gm(3,3)*dgm01(2,2)*dgm10(1,1)+9*gm(1,2)&
&   *gm(1,3)*dgm01(2,3)*dgm10(1,1)+7.5d0*gm(1,2)**2*dgm01(3,3)*dgm10(1,1)&
&   +15*gm(1,2)*gm(3,3)*dgm01(1,1)*dgm10(1,2)+30*gm(1,2)*gm(1,3)&
&   *dgm01(1,3)*dgm10(1,2)+30*gm(1,2)*gm(1,3)*dgm01(1,2)*dgm10(1,3)&
&   +30*gm(1,2)**2*dgm01(1,3)*dgm10(1,3)-3*gm(1,1)*gm(1,3)*dgm01(2,2)&
&   *dgm10(1,3)-6*gm(1,1)*gm(1,2)*dgm01(2,3)*dgm10(1,3)-3*gm(1,3)&
&   **2*dgm01(1,1)*dgm10(2,2)-1.5d0*gm(1,1)*gm(3,3)*dgm01(1,1)*dgm10(2,2)&
&   -3*gm(1,1)*gm(1,3)*dgm01(1,3)*dgm10(2,2)+9*gm(1,2)*gm(1,3)*dgm01(1,1)&
&   *dgm10(2,3)-6*gm(1,1)*gm(1,2)*dgm01(1,3)*dgm10(2,3)+7.5d0*gm(1,2)&
&   **2*dgm01(1,1)*dgm10(3,3)+7.5d0*gm(1,2)**2*gm(3,3)*d2gm(1,1)&
&   +gm(2,3)**2*(-6*dgm01(1,1)*dgm10(1,1)-3*gm(1,1)*d2gm(1,1))+15*gm(1,2)&
&   **2*gm(1,3)*d2gm(1,3)+gm(2,3)*(-6*gm(1,1)*(dgm01(2,3)*dgm10(1,1)&
&   +dgm01(1,3)*dgm10(1,2)+dgm01(1,2)*dgm10(1,3)+dgm01(1,1)*dgm10(2,3))&
&   +9*gm(1,3)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2)+gm(1,2)&
&   *d2gm(1,1))+gm(1,2)*(3*dgm01(1,3)*dgm10(1,1)+3*dgm01(1,1)*dgm10(1,3)&
&   -6*gm(1,1)*d2gm(1,3)))+gm(2,2)*(gm(1,1)*(-1.5d0*dgm01(3,3)*dgm10(1,1)&
&   -6*dgm01(1,3)*dgm10(1,3)-1.5d0*dgm01(1,1)*dgm10(3,3))-3*gm(1,3)&
&   **2*d2gm(1,1)+gm(3,3)*(-3*dgm01(1,1)*dgm10(1,1)-1.5d0*gm(1,1)&
&   *d2gm(1,1))+gm(1,3)*(-9*dgm01(1,3)*dgm10(1,1)-9*dgm01(1,1)*dgm10(1,3)&
&   -3*gm(1,1)*d2gm(1,3)))
   cm(10,5)=gm(3,3)**2*(-3*dgm01(1,1)*dgm10(1,1)-1.5d0*gm(1,1)&
&   *d2gm(1,1))+gm(1,3)*(gm(1,3)*(4.5d0*dgm01(3,3)*dgm10(1,1)+30*dgm01(1,3)&
&   *dgm10(1,3)+4.5d0*dgm01(1,1)*dgm10(3,3))-3*gm(1,1)*(dgm01(3,3)&
&   *dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+5*gm(1,3)**2*d2gm(1,3))+gm(3,3)&
&   *(gm(1,1)*(-3*dgm01(3,3)*dgm10(1,1)-6*dgm01(1,3)*dgm10(1,3)-3*dgm01(1,1)&
&   *dgm10(3,3))+4.5d0*gm(1,3)**2*d2gm(1,1)+gm(1,3)*(6*dgm01(1,3)&
&   *dgm10(1,1)+6*dgm01(1,1)*dgm10(1,3)-3*gm(1,1)*d2gm(1,3)))
   cm(1,6)=gm(1,1)*(gm(1,2)*(12*dgm01(1,1)*dgm10(1,1)+3*gm(1,1)&
&   *d2gm(1,1))+gm(1,1)*(9*dgm01(1,2)*dgm10(1,1)+9*dgm01(1,1)*dgm10(1,2)&
&   +2*gm(1,1)*d2gm(1,2)))
   cm(2,6)=1.5d0*gm(1,2)**3*d2gm(1,1)+gm(1,2)*(gm(1,1)*(7.5d0*dgm01(2,2)&
&   *dgm10(1,1)+36*dgm01(1,2)*dgm10(1,2)+7.5d0*dgm01(1,1)*dgm10(2,2))&
&   +gm(2,2)*(15*dgm01(1,1)*dgm10(1,1)+7.5d0*gm(1,1)*d2gm(1,1)))&
&   +gm(1,2)**2*(13.5d0*dgm01(1,2)*dgm10(1,1)+13.5d0*dgm01(1,1)*dgm10(1,2)&
&   +9*gm(1,1)*d2gm(1,2))+gm(1,1)*(-3*gm(1,1)*(dgm01(2,2)*dgm10(1,2)&
&   +dgm01(1,2)*dgm10(2,2))+gm(2,2)*(1.5d0*dgm01(1,2)*dgm10(1,1)&
&   +1.5d0*dgm01(1,1)*dgm10(1,2)-3*gm(1,1)*d2gm(1,2)))
   cm(3,6)=gm(1,2)*(-4.5d0*gm(1,1)*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)&
&   *dgm10(3,3))+gm(3,3)*(-9*dgm01(1,1)*dgm10(1,1)-4.5d0*gm(1,1)&
&   *d2gm(1,1)))+gm(1,3)*(3*gm(1,2)*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)&
&   *dgm10(1,3))+gm(1,1)*(12*dgm01(2,3)*dgm10(1,1)+18*dgm01(1,3)&
&   *dgm10(1,2)+18*dgm01(1,2)*dgm10(1,3)+12*dgm01(1,1)*dgm10(2,3))&
&   +gm(2,3)*(24*dgm01(1,1)*dgm10(1,1)+12*gm(1,1)*d2gm(1,1)))+gm(1,3)&
&   **2*(10.5d0*dgm01(1,2)*dgm10(1,1)+10.5d0*dgm01(1,1)*dgm10(1,2)&
&   +1.5d0*gm(1,2)*d2gm(1,1)+9*gm(1,1)*d2gm(1,2))+gm(1,1)*(12*gm(2,3)&
&   *(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))-3*gm(1,1)*(dgm01(3,3)&
&   *dgm10(1,2)+dgm01(1,2)*dgm10(3,3))+gm(3,3)*(-10.5d0*dgm01(1,2)&
&   *dgm10(1,1)-10.5d0*dgm01(1,1)*dgm10(1,2)-3*gm(1,1)*d2gm(1,2)))
   cm(4,6)=3*gm(1,2)**2*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))&
&   +gm(1,2)*(gm(1,1)*(3*dgm01(2,3)*dgm10(1,1)+18*dgm01(1,3)*dgm10(1,2)&
&   +18*dgm01(1,2)*dgm10(1,3)+3*dgm01(1,1)*dgm10(2,3))+gm(2,3)*(6*dgm01(1,1)&
&   *dgm10(1,1)+3*gm(1,1)*d2gm(1,1)))+gm(1,1)*(12*gm(2,2)*(dgm01(1,3)&
&   *dgm10(1,1)+dgm01(1,1)*dgm10(1,3))-6*gm(1,1)*(dgm01(2,3)*dgm10(1,2)&
&   +dgm01(1,2)*dgm10(2,3))+gm(2,3)*(-9*dgm01(1,2)*dgm10(1,1)-9*dgm01(1,1)&
&   *dgm10(1,2)-6*gm(1,1)*d2gm(1,2)))+gm(1,3)*(gm(1,1)*(12*dgm01(2,2)&
&   *dgm10(1,1)+36*dgm01(1,2)*dgm10(1,2)+12*dgm01(1,1)*dgm10(2,2))&
&   +3*gm(1,2)**2*d2gm(1,1)+gm(2,2)*(24*dgm01(1,1)*dgm10(1,1)+12*gm(1,1)&
&   *d2gm(1,1))+gm(1,2)*(24*dgm01(1,2)*dgm10(1,1)+24*dgm01(1,1)*dgm10(1,2)&
&   +18*gm(1,1)*d2gm(1,2)))
   cm(5,6)=gm(1,2)*(3*gm(1,1)*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)&
&   *dgm10(1,3))+gm(1,3)*(6*dgm01(1,1)*dgm10(1,1)+3*gm(1,1)*d2gm(1,1)))&
&   +gm(1,1)*(6*gm(1,1)*(dgm01(2,3)*dgm10(1,1)+dgm01(1,3)*dgm10(1,2)&
&   +dgm01(1,2)*dgm10(1,3)+dgm01(1,1)*dgm10(2,3))+gm(2,3)*(24*dgm01(1,1)&
&   *dgm10(1,1)+6*gm(1,1)*d2gm(1,1))+gm(1,3)*(15*dgm01(1,2)*dgm10(1,1)&
&   +15*dgm01(1,1)*dgm10(1,2)+6*gm(1,1)*d2gm(1,2)))
   cm(6,6)=gm(1,2)**2*(6*dgm01(1,1)*dgm10(1,1)+3*gm(1,1)*d2gm(1,1))&
&   +gm(1,1)*(gm(1,1)*(6*dgm01(2,2)*dgm10(1,1)+12*dgm01(1,2)*dgm10(1,2)&
&   +6*dgm01(1,1)*dgm10(2,2))+gm(2,2)*(24*dgm01(1,1)*dgm10(1,1)+6*gm(1,1)&
&   *d2gm(1,1)))+gm(1,1)*gm(1,2)*(18*dgm01(1,2)*dgm10(1,1)+18*dgm01(1,1)&
&   *dgm10(1,2)+6*gm(1,1)*d2gm(1,2))
   cm(7,6)=gm(2,2)**2*(-3*dgm01(1,1)*dgm10(1,1)-1.5d0*gm(1,1)*d2gm(1,1))&
&   +gm(1,2)*(gm(1,2)*(4.5d0*dgm01(2,2)*dgm10(1,1)+30*dgm01(1,2)&
&   *dgm10(1,2)+4.5d0*dgm01(1,1)*dgm10(2,2))-3*gm(1,1)*(dgm01(2,2)&
&   *dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+5*gm(1,2)**2*d2gm(1,2))+gm(2,2)&
&   *(gm(1,1)*(-3*dgm01(2,2)*dgm10(1,1)-6*dgm01(1,2)*dgm10(1,2)-3*dgm01(1,1)&
&   *dgm10(2,2))+4.5d0*gm(1,2)**2*d2gm(1,1)+gm(1,2)*(6*dgm01(1,2)&
&   *dgm10(1,1)+6*dgm01(1,1)*dgm10(1,2)-3*gm(1,1)*d2gm(1,2)))
   cm(8,6)=-9*gm(1,2)*gm(3,3)*dgm01(1,2)*dgm10(1,1)+7.5d0*gm(1,3)&
&   **2*dgm01(2,2)*dgm10(1,1)-1.5d0*gm(1,1)*gm(3,3)*dgm01(2,2)*dgm10(1,1)&
&   +9*gm(1,2)*gm(1,3)*dgm01(2,3)*dgm10(1,1)-3*gm(1,2)**2*dgm01(3,3)&
&   *dgm10(1,1)-9*gm(1,2)*gm(3,3)*dgm01(1,1)*dgm10(1,2)+30*gm(1,3)&
&   **2*dgm01(1,2)*dgm10(1,2)-6*gm(1,1)*gm(3,3)*dgm01(1,2)*dgm10(1,2)&
&   +30*gm(1,2)*gm(1,3)*dgm01(1,3)*dgm10(1,2)-6*gm(1,1)*gm(1,3)*dgm01(2,3)&
&   *dgm10(1,2)-3*gm(1,1)*gm(1,2)*dgm01(3,3)*dgm10(1,2)+30*gm(1,2)&
&   *gm(1,3)*dgm01(1,2)*dgm10(1,3)+7.5d0*gm(1,3)**2*dgm01(1,1)*dgm10(2,2)&
&   -1.5d0*gm(1,1)*gm(3,3)*dgm01(1,1)*dgm10(2,2)+9*gm(1,2)*gm(1,3)&
&   *dgm01(1,1)*dgm10(2,3)-6*gm(1,1)*gm(1,3)*dgm01(1,2)*dgm10(2,3)&
&   -3*gm(1,2)**2*dgm01(1,1)*dgm10(3,3)-3*gm(1,1)*gm(1,2)*dgm01(1,2)&
&   *dgm10(3,3)-3*gm(1,2)**2*gm(3,3)*d2gm(1,1)+gm(2,3)**2*(-6*dgm01(1,1)&
&   *dgm10(1,1)-3*gm(1,1)*d2gm(1,1))+gm(2,2)*(15*gm(1,3)*(dgm01(1,3)&
&   *dgm10(1,1)+dgm01(1,1)*dgm10(1,3))-1.5d0*gm(1,1)*(dgm01(3,3)&
&   *dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+7.5d0*gm(1,3)**2*d2gm(1,1)&
&   +gm(3,3)*(-3*dgm01(1,1)*dgm10(1,1)-1.5d0*gm(1,1)*d2gm(1,1)))&
&   +15*gm(1,2)*gm(1,3)**2*d2gm(1,2)-3*gm(1,1)*gm(1,2)*gm(3,3)*d2gm(1,2)&
&   +gm(2,3)*(9*gm(1,2)*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))&
&   -6*gm(1,1)*(dgm01(2,3)*dgm10(1,1)+dgm01(1,3)*dgm10(1,2)+dgm01(1,2)&
&   *dgm10(1,3)+dgm01(1,1)*dgm10(2,3))+gm(1,3)*(3*dgm01(1,2)*dgm10(1,1)&
&   +3*dgm01(1,1)*dgm10(1,2)+9*gm(1,2)*d2gm(1,1)-6*gm(1,1)*d2gm(1,2)))
   cm(9,6)=gm(1,1)*(gm(2,3)*(-4.5d0*dgm01(2,2)*dgm10(1,1)-12*dgm01(1,2)&
&   *dgm10(1,2)-4.5d0*dgm01(1,1)*dgm10(2,2))-3*gm(1,3)*(dgm01(2,2)&
&   *dgm10(1,2)+dgm01(1,2)*dgm10(2,2)))+gm(1,2)**2*(1.5d0*dgm01(2,3)&
&   *dgm10(1,1)+15*dgm01(1,3)*dgm10(1,2)+15*dgm01(1,2)*dgm10(1,3)&
&   +1.5d0*dgm01(1,1)*dgm10(2,3)+1.5d0*gm(2,3)*d2gm(1,1)+15*gm(1,3)&
&   *d2gm(1,2))+gm(1,2)*(gm(1,3)*(12*dgm01(2,2)*dgm10(1,1)+60*dgm01(1,2)&
&   *dgm10(1,2)+12*dgm01(1,1)*dgm10(2,2))-6*gm(1,1)*(dgm01(2,3)*dgm10(1,2)&
&   +dgm01(1,2)*dgm10(2,3))+gm(2,3)*(-3*dgm01(1,2)*dgm10(1,1)-3*dgm01(1,1)&
&   *dgm10(1,2)-6*gm(1,1)*d2gm(1,2)))+gm(2,2)*(12*gm(1,2)*dgm01(1,3)&
&   *dgm10(1,1)-4.5d0*gm(1,1)*dgm01(2,3)*dgm10(1,1)-3*gm(1,1)*dgm01(1,3)&
&   *dgm10(1,2)+12*gm(1,2)*dgm01(1,1)*dgm10(1,3)-3*gm(1,1)*dgm01(1,2)&
&   *dgm10(1,3)-4.5d0*gm(1,1)*dgm01(1,1)*dgm10(2,3)+gm(2,3)*(-9*dgm01(1,1)&
&   *dgm10(1,1)-4.5d0*gm(1,1)*d2gm(1,1))+gm(1,3)*(9*dgm01(1,2)*dgm10(1,1)&
&   +9*dgm01(1,1)*dgm10(1,2)+12*gm(1,2)*d2gm(1,1)-3*gm(1,1)*d2gm(1,2)))
   cm(10,6)=(gm(1,3)**2*(270*dgm01(2,3)*dgm10(1,1)+540*dgm01(1,3)&
&   *dgm10(1,2)+540*dgm01(1,2)*dgm10(1,3)+270*dgm01(1,1)*dgm10(2,3))&
&   +gm(3,3)*(-108*gm(1,2)*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))&
&   +gm(1,1)*(-54*dgm01(2,3)*dgm10(1,1)-108*dgm01(1,3)*dgm10(1,2)&
&   -108*dgm01(1,2)*dgm10(1,3)-54*dgm01(1,1)*dgm10(2,3)))+gm(2,3)&
&   *(540*gm(1,3)*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3))-54*gm(1,1)&
&   *(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3))+270*gm(1,3)**2*d2gm(1,1)&
&   +gm(3,3)*(-108*dgm01(1,1)*dgm10(1,1)-54*gm(1,1)*d2gm(1,1)))+180*gm(1,3)&
&   **3*d2gm(1,2)+gm(1,3)*(-108*(gm(1,2)*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)&
&   *dgm10(3,3))+gm(1,1)*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3)))&
&   +gm(3,3)*(-216*dgm01(1,2)*dgm10(1,1)-216*dgm01(1,1)*dgm10(1,2)&
&   -108*gm(1,2)*d2gm(1,1)-108*gm(1,1)*d2gm(1,2))))/36.d0
   cm(1,7)=-7.5d0*gm(1,1)*gm(2,2)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)&
&   *dgm10(1,2))+2.5d0*gm(1,2)**3*d2gm(1,1)+gm(1,2)**2*(16.5d0*dgm01(1,2)&
&   *dgm10(1,1)+16.5d0*dgm01(1,1)*dgm10(1,2)+9*gm(1,1)*d2gm(1,2))&
&   +gm(1,1)**2*(0*dgm01(2,2)*dgm10(1,2)+0*dgm01(1,2)*dgm10(2,2)&
&   -3*gm(2,2)*d2gm(1,2))+gm(1,2)*(gm(2,2)*(-3*dgm01(1,1)*dgm10(1,1)&
&   -1.5d0*gm(1,1)*d2gm(1,1))+gm(1,1)*(4.5d0*dgm01(2,2)*dgm10(1,1)&
&   +36*dgm01(1,2)*dgm10(1,2)+4.5d0*dgm01(1,1)*dgm10(2,2)+3*gm(1,1)&
&   *d2gm(2,2)))
   cm(2,7)=gm(2,2)**2*(15*dgm01(1,2)*dgm10(1,1)+15*dgm01(1,1)*dgm10(1,2)&
&   +3*gm(1,2)*d2gm(1,1)+12*gm(1,1)*d2gm(1,2))+gm(1,2)*(15*gm(1,1)&
&   *dgm01(2,2)*dgm10(2,2)+10.5d0*gm(1,2)*(dgm01(2,2)*dgm10(1,2)&
&   +dgm01(1,2)*dgm10(2,2))+1.5d0*gm(1,2)**2*d2gm(2,2))+gm(2,2)*(31.5d0*gm(1,1)&
&   *(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+6*gm(1,2)**2*d2gm(1,2)&
&   +gm(1,2)*(13.5d0*dgm01(2,2)*dgm10(1,1)+24*dgm01(1,2)*dgm10(1,2)&
&   +13.5d0*dgm01(1,1)*dgm10(2,2)+7.5d0*gm(1,1)*d2gm(2,2)))
   cm(3,7)=-6*gm(1,2)*gm(3,3)*dgm01(2,2)*dgm10(1,1)-24*gm(1,2)&
&   *gm(3,3)*dgm01(1,2)*dgm10(1,2)-4.5d0*gm(1,3)**2*dgm01(2,2)*dgm10(1,2)&
&   -7.5d0*gm(1,1)*gm(3,3)*dgm01(2,2)*dgm10(1,2)+18*gm(1,2)*gm(1,3)&
&   *dgm01(2,3)*dgm10(1,2)-6*gm(1,2)**2*dgm01(3,3)*dgm10(1,2)+3*gm(1,2)&
&   *gm(1,3)*dgm01(2,2)*dgm10(1,3)-6*gm(1,2)*gm(3,3)*dgm01(1,1)*dgm10(2,2)&
&   -4.5d0*gm(1,3)**2*dgm01(1,2)*dgm10(2,2)-7.5d0*gm(1,1)*gm(3,3)&
&   *dgm01(1,2)*dgm10(2,2)+3*gm(1,2)*gm(1,3)*dgm01(1,3)*dgm10(2,2)&
&   +12*gm(1,1)*gm(1,3)*dgm01(2,3)*dgm10(2,2)-4.5d0*gm(1,1)*gm(1,2)&
&   *dgm01(3,3)*dgm10(2,2)+18*gm(1,2)*gm(1,3)*dgm01(1,2)*dgm10(2,3)&
&   +12*gm(1,1)*gm(1,3)*dgm01(2,2)*dgm10(2,3)-6*gm(1,2)**2*dgm01(1,2)&
&   *dgm10(3,3)-4.5d0*gm(1,1)*gm(1,2)*dgm01(2,2)*dgm10(3,3)-6*gm(1,2)&
&   **2*gm(3,3)*d2gm(1,2)+gm(2,3)**2*(22.5d0*dgm01(1,2)*dgm10(1,1)&
&   +22.5d0*dgm01(1,1)*dgm10(1,2)+7.5d0*gm(1,2)*d2gm(1,1)+15*gm(1,1)&
&   *d2gm(1,2))+gm(2,2)*(-1.5d0*gm(1,2)*dgm01(3,3)*dgm10(1,1)-3*gm(1,1)&
&   *dgm01(3,3)*dgm10(1,2)+gm(1,3)*(-3*dgm01(2,3)*dgm10(1,1)-12*dgm01(1,3)&
&   *dgm10(1,2)-12*dgm01(1,2)*dgm10(1,3)-3*dgm01(1,1)*dgm10(2,3))&
&   -1.5d0*gm(1,2)*dgm01(1,1)*dgm10(3,3)-3*gm(1,1)*dgm01(1,2)*dgm10(3,3)&
&   -6*gm(1,3)**2*d2gm(1,2)+gm(3,3)*(-4.5d0*dgm01(1,2)*dgm10(1,1)&
&   -4.5d0*dgm01(1,1)*dgm10(1,2)-1.5d0*gm(1,2)*d2gm(1,1)-3*gm(1,1)&
&   *d2gm(1,2)))+1.5d0*gm(1,2)*gm(1,3)**2*d2gm(2,2)-4.5d0*gm(1,1)&
&   *gm(1,2)*gm(3,3)*d2gm(2,2)+gm(2,3)*(15*gm(1,2)*dgm01(2,3)*dgm10(1,1)&
&   +18*gm(1,2)*dgm01(1,3)*dgm10(1,2)+30*gm(1,1)*dgm01(2,3)*dgm10(1,2)&
&   +18*gm(1,2)*dgm01(1,2)*dgm10(1,3)+12*gm(1,1)*dgm01(2,2)*dgm10(1,3)&
&   +12*gm(1,1)*dgm01(1,3)*dgm10(2,2)+15*gm(1,2)*dgm01(1,1)*dgm10(2,3)&
&   +30*gm(1,1)*dgm01(1,2)*dgm10(2,3)-3*gm(2,2)*(dgm01(1,3)*dgm10(1,1)&
&   +dgm01(1,1)*dgm10(1,3)+gm(1,3)*d2gm(1,1))+gm(1,3)*(9*dgm01(2,2)&
&   *dgm10(1,1)+36*dgm01(1,2)*dgm10(1,2)+9*dgm01(1,1)*dgm10(2,2)&
&   +18*gm(1,2)*d2gm(1,2)+12*gm(1,1)*d2gm(2,2)))
   cm(4,7)=gm(1,1)*(24*gm(1,3)*dgm01(2,2)*dgm10(2,2)+27*gm(2,3)&
&   *(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2)))-3*gm(2,2)**2*(dgm01(1,3)&
&   *dgm10(1,1)+dgm01(1,1)*dgm10(1,3)+gm(1,3)*d2gm(1,1))+gm(1,2)&
&   **2*(6*dgm01(2,3)*dgm10(1,2)+3*dgm01(2,2)*dgm10(1,3)+3*dgm01(1,3)&
&   *dgm10(2,2)+6*dgm01(1,2)*dgm10(2,3)+6*gm(2,3)*d2gm(1,2)+3*gm(1,3)&
&   *d2gm(2,2))+gm(1,2)*(12*gm(1,3)*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)&
&   *dgm10(2,2))+3*gm(1,1)*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))&
&   +gm(2,3)*(12*dgm01(2,2)*dgm10(1,1)+24*dgm01(1,2)*dgm10(1,2)+12*dgm01(1,1)&
&   *dgm10(2,2)+3*gm(1,1)*d2gm(2,2)))+gm(2,2)*(9*gm(1,2)*dgm01(2,3)&
&   *dgm10(1,1)+6*gm(1,2)*dgm01(1,3)*dgm10(1,2)+24*gm(1,1)*dgm01(2,3)&
&   *dgm10(1,2)+6*gm(1,2)*dgm01(1,2)*dgm10(1,3)+12*gm(1,1)*dgm01(2,2)&
&   *dgm10(1,3)+12*gm(1,1)*dgm01(1,3)*dgm10(2,2)+9*gm(1,2)*dgm01(1,1)&
&   *dgm10(2,3)+24*gm(1,1)*dgm01(1,2)*dgm10(2,3)+gm(2,3)*(33*dgm01(1,2)&
&   *dgm10(1,1)+33*dgm01(1,1)*dgm10(1,2)+9*gm(1,2)*d2gm(1,1)+24*gm(1,1)&
&   *d2gm(1,2))+gm(1,3)*(6*dgm01(2,2)*dgm10(1,1)+12*dgm01(1,2)*dgm10(1,2)&
&   +6*dgm01(1,1)*dgm10(2,2)+6*gm(1,2)*d2gm(1,2)+12*gm(1,1)*d2gm(2,2)))
   cm(5,7)=gm(1,2)**2*(7.5d0*dgm01(2,3)*dgm10(1,1)+3*dgm01(1,3)&
&   *dgm10(1,2)+3*dgm01(1,2)*dgm10(1,3)+7.5d0*dgm01(1,1)*dgm10(2,3)&
&   +7.5d0*gm(2,3)*d2gm(1,1)+3*gm(1,3)*d2gm(1,2))+gm(2,2)*(-3*gm(1,2)&
&   *dgm01(1,3)*dgm10(1,1)-1.5d0*gm(1,1)*dgm01(2,3)*dgm10(1,1)-9*gm(1,1)&
&   *dgm01(1,3)*dgm10(1,2)-3*gm(1,2)*dgm01(1,1)*dgm10(1,3)-9*gm(1,1)&
&   *dgm01(1,2)*dgm10(1,3)-1.5d0*gm(1,1)*dgm01(1,1)*dgm10(2,3)+gm(2,3)&
&   *(-3*dgm01(1,1)*dgm10(1,1)-1.5d0*gm(1,1)*d2gm(1,1))+gm(1,3)*(-12*dgm01(1,2)&
&   *dgm10(1,1)-12*dgm01(1,1)*dgm10(1,2)-3*gm(1,2)*d2gm(1,1)-9*gm(1,1)&
&   *d2gm(1,2)))+gm(1,2)*(gm(1,1)*(24*dgm01(2,3)*dgm10(1,2)+3*dgm01(2,2)&
&   *dgm10(1,3)+3*dgm01(1,3)*dgm10(2,2)+24*dgm01(1,2)*dgm10(2,3))&
&   +gm(2,3)*(39*dgm01(1,2)*dgm10(1,1)+39*dgm01(1,1)*dgm10(1,2)+24*gm(1,1)&
&   *d2gm(1,2))+gm(1,3)*(0*dgm01(2,2)*dgm10(1,1)+12*dgm01(1,2)*dgm10(1,2)&
&   +0*dgm01(1,1)*dgm10(2,2)+3*gm(1,1)*d2gm(2,2)))+gm(1,1)*(-6*gm(1,3)&
&   *(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+6*gm(1,1)*(dgm01(2,3)&
&   *dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+gm(2,3)*(10.5d0*dgm01(2,2)&
&   *dgm10(1,1)+48*dgm01(1,2)*dgm10(1,2)+10.5d0*dgm01(1,1)*dgm10(2,2)&
&   +6*gm(1,1)*d2gm(2,2)))
   cm(6,7)=12*gm(1,1)**2*dgm01(2,2)*dgm10(2,2)+21*gm(1,1)*gm(1,2)&
&   *(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+gm(2,2)**2*(-3*dgm01(1,1)&
&   *dgm10(1,1)-1.5d0*gm(1,1)*d2gm(1,1))+3*gm(1,2)**3*d2gm(1,2)+gm(1,2)&
&   **2*(7.5d0*dgm01(2,2)*dgm10(1,1)+18*dgm01(1,2)*dgm10(1,2)+7.5d0*dgm01(1,1)&
&   *dgm10(2,2)+3*gm(1,1)*d2gm(2,2))+gm(2,2)*(4.5d0*gm(1,2)**2*d2gm(1,1)&
&   +gm(1,2)*(24*dgm01(1,2)*dgm10(1,1)+24*dgm01(1,1)*dgm10(1,2)+15*gm(1,1)&
&   *d2gm(1,2))+gm(1,1)*(9*dgm01(2,2)*dgm10(1,1)+30*dgm01(1,2)*dgm10(1,2)&
&   +9*dgm01(1,1)*dgm10(2,2)+6*gm(1,1)*d2gm(2,2)))
   cm(7,7)=9*gm(1,2)**2*dgm01(2,2)*dgm10(2,2)+gm(2,2)**3*d2gm(1,1)&
&   +gm(2,2)**2*(1.5d0*dgm01(2,2)*dgm10(1,1)+12*dgm01(1,2)*dgm10(1,2)&
&   +1.5d0*dgm01(1,1)*dgm10(2,2)+6*gm(1,2)*d2gm(1,2)-1.5d0*gm(1,1)&
&   *d2gm(2,2))+gm(2,2)*(-6*gm(1,1)*dgm01(2,2)*dgm10(2,2)+21*gm(1,2)&
&   *(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+4.5d0*gm(1,2)&
&   **2*d2gm(2,2))
   cm(8,7)=-15*gm(1,2)*gm(3,3)*dgm01(2,2)*dgm10(1,2)-15*gm(1,2)&
&   *gm(3,3)*dgm01(1,2)*dgm10(2,2)+15*gm(1,3)**2*dgm01(2,2)*dgm10(2,2)&
&   -3*gm(1,1)*gm(3,3)*dgm01(2,2)*dgm10(2,2)+9*gm(1,2)*gm(1,3)*dgm01(2,3)&
&   *dgm10(2,2)-3*gm(1,2)**2*dgm01(3,3)*dgm10(2,2)+9*gm(1,2)*gm(1,3)&
&   *dgm01(2,2)*dgm10(2,3)-3*gm(1,2)**2*dgm01(2,2)*dgm10(3,3)-1.5d0*gm(2,2)&
&   **2*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3)+gm(3,3)*d2gm(1,1))&
&   -3*gm(1,2)**2*gm(3,3)*d2gm(2,2)+gm(2,3)**2*(1.5d0*dgm01(2,2)&
&   *dgm10(1,1)+6*dgm01(1,2)*dgm10(1,2)+1.5d0*dgm01(1,1)*dgm10(2,2)&
&   +4.5d0*gm(2,2)*d2gm(1,1)+3*gm(1,2)*d2gm(1,2)-3*gm(1,1)*d2gm(2,2))&
&   +gm(2,2)*(-9*gm(1,2)*dgm01(3,3)*dgm10(1,2)-1.5d0*gm(1,1)*dgm01(3,3)&
&   *dgm10(2,2)+gm(1,3)*(24*dgm01(2,3)*dgm10(1,2)+15*dgm01(2,2)*dgm10(1,3)&
&   +15*dgm01(1,3)*dgm10(2,2)+24*dgm01(1,2)*dgm10(2,3))-9*gm(1,2)&
&   *dgm01(1,2)*dgm10(3,3)-1.5d0*gm(1,1)*dgm01(2,2)*dgm10(3,3)+7.5d0*gm(1,3)&
&   **2*d2gm(2,2)+gm(3,3)*(-4.5d0*dgm01(2,2)*dgm10(1,1)-18*dgm01(1,2)&
&   *dgm10(1,2)-4.5d0*dgm01(1,1)*dgm10(2,2)-9*gm(1,2)*d2gm(1,2)-1.5d0*gm(1,1)&
&   *d2gm(2,2)))+gm(2,3)*(6*gm(1,2)*dgm01(2,3)*dgm10(1,2)+9*gm(1,2)&
&   *dgm01(2,2)*dgm10(1,3)+9*gm(1,2)*dgm01(1,3)*dgm10(2,2)-6*gm(1,1)&
&   *dgm01(2,3)*dgm10(2,2)+6*gm(1,2)*dgm01(1,2)*dgm10(2,3)-6*gm(1,1)&
&   *dgm01(2,2)*dgm10(2,3)+gm(2,2)*(9*dgm01(2,3)*dgm10(1,1)+24*dgm01(1,3)&
&   *dgm10(1,2)+24*dgm01(1,2)*dgm10(1,3)+9*dgm01(1,1)*dgm10(2,3)&
&   +24*gm(1,3)*d2gm(1,2))+gm(1,3)*(33*dgm01(2,2)*dgm10(1,2)+33*dgm01(1,2)&
&   *dgm10(2,2)+9*gm(1,2)*d2gm(2,2)))
   cm(9,7)=-9*gm(1,1)*gm(2,3)*dgm01(2,2)*dgm10(2,2)+gm(1,2)*(24*gm(1,3)&
&   *dgm01(2,2)*dgm10(2,2)+9*gm(2,3)*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)&
&   *dgm10(2,2)))+gm(2,2)**2*(3*dgm01(2,3)*dgm10(1,1)+12*dgm01(1,3)&
&   *dgm10(1,2)+12*dgm01(1,2)*dgm10(1,3)+3*dgm01(1,1)*dgm10(2,3)&
&   +3*gm(2,3)*d2gm(1,1)+12*gm(1,3)*d2gm(1,2))+1.5d0*gm(1,2)**2*(dgm01(2,3)&
&   *dgm10(2,2)+dgm01(2,2)*dgm10(2,3)+gm(2,3)*d2gm(2,2))+gm(2,2)&
&   *(6*gm(1,2)*dgm01(2,3)*dgm10(1,2)+12*gm(1,2)*dgm01(2,2)*dgm10(1,3)&
&   +12*gm(1,2)*dgm01(1,3)*dgm10(2,2)-4.5d0*gm(1,1)*dgm01(2,3)*dgm10(2,2)&
&   +6*gm(1,2)*dgm01(1,2)*dgm10(2,3)-4.5d0*gm(1,1)*dgm01(2,2)*dgm10(2,3)&
&   +gm(2,3)*(1.5d0*dgm01(2,2)*dgm10(1,1)+12*dgm01(1,2)*dgm10(1,2)&
&   +1.5d0*dgm01(1,1)*dgm10(2,2)+6*gm(1,2)*d2gm(1,2)-4.5d0*gm(1,1)&
&   *d2gm(2,2))+gm(1,3)*(36*dgm01(2,2)*dgm10(1,2)+36*dgm01(1,2)*dgm10(2,2)&
&   +12*gm(1,2)*d2gm(2,2)))
   cm(10,7)=((270*gm(2,3)**2*dgm01(2,3)-54*gm(2,2)*gm(3,3)*dgm01(2,3)&
&   -54*gm(2,3)*(gm(3,3)*dgm01(2,2)+gm(2,2)*dgm01(3,3)))*dgm10(1,1)&
&   +6*(90*gm(2,3)**2*dgm01(1,3)+gm(3,3)*(-18*gm(1,3)*dgm01(2,2)&
&   -36*gm(1,2)*dgm01(2,3))+gm(2,3)*(-36*gm(3,3)*dgm01(1,2)+180*gm(1,3)&
&   *dgm01(2,3)-36*gm(1,2)*dgm01(3,3))-18*gm(2,2)*(gm(3,3)*dgm01(1,3)&
&   +gm(1,3)*dgm01(3,3)))*dgm10(1,2)+3*(90*gm(1,3)**2*dgm01(2,3)&
&   +gm(3,3)*(-36*gm(1,2)*dgm01(1,3)-18*gm(1,1)*dgm01(2,3))+gm(2,3)&
&   *(-18*gm(3,3)*dgm01(1,1)+180*gm(1,3)*dgm01(1,3)-18*gm(1,1)*dgm01(3,3))&
&   -36*gm(1,3)*(gm(3,3)*dgm01(1,2)+gm(1,2)*dgm01(3,3)))*dgm10(2,2)&
&   +3*dgm01(2,2)*(90*gm(1,3)**2*dgm10(2,3)+gm(3,3)*(-36*gm(1,2)&
&   *dgm10(1,3)-18*gm(1,1)*dgm10(2,3))+gm(2,3)*(-18*gm(3,3)*dgm10(1,1)&
&   +180*gm(1,3)*dgm10(1,3)-18*gm(1,1)*dgm10(3,3))-36*gm(1,3)*(gm(3,3)&
&   *dgm10(1,2)+gm(1,2)*dgm10(3,3)))+6*dgm01(1,2)*(90*gm(2,3)**2*dgm10(1,3)&
&   +gm(3,3)*(-18*gm(1,3)*dgm10(2,2)-36*gm(1,2)*dgm10(2,3))+gm(2,3)&
&   *(-36*gm(3,3)*dgm10(1,2)+180*gm(1,3)*dgm10(2,3)-36*gm(1,2)*dgm10(3,3))&
&   -18*gm(2,2)*(gm(3,3)*dgm10(1,3)+gm(1,3)*dgm10(3,3)))+dgm01(1,1)&
&   *(270*gm(2,3)**2*dgm10(2,3)-54*gm(2,2)*gm(3,3)*dgm10(2,3)-54*gm(2,3)&
&   *(gm(3,3)*dgm10(2,2)+gm(2,2)*dgm10(3,3)))+(90*gm(2,3)**3-54*gm(2,2)&
&   *gm(2,3)*gm(3,3))*d2gm(1,1)+6*(-36*gm(1,2)*gm(2,3)*gm(3,3)+gm(1,3)&
&   *(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))*d2gm(1,2)+3*(90*gm(1,3)&
&   **2*gm(2,3)-18*(2*gm(1,2)*gm(1,3)+gm(1,1)*gm(2,3))*gm(3,3))*d2gm(2,2))&
&   /36.d0
   cm(1,8)=-7.5d0*gm(1,1)*gm(3,3)*dgm01(1,2)*dgm10(1,1)-15*gm(1,1)&
&   *gm(2,3)*dgm01(1,3)*dgm10(1,1)-7.5d0*gm(1,1)*gm(3,3)*dgm01(1,1)&
&   *dgm10(1,2)+0*gm(1,1)**2*dgm01(3,3)*dgm10(1,2)-15*gm(1,1)*gm(2,3)&
&   *dgm01(1,1)*dgm10(1,3)+0*gm(1,1)**2*dgm01(2,3)*dgm10(1,3)+0*gm(1,1)&
&   **2*dgm01(1,3)*dgm10(2,3)+0*gm(1,1)**2*dgm01(1,2)*dgm10(3,3)&
&   -3*gm(1,1)**2*gm(3,3)*d2gm(1,2)+gm(1,3)**2*(16.5d0*dgm01(1,2)&
&   *dgm10(1,1)+16.5d0*dgm01(1,1)*dgm10(1,2)+7.5d0*gm(1,2)*d2gm(1,1)&
&   +9*gm(1,1)*d2gm(1,2))-6*gm(1,1)**2*gm(2,3)*d2gm(1,3)+gm(1,3)&
&   *(gm(2,3)*(-6*dgm01(1,1)*dgm10(1,1)-3*gm(1,1)*d2gm(1,1))+gm(1,2)&
&   *(33*dgm01(1,3)*dgm10(1,1)+33*dgm01(1,1)*dgm10(1,3)+18*gm(1,1)&
&   *d2gm(1,3))+gm(1,1)*(9*dgm01(2,3)*dgm10(1,1)+36*dgm01(1,3)*dgm10(1,2)&
&   +36*dgm01(1,2)*dgm10(1,3)+9*dgm01(1,1)*dgm10(2,3)+6*gm(1,1)*d2gm(2,3)))&
&   +gm(1,2)*(gm(3,3)*(-3*dgm01(1,1)*dgm10(1,1)-1.5d0*gm(1,1)*d2gm(1,1))&
&   +gm(1,1)*(4.5d0*dgm01(3,3)*dgm10(1,1)+36*dgm01(1,3)*dgm10(1,3)&
&   +4.5d0*dgm01(1,1)*dgm10(3,3)+3*gm(1,1)*d2gm(3,3)))
   cm(2,8)=-4.5d0*gm(1,2)*gm(3,3)*dgm01(2,2)*dgm10(1,1)-24*gm(1,2)&
&   *gm(3,3)*dgm01(1,2)*dgm10(1,2)-6*gm(1,3)**2*dgm01(2,2)*dgm10(1,2)&
&   -3*gm(1,1)*gm(3,3)*dgm01(2,2)*dgm10(1,2)+24*gm(1,2)*gm(1,3)*dgm01(2,3)&
&   *dgm10(1,2)-1.5d0*gm(1,2)**2*dgm01(3,3)*dgm10(1,2)+6*gm(1,2)&
&   *gm(1,3)*dgm01(2,2)*dgm10(1,3)+9*gm(1,2)**2*dgm01(2,3)*dgm10(1,3)&
&   -4.5d0*gm(1,2)*gm(3,3)*dgm01(1,1)*dgm10(2,2)-6*gm(1,3)**2*dgm01(1,2)&
&   *dgm10(2,2)-3*gm(1,1)*gm(3,3)*dgm01(1,2)*dgm10(2,2)+6*gm(1,2)&
&   *gm(1,3)*dgm01(1,3)*dgm10(2,2)-9*gm(1,1)*gm(1,3)*dgm01(2,3)*dgm10(2,2)&
&   +7.5d0*gm(1,1)*gm(1,2)*dgm01(3,3)*dgm10(2,2)+24*gm(1,2)*gm(1,3)&
&   *dgm01(1,2)*dgm10(2,3)+9*gm(1,2)**2*dgm01(1,3)*dgm10(2,3)-9*gm(1,1)&
&   *gm(1,3)*dgm01(2,2)*dgm10(2,3)+48*gm(1,1)*gm(1,2)*dgm01(2,3)&
&   *dgm10(2,3)-1.5d0*gm(1,2)**2*dgm01(1,2)*dgm10(3,3)+7.5d0*gm(1,1)&
&   *gm(1,2)*dgm01(2,2)*dgm10(3,3)-6*gm(1,2)**2*gm(3,3)*d2gm(1,2)&
&   +gm(2,3)**2*(16.5d0*dgm01(1,2)*dgm10(1,1)+16.5d0*dgm01(1,1)*dgm10(1,2)&
&   +1.5d0*gm(1,2)*d2gm(1,1)+15*gm(1,1)*d2gm(1,2))+3*gm(1,2)**2*gm(1,3)&
&   *d2gm(2,3)+gm(2,3)*(27*gm(1,2)*dgm01(2,3)*dgm10(1,1)+30*gm(1,2)&
&   *dgm01(1,3)*dgm10(1,2)+54*gm(1,1)*dgm01(2,3)*dgm10(1,2)+30*gm(1,2)&
&   *dgm01(1,2)*dgm10(1,3)+24*gm(1,1)*dgm01(2,2)*dgm10(1,3)+24*gm(1,1)&
&   *dgm01(1,3)*dgm10(2,2)+27*gm(1,2)*dgm01(1,1)*dgm10(2,3)+54*gm(1,1)&
&   *dgm01(1,2)*dgm10(2,3)+gm(1,3)*(12*dgm01(2,2)*dgm10(1,1)+36*dgm01(1,2)&
&   *dgm10(1,2)+12*dgm01(1,1)*dgm10(2,2)+18*gm(1,2)*d2gm(1,2))+6*gm(1,2)&
&   **2*d2gm(1,3)+gm(2,2)*(36*dgm01(1,3)*dgm10(1,1)+36*dgm01(1,1)&
&   *dgm10(1,3)+12*gm(1,3)*d2gm(1,1)+24*gm(1,1)*d2gm(1,3))+24*gm(1,1)&
&   *gm(1,2)*d2gm(2,3))+1.5d0*gm(1,2)**3*d2gm(3,3)+gm(2,2)*(3*gm(1,2)&
&   *dgm01(3,3)*dgm10(1,1)+4.5d0*gm(1,1)*dgm01(3,3)*dgm10(1,2)+12*gm(1,2)&
&   *dgm01(1,3)*dgm10(1,3)+15*gm(1,1)*dgm01(2,3)*dgm10(1,3)+15*gm(1,1)&
&   *dgm01(1,3)*dgm10(2,3)+3*gm(1,2)*dgm01(1,1)*dgm10(3,3)+4.5d0*gm(1,1)&
&   *dgm01(1,2)*dgm10(3,3)-6*gm(1,3)**2*d2gm(1,2)+gm(3,3)*(-7.5d0*dgm01(1,2)&
&   *dgm10(1,1)-7.5d0*dgm01(1,1)*dgm10(1,2)-4.5d0*gm(1,2)*d2gm(1,1)&
&   -3*gm(1,1)*d2gm(1,2))+gm(1,3)*(3*dgm01(2,3)*dgm10(1,1)-6*dgm01(1,3)&
&   *dgm10(1,2)-6*dgm01(1,2)*dgm10(1,3)+3*dgm01(1,1)*dgm10(2,3)+6*gm(1,2)&
&   *d2gm(1,3)-9*gm(1,1)*d2gm(2,3))+7.5d0*gm(1,1)*gm(1,2)*d2gm(3,3))
   cm(3,8)=gm(1,1)*(-9*gm(1,2)*dgm01(3,3)*dgm10(3,3)+36*gm(2,3)&
&   *(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3)))+gm(3,3)**2*(18*dgm01(1,2)&
&   *dgm10(1,1)+18*dgm01(1,1)*dgm10(1,2)+6*gm(1,2)*d2gm(1,1)+12*gm(1,1)&
&   *d2gm(1,2))+3*gm(1,3)**3*d2gm(2,3)+gm(1,3)**2*(7.5d0*dgm01(3,3)&
&   *dgm10(1,2)+15*dgm01(2,3)*dgm10(1,3)+15*dgm01(1,3)*dgm10(2,3)&
&   +7.5d0*dgm01(1,2)*dgm10(3,3)+6*gm(2,3)*d2gm(1,3)+1.5d0*gm(1,2)&
&   *d2gm(3,3))+gm(3,3)*(7.5d0*gm(1,2)*dgm01(3,3)*dgm10(1,1)+19.5d0*gm(1,1)&
&   *dgm01(3,3)*dgm10(1,2)+12*gm(1,2)*dgm01(1,3)*dgm10(1,3)+39*gm(1,1)&
&   *dgm01(2,3)*dgm10(1,3)+39*gm(1,1)*dgm01(1,3)*dgm10(2,3)+7.5d0*gm(1,2)&
&   *dgm01(1,1)*dgm10(3,3)+19.5d0*gm(1,1)*dgm01(1,2)*dgm10(3,3)+6*gm(1,3)&
&   **2*d2gm(1,2)+gm(2,3)*(27*dgm01(1,3)*dgm10(1,1)+27*dgm01(1,1)&
&   *dgm10(1,3)+3*gm(1,3)*d2gm(1,1)+24*gm(1,1)*d2gm(1,3))+gm(1,3)&
&   *(18*dgm01(2,3)*dgm10(1,1)+18*dgm01(1,3)*dgm10(1,2)+18*dgm01(1,2)&
&   *dgm10(1,3)+18*dgm01(1,1)*dgm10(2,3)+6*gm(1,2)*d2gm(1,3)+15*gm(1,1)&
&   *d2gm(2,3))-4.5d0*gm(1,1)*gm(1,2)*d2gm(3,3))+gm(1,3)*(9*gm(1,2)&
&   *(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+27*gm(1,1)*(dgm01(3,3)&
&   *dgm10(2,3)+dgm01(2,3)*dgm10(3,3))+gm(2,3)*(15*dgm01(3,3)*dgm10(1,1)&
&   +24*dgm01(1,3)*dgm10(1,3)+15*dgm01(1,1)*dgm10(3,3)+12*gm(1,1)&
&   *d2gm(3,3)))
   cm(4,8)=12*gm(1,3)*gm(3,3)*dgm01(2,2)*dgm10(1,1)+27*gm(1,2)&
&   *gm(3,3)*dgm01(2,3)*dgm10(1,1)+12*gm(1,3)*gm(3,3)*dgm01(1,2)&
&   *dgm10(1,2)+42*gm(1,2)*gm(3,3)*dgm01(1,3)*dgm10(1,2)+12*gm(1,3)&
&   **2*dgm01(2,3)*dgm10(1,2)+48*gm(1,1)*gm(3,3)*dgm01(2,3)*dgm10(1,2)&
&   +12*gm(1,2)*gm(1,3)*dgm01(3,3)*dgm10(1,2)+42*gm(1,2)*gm(3,3)&
&   *dgm01(1,2)*dgm10(1,3)+18*gm(1,3)**2*dgm01(2,2)*dgm10(1,3)+30*gm(1,1)&
&   *gm(3,3)*dgm01(2,2)*dgm10(1,3)+0*gm(1,2)*gm(1,3)*dgm01(2,3)*dgm10(1,3)&
&   +21*gm(1,2)**2*dgm01(3,3)*dgm10(1,3)+12*gm(1,3)*gm(3,3)*dgm01(1,1)&
&   *dgm10(2,2)+18*gm(1,3)**2*dgm01(1,3)*dgm10(2,2)+30*gm(1,1)*gm(3,3)&
&   *dgm01(1,3)*dgm10(2,2)+12*gm(1,1)*gm(1,3)*dgm01(3,3)*dgm10(2,2)&
&   +27*gm(1,2)*gm(3,3)*dgm01(1,1)*dgm10(2,3)+12*gm(1,3)**2*dgm01(1,2)&
&   *dgm10(2,3)+48*gm(1,1)*gm(3,3)*dgm01(1,2)*dgm10(2,3)+0*gm(1,2)&
&   *gm(1,3)*dgm01(1,3)*dgm10(2,3)+12*gm(1,1)*gm(1,3)*dgm01(2,3)&
&   *dgm10(2,3)+27*gm(1,1)*gm(1,2)*dgm01(3,3)*dgm10(2,3)+12*gm(1,2)&
&   *gm(1,3)*dgm01(1,2)*dgm10(3,3)+21*gm(1,2)**2*dgm01(1,3)*dgm10(3,3)&
&   +12*gm(1,1)*gm(1,3)*dgm01(2,2)*dgm10(3,3)+27*gm(1,1)*gm(1,2)&
&   *dgm01(2,3)*dgm10(3,3)+6*gm(1,2)*gm(1,3)*gm(3,3)*d2gm(1,2)+18*gm(1,2)&
&   **2*gm(3,3)*d2gm(1,3)+gm(2,3)**2*(21*dgm01(1,3)*dgm10(1,1)+21*dgm01(1,1)&
&   *dgm10(1,3)+3*gm(1,3)*d2gm(1,1)+18*gm(1,1)*d2gm(1,3))+6*gm(1,2)&
&   *gm(1,3)**2*d2gm(2,3)+24*gm(1,1)*gm(1,2)*gm(3,3)*d2gm(2,3)+3*gm(1,2)&
&   **2*gm(1,3)*d2gm(3,3)+gm(2,3)*(6*gm(1,2)*dgm01(3,3)*dgm10(1,1)&
&   +27*gm(1,1)*dgm01(3,3)*dgm10(1,2)-24*gm(1,2)*dgm01(1,3)*dgm10(1,3)&
&   +42*gm(1,1)*dgm01(2,3)*dgm10(1,3)+42*gm(1,1)*dgm01(1,3)*dgm10(2,3)&
&   +6*gm(1,2)*dgm01(1,1)*dgm10(3,3)+27*gm(1,1)*dgm01(1,2)*dgm10(3,3)&
&   +6*gm(1,3)**2*d2gm(1,2)+gm(3,3)*(27*dgm01(1,2)*dgm10(1,1)+27*dgm01(1,1)&
&   *dgm10(1,2)+3*gm(1,2)*d2gm(1,1)+24*gm(1,1)*d2gm(1,2))+gm(1,3)&
&   *(12*dgm01(2,3)*dgm10(1,1)+0*dgm01(1,3)*dgm10(1,2)+0*dgm01(1,2)&
&   *dgm10(1,3)+12*dgm01(1,1)*dgm10(2,3)-12*gm(1,2)*d2gm(1,3)+6*gm(1,1)&
&   *d2gm(2,3))+3*gm(1,1)*gm(1,2)*d2gm(3,3))+gm(2,2)*(42*gm(1,1)&
&   *(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+18*gm(1,3)**2*d2gm(1,3)&
&   +gm(3,3)*(42*dgm01(1,3)*dgm10(1,1)+42*dgm01(1,1)*dgm10(1,3)+12*gm(1,3)&
&   *d2gm(1,1)+30*gm(1,1)*d2gm(1,3))+gm(1,3)*(24*dgm01(3,3)*dgm10(1,1)&
&   +72*dgm01(1,3)*dgm10(1,3)+24*dgm01(1,1)*dgm10(3,3)+12*gm(1,1)&
&   *d2gm(3,3)))
   cm(5,8)=36*gm(1,2)*gm(3,3)*dgm01(1,3)*dgm10(1,1)+19.5d0*gm(1,1)&
&   *gm(3,3)*dgm01(2,3)*dgm10(1,1)+39*gm(1,1)*gm(3,3)*dgm01(1,3)&
&   *dgm10(1,2)+36*gm(1,2)*gm(3,3)*dgm01(1,1)*dgm10(1,3)+39*gm(1,1)&
&   *gm(3,3)*dgm01(1,2)*dgm10(1,3)+27*gm(1,1)*gm(1,2)*dgm01(3,3)&
&   *dgm10(1,3)+19.5d0*gm(1,1)*gm(3,3)*dgm01(1,1)*dgm10(2,3)+18*gm(1,1)&
&   **2*dgm01(3,3)*dgm10(2,3)+27*gm(1,1)*gm(1,2)*dgm01(1,3)*dgm10(3,3)&
&   +18*gm(1,1)**2*dgm01(2,3)*dgm10(3,3)+3*gm(1,3)**3*d2gm(1,2)+24*gm(1,1)&
&   *gm(1,2)*gm(3,3)*d2gm(1,3)+12*gm(1,1)**2*gm(3,3)*d2gm(2,3)+gm(1,3)&
&   **2*(7.5d0*dgm01(2,3)*dgm10(1,1)+15*dgm01(1,3)*dgm10(1,2)+15*dgm01(1,2)&
&   *dgm10(1,3)+7.5d0*dgm01(1,1)*dgm10(2,3)+6*gm(1,2)*d2gm(1,3)+6*gm(1,1)&
&   *d2gm(2,3))+gm(1,3)*(18*gm(1,1)*(dgm01(3,3)*dgm10(1,2)+dgm01(2,3)&
&   *dgm10(1,3)+dgm01(1,3)*dgm10(2,3)+dgm01(1,2)*dgm10(3,3))+gm(3,3)&
&   *(27*dgm01(1,2)*dgm10(1,1)+27*dgm01(1,1)*dgm10(1,2)+12*gm(1,2)&
&   *d2gm(1,1)+15*gm(1,1)*d2gm(1,2))+gm(1,2)*(15*dgm01(3,3)*dgm10(1,1)&
&   +24*dgm01(1,3)*dgm10(1,3)+15*dgm01(1,1)*dgm10(3,3)+3*gm(1,1)&
&   *d2gm(3,3)))+gm(2,3)*(1.5d0*gm(1,3)**2*d2gm(1,1)+gm(3,3)*(-9*dgm01(1,1)&
&   *dgm10(1,1)-4.5d0*gm(1,1)*d2gm(1,1))+gm(1,3)*(9*dgm01(1,3)*dgm10(1,1)&
&   +9*dgm01(1,1)*dgm10(1,3)+6*gm(1,1)*d2gm(1,3))+gm(1,1)*(7.5d0*dgm01(3,3)&
&   *dgm10(1,1)+12*dgm01(1,3)*dgm10(1,3)+7.5d0*dgm01(1,1)*dgm10(3,3)&
&   +6*gm(1,1)*d2gm(3,3)))
   cm(6,8)=-15*gm(1,2)*gm(3,3)*dgm01(1,2)*dgm10(1,1)+7.5d0*gm(1,3)&
&   **2*dgm01(2,2)*dgm10(1,1)-1.5d0*gm(1,1)*gm(3,3)*dgm01(2,2)*dgm10(1,1)&
&   +15*gm(1,2)*gm(1,3)*dgm01(2,3)*dgm10(1,1)+0*gm(1,2)**2*dgm01(3,3)&
&   *dgm10(1,1)-15*gm(1,2)*gm(3,3)*dgm01(1,1)*dgm10(1,2)+6*gm(1,3)&
&   **2*dgm01(1,2)*dgm10(1,2)-18*gm(1,1)*gm(3,3)*dgm01(1,2)*dgm10(1,2)&
&   +18*gm(1,2)*gm(1,3)*dgm01(1,3)*dgm10(1,2)+30*gm(1,1)*gm(1,3)&
&   *dgm01(2,3)*dgm10(1,2)-3*gm(1,1)*gm(1,2)*dgm01(3,3)*dgm10(1,2)&
&   +18*gm(1,2)*gm(1,3)*dgm01(1,2)*dgm10(1,3)+12*gm(1,2)**2*dgm01(1,3)&
&   *dgm10(1,3)+24*gm(1,1)*gm(1,3)*dgm01(2,2)*dgm10(1,3)+12*gm(1,1)&
&   *gm(1,2)*dgm01(2,3)*dgm10(1,3)+7.5d0*gm(1,3)**2*dgm01(1,1)*dgm10(2,2)&
&   -1.5d0*gm(1,1)*gm(3,3)*dgm01(1,1)*dgm10(2,2)+24*gm(1,1)*gm(1,3)&
&   *dgm01(1,3)*dgm10(2,2)+6*gm(1,1)**2*dgm01(3,3)*dgm10(2,2)+15*gm(1,2)&
&   *gm(1,3)*dgm01(1,1)*dgm10(2,3)+30*gm(1,1)*gm(1,3)*dgm01(1,2)&
&   *dgm10(2,3)+12*gm(1,1)*gm(1,2)*dgm01(1,3)*dgm10(2,3)+24*gm(1,1)&
&   **2*dgm01(2,3)*dgm10(2,3)+0*gm(1,2)**2*dgm01(1,1)*dgm10(3,3)&
&   -3*gm(1,1)*gm(1,2)*dgm01(1,2)*dgm10(3,3)+6*gm(1,1)**2*dgm01(2,2)&
&   *dgm10(3,3)-3*gm(1,2)**2*gm(3,3)*d2gm(1,1)+gm(2,3)**2*(-6*dgm01(1,1)&
&   *dgm10(1,1)-3*gm(1,1)*d2gm(1,1))+3*gm(1,2)*gm(1,3)**2*d2gm(1,2)&
&   -9*gm(1,1)*gm(1,2)*gm(3,3)*d2gm(1,2)+6*gm(1,2)**2*gm(1,3)*d2gm(1,3)&
&   +6*gm(1,1)*gm(1,2)*gm(1,3)*d2gm(2,3)+gm(2,3)*(gm(1,3)*(33*dgm01(1,2)&
&   *dgm10(1,1)+33*dgm01(1,1)*dgm10(1,2)+9*gm(1,2)*d2gm(1,1)+24*gm(1,1)&
&   *d2gm(1,2))+gm(1,2)*(15*dgm01(1,3)*dgm10(1,1)+15*dgm01(1,1)*dgm10(1,3)&
&   +6*gm(1,1)*d2gm(1,3))+gm(1,1)*(18*dgm01(2,3)*dgm10(1,1)+30*dgm01(1,3)&
&   *dgm10(1,2)+30*dgm01(1,2)*dgm10(1,3)+18*dgm01(1,1)*dgm10(2,3)&
&   +12*gm(1,1)*d2gm(2,3)))+3*gm(1,1)*gm(1,2)**2*d2gm(3,3)+gm(2,2)&
&   *(7.5d0*gm(1,3)**2*d2gm(1,1)+gm(3,3)*(-3*dgm01(1,1)*dgm10(1,1)&
&   -1.5d0*gm(1,1)*d2gm(1,1))+gm(1,3)*(39*dgm01(1,3)*dgm10(1,1)+39*dgm01(1,1)&
&   *dgm10(1,3)+24*gm(1,1)*d2gm(1,3))+gm(1,1)*(10.5d0*dgm01(3,3)&
&   *dgm10(1,1)+48*dgm01(1,3)*dgm10(1,3)+10.5d0*dgm01(1,1)*dgm10(3,3)&
&   +6*gm(1,1)*d2gm(3,3)))
   cm(7,8)=gm(1,2)*(-3*gm(3,3)*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)&
&   *dgm10(2,2))-6*gm(1,3)*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))&
&   +gm(1,2)*(4.5d0*dgm01(3,3)*dgm10(2,2)+30*dgm01(2,3)*dgm10(2,3)&
&   +4.5d0*dgm01(2,2)*dgm10(3,3)))+gm(2,3)**2*(4.5d0*dgm01(2,2)*dgm10(1,1)&
&   +30*dgm01(1,2)*dgm10(1,2)+4.5d0*dgm01(1,1)*dgm10(2,2)+4.5d0*gm(2,2)&
&   *d2gm(1,1)+15*gm(1,2)*d2gm(1,2))+gm(2,3)*(60*gm(1,2)*dgm01(2,3)&
&   *dgm10(1,2)+18*gm(1,2)*dgm01(2,2)*dgm10(1,3)+18*gm(1,2)*dgm01(1,3)&
&   *dgm10(2,2)-3*gm(1,1)*dgm01(2,3)*dgm10(2,2)-6*gm(1,3)*(dgm01(2,2)&
&   *dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+60*gm(1,2)*dgm01(1,2)*dgm10(2,3)&
&   -3*gm(1,1)*dgm01(2,2)*dgm10(2,3)+15*gm(1,2)**2*d2gm(2,3)+gm(2,2)&
&   *(6*dgm01(2,3)*dgm10(1,1)+12*dgm01(1,3)*dgm10(1,2)+12*dgm01(1,2)&
&   *dgm10(1,3)+6*dgm01(1,1)*dgm10(2,3)-6*gm(1,3)*d2gm(1,2)+18*gm(1,2)&
&   *d2gm(1,3)-3*gm(1,1)*d2gm(2,3)))+gm(2,2)**2*(-3*dgm01(3,3)*dgm10(1,1)&
&   -12*dgm01(1,3)*dgm10(1,3)-3*dgm01(1,1)*dgm10(3,3)-1.5d0*gm(3,3)&
&   *d2gm(1,1)-6*gm(1,3)*d2gm(1,3)-1.5d0*gm(1,1)*d2gm(3,3))+gm(2,2)&
&   *(6*gm(1,2)*dgm01(3,3)*dgm10(1,2)+12*gm(1,2)*dgm01(2,3)*dgm10(1,3)&
&   -3*gm(1,1)*dgm01(3,3)*dgm10(2,2)+12*gm(1,2)*dgm01(1,3)*dgm10(2,3)&
&   -6*gm(1,1)*dgm01(2,3)*dgm10(2,3)+6*gm(1,2)*dgm01(1,2)*dgm10(3,3)&
&   -3*gm(1,1)*dgm01(2,2)*dgm10(3,3)+gm(3,3)*(-3*dgm01(2,2)*dgm10(1,1)&
&   -6*dgm01(1,2)*dgm10(1,2)-3*dgm01(1,1)*dgm10(2,2)-3*gm(1,2)*d2gm(1,2))&
&   +gm(1,3)*(-12*dgm01(2,3)*dgm10(1,2)-12*dgm01(2,2)*dgm10(1,3)&
&   -12*dgm01(1,3)*dgm10(2,2)-12*dgm01(1,2)*dgm10(2,3)-6*gm(1,2)&
&   *d2gm(2,3))+4.5d0*gm(1,2)**2*d2gm(3,3))
   cm(8,8)=39*gm(1,3)*gm(2,2)*dgm01(3,3)*dgm10(1,3)+7.5d0*gm(1,3)&
&   **2*dgm01(3,3)*dgm10(2,2)+6*gm(1,3)**2*dgm01(2,3)*dgm10(2,3)&
&   +33*gm(1,2)*gm(1,3)*dgm01(3,3)*dgm10(2,3)+39*gm(1,3)*gm(2,2)&
&   *dgm01(1,3)*dgm10(3,3)+7.5d0*gm(1,3)**2*dgm01(2,2)*dgm10(3,3)&
&   +33*gm(1,2)*gm(1,3)*dgm01(2,3)*dgm10(3,3)-6*gm(1,2)**2*dgm01(3,3)&
&   *dgm10(3,3)-3*gm(1,1)*gm(2,2)*dgm01(3,3)*dgm10(3,3)+gm(3,3)**2*(6*dgm01(2,2)&
&   *dgm10(1,1)+24*dgm01(1,2)*dgm10(1,2)+6*dgm01(1,1)*dgm10(2,2)&
&   +6*gm(2,2)*d2gm(1,1)+12*gm(1,2)*d2gm(1,2))+7.5d0*gm(1,3)**2*gm(2,2)&
&   *d2gm(3,3)+gm(2,3)**2*(0*dgm01(3,3)*dgm10(1,1)+12*dgm01(1,3)&
&   *dgm10(1,3)+0*dgm01(1,1)*dgm10(3,3)+6*gm(1,3)*d2gm(1,3)-3*gm(1,1)&
&   *d2gm(3,3))+gm(3,3)*(30*gm(1,3)*dgm01(2,3)*dgm10(1,2)+18*gm(1,2)&
&   *dgm01(3,3)*dgm10(1,2)+24*gm(1,3)*dgm01(2,2)*dgm10(1,3)+30*gm(1,2)&
&   *dgm01(2,3)*dgm10(1,3)+24*gm(1,3)*dgm01(1,3)*dgm10(2,2)-1.5d0*gm(1,1)&
&   *dgm01(3,3)*dgm10(2,2)+30*gm(1,3)*dgm01(1,2)*dgm10(2,3)+30*gm(1,2)&
&   *dgm01(1,3)*dgm10(2,3)-18*gm(1,1)*dgm01(2,3)*dgm10(2,3)+18*gm(1,2)&
&   *dgm01(1,2)*dgm10(3,3)-1.5d0*gm(1,1)*dgm01(2,2)*dgm10(3,3)+3*gm(2,3)&
&   **2*d2gm(1,1)+24*gm(1,2)*gm(1,3)*d2gm(2,3)+gm(2,3)*(-3*dgm01(2,3)&
&   *dgm10(1,1)+12*dgm01(1,3)*dgm10(1,2)+12*dgm01(1,2)*dgm10(1,3)&
&   -3*dgm01(1,1)*dgm10(2,3)+6*gm(1,3)*d2gm(1,2)+6*gm(1,2)*d2gm(1,3)&
&   -9*gm(1,1)*d2gm(2,3))-3*gm(1,2)**2*d2gm(3,3)+gm(2,2)*(10.5d0*dgm01(3,3)&
&   *dgm10(1,1)+48*dgm01(1,3)*dgm10(1,3)+10.5d0*dgm01(1,1)*dgm10(3,3)&
&   +24*gm(1,3)*d2gm(1,3)-1.5d0*gm(1,1)*d2gm(3,3)))+gm(2,3)*(15*gm(1,2)&
&   *(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))-15*gm(1,1)*(dgm01(3,3)&
&   *dgm10(2,3)+dgm01(2,3)*dgm10(3,3))+3*gm(1,3)**2*d2gm(2,3)+gm(1,3)&
&   *(15*dgm01(3,3)*dgm10(1,2)+18*dgm01(2,3)*dgm10(1,3)+18*dgm01(1,3)&
&   *dgm10(2,3)+15*dgm01(1,2)*dgm10(3,3)+9*gm(1,2)*d2gm(3,3)))
   cm(9,8)=-9*gm(1,3)*gm(3,3)*dgm01(2,2)*dgm10(1,2)+54*gm(1,2)&
&   *gm(3,3)*dgm01(2,3)*dgm10(1,2)+24*gm(1,2)*gm(3,3)*dgm01(2,2)&
&   *dgm10(1,3)-9*gm(1,3)*gm(3,3)*dgm01(1,2)*dgm10(2,2)+24*gm(1,2)&
&   *gm(3,3)*dgm01(1,3)*dgm10(2,2)-6*gm(1,3)**2*dgm01(2,3)*dgm10(2,2)&
&   -3*gm(1,1)*gm(3,3)*dgm01(2,3)*dgm10(2,2)+12*gm(1,2)*gm(1,3)*dgm01(3,3)&
&   *dgm10(2,2)+54*gm(1,2)*gm(3,3)*dgm01(1,2)*dgm10(2,3)-6*gm(1,3)&
&   **2*dgm01(2,2)*dgm10(2,3)-3*gm(1,1)*gm(3,3)*dgm01(2,2)*dgm10(2,3)&
&   +36*gm(1,2)*gm(1,3)*dgm01(2,3)*dgm10(2,3)+16.5d0*gm(1,2)**2*dgm01(3,3)&
&   *dgm10(2,3)+12*gm(1,2)*gm(1,3)*dgm01(2,2)*dgm10(3,3)+16.5d0*gm(1,2)&
&   **2*dgm01(2,3)*dgm10(3,3)+1.5d0*gm(2,3)**3*d2gm(1,1)+15*gm(1,2)&
&   **2*gm(3,3)*d2gm(2,3)+gm(2,3)**2*(-1.5d0*dgm01(2,3)*dgm10(1,1)&
&   +9*dgm01(1,3)*dgm10(1,2)+9*dgm01(1,2)*dgm10(1,3)-1.5d0*dgm01(1,1)&
&   *dgm10(2,3)+3*gm(1,3)*d2gm(1,2)+6*gm(1,2)*d2gm(1,3)-6*gm(1,1)&
&   *d2gm(2,3))+gm(2,3)*(24*gm(1,3)*dgm01(2,3)*dgm10(1,2)+27*gm(1,2)&
&   *dgm01(3,3)*dgm10(1,2)+6*gm(1,3)*dgm01(2,2)*dgm10(1,3)+30*gm(1,2)&
&   *dgm01(2,3)*dgm10(1,3)+6*gm(1,3)*dgm01(1,3)*dgm10(2,2)-4.5d0*gm(1,1)&
&   *dgm01(3,3)*dgm10(2,2)+24*gm(1,3)*dgm01(1,2)*dgm10(2,3)+30*gm(1,2)&
&   *dgm01(1,3)*dgm10(2,3)-24*gm(1,1)*dgm01(2,3)*dgm10(2,3)+27*gm(1,2)&
&   *dgm01(1,2)*dgm10(3,3)-4.5d0*gm(1,1)*dgm01(2,2)*dgm10(3,3)+gm(3,3)&
&   *(7.5d0*dgm01(2,2)*dgm10(1,1)+48*dgm01(1,2)*dgm10(1,2)+7.5d0*dgm01(1,1)&
&   *dgm10(2,2)+7.5d0*gm(2,2)*d2gm(1,1)+24*gm(1,2)*d2gm(1,2))+18*gm(1,2)&
&   *gm(1,3)*d2gm(2,3)+1.5d0*gm(1,2)**2*d2gm(3,3)+gm(2,2)*(3*dgm01(3,3)&
&   *dgm10(1,1)+12*dgm01(1,3)*dgm10(1,3)+3*dgm01(1,1)*dgm10(3,3)&
&   +6*gm(1,3)*d2gm(1,3)-4.5d0*gm(1,1)*d2gm(3,3)))+gm(2,2)*(36*gm(1,2)&
&   *dgm01(3,3)*dgm10(1,3)-7.5d0*gm(1,1)*dgm01(3,3)*dgm10(2,3)+36*gm(1,2)&
&   *dgm01(1,3)*dgm10(3,3)-7.5d0*gm(1,1)*dgm01(2,3)*dgm10(3,3)-6*gm(1,3)&
&   **2*d2gm(2,3)+gm(3,3)*(4.5d0*dgm01(2,3)*dgm10(1,1)+15*dgm01(1,3)&
&   *dgm10(1,2)+15*dgm01(1,2)*dgm10(1,3)+4.5d0*dgm01(1,1)*dgm10(2,3)&
&   -9*gm(1,3)*d2gm(1,2)+24*gm(1,2)*d2gm(1,3)-3*gm(1,1)*d2gm(2,3))&
&   +gm(1,3)*(3*dgm01(3,3)*dgm10(1,2)-6*dgm01(2,3)*dgm10(1,3)-6*dgm01(1,3)&
&   *dgm10(2,3)+3*dgm01(1,2)*dgm10(3,3)+12*gm(1,2)*d2gm(3,3)))
   cm(10,8)=-3*gm(1,1)*gm(2,3)*dgm01(3,3)*dgm10(3,3)+gm(1,3)*(-6*gm(1,2)&
&   *dgm01(3,3)*dgm10(3,3)+33*gm(2,3)*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)&
&   *dgm10(3,3)))+gm(3,3)**2*(0*dgm01(2,3)*dgm10(1,1)+0*dgm01(1,3)&
&   *dgm10(1,2)+0*dgm01(1,2)*dgm10(1,3)+0*dgm01(1,1)*dgm10(2,3)+3*gm(2,3)&
&   *d2gm(1,1)+6*gm(1,3)*d2gm(1,2)-6*gm(1,2)*d2gm(1,3)-3*gm(1,1)&
&   *d2gm(2,3))+gm(1,3)**2*(16.5d0*dgm01(3,3)*dgm10(2,3)+16.5d0*dgm01(2,3)&
&   *dgm10(3,3)+7.5d0*gm(2,3)*d2gm(3,3))+gm(3,3)*(-15*gm(1,2)*dgm01(3,3)&
&   *dgm10(1,3)-7.5d0*gm(1,1)*dgm01(3,3)*dgm10(2,3)-15*gm(1,2)*dgm01(1,3)&
&   *dgm10(3,3)-7.5d0*gm(1,1)*dgm01(2,3)*dgm10(3,3)+9*gm(1,3)**2*d2gm(2,3)&
&   +gm(2,3)*(4.5d0*dgm01(3,3)*dgm10(1,1)+36*dgm01(1,3)*dgm10(1,3)&
&   +4.5d0*dgm01(1,1)*dgm10(3,3)+18*gm(1,3)*d2gm(1,3)-1.5d0*gm(1,1)&
&   *d2gm(3,3))+gm(1,3)*(9*dgm01(3,3)*dgm10(1,2)+36*dgm01(2,3)*dgm10(1,3)&
&   +36*dgm01(1,3)*dgm10(2,3)+9*dgm01(1,2)*dgm10(3,3)-3*gm(1,2)*d2gm(3,3)))
   cm(1,9)=-15*gm(1,1)*gm(2,3)*dgm01(1,2)*dgm10(1,1)-7.5d0*gm(1,1)&
&   *gm(2,2)*dgm01(1,3)*dgm10(1,1)-15*gm(1,1)*gm(2,3)*dgm01(1,1)&
&   *dgm10(1,2)+0*gm(1,1)**2*dgm01(2,3)*dgm10(1,2)-7.5d0*gm(1,1)&
&   *gm(2,2)*dgm01(1,1)*dgm10(1,3)+0*gm(1,1)**2*dgm01(2,2)*dgm10(1,3)&
&   +0*gm(1,1)**2*dgm01(1,3)*dgm10(2,2)+0*gm(1,1)**2*dgm01(1,2)*dgm10(2,3)&
&   -6*gm(1,1)**2*gm(2,3)*d2gm(1,2)-3*gm(1,1)**2*gm(2,2)*d2gm(1,3)&
&   +gm(1,2)**2*(16.5d0*dgm01(1,3)*dgm10(1,1)+16.5d0*dgm01(1,1)*dgm10(1,3)&
&   +9*gm(1,1)*d2gm(1,3))+gm(1,3)*(7.5d0*gm(1,2)**2*d2gm(1,1)+gm(2,2)&
&   *(-3*dgm01(1,1)*dgm10(1,1)-1.5d0*gm(1,1)*d2gm(1,1))+gm(1,2)*(33*dgm01(1,2)&
&   *dgm10(1,1)+33*dgm01(1,1)*dgm10(1,2)+18*gm(1,1)*d2gm(1,2))+gm(1,1)&
&   *(4.5d0*dgm01(2,2)*dgm10(1,1)+36*dgm01(1,2)*dgm10(1,2)+4.5d0*dgm01(1,1)&
&   *dgm10(2,2)+3*gm(1,1)*d2gm(2,2)))+gm(1,2)*(gm(2,3)*(-6*dgm01(1,1)&
&   *dgm10(1,1)-3*gm(1,1)*d2gm(1,1))+gm(1,1)*(9*dgm01(2,3)*dgm10(1,1)&
&   +36*dgm01(1,3)*dgm10(1,2)+36*dgm01(1,2)*dgm10(1,3)+9*dgm01(1,1)&
&   *dgm10(2,3)+6*gm(1,1)*d2gm(2,3)))
   cm(2,9)=gm(1,1)*(-9*gm(1,3)*dgm01(2,2)*dgm10(2,2)+36*gm(2,3)&
&   *(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2)))+gm(2,2)**2*(18*dgm01(1,3)&
&   *dgm10(1,1)+18*dgm01(1,1)*dgm10(1,3)+6*gm(1,3)*d2gm(1,1)+12*gm(1,1)&
&   *d2gm(1,3))+gm(1,2)**2*(15*dgm01(2,3)*dgm10(1,2)+7.5d0*dgm01(2,2)&
&   *dgm10(1,3)+7.5d0*dgm01(1,3)*dgm10(2,2)+15*dgm01(1,2)*dgm10(2,3)&
&   +6*gm(2,3)*d2gm(1,2)+1.5d0*gm(1,3)*d2gm(2,2))+gm(1,2)*(9*gm(1,3)&
&   *(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+27*gm(1,1)*(dgm01(2,3)&
&   *dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+gm(2,3)*(15*dgm01(2,2)*dgm10(1,1)&
&   +24*dgm01(1,2)*dgm10(1,2)+15*dgm01(1,1)*dgm10(2,2)+12*gm(1,1)&
&   *d2gm(2,2)))+3*gm(1,2)**3*d2gm(2,3)+gm(2,2)*(18*gm(1,2)*dgm01(2,3)&
&   *dgm10(1,1)+18*gm(1,2)*dgm01(1,3)*dgm10(1,2)+39*gm(1,1)*dgm01(2,3)&
&   *dgm10(1,2)+18*gm(1,2)*dgm01(1,2)*dgm10(1,3)+19.5d0*gm(1,1)*dgm01(2,2)&
&   *dgm10(1,3)+19.5d0*gm(1,1)*dgm01(1,3)*dgm10(2,2)+18*gm(1,2)*dgm01(1,1)&
&   *dgm10(2,3)+39*gm(1,1)*dgm01(1,2)*dgm10(2,3)+gm(2,3)*(27*dgm01(1,2)&
&   *dgm10(1,1)+27*dgm01(1,1)*dgm10(1,2)+3*gm(1,2)*d2gm(1,1)+24*gm(1,1)&
&   *d2gm(1,2))+6*gm(1,2)**2*d2gm(1,3)+gm(1,3)*(7.5d0*dgm01(2,2)&
&   *dgm10(1,1)+12*dgm01(1,2)*dgm10(1,2)+7.5d0*dgm01(1,1)*dgm10(2,2)&
&   +6*gm(1,2)*d2gm(1,2)-4.5d0*gm(1,1)*d2gm(2,2))+15*gm(1,1)*gm(1,2)&
&   *d2gm(2,3))
   cm(3,9)=3*gm(1,3)*gm(3,3)*dgm01(2,2)*dgm10(1,1)+3*gm(1,2)*gm(3,3)&
&   *dgm01(2,3)*dgm10(1,1)+12*gm(1,3)*gm(3,3)*dgm01(1,2)*dgm10(1,2)&
&   -6*gm(1,2)*gm(3,3)*dgm01(1,3)*dgm10(1,2)+9*gm(1,3)**2*dgm01(2,3)&
&   *dgm10(1,2)+15*gm(1,1)*gm(3,3)*dgm01(2,3)*dgm10(1,2)+6*gm(1,2)&
&   *gm(1,3)*dgm01(3,3)*dgm10(1,2)-6*gm(1,2)*gm(3,3)*dgm01(1,2)*dgm10(1,3)&
&   -1.5d0*gm(1,3)**2*dgm01(2,2)*dgm10(1,3)+4.5d0*gm(1,1)*gm(3,3)&
&   *dgm01(2,2)*dgm10(1,3)+24*gm(1,2)*gm(1,3)*dgm01(2,3)*dgm10(1,3)&
&   -6*gm(1,2)**2*dgm01(3,3)*dgm10(1,3)+3*gm(1,3)*gm(3,3)*dgm01(1,1)&
&   *dgm10(2,2)-1.5d0*gm(1,3)**2*dgm01(1,3)*dgm10(2,2)+4.5d0*gm(1,1)&
&   *gm(3,3)*dgm01(1,3)*dgm10(2,2)+7.5d0*gm(1,1)*gm(1,3)*dgm01(3,3)&
&   *dgm10(2,2)+3*gm(1,2)*gm(3,3)*dgm01(1,1)*dgm10(2,3)+9*gm(1,3)&
&   **2*dgm01(1,2)*dgm10(2,3)+15*gm(1,1)*gm(3,3)*dgm01(1,2)*dgm10(2,3)&
&   +24*gm(1,2)*gm(1,3)*dgm01(1,3)*dgm10(2,3)+48*gm(1,1)*gm(1,3)&
&   *dgm01(2,3)*dgm10(2,3)-9*gm(1,1)*gm(1,2)*dgm01(3,3)*dgm10(2,3)&
&   +6*gm(1,2)*gm(1,3)*dgm01(1,2)*dgm10(3,3)-6*gm(1,2)**2*dgm01(1,3)&
&   *dgm10(3,3)+7.5d0*gm(1,1)*gm(1,3)*dgm01(2,2)*dgm10(3,3)-9*gm(1,1)&
&   *gm(1,2)*dgm01(2,3)*dgm10(3,3)+6*gm(1,2)*gm(1,3)*gm(3,3)*d2gm(1,2)&
&   -6*gm(1,2)**2*gm(3,3)*d2gm(1,3)+gm(2,3)**2*(16.5d0*dgm01(1,3)&
&   *dgm10(1,1)+16.5d0*dgm01(1,1)*dgm10(1,3)+1.5d0*gm(1,3)*d2gm(1,1)&
&   +15*gm(1,1)*d2gm(1,3))+gm(2,2)*(gm(1,3)*(-4.5d0*dgm01(3,3)*dgm10(1,1)&
&   -24*dgm01(1,3)*dgm10(1,3)-4.5d0*dgm01(1,1)*dgm10(3,3))-3*gm(1,1)&
&   *(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))-6*gm(1,3)**2*d2gm(1,3)&
&   +gm(3,3)*(-7.5d0*dgm01(1,3)*dgm10(1,1)-7.5d0*dgm01(1,1)*dgm10(1,3)&
&   -4.5d0*gm(1,3)*d2gm(1,1)-3*gm(1,1)*d2gm(1,3)))+1.5d0*gm(1,3)&
&   **3*d2gm(2,2)+7.5d0*gm(1,1)*gm(1,3)*gm(3,3)*d2gm(2,2)+3*gm(1,2)&
&   *gm(1,3)**2*d2gm(2,3)-9*gm(1,1)*gm(1,2)*gm(3,3)*d2gm(2,3)+gm(2,3)&
&   *(12*gm(1,2)*dgm01(3,3)*dgm10(1,1)+24*gm(1,1)*dgm01(3,3)*dgm10(1,2)&
&   +36*gm(1,2)*dgm01(1,3)*dgm10(1,3)+54*gm(1,1)*dgm01(2,3)*dgm10(1,3)&
&   +54*gm(1,1)*dgm01(1,3)*dgm10(2,3)+12*gm(1,2)*dgm01(1,1)*dgm10(3,3)&
&   +24*gm(1,1)*dgm01(1,2)*dgm10(3,3)+6*gm(1,3)**2*d2gm(1,2)+gm(3,3)&
&   *(36*dgm01(1,2)*dgm10(1,1)+36*dgm01(1,1)*dgm10(1,2)+12*gm(1,2)&
&   *d2gm(1,1)+24*gm(1,1)*d2gm(1,2))+gm(1,3)*(27*dgm01(2,3)*dgm10(1,1)&
&   +30*dgm01(1,3)*dgm10(1,2)+30*dgm01(1,2)*dgm10(1,3)+27*dgm01(1,1)&
&   *dgm10(2,3)+18*gm(1,2)*d2gm(1,3)+24*gm(1,1)*d2gm(2,3)))
   cm(4,9)=24*gm(1,2)*gm(3,3)*dgm01(2,2)*dgm10(1,1)+72*gm(1,2)&
&   *gm(3,3)*dgm01(1,2)*dgm10(1,2)+21*gm(1,3)**2*dgm01(2,2)*dgm10(1,2)&
&   +42*gm(1,1)*gm(3,3)*dgm01(2,2)*dgm10(1,2)+0*gm(1,2)*gm(1,3)*dgm01(2,3)&
&   *dgm10(1,2)+18*gm(1,2)**2*dgm01(3,3)*dgm10(1,2)+12*gm(1,2)*gm(1,3)&
&   *dgm01(2,2)*dgm10(1,3)+12*gm(1,2)**2*dgm01(2,3)*dgm10(1,3)+24*gm(1,2)&
&   *gm(3,3)*dgm01(1,1)*dgm10(2,2)+21*gm(1,3)**2*dgm01(1,2)*dgm10(2,2)&
&   +42*gm(1,1)*gm(3,3)*dgm01(1,2)*dgm10(2,2)+12*gm(1,2)*gm(1,3)&
&   *dgm01(1,3)*dgm10(2,2)+27*gm(1,1)*gm(1,3)*dgm01(2,3)*dgm10(2,2)&
&   +12*gm(1,1)*gm(1,2)*dgm01(3,3)*dgm10(2,2)+0*gm(1,2)*gm(1,3)*dgm01(1,2)&
&   *dgm10(2,3)+12*gm(1,2)**2*dgm01(1,3)*dgm10(2,3)+27*gm(1,1)*gm(1,3)&
&   *dgm01(2,2)*dgm10(2,3)+12*gm(1,1)*gm(1,2)*dgm01(2,3)*dgm10(2,3)&
&   +18*gm(1,2)**2*dgm01(1,2)*dgm10(3,3)+12*gm(1,1)*gm(1,2)*dgm01(2,2)&
&   *dgm10(3,3)+18*gm(1,2)**2*gm(3,3)*d2gm(1,2)+gm(2,3)**2*(21*dgm01(1,2)&
&   *dgm10(1,1)+21*dgm01(1,1)*dgm10(1,2)+3*gm(1,2)*d2gm(1,1)+18*gm(1,1)&
&   *d2gm(1,2))+3*gm(1,2)*gm(1,3)**2*d2gm(2,2)+12*gm(1,1)*gm(1,2)&
&   *gm(3,3)*d2gm(2,2)+6*gm(1,2)**2*gm(1,3)*d2gm(2,3)+gm(2,3)*(12*gm(1,2)&
&   *dgm01(2,3)*dgm10(1,1)+0*gm(1,2)*dgm01(1,3)*dgm10(1,2)+42*gm(1,1)&
&   *dgm01(2,3)*dgm10(1,2)+0*gm(1,2)*dgm01(1,2)*dgm10(1,3)+27*gm(1,1)&
&   *dgm01(2,2)*dgm10(1,3)+27*gm(1,1)*dgm01(1,3)*dgm10(2,2)+12*gm(1,2)&
&   *dgm01(1,1)*dgm10(2,3)+42*gm(1,1)*dgm01(1,2)*dgm10(2,3)+6*gm(1,2)&
&   **2*d2gm(1,3)+gm(2,2)*(27*dgm01(1,3)*dgm10(1,1)+27*dgm01(1,1)&
&   *dgm10(1,3)+3*gm(1,3)*d2gm(1,1)+24*gm(1,1)*d2gm(1,3))+gm(1,3)&
&   *(6*dgm01(2,2)*dgm10(1,1)-24*dgm01(1,2)*dgm10(1,2)+6*dgm01(1,1)&
&   *dgm10(2,2)-12*gm(1,2)*d2gm(1,2)+3*gm(1,1)*d2gm(2,2))+6*gm(1,1)&
&   *gm(1,2)*d2gm(2,3))+gm(2,2)*(12*gm(1,2)*dgm01(3,3)*dgm10(1,1)&
&   +30*gm(1,1)*dgm01(3,3)*dgm10(1,2)+12*gm(1,2)*dgm01(1,3)*dgm10(1,3)&
&   +48*gm(1,1)*dgm01(2,3)*dgm10(1,3)+48*gm(1,1)*dgm01(1,3)*dgm10(2,3)&
&   +12*gm(1,2)*dgm01(1,1)*dgm10(3,3)+30*gm(1,1)*dgm01(1,2)*dgm10(3,3)&
&   +18*gm(1,3)**2*d2gm(1,2)+gm(3,3)*(42*dgm01(1,2)*dgm10(1,1)+42*dgm01(1,1)&
&   *dgm10(1,2)+12*gm(1,2)*d2gm(1,1)+30*gm(1,1)*d2gm(1,2))+gm(1,3)&
&   *(27*dgm01(2,3)*dgm10(1,1)+42*dgm01(1,3)*dgm10(1,2)+42*dgm01(1,2)&
&   *dgm10(1,3)+27*dgm01(1,1)*dgm10(2,3)+6*gm(1,2)*d2gm(1,3)+24*gm(1,1)&
&   *d2gm(2,3)))
   cm(5,9)=39*gm(1,2)*gm(3,3)*dgm01(1,2)*dgm10(1,1)+0*gm(1,3)**2*dgm01(2,2)&
&   *dgm10(1,1)+10.5d0*gm(1,1)*gm(3,3)*dgm01(2,2)*dgm10(1,1)+15*gm(1,2)&
&   *gm(1,3)*dgm01(2,3)*dgm10(1,1)+7.5d0*gm(1,2)**2*dgm01(3,3)*dgm10(1,1)&
&   +39*gm(1,2)*gm(3,3)*dgm01(1,1)*dgm10(1,2)+12*gm(1,3)**2*dgm01(1,2)&
&   *dgm10(1,2)+48*gm(1,1)*gm(3,3)*dgm01(1,2)*dgm10(1,2)+18*gm(1,2)&
&   *gm(1,3)*dgm01(1,3)*dgm10(1,2)+12*gm(1,1)*gm(1,3)*dgm01(2,3)&
&   *dgm10(1,2)+24*gm(1,1)*gm(1,2)*dgm01(3,3)*dgm10(1,2)+18*gm(1,2)&
&   *gm(1,3)*dgm01(1,2)*dgm10(1,3)+6*gm(1,2)**2*dgm01(1,3)*dgm10(1,3)&
&   -3*gm(1,1)*gm(1,3)*dgm01(2,2)*dgm10(1,3)+30*gm(1,1)*gm(1,2)*dgm01(2,3)&
&   *dgm10(1,3)+0*gm(1,3)**2*dgm01(1,1)*dgm10(2,2)+10.5d0*gm(1,1)&
&   *gm(3,3)*dgm01(1,1)*dgm10(2,2)-3*gm(1,1)*gm(1,3)*dgm01(1,3)*dgm10(2,2)&
&   +6*gm(1,1)**2*dgm01(3,3)*dgm10(2,2)+15*gm(1,2)*gm(1,3)*dgm01(1,1)&
&   *dgm10(2,3)+12*gm(1,1)*gm(1,3)*dgm01(1,2)*dgm10(2,3)+30*gm(1,1)&
&   *gm(1,2)*dgm01(1,3)*dgm10(2,3)+24*gm(1,1)**2*dgm01(2,3)*dgm10(2,3)&
&   +7.5d0*gm(1,2)**2*dgm01(1,1)*dgm10(3,3)+24*gm(1,1)*gm(1,2)*dgm01(1,2)&
&   *dgm10(3,3)+6*gm(1,1)**2*dgm01(2,2)*dgm10(3,3)+7.5d0*gm(1,2)&
&   **2*gm(3,3)*d2gm(1,1)+gm(2,3)**2*(-6*dgm01(1,1)*dgm10(1,1)-3*gm(1,1)&
&   *d2gm(1,1))+6*gm(1,2)*gm(1,3)**2*d2gm(1,2)+24*gm(1,1)*gm(1,2)&
&   *gm(3,3)*d2gm(1,2)+3*gm(1,2)**2*gm(1,3)*d2gm(1,3)+gm(2,2)*(gm(1,1)&
&   *(-1.5d0*dgm01(3,3)*dgm10(1,1)-18*dgm01(1,3)*dgm10(1,3)-1.5d0*dgm01(1,1)&
&   *dgm10(3,3))-3*gm(1,3)**2*d2gm(1,1)+gm(3,3)*(-3*dgm01(1,1)*dgm10(1,1)&
&   -1.5d0*gm(1,1)*d2gm(1,1))+gm(1,3)*(-15*dgm01(1,3)*dgm10(1,1)&
&   -15*dgm01(1,1)*dgm10(1,3)-9*gm(1,1)*d2gm(1,3)))+3*gm(1,1)*gm(1,3)&
&   **2*d2gm(2,2)+6*gm(1,1)**2*gm(3,3)*d2gm(2,2)+6*gm(1,1)*gm(1,2)&
&   *gm(1,3)*d2gm(2,3)+gm(2,3)*(gm(1,3)*(15*dgm01(1,2)*dgm10(1,1)&
&   +15*dgm01(1,1)*dgm10(1,2)+9*gm(1,2)*d2gm(1,1)+6*gm(1,1)*d2gm(1,2))&
&   +gm(1,2)*(33*dgm01(1,3)*dgm10(1,1)+33*dgm01(1,1)*dgm10(1,3)+24*gm(1,1)&
&   *d2gm(1,3))+gm(1,1)*(18*dgm01(2,3)*dgm10(1,1)+30*dgm01(1,3)*dgm10(1,2)&
&   +30*dgm01(1,2)*dgm10(1,3)+18*dgm01(1,1)*dgm10(2,3)+12*gm(1,1)&
&   *d2gm(2,3)))
   cm(6,9)=3*gm(1,2)**3*d2gm(1,3)+gm(1,2)*(18*gm(1,1)*(dgm01(2,3)&
&   *dgm10(1,2)+dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2)+dgm01(1,2)&
&   *dgm10(2,3))+gm(2,3)*(9*dgm01(1,2)*dgm10(1,1)+9*dgm01(1,1)*dgm10(1,2)&
&   +6*gm(1,1)*d2gm(1,2))+gm(1,3)*(15*dgm01(2,2)*dgm10(1,1)+24*dgm01(1,2)&
&   *dgm10(1,2)+15*dgm01(1,1)*dgm10(2,2)+3*gm(1,1)*d2gm(2,2)))+gm(1,1)&
&   *(27*gm(1,3)*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+18*gm(1,1)&
&   *(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+gm(2,3)*(7.5d0*dgm01(2,2)&
&   *dgm10(1,1)+12*dgm01(1,2)*dgm10(1,2)+7.5d0*dgm01(1,1)*dgm10(2,2)&
&   +6*gm(1,1)*d2gm(2,2)))+gm(1,2)**2*(7.5d0*dgm01(2,3)*dgm10(1,1)&
&   +15*dgm01(1,3)*dgm10(1,2)+15*dgm01(1,2)*dgm10(1,3)+7.5d0*dgm01(1,1)&
&   *dgm10(2,3)+1.5d0*gm(2,3)*d2gm(1,1)+6*gm(1,3)*d2gm(1,2)+6*gm(1,1)&
&   *d2gm(2,3))+gm(2,2)*(27*gm(1,2)*dgm01(1,3)*dgm10(1,1)+19.5d0*gm(1,1)&
&   *dgm01(2,3)*dgm10(1,1)+39*gm(1,1)*dgm01(1,3)*dgm10(1,2)+27*gm(1,2)&
&   *dgm01(1,1)*dgm10(1,3)+39*gm(1,1)*dgm01(1,2)*dgm10(1,3)+19.5d0*gm(1,1)&
&   *dgm01(1,1)*dgm10(2,3)+gm(2,3)*(-9*dgm01(1,1)*dgm10(1,1)-4.5d0*gm(1,1)&
&   *d2gm(1,1))+gm(1,3)*(36*dgm01(1,2)*dgm10(1,1)+36*dgm01(1,1)*dgm10(1,2)&
&   +12*gm(1,2)*d2gm(1,1)+24*gm(1,1)*d2gm(1,2))+15*gm(1,1)*gm(1,2)&
&   *d2gm(1,3)+12*gm(1,1)**2*d2gm(2,3))
   cm(7,9)=-3*gm(1,1)*gm(2,3)*dgm01(2,2)*dgm10(2,2)+gm(1,2)*(-6*gm(1,3)&
&   *dgm01(2,2)*dgm10(2,2)+33*gm(2,3)*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)&
&   *dgm10(2,2)))+gm(1,2)**2*(16.5d0*dgm01(2,3)*dgm10(2,2)+16.5d0*dgm01(2,2)&
&   *dgm10(2,3)+7.5d0*gm(2,3)*d2gm(2,2))+gm(2,2)**2*(0*dgm01(2,3)&
&   *dgm10(1,1)+0*dgm01(1,3)*dgm10(1,2)+0*dgm01(1,2)*dgm10(1,3)+0*dgm01(1,1)&
&   *dgm10(2,3)+3*gm(2,3)*d2gm(1,1)-6*gm(1,3)*d2gm(1,2)+6*gm(1,2)&
&   *d2gm(1,3)-3*gm(1,1)*d2gm(2,3))+gm(2,2)*(36*gm(1,2)*dgm01(2,3)&
&   *dgm10(1,2)+9*gm(1,2)*dgm01(2,2)*dgm10(1,3)+9*gm(1,2)*dgm01(1,3)&
&   *dgm10(2,2)-7.5d0*gm(1,1)*dgm01(2,3)*dgm10(2,2)+36*gm(1,2)*dgm01(1,2)&
&   *dgm10(2,3)-7.5d0*gm(1,1)*dgm01(2,2)*dgm10(2,3)+gm(2,3)*(4.5d0*dgm01(2,2)&
&   *dgm10(1,1)+36*dgm01(1,2)*dgm10(1,2)+4.5d0*dgm01(1,1)*dgm10(2,2)&
&   +18*gm(1,2)*d2gm(1,2)-1.5d0*gm(1,1)*d2gm(2,2))+gm(1,3)*(-15*dgm01(2,2)&
&   *dgm10(1,2)-15*dgm01(1,2)*dgm10(2,2)-3*gm(1,2)*d2gm(2,2))+9*gm(1,2)&
&   **2*d2gm(2,3))
   cm(8,9)=36*gm(1,3)*gm(3,3)*dgm01(2,2)*dgm10(1,2)-6*gm(1,2)*gm(3,3)&
&   *dgm01(2,3)*dgm10(1,2)+3*gm(1,2)*gm(3,3)*dgm01(2,2)*dgm10(1,3)&
&   +36*gm(1,3)*gm(3,3)*dgm01(1,2)*dgm10(2,2)+3*gm(1,2)*gm(3,3)*dgm01(1,3)&
&   *dgm10(2,2)+16.5d0*gm(1,3)**2*dgm01(2,3)*dgm10(2,2)-7.5d0*gm(1,1)&
&   *gm(3,3)*dgm01(2,3)*dgm10(2,2)+12*gm(1,2)*gm(1,3)*dgm01(3,3)&
&   *dgm10(2,2)-6*gm(1,2)*gm(3,3)*dgm01(1,2)*dgm10(2,3)+16.5d0*gm(1,3)&
&   **2*dgm01(2,2)*dgm10(2,3)-7.5d0*gm(1,1)*gm(3,3)*dgm01(2,2)*dgm10(2,3)&
&   +36*gm(1,2)*gm(1,3)*dgm01(2,3)*dgm10(2,3)-6*gm(1,2)**2*dgm01(3,3)&
&   *dgm10(2,3)+12*gm(1,2)*gm(1,3)*dgm01(2,2)*dgm10(3,3)-6*gm(1,2)&
&   **2*dgm01(2,3)*dgm10(3,3)+1.5d0*gm(2,3)**3*d2gm(1,1)+12*gm(1,2)&
&   *gm(1,3)*gm(3,3)*d2gm(2,2)-6*gm(1,2)**2*gm(3,3)*d2gm(2,3)+gm(2,3)&
&   **2*(-1.5d0*dgm01(2,3)*dgm10(1,1)+9*dgm01(1,3)*dgm10(1,2)+9*dgm01(1,2)&
&   *dgm10(1,3)-1.5d0*dgm01(1,1)*dgm10(2,3)+6*gm(1,3)*d2gm(1,2)+3*gm(1,2)&
&   *d2gm(1,3)-6*gm(1,1)*d2gm(2,3))+gm(2,3)*(30*gm(1,3)*dgm01(2,3)&
&   *dgm10(1,2)+6*gm(1,2)*dgm01(3,3)*dgm10(1,2)+27*gm(1,3)*dgm01(2,2)&
&   *dgm10(1,3)+24*gm(1,2)*dgm01(2,3)*dgm10(1,3)+27*gm(1,3)*dgm01(1,3)&
&   *dgm10(2,2)-4.5d0*gm(1,1)*dgm01(3,3)*dgm10(2,2)+30*gm(1,3)*dgm01(1,2)&
&   *dgm10(2,3)+24*gm(1,2)*dgm01(1,3)*dgm10(2,3)-24*gm(1,1)*dgm01(2,3)&
&   *dgm10(2,3)+6*gm(1,2)*dgm01(1,2)*dgm10(3,3)-4.5d0*gm(1,1)*dgm01(2,2)&
&   *dgm10(3,3)+gm(2,2)*(7.5d0*dgm01(3,3)*dgm10(1,1)+48*dgm01(1,3)&
&   *dgm10(1,3)+7.5d0*dgm01(1,1)*dgm10(3,3)+24*gm(1,3)*d2gm(1,3))&
&   +1.5d0*gm(1,3)**2*d2gm(2,2)+gm(3,3)*(3*dgm01(2,2)*dgm10(1,1)&
&   +12*dgm01(1,2)*dgm10(1,2)+3*dgm01(1,1)*dgm10(2,2)+7.5d0*gm(2,2)&
&   *d2gm(1,1)+6*gm(1,2)*d2gm(1,2)-4.5d0*gm(1,1)*d2gm(2,2))+18*gm(1,2)&
&   *gm(1,3)*d2gm(2,3))+gm(2,2)*(-9*gm(1,2)*dgm01(3,3)*dgm10(1,3)&
&   -3*gm(1,1)*dgm01(3,3)*dgm10(2,3)-9*gm(1,2)*dgm01(1,3)*dgm10(3,3)&
&   -3*gm(1,1)*dgm01(2,3)*dgm10(3,3)+gm(1,3)*(24*dgm01(3,3)*dgm10(1,2)&
&   +54*dgm01(2,3)*dgm10(1,3)+54*dgm01(1,3)*dgm10(2,3)+24*dgm01(1,2)&
&   *dgm10(3,3))+15*gm(1,3)**2*d2gm(2,3)+gm(3,3)*(4.5d0*dgm01(2,3)&
&   *dgm10(1,1)+15*dgm01(1,3)*dgm10(1,2)+15*dgm01(1,2)*dgm10(1,3)&
&   +4.5d0*dgm01(1,1)*dgm10(2,3)+24*gm(1,3)*d2gm(1,2)-9*gm(1,2)*d2gm(1,3)&
&   -3*gm(1,1)*d2gm(2,3)))
   cm(9,9)=39*gm(1,2)*gm(3,3)*dgm01(2,2)*dgm10(1,2)+39*gm(1,2)&
&   *gm(3,3)*dgm01(1,2)*dgm10(2,2)-6*gm(1,3)**2*dgm01(2,2)*dgm10(2,2)&
&   -3*gm(1,1)*gm(3,3)*dgm01(2,2)*dgm10(2,2)+33*gm(1,2)*gm(1,3)*dgm01(2,3)&
&   *dgm10(2,2)+7.5d0*gm(1,2)**2*dgm01(3,3)*dgm10(2,2)+33*gm(1,2)&
&   *gm(1,3)*dgm01(2,2)*dgm10(2,3)+6*gm(1,2)**2*dgm01(2,3)*dgm10(2,3)&
&   +7.5d0*gm(1,2)**2*dgm01(2,2)*dgm10(3,3)+gm(2,2)**2*(6*dgm01(3,3)&
&   *dgm10(1,1)+24*dgm01(1,3)*dgm10(1,3)+6*dgm01(1,1)*dgm10(3,3)&
&   +6*gm(3,3)*d2gm(1,1)+12*gm(1,3)*d2gm(1,3))+7.5d0*gm(1,2)**2*gm(3,3)&
&   *d2gm(2,2)+gm(2,3)**2*(0*dgm01(2,2)*dgm10(1,1)+12*dgm01(1,2)&
&   *dgm10(1,2)+0*dgm01(1,1)*dgm10(2,2)+3*gm(2,2)*d2gm(1,1)+6*gm(1,2)&
&   *d2gm(1,2)-3*gm(1,1)*d2gm(2,2))+gm(2,3)*(18*gm(1,2)*dgm01(2,3)&
&   *dgm10(1,2)+15*gm(1,2)*dgm01(2,2)*dgm10(1,3)+15*gm(1,2)*dgm01(1,3)&
&   *dgm10(2,2)-15*gm(1,1)*dgm01(2,3)*dgm10(2,2)+18*gm(1,2)*dgm01(1,2)&
&   *dgm10(2,3)-15*gm(1,1)*dgm01(2,2)*dgm10(2,3)+gm(1,3)*(15*dgm01(2,2)&
&   *dgm10(1,2)+15*dgm01(1,2)*dgm10(2,2)+9*gm(1,2)*d2gm(2,2))+3*gm(1,2)&
&   **2*d2gm(2,3)+gm(2,2)*(-3*dgm01(2,3)*dgm10(1,1)+12*dgm01(1,3)&
&   *dgm10(1,2)+12*dgm01(1,2)*dgm10(1,3)-3*dgm01(1,1)*dgm10(2,3)&
&   +6*gm(1,3)*d2gm(1,2)+6*gm(1,2)*d2gm(1,3)-9*gm(1,1)*d2gm(2,3)))&
&   +gm(2,2)*(24*gm(1,2)*dgm01(3,3)*dgm10(1,2)+30*gm(1,2)*dgm01(2,3)&
&   *dgm10(1,3)-1.5d0*gm(1,1)*dgm01(3,3)*dgm10(2,2)+30*gm(1,2)*dgm01(1,3)&
&   *dgm10(2,3)-18*gm(1,1)*dgm01(2,3)*dgm10(2,3)+24*gm(1,2)*dgm01(1,2)&
&   *dgm10(3,3)-1.5d0*gm(1,1)*dgm01(2,2)*dgm10(3,3)-3*gm(1,3)**2*d2gm(2,2)&
&   +gm(3,3)*(10.5d0*dgm01(2,2)*dgm10(1,1)+48*dgm01(1,2)*dgm10(1,2)&
&   +10.5d0*dgm01(1,1)*dgm10(2,2)+24*gm(1,2)*d2gm(1,2)-1.5d0*gm(1,1)&
&   *d2gm(2,2))+gm(1,3)*(30*dgm01(2,3)*dgm10(1,2)+18*dgm01(2,2)*dgm10(1,3)&
&   +18*dgm01(1,3)*dgm10(2,2)+30*dgm01(1,2)*dgm10(2,3)+24*gm(1,2)&
&   *d2gm(2,3)))
   cm(10,9)=gm(1,3)*(-3*gm(2,2)*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)&
&   *dgm10(3,3))+gm(1,3)*(4.5d0*dgm01(3,3)*dgm10(2,2)+30*dgm01(2,3)&
&   *dgm10(2,3)+4.5d0*dgm01(2,2)*dgm10(3,3))-6*gm(1,2)*(dgm01(3,3)&
&   *dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))+gm(2,3)**2*(4.5d0*dgm01(3,3)&
&   *dgm10(1,1)+30*dgm01(1,3)*dgm10(1,3)+4.5d0*dgm01(1,1)*dgm10(3,3)&
&   +15*gm(1,3)*d2gm(1,3))+gm(3,3)**2*(-3*dgm01(2,2)*dgm10(1,1)-12*dgm01(1,2)&
&   *dgm10(1,2)-3*dgm01(1,1)*dgm10(2,2)-1.5d0*gm(2,2)*d2gm(1,1)-6*gm(1,2)&
&   *d2gm(1,2)-1.5d0*gm(1,1)*d2gm(2,2))+gm(2,3)*(gm(1,3)*(18*dgm01(3,3)&
&   *dgm10(1,2)+60*dgm01(2,3)*dgm10(1,3)+60*dgm01(1,3)*dgm10(2,3)&
&   +18*dgm01(1,2)*dgm10(3,3))-6*gm(1,2)*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)&
&   *dgm10(3,3))-3*gm(1,1)*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3))&
&   +15*gm(1,3)**2*d2gm(2,3))+gm(3,3)*(12*gm(1,3)*dgm01(2,3)*dgm10(1,2)&
&   -12*gm(1,2)*dgm01(3,3)*dgm10(1,2)+6*gm(1,3)*dgm01(2,2)*dgm10(1,3)&
&   -12*gm(1,2)*dgm01(2,3)*dgm10(1,3)+6*gm(1,3)*dgm01(1,3)*dgm10(2,2)&
&   -3*gm(1,1)*dgm01(3,3)*dgm10(2,2)+12*gm(1,3)*dgm01(1,2)*dgm10(2,3)&
&   -12*gm(1,2)*dgm01(1,3)*dgm10(2,3)-6*gm(1,1)*dgm01(2,3)*dgm10(2,3)&
&   -12*gm(1,2)*dgm01(1,2)*dgm10(3,3)-3*gm(1,1)*dgm01(2,2)*dgm10(3,3)&
&   +4.5d0*gm(2,3)**2*d2gm(1,1)+gm(2,2)*(-3*dgm01(3,3)*dgm10(1,1)&
&   -6*dgm01(1,3)*dgm10(1,3)-3*dgm01(1,1)*dgm10(3,3)-3*gm(1,3)*d2gm(1,3))&
&   +4.5d0*gm(1,3)**2*d2gm(2,2)-6*gm(1,2)*gm(1,3)*d2gm(2,3)+gm(2,3)&
&   *(6*dgm01(2,3)*dgm10(1,1)+12*dgm01(1,3)*dgm10(1,2)+12*dgm01(1,2)&
&   *dgm10(1,3)+6*dgm01(1,1)*dgm10(2,3)+18*gm(1,3)*d2gm(1,2)-6*gm(1,2)&
&   *d2gm(1,3)-3*gm(1,1)*d2gm(2,3)))
   cm(1,10)=-7.5d0*gm(1,1)*gm(3,3)*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)&
&   *dgm10(1,3))+2.5d0*gm(1,3)**3*d2gm(1,1)+gm(1,3)**2*(16.5d0*dgm01(1,3)&
&   *dgm10(1,1)+16.5d0*dgm01(1,1)*dgm10(1,3)+9*gm(1,1)*d2gm(1,3))&
&   +gm(1,1)**2*(0*dgm01(3,3)*dgm10(1,3)+0*dgm01(1,3)*dgm10(3,3)&
&   -3*gm(3,3)*d2gm(1,3))+gm(1,3)*(gm(3,3)*(-3*dgm01(1,1)*dgm10(1,1)&
&   -1.5d0*gm(1,1)*d2gm(1,1))+gm(1,1)*(4.5d0*dgm01(3,3)*dgm10(1,1)&
&   +36*dgm01(1,3)*dgm10(1,3)+4.5d0*dgm01(1,1)*dgm10(3,3)+3*gm(1,1)&
&   *d2gm(3,3)))
   cm(2,10)=-1.5d0*gm(1,3)*gm(3,3)*dgm01(2,2)*dgm10(1,1)-3*gm(1,2)&
&   *gm(3,3)*dgm01(2,3)*dgm10(1,1)-12*gm(1,2)*gm(3,3)*dgm01(1,3)&
&   *dgm10(1,2)+3*gm(1,2)*gm(1,3)*dgm01(3,3)*dgm10(1,2)-12*gm(1,2)&
&   *gm(3,3)*dgm01(1,2)*dgm10(1,3)-6*gm(1,3)**2*dgm01(2,2)*dgm10(1,3)&
&   -3*gm(1,1)*gm(3,3)*dgm01(2,2)*dgm10(1,3)+18*gm(1,2)*gm(1,3)*dgm01(2,3)&
&   *dgm10(1,3)-4.5d0*gm(1,2)**2*dgm01(3,3)*dgm10(1,3)-1.5d0*gm(1,3)&
&   *gm(3,3)*dgm01(1,1)*dgm10(2,2)-6*gm(1,3)**2*dgm01(1,3)*dgm10(2,2)&
&   -3*gm(1,1)*gm(3,3)*dgm01(1,3)*dgm10(2,2)-4.5d0*gm(1,1)*gm(1,3)&
&   *dgm01(3,3)*dgm10(2,2)-3*gm(1,2)*gm(3,3)*dgm01(1,1)*dgm10(2,3)&
&   +18*gm(1,2)*gm(1,3)*dgm01(1,3)*dgm10(2,3)+12*gm(1,1)*gm(1,2)&
&   *dgm01(3,3)*dgm10(2,3)+3*gm(1,2)*gm(1,3)*dgm01(1,2)*dgm10(3,3)&
&   -4.5d0*gm(1,2)**2*dgm01(1,3)*dgm10(3,3)-4.5d0*gm(1,1)*gm(1,3)&
&   *dgm01(2,2)*dgm10(3,3)+12*gm(1,1)*gm(1,2)*dgm01(2,3)*dgm10(3,3)&
&   -6*gm(1,2)**2*gm(3,3)*d2gm(1,3)+gm(2,3)**2*(22.5d0*dgm01(1,3)&
&   *dgm10(1,1)+22.5d0*dgm01(1,1)*dgm10(1,3)+7.5d0*gm(1,3)*d2gm(1,1)&
&   +15*gm(1,1)*d2gm(1,3))+1.5d0*gm(1,2)**2*gm(1,3)*d2gm(3,3)+gm(2,3)&
&   *(9*gm(1,2)*dgm01(3,3)*dgm10(1,1)+12*gm(1,1)*dgm01(3,3)*dgm10(1,2)&
&   +36*gm(1,2)*dgm01(1,3)*dgm10(1,3)+30*gm(1,1)*dgm01(2,3)*dgm10(1,3)&
&   +30*gm(1,1)*dgm01(1,3)*dgm10(2,3)+9*gm(1,2)*dgm01(1,1)*dgm10(3,3)&
&   +12*gm(1,1)*dgm01(1,2)*dgm10(3,3)-3*gm(3,3)*(dgm01(1,2)*dgm10(1,1)&
&   +dgm01(1,1)*dgm10(1,2)+gm(1,2)*d2gm(1,1))+gm(1,3)*(15*dgm01(2,3)&
&   *dgm10(1,1)+18*dgm01(1,3)*dgm10(1,2)+18*dgm01(1,2)*dgm10(1,3)&
&   +15*dgm01(1,1)*dgm10(2,3)+18*gm(1,2)*d2gm(1,3))+12*gm(1,1)*gm(1,2)&
&   *d2gm(3,3))+gm(2,2)*(-7.5d0*gm(1,1)*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)&
&   *dgm10(3,3))-6*gm(1,3)**2*d2gm(1,3)+gm(3,3)*(-4.5d0*dgm01(1,3)&
&   *dgm10(1,1)-4.5d0*dgm01(1,1)*dgm10(1,3)-1.5d0*gm(1,3)*d2gm(1,1)&
&   -3*gm(1,1)*d2gm(1,3))+gm(1,3)*(-6*dgm01(3,3)*dgm10(1,1)-24*dgm01(1,3)&
&   *dgm10(1,3)-6*dgm01(1,1)*dgm10(3,3)-4.5d0*gm(1,1)*d2gm(3,3)))
   cm(3,10)=gm(3,3)**2*(15*dgm01(1,3)*dgm10(1,1)+15*dgm01(1,1)&
&   *dgm10(1,3)+3*gm(1,3)*d2gm(1,1)+12*gm(1,1)*d2gm(1,3))+gm(1,3)&
&   *(15*gm(1,1)*dgm01(3,3)*dgm10(3,3)+10.5d0*gm(1,3)*(dgm01(3,3)&
&   *dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+1.5d0*gm(1,3)**2*d2gm(3,3))&
&   +gm(3,3)*(31.5d0*gm(1,1)*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))&
&   +6*gm(1,3)**2*d2gm(1,3)+gm(1,3)*(13.5d0*dgm01(3,3)*dgm10(1,1)&
&   +24*dgm01(1,3)*dgm10(1,3)+13.5d0*dgm01(1,1)*dgm10(3,3)+7.5d0*gm(1,1)&
&   *d2gm(3,3)))
   cm(4,10)=gm(1,1)*(24*gm(1,2)*dgm01(3,3)*dgm10(3,3)+27*gm(2,3)&
&   *(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3)))-3*gm(3,3)**2*(dgm01(1,2)&
&   *dgm10(1,1)+dgm01(1,1)*dgm10(1,2)+gm(1,2)*d2gm(1,1))+gm(1,3)&
&   **2*(3*dgm01(3,3)*dgm10(1,2)+6*dgm01(2,3)*dgm10(1,3)+6*dgm01(1,3)&
&   *dgm10(2,3)+3*dgm01(1,2)*dgm10(3,3)+6*gm(2,3)*d2gm(1,3)+3*gm(1,2)&
&   *d2gm(3,3))+gm(3,3)*(6*gm(1,2)*dgm01(3,3)*dgm10(1,1)+12*gm(1,1)&
&   *dgm01(3,3)*dgm10(1,2)+12*gm(1,2)*dgm01(1,3)*dgm10(1,3)+24*gm(1,1)&
&   *dgm01(2,3)*dgm10(1,3)+24*gm(1,1)*dgm01(1,3)*dgm10(2,3)+6*gm(1,2)&
&   *dgm01(1,1)*dgm10(3,3)+12*gm(1,1)*dgm01(1,2)*dgm10(3,3)+gm(2,3)&
&   *(33*dgm01(1,3)*dgm10(1,1)+33*dgm01(1,1)*dgm10(1,3)+9*gm(1,3)&
&   *d2gm(1,1)+24*gm(1,1)*d2gm(1,3))+gm(1,3)*(9*dgm01(2,3)*dgm10(1,1)&
&   +6*dgm01(1,3)*dgm10(1,2)+6*dgm01(1,2)*dgm10(1,3)+9*dgm01(1,1)&
&   *dgm10(2,3)+6*gm(1,2)*d2gm(1,3))+12*gm(1,1)*gm(1,2)*d2gm(3,3))&
&   +gm(1,3)*(12*gm(1,2)*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))&
&   +3*gm(1,1)*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3))+gm(2,3)&
&   *(12*dgm01(3,3)*dgm10(1,1)+24*dgm01(1,3)*dgm10(1,3)+12*dgm01(1,1)&
&   *dgm10(3,3)+3*gm(1,1)*d2gm(3,3)))
   cm(5,10)=12*gm(1,1)**2*dgm01(3,3)*dgm10(3,3)+21*gm(1,1)*gm(1,3)&
&   *(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+gm(3,3)**2*(-3*dgm01(1,1)&
&   *dgm10(1,1)-1.5d0*gm(1,1)*d2gm(1,1))+3*gm(1,3)**3*d2gm(1,3)+gm(1,3)&
&   **2*(7.5d0*dgm01(3,3)*dgm10(1,1)+18*dgm01(1,3)*dgm10(1,3)+7.5d0*dgm01(1,1)&
&   *dgm10(3,3)+3*gm(1,1)*d2gm(3,3))+gm(3,3)*(4.5d0*gm(1,3)**2*d2gm(1,1)&
&   +gm(1,3)*(24*dgm01(1,3)*dgm10(1,1)+24*dgm01(1,1)*dgm10(1,3)+15*gm(1,1)&
&   *d2gm(1,3))+gm(1,1)*(9*dgm01(3,3)*dgm10(1,1)+30*dgm01(1,3)*dgm10(1,3)&
&   +9*dgm01(1,1)*dgm10(3,3)+6*gm(1,1)*d2gm(3,3)))
   cm(6,10)=-12*gm(1,2)*gm(3,3)*dgm01(1,3)*dgm10(1,1)-1.5d0*gm(1,1)&
&   *gm(3,3)*dgm01(2,3)*dgm10(1,1)-9*gm(1,1)*gm(3,3)*dgm01(1,3)*dgm10(1,2)&
&   -12*gm(1,2)*gm(3,3)*dgm01(1,1)*dgm10(1,3)-9*gm(1,1)*gm(3,3)*dgm01(1,2)&
&   *dgm10(1,3)-6*gm(1,1)*gm(1,2)*dgm01(3,3)*dgm10(1,3)-1.5d0*gm(1,1)&
&   *gm(3,3)*dgm01(1,1)*dgm10(2,3)+6*gm(1,1)**2*dgm01(3,3)*dgm10(2,3)&
&   -6*gm(1,1)*gm(1,2)*dgm01(1,3)*dgm10(3,3)+6*gm(1,1)**2*dgm01(2,3)&
&   *dgm10(3,3)-9*gm(1,1)*gm(1,2)*gm(3,3)*d2gm(1,3)+gm(1,3)**2*(7.5d0*dgm01(2,3)&
&   *dgm10(1,1)+3*dgm01(1,3)*dgm10(1,2)+3*dgm01(1,2)*dgm10(1,3)+7.5d0*dgm01(1,1)&
&   *dgm10(2,3)+3*gm(1,2)*d2gm(1,3))+gm(1,3)*(gm(1,1)*(3*dgm01(3,3)&
&   *dgm10(1,2)+24*dgm01(2,3)*dgm10(1,3)+24*dgm01(1,3)*dgm10(2,3)&
&   +3*dgm01(1,2)*dgm10(3,3))-3*gm(3,3)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)&
&   *dgm10(1,2)+gm(1,2)*d2gm(1,1))+gm(1,2)*(0*dgm01(3,3)*dgm10(1,1)&
&   +12*dgm01(1,3)*dgm10(1,3)+0*dgm01(1,1)*dgm10(3,3)+3*gm(1,1)*d2gm(3,3)))&
&   +gm(2,3)*(7.5d0*gm(1,3)**2*d2gm(1,1)+gm(3,3)*(-3*dgm01(1,1)*dgm10(1,1)&
&   -1.5d0*gm(1,1)*d2gm(1,1))+gm(1,3)*(39*dgm01(1,3)*dgm10(1,1)+39*dgm01(1,1)&
&   *dgm10(1,3)+24*gm(1,1)*d2gm(1,3))+gm(1,1)*(10.5d0*dgm01(3,3)&
&   *dgm10(1,1)+48*dgm01(1,3)*dgm10(1,3)+10.5d0*dgm01(1,1)*dgm10(3,3)&
&   +6*gm(1,1)*d2gm(3,3)))
   cm(7,10)=((270*gm(2,3)**2*dgm01(2,3)-54*gm(2,2)*gm(3,3)*dgm01(2,3)&
&   -54*gm(2,3)*(gm(3,3)*dgm01(2,2)+gm(2,2)*dgm01(3,3)))*dgm10(1,1)&
&   +6*(90*gm(2,3)**2*dgm01(1,2)-18*gm(1,2)*gm(3,3)*dgm01(2,2)+gm(2,3)&
&   *(-36*gm(2,2)*dgm01(1,3)-36*gm(1,3)*dgm01(2,2)+180*gm(1,2)*dgm01(2,3))&
&   +gm(2,2)*(-18*gm(3,3)*dgm01(1,2)-36*gm(1,3)*dgm01(2,3)-18*gm(1,2)&
&   *dgm01(3,3)))*dgm10(1,3)+3*dgm01(3,3)*(-18*gm(1,1)*gm(2,3)*dgm10(2,2)&
&   +gm(1,2)*(180*gm(2,3)*dgm10(1,2)-36*gm(1,3)*dgm10(2,2))+90*gm(1,2)&
&   **2*dgm10(2,3)+gm(2,2)*(-18*gm(2,3)*dgm10(1,1)-36*gm(1,3)*dgm10(1,2)&
&   -36*gm(1,2)*dgm10(1,3)-18*gm(1,1)*dgm10(2,3)))+3*(-18*gm(1,1)&
&   *gm(2,3)*dgm01(2,2)+gm(1,2)*(180*gm(2,3)*dgm01(1,2)-36*gm(1,3)&
&   *dgm01(2,2))+90*gm(1,2)**2*dgm01(2,3)+gm(2,2)*(-18*gm(2,3)*dgm01(1,1)&
&   -36*gm(1,3)*dgm01(1,2)-36*gm(1,2)*dgm01(1,3)-18*gm(1,1)*dgm01(2,3)))&
&   *dgm10(3,3)+6*dgm01(1,3)*(90*gm(2,3)**2*dgm10(1,2)-18*gm(1,2)&
&   *gm(3,3)*dgm10(2,2)+gm(2,3)*(-36*gm(2,2)*dgm10(1,3)-36*gm(1,3)&
&   *dgm10(2,2)+180*gm(1,2)*dgm10(2,3))+gm(2,2)*(-18*gm(3,3)*dgm10(1,2)&
&   -36*gm(1,3)*dgm10(2,3)-18*gm(1,2)*dgm10(3,3)))+dgm01(1,1)*(270*gm(2,3)&
&   **2*dgm10(2,3)-54*gm(2,2)*gm(3,3)*dgm10(2,3)-54*gm(2,3)*(gm(3,3)&
&   *dgm10(2,2)+gm(2,2)*dgm10(3,3)))+(90*gm(2,3)**3-54*gm(2,2)*gm(2,3)&
&   *gm(3,3))*d2gm(1,1)+6*(-36*gm(1,3)*gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)&
&   **2-18*gm(2,2)*gm(3,3)))*d2gm(1,3)+3*(90*gm(1,2)**2*gm(2,3)-18*gm(2,2)&
&   *(2*gm(1,2)*gm(1,3)+gm(1,1)*gm(2,3)))*d2gm(3,3))/36.d0
   cm(8,10)=-9*gm(1,1)*gm(2,3)*dgm01(3,3)*dgm10(3,3)+gm(1,3)*(24*gm(1,2)&
&   *dgm01(3,3)*dgm10(3,3)+9*gm(2,3)*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)&
&   *dgm10(3,3)))+gm(3,3)**2*(3*dgm01(2,3)*dgm10(1,1)+12*dgm01(1,3)&
&   *dgm10(1,2)+12*dgm01(1,2)*dgm10(1,3)+3*dgm01(1,1)*dgm10(2,3)&
&   +3*gm(2,3)*d2gm(1,1)+12*gm(1,2)*d2gm(1,3))+1.5d0*gm(1,3)**2*(dgm01(3,3)&
&   *dgm10(2,3)+dgm01(2,3)*dgm10(3,3)+gm(2,3)*d2gm(3,3))+gm(3,3)&
&   *(36*gm(1,2)*dgm01(3,3)*dgm10(1,3)-4.5d0*gm(1,1)*dgm01(3,3)*dgm10(2,3)&
&   +36*gm(1,2)*dgm01(1,3)*dgm10(3,3)-4.5d0*gm(1,1)*dgm01(2,3)*dgm10(3,3)&
&   +gm(2,3)*(1.5d0*dgm01(3,3)*dgm10(1,1)+12*dgm01(1,3)*dgm10(1,3)&
&   +1.5d0*dgm01(1,1)*dgm10(3,3)+6*gm(1,3)*d2gm(1,3)-4.5d0*gm(1,1)&
&   *d2gm(3,3))+gm(1,3)*(12*dgm01(3,3)*dgm10(1,2)+6*dgm01(2,3)*dgm10(1,3)&
&   +6*dgm01(1,3)*dgm10(2,3)+12*dgm01(1,2)*dgm10(3,3)+12*gm(1,2)&
&   *d2gm(3,3)))
   cm(9,10)=-15*gm(1,3)*gm(2,2)*dgm01(3,3)*dgm10(1,3)-3*gm(1,3)&
&   **2*dgm01(3,3)*dgm10(2,2)+9*gm(1,2)*gm(1,3)*dgm01(3,3)*dgm10(2,3)&
&   -15*gm(1,3)*gm(2,2)*dgm01(1,3)*dgm10(3,3)-3*gm(1,3)**2*dgm01(2,2)&
&   *dgm10(3,3)+9*gm(1,2)*gm(1,3)*dgm01(2,3)*dgm10(3,3)+15*gm(1,2)&
&   **2*dgm01(3,3)*dgm10(3,3)-3*gm(1,1)*gm(2,2)*dgm01(3,3)*dgm10(3,3)&
&   -1.5d0*gm(3,3)**2*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2)&
&   +gm(2,2)*d2gm(1,1))-3*gm(1,3)**2*gm(2,2)*d2gm(3,3)+gm(2,3)**2*(1.5d0*dgm01(3,3)&
&   *dgm10(1,1)+6*dgm01(1,3)*dgm10(1,3)+1.5d0*dgm01(1,1)*dgm10(3,3)&
&   +3*gm(1,3)*d2gm(1,3)-3*gm(1,1)*d2gm(3,3))+gm(3,3)*(15*gm(1,2)&
&   *dgm01(3,3)*dgm10(1,2)-9*gm(1,3)*dgm01(2,2)*dgm10(1,3)+24*gm(1,2)&
&   *dgm01(2,3)*dgm10(1,3)-9*gm(1,3)*dgm01(1,3)*dgm10(2,2)-1.5d0*gm(1,1)&
&   *dgm01(3,3)*dgm10(2,2)+24*gm(1,2)*dgm01(1,3)*dgm10(2,3)+15*gm(1,2)&
&   *dgm01(1,2)*dgm10(3,3)-1.5d0*gm(1,1)*dgm01(2,2)*dgm10(3,3)+4.5d0*gm(2,3)&
&   **2*d2gm(1,1)+gm(2,3)*(9*dgm01(2,3)*dgm10(1,1)+24*dgm01(1,3)&
&   *dgm10(1,2)+24*dgm01(1,2)*dgm10(1,3)+9*dgm01(1,1)*dgm10(2,3)&
&   +24*gm(1,2)*d2gm(1,3))+7.5d0*gm(1,2)**2*d2gm(3,3)+gm(2,2)*(-4.5d0*dgm01(3,3)&
&   *dgm10(1,1)-18*dgm01(1,3)*dgm10(1,3)-4.5d0*dgm01(1,1)*dgm10(3,3)&
&   -9*gm(1,3)*d2gm(1,3)-1.5d0*gm(1,1)*d2gm(3,3)))+gm(2,3)*(33*gm(1,2)&
&   *(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))-6*gm(1,1)*(dgm01(3,3)&
&   *dgm10(2,3)+dgm01(2,3)*dgm10(3,3))+gm(1,3)*(9*dgm01(3,3)*dgm10(1,2)&
&   +6*dgm01(2,3)*dgm10(1,3)+6*dgm01(1,3)*dgm10(2,3)+9*dgm01(1,2)&
&   *dgm10(3,3)+9*gm(1,2)*d2gm(3,3)))
   cm(10,10)=9*gm(1,3)**2*dgm01(3,3)*dgm10(3,3)+gm(3,3)**3*d2gm(1,1)&
&   +gm(3,3)**2*(1.5d0*dgm01(3,3)*dgm10(1,1)+12*dgm01(1,3)*dgm10(1,3)&
&   +1.5d0*dgm01(1,1)*dgm10(3,3)+6*gm(1,3)*d2gm(1,3)-1.5d0*gm(1,1)&
&   *d2gm(3,3))+gm(3,3)*(-6*gm(1,1)*dgm01(3,3)*dgm10(3,3)+21*gm(1,3)&
&   *(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+4.5d0*gm(1,3)&
&   **2*d2gm(3,3))
   cm(1,11)=5*gm(1,2)**3*d2gm(1,2)+gm(1,2)*(6*gm(1,1)*(dgm01(2,2)&
&   *dgm10(1,2)+dgm01(1,2)*dgm10(2,2))-3*gm(2,2)*(dgm01(1,2)*dgm10(1,1)&
&   +dgm01(1,1)*dgm10(1,2)+gm(1,1)*d2gm(1,2)))+gm(1,2)**2*(4.5d0*dgm01(2,2)&
&   *dgm10(1,1)+30*dgm01(1,2)*dgm10(1,2)+4.5d0*dgm01(1,1)*dgm10(2,2)&
&   +4.5d0*gm(1,1)*d2gm(2,2))+gm(1,1)*(-3*gm(1,1)*dgm01(2,2)*dgm10(2,2)&
&   +gm(2,2)*(-3*dgm01(2,2)*dgm10(1,1)-6*dgm01(1,2)*dgm10(1,2)-3*dgm01(1,1)&
&   *dgm10(2,2)-1.5d0*gm(1,1)*d2gm(2,2)))
   cm(2,11)=6*gm(1,2)**2*dgm01(2,2)*dgm10(2,2)+gm(2,2)*(24*gm(1,1)&
&   *dgm01(2,2)*dgm10(2,2)+18*gm(1,2)*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)&
&   *dgm10(2,2))+3*gm(1,2)**2*d2gm(2,2))+gm(2,2)**2*(6*dgm01(2,2)&
&   *dgm10(1,1)+12*dgm01(1,2)*dgm10(1,2)+6*(dgm01(1,1)*dgm10(2,2)&
&   +gm(1,2)*d2gm(1,2)+gm(1,1)*d2gm(2,2)))
   cm(3,11)=(-108*gm(1,2)*gm(3,3)*dgm01(2,2)*dgm10(1,2)-108*gm(1,2)&
&   *gm(3,3)*dgm01(1,2)*dgm10(2,2)-72*gm(1,3)**2*dgm01(2,2)*dgm10(2,2)&
&   -36*gm(1,1)*gm(3,3)*dgm01(2,2)*dgm10(2,2)+108*gm(1,2)*gm(1,3)&
&   *dgm01(2,3)*dgm10(2,2)-36*gm(1,2)**2*dgm01(3,3)*dgm10(2,2)+108*gm(1,2)&
&   *gm(1,3)*dgm01(2,2)*dgm10(2,3)-36*gm(1,2)**2*dgm01(2,2)*dgm10(3,3)&
&   -36*gm(1,2)**2*gm(3,3)*d2gm(2,2)+gm(2,3)**2*(90*dgm01(2,2)*dgm10(1,1)&
&   +360*dgm01(1,2)*dgm10(1,2)+90*dgm01(1,1)*dgm10(2,2)+180*gm(1,2)&
&   *d2gm(1,2)+90*gm(1,1)*d2gm(2,2))+gm(2,2)*(-36*gm(1,2)*dgm01(3,3)&
&   *dgm10(1,2)-18*gm(1,1)*dgm01(3,3)*dgm10(2,2)-72*gm(1,3)*(dgm01(2,3)&
&   *dgm10(1,2)+dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2)+dgm01(1,2)&
&   *dgm10(2,3))-36*gm(1,2)*dgm01(1,2)*dgm10(3,3)-18*gm(1,1)*dgm01(2,2)&
&   *dgm10(3,3)-36*gm(1,3)**2*d2gm(2,2)+gm(3,3)*(-18*dgm01(2,2)*dgm10(1,1)&
&   -72*dgm01(1,2)*dgm10(1,2)-18*dgm01(1,1)*dgm10(2,2)-36*gm(1,2)&
&   *d2gm(1,2)-18*gm(1,1)*d2gm(2,2)))+gm(2,3)*(360*gm(1,2)*dgm01(2,3)&
&   *dgm10(1,2)+108*gm(1,2)*dgm01(2,2)*dgm10(1,3)+108*gm(1,2)*dgm01(1,3)&
&   *dgm10(2,2)+180*gm(1,1)*dgm01(2,3)*dgm10(2,2)+360*gm(1,2)*dgm01(1,2)&
&   *dgm10(2,3)+180*gm(1,1)*dgm01(2,2)*dgm10(2,3)-72*gm(2,2)*(dgm01(1,3)&
&   *dgm10(1,2)+dgm01(1,2)*dgm10(1,3)+gm(1,3)*d2gm(1,2))+gm(1,3)&
&   *(36*dgm01(2,2)*dgm10(1,2)+36*dgm01(1,2)*dgm10(2,2)+108*gm(1,2)&
&   *d2gm(2,2))))/12.d0
   cm(4,11)=24*gm(1,1)*gm(2,3)*dgm01(2,2)*dgm10(2,2)+gm(1,2)*(6*gm(1,3)&
&   *dgm01(2,2)*dgm10(2,2)+24*gm(2,3)*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)&
&   *dgm10(2,2)))-6*gm(2,2)**2*(dgm01(1,3)*dgm10(1,2)+dgm01(1,2)&
&   *dgm10(1,3)+gm(1,3)*d2gm(1,2))+3*gm(1,2)**2*(dgm01(2,3)*dgm10(2,2)&
&   +dgm01(2,2)*dgm10(2,3)+gm(2,3)*d2gm(2,2))+gm(2,2)*(18*gm(1,2)&
&   *dgm01(2,3)*dgm10(1,2)+3*gm(1,2)*dgm01(2,2)*dgm10(1,3)+3*gm(1,2)&
&   *dgm01(1,3)*dgm10(2,2)+12*gm(1,1)*dgm01(2,3)*dgm10(2,2)+18*gm(1,2)&
&   *dgm01(1,2)*dgm10(2,3)+12*gm(1,1)*dgm01(2,2)*dgm10(2,3)+gm(2,3)&
&   *(12*dgm01(2,2)*dgm10(1,1)+36*dgm01(1,2)*dgm10(1,2)+12*dgm01(1,1)&
&   *dgm10(2,2)+18*gm(1,2)*d2gm(1,2)+12*gm(1,1)*d2gm(2,2))+gm(1,3)&
&   *(-9*dgm01(2,2)*dgm10(1,2)-9*dgm01(1,2)*dgm10(2,2)+3*gm(1,2)&
&   *d2gm(2,2)))
   cm(5,11)=(gm(1,1)*(-108*gm(1,3)*dgm01(2,2)*dgm10(2,2)+108*gm(2,3)&
&   *(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2)))+gm(1,2)**2*(180*dgm01(2,3)&
&   *dgm10(1,2)+18*dgm01(2,2)*dgm10(1,3)+18*dgm01(1,3)*dgm10(2,2)&
&   +180*dgm01(1,2)*dgm10(2,3)+180*gm(2,3)*d2gm(1,2)+18*gm(1,3)*d2gm(2,2))&
&   +gm(2,2)*(-72*gm(1,2)*dgm01(1,3)*dgm10(1,2)-36*gm(1,1)*dgm01(2,3)&
&   *dgm10(1,2)-72*gm(1,2)*dgm01(1,2)*dgm10(1,3)-54*gm(1,1)*dgm01(2,2)&
&   *dgm10(1,3)-54*gm(1,1)*dgm01(1,3)*dgm10(2,2)-36*gm(1,1)*dgm01(1,2)&
&   *dgm10(2,3)-36*gm(2,3)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2)&
&   +gm(1,1)*d2gm(1,2))+gm(1,3)*(-54*dgm01(2,2)*dgm10(1,1)-144*dgm01(1,2)&
&   *dgm10(1,2)-54*dgm01(1,1)*dgm10(2,2)-72*gm(1,2)*d2gm(1,2)-54*gm(1,1)&
&   *d2gm(2,2)))+gm(1,2)*(-36*gm(1,3)*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)&
&   *dgm10(2,2))+144*gm(1,1)*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))&
&   +gm(2,3)*(144*dgm01(2,2)*dgm10(1,1)+720*dgm01(1,2)*dgm10(1,2)&
&   +144*dgm01(1,1)*dgm10(2,2)+144*gm(1,1)*d2gm(2,2))))/12.d0
   cm(6,11)=-3*gm(2,2)**2*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2)&
&   +gm(1,1)*d2gm(1,2))+gm(1,2)*(15*gm(1,1)*dgm01(2,2)*dgm10(2,2)&
&   +13.5d0*gm(1,2)*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))&
&   +1.5d0*gm(1,2)**2*d2gm(2,2))+gm(2,2)*(1.5d0*gm(1,1)*(dgm01(2,2)&
&   *dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+9*gm(1,2)**2*d2gm(1,2)+gm(1,2)&
&   *(7.5d0*dgm01(2,2)*dgm10(1,1)+36*dgm01(1,2)*dgm10(1,2)+7.5d0*dgm01(1,1)&
&   *dgm10(2,2)+7.5d0*gm(1,1)*d2gm(2,2)))
   cm(7,11)=gm(2,2)*(12*gm(1,2)*dgm01(2,2)*dgm10(2,2)+2*gm(2,2)&
&   **2*d2gm(1,2)+gm(2,2)*(9*dgm01(2,2)*dgm10(1,2)+9*dgm01(1,2)*dgm10(2,2)&
&   +3*gm(1,2)*d2gm(2,2)))
   cm(8,11)=-9*gm(1,2)*gm(3,3)*dgm01(2,2)*dgm10(2,2)-3*gm(2,2)&
&   **2*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3)+gm(3,3)*d2gm(1,2))&
&   +gm(2,3)**2*(10.5d0*dgm01(2,2)*dgm10(1,2)+10.5d0*dgm01(1,2)*dgm10(2,2)&
&   +9*gm(2,2)*d2gm(1,2)+1.5d0*gm(1,2)*d2gm(2,2))+gm(2,2)*(12*gm(1,3)&
&   *(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))-4.5d0*gm(1,2)&
&   *(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3))+gm(3,3)*(-10.5d0*dgm01(2,2)&
&   *dgm10(1,2)-10.5d0*dgm01(1,2)*dgm10(2,2)-4.5d0*gm(1,2)*d2gm(2,2)))&
&   +gm(2,3)*(24*gm(1,3)*dgm01(2,2)*dgm10(2,2)+3*gm(1,2)*(dgm01(2,3)&
&   *dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+gm(2,2)*(18*dgm01(2,3)*dgm10(1,2)&
&   +12*dgm01(2,2)*dgm10(1,3)+12*dgm01(1,3)*dgm10(2,2)+18*dgm01(1,2)&
&   *dgm10(2,3)+12*gm(1,3)*d2gm(2,2)))
   cm(9,11)=6*gm(1,2)*gm(2,3)*dgm01(2,2)*dgm10(2,2)+6*gm(2,2)**2*(dgm01(2,3)&
&   *dgm10(1,2)+dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2)+dgm01(1,2)&
&   *dgm10(2,3)+gm(2,3)*d2gm(1,2)+gm(1,3)*d2gm(2,2))+gm(2,2)*(24*gm(1,3)&
&   *dgm01(2,2)*dgm10(2,2)+3*gm(1,2)*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)&
&   *dgm10(2,3))+gm(2,3)*(15*dgm01(2,2)*dgm10(1,2)+15*dgm01(1,2)&
&   *dgm10(2,2)+3*gm(1,2)*d2gm(2,2)))
   cm(10,11)=(-108*gm(3,3)*(gm(1,3)*dgm01(2,2)*dgm10(2,2)+gm(1,2)&
&   *(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3)))+180*gm(2,3)**3*d2gm(1,2)&
&   +gm(2,3)**2*(540*dgm01(2,3)*dgm10(1,2)+270*dgm01(2,2)*dgm10(1,3)&
&   +270*dgm01(1,3)*dgm10(2,2)+540*dgm01(1,2)*dgm10(2,3)+270*gm(1,3)&
&   *d2gm(2,2))+gm(2,3)*(540*gm(1,3)*dgm01(2,3)*dgm10(2,2)-108*gm(1,2)&
&   *dgm01(3,3)*dgm10(2,2)+540*gm(1,3)*dgm01(2,2)*dgm10(2,3)-108*gm(1,2)&
&   *dgm01(2,2)*dgm10(3,3)-108*gm(2,2)*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)&
&   *dgm10(3,3))+gm(3,3)*(-216*dgm01(2,2)*dgm10(1,2)-216*dgm01(1,2)&
&   *dgm10(2,2)-108*gm(2,2)*d2gm(1,2)-108*gm(1,2)*d2gm(2,2)))+gm(2,2)&
&   *(-54*gm(1,3)*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3))+gm(3,3)&
&   *(-108*dgm01(2,3)*dgm10(1,2)-54*dgm01(2,2)*dgm10(1,3)-54*dgm01(1,3)&
&   *dgm10(2,2)-108*dgm01(1,2)*dgm10(2,3)-54*gm(1,3)*d2gm(2,2))))&
&   /36.d0
   cm(1,12)=gm(1,2)*(gm(1,1)*(6*dgm01(3,3)*dgm10(1,2)+12*dgm01(2,3)&
&   *dgm10(1,3)+12*dgm01(1,3)*dgm10(2,3)+6*dgm01(1,2)*dgm10(3,3))&
&   -3*gm(3,3)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2)+gm(1,1)&
&   *d2gm(1,2))-6*gm(2,3)*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3)&
&   +gm(1,1)*d2gm(1,3)))+gm(1,3)**2*(4.5d0*dgm01(2,2)*dgm10(1,1)&
&   +30*dgm01(1,2)*dgm10(1,2)+4.5d0*dgm01(1,1)*dgm10(2,2)+15*gm(1,2)&
&   *d2gm(1,2)+4.5d0*gm(1,1)*d2gm(2,2))+gm(1,3)*(18*gm(1,2)*dgm01(2,3)&
&   *dgm10(1,1)+60*gm(1,2)*dgm01(1,3)*dgm10(1,2)+12*gm(1,1)*dgm01(2,3)&
&   *dgm10(1,2)+60*gm(1,2)*dgm01(1,2)*dgm10(1,3)+6*gm(1,1)*dgm01(2,2)&
&   *dgm10(1,3)+6*gm(1,1)*dgm01(1,3)*dgm10(2,2)+18*gm(1,2)*dgm01(1,1)&
&   *dgm10(2,3)+12*gm(1,1)*dgm01(1,2)*dgm10(2,3)-6*gm(2,3)*(dgm01(1,2)&
&   *dgm10(1,1)+dgm01(1,1)*dgm10(1,2)+gm(1,1)*d2gm(1,2))+15*gm(1,2)&
&   **2*d2gm(1,3)-3*gm(2,2)*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3)&
&   +gm(1,1)*d2gm(1,3))+18*gm(1,1)*gm(1,2)*d2gm(2,3))+gm(1,1)*(-3*gm(2,2)&
&   *dgm01(3,3)*dgm10(1,1)-6*gm(2,2)*dgm01(1,3)*dgm10(1,3)-3*gm(1,1)&
&   *dgm01(3,3)*dgm10(2,2)-12*gm(1,1)*dgm01(2,3)*dgm10(2,3)-3*gm(2,2)&
&   *dgm01(1,1)*dgm10(3,3)-3*gm(1,1)*dgm01(2,2)*dgm10(3,3)+gm(3,3)&
&   *(-3*dgm01(2,2)*dgm10(1,1)-6*dgm01(1,2)*dgm10(1,2)-3*dgm01(1,1)&
&   *dgm10(2,2)-1.5d0*gm(1,1)*d2gm(2,2))+gm(2,3)*(-12*dgm01(2,3)&
&   *dgm10(1,1)-12*dgm01(1,3)*dgm10(1,2)-12*dgm01(1,2)*dgm10(1,3)&
&   -12*dgm01(1,1)*dgm10(2,3)-6*gm(1,1)*d2gm(2,3))-1.5d0*gm(1,1)&
&   *gm(2,2)*d2gm(3,3))+gm(1,2)**2*(4.5d0*dgm01(3,3)*dgm10(1,1)+30*dgm01(1,3)&
&   *dgm10(1,3)+4.5d0*(dgm01(1,1)*dgm10(3,3)+gm(1,1)*d2gm(3,3)))
   cm(2,12)=-15*gm(1,2)*gm(3,3)*dgm01(2,2)*dgm10(1,2)-15*gm(1,2)&
&   *gm(3,3)*dgm01(1,2)*dgm10(2,2)-6*gm(1,3)**2*dgm01(2,2)*dgm10(2,2)&
&   -3*gm(1,1)*gm(3,3)*dgm01(2,2)*dgm10(2,2)+15*gm(1,2)*gm(1,3)*dgm01(2,3)&
&   *dgm10(2,2)+0*gm(1,2)**2*dgm01(3,3)*dgm10(2,2)+15*gm(1,2)*gm(1,3)&
&   *dgm01(2,2)*dgm10(2,3)+12*gm(1,2)**2*dgm01(2,3)*dgm10(2,3)+0*gm(1,2)&
&   **2*dgm01(2,2)*dgm10(3,3)-3*gm(1,2)**2*gm(3,3)*d2gm(2,2)+gm(2,3)&
&   **2*(7.5d0*dgm01(2,2)*dgm10(1,1)+6*dgm01(1,2)*dgm10(1,2)+7.5d0*dgm01(1,1)&
&   *dgm10(2,2)+3*gm(1,2)*d2gm(1,2)+7.5d0*gm(1,1)*d2gm(2,2))+gm(2,3)&
&   *(18*gm(1,2)*dgm01(2,3)*dgm10(1,2)+15*gm(1,2)*dgm01(2,2)*dgm10(1,3)&
&   +15*gm(1,2)*dgm01(1,3)*dgm10(2,2)+39*gm(1,1)*dgm01(2,3)*dgm10(2,2)&
&   +18*gm(1,2)*dgm01(1,2)*dgm10(2,3)+39*gm(1,1)*dgm01(2,2)*dgm10(2,3)&
&   +gm(1,3)*(33*dgm01(2,2)*dgm10(1,2)+33*dgm01(1,2)*dgm10(2,2)+9*gm(1,2)&
&   *d2gm(2,2))+6*gm(1,2)**2*d2gm(2,3)+gm(2,2)*(24*dgm01(2,3)*dgm10(1,1)&
&   +30*dgm01(1,3)*dgm10(1,2)+30*dgm01(1,2)*dgm10(1,3)+24*dgm01(1,1)&
&   *dgm10(2,3)+24*gm(1,3)*d2gm(1,2)+6*gm(1,2)*d2gm(1,3)+24*gm(1,1)&
&   *d2gm(2,3)))+gm(2,2)**2*(6*dgm01(3,3)*dgm10(1,1)+24*dgm01(1,3)&
&   *dgm10(1,3)+6*dgm01(1,1)*dgm10(3,3)+12*gm(1,3)*d2gm(1,3)+6*gm(1,1)&
&   *d2gm(3,3))+gm(2,2)*(-3*gm(1,2)*dgm01(3,3)*dgm10(1,2)+12*gm(1,2)&
&   *dgm01(2,3)*dgm10(1,3)+10.5d0*gm(1,1)*dgm01(3,3)*dgm10(2,2)+12*gm(1,2)&
&   *dgm01(1,3)*dgm10(2,3)+48*gm(1,1)*dgm01(2,3)*dgm10(2,3)-3*gm(1,2)&
&   *dgm01(1,2)*dgm10(3,3)+10.5d0*gm(1,1)*dgm01(2,2)*dgm10(3,3)-3*gm(1,3)&
&   **2*d2gm(2,2)+gm(3,3)*(-1.5d0*dgm01(2,2)*dgm10(1,1)-18*dgm01(1,2)&
&   *dgm10(1,2)-1.5d0*dgm01(1,1)*dgm10(2,2)-9*gm(1,2)*d2gm(1,2)-1.5d0*gm(1,1)&
&   *d2gm(2,2))+gm(1,3)*(30*dgm01(2,3)*dgm10(1,2)+18*dgm01(2,2)*dgm10(1,3)&
&   +18*dgm01(1,3)*dgm10(2,2)+30*dgm01(1,2)*dgm10(2,3)+6*gm(1,2)&
&   *d2gm(2,3))+3*gm(1,2)**2*d2gm(3,3))
   cm(3,12)=-15*gm(1,3)*gm(2,2)*dgm01(3,3)*dgm10(1,3)+0*gm(1,3)&
&   **2*dgm01(3,3)*dgm10(2,2)+12*gm(1,3)**2*dgm01(2,3)*dgm10(2,3)&
&   +15*gm(1,2)*gm(1,3)*dgm01(3,3)*dgm10(2,3)-15*gm(1,3)*gm(2,2)&
&   *dgm01(1,3)*dgm10(3,3)+0*gm(1,3)**2*dgm01(2,2)*dgm10(3,3)+15*gm(1,2)&
&   *gm(1,3)*dgm01(2,3)*dgm10(3,3)-6*gm(1,2)**2*dgm01(3,3)*dgm10(3,3)&
&   -3*gm(1,1)*gm(2,2)*dgm01(3,3)*dgm10(3,3)+gm(3,3)**2*(6*dgm01(2,2)&
&   *dgm10(1,1)+24*dgm01(1,2)*dgm10(1,2)+6*dgm01(1,1)*dgm10(2,2)&
&   +12*gm(1,2)*d2gm(1,2)+6*gm(1,1)*d2gm(2,2))-3*gm(1,3)**2*gm(2,2)&
&   *d2gm(3,3)+gm(2,3)**2*(7.5d0*dgm01(3,3)*dgm10(1,1)+6*dgm01(1,3)&
&   *dgm10(1,3)+7.5d0*dgm01(1,1)*dgm10(3,3)+3*gm(1,3)*d2gm(1,3)+7.5d0*gm(1,1)&
&   *d2gm(3,3))+gm(3,3)*(12*gm(1,3)*dgm01(2,3)*dgm10(1,2)+18*gm(1,2)&
&   *dgm01(3,3)*dgm10(1,2)-3*gm(1,3)*dgm01(2,2)*dgm10(1,3)+30*gm(1,2)&
&   *dgm01(2,3)*dgm10(1,3)-3*gm(1,3)*dgm01(1,3)*dgm10(2,2)+10.5d0*gm(1,1)&
&   *dgm01(3,3)*dgm10(2,2)+12*gm(1,3)*dgm01(1,2)*dgm10(2,3)+30*gm(1,2)&
&   *dgm01(1,3)*dgm10(2,3)+48*gm(1,1)*dgm01(2,3)*dgm10(2,3)+18*gm(1,2)&
&   *dgm01(1,2)*dgm10(3,3)+10.5d0*gm(1,1)*dgm01(2,2)*dgm10(3,3)+3*gm(1,3)&
&   **2*d2gm(2,2)+6*gm(1,2)*gm(1,3)*d2gm(2,3)+gm(2,3)*(24*dgm01(2,3)&
&   *dgm10(1,1)+30*dgm01(1,3)*dgm10(1,2)+30*dgm01(1,2)*dgm10(1,3)&
&   +24*dgm01(1,1)*dgm10(2,3)+6*gm(1,3)*d2gm(1,2)+24*gm(1,2)*d2gm(1,3)&
&   +24*gm(1,1)*d2gm(2,3))-3*gm(1,2)**2*d2gm(3,3)+gm(2,2)*(-1.5d0*dgm01(3,3)&
&   *dgm10(1,1)-18*dgm01(1,3)*dgm10(1,3)-1.5d0*dgm01(1,1)*dgm10(3,3)&
&   -9*gm(1,3)*d2gm(1,3)-1.5d0*gm(1,1)*d2gm(3,3)))+gm(2,3)*(33*gm(1,2)&
&   *(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+39*gm(1,1)*(dgm01(3,3)&
&   *dgm10(2,3)+dgm01(2,3)*dgm10(3,3))+6*gm(1,3)**2*d2gm(2,3)+gm(1,3)&
&   *(15*dgm01(3,3)*dgm10(1,2)+18*dgm01(2,3)*dgm10(1,3)+18*dgm01(1,3)&
&   *dgm10(2,3)+15*dgm01(1,2)*dgm10(3,3)+9*gm(1,2)*d2gm(3,3)))
   cm(4,12)=27*gm(1,3)*gm(3,3)*dgm01(2,2)*dgm10(1,2)+42*gm(1,2)&
&   *gm(3,3)*dgm01(2,3)*dgm10(1,2)+27*gm(1,2)*gm(3,3)*dgm01(2,2)&
&   *dgm10(1,3)+27*gm(1,3)*gm(3,3)*dgm01(1,2)*dgm10(2,2)+27*gm(1,2)&
&   *gm(3,3)*dgm01(1,3)*dgm10(2,2)+21*gm(1,3)**2*dgm01(2,3)*dgm10(2,2)&
&   +42*gm(1,1)*gm(3,3)*dgm01(2,3)*dgm10(2,2)+6*gm(1,2)*gm(1,3)*dgm01(3,3)&
&   *dgm10(2,2)+42*gm(1,2)*gm(3,3)*dgm01(1,2)*dgm10(2,3)+21*gm(1,3)&
&   **2*dgm01(2,2)*dgm10(2,3)+42*gm(1,1)*gm(3,3)*dgm01(2,2)*dgm10(2,3)&
&   -24*gm(1,2)*gm(1,3)*dgm01(2,3)*dgm10(2,3)+21*gm(1,2)**2*dgm01(3,3)&
&   *dgm10(2,3)+6*gm(1,2)*gm(1,3)*dgm01(2,2)*dgm10(3,3)+21*gm(1,2)&
&   **2*dgm01(2,3)*dgm10(3,3)+3*gm(1,2)*gm(1,3)*gm(3,3)*d2gm(2,2)&
&   +18*gm(1,2)**2*gm(3,3)*d2gm(2,3)+gm(2,3)**2*(18*dgm01(2,3)*dgm10(1,1)&
&   +12*dgm01(1,3)*dgm10(1,2)+12*dgm01(1,2)*dgm10(1,3)+18*dgm01(1,1)&
&   *dgm10(2,3)+6*gm(1,3)*d2gm(1,2)+6*gm(1,2)*d2gm(1,3)+18*gm(1,1)&
&   *d2gm(2,3))+gm(2,3)*(0*gm(1,3)*dgm01(2,3)*dgm10(1,2)+12*gm(1,2)&
&   *dgm01(3,3)*dgm10(1,2)+12*gm(1,3)*dgm01(2,2)*dgm10(1,3)+0*gm(1,2)&
&   *dgm01(2,3)*dgm10(1,3)+12*gm(1,3)*dgm01(1,3)*dgm10(2,2)+24*gm(1,1)&
&   *dgm01(3,3)*dgm10(2,2)+0*gm(1,3)*dgm01(1,2)*dgm10(2,3)+0*gm(1,2)&
&   *dgm01(1,3)*dgm10(2,3)+72*gm(1,1)*dgm01(2,3)*dgm10(2,3)+12*gm(1,2)&
&   *dgm01(1,2)*dgm10(3,3)+24*gm(1,1)*dgm01(2,2)*dgm10(3,3)+3*gm(1,3)&
&   **2*d2gm(2,2)+gm(3,3)*(12*dgm01(2,2)*dgm10(1,1)+12*dgm01(1,2)&
&   *dgm10(1,2)+12*dgm01(1,1)*dgm10(2,2)+6*gm(1,2)*d2gm(1,2)+12*gm(1,1)&
&   *d2gm(2,2))-12*gm(1,2)*gm(1,3)*d2gm(2,3)+3*gm(1,2)**2*d2gm(3,3)&
&   +gm(2,2)*(12*dgm01(3,3)*dgm10(1,1)+12*dgm01(1,3)*dgm10(1,3)+12*dgm01(1,1)&
&   *dgm10(3,3)+6*gm(1,3)*d2gm(1,3)+12*gm(1,1)*d2gm(3,3)))+gm(2,2)&
&   *(27*gm(1,2)*dgm01(3,3)*dgm10(1,3)+42*gm(1,1)*dgm01(3,3)*dgm10(2,3)&
&   +27*gm(1,2)*dgm01(1,3)*dgm10(3,3)+42*gm(1,1)*dgm01(2,3)*dgm10(3,3)&
&   +18*gm(1,3)**2*d2gm(2,3)+gm(3,3)*(30*dgm01(2,3)*dgm10(1,1)+48*dgm01(1,3)&
&   *dgm10(1,2)+48*dgm01(1,2)*dgm10(1,3)+30*dgm01(1,1)*dgm10(2,3)&
&   +24*gm(1,3)*d2gm(1,2)+24*gm(1,2)*d2gm(1,3)+30*gm(1,1)*d2gm(2,3))&
&   +gm(1,3)*(27*dgm01(3,3)*dgm10(1,2)+42*dgm01(2,3)*dgm10(1,3)+42*dgm01(1,3)&
&   *dgm10(2,3)+27*dgm01(1,2)*dgm10(3,3)+3*gm(1,2)*d2gm(3,3)))
   cm(5,12)=7.5d0*gm(1,3)*gm(3,3)*dgm01(2,2)*dgm10(1,1)+24*gm(1,2)&
&   *gm(3,3)*dgm01(2,3)*dgm10(1,1)+48*gm(1,3)*gm(3,3)*dgm01(1,2)&
&   *dgm10(1,2)+54*gm(1,2)*gm(3,3)*dgm01(1,3)*dgm10(1,2)+9*gm(1,3)&
&   **2*dgm01(2,3)*dgm10(1,2)+15*gm(1,1)*gm(3,3)*dgm01(2,3)*dgm10(1,2)&
&   +27*gm(1,2)*gm(1,3)*dgm01(3,3)*dgm10(1,2)+54*gm(1,2)*gm(3,3)&
&   *dgm01(1,2)*dgm10(1,3)-1.5d0*gm(1,3)**2*dgm01(2,2)*dgm10(1,3)&
&   +4.5d0*gm(1,1)*gm(3,3)*dgm01(2,2)*dgm10(1,3)+30*gm(1,2)*gm(1,3)&
&   *dgm01(2,3)*dgm10(1,3)+16.5d0*gm(1,2)**2*dgm01(3,3)*dgm10(1,3)&
&   +7.5d0*gm(1,3)*gm(3,3)*dgm01(1,1)*dgm10(2,2)-1.5d0*gm(1,3)**2*dgm01(1,3)&
&   *dgm10(2,2)+4.5d0*gm(1,1)*gm(3,3)*dgm01(1,3)*dgm10(2,2)+3*gm(1,1)&
&   *gm(1,3)*dgm01(3,3)*dgm10(2,2)+24*gm(1,2)*gm(3,3)*dgm01(1,1)&
&   *dgm10(2,3)+9*gm(1,3)**2*dgm01(1,2)*dgm10(2,3)+15*gm(1,1)*gm(3,3)&
&   *dgm01(1,2)*dgm10(2,3)+30*gm(1,2)*gm(1,3)*dgm01(1,3)*dgm10(2,3)&
&   +12*gm(1,1)*gm(1,3)*dgm01(2,3)*dgm10(2,3)+36*gm(1,1)*gm(1,2)&
&   *dgm01(3,3)*dgm10(2,3)+27*gm(1,2)*gm(1,3)*dgm01(1,2)*dgm10(3,3)&
&   +16.5d0*gm(1,2)**2*dgm01(1,3)*dgm10(3,3)+3*gm(1,1)*gm(1,3)*dgm01(2,2)&
&   *dgm10(3,3)+36*gm(1,1)*gm(1,2)*dgm01(2,3)*dgm10(3,3)+24*gm(1,2)&
&   *gm(1,3)*gm(3,3)*d2gm(1,2)+15*gm(1,2)**2*gm(3,3)*d2gm(1,3)-6*gm(2,3)&
&   **2*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3)+gm(1,1)*d2gm(1,3))&
&   +1.5d0*gm(1,3)**3*d2gm(2,2)+7.5d0*gm(1,1)*gm(1,3)*gm(3,3)*d2gm(2,2)&
&   +6*gm(1,2)*gm(1,3)**2*d2gm(2,3)+24*gm(1,1)*gm(1,2)*gm(3,3)*d2gm(2,3)&
&   +1.5d0*gm(1,2)**2*gm(1,3)*d2gm(3,3)+gm(2,3)*(12*gm(1,2)*dgm01(3,3)&
&   *dgm10(1,1)+3*gm(1,1)*dgm01(3,3)*dgm10(1,2)+36*gm(1,2)*dgm01(1,3)&
&   *dgm10(1,3)-6*gm(1,1)*dgm01(2,3)*dgm10(1,3)-6*gm(1,1)*dgm01(1,3)&
&   *dgm10(2,3)+12*gm(1,2)*dgm01(1,1)*dgm10(3,3)+3*gm(1,1)*dgm01(1,2)&
&   *dgm10(3,3)+3*gm(1,3)**2*d2gm(1,2)-9*gm(3,3)*(dgm01(1,2)*dgm10(1,1)&
&   +dgm01(1,1)*dgm10(1,2)+gm(1,1)*d2gm(1,2))+gm(1,3)*(6*dgm01(2,3)&
&   *dgm10(1,1)+24*dgm01(1,3)*dgm10(1,2)+24*dgm01(1,2)*dgm10(1,3)&
&   +6*dgm01(1,1)*dgm10(2,3)+18*gm(1,2)*d2gm(1,3)+6*gm(1,1)*d2gm(2,3))&
&   +12*gm(1,1)*gm(1,2)*d2gm(3,3))+gm(2,2)*(-7.5d0*gm(1,1)*(dgm01(3,3)&
&   *dgm10(1,3)+dgm01(1,3)*dgm10(3,3))-6*gm(1,3)**2*d2gm(1,3)-3*gm(3,3)&
&   *(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3)+gm(1,1)*d2gm(1,3))&
&   +gm(1,3)*(-4.5d0*dgm01(3,3)*dgm10(1,1)-24*dgm01(1,3)*dgm10(1,3)&
&   -4.5d0*dgm01(1,1)*dgm10(3,3)-4.5d0*gm(1,1)*d2gm(3,3)))
   cm(6,12)=-4.5d0*gm(1,2)*gm(3,3)*dgm01(2,2)*dgm10(1,1)-24*gm(1,2)&
&   *gm(3,3)*dgm01(1,2)*dgm10(1,2)+16.5d0*gm(1,3)**2*dgm01(2,2)*dgm10(1,2)&
&   -7.5d0*gm(1,1)*gm(3,3)*dgm01(2,2)*dgm10(1,2)+30*gm(1,2)*gm(1,3)&
&   *dgm01(2,3)*dgm10(1,2)-1.5d0*gm(1,2)**2*dgm01(3,3)*dgm10(1,2)&
&   +27*gm(1,2)*gm(1,3)*dgm01(2,2)*dgm10(1,3)+9*gm(1,2)**2*dgm01(2,3)&
&   *dgm10(1,3)-4.5d0*gm(1,2)*gm(3,3)*dgm01(1,1)*dgm10(2,2)+16.5d0*gm(1,3)&
&   **2*dgm01(1,2)*dgm10(2,2)-7.5d0*gm(1,1)*gm(3,3)*dgm01(1,2)*dgm10(2,2)&
&   +27*gm(1,2)*gm(1,3)*dgm01(1,3)*dgm10(2,2)+36*gm(1,1)*gm(1,3)&
&   *dgm01(2,3)*dgm10(2,2)+3*gm(1,1)*gm(1,2)*dgm01(3,3)*dgm10(2,2)&
&   +30*gm(1,2)*gm(1,3)*dgm01(1,2)*dgm10(2,3)+9*gm(1,2)**2*dgm01(1,3)&
&   *dgm10(2,3)+36*gm(1,1)*gm(1,3)*dgm01(2,2)*dgm10(2,3)+12*gm(1,1)&
&   *gm(1,2)*dgm01(2,3)*dgm10(2,3)-1.5d0*gm(1,2)**2*dgm01(1,2)*dgm10(3,3)&
&   +3*gm(1,1)*gm(1,2)*dgm01(2,2)*dgm10(3,3)-6*gm(1,2)**2*gm(3,3)&
&   *d2gm(1,2)-6*gm(2,3)**2*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2)&
&   +gm(1,1)*d2gm(1,2))+1.5d0*gm(1,2)*gm(1,3)**2*d2gm(2,2)-4.5d0*gm(1,1)&
&   *gm(1,2)*gm(3,3)*d2gm(2,2)+6*gm(1,2)**2*gm(1,3)*d2gm(2,3)+gm(2,3)&
&   *(6*gm(1,2)*dgm01(2,3)*dgm10(1,1)+24*gm(1,2)*dgm01(1,3)*dgm10(1,2)&
&   -6*gm(1,1)*dgm01(2,3)*dgm10(1,2)+24*gm(1,2)*dgm01(1,2)*dgm10(1,3)&
&   +3*gm(1,1)*dgm01(2,2)*dgm10(1,3)+3*gm(1,1)*dgm01(1,3)*dgm10(2,2)&
&   +6*gm(1,2)*dgm01(1,1)*dgm10(2,3)-6*gm(1,1)*dgm01(1,2)*dgm10(2,3)&
&   +3*gm(1,2)**2*d2gm(1,3)-9*gm(2,2)*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)&
&   *dgm10(1,3)+gm(1,1)*d2gm(1,3))+gm(1,3)*(12*dgm01(2,2)*dgm10(1,1)&
&   +36*dgm01(1,2)*dgm10(1,2)+12*dgm01(1,1)*dgm10(2,2)+18*gm(1,2)&
&   *d2gm(1,2)+12*gm(1,1)*d2gm(2,2))+6*gm(1,1)*gm(1,2)*d2gm(2,3))&
&   +1.5d0*gm(1,2)**3*d2gm(3,3)+gm(2,2)*(7.5d0*gm(1,2)*dgm01(3,3)&
&   *dgm10(1,1)+4.5d0*gm(1,1)*dgm01(3,3)*dgm10(1,2)+48*gm(1,2)*dgm01(1,3)&
&   *dgm10(1,3)+15*gm(1,1)*dgm01(2,3)*dgm10(1,3)+15*gm(1,1)*dgm01(1,3)&
&   *dgm10(2,3)+7.5d0*gm(1,2)*dgm01(1,1)*dgm10(3,3)+4.5d0*gm(1,1)&
&   *dgm01(1,2)*dgm10(3,3)+15*gm(1,3)**2*d2gm(1,2)-3*gm(3,3)*(dgm01(1,2)&
&   *dgm10(1,1)+dgm01(1,1)*dgm10(1,2)+gm(1,1)*d2gm(1,2))+gm(1,3)&
&   *(24*dgm01(2,3)*dgm10(1,1)+54*dgm01(1,3)*dgm10(1,2)+54*dgm01(1,2)&
&   *dgm10(1,3)+24*dgm01(1,1)*dgm10(2,3)+24*gm(1,2)*d2gm(1,3)+24*gm(1,1)&
&   *d2gm(2,3))+7.5d0*gm(1,1)*gm(1,2)*d2gm(3,3))
   cm(7,12)=-3*gm(1,2)*gm(3,3)*dgm01(2,2)*dgm10(2,2)+gm(2,3)**2*(16.5d0*dgm01(2,2)&
&   *dgm10(1,2)+16.5d0*dgm01(1,2)*dgm10(2,2)+9*gm(2,2)*d2gm(1,2)&
&   +7.5d0*gm(1,2)*d2gm(2,2))+gm(2,2)*(-15*gm(1,3)*(dgm01(2,3)*dgm10(2,2)&
&   +dgm01(2,2)*dgm10(2,3))+gm(1,2)*(4.5d0*dgm01(3,3)*dgm10(2,2)&
&   +36*dgm01(2,3)*dgm10(2,3)+4.5d0*dgm01(2,2)*dgm10(3,3))+gm(3,3)&
&   *(-7.5d0*dgm01(2,2)*dgm10(1,2)-7.5d0*dgm01(1,2)*dgm10(2,2)-1.5d0*gm(1,2)&
&   *d2gm(2,2)))+gm(2,3)*(-6*gm(1,3)*dgm01(2,2)*dgm10(2,2)+33*gm(1,2)&
&   *(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+6*gm(2,2)**2*d2gm(1,3)&
&   +gm(2,2)*(36*dgm01(2,3)*dgm10(1,2)+9*dgm01(2,2)*dgm10(1,3)+9*dgm01(1,3)&
&   *dgm10(2,2)+36*dgm01(1,2)*dgm10(2,3)-3*gm(1,3)*d2gm(2,2)+18*gm(1,2)&
&   *d2gm(2,3)))+gm(2,2)**2*(0*dgm01(3,3)*dgm10(1,2)+0*dgm01(2,3)&
&   *dgm10(1,3)+0*dgm01(1,3)*dgm10(2,3)+0*dgm01(1,2)*dgm10(3,3)-3*gm(3,3)&
&   *d2gm(1,2)-6*gm(1,3)*d2gm(2,3)+3*gm(1,2)*d2gm(3,3))
   cm(8,12)=gm(2,2)*(-9*gm(1,2)*dgm01(3,3)*dgm10(3,3)+36*gm(1,3)&
&   *(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))+3*gm(2,3)**3*d2gm(1,3)&
&   +gm(3,3)**2*(18*dgm01(2,2)*dgm10(1,2)+18*dgm01(1,2)*dgm10(2,2)&
&   +12*gm(2,2)*d2gm(1,2)+6*gm(1,2)*d2gm(2,2))+gm(2,3)**2*(7.5d0*dgm01(3,3)&
&   *dgm10(1,2)+15*dgm01(2,3)*dgm10(1,3)+15*dgm01(1,3)*dgm10(2,3)&
&   +7.5d0*dgm01(1,2)*dgm10(3,3)+6*gm(1,3)*d2gm(2,3)+1.5d0*gm(1,2)&
&   *d2gm(3,3))+gm(3,3)*(27*gm(1,3)*dgm01(2,3)*dgm10(2,2)+7.5d0*gm(1,2)&
&   *dgm01(3,3)*dgm10(2,2)+27*gm(1,3)*dgm01(2,2)*dgm10(2,3)+12*gm(1,2)&
&   *dgm01(2,3)*dgm10(2,3)+7.5d0*gm(1,2)*dgm01(2,2)*dgm10(3,3)+6*gm(2,3)&
&   **2*d2gm(1,2)+gm(2,3)*(18*dgm01(2,3)*dgm10(1,2)+18*dgm01(2,2)&
&   *dgm10(1,3)+18*dgm01(1,3)*dgm10(2,2)+18*dgm01(1,2)*dgm10(2,3)&
&   +15*gm(2,2)*d2gm(1,3)+3*gm(1,3)*d2gm(2,2)+6*gm(1,2)*d2gm(2,3))&
&   +gm(2,2)*(19.5d0*dgm01(3,3)*dgm10(1,2)+39*dgm01(2,3)*dgm10(1,3)&
&   +39*dgm01(1,3)*dgm10(2,3)+19.5d0*dgm01(1,2)*dgm10(3,3)+24*gm(1,3)&
&   *d2gm(2,3)-4.5d0*gm(1,2)*d2gm(3,3)))+gm(2,3)*(gm(1,3)*(15*dgm01(3,3)&
&   *dgm10(2,2)+24*dgm01(2,3)*dgm10(2,3)+15*dgm01(2,2)*dgm10(3,3))&
&   +9*gm(1,2)*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3))+gm(2,2)&
&   *(27*dgm01(3,3)*dgm10(1,3)+27*dgm01(1,3)*dgm10(3,3)+12*gm(1,3)&
&   *d2gm(3,3)))
   cm(9,12)=gm(3,3)*(-9*gm(1,3)*dgm01(2,2)*dgm10(2,2)+36*gm(1,2)&
&   *(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3)))+3*gm(2,3)**3*d2gm(1,2)&
&   +gm(2,3)**2*(15*dgm01(2,3)*dgm10(1,2)+7.5d0*dgm01(2,2)*dgm10(1,3)&
&   +7.5d0*dgm01(1,3)*dgm10(2,2)+15*dgm01(1,2)*dgm10(2,3)+6*gm(2,2)&
&   *d2gm(1,3)+1.5d0*gm(1,3)*d2gm(2,2)+6*gm(1,2)*d2gm(2,3))+gm(2,2)&
&   *(gm(1,3)*(7.5d0*dgm01(3,3)*dgm10(2,2)+12*dgm01(2,3)*dgm10(2,3)&
&   +7.5d0*dgm01(2,2)*dgm10(3,3))+27*gm(1,2)*(dgm01(3,3)*dgm10(2,3)&
&   +dgm01(2,3)*dgm10(3,3))+gm(3,3)*(39*dgm01(2,3)*dgm10(1,2)+19.5d0*dgm01(2,2)&
&   *dgm10(1,3)+19.5d0*dgm01(1,3)*dgm10(2,2)+39*dgm01(1,2)*dgm10(2,3)&
&   -4.5d0*gm(1,3)*d2gm(2,2)+24*gm(1,2)*d2gm(2,3)))+gm(2,2)**2*(18*dgm01(3,3)&
&   *dgm10(1,3)+18*dgm01(1,3)*dgm10(3,3)+12*gm(3,3)*d2gm(1,3)+6*gm(1,3)&
&   *d2gm(3,3))+gm(2,3)*(9*gm(1,3)*dgm01(2,3)*dgm10(2,2)+15*gm(1,2)&
&   *dgm01(3,3)*dgm10(2,2)+9*gm(1,3)*dgm01(2,2)*dgm10(2,3)+24*gm(1,2)&
&   *dgm01(2,3)*dgm10(2,3)+15*gm(1,2)*dgm01(2,2)*dgm10(3,3)+gm(3,3)&
&   *(27*dgm01(2,2)*dgm10(1,2)+27*dgm01(1,2)*dgm10(2,2)+15*gm(2,2)&
&   *d2gm(1,2)+12*gm(1,2)*d2gm(2,2))+gm(2,2)*(18*dgm01(3,3)*dgm10(1,2)&
&   +18*dgm01(2,3)*dgm10(1,3)+18*dgm01(1,3)*dgm10(2,3)+18*dgm01(1,2)&
&   *dgm10(3,3)+6*gm(1,3)*d2gm(2,3)+3*gm(1,2)*d2gm(3,3)))
   cm(10,12)=-3*gm(1,3)*gm(2,2)*dgm01(3,3)*dgm10(3,3)+gm(2,3)*(-6*gm(1,2)&
&   *dgm01(3,3)*dgm10(3,3)+33*gm(1,3)*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)&
&   *dgm10(3,3)))+gm(3,3)**2*(0*dgm01(2,3)*dgm10(1,2)+0*dgm01(2,2)&
&   *dgm10(1,3)+0*dgm01(1,3)*dgm10(2,2)+0*dgm01(1,2)*dgm10(2,3)+6*gm(2,3)&
&   *d2gm(1,2)-3*gm(2,2)*d2gm(1,3)+3*gm(1,3)*d2gm(2,2)-6*gm(1,2)&
&   *d2gm(2,3))+gm(2,3)**2*(16.5d0*dgm01(3,3)*dgm10(1,3)+16.5d0*dgm01(1,3)&
&   *dgm10(3,3)+7.5d0*gm(1,3)*d2gm(3,3))+gm(3,3)*(4.5d0*gm(1,3)*dgm01(3,3)&
&   *dgm10(2,2)+36*gm(1,3)*dgm01(2,3)*dgm10(2,3)-15*gm(1,2)*dgm01(3,3)&
&   *dgm10(2,3)+4.5d0*gm(1,3)*dgm01(2,2)*dgm10(3,3)-15*gm(1,2)*dgm01(2,3)&
&   *dgm10(3,3)+9*gm(2,3)**2*d2gm(1,3)+gm(2,3)*(9*dgm01(3,3)*dgm10(1,2)&
&   +36*dgm01(2,3)*dgm10(1,3)+36*dgm01(1,3)*dgm10(2,3)+9*dgm01(1,2)&
&   *dgm10(3,3)+18*gm(1,3)*d2gm(2,3)-3*gm(1,2)*d2gm(3,3))+gm(2,2)&
&   *(-7.5d0*dgm01(3,3)*dgm10(1,3)-7.5d0*dgm01(1,3)*dgm10(3,3)-1.5d0*gm(1,3)&
&   *d2gm(3,3)))
   cm(1,13)=5*gm(1,2)**3*d2gm(1,3)+gm(1,2)*(gm(1,1)*(12*dgm01(2,3)&
&   *dgm10(1,2)+6*dgm01(2,2)*dgm10(1,3)+6*dgm01(1,3)*dgm10(2,2)+12*dgm01(1,2)&
&   *dgm10(2,3))-6*gm(2,3)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2)&
&   +gm(1,1)*d2gm(1,2))-3*gm(2,2)*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)&
&   *dgm10(1,3)+gm(1,1)*d2gm(1,3)))+gm(1,3)*(6*gm(1,1)*(dgm01(2,2)&
&   *dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+15*gm(1,2)**2*d2gm(1,2)-3*gm(2,2)&
&   *(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2)+gm(1,1)*d2gm(1,2))&
&   +gm(1,2)*(9*dgm01(2,2)*dgm10(1,1)+60*dgm01(1,2)*dgm10(1,2)+9*dgm01(1,1)&
&   *dgm10(2,2)+9*gm(1,1)*d2gm(2,2)))+gm(1,2)**2*(9*dgm01(2,3)*dgm10(1,1)&
&   +30*dgm01(1,3)*dgm10(1,2)+30*dgm01(1,2)*dgm10(1,3)+9*dgm01(1,1)&
&   *dgm10(2,3)+9*gm(1,1)*d2gm(2,3))+gm(1,1)*(-6*gm(1,1)*(dgm01(2,3)&
&   *dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+gm(2,3)*(-6*dgm01(2,2)*dgm10(1,1)&
&   -12*dgm01(1,2)*dgm10(1,2)-6*dgm01(1,1)*dgm10(2,2)-3*gm(1,1)*d2gm(2,2))&
&   +gm(2,2)*(-6*dgm01(2,3)*dgm10(1,1)-6*dgm01(1,3)*dgm10(1,2)-6*dgm01(1,2)&
&   *dgm10(1,3)-6*dgm01(1,1)*dgm10(2,3)-3*gm(1,1)*d2gm(2,3)))
   cm(2,13)=24*gm(1,1)*gm(2,3)*dgm01(2,2)*dgm10(2,2)+gm(1,2)*(6*gm(1,3)&
&   *dgm01(2,2)*dgm10(2,2)+12*gm(2,3)*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)&
&   *dgm10(2,2)))+gm(1,2)**2*(9*dgm01(2,3)*dgm10(2,2)+9*dgm01(2,2)&
&   *dgm10(2,3)+3*gm(2,3)*d2gm(2,2))+gm(2,2)**2*(12*dgm01(2,3)*dgm10(1,1)&
&   +18*dgm01(1,3)*dgm10(1,2)+18*dgm01(1,2)*dgm10(1,3)+12*dgm01(1,1)&
&   *dgm10(2,3)+12*gm(1,3)*d2gm(1,2)+6*gm(1,2)*d2gm(1,3)+12*gm(1,1)&
&   *d2gm(2,3))+gm(2,2)*(18*gm(1,2)*dgm01(2,3)*dgm10(1,2)+15*gm(1,2)&
&   *dgm01(2,2)*dgm10(1,3)+15*gm(1,2)*dgm01(1,3)*dgm10(2,2)+36*gm(1,1)&
&   *dgm01(2,3)*dgm10(2,2)+18*gm(1,2)*dgm01(1,2)*dgm10(2,3)+36*gm(1,1)&
&   *dgm01(2,2)*dgm10(2,3)+gm(2,3)*(12*dgm01(2,2)*dgm10(1,1)+12*dgm01(1,2)&
&   *dgm10(1,2)+12*dgm01(1,1)*dgm10(2,2)+6*gm(1,2)*d2gm(1,2)+12*gm(1,1)&
&   *d2gm(2,2))+gm(1,3)*(27*dgm01(2,2)*dgm10(1,2)+27*dgm01(1,2)*dgm10(2,2)&
&   +3*gm(1,2)*d2gm(2,2))+6*gm(1,2)**2*d2gm(2,3))
   cm(3,13)=-6*gm(1,3)*gm(3,3)*dgm01(2,2)*dgm10(1,2)+12*gm(1,2)&
&   *gm(3,3)*dgm01(2,3)*dgm10(1,2)+0*gm(1,2)*gm(3,3)*dgm01(2,2)*dgm10(1,3)&
&   -6*gm(1,3)*gm(3,3)*dgm01(1,2)*dgm10(2,2)+0*gm(1,2)*gm(3,3)*dgm01(1,3)&
&   *dgm10(2,2)-3*gm(1,3)**2*dgm01(2,3)*dgm10(2,2)+9*gm(1,1)*gm(3,3)&
&   *dgm01(2,3)*dgm10(2,2)+3*gm(1,2)*gm(1,3)*dgm01(3,3)*dgm10(2,2)&
&   +12*gm(1,2)*gm(3,3)*dgm01(1,2)*dgm10(2,3)-3*gm(1,3)**2*dgm01(2,2)&
&   *dgm10(2,3)+9*gm(1,1)*gm(3,3)*dgm01(2,2)*dgm10(2,3)+36*gm(1,2)&
&   *gm(1,3)*dgm01(2,3)*dgm10(2,3)-6*gm(1,2)**2*dgm01(3,3)*dgm10(2,3)&
&   +3*gm(1,2)*gm(1,3)*dgm01(2,2)*dgm10(3,3)-6*gm(1,2)**2*dgm01(2,3)&
&   *dgm10(3,3)+3*gm(1,2)*gm(1,3)*gm(3,3)*d2gm(2,2)-6*gm(1,2)**2*gm(3,3)&
&   *d2gm(2,3)+gm(2,3)**2*(15*dgm01(2,3)*dgm10(1,1)+18*dgm01(1,3)&
&   *dgm10(1,2)+18*dgm01(1,2)*dgm10(1,3)+15*dgm01(1,1)*dgm10(2,3)&
&   +3*gm(1,3)*d2gm(1,2)+15*gm(1,2)*d2gm(1,3)+15*gm(1,1)*d2gm(2,3))&
&   +gm(2,2)*(gm(1,3)*(-9*dgm01(3,3)*dgm10(1,2)-18*dgm01(2,3)*dgm10(1,3)&
&   -18*dgm01(1,3)*dgm10(2,3)-9*dgm01(1,2)*dgm10(3,3))-3*(gm(1,2)&
&   *(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+gm(1,1)*(dgm01(3,3)&
&   *dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))-6*gm(1,3)**2*d2gm(2,3)+gm(3,3)&
&   *(-3*dgm01(2,3)*dgm10(1,1)-12*dgm01(1,3)*dgm10(1,2)-12*dgm01(1,2)&
&   *dgm10(1,3)-3*dgm01(1,1)*dgm10(2,3)-9*gm(1,3)*d2gm(1,2)-3*gm(1,2)&
&   *d2gm(1,3)-3*gm(1,1)*d2gm(2,3)))+gm(2,3)*(24*gm(1,2)*dgm01(3,3)&
&   *dgm10(1,2)-12*gm(2,2)*dgm01(1,3)*dgm10(1,3)+48*gm(1,2)*dgm01(2,3)&
&   *dgm10(1,3)+12*gm(1,1)*dgm01(3,3)*dgm10(2,2)+48*gm(1,2)*dgm01(1,3)&
&   *dgm10(2,3)+60*gm(1,1)*dgm01(2,3)*dgm10(2,3)+24*gm(1,2)*dgm01(1,2)&
&   *dgm10(3,3)+12*gm(1,1)*dgm01(2,2)*dgm10(3,3)+3*gm(1,3)**2*d2gm(2,2)&
&   +gm(3,3)*(12*dgm01(2,2)*dgm10(1,1)+48*dgm01(1,2)*dgm10(1,2)+12*dgm01(1,1)&
&   *dgm10(2,2)+24*gm(1,2)*d2gm(1,2)+12*gm(1,1)*d2gm(2,2))+gm(1,3)&
&   *(24*dgm01(2,3)*dgm10(1,2)+0*dgm01(2,2)*dgm10(1,3)+0*dgm01(1,3)&
&   *dgm10(2,2)+24*dgm01(1,2)*dgm10(2,3)-6*gm(2,2)*d2gm(1,3)+18*gm(1,2)&
&   *d2gm(2,3)))
   cm(4,13)=42*gm(1,2)*gm(3,3)*dgm01(2,2)*dgm10(1,2)+42*gm(1,2)&
&   *gm(3,3)*dgm01(1,2)*dgm10(2,2)+18*gm(1,3)**2*dgm01(2,2)*dgm10(2,2)&
&   +30*gm(1,1)*gm(3,3)*dgm01(2,2)*dgm10(2,2)+0*gm(1,2)*gm(1,3)*dgm01(2,3)&
&   *dgm10(2,2)+9*gm(1,2)**2*dgm01(3,3)*dgm10(2,2)+0*gm(1,2)*gm(1,3)&
&   *dgm01(2,2)*dgm10(2,3)+12*gm(1,2)**2*dgm01(2,3)*dgm10(2,3)+9*gm(1,2)&
&   **2*dgm01(2,2)*dgm10(3,3)+gm(2,2)**2*(-12*dgm01(1,3)*dgm10(1,3)&
&   -6*gm(1,3)*d2gm(1,3))+9*gm(1,2)**2*gm(3,3)*d2gm(2,2)+gm(2,3)&
&   **2*(9*dgm01(2,2)*dgm10(1,1)+12*dgm01(1,2)*dgm10(1,2)+9*dgm01(1,1)&
&   *dgm10(2,2)+6*gm(1,2)*d2gm(1,2)+9*gm(1,1)*d2gm(2,2))+gm(2,3)&
&   *(24*gm(1,2)*dgm01(2,3)*dgm10(1,2)+12*gm(1,2)*dgm01(2,2)*dgm10(1,3)&
&   +12*gm(1,2)*dgm01(1,3)*dgm10(2,2)+42*gm(1,1)*dgm01(2,3)*dgm10(2,2)&
&   +24*gm(1,2)*dgm01(1,2)*dgm10(2,3)+42*gm(1,1)*dgm01(2,2)*dgm10(2,3)&
&   +gm(1,3)*(0*dgm01(2,2)*dgm10(1,2)+0*dgm01(1,2)*dgm10(2,2)-6*gm(1,2)&
&   *d2gm(2,2))+6*gm(1,2)**2*d2gm(2,3)+gm(2,2)*(24*dgm01(2,3)*dgm10(1,1)&
&   +24*dgm01(1,3)*dgm10(1,2)+24*dgm01(1,2)*dgm10(1,3)+24*dgm01(1,1)&
&   *dgm10(2,3)+6*gm(1,3)*d2gm(1,2)+18*gm(1,2)*d2gm(1,3)+24*gm(1,1)&
&   *d2gm(2,3)))+gm(2,2)*(24*gm(1,2)*dgm01(3,3)*dgm10(1,2)+24*gm(1,2)&
&   *dgm01(2,3)*dgm10(1,3)+15*gm(1,1)*dgm01(3,3)*dgm10(2,2)+24*gm(1,2)&
&   *dgm01(1,3)*dgm10(2,3)+48*gm(1,1)*dgm01(2,3)*dgm10(2,3)+24*gm(1,2)&
&   *dgm01(1,2)*dgm10(3,3)+15*gm(1,1)*dgm01(2,2)*dgm10(3,3)+9*gm(1,3)&
&   **2*d2gm(2,2)+gm(3,3)*(15*dgm01(2,2)*dgm10(1,1)+48*dgm01(1,2)&
&   *dgm10(1,2)+15*dgm01(1,1)*dgm10(2,2)+24*gm(1,2)*d2gm(1,2)+15*gm(1,1)&
&   *d2gm(2,2))+gm(1,3)*(12*dgm01(2,3)*dgm10(1,2)+6*dgm01(2,2)*dgm10(1,3)&
&   +6*dgm01(1,3)*dgm10(2,2)+12*dgm01(1,2)*dgm10(2,3)+6*gm(1,2)*d2gm(2,3)))
   cm(5,13)=12*gm(1,2)*gm(3,3)*dgm01(2,2)*dgm10(1,1)+60*gm(1,2)&
&   *gm(3,3)*dgm01(1,2)*dgm10(1,2)-3*gm(1,3)**2*dgm01(2,2)*dgm10(1,2)&
&   +9*gm(1,1)*gm(3,3)*dgm01(2,2)*dgm10(1,2)+24*gm(1,2)*gm(1,3)*dgm01(2,3)&
&   *dgm10(1,2)+15*gm(1,2)**2*dgm01(3,3)*dgm10(1,2)+0*gm(1,2)*gm(1,3)&
&   *dgm01(2,2)*dgm10(1,3)+18*gm(1,2)**2*dgm01(2,3)*dgm10(1,3)+12*gm(1,2)&
&   *gm(3,3)*dgm01(1,1)*dgm10(2,2)-3*gm(1,3)**2*dgm01(1,2)*dgm10(2,2)&
&   +9*gm(1,1)*gm(3,3)*dgm01(1,2)*dgm10(2,2)+0*gm(1,2)*gm(1,3)*dgm01(1,3)&
&   *dgm10(2,2)-6*gm(1,1)*gm(1,3)*dgm01(2,3)*dgm10(2,2)+12*gm(1,1)&
&   *gm(1,2)*dgm01(3,3)*dgm10(2,2)+24*gm(1,2)*gm(1,3)*dgm01(1,2)&
&   *dgm10(2,3)+18*gm(1,2)**2*dgm01(1,3)*dgm10(2,3)-6*gm(1,1)*gm(1,3)&
&   *dgm01(2,2)*dgm10(2,3)+48*gm(1,1)*gm(1,2)*dgm01(2,3)*dgm10(2,3)&
&   +15*gm(1,2)**2*dgm01(1,2)*dgm10(3,3)+12*gm(1,1)*gm(1,2)*dgm01(2,2)&
&   *dgm10(3,3)+15*gm(1,2)**2*gm(3,3)*d2gm(1,2)-6*gm(2,3)**2*(dgm01(1,2)&
&   *dgm10(1,1)+dgm01(1,1)*dgm10(1,2)+gm(1,1)*d2gm(1,2))+3*gm(1,2)&
&   *gm(1,3)**2*d2gm(2,2)+12*gm(1,1)*gm(1,2)*gm(3,3)*d2gm(2,2)+3*gm(1,2)&
&   **2*gm(1,3)*d2gm(2,3)+gm(2,3)*(24*gm(1,2)*dgm01(2,3)*dgm10(1,1)&
&   +48*gm(1,2)*dgm01(1,3)*dgm10(1,2)+12*gm(1,1)*dgm01(2,3)*dgm10(1,2)&
&   +48*gm(1,2)*dgm01(1,2)*dgm10(1,3)+0*gm(1,1)*dgm01(2,2)*dgm10(1,3)&
&   +0*gm(1,1)*dgm01(1,3)*dgm10(2,2)+24*gm(1,2)*dgm01(1,1)*dgm10(2,3)&
&   +12*gm(1,1)*dgm01(1,2)*dgm10(2,3)+15*gm(1,2)**2*d2gm(1,3)-3*gm(2,2)&
&   *(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3)+gm(1,1)*d2gm(1,3))&
&   +gm(1,3)*(3*dgm01(2,2)*dgm10(1,1)+36*dgm01(1,2)*dgm10(1,2)+3*dgm01(1,1)&
&   *dgm10(2,2)+18*gm(1,2)*d2gm(1,2)+3*gm(1,1)*d2gm(2,2))+24*gm(1,1)&
&   *gm(1,2)*d2gm(2,3))+gm(2,2)*(-3*gm(1,1)*dgm01(3,3)*dgm10(1,2)&
&   -12*gm(1,2)*dgm01(1,3)*dgm10(1,3)-12*gm(1,1)*dgm01(2,3)*dgm10(1,3)&
&   -12*gm(1,1)*dgm01(1,3)*dgm10(2,3)-3*gm(1,1)*dgm01(1,2)*dgm10(3,3)&
&   -6*gm(1,3)**2*d2gm(1,2)-3*gm(3,3)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)&
&   *dgm10(1,2)+gm(1,1)*d2gm(1,2))+gm(1,3)*(-9*dgm01(2,3)*dgm10(1,1)&
&   -18*dgm01(1,3)*dgm10(1,2)-18*dgm01(1,2)*dgm10(1,3)-9*dgm01(1,1)&
&   *dgm10(2,3)-6*gm(1,2)*d2gm(1,3)-9*gm(1,1)*d2gm(2,3)))
   cm(6,13)=gm(1,1)*(24*gm(1,3)*dgm01(2,2)*dgm10(2,2)-6*gm(2,3)&
&   *(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2)))-3*gm(2,2)**2*(dgm01(1,3)&
&   *dgm10(1,1)+dgm01(1,1)*dgm10(1,3)+gm(1,1)*d2gm(1,3))+gm(1,2)&
&   **2*(12*dgm01(2,3)*dgm10(1,2)+12*dgm01(2,2)*dgm10(1,3)+12*dgm01(1,3)&
&   *dgm10(2,2)+12*dgm01(1,2)*dgm10(2,3)+3*gm(2,3)*d2gm(1,2)+3*gm(1,3)&
&   *d2gm(2,2))+gm(1,2)*(30*gm(1,3)*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)&
&   *dgm10(2,2))+18*gm(1,1)*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))&
&   +gm(2,3)*(3*dgm01(2,2)*dgm10(1,1)+12*dgm01(1,2)*dgm10(1,2)+3*dgm01(1,1)&
&   *dgm10(2,2)+3*gm(1,1)*d2gm(2,2)))+3*gm(1,2)**3*d2gm(2,3)+gm(2,2)&
&   *(15*gm(1,2)*dgm01(2,3)*dgm10(1,1)+42*gm(1,2)*dgm01(1,3)*dgm10(1,2)&
&   +6*gm(1,1)*dgm01(2,3)*dgm10(1,2)+42*gm(1,2)*dgm01(1,2)*dgm10(1,3)&
&   +6*gm(1,1)*dgm01(2,2)*dgm10(1,3)+6*gm(1,1)*dgm01(1,3)*dgm10(2,2)&
&   +15*gm(1,2)*dgm01(1,1)*dgm10(2,3)+6*gm(1,1)*dgm01(1,2)*dgm10(2,3)&
&   -9*gm(2,3)*(dgm01(1,2)*dgm10(1,1)+dgm01(1,1)*dgm10(1,2)+gm(1,1)&
&   *d2gm(1,2))+9*gm(1,2)**2*d2gm(1,3)+gm(1,3)*(12*dgm01(2,2)*dgm10(1,1)&
&   +48*dgm01(1,2)*dgm10(1,2)+12*dgm01(1,1)*dgm10(2,2)+24*gm(1,2)&
&   *d2gm(1,2)+12*gm(1,1)*d2gm(2,2))+15*gm(1,1)*gm(1,2)*d2gm(2,3))
   cm(7,13)=18*gm(1,2)*gm(2,3)*dgm01(2,2)*dgm10(2,2)+2*gm(2,2)&
&   **3*d2gm(1,3)+gm(2,2)*(-12*gm(1,3)*dgm01(2,2)*dgm10(2,2)+21*gm(1,2)&
&   *(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+gm(2,3)*(21*dgm01(2,2)&
&   *dgm10(1,2)+21*dgm01(1,2)*dgm10(2,2)+9*gm(1,2)*d2gm(2,2)))+gm(2,2)&
&   **2*(12*dgm01(2,3)*dgm10(1,2)+3*dgm01(2,2)*dgm10(1,3)+3*dgm01(1,3)&
&   *dgm10(2,2)+12*dgm01(1,2)*dgm10(2,3)+6*gm(2,3)*d2gm(1,2)-3*gm(1,3)&
&   *d2gm(2,2)+6*gm(1,2)*d2gm(2,3))
   cm(8,13)=gm(3,3)*(24*gm(1,3)*dgm01(2,2)*dgm10(2,2)-6*gm(1,2)&
&   *(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3)))+3*gm(2,3)**3*d2gm(1,2)&
&   -3*gm(2,2)**2*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3)+gm(3,3)&
&   *d2gm(1,3))+gm(2,3)**2*(12*dgm01(2,3)*dgm10(1,2)+12*dgm01(2,2)&
&   *dgm10(1,3)+12*dgm01(1,3)*dgm10(2,2)+12*dgm01(1,2)*dgm10(2,3)&
&   +9*gm(2,2)*d2gm(1,3)+3*gm(1,3)*d2gm(2,2)+3*gm(1,2)*d2gm(2,3))&
&   +gm(2,2)*(gm(1,3)*(12*dgm01(3,3)*dgm10(2,2)+48*dgm01(2,3)*dgm10(2,3)&
&   +12*dgm01(2,2)*dgm10(3,3))-9*gm(1,2)*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)&
&   *dgm10(3,3))+gm(3,3)*(6*dgm01(2,3)*dgm10(1,2)+6*dgm01(2,2)*dgm10(1,3)&
&   +6*dgm01(1,3)*dgm10(2,2)+6*dgm01(1,2)*dgm10(2,3)+12*gm(1,3)*d2gm(2,2)&
&   -9*gm(1,2)*d2gm(2,3)))+gm(2,3)*(30*gm(1,3)*dgm01(2,3)*dgm10(2,2)&
&   +3*gm(1,2)*dgm01(3,3)*dgm10(2,2)+30*gm(1,3)*dgm01(2,2)*dgm10(2,3)&
&   +12*gm(1,2)*dgm01(2,3)*dgm10(2,3)+3*gm(1,2)*dgm01(2,2)*dgm10(3,3)&
&   +gm(3,3)*(18*dgm01(2,2)*dgm10(1,2)+18*dgm01(1,2)*dgm10(2,2)+15*gm(2,2)&
&   *d2gm(1,2)+3*gm(1,2)*d2gm(2,2))+gm(2,2)*(15*dgm01(3,3)*dgm10(1,2)&
&   +42*dgm01(2,3)*dgm10(1,3)+42*dgm01(1,3)*dgm10(2,3)+15*dgm01(1,2)&
&   *dgm10(3,3)+24*gm(1,3)*d2gm(2,3)))
   cm(9,13)=24*gm(1,2)*gm(3,3)*dgm01(2,2)*dgm10(2,2)+gm(2,3)**2*(9*dgm01(2,2)&
&   *dgm10(1,2)+9*dgm01(1,2)*dgm10(2,2)+6*gm(2,2)*d2gm(1,2)+3*gm(1,2)&
&   *d2gm(2,2))+gm(2,2)*(27*gm(1,3)*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)&
&   *dgm10(2,3))+12*gm(1,2)*(dgm01(3,3)*dgm10(2,2)+dgm01(2,3)*dgm10(2,3)&
&   +dgm01(2,2)*dgm10(3,3))+gm(3,3)*(36*dgm01(2,2)*dgm10(1,2)+36*dgm01(1,2)&
&   *dgm10(2,2)+12*gm(1,2)*d2gm(2,2)))+gm(2,2)**2*(12*dgm01(3,3)&
&   *dgm10(1,2)+18*dgm01(2,3)*dgm10(1,3)+18*dgm01(1,3)*dgm10(2,3)&
&   +12*dgm01(1,2)*dgm10(3,3)+12*gm(3,3)*d2gm(1,2)+12*gm(1,3)*d2gm(2,3))&
&   +gm(2,3)*(6*gm(1,3)*dgm01(2,2)*dgm10(2,2)+12*gm(1,2)*(dgm01(2,3)&
&   *dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+6*gm(2,2)**2*d2gm(1,3)+gm(2,2)&
&   *(18*dgm01(2,3)*dgm10(1,2)+15*dgm01(2,2)*dgm10(1,3)+15*dgm01(1,3)&
&   *dgm10(2,2)+18*dgm01(1,2)*dgm10(2,3)+3*gm(1,3)*d2gm(2,2)+6*gm(1,2)&
&   *d2gm(2,3)))
   cm(10,13)=-3*gm(1,3)*gm(2,2)*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)&
&   *dgm10(3,3))+gm(2,3)*(-3*gm(2,2)*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)&
&   *dgm10(3,3))+gm(1,3)*(9*dgm01(3,3)*dgm10(2,2)+60*dgm01(2,3)*dgm10(2,3)&
&   +9*dgm01(2,2)*dgm10(3,3))-6*gm(1,2)*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)&
&   *dgm10(3,3)))+5*gm(2,3)**3*d2gm(1,3)+gm(3,3)**2*(-6*dgm01(2,2)&
&   *dgm10(1,2)-6*dgm01(1,2)*dgm10(2,2)-3*(gm(2,2)*d2gm(1,2)+gm(1,2)&
&   *d2gm(2,2)))+gm(2,3)**2*(9*dgm01(3,3)*dgm10(1,2)+30*dgm01(2,3)&
&   *dgm10(1,3)+30*dgm01(1,3)*dgm10(2,3)+9*dgm01(1,2)*dgm10(3,3)&
&   +15*gm(1,3)*d2gm(2,3))+gm(3,3)*(6*gm(1,3)*dgm01(2,3)*dgm10(2,2)&
&   -6*gm(1,2)*dgm01(3,3)*dgm10(2,2)+6*gm(1,3)*dgm01(2,2)*dgm10(2,3)&
&   -12*gm(1,2)*dgm01(2,3)*dgm10(2,3)-6*gm(1,2)*dgm01(2,2)*dgm10(3,3)&
&   +9*gm(2,3)**2*d2gm(1,2)+gm(2,3)*(12*dgm01(2,3)*dgm10(1,2)+6*dgm01(2,2)&
&   *dgm10(1,3)+6*dgm01(1,3)*dgm10(2,2)+12*dgm01(1,2)*dgm10(2,3)&
&   -3*gm(2,2)*d2gm(1,3)+9*gm(1,3)*d2gm(2,2)-6*gm(1,2)*d2gm(2,3))&
&   +gm(2,2)*(-6*dgm01(3,3)*dgm10(1,2)-6*dgm01(2,3)*dgm10(1,3)-6*dgm01(1,3)&
&   *dgm10(2,3)-6*dgm01(1,2)*dgm10(3,3)-3*gm(1,3)*d2gm(2,3)))
   cm(1,14)=5*gm(1,3)**3*d2gm(1,2)+gm(1,2)*(6*gm(1,1)*(dgm01(3,3)&
&   *dgm10(1,3)+dgm01(1,3)*dgm10(3,3))-3*gm(3,3)*(dgm01(1,3)*dgm10(1,1)&
&   +dgm01(1,1)*dgm10(1,3)+gm(1,1)*d2gm(1,3)))+gm(1,3)**2*(9*dgm01(2,3)&
&   *dgm10(1,1)+30*dgm01(1,3)*dgm10(1,2)+30*dgm01(1,2)*dgm10(1,3)&
&   +9*dgm01(1,1)*dgm10(2,3)+15*gm(1,2)*d2gm(1,3)+9*gm(1,1)*d2gm(2,3))&
&   +gm(1,3)*(9*gm(1,2)*dgm01(3,3)*dgm10(1,1)+6*gm(1,1)*dgm01(3,3)&
&   *dgm10(1,2)+60*gm(1,2)*dgm01(1,3)*dgm10(1,3)+12*gm(1,1)*dgm01(2,3)&
&   *dgm10(1,3)+12*gm(1,1)*dgm01(1,3)*dgm10(2,3)+9*gm(1,2)*dgm01(1,1)&
&   *dgm10(3,3)+6*gm(1,1)*dgm01(1,2)*dgm10(3,3)-3*gm(3,3)*(dgm01(1,2)&
&   *dgm10(1,1)+dgm01(1,1)*dgm10(1,2)+gm(1,1)*d2gm(1,2))-6*gm(2,3)&
&   *(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3)+gm(1,1)*d2gm(1,3))&
&   +9*gm(1,1)*gm(1,2)*d2gm(3,3))+gm(1,1)*(-6*gm(1,1)*(dgm01(3,3)&
&   *dgm10(2,3)+dgm01(2,3)*dgm10(3,3))+gm(3,3)*(-6*dgm01(2,3)*dgm10(1,1)&
&   -6*dgm01(1,3)*dgm10(1,2)-6*dgm01(1,2)*dgm10(1,3)-6*dgm01(1,1)&
&   *dgm10(2,3)-3*gm(1,1)*d2gm(2,3))+gm(2,3)*(-6*dgm01(3,3)*dgm10(1,1)&
&   -12*dgm01(1,3)*dgm10(1,3)-6*dgm01(1,1)*dgm10(3,3)-3*gm(1,1)*d2gm(3,3)))
   cm(2,14)=-3*gm(1,3)*gm(3,3)*dgm01(2,2)*dgm10(1,2)-18*gm(1,2)&
&   *gm(3,3)*dgm01(2,3)*dgm10(1,2)-9*gm(1,2)*gm(3,3)*dgm01(2,2)*dgm10(1,3)&
&   -3*gm(1,3)*gm(3,3)*dgm01(1,2)*dgm10(2,2)-9*gm(1,2)*gm(3,3)*dgm01(1,3)&
&   *dgm10(2,2)-6*gm(1,3)**2*dgm01(2,3)*dgm10(2,2)-3*gm(1,1)*gm(3,3)&
&   *dgm01(2,3)*dgm10(2,2)+3*gm(1,2)*gm(1,3)*dgm01(3,3)*dgm10(2,2)&
&   -18*gm(1,2)*gm(3,3)*dgm01(1,2)*dgm10(2,3)-6*gm(1,3)**2*dgm01(2,2)&
&   *dgm10(2,3)-3*gm(1,1)*gm(3,3)*dgm01(2,2)*dgm10(2,3)+36*gm(1,2)&
&   *gm(1,3)*dgm01(2,3)*dgm10(2,3)-3*gm(1,2)**2*dgm01(3,3)*dgm10(2,3)&
&   +3*gm(1,2)*gm(1,3)*dgm01(2,2)*dgm10(3,3)-3*gm(1,2)**2*dgm01(2,3)&
&   *dgm10(3,3)-6*gm(1,2)**2*gm(3,3)*d2gm(2,3)+gm(2,3)**2*(15*dgm01(2,3)&
&   *dgm10(1,1)+18*dgm01(1,3)*dgm10(1,2)+18*dgm01(1,2)*dgm10(1,3)&
&   +15*dgm01(1,1)*dgm10(2,3)+15*gm(1,3)*d2gm(1,2)+3*gm(1,2)*d2gm(1,3)&
&   +15*gm(1,1)*d2gm(2,3))+gm(2,3)*(48*gm(1,3)*dgm01(2,3)*dgm10(1,2)&
&   +0*gm(1,2)*dgm01(3,3)*dgm10(1,2)+24*gm(1,3)*dgm01(2,2)*dgm10(1,3)&
&   +24*gm(1,2)*dgm01(2,3)*dgm10(1,3)+24*gm(1,3)*dgm01(1,3)*dgm10(2,2)&
&   +12*gm(1,1)*dgm01(3,3)*dgm10(2,2)+48*gm(1,3)*dgm01(1,2)*dgm10(2,3)&
&   +24*gm(1,2)*dgm01(1,3)*dgm10(2,3)+60*gm(1,1)*dgm01(2,3)*dgm10(2,3)&
&   +0*gm(1,2)*dgm01(1,2)*dgm10(3,3)+12*gm(1,1)*dgm01(2,2)*dgm10(3,3)&
&   +gm(3,3)*(-12*dgm01(1,2)*dgm10(1,2)-6*gm(1,2)*d2gm(1,2))+18*gm(1,2)&
&   *gm(1,3)*d2gm(2,3)+3*gm(1,2)**2*d2gm(3,3)+gm(2,2)*(12*dgm01(3,3)&
&   *dgm10(1,1)+48*dgm01(1,3)*dgm10(1,3)+12*dgm01(1,1)*dgm10(3,3)&
&   +24*gm(1,3)*d2gm(1,3)+12*gm(1,1)*d2gm(3,3)))+gm(2,2)*(-6*gm(1,2)&
&   *dgm01(3,3)*dgm10(1,3)+9*gm(1,1)*dgm01(3,3)*dgm10(2,3)-6*gm(1,2)&
&   *dgm01(1,3)*dgm10(3,3)+9*gm(1,1)*dgm01(2,3)*dgm10(3,3)-6*gm(1,3)&
&   **2*d2gm(2,3)+gm(3,3)*(-3*dgm01(2,3)*dgm10(1,1)-12*dgm01(1,3)&
&   *dgm10(1,2)-12*dgm01(1,2)*dgm10(1,3)-3*dgm01(1,1)*dgm10(2,3)&
&   -3*gm(1,3)*d2gm(1,2)-9*gm(1,2)*d2gm(1,3)-3*gm(1,1)*d2gm(2,3))&
&   +gm(1,3)*(0*dgm01(3,3)*dgm10(1,2)+12*dgm01(2,3)*dgm10(1,3)+12*dgm01(1,3)&
&   *dgm10(2,3)+0*dgm01(1,2)*dgm10(3,3)+3*gm(1,2)*d2gm(3,3)))
   cm(3,14)=24*gm(1,1)*gm(2,3)*dgm01(3,3)*dgm10(3,3)+gm(1,3)*(6*gm(1,2)&
&   *dgm01(3,3)*dgm10(3,3)+12*gm(2,3)*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)&
&   *dgm10(3,3)))+gm(3,3)**2*(12*dgm01(2,3)*dgm10(1,1)+18*dgm01(1,3)&
&   *dgm10(1,2)+18*dgm01(1,2)*dgm10(1,3)+12*dgm01(1,1)*dgm10(2,3)&
&   +6*gm(1,3)*d2gm(1,2)+12*gm(1,2)*d2gm(1,3)+12*gm(1,1)*d2gm(2,3))&
&   +gm(1,3)**2*(9*dgm01(3,3)*dgm10(2,3)+9*dgm01(2,3)*dgm10(3,3)&
&   +3*gm(2,3)*d2gm(3,3))+gm(3,3)*(27*gm(1,2)*dgm01(3,3)*dgm10(1,3)&
&   +36*gm(1,1)*dgm01(3,3)*dgm10(2,3)+27*gm(1,2)*dgm01(1,3)*dgm10(3,3)&
&   +36*gm(1,1)*dgm01(2,3)*dgm10(3,3)+6*gm(1,3)**2*d2gm(2,3)+gm(2,3)&
&   *(12*dgm01(3,3)*dgm10(1,1)+12*dgm01(1,3)*dgm10(1,3)+12*dgm01(1,1)&
&   *dgm10(3,3)+6*gm(1,3)*d2gm(1,3)+12*gm(1,1)*d2gm(3,3))+gm(1,3)&
&   *(15*dgm01(3,3)*dgm10(1,2)+18*dgm01(2,3)*dgm10(1,3)+18*dgm01(1,3)&
&   *dgm10(2,3)+15*dgm01(1,2)*dgm10(3,3)+3*gm(1,2)*d2gm(3,3)))
   cm(4,14)=-12*gm(3,3)**2*dgm01(1,2)*dgm10(1,2)+24*gm(1,3)*gm(3,3)&
&   *dgm01(2,3)*dgm10(1,2)+6*gm(1,2)*gm(3,3)*dgm01(3,3)*dgm10(1,2)&
&   +24*gm(1,3)*gm(3,3)*dgm01(2,2)*dgm10(1,3)+12*gm(1,2)*gm(3,3)&
&   *dgm01(2,3)*dgm10(1,3)+24*gm(1,3)*gm(3,3)*dgm01(1,3)*dgm10(2,2)&
&   +9*gm(1,3)**2*dgm01(3,3)*dgm10(2,2)+15*gm(1,1)*gm(3,3)*dgm01(3,3)&
&   *dgm10(2,2)+24*gm(1,3)*gm(3,3)*dgm01(1,2)*dgm10(2,3)+12*gm(1,2)&
&   *gm(3,3)*dgm01(1,3)*dgm10(2,3)+12*gm(1,3)**2*dgm01(2,3)*dgm10(2,3)&
&   +48*gm(1,1)*gm(3,3)*dgm01(2,3)*dgm10(2,3)+0*gm(1,2)*gm(1,3)*dgm01(3,3)&
&   *dgm10(2,3)+6*gm(1,2)*gm(3,3)*dgm01(1,2)*dgm10(3,3)+9*gm(1,3)&
&   **2*dgm01(2,2)*dgm10(3,3)+15*gm(1,1)*gm(3,3)*dgm01(2,2)*dgm10(3,3)&
&   +0*gm(1,2)*gm(1,3)*dgm01(2,3)*dgm10(3,3)+18*gm(1,2)**2*dgm01(3,3)&
&   *dgm10(3,3)-6*gm(1,2)*gm(3,3)**2*d2gm(1,2)+6*gm(1,2)*gm(1,3)&
&   *gm(3,3)*d2gm(2,3)+9*gm(1,2)**2*gm(3,3)*d2gm(3,3)+gm(2,3)**2*(9*dgm01(3,3)&
&   *dgm10(1,1)+12*dgm01(1,3)*dgm10(1,3)+9*dgm01(1,1)*dgm10(3,3)&
&   +6*gm(1,3)*d2gm(1,3)+9*gm(1,1)*d2gm(3,3))+gm(2,2)*(30*gm(1,1)&
&   *dgm01(3,3)*dgm10(3,3)+42*gm(1,3)*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)&
&   *dgm10(3,3))+9*gm(1,3)**2*d2gm(3,3)+gm(3,3)*(15*dgm01(3,3)*dgm10(1,1)&
&   +48*dgm01(1,3)*dgm10(1,3)+15*dgm01(1,1)*dgm10(3,3)+24*gm(1,3)&
&   *d2gm(1,3)+15*gm(1,1)*d2gm(3,3)))+gm(2,3)*(0*gm(1,2)*dgm01(3,3)&
&   *dgm10(1,3)+42*gm(1,1)*dgm01(3,3)*dgm10(2,3)+0*gm(1,2)*dgm01(1,3)&
&   *dgm10(3,3)+42*gm(1,1)*dgm01(2,3)*dgm10(3,3)+6*gm(1,3)**2*d2gm(2,3)&
&   +gm(3,3)*(24*dgm01(2,3)*dgm10(1,1)+24*dgm01(1,3)*dgm10(1,2)+24*dgm01(1,2)&
&   *dgm10(1,3)+24*dgm01(1,1)*dgm10(2,3)+18*gm(1,3)*d2gm(1,2)+6*gm(1,2)&
&   *d2gm(1,3)+24*gm(1,1)*d2gm(2,3))+gm(1,3)*(12*dgm01(3,3)*dgm10(1,2)&
&   +24*dgm01(2,3)*dgm10(1,3)+24*dgm01(1,3)*dgm10(2,3)+12*dgm01(1,2)&
&   *dgm10(3,3)-6*gm(1,2)*d2gm(3,3)))
   cm(5,14)=gm(1,1)*(24*gm(1,2)*dgm01(3,3)*dgm10(3,3)-6*gm(2,3)&
&   *(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3)))-3*gm(3,3)**2*(dgm01(1,2)&
&   *dgm10(1,1)+dgm01(1,1)*dgm10(1,2)+gm(1,1)*d2gm(1,2))+3*gm(1,3)&
&   **3*d2gm(2,3)+gm(1,3)**2*(12*dgm01(3,3)*dgm10(1,2)+12*dgm01(2,3)&
&   *dgm10(1,3)+12*dgm01(1,3)*dgm10(2,3)+12*dgm01(1,2)*dgm10(3,3)&
&   +3*gm(2,3)*d2gm(1,3)+3*gm(1,2)*d2gm(3,3))+gm(3,3)*(12*gm(1,2)&
&   *dgm01(3,3)*dgm10(1,1)+6*gm(1,1)*dgm01(3,3)*dgm10(1,2)+48*gm(1,2)&
&   *dgm01(1,3)*dgm10(1,3)+6*gm(1,1)*dgm01(2,3)*dgm10(1,3)+6*gm(1,1)&
&   *dgm01(1,3)*dgm10(2,3)+12*gm(1,2)*dgm01(1,1)*dgm10(3,3)+6*gm(1,1)&
&   *dgm01(1,2)*dgm10(3,3)+9*gm(1,3)**2*d2gm(1,2)-9*gm(2,3)*(dgm01(1,3)&
&   *dgm10(1,1)+dgm01(1,1)*dgm10(1,3)+gm(1,1)*d2gm(1,3))+gm(1,3)&
&   *(15*dgm01(2,3)*dgm10(1,1)+42*dgm01(1,3)*dgm10(1,2)+42*dgm01(1,2)&
&   *dgm10(1,3)+15*dgm01(1,1)*dgm10(2,3)+24*gm(1,2)*d2gm(1,3)+15*gm(1,1)&
&   *d2gm(2,3))+12*gm(1,1)*gm(1,2)*d2gm(3,3))+gm(1,3)*(30*gm(1,2)&
&   *(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+18*gm(1,1)*(dgm01(3,3)&
&   *dgm10(2,3)+dgm01(2,3)*dgm10(3,3))+gm(2,3)*(3*dgm01(3,3)*dgm10(1,1)&
&   +12*dgm01(1,3)*dgm10(1,3)+3*dgm01(1,1)*dgm10(3,3)+3*gm(1,1)*d2gm(3,3)))
   cm(6,14)=-9*gm(1,2)*gm(3,3)*dgm01(2,3)*dgm10(1,1)-12*gm(1,3)&
&   *gm(3,3)*dgm01(1,2)*dgm10(1,2)-18*gm(1,2)*gm(3,3)*dgm01(1,3)&
&   *dgm10(1,2)+18*gm(1,3)**2*dgm01(2,3)*dgm10(1,2)-12*gm(1,1)*gm(3,3)&
&   *dgm01(2,3)*dgm10(1,2)+0*gm(1,2)*gm(1,3)*dgm01(3,3)*dgm10(1,2)&
&   -18*gm(1,2)*gm(3,3)*dgm01(1,2)*dgm10(1,3)+15*gm(1,3)**2*dgm01(2,2)&
&   *dgm10(1,3)-3*gm(1,1)*gm(3,3)*dgm01(2,2)*dgm10(1,3)+24*gm(1,2)&
&   *gm(1,3)*dgm01(2,3)*dgm10(1,3)-3*gm(1,2)**2*dgm01(3,3)*dgm10(1,3)&
&   +15*gm(1,3)**2*dgm01(1,3)*dgm10(2,2)-3*gm(1,1)*gm(3,3)*dgm01(1,3)&
&   *dgm10(2,2)+12*gm(1,1)*gm(1,3)*dgm01(3,3)*dgm10(2,2)-9*gm(1,2)&
&   *gm(3,3)*dgm01(1,1)*dgm10(2,3)+18*gm(1,3)**2*dgm01(1,2)*dgm10(2,3)&
&   -12*gm(1,1)*gm(3,3)*dgm01(1,2)*dgm10(2,3)+24*gm(1,2)*gm(1,3)&
&   *dgm01(1,3)*dgm10(2,3)+48*gm(1,1)*gm(1,3)*dgm01(2,3)*dgm10(2,3)&
&   -6*gm(1,1)*gm(1,2)*dgm01(3,3)*dgm10(2,3)+0*gm(1,2)*gm(1,3)*dgm01(1,2)&
&   *dgm10(3,3)-3*gm(1,2)**2*dgm01(1,3)*dgm10(3,3)+12*gm(1,1)*gm(1,3)&
&   *dgm01(2,2)*dgm10(3,3)-6*gm(1,1)*gm(1,2)*dgm01(2,3)*dgm10(3,3)&
&   -6*gm(1,2)*gm(1,3)*gm(3,3)*d2gm(1,2)-6*gm(1,2)**2*gm(3,3)*d2gm(1,3)&
&   -6*gm(2,3)**2*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3)+gm(1,1)&
&   *d2gm(1,3))+3*gm(1,2)*gm(1,3)**2*d2gm(2,3)-9*gm(1,1)*gm(1,2)&
&   *gm(3,3)*d2gm(2,3)+3*gm(1,2)**2*gm(1,3)*d2gm(3,3)+gm(2,3)*(3*gm(1,2)&
&   *dgm01(3,3)*dgm10(1,1)+0*gm(1,1)*dgm01(3,3)*dgm10(1,2)+36*gm(1,2)&
&   *dgm01(1,3)*dgm10(1,3)+12*gm(1,1)*dgm01(2,3)*dgm10(1,3)+12*gm(1,1)&
&   *dgm01(1,3)*dgm10(2,3)+3*gm(1,2)*dgm01(1,1)*dgm10(3,3)+0*gm(1,1)&
&   *dgm01(1,2)*dgm10(3,3)+15*gm(1,3)**2*d2gm(1,2)-3*gm(3,3)*(dgm01(1,2)&
&   *dgm10(1,1)+dgm01(1,1)*dgm10(1,2)+gm(1,1)*d2gm(1,2))+gm(1,3)&
&   *(24*dgm01(2,3)*dgm10(1,1)+48*dgm01(1,3)*dgm10(1,2)+48*dgm01(1,2)&
&   *dgm10(1,3)+24*dgm01(1,1)*dgm10(2,3)+18*gm(1,2)*d2gm(1,3)+24*gm(1,1)&
&   *d2gm(2,3))+3*gm(1,1)*gm(1,2)*d2gm(3,3))+gm(2,2)*(9*gm(1,1)*(dgm01(3,3)&
&   *dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+15*gm(1,3)**2*d2gm(1,3)-3*gm(3,3)&
&   *(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3)+gm(1,1)*d2gm(1,3))&
&   +gm(1,3)*(12*dgm01(3,3)*dgm10(1,1)+60*dgm01(1,3)*dgm10(1,3)+12*dgm01(1,1)&
&   *dgm10(3,3)+12*gm(1,1)*d2gm(3,3)))
   cm(7,14)=-3*gm(1,2)*gm(3,3)*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)&
&   *dgm10(2,3))+5*gm(2,3)**3*d2gm(1,2)+gm(2,3)**2*(30*dgm01(2,3)&
&   *dgm10(1,2)+9*dgm01(2,2)*dgm10(1,3)+9*dgm01(1,3)*dgm10(2,2)+30*dgm01(1,2)&
&   *dgm10(2,3)+9*gm(2,2)*d2gm(1,3)+15*gm(1,2)*d2gm(2,3))+gm(2,2)&
&   *(gm(1,3)*(-6*dgm01(3,3)*dgm10(2,2)-12*dgm01(2,3)*dgm10(2,3)&
&   -6*dgm01(2,2)*dgm10(3,3))+6*gm(1,2)*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)&
&   *dgm10(3,3))+gm(3,3)*(-6*dgm01(2,3)*dgm10(1,2)-6*dgm01(2,2)*dgm10(1,3)&
&   -6*dgm01(1,3)*dgm10(2,2)-6*dgm01(1,2)*dgm10(2,3)-3*gm(1,2)*d2gm(2,3)))&
&   +gm(2,3)*(-6*gm(1,3)*dgm01(2,3)*dgm10(2,2)+9*gm(1,2)*dgm01(3,3)&
&   *dgm10(2,2)-6*gm(1,3)*dgm01(2,2)*dgm10(2,3)+60*gm(1,2)*dgm01(2,3)&
&   *dgm10(2,3)+9*gm(1,2)*dgm01(2,2)*dgm10(3,3)-3*gm(3,3)*(dgm01(2,2)&
&   *dgm10(1,2)+dgm01(1,2)*dgm10(2,2)+gm(2,2)*d2gm(1,2))+gm(2,2)&
&   *(6*dgm01(3,3)*dgm10(1,2)+12*dgm01(2,3)*dgm10(1,3)+12*dgm01(1,3)&
&   *dgm10(2,3)+6*dgm01(1,2)*dgm10(3,3)-6*gm(1,3)*d2gm(2,3)+9*gm(1,2)&
&   *d2gm(3,3)))+gm(2,2)**2*(-6*dgm01(3,3)*dgm10(1,3)-6*dgm01(1,3)&
&   *dgm10(3,3)-3*(gm(3,3)*d2gm(1,3)+gm(1,3)*d2gm(3,3)))
   cm(8,14)=24*gm(1,3)*gm(2,2)*dgm01(3,3)*dgm10(3,3)+gm(2,3)*(6*gm(1,2)&
&   *dgm01(3,3)*dgm10(3,3)+12*gm(1,3)*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)&
&   *dgm10(3,3)))+gm(3,3)**2*(18*dgm01(2,3)*dgm10(1,2)+12*dgm01(2,2)&
&   *dgm10(1,3)+12*dgm01(1,3)*dgm10(2,2)+18*dgm01(1,2)*dgm10(2,3)&
&   +6*gm(2,3)*d2gm(1,2)+12*gm(2,2)*d2gm(1,3)+12*gm(1,2)*d2gm(2,3))&
&   +gm(2,3)**2*(9*dgm01(3,3)*dgm10(1,3)+9*dgm01(1,3)*dgm10(3,3)&
&   +3*gm(1,3)*d2gm(3,3))+gm(3,3)*(12*gm(1,3)*dgm01(3,3)*dgm10(2,2)&
&   +12*gm(1,3)*dgm01(2,3)*dgm10(2,3)+27*gm(1,2)*dgm01(3,3)*dgm10(2,3)&
&   +12*gm(1,3)*dgm01(2,2)*dgm10(3,3)+27*gm(1,2)*dgm01(2,3)*dgm10(3,3)&
&   +6*gm(2,3)**2*d2gm(1,3)+gm(2,3)*(15*dgm01(3,3)*dgm10(1,2)+18*dgm01(2,3)&
&   *dgm10(1,3)+18*dgm01(1,3)*dgm10(2,3)+15*dgm01(1,2)*dgm10(3,3)&
&   +6*gm(1,3)*d2gm(2,3)+3*gm(1,2)*d2gm(3,3))+gm(2,2)*(36*dgm01(3,3)&
&   *dgm10(1,3)+36*dgm01(1,3)*dgm10(3,3)+12*gm(1,3)*d2gm(3,3)))
   cm(9,14)=gm(2,2)*(24*gm(1,2)*dgm01(3,3)*dgm10(3,3)-6*gm(1,3)&
&   *(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))-3*gm(3,3)**2*(dgm01(2,2)&
&   *dgm10(1,2)+dgm01(1,2)*dgm10(2,2)+gm(2,2)*d2gm(1,2))+3*gm(2,3)&
&   **3*d2gm(1,3)+gm(2,3)**2*(12*dgm01(3,3)*dgm10(1,2)+12*dgm01(2,3)&
&   *dgm10(1,3)+12*dgm01(1,3)*dgm10(2,3)+12*dgm01(1,2)*dgm10(3,3)&
&   +3*gm(1,3)*d2gm(2,3)+3*gm(1,2)*d2gm(3,3))+gm(3,3)*(-9*gm(1,3)&
&   *dgm01(2,3)*dgm10(2,2)+12*gm(1,2)*dgm01(3,3)*dgm10(2,2)-9*gm(1,3)&
&   *dgm01(2,2)*dgm10(2,3)+48*gm(1,2)*dgm01(2,3)*dgm10(2,3)+12*gm(1,2)&
&   *dgm01(2,2)*dgm10(3,3)+9*gm(2,3)**2*d2gm(1,2)+gm(2,3)*(42*dgm01(2,3)&
&   *dgm10(1,2)+15*dgm01(2,2)*dgm10(1,3)+15*dgm01(1,3)*dgm10(2,2)&
&   +42*dgm01(1,2)*dgm10(2,3)+15*gm(2,2)*d2gm(1,3)+24*gm(1,2)*d2gm(2,3))&
&   +gm(2,2)*(6*dgm01(3,3)*dgm10(1,2)+6*dgm01(2,3)*dgm10(1,3)+6*dgm01(1,3)&
&   *dgm10(2,3)+6*dgm01(1,2)*dgm10(3,3)-9*gm(1,3)*d2gm(2,3)+12*gm(1,2)&
&   *d2gm(3,3)))+gm(2,3)*(gm(1,3)*(3*dgm01(3,3)*dgm10(2,2)+12*dgm01(2,3)&
&   *dgm10(2,3)+3*dgm01(2,2)*dgm10(3,3))+30*gm(1,2)*(dgm01(3,3)*dgm10(2,3)&
&   +dgm01(2,3)*dgm10(3,3))+gm(2,2)*(18*dgm01(3,3)*dgm10(1,3)+18*dgm01(1,3)&
&   *dgm10(3,3)+3*gm(1,3)*d2gm(3,3)))
   cm(10,14)=18*gm(1,3)*gm(2,3)*dgm01(3,3)*dgm10(3,3)+2*gm(3,3)&
&   **3*d2gm(1,2)+gm(3,3)**2*(3*dgm01(3,3)*dgm10(1,2)+12*dgm01(2,3)&
&   *dgm10(1,3)+12*dgm01(1,3)*dgm10(2,3)+3*dgm01(1,2)*dgm10(3,3)&
&   +6*gm(2,3)*d2gm(1,3)+6*gm(1,3)*d2gm(2,3)-3*gm(1,2)*d2gm(3,3))&
&   +gm(3,3)*(-12*gm(1,2)*dgm01(3,3)*dgm10(3,3)+21*gm(1,3)*(dgm01(3,3)&
&   *dgm10(2,3)+dgm01(2,3)*dgm10(3,3))+gm(2,3)*(21*dgm01(3,3)*dgm10(1,3)&
&   +21*dgm01(1,3)*dgm10(3,3)+9*gm(1,3)*d2gm(3,3)))
   cm(1,15)=5*gm(1,3)**3*d2gm(1,3)+gm(1,3)*(6*gm(1,1)*(dgm01(3,3)&
&   *dgm10(1,3)+dgm01(1,3)*dgm10(3,3))-3*gm(3,3)*(dgm01(1,3)*dgm10(1,1)&
&   +dgm01(1,1)*dgm10(1,3)+gm(1,1)*d2gm(1,3)))+gm(1,3)**2*(4.5d0*dgm01(3,3)&
&   *dgm10(1,1)+30*dgm01(1,3)*dgm10(1,3)+4.5d0*dgm01(1,1)*dgm10(3,3)&
&   +4.5d0*gm(1,1)*d2gm(3,3))+gm(1,1)*(-3*gm(1,1)*dgm01(3,3)*dgm10(3,3)&
&   +gm(3,3)*(-3*dgm01(3,3)*dgm10(1,1)-6*dgm01(1,3)*dgm10(1,3)-3*dgm01(1,1)&
&   *dgm10(3,3)-1.5d0*gm(1,1)*d2gm(3,3)))
   cm(2,15)=(2*(90*gm(2,3)**2*dgm01(1,3)+gm(3,3)*(-18*gm(1,3)*dgm01(2,2)&
&   -36*gm(1,2)*dgm01(2,3))+gm(2,3)*(-36*gm(3,3)*dgm01(1,2)+180*gm(1,3)&
&   *dgm01(2,3)-36*gm(1,2)*dgm01(3,3))-18*gm(2,2)*(gm(3,3)*dgm01(1,3)&
&   +gm(1,3)*dgm01(3,3)))*dgm10(1,3)+3*(30*gm(2,3)**2*dgm01(1,1)&
&   -24*gm(1,2)*gm(3,3)*dgm01(1,2)-12*gm(1,3)**2*dgm01(2,2)-6*gm(1,1)&
&   *gm(3,3)*dgm01(2,2)+36*gm(1,2)*gm(1,3)*dgm01(2,3)+gm(2,3)*(36*gm(1,3)&
&   *dgm01(1,2)+36*gm(1,2)*dgm01(1,3)+60*gm(1,1)*dgm01(2,3))-12*gm(1,2)&
&   **2*dgm01(3,3)+gm(2,2)*(-6*gm(3,3)*dgm01(1,1)-24*gm(1,3)*dgm01(1,3)&
&   -6*gm(1,1)*dgm01(3,3)))*dgm10(3,3)+3*dgm01(3,3)*(30*gm(2,3)**2*dgm10(1,1)&
&   -24*gm(1,2)*gm(3,3)*dgm10(1,2)-12*gm(1,3)**2*dgm10(2,2)-6*gm(1,1)&
&   *gm(3,3)*dgm10(2,2)+36*gm(1,2)*gm(1,3)*dgm10(2,3)+gm(2,3)*(36*gm(1,3)&
&   *dgm10(1,2)+36*gm(1,2)*dgm10(1,3)+60*gm(1,1)*dgm10(2,3))-12*gm(1,2)&
&   **2*dgm10(3,3)+gm(2,2)*(-6*gm(3,3)*dgm10(1,1)-24*gm(1,3)*dgm10(1,3)&
&   -6*gm(1,1)*dgm10(3,3)))+2*dgm01(1,3)*(90*gm(2,3)**2*dgm10(1,3)&
&   +gm(3,3)*(-18*gm(1,3)*dgm10(2,2)-36*gm(1,2)*dgm10(2,3))+gm(2,3)&
&   *(-36*gm(3,3)*dgm10(1,2)+180*gm(1,3)*dgm10(2,3)-36*gm(1,2)*dgm10(3,3))&
&   -18*gm(2,2)*(gm(3,3)*dgm10(1,3)+gm(1,3)*dgm10(3,3)))+2*(-36*gm(1,2)&
&   *gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))&
&   *d2gm(1,3)+3*(-12*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&   -12*gm(1,2)**2*gm(3,3)+gm(1,1)*(30*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&   *d2gm(3,3))/12.d0
   cm(3,15)=6*gm(1,3)**2*dgm01(3,3)*dgm10(3,3)+gm(3,3)*(24*gm(1,1)&
&   *dgm01(3,3)*dgm10(3,3)+18*gm(1,3)*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)&
&   *dgm10(3,3))+3*gm(1,3)**2*d2gm(3,3))+gm(3,3)**2*(6*dgm01(3,3)&
&   *dgm10(1,1)+12*dgm01(1,3)*dgm10(1,3)+6*(dgm01(1,1)*dgm10(3,3)&
&   +gm(1,3)*d2gm(1,3)+gm(1,1)*d2gm(3,3)))
   cm(4,15)=gm(1,3)*(6*gm(1,2)*dgm01(3,3)*dgm10(3,3)+3*gm(1,3)&
&   *(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))-6*gm(3,3)**2*(dgm01(1,3)&
&   *dgm10(1,2)+dgm01(1,2)*dgm10(1,3)+gm(1,2)*d2gm(1,3))+gm(2,3)&
&   *(24*gm(1,1)*dgm01(3,3)*dgm10(3,3)+24*gm(1,3)*(dgm01(3,3)*dgm10(1,3)&
&   +dgm01(1,3)*dgm10(3,3))+3*gm(1,3)**2*d2gm(3,3)+gm(3,3)*(12*dgm01(3,3)&
&   *dgm10(1,1)+36*dgm01(1,3)*dgm10(1,3)+12*dgm01(1,1)*dgm10(3,3)&
&   +18*gm(1,3)*d2gm(1,3)+12*gm(1,1)*d2gm(3,3)))+gm(3,3)*(-9*gm(1,2)&
&   *(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+12*gm(1,1)*(dgm01(3,3)&
&   *dgm10(2,3)+dgm01(2,3)*dgm10(3,3))+gm(1,3)*(3*dgm01(3,3)*dgm10(1,2)&
&   +18*dgm01(2,3)*dgm10(1,3)+18*dgm01(1,3)*dgm10(2,3)+3*dgm01(1,2)&
&   *dgm10(3,3)+3*gm(1,2)*d2gm(3,3)))
   cm(5,15)=-3*gm(3,3)**2*(dgm01(1,3)*dgm10(1,1)+dgm01(1,1)*dgm10(1,3)&
&   +gm(1,1)*d2gm(1,3))+gm(1,3)*(15*gm(1,1)*dgm01(3,3)*dgm10(3,3)&
&   +13.5d0*gm(1,3)*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))&
&   +1.5d0*gm(1,3)**2*d2gm(3,3))+gm(3,3)*(1.5d0*gm(1,1)*(dgm01(3,3)&
&   *dgm10(1,3)+dgm01(1,3)*dgm10(3,3))+9*gm(1,3)**2*d2gm(1,3)+gm(1,3)&
&   *(7.5d0*dgm01(3,3)*dgm10(1,1)+36*dgm01(1,3)*dgm10(1,3)+7.5d0*dgm01(1,1)&
&   *dgm10(3,3)+7.5d0*gm(1,1)*d2gm(3,3)))
   cm(6,15)=(-72*gm(1,3)*gm(3,3)*dgm01(1,3)*dgm10(1,2)+18*gm(1,3)&
&   **2*dgm01(3,3)*dgm10(1,2)-54*gm(1,1)*gm(3,3)*dgm01(3,3)*dgm10(1,2)&
&   -72*gm(1,3)*gm(3,3)*dgm01(1,2)*dgm10(1,3)+180*gm(1,3)**2*dgm01(2,3)&
&   *dgm10(1,3)-36*gm(1,1)*gm(3,3)*dgm01(2,3)*dgm10(1,3)+180*gm(1,3)&
&   **2*dgm01(1,3)*dgm10(2,3)-36*gm(1,1)*gm(3,3)*dgm01(1,3)*dgm10(2,3)&
&   +144*gm(1,1)*gm(1,3)*dgm01(3,3)*dgm10(2,3)+18*gm(1,3)**2*dgm01(1,2)&
&   *dgm10(3,3)-54*gm(1,1)*gm(3,3)*dgm01(1,2)*dgm10(3,3)+144*gm(1,1)&
&   *gm(1,3)*dgm01(2,3)*dgm10(3,3)+gm(1,2)*(-108*gm(1,1)*dgm01(3,3)&
&   *dgm10(3,3)-36*gm(1,3)*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))&
&   +18*gm(1,3)**2*d2gm(3,3)+gm(3,3)*(-54*dgm01(3,3)*dgm10(1,1)-144*dgm01(1,3)&
&   *dgm10(1,3)-54*dgm01(1,1)*dgm10(3,3)-72*gm(1,3)*d2gm(1,3)-54*gm(1,1)&
&   *d2gm(3,3)))+gm(2,3)*(108*gm(1,1)*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)&
&   *dgm10(3,3))+180*gm(1,3)**2*d2gm(1,3)-36*gm(3,3)*(dgm01(1,3)&
&   *dgm10(1,1)+dgm01(1,1)*dgm10(1,3)+gm(1,1)*d2gm(1,3))+gm(1,3)&
&   *(144*dgm01(3,3)*dgm10(1,1)+720*dgm01(1,3)*dgm10(1,3)+144*dgm01(1,1)&
&   *dgm10(3,3)+144*gm(1,1)*d2gm(3,3))))/12.d0
   cm(7,15)=(-54*gm(1,2)*gm(3,3)*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)&
&   *dgm10(3,3))+180*gm(2,3)**3*d2gm(1,3)+gm(2,3)**2*(270*dgm01(3,3)&
&   *dgm10(1,2)+540*dgm01(2,3)*dgm10(1,3)+540*dgm01(1,3)*dgm10(2,3)&
&   +270*dgm01(1,2)*dgm10(3,3)+270*gm(1,2)*d2gm(3,3))+gm(2,2)*(-108*(gm(1,2)&
&   *dgm01(3,3)*dgm10(3,3)+gm(1,3)*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)&
&   *dgm10(3,3)))+gm(3,3)*(-54*dgm01(3,3)*dgm10(1,2)-108*dgm01(2,3)&
&   *dgm10(1,3)-108*dgm01(1,3)*dgm10(2,3)-54*dgm01(1,2)*dgm10(3,3)&
&   -54*gm(1,2)*d2gm(3,3)))+gm(2,3)*(-108*gm(1,3)*dgm01(3,3)*dgm10(2,2)&
&   +540*gm(1,2)*dgm01(3,3)*dgm10(2,3)-108*gm(1,3)*dgm01(2,2)*dgm10(3,3)&
&   +540*gm(1,2)*dgm01(2,3)*dgm10(3,3)-108*gm(3,3)*(dgm01(2,2)*dgm10(1,3)&
&   +dgm01(1,3)*dgm10(2,2)+gm(2,2)*d2gm(1,3))+gm(2,2)*(-216*dgm01(3,3)&
&   *dgm10(1,3)-216*dgm01(1,3)*dgm10(3,3)-108*gm(1,3)*d2gm(3,3))))&
&   /36.d0
   cm(8,15)=6*gm(1,3)*gm(2,3)*dgm01(3,3)*dgm10(3,3)+6*gm(3,3)**2*(dgm01(3,3)&
&   *dgm10(1,2)+dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3)+dgm01(1,2)&
&   *dgm10(3,3)+gm(2,3)*d2gm(1,3)+gm(1,2)*d2gm(3,3))+gm(3,3)*(24*gm(1,2)&
&   *dgm01(3,3)*dgm10(3,3)+3*gm(1,3)*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)&
&   *dgm10(3,3))+gm(2,3)*(15*dgm01(3,3)*dgm10(1,3)+15*dgm01(1,3)&
&   *dgm10(3,3)+3*gm(1,3)*d2gm(3,3)))
   cm(9,15)=-9*gm(1,3)*gm(2,2)*dgm01(3,3)*dgm10(3,3)-3*gm(3,3)&
&   **2*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2)+gm(2,2)*d2gm(1,3))&
&   +gm(2,3)**2*(10.5d0*dgm01(3,3)*dgm10(1,3)+10.5d0*dgm01(1,3)*dgm10(3,3)&
&   +9*gm(3,3)*d2gm(1,3)+1.5d0*gm(1,3)*d2gm(3,3))+gm(2,3)*(24*gm(1,2)&
&   *dgm01(3,3)*dgm10(3,3)+3*gm(1,3)*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)&
&   *dgm10(3,3))+gm(3,3)*(12*dgm01(3,3)*dgm10(1,2)+18*dgm01(2,3)&
&   *dgm10(1,3)+18*dgm01(1,3)*dgm10(2,3)+12*dgm01(1,2)*dgm10(3,3)&
&   +12*gm(1,2)*d2gm(3,3)))+gm(3,3)*(-4.5d0*gm(1,3)*(dgm01(3,3)*dgm10(2,2)&
&   +dgm01(2,2)*dgm10(3,3))+12*gm(1,2)*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)&
&   *dgm10(3,3))+gm(2,2)*(-10.5d0*dgm01(3,3)*dgm10(1,3)-10.5d0*dgm01(1,3)&
&   *dgm10(3,3)-4.5d0*gm(1,3)*d2gm(3,3)))
   cm(10,15)=gm(3,3)*(12*gm(1,3)*dgm01(3,3)*dgm10(3,3)+2*gm(3,3)&
&   **2*d2gm(1,3)+gm(3,3)*(9*dgm01(3,3)*dgm10(1,3)+9*dgm01(1,3)*dgm10(3,3)&
&   +3*gm(1,3)*d2gm(3,3)))
   cm(1,16)=(270*gm(1,2)**2*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))&
&   -54*gm(1,1)*gm(2,2)*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))&
&   +90*gm(1,2)**3*d2gm(2,2)+gm(1,2)*(-108*gm(1,1)*dgm01(2,2)*dgm10(2,2)&
&   -54*gm(2,2)*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2)+gm(1,1)&
&   *d2gm(2,2))))/36.d0
   cm(2,16)=gm(2,2)*(12*gm(1,2)*dgm01(2,2)*dgm10(2,2)+3*gm(2,2)&
&   *(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2)+gm(1,2)*d2gm(2,2)))
   cm(3,16)=((90*gm(2,3)**2*dgm01(1,2)-18*gm(1,2)*gm(3,3)*dgm01(2,2)&
&   +gm(2,3)*(-36*gm(2,2)*dgm01(1,3)-36*gm(1,3)*dgm01(2,2)+180*gm(1,2)&
&   *dgm01(2,3))+gm(2,2)*(-18*gm(3,3)*dgm01(1,2)-36*gm(1,3)*dgm01(2,3)&
&   -18*gm(1,2)*dgm01(3,3)))*dgm10(2,2)+dgm01(2,2)*(90*gm(2,3)**2*dgm10(1,2)&
&   -18*gm(1,2)*gm(3,3)*dgm10(2,2)+gm(2,3)*(-36*gm(2,2)*dgm10(1,3)&
&   -36*gm(1,3)*dgm10(2,2)+180*gm(1,2)*dgm10(2,3))+gm(2,2)*(-18*gm(3,3)&
&   *dgm10(1,2)-36*gm(1,3)*dgm10(2,3)-18*gm(1,2)*dgm10(3,3)))+(-36*gm(1,3)&
&   *gm(2,2)*gm(2,3)+gm(1,2)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))&
&   *d2gm(2,2))/12.d0
   cm(4,16)=18*gm(1,2)*gm(2,3)*dgm01(2,2)*dgm10(2,2)-3*gm(2,2)&
&   **2*(dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2)+gm(1,3)*d2gm(2,2))&
&   +gm(2,2)*(-12*gm(1,3)*dgm01(2,2)*dgm10(2,2)+9*gm(1,2)*(dgm01(2,3)&
&   *dgm10(2,2)+dgm01(2,2)*dgm10(2,3))+9*gm(2,3)*(dgm01(2,2)*dgm10(1,2)&
&   +dgm01(1,2)*dgm10(2,2)+gm(1,2)*d2gm(2,2)))
   cm(5,16)=((-18*gm(1,1)*gm(2,3)*dgm01(2,2)+gm(1,2)*(180*gm(2,3)&
&   *dgm01(1,2)-36*gm(1,3)*dgm01(2,2))+90*gm(1,2)**2*dgm01(2,3)+gm(2,2)&
&   *(-18*gm(2,3)*dgm01(1,1)-36*gm(1,3)*dgm01(1,2)-36*gm(1,2)*dgm01(1,3)&
&   -18*gm(1,1)*dgm01(2,3)))*dgm10(2,2)+dgm01(2,2)*(-18*gm(1,1)*gm(2,3)&
&   *dgm10(2,2)+gm(1,2)*(180*gm(2,3)*dgm10(1,2)-36*gm(1,3)*dgm10(2,2))&
&   +90*gm(1,2)**2*dgm10(2,3)+gm(2,2)*(-18*gm(2,3)*dgm10(1,1)-36*gm(1,3)&
&   *dgm10(1,2)-36*gm(1,2)*dgm10(1,3)-18*gm(1,1)*dgm10(2,3)))+(-36*gm(1,2)&
&   *gm(1,3)*gm(2,2)+90*gm(1,2)**2*gm(2,3)-18*gm(1,1)*gm(2,2)*gm(2,3))&
&   *d2gm(2,2))/12.d0
   cm(6,16)=9*gm(1,2)**2*dgm01(2,2)*dgm10(2,2)-1.5d0*gm(2,2)**2*(dgm01(2,2)&
&   *dgm10(1,1)+dgm01(1,1)*dgm10(2,2)+gm(1,1)*d2gm(2,2))+gm(2,2)&
&   *(-6*gm(1,1)*dgm01(2,2)*dgm10(2,2)+9*gm(1,2)*(dgm01(2,2)*dgm10(1,2)&
&   +dgm01(1,2)*dgm10(2,2))+4.5d0*gm(1,2)**2*d2gm(2,2))
   cm(7,16)=gm(2,2)**2*(6*dgm01(2,2)*dgm10(2,2)+gm(2,2)*d2gm(2,2))
   cm(8,16)=9*gm(2,2)*gm(2,3)*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)&
&   *dgm10(2,3))+gm(2,3)**2*(9*dgm01(2,2)*dgm10(2,2)+4.5d0*gm(2,2)&
&   *d2gm(2,2))+gm(2,2)*(-1.5d0*gm(2,2)*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)&
&   *dgm10(3,3))+gm(3,3)*(-6*dgm01(2,2)*dgm10(2,2)-1.5d0*gm(2,2)&
&   *d2gm(2,2)))
   cm(9,16)=gm(2,2)*(3*gm(2,2)*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)&
&   *dgm10(2,3))+gm(2,3)*(12*dgm01(2,2)*dgm10(2,2)+3*gm(2,2)*d2gm(2,2)))
   cm(10,16)=(270*gm(2,3)**2*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)&
&   *dgm10(2,3))-54*gm(2,2)*gm(3,3)*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)&
&   *dgm10(2,3))+90*gm(2,3)**3*d2gm(2,2)+gm(2,3)*(-54*gm(2,2)*(dgm01(3,3)&
&   *dgm10(2,2)+dgm01(2,2)*dgm10(3,3))+gm(3,3)*(-108*dgm01(2,2)*dgm10(2,2)&
&   -54*gm(2,2)*d2gm(2,2))))/36.d0
   cm(1,17)=(3*(90*gm(1,3)**2*dgm01(1,2)+gm(1,1)*(-18*gm(3,3)*dgm01(1,2)&
&   -36*gm(2,3)*dgm01(1,3))+gm(1,3)*(-36*gm(2,3)*dgm01(1,1)+180*gm(1,2)&
&   *dgm01(1,3)-36*gm(1,1)*dgm01(2,3))-18*gm(1,2)*(gm(3,3)*dgm01(1,1)&
&   +gm(1,1)*dgm01(3,3)))*dgm10(2,2)+dgm01(3,3)*(270*gm(1,2)**2*dgm10(1,2)&
&   -54*gm(1,1)*gm(2,2)*dgm10(1,2)-54*gm(1,2)*(gm(2,2)*dgm10(1,1)&
&   +gm(1,1)*dgm10(2,2)))+6*(90*gm(1,2)**2*dgm01(1,3)+gm(1,1)*(-36*gm(2,3)&
&   *dgm01(1,2)-18*gm(2,2)*dgm01(1,3))+gm(1,3)*(-18*gm(2,2)*dgm01(1,1)&
&   +180*gm(1,2)*dgm01(1,2)-18*gm(1,1)*dgm01(2,2))-36*gm(1,2)*(gm(2,3)&
&   *dgm01(1,1)+gm(1,1)*dgm01(2,3)))*dgm10(2,3)+6*dgm01(2,3)*(90*gm(1,2)&
&   **2*dgm10(1,3)+gm(1,1)*(-36*gm(2,3)*dgm10(1,2)-18*gm(2,2)*dgm10(1,3))&
&   +gm(1,3)*(-18*gm(2,2)*dgm10(1,1)+180*gm(1,2)*dgm10(1,2)-18*gm(1,1)&
&   *dgm10(2,2))-36*gm(1,2)*(gm(2,3)*dgm10(1,1)+gm(1,1)*dgm10(2,3)))&
&   +(270*gm(1,2)**2*dgm01(1,2)-54*gm(1,1)*gm(2,2)*dgm01(1,2)-54*gm(1,2)&
&   *(gm(2,2)*dgm01(1,1)+gm(1,1)*dgm01(2,2)))*dgm10(3,3)+3*dgm01(2,2)&
&   *(90*gm(1,3)**2*dgm10(1,2)+gm(1,1)*(-18*gm(3,3)*dgm10(1,2)-36*gm(2,3)&
&   *dgm10(1,3))+gm(1,3)*(-36*gm(2,3)*dgm10(1,1)+180*gm(1,2)*dgm10(1,3)&
&   -36*gm(1,1)*dgm10(2,3))-18*gm(1,2)*(gm(3,3)*dgm10(1,1)+gm(1,1)&
&   *dgm10(3,3)))+3*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)&
&   **2-18*gm(1,1)*gm(3,3)))*d2gm(2,2)+6*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)&
&   *(gm(1,3)*gm(2,2)+2*gm(1,2)*gm(2,3)))*d2gm(2,3)+(90*gm(1,2)**3-54*gm(1,1)&
&   *gm(1,2)*gm(2,2))*d2gm(3,3))/36.d0
   cm(2,17)=-9*gm(1,2)*gm(3,3)*dgm01(2,2)*dgm10(2,2)+1.5d0*gm(2,3)&
&   **2*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2)+gm(1,2)*d2gm(2,2))&
&   +gm(2,2)*(36*gm(1,3)*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))&
&   +gm(1,2)*(1.5d0*dgm01(3,3)*dgm10(2,2)+12*dgm01(2,3)*dgm10(2,3)&
&   +1.5d0*dgm01(2,2)*dgm10(3,3))-4.5d0*gm(3,3)*(dgm01(2,2)*dgm10(1,2)&
&   +dgm01(1,2)*dgm10(2,2)+gm(1,2)*d2gm(2,2)))+gm(2,3)*(24*gm(1,3)&
&   *dgm01(2,2)*dgm10(2,2)+9*gm(1,2)*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)&
&   *dgm10(2,3))+gm(2,2)*(6*dgm01(2,3)*dgm10(1,2)+12*dgm01(2,2)*dgm10(1,3)&
&   +12*dgm01(1,3)*dgm10(2,2)+6*dgm01(1,2)*dgm10(2,3)+12*gm(1,3)&
&   *d2gm(2,2)+6*gm(1,2)*d2gm(2,3)))+gm(2,2)**2*(3*dgm01(3,3)*dgm10(1,2)&
&   +12*dgm01(2,3)*dgm10(1,3)+12*dgm01(1,3)*dgm10(2,3)+3*dgm01(1,2)&
&   *dgm10(3,3)+12*gm(1,3)*d2gm(2,3)+3*gm(1,2)*d2gm(3,3))
   cm(3,17)=gm(2,2)*(-3*gm(1,2)*dgm01(3,3)*dgm10(3,3)-12*gm(1,3)&
&   *(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))+6*gm(3,3)**2*(dgm01(2,2)&
&   *dgm10(1,2)+dgm01(1,2)*dgm10(2,2)+gm(1,2)*d2gm(2,2))+gm(2,3)&
&   **2*(7.5d0*dgm01(3,3)*dgm10(1,2)+3*dgm01(2,3)*dgm10(1,3)+3*dgm01(1,3)&
&   *dgm10(2,3)+7.5d0*dgm01(1,2)*dgm10(3,3)+3*gm(1,3)*d2gm(2,3)+7.5d0*gm(1,2)&
&   *d2gm(3,3))+gm(3,3)*(-6*gm(1,3)*dgm01(2,3)*dgm10(2,2)+10.5d0*gm(1,2)&
&   *dgm01(3,3)*dgm10(2,2)-6*gm(1,3)*dgm01(2,2)*dgm10(2,3)+48*gm(1,2)&
&   *dgm01(2,3)*dgm10(2,3)+10.5d0*gm(1,2)*dgm01(2,2)*dgm10(3,3)+gm(2,3)&
&   *(24*dgm01(2,3)*dgm10(1,2)+3*dgm01(2,2)*dgm10(1,3)+3*dgm01(1,3)&
&   *dgm10(2,2)+24*dgm01(1,2)*dgm10(2,3)+3*gm(1,3)*d2gm(2,2)+24*gm(1,2)&
&   *d2gm(2,3))+gm(2,2)*(-1.5d0*dgm01(3,3)*dgm10(1,2)-9*dgm01(2,3)&
&   *dgm10(1,3)-9*dgm01(1,3)*dgm10(2,3)-1.5d0*dgm01(1,2)*dgm10(3,3)&
&   -9*gm(1,3)*d2gm(2,3)-1.5d0*gm(1,2)*d2gm(3,3)))+gm(2,3)*(gm(1,3)&
&   *(0*dgm01(3,3)*dgm10(2,2)+12*dgm01(2,3)*dgm10(2,3)+0*dgm01(2,2)&
&   *dgm10(3,3))+39*gm(1,2)*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3))&
&   -3*gm(2,2)*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3)+gm(1,3)&
&   *d2gm(3,3)))
   cm(4,17)=gm(3,3)*(24*gm(1,3)*dgm01(2,2)*dgm10(2,2)+27*gm(1,2)&
&   *(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3)))+gm(2,3)**2*(6*dgm01(2,3)&
&   *dgm10(1,2)+3*dgm01(2,2)*dgm10(1,3)+3*dgm01(1,3)*dgm10(2,2)+6*dgm01(1,2)&
&   *dgm10(2,3)+3*gm(1,3)*d2gm(2,2)+6*gm(1,2)*d2gm(2,3))+gm(2,2)&
&   *(gm(1,3)*(6*dgm01(3,3)*dgm10(2,2)+12*dgm01(2,3)*dgm10(2,3)+6*dgm01(2,2)&
&   *dgm10(3,3))+33*gm(1,2)*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3))&
&   +gm(3,3)*(24*dgm01(2,3)*dgm10(1,2)+12*dgm01(2,2)*dgm10(1,3)+12*dgm01(1,3)&
&   *dgm10(2,2)+24*dgm01(1,2)*dgm10(2,3)+12*gm(1,3)*d2gm(2,2)+24*gm(1,2)&
&   *d2gm(2,3)))-3*gm(2,2)**2*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3)&
&   +gm(1,3)*d2gm(3,3))+gm(2,3)*(12*gm(1,3)*dgm01(2,3)*dgm10(2,2)&
&   +12*gm(1,2)*dgm01(3,3)*dgm10(2,2)+12*gm(1,3)*dgm01(2,2)*dgm10(2,3)&
&   +24*gm(1,2)*dgm01(2,3)*dgm10(2,3)+12*gm(1,2)*dgm01(2,2)*dgm10(3,3)&
&   +3*gm(3,3)*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2)+gm(1,2)&
&   *d2gm(2,2))+gm(2,2)*(9*dgm01(3,3)*dgm10(1,2)+6*dgm01(2,3)*dgm10(1,3)&
&   +6*dgm01(1,3)*dgm10(2,3)+9*dgm01(1,2)*dgm10(3,3)+6*gm(1,3)*d2gm(2,3)&
&   +9*gm(1,2)*d2gm(3,3)))
   cm(5,17)=12*gm(1,3)*gm(3,3)*dgm01(2,2)*dgm10(1,2)+30*gm(1,2)&
&   *gm(3,3)*dgm01(2,3)*dgm10(1,2)+12*gm(1,2)*gm(3,3)*dgm01(2,2)&
&   *dgm10(1,3)+12*gm(1,3)*gm(3,3)*dgm01(1,2)*dgm10(2,2)+12*gm(1,2)&
&   *gm(3,3)*dgm01(1,3)*dgm10(2,2)-4.5d0*gm(1,3)**2*dgm01(2,3)*dgm10(2,2)&
&   -7.5d0*gm(1,1)*gm(3,3)*dgm01(2,3)*dgm10(2,2)+9*gm(1,2)*gm(1,3)&
&   *dgm01(3,3)*dgm10(2,2)+30*gm(1,2)*gm(3,3)*dgm01(1,2)*dgm10(2,3)&
&   -4.5d0*gm(1,3)**2*dgm01(2,2)*dgm10(2,3)-7.5d0*gm(1,1)*gm(3,3)&
&   *dgm01(2,2)*dgm10(2,3)+36*gm(1,2)*gm(1,3)*dgm01(2,3)*dgm10(2,3)&
&   +22.5d0*gm(1,2)**2*dgm01(3,3)*dgm10(2,3)+9*gm(1,2)*gm(1,3)*dgm01(2,2)&
&   *dgm10(3,3)+22.5d0*gm(1,2)**2*dgm01(2,3)*dgm10(3,3)+12*gm(1,2)&
&   *gm(1,3)*gm(3,3)*d2gm(2,2)+15*gm(1,2)**2*gm(3,3)*d2gm(2,3)-6*gm(2,3)&
&   **2*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3)+gm(1,1)*d2gm(2,3))&
&   +gm(2,3)*(18*gm(1,3)*dgm01(2,3)*dgm10(1,2)+15*gm(1,2)*dgm01(3,3)&
&   *dgm10(1,2)+3*gm(1,3)*dgm01(2,2)*dgm10(1,3)+18*gm(1,2)*dgm01(2,3)&
&   *dgm10(1,3)+3*gm(1,3)*dgm01(1,3)*dgm10(2,2)-6*gm(1,1)*dgm01(3,3)&
&   *dgm10(2,2)+18*gm(1,3)*dgm01(1,2)*dgm10(2,3)+18*gm(1,2)*dgm01(1,3)&
&   *dgm10(2,3)-24*gm(1,1)*dgm01(2,3)*dgm10(2,3)+15*gm(1,2)*dgm01(1,2)&
&   *dgm10(3,3)-6*gm(1,1)*dgm01(2,2)*dgm10(3,3)+1.5d0*gm(1,3)**2*d2gm(2,2)&
&   -4.5d0*gm(3,3)*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2)+gm(1,1)&
&   *d2gm(2,2))+18*gm(1,2)*gm(1,3)*d2gm(2,3)+7.5d0*gm(1,2)**2*d2gm(3,3)&
&   -1.5d0*gm(2,2)*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3)+gm(1,1)&
&   *d2gm(3,3)))+gm(2,2)*(-3*gm(1,2)*dgm01(3,3)*dgm10(1,3)-4.5d0*gm(1,1)&
&   *dgm01(3,3)*dgm10(2,3)-3*gm(1,2)*dgm01(1,3)*dgm10(3,3)-4.5d0*gm(1,1)&
&   *dgm01(2,3)*dgm10(3,3)-6*gm(1,3)**2*d2gm(2,3)-3*gm(3,3)*(dgm01(2,3)&
&   *dgm10(1,1)+dgm01(1,1)*dgm10(2,3)+gm(1,1)*d2gm(2,3))+gm(1,3)&
&   *(-3*dgm01(3,3)*dgm10(1,2)-12*dgm01(2,3)*dgm10(1,3)-12*dgm01(1,3)&
&   *dgm10(2,3)-3*dgm01(1,2)*dgm10(3,3)-3*gm(1,2)*d2gm(3,3)))
   cm(6,17)=-6*gm(1,2)*gm(3,3)*dgm01(2,2)*dgm10(1,2)-6*gm(1,2)&
&   *gm(3,3)*dgm01(1,2)*dgm10(2,2)+15*gm(1,3)**2*dgm01(2,2)*dgm10(2,2)&
&   -3*gm(1,1)*gm(3,3)*dgm01(2,2)*dgm10(2,2)+33*gm(1,2)*gm(1,3)*dgm01(2,3)&
&   *dgm10(2,2)+1.5d0*gm(1,2)**2*dgm01(3,3)*dgm10(2,2)+33*gm(1,2)&
&   *gm(1,3)*dgm01(2,2)*dgm10(2,3)+6*gm(1,2)**2*dgm01(2,3)*dgm10(2,3)&
&   +1.5d0*gm(1,2)**2*dgm01(2,2)*dgm10(3,3)-3*gm(1,2)**2*gm(3,3)&
&   *d2gm(2,2)-3*gm(2,3)**2*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2)&
&   +gm(1,1)*d2gm(2,2))+gm(2,3)*(6*gm(1,2)*dgm01(2,3)*dgm10(1,2)&
&   +9*gm(1,2)*dgm01(2,2)*dgm10(1,3)+9*gm(1,2)*dgm01(1,3)*dgm10(2,2)&
&   -15*gm(1,1)*dgm01(2,3)*dgm10(2,2)+6*gm(1,2)*dgm01(1,2)*dgm10(2,3)&
&   -15*gm(1,1)*dgm01(2,2)*dgm10(2,3)+9*gm(1,3)*(dgm01(2,2)*dgm10(1,2)&
&   +dgm01(1,2)*dgm10(2,2)+gm(1,2)*d2gm(2,2))+3*gm(1,2)**2*d2gm(2,3)&
&   -9*gm(2,2)*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3)+gm(1,1)&
&   *d2gm(2,3)))-1.5d0*gm(2,2)**2*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)&
&   *dgm10(3,3)+gm(1,1)*d2gm(3,3))+gm(2,2)*(9*gm(1,2)*dgm01(3,3)&
&   *dgm10(1,2)+24*gm(1,2)*dgm01(2,3)*dgm10(1,3)-4.5d0*gm(1,1)*dgm01(3,3)&
&   *dgm10(2,2)+24*gm(1,2)*dgm01(1,3)*dgm10(2,3)-18*gm(1,1)*dgm01(2,3)&
&   *dgm10(2,3)+9*gm(1,2)*dgm01(1,2)*dgm10(3,3)-4.5d0*gm(1,1)*dgm01(2,2)&
&   *dgm10(3,3)+7.5d0*gm(1,3)**2*d2gm(2,2)-1.5d0*gm(3,3)*(dgm01(2,2)&
&   *dgm10(1,1)+dgm01(1,1)*dgm10(2,2)+gm(1,1)*d2gm(2,2))+gm(1,3)&
&   *(24*dgm01(2,3)*dgm10(1,2)+15*dgm01(2,2)*dgm10(1,3)+15*dgm01(1,3)&
&   *dgm10(2,2)+24*dgm01(1,2)*dgm10(2,3)+24*gm(1,2)*d2gm(2,3))+4.5d0*gm(1,2)&
&   **2*d2gm(3,3))
   cm(7,17)=gm(2,3)**2*(9*dgm01(2,2)*dgm10(2,2)+4.5d0*gm(2,2)*d2gm(2,2))&
&   +gm(2,2)*gm(2,3)*(21*dgm01(2,3)*dgm10(2,2)+21*dgm01(2,2)*dgm10(2,3)&
&   +6*gm(2,2)*d2gm(2,3))+gm(2,2)*(gm(3,3)*(-6*dgm01(2,2)*dgm10(2,2)&
&   -1.5d0*gm(2,2)*d2gm(2,2))+gm(2,2)*(1.5d0*dgm01(3,3)*dgm10(2,2)&
&   +12*dgm01(2,3)*dgm10(2,3)+1.5d0*dgm01(2,2)*dgm10(3,3)+gm(2,2)&
&   *d2gm(3,3)))
   cm(8,17)=-3*gm(2,2)**2*dgm01(3,3)*dgm10(3,3)+24*gm(2,2)*gm(2,3)&
&   *(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3))+gm(3,3)**2*(12*dgm01(2,2)&
&   *dgm10(2,2)+6*gm(2,2)*d2gm(2,2))+3*gm(2,3)**3*d2gm(2,3)+gm(2,3)&
&   **2*(7.5d0*dgm01(3,3)*dgm10(2,2)+18*dgm01(2,3)*dgm10(2,3)+7.5d0*dgm01(2,2)&
&   *dgm10(3,3)+4.5d0*gm(2,2)*d2gm(3,3))+gm(3,3)*(3*gm(2,3)**2*d2gm(2,2)&
&   +gm(2,3)*(21*dgm01(2,3)*dgm10(2,2)+21*dgm01(2,2)*dgm10(2,3)+15*gm(2,2)&
&   *d2gm(2,3))+gm(2,2)*(9*dgm01(3,3)*dgm10(2,2)+30*dgm01(2,3)*dgm10(2,3)&
&   +9*dgm01(2,2)*dgm10(3,3)-1.5d0*gm(2,2)*d2gm(3,3)))
   cm(9,17)=1.5d0*gm(2,3)**3*d2gm(2,2)+gm(2,3)**2*(10.5d0*dgm01(2,3)&
&   *dgm10(2,2)+10.5d0*dgm01(2,2)*dgm10(2,3)+6*gm(2,2)*d2gm(2,3))&
&   +gm(2,2)*(15*gm(2,2)*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3))&
&   +gm(3,3)*(31.5d0*dgm01(2,3)*dgm10(2,2)+31.5d0*dgm01(2,2)*dgm10(2,3)&
&   +12*gm(2,2)*d2gm(2,3)))+gm(2,3)*(gm(3,3)*(15*dgm01(2,2)*dgm10(2,2)&
&   +7.5d0*gm(2,2)*d2gm(2,2))+gm(2,2)*(13.5d0*dgm01(3,3)*dgm10(2,2)&
&   +24*dgm01(2,3)*dgm10(2,3)+13.5d0*dgm01(2,2)*dgm10(3,3)+3*gm(2,2)&
&   *d2gm(3,3)))
   cm(10,17)=gm(3,3)**2*(0*dgm01(2,3)*dgm10(2,2)+0*dgm01(2,2)*dgm10(2,3)&
&   +3*gm(2,3)*d2gm(2,2)-3*gm(2,2)*d2gm(2,3))+gm(2,3)*(-3*gm(2,2)&
&   *dgm01(3,3)*dgm10(3,3)+16.5d0*gm(2,3)*(dgm01(3,3)*dgm10(2,3)&
&   +dgm01(2,3)*dgm10(3,3))+2.5d0*gm(2,3)**2*d2gm(3,3))+gm(3,3)*(-7.5d0*gm(2,2)&
&   *(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3))+9*gm(2,3)**2*d2gm(2,3)&
&   +gm(2,3)*(4.5d0*dgm01(3,3)*dgm10(2,2)+36*dgm01(2,3)*dgm10(2,3)&
&   +4.5d0*dgm01(2,2)*dgm10(3,3)-1.5d0*gm(2,2)*d2gm(3,3)))
   cm(1,18)=(gm(1,2)**2*(540*dgm01(2,3)*dgm10(1,2)+270*dgm01(2,2)&
&   *dgm10(1,3)+270*dgm01(1,3)*dgm10(2,2)+540*dgm01(1,2)*dgm10(2,3))&
&   +gm(1,1)*(-108*gm(2,3)*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2))&
&   +gm(2,2)*(-108*dgm01(2,3)*dgm10(1,2)-54*dgm01(2,2)*dgm10(1,3)&
&   -54*dgm01(1,3)*dgm10(2,2)-108*dgm01(1,2)*dgm10(2,3)))+gm(1,3)&
&   *(-108*gm(1,1)*dgm01(2,2)*dgm10(2,2)+540*gm(1,2)*(dgm01(2,2)&
&   *dgm10(1,2)+dgm01(1,2)*dgm10(2,2))+270*gm(1,2)**2*d2gm(2,2)-54*gm(2,2)&
&   *(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2)+gm(1,1)*d2gm(2,2)))&
&   +180*gm(1,2)**3*d2gm(2,3)+gm(1,2)*(-216*gm(1,1)*(dgm01(2,3)*dgm10(2,2)&
&   +dgm01(2,2)*dgm10(2,3))-108*gm(2,3)*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)&
&   *dgm10(2,2)+gm(1,1)*d2gm(2,2))-108*gm(2,2)*(dgm01(2,3)*dgm10(1,1)&
&   +dgm01(1,1)*dgm10(2,3)+gm(1,1)*d2gm(2,3))))/36.d0
   cm(2,18)=6*gm(1,2)*gm(2,3)*dgm01(2,2)*dgm10(2,2)+gm(2,2)*(24*gm(1,3)&
&   *dgm01(2,2)*dgm10(2,2)+15*gm(1,2)*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)&
&   *dgm10(2,3))+3*gm(2,3)*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2)&
&   +gm(1,2)*d2gm(2,2)))+6*gm(2,2)**2*(dgm01(2,3)*dgm10(1,2)+dgm01(2,2)&
&   *dgm10(1,3)+dgm01(1,3)*dgm10(2,2)+dgm01(1,2)*dgm10(2,3)+gm(1,3)&
&   *d2gm(2,2)+gm(1,2)*d2gm(2,3))
   cm(3,18)=gm(3,3)*(-9*gm(1,3)*dgm01(2,2)*dgm10(2,2)+9*gm(1,2)&
&   *(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3)))+gm(2,3)**2*(15*dgm01(2,3)&
&   *dgm10(1,2)+1.5d0*dgm01(2,2)*dgm10(1,3)+1.5d0*dgm01(1,3)*dgm10(2,2)&
&   +15*dgm01(1,2)*dgm10(2,3)+1.5d0*gm(1,3)*d2gm(2,2)+15*gm(1,2)&
&   *d2gm(2,3))+gm(2,2)*(gm(1,3)*(-4.5d0*dgm01(3,3)*dgm10(2,2)-12*dgm01(2,3)&
&   *dgm10(2,3)-4.5d0*dgm01(2,2)*dgm10(3,3))-3*gm(1,2)*(dgm01(3,3)&
&   *dgm10(2,3)+dgm01(2,3)*dgm10(3,3))+gm(3,3)*(-3*dgm01(2,3)*dgm10(1,2)&
&   -4.5d0*dgm01(2,2)*dgm10(1,3)-4.5d0*dgm01(1,3)*dgm10(2,2)-3*dgm01(1,2)&
&   *dgm10(2,3)-4.5d0*gm(1,3)*d2gm(2,2)-3*gm(1,2)*d2gm(2,3)))+gm(2,3)&
&   *(-3*gm(1,3)*dgm01(2,3)*dgm10(2,2)+12*gm(1,2)*dgm01(3,3)*dgm10(2,2)&
&   -3*gm(1,3)*dgm01(2,2)*dgm10(2,3)+60*gm(1,2)*dgm01(2,3)*dgm10(2,3)&
&   +12*gm(1,2)*dgm01(2,2)*dgm10(3,3)+12*gm(3,3)*(dgm01(2,2)*dgm10(1,2)&
&   +dgm01(1,2)*dgm10(2,2)+gm(1,2)*d2gm(2,2))-6*gm(2,2)*(dgm01(2,3)&
&   *dgm10(1,3)+dgm01(1,3)*dgm10(2,3)+gm(1,3)*d2gm(2,3)))
   cm(4,18)=24*gm(1,2)*gm(3,3)*dgm01(2,2)*dgm10(2,2)+3*gm(2,3)&
&   **2*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2)+gm(1,2)*d2gm(2,2))&
&   +gm(2,2)*(-9*gm(1,3)*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))&
&   +gm(1,2)*(12*dgm01(3,3)*dgm10(2,2)+36*dgm01(2,3)*dgm10(2,3)+12*dgm01(2,2)&
&   *dgm10(3,3))+12*gm(3,3)*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2)&
&   +gm(1,2)*d2gm(2,2)))-6*gm(2,2)**2*(dgm01(2,3)*dgm10(1,3)+dgm01(1,3)&
&   *dgm10(2,3)+gm(1,3)*d2gm(2,3))+gm(2,3)*(6*gm(1,3)*dgm01(2,2)&
&   *dgm10(2,2)+24*gm(1,2)*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3))&
&   +gm(2,2)*(18*dgm01(2,3)*dgm10(1,2)+3*dgm01(2,2)*dgm10(1,3)+3*dgm01(1,3)&
&   *dgm10(2,2)+18*dgm01(1,2)*dgm10(2,3)+3*gm(1,3)*d2gm(2,2)+18*gm(1,2)&
&   *d2gm(2,3)))
   cm(5,18)=15*gm(1,2)*gm(3,3)*dgm01(2,2)*dgm10(1,2)+15*gm(1,2)&
&   *gm(3,3)*dgm01(1,2)*dgm10(2,2)-6*gm(1,3)**2*dgm01(2,2)*dgm10(2,2)&
&   -3*gm(1,1)*gm(3,3)*dgm01(2,2)*dgm10(2,2)+3*gm(1,2)*gm(1,3)*dgm01(2,3)&
&   *dgm10(2,2)+7.5d0*gm(1,2)**2*dgm01(3,3)*dgm10(2,2)+3*gm(1,2)&
&   *gm(1,3)*dgm01(2,2)*dgm10(2,3)+30*gm(1,2)**2*dgm01(2,3)*dgm10(2,3)&
&   +7.5d0*gm(1,2)**2*dgm01(2,2)*dgm10(3,3)+7.5d0*gm(1,2)**2*gm(3,3)&
&   *d2gm(2,2)-3*gm(2,3)**2*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2)&
&   +gm(1,1)*d2gm(2,2))+gm(2,3)*(30*gm(1,2)*dgm01(2,3)*dgm10(1,2)&
&   +9*gm(1,2)*dgm01(2,2)*dgm10(1,3)+9*gm(1,2)*dgm01(1,3)*dgm10(2,2)&
&   -9*gm(1,1)*dgm01(2,3)*dgm10(2,2)+30*gm(1,2)*dgm01(1,2)*dgm10(2,3)&
&   -9*gm(1,1)*dgm01(2,2)*dgm10(2,3)+9*gm(1,3)*(dgm01(2,2)*dgm10(1,2)&
&   +dgm01(1,2)*dgm10(2,2)+gm(1,2)*d2gm(2,2))+15*gm(1,2)**2*d2gm(2,3)&
&   -3*gm(2,2)*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3)+gm(1,1)&
&   *d2gm(2,3)))+gm(2,2)*(-6*gm(1,2)*dgm01(2,3)*dgm10(1,3)-1.5d0*gm(1,1)&
&   *dgm01(3,3)*dgm10(2,2)-6*gm(1,2)*dgm01(1,3)*dgm10(2,3)-6*gm(1,1)&
&   *dgm01(2,3)*dgm10(2,3)-1.5d0*gm(1,1)*dgm01(2,2)*dgm10(3,3)-3*gm(1,3)&
&   **2*d2gm(2,2)-1.5d0*gm(3,3)*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)&
&   *dgm10(2,2)+gm(1,1)*d2gm(2,2))-6*gm(1,3)*(dgm01(2,3)*dgm10(1,2)&
&   +dgm01(2,2)*dgm10(1,3)+dgm01(1,3)*dgm10(2,2)+dgm01(1,2)*dgm10(2,3)&
&   +gm(1,2)*d2gm(2,3)))
   cm(6,18)=-9*gm(1,1)*gm(2,3)*dgm01(2,2)*dgm10(2,2)+gm(1,2)*(24*gm(1,3)&
&   *dgm01(2,2)*dgm10(2,2)+3*gm(2,3)*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)&
&   *dgm10(2,2)))+gm(1,2)**2*(10.5d0*dgm01(2,3)*dgm10(2,2)+10.5d0*dgm01(2,2)&
&   *dgm10(2,3)+1.5d0*gm(2,3)*d2gm(2,2))-3*gm(2,2)**2*(dgm01(2,3)&
&   *dgm10(1,1)+dgm01(1,1)*dgm10(2,3)+gm(1,1)*d2gm(2,3))+gm(2,2)&
&   *(18*gm(1,2)*dgm01(2,3)*dgm10(1,2)+12*gm(1,2)*dgm01(2,2)*dgm10(1,3)&
&   +12*gm(1,2)*dgm01(1,3)*dgm10(2,2)-10.5d0*gm(1,1)*dgm01(2,3)*dgm10(2,2)&
&   +18*gm(1,2)*dgm01(1,2)*dgm10(2,3)-10.5d0*gm(1,1)*dgm01(2,2)*dgm10(2,3)&
&   -4.5d0*gm(2,3)*(dgm01(2,2)*dgm10(1,1)+dgm01(1,1)*dgm10(2,2)+gm(1,1)&
&   *d2gm(2,2))+12*gm(1,3)*(dgm01(2,2)*dgm10(1,2)+dgm01(1,2)*dgm10(2,2)&
&   +gm(1,2)*d2gm(2,2))+9*gm(1,2)**2*d2gm(2,3))
   cm(7,18)=gm(2,2)*(gm(2,3)*(12*dgm01(2,2)*dgm10(2,2)+3*gm(2,2)&
&   *d2gm(2,2))+gm(2,2)*(9*dgm01(2,3)*dgm10(2,2)+9*dgm01(2,2)*dgm10(2,3)&
&   +2*gm(2,2)*d2gm(2,3)))
   cm(8,18)=1.5d0*gm(2,3)**3*d2gm(2,2)+gm(2,3)*(gm(2,2)*(7.5d0*dgm01(3,3)&
&   *dgm10(2,2)+36*dgm01(2,3)*dgm10(2,3)+7.5d0*dgm01(2,2)*dgm10(3,3))&
&   +gm(3,3)*(15*dgm01(2,2)*dgm10(2,2)+7.5d0*gm(2,2)*d2gm(2,2)))&
&   +gm(2,3)**2*(13.5d0*dgm01(2,3)*dgm10(2,2)+13.5d0*dgm01(2,2)*dgm10(2,3)&
&   +9*gm(2,2)*d2gm(2,3))+gm(2,2)*(-3*gm(2,2)*(dgm01(3,3)*dgm10(2,3)&
&   +dgm01(2,3)*dgm10(3,3))+gm(3,3)*(1.5d0*dgm01(2,3)*dgm10(2,2)&
&   +1.5d0*dgm01(2,2)*dgm10(2,3)-3*gm(2,2)*d2gm(2,3)))
   cm(9,18)=gm(2,3)**2*(6*dgm01(2,2)*dgm10(2,2)+3*gm(2,2)*d2gm(2,2))&
&   +gm(2,2)*(gm(2,2)*(6*dgm01(3,3)*dgm10(2,2)+12*dgm01(2,3)*dgm10(2,3)&
&   +6*dgm01(2,2)*dgm10(3,3))+gm(3,3)*(24*dgm01(2,2)*dgm10(2,2)+6*gm(2,2)&
&   *d2gm(2,2)))+gm(2,2)*gm(2,3)*(18*dgm01(2,3)*dgm10(2,2)+18*dgm01(2,2)&
&   *dgm10(2,3)+6*gm(2,2)*d2gm(2,3))
   cm(10,18)=gm(3,3)**2*(-3*dgm01(2,2)*dgm10(2,2)-1.5d0*gm(2,2)&
&   *d2gm(2,2))+gm(2,3)*(gm(2,3)*(4.5d0*dgm01(3,3)*dgm10(2,2)+30*dgm01(2,3)&
&   *dgm10(2,3)+4.5d0*dgm01(2,2)*dgm10(3,3))-3*gm(2,2)*(dgm01(3,3)&
&   *dgm10(2,3)+dgm01(2,3)*dgm10(3,3))+5*gm(2,3)**2*d2gm(2,3))+gm(3,3)&
&   *(gm(2,2)*(-3*dgm01(3,3)*dgm10(2,2)-6*dgm01(2,3)*dgm10(2,3)-3*dgm01(2,2)&
&   *dgm10(3,3))+4.5d0*gm(2,3)**2*d2gm(2,2)+gm(2,3)*(6*dgm01(2,3)&
&   *dgm10(2,2)+6*dgm01(2,2)*dgm10(2,3)-3*gm(2,2)*d2gm(2,3)))
   cm(1,19)=((270*gm(1,3)**2*dgm01(1,3)-54*gm(1,1)*gm(3,3)*dgm01(1,3)&
&   -54*gm(1,3)*(gm(3,3)*dgm01(1,1)+gm(1,1)*dgm01(3,3)))*dgm10(2,2)&
&   +6*(90*gm(1,3)**2*dgm01(1,2)+gm(1,1)*(-18*gm(3,3)*dgm01(1,2)&
&   -36*gm(2,3)*dgm01(1,3))+gm(1,3)*(-36*gm(2,3)*dgm01(1,1)+180*gm(1,2)&
&   *dgm01(1,3)-36*gm(1,1)*dgm01(2,3))-18*gm(1,2)*(gm(3,3)*dgm01(1,1)&
&   +gm(1,1)*dgm01(3,3)))*dgm10(2,3)+3*dgm01(3,3)*(90*gm(1,2)**2*dgm10(1,3)&
&   +gm(1,1)*(-36*gm(2,3)*dgm10(1,2)-18*gm(2,2)*dgm10(1,3))+gm(1,3)&
&   *(-18*gm(2,2)*dgm10(1,1)+180*gm(1,2)*dgm10(1,2)-18*gm(1,1)*dgm10(2,2))&
&   -36*gm(1,2)*(gm(2,3)*dgm10(1,1)+gm(1,1)*dgm10(2,3)))+3*(90*gm(1,2)&
&   **2*dgm01(1,3)+gm(1,1)*(-36*gm(2,3)*dgm01(1,2)-18*gm(2,2)*dgm01(1,3))&
&   +gm(1,3)*(-18*gm(2,2)*dgm01(1,1)+180*gm(1,2)*dgm01(1,2)-18*gm(1,1)&
&   *dgm01(2,2))-36*gm(1,2)*(gm(2,3)*dgm01(1,1)+gm(1,1)*dgm01(2,3)))&
&   *dgm10(3,3)+6*dgm01(2,3)*(90*gm(1,3)**2*dgm10(1,2)+gm(1,1)*(-18*gm(3,3)&
&   *dgm10(1,2)-36*gm(2,3)*dgm10(1,3))+gm(1,3)*(-36*gm(2,3)*dgm10(1,1)&
&   +180*gm(1,2)*dgm10(1,3)-36*gm(1,1)*dgm10(2,3))-18*gm(1,2)*(gm(3,3)&
&   *dgm10(1,1)+gm(1,1)*dgm10(3,3)))+dgm01(2,2)*(270*gm(1,3)**2*dgm10(1,3)&
&   -54*gm(1,1)*gm(3,3)*dgm10(1,3)-54*gm(1,3)*(gm(3,3)*dgm10(1,1)&
&   +gm(1,1)*dgm10(3,3)))+(90*gm(1,3)**3-54*gm(1,1)*gm(1,3)*gm(3,3))&
&   *d2gm(2,2)+6*(-36*gm(1,1)*gm(1,3)*gm(2,3)+gm(1,2)*(90*gm(1,3)&
&   **2-18*gm(1,1)*gm(3,3)))*d2gm(2,3)+3*(90*gm(1,2)**2*gm(1,3)-18*gm(1,1)&
&   *(gm(1,3)*gm(2,2)+2*gm(1,2)*gm(2,3)))*d2gm(3,3))/36.d0
   cm(2,19)=gm(3,3)*(-3*gm(1,3)*dgm01(2,2)*dgm10(2,2)-12*gm(1,2)&
&   *(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3)))+gm(2,3)**2*(3*dgm01(2,3)&
&   *dgm10(1,2)+7.5d0*dgm01(2,2)*dgm10(1,3)+7.5d0*dgm01(1,3)*dgm10(2,2)&
&   +3*dgm01(1,2)*dgm10(2,3)+7.5d0*gm(1,3)*d2gm(2,2)+3*gm(1,2)*d2gm(2,3))&
&   +gm(2,2)*(gm(1,3)*(10.5d0*dgm01(3,3)*dgm10(2,2)+48*dgm01(2,3)&
&   *dgm10(2,3)+10.5d0*dgm01(2,2)*dgm10(3,3))-6*gm(1,2)*(dgm01(3,3)&
&   *dgm10(2,3)+dgm01(2,3)*dgm10(3,3))+gm(3,3)*(-9*dgm01(2,3)*dgm10(1,2)&
&   -1.5d0*dgm01(2,2)*dgm10(1,3)-1.5d0*dgm01(1,3)*dgm10(2,2)-9*dgm01(1,2)&
&   *dgm10(2,3)-1.5d0*gm(1,3)*d2gm(2,2)-9*gm(1,2)*d2gm(2,3)))+6*gm(2,2)&
&   **2*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3)+gm(1,3)*d2gm(3,3))&
&   +gm(2,3)*(39*gm(1,3)*dgm01(2,3)*dgm10(2,2)+0*gm(1,2)*dgm01(3,3)&
&   *dgm10(2,2)+39*gm(1,3)*dgm01(2,2)*dgm10(2,3)+12*gm(1,2)*dgm01(2,3)&
&   *dgm10(2,3)+0*gm(1,2)*dgm01(2,2)*dgm10(3,3)-3*gm(3,3)*(dgm01(2,2)&
&   *dgm10(1,2)+dgm01(1,2)*dgm10(2,2)+gm(1,2)*d2gm(2,2))+gm(2,2)&
&   *(3*dgm01(3,3)*dgm10(1,2)+24*dgm01(2,3)*dgm10(1,3)+24*dgm01(1,3)&
&   *dgm10(2,3)+3*dgm01(1,2)*dgm10(3,3)+24*gm(1,3)*d2gm(2,3)+3*gm(1,2)&
&   *d2gm(3,3)))
   cm(3,19)=-9*gm(1,3)*gm(2,2)*dgm01(3,3)*dgm10(3,3)+gm(2,3)*(24*gm(1,2)&
&   *dgm01(3,3)*dgm10(3,3)+9*gm(1,3)*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)&
&   *dgm10(3,3)))+gm(3,3)**2*(12*dgm01(2,3)*dgm10(1,2)+3*dgm01(2,2)&
&   *dgm10(1,3)+3*dgm01(1,3)*dgm10(2,2)+12*dgm01(1,2)*dgm10(2,3)&
&   +3*gm(1,3)*d2gm(2,2)+12*gm(1,2)*d2gm(2,3))+1.5d0*gm(2,3)**2*(dgm01(3,3)&
&   *dgm10(1,3)+dgm01(1,3)*dgm10(3,3)+gm(1,3)*d2gm(3,3))+gm(3,3)&
&   *(1.5d0*gm(1,3)*dgm01(3,3)*dgm10(2,2)+12*gm(1,3)*dgm01(2,3)*dgm10(2,3)&
&   +36*gm(1,2)*dgm01(3,3)*dgm10(2,3)+1.5d0*gm(1,3)*dgm01(2,2)*dgm10(3,3)&
&   +36*gm(1,2)*dgm01(2,3)*dgm10(3,3)+gm(2,3)*(12*dgm01(3,3)*dgm10(1,2)&
&   +6*dgm01(2,3)*dgm10(1,3)+6*dgm01(1,3)*dgm10(2,3)+12*dgm01(1,2)&
&   *dgm10(3,3)+6*gm(1,3)*d2gm(2,3)+12*gm(1,2)*d2gm(3,3))-4.5d0*gm(2,2)&
&   *(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3)+gm(1,3)*d2gm(3,3)))
   cm(4,19)=gm(2,2)*(24*gm(1,2)*dgm01(3,3)*dgm10(3,3)+27*gm(1,3)&
&   *(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)))-3*gm(3,3)**2*(dgm01(2,2)&
&   *dgm10(1,2)+dgm01(1,2)*dgm10(2,2)+gm(1,2)*d2gm(2,2))+gm(2,3)&
&   **2*(3*dgm01(3,3)*dgm10(1,2)+6*dgm01(2,3)*dgm10(1,3)+6*dgm01(1,3)&
&   *dgm10(2,3)+3*dgm01(1,2)*dgm10(3,3)+6*gm(1,3)*d2gm(2,3)+3*gm(1,2)&
&   *d2gm(3,3))+gm(3,3)*(33*gm(1,3)*dgm01(2,3)*dgm10(2,2)+6*gm(1,2)&
&   *dgm01(3,3)*dgm10(2,2)+33*gm(1,3)*dgm01(2,2)*dgm10(2,3)+12*gm(1,2)&
&   *dgm01(2,3)*dgm10(2,3)+6*gm(1,2)*dgm01(2,2)*dgm10(3,3)+gm(2,3)&
&   *(6*dgm01(2,3)*dgm10(1,2)+9*dgm01(2,2)*dgm10(1,3)+9*dgm01(1,3)&
&   *dgm10(2,2)+6*dgm01(1,2)*dgm10(2,3)+9*gm(1,3)*d2gm(2,2)+6*gm(1,2)&
&   *d2gm(2,3))+gm(2,2)*(12*dgm01(3,3)*dgm10(1,2)+24*dgm01(2,3)*dgm10(1,3)&
&   +24*dgm01(1,3)*dgm10(2,3)+12*dgm01(1,2)*dgm10(3,3)+24*gm(1,3)&
&   *d2gm(2,3)+12*gm(1,2)*d2gm(3,3)))+gm(2,3)*(gm(1,3)*(12*dgm01(3,3)&
&   *dgm10(2,2)+24*dgm01(2,3)*dgm10(2,3)+12*dgm01(2,2)*dgm10(3,3))&
&   +12*gm(1,2)*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3))+3*gm(2,2)&
&   *(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3)+gm(1,3)*d2gm(3,3)))
   cm(5,19)=-6*gm(1,3)*gm(2,2)*dgm01(3,3)*dgm10(1,3)+1.5d0*gm(1,3)&
&   **2*dgm01(3,3)*dgm10(2,2)+6*gm(1,3)**2*dgm01(2,3)*dgm10(2,3)&
&   +33*gm(1,2)*gm(1,3)*dgm01(3,3)*dgm10(2,3)-6*gm(1,3)*gm(2,2)*dgm01(1,3)&
&   *dgm10(3,3)+1.5d0*gm(1,3)**2*dgm01(2,2)*dgm10(3,3)+33*gm(1,2)&
&   *gm(1,3)*dgm01(2,3)*dgm10(3,3)+15*gm(1,2)**2*dgm01(3,3)*dgm10(3,3)&
&   -3*gm(1,1)*gm(2,2)*dgm01(3,3)*dgm10(3,3)-1.5d0*gm(3,3)**2*(dgm01(2,2)&
&   *dgm10(1,1)+dgm01(1,1)*dgm10(2,2)+gm(1,1)*d2gm(2,2))-3*gm(1,3)&
&   **2*gm(2,2)*d2gm(3,3)-3*gm(2,3)**2*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)&
&   *dgm10(3,3)+gm(1,1)*d2gm(3,3))+gm(3,3)*(24*gm(1,3)*dgm01(2,3)&
&   *dgm10(1,2)+15*gm(1,2)*dgm01(3,3)*dgm10(1,2)+9*gm(1,3)*dgm01(2,2)&
&   *dgm10(1,3)+24*gm(1,2)*dgm01(2,3)*dgm10(1,3)+9*gm(1,3)*dgm01(1,3)&
&   *dgm10(2,2)-4.5d0*gm(1,1)*dgm01(3,3)*dgm10(2,2)+24*gm(1,3)*dgm01(1,2)&
&   *dgm10(2,3)+24*gm(1,2)*dgm01(1,3)*dgm10(2,3)-18*gm(1,1)*dgm01(2,3)&
&   *dgm10(2,3)+15*gm(1,2)*dgm01(1,2)*dgm10(3,3)-4.5d0*gm(1,1)*dgm01(2,2)&
&   *dgm10(3,3)+4.5d0*gm(1,3)**2*d2gm(2,2)+24*gm(1,2)*gm(1,3)*d2gm(2,3)&
&   -9*gm(2,3)*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3)+gm(1,1)&
&   *d2gm(2,3))+7.5d0*gm(1,2)**2*d2gm(3,3)-1.5d0*gm(2,2)*(dgm01(3,3)&
&   *dgm10(1,1)+dgm01(1,1)*dgm10(3,3)+gm(1,1)*d2gm(3,3)))+gm(2,3)&
&   *(9*gm(1,2)*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))-15*gm(1,1)&
&   *(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3))+3*gm(1,3)**2*d2gm(2,3)&
&   +gm(1,3)*(9*dgm01(3,3)*dgm10(1,2)+6*dgm01(2,3)*dgm10(1,3)+6*dgm01(1,3)&
&   *dgm10(2,3)+9*dgm01(1,2)*dgm10(3,3)+9*gm(1,2)*d2gm(3,3)))
   cm(6,19)=-3*gm(1,3)*gm(3,3)*dgm01(2,2)*dgm10(1,2)-12*gm(1,2)&
&   *gm(3,3)*dgm01(2,3)*dgm10(1,2)-3*gm(1,2)*gm(3,3)*dgm01(2,2)*dgm10(1,3)&
&   -3*gm(1,3)*gm(3,3)*dgm01(1,2)*dgm10(2,2)-3*gm(1,2)*gm(3,3)*dgm01(1,3)&
&   *dgm10(2,2)+22.5d0*gm(1,3)**2*dgm01(2,3)*dgm10(2,2)-4.5d0*gm(1,1)&
&   *gm(3,3)*dgm01(2,3)*dgm10(2,2)+9*gm(1,2)*gm(1,3)*dgm01(3,3)*dgm10(2,2)&
&   -12*gm(1,2)*gm(3,3)*dgm01(1,2)*dgm10(2,3)+22.5d0*gm(1,3)**2*dgm01(2,2)&
&   *dgm10(2,3)-4.5d0*gm(1,1)*gm(3,3)*dgm01(2,2)*dgm10(2,3)+36*gm(1,2)&
&   *gm(1,3)*dgm01(2,3)*dgm10(2,3)-4.5d0*gm(1,2)**2*dgm01(3,3)*dgm10(2,3)&
&   +9*gm(1,2)*gm(1,3)*dgm01(2,2)*dgm10(3,3)-4.5d0*gm(1,2)**2*dgm01(2,3)&
&   *dgm10(3,3)-3*gm(1,2)*gm(1,3)*gm(3,3)*d2gm(2,2)-6*gm(1,2)**2*gm(3,3)&
&   *d2gm(2,3)-6*gm(2,3)**2*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)*dgm10(2,3)&
&   +gm(1,1)*d2gm(2,3))+gm(2,3)*(18*gm(1,3)*dgm01(2,3)*dgm10(1,2)&
&   +3*gm(1,2)*dgm01(3,3)*dgm10(1,2)+15*gm(1,3)*dgm01(2,2)*dgm10(1,3)&
&   +18*gm(1,2)*dgm01(2,3)*dgm10(1,3)+15*gm(1,3)*dgm01(1,3)*dgm10(2,2)&
&   -6*gm(1,1)*dgm01(3,3)*dgm10(2,2)+18*gm(1,3)*dgm01(1,2)*dgm10(2,3)&
&   +18*gm(1,2)*dgm01(1,3)*dgm10(2,3)-24*gm(1,1)*dgm01(2,3)*dgm10(2,3)&
&   +3*gm(1,2)*dgm01(1,2)*dgm10(3,3)-6*gm(1,1)*dgm01(2,2)*dgm10(3,3)&
&   +7.5d0*gm(1,3)**2*d2gm(2,2)-1.5d0*gm(3,3)*(dgm01(2,2)*dgm10(1,1)&
&   +dgm01(1,1)*dgm10(2,2)+gm(1,1)*d2gm(2,2))+18*gm(1,2)*gm(1,3)&
&   *d2gm(2,3)+1.5d0*gm(1,2)**2*d2gm(3,3)-4.5d0*gm(2,2)*(dgm01(3,3)&
&   *dgm10(1,1)+dgm01(1,1)*dgm10(3,3)+gm(1,1)*d2gm(3,3)))+gm(2,2)&
&   *(12*gm(1,2)*dgm01(3,3)*dgm10(1,3)-7.5d0*gm(1,1)*dgm01(3,3)*dgm10(2,3)&
&   +12*gm(1,2)*dgm01(1,3)*dgm10(3,3)-7.5d0*gm(1,1)*dgm01(2,3)*dgm10(3,3)&
&   +15*gm(1,3)**2*d2gm(2,3)-3*gm(3,3)*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)&
&   *dgm10(2,3)+gm(1,1)*d2gm(2,3))+gm(1,3)*(12*dgm01(3,3)*dgm10(1,2)&
&   +30*dgm01(2,3)*dgm10(1,3)+30*dgm01(1,3)*dgm10(2,3)+12*dgm01(1,2)&
&   *dgm10(3,3)+12*gm(1,2)*d2gm(3,3)))
   cm(7,19)=-7.5d0*gm(2,2)*gm(3,3)*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)&
&   *dgm10(2,3))+2.5d0*gm(2,3)**3*d2gm(2,2)+gm(2,3)**2*(16.5d0*dgm01(2,3)&
&   *dgm10(2,2)+16.5d0*dgm01(2,2)*dgm10(2,3)+9*gm(2,2)*d2gm(2,3))&
&   +gm(2,2)**2*(0*dgm01(3,3)*dgm10(2,3)+0*dgm01(2,3)*dgm10(3,3)&
&   -3*gm(3,3)*d2gm(2,3))+gm(2,3)*(gm(3,3)*(-3*dgm01(2,2)*dgm10(2,2)&
&   -1.5d0*gm(2,2)*d2gm(2,2))+gm(2,2)*(4.5d0*dgm01(3,3)*dgm10(2,2)&
&   +36*dgm01(2,3)*dgm10(2,3)+4.5d0*dgm01(2,2)*dgm10(3,3)+3*gm(2,2)&
&   *d2gm(3,3)))
   cm(8,19)=gm(3,3)**2*(15*dgm01(2,3)*dgm10(2,2)+15*dgm01(2,2)&
&   *dgm10(2,3)+3*gm(2,3)*d2gm(2,2)+12*gm(2,2)*d2gm(2,3))+gm(2,3)&
&   *(15*gm(2,2)*dgm01(3,3)*dgm10(3,3)+10.5d0*gm(2,3)*(dgm01(3,3)&
&   *dgm10(2,3)+dgm01(2,3)*dgm10(3,3))+1.5d0*gm(2,3)**2*d2gm(3,3))&
&   +gm(3,3)*(31.5d0*gm(2,2)*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3))&
&   +6*gm(2,3)**2*d2gm(2,3)+gm(2,3)*(13.5d0*dgm01(3,3)*dgm10(2,2)&
&   +24*dgm01(2,3)*dgm10(2,3)+13.5d0*dgm01(2,2)*dgm10(3,3)+7.5d0*gm(2,2)&
&   *d2gm(3,3)))
   cm(9,19)=12*gm(2,2)**2*dgm01(3,3)*dgm10(3,3)+21*gm(2,2)*gm(2,3)&
&   *(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3))+gm(3,3)**2*(-3*dgm01(2,2)&
&   *dgm10(2,2)-1.5d0*gm(2,2)*d2gm(2,2))+3*gm(2,3)**3*d2gm(2,3)+gm(2,3)&
&   **2*(7.5d0*dgm01(3,3)*dgm10(2,2)+18*dgm01(2,3)*dgm10(2,3)+7.5d0*dgm01(2,2)&
&   *dgm10(3,3)+3*gm(2,2)*d2gm(3,3))+gm(3,3)*(4.5d0*gm(2,3)**2*d2gm(2,2)&
&   +gm(2,3)*(24*dgm01(2,3)*dgm10(2,2)+24*dgm01(2,2)*dgm10(2,3)+15*gm(2,2)&
&   *d2gm(2,3))+gm(2,2)*(9*dgm01(3,3)*dgm10(2,2)+30*dgm01(2,3)*dgm10(2,3)&
&   +9*dgm01(2,2)*dgm10(3,3)+6*gm(2,2)*d2gm(3,3)))
   cm(10,19)=9*gm(2,3)**2*dgm01(3,3)*dgm10(3,3)+gm(3,3)**3*d2gm(2,2)&
&   +gm(3,3)**2*(1.5d0*dgm01(3,3)*dgm10(2,2)+12*dgm01(2,3)*dgm10(2,3)&
&   +1.5d0*dgm01(2,2)*dgm10(3,3)+6*gm(2,3)*d2gm(2,3)-1.5d0*gm(2,2)&
&   *d2gm(3,3))+gm(3,3)*(-6*gm(2,2)*dgm01(3,3)*dgm10(3,3)+21*gm(2,3)&
&   *(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3))+4.5d0*gm(2,3)&
&   **2*d2gm(3,3))
   cm(1,20)=(gm(1,1)*(gm(3,3)*(-54*dgm01(3,3)*dgm10(1,2)-108*dgm01(2,3)&
&   *dgm10(1,3)-108*dgm01(1,3)*dgm10(2,3)-54*dgm01(1,2)*dgm10(3,3))&
&   -108*gm(2,3)*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3)))+180*gm(1,3)&
&   **3*d2gm(2,3)+gm(1,3)**2*(270*dgm01(3,3)*dgm10(1,2)+540*dgm01(2,3)&
&   *dgm10(1,3)+540*dgm01(1,3)*dgm10(2,3)+270*dgm01(1,2)*dgm10(3,3)&
&   +270*gm(1,2)*d2gm(3,3))+gm(1,3)*(540*gm(1,2)*dgm01(3,3)*dgm10(1,3)&
&   -216*gm(1,1)*dgm01(3,3)*dgm10(2,3)+540*gm(1,2)*dgm01(1,3)*dgm10(3,3)&
&   -216*gm(1,1)*dgm01(2,3)*dgm10(3,3)-108*gm(3,3)*(dgm01(2,3)*dgm10(1,1)&
&   +dgm01(1,1)*dgm10(2,3)+gm(1,1)*d2gm(2,3))-108*gm(2,3)*(dgm01(3,3)&
&   *dgm10(1,1)+dgm01(1,1)*dgm10(3,3)+gm(1,1)*d2gm(3,3)))+gm(1,2)&
&   *(-108*gm(1,1)*dgm01(3,3)*dgm10(3,3)-54*gm(3,3)*(dgm01(3,3)*dgm10(1,1)&
&   +dgm01(1,1)*dgm10(3,3)+gm(1,1)*d2gm(3,3))))/36.d0
   cm(2,20)=(gm(3,3)*(-36*gm(1,3)*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)&
&   *dgm10(2,3))+gm(1,2)*(-54*dgm01(3,3)*dgm10(2,2)-144*dgm01(2,3)&
&   *dgm10(2,3)-54*dgm01(2,2)*dgm10(3,3)))+gm(2,3)**2*(18*dgm01(3,3)&
&   *dgm10(1,2)+180*dgm01(2,3)*dgm10(1,3)+180*dgm01(1,3)*dgm10(2,3)&
&   +18*dgm01(1,2)*dgm10(3,3)+180*gm(1,3)*d2gm(2,3)+18*gm(1,2)*d2gm(3,3))&
&   +gm(2,2)*(-108*gm(1,2)*dgm01(3,3)*dgm10(3,3)+108*gm(1,3)*(dgm01(3,3)&
&   *dgm10(2,3)+dgm01(2,3)*dgm10(3,3))+gm(3,3)*(-54*dgm01(3,3)*dgm10(1,2)&
&   -36*dgm01(2,3)*dgm10(1,3)-36*dgm01(1,3)*dgm10(2,3)-54*dgm01(1,2)&
&   *dgm10(3,3)-36*gm(1,3)*d2gm(2,3)-54*gm(1,2)*d2gm(3,3)))+gm(2,3)&
&   *(144*gm(1,3)*dgm01(3,3)*dgm10(2,2)+720*gm(1,3)*dgm01(2,3)*dgm10(2,3)&
&   -36*gm(1,2)*dgm01(3,3)*dgm10(2,3)+144*gm(1,3)*dgm01(2,2)*dgm10(3,3)&
&   -36*gm(1,2)*dgm01(2,3)*dgm10(3,3)-72*gm(3,3)*(dgm01(2,3)*dgm10(1,2)&
&   +dgm01(1,2)*dgm10(2,3)+gm(1,2)*d2gm(2,3))+144*gm(2,2)*(dgm01(3,3)&
&   *dgm10(1,3)+dgm01(1,3)*dgm10(3,3)+gm(1,3)*d2gm(3,3))))/12.d0
   cm(3,20)=6*gm(1,3)*gm(2,3)*dgm01(3,3)*dgm10(3,3)+6*gm(3,3)**2*(dgm01(3,3)&
&   *dgm10(1,2)+dgm01(2,3)*dgm10(1,3)+dgm01(1,3)*dgm10(2,3)+dgm01(1,2)&
&   *dgm10(3,3)+gm(1,3)*d2gm(2,3)+gm(1,2)*d2gm(3,3))+gm(3,3)*(24*gm(1,2)&
&   *dgm01(3,3)*dgm10(3,3)+15*gm(1,3)*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)&
&   *dgm10(3,3))+3*gm(2,3)*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3)&
&   +gm(1,3)*d2gm(3,3)))
   cm(4,20)=24*gm(1,3)*gm(2,2)*dgm01(3,3)*dgm10(3,3)+gm(2,3)*(6*gm(1,2)&
&   *dgm01(3,3)*dgm10(3,3)+24*gm(1,3)*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)&
&   *dgm10(3,3)))-6*gm(3,3)**2*(dgm01(2,3)*dgm10(1,2)+dgm01(1,2)&
&   *dgm10(2,3)+gm(1,2)*d2gm(2,3))+3*gm(2,3)**2*(dgm01(3,3)*dgm10(1,3)&
&   +dgm01(1,3)*dgm10(3,3)+gm(1,3)*d2gm(3,3))+gm(3,3)*(12*gm(1,3)&
&   *dgm01(3,3)*dgm10(2,2)+36*gm(1,3)*dgm01(2,3)*dgm10(2,3)-9*gm(1,2)&
&   *dgm01(3,3)*dgm10(2,3)+12*gm(1,3)*dgm01(2,2)*dgm10(3,3)-9*gm(1,2)&
&   *dgm01(2,3)*dgm10(3,3)+gm(2,3)*(3*dgm01(3,3)*dgm10(1,2)+18*dgm01(2,3)&
&   *dgm10(1,3)+18*dgm01(1,3)*dgm10(2,3)+3*dgm01(1,2)*dgm10(3,3)&
&   +18*gm(1,3)*d2gm(2,3)+3*gm(1,2)*d2gm(3,3))+12*gm(2,2)*(dgm01(3,3)&
&   *dgm10(1,3)+dgm01(1,3)*dgm10(3,3)+gm(1,3)*d2gm(3,3)))
   cm(5,20)=-9*gm(1,1)*gm(2,3)*dgm01(3,3)*dgm10(3,3)+gm(1,3)*(24*gm(1,2)&
&   *dgm01(3,3)*dgm10(3,3)+3*gm(2,3)*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)&
&   *dgm10(3,3)))-3*gm(3,3)**2*(dgm01(2,3)*dgm10(1,1)+dgm01(1,1)&
&   *dgm10(2,3)+gm(1,1)*d2gm(2,3))+gm(1,3)**2*(10.5d0*dgm01(3,3)&
&   *dgm10(2,3)+10.5d0*dgm01(2,3)*dgm10(3,3)+1.5d0*gm(2,3)*d2gm(3,3))&
&   +gm(3,3)*(12*gm(1,2)*dgm01(3,3)*dgm10(1,3)-10.5d0*gm(1,1)*dgm01(3,3)&
&   *dgm10(2,3)+12*gm(1,2)*dgm01(1,3)*dgm10(3,3)-10.5d0*gm(1,1)*dgm01(2,3)&
&   *dgm10(3,3)+9*gm(1,3)**2*d2gm(2,3)-4.5d0*gm(2,3)*(dgm01(3,3)&
&   *dgm10(1,1)+dgm01(1,1)*dgm10(3,3)+gm(1,1)*d2gm(3,3))+gm(1,3)&
&   *(12*dgm01(3,3)*dgm10(1,2)+18*dgm01(2,3)*dgm10(1,3)+18*dgm01(1,3)&
&   *dgm10(2,3)+12*dgm01(1,2)*dgm10(3,3)+12*gm(1,2)*d2gm(3,3)))
   cm(6,20)=(2*(90*gm(1,3)**2*dgm01(2,3)+gm(3,3)*(-36*gm(1,2)*dgm01(1,3)&
&   -18*gm(1,1)*dgm01(2,3))+gm(2,3)*(-18*gm(3,3)*dgm01(1,1)+180*gm(1,3)&
&   *dgm01(1,3)-18*gm(1,1)*dgm01(3,3))-36*gm(1,3)*(gm(3,3)*dgm01(1,2)&
&   +gm(1,2)*dgm01(3,3)))*dgm10(2,3)+3*(-12*gm(2,3)**2*dgm01(1,1)&
&   -24*gm(1,2)*gm(3,3)*dgm01(1,2)+30*gm(1,3)**2*dgm01(2,2)-6*gm(1,1)&
&   *gm(3,3)*dgm01(2,2)+36*gm(1,2)*gm(1,3)*dgm01(2,3)+gm(2,3)*(36*gm(1,3)&
&   *dgm01(1,2)+36*gm(1,2)*dgm01(1,3)-24*gm(1,1)*dgm01(2,3))-12*gm(1,2)&
&   **2*dgm01(3,3)+gm(2,2)*(-6*gm(3,3)*dgm01(1,1)+60*gm(1,3)*dgm01(1,3)&
&   -6*gm(1,1)*dgm01(3,3)))*dgm10(3,3)+3*dgm01(3,3)*(-12*gm(2,3)&
&   **2*dgm10(1,1)-24*gm(1,2)*gm(3,3)*dgm10(1,2)+30*gm(1,3)**2*dgm10(2,2)&
&   -6*gm(1,1)*gm(3,3)*dgm10(2,2)+36*gm(1,2)*gm(1,3)*dgm10(2,3)+gm(2,3)&
&   *(36*gm(1,3)*dgm10(1,2)+36*gm(1,2)*dgm10(1,3)-24*gm(1,1)*dgm10(2,3))&
&   -12*gm(1,2)**2*dgm10(3,3)+gm(2,2)*(-6*gm(3,3)*dgm10(1,1)+60*gm(1,3)&
&   *dgm10(1,3)-6*gm(1,1)*dgm10(3,3)))+2*dgm01(2,3)*(90*gm(1,3)**2*dgm10(2,3)&
&   +gm(3,3)*(-36*gm(1,2)*dgm10(1,3)-18*gm(1,1)*dgm10(2,3))+gm(2,3)&
&   *(-18*gm(3,3)*dgm10(1,1)+180*gm(1,3)*dgm10(1,3)-18*gm(1,1)*dgm10(3,3))&
&   -36*gm(1,3)*(gm(3,3)*dgm10(1,2)+gm(1,2)*dgm10(3,3)))+2*(90*gm(1,3)&
&   **2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))&
&   *d2gm(2,3)+3*(30*gm(1,3)**2*gm(2,2)+36*gm(1,2)*gm(1,3)*gm(2,3)&
&   -12*gm(1,2)**2*gm(3,3)+gm(1,1)*(-12*gm(2,3)**2-6*gm(2,2)*gm(3,3)))&
&   *d2gm(3,3))/12.d0
   cm(7,20)=5*gm(2,3)**3*d2gm(2,3)+gm(2,3)*(6*gm(2,2)*(dgm01(3,3)&
&   *dgm10(2,3)+dgm01(2,3)*dgm10(3,3))-3*gm(3,3)*(dgm01(2,3)*dgm10(2,2)&
&   +dgm01(2,2)*dgm10(2,3)+gm(2,2)*d2gm(2,3)))+gm(2,3)**2*(4.5d0*dgm01(3,3)&
&   *dgm10(2,2)+30*dgm01(2,3)*dgm10(2,3)+4.5d0*dgm01(2,2)*dgm10(3,3)&
&   +4.5d0*gm(2,2)*d2gm(3,3))+gm(2,2)*(-3*gm(2,2)*dgm01(3,3)*dgm10(3,3)&
&   +gm(3,3)*(-3*dgm01(3,3)*dgm10(2,2)-6*dgm01(2,3)*dgm10(2,3)-3*dgm01(2,2)&
&   *dgm10(3,3)-1.5d0*gm(2,2)*d2gm(3,3)))
   cm(8,20)=6*gm(2,3)**2*dgm01(3,3)*dgm10(3,3)+gm(3,3)*(24*gm(2,2)&
&   *dgm01(3,3)*dgm10(3,3)+18*gm(2,3)*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)&
&   *dgm10(3,3))+3*gm(2,3)**2*d2gm(3,3))+gm(3,3)**2*(6*dgm01(3,3)&
&   *dgm10(2,2)+12*dgm01(2,3)*dgm10(2,3)+6*(dgm01(2,2)*dgm10(3,3)&
&   +gm(2,3)*d2gm(2,3)+gm(2,2)*d2gm(3,3)))
   cm(9,20)=-3*gm(3,3)**2*(dgm01(2,3)*dgm10(2,2)+dgm01(2,2)*dgm10(2,3)&
&   +gm(2,2)*d2gm(2,3))+gm(2,3)*(15*gm(2,2)*dgm01(3,3)*dgm10(3,3)&
&   +13.5d0*gm(2,3)*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3))&
&   +1.5d0*gm(2,3)**2*d2gm(3,3))+gm(3,3)*(1.5d0*gm(2,2)*(dgm01(3,3)&
&   *dgm10(2,3)+dgm01(2,3)*dgm10(3,3))+9*gm(2,3)**2*d2gm(2,3)+gm(2,3)&
&   *(7.5d0*dgm01(3,3)*dgm10(2,2)+36*dgm01(2,3)*dgm10(2,3)+7.5d0*dgm01(2,2)&
&   *dgm10(3,3)+7.5d0*gm(2,2)*d2gm(3,3)))
   cm(10,20)=gm(3,3)*(12*gm(2,3)*dgm01(3,3)*dgm10(3,3)+2*gm(3,3)&
&   **2*d2gm(2,3)+gm(3,3)*(9*dgm01(3,3)*dgm10(2,3)+9*dgm01(2,3)*dgm10(3,3)&
&   +3*gm(2,3)*d2gm(3,3)))
   cm(1,21)=(270*gm(1,3)**2*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))&
&   -54*gm(1,1)*gm(3,3)*(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3))&
&   +90*gm(1,3)**3*d2gm(3,3)+gm(1,3)*(-108*gm(1,1)*dgm01(3,3)*dgm10(3,3)&
&   -54*gm(3,3)*(dgm01(3,3)*dgm10(1,1)+dgm01(1,1)*dgm10(3,3)+gm(1,1)&
&   *d2gm(3,3))))/36.d0
   cm(2,21)=((90*gm(2,3)**2*dgm01(1,3)+gm(3,3)*(-18*gm(1,3)*dgm01(2,2)&
&   -36*gm(1,2)*dgm01(2,3))+gm(2,3)*(-36*gm(3,3)*dgm01(1,2)+180*gm(1,3)&
&   *dgm01(2,3)-36*gm(1,2)*dgm01(3,3))-18*gm(2,2)*(gm(3,3)*dgm01(1,3)&
&   +gm(1,3)*dgm01(3,3)))*dgm10(3,3)+dgm01(3,3)*(90*gm(2,3)**2*dgm10(1,3)&
&   +gm(3,3)*(-18*gm(1,3)*dgm10(2,2)-36*gm(1,2)*dgm10(2,3))+gm(2,3)&
&   *(-36*gm(3,3)*dgm10(1,2)+180*gm(1,3)*dgm10(2,3)-36*gm(1,2)*dgm10(3,3))&
&   -18*gm(2,2)*(gm(3,3)*dgm10(1,3)+gm(1,3)*dgm10(3,3)))+(-36*gm(1,2)&
&   *gm(2,3)*gm(3,3)+gm(1,3)*(90*gm(2,3)**2-18*gm(2,2)*gm(3,3)))&
&   *d2gm(3,3))/12.d0
   cm(3,21)=gm(3,3)*(12*gm(1,3)*dgm01(3,3)*dgm10(3,3)+3*gm(3,3)&
&   *(dgm01(3,3)*dgm10(1,3)+dgm01(1,3)*dgm10(3,3)+gm(1,3)*d2gm(3,3)))
   cm(4,21)=18*gm(1,3)*gm(2,3)*dgm01(3,3)*dgm10(3,3)-3*gm(3,3)&
&   **2*(dgm01(3,3)*dgm10(1,2)+dgm01(1,2)*dgm10(3,3)+gm(1,2)*d2gm(3,3))&
&   +gm(3,3)*(-12*gm(1,2)*dgm01(3,3)*dgm10(3,3)+9*gm(1,3)*(dgm01(3,3)&
&   *dgm10(2,3)+dgm01(2,3)*dgm10(3,3))+9*gm(2,3)*(dgm01(3,3)*dgm10(1,3)&
&   +dgm01(1,3)*dgm10(3,3)+gm(1,3)*d2gm(3,3)))
   cm(5,21)=9*gm(1,3)**2*dgm01(3,3)*dgm10(3,3)-1.5d0*gm(3,3)**2*(dgm01(3,3)&
&   *dgm10(1,1)+dgm01(1,1)*dgm10(3,3)+gm(1,1)*d2gm(3,3))+gm(3,3)&
&   *(-6*gm(1,1)*dgm01(3,3)*dgm10(3,3)+9*gm(1,3)*(dgm01(3,3)*dgm10(1,3)&
&   +dgm01(1,3)*dgm10(3,3))+4.5d0*gm(1,3)**2*d2gm(3,3))
   cm(6,21)=((90*gm(1,3)**2*dgm01(2,3)+gm(3,3)*(-36*gm(1,2)*dgm01(1,3)&
&   -18*gm(1,1)*dgm01(2,3))+gm(2,3)*(-18*gm(3,3)*dgm01(1,1)+180*gm(1,3)&
&   *dgm01(1,3)-18*gm(1,1)*dgm01(3,3))-36*gm(1,3)*(gm(3,3)*dgm01(1,2)&
&   +gm(1,2)*dgm01(3,3)))*dgm10(3,3)+dgm01(3,3)*(90*gm(1,3)**2*dgm10(2,3)&
&   +gm(3,3)*(-36*gm(1,2)*dgm10(1,3)-18*gm(1,1)*dgm10(2,3))+gm(2,3)&
&   *(-18*gm(3,3)*dgm10(1,1)+180*gm(1,3)*dgm10(1,3)-18*gm(1,1)*dgm10(3,3))&
&   -36*gm(1,3)*(gm(3,3)*dgm10(1,2)+gm(1,2)*dgm10(3,3)))+(90*gm(1,3)&
&   **2*gm(2,3)-36*gm(1,2)*gm(1,3)*gm(3,3)-18*gm(1,1)*gm(2,3)*gm(3,3))&
&   *d2gm(3,3))/12.d0
   cm(7,21)=(270*gm(2,3)**2*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3))&
&   -54*gm(2,2)*gm(3,3)*(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3))&
&   +90*gm(2,3)**3*d2gm(3,3)+gm(2,3)*(-108*gm(2,2)*dgm01(3,3)*dgm10(3,3)&
&   -54*gm(3,3)*(dgm01(3,3)*dgm10(2,2)+dgm01(2,2)*dgm10(3,3)+gm(2,2)&
&   *d2gm(3,3))))/36.d0
   cm(8,21)=gm(3,3)*(12*gm(2,3)*dgm01(3,3)*dgm10(3,3)+3*gm(3,3)&
&   *(dgm01(3,3)*dgm10(2,3)+dgm01(2,3)*dgm10(3,3)+gm(2,3)*d2gm(3,3)))
   cm(9,21)=9*gm(2,3)**2*dgm01(3,3)*dgm10(3,3)-1.5d0*gm(3,3)**2*(dgm01(3,3)&
&   *dgm10(2,2)+dgm01(2,2)*dgm10(3,3)+gm(2,2)*d2gm(3,3))+gm(3,3)&
&   *(-6*gm(2,2)*dgm01(3,3)*dgm10(3,3)+9*gm(2,3)*(dgm01(3,3)*dgm10(2,3)&
&   +dgm01(2,3)*dgm10(3,3))+4.5d0*gm(2,3)**2*d2gm(3,3))
   cm(10,21)=gm(3,3)**2*(6*dgm01(3,3)*dgm10(3,3)+gm(3,3)*d2gm(3,3))
 end if
!
!contraction to output scalar
!
 e2nl=0.d0
 do jj=1,((rank+3)*(rank+4))/2
   tmp(:)=0.d0
   do ii=1,((rank+1)*(rank+2))/2
     tmp(:)=tmp(:)+aa(:,ii)*cm(ii,jj)
   end do
   e2nl=e2nl+tmp(1)*bb(1,jj)+tmp(2)*bb(2,jj)
 end do
!factor of 2 multiplied in to drop call to conjugate contraction
!e2nl=0.5d0*e2nl

 ABI_DEALLOCATE(cm)

end subroutine contstr23
!!***

end module m_contstr23
!!***
