/* player.c - low-level audio file player implementation
 *
 * This version uses PortAudio (http://www.portaudio.com) for audio
 * output.
 *
 * Copyright 2010, 2016 Petteri Hintsanen <petterih@iki.fi>
 *
 * This file is part of abx.
 *
 * abx is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * abx is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with abx.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "player.h"
#include "soundfile.h"
#include <assert.h>
#include <glib.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/* Ring buffer size in bytes. */
static const int BUFFER_SIZE = 512 * 1024;
/* Minimum acceptable fill level for the buffer.  This is intended to
 * keep PortAudio stream callback from signalling spaces all the time.
 * MIN_BUFFER_LEVEL must be between 0 and 1, though not too close to
 * zero to prevent buffer underruns. */
static const float MIN_BUFFER_LEVEL = 0.5;

/* State data that gets passed around in callbacks etc. */
typedef struct {
    /* Ring buffer.  We always keep one vacant slot, so that it is
     * relatively easy to distinguish between empty (in == out) and
     * full (in + 1 == out) buffer. */
    struct {
        float *begin;
        float *end;
        float *in;
        float *out;
        int bufsize;
        int nframes;
        int space;      
        GCond space_cond;
        GMutex space_mutex;
        int fill;
        GCond fill_cond;
        GMutex fill_mutex;
    } buffer;

    /* Reader thread data. */
    struct {
        GThread* thread;
        int stop;
        int is_eof;
        GMutex eof_mutex;
    } reader;

    /* Sample file handle. */
    Sound_file *sndfile;
    /* Sample metadata. */
    Metadata metadata;
    /* Current playback status. */
    Player_state state;
    /* Location at the most recent seek, in secods. */
    double origin;
    /* Frames played after the most recent seek. */
    int nplayed;
    /* Has reader hit EOF?  (This is used to avoid locking.)  */
    int reader_eof;
    /* PortAudio output stream. */
    PaStream *stream;
    /* PortAudio output device index. */
    PaDeviceIndex outdev;
    /* PortAudio suggested latency, in millisecods. */
    unsigned int latency;
} Player;

/* "The" player. */
static Player player;
/* Whether PortAudio has been intialized or not. */
static int pa_initialized = 0;
/* Whether other player resources have been allocated or not. */
static int initialized = 0;

/*
 * Reader thread.
 *
 * This thread implements the producer part of the producer-consumer
 * model by reading blocks of audio data from the sound file into the
 * ring buffer.
 *
 * Some notes:
 *
 * - Locks and condition variables are used for synchronization.  They
 *   seem to pose no problems with latencies this high.
 *
 * - A new reader is started after each seek operation.
 *
 * - The reader blocks if there is no free space in the buffer.  It
 *   then waits for a space signal from the playback stream callback
 *   (consumer).
 *
 * - The reader reads from the file until EOF is hit, or until it has
 *   been explicitly requested to terminate by stop_reader.
 *
 * - The reader signals fill after sufficient portion of the buffer
 *   has been filled, or EOF has been hit.
 */
static void *
reader_main(void *arg)
{
    Player *player;
    int nchannels;
    int nframes;
    int nread;
    int eof;
    player = (Player *) arg;
    g_debug("reader: starting");
    /* read in data */
    nchannels = player->metadata.channels;
    eof = 0;
    while (!eof) {
        /* Check for full buffer. */
        g_mutex_lock(&player->buffer.space_mutex);
        if ((player->buffer.in == player->buffer.end
             && player->buffer.out == player->buffer.begin)
            || (player->buffer.in +
                player->metadata.channels == player->buffer.out)) {
            player->buffer.space = 0;
        }
        while (!player->buffer.space && !player->reader.stop) {
            g_debug("reader: waiting for space");
            g_cond_wait(&player->buffer.space_cond,
                        &player->buffer.space_mutex);
        }
        /* Check for stop condition. */
        if (player->reader.stop) {
            g_debug("reader: exiting");
            g_mutex_unlock(&player->buffer.space_mutex);
            return NULL;
        }
        g_mutex_unlock(&player->buffer.space_mutex);

        if (player->buffer.in == player->buffer.end) {
            /* rewind the ring buffer */
            g_debug("reader: rewound the ring buffer");
            player->buffer.in = player->buffer.begin;
        }
        if (player->buffer.in >= player->buffer.out) {
            nframes = ((player->buffer.end - player->buffer.in)
                       / nchannels);
        } else {
            nframes = ((player->buffer.out - player->buffer.in)
                       / nchannels) - 1;
        }
        assert(nframes > 0);
        g_debug("reader: reading %d frames, %d / %d frames in buffer, "
                "buffer begin = %p, in = %p, out = %p, end = %p",
                nframes,
                player->buffer.nframes,
                player->buffer.bufsize,
                (void *) player->buffer.begin,
                (void *) player->buffer.in,
                (void *) player->buffer.out,
                (void *) player->buffer.end);
        nread = read_pcm_data(player->sndfile, player->buffer.in, nframes);
        player->buffer.nframes += nread;
        player->buffer.in += (nread * nchannels);
        assert(player->buffer.nframes <= player->buffer.bufsize);
        assert(player->buffer.in <= player->buffer.end);
        g_debug("reader: read %d frames, %d / %d frames in buffer",
                nread,
                player->buffer.nframes,
                player->buffer.bufsize);
        if (nread != nframes) {
            /* EOF */
            g_debug("reader: eof");
            g_mutex_lock(&player->reader.eof_mutex);
            player->reader.is_eof = 1;
            eof = 1;
            g_mutex_unlock(&player->reader.eof_mutex);
        }

        g_mutex_lock(&player->buffer.fill_mutex);
        if ((player->buffer.nframes
             >= (MIN_BUFFER_LEVEL * player->buffer.bufsize))
            || eof) {
            g_debug("reader: signal fill");            
            player->buffer.fill = 1;
            g_cond_signal(&player->buffer.fill_cond);
        }
        g_mutex_unlock(&player->buffer.fill_mutex);
    }

    return NULL;
}

/*
 * Start a new reader thread for the current sound file.  Playback
 * must be stopped, since this function resets the input buffer.
 */
static void
start_reader(void)
{
    size_t nbytes;
    size_t frame_size;
    assert(player.sndfile);
    assert(player.state.playback == STOPPED);
    assert(player.reader.thread == NULL);
    player.metadata = get_metadata(player.sndfile);

    /* Align the buffer size to the frame size. */
    nbytes = BUFFER_SIZE;
    frame_size = player.metadata.channels * sizeof(float);
    nbytes -= nbytes % frame_size;
    player.buffer.bufsize = nbytes / frame_size;
    player.buffer.end = player.buffer.begin + (nbytes / sizeof(float));

    /* Reset buffer. */
    player.buffer.in = player.buffer.out = player.buffer.begin;
    player.buffer.nframes = 0;
    player.buffer.space = 1;
    player.buffer.fill = 0;
    player.reader.stop = 0;
    player.reader.is_eof = 0;
    player.reader_eof = 0;
    player.reader.thread = g_thread_new("reader", reader_main, &player);
}

/*
 * Terminate the reader thread.  Playback must be stopped, since this
 * function can garble the input buffer.
 */
static void
stop_reader(void)
{
    g_debug("telling reader to terminate");
    assert(player.state.playback == STOPPED);
    if (player.reader.thread == NULL) {
        g_debug("reader not running");
        return;
    }
    g_mutex_lock(&player.buffer.space_mutex);
    player.reader.stop = 1;
    /* Signal the reader in case it is blocked. */
    player.buffer.space = 1;
    g_cond_signal(&player.buffer.space_cond);
    g_mutex_unlock(&player.buffer.space_mutex);
    g_debug("waiting for reader to terminate");
    g_thread_join(player.reader.thread);
    player.reader.thread = NULL;
    g_debug("reader has been terminated");
}

/*
 * Stream callback for PortAudio.  Fill the audio output buffer from
 * the ring buffer, and signal space so that the reader thread can
 * fill the ring buffer.
 *
 * Return PaContinue so that PortAudio will keep on playback, or
 * PaComplete if EOF has been reached.
 */
static int
stream_callback(const void *input, void *output,
                unsigned long nframes,
                const PaStreamCallbackTimeInfo *timeinfo,
                PaStreamCallbackFlags statusflags,
                void *userdata)
{
    Player *player = (Player *) userdata;
    int eof = 0;
    unsigned int nchannels = player->metadata.channels;
    float *outbuf = (float *) output;
    player->nplayed += nframes;
    while (nframes > 0) {
        unsigned int i;
        if (player->buffer.out == player->buffer.end) {
            /* rewind the ring buffer */
            player->buffer.out = player->buffer.begin;
        }

        if (player->buffer.out == player->buffer.in) {
            /* buffer underflow or eof, output silence */
            if (!eof) {
                g_mutex_lock(&player->reader.eof_mutex);
                if (!player->reader.is_eof) {
                    g_warning("buffer underflow");
                } else {
                    eof = 1;
                }
                g_mutex_unlock(&player->reader.eof_mutex);
            }
            while (nframes > 0) {
                for (i = 0; i < nchannels; i++) {
                    *outbuf++ = 0;
                }
                nframes--;
            }
        } else {
            for (i = 0; i < nchannels; i++) {
                *outbuf++ = *player->buffer.out++;
            }
            player->buffer.nframes--;
            nframes--;
        }
    }
    /* update current playback location */
    player->state.location = (player->origin
                              + (1.0 * player->nplayed
                                 / player->metadata.rate));

    /* Signal free space. */
    if (player->buffer.nframes
        < (MIN_BUFFER_LEVEL * player->buffer.bufsize)
        && !player->reader_eof) {
        g_debug("player: signal space");
        g_mutex_lock(&player->reader.eof_mutex);
        if (player->reader.is_eof) player->reader_eof = 1;
        g_mutex_unlock(&player->reader.eof_mutex);       
        g_mutex_lock(&player->buffer.space_mutex);
        player->buffer.space = 1;
        g_cond_signal(&player->buffer.space_cond);
        g_mutex_unlock(&player->buffer.space_mutex);
    }
    
    if (!eof) return paContinue;
    else return paComplete;
}

/*
 * Stream finished callback for PortAudio.
 */
static void
stream_finished_callback(void *userdata)
{
    Player *player = (Player *) userdata;
    player->state.playback = STOPPED;
}

/*
 * Scan for default PortAudio host API output device.  Return the
 * device index, or -1 if no device was found.
 */
static PaDeviceIndex
scan_audio_output(void)
{
    const PaDeviceInfo *devinfo;
    PaDeviceIndex i;
    PaDeviceIndex outdev = -1;
    PaDeviceIndex ndevices;

    ndevices = Pa_GetDeviceCount();
    if (ndevices < 0) {
        g_warning("can't find available audio devices: %s",
                  Pa_GetErrorText(ndevices));
    }
    for (i = 0; i < ndevices; i++) {
        devinfo = Pa_GetDeviceInfo(i);
        if (!devinfo) continue;
        if (i == Pa_GetHostApiInfo
            (devinfo->hostApi)->defaultOutputDevice) {
            outdev = i;
            g_debug("found default host api output device index %d",
                    outdev);
            break;
        }
    }
    return outdev;
}

/*
 * Initialize PortAudio stream for playing sample file f.  Return zero
 * on success, nonzero otherwise.
 */
static int
init_playback_stream(Sound_file *f)
{
    PaStreamParameters strparams;
    PaError pa_rval;
    assert(f);
    assert(player.state.playback == STOPPED);  
    if (player.stream) {
        if (Pa_CloseStream(player.stream) != paNoError) {
            g_warning("failed to close playback stream");
        }
        player.stream = NULL;
    }
    player.metadata = get_metadata(f);
    memset(&strparams, 0, sizeof(PaStreamParameters));
    strparams.channelCount = player.metadata.channels;
    strparams.device = player.outdev;
    strparams.sampleFormat = paFloat32;
    strparams.suggestedLatency = player.latency / 100.0;
    strparams.hostApiSpecificStreamInfo = NULL;
    pa_rval = Pa_OpenStream(&player.stream, NULL, &strparams,
                            player.metadata.rate, 0,
                            paNoFlag, stream_callback,
                            &player);
    if (pa_rval != paNoError) {
        g_warning("can't open audio stream for playback: %s",
                  Pa_GetErrorText(pa_rval));
        player.stream = NULL;
        return 1;
    }
    Pa_SetStreamFinishedCallback(player.stream, stream_finished_callback);
    return 0;
}

/*
 * Initialize the player.  Use the given PortAudio device for audio
 * output, or scan for default host API output device if outdev = -1.
 *
 * The suggested latency is in milliseconds.
 *
 * Return 0 on success, nonzero otherwise.
 */
int
init_player(PaDeviceIndex outdev, unsigned int latency)
{
    PaError pa_rval;
    if (initialized) return 0;

    /* Initialize PortAudio. */
    g_debug("initializing PortAudio");
    pa_rval = Pa_Initialize();
    if (pa_rval != paNoError) {
        g_warning("can't initialize audio subsystem: %s",
                  Pa_GetErrorText(pa_rval));
        return 1;
    } else {
        pa_initialized = 1;
    }
    /* Scan for default output device, if needed. */
    if (outdev == -1) outdev = scan_audio_output();
    if (outdev == -1) {
        g_warning("can't find default output audio device");
        Pa_Terminate();
        pa_initialized = 0;
        return 2;
    }

    /* Initialize player structure. */
    player.buffer.begin = (float *) g_malloc(BUFFER_SIZE);
    g_debug("allocated %d bytes of memory for sample buffer", BUFFER_SIZE);
    player.buffer.end = NULL;
    player.buffer.in = player.buffer.out = NULL;
    player.reader.thread = NULL;
    player.reader.is_eof = 0;
    player.sndfile = NULL;
    player.state.playback = STOPPED;
    player.state.location = 0;
    player.origin = 0;
    player.nplayed = 0;
    player.reader_eof = 0;
    player.stream = NULL;
    player.outdev = outdev;
    player.latency = latency;
    g_cond_init(&player.buffer.space_cond);
    g_mutex_init(&player.buffer.space_mutex);
    g_cond_init(&player.buffer.fill_cond);
    g_mutex_init(&player.buffer.fill_mutex);
    g_mutex_init(&player.reader.eof_mutex);    

    g_debug("player init ok");
    initialized = 1;
    return 0;
}

/*
 * Close player and free allocated resources.
 */
void
close_player(void)
{
    g_debug("closing player");
    if (initialized) {
        /* Stop everything. */
        stop_player();
        if (player.stream) {
            if (Pa_CloseStream(player.stream) != paNoError) {
                g_warning("failed to close playback stream");
            }
            player.stream = NULL;
        }
        stop_reader();
        g_free(player.buffer.begin);
        g_cond_clear(&player.buffer.space_cond);
        g_mutex_clear(&player.buffer.space_mutex);
        g_cond_clear(&player.buffer.fill_cond);
        g_mutex_clear(&player.buffer.fill_mutex);
        initialized = 0;
    }

    if (pa_initialized) {
        g_debug("terminating PortAudio");
        if (Pa_Terminate() != paNoError) {
            g_warning("failed to shut down audio subsystem");
        } else {
            pa_initialized = 0;
        }
    }
}

/*
 * Return the current playback state.
 */
Player_state
get_player_state(void)
{
    return player.state;
}

/*
 * Start playing sound file f from the specified offset, counting from
 * the beginning.  This function resets the input sample buffer and
 * PortAudio playback stream.
 *
 * Return 0 on success, nonzero otherwise.
 */
int
start_player(Sound_file* f, double offset)
{
    g_debug("starting player for file %p from offset %lf", f, offset);
    stop_player();
    if (init_playback_stream(f) != 0) return 1;
    player.sndfile = f;
    if (seek_player(offset, SEEK_SET) == -1) return 2;
    g_debug("starting playback");
    Pa_StartStream(player.stream);
    player.state.playback = PLAYING;
    return 0;
}

/*
 * Stop playback.
 */
void
stop_player(void)
{
    if (player.stream) Pa_StopStream(player.stream);
    player.state.playback = STOPPED;
    g_debug("stopped player");
}

/*
 * Pause or resume playback.  Return the new player state.
 */
Player_state
pause_or_resume_player(void)
{
    switch (player.state.playback) {
    case PLAYING:
        Pa_StopStream(player.stream);
        player.state.playback = PAUSED;
        break;
    case PAUSED:
        Pa_StartStream(player.stream);
        player.state.playback = PLAYING;
        break;
    default:
        break;
    }
    return player.state;
}

/*
 * Seek player.  Semantics are as in fseek, offset is in seconds.  If
 * playback is in progress, it is continued from the new location.
 * Return the new location, or -1 on error.
 */
double
seek_player(double offset, int whence)
{
    int was_playing;
    double loc;
    g_debug("seeking to %lf, whence %d", offset, whence);

    /* stop playback and reset state */
    was_playing = (player.state.playback == PLAYING);
    stop_player();
    stop_reader();

    switch (whence) {
    case SEEK_SET:
        loc = offset;
        break;
    case SEEK_CUR:
        loc = player.state.location + offset;
        break;
    case SEEK_END:
        loc = player.metadata.duration + offset;
        break;
    }
    if (loc < 0 || loc > player.metadata.duration) return -1;
    player.state.location = loc;
    player.origin = loc;
    player.nplayed = 0;
    player.buffer.in = player.buffer.out = player.buffer.begin;
    player.buffer.nframes = 0;
    if (seek_sound_file(player.sndfile, loc, SEEK_SET) == -1) {
        return -1;
    }
    g_debug("seek done");

    start_reader();
    g_debug("waiting for reader to fill up buffer");
    g_mutex_lock(&player.buffer.fill_mutex);
    while (!player.buffer.fill) {
        g_cond_wait(&player.buffer.fill_cond,
                    &player.buffer.fill_mutex);
    }
    player.buffer.fill = 0;
    g_mutex_unlock(&player.buffer.fill_mutex);
    g_debug("buffer filled enough or EOF");

    if (was_playing) {
        g_debug("resuming playback");
        Pa_StartStream(player.stream);
        player.state.playback = PLAYING;
    }

    return loc;
}
