## monad-unlift

[![Build Status](https://travis-ci.org/fpco/monad-unlift.svg?branch=master)](https://travis-ci.org/fpco/monad-unlift)

Typeclasses for providing for unlifting of monad transformers and stacks.

Note that concrete implementations of common transformers implementing
these type classes are provided by the monad-unlift-ref package.

## Synopsis

```haskell
import Control.Concurrent.Async
import Control.Monad.Trans.Unlift
import Control.Monad.Trans.RWS.Ref
import Control.Monad.IO.Class
import Data.Mutable

-- Some artbirary data type for the MonadReader
data SomeEnv = SomeEnv Int

myFunc :: RWSRefT
    -- The WriterT piece is contained by an IORef
    IORef
    -- For efficiency, we store the state in a primitive reference
    (PRef RealWorld)
    SomeEnv   -- Reader
    [String]  -- Writer
    Int       -- State
    IO
    (String, String)
myFunc = do
    -- Get the unlift function. Due to weaknesses in ImpredicativeTypes, we
    -- need to use a newtype wrapper. You can also use askRunBase.
    --
    -- If you want to just unwrap one transformer layer, use
    -- askUnlift/askRun/Unlift.
    UnliftBase run <- askUnliftBase

    -- Note that we can use unlift to turn our transformer actions into IO
    -- actions. Unlike the standard RWST, actions from separate threads are
    -- both retained due to mutable references.
    --
    -- In real life: you shouldn't rely on this working, as RWST is not thread
    -- safe. This example is provided as a good demonstration of the type level
    -- functionality.
    liftIO $ concurrently (run foo) (run bar)
  where
    foo = do
        tell ["starting foo"]
        modify (+ 1)
        tell ["leaving foo"]
        return "foo is done"
    bar = do
        tell ["starting bar"]
        SomeEnv e <- ask
        modify (+ e)
        tell ["leaving bar"]
        return "bar is done"

main :: IO ()
main = do
    ((w, x), y, z) <- runRWSRefT myFunc (SomeEnv 5) 6
    print w -- foo is done
    print x -- bar is done
    print y -- 12 = 6 + 5 + 1
    print z -- starting and leaving statements, order ambiguous
```

## Overview

A common pattern is to have some kind of a monad transformer, and want to pass
an action into a function that requires actions in a base monad. That sounds a
bit abstract, so let's give a concrete example:

```haskell
-- From async
concurrently :: IO a -> IO b -> IO (a, b)

func1 :: ReaderT Foo IO String
func2 :: ReaderT Foo IO Double

doBoth :: ReaderT Foo IO (String, Double)
doBoth = _
```

Doing this manually is possible, but a bit tedious:

```haskell
doBoth :: ReaderT Foo IO (String, Double)
doBoth = ReaderT $ \foo -> concurrently
    (runReaderT func1 foo)
    (runReaderT func2 foo)
```

This also doesn't generalize at all; you'll be stuck writing `concurrently`
variants for every monad transformer stack. Fortunately, the `monad-control`
package generalizes this to a large number of transformer stacks. Let's
implement our generalized `concurrently`:

```haskell
concurrentlyG :: MonadBaseControl IO m
              => m a -> m b -> m (StM m a, StM m b)
concurrentlyG f g = liftBaseWith $ \run ->
    concurrently (run f) (run g)
```

Notice how, in the signature for `concurrentlyG`, we no longer return `(a, b)`,
but `(StM m a, StM m b)`. This is because there may be additional monadic
context for each thread of execution, and we have no way of merging these
together in general. Some examples of context are:

* With `WriterT`, it's the values that you called `tell` on
* With `EitherT`, the returned value may not exist at all

In addition to this difficulty, many people find the types in `monad-control`
difficult to navigate, due to their extreme generality (which is in fact the
power of that package!).

There is a subset of these transformer stacks that are in fact [monad
morphisms](http://www.stackage.org/package/mmorph). Simply stated, these are
transformer stacks that are isomorphic to `ReaderT`. For these monads, there is
not context in the returned value. Therefore, there's no need to combine
returned states or deal with possibly missing values.

This concept is represented by the monad-unlift package, which provides a pair of typeclasses for these kinds of transformer stacks. Before we dive in, let's see how we solve our `concurrentlyG` problem with it:

```haskell
concurrentlyG :: MonadBaseUnlift IO m
              => m a -> m b -> m (a, b)
concurrentlyG f g = do
    UnliftBase run <- askUnliftBase
    liftBase $ concurrently (run f) (run g)
```

Notice how we get `(a, b)` in the return type as desired. There's no need to
unwrap values or deal with context.

### MonadTransUnlift

`MonadTransUnlift` is a class for any monad transformer which is isomorphic
to `ReaderT`, in the sense that the environment can be captured and applied
later. Some interesting cases in this space are:

* `IdentityT` and things isomorphic to it; in this case, you can think of the environment as being `()`
* Transformers which contain a mutable reference in their environment. This allows them to behave like stateful transformers (e.g., `StateT` or `WriterT`), but still obey the monad morphism laws. (See below for more details.)

Due to weaknesses in GHC's ImpredicativeTypes, we have a helper datatype to
allow for getting polymorphic unlift functions, appropriately named `Unlift`.
For many common cases, you can get away with using `askRun` instead, e.g.:

```haskell
bar :: ReaderT Foo IO ()

baz :: ReaderT Foo IO ()
baz = do
    run <- askRun
    liftIO $ void $ forkIO $ run bar
```

Using `Unlift`, this would instead be:

```haskell
    Unlift run <- askUnlift
    liftIO $ void $ forkIO $ run bar
```

or equivalently:

```haskell
    u <- askUnlift
    liftIO $ void $ forkIO $ unlift u bar
```

### MonadBaseUnlift

`MonadBaseUnlift` extends this concept to entire transformer stacks. This is
typically the typeclass that people end up using. You can think of these two
typeclasses in exactly the same way as `MonadTrans` and `MonadIO`, or more
precisely `MonadTrans` and `MonadBase`.

For the same ImpredicativeTypes reason, there's a helper type `UnliftBase`.
Everything we just discussed should transfer directly to `MonadBaseUnlift`,
so learning something new isn't necessary. For example, you can rewrite the
last snippet as:

```haskell
    u <- askUnliftBase
    liftIO $ void $ forkIO $ unliftBase u bar
```

### Reference transformers

When playing transformer stack games with a transformer like `StateT`, it's
common to accidentally discard state modifications. Additionally, in the case
of runtime exceptions, it's usually impossible to retain the state. (Similar
statements apply to `WriterT` and `RWST`, both in strict and lazy variants.)

Another approach is to use a `ReaderT` and hold onto a mutable reference. This
is problematic since there's no built in support for operations like `get`,
`put`, or `tell`. What we want is to have a `MonadState` and/or `MonadWriter`
instance.

To address this case, this package includes variants of those transformers that
use mutable references. These references are generic using the
[mutable-containers](http://www.stackage.org/package/mutable-containers)
package, which allows you to have highly efficient references like `PRef`
instead of always using boxed references like `IORef`.

Note that, for generality, the reference transformers take type parameters
indicating which mutable reference type to use. Some examples you may use are:

* `IORef` for boxed references in `IO`
* `STRef s` for boxed references in `ST`
* `PRef RealWorld` for an unboxed reference in `IO`

See the synopsis for a complete example.

### conduit

The `transPipe` function in conduit has caused confusion in the past due to its
requirement of provided functions to obey monad morphism laws. This package
makes a good companion to conduit to simplify that function's usage.

### Other notable instances

Both the `HandlerT` transformer from yesod-core and `LoggingT`/`NoLoggingT` are
valid monad morphisms. `HandlerT` is in fact my first example of using the
"environment holding a mutable reference" technique to overcome exceptions
destroying state.

```haskell
{-# LANGUAGE FlexibleContexts  #-}
{-# LANGUAGE OverloadedStrings #-}
{-# LANGUAGE TemplateHaskell   #-}
import Control.Concurrent.Async
import Control.Monad.IO.Class
import Control.Monad.Logger
import Control.Monad.Trans.Unlift

main :: IO ()
main = runStdoutLoggingT foo

foo :: (MonadLogger m, MonadBaseUnlift IO m, MonadIO m) => m ()
foo = do
    run <- askRunBase
    a <- liftIO $ async $ run $ $logDebug "Hello World!"
    liftIO $ wait a
```
