"""Plugin for testing that data staging works."""

import os
import re
import tempfile
from arcnagios import arcutils, nagutils
from arcnagios.ce.jobplugin import JobPlugin
from arcnagios.utils import log_process_error

def _split_url_urloptions(urlspec):
    if ';' in urlspec:
        xs = urlspec.split(';')
        return (xs[0], xs[1:])
    else:
        return (urlspec, [])

_lfc_re = re.compile(r'lfc://([a-z+-]+://[^@]+)@.*')

class StagingJobPlugin(JobPlugin):

    def staged_outputs(self):
        urls = self.getconf_strlist('staged_outputs', default = [])
        urls = list(map(_split_url_urloptions, urls))
        return [('%s-out-%d'%(self.test_name, i), url, urloptions)
                for i, (url, urloptions) in zip(range(0, len(urls)), urls)]

    def staged_inputs(self):
        urls = self.getconf_strlist('staged_inputs', default = [])
        urls = list(map(_split_url_urloptions, urls))
        return [('%s-in-%d'%(self.test_name, i), url, urloptions)
                for i, (url, urloptions) in zip(range(0, len(urls)), urls)]

    def upload_test_file(self, url):
        try:
            fd, fn = tempfile.mkstemp(prefix = 'tmp-check_arcce_submit-',
                                      text = True)
            fh = os.fdopen(fd, 'w')
            fh.write('This file was uploaded for use by ARC-CE '
                     'staging probes.\n')
            fh.close()
        except OSError as exn:
            self.log.warning('Failed to create a temporary file '
                             'for upload, using /dev/null: %s' % exn)
            fn = '/dev/null'

        self.log.info('Uploading %s to %s.', fn, url)
        r = self.arcclient.arccp(fn, url)
        if fn != '/dev/null':
            os.unlink(fn)
        if r.is_error():
            self.log.warning('Failed to create %s: %s', url, r.error)

    def upload_missing_test_files(self):
        urls = self.getconf_strlist('upload_if_missing', default = [])
        for url in urls:
            try:
                mo = re.match(_lfc_re, url)
                if mo:
                    rep_url = mo.group(1)
                    rep_urls = self.arcclient.arcls_L(url).get()
                    if rep_urls == []:
                        self.upload_test_file(url)
                    elif not rep_url in rep_urls:
                        self.arcclient.arccp(url, rep_url).get()
                        self.arcclient.arccp(rep_url, url, transfer=False).get()
                elif self.arcclient.arcls(url).get() == []:
                    self.upload_test_file(url)
            except arcutils.CalledProcessError as exn:
                log_process_error(self.log, exn, prefix = 'arcls')
                self.upload_test_file(url)

    def write_script(self, fh):
        self.upload_missing_test_files()
        fh.write('# Test "%s" handled by the "staging" job-plugin.\n'
                 % self.test_name)
        for filename, _, _ in self.staged_outputs():
            fh.write('hostname >%s\n'%filename)
        fh.write('\n')

    def check(self, report, jobdir, stored_urls):
        for url in stored_urls:
            try:
                filenames = self.arcclient.arcls(url).get()
                if filenames == []:
                    report.log.error('Could not list %s.', url)
                    report.update_states(nagutils.CRITICAL)
                elif len(filenames) > 1:
                    report.log.warning('Got multiple entries from %s:', url)
                    for filename in filenames:
                        report.log.warning('  %s', filename)
            except arcutils.CalledProcessError as exn:
                log_process_error(report.log, exn)
                report.update_status(nagutils.CRITICAL)
