\name{anova.mppm}
\alias{anova.mppm}
\title{ANOVA for Fitted Point Process Models for Replicated Patterns}
\description{
  Performs analysis of deviance for one or more
  point process models fitted to replicated point pattern data.
}
\usage{
  \method{anova}{mppm}(object, \dots,
                  test=NULL, adjust=TRUE,
                  fine=FALSE, warn=TRUE)
}
\arguments{
  \item{object}{
    Object of class \code{"mppm"} representing a
    point process model that was fitted to replicated point patterns.
  }
  \item{\dots}{
    Optional. Additional objects of class \code{"mppm"}.
  }
  \item{test}{
    Type of hypothesis test to perform. 
    A character string, partially matching one of
    \code{"Chisq"}, \code{"LRT"},
    \code{"Rao"}, \code{"score"}, \code{"F"} or \code{"Cp"},
    or \code{NULL} indicating that no test should be performed.
  }
  \item{adjust}{
    Logical value indicating whether to correct the
    pseudolikelihood ratio when some of the models are not Poisson
    processes.
  }
  \item{fine}{
    Logical value passed to \code{\link{vcov.ppm}}
    indicating whether to use a quick estimate 
    (\code{fine=FALSE}, the default) or a slower, more accurate
    estimate (\code{fine=TRUE}) of the variance of the fitted
    coefficients of each model. 
    Relevant only when some of the models are not Poisson
    and \code{adjust=TRUE}.
  }
  \item{warn}{
    Logical value indicating whether to issue warnings
    if problems arise.
  }
}
\value{
  An object of class \code{"anova"}, or \code{NULL}.
}
\details{
  This is a method for \code{\link{anova}} for comparing several
  fitted point process models of class \code{"mppm"},
  usually generated by the model-fitting function \code{\link{mppm}}).

  If the fitted models are all Poisson point processes,
  then this function performs an Analysis of Deviance of
  the fitted models. The output shows the deviance differences
  (i.e. 2 times log likelihood ratio),
  the difference in degrees of freedom, and (if \code{test="Chi"})
  the two-sided p-values for the chi-squared tests. Their interpretation
  is very similar to that in \code{\link{anova.glm}}.

  If some of the fitted models are \emph{not} Poisson point processes,
  the `deviance' differences in this table are
  'pseudo-deviances' equal to 2 times the differences
  in the maximised values of the log pseudolikelihood (see
  \code{\link{ppm}}). It is not valid to compare these
  values to the chi-squared distribution. In this case,
  if \code{adjust=TRUE} (the default), the
  pseudo-deviances will be adjusted using the method of Pace et al
  (2011) and Baddeley, Turner and Rubak (2015)
  so that the chi-squared test is valid.
  It is strongly advisable to perform this adjustment.

  The argument \code{test} determines which hypothesis test, if any, will
  be performed to compare the models. The argument \code{test}
  should be a character string, partially matching one of
  \code{"Chisq"}, \code{"F"} or \code{"Cp"},
  or \code{NULL}. The first option \code{"Chisq"} gives
  the likelihood ratio test based on the asymptotic chi-squared
  distribution of the deviance difference.
  The meaning of the other options is explained in
  \code{\link{anova.glm}}.
  For random effects models, only \code{"Chisq"} is
  available, and again gives the likelihood ratio test.
}
\section{Error messages}{
  An error message that reports
  \emph{system is computationally singular} indicates that the
  determinant of the Fisher information matrix of one of the models
  was either too large or too small for reliable numerical calculation.
  See \code{\link{vcov.ppm}} for suggestions on how to handle this.
}
\seealso{
  \code{\link{mppm}}
}
\examples{
 H <- hyperframe(X=waterstriders)
 #' test for loglinear trend in x coordinate
 mod0 <- mppm(X~1, data=H, Poisson())
 modx <- mppm(X~x, data=H, Poisson())
 anova(mod0, modx, test="Chi")
 # not significant
 anova(modx, test="Chi")
 # not significant

 #' test for inhibition
 mod0S <- mppm(X~1, data=H, Strauss(2))
 anova(mod0, mod0S, test="Chi")
 # significant! 

 #' test for trend after accounting for inhibition
 modxS <- mppm(X~x, data=H, Strauss(2))
 anova(mod0S, modxS, test="Chi")
 # not significant
}
\references{
  Baddeley, A., Rubak, E. and Turner, R. (2015)
  \emph{Spatial Point Patterns: Methodology and Applications with R}.
  London: Chapman and Hall/CRC Press. 

  Baddeley, A., Turner, R. and Rubak, E. (2015)
  Adjusted composite likelihood ratio test for Gibbs point processes.
  \emph{Journal of Statistical Computation and Simulation}
  \bold{86} (5) 922--941.
  DOI: 10.1080/00949655.2015.1044530.

  Pace, L., Salvan, A. and Sartori, N. (2011)
  Adjusting composite likelihood ratio statistics.
  \emph{Statistica Sinica} \bold{21}, 129--148.
}
\author{
  Adrian Baddeley, Ida-Maria Sintorn and Leanne Bischoff.
  Implemented by 
  \spatstatAuthors.
}
\keyword{spatial}
\keyword{models}
\keyword{methods}
 
 
