/* handsy, Copyright (c) 2018 Jamie Zawinski <jwz@jwz.org>
 *
 * Permission to use, copy, modify, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation.  No representations are made about the suitability of this
 * software for any purpose.  It is provided "as is" without express or 
 * implied warranty.
 *
 * Handsy's various animation key-frames.  I made these by "hand" with -debug.
 *
 * I considered using the Leap Motion Controller API to snapshot real hands,
 * but that device is crap at detecting poses with any precision.
 */

#ifndef __HANDSY_ANIM__
#define __HANDSY_ANIM__

/* For reference:

   hand { joint { thumb  { distal, proximal, metacarpal, 0 }
                  index  { distal, intermediate, proximal, metacarpal }
                  middle { distal, intermediate, proximal, metacarpal }
                  ring   { distal, intermediate, proximal, metacarpal }
                  pinky  { distal, intermediate, proximal, metacarpal }}
           base  { thumb, index, middle, ring, pinky }
           wrist { Z, X, rot }
           pos   { X, Y, Z }}
   duration, pause
   pos[3], rot[3]
 */

/**************************************************************************/

#define NNN 1.8

static const hand open_palm = {
  {{ 0.20, 0.20, 0.20, 0 },
   { 0, 0, 0, 0 },
   { 0, 0, 0, 0 },
   { 0, 0, 0, 0 },
   { 0, 0, 0, 0 }},
  { 0, 0, 0, 0, 0 },
  { 0, 0, 0 },
  { 0, 0, 0 },
  True
};

static const hand closed_hand = {
  {{ 0.86, 1.60, 0, 0 },
   { 1.60, 1.60, 1.60, 0 },
   { 1.60, 1.60, 1.60, 0 },
   { 1.60, 1.60, 1.60, 0 },
   { 1.60, 1.60, 1.60, 0 }},
  { -1.70, 0, 0, 0, 0 },
  { 0.78, 0, 0 },
  { 0, 0, 0 },
  True
};

static const hand closed_flat = {
  {{ 0, 0, 0, 0 },
   { 0, 0, 0, 0 },
   { 0, 0, 0, 0 },
   { 0, 0, 0, 0 },
   { 0, 0, 0, 0 }},
  { 0, 0, 0, 0, 0 },
  { 1.68, 0, 0 },
  { 0, 0, -0.08 },
  True
};

static const hand open_wide = {
  {{ 0, 0, 0, 0 },
   { 0, 0, 0, 0 },
   { 0, 0, 0, 0 },
   { 0, 0, 0, 0 },
   { 0, 0, 0, 0 }},
  { 0, -0.25, -0.08, 0.08, 0.25 },
  { 0, 0, 0 },
  { 0, 0, 0 },
  True
};

static const hand_anim closed_flat_anim[] = {
  { &closed_flat, NNN, 0, { 0, 0, 0 }, { 0, 0, 0 }},
  { 0, }};

static const hand_anim hidden_anim[] = {
  { &closed_flat, NNN, 0, { 0, 0, -0.5 }, { 0, 0, 0 }},
  { 0, }};

static const hand_anim wave_anim[] = {
  { &open_palm, NNN, 0.5, { -0.1, 0, 0 }, { 0, 0, 0 }},
  { &open_palm, 0.05, 0, { -0.1, 0, 0 }, { 0, -0.35, 0 }},
  { &open_palm, 0.10, 0, { -0.1, 0, 0 }, { 0,  0.90, 0 }},
  { &open_palm, 0.10, 0, { -0.1, 0, 0 }, { 0, -0.35, 0 }},
  { &open_palm, 0.10, 0, { -0.1, 0, 0 }, { 0,  0.90, 0 }},
  { &open_palm, 0.10, 0, { -0.1, 0, 0 }, { 0, -0.35, 0 }},
  { &open_palm, 0.10, 0, { -0.1, 0, 0 }, { 0,  0.90, 0 }},
  { &open_palm, 0.10, 0, { -0.1, 0, 0 }, { 0, -0.35, 0 }},
  { &open_palm, 0.10, 0, { -0.1, 0, 0 }, { 0,  0.90, 0 }},
  { &open_palm, 0.10, 0, { -0.1, 0, 0 }, { 0, -0.35, 0 }},
  { &open_palm, 0.10, 0, { -0.1, 0, 0 }, { 0,  0.90, 0 }},
  { &open_palm, 0.10, 0, { -0.1, 0, 0 }, { 0, -0.35, 0 }},
  { &open_palm, 0.05, 1, { -0.1, 0, 0 }, { 0, 0, 0 }},
  { 0, }};

static const hand_anim wave2_anim[] = {
  { &open_palm, NNN, 0, { 0, 0, 0 }, { 0, 0, 0 }},
  { &open_palm, 0.2, 0, { 0, 0, 0 }, { 0,   0, 0 }},
  { &open_palm, 0.2, 0, { 0, 0, 0 }, { 1.3, 0, 0 }},
  { &open_palm, 0.2, 0, { 0, 0, 0 }, { 0,   0, 0 }},
  { &open_palm, 0.2, 0, { 0, 0, 0 }, { 1.3, 0, 0 }},
  { &open_palm, 0.2, 0, { 0, 0, 0 }, { 0,   0, 0 }},
  { &open_palm, 0.2, 0, { 0, 0, 0 }, { 1.3, 0, 0 }},
  { &open_palm, 0.2, 1, { 0, 0, 0 }, { 0,   0.0, 0 }},
  { 0, }};

static const hand_anim jazzhands_anim[] = {
  { &open_palm, NNN, 0.5, { -0.1, 0, 0 }, { 0, 0, 0 }},
  { &open_palm, 0.04, 0, { -0.1, 0, 0 }, { 0, 0.35, -M_PI/4 }},
  { &open_palm, 0.04, 0, { -0.1, 0, 0.1 }, { 0, 0.90, 0 }},
  { &open_palm, 0.04, 0, { -0.1, 0, 0.2 }, { 0, 0.35, -M_PI/4 }},
  { &open_palm, 0.04, 0, { -0.1, 0, 0.3 }, { 0, 0.90, 0 }},
  { &open_palm, 0.04, 0, { -0.1, 0, 0.4 }, { 0, 0.35, -M_PI/4 }},
  { &open_palm, 0.04, 0, { -0.1, 0, 0.4 }, { 0, 0.90, 0 }},
  { &open_palm, 0.04, 0, { -0.1, 0, 0.4 }, { 0, 0.35, -M_PI/4 }},
  { &open_palm, 0.04, 0, { -0.1, 0, 0.4 }, { 0, 0.90, 0 }},
  { &open_palm, 0.04, 0, { -0.1, 0, 0.3 }, { 0, 0.35, -M_PI/4 }},
  { &open_palm, 0.04, 0, { -0.1, 0, 0.2 }, { 0, 0.90, 0 }},
  { &open_palm, 0.04, 0, { -0.1, 0, 0.1 }, { 0, 0.35, -M_PI/4 }},
  { &open_palm, 0.04, 1, { -0.1, 0, 0   }, { 0, 0.35, -M_PI/4 }},
  { 0, }};

/**************************************************************************/

static const hand hand_clap1 = {
  {{ 0.20, 0.20, 0.20, 0 },
   { 0, -0.10, -0.10, 0 },
   { 0, -0.10, -0.10, 0 },
   { 0, -0.10, -0.10, 0 },
   { 0, -0.10, -0.10, 0 }},
  { 0, 0, 0, 0, 0 },
  { 0, 0.9, 0 },
  { 0, 0, 0 },
  True
};

static const hand_anim clap_anim[] = {
  { &open_palm,  NNN, 0,    { -0.3, 0, 0 }, { 0,    0, -M_PI/2 }},
  { &hand_clap1, 0.1, 0.05, { 0.28, 0, 0 }, { -0.5, -1.6, -1.7 }},
  { &open_palm,  0.1, 0,    { -0.3, 0, 0 }, { 0,    0, -M_PI/2 }},
  { &hand_clap1, 0.1, 0.05, { 0.28, 0, 0 }, { -0.5, -1.6, -1.7 }},
  { &open_palm,  0.1, 0,    { -0.3, 0, 0 }, { 0,    0, -M_PI/2 }},
  { &hand_clap1, 0.1, 0.05, { 0.28, 0, 0 }, { -0.5, -1.6, -1.7 }},
  { &open_palm,  0.1, 0,    { -0.3, 0, 0 }, { 0,    0, -M_PI/2 }},
  { &hand_clap1, 0.1, 0.05, { 0.28, 0, 0 }, { -0.5, -1.6, -1.7 }},
  { &open_palm,  0.1, 0,    { -0.3, 0, 0 }, { 0,    0, -M_PI/2 }},
  { &hand_clap1, 0.1, 0.05, { 0.28, 0, 0 }, { -0.5, -1.6, -1.7 }},
  { &open_palm,  0.1, 0,    { -0.3, 0, 0 }, { 0,    0, -M_PI/2 }},
  { &hand_clap1, 0.1, 0.05, { 0.28, 0, 0 }, { -0.5, -1.6, -1.7 }},
  { &open_palm,  0.1, 0,    { -0.3, 0, 0 }, { 0,    0, -M_PI/2 }},
  { &hand_clap1, 0.1, 0.05, { 0.28, 0, 0 }, { -0.5, -1.6, -1.7 }},
  { &open_palm,  0.1, 0,    { -0.3, 0, 0 }, { 0,    0, -M_PI/2 }},
  { &hand_clap1, 0.1, 0.05, { 0.28, 0, 0 }, { -0.5, -1.6, -1.7 }},
  { &open_palm,  0.1, 0,    { -0.3, 0, 0 }, { 0,    0, -M_PI/2 }},
  { &hand_clap1, 0.1, 0.05, { 0.28, 0, 0 }, { -0.5, -1.6, -1.7 }},
  { &open_palm,  0.1, 0,    { -0.3, 0, 0 }, { 0,    0, -M_PI/2 }},
  { &hand_clap1, 0.1, 0.05, { 0.28, 0, 0 }, { -0.5, -1.6, -1.7 }},
  { &open_palm,  0.1, 0,    { -0.3, 0, 0 }, { 0,    0, -M_PI/2 }},
  { 0, }};

static const hand_anim slow_clap_anim[] = {
  { &open_palm,  NNN, 1, { -0.3, 0, 0 }, { 0,    0, -M_PI/2 }},
  { &hand_clap1, 0.1, 1, { 0.28, 0, 0 }, { -0.5, -1.6, -1.7 }},
  { &open_palm,  0.4, 0, { -0.3, 0, 0 }, { 0,    0, -M_PI/2 }},
  { &hand_clap1, 0.1, 1, { 0.28, 0, 0 }, { -0.5, -1.6, -1.7 }},
  { &open_palm,  0.4, 0, { -0.3, 0, 0 }, { 0,    0, -M_PI/2 }},
  { &hand_clap1, 0.1, 1, { 0.28, 0, 0 }, { -0.5, -1.6, -1.7 }},
  { &open_palm,  0.4, 0, { -0.3, 0, 0 }, { 0,    0, -M_PI/2 }},
  { &hand_clap1, 0.1, 1, { 0.28, 0, 0 }, { -0.5, -1.6, -1.7 }},
  { &open_palm,  0.4, 0, { -0.3, 0, 0 }, { 0,    0, -M_PI/2 }},
  { &hand_clap1, 0.1, 1, { 0.28, 0, 0 }, { -0.5, -1.6, -1.7 }},
  { &open_palm,  0.4, 0, { -0.3, 0, 0 }, { 0,    0, -M_PI/2 }},
  { 0, }};

static const hand_anim pray_anim[] = {
  { &hand_clap1, NNN, 0,   { 0, 0, 0 }, { -0.5, -1.6, -1.7 }},
  { &hand_clap1, 0.2, 2.5, { 0.28, 0, 0 }, { -0.5, -1.6, -1.7 }},
  { 0, }};


/**************************************************************************/

static const hand middle_finger = {
  {{ 0.70, 0.70, 0, 0 },
   { 1.60, 1.60, 1.60, 0 },
   { 0, 0, 0, 0 },
   { 1.40, 1.40, 1.40, 0 },
   { 1.60, 1.60, 1.60, 0 }},
  { 0, 0, 0, 0, 0 },
  { -0.04, 0.06, -2.90 },
  { 0, 0, 0 },
  True
};

static const hand_anim middle_anim[] = {
  { &middle_finger, NNN, 1, { 0, 0, 0.0 }, { 0,  0,   0 }},
  { &middle_finger, 0.2, 0, { 0, 0, 0.3 }, { 0,  0.2, 0 }},
  { &middle_finger, 0.2, 0, { 0, 0, 0.0 }, { 0,  0,   0 }},
  { &middle_finger, 0.2, 0, { 0, 0, 0.3 }, { 0, -0.2, 0 }},
  { &middle_finger, 0.2, 0, { 0, 0, 0.0 }, { 0,  0,   0 }},
  { &middle_finger, 0.2, 0, { 0, 0, 0.3 }, { 0,  0.2, 0 }},
  { &middle_finger, 0.2, 0, { 0, 0, 0.0 }, { 0,  0,   0 }},
  { &middle_finger, 0.2, 0, { 0, 0, 0   }, { 0,  0,   0 }},
  { 0, }};

/**************************************************************************/

static const hand hand_ok = {
  {{ 0.87, 0.32, 0.02, 0 },
   { 0.88, 0.91, 0.93, 0 },
   { 0, 0.32, 0.32, 0 },
   { 0, 0.02, 0.02, 0 },
   { -0.20, -0.20, -0.20, 0 }},
  { -0.92, -0.03, -0.06, 0.10, 0.25 },
  { -0.04, 0.06, -2.90 },
  { 0, 0, 0 },
  True
};

static const hand hand_pointing = {
  {{ 0.87, 0.32, 0.02, 0 },
   { 0, 0, 0, 0 },
   { 0.8, 1.60, 1.60, 0 },
   { 0.8, 1.60, 1.60, 0 },
   { 0.8, 1.60, 1.60, 0 }},
  { 0, 0, 0, 0, 0 },
  { 0, 0, 0 },
  { 0, 0, 0 },
  True
};

static const hand_anim ok_anim[] = {
  { &hand_ok, NNN, 1, { 0, 0, 0 }, { 0, 0, M_PI/2 }},
  { 0, }};

static const hand_anim inout1_anim[] = {
  { &hand_ok, NNN, 1, { 0.3, 0.34, 0.4 }, { M_PI/2, 0, M_PI }},
  { 0, }};

static const hand_anim inout2_anim[] = {
  { &hand_pointing, NNN, 1, { -0.8, 0, 0.15 }, { 0, -M_PI/2, -M_PI }},
  { &hand_pointing, 0.2, 0, { -0.1, 0, 0.15 }, { 0, -M_PI/2, -M_PI }},
  { &hand_pointing, 0.2, 0, { -0.8, 0, 0.15 }, { 0, -M_PI/2, -M_PI }},
  { &hand_pointing, 0.2, 0, { -0.1, 0, 0.15 }, { 0, -M_PI/2, -M_PI }},
  { &hand_pointing, 0.2, 0, { -0.8, 0, 0.15 }, { 0, -M_PI/2, -M_PI }},
  { &hand_pointing, 0.2, 0, { -0.1, 0, 0.15 }, { 0, -M_PI/2, -M_PI }},
  { &hand_pointing, 0.2, 0, { -0.8, 0, 0.15 }, { 0, -M_PI/2, -M_PI }},
  { &hand_pointing, 0.2, 0, { -0.1, 0, 0.15 }, { 0, -M_PI/2, -M_PI }},
  { &hand_pointing, 0.2, 1, { -0.8, 0, 0.15 }, { 0, -M_PI/2, -M_PI }},
  { 0, }};


/**************************************************************************/

static const hand tap0 = {
  {{ 0, 0, 0, 0 },
   { 0.06, 0.06, 0.06, 0 },
   { 0.04, 0.04, 0.04, 0 },
   { 0.04, 0.04, 0.04, 0 },
   { 0.02, 0.02, 0.02, 0 }},
  { 0, -0.25, -0.06, 0.10, 0.25 },
  { 1.06, 0.10, 0 },
  { 0, -0.10, 0.04 },
  True
};
static const hand tap1 = {
  {{ 0, 0, 0, 0 },
   { 0.06, 0.06, 0.06, 0 },
   { 0.04, 0.04, 0.04, 0 },
   { 0.24, 0.24, 0.24, 0 },
   { 0.60, 0.60, 0.60, 0 }},
  { 0, -0.25, -0.06, 0.10, 0.25 },
  { 1.06, 0.10, 0 },
  { 0, -0.10, 0.04 },
  True
};
static const hand tap2 = {
  {{ 0, 0, 0, 0 },
   { 0.06, 0.06, 0.06, 0 },
   { 0.30, 0.30, 0.30, 0 },
   { 0.66, 0.66, 0.66, 0 },
   { 0.60, 0.60, 0.60, 0 }},
  { 0, -0.25, -0.06, 0.10, 0.25 },
  { 1.06, 0.10, 0 },
  { 0, -0.10, 0.04 },
  True
};
static const hand tap3 = {
  {{ 0, 0, 0, 0 },
   { 0.26, 0.26, 0.26, 0 },
   { 0.71, 0.71, 0.71, 0 },
   { 0.66, 0.66, 0.66, 0 },
   { 0.60, 0.60, 0.60, 0 }},
  { -0.10, -0.25, -0.06, 0.10, 0.25 },
  { 1.06, 0.10, 0 },
  { 0, -0.10, 0.04 },
  True
};
static const hand tap4 = {
  {{ 0, 0, 0, 0 },
   { 0.72, 0.72, 0.72, 0 },
   { 0.71, 0.71, 0.71, 0 },
   { 0.66, 0.66, 0.66, 0 },
   { 0.60, 0.60, 0.60, 0 }},
  { -0.20, -0.25, -0.06, 0.10, 0.25 },
  { 1.06, 0.10, 0 },
  { 0, -0.10, 0.04 },
  True
};
static const hand tap5 = {
  {{ 0, 0, 0, 0 },
   { 0.72, 0.72, 0.72, 0 },
   { 0.71, 0.71, 0.71, 0 },
   { 0.66, 0.66, 0.66, 0 },
   { 0.60, 0.60, 0.60, 0 }},
  { -0.50, -0.25, -0.06, 0.10, 0.25 },
  { 1.06, 0.10, 0 },
  { 0, -0.10, 0.04 },
  True
};

static const hand_anim tap_anim[] = {
  { &tap5, NNN,  0.5,   { 0, 1, -0.19 }, { 0, 0, 0 }},

  { &tap0, 0.13, 0.8, { 0, 1, -0.19 }, { 0, 0, 0 }},
  { &tap1, 0.13, 0,   { 0, 1, -0.19 }, { 0, 0, 0 }},
  { &tap2, 0.13, 0,   { 0, 1, -0.19 }, { 0, 0, 0 }},
  { &tap3, 0.13, 0,   { 0, 1, -0.19 }, { 0, 0, 0 }},
  { &tap4, 0.13, 0,   { 0, 1, -0.19 }, { 0, 0, 0 }},
  { &tap5, 0.13, 0.8, { 0, 1, -0.19 }, { 0, 0, 0 }},

  { &tap0, 0.13, 0,   { 0, 1, -0.19 }, { 0, 0, 0 }},
  { &tap1, 0.13, 0,   { 0, 1, -0.19 }, { 0, 0, 0 }},
  { &tap2, 0.13, 0,   { 0, 1, -0.19 }, { 0, 0, 0 }},
  { &tap3, 0.13, 0,   { 0, 1, -0.19 }, { 0, 0, 0 }},
  { &tap4, 0.13, 0,   { 0, 1, -0.19 }, { 0, 0, 0 }},
  { &tap5, 0.13, 0.8, { 0, 1, -0.19 }, { 0, 0, 0 }},

  { &tap0, 0.13, 0,   { 0, 1, -0.19 }, { 0, 0, 0 }},
  { &tap1, 0.13, 0,   { 0, 1, -0.19 }, { 0, 0, 0 }},
  { &tap2, 0.13, 0,   { 0, 1, -0.19 }, { 0, 0, 0 }},
  { &tap3, 0.13, 0,   { 0, 1, -0.19 }, { 0, 0, 0 }},
  { &tap4, 0.13, 0,   { 0, 1, -0.19 }, { 0, 0, 0 }},
  { &tap5, 0.13, 0.8, { 0, 1, -0.19 }, { 0, 0, 0 }},

  { &tap0, 0.13, 0,   { 0, 1, -0.19 }, { 0, 0, 0 }},
  { &tap1, 0.13, 0,   { 0, 1, -0.19 }, { 0, 0, 0 }},
  { &tap2, 0.13, 0,   { 0, 1, -0.19 }, { 0, 0, 0 }},
  { &tap3, 0.13, 0,   { 0, 1, -0.19 }, { 0, 0, 0 }},
  { &tap4, 0.13, 0,   { 0, 1, -0.19 }, { 0, 0, 0 }},
  { &tap5, 0.13, 0.8, { 0, 1, -0.19 }, { 0, 0, 0 }},

  { &tap0, 0.13, 0,   { 0, 1, -0.19 }, { 0, 0, 0 }},
  { &tap1, 0.13, 0,   { 0, 1, -0.19 }, { 0, 0, 0 }},
  { &tap2, 0.13, 0,   { 0, 1, -0.19 }, { 0, 0, 0 }},
  { &tap3, 0.13, 0,   { 0, 1, -0.19 }, { 0, 0, 0 }},
  { &tap4, 0.13, 0,   { 0, 1, -0.19 }, { 0, 0, 0 }},
  { &tap5, 0.13, 0.8, { 0, 1, -0.19 }, { 0, 0, 0 }},

  { &tap0, 0.13, 0,   { 0, 1, -0.19 }, { 0, 0, 0 }},
  { &tap1, 0.13, 0,   { 0, 1, -0.19 }, { 0, 0, 0 }},
  { &tap2, 0.13, 0,   { 0, 1, -0.19 }, { 0, 0, 0 }},
  { &tap3, 0.13, 0,   { 0, 1, -0.19 }, { 0, 0, 0 }},
  { &tap4, 0.13, 0,   { 0, 1, -0.19 }, { 0, 0, 0 }},
  { &tap5, 0.13, 0.8, { 0, 1, -0.19 }, { 0, 0, 0 }},
  { 0, }};


/**************************************************************************/

static const hand hand_spock = {
  {{ 0, 0, 0, 0 },
   { 0, 0, 0, 0 },
   { 0, 0, 0, 0 },
   { 0, 0, 0, 0 },
   { 0, 0, 0, 0 }},
  { 0.00, -0.13, -0.25, 0.25, 0.11 },
  { 0, 0.10, 0 },
  { 0, 0, 0 },
  True
};

static const hand_anim spock_anim[] = {
  { &hand_spock, NNN, 1.5, { 0, 0, 0.1 }, { 0, 0, 0 } },
  { 0, }};

/**************************************************************************/

static const hand hand_horns = {
  {{ 0, 1.30, 1.30, 0 },
   { 0, 0, 0, 0 },
   { 0.70, 1.60, 1.60, 0 },
   { 0.70, 1.60, 1.60, 0 },
   { 0, 0, 0, 0 }},
  { -1.50, -0.25, 0, 0, 0.25 },
  { 0, 0.10, 0 },
  { 0, 0, 0 },
  True
};

static const hand_anim horns_anim[] = {
  { &hand_horns, NNN, 1.5, { 0, 0, 0.1 }, { 0, 0, 0 } },
  { &hand_horns, 0.3, 0,   { 0, 0, 0.1 }, { 0.7, 0, 0 } },
  { &hand_horns, 0.3, 0,   { 0, 0, 0.1 }, { 0,   0, 0 } },
  { &hand_horns, 0.3, 0,   { 0, 0, 0.1 }, { 0.7, 0, 0 } },
  { &hand_horns, 0.3, 0.3, { 0, 0, 0.1 }, { 0,   0, 0 } },
  { 0, }};

/**************************************************************************/

static const hand_anim bs1_anim[] = {
  { &hand_horns,  NNN, 1, { 0.2, 0, 0 }, { 0, M_PI/4, M_PI/2 }},
  { 0, }};

static const hand_anim bs2_anim[] = {
  { &closed_hand, NNN, 1, { 0.2, 0, 0 }, { M_PI/4, 0, M_PI/2 }},
  { &open_wide,   0.3, 0, { 0.2, 0, 0 }, { M_PI,   0, M_PI/2 }},
  { &closed_hand, 0.3, 0, { 0.2, 0, 0 }, { M_PI/4, 0, M_PI/2 }},
  { &open_wide,   0.3, 0, { 0.2, 0, 0 }, { M_PI,   0, M_PI/2 }},
  { &closed_hand, 0.3, 0, { 0.2, 0, 0 }, { M_PI/4, 0, M_PI/2 }},
  { &open_wide,   0.3, 0, { 0.2, 0, 0 }, { M_PI,   0, M_PI/2 }},
  { &closed_hand, 0.3, 1, { 0.2, 0, 0 }, { M_PI/4, 0, M_PI/2 }},
  { 0, }};

/**************************************************************************/

static const hand hand_v = {
  {{ 0, 1.60, 1.60, 0 },
   { 0, 0, 0, 0 },
   { 0, 0, 0, 0 },
   { 1.60, 1.60, 1.60, 0 },
   { 1.60, 1.60, 1.60, 0 }},
  { -0.80, -0.22, 0.25, 0, 0 },
  { 0, 0, -M_PI },
  { 0, 0, 0 },
  True
};

static const hand_anim upyours_anim[] = {
  { &hand_v,    NNN, 1, { 0, 0, 0.0 }, { 0,  0,   0 }},
  { &hand_v,    0.2, 0, { 0, 0, 0.3 }, { 0,  0.2, 0 }},
  { &hand_v,    0.2, 0, { 0, 0, 0.0 }, { 0,  0,   0 }},
  { &hand_v,    0.2, 0, { 0, 0, 0.3 }, { 0,  0.2, 0 }},
  { &hand_v,    0.2, 1, { 0, 0, 0.0 }, { 0,  0,   0 }},
  { 0, }};

static const hand_anim peace_anim[] = {
  { &hand_v,    NNN, 0.0, { 0, 0, 0 }, { 0, 0, M_PI }},
  { &hand_v,    0.3, 0.2, { 0, 0, 0 }, { 0, -0.2, M_PI }},
  { &hand_v,    0.8, 2.2, { 0, 0, 0 }, { 0,  0.7, M_PI }},
  { 0, }};

static const hand_anim scissor1_anim[] = {
  { &hand_v,    NNN, 1, { -0.2, 0, 0 }, { M_PI, 0, M_PI }},
  { &hand_v,    0.5, 1, { -0.5, 0, 0 }, { M_PI/2, 0, M_PI/2 }},
  { &hand_v,    0.2, 0, { 0,    0, 0 }, { M_PI/2, 0, M_PI/2 }},
  { &hand_v,    0.2, 0, { -0.2, 0, 0 }, { M_PI/2, 0, M_PI/2 }},
  { &hand_v,    0.2, 0, { 0,    0, 0 }, { M_PI/2, 0, M_PI/2 }},
  { &hand_v,    0.2, 0, { -0.2, 0, 0 }, { M_PI/2, 0, M_PI/2 }},
  { &hand_v,    0.2, 0, { 0,    0, 0 }, { M_PI/2, 0, M_PI/2 }},
  { 0, }};

static const hand_anim scissor2_anim[] = {
  { &hand_v,    NNN, 1, { -0.2, 0, 0 }, { M_PI, 0, M_PI }},
  { &hand_v,    0.5, 1, { -0.5, 0.1, -0.05 }, { 0, -M_PI/2, -M_PI/2 }},
  { &hand_v,    0.2, 0, { 0,    0.1, -0.05 }, { 0, -M_PI/2, -M_PI/2 }},
  { &hand_v,    0.2, 0, { -0.2, 0.1, -0.05 }, { 0, -M_PI/2, -M_PI/2 }},
  { &hand_v,    0.2, 0, { 0,    0.1, -0.05 }, { 0, -M_PI/2, -M_PI/2 }},
  { &hand_v,    0.2, 0, { -0.2, 0.1, -0.05 }, { 0, -M_PI/2, -M_PI/2 }},
  { &hand_v,    0.2, 0, { 0,    0.1, -0.05 }, { 0, -M_PI/2, -M_PI/2 }},
  { 0, }};

/**************************************************************************/

static const hand hand_v2 = {
  {{ 0, 1.60, 1.60, 0 },
   { 0, 0, 0, 0 },
   { 0, 0, 0, 0 },
   { 1.60, 1.60, 1.60, 0 },
   { 1.60, 1.60, 1.60, 0 }},
  { -0.80, -0.06, 0.09, 0, 0 },
  { 0, 0, -M_PI },
  { 0, 0, 0 },
  True
};

static const hand_anim eyeonu_anim[] = {
  { &hand_v2,   NNN, 1,   { 0.3, -1.4, 1.0 }, { M_PI - 1.8, 0, M_PI }},
  { &hand_v2,   0.3, 0.3, { 0.3,  1.4, 0.5 }, { M_PI, 0, -M_PI }},
  { &hand_v2,   0.3, 1,   { 0.3, -1.4, 1.0 }, { M_PI - 1.8, 0, M_PI }},
  { &hand_v2,   0.3, 0.3, { 0.3,  1.4, 0.5 }, { M_PI, 0, -M_PI }},
  { &hand_v2,   0.3, 1,   { 0.3, -1.4, 1.0 }, { M_PI - 1.8, 0, M_PI }},
  { &hand_v2,   0.3, 0.3, { 0.3,  1.4, 0.5 }, { M_PI, 0, -M_PI }},
  { &hand_v2,   0.3, 1,   { 0.3, -1.4, 1.0 }, { M_PI - 1.8, 0, M_PI }},
  { 0, }};

/**************************************************************************/

static const hand hand_trick1 = {
  {{ 0.50, 1.60, 1.60, 0.00 },
   { 0, 1.60, 1.60, 0.00 },
   { 0.36, 0.36, 0.36, 0.00 },
   { 0,    0.16, 0.16, 0.00 },
   { 0.00, 0.00, 0.00, 0.00 }},
  { 0.00, 0.00, 0.00, 0.00, 0.00 },
  { 0.00, 0.00, 0.00 },
  { 0.00, 0.00, 0.00 },
  True
};


static const hand hand_trick2 = {
  {{ 1, 1.60, 1.60, 0.00 },
   { 0, 1.60, 1.60, 0.00 },
   { 0, 1.60, 1.60, 0.00 },
   { 0, 1.60, 1.60, 0.00 },
   { 0, 1.60, 1.60, 0.00 }},
  { 0.00, 0.00, 0.00, 0.00, 0.00 },
  { 0.00, 0.00, 0.00 },
  { 0.00, 0.00, 0.00 },
  True
};

static const hand_anim trick1_anim[] = {
  { &hand_trick2, NNN, 1, { 0.24, 0, 0 }, { 0, M_PI/4, 0 }},
  { &hand_trick1, 0.5, 1, { 0.24, 0, 0 }, { 0, M_PI/4, 0 }},
  { &hand_trick1, 0.3, 0, { 0.24 - 0.4, 0, 0 }, { 0, M_PI/4, 0 }},
  { &hand_trick1, 0.3, 0, { 0.24, 0, 0 }, { 0, M_PI/4, 0 }},
  { &hand_trick1, 0.3, 0, { 0.24 - 0.4, 0, 0 }, { 0, M_PI/4, 0 }},
  { &hand_trick1, 0.3, 0, { 0.24, 0, 0 }, { 0, M_PI/4, 0 }},
  { &hand_trick1, 0.3, 0, { 0.24 - 0.4, 0, 0 }, { 0, M_PI/4, 0 }},
  { &hand_trick1, 0.3, 0, { 0.24, 0, 0 }, { 0, M_PI/4, 0 }},
  { &hand_trick1, 0.3, 1, { 0.24 - 0.4, 0, 0 }, { 0, M_PI/4, 0 }},
  { 0, }};

static const hand_anim trick2_anim[] = {
  { &hand_trick2, NNN, 1, { 0.24, -0.07, 0.51 }, { 0, M_PI + M_PI/4, M_PI }},
  { &hand_trick2, 0.5, 1, { 0.24, -0.07, 0.51 }, { 0, M_PI + M_PI/4, M_PI }},
  { &hand_trick2, 0.3, 0, { 0.24 - 0.4, 0, 0.51 }, { 0, M_PI + M_PI/4, M_PI }},
  { &hand_trick2, 0.3, 0, { 0.24, -0.07, 0.51 }, { 0, M_PI + M_PI/4, M_PI }},
  { &hand_trick2, 0.3, 0, { 0.24 - 0.4, 0, 0.51 }, { 0, M_PI + M_PI/4, M_PI }},
  { &hand_trick2, 0.3, 0, { 0.24, -0.07, 0.51 }, { 0, M_PI + M_PI/4, M_PI }},
  { &hand_trick2, 0.3, 0, { 0.24 - 0.4, 0, 0.51 }, { 0, M_PI + M_PI/4, M_PI }},
  { &hand_trick2, 0.3, 0, { 0.24, -0.07, 0.51 }, { 0, M_PI + M_PI/4, M_PI }},
  { &hand_trick2, 0.3, 1, { 0.24 - 0.4, 0, 0.51 }, { 0, M_PI + M_PI/4, M_PI }},
  { 0, }};

/**************************************************************************/

static const hand thumbs_up = {
  {{ 0.00, 0.00, 0.00, 0.00 },
   { 0.56, 1.60, 1.60, 0.00 },
   { 0.50, 1.60, 1.60, 0.00 },
   { 0.56, 1.60, 1.60, 0.00 },
   { 0.52, 1.60, 1.60, 0.00 }},
  { 0.00, 0.00, 0.00, 0.00, 0.00 },
  { M_PI, 1.52, 0.00 },
  { 0.00, 0.00, 0.00 },
  True
};

static const hand_anim thumbsup_anim[] = {
  { &thumbs_up, NNN,  1, { 0, 0, 0 },    { 0, 0, 0 }},
  { &thumbs_up, 0.25, 0, { 0, 0, 0.25 }, { 0, 0, 0 }},
  { &thumbs_up, 0.25, 0, { 0, 0, 0    }, { 0, 0, 0 }},
  { &thumbs_up, 0.25, 0, { 0, 0, 0.25 }, { 0, 0, 0 }},
  { &thumbs_up, 0.25, 0, { 0, 0, 0    }, { 0, 0, 0 }},
  { &thumbs_up, 0.25, 0, { 0, 0, 0.25 }, { 0, 0, 0 }},
  { &thumbs_up, 0.25, 1, { 0, 0, 0    }, { 0, 0, 0 }},
  { 0, }};

static const hand_anim thumbsdn_anim[] = {
  { &thumbs_up, NNN,  1,   { 0,    0, 0 },    { 0, 0,     0 }},
  { &thumbs_up, 0.25, 0.3, { -0.1, 0, 0.75 }, { 0, -M_PI, 0 }},
  { &thumbs_up, 0.25, 0,   { -0.1, 0, 0.50 }, { 0, -M_PI, 0 }},
  { &thumbs_up, 0.25, 0,   { -0.1, 0, 0.75 }, { 0, -M_PI, 0 }},
  { &thumbs_up, 0.25, 0,   { -0.1, 0, 0.50 }, { 0, -M_PI, 0 }},
  { &thumbs_up, 0.25, 0,   { -0.1, 0, 0.75 }, { 0, -M_PI, 0 }},
  { &thumbs_up, 0.25, 1,   { -0.1, 0, 0.50 }, { 0, -M_PI, 0 }},
  { 0, }};

/**************************************************************************/

static const hand_anim outtahere_anim[] = {
  { &hand_pointing, NNN, 1, { 0, 0,    0 }, { M_PI, 0, 0 }},
  { &thumbs_up,     0.1, 0, { 0, -0.2, 0 }, { M_PI, 0, -M_PI/2 }},
  { &thumbs_up,     0.1, 0, { 0, 0,    0 }, { M_PI, 0, -M_PI/2 }},
  { &thumbs_up,     0.1, 0, { 0, -0.2, 0 }, { M_PI, 0, -M_PI/2 }},
  { &thumbs_up,     0.1, 0, { 0, 0,    0 }, { M_PI, 0, -M_PI/2 }},
  { &thumbs_up,     0.1, 0, { 0, -0.2, 0 }, { M_PI, 0, -M_PI/2 }},
  { &thumbs_up,     0.1, 0, { 0, 0,    0 }, { M_PI, 0, -M_PI/2 }},
  { &thumbs_up,     0.1, 1, { 0, -0.2, 0 }, { M_PI, 0, -M_PI/2 }},
  { 0, }};

static const hand_anim stahhhp_anim[] = {
  { &hand_pointing, NNN, 1,   { 0,    0,    0 }, { M_PI, 0,  0 }},
  { &open_palm,     0.2, 0.5, { -0.1, -0.2, 0 }, { 0,  0,    0 }},
  { &open_palm,     0.2, 0,   { -0.1, -0.2, 0 }, { 0,  -0.2, 0 }},
  { &open_palm,     0.2, 0,   { -0.1, -0.2, 0 }, { 0,   0.2, 0 }},
  { &open_palm,     0.2, 0,   { -0.1, -0.2, 0 }, { 0,  -0.2, 0 }},
  { &open_palm,     0.2, 0,   { -0.1, -0.2, 0 }, { 0,   0.2, 0 }},
  { &open_palm,     0.2, 0,   { -0.1, -0.2, 0 }, { 0,  -0.2, 0 }},
  { &open_palm,     0.2, 1,   { -0.1, -0.2, 0 }, { 0,   0.2, 0 }},
  { &open_palm,     0.2, 0,   { -0.1, -0.6, 0 }, { 0.2, 0.2, 0 }},
  { &open_palm,     0.2, 0,   { -0.1, -0.2, 0 }, { 0.2, 0.2, 0 }},
  { &open_palm,     0.2, 0,   { -0.1, -0.6, 0 }, { 0.2, 0.2, 0 }},
  { &open_palm,     0.2, 0,   { -0.1, -0.2, 0 }, { 0.2, 0.2, 0 }},
  { &open_palm,     0.2, 0,   { -0.1, -0.6, 0 }, { 0.2, 0.2, 0 }},
  { &open_palm,     0.2, 1,   { -0.1, -0.2, 0 }, { 0.2, 0.2, 0 }},
  { 0, }};

static const hand_anim thisguy_anim[] = {
  { &thumbs_up,     NNN, 0.3, { 0, -0.2, 0 }, { M_PI, 0, 0 }},
  { &thumbs_up,     0.5, 0,   { 0, 0,    0 }, { M_PI, 0, -M_PI/2 }},
  { &thumbs_up,     0.1, 0,   { 0, -0.2, 0 }, { M_PI, 0, -M_PI/2 }},
  { &thumbs_up,     0.1, 0,   { 0, 0,    0 }, { M_PI, 0, -M_PI/2 }},
  { &thumbs_up,     0.1, 0,   { 0, -0.2, 0 }, { M_PI, 0, -M_PI/2 }},
  { &thumbs_up,     0.1, 0,   { 0, 0,    0 }, { M_PI, 0, -M_PI/2 }},
  { &thumbs_up,     0.1, 1,   { 0, -0.2, 0 }, { M_PI, 0, -M_PI/2 }},
  { 0, }};


/**************************************************************************/

static const hand gun1 = {
  {{ 0.00, 0.00, 0.00, 0.00 },
   { 0.00, 0.00, 0.00, 0.00 },
   { 0.80, 1.60, 1.60, 0.00 },
   { 0.80, 1.60, 1.60, 0.00 },
   { 0.80, 1.60, 1.60, 0.00 }},
  { 0.00, 0.00, 0.00, 0.00, 0.00 },
  { 1.50, 1.58, 0.00 },
  { 0.00, 0.00, 0.00 },
  True
};

static const hand gun2 = {
  {{ 0.00, 0.84, 0.84, 0.00 },
   { 0.00, 0.00, 0.00, 0.00 },
   { 0.80, 1.60, 1.60, 0.00 },
   { 0.80, 1.60, 1.60, 0.00 },
   { 0.80, 1.60, 1.60, 0.00 }},
  { 0.00, 0.00, 0.00, 0.00, 0.00 },
  { 1.50, 1.58, 0.00 },
  { 0.00, 0.00, 0.00 },
  True
};

static const hand_anim gun_anim[] = {
  { &gun1, NNN, 1, { 0, 0, 0 }, { 0, 0, 0 }},
  { &gun2, 0.1, 0, { 0, 0, 0 }, { 0, 0, -M_PI/5 }},
  { &gun1, 0.3, 0, { 0, 0, 0 }, { 0, 0, 0 }},
  { &gun2, 0.1, 0, { 0, 0, 0 }, { 0, 0, -M_PI/5 }},
  { &gun1, 0.3, 0, { 0, 0, 0 }, { 0, 0, 0 }},
  { &gun2, 0.1, 0, { 0, 0, 0 }, { 0, 0, -M_PI/5 }},
  { &gun1, 0.3, 1, { 0, 0, 0 }, { 0, 0, 0 }},
  { 0, }};


/**************************************************************************/

#define RPS_SHAKE \
  { &closed_hand, 0.2, 0, { -0.4, 0.3, 0.3 }, { 0,    0, -M_PI/2 }}, \
  { &closed_hand, 0.2, 0, { -0.4, 0,   0   }, { M_PI, 0, -M_PI/2 }}, \
  { &closed_hand, 0.2, 0, { -0.4, 0.3, 0.3 }, { 0,    0, -M_PI/2 }}, \
  { &closed_hand, 0.2, 0, { -0.4, 0,   0   }, { M_PI, 0, -M_PI/2 }}, \
  { &closed_hand, 0.2, 0, { -0.4, 0.3, 0.3 }, { 0,    0, -M_PI/2 }}, \
  { &closed_hand, 0.2, 0, { -0.4, 0,   0   }, { M_PI, 0, -M_PI/2 }}
#define RPS_0 \
  { &open_palm,   0.2, 1, { -0.4, 0, 0 }, { 0, 0, 0 }}
#define RPS_R RPS_SHAKE, \
  { &closed_hand, 0.2, 1, { -0.4, 0, 0 }, { M_PI,   0, -M_PI/2 }}
#define RPS_P RPS_SHAKE, \
  { &open_palm,   0.2, 1, { -0.4, 0, 0 }, { M_PI/2, 0, -M_PI/2 }}
#define RPS_S RPS_SHAKE, \
  { &hand_v,      0.2, 1, { -0.4, 0, 0 }, { 0,      -M_PI/2, 0 }}

static const hand_anim rps1_anim[] = {
  { &closed_hand, NNN, 1, { -0.4, 0, 0 }, { M_PI, 0, -M_PI/2 }},
  RPS_S, RPS_P, RPS_R, RPS_0, 
  RPS_S, RPS_P, RPS_R, RPS_0, 
  RPS_P, RPS_R, RPS_S, RPS_0, 
  RPS_R, RPS_S, RPS_P, RPS_0, 
  { 0, }};

static const hand_anim rps2_anim[] = {
  { &closed_hand, NNN, 1, { -0.4, 0, 0 }, { M_PI, 0, -M_PI/2 }},
  RPS_R, RPS_P, RPS_P, RPS_0, 
  RPS_P, RPS_R, RPS_R, RPS_0, 
  RPS_S, RPS_P, RPS_R, RPS_0, 
  RPS_S, RPS_P, RPS_S, RPS_0, 
  { 0, }};


/**************************************************************************/

static const hand walk1 = {
  {{ 0.00, 0.00, 0.00, 0.00 },
   { -0.20, -0.20, -0.20, 0.00 },
   { 0.30, 0.24, 0.24, 0.00 },
   { 0.00, 1.60, 1.60, 0.00 },
   { 0.00, 1.50, 1.50, 0.00 }},
  { -1.16, -0.12, 0.10, 0.00, 0.00 },
  { 0.00, -0.04, 0.02 },
  { 0.00, 0.00, 0.60 },
  True
};

static const hand walk2 = {
  {{ 0.00, 0.00, 0.00, 0.00 },
   { 0.30, 0.24, 0.24, 0.00 },
   { -0.20, -0.20, -0.20, 0.00 },
   { 0.00, 1.60, 1.60, 0.00 },
   { 0.00, 1.50, 1.50, 0.00 }},
  { -1.16, -0.12, 0.10, 0.00, 0.00 },
  { 0.00, 0.00, 0.00 },
  { 0.00, 0.00, 0.60 },
  True
};

static const hand_anim walk_anim[] = {
  { &walk1, NNN, 0, { 0,  0.0, -0.03 }, { 0, M_PI, M_PI }},
  { &walk2, 0.3, 0, { 0, -0.1, -0.03 }, { 0, M_PI, M_PI }},
  { &walk1, 0.3, 0, { 0, -0.2, -0.03 }, { 0, M_PI, M_PI }},
  { &walk2, 0.3, 0, { 0, -0.3, -0.03 }, { 0, M_PI, M_PI }},
  { &walk1, 0.3, 0, { 0, -0.4, -0.03 }, { 0, M_PI, M_PI }},
  { &walk2, 0.3, 0, { 0, -0.5, -0.03 }, { 0, M_PI, M_PI }},
  { &walk1, 0.3, 0, { 0, -0.6, -0.03 }, { 0, M_PI, M_PI }},
  { &walk2, 0.3, 0, { 0, -0.7, -0.03 }, { 0, M_PI, M_PI }},
  { &walk1, 0.3, 0, { 0, -0.8, -0.03 }, { 0, M_PI, M_PI }},
  { &walk2, 0.3, 0, { 0, -0.9, -0.03 }, { 0, M_PI, M_PI }},
  { &walk1, 0.3, 0, { 0, -1.0, -0.03 }, { 0, M_PI, M_PI }},
  { &walk2, 0.3, 0, { 0, -1.1, -0.03 }, { 0, M_PI, M_PI }},
  { &walk1, 0.3, 0, { 0, -1.2, -0.03 }, { 0, M_PI, M_PI }},
  { &walk2, 0.3, 0, { 0, -1.3, -0.03 }, { 0, M_PI, M_PI }},
  { &walk1, 0.3, 0, { 0, -1.4, -0.03 }, { 0, M_PI, M_PI }},
  { &walk2, 0.3, 0, { 0, -1.5, -0.03 }, { 0, M_PI, M_PI }},
  { &walk1, 0.3, 0, { 0, -1.6, -0.03 }, { 0, M_PI, M_PI }},
  { &walk2, 0.3, 0, { 0, -1.7, -0.03 }, { 0, M_PI, M_PI }},
  { &walk1, 0.3, 0, { 0, -1.8, -0.03 }, { 0, M_PI, M_PI }},
  { &walk2, 0.3, 0, { 0, -1.9, -0.03 }, { 0, M_PI, M_PI }},
  { &walk1, 0.3, 0, { 0, -2.0, -0.03 }, { 0, M_PI, M_PI }},
  { 0, }};

/**************************************************************************/

static const hand_anim goatse_anim[] = {
  { &open_palm, NNN, 1, { 0.31, 2,   0.5 }, { M_PI/2, -M_PI/2, 0 }},
  { &open_palm, 2,   0, { 0.31, 0,   0.5 }, { M_PI/2, -M_PI/2, 0 }},
  { &thumbs_up, 2,   1, { 0.31, 0,   0.5 }, { 0,      -M_PI,   0 }},
  { &thumbs_up, 2,   1, { -0.4, 0,   0.5 }, { 0,      -M_PI,   0 }},
  { &open_palm, 2,   1, { -0.4, 0.3, 0.5 }, { M_PI/2, -M_PI/2, 0 }},
  { 0, }};


/**************************************************************************/

static const hand_anim_pair all_hand_anims[] = {
  {{ middle_anim,    middle_anim },    0.2 },
  {{ tap_anim,       tap_anim },       0.7 },
  {{ tap_anim,       tap_anim },       0.7 },
  {{ tap_anim,       tap_anim },       0.7 },
  {{ spock_anim,     hidden_anim},     0 },
  {{ spock_anim,     hidden_anim},     0 },
  {{ horns_anim,     horns_anim},      0 },
  {{ horns_anim,     horns_anim},      0 },
  {{ bs1_anim,       bs2_anim },       0 },
  {{ upyours_anim,   hidden_anim },    0 },
  {{ peace_anim,     hidden_anim },    0 },
  {{ scissor1_anim,  scissor2_anim },  0 },
  {{ wave_anim,      hidden_anim },    0 },
  {{ wave2_anim,     hidden_anim },    0 },
  {{ jazzhands_anim, jazzhands_anim }, 0 },
  {{ ok_anim,        ok_anim },        0 },
  {{ inout1_anim,    inout2_anim },    0 },
  {{ clap_anim,      clap_anim },      0 },
  {{ slow_clap_anim, slow_clap_anim }, 0 },
  {{ pray_anim,      pray_anim },      0 },
  {{ eyeonu_anim,    hidden_anim },    0 },
  {{ trick1_anim,    trick2_anim },    0 },
  {{ thumbsup_anim,  thumbsup_anim },  0 },
  {{ thumbsdn_anim,  thumbsdn_anim },  0 },
  {{ thisguy_anim,   thisguy_anim },   0 },
  {{ outtahere_anim, hidden_anim },    0 },
  {{ stahhhp_anim,   stahhhp_anim },   0 },
  {{ gun_anim,       gun_anim },       0.25 },
  {{ walk_anim,      walk_anim },      0.3 },
  {{ goatse_anim,    goatse_anim },    0 },
  {{ 0, }}};

#endif /* __HANDSY_ANIM__ */
