/* The GIMP -- an image manipulation program
 * Copyright (C) 1995 Spencer Kimball and Peter Mattis
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "config.h"

#include <gtk/gtk.h>

#include "libgimpcolor/gimpcolor.h"
#include "libgimpmath/gimpmath.h"
#include "libgimpwidgets/gimpwidgets.h"

#include "tools-types.h"

#include "base/colorize.h"

#include "core/gimpdrawable.h"
#include "core/gimpimage.h"
#include "core/gimpimagemap.h"

#include "display/gimpdisplay.h"

#include "gimpcolorizetool.h"

#include "gimp-intl.h"


#define SLIDER_WIDTH  200
#define DA_WIDTH       40
#define DA_HEIGHT      20

#define HUE_SLIDER         (1 << 0)
#define LIGHTNESS_SLIDER   (1 << 1)
#define SATURATION_SLIDER  (1 << 2)
#define DRAW               (1 << 3)
#define SLIDERS            (HUE_SLIDER | LIGHTNESS_SLIDER | SATURATION_SLIDER)
#define ALL                (SLIDERS | DRAW)


/*  local function prototypes  */

static void   gimp_colorize_tool_class_init  (GimpColorizeToolClass *klass);
static void   gimp_colorize_tool_init        (GimpColorizeTool      *tool);

static void   gimp_colorize_tool_finalize    (GObject          *object);

static void   gimp_colorize_tool_initialize  (GimpTool         *tool,
                                              GimpDisplay      *gdisp);

static void   gimp_colorize_tool_map         (GimpImageMapTool *image_map_tool);
static void   gimp_colorize_tool_dialog      (GimpImageMapTool *image_map_tool);
static void   gimp_colorize_tool_reset       (GimpImageMapTool *image_map_tool);

static void   colorize_update                (GimpColorizeTool *col_tool,
                                              gint              update);
static void   colorize_hue_adj_update        (GtkAdjustment    *adj,
                                              GimpColorizeTool *col_tool);
static void   colorize_lightness_adj_update  (GtkAdjustment    *adj,
                                              GimpColorizeTool *col_tool);
static void   colorize_saturation_adj_update (GtkAdjustment    *adj,
                                              GimpColorizeTool *col_tool);


/*  private variables  */

static GimpImageMapToolClass *parent_class = NULL;


/*  public functions  */

void
gimp_colorize_tool_register (GimpToolRegisterCallback  callback,
                             gpointer                  data)
{
  (* callback) (GIMP_TYPE_COLORIZE_TOOL,
                G_TYPE_NONE, NULL,
                0,
                "gimp-colorize-tool",
                _("Colorize"),
                _("Colorize the image"),
                N_("/Tools/Color Tools/Colori_ze..."), NULL,
                NULL, "tools/colorize.html",
                GIMP_STOCK_TOOL_COLORIZE,
                data);
}

GType
gimp_colorize_tool_get_type (void)
{
  static GType tool_type = 0;

  if (! tool_type)
    {
      static const GTypeInfo tool_info =
      {
        sizeof (GimpColorizeToolClass),
	(GBaseInitFunc) NULL,
	(GBaseFinalizeFunc) NULL,
	(GClassInitFunc) gimp_colorize_tool_class_init,
	NULL,           /* class_finalize */
	NULL,           /* class_data     */
	sizeof (GimpColorizeTool),
	0,              /* n_preallocs    */
	(GInstanceInitFunc) gimp_colorize_tool_init,
      };

      tool_type = g_type_register_static (GIMP_TYPE_IMAGE_MAP_TOOL,
					  "GimpColorizeTool", 
                                          &tool_info, 0);
    }

  return tool_type;
}


/*  private functions  */

static void
gimp_colorize_tool_class_init (GimpColorizeToolClass *klass)
{
  GObjectClass          *object_class;
  GimpToolClass         *tool_class;
  GimpImageMapToolClass *image_map_tool_class;

  object_class         = G_OBJECT_CLASS (klass);
  tool_class           = GIMP_TOOL_CLASS (klass);
  image_map_tool_class = GIMP_IMAGE_MAP_TOOL_CLASS (klass);

  parent_class = g_type_class_peek_parent (klass);

  object_class->finalize       = gimp_colorize_tool_finalize;

  tool_class->initialize       = gimp_colorize_tool_initialize;

  image_map_tool_class->map    = gimp_colorize_tool_map;
  image_map_tool_class->dialog = gimp_colorize_tool_dialog;
  image_map_tool_class->reset  = gimp_colorize_tool_reset;
}

static void
gimp_colorize_tool_init (GimpColorizeTool *col_tool)
{
  GimpImageMapTool *image_map_tool;

  image_map_tool = GIMP_IMAGE_MAP_TOOL (col_tool);

  image_map_tool->shell_identifier = "gimp-colorize-tool-dialog";
  image_map_tool->shell_desc       = _("Colorize the Image");

  col_tool->colorize = g_new0 (Colorize, 1);

  colorize_init (col_tool->colorize);
}

static void
gimp_colorize_tool_finalize (GObject *object)
{
  GimpColorizeTool *col_tool;

  col_tool = GIMP_COLORIZE_TOOL (object);

  if (col_tool->colorize)
    {
      g_free (col_tool->colorize);
      col_tool->colorize = NULL;
    }

  G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void
gimp_colorize_tool_initialize (GimpTool    *tool,
                               GimpDisplay *gdisp)
{
  GimpColorizeTool *col_tool;

  col_tool = GIMP_COLORIZE_TOOL (tool);

  if (! gimp_drawable_is_rgb (gimp_image_active_drawable (gdisp->gimage)))
    {
      g_message (_("Colorize operates only on RGB color drawables."));
      return;
    }

  colorize_init (col_tool->colorize);

  GIMP_TOOL_CLASS (parent_class)->initialize (tool, gdisp);

  colorize_update (col_tool, ALL);

  gimp_image_map_tool_preview (GIMP_IMAGE_MAP_TOOL (tool));
}

static void
gimp_colorize_tool_map (GimpImageMapTool *image_map_tool)
{
  GimpColorizeTool *col_tool;

  col_tool = GIMP_COLORIZE_TOOL (image_map_tool);

  gimp_image_map_apply (image_map_tool->image_map,
                        (GimpImageMapApplyFunc) colorize,
                        col_tool->colorize);
}


/***************************/
/*  Hue-Saturation dialog  */
/***************************/

static void
gimp_colorize_tool_dialog (GimpImageMapTool *image_map_tool)
{
  GimpColorizeTool *col_tool;
  GtkWidget        *table;
  GtkWidget        *slider;
  GtkWidget        *frame;
  GtkWidget        *vbox;
  GtkObject        *data;

  col_tool = GIMP_COLORIZE_TOOL (image_map_tool);

  frame = gtk_frame_new (_("Select Color"));
  gtk_box_pack_start (GTK_BOX (image_map_tool->main_vbox), frame,
                      FALSE, FALSE, 0);
  gtk_widget_show (frame);

  /*  The table containing sliders  */
  vbox = gtk_vbox_new (FALSE, 4);
  gtk_container_set_border_width (GTK_CONTAINER (vbox), 2);
  gtk_container_add (GTK_CONTAINER (frame), vbox);
  gtk_widget_show (vbox);

  table = gtk_table_new (3, 3, FALSE);
  gtk_table_set_col_spacings (GTK_TABLE (table), 4);
  gtk_table_set_row_spacings (GTK_TABLE (table), 2);
  gtk_box_pack_start (GTK_BOX (vbox), table, FALSE, FALSE, 0);
  gtk_widget_show (table);

  /*  Create the hue scale widget  */
  data = gimp_scale_entry_new (GTK_TABLE (table), 0, 0,
                               _("_Hue:"), SLIDER_WIDTH, -1,
                               0.0, 0.0, 360.0, 1.0, 15.0, 0,
                               TRUE, 0.0, 0.0,
                               NULL, NULL);
  col_tool->hue_data = GTK_ADJUSTMENT (data);
  slider = GIMP_SCALE_ENTRY_SCALE (data);
  gtk_range_set_update_policy (GTK_RANGE (slider), GTK_UPDATE_DELAYED);

  g_signal_connect (data, "value_changed",
                    G_CALLBACK (colorize_hue_adj_update),
                    col_tool);

  /*  Create the lightness scale widget  */
  data = gimp_scale_entry_new (GTK_TABLE (table), 0, 1,
                               _("_Lightness:"), SLIDER_WIDTH, -1,
                               0.0, 0.0, 100.0, 1.0, 10.0, 0,
                               TRUE, 0.0, 0.0,
                               NULL, NULL);
  col_tool->lightness_data = GTK_ADJUSTMENT (data);
  slider = GIMP_SCALE_ENTRY_SCALE (data);
  gtk_range_set_update_policy (GTK_RANGE (slider), GTK_UPDATE_DELAYED);

  g_signal_connect (data, "value_changed",
                    G_CALLBACK (colorize_lightness_adj_update),
                    col_tool);

  /*  Create the saturation scale widget  */
  data = gimp_scale_entry_new (GTK_TABLE (table), 0, 2,
                               _("_Saturation:"), SLIDER_WIDTH, -1,
                               0.0, 0.0, 100.0, 1.0, 10.0, 0,
                               TRUE, 0.0, 0.0,
                               NULL, NULL);
  col_tool->saturation_data = GTK_ADJUSTMENT (data);
  slider = GIMP_SCALE_ENTRY_SCALE (data);
  gtk_range_set_update_policy (GTK_RANGE (slider), GTK_UPDATE_DELAYED);

  g_signal_connect (col_tool->saturation_data, "value_changed",
                    G_CALLBACK (colorize_saturation_adj_update),
                    col_tool);
}

static void
gimp_colorize_tool_reset (GimpImageMapTool *image_map_tool)
{
  GimpColorizeTool *col_tool;

  col_tool = GIMP_COLORIZE_TOOL (image_map_tool);

  colorize_init (col_tool->colorize);

  colorize_update (col_tool, ALL);
}

static void
colorize_update (GimpColorizeTool *col_tool,
                 gint              update)
{
  colorize_calculate (col_tool->colorize);

  if (update & HUE_SLIDER)
    gtk_adjustment_set_value (GTK_ADJUSTMENT (col_tool->hue_data),
                              col_tool->colorize->hue);

  if (update & LIGHTNESS_SLIDER)
    gtk_adjustment_set_value (GTK_ADJUSTMENT (col_tool->lightness_data),
                              col_tool->colorize->lightness);

  if (update & SATURATION_SLIDER)
    gtk_adjustment_set_value (GTK_ADJUSTMENT (col_tool->saturation_data),
                              col_tool->colorize->saturation);
}

static void
colorize_hue_adj_update (GtkAdjustment    *adjustment,
                         GimpColorizeTool *col_tool)
{
  if (col_tool->colorize->hue != adjustment->value)
    {
      col_tool->colorize->hue = adjustment->value;
      colorize_update (col_tool, DRAW);

      gimp_image_map_tool_preview (GIMP_IMAGE_MAP_TOOL (col_tool));
    }
}

static void
colorize_lightness_adj_update (GtkAdjustment    *adjustment,
                               GimpColorizeTool *col_tool)
{
  if (col_tool->colorize->lightness != adjustment->value)
    {
      col_tool->colorize->lightness = adjustment->value;
      colorize_update (col_tool, DRAW);

      gimp_image_map_tool_preview (GIMP_IMAGE_MAP_TOOL (col_tool));
    }
}

static void
colorize_saturation_adj_update (GtkAdjustment    *adjustment,
                                GimpColorizeTool *col_tool)
{
  if (col_tool->colorize->saturation != adjustment->value)
    {
      col_tool->colorize->saturation = adjustment->value;
      colorize_update (col_tool, DRAW);

      gimp_image_map_tool_preview (GIMP_IMAGE_MAP_TOOL (col_tool));
    }
}
