/* The GIMP -- an image manipulation program
 * Copyright (C) 1995 Spencer Kimball and Peter Mattis
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "config.h"

#include <gtk/gtk.h>

#include "display-types.h"

#include "core/gimp.h"
#include "core/gimpcontext.h"
#include "core/gimpimage.h"
#include "core/gimplist.h"

#include "gimpdisplay.h"
#include "gimpdisplay-foreach.h"
#include "gimpdisplayshell.h"
#include "gimpdisplayshell-cursor.h"


gboolean
gimp_displays_dirty (Gimp *gimp)
{
  GList *list;

  g_return_val_if_fail (GIMP_IS_GIMP (gimp), FALSE);

  for (list = GIMP_LIST (gimp->displays)->list;
       list;
       list = g_list_next (list))
    {
      if (((GimpDisplay *) list->data)->gimage->dirty != 0)
	return TRUE;
    }

  return FALSE;
}

void
gimp_displays_delete (Gimp *gimp)
{
  GimpDisplay *gdisp;

  g_return_if_fail (GIMP_IS_GIMP (gimp));

  /*  this removes the GimpDisplay from the list, so do a while loop
   *  "around" the first element to get them all
   */
  while (GIMP_LIST (gimp->displays)->list)
    {
      gdisp = (GimpDisplay *) GIMP_LIST (gimp->displays)->list->data;

      gimp_display_delete (gdisp);
    }
}

/* Force all gdisplays to finish their idlerender projection */
void
gimp_displays_finish_draw (Gimp *gimp)
{
  GList       *list;
  GimpDisplay *gdisp;

  g_return_if_fail (GIMP_IS_GIMP (gimp));

  for (list = GIMP_LIST (gimp->displays)->list;
       list;
       list = g_list_next (list))
    {
      gdisp = (GimpDisplay *) list->data;

      gimp_display_finish_draw (gdisp);
    }
}

void
gimp_displays_reconnect (Gimp      *gimp,
                         GimpImage *old,
                         GimpImage *new)
{
  GList       *list;
  GimpDisplay *gdisp;
  GList       *contexts = NULL;

  g_return_if_fail (GIMP_IS_GIMP (gimp));
  g_return_if_fail (GIMP_IS_IMAGE (old));
  g_return_if_fail (GIMP_IS_IMAGE (new));

  /*  remember which contexts refer to old_gimage  */
  for (list = gimp->context_list; list; list = g_list_next (list))
    {
      GimpContext *context = list->data;

      if (gimp_context_get_image (context) == old)
        contexts = g_list_prepend (contexts, list->data);
    }

  for (list = GIMP_LIST (gimp->displays)->list;
       list;
       list = g_list_next (list))
    {
      gdisp = list->data;

      if (gdisp->gimage == old)
	gimp_display_reconnect (gdisp, new);
    }

  /*  set the new_gimage on the remembered contexts (in reverse
   *  order, since older contexts are usually the parents of
   *  newer ones)
   */
  g_list_foreach (contexts, (GFunc) gimp_context_set_image, new);
  g_list_free (contexts);
}

void
gimp_displays_set_busy (Gimp *gimp)
{
  GList            *list;
  GimpDisplayShell *shell;

  g_return_if_fail (GIMP_IS_GIMP (gimp));

  for (list = GIMP_LIST (gimp->displays)->list;
       list;
       list = g_list_next (list))
    {
      shell = GIMP_DISPLAY_SHELL (GIMP_DISPLAY (list->data)->shell);

      gimp_display_shell_set_override_cursor (shell, GDK_WATCH);
    }
}

void
gimp_displays_unset_busy (Gimp *gimp)
{
  GList            *list;
  GimpDisplayShell *shell;

  g_return_if_fail (GIMP_IS_GIMP (gimp));

  for (list = GIMP_LIST (gimp->displays)->list;
       list;
       list = g_list_next (list))
    {
      shell = GIMP_DISPLAY_SHELL (GIMP_DISPLAY (list->data)->shell);

      gimp_display_shell_unset_override_cursor (shell);
    }
}
