/* GTK - The GIMP Toolkit
 * Copyright (C) 1995-1997 Peter Mattis, Spencer Kimball and Josh MacDonald
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <gtk/gtklabel.h>
#include "gtkhtreeitem.h"


static void gtk_htree_item_class_init     (GtkHTreeItemClass *klass);
static void gtk_htree_item_init           (GtkHTreeItem      *htree_item);
static void gtk_htree_item_size_request   (GtkWidget        *widget,
					   GtkRequisition   *requisition);
static void gtk_htree_item_size_allocate  (GtkWidget        *widget,
					   GtkAllocation    *allocation);


guint
gtk_htree_item_get_type ()
{
  static guint htree_item_type = 0;

  if (!htree_item_type)
    {
      GtkTypeInfo htree_item_info =
      {
	"GtkHTreeItem",
	sizeof (GtkHTreeItem),
	sizeof (GtkHTreeItemClass),
	(GtkClassInitFunc) gtk_htree_item_class_init,
	(GtkObjectInitFunc) gtk_htree_item_init,
	(GtkArgFunc) NULL,
      };

      htree_item_type = gtk_type_unique (gtk_tree_item_get_type (),
					 &htree_item_info);
    }

  return htree_item_type;
}

static void
gtk_htree_item_class_init (GtkHTreeItemClass *klass)
{
  GtkWidgetClass *widget_class;

  widget_class = (GtkWidgetClass*) klass;

  widget_class->size_request = gtk_htree_item_size_request;
  widget_class->size_allocate = gtk_htree_item_size_allocate;
}

static void
gtk_htree_item_init (GtkHTreeItem *htree_item)
{
}

GtkWidget*
gtk_htree_item_new ()
{
  return GTK_WIDGET (gtk_type_new (gtk_htree_item_get_type ()));
}

GtkWidget*
gtk_htree_item_new_with_label (const gchar *label)
{
  GtkWidget *htree_item;
  GtkWidget *label_widget;

  htree_item = gtk_htree_item_new ();
  label_widget = gtk_label_new (label);
  gtk_misc_set_alignment (GTK_MISC (label_widget), 0.0, 0.5);

  gtk_container_add (GTK_CONTAINER (htree_item), label_widget);
  gtk_widget_show (label_widget);

  return htree_item;
}


static void
gtk_htree_item_size_request (GtkWidget      *widget,
			     GtkRequisition *requisition)
{
  GtkBin *bin;
  GtkTreeItem *tree_item;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_HTREE_ITEM (widget));
  g_return_if_fail (requisition != NULL);

  bin = GTK_BIN (widget);
  tree_item = GTK_TREE_ITEM (widget);

  requisition->width = 0;
  requisition->height = 0;

  if (bin->child && GTK_WIDGET_VISIBLE (bin->child))
    {
      gtk_widget_size_request (bin->child, &bin->child->requisition);

      requisition->width += bin->child->requisition.width;
      requisition->height = MAX (requisition->height,
				 bin->child->requisition.height);
    }

  if (tree_item->fork && GTK_WIDGET_VISIBLE (tree_item->fork))
    {
      requisition->width += tree_item->spacing;
      requisition->height += 0;

      gtk_widget_size_request (tree_item->fork,
			       &tree_item->fork->requisition);

      requisition->width += tree_item->fork->requisition.width;
      requisition->height = MAX (requisition->height,
				 tree_item->fork->requisition.height);
    }

  if (tree_item->subtree && GTK_WIDGET_VISIBLE (tree_item->subtree))
    {
      requisition->width += tree_item->spacing;
      requisition->height += 0;

      gtk_widget_size_request (tree_item->subtree,
			       &tree_item->subtree->requisition);

      requisition->width += tree_item->subtree->requisition.width;
      requisition->height = MAX (requisition->height,
				 tree_item->subtree->requisition.height);
    }

  requisition->width += GTK_CONTAINER (tree_item)->border_width * 2;
  requisition->height += GTK_CONTAINER (tree_item)->border_width * 2;
}

static void
gtk_htree_item_size_allocate (GtkWidget     *widget,
			      GtkAllocation *allocation)
{
  GtkBin *bin;
  GtkTreeItem *tree_item;
  GtkAllocation child_allocation;
  gint x;
  gint y;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_HTREE_ITEM (widget));
  g_return_if_fail (allocation != NULL);

  widget->allocation = *allocation;

  if (!GTK_WIDGET_NO_WINDOW (widget) && GTK_WIDGET_REALIZED (widget))
    gdk_window_move_resize (widget->window,
                            allocation->x, allocation->y,
                            allocation->width, allocation->height);

  bin = GTK_BIN (widget);
  tree_item = GTK_TREE_ITEM (widget);

  if (!GTK_WIDGET_NO_WINDOW (widget))
    {
      x = 0;
      y = 0;
    }
  else
    {
      x = allocation->x;
      y = allocation->y;
    }


  x += GTK_CONTAINER (tree_item)->border_width;
  y += GTK_CONTAINER (tree_item)->border_width;

  if (bin->child)
    {
      gint16 child_area_height
	  = allocation->height - (GTK_CONTAINER (tree_item)->border_width * 2);
      gint16 child_height
	  = bin->child->requisition.height;

      child_allocation.x = x;
      child_allocation.y
	  = y + (child_area_height - child_height) * tree_item->child_align;
      child_allocation.width = bin->child->requisition.width;
      child_allocation.height = bin->child->requisition.height;

      gtk_widget_size_allocate (bin->child, &child_allocation);

      x += child_allocation.width;
      y += 0;

      x += tree_item->spacing;
      y += 0;
    }

  if (tree_item->fork)
    {
      child_allocation.x = x;
      child_allocation.y = y;
      child_allocation.width = tree_item->fork->requisition.width;
      child_allocation.height
	  = allocation->height - (GTK_CONTAINER (tree_item)->border_width * 2);

      gtk_widget_size_allocate (tree_item->fork, &child_allocation);

      x += child_allocation.width;
      y += 0;

      x += tree_item->spacing;
      y += 0;
    }

  if (tree_item->subtree)
    {
      child_allocation.x = x;
      child_allocation.y = y;
      child_allocation.width = tree_item->subtree->requisition.width;
      child_allocation.height = tree_item->subtree->requisition.height;

      gtk_widget_size_allocate (tree_item->subtree, &child_allocation);
    }
}
