/* GDK - The GIMP Drawing Kit
 * Copyright (C) 1995-1997 Peter Mattis, Spencer Kimball and Josh MacDonald
 *
 * BeOS Port
 * Copyright (C) 1999 EventLoop, Inc.
 *   Shawn T. Amundson <amundson@gtk.org>
 *   James Mitchell <mitchell@eventloop.com>
 *         
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/*
 * Modified by the GTK+ Team and others 1997-1999.  See the AUTHORS
 * file for a list of people on the GTK+ Team.  See the ChangeLog
 * files for a list of changes.  These files are distributed with
 * GTK+ at ftp://ftp.gtk.org/pub/gtk/. 
 */

#include <Application.h>
#include <Screen.h>
#include <Rect.h>
#include <Window.h>
#include <View.h>

#include <stdlib.h>
#include <stdio.h>
#include <netinet/in.h>

#include "config.h"
#include "gdk/gdk.h"
#include "gdkx.h"
#include "gdkinput.h"
#include "gdkprivate.h"

#include "gdk_bview.h"
#include "gdk_bwindow.h"


/* Forward declarations */
static gboolean gdk_window_gravity_works (void);
static void     gdk_window_set_static_win_gravity (GdkWindow *window, 
						   gboolean   on);
static gboolean gdk_window_have_shape_ext (void);

extern gboolean do_motion_hints;

void
gdk_window_init (void)
{
  BScreen screen(B_MAIN_SCREEN_ID);
  BRect rect;
  unsigned int width;
  unsigned int height;

  rect = screen.Frame();

  if (!screen.IsValid())
    g_error ("BScreen invalid!");

  width = (guint) rect.Width();
  height = (guint) rect.Height();

  gdk_root_parent.xwindow = gdk_root_window;
  gdk_root_parent.window_type = GDK_WINDOW_ROOT;
  gdk_root_parent.window.user_data = NULL;
  gdk_root_parent.width = width;
  gdk_root_parent.height = height;
  gdk_root_parent.children = NULL;
  gdk_root_parent.colormap = NULL;
  gdk_root_parent.ref_count = 1;
  
  gdk_xid_table_insert ((void**)(&gdk_root_window), &gdk_root_parent);
}

GdkWindow*
gdk_window_new (GdkWindow     *parent,
		GdkWindowAttr *attributes,
		gint           attributes_mask)
{
  GdkWindow *window;
  GdkWindowPrivate *priv;
  GdkWindowPrivate *parent_private;
  GdkVisual *visual;
  BView *view;
  BWindow *bwin;
  gchar *title;
  gint x, y, i;
  BView *parent_view;
  
  g_return_val_if_fail (attributes != NULL, NULL);
  
  if (!parent)
    parent = (GdkWindow*) &gdk_root_parent;
  
  parent_private = (GdkWindowPrivate*) parent;
  if (parent_private->destroyed)
    return NULL;

  parent_view = (BView*)(parent_private->xwindow);

  priv = g_new (GdkWindowPrivate, 1);
  window = (GdkWindow*) priv;
  
  priv->parent = parent;
  
  priv->destroyed = FALSE;
  priv->mapped = FALSE;
  priv->guffaw_gravity = FALSE;
  priv->resize_count = 0;
  priv->ref_count = 1;
  
  if (attributes_mask & GDK_WA_X)
    x = attributes->x;
  else
    x = 30;
  
  if (attributes_mask & GDK_WA_Y)
    y = attributes->y;
  else
    y = 55;
  
  priv->x = x;
  priv->y = y;
  priv->width = (attributes->width > 1) ? (attributes->width) : (1);
  priv->height = (attributes->height > 1) ? (attributes->height) : (1);
  priv->window_type = attributes->window_type;
  priv->extension_events = FALSE;
  
  priv->filters = NULL;
  priv->children = NULL;
  
  window->user_data = NULL;
  
  if (attributes_mask & GDK_WA_VISUAL)
    visual = attributes->visual;
  else
    visual = gdk_visual_get_system ();

  /* FOOBAR I moved this up here, because there was a case
     where the colormap was not set before gdk_colormap_ref 
     is called. This may be incorrect. JSM */

  priv->colormap = gdk_colormap_get_system ();
  
  if (attributes->wclass == GDK_INPUT_OUTPUT)
    {
      if (attributes_mask & GDK_WA_COLORMAP)
	priv->colormap = attributes->colormap;
      else
	priv->colormap = gdk_colormap_get_system ();
      
      switch (priv->window_type)
	{
	case GDK_WINDOW_TOPLEVEL:
	case GDK_WINDOW_CHILD:
	case GDK_WINDOW_DIALOG:
	case GDK_WINDOW_TEMP:
	  break;

	case GDK_WINDOW_ROOT:
	  g_error ("cannot make windows of type GDK_WINDOW_ROOT");
	  break;
	case GDK_WINDOW_PIXMAP:
	  g_error ("cannot make windows of type GDK_WINDOW_PIXMAP (use gdk_pixmap_new)");
	  break;
	}
    }
  else
    {
      priv->colormap = NULL;
    }

  if (attributes_mask & GDK_WA_TITLE)
    title = attributes->title;
  else
    title = g_get_prgname ();

  priv->event_mask = attributes->event_mask;

  switch (priv->window_type)
    {
    case GDK_WINDOW_TOPLEVEL:
    case GDK_WINDOW_DIALOG:
     
      bwin = new GdkBWindowPrivate(BRect(x,y,x+priv->width-1,y+priv->height-1),
                                   title, B_TITLED_WINDOW, 0);
      break;
    case GDK_WINDOW_TEMP:
      bwin = new GdkBWindowPrivate(BRect(x,y,x+priv->width-1,y+priv->height-1),
                                   title, B_BORDERED_WINDOW, 0);
      break;
    case GDK_WINDOW_CHILD:
    default:
      bwin = NULL;
      break;
    }

  if (bwin)
    view = new GdkBViewPrivate(BRect(0,0,priv->width-1,priv->height-1),
                               title,
                               B_FOLLOW_LEFT | B_FOLLOW_TOP,
                               B_WILL_DRAW | B_FRAME_EVENTS | 
                               B_FULL_UPDATE_ON_RESIZE);
  else
    view = new GdkBViewPrivate(BRect(x,y,x+priv->width-1,y+priv->height-1),
                               title,
                               B_FOLLOW_LEFT | B_FOLLOW_TOP,
                               B_WILL_DRAW | B_FRAME_EVENTS);
     
  if (attributes->wclass != GDK_INPUT_OUTPUT)
    {
      view->SetFlags(B_FRAME_EVENTS);

      if (parent_view)
        {
          if (parent_view->Window()->Lock())
            {
              (parent_view)->SetFlags(parent_view->Flags() | B_DRAW_ON_CHILDREN);
              parent_view->Window()->Unlock();
            }
        }
    }

  if (bwin)
    bwin->AddChild(view);
  else
    parent_view->AddChild(view);

  if (view->Window()->Lock())
    {
      view->SetViewColor(255,0,0);
      view->SetHighColor(0,255,0);
      view->SetLowColor(0,0,255);
      view->SetFont(new BFont(be_plain_font), B_FONT_ALL); 

      view->Window()->Unlock();
    }

  priv->bewindow = (GdkBWindow*) bwin;
  priv->xwindow = (GdkBView*) view;

  gdk_window_ref (window);

  gdk_xid_table_insert ((void**)(&priv->xwindow), window);
  if (priv->bewindow)
    gdk_xid_table_insert ((void**)(&priv->bewindow), window);
  
  if (priv->colormap)
    gdk_colormap_ref (priv->colormap);

  gdk_window_set_cursor (window, ((attributes_mask & GDK_WA_CURSOR) ?
				  (attributes->cursor) :
				  NULL));
  
  if (parent_private)
    parent_private->children = g_list_prepend (parent_private->children, window);
  
  return window;
}

GdkWindow *
gdk_window_foreign_new (guint32 anid)
{
  g_warning("gdk_window_foreign_new: not implemented!");
  
  return NULL;
}

/* Call this function when you want a window and all its children to
 * disappear.  When xdestroy is true, a request to destroy the XWindow
 * is sent out.  When it is false, it is assumed that the XWindow has
 * been or will be destroyed by destroying some ancestor of this
 * window.
 */
static void
gdk_window_internal_destroy (GdkWindow *window,
			     gboolean   xdestroy,
			     gboolean   our_destroy)
{
  GdkWindowPrivate *priv;

  g_return_if_fail(window != NULL);

  priv = (GdkWindowPrivate*) window;

  if (!priv->destroyed)
    priv->destroyed = TRUE;
  
  g_warning("gdk_window_internal_destroy: not implemented!");
}

/* Like internal_destroy, but also destroys the reference created by
   gdk_window_new. */

void
gdk_window_destroy (GdkWindow *window)
{
  gdk_window_internal_destroy (window, TRUE, TRUE);
  gdk_window_unref (window);
}

/* This function is called when the XWindow is really gone.  */

void
gdk_window_destroy_notify (GdkWindow *window)
{
  GdkWindowPrivate *priv;
  
  g_return_if_fail (window != NULL);
  
  priv = (GdkWindowPrivate*) window;
  
  if (!priv->destroyed)
    {
      if (priv->window_type == GDK_WINDOW_FOREIGN)
	gdk_window_internal_destroy (window, FALSE, FALSE);
      else
	g_warning ("GdkWindow %#lx unexpectedly destroyed", priv->xwindow);
    }
  
  gdk_xid_table_remove (priv->xwindow);
  if (priv->bewindow)
    gdk_xid_table_remove (priv->bewindow);
  gdk_window_unref (window);
}

GdkWindow*
gdk_window_ref (GdkWindow *window)
{
  GdkWindowPrivate *priv= (GdkWindowPrivate *)window;
  
  g_return_val_if_fail (window != NULL, NULL);
  
  priv->ref_count += 1;
  return window;
}

void
gdk_window_unref (GdkWindow *window)
{
  static int warned = 0;
  GdkWindowPrivate *priv = (GdkWindowPrivate *)window;
  g_return_if_fail (window != NULL);
  
  priv->ref_count -= 1;
  if (priv->ref_count == 0)
    {
      if (!priv->destroyed)
	{
	  if (priv->window_type == GDK_WINDOW_FOREIGN)
            {
	      gdk_xid_table_remove (priv->xwindow);
              if (priv->bewindow)
	        gdk_xid_table_remove (priv->bewindow);
            }
	  else
            {
	      g_warning ("losing last reference to undestroyed window\n");
            }
	}
      g_dataset_destroy (window);
      g_free (window);
    }
}

void
gdk_window_show (GdkWindow *window)
{
  GdkWindowPrivate *priv;
  BView *view;
  
  g_return_if_fail (window != NULL);
  
  priv = (GdkWindowPrivate*) window;
  view = (BView*) priv->xwindow;

  if (!priv->destroyed)
    {
      priv->mapped = TRUE;

      if (priv->bewindow)
        if (((BWindow*)(priv->bewindow))->Lock())
          {
            ((BWindow*)(priv->bewindow))->Show();
            ((BWindow*)(priv->bewindow))->Unlock();
	  }

      if (view->Window()->Lock())
        {
          view->Show();
   	  view->Invalidate();
          view->Window()->Unlock();
        }
    }
}

void
gdk_window_hide (GdkWindow *window)
{
  GdkWindowPrivate *priv;
  BView *view;

  g_return_if_fail (window != NULL);
  
  priv = (GdkWindowPrivate*) window;
  view = (BView*) priv->xwindow;

  g_return_if_fail(priv->parent != NULL);

  if (!priv->destroyed)
    {
      priv->mapped = FALSE;

      if (priv->bewindow)
        if (((BWindow*)(priv->bewindow))->Lock())
          {
            ((BWindow*)(priv->bewindow))->Hide();
            ((BWindow*)(priv->bewindow))->Unlock();
          }              

      if (view->Window()->Lock())
        {
	  view->Invalidate();
          view->Hide();
          view->Window()->Unlock();
        }
    }
}

void
gdk_window_withdraw (GdkWindow *window)
{
  GdkWindowPrivate *priv;
  
  g_return_if_fail (window != NULL);
  
  priv = (GdkWindowPrivate*) window;
  if (!priv->destroyed)
    g_warning("gdk_window_withdrawn: not implemented!\n");
}

void
gdk_window_move (GdkWindow *window,
		 gint       x,
		 gint       y)
{
  GdkWindowPrivate *priv;
  BView *view;
  BWindow *bwin;
  
  g_return_if_fail (window != NULL);

  priv = (GdkWindowPrivate*) window;
  view = (BView*) (priv->xwindow);
  bwin = (BWindow*) (priv->bewindow);

  if (!priv->destroyed)
    {
      if (bwin)
          bwin->MoveTo((float)x, (float)y);
      else
        {
          if (view->Window()->Lock())
            {
              view->MoveTo((float)x, (float)y);
              view->Window()->Unlock();
            }
        }
      
      if (priv->window_type == GDK_WINDOW_CHILD)
	{
	  priv->x = x;
	  priv->y = y;
	}
    }
}

void
gdk_window_resize (GdkWindow *window,
		   gint       width,
		   gint       height)
{
  GdkWindowPrivate *priv;
  BView *view;
  BWindow *bwin;
  
  g_return_if_fail (window != NULL);
  
  if (width < 1)
    width = 1;
  if (height < 1)
    height = 1;
  
  priv = (GdkWindowPrivate*) window;
  view = (BView*) (priv->xwindow);
  bwin = (BWindow*) (priv->bewindow);
  
  if (!priv->destroyed &&
      ((priv->resize_count > 0) ||
       (priv->width != (guint16) width) ||
       (priv->height != (guint16) height)))
    {
      if (bwin)
          bwin->ResizeTo((float)width, (float)height);

      if (view->Window()->Lock())
        {
          view->ResizeTo((float)width, (float)height);
          view->Window()->Unlock();
        }

      priv->resize_count += 1;
      
      if (priv->window_type == GDK_WINDOW_CHILD)
	{
	  priv->width = width;
	  priv->height = height;
	}
    }
}

void
gdk_window_move_resize (GdkWindow *window,
			gint       x,
			gint       y,
			gint       width,
			gint       height)
{
  g_return_if_fail (window != NULL);
  
  gdk_window_move (window, x, y);
  gdk_window_resize (window, width, height);
}

void
gdk_window_reparent (GdkWindow *window,
		     GdkWindow *new_parent,
		     gint       x,
		     gint       y)
{
  GdkWindowPrivate *window_private;
  GdkWindowPrivate *parent_private;
  GdkWindowPrivate *old_parent_private;
  
  g_return_if_fail (window != NULL);

  if (!new_parent)
    new_parent = (GdkWindow*) &gdk_root_parent;
  
  window_private = (GdkWindowPrivate*) window;
  old_parent_private = (GdkWindowPrivate*)window_private->parent;
  parent_private = (GdkWindowPrivate*) new_parent;
  
  if (!window_private->destroyed && !parent_private->destroyed)
    {
      BView *view;
      BWindow *bwin;
      BView *old_view;
      BView *new_view;

      view = (BView*) (window_private->xwindow);
      bwin = (BWindow*) (window_private->xwindow);
      old_view = (BView*) (old_parent_private->xwindow);
      new_view = (BView*) (old_parent_private->xwindow);

      if (old_view && old_view->Window()->Lock()) /* !root window? */
        {
          old_view->RemoveChild(view);
          old_view->Window()->Unlock();
        }

      if (bwin && bwin->Lock())
        {
          bwin->RemoveChild(view);
          bwin->Unlock();
          delete bwin;
        }

      if (new_view && new_view->Window()->Lock()) /* !root window? */
        {
          new_view->AddChild(view);
          new_view->Window()->Unlock();
        }
      else
        {
          bwin = new GdkBWindowPrivate(BRect(x,y,
                                       x+window_private->width-1, 
                                       y+window_private->height-1),
                                       "unknown", B_TITLED_WINDOW, 0);
          if(bwin->Lock())
            {
              bwin->AddChild(view);
              bwin->Unlock();
            }
        }
    }
  
  window_private->parent = new_parent;
  
  if (old_parent_private)
    old_parent_private->children = g_list_remove (old_parent_private->children, window);
  
  parent_private->children = g_list_prepend (parent_private->children, window);
}

void
gdk_window_clear (GdkWindow *window)
{
  GdkWindowPrivate *priv;
  
  g_return_if_fail (window != NULL);
  
  priv = (GdkWindowPrivate*) window;
  
  if (!priv->destroyed)
    {
      BView *view;
      BRect rect;
      
      view = (BView*) priv->xwindow;

      if (view->Window()->Lock())
        {
          view = (BView*) (priv->xwindow);
          rect = view->Frame();
          rect.right -= rect.left;
          rect.bottom -= rect.top; 
          rect.left = 0;
          rect.top = 0;

          view->FillRect(rect, B_SOLID_LOW);

          view->Window()->Unlock();
        }
    }
}

void
gdk_window_clear_area (GdkWindow *window,
		       gint       x,
		       gint       y,
		       gint       width,
		       gint       height)
{
  GdkWindowPrivate *priv;
  
  g_return_if_fail (window != NULL);
  
  priv = (GdkWindowPrivate*) window;
  
  if (!priv->destroyed)
    {
      BView *view;
      BRect rect;

      view = (BView*) (priv->xwindow);

      if (view->Window()->Lock())
        {
          rect = view->Frame();
          view->Window()->Unlock();
        }
      
      if (width <= 0)
        width = (gint) rect.Width() + 1 - x;

      if (height <= 0)
        height = (gint) rect.Height() + 1 -  y;

      if (view->Window()->Lock())
        {
          view->FillRect(BRect(x, y, x+width-1, y+height-1), B_SOLID_LOW);
#if 0
          g_print("gdk_window_clear_area: %#x, %d,%d, %dx%d\n", window,
                  x, y, width, height);
#endif

          view->Window()->Unlock();
        }
    }
}

void
gdk_window_clear_area_e (GdkWindow *window,
		         gint       x,
		         gint       y,
		         gint       width,
		         gint       height)
{
  g_print("gdk_window_clear_area_e: ...\n");
  gdk_window_clear_area(window, x, y, width, height);
}

void
gdk_window_copy_area (GdkWindow    *window,
		      GdkGC        *gc,
		      gint          x,
		      gint          y,
		      GdkWindow    *source_window,
		      gint          source_x,
		      gint          source_y,
		      gint          width,
		      gint          height)
{
  GdkWindowPrivate *priv;
  BView *view;
  
  g_return_if_fail (window != NULL);
  g_return_if_fail (gc != NULL);
  
  if (source_window == NULL)
    source_window = window;

  if (window != source_window)
    {
      g_warning("gdk_window_copy_area: window != source_window not supported.\n");
      return;
    }
  
  priv = (GdkWindowPrivate*) window;
  view = (BView*) (priv->xwindow);
  
  if (!priv->destroyed)
    {
      if (view->Window()->Lock())
        {
          view->CopyBits(BRect(source_x, source_y, 
                               source_x + width - 1, source_y + height - 1),
                         BRect(x, y, x + width - 1, y + height - 1));
          view->Window()->Unlock();
        }
    }
}

void
gdk_window_raise (GdkWindow *window)
{
  GdkWindowPrivate *priv;
  
  g_return_if_fail (window != NULL);
  
  priv = (GdkWindowPrivate*) window;
  
  if (!priv->destroyed)
    g_warning("gdk_window_raise: unimplemented!\n");
}

void
gdk_window_lower (GdkWindow *window)
{
  GdkWindowPrivate *priv;
  
  g_return_if_fail (window != NULL);
  
  priv = (GdkWindowPrivate*) window;
  
  if (!priv->destroyed)
    g_warning("gdk_window_lower: unimplemented!\n");
}

void
gdk_window_set_user_data (GdkWindow *window,
			  gpointer   user_data)
{
  g_return_if_fail (window != NULL);
  
  window->user_data = user_data;
}

void
gdk_window_set_hints (GdkWindow *window,
		      gint       x,
		      gint       y,
		      gint       min_width,
		      gint       min_height,
		      gint       max_width,
		      gint       max_height,
		      gint       flags)
{
  GdkWindowPrivate *priv;
  
  g_return_if_fail (window != NULL);
  
  priv = (GdkWindowPrivate*) window;
  if (priv->destroyed)
    return;
  
  g_warning("gdk_window_set_hints: not implemented!");
}

void 
gdk_window_set_geometry_hints (GdkWindow      *window,
			       GdkGeometry    *geometry,
			       GdkWindowHints  geom_mask)
{
  GdkWindowPrivate *priv;
  BWindow *bwin;
  float min_width;
  float min_height;
  float max_width;
  float max_height;
  
  g_return_if_fail (window != NULL);
  
  priv = (GdkWindowPrivate*) window;
  if (priv->destroyed)
    return;

  bwin = (BWindow*) (priv->bewindow);
  g_return_if_fail (bwin != NULL);

  bwin->GetSizeLimits(&min_width, &max_width, &min_height, &max_height);

  if (geom_mask & GDK_HINT_POS)
    {
      /* XXX */
    }

  if (geom_mask & GDK_HINT_MIN_SIZE)
    {
      min_width = (float) geometry->min_width;
      min_height = (float) geometry->min_height;
    }
  
  if (geom_mask & GDK_HINT_MAX_SIZE)
    {
      max_width = (float) geometry->max_width;
      max_height = (float) geometry->max_height;
    }

  GDK_NOTE (MISC, 
            g_print("gdk_window_set_geometry_hints: window %p  min: %d,%d   max: %d,%d\n",
                    window,
                    (gint)min_width, (gint)min_height, 
                    (gint)max_width, (gint)max_height)); 
  bwin->SetSizeLimits(min_width, max_width, min_height, max_height);


  if (geom_mask & GDK_HINT_BASE_SIZE)
    {
      g_warning("gdk_window_set_geometry_hints: GDK_HINT_BASE_SIZE not implemented!");
    }

  if (geom_mask & GDK_HINT_RESIZE_INC)
    {
      g_warning("gdk_window_set_geometry_hints: GDK_HINT_RESIZE_INC not implemented!");
    }

  if (geom_mask & GDK_HINT_ASPECT)
    {
      g_warning("gdk_window_set_geometry_hints: GDK_HINT_ASPECT not implemented!");
    }
}

void
gdk_window_set_title (GdkWindow   *window,
		      const gchar *title)
{
  GdkWindowPrivate *priv;
  BWindow *win;
  
  g_return_if_fail (window != NULL);
  
  priv = (GdkWindowPrivate*) window;
  win = (BWindow*) (priv->bewindow);

  if (!priv->destroyed && win)
    win->SetTitle(title);
}

void          
gdk_window_set_role (GdkWindow   *window,
		     const gchar *role)
{
  GdkWindowPrivate *priv;
  
  g_return_if_fail (window != NULL);
  
  priv = (GdkWindowPrivate*) window;
  
  g_warning("gdk_window_set_role: not implemented!");
}

void          
gdk_window_set_transient_for (GdkWindow *window, 
			      GdkWindow *parent)
{
  GdkWindowPrivate *priv;
  GdkWindowPrivate *parent_private;
  
  g_return_if_fail (window != NULL);
  
  priv = (GdkWindowPrivate*) window;
  parent_private = (GdkWindowPrivate*) parent;
  
  if (!priv->destroyed && !parent_private->destroyed)
    g_warning("gdk_window_set_transient_for: not implemented!");
}

void
gdk_window_set_background (GdkWindow *window,
			   GdkColor  *color)
{
  GdkWindowPrivate *priv;
  BView *view;
  
  g_return_if_fail (window != NULL);
  
  priv = (GdkWindowPrivate*) window;
  view = (BView*) (priv->xwindow);

  if (!priv->destroyed)
    {
      if (view->Window()->Lock())
        {
          GDK_NOTE (MISC, g_print ("gdk_window_set_background: window %p, color rgb %d %d %d\n", window, color->red/256, color->green/256, color->blue/256));
          view->SetViewColor((uchar)(color->red/256), 
                             (uchar)(color->green/256), 
                             (uchar)(color->blue/256));
          view->SetLowColor((uchar)(color->red/256), 
                             (uchar)(color->green/256), 
                             (uchar)(color->blue/256));
          view->Window()->Unlock();
        }
    }
}

void
gdk_window_set_back_pixmap (GdkWindow *window,
			    GdkPixmap *pixmap,
			    gint       parent_relative)
{
  GdkWindowPrivate *window_private;
  GdkPixmapPrivate *pixmap_private;
  
  g_return_if_fail (window != NULL);
  
  window_private = (GdkWindowPrivate*) window;
  pixmap_private = (GdkPixmapPrivate*) pixmap;

  if (!pixmap)
    return;
  
  g_warning("gdk_window_set_back_pixmap: not implemented!");

}

void
gdk_window_set_cursor (GdkWindow *window,
		       GdkCursor *cursor)
{
  GdkWindowPrivate *window_private;
  GdkCursorPrivate *cursor_private;
  static int warned = 0;
  
  g_return_if_fail (window != NULL);
  
  window_private = (GdkWindowPrivate*) window;
  cursor_private = (GdkCursorPrivate*) cursor;
  
  if (!window_private->destroyed)
    {
      if (!warned)
        {
          g_warning("gdk_window_set_cursor: not implemented! (only warned once)");
          warned = 1;
        }
    }
}

void
gdk_window_set_colormap (GdkWindow   *window,
			 GdkColormap *colormap)
{
  GdkWindowPrivate *window_private;
  GdkColormapPrivate *colormap_private;
  
  g_return_if_fail (window != NULL);
  g_return_if_fail (colormap != NULL);
  
  window_private = (GdkWindowPrivate*) window;
  colormap_private = (GdkColormapPrivate*) colormap;
  
  if (!window_private->destroyed)
    g_warning("gdk_window_set_colormap: not implemented!");
}

void
gdk_window_get_user_data (GdkWindow *window,
			  gpointer  *data)
{
  g_return_if_fail (window != NULL);
  
  *data = window->user_data;
}

void
gdk_window_get_geometry (GdkWindow *window,
			 gint      *x,
			 gint      *y,
			 gint      *width,
			 gint      *height,
			 gint      *depth)
{
  GdkWindowPrivate *window_private;
  
  if (!window)
    {
      BScreen screen(B_MAIN_SCREEN_ID);
      BRect rect;
 
      rect = screen.Frame();
      *x = (int) rect.left;
      *y = (int) rect.top;
      *width = (int) rect.Width() + 1;
      *height = (int) rect.Height() + 1;
      *depth = 32;

      return;
    }
  
  window_private = (GdkWindowPrivate*) window;
  
  if (!window_private->destroyed)
    {
      gdk_window_get_position(window, x, y);
      gdk_window_get_size(window, width, height);
      if (depth)
        *depth = 32;
    }
}

void
gdk_window_get_position (GdkWindow *window,
			 gint      *x,
			 gint      *y)
{
  GdkWindowPrivate *window_private;
  BView *view;
  BRect rect;
  BWindow *bwin;
  
  g_return_if_fail (window != NULL);
  
  window_private = (GdkWindowPrivate*) window;

  if (!window_private->destroyed)
    {
      view = (BView*) (window_private->xwindow);
      g_return_if_fail (view != NULL);

      bwin = (BWindow*) (window_private->bewindow);

      if (bwin)
        {
          if (bwin->Lock())
            {
              rect = bwin->Frame();
              bwin->Unlock();
            }
        }
      else 
        {
          if (view->Window()->Lock())
            {
              rect = view->Frame();
              view->Window()->Unlock();
            }
        }

      if (x)
        *x = (int) rect.left;
      if (y)
        *y = (int) rect.top;
    }
}

void
gdk_window_get_size (GdkWindow *window,
		     gint       *width,
		     gint       *height)
{
  GdkWindowPrivate *window_private;
  BView *view;
  BRect rect;
  BWindow *win;
  
  g_return_if_fail (window != NULL);
  
  window_private = (GdkWindowPrivate*) window;
  
  if (!window_private->destroyed)
    {
      view = (BView*) window_private->xwindow;
      g_return_if_fail (view != NULL);

      win = view->Window();
      g_return_if_fail (win != NULL);

      if (win->Lock())
        {
          rect = view->Frame();
          win->Unlock();

          if (width)
            *width = (int) (rect.right - rect.left);
          if (height)
            *height = (int) (rect.bottom - rect.top);
        }
    }
}

GdkVisual*
gdk_window_get_visual (GdkWindow *window)
{
  GdkWindowPrivate *window_private;
  
  g_return_val_if_fail (window != NULL, NULL);
  
  window_private = (GdkWindowPrivate*) window;

  if (window_private && !window_private->destroyed)
    return gdk_visual_get_system();
  
  return NULL;
}

GdkColormap*
gdk_window_get_colormap (GdkWindow *window)
{
  GdkWindowPrivate *window_private;
  
  g_return_val_if_fail (window != NULL, NULL);
  window_private = (GdkWindowPrivate*) window;
  
  g_return_val_if_fail (window_private->window_type != GDK_WINDOW_PIXMAP, NULL);

  if (!window_private->destroyed)
    return gdk_colormap_get_system();
  
  return NULL;
}

GdkWindowType
gdk_window_get_type (GdkWindow *window)
{
  GdkWindowPrivate *window_private;
  
  g_return_val_if_fail (window != NULL, (GdkWindowType) -1);
  
  window_private = (GdkWindowPrivate*) window;
  return (GdkWindowType) (window_private->window_type);
}

gint
gdk_window_get_origin (GdkWindow *window,
		       gint      *x,
		       gint      *y)
{
  GdkWindowPrivate *priv;
  BView *view;
  BWindow *bwin;
  BRect view_rect;
  BRect bwin_rect;
  
  g_return_val_if_fail (window != NULL, 0);
  
  priv = (GdkWindowPrivate*) window;
  view = (BView*) priv->xwindow;
  
  if (!priv->destroyed)
    {
      bwin = view->Window();

      if (bwin->Lock())
        {
          view_rect = view->Frame();
          bwin->Unlock();
        }

      bwin_rect = bwin->Frame();

      if (x)
        *x = bwin_rect.left + view_rect.left;
      if (y)
        *y = bwin_rect.top + view_rect.top;

      return 1;
    }
  
  return 0;
}

gboolean
gdk_window_get_deskrelative_origin (GdkWindow *window,
				    gint      *x,
				    gint      *y)
{
  GdkWindowPrivate *priv;
  
  g_return_val_if_fail (window != NULL, 0);
  
  priv = (GdkWindowPrivate*) window;
  
  if (!priv->destroyed)
    g_warning("gdk_window_get_deskrelative_origin: not implemented!");
  
  return 0;
}

void
gdk_window_get_root_origin (GdkWindow *window,
			    gint      *x,
			    gint      *y)
{
  GdkWindowPrivate *priv;
  
  g_return_if_fail (window != NULL);
  
  priv = (GdkWindowPrivate*) window;

  if (!priv->destroyed)
    g_warning("gdk_window_get_deskrelative_origin: not implemented!");
}

GdkWindow*
gdk_window_get_pointer (GdkWindow       *window,
			gint            *x,
			gint            *y,
			GdkModifierType *mask)
{
  GdkWindowPrivate *priv;
  GdkWindow *return_val;
  BView *view;
  BPoint cursor;
  uint32 buttons;

  do_motion_hints = 1;

  /* Test for cursor wanted in absolute screen coordinates. */
  if (!window)
    {
      BWindow *win;
      BView *top_view;
      GdkWindow *top_window;
      BRect rect;
   
      win = be_app->WindowAt(0);
      g_return_val_if_fail(win != NULL, NULL);

      top_window = gdk_window_lookup(win);
      g_return_val_if_fail(top_window != NULL, NULL);

      top_view = (BView*) ((GdkWindowPrivate*)top_window)->xwindow;

      if (top_view->Window()->Lock())
        {
          top_view->GetMouse(&cursor, &buttons);
          top_view->Window()->Unlock();
        }
      rect = win->Frame(); 
 
      if (x)
        *x = rect.left + cursor.x;
      if (y)
        *y = rect.top + cursor.y;

      return NULL;
    }
  
  priv = (GdkWindowPrivate*) window;
  view = (BView*) priv->xwindow;

  g_return_val_if_fail(view != NULL,  NULL); //testgtk color selector breaks assert
  
  return_val = NULL;

  if (view->Window()->Lock())
    {
      view->GetMouse(&cursor, &buttons);
      view->Window()->Unlock();

      return_val = gdk_window_lookup(view);
    }

  if (x) 
    *x = (int) cursor.x;

  if (y)
    *y = (int) cursor.y;
 
  if (mask)
    {
      *mask = 0;
      if (buttons & B_PRIMARY_MOUSE_BUTTON)
        *mask |= GDK_BUTTON1_MASK;
      if (buttons & B_SECONDARY_MOUSE_BUTTON)
        *mask |= GDK_BUTTON2_MASK;
      if (buttons & B_TERTIARY_MOUSE_BUTTON)
        *mask |= GDK_BUTTON3_MASK;
    }

  return return_val;
}

GdkWindow*
gdk_window_at_pointer (gint *win_x,
		       gint *win_y)
{
  GdkWindow *window;
  
  g_warning("gdk_window_at_pointer: not implemented!");

  window = NULL;
  
  return window;
}

GdkWindow*
gdk_window_get_parent (GdkWindow *window)
{
  g_return_val_if_fail (window != NULL, NULL);
  
  return ((GdkWindowPrivate*) window)->parent;
}

GdkWindow*
gdk_window_get_toplevel (GdkWindow *window)
{
  GdkWindowPrivate *priv;
  
  g_return_val_if_fail (window != NULL, NULL);
  
  priv = (GdkWindowPrivate*) window;
  
  while (priv->window_type == GDK_WINDOW_CHILD)
    {
      window = ((GdkWindowPrivate*) window)->parent;
      priv = (GdkWindowPrivate*) window;
    }
  
  return window;
}

GList*
gdk_window_get_children (GdkWindow *window)
{
  GdkWindowPrivate *priv;
  
  g_return_val_if_fail (window != NULL, NULL);
  
  priv = (GdkWindowPrivate*) window;
  if (priv->destroyed)
    return NULL;

  g_warning("gdk_window_get_children: not implemented!");
  
  return NULL;
}

GdkEventMask  
gdk_window_get_events (GdkWindow *window)
{
  GdkWindowPrivate *priv;
  
  g_return_val_if_fail (window != NULL, (GdkEventMask) 0);
  
  priv = (GdkWindowPrivate*) window;
  if (priv->destroyed)
    return (GdkEventMask) 0;
  
  return (GdkEventMask) priv->event_mask;
}

void          
gdk_window_set_events (GdkWindow       *window,
		       GdkEventMask     event_mask)
{
  GdkWindowPrivate *priv;
  
  g_return_if_fail (window != NULL);
  
  priv = (GdkWindowPrivate*) window;
  if (priv->destroyed)
    return;
  
  priv->event_mask = event_mask;
}

void
gdk_window_add_colormap_windows (GdkWindow *window)
{
  g_warning("gdk_window_add_colormap_windows: not implemented!");
}

/*
 * This needs the X11 shape extension.
 * If not available, shaped windows will look
 * ugly, but programs still work.    Stefan Wille
 */
void
gdk_window_shape_combine_mask (GdkWindow *window,
			       GdkBitmap *mask,
			       gint x, gint y)
{
  g_warning("gdk_window_shape_combine_mask: not implemented!");
}

void          
gdk_window_add_filter (GdkWindow     *window,
		       GdkFilterFunc  function,
		       gpointer       data)
{
  GdkWindowPrivate *priv;
  GList *tmp_list;
  GdkEventFilter *filter;
  
  priv = (GdkWindowPrivate*) window;
  if (priv && priv->destroyed)
    return;
  
  if (priv)
    tmp_list = priv->filters;
  else
    tmp_list = gdk_default_filters;
  
  while (tmp_list)
    {
      filter = (GdkEventFilter *)tmp_list->data;
      if ((filter->function == function) && (filter->data == data))
	return;
      tmp_list = tmp_list->next;
    }
  
  filter = g_new (GdkEventFilter, 1);
  filter->function = function;
  filter->data = data;
  
  if (priv)
    priv->filters = g_list_append (priv->filters, filter);
  else
    gdk_default_filters = g_list_append (gdk_default_filters, filter);
}

void
gdk_window_remove_filter (GdkWindow     *window,
			  GdkFilterFunc  function,
			  gpointer       data)
{
  GdkWindowPrivate *priv;
  GList *tmp_list, *node;
  GdkEventFilter *filter;
  
  priv = (GdkWindowPrivate*) window;
  
  if (priv)
    tmp_list = priv->filters;
  else
    tmp_list = gdk_default_filters;
  
  while (tmp_list)
    {
      filter = (GdkEventFilter *)tmp_list->data;
      node = tmp_list;
      tmp_list = tmp_list->next;
      
      if ((filter->function == function) && (filter->data == data))
	{
	  if (priv)
	    priv->filters = g_list_remove_link (priv->filters, node);
	  else
	    gdk_default_filters = g_list_remove_link (gdk_default_filters, node);
	  g_list_free_1 (node);
	  g_free (filter);
	  
	  return;
	}
    }
}

void
gdk_window_set_override_redirect (GdkWindow *window,
				  gboolean override_redirect)
{
  g_warning("gdk_window_set_override_redirect: not implemented!");
}

void          
gdk_window_set_icon (GdkWindow *window, 
		     GdkWindow *icon_window,
		     GdkPixmap *pixmap,
		     GdkBitmap *mask)
{
  g_warning("gdk_window_set_icon: not implemented!");
}

void          
gdk_window_set_icon_name (GdkWindow *window, 
			  gchar *    name)
{
  g_warning("gdk_window_set_icon_name: not implemented!");
}

void          
gdk_window_set_group (GdkWindow *window, 
		      GdkWindow *leader)
{
  g_warning("gdk_window_set_group: not implemented!");
}

void
gdk_window_set_decorations (GdkWindow      *window,
			    GdkWMDecoration decorations)
{
  g_warning("gdk_window_set_decorations: not implemented!");
}

void
gdk_window_set_functions (GdkWindow    *window,
			  GdkWMFunction functions)
{
  g_warning("gdk_window_set_functions: not implemented!");
}

GList *
gdk_window_get_toplevels (void)
{
  GList *new_list = NULL;
  GList *tmp_list;
  
  tmp_list = gdk_root_parent.children;
  while (tmp_list)
    {
      new_list = g_list_prepend (new_list, tmp_list->data);
      tmp_list = tmp_list->next;
    }
  
  return new_list;
}

void
gdk_window_set_child_shapes (GdkWindow *window)
{
  g_warning("gdk_window_set_child_shapes: not implemented!");
}

void
gdk_window_merge_child_shapes (GdkWindow *window)
{
  g_warning("gdk_window_merge_child_shapes: not implemented!");
}

/*************************************************************
 * gdk_window_is_visible:
 *     Check if the given window is mapped.
 *   arguments:
 *     window: 
 *   results:
 *     is the window mapped
 *************************************************************/

gboolean 
gdk_window_is_visible (GdkWindow *window)
{
  GdkWindowPrivate *priv = (GdkWindowPrivate *)window;
  
  g_return_val_if_fail (window != NULL, FALSE);
  
  return priv->mapped;
}

/*************************************************************
 * gdk_window_is_viewable:
 *     Check if the window and all ancestors of the window
 *     are mapped. (This is not necessarily "viewable" in
 *     the X sense, since we only check as far as we have
 *     GDK window parents, not to the root window)
 *   arguments:
 *     window:
 *   results:
 *     is the window viewable
 *************************************************************/

gboolean 
gdk_window_is_viewable (GdkWindow *window)
{
  GdkWindowPrivate *priv = (GdkWindowPrivate *)window;
  
  g_return_val_if_fail (window != NULL, FALSE);
  
  while (priv && 
	 (priv != &gdk_root_parent) &&
	 (priv->window_type != GDK_WINDOW_FOREIGN))
    {
      if (!priv->mapped)
	return FALSE;
      
      priv = (GdkWindowPrivate *)priv->parent;
    }
  
  return TRUE;
}

void          
gdk_drawable_set_data (GdkDrawable   *drawable,
		       const gchar   *key,
		       gpointer	      data,
		       GDestroyNotify destroy_func)
{
  g_dataset_set_data_full (drawable, key, data, destroy_func);
}


/*************************************************************
 * gdk_window_set_static_gravities:
 *     Set the bit gravity of the given window to static,
 *     and flag it so all children get static subwindow
 *     gravity.
 *   arguments:
 *     window: window for which to set static gravity
 *     use_static: Whether to turn static gravity on or off.
 *   results:
 *     Does the XServer support static gravity?
 *************************************************************/

gboolean 
gdk_window_set_static_gravities (GdkWindow *window,
				 gboolean   use_static)
{
  g_warning("gdk_window_set_static_gravities: not implemented!");
  return FALSE;
}
