/* GDK - The GIMP Drawing Kit
 * Copyright (C) 1995-1997 Peter Mattis, Spencer Kimball and Josh MacDonald
 *
 * BeOS Port
 * Copyright (C) 1999 EventLoop, Inc.
 *   Shawn T. Amundson <amundson@gtk.org>
 *   James Mitchell <mitchell@eventloop.com>
 *         
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/*
 * Modified by the GTK+ Team and others 1997-1999.  See the AUTHORS
 * file for a list of people on the GTK+ Team.  See the ChangeLog
 * files for a list of changes.  These files are distributed with
 * GTK+ at ftp://ftp.gtk.org/pub/gtk/. 
 */

#include "config.h"

#include <Font.h>

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>

#include <gdk/gdk.h>
#include "gdkconfig.h"
#include "gdkprivate.h"

extern "C" {
void gdk_font_get_list (char ***names, int *num);
}

GdkFont*
gdk_font_load (const gchar *font_name)
{
  GdkFont *font;
  GdkFontPrivate *priv;
  BFont *bfont;

  int numfields;
  int n1, n2, c;
  char *p;

  char foundry[32];
  char family[100];
  char weight[32];
  char slant[32];
  char set_width[32];
  char spacing[32];
  char registry[32];
  char encoding[32];
  char pixel_size[10];
  char point_size[10];
  char res_x[10];
  char res_y[10];
  char avg_width[10];

  const char *searchfont;

  int fnHeight, fnFace, fnEncoding, fnSpacing;
  guint *fontxid;
  fnHeight = 10;

  g_warning("gdk_font_load: CALLED with %s", font_name);
  g_return_val_if_fail (font_name != NULL, NULL);

  bfont = new BFont(be_plain_font);
  priv = g_new (GdkFontPrivate, 1);
  font = (GdkFont*) priv;

  numfields = sscanf (font_name,
		      "-%30[^-]-%100[^-]-%30[^-]-%30[^-]-%30[^-]-%n",
                      foundry,
                      family,
                      weight,
                      slant,
                      set_width,
                      &n1);
 

  if (numfields == 0)
   {
     /* Assuming it's a regular BeOS font name */ 
     fnFace = B_REGULAR_FACE;
     fnSpacing = B_CHAR_SPACING;
     fnEncoding = B_UNICODE_UTF8;
     fnHeight = 10;
     searchfont = font_name;
   }
  else if (numfields != 5)
   {
     g_warning ("gdk_font_load: font name %s illegal", font_name);
     g_free (font);
     return NULL;
   }
  else
   {
     /* It must be a XLFD name */

      /* Check for hex escapes in the font family,
       * put in there by gtkfontsel.
       */
      p = family;
      while (*p)
        {
          if (*p == '%' && isxdigit (p[1]) && isxdigit (p[2]))
            {
              sscanf (p+1, "%2x", &c);
              *p = c;
              strcpy (p+1, p+3);
            }
          p++;
        }

      /* Skip add_style which often is empty in the requested font name */
      while (font_name[n1] && font_name[n1] != '-')
        n1++;
      numfields++;

      numfields += sscanf (font_name + n1,
                           "-%8[^-]-%8[^-]-%8[^-]-%8[^-]-%30[^-]-%8[^-]-%30[^-]-
%30[^-]%n",
                           pixel_size,
                           point_size,
                           res_x,
                           res_y,
                           spacing,
                           avg_width,
                           registry,
                           encoding,
                           &n2);

      if (numfields != 14 || font_name[n1 + n2] != '\0')
        {
          g_warning ("gdk_font_load: font name %s illegal", font_name);
          g_free (font);
          return NULL;
        }

      if (strcmp (pixel_size, "*") == 0)
       { 
	 if (strcmp (point_size, "*") == 0)
          fnHeight = 10;
         else
          fnHeight = atoi(point_size) / 10;
       }
      else
        fnHeight = atoi (pixel_size);

      fnFace = 0;

      if (g_strcasecmp (weight, "thin") == 0)
        fnFace |= B_REGULAR_FACE;
      else if (g_strcasecmp (weight, "extralight") == 0)
        fnFace |= B_REGULAR_FACE;
      else if (g_strcasecmp (weight, "ultralight") == 0)
        fnFace |= B_REGULAR_FACE;
      else if (g_strcasecmp (weight, "light") == 0)
        fnFace |= B_REGULAR_FACE;
      else if (g_strcasecmp (weight, "normal") == 0)
        fnFace |= B_REGULAR_FACE;
      else if (g_strcasecmp (weight, "regular") == 0)
        fnFace |= B_REGULAR_FACE;
      else if (g_strcasecmp (weight, "medium") == 0)
        fnFace |= B_REGULAR_FACE;
      else if (g_strcasecmp (weight, "semibold") == 0)
        fnFace |= B_BOLD_FACE;
      else if (g_strcasecmp (weight, "demibold") == 0)
        fnFace |= B_BOLD_FACE;
      else if (g_strcasecmp (weight, "bold") == 0)
        fnFace |= B_BOLD_FACE;
      else if (g_strcasecmp (weight, "extrabold") == 0)
        fnFace |= B_BOLD_FACE;
      else if (g_strcasecmp (weight, "ultrabold") == 0)
        fnFace |= B_BOLD_FACE;
      else if (g_strcasecmp (weight, "heavy") == 0)
        fnFace |= B_BOLD_FACE;
      else if (g_strcasecmp (weight, "black") == 0)
        fnFace |= B_BOLD_FACE;
      else
        fnFace |= B_REGULAR_FACE;

      if (g_strcasecmp (slant, "italic") == 0
          || g_strcasecmp (slant, "oblique") == 0
          || g_strcasecmp (slant, "i") == 0
          || g_strcasecmp (slant, "o") == 0)
        {
          fnFace |= B_ITALIC_FACE;
          fnFace &= ~B_REGULAR_FACE;
        }

      if (g_strcasecmp (registry, "iso8859") == 0)
       {
        if (strcmp (encoding, "1") == 0)
          fnEncoding = B_ISO_8859_1;
        else if (strcmp (encoding, "2") == 0)
	  fnEncoding = B_ISO_8859_2;
	else if (strcmp (encoding, "3") == 0)
	  fnEncoding = B_ISO_8859_3;
	else if (strcmp (encoding, "4") == 0)
	  fnEncoding = B_ISO_8859_4;
	else if (strcmp (encoding, "5") == 0)
	  fnEncoding = B_ISO_8859_5;
	else if (strcmp (encoding, "6") == 0)
	  fnEncoding = B_ISO_8859_6;
	else if (strcmp (encoding, "7") == 0)
	  fnEncoding = B_ISO_8859_7;
	else if (strcmp (encoding, "8") == 0)
	  fnEncoding = B_ISO_8859_8;
	else if (strcmp (encoding, "9") == 0)
	  fnEncoding = B_ISO_8859_9;
	else if (strcmp (encoding, "10") == 0)
	  fnEncoding = B_ISO_8859_10;
	else
	  fnEncoding = B_ISO_8859_1;
       }
      else
       fnEncoding = B_UNICODE_UTF8;

      if (g_strcasecmp (spacing, "m") == 0)
        fnSpacing = B_FIXED_SPACING;
      else if (g_strcasecmp (spacing, "p") == 0)
        fnSpacing = B_STRING_SPACING;

      searchfont = family;
    }

  int numFamilies = count_font_families();

  for (int i = 0; i < numFamilies; i++)
   {
    font_family bfamily;
    uint32 flags;

    if (get_font_family(i, &bfamily, &flags) == B_OK)
      if (g_strcasecmp (searchfont, bfamily) == 0)
       {
	 bfont->SetFamilyAndFace(bfamily, fnFace);
	 break;
       }
#if 0
    g_warning("fam = %s search = %s", bfamily, searchfont);
#endif
   }

  bfont->SetSpacing(fnSpacing);
  bfont->SetEncoding(fnEncoding);
  bfont->SetSize(fnHeight);
  /*bfont->SetFlags(B_DISABLE_ANTIALIASING);*/
  
  /* we hope this contains a semi-valid font at this point */
  
  priv->xfont = (GdkBFont*) bfont;

  priv->ref_count = 1;
  font->type = GDK_FONT_FONT;

  font_height bfontheight;
  bfont->GetHeight(&bfontheight);
  font->ascent = (gint) bfontheight.ascent;
  font->descent = (gint) bfontheight.descent;

  fontxid = g_new(guint, 1);
  *fontxid = (guint) priv->xfont;

  gdk_xid_table_insert((void**)(&fontxid), font);

  return font;
}

GdkFont*
gdk_fontset_load (gchar *fontset_name)
{
  g_warning ("gdk_fontset_load: Not implemented");
  return NULL;
}

GdkFont*
gdk_font_ref (GdkFont *font)
{
  GdkFontPrivate *priv;

  g_return_val_if_fail (font != NULL, NULL);

  priv = (GdkFontPrivate*) font;
  priv->ref_count += 1;
  return font;
}

void
gdk_font_unref (GdkFont *font)
{
  GdkFontPrivate *priv;

  g_return_if_fail (font != NULL);
 
  priv = (GdkFontPrivate*) font;

  priv->ref_count -= 1;
  if (priv->ref_count == 0)
    {
      switch (font->type)
	{
	case GDK_FONT_FONT:
	  gdk_xid_table_remove ((void*) priv->xfont);
	  delete priv->xfont;
	  break;

	default:
	  g_assert_not_reached ();
	}
      g_free (font);
    }
}

gint
gdk_font_id (const GdkFont *font)
{
  const GdkFontPrivate *priv;

  g_return_val_if_fail (font != NULL, 0);

  priv = (const GdkFontPrivate*) font;

  if (font->type == GDK_FONT_FONT)
    return (gint) priv->xfont; 

  g_assert_not_reached ();
  return 0;
}

gint
gdk_font_equal (const GdkFont *fonta,
                const GdkFont *fontb)
{
  const GdkFontPrivate *priva;
  const GdkFontPrivate *privb;

  g_return_val_if_fail (fonta != NULL, FALSE);
  g_return_val_if_fail (fontb != NULL, FALSE);

  priva = (const GdkFontPrivate*) fonta;
  privb = (const GdkFontPrivate*) fontb;

  if (fonta->type == GDK_FONT_FONT && fontb->type == GDK_FONT_FONT)
    return (priva->xfont == privb->xfont); 

  g_assert_not_reached ();
  return 0;
}

gint
gdk_string_width (GdkFont     *font,
		  const gchar *string)
{
  return gdk_text_width (font, string, strlen (string));
}

gint
gdk_text_width (GdkFont      *font,
		const gchar  *text,
		gint          text_length)
{
  GdkFontPrivate *priv;
  BFont *bfont;
  float b_width;
  gint width;

  g_return_val_if_fail (font != NULL, -1);
  g_return_val_if_fail (text != NULL, -1);

  priv = (GdkFontPrivate*) font;
  bfont = (BFont*) priv->xfont;

  switch (font->type)
    {
    case GDK_FONT_FONT:
      width = (gint) bfont->StringWidth(text, text_length);
      break;
    case GDK_FONT_FONTSET:
      g_warning("gdk_text_width: a GDK_FONT_FONTSET! eww");

    default:
      g_assert_not_reached ();
    }
  return width;
}

gint
gdk_text_width_wc (GdkFont	  *font,
		   const GdkWChar *text,
		   gint		   text_length)
{
  gchar *new_text;
  gint width;

  new_text = gdk_wcstombs(text);
  width = gdk_text_width(font, new_text, text_length);

  return width;
}

gint
gdk_char_width (GdkFont *font,
		gchar    character)
{
  return gdk_text_width (font, &character, 1);
}

gint
gdk_char_width_wc (GdkFont *font,
		   GdkWChar character)
{
  return gdk_text_width_wc (font, &character, 1);
}


gint
gdk_string_measure (GdkFont     *font,
                    const gchar *string)
{
  g_return_val_if_fail (font != NULL, -1);
  g_return_val_if_fail (string != NULL, -1);

  return gdk_text_measure (font, string, strlen (string));
}

void
gdk_text_extents (GdkFont     *font,
                  const gchar *text,
                  gint         text_length,
		  gint        *lbearing,
		  gint        *rbearing,
		  gint        *width,
		  gint        *ascent,
		  gint        *descent)
{
  GdkFontPrivate *priv;
  BFont *bfont;
  font_height bfontheight;

  g_return_if_fail (font != NULL);
  g_return_if_fail (text != NULL);

  priv = (GdkFontPrivate*) font;
  bfont = (BFont*) priv->xfont;

  bfont->GetHeight(&bfontheight);

  switch (font->type)
    {
    case GDK_FONT_FONT:
      /* XXX This is all quite bogus */
      if (lbearing)
	*lbearing = 0;
      if (rbearing)
	*rbearing = 0;
      if (width)
	*width = (gint) bfont->StringWidth(text, text_length);
      if (ascent)
	*ascent = (gint) bfontheight.ascent;
      if (descent)
	*descent = (gint) bfontheight.descent;
      break;

    default:
      g_assert_not_reached ();
    }
}

void
gdk_text_extents_wc (GdkFont        *font,
		     const GdkWChar *text,
		     gint            text_length,
		     gint           *lbearing,
		     gint           *rbearing,
		     gint           *width,
		     gint           *ascent,
		     gint           *descent)
{
  g_warning ("gdk_text_extents_wc: Not really implemented. Calling gdk_text_width");
  gdk_text_extents(font, (char*) text, text_length, lbearing, rbearing, width, ascent, descent);
}

void
gdk_string_extents (GdkFont     *font,
		    const gchar *string,
		    gint        *lbearing,
		    gint        *rbearing,
		    gint        *width,
		    gint        *ascent,
		    gint        *descent)
{
  g_return_if_fail (font != NULL);
  g_return_if_fail (string != NULL);

  gdk_text_extents (font, string, strlen (string),
		    lbearing, rbearing, width, ascent, descent);
}


gint
gdk_text_measure (GdkFont     *font,
                  const gchar *text,
                  gint         text_length)
{
  GdkFontPrivate *priv;
  gint width;

  g_return_val_if_fail (font != NULL, -1);
  g_return_val_if_fail (text != NULL, -1);

  priv = (GdkFontPrivate*) font;

  switch (font->type)
    {
    case GDK_FONT_FONT:
      return gdk_text_width (font, text, text_length); /* ??? */
      break;

    default:
      g_assert_not_reached ();
    }
  return 0;
}

gint
gdk_char_measure (GdkFont *font,
                  gchar    character)
{
  g_return_val_if_fail (font != NULL, -1);

  return gdk_text_measure (font, &character, 1);
}

gint
gdk_string_height (GdkFont     *font,
		   const gchar *string)
{
  g_return_val_if_fail (font != NULL, -1);
  g_return_val_if_fail (string != NULL, -1);

  return gdk_text_height (font, string, strlen (string));
}

gint
gdk_text_height (GdkFont     *font,
		 const gchar *text,
		 gint         text_length)
{
  GdkFontPrivate *priv;
  BFont *bfont;
  gint height;

  g_return_val_if_fail (font != NULL, -1);
  g_return_val_if_fail (text != NULL, -1);

  priv= (GdkFontPrivate*) font;
  bfont = (BFont*) priv->xfont;

  switch (font->type)
    {
    case GDK_FONT_FONT:
      height = (gint) bfont->Size();
      break;

    default:
      g_error ("font->type = %d", font->type);
    }
  return height;
}

gint
gdk_char_height (GdkFont *font,
		 gchar    character)
{
  g_return_val_if_fail (font != NULL, -1);

  return gdk_text_height (font, &character, 1);
}

#define MAX_FONTS 4096

void
gdk_font_get_list (char ***names, int *num)
{
  static char *fonts[MAX_FONTS];
  static int32 num_fonts = 0;
  int32 num_families;
  int32 num_styles;
  int32 i, j;
  font_family family; 
  font_style style; 
  uint32 flags; 
  int32 size;

  if (num_fonts > 0)
    {
      *names = fonts;
      *num = num_fonts;
    }

  num_families = count_font_families(); 
  for ( int32 i = 0; i < num_families; i++ ) 
    { 
      if (get_font_family(i, &family, &flags) == B_OK) 
        { 
          num_styles = count_font_styles(family); 

          for (j = 0; j < num_styles; j++ ) 
              if ( get_font_style(family, j, &style, &flags) == B_OK ) 
                { 
                  for (size = 6; size < 20; size = size + 2)
                    {
                      fonts[num_fonts] = g_strdup_printf ("-*-%s-normal-r-normal--*-*-*-*-p-*-*-*",
                                             family);
                      fonts[num_fonts+1] = g_strdup_printf ("-*-%s-bold-r-normal--*-*-*-*-p-*-*-*",
                                             family); 
                      fonts[num_fonts+2] = g_strdup_printf ("-*-%s-normal-i-normal--*-*-*-*-p-*-*-*",
                                             family);
                      fonts[num_fonts+3] = g_strdup_printf ("-*-%s-bold-i-normal--*-*-*-*-p-*-*-*",
                                             family); 
                      num_fonts+=4;
                    }
                } 
        } 
      } 

  *names = fonts;
  *num = num_fonts;
}
