/* GDK - The GIMP Drawing Kit
 * Copyright (C) 1995-1997 Peter Mattis, Spencer Kimball and Josh MacDonald
 *
 * BeOS Port
 * Copyright (C) 1999 EventLoop, Inc.
 *   Shawn T. Amundson <amundson@gtk.org>
 *   James Mitchell <mitchell@eventloop.com>
 *         
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/*
 * Modified by the GTK+ Team and others 1997-1999.  See the AUTHORS
 * file for a list of people on the GTK+ Team.  See the ChangeLog
 * files for a list of changes.  These files are distributed with
 * GTK+ at ftp://ftp.gtk.org/pub/gtk/. 
 */

#include <Bitmap.h>

#include <config.h>

/* gcc -ansi -pedantic on GNU/Linux causes warnings and errors
 * unless this is defined:
 * warning: #warning "Files using this header must be compiled with _SVID_SOURCE or _XOPEN_SOURCE"
 */
#ifndef _XOPEN_SOURCE
#  define _XOPEN_SOURCE 1
#endif

#include <stdlib.h>
#include <sys/types.h>

#include "gdk/gdk.h"
#include "gdkprivate.h"


static void gdk_image_put_normal (GdkDrawable *drawable,
				  GdkGC       *gc,
				  GdkImage    *image,
				  gint         xsrc,
				  gint         ysrc,
				  gint         xdest,
				  gint         ydest,
				  gint         width,
				  gint         height);


static GList *image_list = NULL;


void
gdk_image_exit (void)
{
  GdkImage *image;

  while (image_list)
    {
      image = (GdkImage*) (image_list->data);
      gdk_image_destroy (image);
    }
}

GdkImage *
gdk_image_new_bitmap(GdkVisual *visual, gpointer data, gint w, gint h)
/*
 * Desc: create a new bitmap image
 */
{
  GdkImage *image;
  GdkImagePrivate *priv;
  BBitmap *bitmap;
  BRect rect(0.0, 0.0, (float)w-1.0, (float)h-1.0);

  priv = g_new (GdkImagePrivate, 1);
  image = (GdkImage *) priv;

  bitmap = new BBitmap(rect, B_GRAY1);
  bitmap->SetBits(data, w*h/8, 0, B_GRAY1);

  priv->ximage = (GdkBBitmap*) bitmap;
  priv->image_put = gdk_image_put_normal;

  image->type = GDK_IMAGE_NORMAL;
  image->visual = visual;
  image->byte_order = GDK_LSB_FIRST;
  image->width = w;
  image->height = h;
  image->depth = 1;
  image->bpp = 1;
  image->mem = bitmap->Bits();
  image->bpl = bitmap->BytesPerRow();
  return(image);
} /* gdk_image_new_bitmap() */

void
gdk_image_init (void)
{
}

GdkImage*
gdk_image_new (GdkImageType  type,
	       GdkVisual    *visual,
	       gint          width,
	       gint          height)
{
  GdkImage *image;
  GdkImagePrivate *priv;
  BBitmap *bitmap;
  BRect rect(0.0, 0.0, (float)width - 1.0, (float)height - 1.0);

  switch (type)
    {
    case GDK_IMAGE_SHARED:
      g_warning("gdk_image_new: GDK_IMAGE_SHARED not supported");
      return NULL; /* NOT SUPPORTED */
    case GDK_IMAGE_FASTEST:
      image = gdk_image_new (GDK_IMAGE_NORMAL, visual, width, height);
      break;
    default:
      priv = g_new (GdkImagePrivate, 1);
      image = (GdkImage*) priv;

      bitmap = new BBitmap(rect, B_RGB32);

      image->type = type;
      image->visual = visual;
      image->byte_order = GDK_LSB_FIRST;
      image->width = width;
      image->height = height;
      image->depth = 32;
      image->bpp = 4;
      image->bpl = bitmap->BytesPerRow();
      image->mem = bitmap->Bits();

      priv->ximage = (GdkBBitmap*) bitmap;
      priv->image_put = gdk_image_put_normal;
    }

  return image;
}

GdkImage*
gdk_image_bitmap_new (GdkImageType  type,
                      GdkVisual    *visual,
                      gint          width,
                      gint          height)
{
  GdkImage *image;
  GdkImagePrivate *priv;
  BBitmap *bitmap;
  BRect rect(0.0, 0.0, (float)width - 1.0, (float)height - 1.0);

  priv = g_new (GdkImagePrivate, 1);
  image = (GdkImage*) priv;

  bitmap = new BBitmap(rect, B_GRAY1);

  image->type = type;
  image->visual = visual;
  image->byte_order = GDK_LSB_FIRST;
  image->width = width;
  image->height = height;
  image->depth = 1;
  image->bpp = 1;
  image->bpl = bitmap->BytesPerRow();
  image->mem = bitmap->Bits();

  priv->ximage = (GdkBBitmap*) bitmap;
  priv->image_put = gdk_image_put_normal;

  return image;
}

GdkImage*
gdk_image_get (GdkWindow *window,
	       gint       x,
	       gint       y,
	       gint       width,
	       gint       height)
{
  GdkWindowPrivate *pixmap_priv;
  BBitmap *pixmap_bitmap;
  guchar *pixmap_bits;
  BRect rect(0.0, 0.0, (float)width - 1.0, (float)height - 1.0);
  GdkImagePrivate *priv;
  GdkImage *image;
  BBitmap *bitmap;
  uint32 bits_size;
  int32 i;

  pixmap_priv = (GdkWindowPrivate*) window;

  g_return_val_if_fail (window != NULL, NULL);
  g_return_val_if_fail (pixmap_priv->pixmap != NULL, NULL);

  GDK_NOTE (MISC, g_print("gdk_image_get: window %p: %d,%d %dx%d\n",
                          window, x, y, width, height)); 

  pixmap_bitmap = (BBitmap*) pixmap_priv->pixmap;
  pixmap_bits = pixmap_bitmap->Bits();

  priv = g_new (GdkImagePrivate, 1);
  image = (GdkImage*) priv;

  bitmap = new BBitmap(rect, pixmap_bitmap->ColorSpace());
  bits_size = bitmap->BytesPerRow();
  for (i = 0; i < 10; i++)
    {
      bitmap->SetBits(pixmap_bitmap->Bits(),
                      bits_size, i * width,
                      pixmap_bitmap->ColorSpace()); 
    }

  image->type = GDK_IMAGE_NORMAL;
  image->visual = gdk_window_get_visual(window);
  image->byte_order = GDK_LSB_FIRST;
  image->width = width;
  image->height = height;
  image->depth = 1; /* HACK */
  image->bpp = 1; /* HACK */
  image->bpl = bitmap->BytesPerRow();
  image->mem = bitmap->Bits();

  priv->ximage = (GdkBBitmap*) bitmap;
  priv->image_put = gdk_image_put_normal;

  return image;
}

guint32
gdk_image_get_pixel (GdkImage *image,
		     gint x,
		     gint y)
{
  guint32 *pixel;
  guint32 position;

  g_return_val_if_fail (image != NULL, 0);

  pixel = (guint32 *) (image->mem + y * image->bpl + x * image->bpp);

  return *pixel;
}

void
gdk_image_put_pixel (GdkImage *image,
		     gint x,
		     gint y,
		     guint32 pixel)
{
  guint32 *pixelp;

  g_return_if_fail (image != NULL);
  g_return_if_fail (x >= 0 && 
		    x < image->width && 
  		    y >= 0 && 
		    y < image->height);

  pixelp = (guint32 *) (image->mem + y * image->bpl + x * image->bpp);

  *pixelp = pixel;
}

void
gdk_image_destroy (GdkImage *image)
{
  GdkImagePrivate *priv;

  g_return_if_fail (image != NULL);

  priv = (GdkImagePrivate*) image;
  switch (image->type)
    {
    case GDK_IMAGE_NORMAL:
      {
        BBitmap *bitmap;
        bitmap = (BBitmap*) priv->ximage;
        delete bitmap;
      }
      break;

    case GDK_IMAGE_SHARED:
      g_error ("trying to destroy shared memory image when gdk was compiled without shared memory support");
      break;

    default:
      g_assert_not_reached ();
    }

  g_free (image);
}

static void
gdk_image_put_normal (GdkDrawable *drawable,
		      GdkGC       *gc,
		      GdkImage    *image,
		      gint         xsrc,
		      gint         ysrc,
		      gint         xdest,
		      gint         ydest,
		      gint         width,
		      gint         height)
{
  GdkWindowPrivate *drawable_private;
  GdkImagePrivate *image_private;
  GdkGCPrivate *gc_private;
  BView *view;

  g_return_if_fail (drawable != NULL);
  g_return_if_fail (image != NULL);
  g_return_if_fail (gc != NULL);

  drawable_private = (GdkWindowPrivate*) drawable;
  if (drawable_private->destroyed)
    return;

  view = (BView*) drawable_private->xwindow;
  image_private = (GdkImagePrivate*) image;
  gc_private = (GdkGCPrivate*) gc;

  g_return_if_fail (image->type == GDK_IMAGE_NORMAL);

  gdk_gc_predraw ((GdkBView*)view, gc_private);

  if (view->Window()->Lock())
    {
      view->DrawBitmap((BBitmap*)image_private->ximage,
		       BRect(xsrc, ysrc, width+xsrc-1, height+ysrc-1), 
                       BRect(xdest, ydest, width+xdest-1, height+ydest-1));

      view->Window()->Unlock();
    }

  gdk_gc_postdraw ((GdkBView*)view, gc_private);
}
