/* Rapicorn Examples
 * Copyright (C) 2005 Tim Janik
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General
 * Public License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307, USA.
 */
#include <rapicorn/rapicorn.hh>
#include <gtk/gtk.h>

namespace {
using namespace Rapicorn;

static void     register_builtin_images (void);

static void
drawable_draw (Display  &display,
               Drawable &drawable)
{
  Plane &plane = display.create_plane();
  Painter painter (plane);
  Rect area = drawable.allocation();
  Color fg = 0xff000000;
  double lthickness = 2.25;
  painter.draw_simple_line (area.x + 15, area.y + 15, area.x + 35, area.y + 35, lthickness, fg);
  painter.draw_simple_line (area.x + 35, area.y + 35, area.x + 50, area.y + 20, lthickness, fg);
  painter.draw_simple_line (area.x + 50, area.y + 20, area.x + 75, area.y + 90, lthickness, fg);
  painter.draw_simple_line (area.x + 75, area.y + 90, area.x + 230, area.y + 93, lthickness, fg);
  painter.draw_simple_line (area.x + 75, area.y + 120, area.x + 230, area.y + 110, lthickness * 0.5, fg);
}

static Root*
construct_gui (const char *executable)
{
  Factory::must_parse_file ("tour.xml", "Test", Path::dirname (executable));

  /* create root item */
  Handle<Item> ihandle = Factory::create_item ("Root");
  AutoLocker ilocker (ihandle);
  Item &item = ihandle.get();
  Root &root = item.interface<Root&>();
  root.ref_sink();

  /* create dialog */
  Handle<Item> dhandle = Factory::create_item ("tour-dialog");
  AutoLocker dlocker (dhandle);
  Item &dialog = dhandle.get();
  root.add (dialog);

  /* hook up drawable test */
  Drawable &drawable = root.interface<Drawable&>();
  drawable.sig_draw += slot (&drawable_draw, drawable);

  return &root;
}

static bool
timer ()
{
  g_printerr ("zoot!\n");
  return true;
}

static bool
timer2 ()
{
  g_printerr ("FRUMP!\n");
  return true;
}

extern "C" int
main (int   argc,
      char *argv[])
{
  rapicorn_init_with_foreign_gtk (&argc, &argv, "TourTest");
  register_builtin_images();

  Root *root = construct_gui (argv[0]);
  root->run_async();
  gtk_main();
  return 0;
  
  if (1)
    {
      MainLoop *tloop = glib_loop_create();
      tloop->exec_timer (250, 250, slot (timer));
      MainLoopPool::add_loop (tloop);
    }
  if (1)
    {
      MainLoop *tloop = glib_loop_create();
      tloop->exec_timer (125, 125, slot (timer2));
      MainLoopPool::add_loop (tloop);
    }
  
  Thread::Self::exit();

  return 0;
}

static void
register_builtin_images (void)
{
  /* GdkPixbuf RGBA C-Source image dump 1-byte-run-length-encoded */
#ifdef __SUNPRO_C
#pragma align 4 (alpha_rle)
#endif
#ifdef __GNUC__
  static const uint8 alpha_rle[] __attribute__ ((__aligned__ (4))) = 
#else
    static const uint8 alpha_rle[] = 
#endif
    { ""
      /* Pixbuf magic (0x47646b50) */
      "GdkP"
      /* length: header (24) + pixel_data (2916) */
      "\0\0\13|"
      /* pixdata_type (0x2010002) */
      "\2\1\0\2"
      /* rowstride (128) */
      "\0\0\0\200"
      /* width (32) */
      "\0\0\0\40"
      /* height (32) */
      "\0\0\0\40"
      /* pixel_data: */
      "@\0\0\0\0\304\24\306\354\277\31\310\377\260\31\313\377\240\14\300\301"
      "\234\20\255\307\216\10\252\347\215\1\222\377|\13\220\377u\0\231\377r"
      "\6\213\377n\5\211\377`\13\224\377T\4\210\377@\0\207\377B\0\201\377A\6"
      "\225\377G\13\213\377U\11\201\377L+\202\3779Dp\377&Js\377\40bO\377%nM"
      "\377'|>\377-\206*\377/\2038\377@\221)\377<\227(\3770\236*\3777\271\""
      "\377D\312*\377\0\0\0\0\271\34\271\26\277\"\264\354\271\36\273\377\255"
      "\16\266\356\230\4\262\377\232\0\254\377\221\0\227\377\177\2\216\377w"
      "\4\221\377q\5\204\377j\10\213\377_\13\213\377P\10\207\377S\15z\377O\6"
      "\202\377E\0\204\377M\0\205\377L\17\204\377U!}\377M5q\3773Ql\377'YT\377"
      "/l\77\3775z7\377.x4\377C\2034\377H\2245\377D\226/\377A\254(\377Q\273"
      "2\377W\300.\377\202\0\0\0\0\3\247\32\264\25\263\24\251\353\370\355\370"
      "\377\202\377\377\377\377\22\251W\272\377r\2\212\377t\0\211\377n\0\204"
      "\377h\0\216\377X\14\220\377\\\10\216\377S\6\223\377e\20\213\377Y\13\206"
      "\377Q\0x\377V\33~\377Q\36u\377Q*i\377<Kk\377\77ZL\3777l@\377o\223t\377"
      "\202\377\377\377\377\5l\235[\377[\2512\377Y\267(\377T\266'\377T\306\40"
      "Q\203\0\0\0\0\1\362\331\360]\203\377\377\377\377\22\323\255\332\377l"
      "\2\200\377i\0u\377j\0z\377h\0\211\377k\3\211\377`\22\216\377`\23\206"
      "\377g\23\221\377^\25\215\377e\10~\377S$q\377R-l\377@7f\377E6\\\377DU"
      "U\377PaQ\377p\227\200\377\202\377\377\377\377\4s\235Z\377X\2361\377`"
      "\257$\377]\267\34Q\204\0\0\0\0\26\377\377\377\251\377\377\377\377\370"
      "\361\371\376\377\377\377\377\373\370\373\377o\17\206\377i\0w\377[\2v"
      "\377m\0w\377_\14\200\377j\15\202\377g\3\213\377o\7}\377o\10w\377e\7\177"
      "\377h-y\377d@\201\377Y-c\377a1_\377X@P\377bZQ\377x\214x\377\202\377\377"
      "\377\377\3{\225Y\377t\231&\377}\257\31Q\204\0\0\0\0\11\377\377\377\16"
      "\377\377\377\366\377\377\377\377\366\354\366\204\376\376\376\377\377"
      "\377\377\377\246p\254\365Z\0m\377\224Y\235\377\202\377\377\377\377\5"
      "\312\241\317\377\372\367\372\377\262{\273\377v\17\204\377\315\247\315"
      "\377\202\377\377\377\377\5\314\274\323\377\354\344\352\377\377\377\377"
      "\377\244\235\215\377\220\224v\377\202\377\377\377\377\4\321\334\300\377"
      "\374\375\373\367\377\377\377\347\377\377\377_\203\0\0\0\0\1\377\377\377"
      "`\202\377\377\377\377\6\377\377\377\36\374\372\374\322\377\377\377\377"
      "\326\302\330\377\\\0Y\355\233[\235\377\202\377\377\377\377\4\364\354"
      "\364\377\322\255\327\377\200+\206\377\243[\257\377\202\377\377\377\377"
      "\2\343\320\341\377\350\330\342\377\202\377\377\377\377\2\250\240\225"
      "\377\230\217q\377\202\377\377\377\377\5\370\370\364\344\377\377\377\311"
      "\377\377\377\377\377\377\377\374\377\377\377!\202\0\0\0\0\11\377\377"
      "\377\274\377\377\377\377\377\377\377\332\0\0\0\0\377\377\377\207\377"
      "\377\377\377\376\376\376\377w+d\377\242e\224\367\202\377\377\377\377"
      "\4\253r\253\377y\6y\377z\5}\377\301\221\312\377\202\377\377\377\377\2"
      "\232.x\377\233B{\377\202\377\377\377\377\2\260\240\223\377\244\213z\377"
      "\202\377\377\377\377\2\377\377\3773\377\377\377\22\202\377\377\377\377"
      "\4\377\377\377[\0\0\0\0\377\377\377\32\377\377\377\375\206\377\377\377"
      "\377\2\257\212\253\362\230d\212\377\202\377\377\377\377\4\237N\205\377"
      "|\20n\377y\0w\377\306\212\302\377\202\377\377\377\377\2\2302t\377\233"
      "5{\377\202\377\377\377\377\2\272\226\226\377\242\204u\377\202\377\377"
      "\377\377\2\377\377\3773\377\377\377\24\202\377\377\377\377\3\377\377"
      "\377T\0\0\0\0\377\377\377s\202\377\377\377\377\1\377\377\377\244\202"
      "\377\377\377\234\4\377\377\377\350\377\377\377\377\374\373\374\325\241"
      "b\215\357\202\377\377\377\377\4\233I\202\377\177\35c\377\204\21l\377"
      "\237H\211\377\202\377\377\377\377\2\347\317\337\377\351\326\337\377\202"
      "\377\377\377\377\2\273\223\223\377\306\257\246}\202\377\377\377\377\11"
      "\377\377\377\326\377\377\377\311\377\377\377\377\377\377\377\370\377"
      "\377\377\27\0\0\0\0\377\377\377\316\377\377\377\377\377\377\377\336\203"
      "\0\0\0\0\1\377\377\377\217\202\377\377\377\377\1\366\357\363\213\202"
      "\377\377\377\377\10\241K~\364y\14f\377|\31a\377\211\4p\377\262o\253\377"
      "\364\352\362\377\366\357\364\377\311\240\263\377\202\377\377\377\377"
      "\10\334\304\303\222\377\377\377@\377\377\377\377\377\377\377\370\377"
      "\377\377\211\377\377\377\362\377\377\377\345\377\377\377Z\214\0\0\0\0"
      "\15\213\17]\22y\16g\347}\24`\377\204\17U\362~\22X\377\244Z\226\377\354"
      "\330\350\377\350\322\342\377\321\256\311\377\333\310\324\377\377\377"
      "\377\377\377\377\377\376\377\377\377-\224\0\0\0\0\13v\37]\21\200\25^"
      "\346\177\14U\377~\17U\362\206\23`\377\315\241\312\377\362\350\360\377"
      "\375\374\375\377\374\372\373\377\370\363\365\342\377\377\377]\226\0\0"
      "\0\0\10w\26\\\21\177\7k\346\212\0g\377\220\11k\362\204\17e\377\210\23"
      "g\377\202.f\377\210CbQ\231\0\0\0\0\6\204\5]\20\207\1^\352\206\6U\377"
      "\213\2W\377\212\34Z\377\2121eQ\233\0\0\0\0\5\213\0M\241\205\7[\377\225"
      "\11`\377\233\33^\365\223+[\36\232\0\0\0\0\7\222\16h\234\205\14[\377\204"
      "\22a\377\210\15f\351\217\22[\377\224%W\363\236&n\36\230\0\0\0\0\11\236"
      "\21r\234\226\16y\377\224\16u\377\213\27f\377\204\25a\377\217\23a\343"
      "\246\33c\377\255+o\364\253Dy\37\226\0\0\0\0\13\2532w\234\2551\203\377"
      "\230\33\202\377\227\"u\377\227\32|\377\205\23n\377\214*y\377\230\37o"
      "\343\2569{\377\2659\200\364\263Hj\37\224\0\0\0\0\15\276>s\234\2564\203"
      "\377\262\77\202\377\2360v\377\2408\201\377\237)w\377\222'\201\377\230"
      "#\200\377\236,\217\377\2364\205\342\2533\207\377\244Aq\364\263G_\40\212"
      "\0\0\0\0\204\377\377\377\377\5\377\377\377\375\377\377\377\344\377\377"
      "\377q\0\0\0\0\271Uv\234\202\377\377\377\377\6\317\215\270\377\237L\210"
      "\377\244D\207\377\220/\220\377\241'\233\377\230:\217\377\206\377\377"
      "\377\377\1\377\377\377\330\210\0\0\0\0\202\377\377\377\377\3\377\377"
      "\377\303\377\377\377\235\377\377\377\321\202\377\377\377\377\2\324\217"
      "\215\257\260Yp\377\202\377\377\377\377\6\320\230\262\377\246U\207\377"
      "\231F\211\377\235H\216\377\237:\250\377\240E\233\377\202\377\377\377"
      "\377\5\353\326\343\377\344\304\316\363\347\276\311\377\344\303\302\373"
      "\370\356\355\221\210\0\0\0\0\202\377\377\377\377\3\377\377\377d\0\0\0"
      "\0\377\377\377\21\202\377\377\377\377\2\336\251\251\377\300xz\377\202"
      "\377\377\377\377\6\325\235\275\377\251V\210\377\244V\241\377\231H\255"
      "\377\246>\251\377\262F\252\377\202\377\377\377\377\6\325\236\265\377"
      "\271S\202\377\305]{\341\305cZ\377\261KW\365\247WQ!\207\0\0\0\0\202\377"
      "\377\377\377\3\377\377\377d\377\377\377\3\377\377\377T\202\377\377\377"
      "\377\2\331\250\235\377\300\217\206\377\202\377\377\377\377\6\317\236"
      "\301\377\260]\227\377\250\\\253\377\261S\275\377\252T\277\377\256P\263"
      "\377\202\377\377\377\377\7\357\332\344\377\345\303\321\377\351\302\310"
      "\377\351\277\304\363\317\207\214\377\246X[\366\230^O\"\206\0\0\0\0\205"
      "\377\377\377\377\4\375\374\374\377\335\316\314\377\311\243\215\377\311"
      "\226\226\377\202\377\377\377\377\6\325\254\313\377\273g\244\377\260o"
      "\251\377\242f\301\377\251X\274\377\254f\262\377\206\377\377\377\377\4"
      "\343\262\264\360\254Q_\377\223Z[\366\231UM\"\205\0\0\0\0\202\377\377"
      "\377\377\7\377\377\377\305\372\370\371\367\377\377\377\377\372\371\370"
      "\377\315\274\272\377\311\254\233\377\314\230\242\377\202\377\377\377"
      "\377\6\325\256\320\377\262\200\256\377\262\206\254\377\243|\271\377\242"
      "p\316\377\251p\306\377\202\377\377\377\377\11\326\247\262\377\263`\203"
      "\377\253T\207\377\251cx\377\273^a\377\251L]\336\234E_\377\205ET\366\203"
      "\\>#\204\0\0\0\0\202\377\377\377\377\2\327\305\315\303\301\253\255\377"
      "\202\377\377\377\377\3\367\366\365\377\275\260\247\377\272\251\250\377"
      "\202\377\377\377\377\6\322\275\314\377\266\222\261\377\303\242\260\377"
      "\274\233\266\377\263\223\323\377\243\207\301\377\202\377\377\377\377"
      "\12\331\247\263\377\275v\201\377\264e\216\377\244`s\377\247]i\377\256"
      "Gj\377\250No\336\226F^\377\220KI\366\204A;#\203\0\0\0\0\202\377\377\377"
      "\377\3\320\305\310\377\261\230\251\377\344\342\344\377\202\377\377\377"
      "\377\2\333\330\324\377\272\256\257\377\202\377\377\377\377\6\361\353"
      "\356\377\354\336\343\377\352\340\342\377\351\336\350\377\313\304\331"
      "\377\266\247\267\377\202\377\377\377\377\13\356\342\343\377\345\312\322"
      "\377\340\306\316\377\334\303\314\377\333\270\311\377\244Jq\377\254Fp"
      "\377\227M_\335\230HQ\377\230:Q\367\221>K$\202\0\0\0\0\202\377\377\377"
      "\377\7\312\270\313\377\271\223\257\377\305\273\305\377\376\376\376\377"
      "\377\377\377\377\366\363\363\377\321\300\263\377\206\377\377\377\377"
      "\2\346\327\344\377\301\246\272\377\206\377\377\377\377g\374\372\373\377"
      "\236Tx\377\226Fn\377\243Ap\377\237HR\334\2211U\377\222&L\367\0\0\0\0"
      "\241\220\312\234\247\211\304\377\244\227\274\377\243\216\276\377\257"
      "\230\304\377\275\245\310\377\314\277\326\377\324\270\325\377\333\270"
      "\306\377\333\263\307\377\332\276\307\377\314\254\307\377\313\251\307"
      "\377\322\265\323\377\327\276\315\377\316\265\271\377\325\302\273\377"
      "\277\264\301\377\267\271\272\377\272\247\271\377\265\235\265\377\261"
      "\207\250\377\256m\231\377\246n\241\377\237c\227\377\221b\211\377\223"
      "Q{\377\213<v\377\2238b\377\224.R\333\204\40W\377\0\0\0\0\224\225\333"
      "\377\241\240\325\377\240\245\311\377\247\242\321\377\256\244\310\377"
      "\271\266\317\377\305\306\341\377\315\302\332\377\343\306\325\377\321"
      "\305\305\377\331\273\314\377\316\272\300\377\320\267\314\377\330\263"
      "\274\377\346\306\323\377\324\307\304\377\334\277\267\377\277\264\301"
      "\377\267\271\272\377\272\247\271\377\265\235\265\377\261\207\250\377"
      "\256m\231\377\246n\241\377\237c\227\377\221b\211\377\223Q{\377\213<v"
      "\377\2238b\377\224.R\333\204\40W\377\0\0\0\0\224\225\333\377\241\240"
      "\325\377\240\245\311\377\247\242\321\377\256\244\310\377\271\266\317"
      "\377\305\306\341\377\315\302\332\377\343\306\325\377\321\305\305\377"
      "\331\273\314\377\316\272\300\377\320\267\314\377\330\263\274\377\346"
      "\306\323\377\324\307\304\377\334\277\267\377\304\307\310\377\275\271"
      "\270\377\257\245\301\377\270\223\256\377\254\206\265\377\263w\243\377"
      "\250c\237\377\222W\226\377\211U\224\377\220Fv\377\221Du\377\2078`\377"
      "\212,Y\377\203%]\333" };

  /* GdkPixbuf RGBA C-Source image dump */
  
#ifdef __SUNPRO_C
#pragma align 4 (alpha_raw)
#endif
#ifdef __GNUC__
  static const uint8 alpha_raw[] __attribute__ ((__aligned__ (4))) = 
#else
    static const uint8 alpha_raw[] = 
#endif
    { ""
      /* Pixbuf magic (0x47646b50) */
      "GdkP"
      /* length: header (24) + pixel_data (4096) */
      "\0\0\20\30"
      /* pixdata_type (0x1010002) */
      "\1\1\0\2"
      /* rowstride (128) */
      "\0\0\0\200"
      /* width (32) */
      "\0\0\0\40"
      /* height (32) */
      "\0\0\0\40"
      /* pixel_data: */
      "\0\0\0\0\304\24\306\354\277\31\310\377\260\31\313\377\240\14\300\301"
      "\234\20\255\307\216\10\252\347\215\1\222\377|\13\220\377u\0\231\377r"
      "\6\213\377n\5\211\377`\13\224\377T\4\210\377@\0\207\377B\0\201\377A\6"
      "\225\377G\13\213\377U\11\201\377L+\202\3779Dp\377&Js\377\40bO\377%nM"
      "\377'|>\377-\206*\377/\2038\377@\221)\377<\227(\3770\236*\3777\271\""
      "\377D\312*\377\0\0\0\0\271\34\271\26\277\"\264\354\271\36\273\377\255"
      "\16\266\356\230\4\262\377\232\0\254\377\221\0\227\377\177\2\216\377w"
      "\4\221\377q\5\204\377j\10\213\377_\13\213\377P\10\207\377S\15z\377O\6"
      "\202\377E\0\204\377M\0\205\377L\17\204\377U!}\377M5q\3773Ql\377'YT\377"
      "/l\77\3775z7\377.x4\377C\2034\377H\2245\377D\226/\377A\254(\377Q\273"
      "2\377W\300.\377\0\0\0\0\0\0\0\0\247\32\264\25\263\24\251\353\370\355"
      "\370\377\377\377\377\377\377\377\377\377\251W\272\377r\2\212\377t\0\211"
      "\377n\0\204\377h\0\216\377X\14\220\377\\\10\216\377S\6\223\377e\20\213"
      "\377Y\13\206\377Q\0x\377V\33~\377Q\36u\377Q*i\377<Kk\377\77ZL\3777l@"
      "\377o\223t\377\377\377\377\377\377\377\377\377l\235[\377[\2512\377Y\267"
      "(\377T\266'\377T\306\40Q\0\0\0\0\0\0\0\0\0\0\0\0\362\331\360]\377\377"
      "\377\377\377\377\377\377\377\377\377\377\323\255\332\377l\2\200\377i"
      "\0u\377j\0z\377h\0\211\377k\3\211\377`\22\216\377`\23\206\377g\23\221"
      "\377^\25\215\377e\10~\377S$q\377R-l\377@7f\377E6\\\377DUU\377PaQ\377"
      "p\227\200\377\377\377\377\377\377\377\377\377s\235Z\377X\2361\377`\257"
      "$\377]\267\34Q\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\377\377\377\251\377\377"
      "\377\377\370\361\371\376\377\377\377\377\373\370\373\377o\17\206\377"
      "i\0w\377[\2v\377m\0w\377_\14\200\377j\15\202\377g\3\213\377o\7}\377o"
      "\10w\377e\7\177\377h-y\377d@\201\377Y-c\377a1_\377X@P\377bZQ\377x\214"
      "x\377\377\377\377\377\377\377\377\377{\225Y\377t\231&\377}\257\31Q\0"
      "\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\377\377\377\16\377\377\377\366\377\377"
      "\377\377\366\354\366\204\376\376\376\377\377\377\377\377\246p\254\365"
      "Z\0m\377\224Y\235\377\377\377\377\377\377\377\377\377\312\241\317\377"
      "\372\367\372\377\262{\273\377v\17\204\377\315\247\315\377\377\377\377"
      "\377\377\377\377\377\314\274\323\377\354\344\352\377\377\377\377\377"
      "\244\235\215\377\220\224v\377\377\377\377\377\377\377\377\377\321\334"
      "\300\377\374\375\373\367\377\377\377\347\377\377\377_\0\0\0\0\0\0\0\0"
      "\0\0\0\0\377\377\377`\377\377\377\377\377\377\377\377\377\377\377\36"
      "\374\372\374\322\377\377\377\377\326\302\330\377\\\0Y\355\233[\235\377"
      "\377\377\377\377\377\377\377\377\364\354\364\377\322\255\327\377\200"
      "+\206\377\243[\257\377\377\377\377\377\377\377\377\377\343\320\341\377"
      "\350\330\342\377\377\377\377\377\377\377\377\377\250\240\225\377\230"
      "\217q\377\377\377\377\377\377\377\377\377\370\370\364\344\377\377\377"
      "\311\377\377\377\377\377\377\377\374\377\377\377!\0\0\0\0\0\0\0\0\377"
      "\377\377\274\377\377\377\377\377\377\377\332\0\0\0\0\377\377\377\207"
      "\377\377\377\377\376\376\376\377w+d\377\242e\224\367\377\377\377\377"
      "\377\377\377\377\253r\253\377y\6y\377z\5}\377\301\221\312\377\377\377"
      "\377\377\377\377\377\377\232.x\377\233B{\377\377\377\377\377\377\377"
      "\377\377\260\240\223\377\244\213z\377\377\377\377\377\377\377\377\377"
      "\377\377\3773\377\377\377\22\377\377\377\377\377\377\377\377\377\377"
      "\377[\0\0\0\0\377\377\377\32\377\377\377\375\377\377\377\377\377\377"
      "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377"
      "\377\257\212\253\362\230d\212\377\377\377\377\377\377\377\377\377\237"
      "N\205\377|\20n\377y\0w\377\306\212\302\377\377\377\377\377\377\377\377"
      "\377\2302t\377\2335{\377\377\377\377\377\377\377\377\377\272\226\226"
      "\377\242\204u\377\377\377\377\377\377\377\377\377\377\377\3773\377\377"
      "\377\24\377\377\377\377\377\377\377\377\377\377\377T\0\0\0\0\377\377"
      "\377s\377\377\377\377\377\377\377\377\377\377\377\244\377\377\377\234"
      "\377\377\377\234\377\377\377\350\377\377\377\377\374\373\374\325\241"
      "b\215\357\377\377\377\377\377\377\377\377\233I\202\377\177\35c\377\204"
      "\21l\377\237H\211\377\377\377\377\377\377\377\377\377\347\317\337\377"
      "\351\326\337\377\377\377\377\377\377\377\377\377\273\223\223\377\306"
      "\257\246}\377\377\377\377\377\377\377\377\377\377\377\326\377\377\377"
      "\311\377\377\377\377\377\377\377\370\377\377\377\27\0\0\0\0\377\377\377"
      "\316\377\377\377\377\377\377\377\336\0\0\0\0\0\0\0\0\0\0\0\0\377\377"
      "\377\217\377\377\377\377\377\377\377\377\366\357\363\213\377\377\377"
      "\377\377\377\377\377\241K~\364y\14f\377|\31a\377\211\4p\377\262o\253"
      "\377\364\352\362\377\366\357\364\377\311\240\263\377\377\377\377\377"
      "\377\377\377\377\334\304\303\222\377\377\377@\377\377\377\377\377\377"
      "\377\370\377\377\377\211\377\377\377\362\377\377\377\345\377\377\377"
      "Z\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0"
      "\0\0\0\0\0\0\0\0\0\0\0\0\0\0\213\17]\22y\16g\347}\24`\377\204\17U\362"
      "~\22X\377\244Z\226\377\354\330\350\377\350\322\342\377\321\256\311\377"
      "\333\310\324\377\377\377\377\377\377\377\377\376\377\377\377-\0\0\0\0"
      "\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0"
      "\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0"
      "\0\0\0\0\0\0\0\0v\37]\21\200\25^\346\177\14U\377~\17U\362\206\23`\377"
      "\315\241\312\377\362\350\360\377\375\374\375\377\374\372\373\377\370"
      "\363\365\342\377\377\377]\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0"
      "\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0"
      "\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0w\26"
      "\\\21\177\7k\346\212\0g\377\220\11k\362\204\17e\377\210\23g\377\202."
      "f\377\210CbQ\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0"
      "\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0"
      "\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0"
      "\0\0\0\0\204\5]\20\207\1^\352\206\6U\377\213\2W\377\212\34Z\377\2121"
      "eQ\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0"
      "\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0"
      "\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0"
      "\0\0\0\0\0\0\0\213\0M\241\205\7[\377\225\11`\377\233\33^\365\223+[\36"
      "\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0"
      "\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0"
      "\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0"
      "\0\0\222\16h\234\205\14[\377\204\22a\377\210\15f\351\217\22[\377\224"
      "%W\363\236&n\36\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0"
      "\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0"
      "\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0"
      "\0\236\21r\234\226\16y\377\224\16u\377\213\27f\377\204\25a\377\217\23"
      "a\343\246\33c\377\255+o\364\253Dy\37\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0"
      "\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0"
      "\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0"
      "\0\0\0\0\2532w\234\2551\203\377\230\33\202\377\227\"u\377\227\32|\377"
      "\205\23n\377\214*y\377\230\37o\343\2569{\377\2659\200\364\263Hj\37\0"
      "\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0"
      "\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0"
      "\0\0\0\0\0\0\0\0\0\0\0\276>s\234\2564\203\377\262\77\202\377\2360v\377"
      "\2408\201\377\237)w\377\222'\201\377\230#\200\377\236,\217\377\2364\205"
      "\342\2533\207\377\244Aq\364\263G_\40\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0"
      "\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\377\377\377\377\377"
      "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\375\377\377"
      "\377\344\377\377\377q\0\0\0\0\271Uv\234\273P\205\377\255Q\206\377\261"
      "E\213\377\237L\210\377\244D\207\377\220/\220\377\241'\233\377\230:\217"
      "\377\2406\221\377\242\77\215\377\247C\216\341\260K}\377\272M`\365\302"
      "Tb\40\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0"
      "\0\0\0\0\377\377\377\377\377\377\377\377\377\377\377\303\377\377\377"
      "\235\377\377\377\321\377\377\377\377\377\377\377\377\324\217\215\257"
      "\260Yp\377\256d\206\377\260S\202\377\263W\201\377\246U\207\377\231F\211"
      "\377\235H\216\377\237:\250\377\240E\233\377\257G\222\377\264R\211\377"
      "\253T\213\377\266[w\341\303Yv\377\270c_\365\277YN\34\0\0\0\0\0\0\0\0"
      "\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\377\377\377\377\377"
      "\377\377\377\377\377\377d\0\0\0\0\377\377\377\21\377\377\377\377\377"
      "\377\377\377\336\251\251\377\300xz\377\271s\206\377\272b\206\377\272"
      "_\223\377\251V\210\377\244V\241\377\231H\255\377\246>\251\377\262F\252"
      "\377\262G\235\377\254K\232\377\272`\206\377\271S\202\377\305]{\341\305"
      "cZ\377\261KW\365\247WQ!\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0"
      "\0\0\0\0\0\377\377\377\377\377\377\377\377\377\377\377d\377\377\377\3"
      "\377\377\377T\377\377\377\377\377\377\377\377\331\250\235\377\300\217"
      "\206\377\324\254\257\377\362\351\356\377\375\373\375\377\375\374\375"
      "\377\366\356\366\377\333\260\340\377\252T\277\377\356\333\357\377\377"
      "\377\377\377\332\241\310\377\277d\217\377\343\276\315\377\377\377\377"
      "\377\372\360\362\374\271QW\377\254dg\367\377\377\377\376\377\377\377"
      "\377\377\377\377'\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\377\377\377\377\377"
      "\377\377\377\377\377\377\377\377\377\377\377\377\377\377\377\375\374"
      "\374\377\335\316\314\377\311\243\215\377\314\235\235\377\374\372\372"
      "\377\377\377\377\377\355\334\351\377\352\321\343\377\377\377\377\377"
      "\377\377\377\377\276\202\315\377\315\244\321\377\377\377\377\377\350"
      "\321\343\377\274`\201\377\367\360\362\377\377\377\377\377\377\377\377"
      "\377\322\202\205\350\307\213\224\377\377\377\377\377\373\370\370\315"
      "\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\377\377\377\377\377\377\377"
      "\377\377\377\377\305\372\370\371\367\377\377\377\377\372\371\370\377"
      "\315\274\272\377\311\254\233\377\314\230\242\377\320\222\226\377\321"
      "\224\253\377\315\240\310\377\333\303\331\377\377\377\377\377\377\377"
      "\377\377\306\247\341\377\256x\311\377\374\371\374\377\374\370\373\377"
      "\317\226\243\377\377\377\377\377\364\351\357\377\377\377\377\377\344"
      "\301\302\377\337\275\304\362\377\377\377\377\265\217\230\372\203\\>#"
      "\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\377\377\377\377\377\377\377\377\327"
      "\305\315\303\301\253\255\377\377\377\377\377\377\377\377\377\367\366"
      "\365\377\275\260\247\377\276\256\255\377\362\350\354\377\377\377\377"
      "\377\331\310\325\377\310\255\304\377\377\377\377\377\377\377\377\377"
      "\320\275\344\377\243\207\301\377\342\324\343\377\377\377\377\377\363"
      "\345\350\377\377\377\377\377\313\225\261\377\370\363\364\377\371\364"
      "\365\377\365\350\354\377\373\370\371\376\233Of\377\220KI\366\204A;#\0"
      "\0\0\0\0\0\0\0\0\0\0\0\377\377\377\377\377\377\377\377\320\305\310\377"
      "\261\230\251\377\344\342\344\377\377\377\377\377\377\377\377\377\333"
      "\330\324\377\322\312\312\377\377\377\377\377\377\377\377\377\354\344"
      "\351\377\366\360\362\377\377\377\377\377\377\377\377\377\320\312\334"
      "\377\266\247\267\377\300\263\303\377\377\377\377\377\377\377\377\377"
      "\372\366\367\377\261p\203\377\332\300\311\377\377\377\377\377\377\377"
      "\377\377\340\272\312\377\227M_\335\230HQ\377\230:Q\367\221>K$\0\0\0\0"
      "\0\0\0\0\377\377\377\377\377\377\377\377\312\270\313\377\271\223\257"
      "\377\305\273\305\377\376\376\376\377\377\377\377\377\366\363\363\377"
      "\325\306\272\377\366\362\366\377\377\377\377\377\376\376\376\377\354"
      "\344\353\377\362\350\354\377\377\377\377\377\344\324\342\377\301\246"
      "\272\377\263\251\272\377\363\362\363\377\377\377\377\377\344\313\332"
      "\377\264p\223\377\277\217\237\377\377\377\377\377\377\377\377\377\260"
      "t\222\377\243Ap\377\237HR\334\2211U\377\222&L\367\0\0\0\0\241\220\312"
      "\234\247\211\304\377\244\227\274\377\243\216\276\377\257\230\304\377"
      "\275\245\310\377\314\277\326\377\324\270\325\377\333\270\306\377\333"
      "\263\307\377\332\276\307\377\321\265\315\377\316\256\312\377\322\265"
      "\323\377\327\276\315\377\316\265\271\377\325\302\273\377\277\264\301"
      "\377\267\271\272\377\272\247\271\377\265\235\265\377\261\207\250\377"
      "\256m\231\377\246n\241\377\237c\227\377\221b\211\377\223Q{\377\213<v"
      "\377\2238b\377\224.R\333\204\40W\377\0\0\0\0\224\225\333\377\241\240"
      "\325\377\240\245\311\377\247\242\321\377\256\244\310\377\271\266\317"
      "\377\305\306\341\377\315\302\332\377\343\306\325\377\321\305\305\377"
      "\331\273\314\377\316\272\300\377\320\267\314\377\330\263\274\377\346"
      "\306\323\377\324\307\304\377\334\277\267\377\277\264\301\377\267\271"
      "\272\377\272\247\271\377\265\235\265\377\261\207\250\377\256m\231\377"
      "\246n\241\377\237c\227\377\221b\211\377\223Q{\377\213<v\377\2238b\377"
      "\224.R\333\204\40W\377\0\0\0\0\224\225\333\377\241\240\325\377\240\245"
      "\311\377\247\242\321\377\256\244\310\377\271\266\317\377\305\306\341"
      "\377\315\302\332\377\343\306\325\377\321\305\305\377\331\273\314\377"
      "\316\272\300\377\320\267\314\377\330\263\274\377\346\306\323\377\324"
      "\307\304\377\334\277\267\377\304\307\310\377\275\271\270\377\257\245"
      "\301\377\270\223\256\377\254\206\265\377\263w\243\377\250c\237\377\222"
      "W\226\377\211U\224\377\220Fv\377\221Du\377\2078`\377\212,Y\377\203%]"
      "\333" };
  
  /* GdkPixbuf RGB C-Source image dump 1-byte-run-length-encoded */
#ifdef __SUNPRO_C
#pragma align 4 (rgb_rle)
#endif
#ifdef __GNUC__
  static const uint8 rgb_rle[] __attribute__ ((__aligned__ (4))) = 
#else
    static const uint8 rgb_rle[] = 
#endif
    { ""
      /* Pixbuf magic (0x47646b50) */
      "GdkP"
      /* length: header (24) + pixel_data (2905) */
      "\0\0\13q"
      /* pixdata_type (0x2010001) */
      "\2\1\0\1"
      /* rowstride (96) */
      "\0\0\0`"
      /* width (32) */
      "\0\0\0\40"
      /* height (32) */
      "\0\0\0\40"
      /* pixel_data: */
      "C\217\36S\210\24L\216+M\2062N}>2\221@0\201C&{R/\210[0\224Y\36\234t9\265"
      "qD\300\226<\301\246;\320\2633\335\267=\302\305G\273\303C\273\273:\305"
      "\3039\264\2615\257\306;\263\3036\243\303D\221\262D\212\255(~\306\40m"
      "\277\33i\316\35U\315!Q\326&\77\335\"\203%]\204\40W\222&L\221>K\214>6"
      "\214M.}S+wI0\210W7\235j,\245q<\263zQ\263\220;\263\234;\304\255:\315\265"
      "K\270\304=\260\277;\262\2673\301\2715\264\275:\271\2727\271\264E\253"
      "\262K\241\255:\203\2727{\2610b\272\26T\306\40W\300.D\312*.\321)\212,"
      "Y\224.R\2211U\204\377\377\377\17\375\375\375\362\356\351\275\247\215"
      "\232wE\247xK\242\222I\302\271\202\354\353\324\374\373\366\376\376\375"
      "\370\371\361\344\346\300\275\275R\266\2638\252\256;\205\377\377\377\10"
      "\361\370\353\235\325\204Q\27327\271\"-\260/\2078`\2238b\237HR\202\377"
      "\377\377\3\341\330\321\323\302\262\351\341\332\202\377\377\377\7\244"
      "\211l\237\177M\275\263\211\377\377\377\376\376\376\351\340\313\343\344"
      "\312\202\377\377\377\3\354\355\330\226\234B\237\2242\202\377\377\377"
      "\13\346\345\320\320\333\252\352\362\332\377\377\377\376\376\376^\270"
      "H0\236**\245/\221Du\213<v\243Ap\202\377\377\377\3\264\215\227\231UM\240"
      "aK\202\377\377\377\14\306\244\213\246qK\367\363\357\377\377\377\322\307"
      "\246\246}A\262\215L\332\313\255\341\336\304\325\322\242\260\226D\245"
      "\2038\202\377\377\377\3\266\274\203t\232%\211\251H\202\377\377\377\6"
      "r\257b<\227($\230%\220Fv\223Q{\226Fn\202\377\377\377\3\302\215\235\224"
      "[\\\271\223\210\202\377\377\377\2\264\200r\264\213m\202\377\377\377\10"
      "\302\201l\253oJ\250xI\270tY\250\207J\256\223Q\242\223K\250\200=\202\377"
      "\377\377\13\263\266\211p\221;\232\256\200\377\377\377\336\351\332K\226"
      "8@\221)4\223)\211U\224\221b\211\236Tx\205\377\377\377\4\375\373\373\332"
      "\261\250\261WH\304\217~\202\377\377\377\3\264rW\272lI\363\344\342\203"
      "\377\377\377\2\306\250\200\246vR\205\377\377\377\10\376\376\376\236\300"
      "\232C\2034/\20386\220!\222W\226\237c\227\233X~\202\377\377\377\7\357"
      "\332\333\371\360\360\377\377\377\372\364\365\321\210\200\271TW\307\177"
      "z\202\377\377\377\4\274lm\272V[\337\260\253\346\303\276\202\377\377\377"
      "\2\275\221\207\220hW\202\377\377\377\3\336\334\324\301\310\265\332\341"
      "\326\202\377\377\377\6Y\224^-\206*%\217(\250c\237\246n\241\254m\202\202"
      "\377\377\377\2\312\240\254\316\206\220\202\377\377\377\11\370\357\357"
      "\303Xf\262M_\371\362\363\377\377\377\325\251\247\242UZ\250MT\275qp\202"
      "\377\377\377\2\261\225\206\213\\J\202\377\377\377\3\253\244\214kqIVo"
      "V\202\377\377\377\6}\251\177'|>'x7\263w\243\256m\231\264p\223\202\377"
      "\377\377\3\313\227\266\266Wu\355\316\323\202\377\377\377\14\325\222\236"
      "\263G_\315\221\240\377\377\377\376\376\376\344\320\320\333\303\313\372"
      "\367\370\377\377\377\374\373\373\264\177\177\222SR\202\377\377\377\13"
      "\336\333\327\306\302\270\335\334\332\377\377\377\375\375\375P\203]%n"
      "M\36uI\254\206\265\261\207\250\271u\237\202\377\377\377\21\320\236\263"
      "\260i~\307\177\233\376\374\375\377\377\377\362\342\352\244Aq\263Hj\311"
      "\202\240\354\331\334\373\370\371\375\373\374\361\350\353\331\271\276"
      "\225fl\220Ja\230Hc\204\377\377\377\177\373\373\373\341\344\344\211\232"
      "\221'YT\40bO\40bQ\270\223\256\265\235\265\270\211\241\271\204\212\302"
      "o\203\275o\201\274_\201\277d\217\272`\206\253T\213\247C\216\2533\207"
      "\2659\200\253Dy\225<h\222<^\221GZ\226H]\225L_\203\77^\213HW\217Lf\225"
      "Fl\214In\2106]p2_a1_E6\\<Kk3Ql&Js/Ti\257\245\301\272\247\271\255\245"
      "\255\274\213\242\256|\223\272d\237\254Z\232\273S\233\254K\232\264R\211"
      "\242\77\215\2364\205\2569{\255+o\236&n\2205R\2153\\\210Cb\2000bx0]\213"
      "0[\216.\\\224&r\2221o\201+b](sY-c@7fQ*iM5q9Dp1@v\275\271\270\267\271"
      "\272\263\251\272\254\232\260\257\211\262\263o\262\261^\241\251P\242\262"
      "G\235\257G\222\2406\221\236,\217\230\37o\246\33c\224%W\223+[\2121e\202"
      ".f\205*[}\27g~\23\\\213\26e\221#j\221\34lx\34oa*yV/vR-lQ\36uU!}L+\202"
      "E%v\304\307\310\277\264\301\301\246\272\266\247\267\243\207\301\251p"
      "\306\254f\262\256P\263\262F\252\240E\233\230:\217\230#\200\214*y\217"
      "\23a\217\22[\233\33^\212\34Z\210\23gv\13a\210\23f~\4e\203\14c\215\31"
      "y~\20\201n\23\202\177h\37{]\36oS$qV\33~L\17\204U\11\201;\30\220\334\277"
      "\267\325\302\273\312\253\307\263\251\307\263\223\323\242p\316\251X\274"
      "\252T\277\246>\251\237:\250\241'\233\222'\201\205\23n\204\25a\210\15"
      "f\225\11`\213\2W\204\17ez\0p\203\0m|\12p\211\13m{\16{z\0\207\200\0vu"
      "\12ue\7\177e\10~Q\0xM\0\205G\13\213H\15\226\324\307\304\316\265\271\324"
      "\276\277\307\253\305\274\233\266\243|\271\242f\301\261S\275\231H\255"
      "\235H\216\220/\220\237)w\227\32|\213\27f\204\22a\205\7[\206\6U\220\11"
      "k\201\11Y{\16f\211\4p\204\23g\210\14\200v\11\211r\3\204u\15\203o\10w"
      "^\25\215Y\13\206E\0\204A\6\2259\3\214\346\306\323\327\276\315\325\263"
      "\303\312\261\266\303\242\260\262\206\254\260o\251\250\\\253\244V\241"
      "\231F\211\244D\207\2408\201\227\"u\224\16u\205\14[\213\0M\207\1^\212"
      "\0g~\17U~\22X|\31a\204\21ly\0wz\5}l\12sn\5\177o\7}g\23\221e\20\213O\6"
      "\202B\0\201=\0\227\330\263\274\322\265\323\314\265\311\320\253\271\266"
      "\222\261\262\200\256\273g\244\260]\227\251V\210\246U\207\237L\210\236"
      "0v\230\33\202\226\16y\222\16h\215\0W\204\5]\177\7k\177\14U\204\17Uy\14"
      "f\177\35c|\20ny\6y+x\11\206n\10\177g\3\213`\23\206S\6\223S\15z@\0\207"
      "=\11\235\320\267\314\313\251\307\277\237\311\305\252\272\265\224\254"
      "\272z\262\272w\253\261`\231\272_\223\263W\201\261E\213\262\77\202\255"
      "1\203\236\21r\220\10i\211\0W\200\1bw\26\\\200\25^}\24`\205\26X\200\27"
      "_\202\27_u\30ut\21zx\15\205j\15\202`\22\216\\\10\216P\10\207T\4\210P"
      "\0\236\316\272\300\314\254\307\312\245\300\204\377\377\377\5\376\375"
      "\376\367\356\362\323\237\271\255Q\206\2564\203\202\377\377\377\6\275"
      "n\237\211\15Z\210\15N\204\"Rv\37]y\16g\206\377\377\377\11\346\331\353"
      "k\3\211X\14\220_\13\213`\13\224T\11\243\331\273\314\332\276\307\312\260"
      "\307\202\377\377\377\3\363\345\346\354\325\322\363\350\351\202\377\377"
      "\377\2\307p\233\276>s\202\377\377\377\6\275t\220\204\15K\202\16X\177"
      "\34Z\206\20Z\213\17]\202\377\377\377\15\333\311\327\306\251\305\303\241"
      "\311\301\235\311\270\203\274h\0\211h\0\216j\10\213n\5\211o\0\232\321"
      "\305\305\333\263\307\321\300\263\202\377\377\377\3\340\300\306\311\226"
      "\226\304\226\216\202\377\377\377\2\322\222\247\304D^\202\377\377\377"
      "\6\275y\230\206\10Q\211\32]\212\34a}\10U\200\14S\202\377\377\377\15\242"
      "r\225s\27]k\13m`\10n[\2vj\0zn\0\204q\5\204r\6\213x\0\227\343\306\325"
      "\333\270\306\311\264\270\202\377\377\377\3\336\314\302\311\244\216\337"
      "\265\255\202\377\377\377\2\322w\202\304=Z\202\377\377\377\6\305x\222"
      "\227\20P\233\22K\217\12Y\211\17]\211\24]\202\377\377\377\15\332\311\330"
      "\303\242\273\277\234\276\276\234\306\231R\242i\0ut\0\211w\4\221u\0\231"
      "\201\0\226\315\302\332\324\270\325\307\270\311\205\377\377\377\4\375"
      "\374\374\336\266\264\277Zb\263EL\202\377\377\377\6\320q\223\220\17M\235"
      "\26I\220\36K\217\10U\210\27_\206\377\377\377\11\267\210\303l\2\200r\2"
      "\212\177\2\216|\13\220\224\7\241\305\306\341\314\277\326\303\302\306"
      "\202\377\377\377\7\355\350\346\370\365\365\377\377\377\372\366\366\320"
      "\216\216\273O`\265@P\202\377\377\377\6\312\177\211\236\26;\225\23C\230"
      "\37P\211\13X\212\10U\202\377\377\377\15\252p\246g&yh\23xo\14|v\0\200"
      "w\0\214\202\11\233\221\0\227\215\1\222\223\24\233\271\266\317\275\245"
      "\310\272\257\272\202\377\377\377\2\305\266\266\310\237\235\202\377\377"
      "\377\3\371\357\360\303IU\261E@\202\377\377\377\6\321x\206\240\22""4\244"
      "\36""7\244\30@\220\12O\217\3X\202\377\377\377\15\254v\257g'\213n!\214"
      "\205\37\212~\15\230\207\0\224\211\4\234\232\0\254\216\10\252\216\30\253"
      "\256\244\310\257\230\304\271\223\257\202\377\377\377\3\307\263\274\263"
      "wv\347\323\317\202\377\377\377\2\330\211\230\3029E\202\377\377\377\6"
      "\353\315\317\334\247\251\336\244\257\331\244\255\267Wu\216\21K\202\377"
      "\377\377\15\341\311\340\307\252\321\313\253\327\325\250\327\324\242\327"
      "\244\11\230\242\2\244\230\4\262\234\20\255\236\16\265\247\242\321\243"
      "\216\276\250\213\253\202\377\377\377\7\307\254\274\251q\200\302\204\210"
      "\376\375\375\377\377\377\363\343\342\2532;\206\377\377\377\2\322\220"
      "\252\230\32L\206\377\377\377f\374\370\374\243\27\232\254\24\246\255\16"
      "\266\240\14\300\241\14\275\240\245\311\244\227\274\260\227\274\252\225"
      "\261\237\211\231\247\200\214\251nu\266qs\256ia\240PV\234CA\245>=\236"
      "38\2407.\2422\35\235$\23\244\27+\254\25>\236&>\245$T\243\36Y\224$u\205"
      "(\215u*\226\210\36\225\234\27\250\256\20\240\261\24\244\262\17\250\271"
      "\36\273\260\31\313\257\16\314\241\240\325\247\211\304\243\207\265\256"
      "\207\250\270}\222\255v|\255r\200\254]s\253Oh\250ET\245FL\243@P\252;5"
      "\240)2\251*#\247#\32\233$%\262\40""3\250\34K\237/O\245&e\234\"j\217\33"
      "\177\211.\237\232\"\234\241\36\253\255\27\244\257\35\261\247\32\264\277"
      "\"\264\277\31\310\306\32\333\224\225\333\241\220\312\242\211\252\264"
      "~\237\276i\221\267^u\272Xw\257Tj\245Rk\256Nh\227CS\2243P\24718\247.("
      "\250(\"\243'\17\2342\37\25233\2564>\243'R\245.X\246,k\235\35\211\230"
      "!\235\232%\247\246\40\235\240$\235\245\40\256\250\32\272\271\34\271\304"
      "\24\306\313\31\335"};
  
  /* GdkPixbuf RGB C-Source image dump */
#ifdef __SUNPRO_C
#pragma align 4 (rgb_raw)
#endif
#ifdef __GNUC__
  static const uint8 rgb_raw[] __attribute__ ((__aligned__ (4))) = 
#else
    static const uint8 rgb_raw[] = 
#endif
    { ""
      /* Pixbuf magic (0x47646b50) */
      "GdkP"
      /* length: header (24) + pixel_data (3072) */
      "\0\0\14\30"
      /* pixdata_type (0x1010001) */
      "\1\1\0\1"
      /* rowstride (96) */
      "\0\0\0`"
      /* width (32) */
      "\0\0\0\40"
      /* height (32) */
      "\0\0\0\40"
      /* pixel_data: */
      "\217\36S\210\24L\216+M\2062N}>2\221@0\201C&{R/\210[0\224Y\36\234t9\265"
      "qD\300\226<\301\246;\320\2633\335\267=\302\305G\273\303C\273\273:\305"
      "\3039\264\2615\257\306;\263\3036\243\303D\221\262D\212\255(~\306\40m"
      "\277\33i\316\35U\315!Q\326&\77\335\"\203%]\204\40W\222&L\221>K\214>6"
      "\214M.}S+wI0\210W7\235j,\245q<\263zQ\263\220;\263\234;\304\255:\315\265"
      "K\270\304=\260\277;\262\2673\301\2715\264\275:\271\2727\271\264E\253"
      "\262K\241\255:\203\2727{\2610b\272\26T\306\40W\300.D\312*.\321)\212,"
      "Y\224.R\2211U\230:Q\204A;\207Y3\200O1uU/\206_6\212a4\232wE\247xK\242"
      "\222I\250\232K\253\245\77\312\254C\273\254J\245\262F\260\270H\267\267"
      "A\266\2638\252\256;\255\2304\242\235H\214\246<\177\246)w\265)]\267\34"
      "T\266'Q\27327\271\"-\260/\2078`\2238b\237HR\377\377\377\377\377\377\377"
      "\377\377\377\377\377\376\375\375\362\354\352\273\246\234\220nK\237\177"
      "M\231\211G\271\263\201\351\347\327\373\372\367\376\376\375\367\370\361"
      "\340\342\277\252\262T\226\234B\237\2242\377\377\377\377\377\377\377\377"
      "\377\377\377\377\377\377\377\361\370\354\240\326\204A\254(0\236**\245"
      "/\221Du\213<v\243Ap\377\377\377\377\377\377\342\323\326\327\275\272\354"
      "\340\334\377\377\377\377\377\377\267\213l\245oH\304\251\212\377\377\377"
      "\376\376\376\345\332\311\351\336\314\377\377\377\377\377\377\356\354"
      "\331\254\221;\245\2038\377\377\377\377\377\377\343\345\317\311\330\253"
      "\350\356\334\377\377\377\376\376\376`\246N<\227($\230%\220Fv\223Q{\226"
      "Fn\377\377\377\377\377\377\302\215\235\223Z[\236hZ\377\377\377\377\377"
      "\377\303\231\215\241mH\367\363\360\377\377\377\333\266\252\253oJ\250"
      "xI\334\273\256\343\330\305\326\311\250\246\230T\250\200=\377\377\377"
      "\377\377\377\263\266\211p\221;~\227\\\377\377\377\377\377\377u\256f@"
      "\221)4\223)\211U\224\221b\211\236Tx\377\377\377\377\377\377\312\222\234"
      "\254S`\303\217\221\377\377\377\377\377\377\300ym\275\202o\377\377\377"
      "\377\377\377\272}d\272lI\275aU\262qT\272oF\263qP\252~C\246vR\377\377"
      "\377\377\377\377\256\262\207_|C\232\247\203\377\377\377\334\350\333F"
      "\2057/\20386\220!\222W\226\237c\227\233X~\377\377\377\377\377\377\377"
      "\377\377\377\377\377\377\377\377\375\373\373\340\260\253\271TW\314\214"
      "\207\377\377\377\377\377\377\265_`\272V[\364\344\342\377\377\377\377"
      "\377\377\377\377\377\275\221\207\220hW\377\377\377\377\377\377\377\377"
      "\377\377\377\377\377\377\377\376\376\376\221\264\226.x4-\206*%\217(\250"
      "c\237\246n\241\254m\202\377\377\377\377\377\377\353\333\340\371\362\363"
      "\377\377\377\373\365\365\314\217\214\302Tb\301p\177\377\377\377\377\377"
      "\377\270li\242UZ\324\247\252\342\302\301\377\377\377\377\377\377\261"
      "\225\206\213\\J\377\377\377\377\377\377\336\333\322\306\310\271\331\336"
      "\331\377\377\377\377\377\377^\225`'|>'x7\263w\243\256m\231\264p\223\377"
      "\377\377\377\377\377\313\227\266\307~\225\377\377\377\377\377\377\371"
      "\356\361\273Qd\265Lc\371\362\363\377\377\377\322\244\254\240WW\225Md"
      "\257f{\377\377\377\377\377\377\275\216\215\222SR\377\377\377\377\377"
      "\377\253\243\230ncIkc[\377\377\377\377\377\377z\241\204%nM\36uI\254\206"
      "\265\261\207\250\271u\237\377\377\377\377\377\377\320\236\263\260i~\352"
      "\317\332\377\377\377\377\377\377\316\221\257\244Aq\321\221\246\377\377"
      "\377\376\376\376\337\315\323\336\304\314\372\366\367\377\377\377\374"
      "\373\373\252v\207\230Hc\377\377\377\377\377\377\340\326\327\312\273\271"
      "\333\326\331\377\377\377\375\375\375Iso\40bO\40bQ\270\223\256\265\235"
      "\265\270\211\241\377\377\377\377\377\377\326\247\262\274_\201\311|\241"
      "\376\375\375\377\377\377\361\341\355\2533\207\2659\200\307\204\247\350"
      "\325\337\373\370\371\374\373\373\362\350\353\325\271\300\226]w\213HW"
      "\217Lf\377\377\377\377\377\377\377\377\377\377\377\377\373\372\373\341"
      "\337\345\207\220\2443Ql&Js/Ti\257\245\301\272\247\271\255\245\255\274"
      "\213\242\256|\223\272d\237\254Z\232\273S\233\254K\232\264R\211\242\77"
      "\215\2364\205\2569{\255+o\236&n\2205R\2153\\\210Cb\2000bx0]\2130[\216"
      ".\\\224&r\2221o\201+b](sY-c@7fQ*iM5q9Dp1@v\275\271\270\267\271\272\263"
      "\251\272\254\232\260\257\211\262\263o\262\261^\241\251P\242\262G\235"
      "\257G\222\2406\221\236,\217\230\37o\246\33c\224%W\223+[\2121e\202.f\205"
      "*[}\27g~\23\\\213\26e\221#j\221\34lx\34oa*yV/vR-lQ\36uU!}L+\202E%v\304"
      "\307\310\277\264\301\301\246\272\266\247\267\243\207\301\251p\306\254"
      "f\262\256P\263\262F\252\240E\233\230:\217\230#\200\214*y\217\23a\217"
      "\22[\233\33^\212\34Z\210\23gv\13a\210\23f~\4e\203\14c\215\31y~\20\201"
      "n\23\202h\37{]\36oS$qV\33~L\17\204U\11\201;\30\220\334\277\267\325\302"
      "\273\312\253\307\263\251\307\263\223\323\242p\316\251X\274\252T\277\246"
      ">\251\237:\250\241'\233\222'\201\205\23n\204\25a\210\15f\225\11`\213"
      "\2W\204\17ez\0p\203\0m|\12p\211\13m{\16{z\0\207\200\0vu\12ue\7\177e\10"
      "~Q\0xM\0\205G\13\213H\15\226\324\307\304\316\265\271\324\276\277\307"
      "\253\305\274\233\266\243|\271\242f\301\261S\275\231H\255\235H\216\220"
      "/\220\237)w\227\32|\213\27f\204\22a\205\7[\206\6U\220\11k\201\11Y{\16"
      "f\211\4p\204\23g\210\14\200v\11\211r\3\204u\15\203o\10w^\25\215Y\13\206"
      "E\0\204A\6\2259\3\214\346\306\323\327\276\315\325\263\303\312\261\266"
      "\303\242\260\262\206\254\260o\251\250\\\253\244V\241\231F\211\244D\207"
      "\2408\201\227\"u\224\16u\205\14[\213\0M\207\1^\212\0g~\17U~\22X|\31a"
      "\204\21ly\0wz\5}l\12sn\5\177o\7}g\23\221e\20\213O\6\202B\0\201=\0\227"
      "\330\263\274\322\265\323\314\265\311\320\253\271\266\222\261\262\200"
      "\256\273g\244\260]\227\251V\210\246U\207\237L\210\2360v\230\33\202\226"
      "\16y\222\16h\215\0W\204\5]\177\7k\177\14U\204\17Uy\14f\177\35c|\20ny"
      "\6yx\11\206n\10\177g\3\213`\23\206S\6\223S\15z@\0\207=\11\235\320\267"
      "\314\313\251\307\277\237\311\305\252\272\265\224\254\272z\262\272w\253"
      "\261`\231\272_\223\263W\201\261E\213\262\77\202\2551\203\236\21r\220"
      "\10i\211\0W\200\1bw\26\\\200\25^}\24`\205\26X\200\27_\202\27_u\30ut\21"
      "zx\15\205j\15\202`\22\216\\\10\216P\10\207T\4\210P\0\236\316\272\300"
      "\314\254\307\312\245\300\302\246\255\304\231\261\307}\231\271t\243\261"
      "x\231\272b\206\260S\202\255Q\206\2564\203\2532w\243\31Y\223\22b\211\15"
      "Z\210\15N\204\"Rv\37]y\16gq\10`u\12^q\26^j\25lr\24yh\7x_\14\200k\3\211"
      "X\14\220_\13\213`\13\224T\11\243\331\273\314\332\276\307\312\260\307"
      "\377\377\377\377\377\377\377\377\377\377\377\377\376\376\376\367\360"
      "\362\321\250\273\273P\205\276>s\246,k\233)U\224\33J\204\15K\202\16X\177"
      "\34Z\206\20Z\213\17]\203\27Wj\21Sh\35Ym\"kf\15t`\5um\0wh\0\211h\0\216"
      "j\10\213n\5\211o\0\232\321\305\305\333\263\307\321\300\263\377\377\377"
      "\377\377\377\363\346\351\352\326\326\363\352\351\377\377\377\377\377"
      "\377\306u\217\304D^\2623]\237(V\224#W\206\10Q\211\32]\212\34a}\10U\200"
      "\14Sw\13So\15Pf\30Rs\27]k\13m`\10n[\2vj\0zn\0\204q\5\204r\6\213x\0\227"
      "\343\306\325\333\270\306\311\264\270\377\377\377\377\377\377\336\314"
      "\302\311\243\215\323\231\215\377\377\377\377\377\377\333\222\232\304"
      "=Z\2717Q\263-W\241\"M\227\20P\233\22K\217\12Y\211\17]\211\24]~\37Zj\27"
      "Sc\34\\g\21R\\\0YZ\0mi\0wi\0ut\0\211w\4\221u\0\231\201\0\226\315\302"
      "\332\324\270\325\307\270\311\377\377\377\377\377\377\324\306\304\271"
      "\232\225\322\256\254\377\377\377\377\377\377\314{\202\263EL\325}\213"
      "\361\337\341\375\372\373\375\373\374\365\347\354\314\230\255\217\10U"
      "\346\317\336\377\377\377\263z\243n\21S\300\240\276\377\377\377\361\351"
      "\362g\3\200v\24\211\376\376\376\377\377\377\2200\240\224\7\241\305\306"
      "\341\314\277\326\303\302\306\377\377\377\377\377\377\377\377\377\377"
      "\377\377\377\377\377\375\374\374\340\264\264\273O`\271L[\374\366\367"
      "\377\377\377\351\312\316\341\270\303\377\377\377\377\377\377\247I\202"
      "\271l\232\377\377\377\327\273\322u\26o\360\351\361\377\377\377\377\377"
      "\377\232D\241\244V\262\377\377\377\345\305\347\215\1\222\223\24\233\271"
      "\266\317\275\245\310\272\257\272\377\377\377\377\377\377\351\343\343"
      "\371\364\364\377\377\377\372\366\366\331\212\224\302DQ\261E@\267<A\305"
      "`V\311ap\322\220\240\377\377\377\377\377\377\273i\223\225\21a\372\366"
      "\371\371\365\371\234[\237\377\377\377\354\343\360\377\377\377\315\241"
      "\327\317\232\324\377\377\377\301d\314\216\10\252\216\30\253\256\244\310"
      "\257\230\304\271\223\257\377\377\377\377\377\377\307\263\274\304\227"
      "\226\377\377\377\377\377\377\371\355\357\300BZ\306FQ\357\322\322\377"
      "\377\377\324\224\230\273VZ\377\377\377\377\377\377\277h\204\216\21K\326"
      "\246\310\377\377\377\350\325\347\377\377\377\244j\270\367\356\367\370"
      "\360\370\363\341\362\373\365\373\235\20\265\234\20\255\236\16\265\247"
      "\242\321\243\216\276\250\213\253\377\377\377\377\377\377\307\254\274"
      "\251q\200\350\321\323\377\377\377\377\377\377\321\224\217\310y\177\377"
      "\377\377\377\377\377\337\276\270\356\332\326\377\377\377\377\377\377"
      "\305n\220\230\32L\250M\207\377\377\377\377\377\377\366\361\370\206\36"
      "\220\327\243\325\377\377\377\377\377\377\340\250\336\255\16\266\240\14"
      "\300\241\14\275\240\245\311\244\227\274\260\227\274\377\377\377\377\377"
      "\377\311\261\271\251nu\301\207\211\376\375\375\377\377\377\357\341\341"
      "\255PP\360\337\340\377\377\377\376\375\375\333\257\250\343\270\276\377"
      "\377\377\316\222\236\245$T\243\36Y\360\340\354\377\377\377\313\256\327"
      "\210\36\225\263N\274\377\377\377\377\377\377\305K\275\271\36\273\260"
      "\31\313\257\16\314\241\240\325\247\211\304\243\207\265\256\207\250\270"
      "}\222\255v|\255r\200\254]s\253Oh\250ET\245FL\243@P\252;5\252@H\2566/"
      "\247#\32\233$%\262\40""3\250\34K\237/O\245&e\234\"j\217\33\177\211.\237"
      "\232\"\234\241\36\253\255\27\244\257\35\261\247\32\264\277\"\264\277"
      "\31\310\306\32\333\224\225\333\241\220\312\242\211\252\264~\237\276i"
      "\221\267^u\272Xw\257Tj\245Rk\256Nh\227CS\2243P\24718\247.(\250(\"\243"
      "'\17\2342\37\25233\2564>\243'R\245.X\246,k\235\35\211\230!\235\232%\247"
      "\246\40\235\240$\235\245\40\256\250\32\272\271\34\271\304\24\306\313"
      "\31\335" };

  Image::register_builtin_pixstream ("testimage-alpha-rle", alpha_rle);
  Image::register_builtin_pixstream ("testimage-alpha-raw", alpha_raw);
  Image::register_builtin_pixstream ("testimage-rgb-rle", rgb_rle);
  Image::register_builtin_pixstream ("testimage-rgb-raw", rgb_raw);
}

} // anon
