/* The GIMP -- an image manipulation program
 * Copyright (C) 1995 Spencer Kimball and Peter Mattis
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 *
 * This is based loosely on 'levcon' by Kari Kivisalo (kkivisal@cc.hut.fi),
 * although I completely reimplemented the basic algorithm (and added
 * a few twists).
 */

#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "gimp.h"

typedef struct {
    long free;
    long inithgt;
    long maxiter;
    float limres;
    float force;
    long init_chk;
    int init_img;
    long kernel[3];
    long wrap;
} ContintpValues;

typedef struct {
    int h, w;
    int ci, cj; /* kernel center */
    float scale;
    char *k;
} Kernel;

/* Local functions */
static void free_callback(int, void *, void *);
static void inithgt_callback(int, void *, void *);
static void maxiter_callback(int, void *, void *);
/*
** Not used right now
**static void limres_callback(int, void *, void *);
**static void force_callback(int, void *, void *);
*/
static void button_callback(int, void *, void *);
static void radio_callback(int, void *, void *);
static void init_img_callback(int, void *, void *);
static void ok_callback(int, void *, void *);
static void cancel_callback(int, void *, void *);

static void contintp (Image, Image);

static char *prog_name;
static int dialog_ID;
static ContintpValues vals;
static Kernel *kernel;

static char k0[9] = {
    0, 1, 0,
    1, 0, 1,
    0, 1, 0
};

static char k1[25] = {
     0,  0, -1,  0,  0,
     0,  0,  4,  0,  0,
    -1,  4,  0,  4, -1,
     0,  0,  4,  0,  0,
     0,  0, -1,  0,  0
};

static char kdef[9] = {
    1, 1, 1,
    1, 0, 1,
    1, 1, 1
};

static Kernel kernels[3] = {
    { 3, 3, 1, 1, 0.25, k0 },
    { 5, 5, 2, 2, 0.08333333, k1 },
    { 3, 3, 1, 1, 0.125, kdef }
};

int main (int argc, char *argv[]) {
    Image input, output;
    int group_ID, temp_ID, free_ID, inithgt_ID, maxiter_ID,
        initgrp_ID, init_chk_ID, init_img_ID;
    int kernelgrp_ID, k3x3_ID, k2dpoly4_ID, kcust_ID, wrap_ID;
    /*
    ** Not used right now
    **int limres_ID, force_ID;
    */
    char buf[16];
    void *data;

    /* Save program name for error reporting */
    prog_name = argv[0];

    /* Call 'gimp_init' to initialize this filter */
    if (gimp_init(argc, argv)) {
        input = gimp_get_input_image(0);
        /* I may add an option for in-place conversion, but for now
         * we'll just create a new image, since that's the way most
         * people will probably use it.
         */
        /* output = gimp_get_output_image(0); */
        if (input) {
            /* Both images are available */
            /* For now, the image has to be grayscale.  I may change
             * this later to allow one channel of a color image
             */
            if (gimp_image_type(input) == GRAY_IMAGE) {
                data = gimp_get_params();
                if (data)
                    vals = *((ContintpValues *)data);
                else {
                    vals.free = 255;
                    vals.inithgt = 127;
                    vals.maxiter = 50;
                    vals.limres = 0.0;
                    vals.force = 0.0;
                    vals.init_chk = 0;
                    vals.init_img = 0;
                    vals.kernel[0] = 1;
                    vals.kernel[1] = 0;
                    vals.kernel[2] = 0;
                    vals.wrap = 0;
                }
                /* Set up dialog */
                dialog_ID = gimp_new_dialog("Contour Interpolation");
                gimp_new_label(dialog_ID, DEFAULT, "Options");
                group_ID = gimp_new_row_group(dialog_ID, DEFAULT, NORMAL, "");
                /* -free */
                temp_ID = gimp_new_column_group(dialog_ID, group_ID, NORMAL,
                                                "");
                gimp_new_label(dialog_ID, temp_ID, "Free Pixel:");
                sprintf(buf, "%ld", vals.free);
                free_ID = gimp_new_text(dialog_ID, temp_ID, buf);
                gimp_add_callback(dialog_ID, free_ID, free_callback,
                                  &vals.free);
                /* -inithgt */
                temp_ID = gimp_new_column_group(dialog_ID, group_ID, NORMAL,
                                                "");
                gimp_new_label(dialog_ID, temp_ID, "Init Height:");
                sprintf(buf, "%ld", vals.inithgt);
                inithgt_ID = gimp_new_text(dialog_ID, temp_ID, buf);
                gimp_add_callback(dialog_ID, inithgt_ID, inithgt_callback,
                                  &vals.inithgt);
                /* -maxiter */
                temp_ID = gimp_new_column_group(dialog_ID, group_ID, NORMAL,
                                                "");
                gimp_new_label(dialog_ID, temp_ID, "Iterations:");
                sprintf(buf, "%ld", vals.maxiter);
                maxiter_ID = gimp_new_text(dialog_ID, temp_ID, buf);
                gimp_add_callback(dialog_ID, maxiter_ID, maxiter_callback,
                                  &vals.maxiter);
                /* -limres */
                /*
                **temp_ID = gimp_new_column_group(dialog_ID, group_ID, NORMAL,
                **                                "");
                **gimp_new_label(dialog_ID, temp_ID, "Resolution:");
                **sprintf(buf, "%0.3f", vals.limres);
                **limres_ID = gimp_new_text(dialog_ID, temp_ID, buf);
                **gimp_add_callback(dialog_ID, limres_ID, limres_callback,
                **                  &vals.limres);
                */
                /* -force */
                /*
                **temp_ID = gimp_new_column_group(dialog_ID, group_ID, NORMAL,
                **                                "");
                **gimp_new_label(dialog_ID, temp_ID, "Force:");
                **sprintf(buf, "%0.3f", vals.force);
                **force_ID = gimp_new_text(dialog_ID, temp_ID, buf);
                **gimp_add_callback(dialog_ID, force_ID, force_callback,
                **                  &vals.force);
                */
                /* Edge wrap */
                wrap_ID = gimp_new_check_button(dialog_ID, group_ID,
                                                    "Wrap at edges");
                gimp_change_item(dialog_ID, wrap_ID, sizeof(vals.wrap),
                                 &vals.wrap);
                gimp_add_callback(dialog_ID, wrap_ID, button_callback,
                                  &vals.wrap);
                /* Initial image group */
                initgrp_ID = gimp_new_column_group(dialog_ID, group_ID, NORMAL,
                                                   "");
                init_chk_ID = gimp_new_check_button(dialog_ID, initgrp_ID,
                                                    "Initial Image");
                gimp_change_item(dialog_ID, init_chk_ID, sizeof(vals.init_chk),
                                 &vals.init_chk);
                gimp_add_callback(dialog_ID, init_chk_ID, button_callback,
                                  &vals.init_chk);
                init_img_ID = gimp_new_image_menu(dialog_ID, initgrp_ID,
                                                  IMAGE_CONSTRAIN_GRAY,
                                                  "");
                gimp_add_callback(dialog_ID, init_img_ID, init_img_callback,
                                  &vals.init_img);
                /* Old 3x3 filter */
                kernelgrp_ID = gimp_new_row_group(dialog_ID, group_ID,
                                                  RADIO, "Kernel");
                k3x3_ID = gimp_new_radio_button(dialog_ID, kernelgrp_ID,
                                                "3x3");
                gimp_change_item(dialog_ID, k3x3_ID,
                                 sizeof(vals.kernel[0]), &vals.kernel[0]);
                gimp_add_callback(dialog_ID, k3x3_ID, radio_callback,
                                  &vals.kernel[0]);
                k2dpoly4_ID = gimp_new_radio_button(dialog_ID, kernelgrp_ID,
                                                    "poly4");
                gimp_change_item(dialog_ID, k2dpoly4_ID,
                                 sizeof(vals.kernel[1]), &vals.kernel[1]);
                gimp_add_callback(dialog_ID, k2dpoly4_ID, radio_callback,
                                  &vals.kernel[1]);
                kcust_ID = gimp_new_radio_button(dialog_ID, kernelgrp_ID,
                                                 "custom");
                gimp_change_item(dialog_ID, kcust_ID,
                                 sizeof(vals.kernel[2]), &vals.kernel[2]);
                gimp_add_callback(dialog_ID, kcust_ID, radio_callback,
                                  &vals.kernel[2]);
                /* OK button */
                gimp_add_callback(dialog_ID, gimp_ok_item_id(dialog_ID),
                                  ok_callback, 0);
                /* Cancel button */
                gimp_add_callback(dialog_ID, gimp_cancel_item_id(dialog_ID),
                                  cancel_callback, 0);
                /* Now show the dialog */
                if (gimp_show_dialog(dialog_ID)) {
                    output = gimp_new_image(0,
                                            gimp_image_width(input),
                                            gimp_image_height(input),
                                            gimp_image_type(input));
                    if (output) {
                        gimp_set_params(sizeof(ContintpValues), &vals);
                        gimp_init_progress("Contour interpolation");
                        contintp(input, output);
                        gimp_display_image(output);
                        gimp_update_image(output);
                        gimp_free_image(output);
                    }
                }
            } else {
                gimp_message("contintp: can only operate on grayscale images");
            }
        }
        gimp_free_image(input);
        gimp_quit();
    }
    return 0;
}

/* Callbacks */

static void free_callback(int item_ID, void *client_data, void *call_data) {
    *((long *)client_data) = atoi(call_data);
}

static void inithgt_callback(int item_ID, void *client_data, void *call_data) {
    *((long *)client_data) = atoi(call_data);
}

static void maxiter_callback(int item_ID, void *client_data, void *call_data) {
    *((long *)client_data) = atoi(call_data);
}
/*
** Not used right now
**static void limres_callback(int item_ID, void *client_data,void *call_data) {
**    *((float *)client_data) = atof(call_data);
**}
**
**static void force_callback(int item_ID, void *client_data, void *call_data) {
**    *((float *)client_data) = atof(call_data);
**}
*/
static void button_callback(int item_ID, void *client_data, void *call_data) {
    *((long *)client_data) = *((long *)call_data);
}

static void radio_callback(int item_ID, void *client_data, void *call_data) {
    *((long *)client_data) = *((long *)call_data);
}

static void init_img_callback(int item_ID, void *client_data,
                              void *call_data) {
    *((long *)client_data) = *((long *)call_data);
}

static void ok_callback (item_ID, client_data, call_data)
     int item_ID;
     void *client_data;
     void *call_data;
{
  gimp_close_dialog (dialog_ID, 1);
}

static void cancel_callback (item_ID, client_data, call_data)
     int item_ID;
     void *client_data;
     void *call_data;
{
  gimp_close_dialog (dialog_ID, 0);
}

void interpolate(unsigned char *src, float **dest, long width,
                 long height, long size) {
    long i, j, ki, kj, kimax, kjmax, k, n, si, sj;
    float sum_1;

    kimax = kernel->h - kernel->ci;
    kjmax = kernel->w - kernel->cj;
    n = 0;
    for (i = 0; i < height; i++) {
        for (j = 0; j < width; j++) {
            if (src[n] != vals.free) {
                n++;
                continue;
            }
            k = 0;
            sum_1 = 0.0;
            for (ki = -kernel->ci; ki < kimax; ki++) {
                for (kj = -kernel->cj; kj < kjmax; kj++) {
                    if (kernel->k[k] == 0) {
                        k++;
                        continue;
                    }
                    si = ki + i;
                    if (si < 0) {
                        if (vals.wrap) {
                            si += height;
                        } else {
                            si = 0;
                        }
                    } else if (si >= height) {
                        if (vals.wrap) {
                            si -= height;
                        } else {
                            si = height - 1;
                        }
                    }
                    sj = kj + j;
                    if (sj < 0) {
                        if (vals.wrap) {
                            sj += width;
                        } else {
                            sj = 0;
                        }
                    } else if (sj >= width) {
                        if (vals.wrap) {
                            sj -= width;
                        } else {
                            sj = width - 1;
                        }
                    }
                    if (kernel->k[k] == 1) {
                        sum_1 += dest[si][sj];
                    } else {
                        sum_1 += dest[si][sj] * kernel->k[k];
                    }
                    k++;
                }
            }
            dest[i][j] = sum_1 * kernel->scale;
            n++;
        }
    }
}

void interp_3x3_4c(unsigned char *src, float **dest, long width,
                 long height, long size) {
    /* Optimized for kernels[0] */
    long i, j, n, i_1, j_1, i1, j1, h_1, w_1;
    float sum_1;

    h_1 = height - 1;
    w_1 = width - 1;
    n = 0;
    for (i = 0; i < height; i++) {
        for (j = 0; j < width; j++) {
            if (src[n] != vals.free) {
                n++;
                continue;
            }
            if (i == 0) {
                if (vals.wrap) {
                    i_1 = h_1;
                } else {
                    i_1 = 0;
                }
                i1 = 1;
            } else if (i == h_1) {
                if (vals.wrap) {
                    i1 = 0;
                } else {
                    i1 = h_1;
                }
                i_1 = i-1;
            } else {
                i_1 = i-1;
                i1 = i+1;
            }
            if (j == 0) {
                if (vals.wrap) {
                    j_1 = w_1;
                } else {
                    j_1 = 0;
                }
                j1 = 1;
            } else if (j == w_1) {
                if (vals.wrap) {
                    j1 = 0;
                } else {
                    j1 = w_1;
                }
                j_1 = j-1;
            } else {
                j_1 = j-1;
                j1 = j+1;
            }
            sum_1 = dest[i_1][j] + dest[i1][j] + dest[i][j_1] + dest[i][j1];
            dest[i][j] = sum_1 * 0.25;
            n++;
        }
    }
}

static void contintp(Image input, Image output) {
    long width, height, size;
    long channels, rowstride;
    Image init_img;
    unsigned char *src, *dest, *init = NULL;
    float **destf;
    long n, i, j, k_no, iter = 0;

    width = gimp_image_width(input);
    height = gimp_image_height(input);
    channels = gimp_image_channels(input);
    rowstride = width * channels;
    size = width * height;
    src = gimp_image_data(input);
    dest = gimp_image_data(output);
    if (vals.init_chk) {
        init_img = gimp_get_input_image(vals.init_img);
        init = gimp_image_data(init_img);
    }
    for (k_no = 0; k_no < sizeof(vals.kernel); k_no++) {
        if (vals.kernel[k_no]) {
            if (k_no == sizeof(vals.kernel)-1) {
                /* Custom kernel - not yet implemented */
                kernel = &kernels[k_no];
                break;
            }
            kernel = &kernels[k_no];
            break;
        }
    }
    destf = (float **)malloc(sizeof(float *) * height);
    n = 0;
    for (i = 0; i < height; i++) {
        destf[i] = (float *)malloc(sizeof(float) * width);
        for (j = 0; j < width; j++) {
            if (src[n] == vals.free) {
                if (vals.init_chk) {
                    destf[i][j] = init[n];
                } else {
                    destf[i][j] = vals.inithgt;
                }
            } else {
                destf[i][j] = src[n];
            }
            n++;
        }
    }
    while ((iter < vals.maxiter) && ( /* some sort of delta limit*/ 1)) {
        if ((k_no == 0) || (k_no > sizeof(vals.kernel)))
            interp_3x3_4c(src, destf, width, height, size);
        else
            interpolate(src, destf, width, height, size);
        iter++;
        gimp_do_progress(iter, vals.maxiter);
    }
    n = 0;
    for (i = 0; i < height; i++) {
        for (j = 0; j < width; j++) {
            if (destf[i][j] > 255) {
                dest[n] = 255;
            } else if (destf[i][j] < 0) {
                dest[n] = 0;
            } else {
                dest[n] = destf[i][j];
            }
            n++;
        }
    }
}
