/* xxxxx -- an audio manipulation program
 * Copyright (C) 1997-1998 Larry Ewing
 *
 * Large portions of this code inspired by :)
 * The Gimp - (C) 1995 Spencer Kimball and Peter Mattis 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <glib.h>
#include "at_track.h"
#include "at_clip.h"
#include "at_view.h"
#include "at_type.h"

#define MIX_CHUNK_SIZE 1024

/* static variables */
GSList *clip_list = NULL;
gint global_clip_ID = 1;

/* Clip functions */
static void
at_clip_mix_tracks (Clip *clip, size_t offset, size_t length);

static gint
at_clip_idle_mix (gpointer data);

static void 
at_clip_mixing_complete (Clip *clip);


static Clip *
at_clip_create (void)
{
  Clip *clip;
 
  clip = (Clip *) g_malloc (sizeof (Clip));

  clip->title = NULL;
  clip->filename = NULL;
  clip->has_filename = FALSE;
  
  clip->composite = NULL;
  clip->selection = NULL;
  
  clip->dirty = FALSE;
  clip->undo_on = TRUE;
  
  clip->instance_count = 0;
  clip->ref_count = 0;
  
  clip->ID = global_clip_ID++;
  clip->guides = NULL;

  clip->tracks = NULL;
  clip->channels = NULL;
  clip->track_stack = NULL;
  clip->active_track = 0;
  clip->undo_stack = NULL;
  clip->redo_stack = NULL;
  clip->undo_bytes = 0;
  clip->undo_levels = 0;
  clip->pushing_undo_group = 0;

  clip->recording = FALSE;
  clip->playing = FALSE;
  
  clip_list = g_slist_append (clip_list, clip);

  return clip;
}
  
Clip *
at_clip_new (size_t length, gint base_type, gint sample_rate)
{
  Clip *clip;
  int i;

  clip = at_clip_create ();
  
  clip->filename = NULL;
  clip->length = length;
  
  clip->base_type = base_type;
  clip->sample_rate = sample_rate;
  clip->active_track = -1;
  clip->active_channel = -1;

  for (i = 0; i < MAX_CHANNELS; i++)
    {
      clip->mute[i] = FALSE;
      clip->active[i] = TRUE;
    }   
  
  /* FIXME base type is simply not correct here */
  clip->composite = region_new (length, at_type_bytes (base_type));
  /* FIXME This is wrong too I think.... */
  clip->selection  = region_new (length, at_type_bytes (AT_MONO));

  return clip;

}

Clip *
at_clip_get_named (char *name)
{
  GSList *current = clip_list;
  Clip *clip;

   while (current) 
     {
       clip = (Clip *) (current->data);
       if (strcmp (clip->title, name) == 0)
	 return clip;

       current = current->next;
     }

   return NULL;  
}

Clip *
at_clip_get_ID (int clip_ID)
{
  GSList *current = clip_list;
  Clip *clip;

   while (current) 
     {
       clip = (Clip *) (current->data);
       if (clip->ID == clip_ID)
	 return clip;

       current = current->next;
     }

   return NULL;
}

void 
at_clip_resize (Clip *clip, size_t new_length, size_t new_offset)
{
  GSList *list = clip->tracks;
  Track *track;

  if (new_length <= 0)
    {
      g_warning ("at_clip_resize: length must be positive");
      return;
    }

  /* Push the undo mod onto the stack */
  /* 
     undo_push_group_start (clip, CLIP_MOD_UNDO);
     undo_push_clip_mod (clip);
     */

  g_print ("resizing clip - new_length = %d\n", new_length);

  clip->length = new_length;

  /* resize the channels and the selection mask here */
  if (clip->composite)
    region_resize (clip->composite, new_offset, new_length);
    
  if (clip->selection)
    region_resize (clip->selection, new_offset, new_length);
  
  /* translate all the tracks to the new start offset */
  while (list)
    {
      track = (Track *) list->data;
      at_track_translate (track, new_offset);
      list = list->next;
    }

  /* undo_push_group_end */
  /* at_clip_invalidate_thumbnails (clip);
  at_composite_update_full (clip->ID);*/
}

void 
at_clip_mix_invalidate (Clip *clip)
{
}

Track *
at_clip_add_track (Clip *clip, Track *track,  int position)
{
  if (track->clip_ID != 0 && track->clip_ID != clip->ID)
    {
      g_warning ("at_clip_add_track: attempt to add track to wrong clip");
      return NULL;
    }

  /* FIXME we need to deal with undo here */

  if (position == -1)
    position = at_clip_get_track_index (clip, clip->active_track);

  if (position != -1)
    clip->tracks = g_slist_insert (clip->tracks, track, position);  
  else
    clip->tracks = g_slist_prepend (clip->tracks, track);
  
  clip->track_stack = g_slist_prepend (clip->track_stack, track);
 
  /* update the active track to the new track */
  at_clip_set_active_track (clip, track->ID);

  /* update the region.??? */
  at_clip_mix_invalidate (clip);

  return track;
}

Track *
at_clip_remove_track (Clip *clip, gint track_ID)
{
  Track *track;

  g_warning ("removing track %d", track_ID);
  if ((track = at_track_get_ID (track_ID)))
    {
      /* FIXME undo.... */
      
      clip->tracks = g_slist_remove (clip->tracks, track);
      clip->track_stack = g_slist_remove (clip->track_stack, track);
      
      if (track_ID == clip->active_track) {
	if (clip->tracks) {
	  clip->active_track = ((Track *) clip->track_stack->data)->ID;
	} else {
	  clip->active_track = -1;
	}
      }

      /* at_views_update_area (clip->ID, track->start_offset, track->length); 
       */

      at_clip_mix_invalidate (clip);
    
      return NULL;
    }
  else 
    return NULL;

}

gint
at_clip_raise_track (Clip *clip, gint track_ID)
{
  GSList *list = clip->tracks;
  GSList *prev = NULL;
  gint    pos = 0;
  Track  *track = NULL;

  while (list)
    {
      track = list->data;
      if (track->ID == track_ID)
	break;
      prev = list;
      list = list->next;
      pos++;
    }

  clip->tracks = g_slist_remove (clip->tracks, list->data);
  clip->tracks = g_slist_insert (clip->tracks, (gpointer)track, ++pos);
  
  return pos;
}

gint
at_clip_lower_track (Clip *clip, gint track_ID)
{
  GSList *list = clip->tracks;
  GSList *prev = NULL;
  gint    pos = 0;
  Track  *track = NULL;

  while (list)
    {
      track = list->data;
      if (track->ID == track_ID)
	break;
      prev = list;
      list = list->next;
      pos++;
    }

  clip->tracks = g_slist_remove (clip->tracks, list->data);
  clip->tracks = g_slist_insert (clip->tracks,
				 (gpointer)track,
				 --pos < 0 ? 0 : pos);
  
  return pos;
}
void
at_clip_set_active_track (Clip *clip, gint track_ID) 
{
  clip->active_track = track_ID;

  /* FIXME this needs to update any previews etc... */

}

Track *
at_clip_get_active_track (Clip *clip)
{
  return at_track_get_ID (clip->active_track);
}

/**************************************************
 * Keep tabs on recording and playback of clips
 *************************************************/

void
at_clip_record_start (Clip *clip)
{
  if (clip) {
      clip->recording = TRUE;
  }
}

void
at_clip_record_done (Clip *clip)
{
  if (clip) {
      clip->recording = FALSE;
  }
  
  at_track_record_done (clip);
  clip->dirty = TRUE;
  
  at_clip_update (clip);

}

gint
at_clip_recording (Clip *clip)
{
  return (clip->recording);
}

void
at_clip_play_start (Clip *clip)
{
  if (clip) {
      clip->playing = TRUE;
  }
}

void
at_clip_play_done (Clip *clip)
{
  if (clip) {
      clip->playing = FALSE;
  }
}

gint
at_clip_playing (Clip *clip)
{
  return (clip->playing);
}


gint
at_clip_get_track_index (Clip *clip, gint track_ID)
{
  Track *track;
  GSList *tracks = clip->tracks;
  int index = 1;

  while (tracks)
    {
      track = (Track *) tracks->data;
      if (track->ID == track_ID)
	return index;
      
      index++;
      tracks = tracks->next;
    }
  
  return -1;
}

void
at_clip_update (Clip *clip) 
{
  if (clip->dirty)
    at_clip_mix_tracks (clip, 0, clip->length);
    
} 
    


void
at_clip_new_cmd_callback (GtkWidget *widget, gpointer data)
{
  Clip *clip;
  View *view;
  Track *track;

  clip = at_clip_new (3200, AT_RL, 44100);
  view = at_view_new (clip);

  /* create a track to view the composite on */
  track = at_track_new (view->clip->ID, 0, 200, AT_RL, "Composite", 1.0, 0);
  at_clip_add_track (view->clip, track, 
		     at_clip_get_track_index (view->clip,
					      view->clip->active_track));

  track->data = clip->composite;
  clip->composite_track = track;

  /* create a first track for recording on */
  track = at_track_new (view->clip->ID, 0, 200, AT_RL, "New Track", 1.0, 0);
  at_clip_add_track (view->clip, track, 
		     at_clip_get_track_index (view->clip,
					      view->clip->active_track));

  at_views_update (view->clip->ID);
}


gchar *
at_clip_get_play_file (Clip *clip)
{  
  region_sync (clip->composite);
  return clip->composite->filename;
}

		      

/* Yes, I know, there is nothing in here with offsets.  I'll have to add
 * it in later, but it works much better than the old code IMHO */

static void
at_clip_mix_tracks (Clip *clip, size_t offset, size_t length)
{
  Track  *track;
  size_t  x1, x2;
  GSList  *list = clip->tracks;
  GSList  *track_list = NULL;
  gint track_count = 0;
  TmpRegion *mix_dest;
  Mix *mix_data = &clip->mix;

  if (!clip->dirty)
    return;

  clip->dirty = FALSE;

  g_print ("mixing tracks.... \n");

  region_resize (clip->composite, 0, clip->length);
  /* zero out the composition track */
  mix_dest = tmp_region_new (clip->composite, 0, clip->length);
  tmp_region_zero (mix_dest);

  /* Create a list of all tracks that need mixing */
  while (list)
    {
      track = (Track *) list->data;
      
      if (!track->mute)
	{
	  track_list = g_slist_prepend (track_list, track);
	  track_count ++;
	}

      list = list->next;
    }

  if (track_count == 0)
    return;

  mix_data->track_count = track_count;
  mix_data->track_list = track_list;
  mix_data->list = track_list;
  mix_data->position = 0;
  mix_data->current_track = NULL;
  mix_data->composite_region = mix_dest;
  mix_data->bps = clip->composite->bps;

  /* set up the idle to mix the tracks */
  mix_data->record_tag = gtk_idle_add (at_clip_idle_mix, clip);

  clip->mixing = TRUE;
  
}

static gint
at_clip_idle_mix (gpointer data)
{
  Clip *clip = data;
  Mix *mix = &clip->mix;
  gint track_done = FALSE;
  gint count, i;
  atdata *dest_data, *src_data;
  gint track_size;



  /* check if there's a track on the queue */
  if (!mix->current_track) {

    /* if we're at the end of the track list, 
       we're done mixing */
    if (mix->list == NULL) {
      /* we have now completed the mixing */
      /* divide data by number of tracks */
      dest_data = mix->composite_region->data;
      track_size = mix->composite_region->bps * mix->composite_region->length / sizeof (atdata);
      
      if (mix->track_count != 1) {
	for (i=0; i < track_size; i++)
	  dest_data[i] /= mix->track_count;
      }

      g_print ("Idle mixing completed... \n");

      g_slist_free (mix->track_list);
      tmp_region_destroy (mix->composite_region);
      
      at_clip_mixing_complete(clip);
      
      gtk_idle_remove (mix->record_tag);
      return (FALSE);
    }

    g_print ("queuing new track for mixing ..\n");

    /* get next track from linked list */
    mix->current_track = mix->list->data;

    /* map track */
    mix->track_region = tmp_region_new (mix->current_track->data, 0,
					mix->current_track->length);
    /* since this is a new track, set the position to 0 */
    mix->position = 0;
    mix->list = mix->list->next;
  }

  track_size = mix->current_track->data->bps * mix->current_track->length / sizeof (atdata);

  count = MIX_CHUNK_SIZE * mix->bps;

  if (count + mix->position > track_size) {
    count = track_size - mix->position;
    track_done = TRUE;
  }

  dest_data = mix->composite_region->data;
  dest_data += mix->position;
  src_data = mix->track_region->data;
  src_data += mix->position;
  
  mix->position += count;

  /* perform the actual mixing */
  /* FIXME: handle number of channels correctly */
  for (i=0; i < count; i++) {
      dest_data[i] += src_data[i] * mix->current_track->volume;
  }
  
  if (track_done) {
    g_print ("track completed\n");
    tmp_region_destroy (mix->track_region);
    mix->current_track = NULL;
  }

  return (TRUE);
}


static void 
at_clip_mixing_complete (Clip *clip)
{
  clip->composite_track->length = clip->composite->length;
  at_track_build_preview (clip->composite_track);
  clip->mixing = FALSE;
}











