/* The GIMP -- an image manipulation program
 * Copyright (C) 1995 Spencer Kimball and Peter Mattis
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#ifndef __CANVAS_H__
#define __CANVAS_H__

#include "tag.h"
#include "memory.h"

#define GIMP_BUFFER(b)        ((GimpBuffer*)(b))
#define GIMP_BUFFER_CLASS(b)  ((GimpBufferClass*)(b))

typedef struct _GimpBuffer GimpBuffer;
typedef struct _GimpBufferClass GimpBufferClass;
typedef struct _GimpPoint GimpPoint;
typedef struct _GimpArea GimpArea;
typedef struct _GimpPortion GimpPortion;
typedef struct _GimpPixelArray GimpPixelArray;
typedef struct _GimpMemStatus GimpMemStatus;



/* all the GimpBuffer subclasses that exist */
typedef enum
{
  STORAGE_NONE,
  STORAGE_FLAT,
  STORAGE_TILED,
  STORAGE_SHM
} Storage;

/* how to alloc a portion */
typedef enum
{
  ALLOC_NONE,
  ALLOC_ALLOC,
  ALLOC_UNALLOC
} Alloc;

/* how to initialize a portion */
typedef enum
{
  VALIDATE_NONE,
  VALIDATE_VALIDATE,
  VALIDATE_INVALIDATE
} Validate;

/* how to use a portion */
typedef enum
{
  USE_NONE,
  USE_READ,
  USE_WRITE,
  USE_UPDATE,  /* non-COW WRITE */
  USE_RELEASE
} Use;



struct _GimpPoint {
  gint x;
  gint y;
};

struct _GimpArea {
  GimpPoint a;
  GimpPoint b;
};

struct _GimpPortion
{
  GimpArea focus;
  GimpArea bounds;
};



/* an array of pixels */
struct _GimpPixelArray
{
  /* the layout of each element */
  Tag tag;

  /* the array bounds */
  gint width;
  gint height;

  /* the element spacing */
  gint pixstride;
  gint rowstride;

  /* reference to first element */
  GimpMemPtr * data;
  gint offset;
};

#define gpa_tag(gpa)         ((gpa)->tag)
#define gpa_width(gpa)       ((gpa)->width)
#define gpa_height(gpa)      ((gpa)->height)
#define gpa_pixstride(gpa)   ((gpa)->pixstride)
#define gpa_rowstride(gpa)   ((gpa)->rowstride)
#define gpa_data(gpa)        ((gpa)->data)
#define gpa_offset(gpa)      ((gpa)->offset)

#define gpa_dataptr(gpa)     ((void*) \
                              (guchar_d((gpa)->data) + \
                               (gpa)->offset))


/* info about a GimpMemPtr */
struct _GimpMemStatus
{
  /* does the portion have memory allocated? */
  gboolean alloced;
  
  /* does the memory contain valid data? */
  gboolean valid;

  /* how many places are using it? */
  gint usecount;
};




struct _GimpBuffer
{
  /* the class object */
  void * klass;

  /* buffer size */
  gint width;    /* in pixels */
  gint height;   /* in pixels */
  gint depth;    /* in bytes  */

  /* the tiling characteristics */
  struct {
    gint period;
    gint phase;
  } x, y;

  /* pixel layout */
  Tag tag;

  /* should invalid memory be validated when used? */
  gboolean autovalidate;

  /* should unalloced memory be alloced when validated? */
  gboolean autoalloc;
};


struct _GimpBufferClass
{
  /* subclass type */
  Storage type;

  /* virtual method table */
  void      (* delete)  (GimpBuffer        *buffer);
  gboolean  (* alloc)   (GimpBuffer        *buffer,
                         GimpPortion       *portion,
                         Alloc              how);
  gboolean  (* validate)(GimpBuffer        *buffer,
                         GimpPortion       *portion,
                         Validate           how);
  gboolean  (* use)     (GimpBuffer        *buffer,
                         GimpPortion       *portion, 
                         Use                how);
  gboolean  (* query)   (GimpBuffer        *buffer,
                         GimpPortion       *portion,
                         GimpMemStatus     *status);
  gboolean  (* data)    (GimpBuffer        *buffer,
                         GimpPortion       *portion,
                         GimpPixelArray    *array);
};


/* for use by subclasses */
void        gimp_buffer_init     (GimpBuffer *buffer,
                                  Tag         tag,
                                  gint        width,
                                  gint        height,
                                  gint        x_period,
                                  gint        x_phase,
                                  gint        y_period,
                                  gint        y_phase);
void        gimp_buffer_uninit   (GimpBuffer *buffer);



/* public methods */

/* describe the buffer's xyz dimensions */
gint        gimp_buffer_width    (GimpBuffer        *buffer);
gint        gimp_buffer_height   (GimpBuffer        *buffer);
gint        gimp_buffer_depth    (GimpBuffer        *buffer);

/* add accessors for tag, tiling structure, auto_{alloc,validate} */


/* destroy the buffer */
void        gimp_buffer_delete   (GimpBuffer        *buffer);

/* given a region of interest, find the contiguous bounding box and
   narrow the region if required */
gboolean    gimp_buffer_focus    (GimpBuffer        *buffer,
                                  GimpPortion       *portion,
                                  GimpArea          *roi);

/* given a portion, allocate or deallocate the memory supporting it.
   do not deallocate data with outstanding uses */
gboolean    gimp_buffer_alloc    (GimpBuffer        *buffer,
                                  GimpPortion       *portion,
                                  Alloc              how);

/* given a portion, ensure it has memory supporting it and cause the
   contents to become valid or invalid.  attempt to allocate data if
   required and allowed.  do not invalidate data with outstanding
   uses. */
gboolean    gimp_buffer_validate (GimpBuffer        *buffer,
                                  GimpPortion       *portion,
                                  Validate           how);

/* given a portion, ensure it has valid contents and assert/negate
   your intention to use those contents in some manner.  attempt to
   validate memory if required */
gboolean    gimp_buffer_use      (GimpBuffer        *buffer,
                                  GimpPortion       *portion,
                                  Use                how);

/* given a portion, get the alloc/valid/usecount status of the memory
   supporting it */
gboolean    gimp_buffer_query    (GimpBuffer        *buffer,
                                  GimpPortion       *portion,
                                  GimpMemStatus     *status);

/* given a portion, ensure that it has at least one outstanding use,
   then fill the data structure with information describing the
   current physical location of the memory */
gboolean    gimp_buffer_data     (GimpBuffer        *buffer,
                                  GimpPortion       *portion,
                                  GimpPixelArray    *array);


#endif /* __CANVAS_H__ */
